"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cleanupCdkTfOutDirs = exports.AssertionReturn = exports.TerraformPlan = exports.TerraformApplyCheckAndDestroy = exports.TerraformApplyAndCheckIdempotency = exports.TerraformDestroy = exports.TerraformIdempotentCheck = exports.TerraformPlanExitCode = exports.TerraformApply = exports.execTerraformCommand = exports.BaseTestStack = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const cdktf = require("cdktf");
const cdktf_1 = require("cdktf");
class BaseTestStack extends cdktf_1.TerraformStack {
    constructor(scope, id) {
        super(scope, id);
        const name = new cdktf.TerraformVariable(this, "name", {
            type: "string",
            default: "test",
            description: "System name used to randomize the resources",
        });
        this.name = name.value;
    }
}
exports.BaseTestStack = BaseTestStack;
function execTerraformCommand(command, opts, streamOutput) {
    try {
        if (streamOutput) {
            (0, child_process_1.execSync)(command, { ...opts, stdio: "inherit" });
            return { stdout: "", stderr: "", exitCode: 0 };
        }
        else {
            const stdout = (0, child_process_1.execSync)(command, { ...opts, stdio: "pipe" }).toString();
            return { stdout, stderr: "", exitCode: 0 };
        }
    }
    catch (error) {
        const stdout = error.stdout ? error.stdout.toString() : "";
        const stderr = error.stderr ? error.stderr.toString() : "";
        return { stdout, stderr, exitCode: error.status || 1 };
    }
}
exports.execTerraformCommand = execTerraformCommand;
function TerraformApply(stack, streamOutput = false) {
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(stack, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        for (const [, stackDetails] of stacks) {
            const opts = {
                cwd: path.resolve(stack, stackDetails.workingDirectory),
                env: process.env,
                encoding: "utf-8",
            };
            const initResult = execTerraformCommand(`terraform init`, opts, streamOutput);
            if (initResult.exitCode !== 0)
                throw new Error(`Terraform init failed: ${initResult.stdout} ${initResult.stderr}`);
            const applyResult = execTerraformCommand(`terraform apply -auto-approve`, opts, streamOutput);
            if (applyResult.exitCode !== 0) {
                return {
                    stdout: applyResult.stdout,
                    stderr: applyResult.stderr,
                    error: new Error(`Terraform apply failed: ${applyResult.stdout} ${applyResult.stderr}`),
                };
            }
        }
        return { stdout: "Terraform apply succeeded", stderr: "", error: null };
    }
    catch (error) {
        console.error("Error during Terraform init or apply:", error);
        return { stdout: "", stderr: "", error };
    }
}
exports.TerraformApply = TerraformApply;
function TerraformPlanExitCode(stack, streamOutput = false) {
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(stack, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        for (const [, stackDetails] of stacks) {
            const opts = {
                cwd: path.resolve(stack, stackDetails.workingDirectory),
                env: process.env,
                encoding: "utf-8",
            };
            const initResult = execTerraformCommand(`terraform init`, opts, streamOutput);
            if (initResult.exitCode !== 0)
                throw new Error(`Terraform init failed: ${initResult.stdout} ${initResult.stderr}`);
            const planResult = execTerraformCommand(`terraform plan -input=false -lock=false -detailed-exitcode`, opts, streamOutput);
            return {
                stdout: planResult.stdout,
                stderr: planResult.stderr,
                exitCode: planResult.exitCode,
                error: null,
            };
        }
        return {
            stdout: "Terraform plan succeeded",
            stderr: "",
            exitCode: 0,
            error: null,
        };
    }
    catch (error) {
        console.error("Error during Terraform init or plan:", error);
        return { stdout: "", stderr: "", exitCode: 1, error };
    }
}
exports.TerraformPlanExitCode = TerraformPlanExitCode;
function TerraformIdempotentCheck(stack) {
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(stack, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        for (const [, stackDetails] of stacks) {
            const opts = {
                cwd: path.resolve(stack, stackDetails.workingDirectory),
                env: process.env,
                stdio: "pipe",
            };
            (0, child_process_1.execSync)(`terraform init`, opts);
            let planOutput;
            try {
                planOutput = (0, child_process_1.execSync)(`terraform plan -input=false -detailed-exitcode`, opts).toString();
            }
            catch (error) {
                planOutput = error.stdout ? error.stdout.toString() : "";
            }
            if (planOutput.includes("No changes. Your infrastructure matches the configuration.")) {
                return new AssertionReturn(`Expected subject to be idempotent with no changes`, true);
            }
            else {
                throw new Error(`Plan resulted in changes:\n${planOutput}`);
            }
        }
        return new AssertionReturn(`Expected subject to be idempotent with no changes`, true);
    }
    catch (error) {
        console.error("Error during Terraform init or plan:", error);
        throw error;
    }
}
exports.TerraformIdempotentCheck = TerraformIdempotentCheck;
function TerraformDestroy(stack, streamOutput = false) {
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(stack, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        for (const [, stackDetails] of stacks) {
            const opts = {
                cwd: path.resolve(stack, stackDetails.workingDirectory),
                env: process.env,
                encoding: "utf-8",
            };
            const initResult = execTerraformCommand(`terraform init`, opts, streamOutput);
            if (initResult.exitCode !== 0)
                throw new Error(`Terraform init failed: ${initResult.stdout} ${initResult.stderr}`);
            const destroyResult = execTerraformCommand(`terraform destroy -auto-approve`, opts, streamOutput);
            if (destroyResult.exitCode !== 0)
                throw new Error(`Terraform destroy failed: ${destroyResult.stdout} ${destroyResult.stderr}`);
        }
        return new AssertionReturn(`Expected subject to destroy successfully`, true);
    }
    catch (error) {
        console.error("Error during Terraform init or destroy:", error);
        return new AssertionReturn(`Expected subject to destroy successfully`, false);
    }
}
exports.TerraformDestroy = TerraformDestroy;
function TerraformApplyAndCheckIdempotency(stack, streamOutput = false) {
    const applyResult = TerraformApply(stack, streamOutput);
    if (applyResult.error) {
        return new AssertionReturn(applyResult.error.message, false);
    }
    const planResult = TerraformPlanExitCode(stack, streamOutput);
    if (planResult.error) {
        return new AssertionReturn(planResult.error.message, false);
    }
    if (planResult.exitCode !== 0) {
        return new AssertionReturn(`Terraform configuration not idempotent:\n${planResult.stdout}`, false);
    }
    return new AssertionReturn(`Terraform apply and idempotency check completed successfully`, true);
}
exports.TerraformApplyAndCheckIdempotency = TerraformApplyAndCheckIdempotency;
function TerraformApplyCheckAndDestroy(stack) {
    try {
        const applyAndCheckResult = TerraformApplyAndCheckIdempotency(stack);
        if (!applyAndCheckResult.pass) {
            throw new Error(applyAndCheckResult.message);
        }
    }
    catch (error) {
        console.error("Error during Terraform apply and idempotency check:", error);
        throw error; // Re-throw the error to ensure the test fails
    }
    finally {
        try {
            const destroyResult = TerraformDestroy(stack);
            if (!destroyResult.pass) {
                console.error("Error during Terraform destroy:", destroyResult.message);
            }
        }
        catch (destroyError) {
            console.error("Error during Terraform destroy:", destroyError);
        }
    }
}
exports.TerraformApplyCheckAndDestroy = TerraformApplyCheckAndDestroy;
function TerraformPlan(stack) {
    try {
        // Assuming the presence of a received variable pointing to the relevant directory
        const manifest = JSON.parse(fs.readFileSync(path.resolve(stack, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([, stackDetails]) => {
            const opts = {
                cwd: path.resolve(stack, stackDetails.workingDirectory),
                env: process.env,
                stdio: "pipe",
            };
            (0, child_process_1.execSync)(`terraform init`, opts);
            (0, child_process_1.execSync)(`terraform plan -input=false -lock=false`, opts);
        });
        return new AssertionReturn(`Expected subject not to plan successfully`, true);
    }
    catch (error) {
        console.error("Error during Terraform init or plan:", error);
        throw error;
        return new AssertionReturn("Expected subject to plan successfully, false", false);
    }
}
exports.TerraformPlan = TerraformPlan;
class AssertionReturn {
    /**
     * Create an AssertionReturn
     * @param message - String message containing information about the result of the assertion
     * @param pass - Boolean pass denoting the success of the assertion
     */
    constructor(message, pass) {
        this.message = message;
        this.pass = pass;
        if (!pass) {
            throw new Error(message);
        }
    }
}
exports.AssertionReturn = AssertionReturn;
// Define a cleanup function that can be exported and used in tests.
function cleanupCdkTfOutDirs() {
    try {
        (0, child_process_1.execSync)('find /tmp -name "cdktf.outdir.*" -type d -exec rm -rf {} +');
        // console.log(
        //   "Cleanup of cdktf.outdir.* directories in /tmp was successful.",
        // );
    }
    catch (error) {
        console.error("Error during cleanup:", error);
    }
}
exports.cleanupCdkTfOutDirs = cleanupCdkTfOutDirs;
//# sourceMappingURL=data:application/json;base64,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