"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirewallDomainList = exports.FirewallDomains = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const route53resolver_generated_1 = require("./route53resolver.generated");
/**
 * A list of domains
 */
class FirewallDomains {
    /**
     * Firewall domains created from a list of domains
     *
     * @param list the list of domains
     */
    static fromList(list) {
        for (const domain of list) {
            if (!/^([\w-.]{1,255}|\*[\w-.]{1,254})$/.test(domain)) {
                throw new Error(`Invalid domain: ${domain}. Domain can optionally start with *. Max length of 255. Valid characters: A-Z, a-z, 0-9, _, -, .`);
            }
        }
        return {
            bind(_scope) {
                return { domains: list };
            },
        };
    }
    /**
     * Firewall domains created from the URL of a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param url S3 bucket url (s3://bucket/prefix/objet).
     */
    static fromS3Url(url) {
        if (!core_1.Token.isUnresolved(url) && !url.startsWith('s3://')) {
            throw new Error(`The S3 URL must start with s3://, got ${url}`);
        }
        return {
            bind(_scope) {
                return { domainFileUrl: url };
            },
        };
    }
    /**
     * Firewall domains created from a file stored in Amazon S3.
     * The file must be a text file and must contain a single domain per line.
     * The content type of the S3 object must be `plain/text`.
     *
     * @param bucket S3 bucket
     * @param key S3 key
     */
    static fromS3(bucket, key) {
        return this.fromS3Url(bucket.s3UrlForObject(key));
    }
    /**
     * Firewall domains created from a local disk path to a text file.
     * The file must be a text file (`.txt` extension) and must contain a single
     * domain per line. It will be uploaded to S3.
     *
     * @param assetPath path to the text file
     */
    static fromAsset(assetPath) {
        // cdk-assets will correctly set the content type for the S3 object
        // if the file has the correct extension
        if (path.extname(assetPath) !== '.txt') {
            throw new Error(`FirewallDomains.fromAsset() expects a file with the .txt extension, got ${assetPath}`);
        }
        return {
            bind(scope) {
                const asset = new aws_s3_assets_1.Asset(scope, 'Domains', { path: assetPath });
                if (!asset.isFile) {
                    throw new Error('FirewallDomains.fromAsset() expects a file');
                }
                return { domainFileUrl: asset.s3ObjectUrl };
            },
        };
    }
}
exports.FirewallDomains = FirewallDomains;
_a = JSII_RTTI_SYMBOL_1;
FirewallDomains[_a] = { fqn: "@aws-cdk/aws-route53resolver.FirewallDomains", version: "1.171.0" };
/**
 * A Firewall Domain List
 */
class FirewallDomainList extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_route53resolver_FirewallDomainListProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, FirewallDomainList);
            }
            throw error;
        }
        if (props.name && !core_1.Token.isUnresolved(props.name) && !/^[\w-.]{1,128}$/.test(props.name)) {
            throw new Error(`Invalid domain list name: ${props.name}. The name must have 1-128 characters. Valid characters: A-Z, a-z, 0-9, _, -, .`);
        }
        const domainsConfig = props.domains.bind(this);
        const domainList = new route53resolver_generated_1.CfnFirewallDomainList(this, 'Resource', {
            name: props.name,
            domainFileUrl: domainsConfig.domainFileUrl,
            domains: domainsConfig.domains,
        });
        this.firewallDomainListId = domainList.attrId;
        this.firewallDomainListArn = domainList.attrArn;
        this.firewallDomainListCreationTime = domainList.attrCreationTime;
        this.firewallDomainListCreatorRequestId = domainList.attrCreatorRequestId;
        this.firewallDomainListDomainCount = domainList.attrDomainCount;
        this.firewallDomainListManagedOwnerName = domainList.attrManagedOwnerName;
        this.firewallDomainListModificationTime = domainList.attrModificationTime;
        this.firewallDomainListStatus = domainList.attrStatus;
        this.firewallDomainListStatusMessage = domainList.attrStatusMessage;
    }
    /**
     * Import an existing Firewall Rule Group
     */
    static fromFirewallDomainListId(scope, id, firewallDomainListId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.firewallDomainListId = firewallDomainListId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FirewallDomainList = FirewallDomainList;
_b = JSII_RTTI_SYMBOL_1;
FirewallDomainList[_b] = { fqn: "@aws-cdk/aws-route53resolver.FirewallDomainList", version: "1.171.0" };
//# sourceMappingURL=data:application/json;base64,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