"""
Interface to register functions on PiCloud that can be invoked via the REST API
"""

"""
Copyright (c) 2011 `PiCloud, Inc. <http://www.picloud.com>`_.  All rights reserved.

email: contact@picloud.com

The cloud package is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this package; if not, see 
http://www.gnu.org/licenses/lgpl-2.1.html
"""
import sys
import re

from .util.zip_packer import Packer
from .transport.adapter import SerializingAdapter

_register_query = 'rest/register/'
_deregister_query = 'rest/deregister/'
_list_query = 'rest/list/'

# TODO
#_call_query = 'rest/call/'
_info_query = 'rest/info/'

"""
This module utilizes the cloud object extensively
The functions can be viewed as instance methods of the Cloud (hence accessing of protected variables)
"""


def _get_cloud():
    cl = getattr(sys.modules['cloud'],'__cloud')
    cl._checkOpen()
    return cl

def _getConnection(cloud):
    """Return connection object associated Cloud
    Errors if object is not an HttpConnection"""
    
    if not isinstance(cloud.adapter, SerializingAdapter):
        raise RuntimeError('Unexpected cloud adapter being used')    
    conn = cloud.adapter.connection
    if conn.connection_info()['connection_type'] in ['HTTP', 'HTTPS']:
        #cloud.mp can proxy requests to main cloud adapter
        return conn
    else:
        raise RuntimeError('Cannot use cloud.rest functions when in simulation. conn is %s' % conn.connection_info())
    

    
def register(func, label, out_encoding='json', **kwargs):
    """       
    Register *func* (a callable) so that it can be invoked via the PiCloud Rest API
    
    The REST-registered function can be managed in the future by the specified *label*
    
    *out_encoding* specifies the format that the return value should be in when retrieving the result
    via the REST API. Valid values are "json" for a JSON-encoded object and RAW, where the return value
    must be an str (but can contain any characters).
    
    The return value is the URL which can be HTTP POSTed to to invoke the function.    
    
    Certain special *kwargs* associated with cloud.call can be attached to the periodic jobs: 
        
    * _fast_serialization:
        This keyword can be used to speed up serialization, at the cost of some functionality.
        This affects the serialization of the spawned jobs' return value.
        The stored function will always be serialized by the enhanced serializer, with debugging features.
        Possible values keyword are:
                    
        0. default -- use cloud module's enhanced serialization and debugging info            
        1. no debug -- Disable all debugging features for result            
        2. use cPickle -- Use python's fast serializer, possibly causing PicklingErrors                
    
    * _profile:
            Set this to True to enable profiling of your code. Profiling information is 
            valuable for debugging, but may slow down your jobs.
    * _restartable:
            In the very rare event of hardware failure, this flag indicates that a spawned 
            job can be restarted if the failure happened in the middle of it.
            By default, this is true. This should be unset if the function has external state
            (e.g. it modifies a database entry)
    * _type:
            Select the type of compute resources to use.  PiCloud supports three types,
            specified as strings:
            
            'c1'
                1 compute unit, 300 MB ram, low I/O (default)                    
            'c2'
                2.5 compute units, 800 MB ram, medium I/O                    
            'm1'                    
                3.25 compute units, 8 GB ram, high I/O
                                               
            See http://www.picloud.com/pricing/ for pricing information

    """
    
    cloud = _get_cloud()     
    
    if not callable(func):
        raise TypeError( 'cloud.rest.register first argument (%s) is not callable'  % (str(func) ))        
    
    #ASCII label:
    try:
        label = label.decode('ascii').encode('ascii')
    except (UnicodeDecodeError, UnicodeEncodeError):
        raise TypeError('label must be an ASCII string')
    
    m = re.match(r'^[A-Z0-9a-z_+-.]+$', label)
    if not m:
        raise TypeError('Label can only consist of valid URI characters (alphanumeric or from set(_+-.$)')
        
    try:
        docstring = '' if (func.__doc__ is None) else func.__doc__
        func_desc = (docstring).decode('utf8').encode('utf8')
    except (UnicodeDecodeError, UnicodeEncodeError):
        raise TypeError('function docstring must be an UTF8 compatible unicode string')
    
    if not isinstance(out_encoding, str):
        raise TypeError('out_encoding must be an ASCII string')
    
    params = cloud._getJobParameters(func,
                                     kwargs, 
                                     ignore=['_label', '_depends_on'])
    
    conn = _getConnection(cloud)
    
    sfunc, sarg, logprefix, logcnt = cloud.adapter.cloud_serialize(func,
                                                                   params['fast_serialization'],
                                                                   [],
                                                                   logprefix='rest.')
    
    #Below is derived from HttpAdapter.job_add
    conn._update_params(params)
    
    cloud.adapter.dep_snapshot() #let adapter make any needed calls for dep tracking
    
    data = Packer()
    data.add(sfunc)
    
    params['data'] = data.finish()
    params['label'] = label
    params['description'] = func_desc
    params['out_encoding'] = out_encoding
    
    resp = conn.send_request(_register_query, params)
    
    return resp['uri'] 

def deregister(label):
    """
    Deregister (delete) the REST registered specified by *label*
    """
    cloud = _get_cloud() 

    #ASCII label:
    try:
        label = label.decode('ascii').encode('ascii')
    except (UnicodeDecodeError, UnicodeEncodeError):
        raise TypeError('label must be an ASCII string')
    
    conn = _getConnection(cloud)
    
    conn.send_request(_deregister_query, {'label': label})   
   
def list():
    """
    List labels of REST registered functions
    """ 
    #note beware: List is defined as this function
    cloud = _get_cloud()
    conn = _getConnection(cloud)
    resp = conn.send_request(_list_query, {})
    return resp['labels']

def info(label):
    """
    Retrieve information about a REST registered function specified by *label*
    """
    cloud = _get_cloud()
    conn = _getConnection(cloud)
    resp = conn.send_request(_info_query, {'label':label})
    del resp['data']
    del resp[u'version']
    return resp
