#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""python-gantt-csv manages the arguments of gantt.Task in csv format and
resolves dependencies between tasks. You will be able to edit tasks
without worrying about the order in which you define them.


Author : Shota Horie - horie.shouta at gmail.com


Licence : GPL v3 or any later version


This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""


import csv
import datetime
import re
from collections import OrderedDict
from pathlib import Path
from typing import List, NamedTuple

import gantt


__author__ = 'Shota Horie (horie.shouta at gmail.com)'
__version__ = '0.1.0'


class UniqueResource(gantt.Resource):
    _instances = []

    def __new__(cls, *args, **kwargs):
        for instance in cls._instances:
            if args[0] == instance.name:
                return instance
        new_instance = super().__new__(cls)
        cls._instances.append(new_instance)
        return new_instance


class GCBaseException(Exception):
    pass


class InvalidValueError(GCBaseException):
    def __init__(self, message):
        self.message = message


class UndefinedResourceError(GCBaseException):
    def __init__(self, message):
        self.message = message


class TaskArgs(NamedTuple):
    name: str
    start: str
    depends_of: str
    duration: int
    percent_done: int
    resources: str
    color: str
    id: str


TaskArgsList = List[TaskArgs]
Gantt_Resources = List[gantt.Resource]


def decode_start(start: str) -> datetime.date:
    if start == "today":
        return datetime.date.today()

    match_isodate = re.compile(r"[\d]+-[\d]+-[\d]+")
    m = match_isodate.search(start)
    if m:
        yeay, month, day = [int(s) for s in start.split('-')]
        return datetime.date(yeay, month, day)
    raise InvalidValueError(f'{start} is invalid')


def decode_depends_of(depends_of: str) -> List[str]:
    SEPARATOR_CHARACTER = ':'
    if depends_of == "None":
        return []
    if not depends_of.count(SEPARATOR_CHARACTER):
        return [depends_of]
    return depends_of.split(SEPARATOR_CHARACTER)


def decode_percent_done(percent_done: str) -> int:
    return int(percent_done)


def decode_duration(duration: str) -> int:
    return int(duration)


def decode_resources(resources: str) -> Gantt_Resources:
    SEPARATOR_CHARACTER = ':'
    if resources == 'None':
        return None
    if not resources.count(SEPARATOR_CHARACTER):
        return [UniqueResource(resources)]
    return [UniqueResource(key) for key in resources.split(SEPARATOR_CHARACTER)]


def decode(row: dict) -> dict:
    new_row = row.copy()
    for key in row:
        if key == 'name':
            pass
        if key == 'start':
            new_row[key] = decode_start(row[key])
        if key == 'depends_of':
            new_row[key] = decode_depends_of(row[key])
        if key == 'duration':
            new_row[key] = decode_duration(row[key])
        if key == 'percent_done':
            new_row[key] = decode_percent_done(row[key])
        if key == 'resources':
            new_row[key] = decode_resources(row[key])
        if key == 'color':
            pass
        if key == 'id':
            pass
    return new_row


def is_independent_task(task_args: TaskArgs) -> bool:
    return len(task_args["depends_of"]) == 0


def append_order(task_args: TaskArgs, order: int) -> List[dict]:
    task_args["order"] = order
    return task_args


def remove_order(task_args: dict) -> TaskArgs:
    order = task_args.pop("order")
    return task_args


def get_task_by_id(id_: str, task_args_list: List[dict]) -> TaskArgs:
    for task_args in task_args_list:
        if task_args["id"] == id_:
            return task_args


def get_dependent_tasks(task_args: dict, task_args_list: List[dict]) -> TaskArgsList:
    return [get_task_by_id(id_, task_args_list) for id_ in task_args["depends_of"]]


def get_highest_dependent_order(dependent_tasks: List[dict]) -> List[int]:
    dependent_orders = [task["order"] for task in dependent_tasks]
    if not dependent_orders:
        return 0
    return sorted(dependent_orders)[-1]


def set_task_order(task_args_list: List[dict]) -> List[dict]:
    for task_args in task_args_list:
        dependent_tasks = get_dependent_tasks(task_args, task_args_list)
        highest_order = get_highest_dependent_order(dependent_tasks)
        if highest_order != 0:
            task_args["order"] = highest_order + 1
    return task_args_list


def sort_tasks(task_args_list: TaskArgsList) -> TaskArgsList:
    new_task_args_list = [append_order(task_args._asdict(), i) for i, task_args in enumerate(task_args_list)]
    set_task_order(new_task_args_list)
    new_task_args_list.sort(key=lambda x: x["order"])
    new_task_args_list = [remove_order(task_args) for task_args in new_task_args_list]
    return [TaskArgs(**task_args) for task_args in new_task_args_list]


def create_task(task_args: TaskArgs, task_objs: List[gantt.Task]) -> gantt.Task:
    task_args = task_args._asdict()
    keys = ['name', 'start', 'depends_of', 'duration', 'percent_done', 'resources', 'color']
    kw = {key: task_args[key] for key in keys}
    if not kw["depends_of"]:
        kw["depends_of"] = None
    else:
        kw["depends_of"] = [task_objs[id_] for id_ in kw["depends_of"]]
    return gantt.Task(**kw)


def create_tasks(tasks: TaskArgsList) -> List[gantt.Task]:
    task_objs = OrderedDict()
    for task_args in tasks:
        task_objs[task_args.id] = create_task(task_args, task_objs)
    return list(task_objs.values())


def parse_csv_task(filename: Path) -> TaskArgsList:
    data = []
    with open(filename, 'r', encoding='utf-8') as f:
        reader = csv.reader(f)
        header = reader.__next__()
        for row in reader:
            row_dict = OrderedDict()
            row_dict.update(zip(header, row))
            data.append(TaskArgs(**decode(row_dict)))
    return data


def create_project_from_csv(filename: Path) -> gantt.Project:
    # Create a project
    p1 = gantt.Project(name=filename.stem)

    # Load csv
    tasks = parse_csv_task(filename)

    # Create Tasks
    Tasks = create_tasks(sort_tasks(tasks))
    for Task in Tasks:
        p1.add_task(Task)
    return p1


def get_resources():
    return UniqueResource._instances
