# static analysis: ignore
import ast
from asynq import FutureBase, AsyncTask
import collections
import os
import types

from .analysis_lib import make_module
from .checker import Checker
from .name_check_visitor import (
    _get_task_cls,
    _static_hasattr,
    ClassAttributeChecker,
    NameCheckVisitor,
)
from .implementation import assert_is_value, dump_value
from .error_code import DISABLED_IN_TESTS, ErrorCode
from .stacked_scopes import Composite
from .test_config import TestConfig
from .value import (
    AnnotatedValue,
    AnySource,
    AnyValue,
    AsyncTaskIncompleteValue,
    CallableValue,
    DictIncompleteValue,
    KVPair,
    KnownValue,
    MultiValuedValue,
    NewTypeValue,
    ReferencingValue,
    SequenceIncompleteValue,
    TypedValue,
    TypeVarValue,
    UnboundMethodValue,
    NO_RETURN_VALUE,
    UNINITIALIZED_VALUE,
    UNRESOLVED_VALUE,
    VariableNameValue,
    SubclassValue,
    GenericValue,
    TypedDictValue,
    make_weak,
)
from .tests import proxied_fn, autogenerated, l0cached_async_fn, PropertyObject
from . import test_node_visitor
from .test_node_visitor import assert_passes, assert_fails, only_before


# ===================================================
# Base classes for test_scope tests.
#
# These are also used in scope_lib files.
# ===================================================


class ConfiguredNameCheckVisitor(NameCheckVisitor):
    """NameCheckVisitor configured for testing.

    Should be called TestNameCheckVisitor but that would cause it to be confused with test classes.

    """

    config = TestConfig()


class TestNameCheckVisitorBase(test_node_visitor.BaseNodeVisitorTester):
    visitor_cls = ConfiguredNameCheckVisitor

    def _run_tree(
        self,
        code_str,
        tree,
        check_attributes=True,
        apply_changes=False,
        settings=None,
        **kwargs,
    ):
        # This can happen in Python 2.
        if isinstance(code_str, bytes):
            code_str = code_str.decode("utf-8")
        default_settings = {code: code not in DISABLED_IN_TESTS for code in ErrorCode}
        if settings is not None:
            default_settings.update(settings)
        verbosity = int(os.environ.get("ANS_TEST_SCOPE_VERBOSITY", 0))
        mod = _make_module(code_str)
        kwargs = self.visitor_cls.prepare_constructor_kwargs(kwargs)
        with ClassAttributeChecker(
            self.visitor_cls.config, enabled=check_attributes
        ) as attribute_checker:
            return self.visitor_cls(
                mod.__name__,
                code_str,
                tree,
                module=mod,
                attribute_checker=attribute_checker,
                settings=default_settings,
                verbosity=verbosity,
                **kwargs,
            ).check_for_test(apply_changes=apply_changes)


class TestAnnotatingNodeVisitor(test_node_visitor.BaseNodeVisitorTester):
    """Base class for testing AnnotatingNodeVisitor subclasses."""

    def _run_tree(self, code_str, tree, **kwargs):
        """Runs the visitor on this tree."""
        kwargs["module"] = _make_module(code_str)
        return super(TestAnnotatingNodeVisitor, self)._run_tree(
            code_str, tree, **kwargs
        )


def _make_module(code_str: str) -> types.ModuleType:
    """Creates a Python module with the given code."""
    # make helpers for value inference checking available to all tests
    extra_scope = dict(
        assert_is_value=assert_is_value,
        AsyncTaskIncompleteValue=AsyncTaskIncompleteValue,
        CallableValue=CallableValue,
        DictIncompleteValue=DictIncompleteValue,
        KVPair=KVPair,
        GenericValue=GenericValue,
        KnownValue=KnownValue,
        MultiValuedValue=MultiValuedValue,
        AnnotatedValue=AnnotatedValue,
        SequenceIncompleteValue=SequenceIncompleteValue,
        TypedValue=TypedValue,
        UnboundMethodValue=UnboundMethodValue,
        AnySource=AnySource,
        AnyValue=AnyValue,
        UNRESOLVED_VALUE=UNRESOLVED_VALUE,
        VariableNameValue=VariableNameValue,
        ReferencingValue=ReferencingValue,
        SubclassValue=SubclassValue,
        NewTypeValue=NewTypeValue,
        TypedDictValue=TypedDictValue,
        TypeVarValue=TypeVarValue,
        dump_value=dump_value,
        make_weak=make_weak,
        UNINITIALIZED_VALUE=UNINITIALIZED_VALUE,
        NO_RETURN_VALUE=NO_RETURN_VALUE,
        Composite=Composite,
    )
    return make_module(code_str, extra_scope)


# ===================================================
# Tests for specific functionality.
# ===================================================


def test_annotation():
    tree = ast.Call(ast.Name("int", ast.Load()), [], [])
    checker = Checker(ConfiguredNameCheckVisitor.config)
    ConfiguredNameCheckVisitor(
        "<test input>", "int()", tree, module=ast, annotate=True, checker=checker
    ).check()
    assert TypedValue(int) == tree.inferred_value


class TestNameCheckVisitor(TestNameCheckVisitorBase):
    @assert_passes()
    def test_list_return(self):
        from typing import List

        class A:
            pass

        def func() -> List[A]:
            return [A]  # E: incompatible_return_value

        def func() -> A:
            return A  # E: incompatible_return_value

    @assert_passes()
    def test_known_ordered(self):
        from typing_extensions import OrderedDict

        known_ordered = OrderedDict({1: 2})
        bad_ordered = OrderedDict({"a": "b"})

        def capybara(arg: OrderedDict[int, int]) -> None:
            pass

        def caller() -> None:
            capybara(known_ordered)
            capybara(bad_ordered)  # E: incompatible_argument

    @assert_fails(ErrorCode.undefined_name)
    def test_undefined_name(self):
        def run():
            print(undefined_variable)

    @assert_fails(ErrorCode.undefined_attribute)
    def test_undefined_attribute(self):
        def run():
            lst = []
            print(lst.coruro)

    def test_undefined_name_with_star_import(self):
        # can't use the decorator version because import * isn't allowed with nested functions
        self.assert_fails(
            ErrorCode.undefined_name,
            """
from qcore.asserts import *
def run():
    print(not_in_qcore.asserts)
""",
        )

    @assert_fails(ErrorCode.duplicate_enum_member)
    def test_duplicate_enum_member(self):
        import enum

        class Foo(enum.Enum):
            a = 1
            b = 1

    @assert_fails(ErrorCode.undefined_name)
    def test_undefined_name_in_return(self):
        def what_is_it():
            return tucotuco

    @assert_fails(ErrorCode.undefined_name)
    def test_undefined_name_in_class_kwarg(self):
        def capybara():
            class Capybara(metaclass=Hutia):
                pass

    @assert_passes()
    def test_no_failure_on_builtin(self):
        def run():
            print(len)

    @assert_passes()
    def test_no_failure_on_global(self):
        capybara = 3

        def run():
            print(capybara)

    @assert_passes()
    def test_no_failure_on_global_return(self):
        tucotuco = "a burrowing rodent"

        def what_is_it():
            return tucotuco

    @assert_passes()
    def test_no_failure_on_arg(self):
        def double_it(x):
            return x * 2

    @assert_passes()
    def test_class_scope(self):
        class Porcupine(object):
            def coendou(self):
                return 1

            sphiggurus = coendou

    @assert_fails(ErrorCode.undefined_name)
    def test_class_scope_fails_wrong_order(self):
        def run():
            class Porcupine(object):
                sphiggurus = coendou

                def coendou(self):
                    return 1

    @assert_fails(ErrorCode.undefined_name)
    def test_class_scope_is_not_searched(self):
        class Porcupine(object):
            sphiggurus = 1

            def coendou(self):
                return sphiggurus

    @assert_passes()
    def test_getter_decorator(self):
        class Porcupine(object):
            sphiggurus = property()

            @sphiggurus.getter
            def sphiggurus(self):
                pass

    @assert_passes()
    def test_ipython_whitelisting(self):
        def run():
            print(__IPYTHON__)

    @assert_passes()
    def test_mock_attributes(self):
        def cavy():
            pass

        def run():
            print(cavy.call_count)

    @assert_passes()
    def test_mock_attr(self):
        try:
            from unittest import mock
        except ImportError:
            import mock

        class X:
            a = mock.MagicMock()

        class Y:
            def __init__(self):
                self.x = X()

        def f():
            y = Y()
            assert_is_value(y.x.a, KnownValue(X.a))

    @assert_passes()
    def test_method_mock_attributes(self):
        class Capybara(object):
            def hutia(self):
                pass

            def kerodon(self):
                print(self.hutia.call_count)

    @assert_passes()
    def test_global_assignment(self):
        from qcore.asserts import assert_eq

        fn = assert_eq

        def run():
            assert_is_value(fn, KnownValue(assert_eq))

    @assert_fails(ErrorCode.undefined_attribute)
    def test_builtin_attribute(self):
        def run():
            print(True.hutia)

    @assert_passes()
    def test_module_reassignment(self):
        _std_set = set

        def set(key, value):
            return _std_set([key, value])

        _std_set()

    @assert_fails(ErrorCode.not_callable)
    def test_display(self):
        def run():
            print([1, 2]())

    @assert_fails(ErrorCode.unhashable_key)
    def test_list_in_set(self):
        def run():
            print({[]})

    @assert_passes()
    def test_multiple_assignment_global(self):
        if False:
            goes_in_set = []
        else:
            goes_in_set = "capybara"
        if False:
            # The assignment actually executed at runtime wins
            assert_is_value(goes_in_set, KnownValue("capybara"))
            print({goes_in_set})

    @assert_passes()
    def test_multiple_assignment_function(self):
        def fn(cond):
            if cond:
                goes_in_set = []
            else:
                goes_in_set = "capybara"
            assert_is_value(goes_in_set, KnownValue([]) | KnownValue("capybara"))
            # TODO why isn't this an error?
            print({goes_in_set})

    @assert_fails(ErrorCode.duplicate_dict_key)
    def test_duplicate_dict_key(self):
        def run():
            print({"capybara": 1, "capybara": 2})

    @assert_fails(ErrorCode.unhashable_key)
    def test_unhashable_dict_key(self):
        def run():
            print({[]: 1})

    @assert_fails(ErrorCode.duplicate_dict_key)
    def test_inferred_duplicate_dict_key(self):
        key = "capybara"

        def run():
            print({"capybara": 1, key: 1})

    @assert_fails(ErrorCode.unhashable_key)
    def test_inferred_unhashable_dict_key(self):
        key = []

        def run():
            print({key: 1})

    @assert_passes()
    def test_nested_classes(self):
        class Caviids(object):
            class Capybaras(object):
                if False:
                    print(neochoerus)  # E: undefined_name

            def method(self, cap: Capybaras):
                assert_is_value(cap, TypedValue(Caviids.Capybaras))

    @assert_fails(ErrorCode.undefined_name)
    def test_class_in_function(self):
        def get_capybaras(object):
            class Capybaras(object):
                if False:
                    print(neochoerus)

    @assert_fails(ErrorCode.unsupported_operation)
    def test_cant_del_tuple(self):
        tpl = (1, 2, 3)

        def run():
            del tpl[1]

    @assert_fails(ErrorCode.unsupported_operation)
    def test_cant_del_generator(self):
        tpl = (x for x in (1, 2, 3))

        def run():
            del tpl[1]

    @assert_fails(ErrorCode.unsupported_operation)
    def test_cant_assign_tuple(self):
        tpl = (1, 2, 3)

        def run():
            tpl[1] = 1

    @assert_passes()
    def test_binop(self):
        from typing import Union

        def tucotuco():
            assert_is_value(2 + 3, KnownValue(5))

        def capybara(x: Union[int, float], y: Union[int, float]) -> float:
            return x + y

    @assert_passes()
    def test_inplace_binop(self):
        class Capybara:
            def __add__(self, x: int) -> str:
                return ""

            def __iadd__(self, x: str) -> int:
                return 0

        def tucotuco():
            x = Capybara()
            assert_is_value(x + 1, TypedValue(str))
            x += "a"
            assert_is_value(x, TypedValue(int))

    @assert_passes()
    def test_global_sets_value(self):
        capybara = None

        def set_it():
            global capybara
            capybara = (0,)

        def use_it():
            assert_is_value(capybara, KnownValue((0,)) | KnownValue(None))

    @assert_fails(ErrorCode.unsupported_operation)
    def test_self_type_inference(self):
        class Capybara(object):
            def get(self, i):
                assert_is_value(self, TypedValue(Capybara))
                return self[i]

    @assert_passes()
    def test_self_is_subscriptable(self):
        class Capybara(object):
            def get(self, i):
                return self[i]

            def __getitem__(self, i):
                return i

    @assert_passes()
    def test_cls_type_inference(self):
        class OldStyle:
            def __init_subclass__(cls):
                assert_is_value(cls, SubclassValue(TypedValue(OldStyle)))

            def __new__(cls):
                assert_is_value(cls, SubclassValue(TypedValue(OldStyle)))

    @assert_passes()
    def test_cls_type_inference(self):
        class OldStyle:
            @classmethod
            def capybara(cls):
                pass

    @assert_passes()
    def test_display_type_inference(self):
        UNANNOTATED = AnyValue(AnySource.unannotated)

        def capybara(a, b):
            x = [a, b]
            assert_is_value(
                x, SequenceIncompleteValue(list, [UNANNOTATED, UNANNOTATED])
            )
            y = a, 2
            assert_is_value(
                y, SequenceIncompleteValue(tuple, [UNANNOTATED, KnownValue(2)])
            )

            s = {a, b}
            assert_is_value(s, SequenceIncompleteValue(set, [UNANNOTATED, UNANNOTATED]))
            z = {a: b}
            assert_is_value(
                z, DictIncompleteValue(dict, [KVPair(UNANNOTATED, UNANNOTATED)])
            )
            q = {a: 3, b: 4}
            assert_is_value(
                q,
                DictIncompleteValue(
                    dict,
                    [
                        KVPair(UNANNOTATED, KnownValue(3)),
                        KVPair(UNANNOTATED, KnownValue(4)),
                    ],
                ),
            )

    @assert_passes()
    def test_if_exp(self):
        def capybara(x):
            y = 3 if x else 4
            assert_is_value(y, MultiValuedValue([KnownValue(3), KnownValue(4)]))

    @assert_passes()
    def test_namedtuple(self):
        import collections

        typ = collections.namedtuple("typ", "foo bar")

        def fn():
            t = typ(1, 2)
            print(t.baz)  # E: undefined_attribute

    @assert_passes()
    def test_local_namedtuple(self):
        import collections

        def capybara():
            typ = collections.namedtuple("typ", "foo bar")
            # For now just test that this produces no errors; if we
            # add support for local namedtuples we can assert something
            # more precise here.
            print(typ(1, 2))

    @assert_passes()
    def test_set_after_get(self):
        def fn():
            capybara = None
            for _ in range(5):
                if capybara:
                    print(capybara[0])
                capybara = "foo"

    @assert_passes()
    def test_multiple_anys(self):
        def fn(item):
            if False:
                item = None
            assert_is_value(item, KnownValue(None) | AnyValue(AnySource.unannotated))

    @assert_fails(ErrorCode.undefined_attribute)
    def test_bad_attribute_of_global(self):
        import os

        path = os.path

        def capybara():
            print(path.joyn)

    @assert_passes()
    def test_double_assignment(self):
        from pyanalyze.tests import PropertyObject

        def capybara(aid):
            answer = PropertyObject(aid)
            print(answer)
            answer = PropertyObject(aid)
            assert_is_value(answer, TypedValue(PropertyObject))

    @assert_fails(ErrorCode.class_variable_redefinition)
    def test_duplicate_method(self):
        class Tucotuco(object):
            def __init__(self, fn):
                pass

            def __init__(self, an):
                pass

    @assert_fails(ErrorCode.class_variable_redefinition)
    def test_duplicate_attribute(self):
        import enum

        class Hutia:
            capromys = 1
            capromys = 2

    @assert_passes()
    def test_duplicate_attribute_augassign(self):
        class Capybara:
            x = 1
            x += 1

    @assert_passes()
    def test_duplicate_property_method(self):
        class Capybara(object):
            @property
            def fur(self):
                return "a lot"

            @fur.setter
            def fur(self, value):
                pass

    @assert_fails(ErrorCode.bad_global)
    def test_bad_global(self):
        global x

    @assert_fails(ErrorCode.undefined_name)
    def test_undefined_global(self):
        def fn():
            global x
            return x

    @assert_passes()
    def test_global_value(self):
        x = 3

        def capybara():
            global x
            assert_is_value(x, KnownValue(3))


class TestNoReturn(TestNameCheckVisitorBase):
    @assert_passes()
    def test_no_return(self):
        from typing_extensions import NoReturn
        from typing import Optional

        def f() -> NoReturn:
            raise Exception

        def capybara(x: Optional[int]) -> None:
            if x is None:
                f()
            assert_is_value(x, TypedValue(int))

    @assert_fails(ErrorCode.incompatible_argument)
    def test_no_return_parameter(self):
        from typing_extensions import NoReturn

        def assert_unreachable(x: NoReturn) -> None:
            pass

        def capybara():
            assert_unreachable(1)


class TestSubclassValue(TestNameCheckVisitorBase):
    # In 3.7 the behavior of Type[] changed.
    @only_before((3, 7))
    @assert_passes()
    def test_annotations_in_arguments(self):
        from typing import Type

        TI = Type[int]

        def capybara(x: TI, y: str):
            assert_is_value(x, SubclassValue(TypedValue(int)))
            assert_is_value(y, TypedValue(str))

    @assert_passes()
    def test_type_any(self):
        from typing import Any, Type

        def f(x) -> Type[Any]:
            return type(x)

        def capybara():
            f(1)
            assert_is_value(f(1), TypedValue(type))

    @assert_passes()
    def test_call_method_through_type(self):
        class A:
            def run(self):
                pass

            @classmethod
            def call_on_instance(cls, instance):
                assert_is_value(cls.run, KnownValue(A.run))
                cls.run(instance)

    @assert_passes()
    def test_metaclass_method(self):
        from typing import Type

        class EnumMeta(type):
            def __getitem__(self, x: str) -> float:
                return 42.0

        class Enum(metaclass=EnumMeta):
            pass

        def capybara(enum: Type[Enum]) -> None:
            assert_is_value(enum["x"], TypedValue(float))

    @assert_passes()
    def test_type_union(self):
        from typing import Type, Union

        def capybara(x: Type[Union[int, str]]) -> None:
            assert_is_value(
                x,
                MultiValuedValue(
                    [SubclassValue(TypedValue(int)), SubclassValue(TypedValue(str))]
                ),
            )

        def caller() -> None:
            capybara(int)
            capybara(str)


class TestBoolOp(TestNameCheckVisitorBase):
    @assert_passes()
    def test(self):
        def capybara(x):
            if x:
                cond = str(x)
                cond2 = True
            else:
                cond = None
                cond2 = None
            assert_is_value(cond, MultiValuedValue([TypedValue(str), KnownValue(None)]))
            assert_is_value(
                cond2, MultiValuedValue([KnownValue(True), KnownValue(None)])
            )
            assert_is_value(
                cond and 1,
                MultiValuedValue([TypedValue(str), KnownValue(None), KnownValue(1)]),
            )
            assert_is_value(
                cond2 and 1, MultiValuedValue([KnownValue(None), KnownValue(1)])
            )
            assert_is_value(
                cond or 1, MultiValuedValue([TypedValue(str), KnownValue(1)])
            )
            assert_is_value(
                cond2 or 1, MultiValuedValue([KnownValue(True), KnownValue(1)])
            )

        def hutia(x=None):
            assert_is_value(x, AnyValue(AnySource.unannotated) | KnownValue(None))
            assert_is_value(x or 1, AnyValue(AnySource.unannotated) | KnownValue(1))
            y = x or 1
            assert_is_value(y, AnyValue(AnySource.unannotated) | KnownValue(1))
            assert_is_value(
                (True if x else False) or None, KnownValue(True) | KnownValue(None)
            )


class TestReturn(TestNameCheckVisitorBase):
    @assert_passes()
    def test_type_inference(self):
        from asynq import asynq, async_proxy, AsyncTask, ConstFuture, FutureBase

        def returns_3():
            return 3

        @asynq(pure=True)
        def pure_async_fn():
            return 4

        @asynq()
        def async_fn():
            return 3

        class WithAProperty(object):
            @property
            def this_is_one(self):
                return str(5)

        @async_proxy(pure=True)
        def pure_async_proxy(oid):
            return ConstFuture(oid)

        @async_proxy()
        def impure_async_proxy():
            return ConstFuture(3)

        def capybara(oid):
            assert_is_value(returns_3(), KnownValue(3))
            assert_is_value(
                pure_async_fn(), AsyncTaskIncompleteValue(AsyncTask, KnownValue(4))
            )
            assert_is_value(async_fn(), KnownValue(3))
            assert_is_value(
                async_fn.asynq(), AsyncTaskIncompleteValue(AsyncTask, KnownValue(3))
            )
            assert_is_value(WithAProperty().this_is_one, TypedValue(str))
            assert_is_value(pure_async_proxy(oid), TypedValue(ConstFuture))
            assert_is_value(impure_async_proxy(), AnyValue(AnySource.unannotated))
            assert_is_value(
                impure_async_proxy.asynq(),
                AsyncTaskIncompleteValue(FutureBase, AnyValue(AnySource.unannotated)),
            )

    @assert_passes()
    def test_missing_return(self):
        from abc import abstractmethod
        from typing_extensions import NoReturn

        def foo(cond: bool) -> int:  # E: missing_return
            if cond:
                return 3

        def capybara() -> int:  # E: missing_return
            pass

        class Absy:
            @abstractmethod
            def doesnt_return(self) -> int:  # ok
                pass

        def you_can_skip_return_none() -> None:
            pass

        def no_return_but_does_it() -> NoReturn:  # E: no_return_may_return
            pass

        def return_sometimes(cond: bool) -> NoReturn:  # E: no_return_may_return
            if cond:
                raise Exception

        def no_return_returns() -> NoReturn:
            return 42  # E: no_return_may_return

    # Can't use assert_passes for those two because the location of the error
    # changes between 3.7 and 3.8. Maybe we should hack the error code to
    # always show the error for a function on the def line, not the decorator line.
    @assert_fails(ErrorCode.missing_return)
    def test_asynq_missing_return(self):
        from asynq import asynq

        @asynq()  # E: missing_return
        def f() -> int:
            yield f.asynq()

    @assert_fails(ErrorCode.missing_return)
    def test_asynq_missing_branch(self):
        from asynq import asynq

        @asynq()  # E: missing_return
        def capybara(cond: bool) -> int:
            if cond:
                return 3
            yield capybara.asynq(False)

    @assert_passes()
    def test_async_def(self):
        async def capybara() -> int:  # E: missing_return
            pass

        async def acouchy(cond: bool) -> int:  # E: missing_return
            if cond:
                return 4


class TestUnwrapYield(TestNameCheckVisitorBase):
    @assert_passes()
    def test(self):
        from asynq import asynq
        from typing import Sequence
        from typing_extensions import Literal

        @asynq()
        def async_fn(n):
            return "async_fn"

        @asynq()
        def square(n):
            return int(n * n)

        class Capybara(object):
            @asynq()
            def async_method(self):
                return "capybara"

        @asynq()
        def caller(ints: Sequence[Literal[0, 1, 2]]):
            val1 = yield async_fn.asynq(1)
            assert_is_value(val1, KnownValue("async_fn"))
            val2 = yield square.asynq(3)
            assert_is_value(val2, TypedValue(int))

            val3, val4 = yield async_fn.asynq(1), async_fn.asynq(2)
            assert_is_value(val3, KnownValue("async_fn"))
            assert_is_value(val4, KnownValue("async_fn"))

            val5 = yield Capybara().async_method.asynq()
            assert_is_value(val5, KnownValue("capybara"))

            vals1 = yield [square.asynq(1), square.asynq(2), square.asynq(3)]
            assert_is_value(
                vals1,
                SequenceIncompleteValue(
                    list, [TypedValue(int), TypedValue(int), TypedValue(int)]
                ),
            )

            vals2 = yield [square.asynq(i) for i in ints]
            for val in vals2:
                assert_is_value(val, TypedValue(int))

            vals3 = yield {1: square.asynq(1)}
            assert_is_value(
                vals3,
                DictIncompleteValue(dict, [KVPair(KnownValue(1), TypedValue(int))]),
            )

            vals4 = yield {i: square.asynq(i) for i in ints}
            assert_is_value(
                vals4,
                GenericValue(
                    dict,
                    [
                        MultiValuedValue([KnownValue(0), KnownValue(1), KnownValue(2)]),
                        TypedValue(int),
                    ],
                ),
            )


class TestYieldFrom(TestNameCheckVisitorBase):
    @assert_passes()
    def test(self):
        from typing import Iterator

        def f(x) -> Iterator[int]:
            yield from x

    def capybara(x):
        yield from [1, 2]

    @assert_fails(ErrorCode.bad_yield_from)
    def test_bad_yield_from(self):
        def capybara():
            yield from 3


class TestClassAttributeChecker(TestNameCheckVisitorBase):
    @assert_passes()
    def test_mangled_attributes(self):
        class Capybara(object):
            def __mangled(self):
                pass

            def other_method(self):
                self.__mangled()

    @assert_fails(ErrorCode.attribute_is_never_set)
    def test_never_set(self):
        class Capybara(object):
            def method(self):
                return self.doesnt_exist

    @assert_passes()
    def test_exists_on_class(self):
        class Capybara(object):
            @classmethod
            def type(cls):
                pass

            def method(self):
                return self.__class__.type()

    @assert_fails(ErrorCode.attribute_is_never_set)
    def test_in_classmethod(self):
        class Capybara(object):
            @classmethod
            def do_stuff(cls):
                return cls.stuff

    @assert_passes()
    def test_getattribute_overridden(self):
        class GetAttribute(object):
            def __getattribute__(self, attr):
                return 42

            def foo(self):
                return self.answer

    @assert_passes()
    def test_base_attribute(self):
        class Capybara(object):
            def __init__(self, obj):
                self.obj = str(obj)

        class Neochoerus(Capybara):
            def eat(self):
                assert_is_value(self.obj, TypedValue(str))

    @assert_passes()
    def test_unexamined_base(self):
        from pyanalyze.tests import PropertyObject

        # this base class was not examined, so we don't know if it has the attribute
        class Capybara(PropertyObject):
            def tree(self):
                return self.this_attribute_does_not_exist

    @assert_fails(ErrorCode.attribute_is_never_set)
    def test_cythonized_unexamined_base(self):
        import qcore

        # this base class was also not examined, but it is Cython so we can still know that the
        # attribute does not exist
        class Capybara(qcore.decorators.DecoratorBase):
            def tree(self):
                return self.this_attribute_does_not_exist

    @assert_passes()
    def test_setattr(self):
        class Capybara(object):
            def __init__(self, unannotated):
                for k, v in unannotated:
                    assert_is_value(k, AnyValue(AnySource.generic_argument))
                    setattr(self, k, v)
                assert_is_value(self.grass, AnyValue(AnySource.inference))

    @assert_passes()
    def test_setattr_on_base(self):
        class Capybara(object):
            def __init__(self, unannotated):
                for k, v in unannotated:
                    # Make sure we're not smart enough to infer the attribute
                    assert_is_value(k, AnyValue(AnySource.generic_argument))
                    setattr(self, k, v)
                assert_is_value(self.grass, AnyValue(AnySource.inference))

        class Neochoerus(Capybara):
            def eat(self):
                # this doesn't exist, but we shouldn't alert because there is a setattr() on the
                # base
                self.consume(self.grass)


class TestBadRaise(TestNameCheckVisitorBase):
    @assert_passes()
    def test_raise(self):
        def bad_value():
            raise NotImplemented  # E: bad_exception

        def bad_type():
            # make sure this isn't inferenced to KnownValue, so this tests what it's supposed to
            # test
            assert_is_value(int("3"), TypedValue(int))
            raise int("3")  # E: bad_exception

        def wrong_type():
            raise bool  # E: bad_exception

        def raise_type():
            raise NotImplementedError

        def reraise():
            try:
                pass
            except OSError:
                raise

        def raise_value():
            raise ValueError("not valuable")

    @assert_passes()
    def test_from(self):
        def none():
            raise Exception() from None

        def other_exc():
            raise Exception() from Exception()

        def not_exc():
            raise Exception() from 42  # E: bad_exception


class TestVariableNameValue(TestNameCheckVisitorBase):
    @assert_passes()
    def test(self):
        from typing import NewType, Any

        Uid = NewType("Uid", int)

        def name_ends_with_uid(uid):
            return uid

        def some_func() -> Any:
            return 42

        def test(self, uid: Uid):
            assert_is_value(uid, NewTypeValue(Uid))
            assert_is_value(name_ends_with_uid, KnownValue(name_ends_with_uid))
            uid = some_func()
            assert_is_value(uid, VariableNameValue(["uid"]))
            another_uid = "hello"
            assert_is_value(another_uid, KnownValue("hello"))

            d = {"uid": self}
            assert_is_value(d["uid"], VariableNameValue(["uid"]))
            assert_is_value(self.uid, VariableNameValue(["uid"]))


class TestImports(TestNameCheckVisitorBase):
    def test_star_import(self):
        self.assert_passes(
            """
from qcore.asserts import *

assert_eq(1, 1)
"""
        )

    @assert_passes()
    def test_local_import(self):
        import qcore as _qcore

        def capybara(foo):
            import qcore

            assert_is_value(qcore.utime, KnownValue(_qcore.utime))

    @assert_passes()
    def test_local_import_from(self):
        from qcore.asserts import assert_eq as _assert_eq

        def capybara(foo):
            from qcore.asserts import assert_eq

            assert_is_value(assert_eq, KnownValue(_assert_eq))


class TestNameOfKnownValue(TestNameCheckVisitorBase):
    @assert_passes()
    def test_varname(self):
        from pyanalyze.tests import PropertyObject

        def capybara(proper_capybara):
            assert_is_value(proper_capybara, TypedValue(PropertyObject))

        def hutia(arg):
            proper_capybara = 1
            assert_is_value(proper_capybara, KnownValue(1))

    @assert_passes()
    def test_attribute(self):
        from pyanalyze.tests import PropertyObject

        class Capybara(object):
            def __init__(self, foo, bar):
                self.proper_capybara = foo

            def eat(self):
                assert_is_value(self.proper_capybara, TypedValue(PropertyObject))

        class Hutia(object):
            def __init__(self, arg):
                self.proper_capybara = int(arg)

            def eat(self):
                assert_is_value(self.proper_capybara, TypedValue(int))


class TestComprehensions(TestNameCheckVisitorBase):
    @assert_fails(ErrorCode.undefined_name)
    def test_scoping_in_list_py3(self):
        def capybara(self):
            x = [a for a in (1, 2)]
            return a, x

    @assert_fails(ErrorCode.undefined_name)
    def test_scoping_in_set(self):
        def capybara(self):
            x = {a for a in (1, 2)}
            return a, x

    @assert_fails(ErrorCode.undefined_name)
    def test_scoping_in_generator(self):
        def capybara(self):
            x = (a for a in (1, 2))
            return a, x

    @assert_fails(ErrorCode.undefined_name)
    def test_scoping_in_dict(self):
        def capybara(self):
            x = {a: 3 for a in (1, 2)}
            return a, x

    @assert_passes()
    def test_incomplete_value(self):
        import types

        def capybara(lst):
            a = [int(x) for x in lst]
            assert_is_value(a, make_weak(GenericValue(list, [TypedValue(int)])))

            b = (0 for _ in lst)
            assert_is_value(
                b,
                GenericValue(
                    types.GeneratorType,
                    [KnownValue(0), KnownValue(None), KnownValue(None)],
                ),
            )

            c = {int(x): int(x) for x in lst}
            assert_is_value(
                c, make_weak(GenericValue(dict, [TypedValue(int), TypedValue(int)]))
            )

    @assert_passes()
    def test_sequence_iterable(self):
        # this failed previously because str has no __iter__, but is iterable
        def capybara(oid):
            tmp = str(oid)
            return [s for s in tmp]

    @assert_fails(ErrorCode.undefined_name)
    def test_comprehension_body_within_class(self):
        def capybara():
            class Capybara(object):
                incisors = [1, 2]
                canines = {incisors[0] for _ in incisors}

    @assert_passes()
    def test_comprehension_within_class(self):
        class Capybara(object):
            incisors = [1, 2]
            canines = {i + 1 for i in incisors}


class TestIterationTarget(TestNameCheckVisitorBase):
    @assert_passes()
    def test_known(self):
        def capybara():
            for char in "hello":
                assert_is_value(
                    char,
                    MultiValuedValue(
                        [
                            KnownValue("h"),
                            KnownValue("e"),
                            KnownValue("l"),
                            KnownValue("o"),
                        ]
                    ),
                )

            for num in [1, 2, 3]:
                assert_is_value(
                    num, MultiValuedValue([KnownValue(1), KnownValue(2), KnownValue(3)])
                )

            for elt in [1, None, "hello"]:
                assert_is_value(
                    elt,
                    MultiValuedValue(
                        [KnownValue(1), KnownValue(None), KnownValue("hello")]
                    ),
                )

    @assert_fails(ErrorCode.unsupported_operation)
    def test_known_not_iterable(self):
        def capybara():
            for _ in 3:
                pass

    @assert_fails(ErrorCode.unsupported_operation)
    def test_typed_not_iterable(self):
        def capybara(x):
            for _ in int(x):
                pass

    @assert_passes()
    def test_union_iterable(self):
        from typing import List, Set, Union

        def capybara(x: Union[List[int], Set[str]]) -> None:
            for obj in x:
                assert_is_value(
                    obj, MultiValuedValue([TypedValue(int), TypedValue(str)])
                )

    @assert_passes()
    def test_generic_iterable(self):
        from typing import Iterable, TypeVar, Tuple

        T = TypeVar("T")
        U = TypeVar("U")

        class ItemsView(Iterable[Tuple[T, U]]):
            pass

        def capybara(it: ItemsView[int, str]):
            for k, v in it:
                assert_is_value(k, TypedValue(int))
                assert_is_value(v, TypedValue(str))

    @assert_passes()
    def test_incomplete(self):
        def capybara(x):
            lst = [1, 2, int(x)]
            assert_is_value(
                lst,
                SequenceIncompleteValue(
                    list, [KnownValue(1), KnownValue(2), TypedValue(int)]
                ),
            )
            for elt in lst:
                assert_is_value(
                    elt,
                    MultiValuedValue([KnownValue(1), KnownValue(2), TypedValue(int)]),
                )

    @assert_passes()
    def test_list_comprehension(self):
        from typing_extensions import Literal
        from typing import Sequence

        def capybara(ints: Sequence[Literal[1, 2]]):
            lst = [x for x in ints]
            mvv = MultiValuedValue([KnownValue(1), KnownValue(2)])
            assert_is_value(lst, make_weak(GenericValue(list, [mvv])))
            for y in lst:
                assert_is_value(y, mvv)

            lst2 = [x for x in (1, 2)]
            assert_is_value(
                lst2, SequenceIncompleteValue(list, [KnownValue(1), KnownValue(2)])
            )

            lst3 = [i + j * 10 for i in range(2) for j in range(3)]
            # TODO: should be list[int] instead
            assert_is_value(lst3, TypedValue(list))

    @assert_passes()
    def test_dict_comprehension(self):
        from typing_extensions import Literal
        from typing import Sequence

        def capybara(ints: Sequence[Literal[1, 2, 3]]):
            dct = {x: x for x in ints}
            mvv = MultiValuedValue([KnownValue(1), KnownValue(2), KnownValue(3)])
            assert_is_value(dct, make_weak(GenericValue(dict, [mvv, mvv])))

            for key in dct:
                assert_is_value(key, mvv)

            dct2 = {x: x for x in (1, 2, 3)}
            assert_is_value(
                dct2,
                DictIncompleteValue(
                    dict,
                    [
                        KVPair(KnownValue(1), KnownValue(1)),
                        KVPair(KnownValue(2), KnownValue(2)),
                        KVPair(KnownValue(3), KnownValue(3)),
                    ],
                ),
            )

    @assert_passes()
    def test_maybe_empty(self):
        def capybara(cond):
            lst = []
            if cond:
                lst.append("x")
            assert_is_value(lst, KnownValue(["x"]) | KnownValue([]))
            for c in lst:
                assert_is_value(c, KnownValue("x"))

    @assert_passes()
    def test_old_style(self):
        class HasGetItem:
            def __getitem__(self, i: int) -> str:
                return str(i)

        class BadGetItem:
            def __getitem__(self, i: int, extra: bool) -> str:
                return str(i) + str(extra)

        def capybara():
            for x in HasGetItem():
                assert_is_value(x, TypedValue(str))

            for x in BadGetItem():  # E: unsupported_operation
                assert_is_value(x, AnyValue(AnySource.error))


class TestAddImports(TestNameCheckVisitorBase):
    def test_top_level(self):
        self.assert_is_changed(
            """
import sys

def capybara():
    return qcore.utime()
""",
            """
import qcore
import sys

def capybara():
    return qcore.utime()
""",
        )

    def test_from(self):
        self.assert_is_changed(
            """
from qcore.asserts import assert_eq

def capybara():
    assert_is(3, 4)
""",
            """
from qcore.asserts import assert_eq, assert_is

def capybara():
    assert_is(3, 4)
""",
        )

        self.assert_is_changed(
            """
import sys

def capybara():
    assert_is(3, 4)
""",
            """
from qcore.asserts import assert_is
import sys

def capybara():
    assert_is(3, 4)
""",
        )


class TestNestedFunction(TestNameCheckVisitorBase):
    @assert_passes()
    def test_inference(self):
        def capybara():
            def nested():
                pass

            class NestedClass(object):
                pass

            assert_is_value(nested, KnownValue(nested))
            # Should ideally be something more specific
            assert_is_value(NestedClass, AnyValue(AnySource.inference))

    @assert_passes()
    def test_usage_in_nested_scope():
        def capybara(cond, x):
            if cond:

                def nested(y):
                    pass

                ys = [nested(y) for y in x]

                class Nested(object):
                    xs = ys

    @assert_fails(ErrorCode.incompatible_call)
    def test_argument_mismatch(self):
        def capybara():
            def nested():
                pass

            nested(None)

    @assert_passes()
    def test_async(self):
        from asynq import asynq, result

        @asynq()
        def capybara():
            @asynq()
            def nested():
                return 3

            assert_is_value(nested, KnownValue(nested))
            result((yield nested.asynq()))

    @assert_passes()
    def test_bad_decorator(self):
        import types

        def decorator(fn):
            return fn

        def capybara():
            @decorator
            def nested():
                pass

            assert_is_value(nested, TypedValue(types.FunctionType))

    @assert_passes()
    def test_attribute_set(self):
        def capybara():
            def inner():
                pass

            inner.punare = 3
            print(inner.punare)


class TestYieldInComprehension(TestNameCheckVisitorBase):
    # this became a syntax error in 3.8
    @only_before((3, 8))
    def test_list(self):
        self.assert_fails(
            ErrorCode.yield_in_comprehension,
            """
def capybara():
    [(yield x) for x in []]
""",
        )

    @only_before((3, 8))
    def test_set(self):
        self.assert_fails(
            ErrorCode.yield_in_comprehension,
            """
def capybara():
    {(yield x) for x in []}
""",
        )

    @only_before((3, 8))
    def test_dict(self):
        self.assert_fails(
            ErrorCode.yield_in_comprehension,
            """
def capybara():
    {(yield x): (yield y) for x in []}
""",
        )

    @only_before((3, 8))
    def test_yield_in_condition(self):
        self.assert_fails(
            ErrorCode.yield_in_comprehension,
            """
def capybara():
    [x for x in [] if (yield x)]
""",
        )

    @assert_passes()
    def test_body_only(self):
        def capybara(y):
            [x for x in (yield y)]


class TestUnboundMethodValue(TestNameCheckVisitorBase):
    @assert_passes()
    def test_inference(self):
        from pyanalyze.tests import PropertyObject, ClassWithAsync

        def capybara(oid):
            assert_is_value(
                PropertyObject(oid).non_async_method,
                UnboundMethodValue(
                    "non_async_method", Composite(TypedValue(PropertyObject))
                ),
            )
            assert_is_value(
                PropertyObject(oid).async_method,
                UnboundMethodValue(
                    "async_method", Composite(TypedValue(PropertyObject))
                ),
            )
            assert_is_value(
                ClassWithAsync().get_async,
                UnboundMethodValue("get_async", Composite(TypedValue(ClassWithAsync))),
            )
            assert_is_value(
                ClassWithAsync().get,
                UnboundMethodValue("get", Composite(TypedValue(ClassWithAsync))),
            )
            assert_is_value(
                [oid].append,
                UnboundMethodValue(
                    "append",
                    Composite(
                        SequenceIncompleteValue(list, [AnyValue(AnySource.unannotated)])
                    ),
                ),
            )

    @assert_passes()
    def test_metaclass_super(self):
        class Metaclass(type):
            def __init__(self, name, bases, attrs):
                super(Metaclass, self).__init__(name, bases, attrs)
                # TODO(jelle): the value is inferred correctly but this test fails because identical super
                # objects don't compare equal
                # assert_is_value(super(Metaclass, self).__init__,
                #                 UnboundMethodValue('__init__', super(Metaclass, Metaclass)))
                assert_is_value(
                    self.__init__,
                    UnboundMethodValue(
                        "__init__", Composite(TypedValue(Metaclass), "self")
                    ),
                )


class TestSubscripting(TestNameCheckVisitorBase):
    @assert_passes()
    def test_list_success(self):
        def capybara():
            return [1, 2][0]

    @assert_passes()
    def test_tuple_success(self):
        def capybara():
            return (1, 2)[0]

    @assert_passes()
    def test_str_success(self):
        def capybara():
            return "12"[0]

    @assert_passes()
    def test_custom_index(self):
        class CustomIndex(object):
            def __index__(self):
                return 0

        def capybara():
            return [1, 2][CustomIndex()]

    @assert_passes()
    def test_permissive_subclass(self):
        # Inspired by pyspark.sql.types.Row
        class LetItAllThrough(tuple):
            def __getitem__(self, idx: object) -> object:
                if isinstance(idx, (int, slice)):
                    return super().__getitem__(idx)
                else:
                    return "whatever"

        def capybara(liat: LetItAllThrough) -> None:
            assert_is_value(liat["x"], TypedValue(object))
            assert_is_value(liat[0], TypedValue(object))

    @assert_passes()
    def test_slice(self):
        def capybara():
            return [1, 2][1:]

    @assert_passes()
    def test_failure(self):
        def capybara():
            return [1, 2][3.0]  # E: unsupported_operation

    @assert_passes()
    def test_union(self):
        from typing import Dict, Any, Union

        def capybara(seq: Union[Dict[int, str], Any]) -> None:
            assert_is_value(seq[0], TypedValue(str) | AnyValue(AnySource.from_another))

    @assert_passes()
    def test_weak():
        from typing import Any, Dict, List

        def get_min_max_pk_value(
            min_pks: List[Dict[str, Any]], max_pks: List[Dict[str, Any]]
        ):
            return [r["pk"] for r in [*min_pks, *max_pks]]


class TestPython3Compatibility(TestNameCheckVisitorBase):
    @assert_passes()
    def test_bytes_and_text(self):
        def capybara():
            return b"foo" + "bar"  # E: unsupported_operation

    @assert_passes()
    def test_text_and_bytes(self):
        def capybara():
            return "foo" + b"bar"  # E: unsupported_operation


class TestOperators(TestNameCheckVisitorBase):
    @assert_passes(settings={ErrorCode.value_always_true: False})
    def test_not(self):
        def capybara(x):
            assert_is_value(not x, TypedValue(bool))
            assert_is_value(not True, KnownValue(False))

    @assert_passes()
    def test_unary_op(self):
        def capybara(x):
            assert_is_value(~x, AnyValue(AnySource.from_another))
            assert_is_value(~3, KnownValue(-4))

    @assert_passes()
    def test_binop_type_inference(self):
        def capybara(x):
            assert_is_value(1 + int(x), TypedValue(int))
            assert_is_value(3 * int(x), TypedValue(int))
            assert_is_value("foo" + str(x), TypedValue(str))
            assert_is_value(1 + float(x), TypedValue(float))
            assert_is_value(1.0 + int(x), TypedValue(float))
            assert_is_value(3 * 3.0 + 1, KnownValue(10.0))

    @assert_passes()
    def test_union(self):
        from typing import Union

        def capybara(x: Union[int, str]) -> None:
            assert_is_value(x * 3, MultiValuedValue([TypedValue(int), TypedValue(str)]))

    @assert_passes()
    def test_rop(self):
        class HasAdd:
            def __add__(self, other: int) -> "HasAdd":
                raise NotImplementedError

        class HasRadd:
            def __radd__(self, other: int) -> "HasRadd":
                raise NotImplementedError

        class HasBoth:
            def __add__(self, other: "HasBoth") -> "HasBoth":
                raise NotImplementedError

            def __radd__(self, other: "HasBoth") -> int:
                raise NotImplementedError

        def capybara(x):
            ha = HasAdd()
            hr = HasRadd()
            assert_is_value(1 + hr, TypedValue(HasRadd))
            assert_is_value(x + hr, AnyValue(AnySource.from_another))
            assert_is_value(ha + 1, TypedValue(HasAdd))
            assert_is_value(ha + x, AnyValue(AnySource.from_another))
            assert_is_value(HasBoth() + HasBoth(), TypedValue(HasBoth))

    @assert_passes()
    def test_unsupported_unary_op(self):
        def capybara():
            ~"capybara"  # E: unsupported_operation

    @assert_passes()
    def test_int_float_product(self):
        def capybara(f: float, i: int):
            assert_is_value(i * f, TypedValue(float))


class TestTaskNeedsYield(TestNameCheckVisitorBase):
    @assert_fails(ErrorCode.task_needs_yield)
    def test_constfuture(self):
        from asynq import asynq, ConstFuture

        @asynq()
        def bad_async_fn():
            return ConstFuture(3)

    @assert_fails(ErrorCode.task_needs_yield)
    def test_async(self):
        from asynq import asynq, ConstFuture

        @asynq()
        def async_fn():
            pass

        @asynq()
        def bad_async_fn():
            return async_fn.asynq()

    @assert_fails(ErrorCode.task_needs_yield)
    def test_not_yielded(self):
        from asynq import asynq
        from pyanalyze.tests import async_fn

        @asynq()
        def capybara(oid):
            return async_fn.asynq(oid)

    def test_not_yielded_replacement(self):
        self.assert_is_changed(
            """
from asynq import asynq
from pyanalyze.tests import async_fn

@asynq()
def capybara(oid):
    async_fn.asynq(oid)
""",
            """
from asynq import asynq
from pyanalyze.tests import async_fn

@asynq()
def capybara(oid):
    yield async_fn.asynq(oid)
""",
        )


class TestAsyncAwait(TestNameCheckVisitorBase):
    @assert_passes()
    def test_type_inference(self):
        from collections.abc import Awaitable

        async def capybara(x):
            assert_is_value(x, AnyValue(AnySource.unannotated))
            return "hydrochoerus"

        async def kerodon(x):
            task = capybara(x)
            assert_is_value(task, GenericValue(Awaitable, [KnownValue("hydrochoerus")]))
            val = await task
            assert_is_value(val, KnownValue("hydrochoerus"))

    @assert_fails(ErrorCode.unsupported_operation)
    def test_type_error(self):
        async def capybara():
            await None

    @assert_passes()
    def test_exotic_awaitable(self):
        from typing import TypeVar, Awaitable, Iterable

        T = TypeVar("T")
        U = TypeVar("U")

        class Aww(Iterable[T], Awaitable[U]):
            pass

        async def capybara(aw: Aww[int, str]) -> None:
            assert_is_value(await aw, TypedValue(str))

    @assert_passes()
    def test_async_comprehension(self):
        class ANext:
            async def __anext__(self) -> int:
                return 42

        class AIter:
            def __aiter__(self) -> ANext:
                return ANext()

        async def f():
            x = [y async for y in AIter()]
            assert_is_value(x, make_weak(GenericValue(list, [TypedValue(int)])))

    @assert_passes()
    def test_async_generator(self):
        import collections.abc
        from typing import AsyncIterator

        async def f() -> AsyncIterator[int]:
            yield 1
            yield 2

        async def capybara():
            x = f()
            assert_is_value(
                x, GenericValue(collections.abc.AsyncIterator, [TypedValue(int)])
            )
            ints = [i async for i in x]
            # TODO should be list[int] but we lose the type argument somewhere
            assert_is_value(ints, TypedValue(list))

    @assert_passes()
    def test_bad_async_comprehension(self):
        async def f():
            return [x async for x in []]  # E: unsupported_operation


class TestKeywordOnlyArguments(TestNameCheckVisitorBase):
    @assert_passes()
    def test_success(self):
        def capybara(a, *, b, c=3):
            assert_is_value(a, AnyValue(AnySource.unannotated))
            assert_is_value(b, AnyValue(AnySource.unannotated))
            assert_is_value(c, AnyValue(AnySource.unannotated) | KnownValue(3))
            capybara(1, b=2)

            fn = lambda a, *, b: assert_is_value(b, AnyValue(AnySource.unannotated))
            fn(a, b=b)

    @assert_fails(ErrorCode.incompatible_call)
    def test_failure(self):
        def capybara(a, *, b):
            capybara(1, 2)


class TestMissingAwait(TestNameCheckVisitorBase):
    @assert_fails(ErrorCode.missing_await)
    def test_asyncio_coroutine_internal(self):
        import asyncio

        @asyncio.coroutine
        def f():
            yield from asyncio.sleep(3)

        @asyncio.coroutine
        def g():
            f()

    @assert_passes()
    def test_yield_from(self):
        import asyncio

        @asyncio.coroutine
        def f():
            yield from asyncio.sleep(3)

        @asyncio.coroutine
        def g():
            yield from f()

    @assert_fails(ErrorCode.missing_await)
    def test_asyncio_coroutine_external(self):
        import asyncio

        @asyncio.coroutine
        def f():
            asyncio.sleep(3)

    def test_add_yield_from(self):
        self.assert_is_changed(
            """
import asyncio

@asyncio.coroutine
def f():
    asyncio.sleep(3)
""",
            """
import asyncio

@asyncio.coroutine
def f():
    yield from asyncio.sleep(3)
""",
        )

    @assert_passes()
    def test_has_yield_from_external(self):
        import asyncio

        @asyncio.coroutine
        def f():
            yield from asyncio.sleep(3)

    @assert_fails(ErrorCode.missing_await)
    def test_async_def_internal(self):
        async def f():
            return 42

        async def g():
            f()

    @assert_passes()
    def test_async_def_internal_has_await(self):
        async def f():
            return 42

        async def g():
            await f()

    @assert_fails(ErrorCode.missing_await)
    def test_async_def_external(self):
        import asyncio

        async def f():
            asyncio.sleep(1)

    def test_async_def_external_add_await(self):
        self.assert_is_changed(
            """
import asyncio

async def f():
    asyncio.sleep(1)
""",
            """
import asyncio

async def f():
    await asyncio.sleep(1)
""",
        )

    @assert_passes()
    def test_async_def_external_has_await(self):
        import asyncio

        async def f():
            await asyncio.sleep(1)


class TestNonlocal(TestNameCheckVisitorBase):
    @assert_passes()
    def test_nonlocal(self):
        from qcore.testing import Anything

        def capybara():
            x = 3

            def inner_capybara():
                nonlocal x
                assert_is_value(
                    x, MultiValuedValue([KnownValue(4), KnownValue(3), KnownValue(5)])
                )
                x = 4
                assert_is_value(x, KnownValue(4))

            def second_inner():
                nonlocal x
                # this should not throw unused_variable
                x = 5

            return x

    @assert_passes()
    def test_no_unused_var(self):
        def loop():
            running = True

            def handler():
                nonlocal running
                running = False

            return running


class TestingCallSiteCollector(object):
    """Records as string instead of actual reference so can be tested.

    Replaces name_check_visitor.py:CallSiteCollector, since that class records
    real references, which are hard to test against.

    """

    def __init__(self):
        self.map = collections.defaultdict(list)

    def record_call(self, caller, callee):
        try:
            self.map[callee.__qualname__].append(caller.__qualname__)
        except TypeError:
            # Copied for consistency; see comment in name_check_visitor.py:CallSiteCollector
            pass


class TestCallSiteCollection(TestNameCheckVisitorBase):
    """Base class for testing call site collection."""

    def run_and_get_call_map(self, code_str, **kwargs):
        collector = TestingCallSiteCollector()
        self._run_str(code_str, collector=collector, **kwargs)
        return collector.map

    def test_member_function_call(self):
        call_map = self.run_and_get_call_map(
            """
class TestClass(object):
    def __init__(self):
        self.first_function(5)

    def first_function(self, x):
        print(x)
        self.second_function(x, 4)

    def second_function(self, y, z):
        print(y + z)
"""
        )

        assert "TestClass.first_function" in call_map["TestClass.second_function"]
        assert "TestClass.__init__" in call_map["TestClass.first_function"]
        assert "TestClass.second_function" in call_map["print"]

    def test_nested_function_call(self):
        call_map = self.run_and_get_call_map(
            """
class TestClass(object):
    def __init__(self):
        def second_function(y):
            print(y)
        second_function(3)
"""
        )

        assert "TestClass.__init__" in call_map["second_function"]
        assert "second_function" in call_map["print"]


def test_get_task_cls():
    assert FutureBase is _get_task_cls(proxied_fn)
    assert AsyncTask is _get_task_cls(autogenerated)
    assert AsyncTask is _get_task_cls(l0cached_async_fn)
    assert AsyncTask is _get_task_cls(PropertyObject(1).l0cached_async_method)


class TestBadAsyncYield(TestNameCheckVisitorBase):
    @assert_passes()
    def test_const_future(self):
        from asynq import asynq, ConstFuture, FutureBase

        @asynq()
        def capybara(condition):
            yield FutureBase()
            val = yield ConstFuture(3)
            assert_is_value(val, KnownValue(3))
            val2 = yield None
            assert_is_value(val2, KnownValue(None))

            if condition:
                task = ConstFuture(4)
            else:
                task = capybara.asynq(True)
            val3 = yield task
            assert_is_value(val3, KnownValue(4) | AnyValue(AnySource.inference))


class TestAugAssign(TestNameCheckVisitorBase):
    @assert_passes()
    def test_aug_assign(self):
        def capybara(condition):
            x = 1
            x += 2
            assert_is_value(x, KnownValue(3))


class TestUnpacking(TestNameCheckVisitorBase):
    @assert_passes()
    def test_dict_unpacking(self):
        from typing import Dict, Optional
        from typing_extensions import TypedDict, NotRequired

        class FullTD(TypedDict):
            a: int
            b: str

        class PartialTD(TypedDict):
            a: int
            b: NotRequired[str]

        def capybara(
            d: Dict[str, int],
            ftd: FullTD,
            ptd: PartialTD,
            maybe_ftd: Optional[FullTD] = None,
        ):
            d1 = {1: 2}
            d2 = {3: 4, **d1}
            assert_is_value(
                d2,
                DictIncompleteValue(
                    dict,
                    [
                        KVPair(KnownValue(3), KnownValue(4)),
                        KVPair(KnownValue(1), KnownValue(2)),
                    ],
                ),
            )
            assert_is_value(
                {1: 2, **d},
                DictIncompleteValue(
                    dict,
                    [
                        KVPair(KnownValue(1), KnownValue(2)),
                        KVPair(TypedValue(str), TypedValue(int), is_many=True),
                    ],
                ),
            )
            assert_is_value(
                {1: 2, **ftd},
                DictIncompleteValue(
                    dict,
                    [
                        KVPair(KnownValue(1), KnownValue(2)),
                        KVPair(KnownValue("a"), TypedValue(int)),
                        KVPair(KnownValue("b"), TypedValue(str)),
                    ],
                ),
            )
            assert_is_value(
                {1: 2, **ptd},
                DictIncompleteValue(
                    dict,
                    [
                        KVPair(KnownValue(1), KnownValue(2)),
                        KVPair(KnownValue("a"), TypedValue(int)),
                        KVPair(KnownValue("b"), TypedValue(str), is_required=False),
                    ],
                ),
            )
            assert_is_value(
                {**(maybe_ftd or {})},
                DictIncompleteValue(
                    dict,
                    [
                        KVPair(KnownValue("a"), TypedValue(int), is_required=False),
                        KVPair(KnownValue("b"), TypedValue(str), is_required=False),
                    ],
                ),
            )

    @assert_passes()
    def test_iterable_unpacking(self):
        def capybara(x):
            degu = (1, *x)
            assert_is_value(
                degu,
                make_weak(
                    GenericValue(
                        tuple, [KnownValue(1) | AnyValue(AnySource.generic_argument)]
                    )
                ),
            )

            z = [1, *(2, 3)]
            assert_is_value(
                z,
                SequenceIncompleteValue(
                    list, [KnownValue(1), KnownValue(2), KnownValue(3)]
                ),
            )

    @assert_passes()
    def test_not_iterable(self):
        def capybara(x: int):
            (*x,)  # E: unsupported_operation

    @assert_passes()
    def test_bad_unpack(self):
        def too_many_values():
            a, b = 1, 2, 3  # E: bad_unpack
            return a, b

        def too_few_values():
            a, b, c = 1, 2  # E: bad_unpack
            return a, b, c

        def too_few_values_list():
            [a, b, c] = 1, 2  # E: bad_unpack
            return a, b, c

        def too_short_generalized():
            a, b, *c = (1,)  # E: bad_unpack
            return a, b, c

    @assert_passes()
    def test_correct_unpack(self):
        from typing import List, Union, Any, Tuple

        def run(lst: List[int], union: Union[Any, List[int], Tuple[str, float]]):
            a, b = 1, 2
            assert_is_value(a, KnownValue(1))
            assert_is_value(b, KnownValue(2))

            c, d = lst
            assert_is_value(c, TypedValue(int))
            assert_is_value(d, TypedValue(int))

            e, f = (lst, 42)
            assert_is_value(e, GenericValue(list, [TypedValue(int)]))
            assert_is_value(f, KnownValue(42))

            g, h = union
            assert_is_value(
                g,
                AnyValue(AnySource.generic_argument)
                | TypedValue(int)
                | TypedValue(str),
            )
            assert_is_value(
                h,
                AnyValue(AnySource.generic_argument)
                | TypedValue(int)
                | TypedValue(float),
            )

            long_tuple = (1, 2, 3, 4, 5, 6)
            *i, j, k = long_tuple
            assert_is_value(
                i,
                SequenceIncompleteValue(
                    list, [KnownValue(1), KnownValue(2), KnownValue(3), KnownValue(4)]
                ),
            )
            assert_is_value(j, KnownValue(5))
            assert_is_value(k, KnownValue(6))
            l, m, *n, o, p = long_tuple
            assert_is_value(l, KnownValue(1))
            assert_is_value(m, KnownValue(2))
            assert_is_value(
                n, SequenceIncompleteValue(list, [KnownValue(3), KnownValue(4)])
            )
            assert_is_value(o, KnownValue(5))
            assert_is_value(p, KnownValue(6))

            q, r, *s = (1, 2)
            assert_is_value(q, KnownValue(1))
            assert_is_value(r, KnownValue(2))
            assert_is_value(s, SequenceIncompleteValue(list, []))

            for sprime in []:
                assert_is_value(sprime, AnyValue(AnySource.unreachable))

            for t, u in []:
                assert_is_value(t, AnyValue(AnySource.unreachable))
                assert_is_value(u, AnyValue(AnySource.unreachable))

            known_list = [1, 2]
            v, w = known_list
            assert_is_value(v, KnownValue(1))
            assert_is_value(w, KnownValue(2))

            if lst:
                known_list.append(3)

            # We allow this unsafe code to avoid false positives
            x, y = known_list
            assert_is_value(
                x, MultiValuedValue([KnownValue(1), KnownValue(2), KnownValue(3)])
            )

    @assert_passes()
    def test_unpack_int(self):
        def run():
            a, b = 1, 2
            a()  # E: not_callable
            return a, b


class TestUnusedIgnore(TestNameCheckVisitorBase):
    @assert_fails(ErrorCode.unused_ignore)
    def test_unused(self):
        def capybara(condition):
            x = 1
            print(x)  # static analysis: ignore

    @assert_passes()
    def test_used(self):
        def capybara(condition):
            print(x)  # static analysis: ignore[undefined_name]


class TestMissingF(TestNameCheckVisitorBase):
    @assert_passes()
    def test_no_error(self):
        def capybara(func):
            "x"
            "not valid syntax {}"
            b"no byte {f} {string}"
            "{undefined_name} in this string"
            x = 3
            print("a{x}".format(x=x))
            func("translate {x}", x=x)

    @assert_fails(ErrorCode.missing_f)
    def test_missing_f(self):
        def capybara():
            x = 3
            return "x = {x}"

    def test_autofix(self):
        self.assert_is_changed(
            """
def capybara():
    x = 3
    return "x = {x}"
""",
            """
def capybara():
    x = 3
    return f'x = {x}'
""",
        )


class TestFStrings(TestNameCheckVisitorBase):
    @assert_passes()
    def test_fstr(self):
        def capybara(x):
            y = f"{x} stuff"
            assert_is_value(y, TypedValue(str))

    @assert_fails(ErrorCode.undefined_name)
    def test_undefined_name(self):
        def capybara():
            return f"{x}"


class TestTypedDict(TestNameCheckVisitorBase):
    @assert_passes()
    def test_basic(self):
        from mypy_extensions import TypedDict as METypedDict
        from typing_extensions import TypedDict as TETypedDict

        T = METypedDict("T", {"a": int, "b": str})
        T2 = TETypedDict("T2", {"a": int, "b": str})

        def capybara(x: T, y: T2):
            assert_is_value(
                x,
                TypedDictValue(
                    {"a": (True, TypedValue(int)), "b": (True, TypedValue(str))}
                ),
            )
            assert_is_value(x["a"], TypedValue(int))
            assert_is_value(
                y,
                TypedDictValue(
                    {"a": (True, TypedValue(int)), "b": (True, TypedValue(str))}
                ),
            )
            assert_is_value(y["a"], TypedValue(int))

    @assert_passes()
    def test_unknown_key_unresolved(self):
        from mypy_extensions import TypedDict

        T = TypedDict("T", {"a": int, "b": str})

        def capybara(x: T):
            assert_is_value(x["not a key"], AnyValue(AnySource.inference))

    @assert_passes()
    def test_invalid_key(self):
        from mypy_extensions import TypedDict

        T = TypedDict("T", {"a": int, "b": str})

        def capybara(x: T):
            x[0]  # E: invalid_typeddict_key

    @assert_passes()
    def test_total(self):
        from typing_extensions import TypedDict

        class TD(TypedDict, total=False):
            a: int
            b: str

        class TD2(TD):
            c: float

        def f(td: TD) -> None:
            pass

        def g(td2: TD2) -> None:
            pass

        def caller() -> None:
            f({})
            f({"a": 1})
            f({"a": 1, "b": "c"})
            f({"a": "a"})  # E: incompatible_argument
            g({"c": 1.0})
            g({})  # E: incompatible_argument


_AnnotSettings = {
    ErrorCode.missing_parameter_annotation: True,
    ErrorCode.missing_return_annotation: True,
}


class TestRequireAnnotations(TestNameCheckVisitorBase):
    @assert_passes(settings=_AnnotSettings)
    def test_missing_annotation(self):
        def no_param(x) -> None:  # E: missing_parameter_annotation
            pass

        # E: missing_return_annotation
        def no_return(x: object):
            pass

        class Capybara:
            def f(self, x) -> None:  # E: missing_parameter_annotation
                pass

    @assert_passes(settings=_AnnotSettings)
    def test_dont_annotate_self():
        def f() -> None:
            class X:
                def method(self) -> None:
                    pass

        class X:
            def f(self) -> None:
                pass


class TestAnnAssign(TestNameCheckVisitorBase):
    @assert_passes()
    def test(self):
        from typing_extensions import Final

        def capybara() -> None:
            x: Final = 3
            assert_is_value(x, KnownValue(3))
            y: int = 3
            assert_is_value(y, TypedValue(int))
            z: float
            print(z)  # E: undefined_name


class TestWhile(TestNameCheckVisitorBase):
    @assert_passes()
    def test_while_true_reachability(self):
        def capybara() -> int:  # E: missing_return
            while True:
                break

        def pacarana(i: int) -> int:
            while True:
                if i == 1:
                    return 1


class HasGetattr(object):
    def __getattr__(self, attr):
        return 42

    def real_method(self):
        pass


class HasGetattribute(object):
    def __getattribute__(self, attr):
        return 43

    def real_method(self):
        pass


def test_static_hasattr():
    hga = HasGetattr()
    assert _static_hasattr(hga, "real_method")
    assert _static_hasattr(hga, "__getattr__")
    assert not _static_hasattr(hga, "random_attribute")

    hgat = HasGetattribute()
    assert _static_hasattr(hgat, "real_method")
    assert _static_hasattr(hgat, "__getattribute__")
    assert not _static_hasattr(hgat, "random_attribute")
