"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextProvider = void 0;
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const annotations_1 = require("./annotations");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * (experimental) Base class for the model side of context providers.
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     * @experimental
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    /**
     * @experimental
     */
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = constructs_1.Node.of(scope).tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                annotations_1.Annotations.of(scope).addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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