# Copyright 2025 Advanced Micro Devices, Inc.
#
# Licensed under the Apache License v2.0 with LLVM Exceptions.
# See https://llvm.org/LICENSE.txt for license information.
# SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

from sharktank.kernels.base import *
from sharktank.kernels.mlir_kernel import *
import torch


M = DynDim.M
N = DynDim.N
K = DynDim.K
HALF_K = DynDim.HALF_K
K_OVER_THIRTYTWO = DynDim.K_OVER_THIRTYTWO

U8 = Dtype.U8(torch.uint8)
F16 = Dtype.F16(torch.float16)
F32 = Dtype.F32(torch.float32)


"""
A4W4 asm gemm kernel
D = A*B*alpha + beta*C

A: [M, K/2] f4x2
B: [N, K/2] f4x2
A_scale: [M, K/32] e8m0 padded
B_scale: [N, K/32] e8m0 padded
bias: [M, N] f32
Out: [M, N] bf16
alpha = 1.0, beta = 0.0 by default
"""


# TODO: Embedding kernels as hex
@mlir_kernel(
    inputs=(
        MLIRTensor[M, HALF_K, U8],
        MLIRTensor[N, HALF_K, U8],
        MLIRTensor[M, K_OVER_THIRTYTWO, U8],
        MLIRTensor[N, K_OVER_THIRTYTWO, U8],
        MLIRTensor[M, N, F32],
    ),
    results=(MLIRTensor[M, N, F16],),
)
def asm_fp4_gemm(x, w, x_scale, w_scale, bias, result=None):
    mlir = f"""
#rocm_target = #hal.executable.target<"rocm", "rocm-hsaco-fb", {{target_arch = "gfx950", ukernels = "none"}}>
module {{
{{% raw %}}
    util.func private @asm_mxfp4_gemm(%arg0: tensor<?x?xi8>, %arg1: tensor<?x?xi8>, %arg2: tensor<?x?xi8>, %arg3: tensor<?x?xi8>, %arg4: tensor<?x?xf32>) -> (tensor<?x?xf16>) {{
        %c0 = arith.constant 0 : index
        %c1 = arith.constant 1 : index
        %c2 = arith.constant 2 : index
        %c255 = arith.constant 255 : index
        %c256 = arith.constant 256 : index
        %m = tensor.dim %arg0, %c0 : tensor<?x?xi8>
        %n = tensor.dim %arg1, %c0 : tensor<?x?xi8>
        %k_f4x2 = tensor.dim %arg0, %c1 : tensor<?x?xi8>
        %k_e8m0 = tensor.dim %arg2, %c1 : tensor<?x?xi8>
        %k = arith.muli %k_f4x2, %c2 : index
        // m_256 = (m + 255) // 256 * 256
        %m_256 = affine.apply affine_map<()[s0] -> (s0 ceildiv 256 * 256)>()[%m]
        %hi_pad = arith.subi %m_256, %m : index
        %c0_i8 = arith.constant 1 : i8
        %x_scales_padded = tensor.pad %arg2 low[%c0, %c0] high[%hi_pad, %c0] {{
        ^bb0(%i0: index, %i1: index):
            tensor.yield %c0_i8 : i8
        }} : tensor<?x?xi8> to tensor<?x?xi8>
        %c0_f32 = arith.constant 1.0 : f32
        %bias_padded = tensor.pad %arg4 low[%c0, %c0] high[%hi_pad, %c0] {{
        ^bb0(%i0: index, %i1: index):
            tensor.yield %c0_f32 : f32
        }} : tensor<?x?xf32> to tensor<?x?xf32>
        %alpha = arith.constant 1.0 : f32
        %beta = arith.constant 0.0 : f32
        %alpha_i32 = arith.bitcast %alpha : f32 to i32
        %beta_i32  = arith.bitcast %beta  : f32 to i32
        %m_i32 = arith.index_cast %m : index to i32
        %n_i32 = arith.index_cast %n : index to i32
        %k_i32 = arith.index_cast %k : index to i32
        %k_e8m0_i32 = arith.index_cast %k_e8m0 : index to i32
        %gemm = hal.dispatch.extern "f4gemm_kernel_func"[%m, %n](%alpha_i32, %beta_i32, %k_i32, %k_i32, %n_i32, %m_i32, %n_i32, %k_i32, %k_e8m0_i32, %k_e8m0_i32, %arg0, %arg1, %x_scales_padded, %arg3, %bias_padded) : (i32, i32, i32, i32, i32, i32, i32, i32, i32, i32, tensor<?x?xi8>{{%m, %k_f4x2}}, tensor<?x?xi8>{{%n, %k_f4x2}}, tensor<?x?xi8>{{%m_256, %k_e8m0}}, tensor<?x?xi8>{{%n, %k_e8m0}}, tensor<?x?xf32>{{%m_256, %n}}) -> tensor<?x?xbf16>{{%m_256, %n}}
            count(%device: !hal.device, %m_workload: index, %n_workload: index) -> (index, index, index) {{
                %c1_0 = arith.constant 1 : index
                %subm = arith.constant 256 : index
                %subn = arith.constant 256 : index
                // gdx = (Ndim + SUBN - 1) / SUBN
                // gdy = (Mdim + SUBM - 1) / SUBM
                %subn_sub1 = arith.subi %subn, %c1_0 : index
                %n_add = arith.addi %n_workload, %subn_sub1 : index
                %gdx = arith.divui %n_add, %subn : index
                %subm_sub1 = arith.subi %subm, %c1_0 : index
                %m_add = arith.addi %m_workload, %subm_sub1 : index
                %gdy = arith.divui %m_add, %subm : index
                %gdz = arith.constant 1 : index
                hal.return %gdx, %gdy, %gdz : index, index, index
            }}
            layout(#hal.pipeline.layout<constants = 10, bindings = [
                #hal.pipeline.binding<storage_buffer, ReadOnly>,
                #hal.pipeline.binding<storage_buffer, ReadOnly>,
                #hal.pipeline.binding<storage_buffer, ReadOnly>,
                #hal.pipeline.binding<storage_buffer, ReadOnly>,
                #hal.pipeline.binding<storage_buffer, ReadOnly>,
                #hal.pipeline.binding<storage_buffer>
            ]>)
            objects({{
                #rocm_target ordinal(0) = [
                    #hal.executable.object<{{
                        path = "",
                        data = dense<"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"> : vector<20784xi8>
                    }}>
                ]
            }})
            attributes {{subgroupSize = 64 : i64, workgroup_size = [256 : index, 1 : index, 1 : index]}}
        %gemm_slice = tensor.extract_slice %gemm[0, 0] [%m, %n] [1, 1] : tensor<?x?xbf16> to tensor<?x?xbf16>
        %out_init = tensor.empty(%m, %n) : tensor<?x?xf16>
        %gemm_f16 = linalg.generic {{indexing_maps = [affine_map<(i, j) -> (i, j)>, affine_map<(i, j) -> (i, j)>], iterator_types = ["parallel", "parallel"]}} ins(%gemm_slice : tensor<?x?xbf16>) outs(%out_init : tensor<?x?xf16>) {{
        ^bb0(%in: bf16, %out: f16):
            %in_f32 = arith.extf %in : bf16 to f32
            %in_f16 = arith.truncf %in_f32 : f32 to f16
            linalg.yield %in_f16 : f16
        }} -> tensor<?x?xf16>
        util.return %gemm_f16 : tensor<?x?xf16>
    }}
    util.func private @shuffle_scales(%arg0: tensor<?x?xi8>) -> tensor<?x?xi8> {{
        %c0 = arith.constant 0 : index
        %c1 = arith.constant 1 : index
        %dim0 = tensor.dim %arg0, %c0 : tensor<?x?xi8>
        %dim1 = tensor.dim %arg0, %c1 : tensor<?x?xi8>
        %dim0_i32 = arith.index_cast %dim0 : index to i32
        %dim1_i32 = arith.index_cast %dim1 : index to i32
        %MXFP4_QUANT_BLOCK_SIZE = arith.constant 32 : i32
        %N = arith.muli %dim1_i32, %MXFP4_QUANT_BLOCK_SIZE : i32
        %scaleM_index = affine.apply affine_map<()[s0] -> (s0 ceildiv 32 * 32)>()[%dim0]
        %scaleM = arith.index_cast %scaleM_index : index to i32
        // Note: This is not safe if the dim size exceeds INT32_MAX. To pass a 64
        // bit value it must be broken down into two 32-bit values for the high and
        // low bits.
        // %dim_i32 = arith.index_cast %dim : index to i32
        // Inline external dispatch that conforms to the ABI that the kernel
        // requires. This is the primary reason for the surrounding function as
        // details like tensor shape and push constants need to line up after
        // splicing in the custom dispatch. This allows the kernel author to manage
        // such details by hand without needing the rewrite patterns to worry about
        // things like order of push constants.
        // arg6 = scaleN_pad
        // arg5 = scaleM_pad
        // arg4 = N
        // arg3 = M
        // arg2 = stride_M
        // arg1 = output
        // arg0 = input
        %4 = hal.dispatch.extern "_mxfp4_quant_shuffle"[%dim0, %dim1](%dim1_i32, %dim0_i32, %N, %scaleM, %dim1_i32, %arg0) : (i32, i32, i32, i32, i32, tensor<?x?xi8>{{%dim0, %dim1}}) -> tensor<?x?xi8>{{%dim0, %dim1}}
            count(%device: !hal.device, %dim_m: index,  %dim_n: index) -> (index, index, index) {{
                %x = affine.apply affine_map<()[s0] -> (s0 ceildiv 128)>()[%dim_m]
                %c1_0 = arith.constant 1 : index
                hal.return %x, %dim_n, %c1_0 : index, index, index
            }}
            layout(#hal.pipeline.layout<constants = 5, bindings = [
                #hal.pipeline.binding<storage_buffer, ReadOnly>,
                #hal.pipeline.binding<storage_buffer>
            ]>)
            objects({{
                #rocm_target ordinal(0) = [
                    #hal.executable.object<{{
                        path = "",
                        data = dense<"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"> : vector<6584xi8>
                    }}>
                ]
            }})
            attributes {{subgroupSize = 64, workgroup_size = [128 : index, 1 : index, 1 : index]}}
        util.return %4 : tensor<?x?xi8>
    }}
{{% endraw %}}
    util.func private @{{{{kernel_name}}}}(%x: !x, %w: !w, %x_scale: !x_scale, %w_scale: !w_scale, %bias: !bias) -> !result {{
        %x_scale_shuffle = util.call @shuffle_scales(%x_scale) : (!x_scale) -> !x_scale
        %w_scale_shuffle = util.call @shuffle_scales(%w_scale) : (!w_scale) -> !w_scale
        %result = util.call @asm_mxfp4_gemm(%x, %w, %x_scale_shuffle, %w_scale_shuffle, %bias) : (!x, !w, !x_scale, !x_scale, !bias) -> !result
        util.return %result : !result
    }}
}}
        """

    return MLIRSpec(mlir)
