"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
test('Test cross region deployment with new Lambda function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                BEDROCK_MODEL: "amazon.nova-lite-v1:0",
                BEDROCK_PROFILE: {
                    "Fn::GetAtt": [
                        assertions_1.Match.stringLikeRegexp("testlambdainferencepropstestlambdainferencepropsinference"),
                        "InferenceProfileArn"
                    ]
                }
            }
        },
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole"),
                "Arn"
            ]
        },
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Timeout: 30,
        TracingConfig: {
            Mode: "Active"
        }
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            assertions_1.Match.stringLikeRegexp("testlambdainferencepropstestlambdainferencepropsinference"),
                            "InferenceProfileArn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRoleDefaultPolicy"),
        Roles: [
            {
                "Ref": assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole")
            }
        ]
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Split": [
                            ",",
                            {
                                "Fn::FindInMap": [
                                    "testlambdainferencepropsarearegionmapping",
                                    {
                                        "Fn::Select": [
                                            0,
                                            {
                                                "Fn::Split": [
                                                    "-",
                                                    {
                                                        "Ref": "AWS::Region"
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    "regionalModels"
                                ]
                            }
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctioninlinePolicyAddedToExecutionRole"),
        Roles: [
            {
                "Ref": assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole")
            }
        ]
    });
});
test('Test single region deployment with new Lambda function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        bedrockModelId: "amazon.nova-lite-v1:0",
        deployCrossRegionProfile: false
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                BEDROCK_MODEL: "amazon.nova-lite-v1:0",
                BEDROCK_PROFILE: {
                    "Fn::GetAtt": [
                        "testlambdainferencepropstestlambdainferencepropsinference01BBD7E6",
                        "InferenceProfileArn"
                    ]
                }
            }
        },
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testlambdainferencepropsLambdaFunctionServiceRole102487FD",
                "Arn"
            ]
        },
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Timeout: 30,
        TracingConfig: {
            Mode: "Active"
        }
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            assertions_1.Match.stringLikeRegexp("testlambdainferencepropstestlambdainferencepropsinference"),
                            "InferenceProfileArn"
                        ]
                    }
                },
                {
                    Action: "bedrock:Invoke*",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    "Ref": "AWS::Partition"
                                },
                                ":bedrock:",
                                {
                                    "Ref": "AWS::Region"
                                },
                                "::foundation-model/amazon.nova-lite-v1:0"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRoleDefaultPolicy"),
        Roles: [
            {
                "Ref": assertions_1.Match.stringLikeRegexp("testlambdainferencepropsLambdaFunctionServiceRole")
            }
        ]
    });
});
test('Test deployment with VPC', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        bedrockModelId: "amazon.nova-lite-v1:0",
        deployVpc: true
    };
    const newConstruct = new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    expect(newConstruct.vpc).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.resourceCountIs("AWS::EC2::VPCEndpoint", 2);
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: assertions_1.Match.anyValue(),
            SubnetIds: [
                assertions_1.Match.anyValue(),
                assertions_1.Match.anyValue(),
            ]
        },
    });
});
test('Test deployment overridden Lambda props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            timeout: cdk.Duration.seconds(45)
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Timeout: 45
    });
});
test('Test deployment with overridden inference props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testName = "testName";
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        inferenceProfileProps: {
            inferenceProfileName: testName
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Bedrock::ApplicationInferenceProfile", {
        InferenceProfileName: testName
    });
});
test('Test deployment with invalid inference props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        inferenceProfileProps: {
            inferenceProfileName: "test",
            modelSource: {
                copyFrom: "test"
            }
        },
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    const app = () => {
        new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    };
    expect(app).toThrowError('Error - The construct will create the modelSource value, it cannot be specified in the props.\n');
});
test('Test deployment with invalid Lambda props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
        },
        existingLambdaObj: new lambda.Function(stack, "placeholder", {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
        }),
        bedrockModelId: "amazon.nova-lite-v1:0"
    };
    const app = () => {
        new lib_1.LambdaToBedrockinferenceprofile(stack, 'test-lambda-inferenceprops', props);
    };
    expect(app).toThrowError("Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n");
});
//# sourceMappingURL=data:application/json;base64,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