"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const constructs_1 = require("constructs");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * @summary The ApiGatewayToSageMakerEndpoint class.
 */
class ApiGatewayToSageMakerEndpoint extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSageMakerEndpoint class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSageMakerEndpointProps} props - user provided props for the construct.
     * @since 1.68.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckApiProps(props);
        // CheckSagemakerProps is not called because this construct can't create a Sagemaker resource
        // Setup the API Gateway
        const globalRestApiResponse = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps, props.createUsagePlan);
        this.apiGateway = globalRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalRestApiResponse.role;
        this.apiGatewayLogGroup = globalRestApiResponse.logGroup;
        // Setup the API Gateway role
        if (props.apiGatewayExecutionRole !== undefined) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
            });
            defaults.addCfnGuardSuppressRules(this.apiGatewayRole, ["IAM_NO_INLINE_POLICY_CHECK"]);
            // Setup the IAM policy for SageMaker endpoint
            const invokePolicy = new iam.Policy(this, 'InvokeEndpointPolicy', {
                statements: [
                    new iam.PolicyStatement({
                        actions: ['sagemaker:InvokeEndpoint'],
                        resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:sagemaker:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:endpoint/${props.endpointName}`]
                    })
                ]
            });
            invokePolicy.attachToRole(this.apiGatewayRole);
        }
        // Setup request validation
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-param-validator',
            // Setting this property to true makes sure the following are validated:
            // - Required request parameters in the URI
            // - Query string
            // - Headers
            validateRequestParameters: true
        });
        // Setup method and integration responses
        const methodResponses = [
            { statusCode: '200' },
            { statusCode: '500' },
            { statusCode: '400' }
        ];
        const integResponses = [];
        if (props.responseMappingTemplate !== undefined) {
            integResponses.push({
                statusCode: '200',
                responseTemplates: { 'application/json': props.responseMappingTemplate }
            });
        }
        else {
            integResponses.push({ statusCode: '200' });
        }
        integResponses.push({ statusCode: '500', selectionPattern: '5\\d{2}' }, { statusCode: '400', selectionPattern: '4\\d{2}' });
        // The SageMaker integration can be added either at the root of the API (GET https://execute-api.amazonaws.com/{some-param}),
        // or as a sub-resource (GET https://execute-api.amazonaws.com/inference/{some-param}).
        // The following lines will make sure only the necessary resources are created.
        let apiResource = this.apiGateway.root;
        if (props.resourceName !== undefined) {
            apiResource = apiResource.addResource(props.resourceName);
        }
        apiResource = apiResource.addResource(props.resourcePath);
        // Setup API Gateway method
        defaults.addProxyMethodToApiResource({
            service: 'runtime.sagemaker',
            path: `endpoints/${props.endpointName}/invocations`,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'GET',
            apiResource,
            requestTemplate: props.requestMappingTemplate,
            additionalRequestTemplates: props.additionalRequestTemplates,
            awsIntegrationProps: {
                options: { integrationResponses: integResponses }
            },
            methodOptions: {
                methodResponses,
                requestValidator,
            }
        });
    }
}
exports.ApiGatewayToSageMakerEndpoint = ApiGatewayToSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToSageMakerEndpoint[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-sagemakerendpoint.ApiGatewayToSageMakerEndpoint", version: "2.90.2" };
//# sourceMappingURL=data:application/json;base64,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