"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const iam = require("aws-cdk-lib/aws-iam");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
(0, core_1.SetConsistentFeatureFlags)(stack);
stack.templateOptions.description = 'Integration Test for aws-apigateway-sagemakerendpoint';
const existingRole = new iam.Role(stack, 'api-gateway-role', {
    assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
    description: 'existing role for SageMaker integration',
    inlinePolicies: {
        InvokePolicy: new iam.PolicyDocument({
            statements: [new iam.PolicyStatement({
                    resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:sagemaker:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:endpoint/my-endpoint`],
                    actions: ['sagemaker:InvokeEndpoint']
                })]
        })
    }
});
(0, core_1.addCfnGuardSuppressRules)(existingRole, ["IAM_NO_INLINE_POLICY_CHECK"]);
// Definitions
const requestTemplate = `{
    "instances": [
#set( $user_id = $input.params("user_id") )
#set( $items = $input.params("items") )
#foreach( $item in $items.split(",") )
    {"in0": [$user_id], "in1": [$item]}#if( $foreach.hasNext ),#end
    $esc.newline
#end
    ]
}`;
const responseTemplate = `{
    "ratings": [
#set( $predictions = $input.path("$.predictions") )
#foreach( $item in $predictions )
    $item.scores[0]#if( $foreach.hasNext ),#end
    $esc.newline
#end
    ]
}`;
const props = {
    endpointName: 'my-endpoint',
    resourcePath: '{user_id}',
    resourceName: 'predicted-ratings',
    requestMappingTemplate: requestTemplate,
    responseMappingTemplate: responseTemplate,
    apiGatewayExecutionRole: existingRole
};
new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'test-apigateway-sagemakerendpoint-overwrite', props);
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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