"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplicatedSecret = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const path = require("path");
const secretArnToNameWithSuffix = (secret) => {
    const { resourceName } = aws_cdk_lib_1.Arn.split(secret.secretArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME);
    if (!resourceName) {
        throw new Error('resourceName undefined');
    }
    return resourceName;
};
const secretReplicaArn = (secret, replicaRegion) => {
    return aws_cdk_lib_1.Arn.format({
        region: replicaRegion,
        resource: 'secret',
        service: 'secretsmanager',
        resourceName: secretArnToNameWithSuffix(secret),
        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
    }, aws_cdk_lib_1.Stack.of(secret));
};
class ReplicatedSecret extends aws_secretsmanager_1.Secret {
    constructor(scope, id, props) {
        const stackRegion = aws_cdk_lib_1.Stack.of(scope).region;
        if (aws_cdk_lib_1.Token.isUnresolved(stackRegion)) {
            throw new Error('stack region is not resolved, please be explicit');
        }
        const { replicatedKey, replicaRegions } = props;
        const masterKey = replicatedKey.tryGetRegionalKey(stackRegion);
        if (!masterKey) {
            throw new Error('attempted to create replicated secret with no key available in secret primary region');
        }
        super(scope, id, {
            secretName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            encryptionKey: masterKey,
            replicaRegions: replicaRegions.map((region) => {
                const encryptionKey = replicatedKey.getRegionalKey(region);
                if (!encryptionKey) {
                    throw new Error('attempted to replicate secret into region key is not replicated to: ' + region);
                }
                return {
                    region,
                    encryptionKey,
                };
            }),
            secretObjectValue: props.secretObjectValue,
            secretStringValue: props.secretStringValue,
        });
        this.secrets = {};
        this.masterRegion = stackRegion;
        this.replicatedKey = replicatedKey;
        this.node.addDependency(replicatedKey.dependable);
        props.replicaRegions.forEach((replicaRegion) => {
            this.secrets[replicaRegion] = aws_secretsmanager_1.Secret.fromSecretCompleteArn(this, `${replicaRegion}-replica-secret`, secretReplicaArn(this, replicaRegion));
        });
        const lambda = new aws_lambda_1.Function(this, 'lambda', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'dist', 'lambda')),
            handler: 'lambda.onEvent',
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
        });
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['secretsmanager:DescribeSecret'],
            resources: [this.secretArn],
        }));
        const provider = new custom_resources_1.Provider(this, 'provider', {
            onEventHandler: lambda,
        });
        this.dependable = new aws_cdk_lib_1.CustomResource(this, 'wait-for-replication', {
            serviceToken: provider.serviceToken,
            properties: {
                secretArn: this.secretArn,
                regions: props.replicaRegions,
            },
        });
    }
    getRegionalSecret(region) {
        const secret = this.secrets[region];
        if (region === this.masterRegion) {
            return this;
        }
        if (!secret) {
            throw new Error('No secret in region ' + region);
        }
        return secret;
    }
    grantWrite(grantee) {
        Object.values(this.secrets).forEach((secret) => {
            secret?.grantWrite(grantee);
        });
        this.replicatedKey.grantEncryptDecrypt(grantee);
        return super.grantWrite(grantee);
    }
    grantRead(grantee, versionStages) {
        Object.values(this.secrets).forEach((secret) => {
            secret?.grantRead(grantee, versionStages);
        });
        this.replicatedKey.grantDecrypt(grantee);
        return super.grantRead(grantee, versionStages);
    }
}
exports.ReplicatedSecret = ReplicatedSecret;
_a = JSII_RTTI_SYMBOL_1;
ReplicatedSecret[_a] = { fqn: "@reapit-cdk/replicated-secret.ReplicatedSecret", version: "0.1.0" };
//# sourceMappingURL=data:application/json;base64,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