import importlib
from importlib.util import spec_from_file_location
from pathlib import Path

import click

from lumipy.common import emph
import lumipy.provider as lp
from lumipy.provider.provider_sets import provider_sets


def from_module(target: Path):
    module_name = target.stem
    spec = spec_from_file_location(module_name, target)
    module = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(module)

    objs = [getattr(module, name) for name in dir(module)]
    return [obj for obj in objs if isinstance(obj, lp.BaseProvider)]


def from_csv(target, name):
    if name is None or len(name) == 0:
        raise ValueError(name)
    return [lp.PandasProvider(target, name, None)]


def main(target: str, name: str, user: str, port: int, domain: str, whitelist_me: bool):

    fpath = Path(target)

    if target in provider_sets:
        providers = provider_sets[target]
    elif fpath.suffix == '.py':
        providers = from_module(fpath)
    elif fpath.suffix == '.csv':
        fpath = target if target.startswith('http') else fpath
        providers = from_csv(fpath, name)
    else:
        raise ValueError(f'Unsupported file format or provider set: {emph(target)}')

    lp.ProviderManager(*providers, user=user, port=port, domain=domain, whitelist_me=whitelist_me).run()


@click.command()
@click.argument('target', metavar='TARGET')
@click.option('--name', help='the name to give the provider')
@click.option('--user', help='the user to run with. Can be a user ID or "global". If this argument is not specified a login window will be opened.')
@click.option('--port', help='the port that the python provider API should run at.', default=5001, type=int)
@click.option('--domain', help='the client domain to run the providers in.')
@click.option('--whitelist-me', help='Whitelist your machine name. Required when running globally.', type=bool, is_flag=True)
def run(target: str, name: str, user, port, domain, whitelist_me):
    """Run one or more providers.

    TARGET can be one of the following

    A .py file:

        This will import the module and extract all the objects that inherit from lumipy's BaseProvider class and then start a manager that contains them.

        For example:

            lumipy run path/to/my_providers.py

    A .csv file:

        This will load the CSV into a pandas provider and start it in a manager. You must supply a name in this case.

        For example:

            lumipy run path/to/my_data.csv --name=my.data.csv

        Note: this can be a URL as well.

    A named built-in provider set:

        For example:

            lumipy run demo

    """
    main(target, name, user, port, domain, whitelist_me)
