import time
import datetime
import typer
import httpx
import json
from pathlib import Path

from rich.console import Console

from runlayer_cli.deploy.config import load_config, load_config_raw, update_config_id
from runlayer_cli.api import RunlayerClient
from runlayer_cli.env_substitution import load_env_vars, substitute_env_vars
from runlayer_cli.deploy.docker_builder import (
    check_docker_available,
    build_image,
    authenticate_ecr,
    tag_image,
    push_image,
    DockerBuildError,
    DockerPushError,
)

console = Console()


def _extract_validation_error(exception: Exception) -> str:
    """
    Extract validation error message from HTTP exception.

    For 422 validation errors, extracts the 'detail' field from the JSON response.
    For other errors, returns a generic error message.

    Args:
        exception: Exception raised from API call

    Returns:
        User-friendly error message string
    """
    if isinstance(exception, httpx.HTTPStatusError):
        if exception.response.status_code == 422:
            try:
                error_data = exception.response.json()
                # Extract detail field if present
                if isinstance(error_data, dict) and "detail" in error_data:
                    return str(error_data["detail"])
                # Fallback to full JSON if detail not found
                return json.dumps(error_data)
            except (ValueError, json.JSONDecodeError, KeyError):
                # If JSON parsing fails, return response text
                return exception.response.text or str(exception)
        # For other HTTP errors, return status and text
        return f"HTTP {exception.response.status_code}: {exception.response.text or str(exception)}"

    # For non-HTTP exceptions, return string representation
    return str(exception)


def _validate_runlayer_yaml_config(
    api_client: RunlayerClient, yaml_content: str
) -> None:
    """
    Validate runlayer.yaml configuration via backend API.

    This validates the YAML before proceeding with deployment steps like Docker build.
    Exits with error code if validation fails.

    Args:
        api_client: Runlayer API client
        yaml_content: Raw YAML string to validate

    Raises:
        typer.Exit: If validation fails
    """
    typer.echo("Validating configuration...")

    try:
        result = api_client.validate_yaml(yaml_content)

        if not result.valid:
            error_msg = result.error or "Configuration validation failed"
            typer.secho(
                f"✗ Configuration validation error: {error_msg}",
                fg=typer.colors.RED,
                err=True,
            )
            raise typer.Exit(1)

        typer.secho(
            "✓ Configuration is valid\n",
            fg=typer.colors.GREEN,
            bold=True,
        )
    except typer.Exit:
        # Re-raise typer.Exit to avoid catching it below
        raise
    except httpx.HTTPStatusError as e:
        error_msg = _extract_validation_error(e)
        typer.secho(
            f"✗ Configuration validation error: {error_msg}",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)
    except Exception as e:
        error_msg = _extract_validation_error(e)
        typer.secho(
            f"✗ Failed to validate configuration: {error_msg}",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)


def init_deployment_config(name: str, config_path: str, secret: str, host: str) -> None:
    """
    Initialize a new deployment by creating it in the backend and generating config file.

    Args:
        name: Deployment name (URL-friendly)
        config_path: Path to create the config file
        secret: API secret for authentication
        host: Runlayer API host URL

    Raises:
        typer.Exit: If validation fails or file already exists
    """
    import re
    from pathlib import Path

    # Validate name format (lowercase, alphanumeric, hyphens, underscores)
    if not re.match(r"^[a-z0-9_-]+$", name):
        typer.echo(
            "Error: Name must be lowercase and URL-friendly (a-z, 0-9, -, _)",
            err=True,
        )
        raise typer.Exit(1)

    # Check if file already exists
    path = Path(config_path)
    if path.exists():
        typer.echo(f"Error: Configuration file already exists: {config_path}", err=True)
        raise typer.Exit(1)

    # Initialize API client
    api_client = RunlayerClient(hostname=host, secret=secret)

    typer.echo(f"Creating deployment: {name}...")
    try:
        # Create deployment (template is always generated by backend)
        deployment = api_client.create_deployment(name)
        deployment_id = deployment.id
        typer.secho(
            f"✓ Deployment created with ID: {deployment_id}",
            fg=typer.colors.GREEN,
            bold=True,
        )
    except Exception as e:
        typer.secho(
            f"✗ Failed to create deployment: {e}", fg=typer.colors.RED, err=True
        )
        raise typer.Exit(1)

    if not deployment.template_yaml:
        typer.secho(
            "✗ Failed to get template from backend", fg=typer.colors.RED, err=True
        )
        raise typer.Exit(1)

    with open(path, "w") as f:
        f.write(deployment.template_yaml)

    typer.secho(f"\n✓ Created {config_path}", fg=typer.colors.GREEN, bold=True)
    typer.echo("\nNext steps:")
    typer.echo("  1. Edit the configuration file to match your service")
    typer.echo(
        f"  2. Run: uvx runlayer deploy --config {config_path} --secret <key> --host {host}"
    )


def deploy_service(
    config_path: str,
    secret: str,
    host: str,
    env_file: str | None = None,
) -> None:
    """
    Deploy a service based on runlayer.yaml configuration.

    This function orchestrates the entire deployment process:
    1. Load and validate the configuration
    2. Create or update the deployment
    3. Build the Docker image (if build mode) OR use provided image (if image mode)
    4. Push the image to ECR (if build mode)
    5. Trigger the deployment
    6. Monitor deployment status

    Args:
        config_path: Path to runlayer.yaml configuration file
        secret: API secret for authentication
        host: Runlayer API host URL
        env_file: Optional path to .env file for environment variable substitution

    Raises:
        typer.Exit: If any step fails
    """
    # Track deployment time
    start_time = time.time()

    # Load configuration
    typer.echo(f"Loading configuration from {config_path}...")
    try:
        config = load_config(config_path)
        # Also load raw YAML for backend validation
        yaml_content = load_config_raw(config_path)
    except FileNotFoundError as e:
        typer.secho(f"✗ Error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)
    except ValueError as e:
        typer.secho(f"✗ Configuration error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)

    # Substitute environment variables in YAML content
    # Search for .env file in the same directory as config file if not explicitly provided
    try:
        config_dir = Path(config_path).parent if config_path else None
        env_vars = load_env_vars(env_file, search_dir=config_dir)
        yaml_content = substitute_env_vars(yaml_content, env_vars)
    except FileNotFoundError as e:
        typer.secho(f"✗ Error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)
    except ValueError as e:
        typer.secho(f"✗ Environment variable error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)

    typer.secho(
        f"✓ Configuration loaded: {config.get('name', 'unknown')}\n",
        fg=typer.colors.GREEN,
        bold=True,
    )

    # Initialize API client
    api_client = RunlayerClient(hostname=host, secret=secret)

    # Validate configuration early
    _validate_runlayer_yaml_config(api_client, yaml_content)

    # Create or get deployment
    deployment_id = _get_or_create_deployment(api_client, config, config_path)

    # Detect deployment mode: build if no image in config, else use config image
    docker_image: str | None = None

    if not config.get("image"):
        # Build mode: requires Docker
        if not check_docker_available():
            typer.secho(
                "✗ Error: Docker is not available or not running.",
                fg=typer.colors.RED,
                err=True,
            )
            typer.echo("Please ensure Docker is installed and running.", err=True)
            raise typer.Exit(1)

        # Build Docker image
        image_id = _build_docker_image(config, deployment_id)

        # Push to ECR
        docker_image = _push_to_ecr(api_client, image_id, deployment_id)
    else:
        # Image mode: use image from YAML config
        typer.secho(
            f"Using pre-built image from config: {config.get('image')}\n",
            fg=typer.colors.CYAN,
            bold=True,
        )

    # Update deployment configuration with raw YAML
    _update_deployment_config(api_client, deployment_id, docker_image, yaml_content)

    # Trigger and monitor deployment
    _trigger_and_monitor_deployment(api_client, deployment_id)

    # Calculate and display total deployment time
    elapsed_time = time.time() - start_time
    minutes = int(elapsed_time // 60)
    seconds = int(elapsed_time % 60)

    if minutes > 0:
        time_str = f"{minutes}m {seconds}s"
    else:
        time_str = f"{seconds}s"

    typer.secho(
        f"\n⏱️  Total deployment time: {time_str}",
        fg=typer.colors.CYAN,
        bold=True,
    )


def _get_or_create_deployment(
    api_client: RunlayerClient,
    config: dict,
    config_path: str,
) -> str:
    """
    Get existing deployment or create a new one.

    Args:
        api_client: Runlayer API client
        config: Deployment configuration dict
        config_path: Path to config file (for updating with new ID)

    Returns:
        Deployment ID

    Raises:
        typer.Exit: If deployment creation/retrieval fails
    """
    deployment_id = config.get("id")

    if not deployment_id:
        deployment_name = config.get("name", "unnamed-deployment")
        typer.echo(f"Creating new deployment: {deployment_name}...")
        try:
            deployment = api_client.create_deployment(deployment_name)
            deployment_id = deployment.id
            typer.secho(
                f"✓ Deployment created with ID: {deployment_id}",
                fg=typer.colors.GREEN,
                bold=True,
            )

            update_config_id(config_path, deployment_id)
            typer.secho(
                f"✓ Updated {config_path} with deployment ID\n",
                fg=typer.colors.GREEN,
                bold=True,
            )
        except Exception as e:
            typer.secho(
                f"✗ Failed to create deployment: {e}", fg=typer.colors.RED, err=True
            )
            raise typer.Exit(1)
    else:
        typer.echo(f"Verifying deployment: {deployment_id}...")
        try:
            deployment = api_client.get_deployment(deployment_id)

            # Check if deployment is deleted or being deleted
            if deployment.deletion_status in ["deleted", "deleting"]:
                status_message = (
                    "being deleted"
                    if deployment.deletion_status == "deleting"
                    else "deleted"
                )
                typer.secho(
                    f"✗ Error: Deployment is {status_message}",
                    fg=typer.colors.RED,
                    err=True,
                )
                typer.echo(
                    f"Cannot deploy to a {status_message} deployment.",
                    err=True,
                )
                raise typer.Exit(1)

            typer.secho(
                f"✓ Found deployment: {deployment.name}\n",
                fg=typer.colors.GREEN,
                bold=True,
            )
        except typer.Exit:
            # Re-raise typer.Exit to avoid catching it below
            raise
        except Exception as e:
            typer.secho(
                f"✗ Error: Deployment {deployment_id} not found",
                fg=typer.colors.RED,
                err=True,
            )
            typer.echo(f"Details: {e}", err=True)
            raise typer.Exit(1)

    return deployment_id


def _build_docker_image(config: dict, deployment_id: str) -> str:
    """
    Build Docker image locally.

    Args:
        config: Deployment configuration dict
        deployment_id: Deployment ID for tagging

    Returns:
        Image ID

    Raises:
        typer.Exit: If build fails
    """
    typer.echo("Building Docker image...\n")

    build_config = config.get("build", {})
    platform = build_config.get("platform", "x86")

    docker_platform = None
    if platform == "arm":
        docker_platform = "linux/arm64"
    elif platform == "x86":
        docker_platform = "linux/amd64"

    local_tag = f"runlayer-build:{deployment_id}"
    try:
        image_id = build_image(
            context=build_config.get("context", "."),
            dockerfile=build_config.get("dockerfile", "Dockerfile"),
            tag=local_tag,
            build_args=build_config.get("args", {}),
            platform=docker_platform,
            target=build_config.get("target"),
        )
        typer.secho(
            f"✓ Image built: {image_id[:12]}\n", fg=typer.colors.GREEN, bold=True
        )
        return image_id
    except DockerBuildError as e:
        typer.secho(f"✗ Docker build failed: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)


def _push_to_ecr(
    api_client: RunlayerClient,
    image_id: str,
    deployment_id: str,
) -> str:
    """
    Push Docker image to ECR.

    Args:
        api_client: Runlayer API client
        image_id: Docker image ID to push
        deployment_id: Deployment ID for tagging

    Returns:
        Full image URI with digest (sha256) for immutable reference

    Raises:
        typer.Exit: If push fails
    """
    typer.echo("Getting fresh ECR credentials for push...")
    try:
        ecr_creds = api_client.get_ecr_credentials()

        now = datetime.datetime.now(datetime.timezone.utc)
        expires_at = ecr_creds.expires_at
        if expires_at.tzinfo is None:
            expires_at = expires_at.replace(tzinfo=datetime.timezone.utc)
        time_until_expiry = (expires_at - now).total_seconds()

        typer.secho(
            f"✓ ECR credentials obtained (valid for {int(time_until_expiry / 60)} minutes)\n",
            fg=typer.colors.GREEN,
            bold=True,
        )
    except Exception as e:
        typer.secho(
            f"✗ Failed to get ECR credentials: {e}", fg=typer.colors.RED, err=True
        )
        raise typer.Exit(1)

    if not ecr_creds.repository_url:
        typer.secho(
            "✗ Error: ECR repository not configured on backend",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)

    repository = ecr_creds.repository_url

    typer.echo("Pushing image to ECR...\n")
    try:
        authenticate_ecr(ecr_creds)
        full_image_uri = tag_image(image_id, repository, deployment_id)
        image_digest = push_image(full_image_uri)

        # Use digest-based reference for immutable deployments
        # This ensures ECS detects changes even with the same tag
        repository_without_tag = repository.split(":")[0]
        image_uri_with_digest = f"{repository_without_tag}@{image_digest}"

        typer.secho(
            f"✓ Image pushed: {full_image_uri}\n", fg=typer.colors.GREEN, bold=True
        )
        typer.secho(
            f"✓ Image digest: {image_digest[:19]}...\n",
            fg=typer.colors.GREEN,
            bold=True,
        )

        return image_uri_with_digest
    except DockerPushError as e:
        typer.secho(f"✗ Docker push failed: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)
    except Exception as e:
        typer.secho(f"✗ Unexpected error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)


def _update_deployment_config(
    api_client: RunlayerClient,
    deployment_id: str,
    docker_image: str | None,
    yaml_content: str,
) -> None:
    """
    Update deployment configuration on the backend.

    Backend handles all YAML parsing and validation.
    If docker_image is provided, backend uses it; otherwise uses image from YAML.

    Args:
        api_client: Runlayer API client
        deployment_id: Deployment ID
        docker_image: Docker image URI from build (None if using image from YAML)
        yaml_content: Raw YAML string from config file

    Raises:
        typer.Exit: If update fails or validation fails
    """
    typer.echo("Updating deployment configuration...")

    try:
        # Send raw YAML and docker_image separately to backend
        # Backend will parse/validate YAML and add docker_image to config if provided
        # If docker_image is None and YAML has image field, backend will use YAML image
        api_client.update_deployment(
            deployment_id,
            yaml_content=yaml_content,
            docker_image=docker_image,
        )
        typer.secho(
            "✓ Deployment configuration updated\n", fg=typer.colors.GREEN, bold=True
        )
    except httpx.HTTPStatusError as e:
        error_msg = _extract_validation_error(e)
        # Check if it's a validation error (422)
        if e.response.status_code == 422:
            typer.secho(
                f"✗ Configuration validation error: {error_msg}",
                fg=typer.colors.RED,
                err=True,
            )
        else:
            typer.secho(
                f"✗ Failed to update deployment configuration: {error_msg}",
                fg=typer.colors.RED,
                err=True,
            )
        raise typer.Exit(1)
    except Exception as e:
        error_msg = _extract_validation_error(e)
        typer.secho(
            f"✗ Failed to update deployment configuration: {error_msg}",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)


def _trigger_and_monitor_deployment(
    api_client: RunlayerClient,
    deployment_id: str,
) -> None:
    """
    Trigger deployment and monitor its progress with live log streaming.

    Args:
        api_client: Runlayer API client
        deployment_id: Deployment ID

    Raises:
        typer.Exit: If deployment fails
    """
    typer.echo("Triggering deployment...")
    try:
        trigger_response = api_client.trigger_deployment(deployment_id)
        history_id = trigger_response.history_id
        typer.secho(
            f"✓ Deployment triggered (History ID: {history_id})\n",
            fg=typer.colors.GREEN,
            bold=True,
        )
    except Exception as e:
        typer.secho(
            f"✗ Failed to trigger deployment: {e}", fg=typer.colors.RED, err=True
        )
        raise typer.Exit(1)

    typer.echo("─" * 80)

    max_polls = 60
    poll_count = 0
    last_log_line_count = 0

    while poll_count < max_polls:
        try:
            # Get deployment status
            status_response = api_client.get_deployment_status(deployment_id)
            status = status_response.get("status", "unknown")

            # Get logs
            logs = api_client.get_deployment_logs(history_id)

            # Display new log lines
            if logs:
                log_lines = logs.split("\n")
                new_lines = log_lines[last_log_line_count:]

                for line in new_lines:
                    if line.strip():  # Only print non-empty lines
                        typer.echo(line)

                last_log_line_count = len(log_lines)

            # Check if deployment is complete
            if status in ["completed", "succeeded"]:
                typer.echo("─" * 80)
                typer.secho(
                    "\n✓ Deployment completed successfully!",
                    fg=typer.colors.GREEN,
                    bold=True,
                )
                break
            elif status == "failed":
                typer.echo("─" * 80)
                typer.secho(
                    "\n✗ Deployment failed!", fg=typer.colors.RED, bold=True, err=True
                )
                if "error" in status_response:
                    typer.secho(
                        f"   Error: {status_response.get('error')}",
                        fg=typer.colors.RED,
                        err=True,
                    )
                raise typer.Exit(1)

            # Wait before next poll
            time.sleep(2)
            poll_count += 1

        except typer.Exit:
            raise
        except Exception as e:
            typer.secho(
                f"\n✗ Error monitoring deployment: {e}", fg=typer.colors.RED, err=True
            )
            raise typer.Exit(1)

    if poll_count >= max_polls:
        typer.echo("─" * 80)
        typer.secho(
            "\n⚠ Warning: Deployment is still in progress. Check the web UI for status.",
            fg=typer.colors.YELLOW,
        )


def validate_service(
    config_path: str,
    secret: str,
    host: str,
    env_file: str | None = None,
) -> None:
    """
    Validate runlayer.yaml configuration without deploying.

    This function validates the configuration file by calling the backend
    validation endpoint. It does not create or update any deployments.

    Args:
        config_path: Path to runlayer.yaml configuration file
        secret: API secret for authentication
        host: Runlayer API host URL
        env_file: Optional path to .env file for environment variable substitution

    Raises:
        typer.Exit: If validation fails or file cannot be read
    """
    typer.echo(f"Loading configuration from {config_path}...")

    try:
        yaml_content = load_config_raw(config_path)
    except FileNotFoundError as e:
        typer.secho(f"✗ Error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)
    except ValueError as e:
        typer.secho(f"✗ Configuration error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)

    # Substitute environment variables in YAML content
    # Search for .env file in the same directory as config file if not explicitly provided
    try:
        config_dir = Path(config_path).parent if config_path else None
        env_vars = load_env_vars(env_file, search_dir=config_dir)
        yaml_content = substitute_env_vars(yaml_content, env_vars)
    except FileNotFoundError as e:
        typer.secho(f"✗ Error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)
    except ValueError as e:
        typer.secho(f"✗ Environment variable error: {e}", fg=typer.colors.RED, err=True)
        raise typer.Exit(1)

    # Initialize API client
    api_client = RunlayerClient(hostname=host, secret=secret)

    # Validate configuration
    _validate_runlayer_yaml_config(api_client, yaml_content)

    typer.secho(
        "\n✓ Configuration is valid and ready for deployment!",
        fg=typer.colors.GREEN,
        bold=True,
    )


def destroy_deployment(
    config_path: str,
    secret: str,
    host: str,
    deployment_id: str | None = None,
) -> None:
    """
    Destroy a deployment and its infrastructure.

    Args:
        config_path: Path to runlayer.yaml configuration file (optional if deployment_id provided)
        secret: API secret for authentication
        host: Runlayer API host URL
        deployment_id: Direct deployment ID (optional, takes precedence over config)

    Raises:
        typer.Exit: If destruction fails
    """
    # Determine deployment ID
    if deployment_id:
        deployment_id_to_use = deployment_id
        typer.echo(f"Using deployment ID: {deployment_id_to_use}")
    else:
        # Load config to get deployment ID
        try:
            config = load_config(config_path)
            deployment_id_from_config = config.get("id")
            if not deployment_id_from_config or not isinstance(
                deployment_id_from_config, str
            ):
                typer.secho(
                    f"✗ Error: No deployment ID found in {config_path}",
                    fg=typer.colors.RED,
                    err=True,
                )
                typer.echo(
                    "The configuration file must have an 'id' field with the deployment UUID.",
                    err=True,
                )
                raise typer.Exit(1)
            deployment_id_to_use = deployment_id_from_config
            typer.echo(
                f"Loaded deployment ID from {config_path}: {deployment_id_to_use}"
            )
        except FileNotFoundError:
            typer.secho(
                f"✗ Error: Configuration file not found: {config_path}",
                fg=typer.colors.RED,
                err=True,
            )
            raise typer.Exit(1)
        except ValueError as e:
            typer.secho(f"✗ Configuration error: {e}", fg=typer.colors.RED, err=True)
            raise typer.Exit(1)

    # Initialize API client
    api_client = RunlayerClient(hostname=host, secret=secret)

    # Get deployment details
    try:
        deployment = api_client.get_deployment(deployment_id_to_use)
        typer.echo(f"\nDeployment: {deployment.name}")
        typer.echo(f"Created: {deployment.created_at}")
    except Exception as e:
        typer.secho(
            f"✗ Error fetching deployment details: {e}",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)

    # Check if deployment is already deleted or being deleted
    if deployment.deletion_status in ["deleted", "deleting"]:
        status_message = (
            "being deleted"
            if deployment.deletion_status == "deleting"
            else "already deleted"
        )
        typer.secho(
            f"\n✓ Deployment is {status_message}",
            fg=typer.colors.GREEN,
            bold=True,
        )
        if deployment.deletion_status == "deleted":
            typer.echo("The deployment has already been destroyed.")
        else:
            typer.echo("The deployment destruction is in progress.")
        raise typer.Exit(0)

    # Check if deployment has connected servers
    if deployment.connected_servers:
        typer.secho(
            f"\n⚠ Warning: This deployment has {len(deployment.connected_servers)} connected MCP server(s):",
            fg=typer.colors.YELLOW,
        )
        for server in deployment.connected_servers:
            typer.echo(f"  • {server.get('name', 'Unknown')}")
        typer.echo(
            "\nThese servers will need to be deleted before the deployment can be destroyed.",
        )
        if not typer.confirm("\nDo you still want to proceed?"):
            typer.echo("Deployment destruction cancelled.")
            raise typer.Exit(0)

    # Confirm destruction
    typer.secho(
        f"\n⚠ WARNING: You are about to destroy deployment '{deployment.name}'",
        fg=typer.colors.RED,
        bold=True,
    )
    typer.echo("This will:")
    typer.echo("  • Tear down all infrastructure resources")
    typer.echo("  • Delete the deployment configuration")
    typer.echo("  • This action CANNOT be undone")

    if not typer.confirm("\nAre you sure you want to continue?"):
        typer.echo("Deployment destruction cancelled.")
        raise typer.Exit(0)

    # Delete deployment
    typer.echo(f"\nInitiating destruction of deployment '{deployment.name}'...")
    try:
        api_client.delete_deployment(deployment_id_to_use)
        typer.secho(
            "\n✓ Deployment destruction initiated successfully!",
            fg=typer.colors.GREEN,
            bold=True,
        )
        typer.echo(
            "\nThe infrastructure teardown has been queued and will be processed shortly."
        )
        typer.echo("You can monitor the progress in the web UI.")
    except Exception as e:
        typer.secho(
            f"✗ Error destroying deployment: {e}",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)


def _create_backup_path(config_path: str) -> str:
    """
    Create a timestamped backup path for the config file.

    Args:
        config_path: Original config file path

    Returns:
        Backup file path with timestamp (e.g., runlayer.yaml.backup.2025-12-03-143022)
    """
    timestamp = datetime.datetime.now().strftime("%Y-%m-%d-%H%M%S")
    return f"{config_path}.backup.{timestamp}"


def pull_deployment(
    config_path: str,
    secret: str,
    host: str,
    deployment_id: str | None = None,
) -> None:
    """
    Pull deployment configuration from the backend and save as YAML.

    Args:
        config_path: Path to save the runlayer.yaml configuration file
        secret: API secret for authentication
        host: Runlayer API host URL
        deployment_id: Direct deployment ID (optional, takes precedence over config)

    Raises:
        typer.Exit: If pull fails
    """
    # Determine deployment ID
    deployment_id_to_use: str | None = None

    if deployment_id:
        deployment_id_to_use = deployment_id
        typer.echo(f"Using deployment ID: {deployment_id_to_use}")
    else:
        # Try to load config to get deployment ID
        path = Path(config_path)
        if path.exists():
            try:
                config = load_config(config_path)
                deployment_id_from_config = config.get("id")
                if deployment_id_from_config and isinstance(
                    deployment_id_from_config, str
                ):
                    deployment_id_to_use = deployment_id_from_config
                    typer.echo(
                        f"Loaded deployment ID from {config_path}: {deployment_id_to_use}"
                    )
            except (FileNotFoundError, ValueError):
                pass

    if not deployment_id_to_use:
        typer.secho(
            "✗ Error: No deployment ID provided",
            fg=typer.colors.RED,
            err=True,
        )
        typer.echo(
            "Provide a deployment ID with --deployment-id or use an existing runlayer.yaml file.",
            err=True,
        )
        raise typer.Exit(1)

    # Initialize API client
    api_client = RunlayerClient(hostname=host, secret=secret)

    # Get deployment details first to verify it exists
    try:
        deployment = api_client.get_deployment(deployment_id_to_use)
        typer.echo(f"\nDeployment: {deployment.name}")

        # Check if deployment is deleted or being deleted
        if deployment.deletion_status in ["deleted", "deleting"]:
            status_message = (
                "being deleted"
                if deployment.deletion_status == "deleting"
                else "deleted"
            )
            typer.secho(
                f"✗ Error: Deployment is {status_message}",
                fg=typer.colors.RED,
                err=True,
            )
            raise typer.Exit(1)
    except typer.Exit:
        raise
    except Exception as e:
        typer.secho(
            f"✗ Error fetching deployment details: {e}",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)

    # Export YAML from backend
    typer.echo("\nFetching deployment configuration...")
    try:
        yaml_content = api_client.export_deployment_yaml(deployment_id_to_use)
    except typer.Exit:
        raise
    except Exception as e:
        typer.secho(
            f"✗ Error exporting deployment configuration: {e}",
            fg=typer.colors.RED,
            err=True,
        )
        raise typer.Exit(1)

    # Create backup if file exists
    path = Path(config_path)
    backup_path: str | None = None
    if path.exists():
        backup_path = _create_backup_path(config_path)
        try:
            path.rename(backup_path)
            typer.secho(
                f"✓ Created backup: {backup_path}",
                fg=typer.colors.YELLOW,
                bold=True,
            )
        except Exception as e:
            typer.secho(
                f"✗ Error creating backup: {e}",
                fg=typer.colors.RED,
                err=True,
            )
            raise typer.Exit(1)

    # Write new YAML content
    try:
        with open(config_path, "w") as f:
            f.write(yaml_content)
        typer.secho(
            f"\n✓ Configuration saved to: {config_path}",
            fg=typer.colors.GREEN,
            bold=True,
        )
    except Exception as e:
        typer.secho(
            f"✗ Error writing configuration: {e}",
            fg=typer.colors.RED,
            err=True,
        )
        # Attempt to restore backup if one was created
        if backup_path:
            try:
                Path(backup_path).rename(config_path)
                typer.secho(
                    "✓ Restored original file from backup",
                    fg=typer.colors.YELLOW,
                    err=True,
                )
            except Exception as restore_error:
                typer.secho(
                    f"⚠ Could not restore backup automatically: {restore_error}",
                    fg=typer.colors.YELLOW,
                    err=True,
                )
                typer.echo(
                    f"Your original configuration was saved to: {backup_path}",
                    err=True,
                )
        raise typer.Exit(1)

    typer.echo("\nNext steps:")
    typer.echo(f"  1. Review {config_path} and update any environment variable values")
    typer.echo("  2. Environment variables are shown as ${VAR_NAME} placeholders")
    typer.echo(
        "  3. Set actual values in your environment or .env file before deploying"
    )
