from __future__ import annotations

import asyncio
import hashlib
import json
import socket
import tempfile
import time
import webbrowser
from pathlib import Path
from typing import Any, Literal
from urllib.parse import urlparse

import anyio
import httpx
import structlog
from mcp.client.auth import OAuthClientProvider, TokenStorage
from mcp.shared.auth import (
    OAuthClientInformationFull,
    OAuthClientMetadata,
    OAuthMetadata,
    OAuthToken,
)
from pydantic import AnyHttpUrl, ValidationError
from runlayer_cli.oauth_callback import create_oauth_callback_server
from runlayer_cli.paths import get_runlayer_dir


logger = structlog.get_logger(__name__)


def default_cache_dir() -> Path:
    return get_runlayer_dir() / "oauth-mcp-client-cache"


def get_free_port() -> int:
    """
    Find and return a free port on localhost.

    Returns:
        An available port number
    """
    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
        s.bind(("localhost", 0))
        s.listen(1)
        port = s.getsockname()[1]
        return port


def get_browser_lockfile_path(server_url: str) -> Path:
    """
    Get a deterministic lockfile path for a server URL.

    Args:
        server_url: The server URL to generate a lockfile for

    Returns:
        Path to the lockfile in the system's temporary directory
    """
    # Create a deterministic hash of the server URL
    url_hash = hashlib.sha256(server_url.encode()).hexdigest()[:16]
    temp_dir = Path(tempfile.gettempdir())
    return temp_dir / f"runlayer_oauth_browser_{url_hash}.lock"


def should_open_browser(server_url: str, window_seconds: float = 3.0) -> bool:
    """
    Check if a browser should be opened for the given server.

    Prevents opening multiple browser tabs within the specified time window.
    Uses atomic file creation to avoid race conditions.

    Args:
        server_url: The server URL to check
        window_seconds: Time window in seconds (default: 3.0)

    Returns:
        True if browser should be opened, False otherwise
    """
    lockfile = get_browser_lockfile_path(server_url)
    current_time = time.time()

    try:
        # Try to atomically create the lockfile (prevents race condition)
        with lockfile.open("x") as f:
            f.write(str(current_time))
        return True

    except FileExistsError:
        # Lockfile exists - check if it's still valid
        try:
            last_opened = float(lockfile.read_text().strip())
            time_since_last = current_time - last_opened

            if time_since_last < window_seconds:
                logger.debug(
                    f"Skipping browser open for {server_url} "
                    f"(last opened {time_since_last:.1f}s ago)"
                )
                return False

            # Lockfile is stale - try to update it
            lockfile.write_text(str(current_time))
            return True

        except (OSError, ValueError) as e:
            logger.debug(f"Error reading lockfile: {e}")
            # If we can't read the lockfile, try to recreate it
            try:
                lockfile.unlink()
                lockfile.write_text(str(current_time))
                return True
            except OSError:
                # If all else fails, allow browser to open
                return True

    except (OSError, ValueError) as e:
        # If anything goes wrong with lockfile creation, err on the side of opening the browser
        logger.debug(f"Error checking browser lockfile: {e}")
        return True


class FileTokenStorage(TokenStorage):
    """
    File-based token storage implementation for OAuth credentials and tokens.
    Implements the mcp.client.auth.TokenStorage protocol.

    Each instance is tied to a specific server URL for proper token isolation.
    """

    def __init__(self, server_url: str, cache_dir: Path | None = None):
        """Initialize storage for a specific server URL."""
        self.server_url = server_url
        self.cache_dir = cache_dir or default_cache_dir()
        self.cache_dir.mkdir(exist_ok=True, parents=True)

    @staticmethod
    def get_base_url(url: str) -> str:
        """Extract the base URL (scheme + host) from a URL."""
        parsed = urlparse(url)
        return f"{parsed.scheme}://{parsed.netloc}"

    def get_cache_key(self) -> str:
        """Generate a safe filesystem key from the server's base URL."""
        base_url = self.get_base_url(self.server_url)
        return (
            base_url.replace("://", "_")
            .replace(".", "_")
            .replace("/", "_")
            .replace(":", "_")
        )

    def _get_file_path(self, file_type: Literal["client_info", "tokens"]) -> Path:
        """Get the file path for the specified cache file type."""
        key = self.get_cache_key()
        return self.cache_dir / f"{key}_{file_type}.json"

    async def get_tokens(self) -> OAuthToken | None:
        """Load tokens from file storage."""
        path = self._get_file_path("tokens")

        try:
            data = json.loads(path.read_text())

            # Load token from format
            if "token" in data:
                token = OAuthToken.model_validate(data["token"])
            else:
                # Old format - just the token fields directly
                token = OAuthToken.model_validate(data)

            return token
        except (FileNotFoundError, json.JSONDecodeError, ValidationError) as e:
            logger.info(
                f"Could not load tokens for {self.get_base_url(self.server_url)}: {e}"
            )
            return None

    def get_token_expiry_time(self) -> float | None:
        """Load the absolute expiration timestamp from file storage."""
        path = self._get_file_path("tokens")
        try:
            data = json.loads(path.read_text())
            # New format includes expiry_time alongside token
            expiry_time = data.get("expiry_time")
            if expiry_time is not None and not isinstance(expiry_time, (int, float)):
                logger.warning(
                    "Invalid expiry_time in token cache",
                    expiry_time=expiry_time,
                    server_url=self.get_base_url(self.server_url),
                )
                return None
            return expiry_time
        except (FileNotFoundError, json.JSONDecodeError):
            return None

    def get_oauth_metadata(self) -> dict[str, Any] | None:
        """Load OAuth metadata from client_info file storage."""
        path = self._get_file_path("client_info")
        try:
            data = json.loads(path.read_text())
            return data.get("oauth_metadata")
        except (FileNotFoundError, json.JSONDecodeError):
            return None

    def set_oauth_metadata(self, oauth_metadata: dict[str, Any]) -> None:
        """Save OAuth metadata to client_info file storage."""
        path = self._get_file_path("client_info")
        try:
            # Load existing data
            data = json.loads(path.read_text())
        except (FileNotFoundError, json.JSONDecodeError):
            # If file doesn't exist or is invalid, create new structure
            data = {}

        # Add/update oauth_metadata
        data["oauth_metadata"] = oauth_metadata
        path.write_text(json.dumps(data, indent=2))
        logger.info(f"Saved OAuth metadata for {self.get_base_url(self.server_url)}")

    async def set_tokens(self, tokens: OAuthToken) -> None:
        """Save tokens to file storage."""

        # Calculate absolute expiration timestamp
        # We store this alongside the token because OAuthToken only contains a relative `expires_in` field
        expiry_time = (
            time.time() + tokens.expires_in if tokens.expires_in is not None else None
        )

        # Store token and expiry_time
        path = self._get_file_path("tokens")
        data = {
            "token": tokens.model_dump(mode="json"),
            "expiry_time": expiry_time,
        }
        path.write_text(json.dumps(data, indent=2))

        logger.info(
            f"Saved tokens for {self.get_base_url(self.server_url)}",
            expiry_time=expiry_time,
        )

    async def get_client_info(self) -> OAuthClientInformationFull | None:
        """Load client information from file storage."""
        path = self._get_file_path("client_info")

        try:
            data = json.loads(path.read_text())

            # Extract client_info (the file contains both client_info and oauth_metadata)
            # The parent class expects just the client_info fields
            if "client_id" in data:
                # New format: client_info fields are at root level alongside oauth_metadata
                # Create a copy without oauth_metadata for validation
                client_info_data = {
                    k: v for k, v in data.items() if k != "oauth_metadata"
                }
                return OAuthClientInformationFull.model_validate(client_info_data)
            else:
                # Invalid format
                logger.warning(
                    f"Invalid cache format for {self.get_base_url(self.server_url)} "
                    "(missing client_id). Clearing cache."
                )
                self.clear()
                return None

        except (FileNotFoundError, json.JSONDecodeError, ValidationError) as e:
            logger.info(
                f"Could not load client info for {self.get_base_url(self.server_url)}: {e}"
            )
            return None

    async def set_client_info(self, client_info: OAuthClientInformationFull) -> None:
        """Save client information to file storage, preserving oauth_metadata if it exists."""
        path = self._get_file_path("client_info")

        # Try to preserve existing oauth_metadata
        existing_oauth_metadata = None
        try:
            existing_data = json.loads(path.read_text())
            existing_oauth_metadata = existing_data.get("oauth_metadata")
        except (FileNotFoundError, json.JSONDecodeError):
            pass

        # Start with client_info data
        data = client_info.model_dump(mode="json")

        # Preserve oauth_metadata if it exists
        if existing_oauth_metadata is not None:
            data["oauth_metadata"] = existing_oauth_metadata

        path.write_text(json.dumps(data, indent=2))
        logger.info(f"Saved client info for {self.get_base_url(self.server_url)}")

    def clear(self) -> None:
        """Clear all cached data for this server."""
        file_types: list[Literal["client_info", "tokens"]] = [
            "client_info",
            "tokens",
        ]
        for file_type in file_types:
            path = self._get_file_path(file_type)
            path.unlink(missing_ok=True)
        logger.info(f"Cleared OAuth cache for {self.get_base_url(self.server_url)}")

    @classmethod
    def clear_all(cls, cache_dir: Path | None = None) -> None:
        """Clear all cached data for all servers."""
        cache_dir = cache_dir or default_cache_dir()
        if not cache_dir.exists():
            return

        file_types: list[Literal["client_info", "tokens"]] = [
            "client_info",
            "tokens",
        ]
        for file_type in file_types:
            for file in cache_dir.glob(f"*_{file_type}.json"):
                file.unlink(missing_ok=True)
        logger.info("Cleared all OAuth client cache data.")


class OAuth(OAuthClientProvider):
    """
    OAuth client provider for MCP servers with browser-based authentication.

    This class provides OAuth authentication for Runlayer clients by opening
    a browser for user authorization and running a local callback server.
    """

    def __init__(
        self,
        mcp_url: str,
        scopes: str | list[str] | None = None,
        client_name: str = "Runlayer Client",
        token_storage_cache_dir: Path | None = None,
        additional_client_metadata: dict[str, Any] | None = None,
        callback_port: int | None = None,
    ):
        """
        Initialize OAuth client provider for an MCP server.

        Args:
            mcp_url: Full URL to the MCP endpoint (e.g. "http://host/mcp/sse/")
            scopes: OAuth scopes to request. Can be a
            space-separated string or a list of strings.
            client_name: Name for this client during registration
            token_storage_cache_dir: Directory for FileTokenStorage
            additional_client_metadata: Extra fields for OAuthClientMetadata
            callback_port: Fixed port for OAuth callback (default: random available port)
        """
        parsed_url = urlparse(mcp_url)
        server_base_url = f"{parsed_url.scheme}://{parsed_url.netloc}"

        # Setup OAuth client
        self.redirect_port = callback_port or get_free_port()
        redirect_uri = f"http://localhost:{self.redirect_port}/callback"

        if isinstance(scopes, list):
            scopes = " ".join(scopes)

        client_metadata = OAuthClientMetadata(
            client_name=client_name,
            redirect_uris=[AnyHttpUrl(redirect_uri)],
            grant_types=["authorization_code", "refresh_token"],
            response_types=["code"],
            # token_endpoint_auth_method="client_secret_post",
            scope=scopes,
            **(additional_client_metadata or {}),
        )

        # Create server-specific token storage
        storage = FileTokenStorage(
            server_url=server_base_url, cache_dir=token_storage_cache_dir
        )

        # Store server_base_url for use in callback_handler
        self.server_base_url = server_base_url

        # Initialize parent class
        super().__init__(
            server_url=server_base_url,
            client_metadata=client_metadata,
            storage=storage,
            redirect_handler=self.redirect_handler,
            callback_handler=self.callback_handler,
        )

    async def _initialize(self) -> None:
        """
        Override parent's _initialize to properly calculate token expiry and restore OAuth metadata.

        The parent class loads tokens from storage but doesn't calculate
        the token_expiry_time, which causes is_token_valid() to always
        return True (since token_expiry_time is None). This prevents
        token refresh from ever being triggered.

        We use an absolute expiration timestamp stored separately to ensure
        the relative `expires_in` field that we receive when we save the tokens is respected across CLI restarts.

        Additionally, we restore oauth_metadata which is needed for token refresh but not cached by the parent class.
        """
        # Call parent's initialization to load tokens and client info
        await super()._initialize()

        logger.info(
            "OAuth context after initialization",
            has_tokens=bool(self.context.current_tokens),
            has_client_info=bool(self.context.client_info),
            token_expiry_time=self.context.token_expiry_time,
            has_oauth_metadata=bool(self.context.oauth_metadata),
        )

        if isinstance(self.context.storage, FileTokenStorage):
            # If we loaded tokens, restore the absolute expiry time
            if self.context.current_tokens:
                # Get expiry time if using FileTokenStorage
                expiry_time = self.context.storage.get_token_expiry_time()

                if expiry_time:
                    self.context.token_expiry_time = expiry_time
                    logger.info(
                        "Initialized tokens with stored expiry",
                        token_expiry_time=expiry_time,
                        seconds_until_expiry=expiry_time - time.time(),
                    )

            # Restore oauth_metadata if available - critical for token refresh
            oauth_metadata_dict = self.context.storage.get_oauth_metadata()
            if oauth_metadata_dict:
                try:
                    self.context.oauth_metadata = OAuthMetadata.model_validate(
                        oauth_metadata_dict
                    )
                    logger.info(
                        "Restored OAuth metadata from cache",
                        token_endpoint=str(self.context.oauth_metadata.token_endpoint)
                        if self.context.oauth_metadata.token_endpoint
                        else None,
                    )
                except ValidationError as e:
                    logger.warning(
                        f"Failed to validate cached OAuth metadata: {e}",
                    )

    async def _handle_token_response(self, response: httpx.Response) -> None:
        """
        Override parent's _handle_token_response to persist oauth_metadata after successful token acquisition.

        The parent class discovers oauth_metadata during the 401 handling in async_auth_flow,
        but doesn't persist it. We need to save it so that token refresh can use the correct
        token endpoint URL on subsequent CLI runs.
        """
        # Let parent handle the token response (validates and saves tokens)
        await super()._handle_token_response(response)

        # Now save oauth_metadata if we have it - this is critical for token refresh
        if self.context.oauth_metadata and isinstance(
            self.context.storage, FileTokenStorage
        ):
            self.context.storage.set_oauth_metadata(
                self.context.oauth_metadata.model_dump(mode="json")
            )
            logger.info(
                "Saved OAuth metadata to cache",
                token_endpoint=str(self.context.oauth_metadata.token_endpoint)
                if self.context.oauth_metadata.token_endpoint
                else None,
            )

    async def redirect_handler(self, authorization_url: str) -> None:
        """Open browser for authorization, with protection against multiple tabs."""
        logger.info(f"OAuth authorization URL: {authorization_url}")

        # Only open browser if we haven't opened one recently for this server
        if should_open_browser(self.server_base_url):
            webbrowser.open(authorization_url)
        else:
            logger.info(
                "Skipping browser open - a tab was recently opened for this server. "
                "Please use the existing browser tab to complete authorization."
            )

    async def callback_handler(self) -> tuple[str, str | None]:
        """Handle OAuth callback and return (auth_code, state)."""
        # Create a future to capture the OAuth response
        response_future = asyncio.get_running_loop().create_future()

        # Create server with the future
        server = create_oauth_callback_server(
            port=self.redirect_port,
            server_url=self.server_base_url,
            response_future=response_future,
        )

        # Run server until response is received with timeout logic
        async with anyio.create_task_group() as tg:
            tg.start_soon(server.serve)
            logger.info(
                f"OAuth callback server started at http://localhost:{self.redirect_port}"
            )

            TIMEOUT = 300.0  # 5 minute timeout
            try:
                with anyio.fail_after(TIMEOUT):
                    auth_code, state = await response_future
                    return auth_code, state
            except TimeoutError:
                raise TimeoutError(f"OAuth callback timed out after {TIMEOUT} seconds")
            finally:
                server.should_exit = True
                await asyncio.sleep(0.1)  # Allow server to shutdown gracefully
                tg.cancel_scope.cancel()
