"""Mouse anatomy"""

import requests
from pydantic import BaseModel, ConfigDict

from aind_data_schema_models.registries import Registry

"""Mouse anatomy"""


class MouseAnatomyModel(BaseModel):
    """Base model for mouse anatomy"""

    model_config = ConfigDict(frozen=True)
    name: str
    registry: Registry.ONE_OF
    registry_identifier: str


def search_emapa_exact_match(class_name):
    """Pull the exact name match from the EMAPA ontology

    Parameters
    ----------
    class_name : str
        Name of class

    Returns
    -------
    list

    Raises
    ------
    Exception
        OLS query failed on any status code other than 200
    """
    base_url = "https://www.ebi.ac.uk/ols/api/search"
    params = {
        "q": class_name,
        "ontology": "emapa",  # Specify the ontology
        "type": "class",  # Search for classes
    }
    response = requests.get(base_url, params=params)
    if response.status_code == 200:
        results = response.json()
        # Extract terms with exact label match
        exact_matches = [
            {
                "iri": entry["iri"],
                "label": entry["label"],
            }
            for entry in results.get("response", {}).get("docs", [])
            if entry["label"].lower() == class_name.lower()
        ]
        return exact_matches
    else:
        raise Exception(f"OLS query failed: {response.status_code}, {response.text}")


def get_emapa_id(class_name):
    """Get the EMAPA ID for a given class name

    Parameters
    ----------
    class_name : str

    Returns
    -------
    int
    """
    results = search_emapa_exact_match(class_name)

    if results:
        return results[0]["iri"].split("_")[1]
    else:
        return None


class MouseAnatomyMeta(type):
    """Meta class for MouseAnatomy groups"""

    def __getattribute__(cls, name):
        """Custom get attribute function, validates anatomy names against external EMAPA registry

        Parameters
        ----------
        name : str
            Attribute name

        Raises
        ------
        ValueError
            Name not found in the EMAPA registry
        """

        # bypass
        if name.startswith("__"):
            return object.__getattribute__(cls, name)

        class_dict = object.__getattribute__(cls, "__dict__")

        if name in class_dict:
            original_name = super().__getattribute__(name)

            emapa_id = get_emapa_id(original_name)

            if emapa_id:
                return MouseAnatomyModel(
                    name=original_name,
                    registry=Registry.EMAPA,
                    registry_identifier=emapa_id,
                )
            else:
                raise ValueError(f"Could not find EMAPA ID for {original_name}")

        # second bypass for defined values
        return super().__getattribute__(name)


class MouseAnatomy(metaclass=MouseAnatomyMeta):
    """MouseAnatomicalStructure"""

    ANATOMICAL_STRUCTURE = "Anatomical structure"
    FIRST_POLAR_BODY = "first polar body"
    _1_CELL_STAGE_EMBRYO = "1-cell stage embryo"
    SECOND_POLAR_BODY = "second polar body"
    ZONA_PELLUCIDA = "zona pellucida"
    _2_CELL_STAGE_EMBRYO = "2-cell stage embryo"
    _4_8_CELL_STAGE_EMBRYO = "4-8 cell stage embryo"
    EMBRYO = "embryo"
    COMPACTED_MORULA = "compacted morula"
    INNER_CELL_MASS = "inner cell mass"
    EXTRAEMBRYONIC_COMPONENT = "extraembryonic component"
    EXTRAEMBRYONIC_CAVITY = "extraembryonic cavity"
    BLASTOCOELIC_CAVITY = "blastocoelic cavity"
    TROPHECTODERM = "trophectoderm"
    MURAL_TROPHECTODERM = "mural trophectoderm"
    POLAR_TROPHECTODERM = "polar trophectoderm"
    EPIBLAST = "epiblast"
    PRIMITIVE_ENDODERM = "primitive endoderm"
    EXTRAEMBRYONIC_ENDODERM = "extraembryonic endoderm"
    PARIETAL_ENDODERM = "parietal endoderm"
    EXTRAEMBRYONIC_COMPONENT_CAVITY = "extraembryonic component cavity"
    YOLK_SAC_CAVITY = "yolk sac cavity"
    INTERMEDIATE_ENDODERM = "intermediate endoderm"
    REICHERT_S_MEMBRANE = "Reichert's membrane"
    EXTRAEMBRYONIC_VISCERAL_ENDODERM = "extraembryonic visceral endoderm"
    ECTOPLACENTAL_CONE = "ectoplacental cone"
    BODY_CAVITY_OR_LINING = "body cavity or lining"
    PROAMNIOTIC_CAVITY_EMBRYONIC_PART = "proamniotic cavity embryonic part"
    EMBRYO_ENDODERM = "embryo endoderm"
    PROAMNIOTIC_CAVITY_EXTRAEMBRYONIC_PART = "proamniotic cavity extraembryonic part"
    EXTRAEMBRYONIC_ECTODERM = "extraembryonic ectoderm"
    PRIMARY_TROPHOBLAST_GIANT_CELL = "primary trophoblast giant cell"
    CYTOTROPHOBLAST = "cytotrophoblast"
    SYNCYTIOTROPHOBLAST = "syncytiotrophoblast"
    EMBRYO_ECTODERM = "embryo ectoderm"
    DEFINITIVE_ENDODERM = "definitive endoderm"
    EMBRYO_MESODERM = "embryo mesoderm"
    PRIMITIVE_STREAK = "primitive streak"
    NEURAL_ECTODERM = "neural ectoderm"
    NON_NEURAL_ECTODERM = "non-neural ectoderm"
    NODE = "node"
    AMNIOTIC_FOLD = "amniotic fold"
    AMNIOTIC_FOLD_ECTODERM = "amniotic fold ectoderm"
    AMNIOTIC_FOLD_MESODERM = "amniotic fold mesoderm"
    AMNIOTIC_CAVITY = "amniotic cavity"
    ECTOPLACENTAL_CAVITY = "ectoplacental cavity"
    EXOCOELOMIC_CAVITY = "exocoelomic cavity"
    CAVITY_WITHIN_AMNIOTIC_FOLD = "cavity within amniotic fold"
    YOLK_SAC = "yolk sac"
    YOLK_SAC_ENDODERM = "yolk sac endoderm"
    YOLK_SAC_MESODERM = "yolk sac mesoderm"
    INTRAEMBRYONIC_COELOM = "intraembryonic coelom"
    FUTURE_BRAIN = "future brain"
    FUTURE_BRAIN_NEURAL_FOLD = "future brain neural fold"
    FUTURE_BRAIN_NEURAL_CREST = "future brain neural crest"
    FUTURE_SPINAL_CORD = "future spinal cord"
    FUTURE_SPINAL_CORD_NEURAL_PLATE = "future spinal cord neural plate"
    SURFACE_ECTODERM = "surface ectoderm"
    MESENCHYME = "mesenchyme"
    HEAD_MESENCHYME = "head mesenchyme"
    MESENCHYME_DERIVED_FROM_HEAD_MESODERM = "mesenchyme derived from head mesoderm"
    NOTOCHORDAL_PLATE = "notochordal plate"
    NOTOCHORDAL_PROCESS = "notochordal process"
    ORGAN_SYSTEM = "organ system"
    CARDIOVASCULAR_SYSTEM = "cardiovascular system"
    HEART = "heart"
    CARDIOGENIC_PLATE = "cardiogenic plate"
    ALLANTOIS = "allantois"
    ALLANTOIS_MESODERM = "allantois mesoderm"
    AMNION = "amnion"
    AMNION_ECTODERM = "amnion ectoderm"
    AMNION_MESODERM = "amnion mesoderm"
    CHORION = "chorion"
    CHORION_ECTODERM = "chorion ectoderm"
    CHORION_MESODERM = "chorion mesoderm"
    BLOOD_ISLAND = "blood island"
    PRIMORDIAL_GERM_CELL_OF_YOLK_SAC = "primordial germ cell of yolk sac"
    BRANCHIAL_ARCH = "branchial arch"
    _1ST_BRANCHIAL_ARCH = "1st branchial arch"
    _1ST_BRANCHIAL_GROOVE = "1st branchial groove"
    _1ST_BRANCHIAL_GROOVE_ECTODERM = "1st branchial groove ectoderm"
    _1ST_BRANCHIAL_MEMBRANE = "1st branchial membrane"
    _1ST_BRANCHIAL_MEMBRANE_ECTODERM = "1st branchial membrane ectoderm"
    _1ST_BRANCHIAL_MEMBRANE_ENDODERM = "1st branchial membrane endoderm"
    _1ST_BRANCHIAL_POUCH = "1st branchial pouch"
    _1ST_BRANCHIAL_POUCH_ENDODERM = "1st branchial pouch endoderm"
    _1ST_BRANCHIAL_ARCH_ECTODERM = "1st branchial arch ectoderm"
    _1ST_BRANCHIAL_ARCH_ENDODERM = "1st branchial arch endoderm"
    _1ST_BRANCHIAL_ARCH_MESENCHYME = "1st branchial arch mesenchyme"
    _1ST_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "1st branchial arch mesenchyme derived from neural crest"
    _1ST_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_HEAD_MESODERM = (
        "1st branchial arch mesenchyme derived from head mesoderm"
    )
    PERICARDIAL_COMPONENT = "pericardial component"
    PERICARDIAL_CAVITY = "pericardial cavity"
    PERICARDIUM = "pericardium"
    PERICARDIO_PERITONEAL_COMPONENT = "pericardio-peritoneal component"
    PERICARDIO_PERITONEAL_COMPONENT_CAVITY = "pericardio-peritoneal component cavity"
    PERICARDIO_PERITONEAL_COMPONENT_MESOTHELIUM = "pericardio-peritoneal component mesothelium"
    PERITONEAL_COMPONENT = "peritoneal component"
    PERITONEAL_CAVITY = "peritoneal cavity"
    PERITONEUM = "peritoneum"
    FUTURE_MIDBRAIN = "future midbrain"
    FUTURE_MIDBRAIN_FLOOR_PLATE = "future midbrain floor plate"
    FUTURE_MIDBRAIN_NEURAL_FOLD = "future midbrain neural fold"
    FUTURE_MIDBRAIN_NEURAL_CREST = "future midbrain neural crest"
    FUTURE_PROSENCEPHALON = "future prosencephalon"
    FUTURE_PROSENCEPHALON_FLOOR_PLATE = "future prosencephalon floor plate"
    FUTURE_PROSENCEPHALON_NEURAL_FOLD = "future prosencephalon neural fold"
    FUTURE_PROSENCEPHALON_NEURAL_CREST = "future prosencephalon neural crest"
    FUTURE_HINDBRAIN = "future hindbrain"
    ANTERIOR_PRO_RHOMBOMERE = "anterior pro-rhombomere"
    ANTERIOR_PRO_RHOMBOMERE_FLOOR_PLATE = "anterior pro-rhombomere floor plate"
    ANTERIOR_PRO_RHOMBOMERE_NEURAL_FOLD = "anterior pro-rhombomere neural fold"
    ANTERIOR_PRO_RHOMBOMERE_NEURAL_CREST = "anterior pro-rhombomere neural crest"
    FUTURE_HINDBRAIN_NEURAL_ECTODERM = "future hindbrain neural ectoderm"
    FUTURE_HINDBRAIN_FLOOR_PLATE = "future hindbrain floor plate"
    FUTURE_HINDBRAIN_NEURAL_FOLD = "future hindbrain neural fold"
    FUTURE_HINDBRAIN_NEURAL_CREST = "future hindbrain neural crest"
    POSTERIOR_PRO_RHOMBOMERE = "posterior pro-rhombomere"
    POSTERIOR_PRO_RHOMBOMERE_FLOOR_PLATE = "posterior pro-rhombomere floor plate"
    POSTERIOR_PRO_RHOMBOMERE_NEURAL_FOLD = "posterior pro-rhombomere neural fold"
    POSTERIOR_PRO_RHOMBOMERE_NEURAL_CREST = "posterior pro-rhombomere neural crest"
    PRE_OTIC_SULCUS = "pre-otic sulcus"
    FUTURE_SPINAL_CORD_NEURAL_FOLD = "future spinal cord neural fold"
    FUTURE_SPINAL_CORD_NEURAL_CREST = "future spinal cord neural crest"
    NEURAL_TUBE = "neural tube"
    NEURAL_TUBE_FLOOR_PLATE = "neural tube floor plate"
    NEURAL_TUBE_LATERAL_WALL = "neural tube lateral wall"
    NEURAL_LUMEN = "neural lumen"
    NEURAL_TUBE_ROOF_PLATE = "neural tube roof plate"
    HEAD_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "head mesenchyme derived from neural crest"
    TRIGEMINAL_NEURAL_CREST = "trigeminal neural crest"
    HEAD_PARAXIAL_MESENCHYME = "head paraxial mesenchyme"
    HEAD_SOMITE = "head somite"
    SOMITE_01 = "somite 01"
    SOMITE_02 = "somite 02"
    SOMITE_03 = "somite 03"
    SOMITE_04 = "somite 04"
    TRUNK_MESENCHYME = "trunk mesenchyme"
    INTERMEDIATE_MESENCHYME = "intermediate mesenchyme"
    LATERAL_PLATE_MESODERM = "lateral plate mesoderm"
    SOMATOPLEURE = "somatopleure"
    SPLANCHNOPLEURE = "splanchnopleure"
    TRUNK_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "trunk mesenchyme derived from neural crest"
    TRUNK_PARAXIAL_MESENCHYME = "trunk paraxial mesenchyme"
    TRUNK_SOMITE = "trunk somite"
    SOMITE_05 = "somite 05"
    SOMITE_06 = "somite 06"
    SOMITE_07 = "somite 07"
    SOMITE_08 = "somite 08"
    TRUNK_UNSEGMENTED_MESENCHYME = "trunk unsegmented mesenchyme"
    PRIMORDIAL_GERM_CELL_OF_TRUNK_MESENCHYME = "primordial germ cell of trunk mesenchyme"
    NOTOCHORD = "notochord"
    SENSORY_ORGAN_SYSTEM = "sensory organ system"
    EAR = "ear"
    INNER_EAR = "inner ear"
    OTIC_PLACODE = "otic placode"
    OTIC_PLACODE_MESENCHYME = "otic placode mesenchyme"
    OTIC_PLACODE_EPITHELIUM = "otic placode epithelium"
    EYE = "eye"
    OPTIC_SULCUS = "optic sulcus"
    OPTIC_SULCUS_NEURAL_ECTODERM = "optic sulcus neural ectoderm"
    ARTERIAL_SYSTEM = "arterial system"
    DORSAL_AORTA = "dorsal aorta"
    LEFT_DORSAL_AORTA = "left dorsal aorta"
    RIGHT_DORSAL_AORTA = "right dorsal aorta"
    VITELLINE_ARTERY = "vitelline artery"
    PRIMITIVE_HEART_TUBE_CARDIAC_JELLY = "primitive heart tube cardiac jelly"
    PRIMITIVE_HEART_TUBE_ENDOCARDIAL_TUBE = "primitive heart tube endocardial tube"
    PRIMITIVE_HEART_TUBE_CARDIAC_MUSCLE = "primitive heart tube cardiac muscle"
    HEART_MESENTERY = "heart mesentery"
    DORSAL_MESOCARDIUM = "dorsal mesocardium"
    TRANSVERSE_PERICARDIAL_SINUS = "transverse pericardial sinus"
    PRIMITIVE_HEART_TUBE = "primitive heart tube"
    SINUS_VENOSUS = "sinus venosus"
    SINUS_VENOSUS_LEFT_HORN = "sinus venosus left horn"
    SINUS_VENOSUS_RIGHT_HORN = "sinus venosus right horn"
    VENOUS_SYSTEM = "venous system"
    ANTERIOR_CARDINAL_VEIN = "anterior cardinal vein"
    PRIMARY_HEAD_VEIN = "primary head vein"
    UMBILICAL_VEIN_EMBRYONIC_PART = "umbilical vein embryonic part"
    VITELLINE_VEIN_EMBRYONIC_PART = "vitelline vein embryonic part"
    VISCERAL_ORGAN_SYSTEM = "visceral organ system"
    ALIMENTARY_SYSTEM = "alimentary system"
    GUT = "gut"
    FOREGUT_DIVERTICULUM = "foregut diverticulum"
    FOREGUT_DIVERTICULUM_MESENCHYME = "foregut diverticulum mesenchyme"
    FOREGUT_DIVERTICULUM_ENDODERM = "foregut diverticulum endoderm"
    HINDGUT_DIVERTICULUM = "hindgut diverticulum"
    HINDGUT_DIVERTICULUM_MESENCHYME = "hindgut diverticulum mesenchyme"
    HINDGUT_DIVERTICULUM_ENDODERM = "hindgut diverticulum endoderm"
    PRIMORDIAL_GERM_CELL_OF_HINDGUT = "primordial germ cell of hindgut"
    MIDGUT = "midgut"
    MIDGUT_MESENCHYME = "midgut mesenchyme"
    MIDGUT_ENDODERM = "midgut endoderm"
    MOUTH_FOREGUT_JUNCTION = "mouth-foregut junction"
    BUCCOPHARYNGEAL_MEMBRANE = "buccopharyngeal membrane"
    BUCCOPHARYNGEAL_MEMBRANE_ECTODERM = "buccopharyngeal membrane ectoderm"
    BUCCOPHARYNGEAL_MEMBRANE_ENDODERM = "buccopharyngeal membrane endoderm"
    ORAL_REGION = "oral region"
    STOMATODEUM = "stomatodeum"
    ALLANTOIS_MESENCHYME = "allantois mesenchyme"
    AMNION_MESENCHYME = "amnion mesenchyme"
    CHORION_MESENCHYME = "chorion mesenchyme"
    YOLK_SAC_MESENCHYME = "yolk sac mesenchyme"
    _2ND_BRANCHIAL_ARCH = "2nd branchial arch"
    _2ND_BRANCHIAL_MEMBRANE = "2nd branchial membrane"
    _2ND_BRANCHIAL_MEMBRANE_ENDODERM = "2nd branchial membrane endoderm"
    _2ND_BRANCHIAL_POUCH = "2nd branchial pouch"
    _2ND_BRANCHIAL_POUCH_ENDODERM = "2nd branchial pouch endoderm"
    _2ND_BRANCHIAL_ARCH_ECTODERM = "2nd branchial arch ectoderm"
    _2ND_BRANCHIAL_ARCH_ENDODERM = "2nd branchial arch endoderm"
    _2ND_BRANCHIAL_ARCH_MESENCHYME = "2nd branchial arch mesenchyme"
    _2ND_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_HEAD_MESODERM = (
        "2nd branchial arch mesenchyme derived from head mesoderm"
    )
    _2ND_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "2nd branchial arch mesenchyme derived from neural crest"
    PERICARDIO_PERITONEAL_CANAL = "pericardio-peritoneal canal"
    PERICARDIO_PERITONEAL_CANAL_CAVITY = "pericardio-peritoneal canal cavity"
    PERICARDIO_PERITONEAL_CANAL_MESOTHELIUM = "pericardio-peritoneal canal mesothelium"
    FUTURE_MIDBRAIN_ROOF_PLATE = "future midbrain roof plate"
    FUTURE_HINDBRAIN_POSTERIOR_TO_RHOMBOMERE_5 = "future hindbrain posterior to rhombomere 5"
    FUTURE_HINDBRAIN_POSTERIOR_TO_RHOMBOMERE_5_FLOOR_PLATE = "future hindbrain posterior to rhombomere 5 floor plate"
    FUTURE_HINDBRAIN_POSTERIOR_TO_RHOMBOMERE_5_NEURAL_FOLD = "future hindbrain posterior to rhombomere 5 neural fold"
    FUTURE_HINDBRAIN_POSTERIOR_TO_RHOMBOMERE_5_NEURAL_CREST = "future hindbrain posterior to rhombomere 5 neural crest"
    RHOMBOMERE_1 = "rhombomere 1"
    RHOMBOMERE_1_FLOOR_PLATE = "rhombomere 1 floor plate"
    RHOMBOMERE_1_NEURAL_FOLD = "rhombomere 1 neural fold"
    RHOMBOMERE_2 = "rhombomere 2"
    RHOMBOMERE_2_FLOOR_PLATE = "rhombomere 2 floor plate"
    RHOMBOMERE_2_NEURAL_FOLD = "rhombomere 2 neural fold"
    RHOMBOMERE_2_NEURAL_CREST = "rhombomere 2 neural crest"
    RHOMBOMERE_3 = "rhombomere 3"
    RHOMBOMERE_3_FLOOR_PLATE = "rhombomere 3 floor plate"
    RHOMBOMERE_3_NEURAL_FOLD = "rhombomere 3 neural fold"
    RHOMBOMERE_3_NEURAL_CREST = "rhombomere 3 neural crest"
    RHOMBOMERE_4 = "rhombomere 4"
    RHOMBOMERE_4_FLOOR_PLATE = "rhombomere 4 floor plate"
    RHOMBOMERE_4_NEURAL_FOLD = "rhombomere 4 neural fold"
    RHOMBOMERE_4_NEURAL_CREST = "rhombomere 4 neural crest"
    RHOMBOMERE_5 = "rhombomere 5"
    RHOMBOMERE_5_FLOOR_PLATE = "rhombomere 5 floor plate"
    RHOMBOMERE_5_NEURAL_FOLD = "rhombomere 5 neural fold"
    RHOMBOMERE_5_NEURAL_CREST = "rhombomere 5 neural crest"
    RHOMBOMERE_1_NEURAL_CREST = "rhombomere 1 neural crest"
    NEURAL_LUMINAL_OCCLUSION = "neural luminal occlusion"
    FACIAL_NEURAL_CREST = "facial neural crest"
    FACIO_ACOUSTIC_NEURAL_CREST = "facio-acoustic neural crest"
    SOMITE_09 = "somite 09"
    SOMITE_10 = "somite 10"
    SOMITE_11 = "somite 11"
    SOMITE_12 = "somite 12"
    SOMITE_13 = "somite 13"
    SEPTUM_TRANSVERSUM = "septum transversum"
    SEPTUM_TRANSVERSUM_HEPATIC_COMPONENT = "septum transversum hepatic component"
    SEPTUM_TRANSVERSUM_NON_HEPATIC_COMPONENT = "septum transversum non-hepatic component"
    OPTIC_EMINENCE = "optic eminence"
    OPTIC_EMINENCE_MESENCHYME = "optic eminence mesenchyme"
    OPTIC_PIT = "optic pit"
    OPTIC_PIT_NEURAL_ECTODERM = "optic pit neural ectoderm"
    COMMON_DORSAL_AORTA = "common dorsal aorta"
    INTERNAL_CAROTID_ARTERY = "internal carotid artery"
    LEFT_INTERNAL_CAROTID_ARTERY = "left internal carotid artery"
    RIGHT_INTERNAL_CAROTID_ARTERY = "right internal carotid artery"
    UMBILICAL_ARTERY_EMBRYONIC_PART = "umbilical artery embryonic part"
    BLOOD = "blood"
    BULBUS_CORDIS = "bulbus cordis"
    BULBUS_CORDIS_CAUDAL_HALF = "bulbus cordis caudal half"
    BULBUS_CORDIS_CAUDAL_HALF_CARDIAC_JELLY = "bulbus cordis caudal half cardiac jelly"
    BULBUS_CORDIS_CAUDAL_HALF_ENDOCARDIAL_TUBE = "bulbus cordis caudal half endocardial tube"
    BULBUS_CORDIS_CAUDAL_HALF_CARDIAC_MUSCLE = "bulbus cordis caudal half cardiac muscle"
    BULBUS_CORDIS_ROSTRAL_HALF = "bulbus cordis rostral half"
    BULBUS_CORDIS_ROSTRAL_HALF_CARDIAC_JELLY = "bulbus cordis rostral half cardiac jelly"
    BULBUS_CORDIS_ROSTRAL_HALF_ENDOCARDIAL_TUBE = "bulbus cordis rostral half endocardial tube"
    BULBUS_CORDIS_ROSTRAL_HALF_CARDIAC_MUSCLE = "bulbus cordis rostral half cardiac muscle"
    COMMON_ATRIAL_CHAMBER = "common atrial chamber"
    COMMON_ATRIAL_CHAMBER_CARDIAC_JELLY = "common atrial chamber cardiac jelly"
    COMMON_ATRIAL_CHAMBER_ENDOCARDIAL_TUBE = "common atrial chamber endocardial tube"
    COMMON_ATRIAL_CHAMBER_CARDIAC_MUSCLE = "common atrial chamber cardiac muscle"
    OUTFLOW_TRACT = "outflow tract"
    OUTFLOW_TRACT_CARDIAC_JELLY = "outflow tract cardiac jelly"
    OUTFLOW_TRACT_ENDOCARDIAL_TUBE = "outflow tract endocardial tube"
    OUTFLOW_TRACT_MESENCHYME = "outflow tract mesenchyme"
    PRIMITIVE_VENTRICLE = "primitive ventricle"
    PRIMITIVE_VENTRICLE_CARDIAC_JELLY = "primitive ventricle cardiac jelly"
    PRIMITIVE_VENTRICLE_ENDOCARDIAL_TUBE = "primitive ventricle endocardial tube"
    PRIMITIVE_VENTRICLE_CARDIAC_MUSCLE = "primitive ventricle cardiac muscle"
    CARDINAL_VEIN = "cardinal vein"
    COMMON_CARDINAL_VEIN = "common cardinal vein"
    POSTERIOR_CARDINAL_VEIN = "posterior cardinal vein"
    LEFT_UMBILICAL_VEIN_EMBRYONIC_PART = "left umbilical vein embryonic part"
    RIGHT_UMBILICAL_VEIN_EMBRYONIC_PART = "right umbilical vein embryonic part"
    THYROID_PRIMORDIUM = "thyroid primordium"
    THYROID_PRIMORDIUM_MESENCHYME = "thyroid primordium mesenchyme"
    FOREGUT_MIDGUT_JUNCTION = "foregut-midgut junction"
    FOREGUT_MIDGUT_JUNCTION_MESENCHYME = "foregut-midgut junction mesenchyme"
    FOREGUT_MIDGUT_JUNCTION_MESENTERY = "foregut-midgut junction mesentery"
    FOREGUT_MIDGUT_JUNCTION_DORSAL_MESENTERY = "foregut-midgut junction dorsal mesentery"
    GENITOURINARY_SYSTEM = "genitourinary system"
    NEPHROGENIC_CORD = "nephrogenic cord"
    PRESUMPTIVE_NEPHRIC_DUCT = "presumptive nephric duct"
    EXTRAEMBRYONIC_VASCULAR_SYSTEM = "extraembryonic vascular system"
    EXTRAEMBRYONIC_ARTERIAL_SYSTEM = "extraembryonic arterial system"
    UMBILICAL_ARTERY_EXTRAEMBRYONIC_PART = "umbilical artery extraembryonic part"
    COMMON_UMBILICAL_ARTERY = "common umbilical artery"
    EXTRAEMBRYONIC_VENOUS_SYSTEM = "extraembryonic venous system"
    UMBILICAL_VEIN_EXTRAEMBRYONIC_PART = "umbilical vein extraembryonic part"
    COMMON_UMBILICAL_VEIN = "common umbilical vein"
    LEFT_UMBILICAL_VEIN_EXTRAEMBRYONIC_PART = "left umbilical vein extraembryonic part"
    RIGHT_UMBILICAL_VEIN_EXTRAEMBRYONIC_PART = "right umbilical vein extraembryonic part"
    VITELLINE_VEIN_EXTRAEMBRYONIC_PART = "vitelline vein extraembryonic part"
    LEFT_VITELLINE_VEIN_EXTRAEMBRYONIC_PART = "left vitelline vein extraembryonic part"
    RIGHT_VITELLINE_VEIN_EXTRAEMBRYONIC_PART = "right vitelline vein extraembryonic part"
    _1ST_BRANCHIAL_ARCH_MANDIBULAR_COMPONENT = "1st branchial arch mandibular component"
    _1ST_BRANCHIAL_ARCH_MANDIBULAR_COMPONENT_ECTODERM = "1st branchial arch mandibular component ectoderm"
    _1ST_BRANCHIAL_ARCH_MANDIBULAR_COMPONENT_ENDODERM = "1st branchial arch mandibular component endoderm"
    _1ST_BRANCHIAL_ARCH_MANDIBULAR_COMPONENT_MESENCHYME = "1st branchial arch mandibular component mesenchyme"
    _1ST_BRANCHIAL_ARCH_MANDIBULAR_COMPONENT_MESENCHYME_DERIVED_FROM_HEAD_MESODERM = (
        "1st branchial arch mandibular component mesenchyme derived from head mesoderm"
    )
    _1ST_BRANCHIAL_ARCH_MANDIBULAR_COMPONENT_MESENCHYME_DERIVED_FROM_NEURAL_CREST = (
        "1st branchial arch mandibular component mesenchyme derived from neural crest"
    )
    _1ST_BRANCHIAL_ARCH_MAXILLARY_COMPONENT = "1st branchial arch maxillary component"
    _1ST_BRANCHIAL_ARCH_MAXILLARY_COMPONENT_ECTODERM = "1st branchial arch maxillary component ectoderm"
    _1ST_BRANCHIAL_ARCH_MAXILLARY_COMPONENT_ENDODERM = "1st branchial arch maxillary component endoderm"
    _1ST_BRANCHIAL_ARCH_MAXILLARY_COMPONENT_MESENCHYME = "1st branchial arch maxillary component mesenchyme"
    _1ST_BRANCHIAL_ARCH_MAXILLARY_COMPONENT_MESENCHYME_DERIVED_FROM_HEAD_MESODERM = (
        "1st branchial arch maxillary component mesenchyme derived from head mesoderm"
    )
    _1ST_BRANCHIAL_ARCH_MAXILLARY_COMPONENT_MESENCHYME_DERIVED_FROM_NEURAL_CREST = (
        "1st branchial arch maxillary component mesenchyme derived from neural crest"
    )
    _1ST_BRANCHIAL_ARCH_MAXILLARY_MANDIBULAR_GROOVE = "1st branchial arch maxillary-mandibular groove"
    _1ST_BRANCHIAL_ARCH_MAXILLARY_MANDIBULAR_GROOVE_ECTODERM = "1st branchial arch maxillary-mandibular groove ectoderm"
    _2ND_BRANCHIAL_GROOVE = "2nd branchial groove"
    _2ND_BRANCHIAL_GROOVE_ECTODERM = "2nd branchial groove ectoderm"
    _2ND_BRANCHIAL_MEMBRANE_ECTODERM = "2nd branchial membrane ectoderm"
    _3RD_BRANCHIAL_ARCH = "3rd branchial arch"
    _3RD_BRANCHIAL_ARCH_ECTODERM = "3rd branchial arch ectoderm"
    _3RD_BRANCHIAL_ARCH_ENDODERM = "3rd branchial arch endoderm"
    _3RD_BRANCHIAL_ARCH_MESENCHYME = "3rd branchial arch mesenchyme"
    _3RD_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_HEAD_MESODERM = (
        "3rd branchial arch mesenchyme derived from head mesoderm"
    )
    _3RD_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "3rd branchial arch mesenchyme derived from neural crest"
    LIMB = "limb"
    FORELIMB_BUD = "forelimb bud"
    FORELIMB_BUD_ECTODERM = "forelimb bud ectoderm"
    FORELIMB_BUD_MESENCHYME = "forelimb bud mesenchyme"
    MESENCHYME_DERIVED_FROM_LATERAL_PLATE = "mesenchyme derived from lateral plate"
    MESENCHYME_DERIVED_FROM_SOMATOPLEURE = "mesenchyme derived from somatopleure"
    MESENCHYME_DERIVED_FROM_SPLANCHNOPLEURE = "mesenchyme derived from splanchnopleure"
    SOMITE_05_DERMOMYOTOME = "somite 05 dermomyotome"
    SOMITE_05_MYOCOELE = "somite 05 myocoele"
    SOMITE_05_SCLEROTOME = "somite 05 sclerotome"
    SOMITE_06_DERMOMYOTOME = "somite 06 dermomyotome"
    SOMITE_06_MYOCOELE = "somite 06 myocoele"
    SOMITE_06_SCLEROTOME = "somite 06 sclerotome"
    SOMITE_07_DERMOMYOTOME = "somite 07 dermomyotome"
    SOMITE_07_MYOCOELE = "somite 07 myocoele"
    SOMITE_07_SCLEROTOME = "somite 07 sclerotome"
    SOMITE_08_DERMOMYOTOME = "somite 08 dermomyotome"
    SOMITE_08_MYOCOELE = "somite 08 myocoele"
    SOMITE_08_SCLEROTOME = "somite 08 sclerotome"
    SOMITE_09_DERMOMYOTOME = "somite 09 dermomyotome"
    SOMITE_09_MYOCOELE = "somite 09 myocoele"
    SOMITE_09_SCLEROTOME = "somite 09 sclerotome"
    SOMITE_10_DERMOMYOTOME = "somite 10 dermomyotome"
    SOMITE_10_MYOCOELE = "somite 10 myocoele"
    SOMITE_10_SCLEROTOME = "somite 10 sclerotome"
    SOMITE_11_DERMOMYOTOME = "somite 11 dermomyotome"
    SOMITE_11_MYOCOELE = "somite 11 myocoele"
    SOMITE_11_SCLEROTOME = "somite 11 sclerotome"
    SOMITE_12_DERMOMYOTOME = "somite 12 dermomyotome"
    SOMITE_12_MYOCOELE = "somite 12 myocoele"
    SOMITE_12_SCLEROTOME = "somite 12 sclerotome"
    SOMITE_13_DERMOMYOTOME = "somite 13 dermomyotome"
    SOMITE_13_MYOCOELE = "somite 13 myocoele"
    SOMITE_13_SCLEROTOME = "somite 13 sclerotome"
    SOMITE_14 = "somite 14"
    SOMITE_14_DERMOMYOTOME = "somite 14 dermomyotome"
    SOMITE_14_MYOCOELE = "somite 14 myocoele"
    SOMITE_14_SCLEROTOME = "somite 14 sclerotome"
    SOMITE_15 = "somite 15"
    SOMITE_15_DERMOMYOTOME = "somite 15 dermomyotome"
    SOMITE_15_MYOCOELE = "somite 15 myocoele"
    SOMITE_15_SCLEROTOME = "somite 15 sclerotome"
    SOMITE_16 = "somite 16"
    SOMITE_16_DERMOMYOTOME = "somite 16 dermomyotome"
    SOMITE_16_MYOCOELE = "somite 16 myocoele"
    SOMITE_16_SCLEROTOME = "somite 16 sclerotome"
    SOMITE_17 = "somite 17"
    SOMITE_17_DERMOMYOTOME = "somite 17 dermomyotome"
    SOMITE_17_MYOCOELE = "somite 17 myocoele"
    SOMITE_17_SCLEROTOME = "somite 17 sclerotome"
    SOMITE_18 = "somite 18"
    SOMITE_18_DERMOMYOTOME = "somite 18 dermomyotome"
    SOMITE_18_MYOCOELE = "somite 18 myocoele"
    SOMITE_18_SCLEROTOME = "somite 18 sclerotome"
    SOMITE_19 = "somite 19"
    SOMITE_19_DERMOMYOTOME = "somite 19 dermomyotome"
    SOMITE_19_MYOCOELE = "somite 19 myocoele"
    SOMITE_19_SCLEROTOME = "somite 19 sclerotome"
    SOMITE_20 = "somite 20"
    SOMITE_20_DERMOMYOTOME = "somite 20 dermomyotome"
    SOMITE_20_MYOCOELE = "somite 20 myocoele"
    SOMITE_20_SCLEROTOME = "somite 20 sclerotome"
    NERVOUS_SYSTEM = "nervous system"
    CENTRAL_NERVOUS_SYSTEM = "central nervous system"
    FUTURE_MIDBRAIN_LATERAL_WALL = "future midbrain lateral wall"
    RHOMBOMERE_1_LATERAL_WALL = "rhombomere 1 lateral wall"
    RHOMBOMERE_1_ROOF_PLATE = "rhombomere 1 roof plate"
    RHOMBOMERE_2_LATERAL_WALL = "rhombomere 2 lateral wall"
    RHOMBOMERE_2_ROOF_PLATE = "rhombomere 2 roof plate"
    RHOMBOMERE_3_LATERAL_WALL = "rhombomere 3 lateral wall"
    RHOMBOMERE_3_ROOF_PLATE = "rhombomere 3 roof plate"
    RHOMBOMERE_4_LATERAL_WALL = "rhombomere 4 lateral wall"
    RHOMBOMERE_4_ROOF_PLATE = "rhombomere 4 roof plate"
    RHOMBOMERE_5_LATERAL_WALL = "rhombomere 5 lateral wall"
    RHOMBOMERE_5_ROOF_PLATE = "rhombomere 5 roof plate"
    RHOMBOMERE_6 = "rhombomere 6"
    RHOMBOMERE_6_FLOOR_PLATE = "rhombomere 6 floor plate"
    RHOMBOMERE_6_LATERAL_WALL = "rhombomere 6 lateral wall"
    RHOMBOMERE_6_ROOF_PLATE = "rhombomere 6 roof plate"
    RHOMBOMERE_7 = "rhombomere 7"
    RHOMBOMERE_7_FLOOR_PLATE = "rhombomere 7 floor plate"
    RHOMBOMERE_7_LATERAL_WALL = "rhombomere 7 lateral wall"
    RHOMBOMERE_7_ROOF_PLATE = "rhombomere 7 roof plate"
    RHOMBOMERE_8 = "rhombomere 8"
    RHOMBOMERE_8_FLOOR_PLATE = "rhombomere 8 floor plate"
    RHOMBOMERE_8_LATERAL_WALL = "rhombomere 8 lateral wall"
    RHOMBOMERE_8_ROOF_PLATE = "rhombomere 8 roof plate"
    PROSENCEPHALON = "prosencephalon"
    PROSENCEPHALON_FLOOR_PLATE = "prosencephalon floor plate"
    FUTURE_DIENCEPHALON = "future diencephalon"
    FUTURE_DIENCEPHALON_FLOOR_PLATE = "future diencephalon floor plate"
    FUTURE_DIENCEPHALON_LATERAL_WALL = "future diencephalon lateral wall"
    FUTURE_DIENCEPHALON_NEURAL_CREST = "future diencephalon neural crest"
    FUTURE_DIENCEPHALON_ROOF_PLATE = "future diencephalon roof plate"
    PROSENCEPHALON_LATERAL_WALL = "prosencephalon lateral wall"
    PROSENCEPHALIC_VESICLE = "prosencephalic vesicle"
    PROSENCEPHALON_ROOF_PLATE = "prosencephalon roof plate"
    ROSTRAL_NEUROPORE = "rostral neuropore"
    PROSENCEPHALON_NEURAL_CREST = "prosencephalon neural crest"
    CAUDAL_NEUROPORE = "caudal neuropore"
    OTIC_PIT = "otic pit"
    OTIC_PIT_MESENCHYME = "otic pit mesenchyme"
    OTIC_PIT_EPITHELIUM = "otic pit epithelium"
    OPTIC_EMINENCE_SURFACE_ECTODERM = "optic eminence surface ectoderm"
    OPTIC_VESICLE = "optic vesicle"
    OPTIC_VESICLE_NEURAL_ECTODERM = "optic vesicle neural ectoderm"
    NOSE = "nose"
    OLFACTORY_PLACODE = "olfactory placode"
    LEFT_UMBILICAL_ARTERY = "left umbilical artery"
    RIGHT_UMBILICAL_ARTERY = "right umbilical artery"
    ATRIOVENTRICULAR_CANAL = "atrioventricular canal"
    VESSELS_CONNECTING_INTERNAL_CAROTID_ARTERY_TO_PRIMARY_HEAD_VEINS = (
        "vessels connecting internal carotid artery to primary head veins"
    )
    FOREGUT = "foregut"
    PHARYNGEAL_REGION = "pharyngeal region"
    PHARYNGEAL_REGION_MESENCHYME = "pharyngeal region mesenchyme"
    PHARYNGEAL_REGION_EPITHELIUM = "pharyngeal region epithelium"
    PHARYNGEAL_REGION_VASCULAR_ELEMENT = "pharyngeal region vascular element"
    FOREGUT_VASCULAR_ELEMENT = "foregut vascular element"
    FOREGUT_GLAND = "foregut gland"
    BILIARY_BUD = "biliary bud"
    BILIARY_BUD_EXTRAHEPATIC_PART = "biliary bud extrahepatic part"
    BILIARY_BUD_INTRAHEPATIC_PART = "biliary bud intrahepatic part"
    FOREGUT_MIDGUT_JUNCTION_EPITHELIUM = "foregut-midgut junction epithelium"
    FOREGUT_MIDGUT_JUNCTION_VASCULAR_ELEMENT = "foregut-midgut junction vascular element"
    HINDGUT_DIVERTICULUM_EPITHELIUM = "hindgut diverticulum epithelium"
    HINDGUT_DIVERTICULUM_VASCULAR_ELEMENT = "hindgut diverticulum vascular element"
    MIDGUT_EPITHELIUM = "midgut epithelium"
    MIDGUT_VASCULAR_ELEMENT = "midgut vascular element"
    STOMATODEUM_GLAND = "stomatodeum gland"
    STOMATODEUM_ECTODERM = "stomatodeum ectoderm"
    GENITOURINARY_SYSTEM_MESENCHYME = "genitourinary system mesenchyme"
    NEPHRIC_DUCT = "nephric duct"
    PRIMORDIAL_GERM_CELL_OF_GONAD = "primordial germ cell of gonad"
    PRONEPHROS = "pronephros"
    TAIL_BUD = "tail bud"
    _3RD_BRANCHIAL_GROOVE = "3rd branchial groove"
    _3RD_BRANCHIAL_GROOVE_ECTODERM = "3rd branchial groove ectoderm"
    _3RD_BRANCHIAL_MEMBRANE = "3rd branchial membrane"
    _3RD_BRANCHIAL_MEMBRANE_ECTODERM = "3rd branchial membrane ectoderm"
    _3RD_BRANCHIAL_MEMBRANE_ENDODERM = "3rd branchial membrane endoderm"
    _3RD_BRANCHIAL_POUCH = "3rd branchial pouch"
    _3RD_BRANCHIAL_POUCH_ENDODERM = "3rd branchial pouch endoderm"
    PARIETAL_PERICARDIUM = "parietal pericardium"
    VISCERAL_PERICARDIUM = "visceral pericardium"
    PERICARDIO_PERITONEAL_CANAL_PARIETAL_MESOTHELIUM = "pericardio-peritoneal canal parietal mesothelium"
    PERITONEUM_PARIETAL_MESOTHELIUM = "peritoneum parietal mesothelium"
    PERITONEUM_VISCERAL_MESOTHELIUM = "peritoneum visceral mesothelium"
    HINDLIMB_RIDGE = "hindlimb ridge"
    HINDLIMB_RIDGE_ECTODERM = "hindlimb ridge ectoderm"
    HINDLIMB_RIDGE_MESENCHYME = "hindlimb ridge mesenchyme"
    TRUNK_MESENCHYME_DERIVED_FROM_NEURAL_CREST_LATERAL_MIGRATION_PATHWAY = (
        "trunk mesenchyme derived from neural crest lateral migration pathway"
    )
    TRUNK_MESENCHYME_DERIVED_FROM_NEURAL_CREST_MEDIAL_MIGRATION_PATHWAY = (
        "trunk mesenchyme derived from neural crest medial migration pathway"
    )
    SOMITE_21 = "somite 21"
    SOMITE_21_DERMOMYOTOME = "somite 21 dermomyotome"
    SOMITE_21_MYOCOELE = "somite 21 myocoele"
    SOMITE_21_SCLEROTOME = "somite 21 sclerotome"
    SOMITE_22 = "somite 22"
    SOMITE_22_DERMOMYOTOME = "somite 22 dermomyotome"
    SOMITE_22_MYOCOELE = "somite 22 myocoele"
    SOMITE_22_SCLEROTOME = "somite 22 sclerotome"
    SOMITE_23 = "somite 23"
    SOMITE_23_DERMOMYOTOME = "somite 23 dermomyotome"
    SOMITE_23_MYOCOELE = "somite 23 myocoele"
    SOMITE_23_SCLEROTOME = "somite 23 sclerotome"
    SOMITE_24 = "somite 24"
    SOMITE_24_DERMOMYOTOME = "somite 24 dermomyotome"
    SOMITE_24_MYOCOELE = "somite 24 myocoele"
    SOMITE_24_SCLEROTOME = "somite 24 sclerotome"
    SOMITE_25 = "somite 25"
    SOMITE_25_DERMOMYOTOME = "somite 25 dermomyotome"
    SOMITE_25_MYOCOELE = "somite 25 myocoele"
    SOMITE_25_SCLEROTOME = "somite 25 sclerotome"
    SOMITE_26 = "somite 26"
    SOMITE_26_DERMOMYOTOME = "somite 26 dermomyotome"
    SOMITE_26_MYOCOELE = "somite 26 myocoele"
    SOMITE_26_SCLEROTOME = "somite 26 sclerotome"
    SOMITE_27 = "somite 27"
    SOMITE_27_DERMOMYOTOME = "somite 27 dermomyotome"
    SOMITE_27_MYOCOELE = "somite 27 myocoele"
    SOMITE_27_SCLEROTOME = "somite 27 sclerotome"
    SOMITE_28 = "somite 28"
    SOMITE_28_DERMOMYOTOME = "somite 28 dermomyotome"
    SOMITE_28_MYOCOELE = "somite 28 myocoele"
    SOMITE_28_SCLEROTOME = "somite 28 sclerotome"
    SOMITE_29 = "somite 29"
    SOMITE_29_DERMOMYOTOME = "somite 29 dermomyotome"
    SOMITE_29_MYOCOELE = "somite 29 myocoele"
    SOMITE_29_SCLEROTOME = "somite 29 sclerotome"
    SOMITE_30 = "somite 30"
    SOMITE_30_DERMOMYOTOME = "somite 30 dermomyotome"
    SOMITE_30_MYOCOELE = "somite 30 myocoele"
    SOMITE_30_SCLEROTOME = "somite 30 sclerotome"
    FUTURE_FOREBRAIN = "future forebrain"
    FUTURE_NEUROHYPOPHYSIS = "future neurohypophysis"
    TELENCEPHALIC_VESICLE = "telencephalic vesicle"
    TELENCEPHALON_FLOOR_PLATE = "telencephalon floor plate"
    TELENCEPHALON_LATERAL_WALL = "telencephalon lateral wall"
    TELENCEPHALON_ROOF_PLATE = "telencephalon roof plate"
    CENTRAL_NERVOUS_SYSTEM_GANGLION = "central nervous system ganglion"
    CRANIAL_GANGLION = "cranial ganglion"
    FACIO_ACOUSTIC_PREGANGLION_COMPLEX = "facio-acoustic preganglion complex"
    GLOSSOPHARYNGEAL_PREGANGLION = "glossopharyngeal preganglion"
    GLOSSOPHARYNGEAL_VAGUS_PREGANGLION_COMPLEX = "glossopharyngeal-vagus preganglion complex"
    TRIGEMINAL_PREGANGLION = "trigeminal preganglion"
    VAGUS_PREGANGLION = "vagus preganglion"
    PERIPHERAL_NERVOUS_SYSTEM = "peripheral nervous system"
    SPINAL_COMPONENT_OF_PERIPHERAL_NERVOUS_SYSTEM = "spinal component of peripheral nervous system"
    SPINAL_GANGLION = "spinal ganglion"
    DORSAL_ROOT_GANGLION = "dorsal root ganglion"
    OTOCYST = "otocyst"
    OTOCYST_MESENCHYME = "otocyst mesenchyme"
    OTOCYST_EPITHELIUM = "otocyst epithelium"
    LENS_PLACODE = "lens placode"
    EYE_MESENCHYME = "eye mesenchyme"
    OPTIC_CUP = "optic cup"
    OPTIC_CUP_INNER_LAYER = "optic cup inner layer"
    OPTIC_CUP_INTRARETINAL_SPACE = "optic cup intraretinal space"
    OPTIC_CUP_OUTER_LAYER = "optic cup outer layer"
    OPTIC_STALK = "optic stalk"
    NASO_LACRIMAL_GROOVE = "naso-lacrimal groove"
    NASAL_PROCESS = "nasal process"
    FRONTO_NASAL_PROCESS = "fronto-nasal process"
    FRONTO_NASAL_PROCESS_ECTODERM = "fronto-nasal process ectoderm"
    FRONTO_NASAL_PROCESS_MESENCHYME = "fronto-nasal process mesenchyme"
    BRANCHIAL_ARCH_ARTERY = "branchial arch artery"
    _1ST_ARCH_ARTERY = "1st arch artery"
    _2ND_ARCH_ARTERY = "2nd arch artery"
    _3RD_ARCH_ARTERY = "3rd arch artery"
    HEART_ATRIUM = "heart atrium"
    COMMON_ATRIAL_CHAMBER_ENDOCARDIAL_LINING = "common atrial chamber endocardial lining"
    BULBOVENTRICULAR_GROOVE = "bulboventricular groove"
    BULBUS_CORDIS_CAUDAL_HALF_ENDOCARDIAL_LINING = "bulbus cordis caudal half endocardial lining"
    BULBUS_CORDIS_ROSTRAL_HALF_ENDOCARDIAL_LINING = "bulbus cordis rostral half endocardial lining"
    ENDOCARDIAL_CUSHION = "endocardial cushion"
    PRIMITIVE_VENTRICLE_ENDOCARDIAL_LINING = "primitive ventricle endocardial lining"
    LEFT_VITELLINE_VEIN_EMBRYONIC_PART = "left vitelline vein embryonic part"
    RIGHT_VITELLINE_VEIN_EMBRYONIC_PART = "right vitelline vein embryonic part"
    ESOPHAGEAL_REGION = "esophageal region"
    ESOPHAGEAL_REGION_MESENCHYME = "esophageal region mesenchyme"
    ESOPHAGEAL_REGION_EPITHELIUM = "esophageal region epithelium"
    ESOPHAGEAL_REGION_MESENTERY = "esophageal region mesentery"
    DORSAL_MESO_ESOPHAGUS = "dorsal meso-esophagus"
    ESOPHAGEAL_REGION_VASCULAR_ELEMENT = "esophageal region vascular element"
    PHARYNX = "pharynx"
    PHARYNX_MESENCHYME = "pharynx mesenchyme"
    PHARYNX_EPITHELIUM = "pharynx epithelium"
    PHARYNX_VASCULAR_ELEMENT = "pharynx vascular element"
    FOREGUT_MESENTERY = "foregut mesentery"
    FOREGUT_DORSAL_MESENTERY = "foregut dorsal mesentery"
    FOREGUT_VENTRAL_MESENTERY = "foregut ventral mesentery"
    GALLBLADDER_PRIMORDIUM = "gallbladder primordium"
    FOREGUT_MIDGUT_JUNCTION_VENTRAL_MESENTERY = "foregut-midgut junction ventral mesentery"
    HINDGUT = "hindgut"
    HINDGUT_MESENCHYME = "hindgut mesenchyme"
    HINDGUT_EPITHELIUM = "hindgut epithelium"
    HINDGUT_MESENTERY = "hindgut mesentery"
    HINDGUT_DORSAL_MESENTERY = "hindgut dorsal mesentery"
    HINDGUT_VASCULAR_ELEMENT = "hindgut vascular element"
    MIDGUT_MESENTERY = "midgut mesentery"
    MIDGUT_DORSAL_MESENTERY = "midgut dorsal mesentery"
    ORAL_REGION_GLAND = "oral region gland"
    FUTURE_ADENOHYPOPHYSIS = "future adenohypophysis"
    RATHKE_S_POUCH = "Rathke's pouch"
    ORAL_REGION_ECTODERM = "oral region ectoderm"
    RESPIRATORY_SYSTEM = "respiratory system"
    LUNG = "lung"
    LEFT_LUNG_RUDIMENT = "left lung rudiment"
    LEFT_LUNG_RUDIMENT_MESENCHYME = "left lung rudiment mesenchyme"
    LEFT_LUNG_RUDIMENT_EPITHELIUM = "left lung rudiment epithelium"
    LEFT_LUNG_RUDIMENT_VASCULAR_ELEMENT = "left lung rudiment vascular element"
    RIGHT_LUNG_RUDIMENT = "right lung rudiment"
    RIGHT_LUNG_RUDIMENT_MESENCHYME = "right lung rudiment mesenchyme"
    RIGHT_LUNG_RUDIMENT_EPITHELIUM = "right lung rudiment epithelium"
    RIGHT_LUNG_RUDIMENT_VASCULAR_ELEMENT = "right lung rudiment vascular element"
    RESPIRATORY_TRACT = "respiratory tract"
    LOWER_RESPIRATORY_TRACT = "lower respiratory tract"
    LARYNGO_TRACHEAL_GROOVE = "laryngo-tracheal groove"
    TRACHEAL_DIVERTICULUM = "tracheal diverticulum"
    TRACHEAL_DIVERTICULUM_MESENCHYME = "tracheal diverticulum mesenchyme"
    TRACHEAL_DIVERTICULUM_EPITHELIUM = "tracheal diverticulum epithelium"
    TRACHEAL_DIVERTICULUM_VASCULAR_ELEMENT = "tracheal diverticulum vascular element"
    MESONEPHROS = "mesonephros"
    MESONEPHROS_MESENCHYME = "mesonephros mesenchyme"
    MESONEPHRIC_TUBULE = "mesonephric tubule"
    TAIL = "tail"
    TAIL_MESENCHYME = "tail mesenchyme"
    TAIL_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "tail mesenchyme derived from neural crest"
    TAIL_PARAXIAL_MESENCHYME = "tail paraxial mesenchyme"
    TAIL_UNSEGMENTED_MESENCHYME = "tail unsegmented mesenchyme"
    TAIL_NERVOUS_SYSTEM = "tail nervous system"
    TAIL_CENTRAL_NERVOUS_SYSTEM = "tail central nervous system"
    TAIL_FUTURE_SPINAL_CORD = "tail future spinal cord"
    TAIL_NEURAL_PLATE = "tail neural plate"
    TAIL_NEURAL_TUBE = "tail neural tube"
    _3RD_BRANCHIAL_POUCH_DORSAL_ENDODERM = "3rd branchial pouch dorsal endoderm"
    _3RD_BRANCHIAL_POUCH_VENTRAL_ENDODERM = "3rd branchial pouch ventral endoderm"
    _4TH_BRANCHIAL_ARCH = "4th branchial arch"
    _4TH_BRANCHIAL_GROOVE = "4th branchial groove"
    _4TH_BRANCHIAL_GROOVE_ECTODERM = "4th branchial groove ectoderm"
    _4TH_BRANCHIAL_MEMBRANE = "4th branchial membrane"
    _4TH_BRANCHIAL_MEMBRANE_ECTODERM = "4th branchial membrane ectoderm"
    _4TH_BRANCHIAL_MEMBRANE_ENDODERM = "4th branchial membrane endoderm"
    _4TH_BRANCHIAL_POUCH = "4th branchial pouch"
    _4TH_BRANCHIAL_POUCH_ENDODERM = "4th branchial pouch endoderm"
    _4TH_BRANCHIAL_ARCH_ECTODERM = "4th branchial arch ectoderm"
    _4TH_BRANCHIAL_ARCH_ENDODERM = "4th branchial arch endoderm"
    _4TH_BRANCHIAL_ARCH_MESENCHYME = "4th branchial arch mesenchyme"
    _4TH_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_HEAD_MESODERM = (
        "4th branchial arch mesenchyme derived from head mesoderm"
    )
    _4TH_BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "4th branchial arch mesenchyme derived from neural crest"
    PLEURAL_COMPONENT = "pleural component"
    PLEURAL_CAVITY = "pleural cavity"
    PLEURA = "pleura"
    PARIETAL_PLEURA = "parietal pleura"
    VISCERAL_PLEURA = "visceral pleura"
    FORELIMB_BUD_APICAL_ECTODERMAL_RIDGE = "forelimb bud apical ectodermal ridge"
    HINDLIMB_BUD = "hindlimb bud"
    HINDLIMB_BUD_ECTODERM = "hindlimb bud ectoderm"
    HINDLIMB_BUD_MESENCHYME = "hindlimb bud mesenchyme"
    FUTURE_MIDBRAIN_VENTRICULAR_LAYER = "future midbrain ventricular layer"
    RHOMBOMERE_1_VENTRICULAR_LAYER = "rhombomere 1 ventricular layer"
    RHOMBOMERE_2_VENTRICULAR_LAYER = "rhombomere 2 ventricular layer"
    RHOMBOMERE_3_VENTRICULAR_LAYER = "rhombomere 3 ventricular layer"
    RHOMBOMERE_4_VENTRICULAR_LAYER = "rhombomere 4 ventricular layer"
    RHOMBOMERE_5_VENTRICULAR_LAYER = "rhombomere 5 ventricular layer"
    RHOMBOMERE_6_VENTRICULAR_LAYER = "rhombomere 6 ventricular layer"
    RHOMBOMERE_7_VENTRICULAR_LAYER = "rhombomere 7 ventricular layer"
    RHOMBOMERE_8_VENTRICULAR_LAYER = "rhombomere 8 ventricular layer"
    FACIO_ACOUSTIC_GANGLION_COMPLEX = "facio-acoustic ganglion complex"
    GLOSSOPHARYNGEAL_GANGLION = "glossopharyngeal ganglion"
    GLOSSOPHARYNGEAL_VAGUS_GANGLION_COMPLEX = "glossopharyngeal-vagus ganglion complex"
    TRIGEMINAL_GANGLION = "trigeminal ganglion"
    VAGUS_GANGLION = "vagus ganglion"
    LENS_PIT = "lens pit"
    NASAL_PIT = "nasal pit"
    NASAL_PIT_EPITHELIUM = "nasal pit epithelium"
    FRONTAL_PROCESS = "frontal process"
    FRONTAL_PROCESS_ECTODERM = "frontal process ectoderm"
    FRONTAL_PROCESS_MESENCHYME = "frontal process mesenchyme"
    LATERO_NASAL_PROCESS = "latero-nasal process"
    LATERO_NASAL_PROCESS_ECTODERM = "latero-nasal process ectoderm"
    LATERO_NASAL_PROCESS_MESENCHYME = "latero-nasal process mesenchyme"
    MEDIAL_NASAL_PROCESS = "medial-nasal process"
    MEDIAL_NASAL_PROCESS_ECTODERM = "medial-nasal process ectoderm"
    MEDIAL_NASAL_PROCESS_MESENCHYME = "medial-nasal process mesenchyme"
    PRINCIPAL_ARTERY_TO_FORELIMB = "principal artery to forelimb"
    PRINCIPAL_ARTERY_TO_HINDLIMB = "principal artery to hindlimb"
    COMMON_ATRIAL_CHAMBER_LEFT_PART = "common atrial chamber left part"
    COMMON_ATRIAL_CHAMBER_LEFT_PART_CARDIAC_JELLY = "common atrial chamber left part cardiac jelly"
    COMMON_ATRIAL_CHAMBER_LEFT_PART_ENDOCARDIAL_LINING = "common atrial chamber left part endocardial lining"
    LEFT_AURICULAR_REGION = "left auricular region"
    LEFT_AURICULAR_REGION_ENDOCARDIAL_LINING = "left auricular region endocardial lining"
    LEFT_AURICULAR_REGION_CARDIAC_MUSCLE = "left auricular region cardiac muscle"
    COMMON_ATRIAL_CHAMBER_LEFT_PART_CARDIAC_MUSCLE = "common atrial chamber left part cardiac muscle"
    COMMON_ATRIAL_CHAMBER_RIGHT_PART = "common atrial chamber right part"
    COMMON_ATRIAL_CHAMBER_RIGHT_PART_CARDIAC_JELLY = "common atrial chamber right part cardiac jelly"
    COMMON_ATRIAL_CHAMBER_RIGHT_PART_ENDOCARDIAL_LINING = "common atrial chamber right part endocardial lining"
    COMMON_ATRIAL_CHAMBER_RIGHT_PART_CARDIAC_MUSCLE = "common atrial chamber right part cardiac muscle"
    RIGHT_AURICULAR_REGION = "right auricular region"
    RIGHT_AURICULAR_REGION_ENDOCARDIAL_LINING = "right auricular region endocardial lining"
    RIGHT_AURICULAR_REGION_CARDIAC_MUSCLE = "right auricular region cardiac muscle"
    AORTICO_PULMONARY_SPIRAL_SEPTUM = "aortico-pulmonary spiral septum"
    INTER_SUBCARDINAL_VENOUS_ANASTOMOSIS = "inter-subcardinal venous anastomosis"
    SUBCARDINAL_VEIN = "subcardinal vein"
    SUPRACARDINAL_VEIN = "supracardinal vein"
    ANAL_REGION = "anal region"
    CLOACAL_MEMBRANE = "cloacal membrane"
    ESOPHAGUS = "esophagus"
    ESOPHAGUS_MESENCHYME = "esophagus mesenchyme"
    ESOPHAGUS_EPITHELIUM = "esophagus epithelium"
    ESOPHAGUS_MESENTERY = "esophagus mesentery"
    ESOPHAGUS_VASCULAR_ELEMENT = "esophagus vascular element"
    ORAL_REGION_EPITHELIUM = "oral region epithelium"
    LIVER_AND_BILIARY_SYSTEM = "liver and biliary system"
    CYSTIC_DUCT = "cystic duct"
    HEPATIC_DUCT = "hepatic duct"
    HEPATIC_DUCT_EXTRAHEPATIC_PART = "hepatic duct extrahepatic part"
    HEPATIC_DUCT_INTRAHEPATIC_PART = "hepatic duct intrahepatic part"
    LIVER = "liver"
    HEPATIC_PRIMORDIUM = "hepatic primordium"
    HEPATIC_PRIMORDIUM_PARENCHYMA = "hepatic primordium parenchyma"
    MAIN_BRONCHUS = "main bronchus"
    MAIN_BRONCHUS_MESENCHYME = "main bronchus mesenchyme"
    MAIN_BRONCHUS_EPITHELIUM = "main bronchus epithelium"
    MAIN_BRONCHUS_VASCULAR_ELEMENT = "main bronchus vascular element"
    TRACHEA = "trachea"
    TRACHEA_MESENCHYME = "trachea mesenchyme"
    TRACHEA_EPITHELIUM = "trachea epithelium"
    TRACHEA_VASCULAR_ELEMENT = "trachea vascular element"
    GONAD_PRIMORDIUM = "gonad primordium"
    GONAD_PRIMORDIUM_MESENCHYME = "gonad primordium mesenchyme"
    TAIL_SOMITE = "tail somite"
    SOMITE_31 = "somite 31"
    SOMITE_31_DERMOMYOTOME = "somite 31 dermomyotome"
    SOMITE_31_MYOCOELE = "somite 31 myocoele"
    SOMITE_31_SCLEROTOME = "somite 31 sclerotome"
    SOMITE_32 = "somite 32"
    SOMITE_32_DERMOMYOTOME = "somite 32 dermomyotome"
    SOMITE_32_MYOCOELE = "somite 32 myocoele"
    SOMITE_32_SCLEROTOME = "somite 32 sclerotome"
    SOMITE_33 = "somite 33"
    SOMITE_33_DERMOMYOTOME = "somite 33 dermomyotome"
    SOMITE_33_MYOCOELE = "somite 33 myocoele"
    SOMITE_33_SCLEROTOME = "somite 33 sclerotome"
    SOMITE_34 = "somite 34"
    SOMITE_34_DERMOMYOTOME = "somite 34 dermomyotome"
    SOMITE_34_MYOCOELE = "somite 34 myocoele"
    SOMITE_34_SCLEROTOME = "somite 34 sclerotome"
    SOMITE_35 = "somite 35"
    SOMITE_35_DERMOMYOTOME = "somite 35 dermomyotome"
    SOMITE_35_MYOCOELE = "somite 35 myocoele"
    SOMITE_35_SCLEROTOME = "somite 35 sclerotome"
    TAIL_NEURAL_CREST = "tail neural crest"
    _2ND_BRANCHIAL_ARCH_EPITHELIUM = "2nd branchial arch epithelium"
    _3RD_BRANCHIAL_GROOVE_EPITHELIUM = "3rd branchial groove epithelium"
    CERVICAL_SINUS = "cervical sinus"
    GREATER_SAC = "greater sac"
    GREATER_SAC_CAVITY = "greater sac cavity"
    GREATER_SAC_MESOTHELIUM = "greater sac mesothelium"
    GREATER_SAC_PARIETAL_MESOTHELIUM = "greater sac parietal mesothelium"
    OMENTAL_BURSA = "omental bursa"
    OMENTAL_BURSA_MESOTHELIUM = "omental bursa mesothelium"
    OMENTAL_BURSA_PARIETAL_MESOTHELIUM = "omental bursa parietal mesothelium"
    HINDLIMB_BUD_APICAL_ECTODERMAL_RIDGE = "hindlimb bud apical ectodermal ridge"
    BODY_WALL_MESENCHYME = "body wall mesenchyme"
    BRAIN = "brain"
    FOREBRAIN = "forebrain"
    DIENCEPHALON = "diencephalon"
    DIENCEPHALON_GLAND = "diencephalon gland"
    INFUNDIBULAR_RECESS_OF_3RD_VENTRICLE = "infundibular recess of 3rd ventricle"
    _3RD_VENTRICLE = "3rd ventricle"
    DIENCEPHALIC_PART_OF_INTERVENTRICULAR_FORAMEN = "diencephalic part of interventricular foramen"
    OPTIC_RECESS = "optic recess"
    DIENCEPHALON_FLOOR_PLATE = "diencephalon floor plate"
    DIENCEPHALON_LAMINA_TERMINALIS = "diencephalon lamina terminalis"
    DIENCEPHALON_LATERAL_WALL = "diencephalon lateral wall"
    DIENCEPHALON_LATERAL_WALL_MANTLE_LAYER = "diencephalon lateral wall mantle layer"
    DIENCEPHALON_LATERAL_WALL_MARGINAL_LAYER = "diencephalon lateral wall marginal layer"
    DIENCEPHALON_LATERAL_WALL_VENTRICULAR_LAYER = "diencephalon lateral wall ventricular layer"
    DIENCEPHALON_ROOF_PLATE = "diencephalon roof plate"
    TELENCEPHALON = "telencephalon"
    TELENCEPHALON_MANTLE_LAYER = "telencephalon mantle layer"
    TELENCEPHALON_MARGINAL_LAYER = "telencephalon marginal layer"
    TELENCEPHALON_VENTRICULAR_LAYER = "telencephalon ventricular layer"
    LATERAL_VENTRICLE = "lateral ventricle"
    TELENCEPHALIC_PART_OF_INTERVENTRICULAR_FORAMEN = "telencephalic part of interventricular foramen"
    HINDBRAIN = "hindbrain"
    _4TH_VENTRICLE = "4th ventricle"
    RHOMBOMERE_1_MANTLE_LAYER = "rhombomere 1 mantle layer"
    RHOMBOMERE_1_MARGINAL_LAYER = "rhombomere 1 marginal layer"
    RHOMBOMERE_2_MANTLE_LAYER = "rhombomere 2 mantle layer"
    RHOMBOMERE_2_MARGINAL_LAYER = "rhombomere 2 marginal layer"
    RHOMBOMERE_3_MANTLE_LAYER = "rhombomere 3 mantle layer"
    RHOMBOMERE_3_MARGINAL_LAYER = "rhombomere 3 marginal layer"
    RHOMBOMERE_4_MANTLE_LAYER = "rhombomere 4 mantle layer"
    RHOMBOMERE_4_MARGINAL_LAYER = "rhombomere 4 marginal layer"
    RHOMBOMERE_5_MANTLE_LAYER = "rhombomere 5 mantle layer"
    RHOMBOMERE_5_MARGINAL_LAYER = "rhombomere 5 marginal layer"
    RHOMBOMERE_6_MANTLE_LAYER = "rhombomere 6 mantle layer"
    RHOMBOMERE_6_MARGINAL_LAYER = "rhombomere 6 marginal layer"
    RHOMBOMERE_7_MANTLE_LAYER = "rhombomere 7 mantle layer"
    RHOMBOMERE_7_MARGINAL_LAYER = "rhombomere 7 marginal layer"
    RHOMBOMERE_8_MANTLE_LAYER = "rhombomere 8 mantle layer"
    RHOMBOMERE_8_MARGINAL_LAYER = "rhombomere 8 marginal layer"
    MIDBRAIN = "midbrain"
    MIDBRAIN_FLOOR_PLATE = "midbrain floor plate"
    MIDBRAIN_LATERAL_WALL = "midbrain lateral wall"
    MIDBRAIN_MANTLE_LAYER = "midbrain mantle layer"
    MIDBRAIN_MARGINAL_LAYER = "midbrain marginal layer"
    MIDBRAIN_VENTRICULAR_LAYER = "midbrain ventricular layer"
    MESENCEPHALIC_VESICLE = "mesencephalic vesicle"
    MIDBRAIN_ROOF_PLATE = "midbrain roof plate"
    ACOUSTIC_GANGLION = "acoustic ganglion"
    AUTONOMIC_NERVOUS_SYSTEM = "autonomic nervous system"
    SYMPATHETIC_NERVOUS_SYSTEM = "sympathetic nervous system"
    PERIPHERAL_NERVE_TRUNK = "peripheral nerve trunk"
    SPINAL_NERVE_PLEXUS = "spinal nerve plexus"
    BRACHIAL_PLEXUS = "brachial plexus"
    SPINAL_NERVE = "spinal nerve"
    SEGMENTAL_SPINAL_NERVE = "segmental spinal nerve"
    OUTER_EAR = "outer ear"
    BRANCHIAL_MEMBRANE = "branchial membrane"
    BRANCHIAL_MEMBRANE_EPITHELIUM = "branchial membrane epithelium"
    ENDOLYMPHATIC_APPENDAGE = "endolymphatic appendage"
    ENDOLYMPHATIC_APPENDAGE_MESENCHYME = "endolymphatic appendage mesenchyme"
    ENDOLYMPHATIC_APPENDAGE_EPITHELIUM = "endolymphatic appendage epithelium"
    REST_OF_OTOCYST = "rest of otocyst"
    REST_OF_OTOCYST_MESENCHYME = "rest of otocyst mesenchyme"
    REST_OF_OTOCYST_EPITHELIUM = "rest of otocyst epithelium"
    MIDDLE_EAR = "middle ear"
    MIDDLE_EAR_MESENCHYME = "middle ear mesenchyme"
    TUBOTYMPANIC_RECESS = "tubotympanic recess"
    EYE_SURFACE_ECTODERM = "eye surface ectoderm"
    _4TH_ARCH_ARTERY = "4th arch artery"
    _6TH_ARCH_ARTERY = "6th arch artery"
    DUCTUS_CAROTICUS = "ductus caroticus"
    OPHTHALMIC_ARTERY = "ophthalmic artery"
    PULMONARY_ARTERY = "pulmonary artery"
    COMMON_ATRIAL_CHAMBER_RIGHT_PART_VALVE = "common atrial chamber right part valve"
    COMMON_ATRIAL_CHAMBER_RIGHT_PART_VENOUS_VALVE = "common atrial chamber right part venous valve"
    INTERATRIAL_SEPTUM = "interatrial septum"
    FORAMEN_PRIMUM = "foramen primum"
    SEPTUM_PRIMUM = "septum primum"
    ASCENDING_AORTA = "ascending aorta"
    PULMONARY_TRUNK = "pulmonary trunk"
    MARGINAL_VEIN_OF_FORELIMB = "marginal vein of forelimb"
    MARGINAL_VEIN_OF_HINDLIMB = "marginal vein of hindlimb"
    PRINCIPAL_VEIN_OF_FORELIMB = "principal vein of forelimb"
    PRINCIPAL_VEIN_OF_HINDLIMB = "principal vein of hindlimb"
    VITELLINE_VENOUS_PLEXUS = "vitelline venous plexus"
    STOMACH = "stomach"
    STOMACH_MESENCHYME = "stomach mesenchyme"
    STOMACH_EPITHELIUM = "stomach epithelium"
    STOMACH_MESENTERY = "stomach mesentery"
    DORSAL_MESOGASTRIUM = "dorsal mesogastrium"
    VENTRAL_MESOGASTRIUM = "ventral mesogastrium"
    STOMACH_VASCULAR_ELEMENT = "stomach vascular element"
    HINDGUT_DIVERTICULUM_POSTANAL_COMPONENT = "hindgut diverticulum postanal component"
    HINDGUT_DIVERTICULUM_POSTANAL_COMPONENT_MESENCHYME = "hindgut diverticulum postanal component mesenchyme"
    HINDGUT_DIVERTICULUM_POSTANAL_COMPONENT_EPITHELIUM = "hindgut diverticulum postanal component epithelium"
    HINDGUT_DIVERTICULUM_POSTANAL_COMPONENT_VASCULAR_ELEMENT = (
        "hindgut diverticulum postanal component vascular element"
    )
    HINDGUT_DIVERTICULUM_PREANAL_COMPONENT = "hindgut diverticulum preanal component"
    HINDGUT_DIVERTICULUM_PREANAL_COMPONENT_MESENCHYME = "hindgut diverticulum preanal component mesenchyme"
    HINDGUT_DIVERTICULUM_PREANAL_COMPONENT_EPITHELIUM = "hindgut diverticulum preanal component epithelium"
    HINDGUT_DIVERTICULUM_PREANAL_COMPONENT_VASCULAR_ELEMENT = "hindgut diverticulum preanal component vascular element"
    MIDGUT_LOOP = "midgut loop"
    MIDGUT_LOOP_MESENCHYME = "midgut loop mesenchyme"
    MIDGUT_LOOP_EPITHELIUM = "midgut loop epithelium"
    MIDGUT_LOOP_MESENTERY = "midgut loop mesentery"
    MIDGUT_LOOP_DORSAL_MESENTERY = "midgut loop dorsal mesentery"
    MIDGUT_LOOP_VASCULAR_ELEMENT = "midgut loop vascular element"
    SOMITE_36 = "somite 36"
    SOMITE_36_DERMOMYOTOME = "somite 36 dermomyotome"
    SOMITE_36_MYOCOELE = "somite 36 myocoele"
    SOMITE_36_SCLEROTOME = "somite 36 sclerotome"
    SOMITE_37 = "somite 37"
    SOMITE_37_DERMOMYOTOME = "somite 37 dermomyotome"
    SOMITE_37_MYOCOELE = "somite 37 myocoele"
    SOMITE_37_SCLEROTOME = "somite 37 sclerotome"
    SOMITE_38 = "somite 38"
    SOMITE_38_DERMOMYOTOME = "somite 38 dermomyotome"
    SOMITE_38_MYOCOELE = "somite 38 myocoele"
    SOMITE_38_SCLEROTOME = "somite 38 sclerotome"
    SOMITE_39 = "somite 39"
    SOMITE_39_DERMOMYOTOME = "somite 39 dermomyotome"
    SOMITE_39_MYOCOELE = "somite 39 myocoele"
    SOMITE_39_SCLEROTOME = "somite 39 sclerotome"
    FOREGUT_MIDGUT_JUNCTION_GLAND = "foregut-midgut junction gland"
    PANCREAS_PRIMORDIUM = "pancreas primordium"
    PANCREAS_PRIMORDIUM_DORSAL_BUD = "pancreas primordium dorsal bud"
    THYROID_GLAND = "thyroid gland"
    FORAMEN_CAECUM = "foramen caecum"
    THYROGLOSSAL_DUCT = "thyroglossal duct"
    METENCEPHALON = "metencephalon"
    METENCEPHALON_LATERAL_WALL = "metencephalon lateral wall"
    METENCEPHALON_ALAR_PLATE = "metencephalon alar plate"
    CEREBELLUM_PRIMORDIUM = "cerebellum primordium"
    METENCEPHALON_ALAR_PLATE_MANTLE_LAYER = "metencephalon alar plate mantle layer"
    METENCEPHALON_ALAR_PLATE_MARGINAL_LAYER = "metencephalon alar plate marginal layer"
    METENCEPHALON_ALAR_PLATE_VENTRICULAR_LAYER = "metencephalon alar plate ventricular layer"
    METENCEPHALON_BASAL_PLATE = "metencephalon basal plate"
    METENCEPHALON_BASAL_PLATE_MANTLE_LAYER = "metencephalon basal plate mantle layer"
    METENCEPHALON_BASAL_PLATE_MARGINAL_LAYER = "metencephalon basal plate marginal layer"
    METENCEPHALON_BASAL_PLATE_VENTRICULAR_LAYER = "metencephalon basal plate ventricular layer"
    MYELENCEPHALON = "myelencephalon"
    MYELENCEPHALON_LATERAL_WALL = "myelencephalon lateral wall"
    MYELENCEPHALON_ALAR_PLATE = "myelencephalon alar plate"
    MYELENCEPHALON_ALAR_PLATE_MANTLE_LAYER = "myelencephalon alar plate mantle layer"
    MYELENCEPHALON_ALAR_PLATE_MARGINAL_LAYER = "myelencephalon alar plate marginal layer"
    MYELENCEPHALON_ALAR_PLATE_VENTRICULAR_LAYER = "myelencephalon alar plate ventricular layer"
    MYELENCEPHALON_BASAL_PLATE = "myelencephalon basal plate"
    MYELENCEPHALON_BASAL_PLATE_MANTLE_LAYER = "myelencephalon basal plate mantle layer"
    MYELENCEPHALON_BASAL_PLATE_MARGINAL_LAYER = "myelencephalon basal plate marginal layer"
    MYELENCEPHALON_BASAL_PLATE_VENTRICULAR_LAYER = "myelencephalon basal plate ventricular layer"
    NEURAL_TUBE_MANTLE_LAYER = "neural tube mantle layer"
    NEURAL_TUBE_ALAR_COLUMNS = "neural tube alar columns"
    NEURAL_TUBE_BASAL_COLUMNS = "neural tube basal columns"
    NEURAL_TUBE_MARGINAL_LAYER = "neural tube marginal layer"
    NEURAL_TUBE_VENTRICULAR_LAYER = "neural tube ventricular layer"
    INFERIOR_GLOSSOPHARYNGEAL_GANGLION = "inferior glossopharyngeal ganglion"
    SUPERIOR_GLOSSOPHARYNGEAL_GANGLION = "superior glossopharyngeal ganglion"
    INFERIOR_VAGUS_GANGLION = "inferior vagus ganglion"
    SUPERIOR_VAGUS_GANGLION = "superior vagus ganglion"
    SYMPATHETIC_GANGLION = "sympathetic ganglion"
    THORACIC_GANGLION = "thoracic ganglion"
    CORNEA = "cornea"
    CORNEA_EPITHELIUM = "cornea epithelium"
    LENS_VESICLE = "lens vesicle"
    LENS_VESICLE_CAVITY = "lens vesicle cavity"
    LENS_VESICLE_EPITHELIUM = "lens vesicle epithelium"
    OPTIC_FISSURE = "optic fissure"
    PERIOPTIC_MESENCHYME = "perioptic mesenchyme"
    RETINA = "retina"
    EMBRYONIC_FISSURE = "embryonic fissure"
    INTRARETINAL_SPACE = "intraretinal space"
    NEURAL_RETINAL_EPITHELIUM = "neural retinal epithelium"
    PIGMENTED_RETINAL_EPITHELIUM = "pigmented retinal epithelium"
    ENDODERM_COMPONENT_OF_ANAL_MEMBRANE = "endoderm component of anal membrane"
    FOREGUT_DUODENUM = "foregut duodenum"
    DUODENUM_ROSTRAL_PART = "duodenum rostral part"
    DUODENUM_ROSTRAL_PART_MESENCHYME = "duodenum rostral part mesenchyme"
    DUODENUM_ROSTRAL_PART_EPITHELIUM = "duodenum rostral part epithelium"
    DUODENUM_ROSTRAL_PART_MESENTERY = "duodenum rostral part mesentery"
    DUODENUM_ROSTRAL_PART_MESODUODENUM = "duodenum rostral part mesoduodenum"
    DUODENUM_ROSTRAL_PART_VASCULAR_ELEMENT = "duodenum rostral part vascular element"
    TONGUE = "tongue"
    LINGUAL_SWELLING = "lingual swelling"
    MEDIAN_LINGUAL_SWELLING = "median lingual swelling"
    MEDIAN_LINGUAL_SWELLING_EPITHELIUM = "median lingual swelling epithelium"
    LATERAL_LINGUAL_SWELLING = "lateral lingual swelling"
    LATERAL_LINGUAL_SWELLING_EPITHELIUM = "lateral lingual swelling epithelium"
    LATERAL_LINGUAL_SWELLING_MESENCHYME = "lateral lingual swelling mesenchyme"
    MEDIAN_LINGUAL_SWELLING_MESENCHYME = "median lingual swelling mesenchyme"
    MIDGUT_DERIVED_DUODENUM = "midgut derived duodenum"
    DUODENUM_CAUDAL_PART = "duodenum caudal part"
    DUODENUM_CAUDAL_PART_MESENCHYME = "duodenum caudal part mesenchyme"
    DUODENUM_CAUDAL_PART_EPITHELIUM = "duodenum caudal part epithelium"
    DUODENUM_CAUDAL_PART_MESENTERY = "duodenum caudal part mesentery"
    DUODENUM_CAUDAL_PART_MESODUODENUM = "duodenum caudal part mesoduodenum"
    DUODENUM_CAUDAL_PART_VASCULAR_ELEMENT = "duodenum caudal part vascular element"
    ORONASAL_CAVITY = "oronasal cavity"
    COMMON_BILE_DUCT = "common bile duct"
    GALLBLADDER = "gallbladder"
    LIVER_PARENCHYMA = "liver parenchyma"
    STOMACH_PYLORIC_REGION_MUSCULARIS = "stomach pyloric region muscularis"
    URETERIC_BUD = "ureteric bud"
    URORECTAL_SEPTUM = "urorectal septum"
    SKELETON = "skeleton"
    AXIAL_SKELETON = "axial skeleton"
    AXIAL_SKELETON_CERVICAL_REGION = "axial skeleton cervical region"
    CERVICAL_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "cervical vertebral pre-cartilage condensation"
    SOMITE_40 = "somite 40"
    SOMITE_40_DERMOMYOTOME = "somite 40 dermomyotome"
    SOMITE_40_MYOCOELE = "somite 40 myocoele"
    SOMITE_40_SCLEROTOME = "somite 40 sclerotome"
    SOMITE_41 = "somite 41"
    SOMITE_41_DERMOMYOTOME = "somite 41 dermomyotome"
    SOMITE_41_MYOCOELE = "somite 41 myocoele"
    SOMITE_41_SCLEROTOME = "somite 41 sclerotome"
    SOMITE_42 = "somite 42"
    SOMITE_42_DERMOMYOTOME = "somite 42 dermomyotome"
    SOMITE_42_MYOCOELE = "somite 42 myocoele"
    SOMITE_42_SCLEROTOME = "somite 42 sclerotome"
    SOMITE_43 = "somite 43"
    SOMITE_43_DERMOMYOTOME = "somite 43 dermomyotome"
    SOMITE_43_MYOCOELE = "somite 43 myocoele"
    SOMITE_43_SCLEROTOME = "somite 43 sclerotome"
    SOMITE_44 = "somite 44"
    SOMITE_44_DERMOMYOTOME = "somite 44 dermomyotome"
    SOMITE_44_MYOCOELE = "somite 44 myocoele"
    SOMITE_44_SCLEROTOME = "somite 44 sclerotome"
    SOMITE_45 = "somite 45"
    SOMITE_45_DERMOMYOTOME = "somite 45 dermomyotome"
    SOMITE_45_MYOCOELE = "somite 45 myocoele"
    SOMITE_45_SCLEROTOME = "somite 45 sclerotome"
    _1ST_BRANCHIAL_GROOVE_EPITHELIUM = "1st branchial groove epithelium"
    FUTURE_ARM = "future arm"
    FUTURE_ARM_ECTODERM = "future arm ectoderm"
    FUTURE_ARM_MESENCHYME = "future arm mesenchyme"
    HANDPLATE = "handplate"
    HANDPLATE_ECTODERM = "handplate ectoderm"
    HANDPLATE_APICAL_ECTODERMAL_RIDGE = "handplate apical ectodermal ridge"
    HANDPLATE_MESENCHYME = "handplate mesenchyme"
    FOOTPLATE = "footplate"
    FOOTPLATE_ECTODERM = "footplate ectoderm"
    FOOTPLATE_APICAL_ECTODERMAL_RIDGE = "footplate apical ectodermal ridge"
    FOOTPLATE_MESENCHYME = "footplate mesenchyme"
    FUTURE_LEG = "future leg"
    FUTURE_LEG_ECTODERM = "future leg ectoderm"
    FUTURE_LEG_MESENCHYME = "future leg mesenchyme"
    PANCREAS_PRIMORDIUM_VENTRAL_BUD = "pancreas primordium ventral bud"
    _4TH_VENTRICLE_LATERAL_RECESS = "4th ventricle lateral recess"
    METENCEPHALON_FLOOR_PLATE = "metencephalon floor plate"
    METENCEPHALON_ROOF = "metencephalon roof"
    MYELENCEPHALON_FLOOR_PLATE = "myelencephalon floor plate"
    MYELENCEPHALON_ROOF_PLATE = "myelencephalon roof plate"
    CENTRAL_NERVOUS_SYSTEM_NERVE = "central nervous system nerve"
    CRANIAL_NERVE = "cranial nerve"
    ACCESSORY_NERVE = "accessory nerve"
    ACCESSORY_NERVE_CRANIAL_COMPONENT = "accessory nerve cranial component"
    ACCESSORY_NERVE_SPINAL_COMPONENT = "accessory nerve spinal component"
    GLOSSOPHARYNGEAL_NERVE = "glossopharyngeal nerve"
    HYPOGLOSSAL_NERVE = "hypoglossal nerve"
    PARASYMPATHETIC_NERVOUS_SYSTEM = "parasympathetic nervous system"
    PARASYMPATHETIC_NERVE = "parasympathetic nerve"
    VAGAL_NERVE_TRUNK = "vagal nerve trunk"
    VAGUS_NERVE_RIGHT_RECURRENT_LARYNGEAL_BRANCH = "vagus nerve right recurrent laryngeal branch"
    MEDIAN_NERVE = "median nerve"
    RADIAL_NERVE = "radial nerve"
    ULNAR_NERVE = "ulnar nerve"
    AUDITORY_HILLOCKS = "auditory hillocks"
    FUTURE_TYMPANIC_MEMBRANE = "future tympanic membrane"
    FUTURE_TYMPANIC_MEMBRANE_EPITHELIUM = "future tympanic membrane epithelium"
    INNER_EAR_VESTIBULAR_COMPONENT = "inner ear vestibular component"
    DUCTUS_REUNIENS = "ductus reuniens"
    DUCTUS_REUNIENS_MESENCHYME = "ductus reuniens mesenchyme"
    DUCTUS_REUNIENS_EPITHELIUM = "ductus reuniens epithelium"
    SACCULE = "saccule"
    SACCULE_MESENCHYME = "saccule mesenchyme"
    SACCULE_EPITHELIUM = "saccule epithelium"
    UTRICLE = "utricle"
    UTRICLE_MESENCHYME = "utricle mesenchyme"
    UTRICLE_EPITHELIUM = "utricle epithelium"
    POSTERIOR_SEMICIRCULAR_CANAL = "posterior semicircular canal"
    POSTERIOR_SEMICIRCULAR_CANAL_MESENCHYME = "posterior semicircular canal mesenchyme"
    POSTERIOR_SEMICIRCULAR_CANAL_EPITHELIUM = "posterior semicircular canal epithelium"
    SUPERIOR_SEMICIRCULAR_CANAL = "superior semicircular canal"
    SUPERIOR_SEMICIRCULAR_CANAL_MESENCHYME = "superior semicircular canal mesenchyme"
    SUPERIOR_SEMICIRCULAR_CANAL_EPITHELIUM = "superior semicircular canal epithelium"
    HYALOID_CAVITY = "hyaloid cavity"
    HYALOID_VASCULAR_PLEXUS = "hyaloid vascular plexus"
    LENS_VESICLE_ANTERIOR_EPITHELIUM = "lens vesicle anterior epithelium"
    LENS_VESICLE_EQUATORIAL_EPITHELIUM = "lens vesicle equatorial epithelium"
    LENS_VESICLE_POSTERIOR_EPITHELIUM = "lens vesicle posterior epithelium"
    BASILAR_ARTERY = "basilar artery"
    HYOID_ARTERY = "hyoid artery"
    ILIAC_ARTERY = "iliac artery"
    COMMON_ILIAC_ARTERY = "common iliac artery"
    MAXILLARY_ARTERY = "maxillary artery"
    MESENTERIC_ARTERY = "mesenteric artery"
    STAPEDIAL_ARTERY = "stapedial artery"
    VERTEBRAL_ARTERY = "vertebral artery"
    LEFT_ATRIUM = "left atrium"
    LEFT_ATRIUM_AURICULAR_REGION = "left atrium auricular region"
    LEFT_ATRIUM_AURICULAR_REGION_ENDOCARDIAL_LINING = "left atrium auricular region endocardial lining"
    LEFT_ATRIUM_AURICULAR_REGION_CARDIAC_MUSCLE = "left atrium auricular region cardiac muscle"
    LEFT_ATRIUM_ENDOCARDIAL_LINING = "left atrium endocardial lining"
    LEFT_ATRIUM_CARDIAC_MUSCLE = "left atrium cardiac muscle"
    RIGHT_ATRIUM = "right atrium"
    RIGHT_ATRIUM_AURICULAR_REGION = "right atrium auricular region"
    RIGHT_ATRIUM_AURICULAR_REGION_ENDOCARDIAL_LINING = "right atrium auricular region endocardial lining"
    RIGHT_ATRIUM_AURICULAR_REGION_CARDIAC_MUSCLE = "right atrium auricular region cardiac muscle"
    RIGHT_ATRIUM_ENDOCARDIAL_LINING = "right atrium endocardial lining"
    RIGHT_ATRIUM_CARDIAC_MUSCLE = "right atrium cardiac muscle"
    RIGHT_ATRIUM_VALVE = "right atrium valve"
    RIGHT_ATRIUM_VENOUS_VALVE = "right atrium venous valve"
    OUTFLOW_TRACT_AORTIC_COMPONENT = "outflow tract aortic component"
    OUTFLOW_TRACT_PULMONARY_COMPONENT = "outflow tract pulmonary component"
    HEART_VENTRICLE = "heart ventricle"
    BULBAR_RIDGE = "bulbar ridge"
    INTERVENTRICULAR_SEPTUM = "interventricular septum"
    INTERVENTRICULAR_SEPTUM_ENDOCARDIAL_LINING = "interventricular septum endocardial lining"
    INTERVENTRICULAR_SEPTUM_CARDIAC_MUSCLE = "interventricular septum cardiac muscle"
    INTERVENTRICULAR_GROOVE = "interventricular groove"
    HEART_LEFT_VENTRICLE = "heart left ventricle"
    LEFT_VENTRICLE_ENDOCARDIAL_LINING = "left ventricle endocardial lining"
    LEFT_VENTRICLE_CARDIAC_MUSCLE = "left ventricle cardiac muscle"
    HEART_RIGHT_VENTRICLE = "heart right ventricle"
    RIGHT_VENTRICLE_ENDOCARDIAL_LINING = "right ventricle endocardial lining"
    RIGHT_VENTRICLE_CARDIAC_MUSCLE = "right ventricle cardiac muscle"
    DUCTUS_VENOSUS = "ductus venosus"
    MESENTERIC_VEIN = "mesenteric vein"
    PORTAL_VEIN = "portal vein"
    ROSTRAL_TWO_THIRDS_OF_ANAL_CANAL = "rostral two-thirds of anal canal"
    HYOID_PRE_CARTILAGE_CONDENSATION = "hyoid pre-cartilage condensation"
    HYPOBRANCHIAL_EMINENCE = "hypobranchial eminence"
    OCCIPITAL_MYOTOME = "occipital myotome"
    EPIGLOTTAL_PRIMORDIUM = "epiglottal primordium"
    MANDIBULAR_PROCESS = "mandibular process"
    MANDIBLE_PRIMORDIUM = "mandible primordium"
    MANDIBULAR_PROCESS_MESENCHYME = "mandibular process mesenchyme"
    MECKEL_S_CARTILAGE_PRE_CARTILAGE_CONDENSATION = "Meckel's cartilage pre-cartilage condensation"
    MAXILLARY_PROCESS = "maxillary process"
    MAXILLA_PRIMORDIUM = "maxilla primordium"
    MAXILLARY_PROCESS_MESENCHYME = "maxillary process mesenchyme"
    PALATAL_SHELF = "palatal shelf"
    PALATAL_SHELF_EPITHELIUM = "palatal shelf epithelium"
    PALATAL_SHELF_MESENCHYME = "palatal shelf mesenchyme"
    LIVER_SINUSOID = "liver sinusoid"
    URINARY_SYSTEM = "urinary system"
    URINARY_SYSTEM_MESENTERY = "urinary system mesentery"
    UROGENITAL_MESENTERY = "urogenital mesentery"
    METANEPHROS = "metanephros"
    METANEPHRIC_MESENCHYME = "metanephric mesenchyme"
    UROGENITAL_MEMBRANE = "urogenital membrane"
    UROGENITAL_SINUS = "urogenital sinus"
    REPRODUCTIVE_SYSTEM = "reproductive system"
    GENITAL_TUBERCLE = "genital tubercle"
    GONAD = "gonad"
    COELOMIC_EPITHELIUM_OF_GONAD = "coelomic epithelium of gonad"
    GONADAL_MESENCHYME = "gonadal mesenchyme"
    ARYTENOID_SWELLING_PRE_CARTILAGE_CONDENSATION = "arytenoid swelling pre-cartilage condensation"
    LEFT_LUNG_RUDIMENT_LOBAR_BRONCHUS = "left lung rudiment lobar bronchus"
    LEFT_LUNG_RUDIMENT_LOBAR_BRONCHUS_MESENCHYME = "left lung rudiment lobar bronchus mesenchyme"
    LEFT_LUNG_RUDIMENT_LOBAR_BRONCHUS_EPITHELIUM = "left lung rudiment lobar bronchus epithelium"
    LEFT_LUNG_RUDIMENT_LOBAR_BRONCHUS_VASCULAR_ELEMENT = "left lung rudiment lobar bronchus vascular element"
    CERVICAL_VERTEBRAL_CARTILAGE_CONDENSATION = "cervical vertebral cartilage condensation"
    AXIAL_SKELETON_LUMBAR_REGION = "axial skeleton lumbar region"
    LUMBAR_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "lumbar vertebral pre-cartilage condensation"
    SCLEROTOME_CONDENSATION = "sclerotome condensation"
    AXIAL_SKELETON_THORACIC_REGION = "axial skeleton thoracic region"
    RIB_PRE_CARTILAGE_CONDENSATION = "rib pre-cartilage condensation"
    THORACIC_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "thoracic vertebral pre-cartilage condensation"
    SOMITE_46 = "somite 46"
    SOMITE_46_DERMOMYOTOME = "somite 46 dermomyotome"
    SOMITE_46_MYOCOELE = "somite 46 myocoele"
    SOMITE_46_SCLEROTOME = "somite 46 sclerotome"
    SOMITE_47 = "somite 47"
    SOMITE_47_DERMOMYOTOME = "somite 47 dermomyotome"
    SOMITE_47_MYOCOELE = "somite 47 myocoele"
    SOMITE_47_SCLEROTOME = "somite 47 sclerotome"
    SOMITE_48 = "somite 48"
    SOMITE_48_DERMOMYOTOME = "somite 48 dermomyotome"
    SOMITE_48_MYOCOELE = "somite 48 myocoele"
    SOMITE_48_SCLEROTOME = "somite 48 sclerotome"
    ORAL_CAVITY = "oral cavity"
    FORELIMB = "forelimb"
    ARM = "arm"
    ELBOW = "elbow"
    ELBOW_EPITHELIUM = "elbow epithelium"
    ELBOW_MESENCHYME = "elbow mesenchyme"
    FOREARM = "forearm"
    FOREARM_EPITHELIUM = "forearm epithelium"
    FOREARM_MESENCHYME = "forearm mesenchyme"
    RADIUS_ULNA_PRE_CARTILAGE_CONDENSATION = "radius-ulna pre-cartilage condensation"
    SHOULDER = "shoulder"
    SHOULDER_EPITHELIUM = "shoulder epithelium"
    SHOULDER_MESENCHYME = "shoulder mesenchyme"
    UPPER_ARM = "upper arm"
    UPPER_ARM_EPITHELIUM = "upper arm epithelium"
    UPPER_ARM_MESENCHYME = "upper arm mesenchyme"
    HUMERUS_PRE_CARTILAGE_CONDENSATION = "humerus pre-cartilage condensation"
    HAND = "hand"
    FORELIMB_DIGIT_1 = "forelimb digit 1"
    FORELIMB_DIGIT_1_EPITHELIUM = "forelimb digit 1 epithelium"
    FORELIMB_DIGIT_1_MESENCHYME = "forelimb digit 1 mesenchyme"
    FORELIMB_DIGIT_2 = "forelimb digit 2"
    FORELIMB_DIGIT_2_EPITHELIUM = "forelimb digit 2 epithelium"
    FORELIMB_DIGIT_2_MESENCHYME = "forelimb digit 2 mesenchyme"
    FORELIMB_DIGIT_3 = "forelimb digit 3"
    FORELIMB_DIGIT_3_EPITHELIUM = "forelimb digit 3 epithelium"
    FORELIMB_DIGIT_3_MESENCHYME = "forelimb digit 3 mesenchyme"
    FORELIMB_DIGIT_4 = "forelimb digit 4"
    FORELIMB_DIGIT_4_EPITHELIUM = "forelimb digit 4 epithelium"
    FORELIMB_DIGIT_4_MESENCHYME = "forelimb digit 4 mesenchyme"
    FORELIMB_DIGIT_5 = "forelimb digit 5"
    FORELIMB_DIGIT_5_EPITHELIUM = "forelimb digit 5 epithelium"
    FORELIMB_DIGIT_5_MESENCHYME = "forelimb digit 5 mesenchyme"
    HAND_REST_OF_EPITHELIUM = "hand rest of epithelium"
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_1_AND_2 = "interdigital region between forelimb digits 1 and 2"
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_1_AND_2_EPITHELIUM = (
        "interdigital region between forelimb digits 1 and 2 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_1_AND_2_MESENCHYME = (
        "interdigital region between forelimb digits 1 and 2 mesenchyme"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_2_AND_3 = "interdigital region between forelimb digits 2 and 3"
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_2_AND_3_EPITHELIUM = (
        "interdigital region between forelimb digits 2 and 3 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_2_AND_3_MESENCHYME = (
        "interdigital region between forelimb digits 2 and 3 mesenchyme"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_3_AND_4 = "interdigital region between forelimb digits 3 and 4"
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_3_AND_4_EPITHELIUM = (
        "interdigital region between forelimb digits 3 and 4 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_3_AND_4_MESENCHYME = (
        "interdigital region between forelimb digits 3 and 4 mesenchyme"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_4_AND_5 = "interdigital region between forelimb digits 4 and 5"
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_4_AND_5_EPITHELIUM = (
        "interdigital region between forelimb digits 4 and 5 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_4_AND_5_MESENCHYME = (
        "interdigital region between forelimb digits 4 and 5 mesenchyme"
    )
    HAND_MESENCHYME = "hand mesenchyme"
    HINDLIMB = "hindlimb"
    FOOT = "foot"
    HINDLIMB_DIGIT_1 = "hindlimb digit 1"
    HINDLIMB_DIGIT_1_EPITHELIUM = "hindlimb digit 1 epithelium"
    HINDLIMB_DIGIT_1_MESENCHYME = "hindlimb digit 1 mesenchyme"
    HINDLIMB_DIGIT_2 = "hindlimb digit 2"
    HINDLIMB_DIGIT_2_EPITHELIUM = "hindlimb digit 2 epithelium"
    HINDLIMB_DIGIT_2_MESENCHYME = "hindlimb digit 2 mesenchyme"
    HINDLIMB_DIGIT_3 = "hindlimb digit 3"
    HINDLIMB_DIGIT_3_EPITHELIUM = "hindlimb digit 3 epithelium"
    HINDLIMB_DIGIT_3_MESENCHYME = "hindlimb digit 3 mesenchyme"
    HINDLIMB_DIGIT_4 = "hindlimb digit 4"
    HINDLIMB_DIGIT_4_EPITHELIUM = "hindlimb digit 4 epithelium"
    HINDLIMB_DIGIT_4_MESENCHYME = "hindlimb digit 4 mesenchyme"
    HINDLIMB_DIGIT_5 = "hindlimb digit 5"
    HINDLIMB_DIGIT_5_EPITHELIUM = "hindlimb digit 5 epithelium"
    HINDLIMB_DIGIT_5_MESENCHYME = "hindlimb digit 5 mesenchyme"
    FOOT_REST_OF_EPITHELIUM = "foot rest of epithelium"
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_1_AND_2 = "interdigital region between hindlimb digits 1 and 2"
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_1_AND_2_EPITHELIUM = (
        "interdigital region between hindlimb digits 1 and 2 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_1_AND_2_MESENCHYME = (
        "interdigital region between hindlimb digits 1 and 2 mesenchyme"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_2_AND_3 = "interdigital region between hindlimb digits 2 and 3"
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_2_AND_3_EPITHELIUM = (
        "interdigital region between hindlimb digits 2 and 3 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_2_AND_3_MESENCHYME = (
        "interdigital region between hindlimb digits 2 and 3 mesenchyme"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_3_AND_4 = "interdigital region between hindlimb digits 3 and 4"
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_3_AND_4_EPITHELIUM = (
        "interdigital region between hindlimb digits 3 and 4 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_3_AND_4_MESENCHYME = (
        "interdigital region between hindlimb digits 3 and 4 mesenchyme"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_4_AND_5 = "interdigital region between hindlimb digits 4 and 5"
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_4_AND_5_EPITHELIUM = (
        "interdigital region between hindlimb digits 4 and 5 epithelium"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_4_AND_5_MESENCHYME = (
        "interdigital region between hindlimb digits 4 and 5 mesenchyme"
    )
    FOOT_MESENCHYME = "foot mesenchyme"
    LEG = "leg"
    HIP = "hip"
    HIP_EPITHELIUM = "hip epithelium"
    HIP_MESENCHYME = "hip mesenchyme"
    KNEE = "knee"
    KNEE_EPITHELIUM = "knee epithelium"
    KNEE_MESENCHYME = "knee mesenchyme"
    LOWER_LEG = "lower leg"
    LOWER_LEG_EPITHELIUM = "lower leg epithelium"
    LOWER_LEG_MESENCHYME = "lower leg mesenchyme"
    UPPER_LEG = "upper leg"
    UPPER_LEG_EPITHELIUM = "upper leg epithelium"
    UPPER_LEG_MESENCHYME = "upper leg mesenchyme"
    FEMUR_PRE_CARTILAGE_CONDENSATION = "femur pre-cartilage condensation"
    PANCREAS = "pancreas"
    BODY_OF_PANCREAS = "body of pancreas"
    DORSAL_PANCREATIC_DUCT_OF_PANCREAS_BODY = "dorsal pancreatic duct of pancreas body"
    PANCREAS_BODY_PARENCHYMA = "pancreas body parenchyma"
    HEAD_OF_PANCREAS = "head of pancreas"
    PANCREAS_HEAD_PARENCHYMA = "pancreas head parenchyma"
    UNCINATE_PROCESS = "uncinate process"
    VENTRAL_PANCREATIC_DUCT = "ventral pancreatic duct"
    TAIL_OF_PANCREAS = "tail of pancreas"
    DORSAL_PANCREATIC_DUCT_OF_PANCREAS_TAIL = "dorsal pancreatic duct of pancreas tail"
    PANCREAS_TAIL_PARENCHYMA = "pancreas tail parenchyma"
    ADENOHYPOPHYSIS = "adenohypophysis"
    ADENOHYPOPHYSIS_PARS_ANTERIOR = "adenohypophysis pars anterior"
    ADENOHYPOPHYSIS_PARS_INTERMEDIA = "adenohypophysis pars intermedia"
    ADENOHYPOPHYSIS_PARS_TUBERALIS = "adenohypophysis pars tuberalis"
    REMNANT_OF_RATHKE_S_POUCH = "remnant of Rathke's pouch"
    NEUROHYPOPHYSIS = "neurohypophysis"
    NEUROHYPOPHYSIS_INFUNDIBULUM = "neurohypophysis infundibulum"
    MEDIAN_EMINENCE = "median eminence"
    THYMUS_PRIMORDIUM = "thymus primordium"
    INTEGUMENTAL_SYSTEM = "integumental system"
    SKIN = "skin"
    DERMIS = "dermis"
    EPIDERMIS = "epidermis"
    VIBRISSA = "vibrissa"
    VIBRISSA_EPITHELIUM = "vibrissa epithelium"
    VIBRISSA_MESENCHYME = "vibrissa mesenchyme"
    EPITHALAMUS = "epithalamus"
    EPITHALAMUS_MANTLE_LAYER = "epithalamus mantle layer"
    EPITHALAMUS_MARGINAL_LAYER = "epithalamus marginal layer"
    EPITHALAMUS_VENTRICULAR_LAYER = "epithalamus ventricular layer"
    HYPOTHALAMUS = "hypothalamus"
    HYPOTHALAMUS_MANTLE_LAYER = "hypothalamus mantle layer"
    HYPOTHALAMUS_MARGINAL_LAYER = "hypothalamus marginal layer"
    HYPOTHALAMUS_VENTRICULAR_LAYER = "hypothalamus ventricular layer"
    THALAMUS = "thalamus"
    THALAMUS_MANTLE_LAYER = "thalamus mantle layer"
    THALAMUS_MARGINAL_LAYER = "thalamus marginal layer"
    THALAMUS_VENTRICULAR_LAYER = "thalamus ventricular layer"
    CEREBRAL_CORTEX = "cerebral cortex"
    CEREBRAL_CORTEX_MANTLE_LAYER = "cerebral cortex mantle layer"
    CEREBRAL_CORTEX_MARGINAL_LAYER = "cerebral cortex marginal layer"
    CEREBRAL_CORTEX_VENTRICULAR_LAYER = "cerebral cortex ventricular layer"
    CHOROID_INVAGINATION = "choroid invagination"
    CORPUS_STRIATUM = "corpus striatum"
    MEDULLA_OBLONGATA = "medulla oblongata"
    MEDULLA_OBLONGATA_FLOOR_PLATE = "medulla oblongata floor plate"
    MEDULLA_OBLONGATA_LATERAL_WALL = "medulla oblongata lateral wall"
    MEDULLA_OBLONGATA_ALAR_PLATE = "medulla oblongata alar plate"
    MEDULLA_OBLONGATA_ALAR_PLATE_MANTLE_LAYER = "medulla oblongata alar plate mantle layer"
    MEDULLA_OBLONGATA_ALAR_PLATE_MARGINAL_LAYER = "medulla oblongata alar plate marginal layer"
    MEDULLA_OBLONGATA_ALAR_PLATE_VENTRICULAR_LAYER = "medulla oblongata alar plate ventricular layer"
    MEDULLA_OBLONGATA_BASAL_PLATE = "medulla oblongata basal plate"
    MEDULLA_OBLONGATA_BASAL_PLATE_MANTLE_LAYER = "medulla oblongata basal plate mantle layer"
    MEDULLA_OBLONGATA_BASAL_PLATE_MARGINAL_LAYER = "medulla oblongata basal plate marginal layer"
    MEDULLA_OBLONGATA_BASAL_PLATE_VENTRICULAR_LAYER = "medulla oblongata basal plate ventricular layer"
    MEDULLA_OBLONGATA_SULCUS_LIMITANS = "medulla oblongata sulcus limitans"
    MEDULLA_OBLONGATA_ROOF = "medulla oblongata roof"
    PONS = "pons"
    PONS_MANTLE_LAYER = "pons mantle layer"
    PONS_MARGINAL_LAYER = "pons marginal layer"
    PONS_VENTRICULAR_LAYER = "pons ventricular layer"
    METENCEPHALON_SULCUS_LIMITANS = "metencephalon sulcus limitans"
    METENCEPHALON_PART_OF_4TH_VENTRICLE_CHOROID_PLEXUS = "metencephalon part of 4th ventricle choroid plexus"
    FACIAL_GANGLION = "facial ganglion"
    VESTIBULO_COCHLEAR_GANGLION_COMPLEX = "vestibulo-cochlear ganglion complex"
    VESTIBULO_COCHLEAR_GANGLION = "vestibulo-cochlear ganglion"
    COCHLEAR_GANGLION = "cochlear ganglion"
    VESTIBULAR_GANGLION = "vestibular ganglion"
    OCULOMOTOR_NERVE = "oculomotor nerve"
    OPTIC_NERVE = "optic nerve"
    TRIGEMINAL_NERVE = "trigeminal nerve"
    SPINAL_CORD = "spinal cord"
    SPINAL_CORD_FLOOR_PLATE = "spinal cord floor plate"
    SPINAL_CORD_LATERAL_WALL = "spinal cord lateral wall"
    SPINAL_CORD_MANTLE_LAYER = "spinal cord mantle layer"
    ALAR_COLUMNS = "alar columns"
    BASAL_COLUMNS = "basal columns"
    SPINAL_CORD_MARGINAL_LAYER = "spinal cord marginal layer"
    SPINAL_CORD_SULCUS_LIMITANS = "spinal cord sulcus limitans"
    SPINAL_CORD_VENTRICULAR_LAYER = "spinal cord ventricular layer"
    SPINAL_CORD_ROOF_PLATE = "spinal cord roof plate"
    LUMBOSACRAL_PLEXUS = "lumbosacral plexus"
    EXTERNAL_AUDITORY_MEATUS = "external auditory meatus"
    PINNA = "pinna"
    PINNA_MESENCHYME = "pinna mesenchyme"
    PINNA_SURFACE_EPITHELIUM = "pinna surface epithelium"
    ENDOLYMPHATIC_SAC = "endolymphatic sac"
    ENDOLYMPHATIC_SAC_MESENCHYME = "endolymphatic sac mesenchyme"
    ENDOLYMPHATIC_SAC_EPITHELIUM = "endolymphatic sac epithelium"
    OTIC_CAPSULE = "otic capsule"
    COCHLEA = "cochlea"
    COCHLEAR_DUCT = "cochlear duct"
    COCHLEAR_DUCT_MESENCHYME = "cochlear duct mesenchyme"
    COCHLEAR_DUCT_EPITHELIUM = "cochlear duct epithelium"
    PHARYNGO_TYMPANIC_TUBE = "pharyngo-tympanic tube"
    CORNEA_STROMA = "cornea stroma"
    OPTIC_CHIASMA = "optic chiasma"
    NASAL_CAVITY = "nasal cavity"
    NASAL_CAVITY_EPITHELIUM = "nasal cavity epithelium"
    OLFACTORY_EPITHELIUM = "olfactory epithelium"
    NASAL_CAVITY_RESPIRATORY_EPITHELIUM = "nasal cavity respiratory epithelium"
    NASAL_SEPTUM = "nasal septum"
    FRONTAL_PROCESS_EPITHELIUM = "frontal process epithelium"
    LATERO_NASAL_PROCESS_EPITHELIUM = "latero-nasal process epithelium"
    MEDIAL_NASAL_PROCESS_EPITHELIUM = "medial-nasal process epithelium"
    VOMERONASAL_ORGAN = "vomeronasal organ"
    ARCH_OF_AORTA = "arch of aorta"
    EXTERNAL_ILIAC_ARTERY = "external iliac artery"
    INNOMINATE_ARTERY = "innominate artery"
    INTERSEGMENTAL_ARTERY = "intersegmental artery"
    SUBCLAVIAN_ARTERY = "subclavian artery"
    ATRIOVENTRICULAR_ENDOCARDIAL_CUSHION = "atrioventricular endocardial cushion"
    OUTFLOW_TRACT_PULMONARY_COMPONENT_PROXIMAL_PART = "outflow tract pulmonary component proximal part"
    INTERNAL_JUGULAR_VEIN = "internal jugular vein"
    SUBCLAVIAN_VEIN = "subclavian vein"
    HYOID_CARTILAGE_CONDENSATION = "hyoid cartilage condensation"
    STOMACH_GLANDULAR_REGION = "stomach glandular region"
    STOMACH_GLANDULAR_REGION_MESENCHYME = "stomach glandular region mesenchyme"
    STOMACH_GLANDULAR_REGION_EPITHELIUM = "stomach glandular region epithelium"
    STOMACH_GLANDULAR_REGION_VASCULAR_ELEMENT = "stomach glandular region vascular element"
    STOMACH_PROVENTRICULAR_REGION = "stomach proventricular region"
    STOMACH_PROVENTRICULAR_REGION_MESENCHYME = "stomach proventricular region mesenchyme"
    STOMACH_PROVENTRICULAR_REGION_EPITHELIUM = "stomach proventricular region epithelium"
    STOMACH_PROVENTRICULAR_REGION_VASCULAR_ELEMENT = "stomach proventricular region vascular element"
    STOMACH_PYLORIC_REGION = "stomach pyloric region"
    STOMACH_PYLORIC_REGION_MESENCHYME = "stomach pyloric region mesenchyme"
    STOMACH_PYLORIC_REGION_EPITHELIUM = "stomach pyloric region epithelium"
    STOMACH_PYLORIC_REGION_VASCULAR_ELEMENT = "stomach pyloric region vascular element"
    MECKEL_S_CARTILAGE = "Meckel's cartilage"
    MANDIBLE_PRIMORDIUM_ALVEOLAR_SULCUS = "mandible primordium alveolar sulcus"
    MAXILLARY_PROCESS_EPITHELIUM = "maxillary process epithelium"
    MAXILLARY_PROCESS_SKELETON = "maxillary process skeleton"
    MAXILLA = "maxilla"
    MAXILLARY_PROCESS_ALVEOLAR_SULCUS = "maxillary process alveolar sulcus"
    PREMAXILLA = "premaxilla"
    PRIMARY_PALATE = "primary palate"
    PRIMARY_PALATE_EPITHELIUM = "primary palate epithelium"
    PRIMARY_PALATE_MESENCHYME = "primary palate mesenchyme"
    PRIMITIVE_COLLECTING_DUCT = "primitive collecting duct"
    ARYTENOID_SWELLING_CARTILAGE_CONDENSATION = "arytenoid swelling cartilage condensation"
    LEFT_LUNG = "left lung"
    LEFT_LUNG_MESENCHYME = "left lung mesenchyme"
    LEFT_LUNG_EPITHELIUM = "left lung epithelium"
    LEFT_LUNG_LOBAR_BRONCHUS = "left lung lobar bronchus"
    LEFT_LUNG_LOBAR_BRONCHUS_MESENCHYME = "left lung lobar bronchus mesenchyme"
    LEFT_LUNG_LOBAR_BRONCHUS_EPITHELIUM = "left lung lobar bronchus epithelium"
    LEFT_LUNG_LOBAR_BRONCHUS_VASCULAR_ELEMENT = "left lung lobar bronchus vascular element"
    LEFT_LUNG_VASCULAR_ELEMENT = "left lung vascular element"
    RIGHT_LUNG = "right lung"
    RIGHT_LUNG_MESENCHYME = "right lung mesenchyme"
    RIGHT_LUNG_EPITHELIUM = "right lung epithelium"
    RIGHT_LUNG_LOBAR_BRONCHUS = "right lung lobar bronchus"
    RIGHT_LUNG_LOBAR_BRONCHUS_MESENCHYME = "right lung lobar bronchus mesenchyme"
    RIGHT_LUNG_LOBAR_BRONCHUS_EPITHELIUM = "right lung lobar bronchus epithelium"
    RIGHT_LUNG_LOBAR_BRONCHUS_VASCULAR_ELEMENT = "right lung lobar bronchus vascular element"
    RIGHT_LUNG_VASCULAR_ELEMENT = "right lung vascular element"
    UPPER_RESPIRATORY_TRACT = "upper respiratory tract"
    NASOPHARYNX = "nasopharynx"
    NASOPHARYNX_MESENCHYME = "nasopharynx mesenchyme"
    NASOPHARYNX_EPITHELIUM = "nasopharynx epithelium"
    NASOPHARYNX_VASCULAR_ELEMENT = "nasopharynx vascular element"
    CERVICAL_VERTEBRA = "cervical vertebra"
    LUMBAR_VERTEBRAL_CARTILAGE_CONDENSATION = "lumbar vertebral cartilage condensation"
    AXIAL_SKELETON_SACRAL_REGION = "axial skeleton sacral region"
    SACRAL_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "sacral vertebral pre-cartilage condensation"
    RIB_CARTILAGE_CONDENSATION = "rib cartilage condensation"
    THORACIC_VERTEBRAL_CARTILAGE_CONDENSATION = "thoracic vertebral cartilage condensation"
    CRANIUM = "cranium"
    CHONDROCRANIUM = "chondrocranium"
    TEMPORAL_BONE = "temporal bone"
    TEMPORAL_BONE_PETROUS_PART = "temporal bone petrous part"
    SOMITE_49 = "somite 49"
    SOMITE_49_DERMOMYOTOME = "somite 49 dermomyotome"
    SOMITE_49_MYOCOELE = "somite 49 myocoele"
    SOMITE_49_SCLEROTOME = "somite 49 sclerotome"
    SOMITE_50 = "somite 50"
    SOMITE_50_DERMOMYOTOME = "somite 50 dermomyotome"
    SOMITE_50_MYOCOELE = "somite 50 myocoele"
    SOMITE_50_SCLEROTOME = "somite 50 sclerotome"
    SOMITE_51 = "somite 51"
    SOMITE_51_DERMOMYOTOME = "somite 51 dermomyotome"
    SOMITE_51_MYOCOELE = "somite 51 myocoele"
    SOMITE_51_SCLEROTOME = "somite 51 sclerotome"
    SOMITE_52 = "somite 52"
    SOMITE_52_DERMOMYOTOME = "somite 52 dermomyotome"
    SOMITE_52_MYOCOELE = "somite 52 myocoele"
    SOMITE_52_SCLEROTOME = "somite 52 sclerotome"
    TAIL_SPINAL_CORD = "tail spinal cord"
    DIAPHRAGM = "diaphragm"
    DIAPHRAGM_CENTRAL_TENDON = "diaphragm central tendon"
    CRUS_OF_DIAPHRAGM = "crus of diaphragm"
    PLEURO_PERICARDIAL_CANAL = "pleuro-pericardial canal"
    PLEURO_PERICARDIAL_CANAL_MESOTHELIAL_LINING = "pleuro-pericardial canal mesothelial lining"
    PLEURO_PERICARDIAL_FOLDS = "pleuro-pericardial folds"
    PLEURO_PERITONEAL_CANAL = "pleuro-peritoneal canal"
    PLEURO_PERITONEAL_CANAL_MESOTHELIAL_LINING = "pleuro-peritoneal canal mesothelial lining"
    PLEURO_PERITONEAL_MEMBRANE = "pleuro-peritoneal membrane"
    ELBOW_JOINT_PRIMORDIUM = "elbow joint primordium"
    RADIUS_ULNA_CARTILAGE_CONDENSATION = "radius-ulna cartilage condensation"
    SHOULDER_JOINT_PRIMORDIUM = "shoulder joint primordium"
    HUMERUS_CARTILAGE_CONDENSATION = "humerus cartilage condensation"
    FORELIMB_DIGIT_1_METACARPUS_PRE_CARTILAGE_CONDENSATION = "forelimb digit 1 metacarpus pre-cartilage condensation"
    FORELIMB_DIGIT_1_PHALANX_PRE_CARTILAGE_CONDENSATION = "forelimb digit 1 phalanx pre-cartilage condensation"
    FORELIMB_DIGIT_2_METACARPUS_PRE_CARTILAGE_CONDENSATION = "forelimb digit 2 metacarpus pre-cartilage condensation"
    FORELIMB_DIGIT_2_PHALANX_PRE_CARTILAGE_CONDENSATION = "forelimb digit 2 phalanx pre-cartilage condensation"
    FORELIMB_DIGIT_3_METACARPUS_PRE_CARTILAGE_CONDENSATION = "forelimb digit 3 metacarpus pre-cartilage condensation"
    FORELIMB_DIGIT_3_PHALANX_PRE_CARTILAGE_CONDENSATION = "forelimb digit 3 phalanx pre-cartilage condensation"
    FORELIMB_DIGIT_4_METACARPUS_PRE_CARTILAGE_CONDENSATION = "forelimb digit 4 metacarpus pre-cartilage condensation"
    FORELIMB_DIGIT_4_PHALANX_PRE_CARTILAGE_CONDENSATION = "forelimb digit 4 phalanx pre-cartilage condensation"
    FORELIMB_DIGIT_5_METACARPUS_PRE_CARTILAGE_CONDENSATION = "forelimb digit 5 metacarpus pre-cartilage condensation"
    FORELIMB_DIGIT_5_PHALANX_PRE_CARTILAGE_CONDENSATION = "forelimb digit 5 phalanx pre-cartilage condensation"
    CARPUS_PRE_CARTILAGE_CONDENSATION = "carpus pre-cartilage condensation"
    HINDLIMB_DIGIT_1_METATARSUS_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 1 metatarsus pre-cartilage condensation"
    HINDLIMB_DIGIT_1_PHALANX_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 1 phalanx pre-cartilage condensation"
    HINDLIMB_DIGIT_2_METATARSUS_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 2 metatarsus pre-cartilage condensation"
    HINDLIMB_DIGIT_2_PHALANX_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 2 phalanx pre-cartilage condensation"
    HINDLIMB_DIGIT_3_METATARSUS_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 3 metatarsus pre-cartilage condensation"
    HINDLIMB_DIGIT_3_PHALANX_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 3 phalanx pre-cartilage condensation"
    HINDLIMB_DIGIT_4_METATARSUS_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 4 metatarsus pre-cartilage condensation"
    HINDLIMB_DIGIT_4_PHALANX_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 4 phalanx pre-cartilage condensation"
    HINDLIMB_DIGIT_5_METATARSUS_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 5 metatarsus pre-cartilage condensation"
    HINDLIMB_DIGIT_5_PHALANX_PRE_CARTILAGE_CONDENSATION = "hindlimb digit 5 phalanx pre-cartilage condensation"
    FUTURE_TARSUS = "future tarsus"
    CALCANEUM_PRE_CARTILAGE_CONDENSATION = "calcaneum pre-cartilage condensation"
    FUTURE_TARSUS_REST_OF_MESENCHYME = "future tarsus rest of mesenchyme"
    TALUS_PRE_CARTILAGE_CONDENSATION = "talus pre-cartilage condensation"
    HIP_JOINT_PRIMORDIUM = "hip joint primordium"
    KNEE_JOINT_PRIMORDIUM = "knee joint primordium"
    TIBIA_FIBULA_PRE_CARTILAGE_CONDENSATION = "tibia-fibula pre-cartilage condensation"
    FEMUR_CARTILAGE_CONDENSATION = "femur cartilage condensation"
    VERTEBRAL_AXIS_MUSCULATURE = "vertebral axis musculature"
    ANTERIOR_ABDOMINAL_WALL_MUSCULATURE = "anterior abdominal wall musculature"
    ANTERIOR_ABDOMINAL_WALL_MUSCLE = "anterior abdominal wall muscle"
    ANTERIOR_ABDOMINAL_WALL_UNDIFFERENTIATED_PRE_MUSCLE_MASS = (
        "anterior abdominal wall undifferentiated pre-muscle mass"
    )
    PECTORAL_GIRDLE_AND_THORACIC_BODY_WALL_MUSCULATURE = "pectoral girdle and thoracic body wall musculature"
    PECTORAL_GIRDLE_AND_THORACIC_BODY_WALL_MUSCLE = "pectoral girdle and thoracic body wall muscle"
    DELTOID_PRE_MUSCLE_MASS = "deltoid pre-muscle mass"
    SUBSCAPULARIS_PRE_MUSCLE_MASS = "subscapularis pre-muscle mass"
    SALIVARY_GLAND = "salivary gland"
    SUBLINGUAL_GLAND_PRIMORDIUM = "sublingual gland primordium"
    SUBLINGUAL_GLAND_PRIMORDIUM_EPITHELIUM = "sublingual gland primordium epithelium"
    SUBLINGUAL_GLAND_PRIMORDIUM_MESENCHYME = "sublingual gland primordium mesenchyme"
    SUBMANDIBULAR_GLAND_PRIMORDIUM = "submandibular gland primordium"
    SUBMANDIBULAR_GLAND_PRIMORDIUM_EPITHELIUM = "submandibular gland primordium epithelium"
    SUBMANDIBULAR_GLAND_PRIMORDIUM_MESENCHYME = "submandibular gland primordium mesenchyme"
    INTEGUMENTAL_SYSTEM_GLAND = "integumental system gland"
    MAMMARY_GLAND = "mammary gland"
    MAMMARY_GLAND_EPITHELIUM = "mammary gland epithelium"
    MAMMARY_MESENCHYME = "mammary mesenchyme"
    VIBRISSA_DERMAL_COMPONENT = "vibrissa dermal component"
    VIBRISSA_EPIDERMAL_COMPONENT = "vibrissa epidermal component"
    DIENCEPHALON_MENINGES = "diencephalon meninges"
    DIENCEPHALON_ARACHNOID_MATER = "diencephalon arachnoid mater"
    DIENCEPHALON_DURA_MATER = "diencephalon dura mater"
    DIENCEPHALON_PIA_MATER = "diencephalon pia mater"
    CHOROID_FISSURE = "choroid fissure"
    LATERAL_VENTRICLE_CHOROID_PLEXUS = "lateral ventricle choroid plexus"
    ANTERIOR_HORN_OF_LATERAL_VENTRICLE = "anterior horn of lateral ventricle"
    INFERIOR_HORN_OF_LATERAL_VENTRICLE = "inferior horn of lateral ventricle"
    POSTERIOR_HORN_OF_LATERAL_VENTRICLE = "posterior horn of lateral ventricle"
    SUPERIOR_HORN_OF_LATERAL_VENTRICLE = "superior horn of lateral ventricle"
    TELENCEPHALON_MENINGES = "telencephalon meninges"
    TELENCEPHALON_ARACHNOID_MATER = "telencephalon arachnoid mater"
    TELENCEPHALON_DURA_MATER = "telencephalon dura mater"
    TELENCEPHALON_PIA_MATER = "telencephalon pia mater"
    OLFACTORY_LOBE = "olfactory lobe"
    OLFACTORY_CORTEX = "olfactory cortex"
    OLFACTORY_CORTEX_MANTLE_LAYER = "olfactory cortex mantle layer"
    OLFACTORY_CORTEX_MARGINAL_LAYER = "olfactory cortex marginal layer"
    OLFACTORY_CORTEX_VENTRICULAR_LAYER = "olfactory cortex ventricular layer"
    HINDBRAIN_MENINGES = "hindbrain meninges"
    HINDBRAIN_ARACHNOID_MATER = "hindbrain arachnoid mater"
    HINDBRAIN_DURA_MATER = "hindbrain dura mater"
    HINDBRAIN_PIA_MATER = "hindbrain pia mater"
    CEREBELLUM = "cerebellum"
    CEREBELLUM_MANTLE_LAYER = "cerebellum mantle layer"
    CEREBELLUM_MARGINAL_LAYER = "cerebellum marginal layer"
    CEREBELLUM_VENTRICULAR_LAYER = "cerebellum ventricular layer"
    MIDBRAIN_MENINGES = "midbrain meninges"
    MIDBRAIN_ARACHNOID_MATER = "midbrain arachnoid mater"
    MIDBRAIN_DURA_MATER = "midbrain dura mater"
    MIDBRAIN_PIA_MATER = "midbrain pia mater"
    CEREBRAL_AQUEDUCT = "cerebral aqueduct"
    FACIAL_NERVE = "facial nerve"
    OLFACTORY_NERVE = "olfactory nerve"
    TRIGEMINAL_NERVE_MANDIBULAR_DIVISION = "trigeminal nerve mandibular division"
    TRIGEMINAL_NERVE_MAXILLARY_DIVISION = "trigeminal nerve maxillary division"
    TRIGEMINAL_NERVE_OPHTHALMIC_DIVISION = "trigeminal nerve ophthalmic division"
    VESTIBULOCOCHLEAR_NERVE = "vestibulocochlear nerve"
    VESTIBULOCOCHLEAR_NERVE_COCHLEAR_COMPONENT = "vestibulocochlear nerve cochlear component"
    VESTIBULOCOCHLEAR_NERVE_VESTIBULAR_COMPONENT = "vestibulocochlear nerve vestibular component"
    SPINAL_CORD_MENINGES = "spinal cord meninges"
    SPINAL_CORD_ARACHNOID_MATER = "spinal cord arachnoid mater"
    SPINAL_CORD_DURA_MATER = "spinal cord dura mater"
    SPINAL_CORD_PIA_MATER = "spinal cord pia mater"
    PHRENIC_NERVE = "phrenic nerve"
    INNER_EAR_LABYRINTH = "inner ear labyrinth"
    LABYRINTH_MESENCHYME = "labyrinth mesenchyme"
    LABYRINTH_EPITHELIUM = "labyrinth epithelium"
    CRUS_COMMUNE = "crus commune"
    CRUS_COMMUNE_MESENCHYME = "crus commune mesenchyme"
    CRUS_COMMUNE_EPITHELIUM = "crus commune epithelium"
    LATERAL_SEMICIRCULAR_CANAL = "lateral semicircular canal"
    LATERAL_SEMICIRCULAR_CANAL_MESENCHYME = "lateral semicircular canal mesenchyme"
    LATERAL_SEMICIRCULAR_CANAL_EPITHELIUM = "lateral semicircular canal epithelium"
    OSSICLE = "ossicle"
    INCUS_PRE_CARTILAGE_CONDENSATION = "incus pre-cartilage condensation"
    MALLEUS_PRE_CARTILAGE_CONDENSATION = "malleus pre-cartilage condensation"
    STAPES_PRE_CARTILAGE_CONDENSATION = "stapes pre-cartilage condensation"
    CHOROIDAL_VASCULATURE = "choroidal vasculature"
    EYELID = "eyelid"
    INNER_CANTHUS = "inner canthus"
    LOWER_EYELID = "lower eyelid"
    LOWER_EYELID_EPITHELIUM = "lower eyelid epithelium"
    LOWER_EYELID_MESENCHYME = "lower eyelid mesenchyme"
    UPPER_EYELID = "upper eyelid"
    UPPER_EYELID_EPITHELIUM = "upper eyelid epithelium"
    UPPER_EYELID_MESENCHYME = "upper eyelid mesenchyme"
    VITREOUS_HUMOUR = "vitreous humour"
    LENS = "lens"
    LENS_ANTERIOR_EPITHELIUM = "lens anterior epithelium"
    LENS_EQUATORIAL_EPITHELIUM = "lens equatorial epithelium"
    LENS_FIBER = "lens fiber"
    ANTERIOR_LENS_FIBER = "anterior lens fiber"
    POSTERIOR_LENS_FIBER = "posterior lens fiber"
    EYE_SKELETAL_MUSCLE = "eye skeletal muscle"
    EXTRINSIC_OCULAR_PRE_MUSCLE_MASS = "extrinsic ocular pre-muscle mass"
    NARIS = "naris"
    EXTERNAL_NARIS = "external naris"
    POSTERIOR_NARIS = "posterior naris"
    POSTERIOR_NARIS_EPITHELIUM = "posterior naris epithelium"
    PRIMARY_CHOANA = "primary choana"
    NASAL_CAPSULE = "nasal capsule"
    NASO_LACRIMAL_DUCT = "naso-lacrimal duct"
    ANTERIOR_CEREBRAL_ARTERY = "anterior cerebral artery"
    COMMON_CAROTID_ARTERY = "common carotid artery"
    ABDOMINAL_AORTA = "abdominal aorta"
    POST_DUCTAL_PART_OF_THORACIC_AORTA = "post-ductal part of thoracic aorta"
    DUCTUS_ARTERIOSUS = "ductus arteriosus"
    HEPATIC_ARTERY = "hepatic artery"
    INTERNAL_ILIAC_ARTERY = "internal iliac artery"
    POSTERIOR_CEREBRAL_ARTERY = "posterior cerebral artery"
    FORAMEN_SECUNDUM = "foramen secundum"
    SEPTUM_SECUNDUM = "septum secundum"
    BULBOVENTRICULAR_REGION = "bulboventricular region"
    BULBAR_CUSHION = "bulbar cushion"
    ENDOCARDIAL_TISSUE = "endocardial tissue"
    HEART_VALVE = "heart valve"
    AORTIC_VALVE = "aortic valve"
    MITRAL_VALVE = "mitral valve"
    PULMONARY_VALVE = "pulmonary valve"
    TRICUSPID_VALVE = "tricuspid valve"
    COMMON_ILIAC_VEIN = "common iliac vein"
    EXTERNAL_JUGULAR_VEIN = "external jugular vein"
    FEMORAL_VEIN = "femoral vein"
    TONGUE_EPITHELIUM = "tongue epithelium"
    TONGUE_MESENCHYME = "tongue mesenchyme"
    TONGUE_MUSCLE = "tongue muscle"
    TONGUE_EXTRINSIC_PRE_MUSCLE_MASS = "tongue extrinsic pre-muscle mass"
    STOMACH_FUNDUS = "stomach fundus"
    STOMACH_FUNDUS_MESENCHYME = "stomach fundus mesenchyme"
    STOMACH_FUNDUS_EPITHELIUM = "stomach fundus epithelium"
    STOMACH_FUNDUS_VASCULAR_ELEMENT = "stomach fundus vascular element"
    LESSER_OMENTUM = "lesser omentum"
    PYLORIC_ANTRUM = "pyloric antrum"
    PERINEAL_BODY = "perineal body"
    PERINEAL_BODY_MESENCHYME = "perineal body mesenchyme"
    PERINEAL_BODY_EPITHELIUM = "perineal body epithelium"
    PERINEAL_BODY_VASCULAR_ELEMENT = "perineal body vascular element"
    RECTUM = "rectum"
    RECTUM_MESENCHYME = "rectum mesenchyme"
    RECTUM_EPITHELIUM = "rectum epithelium"
    RECTUM_VASCULAR_ELEMENT = "rectum vascular element"
    LOWER_JAW = "lower jaw"
    LOWER_LIP = "lower lip"
    LOWER_JAW_SKELETON = "lower jaw skeleton"
    MANDIBLE_MESENCHYME = "mandible mesenchyme"
    LOWER_JAW_MESENCHYME = "lower jaw mesenchyme"
    LOWER_JAW_TOOTH = "lower jaw tooth"
    LOWER_JAW_INCISOR = "lower jaw incisor"
    LOWER_JAW_INCISOR_EPITHELIUM = "lower jaw incisor epithelium"
    LOWER_JAW_INCISOR_MESENCHYME = "lower jaw incisor mesenchyme"
    LOWER_JAW_MOLAR = "lower jaw molar"
    LOWER_JAW_MOLAR_EPITHELIUM = "lower jaw molar epithelium"
    LOWER_JAW_MOLAR_MESENCHYME = "lower jaw molar mesenchyme"
    UPPER_JAW = "upper jaw"
    UPPER_LIP = "upper lip"
    PHILTRUM = "philtrum"
    UPPER_JAW_SKELETON = "upper jaw skeleton"
    UPPER_JAW_MESENCHYME = "upper jaw mesenchyme"
    UPPER_JAW_ALVEOLAR_SULCUS = "upper jaw alveolar sulcus"
    UPPER_JAW_TOOTH = "upper jaw tooth"
    UPPER_JAW_INCISOR = "upper jaw incisor"
    UPPER_JAW_INCISOR_EPITHELIUM = "upper jaw incisor epithelium"
    UPPER_JAW_INCISOR_MESENCHYME = "upper jaw incisor mesenchyme"
    UPPER_JAW_MOLAR = "upper jaw molar"
    UPPER_JAW_MOLAR_EPITHELIUM = "upper jaw molar epithelium"
    UPPER_JAW_MOLAR_MESENCHYME = "upper jaw molar mesenchyme"
    LIVER_BARE_AREA = "liver bare area"
    RENAL_PELVIS = "renal pelvis"
    URETER = "ureter"
    RENAL_CORTEX = "renal cortex"
    NEPHROGENIC_INTERSTITIUM = "nephrogenic interstitium"
    EARLY_NEPHRON = "early nephron"
    PHALLIC_URETHRA_OF_MALE = "phallic urethra of male"
    FEMALE_REPRODUCTIVE_SYSTEM = "female reproductive system"
    MESOVARIUM = "mesovarium"
    OVARY = "ovary"
    SURFACE_EPITHELIUM_OF_OVARY = "surface epithelium of ovary"
    GERM_CELL_OF_OVARY = "germ cell of ovary"
    RETE_OVARII_OF_OVARY = "rete ovarii of ovary"
    MALE_REPRODUCTIVE_SYSTEM = "male reproductive system"
    GENITAL_TUBERCLE_OF_MALE = "genital tubercle of male"
    NEPHRIC_DUCT_OF_MALE__MESONEPHRIC_PORTION = "nephric duct of male, mesonephric portion"
    MESORCHIUM = "mesorchium"
    TESTIS = "testis"
    COELOMIC_EPITHELIUM_OF_TESTIS = "coelomic epithelium of testis"
    GERM_CELL_OF_TESTIS = "germ cell of testis"
    LEFT_LUNG_HILUS = "left lung hilus"
    RIGHT_LUNG_ACCESSORY_LOBE = "right lung accessory lobe"
    RIGHT_LUNG_ACCESSORY_LOBE_LOBAR_BRONCHUS = "right lung accessory lobe lobar bronchus"
    RIGHT_LUNG_ACCESSORY_LOBE_LOBAR_BRONCHUS_MESENCHYME = "right lung accessory lobe lobar bronchus mesenchyme"
    RIGHT_LUNG_ACCESSORY_LOBE_LOBAR_BRONCHUS_EPITHELIUM = "right lung accessory lobe lobar bronchus epithelium"
    RIGHT_LUNG_ACCESSORY_LOBE_LOBAR_BRONCHUS_VASCULAR_ELEMENT = (
        "right lung accessory lobe lobar bronchus vascular element"
    )
    RIGHT_LUNG_CAUDAL_LOBE = "right lung caudal lobe"
    RIGHT_LUNG_CAUDAL_LOBE_LOBAR_BRONCHUS = "right lung caudal lobe lobar bronchus"
    RIGHT_LUNG_CAUDAL_LOBE_LOBAR_BRONCHUS_MESENCHYME = "right lung caudal lobe lobar bronchus mesenchyme"
    RIGHT_LUNG_CAUDAL_LOBE_LOBAR_BRONCHUS_EPITHELIUM = "right lung caudal lobe lobar bronchus epithelium"
    RIGHT_LUNG_CAUDAL_LOBE_LOBAR_BRONCHUS_VASCULAR_ELEMENT = "right lung caudal lobe lobar bronchus vascular element"
    RIGHT_LUNG_CRANIAL_LOBE = "right lung cranial lobe"
    RIGHT_LUNG_CRANIAL_LOBE_LOBAR_BRONCHUS = "right lung cranial lobe lobar bronchus"
    RIGHT_LUNG_CRANIAL_LOBE_LOBAR_BRONCHUS_MESENCHYME = "right lung cranial lobe lobar bronchus mesenchyme"
    RIGHT_LUNG_CRANIAL_LOBE_LOBAR_BRONCHUS_EPITHELIUM = "right lung cranial lobe lobar bronchus epithelium"
    RIGHT_LUNG_CRANIAL_LOBE_LOBAR_BRONCHUS_VASCULAR_ELEMENT = "right lung cranial lobe lobar bronchus vascular element"
    RIGHT_LUNG_HILUS = "right lung hilus"
    RIGHT_LUNG_MIDDLE_LOBE = "right lung middle lobe"
    RIGHT_LUNG_MIDDLE_LOBE_LOBAR_BRONCHUS = "right lung middle lobe lobar bronchus"
    RIGHT_LUNG_MIDDLE_LOBE_LOBAR_BRONCHUS_MESENCHYME = "right lung middle lobe lobar bronchus mesenchyme"
    RIGHT_LUNG_MIDDLE_LOBE_LOBAR_BRONCHUS_EPITHELIUM = "right lung middle lobe lobar bronchus epithelium"
    RIGHT_LUNG_MIDDLE_LOBE_LOBAR_BRONCHUS_VASCULAR_ELEMENT = "right lung middle lobe lobar bronchus vascular element"
    LUMBAR_VERTEBRA = "lumbar vertebra"
    NUCLEUS_PULPOSUS = "nucleus pulposus"
    SACRAL_VERTEBRAL_CARTILAGE_CONDENSATION = "sacral vertebral cartilage condensation"
    RIB = "rib"
    THORACIC_VERTEBRA = "thoracic vertebra"
    BASIOCCIPITAL_PRE_CARTILAGE_CONDENSATION = "basioccipital pre-cartilage condensation"
    BASISPHENOID_PRE_CARTILAGE_CONDENSATION = "basisphenoid pre-cartilage condensation"
    EXOCCIPITAL_PRE_CARTILAGE_CONDENSATION = "exoccipital pre-cartilage condensation"
    SPHENOID_BONE_PRE_CARTILAGE_CONDENSATION = "sphenoid bone pre-cartilage condensation"
    VAULT_OF_SKULL = "vault of skull"
    FRONTAL_BONE_PRIMORDIUM = "frontal bone primordium"
    INTER_PARIETAL_BONE_PRIMORDIUM = "inter-parietal bone primordium"
    PARIETAL_BONE_PRIMORDIUM = "parietal bone primordium"
    TEMPORAL_BONE_SQUAMOUS_PART = "temporal bone squamous part"
    VISCEROCRANIUM = "viscerocranium"
    ETHMOID_BONE_PRIMORDIUM = "ethmoid bone primordium"
    FACIAL_BONE_PRIMORDIUM = "facial bone primordium"
    THORACIC_SEGMENT_SKELETON = "thoracic segment skeleton"
    CLAVICLE_PRE_CARTILAGE_CONDENSATION = "clavicle pre-cartilage condensation"
    SCAPULA_PRE_CARTILAGE_CONDENSATION = "scapula pre-cartilage condensation"
    PELVIC_GIRDLE_SKELETON = "pelvic girdle skeleton"
    ILIUM_PRE_CARTILAGE_CONDENSATION = "ilium pre-cartilage condensation"
    SOMITE_53 = "somite 53"
    SOMITE_53_DERMOMYOTOME = "somite 53 dermomyotome"
    SOMITE_53_MYOCOELE = "somite 53 myocoele"
    SOMITE_53_SCLEROTOME = "somite 53 sclerotome"
    SOMITE_54 = "somite 54"
    SOMITE_54_DERMOMYOTOME = "somite 54 dermomyotome"
    SOMITE_54_MYOCOELE = "somite 54 myocoele"
    SOMITE_54_SCLEROTOME = "somite 54 sclerotome"
    SOMITE_55 = "somite 55"
    SOMITE_55_DERMOMYOTOME = "somite 55 dermomyotome"
    SOMITE_55_MYOCOELE = "somite 55 myocoele"
    SOMITE_55_SCLEROTOME = "somite 55 sclerotome"
    TAIL_SKELETON = "tail skeleton"
    AXIAL_SKELETON_TAIL_REGION = "axial skeleton tail region"
    TAIL_VERTEBRAL_CARTILAGE_CONDENSATION = "tail vertebral cartilage condensation"
    TAIL_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "tail vertebral pre-cartilage condensation"
    DIAPHRAGM_DOME = "diaphragm dome"
    LEFT_HEPATIC_RECESS = "left hepatic recess"
    OMENTAL_BURSA_INFERIOR_RECESS = "omental bursa inferior recess"
    OMENTAL_BURSA_SUPERIOR_RECESS = "omental bursa superior recess"
    RIGHT_HEPATIC_RECESS = "right hepatic recess"
    ELBOW_SKIN = "elbow skin"
    ELBOW_DERMIS = "elbow dermis"
    ELBOW_EPIDERMIS = "elbow epidermis"
    RADIUS_CARTILAGE_CONDENSATION = "radius cartilage condensation"
    ULNA_CARTILAGE_CONDENSATION = "ulna cartilage condensation"
    FOREARM_SKIN = "forearm skin"
    FOREARM_DERMIS = "forearm dermis"
    FOREARM_EPIDERMIS = "forearm epidermis"
    SHOULDER_SKIN = "shoulder skin"
    SHOULDER_DERMIS = "shoulder dermis"
    SHOULDER_EPIDERMIS = "shoulder epidermis"
    UPPER_ARM_SKIN = "upper arm skin"
    UPPER_ARM_DERMIS = "upper arm dermis"
    UPPER_ARM_EPIDERMIS = "upper arm epidermis"
    FORELIMB_DIGIT_1_METACARPUS_CARTILAGE_CONDENSATION = "forelimb digit 1 metacarpus cartilage condensation"
    FORELIMB_DIGIT_1_PHALANX_CARTILAGE_CONDENSATION = "forelimb digit 1 phalanx cartilage condensation"
    FORELIMB_DIGIT_1_SKIN = "forelimb digit 1 skin"
    FORELIMB_DIGIT_1_DERMIS = "forelimb digit 1 dermis"
    FORELIMB_DIGIT_1_EPIDERMIS = "forelimb digit 1 epidermis"
    FORELIMB_DIGIT_2_METACARPUS_CARTILAGE_CONDENSATION = "forelimb digit 2 metacarpus cartilage condensation"
    FORELIMB_DIGIT_2_PHALANX_CARTILAGE_CONDENSATION = "forelimb digit 2 phalanx cartilage condensation"
    FORELIMB_DIGIT_2_SKIN = "forelimb digit 2 skin"
    FORELIMB_DIGIT_2_DERMIS = "forelimb digit 2 dermis"
    FORELIMB_DIGIT_2_EPIDERMIS = "forelimb digit 2 epidermis"
    FORELIMB_DIGIT_3_METACARPUS_CARTILAGE_CONDENSATION = "forelimb digit 3 metacarpus cartilage condensation"
    FORELIMB_DIGIT_3_PHALANX_CARTILAGE_CONDENSATION = "forelimb digit 3 phalanx cartilage condensation"
    FORELIMB_DIGIT_3_SKIN = "forelimb digit 3 skin"
    FORELIMB_DIGIT_3_DERMIS = "forelimb digit 3 dermis"
    FORELIMB_DIGIT_3_EPIDERMIS = "forelimb digit 3 epidermis"
    FORELIMB_DIGIT_4_METACARPUS_CARTILAGE_CONDENSATION = "forelimb digit 4 metacarpus cartilage condensation"
    FORELIMB_DIGIT_4_PHALANX_CARTILAGE_CONDENSATION = "forelimb digit 4 phalanx cartilage condensation"
    FORELIMB_DIGIT_4_SKIN = "forelimb digit 4 skin"
    FORELIMB_DIGIT_4_DERMIS = "forelimb digit 4 dermis"
    FORELIMB_DIGIT_4_EPIDERMIS = "forelimb digit 4 epidermis"
    FORELIMB_DIGIT_5_METACARPUS_CARTILAGE_CONDENSATION = "forelimb digit 5 metacarpus cartilage condensation"
    FORELIMB_DIGIT_5_PHALANX_CARTILAGE_CONDENSATION = "forelimb digit 5 phalanx cartilage condensation"
    FORELIMB_DIGIT_5_SKIN = "forelimb digit 5 skin"
    FORELIMB_DIGIT_5_DERMIS = "forelimb digit 5 dermis"
    FORELIMB_DIGIT_5_EPIDERMIS = "forelimb digit 5 epidermis"
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_1_AND_2_SKIN = (
        "interdigital region between forelimb digits 1 and 2 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_1_AND_2_DERMIS = (
        "interdigital region between forelimb digits 1 and 2 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_1_AND_2_EPIDERMIS = (
        "interdigital region between forelimb digits 1 and 2 epidermis"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_2_AND_3_SKIN = (
        "interdigital region between forelimb digits 2 and 3 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_2_AND_3_DERMIS = (
        "interdigital region between forelimb digits 2 and 3 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_2_AND_3_EPIDERMIS = (
        "interdigital region between forelimb digits 2 and 3 epidermis"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_3_AND_4_SKIN = (
        "interdigital region between forelimb digits 3 and 4 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_3_AND_4_DERMIS = (
        "interdigital region between forelimb digits 3 and 4 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_3_AND_4_EPIDERMIS = (
        "interdigital region between forelimb digits 3 and 4 epidermis"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_4_AND_5_SKIN = (
        "interdigital region between forelimb digits 4 and 5 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_4_AND_5_DERMIS = (
        "interdigital region between forelimb digits 4 and 5 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_FORELIMB_DIGITS_4_AND_5_EPIDERMIS = (
        "interdigital region between forelimb digits 4 and 5 epidermis"
    )
    CARPUS_CARTILAGE_CONDENSATION = "carpus cartilage condensation"
    HAND_REST_OF_SKIN = "hand rest of skin"
    HAND_REST_OF_SKIN_DERMIS = "hand rest of skin dermis"
    HAND_REST_OF_SKIN_EPIDERMIS = "hand rest of skin epidermis"
    HINDLIMB_DIGIT_1_METATARSUS_CARTILAGE_CONDENSATION = "hindlimb digit 1 metatarsus cartilage condensation"
    HINDLIMB_DIGIT_1_PHALANX_CARTILAGE_CONDENSATION = "hindlimb digit 1 phalanx cartilage condensation"
    HINDLIMB_DIGIT_1_SKIN = "hindlimb digit 1 skin"
    HINDLIMB_DIGIT_1_DERMIS = "hindlimb digit 1 dermis"
    HINDLIMB_DIGIT_1_EPIDERMIS = "hindlimb digit 1 epidermis"
    HINDLIMB_DIGIT_2_METATARSUS_CARTILAGE_CONDENSATION = "hindlimb digit 2 metatarsus cartilage condensation"
    HINDLIMB_DIGIT_2_PHALANX_CARTILAGE_CONDENSATION = "hindlimb digit 2 phalanx cartilage condensation"
    HINDLIMB_DIGIT_2_SKIN = "hindlimb digit 2 skin"
    HINDLIMB_DIGIT_2_DERMIS = "hindlimb digit 2 dermis"
    HINDLIMB_DIGIT_2_EPIDERMIS = "hindlimb digit 2 epidermis"
    HINDLIMB_DIGIT_3_METATARSUS_CARTILAGE_CONDENSATION = "hindlimb digit 3 metatarsus cartilage condensation"
    HINDLIMB_DIGIT_3_PHALANX_CARTILAGE_CONDENSATION = "hindlimb digit 3 phalanx cartilage condensation"
    HINDLIMB_DIGIT_3_SKIN = "hindlimb digit 3 skin"
    HINDLIMB_DIGIT_3_DERMIS = "hindlimb digit 3 dermis"
    HINDLIMB_DIGIT_3_EPIDERMIS = "hindlimb digit 3 epidermis"
    HINDLIMB_DIGIT_4_METATARSUS_CARTILAGE_CONDENSATION = "hindlimb digit 4 metatarsus cartilage condensation"
    HINDLIMB_DIGIT_4_PHALANX_CARTILAGE_CONDENSATION = "hindlimb digit 4 phalanx cartilage condensation"
    HINDLIMB_DIGIT_4_SKIN = "hindlimb digit 4 skin"
    HINDLIMB_DIGIT_4_DERMIS = "hindlimb digit 4 dermis"
    HINDLIMB_DIGIT_4_EPIDERMIS = "hindlimb digit 4 epidermis"
    HINDLIMB_DIGIT_5_METATARSUS_CARTILAGE_CONDENSATION = "hindlimb digit 5 metatarsus cartilage condensation"
    HINDLIMB_DIGIT_5_PHALANX_CARTILAGE_CONDENSATION = "hindlimb digit 5 phalanx cartilage condensation"
    HINDLIMB_DIGIT_5_SKIN = "hindlimb digit 5 skin"
    HINDLIMB_DIGIT_5_DERMIS = "hindlimb digit 5 dermis"
    HINDLIMB_DIGIT_5_EPIDERMIS = "hindlimb digit 5 epidermis"
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_1_AND_2_SKIN = (
        "interdigital region between hindlimb digits 1 and 2 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_1_AND_2_DERMIS = (
        "interdigital region between hindlimb digits 1 and 2 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_1_AND_2_EPIDERMIS = (
        "interdigital region between hindlimb digits 1 and 2 epidermis"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_2_AND_3_SKIN = (
        "interdigital region between hindlimb digits 2 and 3 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_2_AND_3_DERMIS = (
        "interdigital region between hindlimb digits 2 and 3 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_2_AND_3_EPIDERMIS = (
        "interdigital region between hindlimb digits 2 and 3 epidermis"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_3_AND_4_SKIN = (
        "interdigital region between hindlimb digits 3 and 4 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_3_AND_4_DERMIS = (
        "interdigital region between hindlimb digits 3 and 4 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_3_AND_4_EPIDERMIS = (
        "interdigital region between hindlimb digits 3 and 4 epidermis"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_4_AND_5_SKIN = (
        "interdigital region between hindlimb digits 4 and 5 skin"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_4_AND_5_DERMIS = (
        "interdigital region between hindlimb digits 4 and 5 dermis"
    )
    INTERDIGITAL_REGION_BETWEEN_HINDLIMB_DIGITS_4_AND_5_EPIDERMIS = (
        "interdigital region between hindlimb digits 4 and 5 epidermis"
    )
    CALCANEUM_CARTILAGE_CONDENSATION = "calcaneum cartilage condensation"
    TALUS_CARTILAGE_CONDENSATION = "talus cartilage condensation"
    FOOT_REST_OF_SKIN = "foot rest of skin"
    FOOT_REST_OF_SKIN_DERMIS = "foot rest of skin dermis"
    FOOT_REST_OF_SKIN_EPIDERMIS = "foot rest of skin epidermis"
    HIP_SKIN = "hip skin"
    HIP_DERMIS = "hip dermis"
    HIP_EPIDERMIS = "hip epidermis"
    KNEE_SKIN = "knee skin"
    KNEE_DERMIS = "knee dermis"
    KNEE_EPIDERMIS = "knee epidermis"
    FIBULA_CARTILAGE_CONDENSATION = "fibula cartilage condensation"
    TIBIA_CARTILAGE_CONDENSATION = "tibia cartilage condensation"
    LOWER_LEG_SKIN = "lower leg skin"
    LOWER_LEG_DERMIS = "lower leg dermis"
    LOWER_LEG_EPIDERMIS = "lower leg epidermis"
    UPPER_LEG_SKIN = "upper leg skin"
    UPPER_LEG_DERMIS = "upper leg dermis"
    UPPER_LEG_EPIDERMIS = "upper leg epidermis"
    EXTERNAL_OBLIQUE_MUSCLE = "external oblique muscle"
    INTERNAL_OBLIQUE_MUSCLE = "internal oblique muscle"
    RECTUS_ABDOMINIS = "rectus abdominis"
    TRANSVERSUS_ABDOMINIS = "transversus abdominis"
    AXIAL_MUSCULATURE = "axial musculature"
    AXIAL_MUSCLE = "axial muscle"
    ERECTOR_SPINAE = "erector spinae"
    PSOAS_MAJOR = "psoas major"
    PSOAS_MINOR = "psoas minor"
    CRANIAL_MUSCULATURE = "cranial musculature"
    CRANIAL_MUSCLE = "cranial muscle"
    DELTOID = "deltoid"
    LATISSIMUS_DORSI = "latissimus dorsi"
    PECTORALIS_MAJOR = "pectoralis major"
    PECTORALIS_MINOR = "pectoralis minor"
    SUBSCAPULARIS = "subscapularis"
    SUPRASPINATUS = "supraspinatus"
    TRAPEZIUS = "trapezius"
    PELVIC_GIRDLE_MUSCULATURE = "pelvic girdle musculature"
    PELVIC_GIRDLE_MUSCLE = "pelvic girdle muscle"
    GLUTEAL_MUSCLE = "gluteal muscle"
    INTEGUMENTAL_SYSTEM_MUSCLE = "integumental system muscle"
    INTEGUMENTAL_SYSTEM_SUPERFICIAL_MUSCLE = "integumental system superficial muscle"
    PANNICULUS_CARNOSUS_GROUP = "panniculus carnosus group"
    PLATYSMA = "platysma"
    THYROID_GLAND_ISTHMUS = "thyroid gland isthmus"
    THYROID_GLAND_LOBE = "thyroid gland lobe"
    THYROID_GLAND_LEFT_LOBE = "thyroid gland left lobe"
    THYROID_GLAND_RIGHT_LOBE = "thyroid gland right lobe"
    DIENCEPHALON_INTERNAL_CAPSULE = "diencephalon internal capsule"
    CAUDATE_NUCLEUS = "caudate nucleus"
    LENTIFORM_NUCLEUS = "lentiform nucleus"
    TENTORIUM_CEREBELLI = "tentorium cerebelli"
    HINDBRAIN_SUBARACHNOID_SPACE = "hindbrain subarachnoid space"
    PONTINE_CISTERN = "pontine cistern"
    MIDBRAIN_TEGMENTUM = "midbrain tegmentum"
    ABDUCENS_NERVE = "abducens nerve"
    FACIAL_NERVE_CHORDA_TYMPANI_BRANCH = "facial nerve chorda tympani branch"
    TROCHLEAR_NERVE = "trochlear nerve"
    CONUS_MEDULLARIS = "conus medullaris"
    FILUM_TERMINALE = "filum terminale"
    AUTONOMIC_GANGLION = "autonomic ganglion"
    CILIARY_GANGLION = "ciliary ganglion"
    PARAGANGLION_OF_ZUCKERKANDL = "paraganglion of Zuckerkandl"
    AUTONOMIC_NERVE_PLEXUS = "autonomic nerve plexus"
    HYPOGASTRIC_PLEXUS = "hypogastric plexus"
    CERVICO_THORACIC_GANGLION = "cervico-thoracic ganglion"
    INTERCOSTAL_NERVE = "intercostal nerve"
    INCUS_CARTILAGE_CONDENSATION = "incus cartilage condensation"
    MALLEUS_CARTILAGE_CONDENSATION = "malleus cartilage condensation"
    STAPES_CARTILAGE_CONDENSATION = "stapes cartilage condensation"
    EYE_ANTERIOR_CHAMBER = "eye anterior chamber"
    AQUEOUS_HUMOUR = "aqueous humour"
    CONJUNCTIVAL_SAC = "conjunctival sac"
    EXTRINSIC_OCULAR_MUSCLE = "extrinsic ocular muscle"
    TUNICA_VASCULOSA_LENTIS = "tunica vasculosa lentis"
    VASA_HYALOIDEA_PROPRIA = "vasa hyaloidea propria"
    LENS_CAPSULE = "lens capsule"
    OPTIC_DISC = "optic disc"
    MIDDLE_CEREBRAL_ARTERY = "middle cerebral artery"
    SPINAL_ARTERY = "spinal artery"
    ANTERIOR_SPINAL_ARTERY = "anterior spinal artery"
    POSTERIOR_SPINAL_ARTERY = "posterior spinal artery"
    FORAMEN_OVALE_OF_INTERATRIAL_SEPTUM = "foramen ovale of interatrial septum"
    INTERVENTRICULAR_SEPTUM_MEMBRANOUS_PART = "interventricular septum membranous part"
    INTERVENTRICULAR_SEPTUM_MUSCULAR_PART = "interventricular septum muscular part"
    LYMPHATIC_SYSTEM = "lymphatic system"
    JUGULAR_LYMPH_SAC = "jugular lymph sac"
    AZYGOS_VEIN = "azygos vein"
    HEMIAZYGOS_VEIN = "hemiazygos vein"
    HEPATICO_CARDIAC_VEIN = "hepatico-cardiac vein"
    LEFT_HEPATICO_CARDIAC_VEIN = "left hepatico-cardiac vein"
    RIGHT_HEPATICO_CARDIAC_VEIN = "right hepatico-cardiac vein"
    SPINAL_VEIN = "spinal vein"
    ANAL_CANAL = "anal canal"
    ANAL_CANAL_ROSTRAL_PART = "anal canal rostral part"
    ANAL_CANAL_ROSTRAL_PART_MESENCHYME = "anal canal rostral part mesenchyme"
    ANAL_CANAL_ROSTRAL_PART_EPITHELIUM = "anal canal rostral part epithelium"
    ANAL_CANAL_ROSTRAL_PART_VASCULAR_ELEMENT = "anal canal rostral part vascular element"
    ANAL_CANAL_CAUDAL_PART = "anal canal caudal part"
    ANAL_CANAL_CAUDAL_PART_MESENCHYME = "anal canal caudal part mesenchyme"
    ANAL_CANAL_CAUDAL_PART_EPITHELIUM = "anal canal caudal part epithelium"
    ANAL_CANAL_CAUDAL_PART_VASCULAR_ELEMENT = "anal canal caudal part vascular element"
    EXTERNAL_ANAL_SPHINCTER = "external anal sphincter"
    INTERNAL_ANAL_SPHINCTER = "internal anal sphincter"
    GASTRO_ESOPHAGEAL_JUNCTION = "gastro-esophageal junction"
    GASTRO_ESOPHAGEAL_JUNCTION_MESENCHYME = "gastro-esophageal junction mesenchyme"
    GASTRO_ESOPHAGEAL_JUNCTION_EPITHELIUM = "gastro-esophageal junction epithelium"
    GASTRO_ESOPHAGEAL_JUNCTION_VASCULAR_ELEMENT = "gastro-esophageal junction vascular element"
    CIRCUMVALLATE_PAPILLAE = "circumvallate papillae"
    FUNGIFORM_PAPILLAE = "fungiform papillae"
    INTERMOLAR_EMINENCE = "intermolar eminence"
    MEDIAN_FIBROUS_SEPTUM = "median fibrous septum"
    TONGUE_EXTRINSIC_MUSCLE = "tongue extrinsic muscle"
    GENIOGLOSSUS = "genioglossus"
    HYOGLOSSUS = "hyoglossus"
    PALATOGLOSSUS = "palatoglossus"
    STYLOGLOSSUS = "styloglossus"
    TONGUE_INTRINSIC_MUSCLE = "tongue intrinsic muscle"
    TONGUE_INTRINSIC_MUSCLE_TRANSVERSE_COMPONENT = "tongue intrinsic muscle transverse component"
    TONGUE_INTRINSIC_MUSCLE_VERTICAL_COMPONENT = "tongue intrinsic muscle vertical component"
    STOMACH_FUNDUS_GLANDULAR_MUCOUS_MEMBRANE = "stomach fundus glandular mucous membrane"
    GASTRO_SPLENIC_LIGAMENT = "gastro-splenic ligament"
    LIENO_RENAL_LIGAMENT = "lieno-renal ligament"
    FALCIFORM_LIGAMENT = "falciform ligament"
    EPIGLOTTIS = "epiglottis"
    MANDIBLE = "mandible"
    LOWER_JAW_ALVEOLAR_SULCUS = "lower jaw alveolar sulcus"
    LOWER_JAW_INCISOR_DENTAL_LAMINA = "lower jaw incisor dental lamina"
    LOWER_JAW_INCISOR_ENAMEL_ORGAN = "lower jaw incisor enamel organ"
    LOWER_JAW_INCISOR_DENTAL_PAPILLA = "lower jaw incisor dental papilla"
    LOWER_JAW_MOLAR_DENTAL_LAMINA = "lower jaw molar dental lamina"
    LOWER_JAW_MOLAR_ENAMEL_ORGAN = "lower jaw molar enamel organ"
    LOWER_JAW_MOLAR_DENTAL_PAPILLA = "lower jaw molar dental papilla"
    UPPER_JAW_INCISOR_DENTAL_LAMINA = "upper jaw incisor dental lamina"
    UPPER_JAW_INCISOR_ENAMEL_ORGAN = "upper jaw incisor enamel organ"
    UPPER_JAW_INCISOR_DENTAL_PAPILLA = "upper jaw incisor dental papilla"
    UPPER_JAW_MOLAR_DENTAL_LAMINA = "upper jaw molar dental lamina"
    UPPER_JAW_MOLAR_ENAMEL_ORGAN = "upper jaw molar enamel organ"
    UPPER_JAW_MOLAR_DENTAL_PAPILLA = "upper jaw molar dental papilla"
    LEFT_HEPATIC_DUCT_INTRAHEPATIC_PART = "left hepatic duct intrahepatic part"
    RIGHT_HEPATIC_DUCT_INTRAHEPATIC_PART = "right hepatic duct intrahepatic part"
    LIVER_LOBE = "liver lobe"
    LIVER_LEFT_LOBE = "liver left lobe"
    LIVER_LEFT_LOBE_BARE_AREA = "liver left lobe bare area"
    LIVER_LEFT_LOBE_HEPATIC_SINUSOID = "liver left lobe hepatic sinusoid"
    LIVER_LEFT_LOBE_PARENCHYMA = "liver left lobe parenchyma"
    LIVER_RIGHT_LOBE = "liver right lobe"
    LIVER_RIGHT_LOBE_BARE_AREA = "liver right lobe bare area"
    LIVER_CAUDATE_LOBE = "liver caudate lobe"
    LIVER_CAUDATE_LOBE_HEPATIC_SINUSOID = "liver caudate lobe hepatic sinusoid"
    LIVER_CAUDATE_LOBE_PARENCHYMA = "liver caudate lobe parenchyma"
    LIVER_RIGHT_LOBE_HEPATIC_SINUSOID = "liver right lobe hepatic sinusoid"
    LIVER_RIGHT_LOBE_PARENCHYMA = "liver right lobe parenchyma"
    LIVER_QUADRATE_LOBE = "liver quadrate lobe"
    LIVER_QUADRATE_LOBE_HEPATIC_SINUSOID = "liver quadrate lobe hepatic sinusoid"
    LIVER_QUADRATE_LOBE_PARENCHYMA = "liver quadrate lobe parenchyma"
    BLADDER = "bladder"
    URACHUS = "urachus"
    RENAL_MEDULLARY_INTERSTITIUM = "renal medullary interstitium"
    MULLERIAN_TUBERCLE = "Mullerian tubercle"
    LABIAL_FOLD = "labial fold"
    MEDIASTINUM_TESTIS = "mediastinum testis"
    RETE_TESTIS = "rete testis"
    LARYNX = "larynx"
    LARYNX_MUCOUS_MEMBRANE = "larynx mucous membrane"
    SACRAL_VERTEBRA = "sacral vertebra"
    BASIOCCIPITAL_CARTILAGE_CONDENSATION = "basioccipital cartilage condensation"
    BASISPHENOID_CARTILAGE_CONDENSATION = "basisphenoid cartilage condensation"
    EXOCCIPITAL_CARTILAGE_CONDENSATION = "exoccipital cartilage condensation"
    SPHENOID_BONE = "sphenoid bone"
    SUPRAOCCIPITAL_CARTILAGE_CONDENSATION = "supraoccipital cartilage condensation"
    CLAVICLE_CARTILAGE_CONDENSATION = "clavicle cartilage condensation"
    SCAPULA_CARTILAGE_CONDENSATION = "scapula cartilage condensation"
    STERNUM = "sternum"
    STERNEBRA_PRE_CARTILAGE_CONDENSATION = "sternebra pre-cartilage condensation"
    ACETABULAR_REGION = "acetabular region"
    ILIUM_CARTILAGE_CONDENSATION = "ilium cartilage condensation"
    ISCHIUM_PRE_CARTILAGE_CONDENSATION = "ischium pre-cartilage condensation"
    PUBIS_PRE_CARTILAGE_CONDENSATION = "pubis pre-cartilage condensation"
    SOMITE_56 = "somite 56"
    SOMITE_56_DERMOMYOTOME = "somite 56 dermomyotome"
    SOMITE_56_MYOCOELE = "somite 56 myocoele"
    SOMITE_56_SCLEROTOME = "somite 56 sclerotome"
    SOMITE_57 = "somite 57"
    SOMITE_57_DERMOMYOTOME = "somite 57 dermomyotome"
    SOMITE_57_MYOCOELE = "somite 57 myocoele"
    SOMITE_57_SCLEROTOME = "somite 57 sclerotome"
    SOMITE_58 = "somite 58"
    SOMITE_58_DERMOMYOTOME = "somite 58 dermomyotome"
    SOMITE_58_MYOCOELE = "somite 58 myocoele"
    SOMITE_58_SCLEROTOME = "somite 58 sclerotome"
    SOMITE_59 = "somite 59"
    SOMITE_59_DERMOMYOTOME = "somite 59 dermomyotome"
    SOMITE_59_MYOCOELE = "somite 59 myocoele"
    SOMITE_59_SCLEROTOME = "somite 59 sclerotome"
    SOMITE_60 = "somite 60"
    SOMITE_60_DERMOMYOTOME = "somite 60 dermomyotome"
    SOMITE_60_MYOCOELE = "somite 60 myocoele"
    SOMITE_60_SCLEROTOME = "somite 60 sclerotome"
    TAIL_PERIPHERAL_NERVOUS_SYSTEM = "tail peripheral nervous system"
    TAIL_GANGLION = "tail ganglion"
    TAIL_DORSAL_ROOT_GANGLION = "tail dorsal root ganglion"
    TAIL_VERTEBRA = "tail vertebra"
    FOREGUT_DIVERTICULUM_LUMEN = "foregut diverticulum lumen"
    HINDGUT_DIVERTICULUM_LUMEN = "hindgut diverticulum lumen"
    MIDGUT_LUMEN = "midgut lumen"
    DUODENUM_ROSTRAL_PART_LUMEN = "duodenum rostral part lumen"
    GASTRO_ESOPHAGEAL_JUNCTION_LUMEN = "gastro-esophageal junction lumen"
    ESOPHAGUS_LUMEN = "esophagus lumen"
    PHARYNX_LUMEN = "pharynx lumen"
    STOMACH_FUNDUS_LUMEN = "stomach fundus lumen"
    STOMACH_GLANDULAR_REGION_LUMEN = "stomach glandular region lumen"
    STOMACH_PROVENTRICULAR_REGION_LUMEN = "stomach proventricular region lumen"
    STOMACH_PYLORIC_REGION_LUMEN = "stomach pyloric region lumen"
    FOREGUT_MIDGUT_JUNCTION_LUMEN = "foregut-midgut junction lumen"
    PERINEAL_BODY_LUMEN = "perineal body lumen"
    RECTUM_LUMEN = "rectum lumen"
    DUODENUM_CAUDAL_PART_LUMEN = "duodenum caudal part lumen"
    MIDGUT_LOOP_LUMEN = "midgut loop lumen"
    ANAL_CANAL_CAUDAL_PART_LUMEN = "anal canal caudal part lumen"
    ANAL_CANAL_ROSTRAL_PART_LUMEN = "anal canal rostral part lumen"
    STOMACH_LUMEN = "stomach lumen"
    HINDGUT_LUMEN = "hindgut lumen"
    HINDGUT_DIVERTICULUM_POSTANAL_COMPONENT_LUMEN = "hindgut diverticulum postanal component lumen"
    HINDGUT_DIVERTICULUM_PREANAL_COMPONENT_LUMEN = "hindgut diverticulum preanal component lumen"
    ORAL_REGION_CAVITY = "oral region cavity"
    FOREGUT_LUMEN = "foregut lumen"
    ESOPHAGEAL_REGION_LUMEN = "esophageal region lumen"
    PHARYNGEAL_REGION_LUMEN = "pharyngeal region lumen"
    FOREGUT_MIDGUT_JUNCTION_ENDODERM = "foregut-midgut junction endoderm"
    PHYSIOLOGICAL_UMBILICAL_HERNIA = "physiological umbilical hernia"
    PHYSIOLOGICAL_UMBILICAL_HERNIA_DERMIS = "physiological umbilical hernia dermis"
    PHYSIOLOGICAL_UMBILICAL_HERNIA_EPIDERMIS = "physiological umbilical hernia epidermis"
    CARINA_TRACHEA = "carina trachea"
    CARINA_TRACHEA_MESENCHYME = "carina trachea mesenchyme"
    CARINA_TRACHEA_EPITHELIUM = "carina trachea epithelium"
    CARINA_TRACHEA_VASCULAR_ELEMENT = "carina trachea vascular element"
    GREATER_SAC_VISCERAL_MESOTHELIUM = "greater sac visceral mesothelium"
    VENA_CAVA = "vena cava"
    INFERIOR_VENA_CAVA = "inferior vena cava"
    SUPERIOR_VENA_CAVA = "superior vena cava"
    LEFT_SUPERIOR_VENA_CAVA = "left superior vena cava"
    RIGHT_SUPERIOR_VENA_CAVA = "right superior vena cava"
    INFERIOR_VENA_CAVA_POST_HEPATIC_PART = "inferior vena cava post-hepatic part"
    INFERIOR_VENA_CAVA_PRE_HEPATIC_PART = "inferior vena cava pre-hepatic part"
    INFERIOR_VENA_CAVA_LEFT_POST_HEPATIC_PART = "inferior vena cava left post-hepatic part"
    INFERIOR_VENA_CAVA_RIGHT_POST_HEPATIC_PART = "inferior vena cava right post-hepatic part"
    GLAND = "gland"
    ADRENAL_GLAND = "adrenal gland"
    ADRENAL_CORTEX = "adrenal cortex"
    ADRENAL_MEDULLA = "adrenal medulla"
    DIAPHRAGM_MESOTHELIUM = "diaphragm mesothelium"
    ELBOW_REST_OF_MESENCHYME = "elbow rest of mesenchyme"
    FOREARM_REST_OF_MESENCHYME = "forearm rest of mesenchyme"
    SHOULDER_REST_OF_MESENCHYME = "shoulder rest of mesenchyme"
    UPPER_ARM_REST_OF_MESENCHYME = "upper arm rest of mesenchyme"
    HIP_REST_OF_MESENCHYME = "hip rest of mesenchyme"
    KNEE_REST_OF_MESENCHYME = "knee rest of mesenchyme"
    LOWER_LEG_REST_OF_MESENCHYME = "lower leg rest of mesenchyme"
    UPPER_LEG_REST_OF_MESENCHYME = "upper leg rest of mesenchyme"
    CERVICAL_GANGLION = "cervical ganglion"
    INFERIOR_CERVICAL_GANGLION = "inferior cervical ganglion"
    MIDDLE_CERVICAL_GANGLION = "middle cervical ganglion"
    SUPERIOR_CERVICAL_GANGLION = "superior cervical ganglion"
    NASAL_SEPTUM_EPITHELIUM = "nasal septum epithelium"
    NASAL_SEPTUM_MESENCHYME = "nasal septum mesenchyme"
    VOMERONASAL_ORGAN_EPITHELIUM = "vomeronasal organ epithelium"
    VOMERONASAL_ORGAN_MESENCHYME = "vomeronasal organ mesenchyme"
    OTIC_PIT_LUMEN = "otic pit lumen"
    OMENTAL_BURSA_CAVITY = "omental bursa cavity"
    HAND_SKIN = "hand skin"
    HAND_DERMIS = "hand dermis"
    HAND_EPIDERMIS = "hand epidermis"
    PALMAR_PAD = "palmar pad"
    TARSUS_REST_OF_MESENCHYME = "tarsus rest of mesenchyme"
    FOOT_SKIN = "foot skin"
    FOOT_DERMIS = "foot dermis"
    FOOT_EPIDERMIS = "foot epidermis"
    PLANTAR_PAD = "plantar pad"
    FABELLA_PRE_CARTILAGE_CONDENSATION = "fabella pre-cartilage condensation"
    PATELLAR_LIGAMENT = "patellar ligament"
    PATELLA_PRE_CARTILAGE_CONDENSATION = "patella pre-cartilage condensation"
    ILIO_PSOAS = "ilio-psoas"
    INTERCOSTAL_MUSCLE = "intercostal muscle"
    INTERCOSTAL_MUSCLE_EXTERNAL_LAYER = "intercostal muscle external layer"
    INTERCOSTAL_MUSCLE_INTERNAL_LAYER = "intercostal muscle internal layer"
    QUADRATUS_LUMBORUM = "quadratus lumborum"
    SERRATUS_VENTRALIS = "serratus ventralis"
    INFRASPINATUS = "infraspinatus"
    GLUTEUS_MAXIMUS = "gluteus maximus"
    SPLEEN_PRIMORDIUM = "spleen primordium"
    PAROTID_GLAND = "parotid gland"
    PAROTID_GLAND_EPITHELIUM = "parotid gland epithelium"
    PAROTID_GLAND_MESENCHYME = "parotid gland mesenchyme"
    THYROGLOSSAL_DUCT_MEDIAN_LOBE = "thyroglossal duct median lobe"
    MAMMARY_LINE = "mammary line"
    EPITHALAMIC_RECESS = "epithalamic recess"
    _3RD_VENTRICLE_CHOROID_PLEXUS = "3rd ventricle choroid plexus"
    PINEAL_PRIMORDIUM = "pineal primordium"
    INTERTHALAMIC_ADHESION = "interthalamic adhesion"
    HEAD_OF_CAUDATE_NUCLEUS = "head of caudate nucleus"
    TAIL_OF_CAUDATE_NUCLEUS = "tail of caudate nucleus"
    HEAD_OF_LATERAL_VENTRICLE = "head of lateral ventricle"
    TAIL_OF_LATERAL_VENTRICLE = "tail of lateral ventricle"
    FALX_CEREBRI = "falx cerebri"
    MEDULLARY_RAPHE = "medullary raphe"
    CEREBELLUM_INTRAVENTRICULAR_PORTION = "cerebellum intraventricular portion"
    CEREBELLUM_INTRAVENTRICULAR_PORTION_MANTLE_LAYER = "cerebellum intraventricular portion mantle layer"
    CEREBELLUM_INTRAVENTRICULAR_PORTION_MARGINAL_LAYER = "cerebellum intraventricular portion marginal layer"
    CEREBELLUM_INTRAVENTRICULAR_PORTION_VENTRICULAR_LAYER = "cerebellum intraventricular portion ventricular layer"
    REST_OF_CEREBELLUM = "rest of cerebellum"
    REST_OF_CEREBELLUM_MANTLE_LAYER = "rest of cerebellum mantle layer"
    REST_OF_CEREBELLUM_MARGINAL_LAYER = "rest of cerebellum marginal layer"
    REST_OF_CEREBELLUM_VENTRICULAR_LAYER = "rest of cerebellum ventricular layer"
    METENCEPHALON_REST_OF_ALAR_PLATE = "metencephalon rest of alar plate"
    METENCEPHALON_REST_OF_ALAR_PLATE_MANTLE_LAYER = "metencephalon rest of alar plate mantle layer"
    METENCEPHALON_REST_OF_ALAR_PLATE_MARGINAL_LAYER = "metencephalon rest of alar plate marginal layer"
    METENCEPHALON_REST_OF_ALAR_PLATE_VENTRICULAR_LAYER = "metencephalon rest of alar plate ventricular layer"
    DORSAL_GREY_HORN = "dorsal grey horn"
    VENTRAL_GREY_HORN = "ventral grey horn"
    INTERMEDIATE_GREY_HORN = "intermediate grey horn"
    SCIATIC_NERVE = "sciatic nerve"
    FUTURE_TYMPANIC_MEMBRANE_ENDOTHELIUM = "future tympanic membrane endothelium"
    ENDOLYMPHATIC_DUCT = "endolymphatic duct"
    ENDOLYMPHATIC_DUCT_MESENCHYME = "endolymphatic duct mesenchyme"
    ENDOLYMPHATIC_DUCT_EPITHELIUM = "endolymphatic duct epithelium"
    INCUS = "incus"
    MALLEUS = "malleus"
    STAPES = "stapes"
    STAPEDIUS_MUSCLE = "stapedius muscle"
    TENSOR_TYMPANI_MUSCLE = "tensor tympani muscle"
    CONJUNCTIVAL_SAC_LOWER_RECESS = "conjunctival sac lower recess"
    CONJUNCTIVAL_SAC_UPPER_RECESS = "conjunctival sac upper recess"
    NEURAL_RETINA = "neural retina"
    RETINA_NERVE_FIBER_LAYER = "retina nerve fiber layer"
    RETINA_NUCLEAR_LAYER = "retina nuclear layer"
    NASAL_MEATUS = "nasal meatus"
    ANTERIOR_NARIS = "anterior naris"
    ANTERIOR_NARIS_EPITHELIUM = "anterior naris epithelium"
    EXTERNAL_NARIS_EPITHELIUM = "external naris epithelium"
    COMMUNICATING_ARTERY = "communicating artery"
    ANTERIOR_COMMUNICATING_ARTERY = "anterior communicating artery"
    POSTERIOR_COMMUNICATING_ARTERY = "posterior communicating artery"
    AORTA = "aorta"
    THORACIC_AORTA = "thoracic aorta"
    DESCENDING_THORACIC_AORTA = "descending thoracic aorta"
    DESCENDING_AORTA = "descending aorta"
    BRONCHIAL_ARTERY = "bronchial artery"
    CAROTID_ARTERY = "carotid artery"
    EXTERNAL_CAROTID_ARTERY = "external carotid artery"
    CEREBRAL_ARTERY = "cerebral artery"
    INTERNAL_THORACIC_ARTERY = "internal thoracic artery"
    INFERIOR_MESENTERIC_ARTERY = "inferior mesenteric artery"
    SUPERIOR_MESENTERIC_ARTERY = "superior mesenteric artery"
    OVARIAN_ARTERY = "ovarian artery"
    TESTICULAR_ARTERY = "testicular artery"
    THYROID_ARTERY = "thyroid artery"
    INFERIOR_THYROID_ARTERY = "inferior thyroid artery"
    SUPERIOR_THYROID_ARTERY = "superior thyroid artery"
    AORTIC_SINUS = "aortic sinus"
    TRABECULAE_CARNEAE = "trabeculae carneae"
    AORTIC_VALVE_LEAFLET = "aortic valve leaflet"
    MITRAL_VALVE_LEAFLET = "mitral valve leaflet"
    PULMONARY_VALVE_LEAFLET = "pulmonary valve leaflet"
    TRICUSPID_VALVE_LEAFLET = "tricuspid valve leaflet"
    BRAIN_VENOUS_SYSTEM = "brain venous system"
    HINDBRAIN_VENOUS_SYSTEM = "hindbrain venous system"
    HINDBRAIN_VENOUS_SINUS = "hindbrain venous sinus"
    HINDBRAIN_VENOUS_DURAL_SINUS = "hindbrain venous dural sinus"
    INFERIOR_SAGITTAL_SINUS = "inferior sagittal sinus"
    SUPERIOR_SAGITTAL_SINUS = "superior sagittal sinus"
    TRANSVERSE_SINUS = "transverse sinus"
    JUGULAR_VEIN = "jugular vein"
    INTERNAL_THORACIC_VEIN = "internal thoracic vein"
    INFERIOR_MESENTERIC_VEIN = "inferior mesenteric vein"
    SUPERIOR_MESENTERIC_VEIN = "superior mesenteric vein"
    OVARIAN_VEIN = "ovarian vein"
    PULMONARY_VEIN = "pulmonary vein"
    SUPERIOR_VESICAL_VEIN = "superior vesical vein"
    TESTICULAR_VEIN = "testicular vein"
    THYROID_VEIN = "thyroid vein"
    PHARYNX_SKELETON = "pharynx skeleton"
    HYOID = "hyoid"
    HYOID_GREATER_HORN = "hyoid greater horn"
    HYOID_LESSER_HORN = "hyoid lesser horn"
    HYOID_BODY = "hyoid body"
    FRENULUM = "frenulum"
    GREATER_OMENTUM = "greater omentum"
    FOREGUT_MIDGUT_JUNCTION_DUODENUM = "foregut-midgut junction duodenum"
    DUODENAL_PAPILLA = "duodenal papilla"
    SPHINCTER_OF_ODDI = "sphincter of Oddi"
    RECTUM_MESENTERY = "rectum mesentery"
    RECTUM_DORSAL_MESENTERY = "rectum dorsal mesentery"
    JEJUNUM = "jejunum"
    JEJUNUM_EPITHELIUM = "jejunum epithelium"
    JEJUNUM_VASCULAR_ELEMENT = "jejunum vascular element"
    JEJUNUM_MESENTERY = "jejunum mesentery"
    JEJUNUM_DORSAL_MESENTERY = "jejunum dorsal mesentery"
    LIVER_LEFT_LOBE_HEPATIC_DUCT = "liver left lobe hepatic duct"
    RENAL_CALYX = "renal calyx"
    MAJOR_CALYX = "major calyx"
    MINOR_CALYX = "minor calyx"
    RENAL_CAPSULE = "renal capsule"
    GENITAL_TUBERCLE_OF_FEMALE = "genital tubercle of female"
    DUCTUS_DEFERENS = "ductus deferens"
    PENIS = "penis"
    SEMINIFEROUS_CORD = "seminiferous cord"
    INTERSTITIUM_OF_THE_TESTIS = "interstitium of the testis"
    URETHRA_OF_MALE = "urethra of male"
    URETHRAL_FOLD_OF_MALE = "urethral fold of male"
    RESPIRATORY_SYSTEM_CARTILAGE = "respiratory system cartilage"
    ARYTENOID_CARTILAGE = "arytenoid cartilage"
    CRICOID_CARTILAGE = "cricoid cartilage"
    LARYNGEAL_CARTILAGE = "laryngeal cartilage"
    THYROID_CARTILAGE = "thyroid cartilage"
    TRACHEA_CARTILAGINOUS_RING = "trachea cartilaginous ring"
    BASIOCCIPITAL_BONE = "basioccipital bone"
    BASISPHENOID_BONE = "basisphenoid bone"
    CAROTID_CANAL = "carotid canal"
    EXOCCIPITAL_BONE = "exoccipital bone"
    FORAMEN_OVALE_OF_CHONDROCRANIUM = "foramen ovale of chondrocranium"
    FORAMEN_ROTUNDUM = "foramen rotundum"
    JUGULAR_FORAMEN = "jugular foramen"
    TEMPORO_MANDIBULAR_JOINT_PRIMORDIUM = "temporo-mandibular joint primordium"
    PARIETAL_BONE = "parietal bone"
    OPTIC_FORAMEN = "optic foramen"
    ORBITAL_FISSURE = "orbital fissure"
    SUPERIOR_ORBITAL_FISSURE = "superior orbital fissure"
    ORBITO_SPHENOID = "orbito-sphenoid"
    CLAVICLE = "clavicle"
    SCAPULA = "scapula"
    MANUBRIUM_STERNI = "manubrium sterni"
    STERNEBRA = "sternebra"
    XIPHISTERNUM = "xiphisternum"
    ILIUM = "ilium"
    ISCHIUM = "ischium"
    PUBIS = "pubis"
    JEJUNUM_LUMEN = "jejunum lumen"
    FORELIMB_DIGIT_1_NAIL = "forelimb digit 1 nail"
    FORELIMB_DIGIT_2_NAIL = "forelimb digit 2 nail"
    FORELIMB_DIGIT_3_NAIL = "forelimb digit 3 nail"
    FORELIMB_DIGIT_4_NAIL = "forelimb digit 4 nail"
    FORELIMB_DIGIT_5_NAIL = "forelimb digit 5 nail"
    HINDLIMB_DIGIT_1_NAIL = "hindlimb digit 1 nail"
    HINDLIMB_DIGIT_2_NAIL = "hindlimb digit 2 nail"
    HINDLIMB_DIGIT_3_NAIL = "hindlimb digit 3 nail"
    HINDLIMB_DIGIT_4_NAIL = "hindlimb digit 4 nail"
    HINDLIMB_DIGIT_5_NAIL = "hindlimb digit 5 nail"
    AXILLARY_ARTERY = "axillary artery"
    EPIGASTRIC_ARTERY = "epigastric artery"
    INFERIOR_EPIGASTRIC_ARTERY = "inferior epigastric artery"
    SUPERIOR_EPIGASTRIC_ARTERY = "superior epigastric artery"
    INTERCOSTAL_ARTERY = "intercostal artery"
    GREATER_PALATINE_ARTERY = "greater palatine artery"
    AXILLARY_VEIN = "axillary vein"
    EPIGASTRIC_VEIN = "epigastric vein"
    INFERIOR_EPIGASTRIC_VEIN = "inferior epigastric vein"
    SUPERIOR_EPIGASTRIC_VEIN = "superior epigastric vein"
    INTERCOSTAL_VEIN = "intercostal vein"
    INFERIOR_THYROID_VEIN = "inferior thyroid vein"
    SUPERIOR_THYROID_VEIN = "superior thyroid vein"
    HEMOLYMPHOID_SYSTEM = "hemolymphoid system"
    HEMOLYMPHOID_SYSTEM_GLAND = "hemolymphoid system gland"
    SPLEEN = "spleen"
    THYMUS = "thymus"
    HAIR = "hair"
    HAIR_BULB = "hair bulb"
    HAIR_FOLLICLE = "hair follicle"
    HAIR_ROOT_SHEATH = "hair root sheath"
    PINEAL_GLAND = "pineal gland"
    ANTERIOR_HORN_OF_LATERAL_VENTRICLE_CHOROID_PLEXUS = "anterior horn of lateral ventricle choroid plexus"
    LATERAL_VENTRICLE_INFERIOR_HORN_CHOROID_PLEXUS = "lateral ventricle inferior horn choroid plexus"
    POSTERIOR_HORN_OF_LATERAL_VENTRICLE_CHOROID_PLEXUS = "posterior horn of lateral ventricle choroid plexus"
    SUPERIOR_HORN_OF_LATERAL_VENTRICLE_CHOROID_PLEXUS = "superior horn of lateral ventricle choroid plexus"
    HEAD_OF_LATERAL_VENTRICLE_CHOROID_PLEXUS = "head of lateral ventricle choroid plexus"
    TAIL_OF_LATERAL_VENTRICLE_CHOROID_PLEXUS = "tail of lateral ventricle choroid plexus"
    MEDULLA_OBLONGATA_REST_OF_BASAL_PLATE = "medulla oblongata rest of basal plate"
    MEDULLA_OBLONGATA_REST_OF_BASAL_PLATE_MANTLE_LAYER = "medulla oblongata rest of basal plate mantle layer"
    MEDULLA_OBLONGATA_REST_OF_BASAL_PLATE_MARGINAL_LAYER = "medulla oblongata rest of basal plate marginal layer"
    MEDULLA_OBLONGATA_REST_OF_BASAL_PLATE_VENTRICULAR_LAYER = "medulla oblongata rest of basal plate ventricular layer"
    MEDULLARY_RAPHE_MANTLE_LAYER = "medullary raphe mantle layer"
    MEDULLARY_RAPHE_MARGINAL_LAYER = "medullary raphe marginal layer"
    MEDULLARY_RAPHE_VENTRICULAR_LAYER = "medullary raphe ventricular layer"
    TEMPORAL_LOBE = "temporal lobe"
    TEMPORAL_LOBE_MARGINAL_LAYER = "temporal lobe marginal layer"
    TEMPORAL_LOBE_VENTRICULAR_LAYER = "temporal lobe ventricular layer"
    AXILLARY_NERVE = "axillary nerve"
    OTIC_CARTILAGE = "otic cartilage"
    DESCEMET_S_MEMBRANE = "Descemet's membrane"
    EYE_SMOOTH_MUSCLE = "eye smooth muscle"
    INTRINSIC_OCULAR_MUSCLE = "intrinsic ocular muscle"
    SUBLINGUAL_GLAND = "sublingual gland"
    SUBLINGUAL_GLAND_EPITHELIUM = "sublingual gland epithelium"
    SUBLINGUAL_GLAND_MESENCHYME = "sublingual gland mesenchyme"
    SUBMANDIBULAR_GLAND = "submandibular gland"
    SUBMANDIBULAR_GLAND_EPITHELIUM = "submandibular gland epithelium"
    SUBMANDIBULAR_GLAND_MESENCHYME = "submandibular gland mesenchyme"
    GUT_GLAND = "gut gland"
    ANAL_CANAL_EPITHELIUM = "anal canal epithelium"
    ANAL_CANAL_LUMEN = "anal canal lumen"
    ANAL_CANAL_VASCULAR_ELEMENT = "anal canal vascular element"
    DUODENUM = "duodenum"
    FILIFORM_PAPILLAE = "filiform papillae"
    PERINEAL_BODY_MUSCLE = "perineal body muscle"
    COLON = "colon"
    COLON_EPITHELIUM = "colon epithelium"
    COLON_MESENTERY = "colon mesentery"
    COLON_DORSAL_MESENTERY = "colon dorsal mesentery"
    COLON_VASCULAR_ELEMENT = "colon vascular element"
    COLON_LUMEN = "colon lumen"
    SECONDARY_PALATE = "secondary palate"
    SOFT_PALATE = "soft palate"
    SOFT_PALATE_EPITHELIUM = "soft palate epithelium"
    SOFT_PALATE_MESENCHYME = "soft palate mesenchyme"
    HARD_PALATE = "hard palate"
    HARD_PALATE_EPITHELIUM = "hard palate epithelium"
    HARD_PALATE_MESENCHYME = "hard palate mesenchyme"
    NASOPALATINE_DUCT = "nasopalatine duct"
    SUBLINGUAL_CARUNCLE = "sublingual caruncle"
    SULCUS_TERMINALIS = "sulcus terminalis"
    PHARYNX_SKELETAL_MUSCLE = "pharynx skeletal muscle"
    PHARYNGEAL_CONSTRICTOR_MUSCLE = "pharyngeal constrictor muscle"
    INFERIOR_PHARYNGEAL_CONSTRICTOR_MUSCLE = "inferior pharyngeal constrictor muscle"
    SUPERIOR_PHARYNGEAL_CONSTRICTOR_MUSCLE = "superior pharyngeal constrictor muscle"
    LOWER_JAW_GUM = "lower jaw gum"
    LOWER_JAW_GUM_EPITHELIUM = "lower jaw gum epithelium"
    LOWER_JAW_GUM_MESENCHYME = "lower jaw gum mesenchyme"
    UPPER_JAW_GUM = "upper jaw gum"
    UPPER_JAW_GUM_EPITHELIUM = "upper jaw gum epithelium"
    UPPER_JAW_GUM_MESENCHYME = "upper jaw gum mesenchyme"
    CORTICAL_RENAL_TUBULE = "cortical renal tubule"
    FEMALE_CYSTIC_VESICULAR_APPENDAGE = "female cystic vesicular appendage"
    OVIDUCT = "oviduct"
    UTERINE_HORN = "uterine horn"
    VAGINA = "vagina"
    UPPER_VAGINA = "upper vagina"
    GLANS_PENIS = "glans penis"
    PREPUCE_OF_MALE = "prepuce of male"
    EFFERENT_DUCT_OF_SEMINAL_VESICLE = "efferent duct of seminal vesicle"
    PELVIC_URETHRA_OF_MALE = "pelvic urethra of male"
    SINOVAGINAL_BULB = "sinovaginal bulb"
    LEFT_LUNG_BRONCHIOLE = "left lung bronchiole"
    LEFT_LUNG_ALVEOLAR_SYSTEM = "left lung alveolar system"
    RIGHT_LUNG_ACCESSORY_LOBE_BRONCHIOLE = "right lung accessory lobe bronchiole"
    RIGHT_LUNG_ACCESSORY_LOBE_ALVEOLAR_SYSTEM = "right lung accessory lobe alveolar system"
    RIGHT_LUNG_CAUDAL_LOBE_BRONCHIOLE = "right lung caudal lobe bronchiole"
    RIGHT_LUNG_CAUDAL_LOBE_ALVEOLAR_SYSTEM = "right lung caudal lobe alveolar system"
    RIGHT_LUNG_CRANIAL_LOBE_BRONCHIOLE = "right lung cranial lobe bronchiole"
    RIGHT_LUNG_CRANIAL_LOBE_ALVEOLAR_SYSTEM = "right lung cranial lobe alveolar system"
    RIGHT_LUNG_MIDDLE_LOBE_BRONCHIOLE = "right lung middle lobe bronchiole"
    RIGHT_LUNG_MIDDLE_LOBE_ALVEOLAR_SYSTEM = "right lung middle lobe alveolar system"
    CRUCIATE_LIGAMENT_OF_ATLAS = "cruciate ligament of atlas"
    HYPOGLOSSAL_CANAL = "hypoglossal canal"
    SPHENOIDAL_CANAL = "sphenoidal canal"
    ZYGOMATIC_PROCESS = "zygomatic process"
    FRONTAL_BONE = "frontal bone"
    INTER_PARIETAL_BONE = "inter-parietal bone"
    ETHMOID_BONE = "ethmoid bone"
    FACIAL_BONE = "facial bone"
    COSTO_VERTEBRAL_JOINT = "costo-vertebral joint"
    SYNOVIAL_CAVITY_OF_HIP_BONE = "synovial cavity of hip bone"
    TEMPORAL_LOBE_MANTLE_LAYER = "temporal lobe mantle layer"
    SCLERA = "sclera"
    PEYER_S_PATCH = "Peyer's patch"
    FIBROUS_PERICARDIUM = "fibrous pericardium"
    SEROSAL_PERICARDIUM = "serosal pericardium"
    PARIETAL_SEROSAL_PERICARDIUM = "parietal serosal pericardium"
    VISCERAL_SEROSAL_PERICARDIUM = "visceral serosal pericardium"
    INCISIVE_CANAL = "incisive canal"
    C1_VERTEBRA_HYPOCHORDAL_ARCH = "C1 vertebra hypochordal arch"
    PINEAL_GLAND_STALK = "pineal gland stalk"
    DENTATE_GYRUS = "dentate gyrus"
    ANTERIOR_COMMISSURE = "anterior commissure"
    HIPPOCAMPUS_MANTLE_LAYER = "hippocampus mantle layer"
    HIPPOCAMPUS_MARGINAL_LAYER = "hippocampus marginal layer"
    ALLOCORTEX = "allocortex"
    HIPPOCAMPUS_VENTRICULAR_LAYER = "hippocampus ventricular layer"
    ALLOCORTEX_MANTLE_LAYER = "allocortex mantle layer"
    ALLOCORTEX_MARGINAL_LAYER = "allocortex marginal layer"
    ALLOCORTEX_VENTRICULAR_LAYER = "allocortex ventricular layer"
    CEREBELLUM_DORSAL_PART = "cerebellum dorsal part"
    CEREBELLUM_DORSAL_PART_MANTLE_LAYER = "cerebellum dorsal part mantle layer"
    CEREBELLUM_DORSAL_PART_MARGINAL_LAYER = "cerebellum dorsal part marginal layer"
    CEREBELLUM_DORSAL_PART_VENTRICULAR_LAYER = "cerebellum dorsal part ventricular layer"
    TECTUM = "tectum"
    MIDBRAIN_ROOF_PLATE_CAUDAL_PART = "midbrain roof plate caudal part"
    DIENCEPHALON_SUBARACHNOID_SPACE = "diencephalon subarachnoid space"
    TELENCEPHALON_SUBARACHNOID_SPACE = "telencephalon subarachnoid space"
    MIDBRAIN_SUBARACHNOID_SPACE = "midbrain subarachnoid space"
    INTERPEDUNCULAR_CISTERN = "interpeduncular cistern"
    CISTERNA_CHIASMATICA = "cisterna chiasmatica"
    CAVERNOUS_SINUS = "cavernous sinus"
    SIGMOID_SINUS = "sigmoid sinus"
    ORGAN_OF_CORTI = "organ of Corti"
    TYMPANIC_MEMBRANE = "tympanic membrane"
    TYMPANIC_MEMBRANE_ENDOTHELIUM = "tympanic membrane endothelium"
    TYMPANIC_MEMBRANE_EPITHELIUM = "tympanic membrane epithelium"
    CILIARY_BODY = "ciliary body"
    MEDULLA_OBLONGATA_PART_OF_4TH_VENTRICLE_CHOROID_PLEXUS = "medulla oblongata part of 4th ventricle choroid plexus"
    OPTIC_CHOROID = "optic choroid"
    DUODENUM_EPITHELIUM = "duodenum epithelium"
    DUODENUM_LUMEN = "duodenum lumen"
    DUODENUM_MESENTERY = "duodenum mesentery"
    MESODUODENUM = "mesoduodenum"
    DUODENUM_VASCULAR_ELEMENT = "duodenum vascular element"
    GASTRO_ESOPHAGEAL_JUNCTION_MESENTERY = "gastro-esophageal junction mesentery"
    SUSPENSORY_LIGAMENT_OF_OVARY = "suspensory ligament of ovary"
    APPENDIX_EPIDIDYMIS = "appendix epididymis"
    APPENDIX_TESTIS = "appendix testis"
    EJACULATORY_DUCT = "ejaculatory duct"
    GUBERNACULUM_OF_MALE = "gubernaculum of male"
    LARYNX_ADITUS = "larynx aditus"
    MAIN_BRONCHUS_CARTILAGE = "main bronchus cartilage"
    CAUDATE_PUTAMEN = "caudate-putamen"
    LUMBAR_ARTERY = "lumbar artery"
    SUBCOSTAL_ARTERY = "subcostal artery"
    REST_OF_UPPER_LIP = "rest of upper lip"
    LEFT_HEPATIC_DUCT_EXTRAHEPATIC_PART = "left hepatic duct extrahepatic part"
    RIGHT_HEPATIC_DUCT_EXTRAHEPATIC_PART = "right hepatic duct extrahepatic part"
    COMMON_HEPATIC_DUCT = "common hepatic duct"
    ELBOW_JOINT = "elbow joint"
    RADIUS = "radius"
    ULNA = "ulna"
    SHOULDER_JOINT = "shoulder joint"
    HUMERUS = "humerus"
    UPPER_ARM_MUSCLE = "upper arm muscle"
    BICEPS_BRACHII = "biceps brachii"
    BRACHIALIS = "brachialis"
    TERES_MAJOR = "teres major"
    TRICEPS_BRACHII = "triceps brachii"
    FORELIMB_DIGIT_1_METACARPAL = "forelimb digit 1 metacarpal"
    FORELIMB_DIGIT_1_PHALANX = "forelimb digit 1 phalanx"
    FORELIMB_DIGIT_2_METACARPAL = "forelimb digit 2 metacarpal"
    FORELIMB_DIGIT_2_PHALANX = "forelimb digit 2 phalanx"
    FORELIMB_DIGIT_3_METACARPAL = "forelimb digit 3 metacarpal"
    FORELIMB_DIGIT_3_PHALANX = "forelimb digit 3 phalanx"
    FORELIMB_DIGIT_4_METACARPAL = "forelimb digit 4 metacarpal"
    FORELIMB_DIGIT_4_PHALANX = "forelimb digit 4 phalanx"
    FORELIMB_DIGIT_5_METACARPAL = "forelimb digit 5 metacarpal"
    FORELIMB_DIGIT_5_PHALANX = "forelimb digit 5 phalanx"
    CARPUS = "carpus"
    HINDLIMB_DIGIT_1_METATARSAL = "hindlimb digit 1 metatarsal"
    HINDLIMB_DIGIT_1_PHALANX = "hindlimb digit 1 phalanx"
    HINDLIMB_DIGIT_2_METATARSAL = "hindlimb digit 2 metatarsal"
    HINDLIMB_DIGIT_2_PHALANX = "hindlimb digit 2 phalanx"
    HINDLIMB_DIGIT_3_METATARSAL = "hindlimb digit 3 metatarsal"
    HINDLIMB_DIGIT_3_PHALANX = "hindlimb digit 3 phalanx"
    HINDLIMB_DIGIT_4_METATARSAL = "hindlimb digit 4 metatarsal"
    HINDLIMB_DIGIT_4_PHALANX = "hindlimb digit 4 phalanx"
    HINDLIMB_DIGIT_5_METATARSAL = "hindlimb digit 5 metatarsal"
    HINDLIMB_DIGIT_5_PHALANX = "hindlimb digit 5 phalanx"
    TARSUS = "tarsus"
    CALCANEUS = "calcaneus"
    TALUS = "talus"
    HIP_JOINT = "hip joint"
    KNEE_JOINT = "knee joint"
    FABELLA_CARTILAGE_CONDENSATION = "fabella cartilage condensation"
    PATELLA_CARTILAGE_CONDENSATION = "patella cartilage condensation"
    FIBULA = "fibula"
    TIBIA = "tibia"
    FEMUR = "femur"
    UPPER_LEG_MUSCLE = "upper leg muscle"
    HAMSTRING_MUSCLE = "hamstring muscle"
    QUADRICEPS_FEMORIS_MUSCLE = "quadriceps femoris muscle"
    LENTICULAR_NUCLEUS = "lenticular nucleus"
    ORBITAL_FAT_PAD = "orbital fat pad"
    HARDERIAN_GLAND = "Harderian gland"
    IRIS = "iris"
    RETINA_INNER_NUCLEAR_LAYER = "retina inner nuclear layer"
    NEURAL_RETINA_INTERMEDIATE_ANUCLEAR_LAYER = "neural retina intermediate anuclear layer"
    RETINA_OUTER_NUCLEAR_LAYER = "retina outer nuclear layer"
    PECTINATE_LINE = "pectinate line"
    CORONARY_ARTERY = "coronary artery"
    MANDIBULAR_CANAL = "mandibular canal"
    GLUTEUS_MEDIUS = "gluteus medius"
    ILIACUS = "iliacus"
    LEVATOR_ANI = "levator ani"
    ADDUCTOR_MUSCLE_OF_LEG = "adductor muscle of leg"
    TRIGEMINAL_NERVE_LINGUAL_BRANCH = "trigeminal nerve lingual branch"
    GLANS_OF_FEMALE_GENITAL_TUBERCLE = "glans of female genital tubercle"
    FEMALE_INGUINAL_CANAL = "female inguinal canal"
    FEMALE_INGUINAL_RING = "female inguinal ring"
    FEMALE_INTERNAL_INGUINAL_RING = "female internal inguinal ring"
    FEMALE_EXTERNAL_INGUINAL_RING = "female external inguinal ring"
    REST_OF_INTERSTITIUM_OF_THE_TESTIS = "rest of interstitium of the testis"
    TUNICA_VAGINALIS_TESTIS = "tunica vaginalis testis"
    SEMINAL_VESICLE = "seminal vesicle"
    VOCAL_FOLDS = "vocal folds"
    LEFT_LUNG_ALVEOLUS = "left lung alveolus"
    LEFT_LUNG_ALVEOLAR_DUCT = "left lung alveolar duct"
    RIGHT_LUNG_MIDDLE_LOBE_ALVEOLUS = "right lung middle lobe alveolus"
    RIGHT_LUNG_MIDDLE_LOBE_ALVEOLAR_DUCT = "right lung middle lobe alveolar duct"
    RIGHT_LUNG_CRANIAL_LOBE_ALVEOLUS = "right lung cranial lobe alveolus"
    RIGHT_LUNG_CRANIAL_LOBE_ALVEOLAR_DUCT = "right lung cranial lobe alveolar duct"
    RIGHT_LUNG_CAUDAL_LOBE_ALVEOLUS = "right lung caudal lobe alveolus"
    RIGHT_LUNG_CAUDAL_LOBE_ALVEOLAR_DUCT = "right lung caudal lobe alveolar duct"
    RIGHT_LUNG_ACCESSORY_LOBE_ALVEOLUS = "right lung accessory lobe alveolus"
    RIGHT_LUNG_ACCESSORY_LOBE_ALVEOLAR_DUCT = "right lung accessory lobe alveolar duct"
    TRACHEA_SMOOTH_MUSCLE = "trachea smooth muscle"
    TRACHEALIS = "trachealis"
    CRANIOPHARYNGEAL_CANAL = "craniopharyngeal canal"
    PITUITARY_FOSSA = "pituitary fossa"
    TEMPORO_MANDIBULAR_JOINT = "temporo-mandibular joint"
    TYMPANIC_RING = "tympanic ring"
    SUTURE = "suture"
    ANTERIOR_FONTANELLE = "anterior fontanelle"
    POSTERIOR_FONTANELLE = "posterior fontanelle"
    ETHMOID_BONE_CRIBRIFORM_PLATE = "ethmoid bone cribriform plate"
    NASAL_BONE = "nasal bone"
    STERNO_CLAVICULAR_JOINT = "sterno-clavicular joint"
    FORELIMB_JOINT = "forelimb joint"
    RADIO_CARPAL_JOINT = "radio-carpal joint"
    RADIO_ULNAR_JOINT = "radio-ulnar joint"
    LIGAMENTUM_NUCHAE = "ligamentum nuchae"
    BROWN_FAT = "brown fat"
    EPICRANIAL_APONEUROSIS = "epicranial aponeurosis"
    CAROTID_BODY = "carotid body"
    CELIAC_TRUNK = "celiac trunk"
    FACIAL_ARTERY = "facial artery"
    MEDIAN_SACRAL_ARTERY = "median sacral artery"
    MUSCULO_PHRENIC_ARTERY = "musculo-phrenic artery"
    SPLENIC_ARTERY = "splenic artery"
    ACCESSORY_HEMIAZYGOS_VEIN = "accessory hemiazygos vein"
    DEEP_DORSAL_VEIN_OF_CLITORIS = "deep dorsal vein of clitoris"
    DEEP_DORSAL_VEIN_OF_PENIS = "deep dorsal vein of penis"
    FACIAL_VEIN = "facial vein"
    HEPATIC_VEIN = "hepatic vein"
    MUSCULO_PHRENIC_VEIN = "musculo-phrenic vein"
    SAGITTAL_SUTURE = "sagittal suture"
    CORONAL_SUTURE = "coronal suture"
    FRONTAL_SUTURE = "frontal suture"
    LAMBDOID_SUTURE = "lambdoid suture"
    SQUAMO_PARIETAL_SUTURE = "squamo-parietal suture"
    SPHENOIDAL_FONTANELLE = "sphenoidal fontanelle"
    INFERIOR_PETROSAL_SINUS = "inferior petrosal sinus"
    PINEAL_VENOUS_PLEXUS = "pineal venous plexus"
    ARCUATE_LIGAMENT = "arcuate ligament"
    MEDIAL_ARCUATE_LIGAMENT = "medial arcuate ligament"
    LATERAL_ARCUATE_LIGAMENT = "lateral arcuate ligament"
    FENESTRA_COCHLEAE = "fenestra cochleae"
    FENESTRA_VESTIBULI = "fenestra vestibuli"
    INFERIOR_OLIVARY_NUCLEUS = "inferior olivary nucleus"
    HYALOID_ARTERY = "hyaloid artery"
    LENS_EMBRYONIC_NUCLEUS = "lens embryonic nucleus"
    LENS_POSTERIOR_SUTURE = "lens posterior suture"
    LACRIMAL_DRAINAGE_SYSTEM = "lacrimal drainage system"
    PANCREAS_BODY_ISLETS_OF_LANGERHANS = "pancreas body islets of Langerhans"
    PANCREAS_HEAD_ISLETS_OF_LANGERHANS = "pancreas head islets of Langerhans"
    PANCREAS_TAIL_ISLETS_OF_LANGERHANS = "pancreas tail islets of Langerhans"
    PHARYNGEAL_LIGAMENT = "pharyngeal ligament"
    PHARYNGEAL_TUBERCLE = "pharyngeal tubercle"
    PYLORIC_SPHINCTER = "pyloric sphincter"
    LARGE_INTESTINE = "large intestine"
    MIDGUT_DERIVED_LARGE_INTESTINE = "midgut derived large intestine"
    HINDGUT_DERIVED_LARGE_INTESTINE = "hindgut derived large intestine"
    MIDGUT_DERIVED_LARGE_INTESTINE_MESENCHYME = "midgut derived large intestine mesenchyme"
    HINDGUT_DERIVED_LARGE_INTESTINE_MESENCHYME = "hindgut derived large intestine mesenchyme"
    MIDGUT_DERIVED_LARGE_INTESTINE_EPITHELIUM = "midgut derived large intestine epithelium"
    MIDGUT_DERIVED_LARGE_INTESTINE_LUMEN = "midgut derived large intestine lumen"
    MIDGUT_DERIVED_LARGE_INTESTINE_MESENTERY = "midgut derived large intestine mesentery"
    MIDGUT_DERIVED_LARGE_INTESTINE_DORSAL_MESENTERY = "midgut derived large intestine dorsal mesentery"
    MIDGUT_DERIVED_LARGE_INTESTINE_VASCULAR_ELEMENT = "midgut derived large intestine vascular element"
    HINDGUT_DERIVED_LARGE_INTESTINE_EPITHELIUM = "hindgut derived large intestine epithelium"
    HINDGUT_DERIVED_LARGE_INTESTINE_LUMEN = "hindgut derived large intestine lumen"
    HINDGUT_DERIVED_LARGE_INTESTINE_MESENTERY = "hindgut derived large intestine mesentery"
    HINDGUT_DERIVED_LARGE_INTESTINE_DORSAL_MESENTERY = "hindgut derived large intestine dorsal mesentery"
    HINDGUT_DERIVED_LARGE_INTESTINE_VASCULAR_ELEMENT = "hindgut derived large intestine vascular element"
    TRANSVERSUS_THORACIS = "transversus thoracis"
    INFRAHYOID_MUSCLE = "infrahyoid muscle"
    CRICOTHYROID = "cricothyroid"
    STERNOHYOID_MUSCLE = "sternohyoid muscle"
    STERNOTHYROID_MUSCLE = "sternothyroid muscle"
    THYROHYOID_MUSCLE = "thyrohyoid muscle"
    SUPERIOR_HYPOGASTRIC_PLEXUS = "superior hypogastric plexus"
    INFERIOR_HYPOGASTRIC_PLEXUS = "inferior hypogastric plexus"
    SUPERIOR_NASAL_MEATUS = "superior nasal meatus"
    MIDDLE_NASAL_MEATUS = "middle nasal meatus"
    INFERIOR_NASAL_MEATUS = "inferior nasal meatus"
    RENAL_MEDULLA = "renal medulla"
    LOOP_OF_HENLE = "loop of Henle"
    MALE_ACCESSORY_GLAND = "male accessory gland"
    BULBOURETHRAL_GLAND_OF_MALE = "bulbourethral gland of male"
    PROSTATE_GLAND = "prostate gland"
    TUNICA_ALBUGINEA_OF_TESTIS = "tunica albuginea of testis"
    EPIDIDYMIS = "epididymis"
    CAPUT_EPIDIDYMIS = "caput epididymis"
    CAUDA_EPIDIDYMIS = "cauda epididymis"
    CORPUS_EPIDIDYMIS = "corpus epididymis"
    PENILE_BODY = "penile body"
    SCROTUM = "scrotum"
    SACRO_ILIAC_JOINT = "sacro-iliac joint"
    METACARPAL_PHALANGEAL_JOINT = "metacarpal-phalangeal joint"
    SPLEEN_HILUM = "spleen hilum"
    SPLEEN_MEDULLARY_REGION = "spleen medullary region"
    THYMUS_MEDULLARY_CORE = "thymus medullary core"
    THYMUS_THIN_FIBROUS_CAPSULE = "thymus thin fibrous capsule"
    THYROID_GLAND_LEFT_LOBE_COLLOID_FILLED_FOLLICLES = "thyroid gland left lobe colloid-filled follicles"
    THYROID_GLAND_RIGHT_LOBE_COLLOID_FILLED_FOLLICLES = "thyroid gland right lobe colloid-filled follicles"
    SUPERIOR_CEREBELLAR_ARTERY = "superior cerebellar artery"
    THORACIC_DUCT = "thoracic duct"
    GREAT_CEREBRAL_VEIN_OF_GALEN = "great cerebral vein of Galen"
    LINGUAL_VEIN = "lingual vein"
    OPHTHALMIC_VEIN = "ophthalmic vein"
    GREATER_PALATINE_VEIN = "greater palatine vein"
    SUPERFICIAL_TEMPORAL_VEIN = "superficial temporal vein"
    DISTAL_RADIO_ULNAR_JOINT = "distal radio-ulnar joint"
    PROXIMAL_RADIO_ULNAR_JOINT = "proximal radio-ulnar joint"
    LOWER_LEG_MUSCLE = "lower leg muscle"
    FLEXOR_DIGITORUM_PROFUNDUS = "flexor digitorum profundus"
    CAROTID_SINUS = "carotid sinus"
    CELIAC_GANGLION = "celiac ganglion"
    ANNULAR_LIGAMENT = "annular ligament"
    C1_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "C1 vertebral pre-cartilage condensation"
    C2_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "C2 vertebral pre-cartilage condensation"
    C3_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "C3 vertebral pre-cartilage condensation"
    C4_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "C4 vertebral pre-cartilage condensation"
    C5_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "C5 vertebral pre-cartilage condensation"
    C6_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "C6 vertebral pre-cartilage condensation"
    C7_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "C7 vertebral pre-cartilage condensation"
    T1_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T1 vertebral pre-cartilage condensation"
    T2_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T2 vertebral pre-cartilage condensation"
    T3_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T3 vertebral pre-cartilage condensation"
    T4_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T4 vertebral pre-cartilage condensation"
    T5_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T5 vertebral pre-cartilage condensation"
    C1_VERTEBRAL_CARTILAGE_CONDENSATION = "C1 vertebral cartilage condensation"
    C2_VERTEBRAL_CARTILAGE_CONDENSATION = "C2 vertebral cartilage condensation"
    C3_VERTEBRAL_CARTILAGE_CONDENSATION = "C3 vertebral cartilage condensation"
    C4_VERTEBRAL_CARTILAGE_CONDENSATION = "C4 vertebral cartilage condensation"
    C5_VERTEBRAL_CARTILAGE_CONDENSATION = "C5 vertebral cartilage condensation"
    L1_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "L1 vertebral pre-cartilage condensation"
    T6_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T6 vertebral pre-cartilage condensation"
    T7_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T7 vertebral pre-cartilage condensation"
    T8_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T8 vertebral pre-cartilage condensation"
    T9_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T9 vertebral pre-cartilage condensation"
    T10_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T10 vertebral pre-cartilage condensation"
    T11_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T11 vertebral pre-cartilage condensation"
    T12_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T12 vertebral pre-cartilage condensation"
    T13_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "T13 vertebral pre-cartilage condensation"
    C6_VERTEBRAL_CARTILAGE_CONDENSATION = "C6 vertebral cartilage condensation"
    C7_VERTEBRAL_CARTILAGE_CONDENSATION = "C7 vertebral cartilage condensation"
    L2_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "L2 vertebral pre-cartilage condensation"
    L3_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "L3 vertebral pre-cartilage condensation"
    L4_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "L4 vertebral pre-cartilage condensation"
    L5_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "L5 vertebral pre-cartilage condensation"
    L6_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "L6 vertebral pre-cartilage condensation"
    S1_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "S1 vertebral pre-cartilage condensation"
    S2_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "S2 vertebral pre-cartilage condensation"
    T1_VERTEBRAL_CARTILAGE_CONDENSATION = "T1 vertebral cartilage condensation"
    T2_VERTEBRAL_CARTILAGE_CONDENSATION = "T2 vertebral cartilage condensation"
    T3_VERTEBRAL_CARTILAGE_CONDENSATION = "T3 vertebral cartilage condensation"
    T4_VERTEBRAL_CARTILAGE_CONDENSATION = "T4 vertebral cartilage condensation"
    T5_VERTEBRAL_CARTILAGE_CONDENSATION = "T5 vertebral cartilage condensation"
    L1_VERTEBRAL_CARTILAGE_CONDENSATION = "L1 vertebral cartilage condensation"
    S3_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "S3 vertebral pre-cartilage condensation"
    S4_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "S4 vertebral pre-cartilage condensation"
    TL1_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl1 vertebral pre-cartilage condensation"
    TL2_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl2 vertebral pre-cartilage condensation"
    TL3_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl3 vertebral pre-cartilage condensation"
    T1_RIB_PRE_CARTILAGE_CONDENSATION = "T1 rib pre-cartilage condensation"
    T2_RIB_PRE_CARTILAGE_CONDENSATION = "T2 rib pre-cartilage condensation"
    T3_RIB_PRE_CARTILAGE_CONDENSATION = "T3 rib pre-cartilage condensation"
    T4_RIB_PRE_CARTILAGE_CONDENSATION = "T4 rib pre-cartilage condensation"
    T5_RIB_PRE_CARTILAGE_CONDENSATION = "T5 rib pre-cartilage condensation"
    T6_VERTEBRAL_CARTILAGE_CONDENSATION = "T6 vertebral cartilage condensation"
    T7_VERTEBRAL_CARTILAGE_CONDENSATION = "T7 vertebral cartilage condensation"
    T8_VERTEBRAL_CARTILAGE_CONDENSATION = "T8 vertebral cartilage condensation"
    T9_VERTEBRAL_CARTILAGE_CONDENSATION = "T9 vertebral cartilage condensation"
    T10_VERTEBRAL_CARTILAGE_CONDENSATION = "T10 vertebral cartilage condensation"
    T11_VERTEBRAL_CARTILAGE_CONDENSATION = "T11 vertebral cartilage condensation"
    T12_VERTEBRAL_CARTILAGE_CONDENSATION = "T12 vertebral cartilage condensation"
    T13_VERTEBRAL_CARTILAGE_CONDENSATION = "T13 vertebral cartilage condensation"
    L2_VERTEBRAL_CARTILAGE_CONDENSATION = "L2 vertebral cartilage condensation"
    L3_VERTEBRAL_CARTILAGE_CONDENSATION = "L3 vertebral cartilage condensation"
    L4_VERTEBRAL_CARTILAGE_CONDENSATION = "L4 vertebral cartilage condensation"
    L5_VERTEBRAL_CARTILAGE_CONDENSATION = "L5 vertebral cartilage condensation"
    L6_VERTEBRAL_CARTILAGE_CONDENSATION = "L6 vertebral cartilage condensation"
    S1_VERTEBRAL_CARTILAGE_CONDENSATION = "S1 vertebral cartilage condensation"
    S2_VERTEBRAL_CARTILAGE_CONDENSATION = "S2 vertebral cartilage condensation"
    TL4_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl4 vertebral pre-cartilage condensation"
    TL5_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl5 vertebral pre-cartilage condensation"
    TL6_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl6 vertebral pre-cartilage condensation"
    TL7_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl7 vertebral pre-cartilage condensation"
    TL8_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl8 vertebral pre-cartilage condensation"
    T6_RIB_PRE_CARTILAGE_CONDENSATION = "T6 rib pre-cartilage condensation"
    T7_RIB_PRE_CARTILAGE_CONDENSATION = "T7 rib pre-cartilage condensation"
    T8_RIB_PRE_CARTILAGE_CONDENSATION = "T8 rib pre-cartilage condensation"
    T9_RIB_PRE_CARTILAGE_CONDENSATION = "T9 rib pre-cartilage condensation"
    T10_RIB_PRE_CARTILAGE_CONDENSATION = "T10 rib pre-cartilage condensation"
    T11_RIB_PRE_CARTILAGE_CONDENSATION = "T11 rib pre-cartilage condensation"
    T12_RIB_PRE_CARTILAGE_CONDENSATION = "T12 rib pre-cartilage condensation"
    T13_RIB_PRE_CARTILAGE_CONDENSATION = "T13 rib pre-cartilage condensation"
    S3_VERTEBRAL_CARTILAGE_CONDENSATION = "S3 vertebral cartilage condensation"
    S4_VERTEBRAL_CARTILAGE_CONDENSATION = "S4 vertebral cartilage condensation"
    TL1_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl1 vertebral cartilage condensation"
    TL2_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl2 vertebral cartilage condensation"
    TL3_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl3 vertebral cartilage condensation"
    TL9_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl9 vertebral pre-cartilage condensation"
    TL10_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl10 vertebral pre-cartilage condensation"
    TL11_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl11 vertebral pre-cartilage condensation"
    T1_RIB_CARTILAGE_CONDENSATION = "T1 rib cartilage condensation"
    T2_RIB_CARTILAGE_CONDENSATION = "T2 rib cartilage condensation"
    T3_RIB_CARTILAGE_CONDENSATION = "T3 rib cartilage condensation"
    T4_RIB_CARTILAGE_CONDENSATION = "T4 rib cartilage condensation"
    T5_RIB_CARTILAGE_CONDENSATION = "T5 rib cartilage condensation"
    CERVICAL_INTERVERTEBRAL_DISC = "cervical intervertebral disc"
    C1_INTERVERTEBRAL_DISC = "C1 intervertebral disc"
    C1_NUCLEUS_PULPOSUS = "C1 nucleus pulposus"
    C1_ANNULUS_FIBROSUS = "C1 annulus fibrosus"
    C2_INTERVERTEBRAL_DISC = "C2 intervertebral disc"
    C2_NUCLEUS_PULPOSUS = "C2 nucleus pulposus"
    C2_ANNULUS_FIBROSUS = "C2 annulus fibrosus"
    C3_INTERVERTEBRAL_DISC = "C3 intervertebral disc"
    C3_NUCLEUS_PULPOSUS = "C3 nucleus pulposus"
    C3_ANNULUS_FIBROSUS = "C3 annulus fibrosus"
    C4_INTERVERTEBRAL_DISC = "C4 intervertebral disc"
    C4_NUCLEUS_PULPOSUS = "C4 nucleus pulposus"
    C4_ANNULUS_FIBROSUS = "C4 annulus fibrosus"
    C5_INTERVERTEBRAL_DISC = "C5 intervertebral disc"
    C5_NUCLEUS_PULPOSUS = "C5 nucleus pulposus"
    C5_ANNULUS_FIBROSUS = "C5 annulus fibrosus"
    C1_VERTEBRA = "C1 vertebra"
    C2_VERTEBRA = "C2 vertebra"
    C3_VERTEBRA = "C3 vertebra"
    C4_VERTEBRA = "C4 vertebra"
    C5_VERTEBRA = "C5 vertebra"
    TL4_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl4 vertebral cartilage condensation"
    TL5_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl5 vertebral cartilage condensation"
    TL6_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl6 vertebral cartilage condensation"
    TL7_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl7 vertebral cartilage condensation"
    TL8_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl8 vertebral cartilage condensation"
    TL9_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl9 vertebral cartilage condensation"
    TL10_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl10 vertebral cartilage condensation"
    TL11_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl11 vertebral cartilage condensation"
    TL12_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl12 vertebral pre-cartilage condensation"
    TL13_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl13 vertebral pre-cartilage condensation"
    TL14_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl14 vertebral pre-cartilage condensation"
    TL15_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl15 vertebral pre-cartilage condensation"
    TL16_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl16 vertebral pre-cartilage condensation"
    TL17_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl17 vertebral pre-cartilage condensation"
    TL18_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl18 vertebral pre-cartilage condensation"
    TL19_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl19 vertebral pre-cartilage condensation"
    T6_RIB_CARTILAGE_CONDENSATION = "T6 rib cartilage condensation"
    T7_RIB_CARTILAGE_CONDENSATION = "T7 rib cartilage condensation"
    T8_RIB_CARTILAGE_CONDENSATION = "T8 rib cartilage condensation"
    T9_RIB_CARTILAGE_CONDENSATION = "T9 rib cartilage condensation"
    T10_RIB_CARTILAGE_CONDENSATION = "T10 rib cartilage condensation"
    T11_RIB_CARTILAGE_CONDENSATION = "T11 rib cartilage condensation"
    T12_RIB_CARTILAGE_CONDENSATION = "T12 rib cartilage condensation"
    T13_RIB_CARTILAGE_CONDENSATION = "T13 rib cartilage condensation"
    C6_INTERVERTEBRAL_DISC = "C6 intervertebral disc"
    C6_NUCLEUS_PULPOSUS = "C6 nucleus pulposus"
    C6_ANNULUS_FIBROSUS = "C6 annulus fibrosus"
    C7_INTERVERTEBRAL_DISC = "C7 intervertebral disc"
    C7_NUCLEUS_PULPOSUS = "C7 nucleus pulposus"
    C7_ANNULUS_FIBROSUS = "C7 annulus fibrosus"
    C6_VERTEBRA = "C6 vertebra"
    C7_VERTEBRA = "C7 vertebra"
    LUMBAR_INTERVERTEBRAL_DISC = "lumbar intervertebral disc"
    L1_INTERVERTEBRAL_DISC = "L1 intervertebral disc"
    L1_NUCLEUS_PULPOSUS = "L1 nucleus pulposus"
    L1_ANNULUS_FIBROSUS = "L1 annulus fibrosus"
    L1_VERTEBRA = "L1 vertebra"
    TL12_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl12 vertebral cartilage condensation"
    TL13_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl13 vertebral cartilage condensation"
    TL14_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl14 vertebral cartilage condensation"
    TL15_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl15 vertebral cartilage condensation"
    TL16_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl16 vertebral cartilage condensation"
    TL17_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl17 vertebral cartilage condensation"
    TL18_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl18 vertebral cartilage condensation"
    TL19_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl19 vertebral cartilage condensation"
    TL20_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl20 vertebral pre-cartilage condensation"
    TL21_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl21 vertebral pre-cartilage condensation"
    TL22_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl22 vertebral pre-cartilage condensation"
    TL23_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl23 vertebral pre-cartilage condensation"
    TL24_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl24 vertebral pre-cartilage condensation"
    TL25_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl25 vertebral pre-cartilage condensation"
    TL26_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl26 vertebral pre-cartilage condensation"
    TL27_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl27 vertebral pre-cartilage condensation"
    THORACIC_INTERVERTEBRAL_DISC = "thoracic intervertebral disc"
    T1_INTERVERTEBRAL_DISC = "T1 intervertebral disc"
    T1_NUCLEUS_PULPOSUS = "T1 nucleus pulposus"
    T1_ANNULUS_FIBROSUS = "T1 annulus fibrosus"
    T2_INTERVERTEBRAL_DISC = "T2 intervertebral disc"
    T2_NUCLEUS_PULPOSUS = "T2 nucleus pulposus"
    T2_ANNULUS_FIBROSUS = "T2 annulus fibrosus"
    T3_INTERVERTEBRAL_DISC = "T3 intervertebral disc"
    T3_NUCLEUS_PULPOSUS = "T3 nucleus pulposus"
    T3_ANNULUS_FIBROSUS = "T3 annulus fibrosus"
    T4_INTERVERTEBRAL_DISC = "T4 intervertebral disc"
    T4_NUCLEUS_PULPOSUS = "T4 nucleus pulposus"
    T4_ANNULUS_FIBROSUS = "T4 annulus fibrosus"
    T5_INTERVERTEBRAL_DISC = "T5 intervertebral disc"
    T5_NUCLEUS_PULPOSUS = "T5 nucleus pulposus"
    T5_ANNULUS_FIBROSUS = "T5 annulus fibrosus"
    T6_INTERVERTEBRAL_DISC = "T6 intervertebral disc"
    T6_NUCLEUS_PULPOSUS = "T6 nucleus pulposus"
    T6_ANNULUS_FIBROSUS = "T6 annulus fibrosus"
    T7_INTERVERTEBRAL_DISC = "T7 intervertebral disc"
    T7_NUCLEUS_PULPOSUS = "T7 nucleus pulposus"
    T7_ANNULUS_FIBROSUS = "T7 annulus fibrosus"
    T8_INTERVERTEBRAL_DISC = "T8 intervertebral disc"
    T8_NUCLEUS_PULPOSUS = "T8 nucleus pulposus"
    T8_ANNULUS_FIBROSUS = "T8 annulus fibrosus"
    T9_INTERVERTEBRAL_DISC = "T9 intervertebral disc"
    T9_NUCLEUS_PULPOSUS = "T9 nucleus pulposus"
    T9_ANNULUS_FIBROSUS = "T9 annulus fibrosus"
    T10_INTERVERTEBRAL_DISC = "T10 intervertebral disc"
    T10_NUCLEUS_PULPOSUS = "T10 nucleus pulposus"
    T10_ANNULUS_FIBROSUS = "T10 annulus fibrosus"
    T11_INTERVERTEBRAL_DISC = "T11 intervertebral disc"
    T11_NUCLEUS_PULPOSUS = "T11 nucleus pulposus"
    T11_ANNULUS_FIBROSUS = "T11 annulus fibrosus"
    T12_INTERVERTEBRAL_DISC = "T12 intervertebral disc"
    T12_NUCLEUS_PULPOSUS = "T12 nucleus pulposus"
    T12_ANNULUS_FIBROSUS = "T12 annulus fibrosus"
    T13_INTERVERTEBRAL_DISC = "T13 intervertebral disc"
    T13_NUCLEUS_PULPOSUS = "T13 nucleus pulposus"
    T13_ANNULUS_FIBROSUS = "T13 annulus fibrosus"
    T1_RIB = "T1 rib"
    T2_RIB = "T2 rib"
    T3_RIB = "T3 rib"
    T4_RIB = "T4 rib"
    T5_RIB = "T5 rib"
    T6_RIB = "T6 rib"
    T7_RIB = "T7 rib"
    T8_RIB = "T8 rib"
    T9_RIB = "T9 rib"
    T10_RIB = "T10 rib"
    T11_RIB = "T11 rib"
    T12_RIB = "T12 rib"
    T13_RIB = "T13 rib"
    T1_VERTEBRA = "T1 vertebra"
    T2_VERTEBRA = "T2 vertebra"
    T3_VERTEBRA = "T3 vertebra"
    T4_VERTEBRA = "T4 vertebra"
    T5_VERTEBRA = "T5 vertebra"
    T6_VERTEBRA = "T6 vertebra"
    T7_VERTEBRA = "T7 vertebra"
    T8_VERTEBRA = "T8 vertebra"
    T9_VERTEBRA = "T9 vertebra"
    T10_VERTEBRA = "T10 vertebra"
    T11_VERTEBRA = "T11 vertebra"
    T12_VERTEBRA = "T12 vertebra"
    T13_VERTEBRA = "T13 vertebra"
    L2_INTERVERTEBRAL_DISC = "L2 intervertebral disc"
    L2_NUCLEUS_PULPOSUS = "L2 nucleus pulposus"
    L2_ANNULUS_FIBROSUS = "L2 annulus fibrosus"
    L3_INTERVERTEBRAL_DISC = "L3 intervertebral disc"
    L3_NUCLEUS_PULPOSUS = "L3 nucleus pulposus"
    L3_ANNULUS_FIBROSUS = "L3 annulus fibrosus"
    L4_INTERVERTEBRAL_DISC = "L4 intervertebral disc"
    L4_NUCLEUS_PULPOSUS = "L4 nucleus pulposus"
    L4_ANNULUS_FIBROSUS = "L4 annulus fibrosus"
    L5_INTERVERTEBRAL_DISC = "L5 intervertebral disc"
    L5_NUCLEUS_PULPOSUS = "L5 nucleus pulposus"
    L5_ANNULUS_FIBROSUS = "L5 annulus fibrosus"
    L6_INTERVERTEBRAL_DISC = "L6 intervertebral disc"
    L6_NUCLEUS_PULPOSUS = "L6 nucleus pulposus"
    L6_ANNULUS_FIBROSUS = "L6 annulus fibrosus"
    L2_VERTEBRA = "L2 vertebra"
    L3_VERTEBRA = "L3 vertebra"
    L4_VERTEBRA = "L4 vertebra"
    L5_VERTEBRA = "L5 vertebra"
    L6_VERTEBRA = "L6 vertebra"
    SACRAL_INTERVERTEBRAL_DISC = "sacral intervertebral disc"
    S1_INTERVERTEBRAL_DISC = "S1 intervertebral disc"
    S1_NUCLEUS_PULPOSUS = "S1 nucleus pulposus"
    S1_ANNULUS_FIBROSUS = "S1 annulus fibrosus"
    S2_INTERVERTEBRAL_DISC = "S2 intervertebral disc"
    S2_NUCLEUS_PULPOSUS = "S2 nucleus pulposus"
    S2_ANNULUS_FIBROSUS = "S2 annulus fibrosus"
    S3_INTERVERTEBRAL_DISC = "S3 intervertebral disc"
    S3_NUCLEUS_PULPOSUS = "S3 nucleus pulposus"
    S3_ANNULUS_FIBROSUS = "S3 annulus fibrosus"
    S4_INTERVERTEBRAL_DISC = "S4 intervertebral disc"
    S4_NUCLEUS_PULPOSUS = "S4 nucleus pulposus"
    S4_ANNULUS_FIBROSUS = "S4 annulus fibrosus"
    S1_VERTEBRA = "S1 vertebra"
    S2_VERTEBRA = "S2 vertebra"
    S3_VERTEBRA = "S3 vertebra"
    S4_VERTEBRA = "S4 vertebra"
    TAIL_INTERVERTEBRAL_DISC = "tail intervertebral disc"
    TL1_INTERVERTEBRAL_DISC = "Tl1 intervertebral disc"
    TL1_NUCLEUS_PULPOSUS = "Tl1 nucleus pulposus"
    TL1_ANNULUS_FIBROSUS = "Tl1 annulus fibrosus"
    TL2_INTERVERTEBRAL_DISC = "Tl2 intervertebral disc"
    TL2_NUCLEUS_PULPOSUS = "Tl2 nucleus pulposus"
    TL2_ANNULUS_FIBROSUS = "Tl2 annulus fibrosus"
    TL3_INTERVERTEBRAL_DISC = "Tl3 intervertebral disc"
    TL3_NUCLEUS_PULPOSUS = "Tl3 nucleus pulposus"
    TL3_ANNULUS_FIBROSUS = "Tl3 annulus fibrosus"
    TL1_VERTEBRA = "Tl1 vertebra"
    TL2_VERTEBRA = "Tl2 vertebra"
    TL3_VERTEBRA = "Tl3 vertebra"
    TL20_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl20 vertebral cartilage condensation"
    TL21_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl21 vertebral cartilage condensation"
    TL22_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl22 vertebral cartilage condensation"
    TL23_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl23 vertebral cartilage condensation"
    TL24_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl24 vertebral cartilage condensation"
    TL25_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl25 vertebral cartilage condensation"
    TL26_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl26 vertebral cartilage condensation"
    TL27_VERTEBRAL_CARTILAGE_CONDENSATION = "Tl27 vertebral cartilage condensation"
    TL28_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl28 vertebral pre-cartilage condensation"
    TL29_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl29 vertebral pre-cartilage condensation"
    TL30_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl30 vertebral pre-cartilage condensation"
    TL31_VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "Tl31 vertebral pre-cartilage condensation"
    TL4_INTERVERTEBRAL_DISC = "Tl4 intervertebral disc"
    TL4_NUCLEUS_PULPOSUS = "Tl4 nucleus pulposus"
    TL4_ANNULUS_FIBROSUS = "Tl4 annulus fibrosus"
    TL5_INTERVERTEBRAL_DISC = "Tl5 intervertebral disc"
    TL5_NUCLEUS_PULPOSUS = "Tl5 nucleus pulposus"
    TL5_ANNULUS_FIBROSUS = "Tl5 annulus fibrosus"
    TL6_INTERVERTEBRAL_DISC = "Tl6 intervertebral disc"
    TL6_NUCLEUS_PULPOSUS = "Tl6 nucleus pulposus"
    TL6_ANNULUS_FIBROSUS = "Tl6 annulus fibrosus"
    TL7_INTERVERTEBRAL_DISC = "Tl7 intervertebral disc"
    TL7_NUCLEUS_PULPOSUS = "Tl7 nucleus pulposus"
    TL7_ANNULUS_FIBROSUS = "Tl7 annulus fibrosus"
    TL8_INTERVERTEBRAL_DISC = "Tl8 intervertebral disc"
    TL8_NUCLEUS_PULPOSUS = "Tl8 nucleus pulposus"
    TL8_ANNULUS_FIBROSUS = "Tl8 annulus fibrosus"
    TL9_INTERVERTEBRAL_DISC = "Tl9 intervertebral disc"
    TL9_NUCLEUS_PULPOSUS = "Tl9 nucleus pulposus"
    TL9_ANNULUS_FIBROSUS = "Tl9 annulus fibrosus"
    TL10_INTERVERTEBRAL_DISC = "Tl10 intervertebral disc"
    TL10_NUCLEUS_PULPOSUS = "Tl10 nucleus pulposus"
    TL10_ANNULUS_FIBROSUS = "Tl10 annulus fibrosus"
    TL11_INTERVERTEBRAL_DISC = "Tl11 intervertebral disc"
    TL11_NUCLEUS_PULPOSUS = "Tl11 nucleus pulposus"
    TL11_ANNULUS_FIBROSUS = "Tl11 annulus fibrosus"
    TL4_VERTEBRA = "Tl4 vertebra"
    TL5_VERTEBRA = "Tl5 vertebra"
    TL6_VERTEBRA = "Tl6 vertebra"
    TL7_VERTEBRA = "Tl7 vertebra"
    TL8_VERTEBRA = "Tl8 vertebra"
    TL9_VERTEBRA = "Tl9 vertebra"
    TL10_VERTEBRA = "Tl10 vertebra"
    TL11_VERTEBRA = "Tl11 vertebra"
    TL12_INTERVERTEBRAL_DISC = "Tl12 intervertebral disc"
    TL12_NUCLEUS_PULPOSUS = "Tl12 nucleus pulposus"
    TL12_ANNULUS_FIBROSUS = "Tl12 annulus fibrosus"
    TL13_INTERVERTEBRAL_DISC = "Tl13 intervertebral disc"
    TL13_NUCLEUS_PULPOSUS = "Tl13 nucleus pulposus"
    TL13_ANNULUS_FIBROSUS = "Tl13 annulus fibrosus"
    TL14_INTERVERTEBRAL_DISC = "Tl14 intervertebral disc"
    TL14_NUCLEUS_PULPOSUS = "Tl14 nucleus pulposus"
    TL14_ANNULUS_FIBROSUS = "Tl14 annulus fibrosus"
    TL15_INTERVERTEBRAL_DISC = "Tl15 intervertebral disc"
    TL15_NUCLEUS_PULPOSUS = "Tl15 nucleus pulposus"
    TL15_ANNULUS_FIBROSUS = "Tl15 annulus fibrosus"
    TL16_INTERVERTEBRAL_DISC = "Tl16 intervertebral disc"
    TL16_NUCLEUS_PULPOSUS = "Tl16 nucleus pulposus"
    TL16_ANNULUS_FIBROSUS = "Tl16 annulus fibrosus"
    TL17_INTERVERTEBRAL_DISC = "Tl17 intervertebral disc"
    TL17_NUCLEUS_PULPOSUS = "Tl17 nucleus pulposus"
    TL17_ANNULUS_FIBROSUS = "Tl17 annulus fibrosus"
    TL18_INTERVERTEBRAL_DISC = "Tl18 intervertebral disc"
    TL18_NUCLEUS_PULPOSUS = "Tl18 nucleus pulposus"
    TL18_ANNULUS_FIBROSUS = "Tl18 annulus fibrosus"
    TL19_INTERVERTEBRAL_DISC = "Tl19 intervertebral disc"
    TL19_NUCLEUS_PULPOSUS = "Tl19 nucleus pulposus"
    TL19_ANNULUS_FIBROSUS = "Tl19 annulus fibrosus"
    TL12_VERTEBRA = "Tl12 vertebra"
    TL13_VERTEBRA = "Tl13 vertebra"
    TL14_VERTEBRA = "Tl14 vertebra"
    TL15_VERTEBRA = "Tl15 vertebra"
    TL16_VERTEBRA = "Tl16 vertebra"
    TL17_VERTEBRA = "Tl17 vertebra"
    TL18_VERTEBRA = "Tl18 vertebra"
    TL19_VERTEBRA = "Tl19 vertebra"
    PLEURO_PERITONEAL_CANAL_MESOTHELIUM = "pleuro-peritoneal canal mesothelium"
    PLEURO_PERITONEAL_CANAL_PARIETAL_MESOTHELIUM = "pleuro-peritoneal canal parietal mesothelium"
    PLEURO_PERITONEAL_CANAL_CAVITY = "pleuro-peritoneal canal cavity"
    FUTURE_CENTRAL_TENDON = "future central tendon"
    RATHKE_S_POUCH_WALL = "Rathke's pouch wall"
    RATHKE_S_POUCH_LUMEN = "Rathke's pouch lumen"
    PROCTODEUM = "proctodeum"
    CAUDAL_ONE_THIRD_OF_ANAL_CANAL = "caudal one-third of anal canal"
    ECTODERM_COMPONENT_OF_ANAL_MEMBRANE = "ectoderm component of anal membrane"
    FUTURE_DOME = "future dome"
    SPINAL_CORD_LUMEN = "spinal cord lumen"
    DIAPHRAGM_LEFT_DOME = "diaphragm left dome"
    DIAPHRAGM_RIGHT_DOME = "diaphragm right dome"
    HUMERUS_CREST = "humerus crest"
    HEAD_OF_HUMERUS = "head of humerus"
    HUMERUS_SHAFT = "humerus shaft"
    HUMERUS_DELTOID_TUBEROSITY = "humerus deltoid tuberosity"
    TENDON_OF_BICEPS_BRACHII = "tendon of biceps brachii"
    LONG_HEAD_OF_BICEPS_BRACHII = "long head of biceps brachii"
    SHORT_HEAD_OF_BICEPS_BRACHII = "short head of biceps brachii"
    CARPAL_BONE = "carpal bone"
    FORELIMB_DIGIT_1_METACARPAL_PHALANGEAL_JOINT = "forelimb digit 1 metacarpal-phalangeal joint"
    FORELIMB_DIGIT_1_INTERPHALANGEAL_JOINT = "forelimb digit 1 interphalangeal joint"
    FORELIMB_DIGIT_1_EPIPHYSEAL_PLATE = "forelimb digit 1 epiphyseal plate"
    FORELIMB_DIGIT_2_METACARPAL_PHALANGEAL_JOINT = "forelimb digit 2 metacarpal-phalangeal joint"
    FORELIMB_DIGIT_2_INTERPHALANGEAL_JOINT = "forelimb digit 2 interphalangeal joint"
    FORELIMB_DIGIT_2_EPIPHYSEAL_PLATE = "forelimb digit 2 epiphyseal plate"
    FORELIMB_DIGIT_3_METACARPAL_PHALANGEAL_JOINT = "forelimb digit 3 metacarpal-phalangeal joint"
    FORELIMB_DIGIT_3_INTERPHALANGEAL_JOINT = "forelimb digit 3 interphalangeal joint"
    FORELIMB_DIGIT_3_EPIPHYSEAL_PLATE = "forelimb digit 3 epiphyseal plate"
    FORELIMB_DIGIT_4_METACARPAL_PHALANGEAL_JOINT = "forelimb digit 4 metacarpal-phalangeal joint"
    FORELIMB_DIGIT_4_INTERPHALANGEAL_JOINT = "forelimb digit 4 interphalangeal joint"
    FORELIMB_DIGIT_4_EPIPHYSEAL_PLATE = "forelimb digit 4 epiphyseal plate"
    FORELIMB_DIGIT_5_METACARPAL_PHALANGEAL_JOINT = "forelimb digit 5 metacarpal-phalangeal joint"
    FORELIMB_DIGIT_5_INTERPHALANGEAL_JOINT = "forelimb digit 5 interphalangeal joint"
    FORELIMB_DIGIT_5_EPIPHYSEAL_PLATE = "forelimb digit 5 epiphyseal plate"
    TARSAL_BONE = "tarsal bone"
    CEPHALIC_VEIN = "cephalic vein"
    CEREBELLUM_LATERAL_PART = "cerebellum lateral part"
    CEREBELLUM_LATERAL_PART_MANTLE_LAYER = "cerebellum lateral part mantle layer"
    CEREBELLUM_LATERAL_PART_MARGINAL_LAYER = "cerebellum lateral part marginal layer"
    CEREBELLUM_LATERAL_PART_VENTRICULAR_LAYER = "cerebellum lateral part ventricular layer"
    CHOROID_LAYER = "choroid layer"
    SEROUS_GLAND_EPITHELIAL_COMPONENT = "serous gland epithelial component"
    NASAL_SEPTUM_CARTILAGE = "nasal septum cartilage"
    TURBINATE_BONE = "turbinate bone"
    OROPHARYNX = "oropharynx"
    OROPHARYNX_GLAND = "oropharynx gland"
    OROPHARYNX_GLAND_DUCT = "oropharynx gland duct"
    TONGUE_GLAND = "tongue gland"
    TONGUE_GLAND_DUCT = "tongue gland duct"
    LOWER_JAW_ALVEOLAR_CANAL = "lower jaw alveolar canal"
    MANDIBLE_RAMUS = "mandible ramus"
    L1_VERTEBRA_CENTRUM = "L1 vertebra centrum"
    L1_VERTEBRA_NEURAL_ARCH = "L1 vertebra neural arch"
    L1_FUTURE_VERTEBRAL_CANAL = "L1 future vertebral canal"
    L1_NEURAL_ARCH_LAMINA = "L1 neural arch lamina"
    L1_NEURAL_ARCH_PEDICLE = "L1 neural arch pedicle"
    L1_NEURAL_ARCH_TRANSVERSE_PROCESS = "L1 neural arch transverse process"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_L1_SPINOUS_PROCESS = "cartilage primordium of future L1 spinous process"
    ZYGOMATIC_BONE = "zygomatic bone"
    OCCIPITAL_BONE = "occipital bone"
    OCCIPITAL_BONE_SQUAMOUS_PART = "occipital bone squamous part"
    ETHMOID_BONE_PERPENDICULAR_PLATE = "ethmoid bone perpendicular plate"
    T1_RIB_COSTAL_CARTILAGE = "T1 rib costal cartilage"
    T1_RIB_HEAD_CARTILAGE = "T1 rib head cartilage"
    T1_RIB_NECK_CARTILAGE = "T1 rib neck cartilage"
    ACROMION = "acromion"
    BLADE_OF_SCAPULA = "blade of scapula"
    CORACOID_PROCESS = "coracoid process"
    GLENOID_FOSSA_OF_SCAPULA = "glenoid fossa of scapula"
    SCAPULA_SPINE = "scapula spine"
    SCAPULA_BORDER = "scapula border"
    SCAPULA_ANGLE = "scapula angle"
    SCAPULA_LATERAL_BORDER = "scapula lateral border"
    SCAPULA_MEDIAL_BORDER = "scapula medial border"
    SCAPULA_INFERIOR_ANGLE = "scapula inferior angle"
    SCAPULA_SUPERIOR_ANGLE = "scapula superior angle"
    PRE_VERTEBRAL_MUSCLE_OF_NECK = "pre-vertebral muscle of neck"
    LONGUS_CAPITIS = "longus capitis"
    LONGUS_CERVICIS = "longus cervicis"
    FACIAL_MUSCLE = "facial muscle"
    DIGASTRIC = "digastric"
    MASSETER = "masseter"
    MYLOHYOID = "mylohyoid"
    STERNO_MASTOID = "sterno-mastoid"
    TEMPORALIS = "temporalis"
    PTERYGOID = "pterygoid"
    LATERAL_PTERYGOID = "lateral pterygoid"
    MEDIAL_PTERYGOID = "medial pterygoid"
    C1_DORSAL_ROOT_GANGLION = "C1 dorsal root ganglion"
    C1_SEGMENTAL_SPINAL_NERVE = "C1 segmental spinal nerve"
    C1_DORSAL_ROOT = "C1 dorsal root"
    C1_VENTRAL_ROOT = "C1 ventral root"
    TAIL_NERVE = "tail nerve"
    TAIL_SEGMENTAL_SPINAL_NERVE = "tail segmental spinal nerve"
    TL1_SEGMENTAL_SPINAL_NERVE = "Tl1 segmental spinal nerve"
    TL1_DORSAL_ROOT = "Tl1 dorsal root"
    TL1_VENTRAL_ROOT = "Tl1 ventral root"
    TL1_DORSAL_ROOT_GANGLION = "Tl1 dorsal root ganglion"
    C2_DORSAL_ROOT_GANGLION = "C2 dorsal root ganglion"
    C3_DORSAL_ROOT_GANGLION = "C3 dorsal root ganglion"
    C4_DORSAL_ROOT_GANGLION = "C4 dorsal root ganglion"
    C5_DORSAL_ROOT_GANGLION = "C5 dorsal root ganglion"
    C6_DORSAL_ROOT_GANGLION = "C6 dorsal root ganglion"
    C7_DORSAL_ROOT_GANGLION = "C7 dorsal root ganglion"
    L1_DORSAL_ROOT_GANGLION = "L1 dorsal root ganglion"
    L2_DORSAL_ROOT_GANGLION = "L2 dorsal root ganglion"
    L3_DORSAL_ROOT_GANGLION = "L3 dorsal root ganglion"
    L4_DORSAL_ROOT_GANGLION = "L4 dorsal root ganglion"
    L5_DORSAL_ROOT_GANGLION = "L5 dorsal root ganglion"
    L6_DORSAL_ROOT_GANGLION = "L6 dorsal root ganglion"
    S1_DORSAL_ROOT_GANGLION = "S1 dorsal root ganglion"
    S2_DORSAL_ROOT_GANGLION = "S2 dorsal root ganglion"
    S3_DORSAL_ROOT_GANGLION = "S3 dorsal root ganglion"
    S4_DORSAL_ROOT_GANGLION = "S4 dorsal root ganglion"
    T1_DORSAL_ROOT_GANGLION = "T1 dorsal root ganglion"
    T2_DORSAL_ROOT_GANGLION = "T2 dorsal root ganglion"
    T3_DORSAL_ROOT_GANGLION = "T3 dorsal root ganglion"
    T4_DORSAL_ROOT_GANGLION = "T4 dorsal root ganglion"
    T6_DORSAL_ROOT_GANGLION = "T6 dorsal root ganglion"
    T5_DORSAL_ROOT_GANGLION = "T5 dorsal root ganglion"
    T7_DORSAL_ROOT_GANGLION = "T7 dorsal root ganglion"
    T8_DORSAL_ROOT_GANGLION = "T8 dorsal root ganglion"
    T9_DORSAL_ROOT_GANGLION = "T9 dorsal root ganglion"
    T10_DORSAL_ROOT_GANGLION = "T10 dorsal root ganglion"
    T11_DORSAL_ROOT_GANGLION = "T11 dorsal root ganglion"
    T12_DORSAL_ROOT_GANGLION = "T12 dorsal root ganglion"
    T13_DORSAL_ROOT_GANGLION = "T13 dorsal root ganglion"
    TL2_DORSAL_ROOT_GANGLION = "Tl2 dorsal root ganglion"
    TL3_DORSAL_ROOT_GANGLION = "Tl3 dorsal root ganglion"
    TL4_DORSAL_ROOT_GANGLION = "Tl4 dorsal root ganglion"
    TL5_DORSAL_ROOT_GANGLION = "Tl5 dorsal root ganglion"
    TL6_DORSAL_ROOT_GANGLION = "Tl6 dorsal root ganglion"
    TL7_DORSAL_ROOT_GANGLION = "Tl7 dorsal root ganglion"
    TL8_DORSAL_ROOT_GANGLION = "Tl8 dorsal root ganglion"
    TL9_DORSAL_ROOT_GANGLION = "Tl9 dorsal root ganglion"
    TL10_DORSAL_ROOT_GANGLION = "Tl10 dorsal root ganglion"
    TL11_DORSAL_ROOT_GANGLION = "Tl11 dorsal root ganglion"
    TL12_DORSAL_ROOT_GANGLION = "Tl12 dorsal root ganglion"
    TL13_DORSAL_ROOT_GANGLION = "Tl13 dorsal root ganglion"
    TL14_DORSAL_ROOT_GANGLION = "Tl14 dorsal root ganglion"
    TL15_DORSAL_ROOT_GANGLION = "Tl15 dorsal root ganglion"
    TL16_DORSAL_ROOT_GANGLION = "Tl16 dorsal root ganglion"
    TL17_DORSAL_ROOT_GANGLION = "Tl17 dorsal root ganglion"
    TL18_DORSAL_ROOT_GANGLION = "Tl18 dorsal root ganglion"
    TL19_DORSAL_ROOT_GANGLION = "Tl19 dorsal root ganglion"
    TL2_SEGMENTAL_SPINAL_NERVE = "Tl2 segmental spinal nerve"
    TL3_SEGMENTAL_SPINAL_NERVE = "Tl3 segmental spinal nerve"
    TL4_SEGMENTAL_SPINAL_NERVE = "Tl4 segmental spinal nerve"
    TL5_SEGMENTAL_SPINAL_NERVE = "Tl5 segmental spinal nerve"
    TL6_SEGMENTAL_SPINAL_NERVE = "Tl6 segmental spinal nerve"
    TL7_SEGMENTAL_SPINAL_NERVE = "Tl7 segmental spinal nerve"
    TL8_SEGMENTAL_SPINAL_NERVE = "Tl8 segmental spinal nerve"
    TL9_SEGMENTAL_SPINAL_NERVE = "Tl9 segmental spinal nerve"
    TL10_SEGMENTAL_SPINAL_NERVE = "Tl10 segmental spinal nerve"
    TL11_SEGMENTAL_SPINAL_NERVE = "Tl11 segmental spinal nerve"
    TL12_SEGMENTAL_SPINAL_NERVE = "Tl12 segmental spinal nerve"
    TL13_SEGMENTAL_SPINAL_NERVE = "Tl13 segmental spinal nerve"
    TL14_SEGMENTAL_SPINAL_NERVE = "Tl14 segmental spinal nerve"
    TL15_SEGMENTAL_SPINAL_NERVE = "Tl15 segmental spinal nerve"
    TL16_SEGMENTAL_SPINAL_NERVE = "Tl16 segmental spinal nerve"
    TL17_SEGMENTAL_SPINAL_NERVE = "Tl17 segmental spinal nerve"
    TL18_SEGMENTAL_SPINAL_NERVE = "Tl18 segmental spinal nerve"
    TL19_SEGMENTAL_SPINAL_NERVE = "Tl19 segmental spinal nerve"
    C2_SEGMENTAL_SPINAL_NERVE = "C2 segmental spinal nerve"
    C3_SEGMENTAL_SPINAL_NERVE = "C3 segmental spinal nerve"
    C4_SEGMENTAL_SPINAL_NERVE = "C4 segmental spinal nerve"
    C5_SEGMENTAL_SPINAL_NERVE = "C5 segmental spinal nerve"
    C6_SEGMENTAL_SPINAL_NERVE = "C6 segmental spinal nerve"
    C7_SEGMENTAL_SPINAL_NERVE = "C7 segmental spinal nerve"
    L1_SEGMENTAL_SPINAL_NERVE = "L1 segmental spinal nerve"
    L2_SEGMENTAL_SPINAL_NERVE = "L2 segmental spinal nerve"
    L3_SEGMENTAL_SPINAL_NERVE = "L3 segmental spinal nerve"
    L4_SEGMENTAL_SPINAL_NERVE = "L4 segmental spinal nerve"
    L5_SEGMENTAL_SPINAL_NERVE = "L5 segmental spinal nerve"
    L6_SEGMENTAL_SPINAL_NERVE = "L6 segmental spinal nerve"
    S1_SEGMENTAL_SPINAL_NERVE = "S1 segmental spinal nerve"
    S2_SEGMENTAL_SPINAL_NERVE = "S2 segmental spinal nerve"
    S3_SEGMENTAL_SPINAL_NERVE = "S3 segmental spinal nerve"
    S4_SEGMENTAL_SPINAL_NERVE = "S4 segmental spinal nerve"
    T1_SEGMENTAL_SPINAL_NERVE = "T1 segmental spinal nerve"
    T2_SEGMENTAL_SPINAL_NERVE = "T2 segmental spinal nerve"
    T3_SEGMENTAL_SPINAL_NERVE = "T3 segmental spinal nerve"
    T4_SEGMENTAL_SPINAL_NERVE = "T4 segmental spinal nerve"
    T5_SEGMENTAL_SPINAL_NERVE = "T5 segmental spinal nerve"
    T6_SEGMENTAL_SPINAL_NERVE = "T6 segmental spinal nerve"
    T7_SEGMENTAL_SPINAL_NERVE = "T7 segmental spinal nerve"
    T8_SEGMENTAL_SPINAL_NERVE = "T8 segmental spinal nerve"
    T9_SEGMENTAL_SPINAL_NERVE = "T9 segmental spinal nerve"
    T10_SEGMENTAL_SPINAL_NERVE = "T10 segmental spinal nerve"
    T11_SEGMENTAL_SPINAL_NERVE = "T11 segmental spinal nerve"
    T12_SEGMENTAL_SPINAL_NERVE = "T12 segmental spinal nerve"
    T13_SEGMENTAL_SPINAL_NERVE = "T13 segmental spinal nerve"
    C2_DORSAL_ROOT = "C2 dorsal root"
    C2_VENTRAL_ROOT = "C2 ventral root"
    C3_DORSAL_ROOT = "C3 dorsal root"
    C3_VENTRAL_ROOT = "C3 ventral root"
    C4_DORSAL_ROOT = "C4 dorsal root"
    C4_VENTRAL_ROOT = "C4 ventral root"
    C5_DORSAL_ROOT = "C5 dorsal root"
    C5_VENTRAL_ROOT = "C5 ventral root"
    C6_DORSAL_ROOT = "C6 dorsal root"
    C6_VENTRAL_ROOT = "C6 ventral root"
    C7_DORSAL_ROOT = "C7 dorsal root"
    C7_VENTRAL_ROOT = "C7 ventral root"
    L1_DORSAL_ROOT = "L1 dorsal root"
    L1_VENTRAL_ROOT = "L1 ventral root"
    L2_DORSAL_ROOT = "L2 dorsal root"
    L2_VENTRAL_ROOT = "L2 ventral root"
    L3_DORSAL_ROOT = "L3 dorsal root"
    L3_VENTRAL_ROOT = "L3 ventral root"
    L4_DORSAL_ROOT = "L4 dorsal root"
    L4_VENTRAL_ROOT = "L4 ventral root"
    L5_DORSAL_ROOT = "L5 dorsal root"
    L5_VENTRAL_ROOT = "L5 ventral root"
    L6_DORSAL_ROOT = "L6 dorsal root"
    L6_VENTRAL_ROOT = "L6 ventral root"
    S1_DORSAL_ROOT = "S1 dorsal root"
    S1_VENTRAL_ROOT = "S1 ventral root"
    S2_DORSAL_ROOT = "S2 dorsal root"
    S2_VENTRAL_ROOT = "S2 ventral root"
    S3_DORSAL_ROOT = "S3 dorsal root"
    S3_VENTRAL_ROOT = "S3 ventral root"
    S4_DORSAL_ROOT = "S4 dorsal root"
    S4_VENTRAL_ROOT = "S4 ventral root"
    T1_DORSAL_ROOT = "T1 dorsal root"
    T1_VENTRAL_ROOT = "T1 ventral root"
    T10_DORSAL_ROOT = "T10 dorsal root"
    T11_DORSAL_ROOT = "T11 dorsal root"
    T11_VENTRAL_ROOT = "T11 ventral root"
    T10_VENTRAL_ROOT = "T10 ventral root"
    T12_DORSAL_ROOT = "T12 dorsal root"
    T12_VENTRAL_ROOT = "T12 ventral root"
    T13_DORSAL_ROOT = "T13 dorsal root"
    T2_DORSAL_ROOT = "T2 dorsal root"
    T2_VENTRAL_ROOT = "T2 ventral root"
    T3_DORSAL_ROOT = "T3 dorsal root"
    T3_VENTRAL_ROOT = "T3 ventral root"
    T4_DORSAL_ROOT = "T4 dorsal root"
    T4_VENTRAL_ROOT = "T4 ventral root"
    T5_DORSAL_ROOT = "T5 dorsal root"
    T5_VENTRAL_ROOT = "T5 ventral root"
    T6_DORSAL_ROOT = "T6 dorsal root"
    T6_VENTRAL_ROOT = "T6 ventral root"
    T7_DORSAL_ROOT = "T7 dorsal root"
    T7_VENTRAL_ROOT = "T7 ventral root"
    T8_DORSAL_ROOT = "T8 dorsal root"
    T8_VENTRAL_ROOT = "T8 ventral root"
    T9_DORSAL_ROOT = "T9 dorsal root"
    T9_VENTRAL_ROOT = "T9 ventral root"
    T13_VENTRAL_ROOT = "T13 ventral root"
    TL10_DORSAL_ROOT = "Tl10 dorsal root"
    TL10_VENTRAL_ROOT = "Tl10 ventral root"
    TL11_DORSAL_ROOT = "Tl11 dorsal root"
    TL11_VENTRAL_ROOT = "Tl11 ventral root"
    TL12_DORSAL_ROOT = "Tl12 dorsal root"
    TL12_VENTRAL_ROOT = "Tl12 ventral root"
    TL13_DORSAL_ROOT = "Tl13 dorsal root"
    TL13_VENTRAL_ROOT = "Tl13 ventral root"
    TL14_DORSAL_ROOT = "Tl14 dorsal root"
    TL14_VENTRAL_ROOT = "Tl14 ventral root"
    TL15_DORSAL_ROOT = "Tl15 dorsal root"
    TL15_VENTRAL_ROOT = "Tl15 ventral root"
    TL16_DORSAL_ROOT = "Tl16 dorsal root"
    TL16_VENTRAL_ROOT = "Tl16 ventral root"
    TL17_DORSAL_ROOT = "Tl17 dorsal root"
    TL17_VENTRAL_ROOT = "Tl17 ventral root"
    TL18_DORSAL_ROOT = "Tl18 dorsal root"
    TL18_VENTRAL_ROOT = "Tl18 ventral root"
    TL19_DORSAL_ROOT = "Tl19 dorsal root"
    TL19_VENTRAL_ROOT = "Tl19 ventral root"
    TL2_DORSAL_ROOT = "Tl2 dorsal root"
    TL2_VENTRAL_ROOT = "Tl2 ventral root"
    TL3_DORSAL_ROOT = "Tl3 dorsal root"
    TL3_VENTRAL_ROOT = "Tl3 ventral root"
    TL4_DORSAL_ROOT = "Tl4 dorsal root"
    TL4_VENTRAL_ROOT = "Tl4 ventral root"
    TL5_DORSAL_ROOT = "Tl5 dorsal root"
    TL5_VENTRAL_ROOT = "Tl5 ventral root"
    TL6_DORSAL_ROOT = "Tl6 dorsal root"
    TL6_VENTRAL_ROOT = "Tl6 ventral root"
    TL7_DORSAL_ROOT = "Tl7 dorsal root"
    TL7_VENTRAL_ROOT = "Tl7 ventral root"
    TL8_DORSAL_ROOT = "Tl8 dorsal root"
    TL8_VENTRAL_ROOT = "Tl8 ventral root"
    TL9_DORSAL_ROOT = "Tl9 dorsal root"
    TL9_VENTRAL_ROOT = "Tl9 ventral root"
    VAGUS_NERVE_LEFT_RECURRENT_LARYNGEAL_BRANCH = "vagus nerve left recurrent laryngeal branch"
    PINNA_MESENCHYME_CONDENSATION = "pinna mesenchyme condensation"
    PINNA_REST_OF_MESENCHYME = "pinna rest of mesenchyme"
    PINNA_CARTILAGE_CONDENSATION = "pinna cartilage condensation"
    SUBCUTANEOUS_VEIN = "subcutaneous vein"
    TRUNK_MESENCHYME_DERIVED_FROM_NEURAL_CREST_NON_MIGRATING_CELLS = (
        "trunk mesenchyme derived from neural crest non-migrating cells"
    )
    MESENCHYME_DERIVED_FROM_SOMITES_01_04 = "mesenchyme derived from somites 01-04"
    THYROID_PRIMORDIUM_ENDODERM = "thyroid primordium endoderm"
    RHOMBOMERE_1_REST_OF_NEURAL_FOLD = "rhombomere 1 rest of neural fold"
    RHOMBOMERE_2_REST_OF_NEURAL_FOLD = "rhombomere 2 rest of neural fold"
    RHOMBOMERE_3_REST_OF_NEURAL_FOLD = "rhombomere 3 rest of neural fold"
    RHOMBOMERE_4_REST_OF_NEURAL_FOLD = "rhombomere 4 rest of neural fold"
    RHOMBOMERE_5_REST_OF_NEURAL_FOLD = "rhombomere 5 rest of neural fold"
    OTOCYST_LUMEN = "otocyst lumen"
    L2_VERTEBRA_CENTRUM = "L2 vertebra centrum"
    L2_FUTURE_VERTEBRAL_CANAL = "L2 future vertebral canal"
    L2_VERTEBRA_NEURAL_ARCH = "L2 vertebra neural arch"
    L2_NEURAL_ARCH_LAMINA = "L2 neural arch lamina"
    L2_NEURAL_ARCH_PEDICLE = "L2 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_L2_SPINOUS_PROCESS = "cartilage primordium of future L2 spinous process"
    L2_NEURAL_ARCH_TRANSVERSE_PROCESS = "L2 neural arch transverse process"
    L3_VERTEBRA_CENTRUM = "L3 vertebra centrum"
    L3_FUTURE_VERTEBRAL_CANAL = "L3 future vertebral canal"
    L3_VERTEBRA_NEURAL_ARCH = "L3 vertebra neural arch"
    L3_NEURAL_ARCH_LAMINA = "L3 neural arch lamina"
    L3_NEURAL_ARCH_PEDICLE = "L3 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_L3_SPINOUS_PROCESS = "cartilage primordium of future L3 spinous process"
    L3_NEURAL_ARCH_TRANSVERSE_PROCESS = "L3 neural arch transverse process"
    L4_VERTEBRA_CENTRUM = "L4 vertebra centrum"
    L4_FUTURE_VERTEBRAL_CANAL = "L4 future vertebral canal"
    L4_VERTEBRA_NEURAL_ARCH = "L4 vertebra neural arch"
    L4_NEURAL_ARCH_LAMINA = "L4 neural arch lamina"
    L4_NEURAL_ARCH_PEDICLE = "L4 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_L4_SPINOUS_PROCESS = "cartilage primordium of future L4 spinous process"
    L4_NEURAL_ARCH_TRANSVERSE_PROCESS = "L4 neural arch transverse process"
    L5_VERTEBRA_CENTRUM = "L5 vertebra centrum"
    L5_FUTURE_VERTEBRAL_CANAL = "L5 future vertebral canal"
    L5_VERTEBRA_NEURAL_ARCH = "L5 vertebra neural arch"
    L5_NEURAL_ARCH_LAMINA = "L5 neural arch lamina"
    L5_NEURAL_ARCH_PEDICLE = "L5 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_L5_SPINOUS_PROCESS = "cartilage primordium of future L5 spinous process"
    L5_NEURAL_ARCH_TRANSVERSE_PROCESS = "L5 neural arch transverse process"
    L6_VERTEBRA_CENTRUM = "L6 vertebra centrum"
    L6_FUTURE_VERTEBRAL_CANAL = "L6 future vertebral canal"
    L6_VERTEBRA_NEURAL_ARCH = "L6 vertebra neural arch"
    L6_NEURAL_ARCH_LAMINA = "L6 neural arch lamina"
    L6_NEURAL_ARCH_PEDICLE = "L6 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_L6_SPINOUS_PROCESS = "cartilage primordium of future L6 spinous process"
    L6_NEURAL_ARCH_TRANSVERSE_PROCESS = "L6 neural arch transverse process"
    S1_VERTEBRA_CENTRUM = "S1 vertebra centrum"
    S1_FUTURE_VERTEBRAL_CANAL = "S1 future vertebral canal"
    S1_VERTEBRA_NEURAL_ARCH = "S1 vertebra neural arch"
    S1_NEURAL_ARCH_LAMINA = "S1 neural arch lamina"
    S1_NEURAL_ARCH_PEDICLE = "S1 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_S1_SPINOUS_PROCESS = "cartilage primordium of future S1 spinous process"
    S1_NEURAL_ARCH_TRANSVERSE_PROCESS = "S1 neural arch transverse process"
    S2_VERTEBRA_CENTRUM = "S2 vertebra centrum"
    S2_FUTURE_VERTEBRAL_CANAL = "S2 future vertebral canal"
    S2_VERTEBRA_NEURAL_ARCH = "S2 vertebra neural arch"
    S2_NEURAL_ARCH_LAMINA = "S2 neural arch lamina"
    S2_NEURAL_ARCH_PEDICLE = "S2 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_S2_SPINOUS_PROCESS = "cartilage primordium of future S2 spinous process"
    S2_NEURAL_ARCH_TRANSVERSE_PROCESS = "S2 neural arch transverse process"
    S3_VERTEBRA_CENTRUM = "S3 vertebra centrum"
    S3_FUTURE_VERTEBRAL_CANAL = "S3 future vertebral canal"
    S3_VERTEBRA_NEURAL_ARCH = "S3 vertebra neural arch"
    S3_NEURAL_ARCH_LAMINA = "S3 neural arch lamina"
    S3_NEURAL_ARCH_PEDICLE = "S3 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_S3_SPINOUS_PROCESS = "cartilage primordium of future S3 spinous process"
    S3_NEURAL_ARCH_TRANSVERSE_PROCESS = "S3 neural arch transverse process"
    S4_VERTEBRA_CENTRUM = "S4 vertebra centrum"
    S4_FUTURE_VERTEBRAL_CANAL = "S4 future vertebral canal"
    S4_VERTEBRA_NEURAL_ARCH = "S4 vertebra neural arch"
    S4_NEURAL_ARCH_LAMINA = "S4 neural arch lamina"
    S4_NEURAL_ARCH_PEDICLE = "S4 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_S4_SPINOUS_PROCESS = "cartilage primordium of future S4 spinous process"
    S4_NEURAL_ARCH_TRANSVERSE_PROCESS = "S4 neural arch transverse process"
    T1_VERTEBRA_CENTRUM = "T1 vertebra centrum"
    T1_VERTEBRAL_CANAL = "T1 vertebral canal"
    T1_VERTEBRA_NEURAL_ARCH = "T1 vertebra neural arch"
    T1_NEURAL_ARCH_LAMINA = "T1 neural arch lamina"
    T1_NEURAL_ARCH_PEDICLE = "T1 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T1_SPINOUS_PROCESS = "cartilage primordium of future T1 spinous process"
    T1_NEURAL_ARCH_TRANSVERSE_PROCESS = "T1 neural arch transverse process"
    T10_VERTEBRA_CENTRUM = "T10 vertebra centrum"
    T10_VERTEBRAL_CANAL = "T10 vertebral canal"
    T10_VERTEBRA_NEURAL_ARCH = "T10 vertebra neural arch"
    T10_NEURAL_ARCH_LAMINA = "T10 neural arch lamina"
    T10_NEURAL_ARCH_PEDICLE = "T10 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T10_SPINOUS_PROCESS = "cartilage primordium of future T10 spinous process"
    T10_NEURAL_ARCH_TRANSVERSE_PROCESS = "T10 neural arch transverse process"
    T11_VERTEBRA_CENTRUM = "T11 vertebra centrum"
    T11_VERTEBRAL_CANAL = "T11 vertebral canal"
    T11_VERTEBRA_NEURAL_ARCH = "T11 vertebra neural arch"
    T11_NEURAL_ARCH_LAMINA = "T11 neural arch lamina"
    T11_NEURAL_ARCH_PEDICLE = "T11 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T11_SPINOUS_PROCESS = "cartilage primordium of future T11 spinous process"
    T11_NEURAL_ARCH_TRANSVERSE_PROCESS = "T11 neural arch transverse process"
    T12_VERTEBRA_CENTRUM = "T12 vertebra centrum"
    T12_VERTEBRAL_CANAL = "T12 vertebral canal"
    T12_VERTEBRA_NEURAL_ARCH = "T12 vertebra neural arch"
    T12_NEURAL_ARCH_LAMINA = "T12 neural arch lamina"
    T12_NEURAL_ARCH_PEDICLE = "T12 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T12_SPINOUS_PROCESS = "cartilage primordium of future T12 spinous process"
    T12_NEURAL_ARCH_TRANSVERSE_PROCESS = "T12 neural arch transverse process"
    T13_VERTEBRA_CENTRUM = "T13 vertebra centrum"
    T13_VERTEBRAL_CANAL = "T13 vertebral canal"
    T13_VERTEBRA_NEURAL_ARCH = "T13 vertebra neural arch"
    T13_NEURAL_ARCH_LAMINA = "T13 neural arch lamina"
    T13_NEURAL_ARCH_PEDICLE = "T13 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T13_SPINOUS_PROCESS = "cartilage primordium of future T13 spinous process"
    T13_NEURAL_ARCH_TRANSVERSE_PROCESS = "T13 neural arch transverse process"
    T2_VERTEBRA_CENTRUM = "T2 vertebra centrum"
    T2_VERTEBRAL_CANAL = "T2 vertebral canal"
    T2_VERTEBRA_NEURAL_ARCH = "T2 vertebra neural arch"
    T2_NEURAL_ARCH_LAMINA = "T2 neural arch lamina"
    T2_NEURAL_ARCH_PEDICLE = "T2 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T2_SPINOUS_PROCESS = "cartilage primordium of future T2 spinous process"
    T2_NEURAL_ARCH_TRANSVERSE_PROCESS = "T2 neural arch transverse process"
    T3_VERTEBRA_CENTRUM = "T3 vertebra centrum"
    T3_VERTEBRAL_CANAL = "T3 vertebral canal"
    T3_VERTEBRA_NEURAL_ARCH = "T3 vertebra neural arch"
    T3_NEURAL_ARCH_LAMINA = "T3 neural arch lamina"
    T3_NEURAL_ARCH_PEDICLE = "T3 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T3_SPINOUS_PROCESS = "cartilage primordium of future T3 spinous process"
    T3_NEURAL_ARCH_TRANSVERSE_PROCESS = "T3 neural arch transverse process"
    T4_VERTEBRA_CENTRUM = "T4 vertebra centrum"
    T4_VERTEBRAL_CANAL = "T4 vertebral canal"
    T4_VERTEBRA_NEURAL_ARCH = "T4 vertebra neural arch"
    T4_NEURAL_ARCH_LAMINA = "T4 neural arch lamina"
    T4_NEURAL_ARCH_PEDICLE = "T4 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T4_SPINOUS_PROCESS = "cartilage primordium of future T4 spinous process"
    T4_NEURAL_ARCH_TRANSVERSE_PROCESS = "T4 neural arch transverse process"
    T5_VERTEBRA_CENTRUM = "T5 vertebra centrum"
    T5_VERTEBRAL_CANAL = "T5 vertebral canal"
    T5_VERTEBRA_NEURAL_ARCH = "T5 vertebra neural arch"
    T5_NEURAL_ARCH_LAMINA = "T5 neural arch lamina"
    T5_NEURAL_ARCH_PEDICLE = "T5 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T5_SPINOUS_PROCESS = "cartilage primordium of future T5 spinous process"
    T5_NEURAL_ARCH_TRANSVERSE_PROCESS = "T5 neural arch transverse process"
    T6_VERTEBRA_CENTRUM = "T6 vertebra centrum"
    T6_VERTEBRAL_CANAL = "T6 vertebral canal"
    T6_VERTEBRA_NEURAL_ARCH = "T6 vertebra neural arch"
    T6_NEURAL_ARCH_LAMINA = "T6 neural arch lamina"
    T6_NEURAL_ARCH_PEDICLE = "T6 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T6_SPINOUS_PROCESS = "cartilage primordium of future T6 spinous process"
    T6_NEURAL_ARCH_TRANSVERSE_PROCESS = "T6 neural arch transverse process"
    T7_VERTEBRA_CENTRUM = "T7 vertebra centrum"
    T7_VERTEBRAL_CANAL = "T7 vertebral canal"
    T7_VERTEBRA_NEURAL_ARCH = "T7 vertebra neural arch"
    T7_NEURAL_ARCH_LAMINA = "T7 neural arch lamina"
    T7_NEURAL_ARCH_PEDICLE = "T7 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T7_SPINOUS_PROCESS = "cartilage primordium of future T7 spinous process"
    T7_NEURAL_ARCH_TRANSVERSE_PROCESS = "T7 neural arch transverse process"
    T8_VERTEBRA_CENTRUM = "T8 vertebra centrum"
    T8_VERTEBRAL_CANAL = "T8 vertebral canal"
    T8_VERTEBRA_NEURAL_ARCH = "T8 vertebra neural arch"
    T8_NEURAL_ARCH_LAMINA = "T8 neural arch lamina"
    T8_NEURAL_ARCH_PEDICLE = "T8 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T8_SPINOUS_PROCESS = "cartilage primordium of future T8 spinous process"
    T8_NEURAL_ARCH_TRANSVERSE_PROCESS = "T8 neural arch transverse process"
    T9_VERTEBRA_CENTRUM = "T9 vertebra centrum"
    T9_VERTEBRAL_CANAL = "T9 vertebral canal"
    T9_VERTEBRA_NEURAL_ARCH = "T9 vertebra neural arch"
    T9_NEURAL_ARCH_LAMINA = "T9 neural arch lamina"
    T9_NEURAL_ARCH_PEDICLE = "T9 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_T9_SPINOUS_PROCESS = "cartilage primordium of future T9 spinous process"
    T9_NEURAL_ARCH_TRANSVERSE_PROCESS = "T9 neural arch transverse process"
    TL1_VERTEBRA_CENTRUM = "Tl1 vertebra centrum"
    TL1_VERTEBRAL_CANAL = "Tl1 vertebral canal"
    TL1_VERTEBRA_NEURAL_ARCH = "Tl1 vertebra neural arch"
    TL1_NEURAL_ARCH_LAMINA = "Tl1 neural arch lamina"
    TL1_NEURAL_ARCH_PEDICLE = "Tl1 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL1_SPINOUS_PROCESS = "cartilage primordium of future Tl1 spinous process"
    TL1_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl1 neural arch transverse process"
    TL10_VERTEBRA_CENTRUM = "Tl10 vertebra centrum"
    TL10_VERTEBRAL_CANAL = "Tl10 vertebral canal"
    TL10_VERTEBRA_NEURAL_ARCH = "Tl10 vertebra neural arch"
    TL10_NEURAL_ARCH_LAMINA = "Tl10 neural arch lamina"
    TL10_NEURAL_ARCH_PEDICLE = "Tl10 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL10_SPINOUS_PROCESS = "cartilage primordium of future Tl10 spinous process"
    TL10_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl10 neural arch transverse process"
    TL11_VERTEBRA_CENTRUM = "Tl11 vertebra centrum"
    TL11_VERTEBRAL_CANAL = "Tl11 vertebral canal"
    TL11_VERTEBRA_NEURAL_ARCH = "Tl11 vertebra neural arch"
    TL11_NEURAL_ARCH_LAMINA = "Tl11 neural arch lamina"
    TL11_NEURAL_ARCH_PEDICLE = "Tl11 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL11_SPINOUS_PROCESS = "cartilage primordium of future Tl11 spinous process"
    TL11_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl11 neural arch transverse process"
    TL12_VERTEBRA_CENTRUM = "Tl12 vertebra centrum"
    TL12_VERTEBRAL_CANAL = "Tl12 vertebral canal"
    TL12_VERTEBRA_NEURAL_ARCH = "Tl12 vertebra neural arch"
    TL12_NEURAL_ARCH_LAMINA = "Tl12 neural arch lamina"
    TL12_NEURAL_ARCH_PEDICLE = "Tl12 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL12_SPINOUS_PROCESS = "cartilage primordium of future Tl12 spinous process"
    TL12_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl12 neural arch transverse process"
    TL13_VERTEBRA_CENTRUM = "Tl13 vertebra centrum"
    TL13_VERTEBRAL_CANAL = "Tl13 vertebral canal"
    TL13_VERTEBRA_NEURAL_ARCH = "Tl13 vertebra neural arch"
    TL13_NEURAL_ARCH_LAMINA = "Tl13 neural arch lamina"
    TL13_NEURAL_ARCH_PEDICLE = "Tl13 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL13_SPINOUS_PROCESS = "cartilage primordium of future Tl13 spinous process"
    TL13_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl13 neural arch transverse process"
    TL14_VERTEBRA_CENTRUM = "Tl14 vertebra centrum"
    TL14_VERTEBRAL_CANAL = "Tl14 vertebral canal"
    TL14_VERTEBRA_NEURAL_ARCH = "Tl14 vertebra neural arch"
    TL14_NEURAL_ARCH_LAMINA = "Tl14 neural arch lamina"
    TL14_NEURAL_ARCH_PEDICLE = "Tl14 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL14_SPINOUS_PROCESS = "cartilage primordium of future Tl14 spinous process"
    TL14_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl14 neural arch transverse process"
    TL15_VERTEBRA_CENTRUM = "Tl15 vertebra centrum"
    TL15_VERTEBRAL_CANAL = "Tl15 vertebral canal"
    TL15_VERTEBRA_NEURAL_ARCH = "Tl15 vertebra neural arch"
    TL15_NEURAL_ARCH_LAMINA = "Tl15 neural arch lamina"
    TL15_NEURAL_ARCH_PEDICLE = "Tl15 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL15_SPINOUS_PROCESS = "cartilage primordium of future Tl15 spinous process"
    TL15_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl15 neural arch transverse process"
    TL16_VERTEBRA_CENTRUM = "Tl16 vertebra centrum"
    TL16_VERTEBRAL_CANAL = "Tl16 vertebral canal"
    TL16_VERTEBRA_NEURAL_ARCH = "Tl16 vertebra neural arch"
    TL16_NEURAL_ARCH_LAMINA = "Tl16 neural arch lamina"
    TL16_NEURAL_ARCH_PEDICLE = "Tl16 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL16_SPINOUS_PROCESS = "cartilage primordium of future Tl16 spinous process"
    TL16_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl16 neural arch transverse process"
    TL17_VERTEBRA_CENTRUM = "Tl17 vertebra centrum"
    TL17_VERTEBRAL_CANAL = "Tl17 vertebral canal"
    TL17_VERTEBRA_NEURAL_ARCH = "Tl17 vertebra neural arch"
    TL17_NEURAL_ARCH_LAMINA = "Tl17 neural arch lamina"
    TL17_NEURAL_ARCH_PEDICLE = "Tl17 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL17_SPINOUS_PROCESS = "cartilage primordium of future Tl17 spinous process"
    TL17_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl17 neural arch transverse process"
    TL18_VERTEBRA_CENTRUM = "Tl18 vertebra centrum"
    TL18_VERTEBRAL_CANAL = "Tl18 vertebral canal"
    TL18_VERTEBRA_NEURAL_ARCH = "Tl18 vertebra neural arch"
    TL18_NEURAL_ARCH_LAMINA = "Tl18 neural arch lamina"
    TL18_NEURAL_ARCH_PEDICLE = "Tl18 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL18_SPINOUS_PROCESS = "cartilage primordium of future Tl18 spinous process"
    TL18_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl18 neural arch transverse process"
    TL19_VERTEBRA_CENTRUM = "Tl19 vertebra centrum"
    TL19_VERTEBRAL_CANAL = "Tl19 vertebral canal"
    TL19_VERTEBRA_NEURAL_ARCH = "Tl19 vertebra neural arch"
    TL19_NEURAL_ARCH_LAMINA = "Tl19 neural arch lamina"
    TL19_NEURAL_ARCH_PEDICLE = "Tl19 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL19_SPINOUS_PROCESS = "cartilage primordium of future Tl19 spinous process"
    TL19_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl19 neural arch transverse process"
    TL2_VERTEBRA_CENTRUM = "Tl2 vertebra centrum"
    TL2_VERTEBRAL_CANAL = "Tl2 vertebral canal"
    TL2_VERTEBRA_NEURAL_ARCH = "Tl2 vertebra neural arch"
    TL2_NEURAL_ARCH_LAMINA = "Tl2 neural arch lamina"
    TL2_NEURAL_ARCH_PEDICLE = "Tl2 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL2_SPINOUS_PROCESS = "cartilage primordium of future Tl2 spinous process"
    TL2_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl2 neural arch transverse process"
    TL3_VERTEBRA_CENTRUM = "Tl3 vertebra centrum"
    TL3_VERTEBRAL_CANAL = "Tl3 vertebral canal"
    TL3_VERTEBRA_NEURAL_ARCH = "Tl3 vertebra neural arch"
    TL3_NEURAL_ARCH_LAMINA = "Tl3 neural arch lamina"
    TL3_NEURAL_ARCH_PEDICLE = "Tl3 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL3_SPINOUS_PROCESS = "cartilage primordium of future Tl3 spinous process"
    TL3_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl3 neural arch transverse process"
    TL4_VERTEBRA_CENTRUM = "Tl4 vertebra centrum"
    TL4_VERTEBRAL_CANAL = "Tl4 vertebral canal"
    TL4_VERTEBRA_NEURAL_ARCH = "Tl4 vertebra neural arch"
    TL4_NEURAL_ARCH_LAMINA = "Tl4 neural arch lamina"
    TL4_NEURAL_ARCH_PEDICLE = "Tl4 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL4_SPINOUS_PROCESS = "cartilage primordium of future Tl4 spinous process"
    TL4_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl4 neural arch transverse process"
    TL5_VERTEBRA_CENTRUM = "Tl5 vertebra centrum"
    TL5_VERTEBRAL_CANAL = "Tl5 vertebral canal"
    TL5_VERTEBRA_NEURAL_ARCH = "Tl5 vertebra neural arch"
    TL5_NEURAL_ARCH_LAMINA = "Tl5 neural arch lamina"
    TL5_NEURAL_ARCH_PEDICLE = "Tl5 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL5_SPINOUS_PROCESS = "cartilage primordium of future Tl5 spinous process"
    TL5_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl5 neural arch transverse process"
    TL6_VERTEBRA_CENTRUM = "Tl6 vertebra centrum"
    TL6_VERTEBRAL_CANAL = "Tl6 vertebral canal"
    TL6_VERTEBRA_NEURAL_ARCH = "Tl6 vertebra neural arch"
    TL6_NEURAL_ARCH_LAMINA = "Tl6 neural arch lamina"
    TL6_NEURAL_ARCH_PEDICLE = "Tl6 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL6_SPINOUS_PROCESS = "cartilage primordium of future Tl6 spinous process"
    TL6_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl6 neural arch transverse process"
    TL7_VERTEBRA_CENTRUM = "Tl7 vertebra centrum"
    TL7_VERTEBRAL_CANAL = "Tl7 vertebral canal"
    TL7_VERTEBRA_NEURAL_ARCH = "Tl7 vertebra neural arch"
    TL7_NEURAL_ARCH_LAMINA = "Tl7 neural arch lamina"
    TL7_NEURAL_ARCH_PEDICLE = "Tl7 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL7_SPINOUS_PROCESS = "cartilage primordium of future Tl7 spinous process"
    TL7_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl7 neural arch transverse process"
    TL8_VERTEBRA_CENTRUM = "Tl8 vertebra centrum"
    TL8_VERTEBRAL_CANAL = "Tl8 vertebral canal"
    TL8_VERTEBRA_NEURAL_ARCH = "Tl8 vertebra neural arch"
    TL8_NEURAL_ARCH_LAMINA = "Tl8 neural arch lamina"
    TL8_NEURAL_ARCH_PEDICLE = "Tl8 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL8_SPINOUS_PROCESS = "cartilage primordium of future Tl8 spinous process"
    TL8_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl8 neural arch transverse process"
    TL9_VERTEBRA_CENTRUM = "Tl9 vertebra centrum"
    TL9_VERTEBRAL_CANAL = "Tl9 vertebral canal"
    TL9_VERTEBRA_NEURAL_ARCH = "Tl9 vertebra neural arch"
    TL9_NEURAL_ARCH_LAMINA = "Tl9 neural arch lamina"
    TL9_NEURAL_ARCH_PEDICLE = "Tl9 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_TL9_SPINOUS_PROCESS = "cartilage primordium of future Tl9 spinous process"
    TL9_NEURAL_ARCH_TRANSVERSE_PROCESS = "Tl9 neural arch transverse process"
    T10_RIB_COSTAL_CARTILAGE = "T10 rib costal cartilage"
    T10_RIB_HEAD_CARTILAGE = "T10 rib head cartilage"
    T10_RIB_NECK_CARTILAGE = "T10 rib neck cartilage"
    T11_RIB_COSTAL_CARTILAGE = "T11 rib costal cartilage"
    T11_RIB_HEAD_CARTILAGE = "T11 rib head cartilage"
    T11_RIB_NECK_CARTILAGE = "T11 rib neck cartilage"
    T12_RIB_COSTAL_CARTILAGE = "T12 rib costal cartilage"
    T12_RIB_HEAD_CARTILAGE = "T12 rib head cartilage"
    T12_RIB_NECK_CARTILAGE = "T12 rib neck cartilage"
    T13_RIB_COSTAL_CARTILAGE = "T13 rib costal cartilage"
    T13_RIB_HEAD_CARTILAGE = "T13 rib head cartilage"
    T13_RIB_NECK_CARTILAGE = "T13 rib neck cartilage"
    T2_RIB_COSTAL_CARTILAGE = "T2 rib costal cartilage"
    T2_RIB_HEAD_CARTILAGE = "T2 rib head cartilage"
    T2_RIB_NECK_CARTILAGE = "T2 rib neck cartilage"
    T3_RIB_COSTAL_CARTILAGE = "T3 rib costal cartilage"
    T3_RIB_HEAD_CARTILAGE = "T3 rib head cartilage"
    T3_RIB_NECK_CARTILAGE = "T3 rib neck cartilage"
    T4_RIB_COSTAL_CARTILAGE = "T4 rib costal cartilage"
    T4_RIB_HEAD_CARTILAGE = "T4 rib head cartilage"
    T4_RIB_NECK_CARTILAGE = "T4 rib neck cartilage"
    T5_RIB_COSTAL_CARTILAGE = "T5 rib costal cartilage"
    T5_RIB_HEAD_CARTILAGE = "T5 rib head cartilage"
    T5_RIB_NECK_CARTILAGE = "T5 rib neck cartilage"
    T6_RIB_COSTAL_CARTILAGE = "T6 rib costal cartilage"
    T6_RIB_HEAD_CARTILAGE = "T6 rib head cartilage"
    T6_RIB_NECK_CARTILAGE = "T6 rib neck cartilage"
    T7_RIB_COSTAL_CARTILAGE = "T7 rib costal cartilage"
    T7_RIB_HEAD_CARTILAGE = "T7 rib head cartilage"
    T7_RIB_NECK_CARTILAGE = "T7 rib neck cartilage"
    T8_RIB_COSTAL_CARTILAGE = "T8 rib costal cartilage"
    T8_RIB_HEAD_CARTILAGE = "T8 rib head cartilage"
    T8_RIB_NECK_CARTILAGE = "T8 rib neck cartilage"
    T9_RIB_COSTAL_CARTILAGE = "T9 rib costal cartilage"
    T9_RIB_HEAD_CARTILAGE = "T9 rib head cartilage"
    T9_RIB_NECK_CARTILAGE = "T9 rib neck cartilage"
    C3_VERTEBRA_CENTRUM = "C3 vertebra centrum"
    C3_VERTEBRAL_CANAL = "C3 vertebral canal"
    C3_VERTEBRA_NEURAL_ARCH = "C3 vertebra neural arch"
    C3_NEURAL_ARCH_LAMINA = "C3 neural arch lamina"
    C3_NEURAL_ARCH_PEDICLE = "C3 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_C3_SPINOUS_PROCESS = "cartilage primordium of future C3 spinous process"
    C3_NEURAL_ARCH_TRANSVERSE_PROCESS = "C3 neural arch transverse process"
    C4_VERTEBRA_CENTRUM = "C4 vertebra centrum"
    C4_VERTEBRAL_CANAL = "C4 vertebral canal"
    C4_VERTEBRA_NEURAL_ARCH = "C4 vertebra neural arch"
    C4_NEURAL_ARCH_LAMINA = "C4 neural arch lamina"
    C4_NEURAL_ARCH_PEDICLE = "C4 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_C4_SPINOUS_PROCESS = "cartilage primordium of future C4 spinous process"
    C4_NEURAL_ARCH_TRANSVERSE_PROCESS = "C4 neural arch transverse process"
    C5_VERTEBRA_CENTRUM = "C5 vertebra centrum"
    C5_VERTEBRAL_CANAL = "C5 vertebral canal"
    C5_VERTEBRA_NEURAL_ARCH = "C5 vertebra neural arch"
    C5_NEURAL_ARCH_LAMINA = "C5 neural arch lamina"
    C5_NEURAL_ARCH_PEDICLE = "C5 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_C5_SPINOUS_PROCESS = "cartilage primordium of future C5 spinous process"
    C5_NEURAL_ARCH_TRANSVERSE_PROCESS = "C5 neural arch transverse process"
    C6_VERTEBRA_CENTRUM = "C6 vertebra centrum"
    C6_VERTEBRAL_CANAL = "C6 vertebral canal"
    C6_VERTEBRA_NEURAL_ARCH = "C6 vertebra neural arch"
    C6_NEURAL_ARCH_LAMINA = "C6 neural arch lamina"
    C6_NEURAL_ARCH_PEDICLE = "C6 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_C6_SPINOUS_PROCESS = "cartilage primordium of future C6 spinous process"
    C6_NEURAL_ARCH_TRANSVERSE_PROCESS = "C6 neural arch transverse process"
    C7_VERTEBRA_CENTRUM = "C7 vertebra centrum"
    C7_VERTEBRAL_CANAL = "C7 vertebral canal"
    C7_VERTEBRA_NEURAL_ARCH = "C7 vertebra neural arch"
    C7_NEURAL_ARCH_LAMINA = "C7 neural arch lamina"
    C7_NEURAL_ARCH_PEDICLE = "C7 neural arch pedicle"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_C7_SPINOUS_PROCESS = "cartilage primordium of future C7 spinous process"
    C7_NEURAL_ARCH_TRANSVERSE_PROCESS = "C7 neural arch transverse process"
    MOUSE = "mouse"
    LEFT_ATRIUM_LUMEN = "left atrium lumen"
    RIGHT_ATRIUM_LUMEN = "right atrium lumen"
    INTERVENTRICULAR_CANAL = "interventricular canal"
    LEFT_VENTRICLE_LUMEN = "left ventricle lumen"
    RIGHT_VENTRICLE_LUMEN = "right ventricle lumen"
    INTERSEGMENTAL_VEIN = "intersegmental vein"
    MEDIAN_SACRAL_VEIN = "median sacral vein"
    INFERIOR_VENA_CAVA_INTRAHEPATIC_PART = "inferior vena cava intrahepatic part"
    COMMUNICATION_DUCTUS_VENOSUS_TO_PORTAL_VEIN = "communication ductus venosus to portal vein"
    PHYSIOLOGICAL_UMBILICAL_HERNIA_MESENCHYME = "physiological umbilical hernia mesenchyme"
    PHYSIOLOGICAL_UMBILICAL_HERNIA_MESOTHELIUM = "physiological umbilical hernia mesothelium"
    UMBILICAL_CORD = "umbilical cord"
    UMBILICAL_CORD_MESENCHYME = "umbilical cord mesenchyme"
    VITELLINE_DUCT = "vitelline duct"
    VITELLINE_VEIN_VITELLINE_DUCT_PART = "vitelline vein vitelline duct part"
    CAVITY_OF_PHYSIOLOGICAL_UMBILICAL_HERNIA = "cavity of physiological umbilical hernia"
    CAVITY_OF_PHYSIOLOGICAL_UMBILICAL_HERNIA_MESOTHELIUM = "cavity of physiological umbilical hernia mesothelium"
    PANCREAS_MESENCHYME = "pancreas mesenchyme"
    HINDLIMB_DIGIT_1_INTERPHALANGEAL_JOINT = "hindlimb digit 1 interphalangeal joint"
    HINDLIMB_DIGIT_1_METATARSAL_PHALANGEAL_JOINT = "hindlimb digit 1 metatarsal-phalangeal joint"
    HINDLIMB_DIGIT_2_METATARSAL_PHALANGEAL_JOINT = "hindlimb digit 2 metatarsal-phalangeal joint"
    HINDLIMB_DIGIT_2_INTERPHALANGEAL_JOINT = "hindlimb digit 2 interphalangeal joint"
    HINDLIMB_DIGIT_3_INTERPHALANGEAL_JOINT = "hindlimb digit 3 interphalangeal joint"
    HINDLIMB_DIGIT_3_METATARSAL_PHALANGEAL_JOINT = "hindlimb digit 3 metatarsal-phalangeal joint"
    HINDLIMB_DIGIT_4_INTERPHALANGEAL_JOINT = "hindlimb digit 4 interphalangeal joint"
    HINDLIMB_DIGIT_4_METATARSAL_PHALANGEAL_JOINT = "hindlimb digit 4 metatarsal-phalangeal joint"
    HINDLIMB_DIGIT_5_INTERPHALANGEAL_JOINT = "hindlimb digit 5 interphalangeal joint"
    HINDLIMB_DIGIT_5_METATARSAL_PHALANGEAL_JOINT = "hindlimb digit 5 metatarsal-phalangeal joint"
    C1_VERTEBRAL_CANAL = "C1 vertebral canal"
    C1_VERTEBRA_NEURAL_ARCH = "C1 vertebra neural arch"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_C1_SPINOUS_PROCESS = "cartilage primordium of future C1 spinous process"
    C1_NEURAL_ARCH_LAMINA = "C1 neural arch lamina"
    C1_NEURAL_ARCH_PEDICLE = "C1 neural arch pedicle"
    C2_VERTEBRA_CENTRUM = "C2 vertebra centrum"
    ODONTOID_PROCESS = "odontoid process"
    C2_VERTEBRAL_CANAL = "C2 vertebral canal"
    C2_VERTEBRA_NEURAL_ARCH = "C2 vertebra neural arch"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_C2_SPINOUS_PROCESS = "cartilage primordium of future C2 spinous process"
    C2_NEURAL_ARCH_LAMINA = "C2 neural arch lamina"
    C2_NEURAL_ARCH_PEDICLE = "C2 neural arch pedicle"
    C3_FORAMEN_TRANSVERSARIUM = "C3 foramen transversarium"
    C4_FORAMEN_TRANSVERSARIUM = "C4 foramen transversarium"
    C5_FORAMEN_TRANSVERSARIUM = "C5 foramen transversarium"
    C6_FORAMEN_TRANSVERSARIUM = "C6 foramen transversarium"
    C7_FORAMEN_TRANSVERSARIUM = "C7 foramen transversarium"
    C1_MIXED_SPINAL_NERVE = "C1 mixed spinal nerve"
    C2_MIXED_SPINAL_NERVE = "C2 mixed spinal nerve"
    C3_MIXED_SPINAL_NERVE = "C3 mixed spinal nerve"
    C4_MIXED_SPINAL_NERVE = "C4 mixed spinal nerve"
    C5_MIXED_SPINAL_NERVE = "C5 mixed spinal nerve"
    C6_MIXED_SPINAL_NERVE = "C6 mixed spinal nerve"
    C7_MIXED_SPINAL_NERVE = "C7 mixed spinal nerve"
    L1_MIXED_SPINAL_NERVE = "L1 mixed spinal nerve"
    L2_MIXED_SPINAL_NERVE = "L2 mixed spinal nerve"
    L3_MIXED_SPINAL_NERVE = "L3 mixed spinal nerve"
    L4_MIXED_SPINAL_NERVE = "L4 mixed spinal nerve"
    L5_MIXED_SPINAL_NERVE = "L5 mixed spinal nerve"
    L6_MIXED_SPINAL_NERVE = "L6 mixed spinal nerve"
    S1_MIXED_SPINAL_NERVE = "S1 mixed spinal nerve"
    S2_MIXED_SPINAL_NERVE = "S2 mixed spinal nerve"
    S3_MIXED_SPINAL_NERVE = "S3 mixed spinal nerve"
    S4_MIXED_SPINAL_NERVE = "S4 mixed spinal nerve"
    T1_MIXED_SPINAL_NERVE = "T1 mixed spinal nerve"
    T2_MIXED_SPINAL_NERVE = "T2 mixed spinal nerve"
    T3_MIXED_SPINAL_NERVE = "T3 mixed spinal nerve"
    T4_MIXED_SPINAL_NERVE = "T4 mixed spinal nerve"
    T5_MIXED_SPINAL_NERVE = "T5 mixed spinal nerve"
    T6_MIXED_SPINAL_NERVE = "T6 mixed spinal nerve"
    T7_MIXED_SPINAL_NERVE = "T7 mixed spinal nerve"
    T8_MIXED_SPINAL_NERVE = "T8 mixed spinal nerve"
    T9_MIXED_SPINAL_NERVE = "T9 mixed spinal nerve"
    T10_MIXED_SPINAL_NERVE = "T10 mixed spinal nerve"
    T11_MIXED_SPINAL_NERVE = "T11 mixed spinal nerve"
    T12_MIXED_SPINAL_NERVE = "T12 mixed spinal nerve"
    T13_MIXED_SPINAL_NERVE = "T13 mixed spinal nerve"
    T1_RIB_SHAFT_OSSIFIED_REGION = "T1 rib shaft ossified region"
    T1_RIB_SHAFT_NON_OSSIFIED_REGION = "T1 rib shaft non-ossified region"
    T2_RIB_SHAFT_OSSIFIED_REGION = "T2 rib shaft ossified region"
    T2_RIB_SHAFT_NON_OSSIFIED_REGION = "T2 rib shaft non-ossified region"
    T3_RIB_SHAFT_OSSIFIED_REGION = "T3 rib shaft ossified region"
    T3_RIB_SHAFT_NON_OSSIFIED_REGION = "T3 rib shaft non-ossified region"
    T4_RIB_SHAFT_OSSIFIED_REGION = "T4 rib shaft ossified region"
    T4_RIB_SHAFT_NON_OSSIFIED_REGION = "T4 rib shaft non-ossified region"
    T5_RIB_SHAFT_OSSIFIED_REGION = "T5 rib shaft ossified region"
    T5_RIB_SHAFT_NON_OSSIFIED_REGION = "T5 rib shaft non-ossified region"
    T6_RIB_SHAFT_OSSIFIED_REGION = "T6 rib shaft ossified region"
    T6_RIB_SHAFT_NON_OSSIFIED_REGION = "T6 rib shaft non-ossified region"
    T7_RIB_SHAFT_OSSIFIED_REGION = "T7 rib shaft ossified region"
    T7_RIB_SHAFT_NON_OSSIFIED_REGION = "T7 rib shaft non-ossified region"
    T8_RIB_SHAFT_OSSIFIED_REGION = "T8 rib shaft ossified region"
    T8_RIB_SHAFT_NON_OSSIFIED_REGION = "T8 rib shaft non-ossified region"
    T9_RIB_SHAFT_OSSIFIED_REGION = "T9 rib shaft ossified region"
    T9_RIB_SHAFT_NON_OSSIFIED_REGION = "T9 rib shaft non-ossified region"
    T10_RIB_SHAFT_OSSIFIED_REGION = "T10 rib shaft ossified region"
    T10_RIB_SHAFT_NON_OSSIFIED_REGION = "T10 rib shaft non-ossified region"
    T11_RIB_SHAFT_OSSIFIED_REGION = "T11 rib shaft ossified region"
    T11_RIB_SHAFT_NON_OSSIFIED_REGION = "T11 rib shaft non-ossified region"
    T12_RIB_SHAFT_NON_OSSIFIED_REGION = "T12 rib shaft non-ossified region"
    T12_RIB_SHAFT_OSSIFIED_REGION = "T12 rib shaft ossified region"
    T13_RIB_SHAFT_OSSIFIED_REGION = "T13 rib shaft ossified region"
    T13_RIB_SHAFT_NON_OSSIFIED_REGION = "T13 rib shaft non-ossified region"
    C1_NEURAL_ARCH_TRANSVERSE_PROCESS = "C1 neural arch transverse process"
    C1_FORAMEN_TRANSVERSARIUM = "C1 foramen transversarium"
    C2_NEURAL_ARCH_TRANSVERSE_PROCESS = "C2 neural arch transverse process"
    C2_FORAMEN_TRANSVERSARIUM = "C2 foramen transversarium"
    VIBRISSA_BULB = "vibrissa bulb"
    VIBRISSA_FOLLICLE = "vibrissa follicle"
    VIBRISSA_ROOT_SHEATH = "vibrissa root sheath"
    VIBRISSA_HAIR = "vibrissa hair"
    VIBRISSA_ARRECTOR_PILI_MUSCLE = "vibrissa arrector pili muscle"
    HAIR_ARRECTOR_PILI_MUSCLE = "hair arrector pili muscle"
    MALE_INGUINAL_CANAL = "male inguinal canal"
    ENTERIC_NERVOUS_SYSTEM = "enteric nervous system"
    PUBIS_CARTILAGE_CONDENSATION = "pubis cartilage condensation"
    ISCHIUM_CARTILAGE_CONDENSATION = "ischium cartilage condensation"
    ORAL_MUCOSA = "oral mucosa"
    ORAL_EPITHELIUM = "oral epithelium"
    ORAL_LAMINA_PROPRIA = "oral lamina propria"
    ORAL_REGION_MUSCULARIS = "oral region muscularis"
    PHARYNX_MUCOSA = "pharynx mucosa"
    PHARYNX_MUSCULARIS = "pharynx muscularis"
    PHARYNX_LAMINA_PROPRIA = "pharynx lamina propria"
    ESOPHAGUS_MUCOSA = "esophagus mucosa"
    ESOPHAGUS_MUSCULARIS = "esophagus muscularis"
    ESOPHAGUS_LAMINA_PROPRIA = "esophagus lamina propria"
    ESOPHAGUS_MUSCULARIS_MUCOSA = "esophagus muscularis mucosa"
    FOREGUT_MIDGUT_JUNCTION_MUCOSA = "foregut-midgut junction mucosa"
    FOREGUT_MIDGUT_JUNCTION_SUBMUCOSA = "foregut-midgut junction submucosa"
    FOREGUT_MIDGUT_JUNCTION_MUSCULARIS = "foregut-midgut junction muscularis"
    FOREGUT_MIDGUT_JUNCTION_SEROSA = "foregut-midgut junction serosa"
    FOREGUT_MIDGUT_JUNCTION_LAMINA_PROPRIA = "foregut-midgut junction lamina propria"
    FOREGUT_MIDGUT_JUNCTION_MUSCULARIS_MUCOSA = "foregut-midgut junction muscularis mucosa"
    DUODENUM_ROSTRAL_PART_MUCOSA = "duodenum rostral part mucosa"
    DUODENUM_ROSTRAL_PART_SUBMUCOSA = "duodenum rostral part submucosa"
    DUODENUM_ROSTRAL_PART_MUSCULARIS = "duodenum rostral part muscularis"
    DUODENUM_ROSTRAL_PART_SEROSA = "duodenum rostral part serosa"
    DUODENUM_ROSTRAL_PART_LAMINA_PROPRIA = "duodenum rostral part lamina propria"
    DUODENUM_ROSTRAL_PART_MUSCULARIS_MUCOSA = "duodenum rostral part muscularis mucosa"
    GASTRO_ESOPHAGEAL_JUNCTION_MUCOSA = "gastro-esophageal junction mucosa"
    GASTRO_ESOPHAGEAL_JUNCTION_SUBMUCOSA = "gastro-esophageal junction submucosa"
    GASTRO_ESOPHAGEAL_JUNCTION_MUSCULARIS = "gastro-esophageal junction muscularis"
    GASTRO_ESOPHAGEAL_JUNCTION_SEROSA = "gastro-esophageal junction serosa"
    GASTRO_ESOPHAGEAL_JUNCTION_LAMINA_PROPRIA = "gastro-esophageal junction lamina propria"
    GASTRO_ESOPHAGEAL_JUNCTION_MUSCULARIS_MUCOSA = "gastro-esophageal junction muscularis mucosa"
    PERINEAL_BODY_MUCOSA = "perineal body mucosa"
    PERINEAL_BODY_SUBMUCOSA = "perineal body submucosa"
    PERINEAL_BODY_MUSCULARIS = "perineal body muscularis"
    PERINEAL_BODY_SEROSA = "perineal body serosa"
    PERINEAL_BODY_LAMINA_PROPRIA = "perineal body lamina propria"
    PERINEAL_BODY_MUSCULARIS_MUCOSA = "perineal body muscularis mucosa"
    HINDGUT_MUCOSA = "hindgut mucosa"
    HINDGUT_SUBMUCOSA = "hindgut submucosa"
    HINDGUT_MUSCULARIS = "hindgut muscularis"
    HINDGUT_SEROSA = "hindgut serosa"
    HINDGUT_LAMINA_PROPRIA = "hindgut lamina propria"
    HINDGUT_MUSCULARIS_MUCOSA = "hindgut muscularis mucosa"
    RECTUM_MUCOSA = "rectum mucosa"
    RECTUM_SUBMUCOSA = "rectum submucosa"
    RECTUM_MUSCULARIS = "rectum muscularis"
    RECTUM_SEROSA = "rectum serosa"
    RECTUM_LAMINA_PROPRIA = "rectum lamina propria"
    RECTUM_MUSCULARIS_MUCOSA = "rectum muscularis mucosa"
    DUODENUM_CAUDAL_PART_MUCOSA = "duodenum caudal part mucosa"
    DUODENUM_CAUDAL_PART_SUBMUCOSA = "duodenum caudal part submucosa"
    DUODENUM_CAUDAL_PART_MUSCULARIS = "duodenum caudal part muscularis"
    DUODENUM_CAUDAL_PART_SEROSA = "duodenum caudal part serosa"
    DUODENUM_CAUDAL_PART_LAMINA_PROPRIA = "duodenum caudal part lamina propria"
    DUODENUM_CAUDAL_PART_MUSCULARIS_MUCOSA = "duodenum caudal part muscularis mucosa"
    JEJUNUM_MUCOSA = "jejunum mucosa"
    JEJUNUM_SUBMUCOSA = "jejunum submucosa"
    JEJUNUM_MUSCULARIS = "jejunum muscularis"
    JEJUNUM_SEROSA = "jejunum serosa"
    JEJUNUM_LAMINA_PROPRIA = "jejunum lamina propria"
    JEJUNUM_MUSCULARIS_MUCOSA = "jejunum muscularis mucosa"
    MIDGUT_LOOP_MUCOSA = "midgut loop mucosa"
    MIDGUT_LOOP_SUBMUCOSA = "midgut loop submucosa"
    MIDGUT_LOOP_MUSCULARIS = "midgut loop muscularis"
    MIDGUT_LOOP_SEROSA = "midgut loop serosa"
    MIDGUT_LOOP_LAMINA_PROPRIA = "midgut loop lamina propria"
    MIDGUT_LOOP_MUSCULARIS_MUCOSA = "midgut loop muscularis mucosa"
    MIDGUT_MUCOSA = "midgut mucosa"
    MIDGUT_SUBMUCOSA = "midgut submucosa"
    MIDGUT_MUSCULARIS = "midgut muscularis"
    MIDGUT_SEROSA = "midgut serosa"
    MIDGUT_LAMINA_PROPRIA = "midgut lamina propria"
    MIDGUT_MUSCULARIS_MUCOSA = "midgut muscularis mucosa"
    STOMACH_FUNDUS_MUCOSA = "stomach fundus mucosa"
    STOMACH_FUNDUS_SUBMUCOSA = "stomach fundus submucosa"
    STOMACH_FUNDUS_MUSCULARIS = "stomach fundus muscularis"
    STOMACH_FUNDUS_SEROSA = "stomach fundus serosa"
    STOMACH_FUNDUS_GASTRIC_GLAND = "stomach fundus gastric gland"
    STOMACH_FUNDUS_LAMINA_PROPRIA = "stomach fundus lamina propria"
    STOMACH_FUNDUS_MUSCULARIS_MUCOSA = "stomach fundus muscularis mucosa"
    STOMACH_GLANDULAR_REGION_MUCOSA = "stomach glandular region mucosa"
    STOMACH_GLANDULAR_REGION_SUBMUCOSA = "stomach glandular region submucosa"
    STOMACH_GLANDULAR_REGION_MUSCULARIS = "stomach glandular region muscularis"
    STOMACH_GLANDULAR_REGION_SEROSA = "stomach glandular region serosa"
    STOMACH_GLANDULAR_REGION_LAMINA_PROPRIA = "stomach glandular region lamina propria"
    STOMACH_GLANDULAR_REGION_MUSCULARIS_MUCOSA = "stomach glandular region muscularis mucosa"
    STOMACH_GLANDULAR_REGION_GASTRIC_GLAND = "stomach glandular region gastric gland"
    STOMACH_PROVENTRICULAR_REGION_MUCOSA = "stomach proventricular region mucosa"
    STOMACH_PROVENTRICULAR_REGION_SUBMUCOSA = "stomach proventricular region submucosa"
    STOMACH_PROVENTRICULAR_REGION_MUSCULARIS = "stomach proventricular region muscularis"
    STOMACH_PROVENTRICULAR_REGION_SEROSA = "stomach proventricular region serosa"
    STOMACH_PROVENTRICULAR_REGION_LAMINA_PROPRIA = "stomach proventricular region lamina propria"
    STOMACH_PROVENTRICULAR_REGION_GASTRIC_GLAND = "stomach proventricular region gastric gland"
    STOMACH_PROVENTRICULAR_REGION_MUSCULARIS_MUCOSA = "stomach proventricular region muscularis mucosa"
    STOMACH_PYLORIC_REGION_MUCOSA = "stomach pyloric region mucosa"
    STOMACH_PYLORIC_REGION_SUBMUCOSA = "stomach pyloric region submucosa"
    STOMACH_PYLORIC_REGION_SEROSA = "stomach pyloric region serosa"
    STOMACH_PYLORIC_REGION_LAMINA_PROPRIA = "stomach pyloric region lamina propria"
    STOMACH_PYLORIC_REGION_GASTRIC_GLAND = "stomach pyloric region gastric gland"
    STOMACH_PYLORIC_REGION_MUSCULARIS_MUCOSA = "stomach pyloric region muscularis mucosa"
    ANAL_CANAL_CAUDAL_PART_MUCOSA = "anal canal caudal part mucosa"
    ANAL_CANAL_CAUDAL_PART_SUBMUCOSA = "anal canal caudal part submucosa"
    ANAL_CANAL_CAUDAL_PART_GLAND = "anal canal caudal part gland"
    ANAL_CANAL_CAUDAL_PART_LAMINA_PROPRIA = "anal canal caudal part lamina propria"
    ANAL_CANAL_ROSTRAL_PART_MUCOSA = "anal canal rostral part mucosa"
    ANAL_CANAL_ROSTRAL_PART_SUBMUCOSA = "anal canal rostral part submucosa"
    ANAL_CANAL_ROSTRAL_PART_LAMINA_PROPRIA = "anal canal rostral part lamina propria"
    ANAL_CANAL_ROSTRAL_PART_GLAND = "anal canal rostral part gland"
    DUODENUM_MUCOSA = "duodenum mucosa"
    DUODENUM_SUBMUCOSA = "duodenum submucosa"
    DUODENUM_MUSCULARIS = "duodenum muscularis"
    DUODENUM_SEROSA = "duodenum serosa"
    DUODENUM_LAMINA_PROPRIA = "duodenum lamina propria"
    DUODENUM_MUSCULARIS_MUCOSA = "duodenum muscularis mucosa"
    COLON_MUCOSA = "colon mucosa"
    COLON_SUBMUCOSA = "colon submucosa"
    COLON_MUSCULARIS = "colon muscularis"
    COLON_SEROSA = "colon serosa"
    COLON_LAMINA_PROPRIA = "colon lamina propria"
    COLON_MUSCULARIS_MUCOSA = "colon muscularis mucosa"
    ANAL_CANAL_MUCOSA = "anal canal mucosa"
    ANAL_CANAL_SUBMUCOSA = "anal canal submucosa"
    ANAL_CANAL_GLAND = "anal canal gland"
    ANAL_CANAL_LAMINA_PROPRIA = "anal canal lamina propria"
    COELOMIC_EPITHELIUM_OF_GONAD_PRIMORDIUM = "coelomic epithelium of gonad primordium"
    ANTERIOR_GONAD_PRIMORDIUM_MESENCHYME = "anterior gonad primordium mesenchyme"
    CENTRAL_GONAD_PRIMORDIUM_MESENCHYME = "central gonad primordium mesenchyme"
    POSTERIOR_GONAD_PRIMORDIUM_MESENCHYME = "posterior gonad primordium mesenchyme"
    CLOACA = "cloaca"
    CLOACAL_EPITHELIUM = "cloacal epithelium"
    CLOACA_ASSOCIATED_MESENCHYME = "cloaca associated mesenchyme"
    NEPHRIC_DUCT__MESONEPHRIC_PORTION = "nephric duct, mesonephric portion"
    NEPHRIC_DUCT__METANEPHRIC_PORTION = "nephric duct, metanephric portion"
    NEPHRIC_DUCT__PRONEPHRIC_PORTION = "nephric duct, pronephric portion"
    URETERIC_TIP = "ureteric tip"
    URETERIC_STALK = "ureteric stalk"
    PARAMETANEPHRIC_MESENCHYME = "parametanephric mesenchyme"
    ROSTRAL_PARAMETANEPHRIC_MESENCHYME = "rostral parametanephric mesenchyme"
    CAUDAL_PARAMESONEPHRIC_MESENCHYME = "caudal paramesonephric mesenchyme"
    CAP_MESENCHYME = "cap mesenchyme"
    PERIPHERAL_BLASTEMA = "peripheral blastema"
    ANTERIOR_GONADAL_MESENCHYME = "anterior gonadal mesenchyme"
    CENTRAL_GONADAL_MESENCHYME = "central gonadal mesenchyme"
    POSTERIOR_GONADAL_MESENCHYME = "posterior gonadal mesenchyme"
    PARAMESONEPHRIC_DUCT = "paramesonephric duct"
    PRETUBULAR_AGGREGATE = "pretubular aggregate"
    RENAL_VESICLE = "renal vesicle"
    COMMA_SHAPED_BODY = "comma-shaped body"
    UPPER_LIMB_OF_COMMA_SHAPED_BODY = "upper limb of comma-shaped body"
    URETERIC_TRUNK = "ureteric trunk"
    EPITHELIAL_LAYER_OF_URETER = "epithelial layer of ureter"
    URETERAL_MESENCHYME = "ureteral mesenchyme"
    NEPHROGENIC_ZONE = "nephrogenic zone"
    S_SHAPED_BODY = "s-shaped body"
    RENAL_CONNECTING_TUBULE_OF_S_SHAPED_BODY = "renal connecting tubule of s-shaped body"
    DISTAL_SEGMENT_OF_S_SHAPED_BODY = "distal segment of s-shaped body"
    MEDIAL_SEGMENT_OF_S_SHAPED_BODY = "medial segment of s-shaped body"
    PROXIMAL_SEGMENT_OF_S_SHAPED_BODY = "proximal segment of s-shaped body"
    VISCERAL_EPITHELIUM_OF_S_SHAPED_BODY = "visceral epithelium of s-shaped body"
    PARIETAL_EPITHELIUM_OF_S_SHAPED_BODY = "parietal epithelium of s-shaped body"
    CAPILLARY_LOOP_RENAL_CORPUSCLE = "capillary loop renal corpuscle"
    VISCERAL_EPITHELIUM_OF_CAPILLARY_LOOP_RENAL_CORPUSCLE = "visceral epithelium of capillary loop renal corpuscle"
    PARIETAL_EPITHELIUM_OF_CAPILLARY_LOOP_RENAL_CORPUSCLE = "parietal epithelium of capillary loop renal corpuscle"
    ASSOCIATED_INTERSTITIUM_OF_CAPILLARY_LOOP_RENAL_CORPUSCLE = (
        "associated interstitium of capillary loop renal corpuscle"
    )
    PRESUMPTIVE_MESANGIUM_OF_CAPILLARY_LOOP_RENAL_CORPUSCLE = "presumptive mesangium of capillary loop renal corpuscle"
    PRESUMPTIVE_ENDOTHELIUM_OF_CAPILLARY_LOOP_RENAL_CORPUSCLE = (
        "presumptive endothelium of capillary loop renal corpuscle"
    )
    RENAL_TUBULE = "renal tubule"
    EARLY_PROXIMAL_TUBULE = "early proximal tubule"
    ANLAGE_OF_LOOP_OF_HENLE = "anlage of loop of Henle"
    EARLY_DISTAL_TUBULE = "early distal tubule"
    RENAL_CONNECTING_TUBULE_OF_CAPILLARY_LOOP_NEPHRON = "renal connecting tubule of capillary loop nephron"
    DEVELOPING_VASCULATURE_OF_METANEPHROS = "developing vasculature of metanephros"
    LOWER_LIMB_OF_COMMA_SHAPED_BODY = "lower limb of comma-shaped body"
    MATURING_RENAL_CORPUSCLE = "maturing renal corpuscle"
    MATURING_GLOMERULAR_TUFT = "maturing glomerular tuft"
    GLOMERULAR_CAPILLARY_SYSTEM_OF_MATURING_GLOMERULAR_TUFT = "glomerular capillary system of maturing glomerular tuft"
    GLOMERULAR_MESANGIUM_OF_MATURING_GLOMERULAR_TUFT = "glomerular mesangium of maturing glomerular tuft"
    VISCERAL_EPITHELIUM_OF_MATURING_RENAL_CORPUSCLE = "visceral epithelium of maturing renal corpuscle"
    BOWMAN_S_CAPSULE_OF_MATURING_RENAL_CORPUSCLE = "Bowman's capsule of maturing renal corpuscle"
    PARIETAL_EPITHELIUM_OF_MATURING_RENAL_CORPUSCLE = "parietal epithelium of maturing renal corpuscle"
    URINARY_SPACE_OF_MATURING_RENAL_CORPUSCLE = "urinary space of maturing renal corpuscle"
    JUXTAGLOMERULAR_ARTERIOLE = "juxtaglomerular arteriole"
    CORTICAL_EARLY_PROXIMAL_TUBULE = "cortical early proximal tubule"
    CORTICAL_EARLY_DISTAL_TUBULE = "cortical early distal tubule"
    RENAL_CONNECTING_TUBULE = "renal connecting tubule"
    DEVELOPING_VASCULATURE_OF_RENAL_CORTEX = "developing vasculature of renal cortex"
    DEVELOPING_ARTERIAL_VASCULATURE_OF_RENAL_CORTEX = "developing arterial vasculature of renal cortex"
    DEVELOPING_VENOUS_VASCULATURE_OF_RENAL_CORTEX = "developing venous vasculature of renal cortex"
    DEVELOPING_CAPILLARY_OF_RENAL_CORTEX = "developing capillary of renal cortex"
    IMMATURE_LOOP_OF_HENLE = "immature loop of Henle"
    MEDULLARY_COLLECTING_DUCT = "medullary collecting duct"
    DEVELOPING_VASCULATURE_OF_RENAL_MEDULLA = "developing vasculature of renal medulla"
    DEVELOPING_ARTERIAL_VASCULATURE_OF_RENAL_MEDULLA = "developing arterial vasculature of renal medulla"
    DEVELOPING_VENOUS_VASCULATURE_OF_RENAL_MEDULLA = "developing venous vasculature of renal medulla"
    DEVELOPING_CAPILLARY_OF_RENAL_MEDULLA = "developing capillary of renal medulla"
    PELVIC_UROTHELIAL_LINING = "pelvic urothelial lining"
    PERIHILAR_INTERSTITIUM = "perihilar interstitium"
    UROTHELIUM_OF_URETER = "urothelium of ureter"
    ADVENTITIA_OF_URETER = "adventitia of ureter"
    PELVIC_SMOOTH_MUSCLE = "pelvic smooth muscle"
    CORTICAL_COLLECTING_DUCT = "cortical collecting duct"
    RENAL_CORTICAL_INTERSTITIUM = "renal cortical interstitium"
    RENAL_CORTICAL_VASCULATURE = "renal cortical vasculature"
    RENAL_CORTICAL_ARTERIAL_SYSTEM = "renal cortical arterial system"
    RENAL_CORTICAL_VENOUS_SYSTEM = "renal cortical venous system"
    RENAL_MEDULLARY_VASCULATURE = "renal medullary vasculature"
    RENAL_MEDULLARY_ARTERIAL_SYSTEM = "renal medullary arterial system"
    RENAL_MEDULLARY_VENOUS_SYSTEM = "renal medullary venous system"
    INTERLOBULAR_ARTERY = "interlobular artery"
    ARCUATE_ARTERY = "arcuate artery"
    INTERLOBULAR_VEIN = "interlobular vein"
    ARCUATE_VEIN = "arcuate vein"
    AFFERENT_ARTERIOLE_OF_MATURING_RENAL_CORPUSCLE = "afferent arteriole of maturing renal corpuscle"
    EFFERENT_ARTERIOLE_OF_MATURING_RENAL_CORPUSCLE = "efferent arteriole of maturing renal corpuscle"
    MATURE_RENAL_CORPUSCLE = "mature renal corpuscle"
    GLOMERULAR_TUFT = "glomerular tuft"
    GLOMERULAR_CAPILLARY_SYSTEM = "glomerular capillary system"
    GLOMERULAR_MESANGIUM_OF_GLOMERULAR_TUFT = "glomerular mesangium of glomerular tuft"
    GLOMERULAR_BASEMENT_MEMBRANE_OF_MATURE_RENAL_CORPUSCLE = "glomerular basement membrane of mature renal corpuscle"
    VISCERAL_EPITHELIUM_OF_MATURE_RENAL_CORPUSCLE = "visceral epithelium of mature renal corpuscle"
    BOWMAN_S_CAPSULE_OF_MATURE_RENAL_CORPUSCLE = "Bowman's capsule of mature renal corpuscle"
    PARIETAL_EPITHELIUM_OF_MATURE_RENAL_CORPUSCLE = "parietal epithelium of mature renal corpuscle"
    URINARY_SPACE_OF_MATURE_RENAL_CORPUSCLE = "urinary space of mature renal corpuscle"
    AFFERENT_ARTERIOLE_OF_MATURE_RENAL_CORPUSCLE = "afferent arteriole of mature renal corpuscle"
    PART_OF_AFFERENT_ARTERIOLE_FORMING_THE_JUXTAGLOMERULAR_COMPLEX = (
        "part of afferent arteriole forming the juxtaglomerular complex"
    )
    REST_OF_AFFERENT_ARTERIOLE = "rest of afferent arteriole"
    EFFERENT_ARTERIOLE_OF_MATURE_RENAL_CORPUSCLE = "efferent arteriole of mature renal corpuscle"
    EXTRAGLOMERULAR_MESANGIUM = "extraglomerular mesangium"
    RENAL_PROXIMAL_TUBULE = "renal proximal tubule"
    RENAL_PROXIMAL_CONVOLUTED_TUBULE = "renal proximal convoluted tubule"
    OUTER_RENAL_MEDULLA = "outer renal medulla"
    LOOP_OF_HENLE__OUTER_MEDULLARY_PORTION = "loop of Henle, outer medullary portion"
    THIN_DESCENDING_LIMB_OF_OUTER_MEDULLA = "thin descending limb of outer medulla"
    THIN_ASCENDING_LIMB_OF_OUTER_MEDULLA = "thin ascending limb of outer medulla"
    PREMACULA_SEGMENT_OF_DISTAL_STRAIGHT_TUBULE_OF_OUTER_MEDULLA = (
        "premacula segment of distal straight tubule of outer medulla"
    )
    OUTER_MEDULLARY_INTERSTITIUM = "outer medullary interstitium"
    OUTER_MEDULLARY_COLLECTING_DUCT = "outer medullary collecting duct"
    INNER_RENAL_MEDULLA = "inner renal medulla"
    LOOP_OF_HENLE__INNER_MEDULLARY_PORTION = "loop of Henle, inner medullary portion"
    THIN_DESCENDING_LIMB_OF_INNER_MEDULLA = "thin descending limb of inner medulla"
    BEND_OF_LOOP_OF_HENLE_INNER_MEDULLARY_PORTION = "bend of loop of Henle inner medullary portion"
    THIN_ASCENDING_LIMB_OF_INNER_MEDULLA = "thin ascending limb of inner medulla"
    INNER_MEDULLARY_INTERSTITIUM = "inner medullary interstitium"
    INNER_MEDULLARY_COLLECTING_DUCT = "inner medullary collecting duct"
    RENAL_ARTERY = "renal artery"
    RENAL_VEIN = "renal vein"
    RENAL_DISTAL_TUBULE = "renal distal tubule"
    DISTAL_CONVOLUTED_TUBULE = "distal convoluted tubule"
    DISTAL_STRAIGHT_TUBULE_OF_RENAL_CORTEX = "distal straight tubule of renal cortex"
    MACULA_DENSA = "macula densa"
    POSTMACULA_SEGMENT_OF_DISTAL_STRAIGHT_TUBULE = "postmacula segment of distal straight tubule"
    COLLECTING_DUCT = "collecting duct"
    STAGE_IV_IMMATURE_NEPHRON = "stage IV immature nephron"
    RENAL_VASCULATURE = "renal vasculature"
    MATURING_NEPHRON = "maturing nephron"
    MATURE_NEPHRON = "mature nephron"
    EARLY_TUBULE = "early tubule"
    DEVELOPING_CAPILLARY_LOOP_STAGE_NEPHRON_GROUP = "developing capillary loop stage nephron group"
    RENAL_INTERSTITIUM = "renal interstitium"
    LATE_TUBULE = "late tubule"
    MALE_ASSOCIATED_REPRODUCTIVE_STRUCTURE = "male associated reproductive structure"
    FEMALE_ASSOCIATED_REPRODUCTIVE_STRUCTURE = "female associated reproductive structure"
    ALLANTOIS_OF_EMBRYONIC_URINARY_SYSTEM = "allantois of embryonic urinary system"
    URETHRAL_PLATE = "urethral plate"
    URETERAL_MESENCHYME_SUBEPITHELIAL_LAYER = "ureteral mesenchyme subepithelial layer"
    URETERAL_MESENCHYME_MIDDLE_LAYER = "ureteral mesenchyme middle layer"
    URETERAL_MESENCHYME_OUTER_LAYER = "ureteral mesenchyme outer layer"
    SUPERFICIAL_CELL_LAYER_OF_URETER = "superficial cell layer of ureter"
    UROTHELIUM_OF_BLADDER = "urothelium of bladder"
    UROTHELIUM_OF_BLADDER_FUNDUS = "urothelium of bladder fundus"
    UROTHELIUM_OF_BLADDER_TRIGONE = "urothelium of bladder trigone"
    MUSCULARIS_MUCOSA_OF_BLADDER = "muscularis mucosa of bladder"
    MUSCULARIS_MUCOSA_OF_FUNDUS = "muscularis mucosa of fundus"
    MUSCULARIS_MUCOSA_OF_TRIGONE = "muscularis mucosa of trigone"
    DETRUSOR_MUSCLE_OF_BLADDER = "detrusor muscle of bladder"
    DETRUSOR_MUSCLE_OF_BLADDER_FUNDUS = "detrusor muscle of bladder fundus"
    DETRUSOR_MUSCLE_OF_BLADDER_TRIGONE = "detrusor muscle of bladder trigone"
    SEROSA_OF_BLADDER = "serosa of bladder"
    SEROSA_OF_BLADDER_FUNDUS = "serosa of bladder fundus"
    SEROSA_OF_TRIGONE = "serosa of trigone"
    VASCULATURE_OF_BLADDER = "vasculature of bladder"
    EPITHELIUM_OF_PELVIC_URETHRA_OF_MALE = "epithelium of pelvic urethra of male"
    MESENCHYMAL_LAYER_OF_PELVIC_URETHRA_OF_MALE = "mesenchymal layer of pelvic urethra of male"
    MUSCLE_LAYER_OF_PELVIC_URETHRA_OF_MALE = "muscle layer of pelvic urethra of male"
    URETHRA_OF_FEMALE = "urethra of female"
    PELVIC_URETHRA_OF_FEMALE = "pelvic urethra of female"
    EPITHELIUM_OF_PELVIC_URETHRA_OF_FEMALE = "epithelium of pelvic urethra of female"
    MESENCHYMAL_LAYER_OF_PELVIC_URETHRA_OF_FEMALE = "mesenchymal layer of pelvic urethra of female"
    MUSCLE_LAYER_OF_PELVIC_URETHRA_OF_FEMALE = "muscle layer of pelvic urethra of female"
    INTERMEDIATE_CELL_LAYER_OF_URETER = "intermediate cell layer of ureter"
    BASAL_CELL_LAYER_OF_URETER = "basal cell layer of ureter"
    URETERAL_SMOOTH_MUSCLE_LAYER = "ureteral smooth muscle layer"
    URETERAL_SMOOTH_MUSCLE_LAYER_SMOOTH_MUSCLE_COMPONENT = "ureteral smooth muscle layer smooth muscle component"
    ADVENTITIA_OF_BLADDER = "adventitia of bladder"
    ADVENTITIA_OF_FUNDUS = "adventitia of fundus"
    ADVENTITIA_OF_BLADDER_TRIGONE = "adventitia of bladder trigone"
    LAMINA_PROPRIA_OF_URETER = "lamina propria of ureter"
    MESENCHYMAL_LAYER_OF_PHALLIC_URETHRA_OF_MALE = "mesenchymal layer of phallic urethra of male"
    MUSCLE_LAYER_OF_PHALLIC_URETHRA_OF_MALE = "muscle layer of phallic urethra of male"
    PROXIMAL_STRAIGHT_TUBULE__OUTER_MEDULLARY_PORTION = "proximal straight tubule, outer medullary portion"
    MESENCHYMAL_STROMA_OF_OVARY = "mesenchymal stroma of ovary"
    DEVELOPING_VASCULATURE_OF_OVARY = "developing vasculature of ovary"
    MESONEPHROS_OF_FEMALE = "mesonephros of female"
    MESONEPHRIC_MESENCHYME_OF_FEMALE = "mesonephric mesenchyme of female"
    MESONEPHRIC_TUBULE_OF_FEMALE = "mesonephric tubule of female"
    NEPHRIC_DUCT_OF_FEMALE__MESONEPHRIC_PORTION = "nephric duct of female, mesonephric portion"
    RETE_OVARII_OF_MESONEPHROS = "rete ovarii of mesonephros"
    COELOMIC_EPITHELIUM_OF_FEMALE_MESONEPHROS = "coelomic epithelium of female mesonephros"
    PARAMESONEPHRIC_DUCT_OF_FEMALE__MESONEPHRIC_PORTION = "paramesonephric duct of female, mesonephric portion"
    PARAMESONEPHRIC_DUCT_OF_FEMALE__REST_OF = "paramesonephric duct of female, rest of"
    NEPHRIC_DUCT_OF_FEMALE__REST_OF = "nephric duct of female, rest of"
    EPITHELIUM_OF_NEPHRIC_DUCT_OF_FEMALE = "epithelium of nephric duct of female"
    MESENCHYME_OF_NEPHRIC_DUCT_OF_FEMALE = "mesenchyme of nephric duct of female"
    EPITHELIUM_OF_REST_OF_PARAMESONEPHRIC_DUCT_OF_FEMALE = "epithelium of rest of paramesonephric duct of female"
    MESENCHYME_OF_REST_OF_PARAMESONEPHRIC_DUCT_OF_FEMALE = "mesenchyme of rest of paramesonephric duct of female"
    MEIOTIC_GERM_CELL_OF_OVARY = "meiotic germ cell of ovary"
    PREMEIOTIC_GERM_CELL_OF_OVARY = "premeiotic germ cell of ovary"
    DEGENERATING_MESONEPHRIC_TUBULE_OF_FEMALE = "degenerating mesonephric tubule of female"
    DEGENERATING_MESONEPHRIC_PORTION_OF_NEPHRIC_DUCT_OF_FEMALE = (
        "degenerating mesonephric portion of nephric duct of female"
    )
    CORTICAL_COMPONENT_OF_MESENCHYMAL_STROMA = "cortical component of mesenchymal stroma"
    MEDULLARY_COMPONENT_OF_MESENCHYMAL_STROMA = "medullary component of mesenchymal stroma"
    UTERINE_HORN_EPITHELIUM = "uterine horn epithelium"
    UTERINE_HORN_MUSCULAR_LAYER = "uterine horn muscular layer"
    EPITHELIAL_LAYER_OF_OVIDUCT = "epithelial layer of oviduct"
    MUSCLE_LAYER_OF_OVIDUCT = "muscle layer of oviduct"
    OUTER_LAYER_OF_OVIDUCT = "outer layer of oviduct"
    EPITHELIUM_OF_UPPER_VAGINA = "epithelium of upper vagina"
    MUSCLE_OF_UPPER_VAGINA = "muscle of upper vagina"
    EPITHELIUM_OF_SINOVAGINAL_BULB = "epithelium of sinovaginal bulb"
    MUSCLE_OF_SINOVAGINAL_BULB = "muscle of sinovaginal bulb"
    OOCYTE_OF_PRIMARY_FOLLICLE = "oocyte of primary follicle"
    FOLLICLE_CELL_OF_PRIMARY_FOLLICLE = "follicle cell of primary follicle"
    PRIMARY_SEX_CORD = "primary sex cord"
    SERTOLI_CELL = "Sertoli cell"
    PERITUBULAR_MYOID_CELL = "peritubular myoid cell"
    DEVELOPING_VASCULATURE_OF_TESTIS = "developing vasculature of testis"
    COELOMIC_VESSEL = "coelomic vessel"
    INTERSTITIAL_VESSEL = "interstitial vessel"
    MESONEPHROS_OF_MALE = "mesonephros of male"
    MESONEPHRIC_MESENCHYME_OF_MALE = "mesonephric mesenchyme of male"
    MESONEPHRIC_TUBULE_OF_MALE = "mesonephric tubule of male"
    PARAMESONEPHRIC_DUCT_OF_MALE__MESONEPHRIC_PORTION = "paramesonephric duct of male, mesonephric portion"
    DEGENERATING_MESONEPHRIC_PORTION_OF_MALE_PARAMESONEPHRIC_DUCT = (
        "degenerating mesonephric portion of male paramesonephric duct"
    )
    GENITAL_TUBERCLE_MESENCHYME_OF_MALE = "genital tubercle mesenchyme of male"
    GENITAL_TUBERCLE_SKIN_OF_MALE = "genital tubercle skin of male"
    NEPHRIC_DUCT_OF_MALE__REST_OF = "nephric duct of male, rest of"
    MESENCHYME_OF_NEPHRIC_DUCT_OF_MALE = "mesenchyme of nephric duct of male"
    EPITHELIUM_OF_NEPHRIC_DUCT_OF_MALE = "epithelium of nephric duct of male"
    PARAMESONEPHRIC_DUCT_OF_MALE__REST_OF = "paramesonephric duct of male, rest of"
    EPITHELIUM_OF_REST_OF_PARAMESONEPHRIC_DUCT_OF_MALE = "epithelium of rest of paramesonephric duct of male"
    MESENCHYME_OF_REST_OF_PARAMESONEPHRIC_DUCT_OF_MALE = "mesenchyme of rest of paramesonephric duct of male"
    DEGENERATING_REST_OF_PARAMESONEPHRIC_DUCT_OF_MALE = "degenerating rest of paramesonephric duct of male"
    EPITHELIUM_OF_DEGENERATING_REST_OF_PARAMESONEPHRIC_DUCT_OF_MALE = (
        "epithelium of degenerating rest of paramesonephric duct of male"
    )
    MESENCHYME_OF_DEGENERATING_REST_OF_PARAMESONEPHRIC_DUCT_OF_MALE = (
        "mesenchyme of degenerating rest of paramesonephric duct of male"
    )
    DISTAL_GENITAL_TUBERCLE_OF_MALE = "distal genital tubercle of male"
    MESENCHYME_OF_DISTAL_GENITAL_TUBERCLE_OF_MALE = "mesenchyme of distal genital tubercle of male"
    SKIN_OF_DISTAL_GENITAL_TUBERCLE_OF_MALE = "skin of distal genital tubercle of male"
    PREPUTIAL_SWELLING_OF_MALE = "preputial swelling of male"
    SKIN_OF_MALE_PREPUTIAL_SWELLING = "skin of male preputial swelling"
    MESENCHYME_OF_MALE_PREPUTIAL_SWELLING = "mesenchyme of male preputial swelling"
    GLANS_OF_MALE_GENITAL_TUBERCLE = "glans of male genital tubercle"
    MESENCHYME_OF_GLANS_OF_MALE_GENITAL_TUBERCLE = "mesenchyme of glans of male genital tubercle"
    SKIN_OF_GLANS_OF_MALE_GENITAL_TUBERCLE = "skin of glans of male genital tubercle"
    MUSCLE_OF_DUCTUS_DEFERENS = "muscle of ductus deferens"
    EPITHELIUM_OF_DUCTUS_DEFERENS = "epithelium of ductus deferens"
    MALE_CYSTIC_VESICULAR_APPENDAGE = "male cystic vesicular appendage"
    MALE_INGUINAL_RING = "male inguinal ring"
    MALE_EXTERNAL_INGUINAL_RING = "male external inguinal ring"
    MALE_INTERNAL_INGUINAL_RING = "male internal inguinal ring"
    SCROTAL_FOLD = "scrotal fold"
    VASCULATURE_OF_TESTIS = "vasculature of testis"
    REST_OF_MESENTERY = "rest of mesentery"
    GENITAL_TUBERCLE_MESENCHYME_OF_FEMALE = "genital tubercle mesenchyme of female"
    GENITAL_TUBERCLE_SKIN_OF_FEMALE = "genital tubercle skin of female"
    VASCULATURE_OF_OVARY = "vasculature of ovary"
    URETERAL_SMOOTH_MUSCLE_LAYER_INNER_LONGITUDINAL_MUSCLE = "ureteral smooth muscle layer inner longitudinal muscle"
    URETERAL_SMOOTH_MUSCLE_LAYER_OUTER_CIRCULAR_MUSCLE = "ureteral smooth muscle layer outer circular muscle"
    MEDIAN_UMBILICAL_LIGAMENT = "median umbilical ligament"
    BULBAR_PART_OF_URETHRA_OF_MALE = "bulbar part of urethra of male"
    MALE_URETHRAL_GLAND = "male urethral gland"
    FEMALE_URETHRAL_GLAND = "female urethral gland"
    LEYDIG_CELL = "Leydig cell"
    SEGMENT_1_OF_PROXIMAL_TUBULE = "segment 1 of proximal tubule"
    SEGMENT_2_OF_PROXIMAL_TUBULE = "segment 2 of proximal tubule"
    PROXIMAL_STRAIGHT_TUBULE__CORTICAL_PORTION = "proximal straight tubule, cortical portion"
    OUTER_STRIPE_OF_OUTER_MEDULLA = "outer stripe of outer medulla"
    LOOP_OF_HENLE_OF_OUTER_STRIPE_OF_OUTER_MEDULLA = "loop of Henle of outer stripe of outer medulla"
    PROXIMAL_STRAIGHT_TUBULE_OF_OUTER_STRIPE_OF_OUTER_MEDULLA = (
        "proximal straight tubule of outer stripe of outer medulla"
    )
    DISTAL_STRAIGHT_TUBULE_OF_OUTER_STRIPE_OF_OUTER_MEDULLA = "distal straight tubule of outer stripe of outer medulla"
    INNER_STRIPE_OF_OUTER_MEDULLA = "inner stripe of outer medulla"
    LOOP_OF_HENLE_OF_INNER_STRIPE_OF_OUTER_MEDULLA = "loop of Henle of inner stripe of outer medulla"
    THIN_DESCENDING_LIMB_OF_INNER_STRIPE_OF_OUTER_MEDULLA = "thin descending limb of inner stripe of outer medulla"
    PREMACULA_SEGMENT_OF_DISTAL_STRAIGHT_TUBULE_OF_INNER_STRIPE_OF_INNER_MEDULLA = (
        "premacula segment of distal straight tubule of inner stripe of inner medulla"
    )
    RENAL_NERVE = "renal nerve"
    EPITHELIUM_OF_CAUDA_OF_EPIDIDYMIS = "epithelium of cauda of epididymis"
    TRUNK_DERMOMYOTOME = "trunk dermomyotome"
    TAIL_DERMOMYOTOME = "tail dermomyotome"
    TESTICULAR_BRANCH_OF_GENITAL_NERVE = "testicular branch of genital nerve"
    EPIDERMAL_LAYER_OF_SCROTUM = "epidermal layer of scrotum"
    BASAL_EPIDERMAL_LAYER_OF_SCROTUM = "basal epidermal layer of scrotum"
    STRATIFIED_EPIDERMAL_LAYER_OF_SCROTUM = "stratified epidermal layer of scrotum"
    DERMAL_LAYER_OF_SCROTUM = "dermal layer of scrotum"
    HAIR_FOLLICLE_OF_SCROTUM_DERMAL_LAYER = "hair follicle of scrotum dermal layer"
    SWEAT_GLAND_OF_SCROTUM_DERMAL_LAYER = "sweat gland of scrotum dermal layer"
    SEPTUM_OF_SCROTUM = "septum of scrotum"
    EPITHELIAL_LAYER_OF_EFFERENT_DUCT = "epithelial layer of efferent duct"
    REST_OF_EFFERENT_DUCT = "rest of efferent duct"
    MUSCLE_LAYER_OF_CAUDA_OF_EPIDIDYMIS = "muscle layer of cauda of epididymis"
    ADVENTITIA_OF_CAUDA_OF_EPIDIDYMIS = "adventitia of cauda of epididymis"
    EPITHELIUM_OF_CORPUS_EPIDIDYMIS = "epithelium of corpus epididymis"
    MUSCLE_LAYER_OF_CORPUS_OF_EPIDIDYMIS = "muscle layer of corpus of epididymis"
    ADVENTITIA_OF_CORPUS_OF_EPIDIDYMIS = "adventitia of corpus of epididymis"
    EPITHELIUM_OF_CAPUT_OF_EPIDIDYMIS = "epithelium of caput of epididymis"
    MUSCLE_LAYER_OF_CAPUT_OF_EPIDIDYMIS = "muscle layer of caput of epididymis"
    ADVENTITIA_OF_CAPUT_OF_EPIDIDYMIS = "adventitia of caput of epididymis"
    SEROSA_OF_DUCTUS_DEFERENS = "serosa of ductus deferens"
    OS_PENIS = "os penis"
    EPITHELIUM_OF_SEMINAL_VESICLE = "epithelium of seminal vesicle"
    MUSCLE_OF_SEMINAL_VESICLE = "muscle of seminal vesicle"
    SEROSA_OF_SEMINAL_VESICLE = "serosa of seminal vesicle"
    EPITHELIUM_OF_EJACULATORY_DUCT = "epithelium of ejaculatory duct"
    EPITHELIUM_OF_BULBOURETHRAL_GLAND_OF_MALE = "epithelium of bulbourethral gland of male"
    LAMINA_PROPRIA_OF_BULBOURETHRAL_GLAND_OF_MALE = "lamina propria of bulbourethral gland of male"
    ANTERIOR_PROSTATE_GLAND = "anterior prostate gland"
    ANTERIOR_PROSTATIC_EPITHELIUM = "anterior prostatic epithelium"
    GLANDULAR_EPITHELIUM_OF_ANTERIOR_PROSTATE = "glandular epithelium of anterior prostate"
    BASAL_EPITHELIUM_OF_ANTERIOR_PROSTATE = "basal epithelium of anterior prostate"
    ANTERIOR_PROSTATIC_INTERSTITIUM = "anterior prostatic interstitium"
    SMOOTH_MUSCLE_OF_ANTERIOR_PROSTATE = "smooth muscle of anterior prostate"
    REST_OF_INTERSTITIUM_OF_ANTERIOR_PROSTATE = "rest of interstitium of anterior prostate"
    DORSAL_PROSTATE_GLAND = "dorsal prostate gland"
    DORSAL_PROSTATIC_EPITHELIUM = "dorsal prostatic epithelium"
    GLANDULAR_EPITHELIUM_OF_DORSAL_PROSTATE = "glandular epithelium of dorsal prostate"
    BASAL_EPITHELIUM_OF_DORSAL_PROSTATE = "basal epithelium of dorsal prostate"
    DORSAL_PROSTATIC_INTERSTITIUM = "dorsal prostatic interstitium"
    SMOOTH_MUSCLE_OF_DORSAL_PROSTATE = "smooth muscle of dorsal prostate"
    REST_OF_INTERSTITIUM_OF_DORSAL_PROSTATE = "rest of interstitium of dorsal prostate"
    LATERAL_PROSTATE_GLAND = "lateral prostate gland"
    LATERAL_PROSTATIC_EPITHELIUM = "lateral prostatic epithelium"
    GLANDULAR_EPITHELIUM_OF_LATERAL_PROSTATE = "glandular epithelium of lateral prostate"
    BASAL_EPITHELIUM_OF_LATERAL_PROSTATE = "basal epithelium of lateral prostate"
    LATERAL_PROSTATIC_INTERSTITIUM = "lateral prostatic interstitium"
    SMOOTH_MUSCLE_OF_LATERAL_PROSTATE = "smooth muscle of lateral prostate"
    REST_OF_INTERSTITIUM_OF_LATERAL_PROSTATE = "rest of interstitium of lateral prostate"
    VENTRAL_PROSTATE_GLAND = "ventral prostate gland"
    VENTRAL_PROSTATIC_EPITHELIUM = "ventral prostatic epithelium"
    GLANDULAR_EPITHELIUM_OF_VENTRAL_PROSTATE = "glandular epithelium of ventral prostate"
    BASAL_EPITHELIUM_OF_VENTRAL_PROSTATE = "basal epithelium of ventral prostate"
    VENTRAL_PROSTATIC_INTERSTITIUM = "ventral prostatic interstitium"
    SMOOTH_MUSCLE_OF_VENTRAL_PROSTATE = "smooth muscle of ventral prostate"
    REST_OF_INTERSTITIUM_OF_VENTRAL_PROSTATE = "rest of interstitium of ventral prostate"
    CORPUS_CAVERNOSUM_OF_PENILE_BODY = "corpus cavernosum of penile body"
    LAMINA_PROPRIA_OF_PENILE_BODY = "lamina propria of penile body"
    CORPUS_CAVERNOSUM_OF_GLANS_PENIS = "corpus cavernosum of glans penis"
    LAMINA_PROPRIA_OF_GLANS_PENIS = "lamina propria of glans penis"
    OOCYTE_OF_SECONDARY_FOLLICLE = "oocyte of secondary follicle"
    FOLLICLE_CELL_OF_SECONDARY_FOLLICLE = "follicle cell of secondary follicle"
    CUMULUS_OF_SECONDARY_FOLLICLE = "cumulus of secondary follicle"
    MURAL_OF_SECONDARY_FOLLICLE = "mural of secondary follicle"
    THECA_OF_SECONDARY_FOLLICLE = "theca of secondary follicle"
    OVARY_ATRETIC_FOLLICLE = "ovary atretic follicle"
    OOCYTE_OF_ATRETIC_FOLLICLE = "oocyte of atretic follicle"
    APOPTOTIC_GERMINAL_CELL = "apoptotic germinal cell"
    OVARY_RUPTURED_FOLLICLE = "ovary ruptured follicle"
    THECA_OF_RUPTURED_FOLLICLE = "theca of ruptured follicle"
    GERMINAL_CELL_OF_RUPTURED_FOLLICLE = "germinal cell of ruptured follicle"
    CORPUS_LUTEUM = "corpus luteum"
    CORPUS_ALBICANS = "corpus albicans"
    OVIDUCT_INFUNDIBULUM = "oviduct infundibulum"
    EPITHELIAL_LAYER_OF_INFUNDIBULUM = "epithelial layer of infundibulum"
    SEROSA_OF_INFUNDIBULUM = "serosa of infundibulum"
    REST_OF_OVIDUCT = "rest of oviduct"
    EPITHELIAL_LAYER_OF_REST_OF_OVIDUCT = "epithelial layer of rest of oviduct"
    SEROSA_OF_REST_OF_OVIDUCT = "serosa of rest of oviduct"
    UTERUS = "uterus"
    ENDOMETRIUM = "endometrium"
    UTERINE_GLAND = "uterine gland"
    REST_OF_ENDOMETRIUM = "rest of endometrium"
    MYOMETRIUM = "myometrium"
    ADVENTITIA_OF_UTERUS = "adventitia of uterus"
    CERVIX_UTERI = "cervix uteri"
    EPITHELIAL_LAYER_OF_CERVIX_UTERI = "epithelial layer of cervix uteri"
    MUSCLE_LAYER_OF_CERVIX_UTERI = "muscle layer of cervix uteri"
    ADVENTITIA_OF_CERVIX_UTERI = "adventitia of cervix uteri"
    GLAND_OF_LOWER_VAGINAL_EPITHELIUM = "gland of lower vaginal epithelium"
    REST_OF_EPITHELIAL_LAYER_OF_LOWER_VAGINA = "rest of epithelial layer of lower vagina"
    ADVENTITIA_OF_SINOVAGINAL_BULB = "adventitia of sinovaginal bulb"
    GLAND_OF_UPPER_VAGINAL_EPITHELIUM = "gland of upper vaginal epithelium"
    REST_OF_EPITHELIUM_OF_UPPER_VAGINA = "rest of epithelium of upper vagina"
    ADVENTITIA_OF_UPPER_VAGINA = "adventitia of upper vagina"
    URETERAL_MESENCHYME__REST_OF = "ureteral mesenchyme, rest of"
    ENDOTHELIUM_OF_RENAL_ARTERY = "endothelium of renal artery"
    SMOOTH_MUSCLE_LAYER_OF_RENAL_ARTERY = "smooth muscle layer of renal artery"
    ENDOTHELIUM_OF_RENAL_VEIN = "endothelium of renal vein"
    SMOOTH_MUSCLE_LAYER_OF_RENAL_VEIN = "smooth muscle layer of renal vein"
    GLOMERULAR_MESANGIUM_OF_MATURING_RENAL_CORPUSCLE = "glomerular mesangium of maturing renal corpuscle"
    RENAL_CORTICAL_CAPILLARY = "renal cortical capillary"
    RENAL_MEDULLARY_CAPILLARY = "renal medullary capillary"
    MUSCLE_LAYER_OF_REST_OF_OVIDUCT = "muscle layer of rest of oviduct"
    CIRCULAR_MUSCLE_LAYER_OF_REST_OF_OVIDUCT = "circular muscle layer of rest of oviduct"
    LONGITUDINAL_MUSCLE_LAYER_OF_REST_OF_OVIDUCT = "longitudinal muscle layer of rest of oviduct"
    CIRCULAR_MUSCLE_OF_DUCTUS_DEFERENS = "circular muscle of ductus deferens"
    LONGITUDINAL_MUSCLE_OF_DUCTUS_DEFERENS = "longitudinal muscle of ductus deferens"
    CRANIAL_SUSPENSORY_LIGAMENT = "cranial suspensory ligament"
    EPITHELIUM_OF_PARAMESONEPHRIC_DUCT_OF_FEMALE__MESONEPHRIC_PORTION = (
        "epithelium of paramesonephric duct of female, mesonephric portion"
    )
    MESENCHYME_OF_PARAMESONEPHRIC_DUCT_OF_FEMALE__MESONEPHRIC_PORTION = (
        "mesenchyme of paramesonephric duct of female, mesonephric portion"
    )
    GENITAL_SWELLING = "genital swelling"
    DORSAL_GENITAL_SWELLING = "dorsal genital swelling"
    VENTROLATERAL_GENITAL_SWELLING = "ventrolateral genital swelling"
    UROTHELIUM_OF_VENTRAL_BLADDER_NECK = "urothelium of ventral bladder neck"
    LAMINA_PROPRIA_OF_BLADDER = "lamina propria of bladder"
    LAMINA_PROPRIA_OF_BLADDER_FUNDUS = "lamina propria of bladder fundus"
    LAMINA_PROPRIA_OF_BLADDER_TRIGONE = "lamina propria of bladder trigone"
    LAMINA_PROPRIA_OF_VENTRAL_BLADDER_NECK = "lamina propria of ventral bladder neck"
    DETRUSOR_MUSCLE_OF_VENTRAL_BLADDER_NECK = "detrusor muscle of ventral bladder neck"
    SEROSA_OF_NECK = "serosa of neck"
    MUSCULARIS_MUCOSA_OF_NECK = "muscularis mucosa of neck"
    ADVENTITIA_OF_VENTRAL_BLADDER_NECK = "adventitia of ventral bladder neck"
    URETERAL_SMOOTH_MUSCLE_LAYER_INTERSTITIUM = "ureteral smooth muscle layer interstitium"
    LABIAL_SWELLING = "labial swelling"
    MESENCHYME_OF_LABIAL_SWELLING = "mesenchyme of labial swelling"
    LABIAL_SWELLING_SKIN = "labial swelling skin"
    MESENCHYME_OF_LABIAL_FOLD = "mesenchyme of labial fold"
    LABIAL_FOLD_SKIN = "labial fold skin"
    ADVENTITIA_OF_EPIDIDYMIS = "adventitia of epididymis"
    EPITHELIUM_OF_EPIDIDYMIS = "epithelium of epididymis"
    JUXTAGLOMERULAR_COMPLEX = "juxtaglomerular complex"
    COELOMIC_EPITHELIUM_OF_MALE_MESONEPHROS = "coelomic epithelium of male mesonephros"
    EPITHELIUM_OF_PHALLIC_URETHRA_OF_MALE = "epithelium of phallic urethra of male"
    EPITHELIUM_OF_PHALLIC_URETHRA_OF_FEMALE = "epithelium of phallic urethra of female"
    PREPUTIAL_SWELLING_OF_FEMALE = "preputial swelling of female"
    DISTAL_GENITAL_TUBERCLE_OF_FEMALE = "distal genital tubercle of female"
    MESENCHYME_OF_DISTAL_GENITAL_TUBERCLE_OF_FEMALE = "mesenchyme of distal genital tubercle of female"
    SKIN_OF_DISTAL_GENITAL_TUBERCLE_OF_FEMALE = "skin of distal genital tubercle of female"
    MESENCHYME_OF_FEMALE_PREPUTIAL_SWELLING = "mesenchyme of female preputial swelling"
    SKIN_OF_FEMALE_PREPUTIAL_SWELLING = "skin of female preputial swelling"
    SCROTAL_SWELLING = "scrotal swelling"
    MESENCHYME_OF_SCROTAL_SWELLING = "mesenchyme of scrotal swelling"
    SCROTAL_SWELLING_SKIN = "scrotal swelling skin"
    TUNICA_ALBUGINEA_OF_OVARY = "tunica albuginea of ovary"
    CRANIAL_MESONEPHRIC_TUBULE_OF_MALE = "cranial mesonephric tubule of male"
    MESONEPHRIC_GLOMERULUS_OF_MALE = "mesonephric glomerulus of male"
    REST_OF_CRANIAL_MESONEPHRIC_TUBULE_OF_MALE = "rest of cranial mesonephric tubule of male"
    CAUDAL_MESONEPHRIC_TUBULE_OF_MALE = "caudal mesonephric tubule of male"
    CRANIAL_MESONEPHRIC_TUBULE_OF_FEMALE = "cranial mesonephric tubule of female"
    MESONEPHRIC_GLOMERULUS_OF_FEMALE = "mesonephric glomerulus of female"
    REST_OF_CRANIAL_MESONEPHRIC_TUBULE_OF_FEMALE = "rest of cranial mesonephric tubule of female"
    CAUDAL_MESONEPHRIC_TUBULE_OF_FEMALE = "caudal mesonephric tubule of female"
    CRANIAL_MESONEPHRIC_TUBULE = "cranial mesonephric tubule"
    REST_OF_CRANIAL_MESONEPHRIC_TUBULE = "rest of cranial mesonephric tubule"
    MESONEPHRIC_GLOMERULUS = "mesonephric glomerulus"
    CAUDAL_MESONEPHRIC_TUBULE = "caudal mesonephric tubule"
    MESENCHYME_OF_SCROTAL_FOLD = "mesenchyme of scrotal fold"
    SCROTAL_FOLD_SKIN = "scrotal fold skin"
    MESENCHYME_OF_GLANS_PENIS = "mesenchyme of glans penis"
    GLANS_PENIS_SKIN = "glans penis skin"
    PREPUCE_MESENCHYME_OF_MALE = "prepuce mesenchyme of male"
    MALE_PREPUCE_SKIN = "male prepuce skin"
    PREPUTIAL_GLAND_OF_MALE = "preputial gland of male"
    MESENCHYME_OF_PENILE_BODY = "mesenchyme of penile body"
    PENILE_BODY_SKIN = "penile body skin"
    OUTER_LAYER_OF_UTERINE_HORN = "outer layer of uterine horn"
    PREPUCE_OF_FEMALE = "prepuce of female"
    FEMALE_PREPUCE_MESENCHYME = "female prepuce mesenchyme"
    FEMALE_PREPUCE_SKIN = "female prepuce skin"
    PREPUTIAL_GLAND_OF_FEMALE = "preputial gland of female"
    MESENCHYME_OF_GLANS_CLITORIS = "mesenchyme of glans clitoris"
    SKIN_OF_GLANS_OF_FEMALE_GENITAL_TUBERCLE = "skin of glans of female genital tubercle"
    MUSCLE_LAYER_OF_EPIDIDYMIS = "muscle layer of epididymis"
    OVARY_PRIMORDIAL_FOLLICLE = "ovary primordial follicle"
    OOCYTE_OF_PRIMORDIAL_FOLLICLE = "oocyte of primordial follicle"
    OVARY_TERTIARY_FOLLICLE = "ovary tertiary follicle"
    OOCYTE_OF_TERTIARY_FOLLICLE = "oocyte of tertiary follicle"
    FOLLICLE_CELL_OF_TERTIARY_FOLLICLE = "follicle cell of tertiary follicle"
    THECA_OF_TERTIARY_FOLLICLE = "theca of tertiary follicle"
    ANTRUM_OF_TERTIARY_FOLLICLE = "antrum of tertiary follicle"
    CUMULUS_OF_TERTIARY_FOLLICLE = "cumulus of tertiary follicle"
    MURAL_OF_TERTIARY_FOLLICLE = "mural of tertiary follicle"
    OVARY_MATURE_FOLLICLE = "ovary mature follicle"
    OOCYTE_OF_MATURE_FOLLICLE = "oocyte of mature follicle"
    GERMINAL_CELL_OF_MATURE_FOLLICLE = "germinal cell of mature follicle"
    CUMULUS_OF_MATURE_FOLLICLE = "cumulus of mature follicle"
    MURAL_OF_MATURE_FOLLICLE = "mural of mature follicle"
    THECA_OF_MATURE_FOLLICLE = "theca of mature follicle"
    ANTRUM_OF_MATURE_FOLLICLE = "antrum of mature follicle"
    VASCULATURE_OF_SCROTUM = "vasculature of scrotum"
    VASCULATURE_OF_DUCTUS_DEFERENS = "vasculature of ductus deferens"
    VASCULATURE_OF_EPIDIDYMIS = "vasculature of epididymis"
    VASCULATURE_OF_SEMINAL_VESICLE = "vasculature of seminal vesicle"
    VASCULATURE_OF_EJACULATORY_DUCT = "vasculature of ejaculatory duct"
    VASCULATURE_OF_BULBOURETHRAL_GLAND_OF_MALE = "vasculature of bulbourethral gland of male"
    VASCULATURE_OF_PROSTATE_GLAND = "vasculature of prostate gland"
    VASCULATURE_OF_PENIS = "vasculature of penis"
    VASCULATURE_OF_OVIDUCT = "vasculature of oviduct"
    VASCULATURE_OF_UTERUS = "vasculature of uterus"
    VASCULATURE_OF_VAGINA = "vasculature of vagina"
    CLITORIS = "clitoris"
    BODY_OF_CLITORIS = "body of clitoris"
    OS_CLITORIDIS = "os clitoridis"
    EPITHELIUM_OF_BODY_OF_CLITORIS = "epithelium of body of clitoris"
    LAMINA_PROPRIA_OF_BODY_OF_CLITORIS = "lamina propria of body of clitoris"
    GLANS_CLITORIS = "glans clitoris"
    EPITHELIUM_OF_GLANS_CLITORIS = "epithelium of glans clitoris"
    LAMINA_PROPRIA_OF_GLANS_CLITORIS = "lamina propria of glans clitoris"
    VASCULATURE_OF_CLITORIS = "vasculature of clitoris"
    LABIA = "labia"
    LABIA_SKIN = "labia skin"
    LAMINA_PROPRIA_OF_LABIA = "lamina propria of labia"
    GLAND_OF_LABIA = "gland of labia"
    VASCULATURE_OF_LABIA = "vasculature of labia"
    URETERAL_VASCULATURE = "ureteral vasculature"
    PRIMITIVE_BLADDER = "primitive bladder"
    UROTHELIUM_OF_PRIMITIVE_BLADDER = "urothelium of primitive bladder"
    MESENCHYMAL_LAYER_OF_PRIMITIVE_BLADDER = "mesenchymal layer of primitive bladder"
    CAUDAL_UROGENITAL_SINUS = "caudal urogenital sinus"
    CAUDAL_UROGENITAL_SINUS_EPITHELIUM = "caudal urogenital sinus epithelium"
    CAUDAL_UROGENITAL_SINUS_MESENCHYME = "caudal urogenital sinus mesenchyme"
    URETHRAL_FOLD = "urethral fold"
    URETHRA = "urethra"
    PELVIC_URETHRA = "pelvic urethra"
    EPITHELIUM_OF_PELVIC_URETHRA = "epithelium of pelvic urethra"
    MESENCHYMAL_LAYER_OF_PELVIC_URETHRA = "mesenchymal layer of pelvic urethra"
    MUSCLE_LAYER_OF_PELVIC_URETHRA = "muscle layer of pelvic urethra"
    PHALLIC_URETHRA = "phallic urethra"
    URETHRAL_PLATE_OF_PHALLIC_URETHRA = "urethral plate of phallic urethra"
    CLITORAL_URETHRA = "clitoral urethra"
    URETHRAL_PLATE_EPITHELIUM_OF_CLITORAL_URETHRA = "urethral plate epithelium of clitoral urethra"
    URETHRAL_FOLD_OF_FEMALE = "urethral fold of female"
    EPITHELIUM_OF_CLITORAL_URETHRA = "epithelium of clitoral urethra"
    MESENCHYMAL_LAYER_OF_PHALLIC_URETHRA_OF_FEMALE = "mesenchymal layer of phallic urethra of female"
    MUSCLE_LAYER_OF_PHALLIC_URETHRA_OF_FEMALE = "muscle layer of phallic urethra of female"
    PROSTATIC_URETHRA = "prostatic urethra"
    PENILE_URETHRA = "penile urethra"
    EPITHELIUM_OF_PENILE_URETHRA = "epithelium of penile urethra"
    MESENCHYMAL_LAYER_OF_PENILE_URETHRA = "mesenchymal layer of penile urethra"
    MUSCLE_LAYER_OF_PENILE_URETHRA = "muscle layer of penile urethra"
    EPITHELIUM_OF_PROSTATIC_URETHRA = "epithelium of prostatic urethra"
    MESENCHYMAL_LAYER_OF_PROSTATIC_URETHRA = "mesenchymal layer of prostatic urethra"
    MUSCLE_LAYER_OF_PROSTATIC_URETHRA = "muscle layer of prostatic urethra"
    GLOMERULAR_BASEMENT_MEMBRANE_OF_MATURING_RENAL_CORPUSCLE = (
        "glomerular basement membrane of maturing renal corpuscle"
    )
    MALE_EXTERNAL_GENITALIA = "male external genitalia"
    FEMALE_EXTERNAL_GENITALIA = "female external genitalia"
    EPITHELIUM_OF_VAGINA = "epithelium of vagina"
    MUSCLE_OF_VAGINA = "muscle of vagina"
    ADVENTITIA_OF_VAGINA = "adventitia of vagina"
    STRAIGHT_LIMB_OF_IMMATURE_LOOP_OF_HENLE = "straight limb of immature loop of Henle"
    BEND_OF_IMMATURE_LOOP_OF_HENLE = "bend of immature loop of Henle"
    OOCYTE = "oocyte"
    PRE_GRANULOSA_CELL = "pre-granulosa cell"
    REST_OF_CORTICAL_COMPONENT_OF_MESENCHYMAL_STROMA_OF_OVARY = (
        "rest of cortical component of mesenchymal stroma of ovary"
    )
    URETERIC_TREE_TERMINAL_BRANCH_EXCLUDING_TIP_ITSELF = "ureteric tree terminal branch excluding tip itself"
    OVIGEROUS_CORD = "ovigerous cord"
    DERMOMYOTOME = "dermomyotome"
    TRUNK_MYOCOELE = "trunk myocoele"
    TAIL_MYOCOELE = "tail myocoele"
    MYOCOELE = "myocoele"
    TRUNK_SCLEROTOME = "trunk sclerotome"
    TAIL_SCLEROTOME = "tail sclerotome"
    SCLEROTOME = "sclerotome"
    SOMITE = "somite"
    DEVELOPING_VASCULATURE_OF_CAUDA_EPIDIDYMIS = "developing vasculature of cauda epididymis"
    DEVELOPING_VASCULATURE_OF_CORPUS_EPIDIDYMIS = "developing vasculature of corpus epididymis"
    DEVELOPING_VASCULATURE_OF_CAPUT_EPIDIDYMIS = "developing vasculature of caput epididymis"
    DEVELOPING_VASCULATURE_OF_UTERINE_HORN = "developing vasculature of uterine horn"
    DEVELOPING_VASCULATURE_OF_OVIDUCT = "developing vasculature of oviduct"
    VASCULATURE_OF_UPPER_VAGINA = "vasculature of upper vagina"
    VASCULATURE_OF_SINOVAGINAL_BULB = "vasculature of sinovaginal bulb"
    DEVELOPING_VASCULATURE_OF_VAGINA = "developing vasculature of vagina"
    DEVELOPING_VASCULATURE_SYSTEM_OF_EPIDIDYMIS = "developing vasculature system of epididymis"
    SURFACE_EPITHELIUM_OF_TESTIS = "surface epithelium of testis"
    FOLLICLE_CELL_OF_PRIMORDIAL_FOLLICLE = "follicle cell of primordial follicle"
    CREMASTER_MUSCLE = "cremaster muscle"
    THECA_OF_ATRETIC_FOLLICLE = "theca of atretic follicle"
    GRANULOSA_OF_RUPTURED_FOLLICLE = "granulosa of ruptured follicle"
    MUSCLE_LAYER_OF_INFUNDIBULUM = "muscle layer of infundibulum"
    LONGITUDINAL_MUSCLE_LAYER_OF_INFUNDIBULUM = "longitudinal muscle layer of infundibulum"
    CIRCULAR_MUSCLE_LAYER_OF_INFUNDIBULUM = "circular muscle layer of infundibulum"
    RESIDENT_MACROPHAGE = "resident macrophage"
    REST_OF_RENAL_INTERSTITIUM = "rest of renal interstitium"
    RENAL_FIBROBLASTS = "renal fibroblasts"
    ANLAGE_OF_LOOP_OF_HENLE_OF_CORTICAL_RENAL_TUBULE = "anlage of loop of Henle of cortical renal tubule"
    RENAL_CORTICAL_FIBROBLASTS = "renal cortical fibroblasts"
    RESIDENT_MACROPHAGE_OF_THE_CORTEX = "resident macrophage of the cortex"
    REST_OF_RENAL_CORTICAL_INTERSTITIUM = "rest of renal cortical interstitium"
    RENAL_CORTICAL_LYMPHATICS = "renal cortical lymphatics"
    RENAL_CORTICAL_NERVES = "renal cortical nerves"
    RENAL_MEDULLARY_FIBROBLASTS = "renal medullary fibroblasts"
    RESIDENT_MACROPHAGE_OF_THE_MEDULLA = "resident macrophage of the medulla"
    REST_OF_RENAL_MEDULLARY_INTERSTITIUM = "rest of renal medullary interstitium"
    RENAL_MEDULLARY_LYMPHATICS = "renal medullary lymphatics"
    RENAL_MEDULLARY_NERVES = "renal medullary nerves"
    PERIHILAR_FIBROBLAST = "perihilar fibroblast"
    RESIDENT_MACROPHAGE_OF_THE_PERIHILAR_INTERSTITIUM = "resident macrophage of the perihilar interstitium"
    REST_OF_PERIHILAR_INTERSTITIUM = "rest of perihilar interstitium"
    PREMACULA_SEGMENT_OF_DISTAL_STRAIGHT_TUBULE_OF_RENAL_CORTEX = (
        "premacula segment of distal straight tubule of renal cortex"
    )
    OUTER_MEDULLARY_FIBROBLAST = "outer medullary fibroblast"
    RESIDENT_MACROPHAGE_OF_THE_OUTER_MEDULLA = "resident macrophage of the outer medulla"
    REST_OF_OUTER_MEDULLARY_INTERSTITIUM = "rest of outer medullary interstitium"
    INNER_MEDULLARY_FIBROBLAST = "inner medullary fibroblast"
    RESIDENT_MACROPHAGE_OF_THE_INNER_MEDULLA = "resident macrophage of the inner medulla"
    REST_OF_INNER_MEDULLARY_INTERSTITIUM = "rest of inner medullary interstitium"
    MEDULLARY_RENAL_TUBULE = "medullary renal tubule"
    REST_OF_MEDULLARY_RENAL_TUBULE = "rest of medullary renal tubule"
    PERITUBULAR_FIBROBLASTS_OF_CORTEX = "peritubular fibroblasts of cortex"
    PERIARTERIAL_FIBROBLASTS_OF_CORTEX = "periarterial fibroblasts of cortex"
    PERITUBULAR_FIBROBLASTS_OF_OUTER_MEDULLA = "peritubular fibroblasts of outer medulla"
    PERIARTERIAL_FIBROBLASTS_OF_OUTER_MEDULLA = "periarterial fibroblasts of outer medulla"
    PERITUBULAR_FIBROBLASTS_OF_INNER_MEDULLA = "peritubular fibroblasts of inner medulla"
    PERIARTERIAL_FIBROBLASTS_OF_INNER_MEDULLA = "periarterial fibroblasts of inner medulla"
    RENAL_ARTERIAL_SYSTEM = "renal arterial system"
    RENAL_CAPILLARY = "renal capillary"
    RENAL_VENOUS_SYSTEM = "renal venous system"
    RENAL_LARGE_BLOOD_VESSEL = "renal large blood vessel"
    RENAL_SMALL_BLOOD_VESSEL = "renal small blood vessel"
    SEMINIFEROUS_TUBULE = "seminiferous tubule"
    SPERMATOGONIUM = "spermatogonium"
    SPERMATOCYTE = "spermatocyte"
    SPERMATID = "spermatid"
    SPERMATOZOON = "spermatozoon"
    MESENCHYME_OF_UROGENITAL_SINUS = "mesenchyme of urogenital sinus"
    MESENCHYME_OF_PROXIMAL_GENITAL_TUBERCLE_OF_MALE = "mesenchyme of proximal genital tubercle of male"
    SKIN_OF_PROXIMAL_GENITAL_TUBERCLE_OF_MALE = "skin of proximal genital tubercle of male"
    DEVELOPING_VASCULATURE_OF_FEMALE_GENITAL_TUBERCLE = "developing vasculature of female genital tubercle"
    PROXIMAL_GENITAL_TUBERCLE_OF_MALE = "proximal genital tubercle of male"
    PROXIMAL_URETHRAL_MEATUS_OF_MALE = "proximal urethral meatus of male"
    PROXIMAL_GENITAL_TUBERCLE_OF_FEMALE = "proximal genital tubercle of female"
    URETHRAL_PLATE_EPITHELIUM_OF_MALE = "urethral plate epithelium of male"
    URETHRAL_TUBE_EPITHELIUM_OF_MALE = "urethral tube epithelium of male"
    EPITHELIUM_OF_UROGENITAL_SINUS = "epithelium of urogenital sinus"
    DEVELOPING_VASCULATURE_OF_MALE_GENITAL_TUBERCLE = "developing vasculature of male genital tubercle"
    EARLY_DISTAL_TUBULE_OF_CAPILLARY_LOOP_NEPHRON = "early distal tubule of capillary loop nephron"
    EARLY_DISTAL_TUBULE_OF_MATURING_NEPHRON = "early distal tubule of maturing nephron"
    RENAL_CONNECTING_TUBULE_OF_MATURING_NEPHRON = "renal connecting tubule of maturing nephron"
    RENAL_CONNECTING_TUBULE_OF_MATURE_NEPHRON = "renal connecting tubule of mature nephron"
    URETHRAL_PLATE_EPITHELIUM = "urethral plate epithelium"
    NERVE_OF_PELVIC_URETHRA = "nerve of pelvic urethra"
    NERVE_OF_PHALLIC_URETHRA = "nerve of phallic urethra"
    MESENCHYMAL_LAYER_OF_VENTRAL_PRIMITIVE_BLADDER = "mesenchymal layer of ventral primitive bladder"
    NERVE_OF_BLADDER = "nerve of bladder"
    PELVIC_GANGLION = "pelvic ganglion"
    NERVE_OF_UROGENITAL_SINUS = "nerve of urogenital sinus"
    MESENCHYMAL_LAYER_OF_DORSAL_PRIMITIVE_BLADDER = "mesenchymal layer of dorsal primitive bladder"
    URETHRAL_TUBE_EPITHELIUM_OF_FEMALE = "urethral tube epithelium of female"
    PROXIMAL_URETHRAL_MEATUS_OF_FEMALE = "proximal urethral meatus of female"
    DEVELOPING_VASCULATURE_OF_UROGENITAL_SINUS = "developing vasculature of urogenital sinus"
    NERVES_OF_UROGENITAL_SINUS = "nerves of urogenital sinus"
    MESENCHYME_OF_PROXIMAL_GENITAL_TUBERCLE_OF_FEMALE = "mesenchyme of proximal genital tubercle of female"
    SKIN_OF_PROXIMAL_GENITAL_TUBERCLE_OF_FEMALE = "skin of proximal genital tubercle of female"
    URETHRAL_PLATE_EPITHELIUM_OF_FEMALE = "urethral plate epithelium of female"
    DEVELOPING_VASCULATURE_OF_MALE_MESONEPHROS = "developing vasculature of male mesonephros"
    DEVELOPING_VASCULATURE_OF_FEMALE_MESONEPHROS = "developing vasculature of female mesonephros"
    DEVELOPING_VASCULATURE_OF_PRIMITIVE_BLADDER = "developing vasculature of primitive bladder"
    RENAL_CONNECTING_SEGMENT_OF_RENAL_VESICLE = "renal connecting segment of renal vesicle"
    EARLY_PROXIMAL_TUBULE_OF_MATURING_NEPHRON = "early proximal tubule of maturing nephron"
    EARLY_PROXIMAL_TUBULE_OF_CAPILLARY_LOOP_NEPHRON = "early proximal tubule of capillary loop nephron"
    RENAL_CONNECTING_SEGMENT_OF_COMMA_SHAPED_BODY = "renal connecting segment of comma-shaped body"
    PROXIMAL_RENAL_VESICLE = "proximal renal vesicle"
    MESENCHYMAL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = "mesenchymal layer of ventral pelvic urethra of male"
    MESENCHYMAL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "mesenchymal layer of dorsal pelvic urethra of male"
    MESENCHYMAL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = "mesenchymal layer of ventral pelvic urethra of female"
    MESENCHYMAL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "mesenchymal layer of dorsal pelvic urethra of female"
    MUSCLE_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = "muscle layer of ventral pelvic urethra of male"
    MUSCLE_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "muscle layer of dorsal pelvic urethra of male"
    MUSCLE_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = "muscle layer of ventral pelvic urethra of female"
    MUSCLE_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "muscle layer of dorsal pelvic urethra of female"
    NERVE_OF_PELVIC_URETHRA_OF_MALE = "nerve of pelvic urethra of male"
    CORTICAL_RENAL_TUBULE_OF_MATURE_NEPHRON_GROUP = "cortical renal tubule of mature nephron group"
    NERVE_OF_PELVIC_URETHRA_OF_FEMALE = "nerve of pelvic urethra of female"
    NERVE_OF_URETHRA = "nerve of urethra"
    CORTICAL_RENAL_TUBULE_OF_MATURING_NEPHRON_GROUP = "cortical renal tubule of maturing nephron group"
    CORTICAL_RENAL_TUBULE_OF_CAPILLARY_LOOP_NEPHRON_GROUP = "cortical renal tubule of capillary loop nephron group"
    MESENCHYMAL_LAYER_OF_VENTRAL_PELVIC_URETHRA = "mesenchymal layer of ventral pelvic urethra"
    MESENCHYMAL_LAYER_OF_DORSAL_PELVIC_URETHRA = "mesenchymal layer of dorsal pelvic urethra"
    MUSCLE_LAYER_OF_VENTRAL_PELVIC_URETHRA = "muscle layer of ventral pelvic urethra"
    MUSCLE_LAYER_OF_DORSAL_PELVIC_URETHRA = "muscle layer of dorsal pelvic urethra"
    DEVELOPING_VASCULATURE_OF_GENITOURINARY_SYSTEM = "developing vasculature of genitourinary system"
    DEVELOPING_VASCULATURE_OF_URINARY_SYSTEM = "developing vasculature of urinary system"
    DEVELOPING_VASCULATURE_OF_REPRODUCTIVE_SYSTEM = "developing vasculature of reproductive system"
    DISTAL_RENAL_VESICLE = "distal renal vesicle"
    TRUNK = "trunk"
    HEAD = "head"
    POLAR_BODY = "polar body"
    ALLANTOIS_BLOOD_VESSEL = "allantois blood vessel"
    ALLANTOIS_ENDOTHELIUM = "allantois endothelium"
    EXTRAEMBRYONIC_MESODERM = "extraembryonic mesoderm"
    EXTRAEMBRYONIC_MESENCHYME = "extraembryonic mesenchyme"
    _4_CELL_STAGE_EMBRYO = "4-cell stage embryo"
    _8_CELL_STAGE_EMBRYO = "8-cell stage embryo"
    YOLK_SAC_BLOOD_VESSEL = "yolk sac blood vessel"
    TRABECULAR_CORD_MESODERMAL_CORE = "trabecular cord mesodermal core"
    UMBILICAL_RING = "umbilical ring"
    TRABECULAR_CORD = "trabecular cord"
    INTERPLACENTAL_SPACE = "interplacental space"
    CHORIONIC_PLATE = "chorionic plate"
    CHORIOALLANTOIC_PLACENTA = "chorioallantoic placenta"
    SECONDARY_TROPHOBLAST_GIANT_CELL = "secondary trophoblast giant cell"
    SPONGIOTROPHOBLAST = "spongiotrophoblast"
    TROPHOBLAST_GIANT_CELL = "trophoblast giant cell"
    LABYRINTHINE_ZONE = "labyrinthine zone"
    EPITHELIUM_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = "epithelium of ventral pelvic urethra of male"
    LAMINA_PROPRIA_OF_SINOVAGINAL_BULB = "lamina propria of sinovaginal bulb"
    BASAL_CELL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "basal cell layer of dorsal pelvic urethra of male"
    BASAL_CELL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = "basal cell layer of ventral pelvic urethra of male"
    VASCULATURE_OF_PELVIC_URETHRA_OF_MALE = "vasculature of pelvic urethra of male"
    SUPERFICIAL_CELL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "superficial cell layer of dorsal pelvic urethra of male"
    EPITHELIUM_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "epithelium of dorsal pelvic urethra of female"
    VASCULATURE_OF_PROSTATIC_URETHRA = "vasculature of prostatic urethra"
    SUPERFICIAL_CELL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = (
        "superficial cell layer of dorsal pelvic urethra of female"
    )
    BASAL_CELL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "basal cell layer of dorsal pelvic urethra of female"
    SUPERFICIAL_CELL_LAYER_OF_PROSTATIC_URETHRA = "superficial cell layer of prostatic urethra"
    RHABDOSPHINCTER_OF_MALE = "rhabdosphincter of male"
    LAMINA_PROPRIA_OF_EJACULATORY_DUCT = "lamina propria of ejaculatory duct"
    BASAL_CELL_LAYER_OF_PROSTATIC_URETHRA = "basal cell layer of prostatic urethra"
    LAMINA_PROPRIA_OF_UPPER_VAGINA = "lamina propria of upper vagina"
    SUPERFICIAL_CELL_LAYER_OF_BLADDER_UROTHELIUM = "superficial cell layer of bladder urothelium"
    BASAL_CELL_LAYER_OF_BLADDER_UROTHELIUM = "basal cell layer of bladder urothelium"
    SUBMUCOSA_OF_BLADDER = "submucosa of bladder"
    SUBMUCOSA_OF_NECK = "submucosa of neck"
    SUBMUCOSA_OF_FUNDUS = "submucosa of fundus"
    SMOOTH_MUSCLE_LAYER_OF_PROSTATIC_URETHRA = "smooth muscle layer of prostatic urethra"
    NERVE_OF_PROSTATIC_URETHRA = "nerve of prostatic urethra"
    MESENCHYMAL_PAD_OF_PELVIC_URETHRA_OF_MALE = "mesenchymal pad of pelvic urethra of male"
    PERI_PROSTATIC_MESENCHYME = "peri-prostatic mesenchyme"
    EPITHELIUM_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = "epithelium of ventral pelvic urethra of female"
    VASCULATURE_OF_PELVIC_URETHRA_OF_FEMALE = "vasculature of pelvic urethra of female"
    SUPERFICIAL_CELL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = (
        "superficial cell layer of ventral pelvic urethra of male"
    )
    NERVE_OF_PROSTATE_GLAND = "nerve of prostate gland"
    PRESUMPTIVE_NEURAL_CREST_DERIVATIVE_OF_PELVIC_URETHRA_OF_MALE = (
        "presumptive neural crest derivative of pelvic urethra of male"
    )
    MESENCHYMAL_PAD_OF_PELVIC_URETHRA_OF_FEMALE = "mesenchymal pad of pelvic urethra of female"
    PRESUMPTIVE_NEURAL_CREST_DERIVATIVE_OF_PELVIC_URETHRA_OF_FEMALE = (
        "presumptive neural crest derivative of pelvic urethra of female"
    )
    MUSCLE_OF_EJACULATORY_DUCT = "muscle of ejaculatory duct"
    SUPERFICIAL_CELL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = (
        "superficial cell layer of ventral pelvic urethra of female"
    )
    BASAL_CELL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = "basal cell layer of ventral pelvic urethra of female"
    SUBMUCOSA_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "submucosa of dorsal pelvic urethra of male"
    LAMINA_PROPRIA_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = "lamina propria of ventral pelvic urethra of male"
    MUSCULARIS_MUCOSA_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "muscularis mucosa of dorsal pelvic urethra of female"
    LAMINA_PROPRIA_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "lamina propria of dorsal pelvic urethra of female"
    MUSCULARIS_MUCOSA_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = "muscularis mucosa of ventral pelvic urethra of male"
    SUBMUCOSA_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = "submucosa of ventral pelvic urethra of male"
    INTERMEDIATE_CELL_LAYER_OF_PROSTATIC_URETHRA = "intermediate cell layer of prostatic urethra"
    EPITHELIUM_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "epithelium of dorsal pelvic urethra of male"
    SUBMUCOSA_OF_PROSTATIC_URETHRA = "submucosa of prostatic urethra"
    MUSCULARIS_MUCOSA_OF_PROSTATIC_URETHRA = "muscularis mucosa of prostatic urethra"
    LAMINA_PROPRIA_OF_PROSTATIC_URETHRA = "lamina propria of prostatic urethra"
    SUBMUCOSA_OF_TRIGONE = "submucosa of trigone"
    INTERMEDIATE_CELL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = (
        "intermediate cell layer of ventral pelvic urethra of female"
    )
    INTERMEDIATE_CELL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = (
        "intermediate cell layer of dorsal pelvic urethra of female"
    )
    INTERMEDIATE_CELL_LAYER_OF_DORSAL_PELVIC_URETHRA_OF_MALE = (
        "intermediate cell layer of dorsal pelvic urethra of male"
    )
    SUBMUCOSA_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = "submucosa of ventral pelvic urethra of female"
    MUSCULARIS_MUCOSA_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = "muscularis mucosa of ventral pelvic urethra of female"
    LAMINA_PROPRIA_OF_VENTRAL_PELVIC_URETHRA_OF_FEMALE = "lamina propria of ventral pelvic urethra of female"
    LAMINA_PROPRIA_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "lamina propria of dorsal pelvic urethra of male"
    MUSCULARIS_MUCOSA_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "muscularis mucosa of dorsal pelvic urethra of male"
    SUBMUCOSA_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "submucosa of dorsal pelvic urethra of female"
    INTERMEDIATE_CELL_LAYER_OF_VENTRAL_PELVIC_URETHRA_OF_MALE = (
        "intermediate cell layer of ventral pelvic urethra of male"
    )
    LIMB_INTERDIGITAL_REGION_MESENCHYME = "limb interdigital region mesenchyme"
    LIMB_INTERDIGITAL_REGION_SKIN = "limb interdigital region skin"
    FORELIMB_EPITHELIUM = "forelimb epithelium"
    FORELIMB_SKIN = "forelimb skin"
    FORELIMB_DERMIS = "forelimb dermis"
    PHALANX = "phalanx"
    DIGIT_SKIN = "digit skin"
    DIGIT_EPITHELIUM = "digit epithelium"
    INTERDIGITAL_REGION_EPITHELIUM = "interdigital region epithelium"
    LIMB_INTERDIGITAL_REGION = "limb interdigital region"
    HINDLIMB_DERMIS = "hindlimb dermis"
    HINDLIMB_SKIN = "hindlimb skin"
    HINDLIMB_EPITHELIUM = "hindlimb epithelium"
    FOREARM_MUSCLE = "forearm muscle"
    FORELIMB_SKELETON = "forelimb skeleton"
    FORELIMB_MUSCULATURE = "forelimb musculature"
    FORELIMB_PRE_CARTILAGE_CONDENSATION = "forelimb pre-cartilage condensation"
    FORELIMB_CARTILAGE_CONDENSATION = "forelimb cartilage condensation"
    FORELIMB_EPIDERMIS = "forelimb epidermis"
    FORELIMB_MESENCHYME = "forelimb mesenchyme"
    HAND_PRE_CARTILAGE_CONDENSATION = "hand pre-cartilage condensation"
    HAND_EPITHELIUM = "hand epithelium"
    HAND_CARTILAGE_CONDENSATION = "hand cartilage condensation"
    HINDLIMB_CARTILAGE_CONDENSATION = "hindlimb cartilage condensation"
    HINDLIMB_SKELETON = "hindlimb skeleton"
    HINDLIMB_MUSCULATURE = "hindlimb musculature"
    HINDLIMB_JOINT = "hindlimb joint"
    HINDLIMB_EPIDERMIS = "hindlimb epidermis"
    HINDLIMB_MESENCHYME = "hindlimb mesenchyme"
    HINDLIMB_PRE_CARTILAGE_CONDENSATION = "hindlimb pre-cartilage condensation"
    FORELIMB_PHALANX_PRE_CARTILAGE_CONDENSATION = "forelimb phalanx pre-cartilage condensation"
    FORELIMB_DIGIT_CARTILAGE_CONDENSATION = "forelimb digit cartilage condensation"
    HAND_JOINT = "hand joint"
    FORELIMB_DIGIT = "forelimb digit"
    HAND_SKELETON = "hand skeleton"
    HAND_MUSCLE = "hand muscle"
    FORELIMB_DIGIT_MESENCHYME = "forelimb digit mesenchyme"
    FORELIMB_DIGIT_PRE_CARTILAGE_CONDENSATION = "forelimb digit pre-cartilage condensation"
    FORELIMB_DIGIT_EPITHELIUM = "forelimb digit epithelium"
    FORELIMB_DIGIT_SKIN = "forelimb digit skin"
    HINDLIMB_DIGIT = "hindlimb digit"
    FORELIMB_PHALANX = "forelimb phalanx"
    FORELIMB_INTERDIGITAL_REGION = "forelimb interdigital region"
    FORELIMB_INTERDIGITAL_REGION_EPITHELIUM = "forelimb interdigital region epithelium"
    FORELIMB_INTERDIGITAL_REGION_MESENCHYME = "forelimb interdigital region mesenchyme"
    METACARPUS = "metacarpus"
    FOOT_EPITHELIUM = "foot epithelium"
    FOOT_PRE_CARTILAGE_CONDENSATION = "foot pre-cartilage condensation"
    FOOT_CARTILAGE_CONDENSATION = "foot cartilage condensation"
    FORELIMB_PHALANX_CARTILAGE_CONDENSATION = "forelimb phalanx cartilage condensation"
    ARACHNOID_MATER = "arachnoid mater"
    MENINGES = "meninges"
    BRAIN_ARACHNOID_MATER = "brain arachnoid mater"
    BRAIN_MENINGES = "brain meninges"
    PIA_MATER = "pia mater"
    DURA_MATER = "dura mater"
    GANGLIONIC_EMINENCE = "ganglionic eminence"
    BRAIN_SUBARACHNOID_SPACE = "brain subarachnoid space"
    BRAIN_PIA_MATER = "brain pia mater"
    BRAIN_DURA_MATER = "brain dura mater"
    VERTEBRAL_CARTILAGE_CONDENSATION = "vertebral cartilage condensation"
    ANNULUS_FIBROSUS = "annulus fibrosus"
    VERTEBRAL_PRE_CARTILAGE_CONDENSATION = "vertebral pre-cartilage condensation"
    AMYGDALA = "amygdala"
    BASAL_GANGLIA = "basal ganglia"
    BRAIN_VENTRICLE = "brain ventricle"
    BRAIN_FLOOR_PLATE = "brain floor plate"
    FUTURE_BRAIN_FLOOR_PLATE = "future brain floor plate"
    BRAINSTEM = "brainstem"
    BRAIN_VENTRICULAR_LAYER = "brain ventricular layer"
    BRAIN_MARGINAL_LAYER = "brain marginal layer"
    BRAIN_MANTLE_LAYER = "brain mantle layer"
    LUNG_ALVEOLUS = "lung alveolus"
    ALIMENTARY_SYSTEM_EPITHELIUM = "alimentary system epithelium"
    FLOOR_PLATE = "floor plate"
    ENDOCARDIAL_TUBE = "endocardial tube"
    ENDOCARDIAL_LINING = "endocardial lining"
    CARDIAC_JELLY = "cardiac jelly"
    CARDIAC_MUSCLE_TISSUE = "cardiac muscle tissue"
    BRONCHUS = "bronchus"
    BRONCHUS_MESENCHYME = "bronchus mesenchyme"
    BRONCHUS_EPITHELIUM = "bronchus epithelium"
    ALVEOLAR_SYSTEM = "alveolar system"
    BRONCHIOLE_EPITHELIUM = "bronchiole epithelium"
    ALVEOLAR_DUCT = "alveolar duct"
    LOBAR_BRONCHUS_EPITHELIUM = "lobar bronchus epithelium"
    LOBAR_BRONCHUS = "lobar bronchus"
    BRONCHIOLE = "bronchiole"
    LOBAR_BRONCHUS_MESENCHYME = "lobar bronchus mesenchyme"
    LIMB_JOINT_PRIMORDIUM = "limb joint primordium"
    LIMB_MUSCULATURE = "limb musculature"
    LIMB_CARTILAGE_CONDENSATION = "limb cartilage condensation"
    LIMB_SKELETON = "limb skeleton"
    CORTICAL_SUBPLATE = "cortical subplate"
    CEREBRAL_CORTEX_SUBVENTRICULAR_ZONE = "cerebral cortex subventricular zone"
    LIMB_MESENCHYME = "limb mesenchyme"
    LIMB_PRE_CARTILAGE_CONDENSATION = "limb pre-cartilage condensation"
    CEREBRAL_CORTEX_VENTRICULAR_ZONE = "cerebral cortex ventricular zone"
    LIMB_ECTODERM = "limb ectoderm"
    DIENCEPHALON_MARGINAL_LAYER = "diencephalon marginal layer"
    DIENCEPHALON_VENTRICULAR_LAYER = "diencephalon ventricular layer"
    CORTICAL_PLATE = "cortical plate"
    CEREBRAL_CORTEX_INTERMEDIATE_ZONE = "cerebral cortex intermediate zone"
    CEREBRAL_CORTEX_MARGINAL_ZONE = "cerebral cortex marginal zone"
    MUSCULOSKELETAL_SYSTEM = "musculoskeletal system"
    MUSCLE_TISSUE = "muscle tissue"
    SKELETAL_MUSCLE_TISSUE = "skeletal muscle tissue"
    SMOOTH_MUSCLE_TISSUE = "smooth muscle tissue"
    DIENCEPHALON_MANTLE_LAYER = "diencephalon mantle layer"
    DIGIT_MESENCHYME = "digit mesenchyme"
    PHALANX_PRE_CARTILAGE_CONDENSATION = "phalanx pre-cartilage condensation"
    PHALANX_CARTILAGE_CONDENSATION = "phalanx cartilage condensation"
    AUTOPOD = "autopod"
    AUTOPOD_SKIN = "autopod skin"
    LIMB_EPITHELIUM = "limb epithelium"
    LIMB_DIGIT = "limb digit"
    LIMB_SKIN = "limb skin"
    LIMB_DERMIS = "limb dermis"
    LIMB_EPIDERMIS = "limb epidermis"
    APPENDICULAR_SKELETON = "appendicular skeleton"
    CARTILAGE = "cartilage"
    CARTILAGE_CONDENSATION = "cartilage condensation"
    NEURAL_FOLD = "neural fold"
    EXTRAEMBRYONIC_BLOOD_VESSEL = "extraembryonic blood vessel"
    MESENCHYME_DERIVED_FROM_NEURAL_CREST = "mesenchyme derived from neural crest"
    PRE_CARTILAGE_CONDENSATION = "pre-cartilage condensation"
    NEURAL_CREST = "neural crest"
    EPITHELIUM = "epithelium"
    INTERVERTEBRAL_DISC = "intervertebral disc"
    VERTEBRA = "vertebra"
    CHOROID_PLEXUS = "choroid plexus"
    _4TH_VENTRICLE_CHOROID_PLEXUS = "4th ventricle choroid plexus"
    BLOOD_VESSEL = "blood vessel"
    APICAL_ECTODERMAL_RIDGE = "apical ectodermal ridge"
    ATRIUM_ENDOCARDIAL_LINING = "atrium endocardial lining"
    ATRIUM_CARDIAC_MUSCLE = "atrium cardiac muscle"
    VENTRICLE_ENDOCARDIAL_LINING = "ventricle endocardial lining"
    VENTRICLE_CARDIAC_MUSCLE = "ventricle cardiac muscle"
    BRANCHIAL_ARCH_MEMBRANE = "branchial arch membrane"
    BRANCHIAL_GROOVE = "branchial groove"
    BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_HEAD_MESODERM = "branchial arch mesenchyme derived from head mesoderm"
    BRANCHIAL_POUCH = "branchial pouch"
    BRANCHIAL_ARCH_ECTODERM = "branchial arch ectoderm"
    BRANCHIAL_ARCH_ENDODERM = "branchial arch endoderm"
    BRANCHIAL_ARCH_MESENCHYME = "branchial arch mesenchyme"
    BRANCHIAL_ARCH_MESENCHYME_DERIVED_FROM_NEURAL_CREST = "branchial arch mesenchyme derived from neural crest"
    PARAXIAL_MESENCHYME = "paraxial mesenchyme"
    EYELID_EPITHELIUM = "eyelid epithelium"
    EYELID_MESENCHYME = "eyelid mesenchyme"
    BONE_MARROW = "bone marrow"
    BODY_WALL = "body wall"
    MESENDODERM = "mesendoderm"
    FACE = "face"
    ILEUM = "ileum"
    _6TH_BRANCHIAL_ARCH_MESENCHYME = "6th branchial arch mesenchyme"
    _6TH_BRANCHIAL_ARCH = "6th branchial arch"
    PIRIFORM_CORTEX = "piriform cortex"
    HIPPOCAMPUS_CA1 = "hippocampus CA1"
    HIPPOCAMPUS_CA2 = "hippocampus CA2"
    HIPPOCAMPUS_CA3 = "hippocampus CA3"
    HIPPOCAMPUS_CA4 = "hippocampus CA4"
    HIPPOCAMPUS_REGION = "hippocampus region"
    TAIL_SURFACE_ECTODERM = "tail surface ectoderm"
    LINGUAL_SWELLING_MESENCHYME = "lingual swelling mesenchyme"
    LINGUAL_SWELLING_EPITHELIUM = "lingual swelling epithelium"
    FORELIMB_BUD_PRESUMPTIVE_APICAL_ECTODERMAL_RIDGE = "forelimb bud presumptive apical ectodermal ridge"
    TONGUE_PAPILLAE = "tongue papillae"
    PRETECTUM = "pretectum"
    EPIDERMIS_BASAL_LAYER = "epidermis basal layer"
    PALLIDUM = "pallidum"
    PERIAQUEDUCTAL_GREY = "periaqueductal grey"
    BONE = "bone"
    ANKLE = "ankle"
    WRIST = "wrist"
    EPIDERMIS_SPINOUS_LAYER = "epidermis spinous layer"
    EPIDERMIS_GRANULAR_LAYER = "epidermis granular layer"
    EPIDERMIS_CORNIFIED_LAYER = "epidermis cornified layer"
    FOLIATE_PAPILLAE = "foliate papillae"
    ACCUMBENS_NUCLEUS = "accumbens nucleus"
    BRAINSTEM_NUCLEUS = "brainstem nucleus"
    CRISTA_AMPULLARIS = "crista ampullaris"
    HAIR_INNER_ROOT_SHEATH = "hair inner root sheath"
    HAIR_OUTER_ROOT_SHEATH = "hair outer root sheath"
    OLFACTORY_CORTEX_SUBVENTRICULAR_ZONE = "olfactory cortex subventricular zone"
    LEFT_HEPATIC_DUCT = "left hepatic duct"
    FUTURE_PITUITARY = "future pituitary"
    RIGHT_HEPATIC_DUCT = "right hepatic duct"
    TAIL_NEURAL_TUBE_FLOOR_PLATE = "tail neural tube floor plate"
    CHONDROCRANIUM_PRE_CARTILAGE_CONDENSATION = "chondrocranium pre-cartilage condensation"
    EAR_EPITHELIUM = "ear epithelium"
    PHARYNGO_TYMPANIC_TUBE_EPITHELIUM = "pharyngo-tympanic tube epithelium"
    CHONDROCRANIUM_CARTILAGE_CONDENSATION = "chondrocranium cartilage condensation"
    BRAIN_SUBVENTRICULAR_ZONE = "brain subventricular zone"
    ALVEOLAR_SULCUS = "alveolar sulcus"
    SNOUT = "snout"
    PRESUMPTIVE_IRIS = "presumptive iris"
    NERVE = "nerve"
    MAIN_OLFACTORY_BULB = "main olfactory bulb"
    PALATE_EPITHELIUM = "palate epithelium"
    PALATE_MESENCHYME = "palate mesenchyme"
    PARATHYROID_GLAND = "parathyroid gland"
    PARASYMPATHETIC_GANGLION = "parasympathetic ganglion"
    PERIPHERAL_NERVOUS_SYSTEM_GANGLION = "peripheral nervous system ganglion"
    PERIPHERAL_NERVE = "peripheral nerve"
    SALIVARY_GLAND_EPITHELIUM = "salivary gland epithelium"
    SALIVARY_GLAND_MESENCHYME = "salivary gland mesenchyme"
    RHOMBOMERE_FLOOR_PLATE = "rhombomere floor plate"
    RHOMBOMERE_LATERAL_WALL = "rhombomere lateral wall"
    RHOMBOMERE = "rhombomere"
    RHOMBOMERE_MARGINAL_LAYER = "rhombomere marginal layer"
    RHOMBOMERE_VENTRICULAR_LAYER = "rhombomere ventricular layer"
    RHOMBOMERE_ROOF_PLATE = "rhombomere roof plate"
    RHOMBOMERE_MANTLE_LAYER = "rhombomere mantle layer"
    RESPIRATORY_SYSTEM_EPITHELIUM = "respiratory system epithelium"
    RESPIRATORY_TRACT_EPITHELIUM = "respiratory tract epithelium"
    SEMICIRCULAR_CANAL_EPITHELIUM = "semicircular canal epithelium"
    SEMICIRCULAR_CANAL_MESENCHYME = "semicircular canal mesenchyme"
    SEMICIRCULAR_DUCT = "semicircular duct"
    SEMICIRCULAR_DUCT_AMPULLA = "semicircular duct ampulla"
    SEMICIRCULAR_CANAL = "semicircular canal"
    SMALL_INTESTINE = "small intestine"
    TELENCEPHALON_SEPTUM = "telencephalon septum"
    DERMATOME = "dermatome"
    LIP = "lip"
    TRUNK_DERMATOME = "trunk dermatome"
    MYOTOME = "myotome"
    NEOCORTEX = "neocortex"
    TRUNK_MYOTOME = "trunk myotome"
    MYELENCEPHALON_VENTRICULAR_LAYER = "myelencephalon ventricular layer"
    HIPPOCAMPUS = "hippocampus"
    GANGLION = "ganglion"
    MYENTERIC_NERVE_PLEXUS = "myenteric nerve plexus"
    MEDULLA_OBLONGATA_MANTLE_LAYER = "medulla oblongata mantle layer"
    METACARPUS_CARTILAGE_CONDENSATION = "metacarpus cartilage condensation"
    METACARPUS_PRE_CARTILAGE_CONDENSATION = "metacarpus pre-cartilage condensation"
    MEDULLA_OBLONGATA_VENTRICULAR_LAYER = "medulla oblongata ventricular layer"
    MEDULLA_OBLONGATA_MARGINAL_LAYER = "medulla oblongata marginal layer"
    METENCEPHALON_VENTRICULAR_LAYER = "metencephalon ventricular layer"
    MESOTHELIUM = "mesothelium"
    MIDBRAIN_HINDBRAIN_JUNCTION = "midbrain-hindbrain junction"
    UPPER_JAW_TOOTH_ENAMEL_ORGAN = "upper jaw tooth enamel organ"
    UPPER_JAW_TOOTH_DENTAL_LAMINA = "upper jaw tooth dental lamina"
    LUNG_EPITHELIUM = "lung epithelium"
    UPPER_JAW_TOOTH_DENTAL_PAPILLA = "upper jaw tooth dental papilla"
    UPPER_JAW_EPITHELIUM = "upper jaw epithelium"
    LOWER_JAW_TOOTH_DENTAL_PAPILLA = "lower jaw tooth dental papilla"
    UPPER_JAW_TOOTH_MESENCHYME = "upper jaw tooth mesenchyme"
    UPPER_JAW_TOOTH_EPITHELIUM = "upper jaw tooth epithelium"
    LUNG_MESENCHYME = "lung mesenchyme"
    LUNG_VASCULAR_ELEMENT = "lung vascular element"
    LOWER_JAW_TOOTH_EPITHELIUM = "lower jaw tooth epithelium"
    SUPERIOR_COLLICULUS = "superior colliculus"
    INFERIOR_COLLICULUS = "inferior colliculus"
    LENS_EPITHELIUM = "lens epithelium"
    INTESTINE_MESENCHYME = "intestine mesenchyme"
    INTESTINE_EPITHELIUM = "intestine epithelium"
    INTESTINE = "intestine"
    LOWER_JAW_TOOTH_ENAMEL_ORGAN = "lower jaw tooth enamel organ"
    LOWER_JAW_TOOTH_MESENCHYME = "lower jaw tooth mesenchyme"
    LOWER_JAW_TOOTH_DENTAL_LAMINA = "lower jaw tooth dental lamina"
    MOLAR_EPITHELIUM = "molar epithelium"
    MOLAR_DENTAL_LAMINA = "molar dental lamina"
    MOLAR_ENAMEL_ORGAN = "molar enamel organ"
    MOLAR_MESENCHYME = "molar mesenchyme"
    MOLAR_DENTAL_PAPILLA = "molar dental papilla"
    INNER_EAR_EPITHELIUM = "inner ear epithelium"
    VESTIBULAR_COMPONENT_EPITHELIUM = "vestibular component epithelium"
    INNER_EAR_MESENCHYME = "inner ear mesenchyme"
    ALIMENTARY_SYSTEM_MESENCHYME = "alimentary system mesenchyme"
    VESTIBULAR_COMPONENT_MESENCHYME = "vestibular component mesenchyme"
    ENAMEL_ORGAN = "enamel organ"
    INCISOR = "incisor"
    INCISOR_EPITHELIUM = "incisor epithelium"
    TOOTH_MESENCHYME = "tooth mesenchyme"
    DENTAL_PAPILLA = "dental papilla"
    INCISOR_DENTAL_PAPILLA = "incisor dental papilla"
    INCISOR_MESENCHYME = "incisor mesenchyme"
    INCISOR_ENAMEL_ORGAN = "incisor enamel organ"
    INCISOR_DENTAL_LAMINA = "incisor dental lamina"
    MOLAR = "molar"
    HINDBRAIN_VENTRICULAR_LAYER = "hindbrain ventricular layer"
    HINDBRAIN_ROOF_PLATE = "hindbrain roof plate"
    HINDBRAIN_ROOF = "hindbrain roof"
    JAW_PRIMORDIUM = "jaw primordium"
    DENTAL_LAMINA = "dental lamina"
    TOOTH_EPITHELIUM = "tooth epithelium"
    JAW_EPITHELIUM = "jaw epithelium"
    JAW = "jaw"
    TOOTH = "tooth"
    JAW_MESENCHYME = "jaw mesenchyme"
    GUT_LUMEN = "gut lumen"
    GUT_MESENCHYME = "gut mesenchyme"
    GUT_DORSAL_MESENTERY = "gut dorsal mesentery"
    GUT_MESENTERY = "gut mesentery"
    HINDBRAIN_LATERAL_WALL = "hindbrain lateral wall"
    HINDBRAIN_FLOOR_PLATE = "hindbrain floor plate"
    HINDBRAIN_BASAL_PLATE = "hindbrain basal plate"
    HINDBRAIN_ALAR_PLATE = "hindbrain alar plate"
    HINDBRAIN_MARGINAL_LAYER = "hindbrain marginal layer"
    HINDBRAIN_MANTLE_LAYER = "hindbrain mantle layer"
    FRONTO_NASAL_PROCESS_EPITHELIUM = "fronto-nasal process epithelium"
    FUTURE_FOREBRAIN_FLOOR_PLATE = "future forebrain floor plate"
    FOREBRAIN_MARGINAL_LAYER = "forebrain marginal layer"
    FOREBRAIN_VENTRICULAR_LAYER = "forebrain ventricular layer"
    FOREGUT_EPITHELIUM = "foregut epithelium"
    FOREGUT_MESENCHYME = "foregut mesenchyme"
    PANCREAS_PARENCHYMA = "pancreas parenchyma"
    ISLETS_OF_LANGERHANS = "islets of Langerhans"
    GUT_EPITHELIUM = "gut epithelium"
    GUT_ENDODERM = "gut endoderm"
    FUTURE_FOREBRAIN_ROOF_PLATE = "future forebrain roof plate"
    FUTURE_FOREBRAIN_LATERAL_WALL = "future forebrain lateral wall"
    HINDLIMB_DIGIT_PRE_CARTILAGE_CONDENSATION = "hindlimb digit pre-cartilage condensation"
    HINDLIMB_DIGIT_CARTILAGE_CONDENSATION = "hindlimb digit cartilage condensation"
    METATARSUS = "metatarsus"
    HINDLIMB_PHALANX = "hindlimb phalanx"
    HINDLIMB_DIGIT_SKIN = "hindlimb digit skin"
    HINDLIMB_DIGIT_DERMIS = "hindlimb digit dermis"
    HINDLIMB_DIGIT_EPIDERMIS = "hindlimb digit epidermis"
    HINDLIMB_INTERDIGITAL_REGION = "hindlimb interdigital region"
    HINDLIMB_DIGIT_EPITHELIUM = "hindlimb digit epithelium"
    HINDLIMB_DIGIT_MESENCHYME = "hindlimb digit mesenchyme"
    HINDLIMB_PHALANX_CARTILAGE_CONDENSATION = "hindlimb phalanx cartilage condensation"
    FOOT_JOINT = "foot joint"
    HINDLIMB_PHALANX_PRE_CARTILAGE_CONDENSATION = "hindlimb phalanx pre-cartilage condensation"
    METATARSUS_CARTILAGE_CONDENSATION = "metatarsus cartilage condensation"
    PANCREATIC_DUCT = "pancreatic duct"
    DORSAL_PANCREATIC_DUCT = "dorsal pancreatic duct"
    HINDLIMB_INTERDIGITAL_REGION_MESENCHYME = "hindlimb interdigital region mesenchyme"
    METATARSUS_PRE_CARTILAGE_CONDENSATION = "metatarsus pre-cartilage condensation"
    HINDLIMB_INTERDIGITAL_REGION_EPITHELIUM = "hindlimb interdigital region epithelium"
    NERVE_OF_UTERUS = "nerve of uterus"
    NERVE_OF_PRIMITIVE_BLADDER = "nerve of primitive bladder"
    NERVE_OF_LABIA = "nerve of labia"
    NERVE_OF_CLITORIS = "nerve of clitoris"
    NERVE_OF_GENITAL_TUBERCLE_OF_FEMALE = "nerve of genital tubercle of female"
    NERVE_OF_VAGINA = "nerve of vagina"
    NERVE_OF_EPIDIDYMIS = "nerve of epididymis"
    NERVE_OF_SEMINAL_VESICLE = "nerve of seminal vesicle"
    NERVE_OF_DUCTUS_DEFERENS = "nerve of ductus deferens"
    NERVE_OF_SCROTUM = "nerve of scrotum"
    NERVE_OF_OVARY = "nerve of ovary"
    NERVE_OF_OVIDUCT = "nerve of oviduct"
    NERVE_OF_UTERINE_HORN = "nerve of uterine horn"
    NERVE_OF_EJACULATORY_DUCT = "nerve of ejaculatory duct"
    NERVE_OF_BULBOURETHRAL_GLAND_OF_MALE = "nerve of bulbourethral gland of male"
    NERVE_OF_GENITAL_TUBERCLE_OF_MALE = "nerve of genital tubercle of male"
    NERVE_OF_PENIS = "nerve of penis"
    NERVE_OF_PENILE_URETHRA = "nerve of penile urethra"
    NERVE_OF_URETER = "nerve of ureter"
    _3RD_VENTRICLE_EPENDYMA = "3rd ventricle ependyma"
    _6TH_BRANCHIAL_ARCH_ECTODERM = "6th branchial arch ectoderm"
    ABDOMEN = "abdomen"
    ABDOMEN_MUSCULATURE = "abdomen musculature"
    ABDOMINAL_SEGMENT_OF_TRUNK = "abdominal segment of trunk"
    ABDUCENS_NUCLEUS = "abducens nucleus"
    ACCESSORY_OLFACTORY_BULB = "accessory olfactory bulb"
    ACCESSORY_OLFACTORY_BULB_EXTERNAL_PLEXIFORM_LAYER = "accessory olfactory bulb external plexiform layer"
    ACCESSORY_OLFACTORY_BULB_GLOMERULAR_LAYER = "accessory olfactory bulb glomerular layer"
    ACCESSORY_OLFACTORY_BULB_GRANULE_CELL_LAYER = "accessory olfactory bulb granule cell layer"
    ACCESSORY_OLFACTORY_BULB_INTERNAL_PLEXIFORM_LAYER = "accessory olfactory bulb internal plexiform layer"
    ACCESSORY_OLFACTORY_BULB_MITRAL_CELL_LAYER = "accessory olfactory bulb mitral cell layer"
    ADIPOSE_TISSUE = "adipose tissue"
    ADRENAL_CORTEX_ZONE = "adrenal cortex zone"
    ADRENAL_GLAND_ZONA_FASCICULATA = "adrenal gland zona fasciculata"
    ADRENAL_GLAND_ZONA_GLOMERULOSA = "adrenal gland zona glomerulosa"
    ADRENAL_GLAND_ZONA_RETICULARIS = "adrenal gland zona reticularis"
    ALIMENTARY_SYSTEM_SMOOTH_MUSCLE = "alimentary system smooth muscle"
    ALVEOLAR_PROCESS = "alveolar process"
    ALVEOLAR_SAC = "alveolar sac"
    ALVEOLAR_WALL = "alveolar wall"
    ALVEOLUS_EPITHELIUM = "alveolus epithelium"
    ALVEOLUS_SMOOTH_MUSCLE = "alveolus smooth muscle"
    AMPULLARY_GLAND = "ampullary gland"
    AMYGDALOHIPPOCAMPAL_AREA = "amygdalohippocampal area"
    AMYGDALOPIRIFORM_TRANSITION_AREA = "amygdalopiriform transition area"
    ANKLE_JOINT = "ankle joint"
    ANTERIOR_AMYGDALOID_AREA = "anterior amygdaloid area"
    ANTERIOR_CORTICAL_AMYGDALOID_NUCLEUS = "anterior cortical amygdaloid nucleus"
    ANTERIOR_DORSAL_THALAMIC_NUCLEUS = "anterior dorsal thalamic nucleus"
    ANTERIOR_MEDIAL_CORTEX = "anterior medial cortex"
    ANTERIOR_OLFACTORY_NUCLEUS = "anterior olfactory nucleus"
    ANTERIOR_THALAMIC_GROUP = "anterior thalamic group"
    ANTERIOR_VENTRAL_THALAMIC_NUCLEUS = "anterior ventral thalamic nucleus"
    AORTA_ENDOTHELIUM = "aorta endothelium"
    AORTA_SMOOTH_MUSCLE = "aorta smooth muscle"
    AORTA_TUNICA_ADVENTITIA = "aorta tunica adventitia"
    AORTA_TUNICA_INTIMA = "aorta tunica intima"
    AORTA_TUNICA_MEDIA = "aorta tunica media"
    AORTA_WALL = "aorta wall"
    APEX_OF_CECUM = "apex of cecum"
    APOCRINE_GLAND = "apocrine gland"
    ARCUATE_NUCLEUS = "arcuate nucleus"
    AREA_POSTREMA = "area postrema"
    ARTERIAL_BLOOD_VESSEL = "arterial blood vessel"
    ARTERIAL_SYSTEM_ENDOTHELIUM = "arterial system endothelium"
    ARTERIOLE = "arteriole"
    ARTERY = "artery"
    ARTERY_ENDOTHELIUM = "artery endothelium"
    ARTICULAR_CARTILAGE = "articular cartilage"
    ARTICULAR_SYSTEM = "articular system"
    ASCENDING_COLON = "ascending colon"
    ATRIOVENTRICULAR_BUNDLE = "atrioventricular bundle"
    ATRIOVENTRICULAR_NODE = "atrioventricular node"
    ATRIOVENTRICULAR_VALVE = "atrioventricular valve"
    RIGHT_LUNG_LOBE = "right lung lobe"
    AUDITORY_CORTEX = "auditory cortex"
    PINNA_CARTILAGE = "pinna cartilage"
    PINNA_MUSCLE = "pinna muscle"
    AUTOPOD_JOINT = "autopod joint"
    BACK_MUSCULATURE = "back musculature"
    BACK_OF_TRUNK = "back of trunk"
    BARREL_CORTEX = "barrel cortex"
    BASAL_FOREBRAIN = "basal forebrain"
    BASAL_VENTRAL_MEDIAL_THALAMIC_NUCLEUS = "basal ventral medial thalamic nucleus"
    BASILAR_MEMBRANE = "basilar membrane"
    BASOLATERAL_AMYGDALOID_NUCLEUS = "basolateral amygdaloid nucleus"
    BASOMEDIAL_AMYGDALOID_NUCLEUS = "basomedial amygdaloid nucleus"
    BED_NUCLEUS_OF_STRIA_TERMINALIS = "bed nucleus of stria terminalis"
    BICEPS_FEMORIS = "biceps femoris"
    BILE_DUCT = "bile duct"
    BILIARY_TRACT = "biliary tract"
    BLADDER_FUNDUS = "bladder fundus"
    BLADDER_TRIGONE = "bladder trigone"
    BLADDER_WALL = "bladder wall"
    BLOOD_VESSEL_ENDOTHELIUM = "blood vessel endothelium"
    BLOOD_VESSEL_SMOOTH_MUSCLE = "blood vessel smooth muscle"
    BODY_FLUID_OR_SUBSTANCE = "body fluid or substance"
    BONE_TISSUE = "bone tissue"
    BRAIN_BLOOD_VESSEL = "brain blood vessel"
    BRAIN_EPENDYMA = "brain ependyma"
    BRAIN_GRAY_MATTER = "brain gray matter"
    BRAIN_NUCLEUS = "brain nucleus"
    BRAIN_VASCULAR_ELEMENT = "brain vascular element"
    BRAIN_WHITE_MATTER = "brain white matter"
    BRAINSTEM_RETICULAR_FORMATION = "brainstem reticular formation"
    BRAINSTEM_WHITE_MATTER = "brainstem white matter"
    BRANCHIAL_GROOVE_ECTODERM = "branchial groove ectoderm"
    BRANCHIAL_POUCH_ENDODERM = "branchial pouch endoderm"
    BRONCHUS_CARTILAGE = "bronchus cartilage"
    BRONCHUS_CONNECTIVE_TISSUE = "bronchus connective tissue"
    BRONCHUS_LAMINA_PROPRIA = "bronchus lamina propria"
    BRONCHUS_SMOOTH_MUSCLE = "bronchus smooth muscle"
    CA1_PYRAMIDAL_CELL_LAYER = "CA1 pyramidal cell layer"
    CECUM = "cecum"
    CAPILLARY = "capillary"
    CARDIOVASCULAR_SYSTEM_ENDOTHELIUM = "cardiovascular system endothelium"
    CELIAC_ARTERY = "celiac artery"
    CEMENTUM = "cementum"
    CENTRAL_AMYGDALOID_NUCLEUS = "central amygdaloid nucleus"
    CENTRAL_LATERAL_THALAMIC_NUCLEUS = "central lateral thalamic nucleus"
    CENTRAL_MEDIAL_THALAMIC_NUCLEUS = "central medial thalamic nucleus"
    CENTRAL_NERVOUS_SYSTEM_ROOF_PLATE = "central nervous system roof plate"
    CENTRAL_NERVOUS_SYSTEM_VASCULAR_ELEMENT = "central nervous system vascular element"
    VENTRICULAR_LAYER = "ventricular layer"
    CENTRAL_THALAMIC_NUCLEUS = "central thalamic nucleus"
    CEREBELLAR_CORTEX = "cerebellar cortex"
    CEREBELLUM_LAYER = "cerebellum layer"
    CEREBELLAR_PEDUNCLE = "cerebellar peduncle"
    CEREBELLUM_VERMIS = "cerebellum vermis"
    CEREBELLUM_ANTERIOR_LOBE = "cerebellum anterior lobe"
    CEREBELLUM_EXTERNAL_GRANULE_CELL_LAYER = "cerebellum external granule cell layer"
    CEREBELLUM_GRANULE_CELL_LAYER = "cerebellum granule cell layer"
    CEREBELLAR_HEMISPHERE = "cerebellar hemisphere"
    CEREBELLUM_INTERNAL_GRANULE_CELL_LAYER = "cerebellum internal granule cell layer"
    CEREBELLUM_LOBE = "cerebellum lobe"
    CEREBELLUM_MOLECULAR_LAYER = "cerebellum molecular layer"
    CEREBELLUM_POSTERIOR_LOBE = "cerebellum posterior lobe"
    CEREBELLUM_PURKINJE_CELL_LAYER = "cerebellum Purkinje cell layer"
    CEREBELLUM_VERMIS_LOBULE = "cerebellum vermis lobule"
    CEREBELLUM_VERMIS_LOBULE_I = "cerebellum vermis lobule I"
    CEREBELLUM_VERMIS_LOBULE_II = "cerebellum vermis lobule II"
    CEREBELLUM_VERMIS_LOBULE_III = "cerebellum vermis lobule III"
    CEREBELLUM_VERMIS_LOBULE_IV = "cerebellum vermis lobule IV"
    CEREBELLUM_VERMIS_LOBULE_IX = "cerebellum vermis lobule IX"
    CEREBELLUM_VERMIS_LOBULE_V = "cerebellum vermis lobule V"
    CEREBELLUM_VERMIS_LOBULE_VI = "cerebellum vermis lobule VI"
    CEREBELLUM_VERMIS_LOBULE_VII = "cerebellum vermis lobule VII"
    CEREBELLUM_VERMIS_LOBULE_VIII = "cerebellum vermis lobule VIII"
    CEREBELLUM_VERMIS_LOBULE_X = "cerebellum vermis lobule X"
    CEREBELLUM_WHITE_MATTER = "cerebellum white matter"
    CEREBRAL_CORTEX_VASCULAR_ELEMENT = "cerebral cortex vascular element"
    CEREBRAL_HEMISPHERE_WHITE_MATTER = "cerebral hemisphere white matter"
    CHONDROCRANIUM_BONE = "chondrocranium bone"
    CILIARY_EPITHELIUM = "ciliary epithelium"
    CILIARY_MUSCLE = "ciliary muscle"
    CILIARY_STROMA = "ciliary stroma"
    CINGULATE_CORTEX = "cingulate cortex"
    CIRCUMVENTRICULAR_ORGAN = "circumventricular organ"
    CLAUSTRUM = "claustrum"
    COAT_HAIR_FOLLICLE = "coat hair follicle"
    COCHLEA_EPITHELIUM = "cochlea epithelium"
    COCHLEAR_NUCLEUS = "cochlear nucleus"
    COELOMIC_EPITHELIUM_OF_REPRODUCTIVE_SYSTEM = "coelomic epithelium of reproductive system"
    COMPACT_BONE = "compact bone"
    CONNECTIVE_TISSUE = "connective tissue"
    CORONARY_VASCULAR_PLEXUS = "coronary vascular plexus"
    CORPUS_CALLOSUM = "corpus callosum"
    CORTICAL_AMYGDALOID_NUCLEUS = "cortical amygdaloid nucleus"
    CORTICAL_LAYER_I = "cortical layer I"
    CORTICAL_LAYER_II = "cortical layer II"
    CORTICAL_LAYER_III = "cortical layer III"
    CORTICAL_LAYER_IV = "cortical layer IV"
    CORTICAL_LAYER_V = "cortical layer V"
    CORTICAL_LAYER_VI = "cortical layer VI"
    CORTICAL_PREPLATE = "cortical preplate"
    COSTAL_CARTILAGE = "costal cartilage"
    CRANIAL_SUTURE = "cranial suture"
    CRANIAL_SYNCHONDROSIS = "cranial synchondrosis"
    CRYPT_OF_LIEBERKUHN = "crypt of lieberkuhn"
    CUMULUS_OOPHORUS = "cumulus oophorus"
    CUNEATE_NUCLEUS = "cuneate nucleus"
    DECIDUA = "decidua"
    DECIDUA_BASALIS = "decidua basalis"
    DECIDUA_CAPSULARIS = "decidua capsularis"
    DEEP_CEREBELLAR_NUCLEUS = "deep cerebellar nucleus"
    DEGENERATING_PARAMESONEPHRIC_DUCT_OF_MALE = "degenerating paramesonephric duct of male"
    DENTAL_PULP = "dental pulp"
    DENTAL_SAC = "dental sac"
    DENTATE_GYRUS_GRANULE_CELL_LAYER = "dentate gyrus granule cell layer"
    DENTATE_GYRUS_HILUS = "dentate gyrus hilus"
    DENTATE_GYRUS_LAYER = "dentate gyrus layer"
    DENTATE_GYRUS_MOLECULAR_LAYER = "dentate gyrus molecular layer"
    DENTATE_GYRUS_POLYMORPHIC_LAYER = "dentate gyrus polymorphic layer"
    DENTIN = "dentin"
    DERMIS_PAPILLARY_LAYER = "dermis papillary layer"
    DERMIS_RETICULAR_LAYER = "dermis reticular layer"
    DESCENDING_COLON = "descending colon"
    DILATATOR_PUPILLAE = "dilatator pupillae"
    DISTAL_PHALANX = "distal phalanx"
    DISTAL_URETHRAL_EPITHELIUM = "distal urethral epithelium"
    DORSAL_COCHLEAR_NUCLEUS = "dorsal cochlear nucleus"
    DORSAL_LATERAL_GENICULATE_NUCLEUS = "dorsal lateral geniculate nucleus"
    DORSAL_MOTOR_NUCLEUS_OF_VAGUS_NERVE = "dorsal motor nucleus of vagus nerve"
    DORSAL_NASAL_ARTERY = "dorsal nasal artery"
    DORSAL_RAPHE_NUCLEUS = "dorsal raphe nucleus"
    DORSAL_TEGMENTAL_NUCLEUS = "dorsal tegmental nucleus"
    DORSAL_THALAMUS = "dorsal thalamus"
    MEDIAL_THALAMIC_GROUP = "medial thalamic group"
    DORSOMEDIAL_HYPOTHALAMIC_NUCLEUS = "dorsomedial hypothalamic nucleus"
    EAR_SKIN = "ear skin"
    EDINGER_WESTPHAL_NUCLEUS = "edinger-westphal nucleus"
    ENAMEL = "enamel"
    ENDOCHONDRAL_BONE = "endochondral bone"
    ENDOCRINE_PANCREAS = "endocrine pancreas"
    ENDOCRINE_SYSTEM = "endocrine system"
    ENDOMETRIUM_EPITHELIUM = "endometrium epithelium"
    ENDOMETRIUM_GLANDULAR_EPITHELIUM = "endometrium glandular epithelium"
    ENDOMETRIUM_LUMINAL_EPITHELIUM = "endometrium luminal epithelium"
    ENDOMETRIUM_STROMA = "endometrium stroma"
    ENDOPIRIFORM_NUCLEUS = "endopiriform nucleus"
    ENTERIC_GANGLION = "enteric ganglion"
    ENTORHINAL_CORTEX = "entorhinal cortex"
    EPIDERMIS_STRATUM_LUCIDUM = "epidermis stratum lucidum"
    EPIDERMIS_SUPRABASAL_LAYER = "epidermis suprabasal layer"
    EPIDIDYMAL_FAT_PAD = "epididymal fat pad"
    EPITHELIUM_OF_CAPILLARY_LOOP_RENAL_CORPUSCLE = "epithelium of capillary loop renal corpuscle"
    EPITHELIUM_OF_DEGENERATING_PARAMESONEPHRIC_DUCT_OF_MALE = "epithelium of degenerating paramesonephric duct of male"
    EPITHELIUM_OF_PARAMESONEPHRIC_DUCT_OF_FEMALE = "epithelium of paramesonephric duct of female"
    EPITHELIUM_OF_PARAMESONEPHRIC_DUCT_OF_MALE = "epithelium of paramesonephric duct of male"
    ESOPHAGUS_BASAL_EPITHELIUM = "esophagus basal epithelium"
    ESOPHAGUS_SEROSA = "esophagus serosa"
    ESOPHAGUS_SMOOTH_MUSCLE = "esophagus smooth muscle"
    ESOPHAGUS_SQUAMOUS_EPITHELIUM = "esophagus squamous epithelium"
    ESOPHAGUS_SUBMUCOSA = "esophagus submucosa"
    ESOPHAGUS_WALL = "esophagus wall"
    EXOCRINE_GLAND = "exocrine gland"
    EXOCRINE_PANCREAS = "exocrine pancreas"
    EXOCRINE_SYSTEM = "exocrine system"
    EXTENSOR_DIGITORUM_LONGUS = "extensor digitorum longus"
    EXTERNAL_CAPSULE = "external capsule"
    EXTERNAL_SPIRAL_SULCUS = "external spiral sulcus"
    EYE_GLAND = "eye gland"
    EYE_MUSCLE = "eye muscle"
    EYE_SURFACE_REGION = "eye surface region"
    FACE_MESENCHYME = "face mesenchyme"
    FACIAL_NUCLEUS = "facial nucleus"
    FASCIOLA_CINEREUM = "fasciola cinereum"
    FAT_PAD = "fat pad"
    FEMORAL_ARTERY = "femoral artery"
    FEMORAL_NERVE = "femoral nerve"
    FIBROUS_JOINT = "fibrous joint"
    FIMBRIA_HIPPOCAMPUS = "fimbria hippocampus"
    FLOCCULUS = "flocculus"
    FOREARM_BONE = "forearm bone"
    FOREBRAIN_VASCULAR_ELEMENT = "forebrain vascular element"
    FORELIMB_NAIL = "forelimb nail"
    FORELIMB_LONG_BONE = "forelimb long bone"
    FORNIX = "fornix"
    NASAL_ARTERY = "nasal artery"
    FOOT_BONE = "foot bone"
    _4TH_VENTRICLE_EPENDYMA = "4th ventricle ependyma"
    FRONTAL_ASSOCIATION_CORTEX = "frontal association cortex"
    FRONTAL_CORTEX = "frontal cortex"
    FRONTONASAL_SUTURE = "frontonasal suture"
    FUTURE_BRAIN_MARGINAL_LAYER = "future brain marginal layer"
    FUTURE_BRAIN_ROOF_PLATE = "future brain roof plate"
    FUTURE_BRAIN_VENTRICULAR_LAYER = "future brain ventricular layer"
    FUTURE_HINDBRAIN_LATERAL_WALL = "future hindbrain lateral wall"
    FUTURE_HINDBRAIN_ROOF_PLATE = "future hindbrain roof plate"
    FUTURE_HINDBRAIN_VENTRICULAR_LAYER = "future hindbrain ventricular layer"
    FUTURE_SPINAL_CORD_VASCULAR_ELEMENT = "future spinal cord vascular element"
    GALLBLADDER_EPITHELIUM = "gallbladder epithelium"
    GALLBLADDER_LAMINA_PROPRIA = "gallbladder lamina propria"
    GALLBLADDER_SEROSA = "gallbladder serosa"
    GALLBLADDER_SMOOTH_MUSCLE = "gallbladder smooth muscle"
    GASTRIC_ARTERY = "gastric artery"
    GASTROCNEMIUS_MUSCLE = "gastrocnemius muscle"
    GENICULATE_THALAMIC_GROUP = "geniculate thalamic group"
    GENITAL_SWELLING_OF_GENITAL_TUBERCLE = "genital swelling of genital tubercle"
    GIGANTOCELLULAR_RETICULAR_NUCLEUS = "gigantocellular reticular nucleus"
    GIRDLE_BONE = "girdle bone"
    GLOBUS_PALLIDUS = "globus pallidus"
    GONAD_GERMINAL_EPITHELIUM = "gonad germinal epithelium"
    GREATER_EPITHELIAL_RIDGE = "greater epithelial ridge"
    GUT_MUSCULARIS = "gut muscularis"
    HABENULA = "habenula"
    HAIR_CORTEX = "hair cortex"
    HAIR_CUTICLE = "hair cuticle"
    HAIR_MEDULLA = "hair medulla"
    HAIR_ROOT_SHEATH_MATRIX = "hair root sheath matrix"
    HAND_BONE = "hand bone"
    HAND_CONNECTIVE_TISSUE = "hand connective tissue"
    FORELIMB_DISTAL_PHALANX = "forelimb distal phalanx"
    HASSALL_S_CORPUSCLE = "hassall's corpuscle"
    HEART_BLOOD_VESSEL = "heart blood vessel"
    FORELIMB_DIGIT_DERMIS = "forelimb digit dermis"
    HEART_SEPTUM = "heart septum"
    BRANCHIAL_ARCH_EPITHELIUM = "branchial arch epithelium"
    HEMATOPOIETIC_SYSTEM = "hematopoietic system"
    HINDLIMB_LONG_BONE = "hindlimb long bone"
    HIPPOCAMPAL_COMMISSURE = "hippocampal commissure"
    HIPPOCAMPAL_FORMATION = "hippocampal formation"
    HIPPOCAMPUS_GRANULE_CELL_LAYER = "hippocampus granule cell layer"
    HIPPOCAMPUS_LAYER = "hippocampus layer"
    HIPPOCAMPUS_MOLECULAR_LAYER = "hippocampus molecular layer"
    HIPPOCAMPUS_PYRAMIDAL_CELL_LAYER = "hippocampus pyramidal cell layer"
    HIPPOCAMPUS_STRATUM_LACUNOSUM = "hippocampus stratum lacunosum"
    HIPPOCAMPUS_STRATUM_LUCIDUM = "hippocampus stratum lucidum"
    HIPPOCAMPUS_STRATUM_ORIENS = "hippocampus stratum oriens"
    HIPPOCAMPUS_STRATUM_RADIATUM = "hippocampus stratum radiatum"
    HYALINE_CARTILAGE = "hyaline cartilage"
    HYPERTROPHIC_CARTILAGE_ZONE = "hypertrophic cartilage zone"
    HYPOGLOSSAL_NUCLEUS = "hypoglossal nucleus"
    HYPOTHALAMIC_NUCLEUS = "hypothalamic nucleus"
    HYPOTHALAMUS_LATERAL_ZONE = "hypothalamus lateral zone"
    HYPOTHALAMUS_MEDIAL_ZONE = "hypothalamus medial zone"
    HYPOTHALAMUS_MEDIAL_ZONE_TUBERAL_AREA = "hypothalamus medial zone tuberal area"
    HYPOTHALAMUS_PERIVENTRICULAR_ZONE = "hypothalamus periventricular zone"
    HYPOTHALAMUS_VASCULAR_ELEMENT = "hypothalamus vascular element"
    ILEUM_EPITHELIUM = "ileum epithelium"
    ILEUM_MUCOSA = "ileum mucosa"
    RIGHT_LUNG_BRONCHIOLE = "right lung bronchiole"
    IMMATURE_LOOP_OF_HENLE_ASCENDING_LIMB = "immature loop of Henle ascending limb"
    IMMATURE_LOOP_OF_HENLE_DESCENDING_LIMB = "immature loop of Henle descending limb"
    IMPULSE_CONDUCTING_SYSTEM = "impulse conducting system"
    INCISOR_DENTAL_PULP = "incisor dental pulp"
    INDUSIUM_GRISEUM = "indusium griseum"
    INGUINAL_LYMPH_NODE = "inguinal lymph node"
    INSULAR_CORTEX = "insular cortex"
    ALVEOLAR_SEPTUM = "alveolar septum"
    INTERCALATED_AMYGDALOID_NUCLEUS = "intercalated amygdaloid nucleus"
    INTERCEREBRAL_COMMISSURE = "intercerebral commissure"
    CEREBRAL_INTERNAL_CAPSULE = "cerebral internal capsule"
    INTERNAL_SPIRAL_SULCUS = "internal spiral sulcus"
    INTERPEDUNCULAR_NUCLEUS = "interpeduncular nucleus"
    INTESTINE_MUCOSA = "intestine mucosa"
    INTESTINE_MUSCULARIS = "intestine muscularis"
    INTESTINE_MUSCULARIS_MUCOSA = "intestine muscularis mucosa"
    INTESTINE_SMOOTH_MUSCLE = "intestine smooth muscle"
    INTESTINE_WALL = "intestine wall"
    INTRALAMINAR_THALAMIC_GROUP = "intralaminar thalamic group"
    INTRINSIC_NERVE_PLEXUS = "intrinsic nerve plexus"
    IRIS_EPITHELIUM = "iris epithelium"
    IRIS_MUSCLE = "iris muscle"
    IRIS_NERVE = "iris nerve"
    IRIS_STROMA = "iris stroma"
    ISLANDS_OF_CALLEJA = "islands of Calleja"
    JAW_BONE = "jaw bone"
    JAW_PRIMORDIUM_MESENCHYME = "jaw primordium mesenchyme"
    JAW_SKELETON = "jaw skeleton"
    SKELETAL_JOINT = "skeletal joint"
    METANEPHROS_EPITHELIUM = "metanephros epithelium"
    METANEPHROS_GLOMERULAR_EPITHELIUM = "metanephros glomerular epithelium"
    LOOP_OF_HENLE_OF_RENAL_MEDULLA = "loop of Henle of renal medulla"
    KIDNEY_OUTER_MEDULLA_PERITUBULAR_CAPILLARY = "kidney outer medulla peritubular capillary"
    LACRIMAL_APPARATUS = "lacrimal apparatus"
    LACRIMAL_GLAND = "lacrimal gland"
    LACRIMAL_GLAND_BUD = "lacrimal gland bud"
    LARGE_INTESTINE_CRYPT_OF_LIEBERKUHN = "large intestine crypt of lieberkuhn"
    LARGE_INTESTINE_EPITHELIUM = "large intestine epithelium"
    LARGE_INTESTINE_MUCOSA = "large intestine mucosa"
    LARGE_INTESTINE_PEYER_S_PATCH = "large intestine Peyer's patch"
    LARGE_INTESTINE_SMOOTH_MUSCLE = "large intestine smooth muscle"
    LARGE_INTESTINE_WALL = "large intestine wall"
    LARYNX_CONNECTIVE_TISSUE = "larynx connective tissue"
    LARYNX_EPITHELIUM = "larynx epithelium"
    LARYNX_MUSCLE = "larynx muscle"
    LARYNX_SUBMUCOSA = "larynx submucosa"
    LARYNX_SUBMUCOSA_GLAND = "larynx submucosa gland"
    LATERAL_AMYGDALOID_NUCLEUS = "lateral amygdaloid nucleus"
    LATERAL_ENTORHINAL_CORTEX = "lateral entorhinal cortex"
    LATERAL_GENICULATE_NUCLEUS = "lateral geniculate nucleus"
    LATERAL_HABENULAR_NUCLEUS = "lateral habenular nucleus"
    LATERAL_HYPOTHALAMIC_AREA = "lateral hypothalamic area"
    LATERAL_MAMMILLARY_NUCLEUS = "lateral mammillary nucleus"
    LATERAL_MEDULLARY_RETICULAR_FORMATION = "lateral medullary reticular formation"
    LATERAL_PREOPTIC_AREA = "lateral preoptic area"
    LATERAL_RETICULAR_NUCLEUS = "lateral reticular nucleus"
    LATERAL_SEPTAL_COMPLEX = "lateral septal complex"
    LATERAL_SEPTAL_NUCLEUS = "lateral septal nucleus"
    LATERAL_THALAMIC_GROUP = "lateral thalamic group"
    LATERAL_VENTRICLE_EPENDYMA = "lateral ventricle ependyma"
    LATERAL_VENTRICLE_SUBEPENDYMAL_LAYER = "lateral ventricle subependymal layer"
    LATERODORSAL_TEGMENTAL_NUCLEUS = "laterodorsal tegmental nucleus"
    LEFT_COMMON_CAROTID_ARTERY = "left common carotid artery"
    LESSER_EPITHELIAL_RIDGE = "lesser epithelial ridge"
    LIGAMENT = "ligament"
    LIMB_BONE = "limb bone"
    LIMB_LONG_BONE = "limb long bone"
    SPIRAL_LIMBUS = "spiral limbus"
    LIVER_ACINUS = "liver acinus"
    LIVER_VASCULATURE = "liver vasculature"
    LIVER_LOBULE = "liver lobule"
    LIVER_VASCULAR_ELEMENT = "liver vascular element"
    LOCUS_COERULEUS = "locus coeruleus"
    LONG_BONE = "long bone"
    LONG_BONE_DIAPHYSIS = "long bone diaphysis"
    LONG_BONE_EPIPHYSEAL_PLATE = "long bone epiphyseal plate"
    LONG_BONE_EPIPHYSEAL_PLATE_HYPERTROPHIC_ZONE = "long bone epiphyseal plate hypertrophic zone"
    LONG_BONE_EPIPHYSEAL_PLATE_PROLIFERATIVE_ZONE = "long bone epiphyseal plate proliferative zone"
    LONG_BONE_EPIPHYSIS = "long bone epiphysis"
    LONG_BONE_METAPHYSIS = "long bone metaphysis"
    LOOP_OF_HENLE_ASCENDING_LIMB = "loop of Henle ascending limb"
    LOOP_OF_HENLE_DESCENDING_LIMB = "loop of Henle descending limb"
    LOOP_OF_HENLE_DISTAL_STRAIGHT_TUBULE = "loop of Henle distal straight tubule"
    LOOP_OF_HENLE_THICK_LIMB = "loop of Henle thick limb"
    LOOP_OF_HENLE_THIN_DESCENDING_LIMB = "loop of Henle thin descending limb"
    LOOP_OF_HENLE_THIN_LIMB = "loop of Henle thin limb"
    LOWER_JAW_EPITHELIUM = "lower jaw epithelium"
    LOWER_JAW_INCISOR_DENTAL_FOLLICLE = "lower jaw incisor dental follicle"
    LOWER_JAW_MOLAR_DENTAL_FOLLICLE = "lower jaw molar dental follicle"
    LOWER_LEG_BONE = "lower leg bone"
    LOWER_RESPIRATORY_TRACT_SMOOTH_MUSCLE = "lower respiratory tract smooth muscle"
    LUNG_CONNECTIVE_TISSUE = "lung connective tissue"
    LUNG_PARENCHYMA = "lung parenchyma"
    LYMPH_NODE = "lymph node"
    LYMPH_NODE_CAPSULE = "lymph node capsule"
    LYMPH_NODE_CORTEX = "lymph node cortex"
    LYMPH_NODE_ENDOTHELIUM = "lymph node endothelium"
    LYMPH_NODE_FOLLICLE = "lymph node follicle"
    LYMPH_NODE_GERMINAL_CENTER = "lymph node germinal center"
    LYMPH_NODE_MEDULLA = "lymph node medulla"
    LYMPH_NODE_PRIMARY_FOLLICLE = "lymph node primary follicle"
    LYMPH_NODE_SECONDARY_FOLLICLE = "lymph node secondary follicle"
    LYMPHATIC_VESSEL = "lymphatic vessel"
    LYMPHATIC_VESSEL_ENDOTHELIUM = "lymphatic vessel endothelium"
    MACULA_OF_SACCULE = "macula of saccule"
    MACULA_OF_UTRICLE = "macula of utricle"
    MAMMARY_BUD = "mammary bud"
    MAMMARY_GLAND_CONNECTIVE_TISSUE = "mammary gland connective tissue"
    MAMMARY_GLAND_DUCT = "mammary gland duct"
    MAMMARY_GLAND_FAT = "mammary gland fat"
    MAMMARY_PLACODE = "mammary placode"
    MAMMILLARY_BODY = "mammillary body"
    MEDIAL_AMYGDALOID_NUCLEUS = "medial amygdaloid nucleus"
    MEDIAL_DORSAL_THALAMIC_NUCLEUS = "medial dorsal thalamic nucleus"
    MEDIAL_ENTORHINAL_CORTEX = "medial entorhinal cortex"
    MEDIAL_GENICULATE_NUCLEUS = "medial geniculate nucleus"
    MEDIAL_HABENULAR_NUCLEUS = "medial habenular nucleus"
    MEDIAL_MAMMILLARY_NUCLEUS = "medial mammillary nucleus"
    MEDIAL_PREOPTIC_NUCLEUS = "medial preoptic nucleus"
    MEDIAL_PREOPTIC_AREA = "medial preoptic area"
    MEDIAN_RAPHE_NUCLEUS = "median raphe nucleus"
    MEDIAL_SEPTAL_COMPLEX = "medial septal complex"
    MEDIAL_SEPTAL_NUCLEUS = "medial septal nucleus"
    MEDIAL_VESTIBULAR_NUCLEUS = "medial vestibular nucleus"
    MEDIASTINAL_LYMPH_NODE = "mediastinal lymph node"
    MEDULLARY_RETICULAR_FORMATION = "medullary reticular formation"
    MEDULLARY_TUBULE = "medullary tubule"
    MESENCHYME_OF_DEGENERATING_PARAMESONEPHRIC_DUCT_OF_MALE = "mesenchyme of degenerating paramesonephric duct of male"
    MESENCHYME_OF_PARAMESONEPHRIC_DUCT_OF_FEMALE = "mesenchyme of paramesonephric duct of female"
    MESENCHYME_OF_PARAMESONEPHRIC_DUCT_OF_MALE = "mesenchyme of paramesonephric duct of male"
    MESENTERIC_LYMPH_NODE = "mesenteric lymph node"
    MESOMETRIUM = "mesometrium"
    BRANCHIAL_GROOVE_EPITHELIUM = "branchial groove epithelium"
    METATARSUS_MESENCHYME = "metatarsus mesenchyme"
    MIDBRAIN_RETICULAR_FORMATION = "midbrain reticular formation"
    MIDBRAIN_VASCULAR_ELEMENT = "midbrain vascular element"
    MIDDLE_PHALANX = "middle phalanx"
    MIDLINE_THALAMIC_GROUP = "midline thalamic group"
    MINOR_SALIVARY_GLAND = "minor salivary gland"
    MODIOLUS = "modiolus"
    MOLAR_DENTAL_PULP = "molar dental pulp"
    MUSCULATURE = "musculature"
    SKELETAL_MUSCULATURE = "skeletal musculature"
    NAIL = "nail"
    NAIL_BED = "nail bed"
    NAIL_MATRIX = "nail matrix"
    NAIL_PLATE = "nail plate"
    NASAL_MESENCHYME = "nasal mesenchyme"
    NASAL_PROCESS_ECTODERM = "nasal process ectoderm"
    NASAL_PROCESS_MESENCHYME = "nasal process mesenchyme"
    NECK = "neck"
    NEOCORTEX_LAYER = "neocortex layer"
    NEOCORTEX_REGION = "neocortex region"
    NEPHRIC_DUCT_OF_FEMALE = "nephric duct of female"
    NEPHRIC_DUCT_OF_MALE = "nephric duct of male"
    NEPHRON = "nephron"
    NEURAL_PLATE = "neural plate"
    NEURAL_GROOVE = "neural groove"
    NEUROHYPOPHYSIS_PARS_NERVOSA = "neurohypophysis pars nervosa"
    NUCLEUS_AMBIGUUS = "nucleus ambiguus"
    NUCLEUS_OF_DARKSCHEWITSCH = "nucleus of darkschewitsch"
    NUCLEUS_OF_DIAGONAL_BAND = "nucleus of diagonal band"
    NUCLEUS_OF_LATERAL_OLFACTORY_TRACT = "nucleus of lateral olfactory tract"
    NUCLEUS_OF_TRAPEZOID_BODY = "nucleus of trapezoid body"
    NUCLEUS_RELATED_TO_VISUAL_CORTEX = "nucleus related to visual cortex"
    NUCLEUS_REUNIENS = "nucleus reuniens"
    OBLIQUE_ABDOMINAL_MUSCLE = "oblique abdominal muscle"
    OCCIPITAL_CORTEX = "occipital cortex"
    OCULOMOTOR_NUCLEUS = "oculomotor nucleus"
    MAIN_OLFACTORY_BULB_EXTERNAL_PLEXIFORM_LAYER = "main olfactory bulb external plexiform layer"
    MAIN_OLFACTORY_BULB_GLOMERULAR_LAYER = "main olfactory bulb glomerular layer"
    MAIN_OLFACTORY_BULB_GRANULE_CELL_LAYER = "main olfactory bulb granule cell layer"
    MAIN_OLFACTORY_BULB_INTERNAL_PLEXIFORM_LAYER = "main olfactory bulb internal plexiform layer"
    MAIN_OLFACTORY_BULB_MITRAL_CELL_LAYER = "main olfactory bulb mitral cell layer"
    MAIN_OLFACTORY_BULB_SUBEPENDYMAL_ZONE = "main olfactory bulb subependymal zone"
    OLFACTORY_GLAND = "olfactory gland"
    MAIN_OLFACTORY_BULB_OLFACTORY_NERVE_LAYER = "main olfactory bulb olfactory nerve layer"
    OLFACTORY_TRACT = "olfactory tract"
    OLFACTORY_TUBERCLE = "olfactory tubercle"
    OPTIC_TRACT = "optic tract"
    ORAL_REGION_MESENCHYME = "oral region mesenchyme"
    ORBICULARIS_OCULI = "orbicularis oculi"
    OSSEUS_COCHLEA = "osseus cochlea"
    OSSEUS_COCHLEAR_CANAL = "osseus cochlear canal"
    OSSEUS_SPIRAL_LAMINA = "osseus spiral lamina"
    OUTFLOW_TRACT_CARDIAC_MUSCLE = "outflow tract cardiac muscle"
    OUTFLOW_TRACT_PULMONARY_COMPONENT_DISTAL_PART = "outflow tract pulmonary component distal part"
    OVARY_ANTRAL_FOLLICLE = "ovary antral follicle"
    OVARY_CAPSULE = "ovary capsule"
    OVARY_FOLLICLE = "ovary follicle"
    OVARY_GROWING_FOLLICLE = "ovary growing follicle"
    OVARY_PREANTRAL_FOLLICLE = "ovary preantral follicle"
    OVARY_PRIMARY_FOLLICLE = "ovary primary follicle"
    OVARY_SECONDARY_FOLLICLE = "ovary secondary follicle"
    OVARY_STRATUM_GRANULOSUM = "ovary stratum granulosum"
    OVARY_THECA = "ovary theca"
    PALATAL_RUGAE = "palatal rugae"
    HARD_PALATE_PALATAL_RUGAE = "hard palate palatal rugae"
    PALATAL_RUGAE_EPITHELIUM = "palatal rugae epithelium"
    HARD_PALATE_PALATAL_RUGAE_EPITHELIUM = "hard palate palatal rugae epithelium"
    PALATAL_RUGAE_MESENCHYME = "palatal rugae mesenchyme"
    HARD_PALATE_PALATAL_RUGAE_MESENCHYME = "hard palate palatal rugae mesenchyme"
    PALATINE_GLAND = "palatine gland"
    PALATINE_TONSIL = "palatine tonsil"
    PANCREAS_EPITHELIUM = "pancreas epithelium"
    PANCREAS_PRIMORDIUM_EPITHELIUM = "pancreas primordium epithelium"
    PANCREAS_PRIMORDIUM_MESENCHYME = "pancreas primordium mesenchyme"
    PANCREAS_TIP_EPITHELIUM = "pancreas tip epithelium"
    PANCREAS_TRUNK_EPITHELIUM = "pancreas trunk epithelium"
    PANCREATIC_ACINUS = "pancreatic acinus"
    PANCREATIC_ISLET_CORE = "pancreatic islet core"
    PANCREATIC_ISLET_MANTLE = "pancreatic islet mantle"
    PAPILLARY_DUCT = "papillary duct"
    PARABIGEMINAL_NUCLEUS = "parabigeminal nucleus"
    PARABRACHIAL_NUCLEUS = "parabrachial nucleus"
    PARACENTRAL_NUCLEUS = "paracentral nucleus"
    PARAFASCICULAR_NUCLEUS = "parafascicular nucleus"
    PARAMESONEPHRIC_DUCT_OF_FEMALE = "paramesonephric duct of female"
    PARAMESONEPHRIC_DUCT_OF_MALE = "paramesonephric duct of male"
    PARANASAL_SINUS = "paranasal sinus"
    PARATHYROID_GLAND_CAPSULE = "parathyroid gland capsule"
    PARATHYROID_GLAND_PARENCHYMA = "parathyroid gland parenchyma"
    PARAVENTRICULAR_HYPOTHALAMIC_NUCLEUS = "paraventricular hypothalamic nucleus"
    PARAVENTRICULAR_THALAMIC_NUCLEUS = "paraventricular thalamic nucleus"
    PARIETAL_CORTEX = "parietal cortex"
    PATELLA = "patella"
    PECTORAL_GIRDLE_SKELETON = "pectoral girdle skeleton"
    PECTORALIS = "pectoralis"
    FORELIMB_DIGIT_EPIDERMIS = "forelimb digit epidermis"
    PEDUNCULAR_PONTINE_NUCLEUS = "peduncular pontine nucleus"
    MIDDLE_EAR_EPITHELIUM = "middle ear epithelium"
    PENIS_EPITHELIUM = "penis epithelium"
    PENIS_SKIN = "penis skin"
    PERICHONDRIUM = "perichondrium"
    PERILYMPHATIC_CHANNEL = "perilymphatic channel"
    PERINEURAL_VASCULAR_PLEXUS = "perineural vascular plexus"
    PERIODONTAL_LIGAMENT = "periodontal ligament"
    PERIOLIVARY_NUCLEUS = "periolivary nucleus"
    PERIOSTEUM = "periosteum"
    PERIRHINAL_CORTEX = "perirhinal cortex"
    PERITONEAL_FLUID = "peritoneal fluid"
    PEYER_S_PATCH_GERMINAL_CENTER = "Peyer's patch germinal center"
    PHOTORECEPTOR_LAYER = "photoreceptor layer"
    PHOTORECEPTOR_LAYER_INNER_SEGMENT = "photoreceptor layer inner segment"
    PHOTORECEPTOR_LAYER_OUTER_SEGMENT = "photoreceptor layer outer segment"
    PLACENTA = "placenta"
    PLASMA = "plasma"
    SPINAL_CORD_INTERMEDIATE_GREY = "spinal cord intermediate grey"
    PONTINE_NUCLEUS = "pontine nucleus"
    PONTINE_RETICULAR_FORMATION = "pontine reticular formation"
    POSTERIOR_AMYGDALOID_NUCLEUS = "posterior amygdaloid nucleus"
    POSTERIOR_COMMISSURE = "posterior commissure"
    POSTERIOR_SEPTAL_COMPLEX = "posterior septal complex"
    POSTERIOR_THALAMIC_GROUP = "posterior thalamic group"
    POSTERIOR_THALAMIC_NUCLEUS = "posterior thalamic nucleus"
    POSTEROLATERAL_CORTICAL_AMYGDALOID_NUCLEUS = "posterolateral cortical amygdaloid nucleus"
    POSTEROMEDIAL_CORTICAL_AMYGDALOID_NUCLEUS = "posteromedial cortical amygdaloid nucleus"
    PREPUTIAL_GLAND = "preputial gland"
    PREPUTIAL_SWELLING = "preputial swelling"
    PRESUBICULUM = "presubiculum"
    PRIMARY_MOTOR_CORTEX = "primary motor cortex"
    PRIMARY_OOCYTE = "primary oocyte"
    PRIMARY_SOMATOSENSORY_CORTEX = "primary somatosensory cortex"
    PRIMARY_SPERMATOCYTE = "primary spermatocyte"
    PRIMARY_VISUAL_CORTEX = "primary visual cortex"
    FORELIMB_INTERDIGITAL_REGION_SKIN = "forelimb interdigital region skin"
    PROSTATE_GLAND_DORSOLATERAL_LOBE = "prostate gland dorsolateral lobe"
    PROSTATE_EPITHELIUM = "prostate epithelium"
    PROSTATE_GLAND_LOBE = "prostate gland lobe"
    PROSTATE_GLAND_SMOOTH_MUSCLE = "prostate gland smooth muscle"
    PROSTATE_STROMA = "prostate stroma"
    FORELIMB_INTERDIGITAL_REGION_DERMIS = "forelimb interdigital region dermis"
    PROXIMAL_PHALANX = "proximal phalanx"
    RIGHT_LUNG_ALVEOLAR_SYSTEM = "right lung alveolar system"
    PURKINJE_FIBER = "purkinje fiber"
    PUTAMEN = "putamen"
    RAPHE_MAGNUS_NUCLEUS = "raphe magnus nucleus"
    RAPHE_NUCLEUS = "raphe nucleus"
    RAPHE_PALLIDUS_NUCLEUS = "raphe pallidus nucleus"
    RED_NUCLEUS = "red nucleus"
    RENAL_CONVOLUTED_TUBULE = "renal convoluted tubule"
    RENAL_CORPUSCLE = "renal corpuscle"
    RENAL_PAPILLA = "renal papilla"
    REPRODUCTIVE_SYSTEM_MESENCHYME = "reproductive system mesenchyme"
    RESPIRATORY_BRONCHIOLE = "respiratory bronchiole"
    RESPIRATORY_BRONCHIOLE_EPITHELIUM = "respiratory bronchiole epithelium"
    RESPIRATORY_SYSTEM_BLOOD_VESSEL = "respiratory system blood vessel"
    RESPIRATORY_SYSTEM_BLOOD_VESSEL_ENDOTHELIUM = "respiratory system blood vessel endothelium"
    RESPIRATORY_SYSTEM_MUSCLE = "respiratory system muscle"
    RESPIRATORY_SYSTEM_SMOOTH_MUSCLE = "respiratory system smooth muscle"
    RETICULAR_MEMBRANE = "reticular membrane"
    RETICULAR_TEGMENTAL_NUCLEUS = "reticular tegmental nucleus"
    RETICULAR_THALAMIC_NUCLEUS = "reticular thalamic nucleus"
    RETINA_EPITHELIUM = "retina epithelium"
    RETINA_GANGLION_CELL_LAYER = "retina ganglion cell layer"
    RETINA_INNER_PLEXIFORM_LAYER = "retina inner plexiform layer"
    RETINA_LAYER = "retina layer"
    RETINA_OUTER_PLEXIFORM_LAYER = "retina outer plexiform layer"
    RETROHIPPOCAMPAL_CORTEX = "retrohippocampal cortex"
    RHOMBOMERE_NEURAL_CREST = "rhombomere neural crest"
    RHOMBOMERE_NEURAL_FOLD = "rhombomere neural fold"
    RIGHT_COMMON_CAROTID_ARTERY = "right common carotid artery"
    SALIVARY_DUCT = "salivary duct"
    SCALA_MEDIA = "scala media"
    SCALA_TYMPANI = "scala tympani"
    SCALA_VESTIBULI = "scala vestibuli"
    SEBACEOUS_GLAND = "sebaceous gland"
    SECONDARY_OOCYTE = "secondary oocyte"
    SECONDARY_SOMATOSENSORY_CORTEX = "secondary somatosensory cortex"
    SECONDARY_SPERMATOCYTE = "secondary spermatocyte"
    SECONDARY_VISUAL_CORTEX = "secondary visual cortex"
    SECRETORY_CIRCUMVENTRICULAR_ORGAN = "secretory circumventricular organ"
    SEMILUNAR_VALVE = "semilunar valve"
    SEMINIFEROUS_TUBULE_EPITHELIUM = "seminiferous tubule epithelium"
    SEMITENDINOSUS = "semitendinosus"
    SEPTAL_NUCLEUS = "septal nucleus"
    SEPTAL_OLFACTORY_ORGAN = "septal olfactory organ"
    GRUENEBERG_GANGLION = "Grueneberg ganglion"
    SEPTAL_ORGAN_OF_MASERA = "septal organ of Masera"
    SEPTOFIMBRIAL_NUCLEUS = "septofimbrial nucleus"
    SEPTOHIPPOCAMPAL_NUCLEUS = "septohippocampal nucleus"
    SERRATUS_MUSCLE = "serratus muscle"
    SERUM = "serum"
    SESAMOID_BONE_OF_GASTROCNEMIUS = "sesamoid bone of gastrocnemius"
    SINOATRIAL_NODE = "sinoatrial node"
    SKELETAL_SYSTEM = "skeletal system"
    SKIN_GLAND = "skin gland"
    SKIN_MUCOUS_GLAND = "skin mucous gland"
    SKIN_MUSCLE = "skin muscle"
    SMALL_INTESTINE_CRYPT_OF_LIEBERKUHN = "small intestine crypt of lieberkuhn"
    SMALL_INTESTINE_EPITHELIUM = "small intestine epithelium"
    SMALL_INTESTINE_LAMINA_PROPRIA = "small intestine lamina propria"
    SMALL_INTESTINE_MUCOSA = "small intestine mucosa"
    SMALL_INTESTINE_MUSCULARIS_MUCOSA = "small intestine muscularis mucosa"
    SMALL_INTESTINE_PEYER_S_PATCH = "small intestine Peyer's patch"
    SMALL_INTESTINE_SMOOTH_MUSCLE = "small intestine smooth muscle"
    INTESTINAL_VILLUS = "intestinal villus"
    SMALL_INTESTINE_WALL = "small intestine wall"
    SOLEUS = "soleus"
    SOLITARY_TRACT_NUCLEUS = "solitary tract nucleus"
    SPHINCTER_PUPILLAE = "sphincter pupillae"
    SPINAL_CORD_CENTRAL_CANAL = "spinal cord central canal"
    SPINAL_CORD_DORSAL_HORN = "spinal cord dorsal horn"
    SPINAL_CORD_EPENDYMAL_LAYER = "spinal cord ependymal layer"
    SPINAL_CORD_GRAY_MATTER = "spinal cord gray matter"
    SPINAL_CORD_VASCULAR_ELEMENT = "spinal cord vascular element"
    SPINAL_CORD_VENTRAL_HORN = "spinal cord ventral horn"
    SPINAL_CORD_WHITE_MATTER = "spinal cord white matter"
    SPINAL_GANGLION_OR_NERVE = "spinal ganglion or nerve"
    SPINAL_VESTIBULAR_NUCLEUS = "spinal vestibular nucleus"
    SPIRAL_LIGAMENT = "spiral ligament"
    SPIRAL_PROMINENCE = "spiral prominence"
    SPIRAL_SULCUS = "spiral sulcus"
    SPIRAL_VESSEL = "spiral vessel"
    SPLEEN_CAPSULE = "spleen capsule"
    SPLEEN_GERMINAL_CENTER = "spleen germinal center"
    SPLEEN_LYMPHOID_FOLLICLE = "spleen lymphoid follicle"
    SPLEEN_PULP = "spleen pulp"
    SPLEEN_RED_PULP = "spleen red pulp"
    SPLEEN_SMOOTH_MUSCLE = "spleen smooth muscle"
    SPLEEN_TRABECULUM = "spleen trabeculum"
    SPLEEN_WHITE_PULP = "spleen white pulp"
    BODY_OF_STOMACH = "body of stomach"
    STOMACH_CARDIAC_REGION = "stomach cardiac region"
    STOMACH_GLANDULAR_EPITHELIUM = "stomach glandular epithelium"
    STOMACH_MUCOSA = "stomach mucosa"
    STOMACH_MUSCULARIS_MUCOSA = "stomach muscularis mucosa"
    RIGHT_LUNG_ALVEOLAR_DUCT = "right lung alveolar duct"
    STOMACH_REGION = "stomach region"
    STOMACH_SEROSA = "stomach serosa"
    STOMACH_SMOOTH_MUSCLE = "stomach smooth muscle"
    STOMACH_SQUAMOUS_EPITHELIUM = "stomach squamous epithelium"
    STOMACH_SUBMUCOSA = "stomach submucosa"
    STOMACH_WALL = "stomach wall"
    STRIA_VASCULARIS = "stria vascularis"
    SUBCOMMISSURAL_ORGAN = "subcommissural organ"
    SUBCUTANEOUS_ADIPOSE_TISSUE = "subcutaneous adipose tissue"
    SUBFORNICAL_ORGAN = "subfornical organ"
    DENTATE_GYRUS_SUBGRANULAR_ZONE = "dentate gyrus subgranular zone"
    SUBICULUM = "subiculum"
    SUBMANDIBULAR_DUCT = "submandibular duct"
    SUBMUCOUS_NERVE_PLEXUS = "submucous nerve plexus"
    SUBSTANTIA_NIGRA = "substantia nigra"
    SUBSTANTIA_NIGRA_PARS_COMPACTA = "substantia nigra pars compacta"
    SUBSTANTIA_NIGRA_PARS_LATERALIS = "substantia nigra pars lateralis"
    SUBSTANTIA_NIGRA_PARS_RETICULATA = "substantia nigra pars reticulata"
    SUBTHALAMIC_NUCLEUS = "subthalamic nucleus"
    SUPERIOR_OLIVARY_NUCLEUS = "superior olivary nucleus"
    SUPRACHIASMATIC_NUCLEUS = "suprachiasmatic nucleus"
    SUPRAMAMMILLARY_NUCLEUS = "supramammillary nucleus"
    SUPRAOPTIC_NUCLEUS = "supraoptic nucleus"
    SWEAT_GLAND = "sweat gland"
    SYMPATHETIC_NERVE_TRUNK = "sympathetic nerve trunk"
    SYNCHONDROSIS = "synchondrosis"
    TAENIA_TECTA = "taenia tecta"
    TAIL_BLOOD_VESSEL = "tail blood vessel"
    TAIL_MYOTOME = "tail myotome"
    TASTE_BUD = "taste bud"
    TECTORIAL_MEMBRANE = "tectorial membrane"
    TEMPORAL_CORTEX_ASSOCIATION_AREA = "temporal cortex association area"
    TENDO_CALCANEUS = "tendo calcaneus"
    TENDON = "tendon"
    TENSOR_FASCIAE_LATA = "tensor fasciae lata"
    TESTIS_CORTICAL_REGION = "testis cortical region"
    TESTIS_MEDULLARY_REGION = "testis medullary region"
    TESTIS_MESENCHYME = "testis mesenchyme"
    THALAMIC_NUCLEUS = "thalamic nucleus"
    THALAMUS_VASCULAR_ELEMENT = "thalamus vascular element"
    THIN_ASCENDING_LIMB_OF_MEDULLA = "thin ascending limb of medulla"
    THORACIC_SEGMENT_OF_TRUNK = "thoracic segment of trunk"
    THYMUS_CORTEX = "thymus cortex"
    THYMUS_EPITHELIUM = "thymus epithelium"
    THYMUS_PARATHYROID_PRIMORDIUM = "thymus/parathyroid primordium"
    TIBIALIS_ANTERIOR = "tibialis anterior"
    TIBIALIS_MUSCLE = "tibialis muscle"
    TISSUE = "tissue"
    TONGUE_PAPILLA_EPITHELIUM = "tongue papilla epithelium"
    TONGUE_VASCULAR_ELEMENT = "tongue vascular element"
    TONSIL = "tonsil"
    TOOTH_SUBSTANCE = "tooth substance"
    TRABECULAR_BONE = "trabecular bone"
    TRACHEA_BLOOD_VESSEL = "trachea blood vessel"
    TRACHEA_CARTILAGE = "trachea cartilage"
    TRACHEA_CONNECTIVE_TISSUE = "trachea connective tissue"
    TRACHEA_LAMINA_PROPRIA = "trachea lamina propria"
    TRACHEA_NON_CARTILAGE_CONNECTIVE_TISSUE = "trachea non-cartilage connective tissue"
    TRACHEA_SUBMUCOSA = "trachea submucosa"
    TRANSVERSE_COLON = "transverse colon"
    TRIGEMINAL_V_MESENCEPHALIC_NUCLEUS = "trigeminal V mesencephalic nucleus"
    TRIGEMINAL_V_MOTOR_NUCLEUS = "trigeminal V motor nucleus"
    TRIGEMINAL_NUCLEUS = "trigeminal nucleus"
    TRIGEMINAL_V_SENSORY_NUCLEUS = "trigeminal V sensory nucleus"
    TRIGEMINAL_V_SPINAL_SENSORY_NUCLEUS = "trigeminal V spinal sensory nucleus"
    TROCHLEAR_NUCLEUS = "trochlear nucleus"
    TRUNCUS_ARTERIOSUS = "truncus arteriosus"
    TRUNK_MUSCULATURE = "trunk musculature"
    TUBEROMAMMILLARY_NUCLEUS = "tuberomammillary nucleus"
    TURBINATE_BONE_PRIMORDIUM = "turbinate bone primordium"
    UPPER_ARM_BONE = "upper arm bone"
    UPPER_LEG_BONE = "upper leg bone"
    UPPER_RESPIRATORY_TRACT_EPITHELIUM = "upper respiratory tract epithelium"
    URETER_INTERSTITIUM = "ureter interstitium"
    URETHRA_EPITHELIUM = "urethra epithelium"
    BLADDER_MUCOSA = "bladder mucosa"
    URINARY_SYSTEM_MESENCHYME = "urinary system mesenchyme"
    UROGENITAL_RIDGE = "urogenital ridge"
    UTERUS_STROMA = "uterus stroma"
    UVEA = "uvea"
    LAMINA_PROPRIA_OF_VAGINA = "lamina propria of vagina"
    CARDIAC_VALVE_LEAFLET = "cardiac valve leaflet"
    VASCULAR_PLEXUS = "vascular plexus"
    VASCULAR_SYSTEM = "vascular system"
    VEIN = "vein"
    VENOUS_SYSTEM_ENDOTHELIUM = "venous system endothelium"
    VENTRAL_LATERAL_GENICULATE_NUCLEUS = "ventral lateral geniculate nucleus"
    VENTRAL_NASAL_ARTERY = "ventral nasal artery"
    VENTRAL_POSTERIOR_LATERAL_THALAMIC_NUCLEUS = "ventral posterior lateral thalamic nucleus"
    VENTRAL_POSTERIOR_MEDIAL_THALAMIC_NUCLEUS = "ventral posterior medial thalamic nucleus"
    VENTRAL_POSTERIOR_THALAMIC_GROUP = "ventral posterior thalamic group"
    VENTRAL_TEGMENTAL_AREA = "ventral tegmental area"
    VENTRAL_TEGMENTAL_NUCLEUS = "ventral tegmental nucleus"
    PRETHALAMUS = "prethalamus"
    VENTROMEDIAL_HYPOTHALAMIC_NUCLEUS = "ventromedial hypothalamic nucleus"
    VENULE = "venule"
    VERTEBRA_CAUDAL_ARTICULAR_PROCESS = "vertebra caudal articular process"
    VERTEBRAL_COLUMN = "vertebral column"
    VESTIBULAR_EPITHELIUM = "vestibular epithelium"
    VESTIBULAR_MEMBRANE = "vestibular membrane"
    VESTIBULAR_NUCLEUS = "vestibular nucleus"
    VISCEROCRANIUM_BONE = "viscerocranium bone"
    WHARTON_S_JELLY = "Wharton's jelly"
    WHITE_FAT = "white fat"
    WHITE_MATTER = "white matter"
    YOLK_SAC_PARIETAL_ENDODERM = "yolk sac parietal endoderm"
    YOLK_SAC_VISCERAL_ENDODERM = "yolk sac visceral endoderm"
    ZONA_INCERTA = "zona incerta"
    PELVIC_REGION_OF_TRUNK = "pelvic region of trunk"
    VENOUS_BLOOD_VESSEL = "venous blood vessel"
    FORELIMB_BONE = "forelimb bone"
    HINDLIMB_BONE = "hindlimb bone"
    CORNEA_ENDOTHELIUM = "cornea endothelium"
    FORELIMB_INTERDIGITAL_REGION_EPIDERMIS = "forelimb interdigital region epidermis"
    HINDLIMB_INTERDIGITAL_REGION_SKIN = "hindlimb interdigital region skin"
    HINDLIMB_INTERDIGITAL_REGION_DERMIS = "hindlimb interdigital region dermis"
    HINDLIMB_INTERDIGITAL_REGION_EPIDERMIS = "hindlimb interdigital region epidermis"
    LIMB_INTERDIGITAL_REGION_DERMIS = "limb interdigital region dermis"
    LIMB_INTERDIGITAL_REGION_EPIDERMIS = "limb interdigital region epidermis"
    FUTURE_FOREBRAIN_VENTRICULAR_LAYER = "future forebrain ventricular layer"
    LIMB_BUD = "limb bud"
    GUM = "gum"
    NERVE_OF_FEMALE_REPRODUCTIVE_SYSTEM = "nerve of female reproductive system"
    NERVE_OF_MALE_REPRODUCTIVE_SYSTEM = "nerve of male reproductive system"
    NERVE_OF_URINARY_SYSTEM = "nerve of urinary system"
    ORGAN = "organ"
    LIVER_AND_BILIARY_SYSTEM_EPITHELIUM = "liver and biliary system epithelium"
    REPRODUCTIVE_SYSTEM_EPITHELIUM = "reproductive system epithelium"
    SENSORY_ORGAN_EPITHELIUM = "sensory organ epithelium"
    URINARY_SYSTEM_EPITHELIUM = "urinary system epithelium"
    GENITOURINARY_SYSTEM_EPITHELIUM = "genitourinary system epithelium"
    SENSORY_ORGAN = "sensory organ"
    OUTER_EAR_EPITHELIUM = "outer ear epithelium"
    EYE_EPITHELIUM = "eye epithelium"
    NASAL_EPITHELIUM = "nasal epithelium"
    NARIS_EPITHELIUM = "naris epithelium"
    NASAL_PROCESS_EPITHELIUM = "nasal process epithelium"
    EMBRYO_MESENCHYME = "embryo mesenchyme"
    FEMALE_REPRODUCTIVE_SYSTEM_EPITHELIUM = "female reproductive system epithelium"
    MALE_REPRODUCTIVE_SYSTEM_EPITHELIUM = "male reproductive system epithelium"
    PROXIMAL_URETHRAL_EPITHELIUM = "proximal urethral epithelium"
    PROXIMAL_URETHRAL_MEATUS = "proximal urethral meatus"
    GUM_EPITHELIUM = "gum epithelium"
    INTEGUMENTAL_SYSTEM_MESENCHYME = "integumental system mesenchyme"
    RESPIRATORY_SYSTEM_MESENCHYME = "respiratory system mesenchyme"
    METANEPHROS_PARIETAL_EPITHELIUM = "metanephros parietal epithelium"
    METANEPHROS_VISCERAL_EPITHELIUM = "metanephros visceral epithelium"
    SENSORY_ORGAN_MESENCHYME = "sensory organ mesenchyme"
    EAR_MESENCHYME = "ear mesenchyme"
    CARDIOVASCULAR_SYSTEM_MESENCHYME = "cardiovascular system mesenchyme"
    FEMALE_REPRODUCTIVE_SYSTEM_MESENCHYME = "female reproductive system mesenchyme"
    MALE_REPRODUCTIVE_SYSTEM_MESENCHYME = "male reproductive system mesenchyme"
    CARTILAGE_PRIMORDIUM_OF_FUTURE_SPINOUS_PROCESS = "cartilage primordium of future spinous process"
    RESPIRATORY_SYSTEM_CONNECTIVE_TISSUE = "respiratory system connective tissue"
    SENSORY_ORGAN_CARTILAGE = "sensory organ cartilage"
    OSSICLE_CARTILAGE_CONDENSATION = "ossicle cartilage condensation"
    PECTORAL_THORACIC_CARTILAGE_CONDENSATION = "pectoral-thoracic cartilage condensation"
    PELVIC_GIRDLE_CARTILAGE_CONDENSATION = "pelvic girdle cartilage condensation"
    OSSICLE_PRE_CARTILAGE_CONDENSATION = "ossicle pre-cartilage condensation"
    PECTORAL_THORACIC_PRE_CARTILAGE_CONDENSATION = "pectoral-thoracic pre-cartilage condensation"
    PELVIC_GIRDLE_PRE_CARTILAGE_CONDENSATION = "pelvic girdle pre-cartilage condensation"
    ECTODERM = "ectoderm"
    ENDODERM = "endoderm"
    MESODERM = "mesoderm"
    SKELETAL_MUSCLE = "skeletal muscle"
    EYE_NEURAL_ECTODERM = "eye neural ectoderm"
    ALIMENTARY_SYSTEM_ECTODERM = "alimentary system ectoderm"
    ALIMENTARY_SYSTEM_ENDODERM = "alimentary system endoderm"
    ABDOMINAL_LYMPH_NODE = "abdominal lymph node"
    VASCULAR_ELEMENT = "vascular element"
    ALIMENTARY_SYSTEM_VASCULAR_ELEMENT = "alimentary system vascular element"
    RESPIRATORY_SYSTEM_VASCULAR_ELEMENT = "respiratory system vascular element"
    HEAD_BONE = "head bone"
    RIGHT_LUNG_ALVEOLUS = "right lung alveolus"
    PITUITARY_GLAND = "pituitary gland"
    ENDOCRINE_GLAND = "endocrine gland"
    GERM_CELL_OF_GONAD = "germ cell of gonad"
    MEDIASTINUM = "mediastinum"
    AUDITORY_SYSTEM = "auditory system"
    VISUAL_SYSTEM = "visual system"
    OLFACTORY_SYSTEM = "olfactory system"
    LUNG_HILUS = "lung hilus"
    LOBAR_BRONCHUS_VASCULAR_ELEMENT = "lobar bronchus vascular element"
    LUNG_RUDIMENT = "lung rudiment"
    LUNG_RUDIMENT_EPITHELIUM = "lung rudiment epithelium"
    LUNG_RUDIMENT_MESENCHYME = "lung rudiment mesenchyme"
    LUNG_RUDIMENT_VASCULAR_ELEMENT = "lung rudiment vascular element"
    RIB_HEAD_CARTILAGE = "rib head cartilage"
    RIB_NECK_CARTILAGE = "rib neck cartilage"
    RIB_SHAFT = "rib shaft"
    RIB_SHAFT_NON_OSSIFIED_REGION = "rib shaft non-ossified region"
    RIB_SHAFT_OSSIFIED_REGION = "rib shaft ossified region"
    UMBILICAL_ARTERY = "umbilical artery"
    UMBILICAL_VEIN = "umbilical vein"
    LEFT_UMBILICAL_VEIN = "left umbilical vein"
    RIGHT_UMBILICAL_VEIN = "right umbilical vein"
    VITELLINE_VEIN = "vitelline vein"
    LEFT_VITELLINE_VEIN = "left vitelline vein"
    RIGHT_VITELLINE_VEIN = "right vitelline vein"
    FUTURE_TELENCEPHALON = "future telencephalon"
    PROEPICARDIUM = "proepicardium"
    NASAL_CAVITY_MUCOSA = "nasal cavity mucosa"
    NASAL_CAVITY_LAMINA_PROPRIA = "nasal cavity lamina propria"
    NASAL_SEPTUM_MUCOSA = "nasal septum mucosa"
    NASAL_SEPTUM_LAMINA_PROPRIA = "nasal septum lamina propria"
    ULTIMOBRANCHIAL_BODY = "ultimobranchial body"
    BODY_REGION = "body region"
    EARLY_EMBRYO = "early embryo"
    GERM_LAYER = "germ layer"
    UMBILICAL_OR_VITELLINE_VESSEL = "umbilical or vitelline vessel"
    BLASTOCYST = "blastocyst"
    RIB_CARTILAGE = "rib cartilage"
    CAVITY_OR_LINING = "cavity or lining"
    PROAMNIOTIC_CAVITY = "proamniotic cavity"
    PRIMARY_YOLK_SAC_CAVITY = "primary yolk sac cavity"
    CONCEPTUS = "conceptus"
    _1_CELL_STAGE_CONCEPTUS = "1-cell stage conceptus"
    _2_CELL_STAGE_CONCEPTUS = "2-cell stage conceptus"
    _4_CELL_STAGE_CONCEPTUS = "4-cell stage conceptus"
    _8_CELL_STAGE_CONCEPTUS = "8-cell stage conceptus"
    MORULA_STAGE_CONCEPTUS = "morula-stage conceptus"
    BLASTOCYST_STAGE_CONCEPTUS = "blastocyst-stage conceptus"
    LIMB_MUSCLE = "limb muscle"
    HINDLIMB_MUSCLE = "hindlimb muscle"
    FORELIMB_MUSCLE = "forelimb muscle"
    NECK_MUSCLE = "neck muscle"
    AMYGDALOID_NUCLEUS = "amygdaloid nucleus"
    ANAL_SPHINCTER = "anal sphincter"
    ARM_MUSCLE = "arm muscle"
    LEG_MUSCLE = "leg muscle"
    ATRIUM_AURICULAR_REGION = "atrium auricular region"
    ATRIUM_AURICULAR_REGION_CARDIAC_MUSCLE = "atrium auricular region cardiac muscle"
    ATRIUM_AURICULAR_REGION_ENDOCARDIAL_LINING = "atrium auricular region endocardial lining"
    ATRIUM_LUMEN = "atrium lumen"
    COMMON_ATRIAL_CHAMBER_AURICULAR_REGION = "common atrial chamber auricular region"
    COMMON_ATRIAL_CHAMBER_AURICULAR_REGION_CARDIAC_MUSCLE = "common atrial chamber auricular region cardiac muscle"
    COMMON_ATRIAL_CHAMBER_AURICULAR_REGION_ENDOCARDIAL_LINING = (
        "common atrial chamber auricular region endocardial lining"
    )
    CRANIUM_BONE_PRIMORDIUM = "cranium bone primordium"
    _4_8_CELL_STAGE_CONCEPTUS = "4-8 cell stage conceptus"
    CRANIAL_PREGANGLION = "cranial preganglion"
    HYOID_CARTILAGE_PRIMORDIUM = "hyoid cartilage primordium"
    INTERVENTRICULAR_FORAMEN = "interventricular foramen"
    ANTERIOR_VISCERAL_ENDODERM = "anterior visceral endoderm"
    POSTERIOR_VISCERAL_ENDODERM = "posterior visceral endoderm"
    CLOACAL_EPITHELIUM_OF_CLOACAL_MEMBRANE = "cloacal epithelium of cloacal membrane"
    SURFACE_ECTODERM_OF_CLOACAL_MEMBRANE = "surface ectoderm of cloacal membrane"
    SUPERFICIAL_CELL_OF_BLADDER_UROTHELIUM = "superficial cell of bladder urothelium"
    INTERMEDIATE_CELL_LAYER_OF_BLADDER_UROTHELIUM = "intermediate cell layer of bladder urothelium"
    KRT5_EXPRESSING_BASAL_CELL_OF_BLADDER_UROTHELIUM = "Krt5-expressing basal cell of bladder urothelium"
    SUBUROTHELIAL_LAMINA_PROPRIA_OF_BLADDER = "suburothelial lamina propria of bladder"
    OUTER_LAMINA_PROPRIA_OF_BLADDER = "outer lamina propria of bladder"
    OUTER_LAYER_OF_BLADDER = "outer layer of bladder"
    BLADDER_NECK = "bladder neck"
    VENTRAL_BLADDER_NECK = "ventral bladder neck"
    URETER_JUNCTION_OF_BLADDER_TRIGONE = "ureter junction of bladder trigone"
    URETHRA_JUNCTION_OF_BLADDER_TRIGONE = "urethra junction of bladder trigone"
    URETERAL_SMOOTH_MUSCLE_OF_TRIGONE = "ureteral smooth muscle of trigone"
    FUNDUS_JUNCTION_OF_VENTRAL_BLADDER_NECK = "fundus junction of ventral bladder neck"
    URETHRA_JUNCTION_OF_VENTRAL_BLADDER_NECK = "urethra junction of ventral bladder neck"
    UNDIFFERENTIATED_EPITHELIAL_CELL_OF_UROTHELIUM_OF_PRIMITIVE_BLADDER = (
        "undifferentiated epithelial cell of urothelium of primitive bladder"
    )
    INTERMEDIATE_CELL_OF_UROTHELIUM_OF_PRIMITIVE_BLADDER = "intermediate cell of urothelium of primitive bladder"
    P_0_CELL_OF_UROTHELIUM_OF_PRIMITIVE_BLADDER = "P-0 cell of urothelium of primitive bladder"
    INNER_MESENCHYMAL_LAYER_OF_PRIMITIVE_BLADDER = "inner mesenchymal layer of primitive bladder"
    OUTER_MESENCHYMAL_LAYER_OF_PRIMITIVE_BLADDER = "outer mesenchymal layer of primitive bladder"
    MUSCLE_LAYER_OF_PRIMITIVE_BLADDER = "muscle layer of primitive bladder"
    SMOOTH_MUSCLE_CELL_OF_OUTER_MESENCHYMAL_LAYER_OF_PRIMITIVE_BLADDER = (
        "smooth muscle cell of outer mesenchymal layer of primitive bladder"
    )
    MESENCHYMAL_CELL_OF_OUTER_MESENCHYMAL_LAYER_OF_PRIMITIVE_BLADDER = (
        "mesenchymal cell of outer mesenchymal layer of primitive bladder"
    )
    OUTER_LAYER_OF_PRIMITIVE_BLADDER = "outer layer of primitive bladder"
    SEROSA_OF_PRIMITIVE_BLADDER = "serosa of primitive bladder"
    ADVENTITIA_OF_PRIMITIVE_BLADDER = "adventitia of primitive bladder"
    PRIMITIVE_BLADDER_CAUDAL_UROGENITAL_SINUS_TRANSITION_ZONE = (
        "primitive bladder-caudal urogenital sinus transition zone"
    )
    EPITHELIUM_OF_PRIMITIVE_BLADDER_CAUDAL_UROGENITAL_SINUS_TRANSITION_ZONE = (
        "epithelium of primitive bladder-caudal urogenital sinus transition zone"
    )
    MESENCHYME_OF_PRIMITIVE_BLADDER_CAUDAL_UROGENITAL_SINUS_TRANSITION_ZONE = (
        "mesenchyme of primitive bladder-caudal urogenital sinus transition zone"
    )
    UROGENITAL_SINUS_RIDGE = "urogenital sinus ridge"
    REST_OF_CAUDAL_UROGENITAL_SINUS_EPITHELIUM = "rest of caudal urogenital sinus epithelium"
    COMMON_NEPHRIC_DUCT = "common nephric duct"
    DEVELOPING_VASCULATURE_OF_PELVIC_URETHRA = "developing vasculature of pelvic urethra"
    ADVENTITIA_OF_PELVIC_URETHRA = "adventitia of pelvic urethra"
    REST_OF_EPITHELIUM_OF_PELVIC_URETHRA = "rest of epithelium of pelvic urethra"
    SUPERFICIAL_CELL_LAYER_OF_PELVIC_URETHRA_OF_FEMALE = "superficial cell layer of pelvic urethra of female"
    INTERMEDIATE_CELL_LAYER_OF_PELVIC_URETHRA_OF_FEMALE = "intermediate cell layer of pelvic urethra of female"
    BASAL_CELL_LAYER_OF_PELVIC_URETHRA_OF_FEMALE = "basal cell layer of pelvic urethra of female"
    UROGENITAL_SINUS_RIDGE_OF_FEMALE = "urogenital sinus ridge of female"
    REST_OF_EPITHELIUM_OF_DORSAL_PELVIC_URETHRA_OF_FEMALE = "rest of epithelium of dorsal pelvic urethra of female"
    LAMINA_PROPRIA_OF_PELVIC_URETHRA_OF_FEMALE = "lamina propria of pelvic urethra of female"
    MUSCULARIS_MUCOSA_OF_PELVIC_URETHRA_OF_FEMALE = "muscularis mucosa of pelvic urethra of female"
    SUBMUCOSA_OF_PELVIC_URETHRA_OF_FEMALE = "submucosa of pelvic urethra of female"
    ANTERIOR_MESENCHYMAL_PAD_OF_FEMALE = "anterior mesenchymal pad of female"
    DORSAL_MESENCHYMAL_PAD_OF_FEMALE = "dorsal mesenchymal pad of female"
    VENTRAL_MESENCHYMAL_PAD_OF_FEMALE = "ventral mesenchymal pad of female"
    SMOOTH_MUSCLE_LAYER_OF_PELVIC_URETHRA_OF_FEMALE = "smooth muscle layer of pelvic urethra of female"
    RHABDOSPHINCTER_OF_FEMALE = "rhabdosphincter of female"
    ADVENTITIA_OF_PELVIC_URETHRA_OF_FEMALE = "adventitia of pelvic urethra of female"
    EGG_CYLINDER = "egg cylinder"
    SUPERFICIAL_CELL_LAYER_OF_PELVIC_URETHRA_OF_MALE = "superficial cell layer of pelvic urethra of male"
    INTERMEDIATE_CELL_LAYER_OF_PELVIC_URETHRA_OF_MALE = "intermediate cell layer of pelvic urethra of male"
    BASAL_CELL_LAYER_OF_PELVIC_URETHRA_OF_MALE = "basal cell layer of pelvic urethra of male"
    UROGENITAL_SINUS_RIDGE_OF_MALE = "urogenital sinus ridge of male"
    REST_OF_EPITHELIUM_OF_DORSAL_PELVIC_URETHRA_OF_MALE = "rest of epithelium of dorsal pelvic urethra of male"
    LAMINA_PROPRIA_OF_PELVIC_URETHRA_OF_MALE = "lamina propria of pelvic urethra of male"
    MUSCULARIS_MUCOSA_OF_PELVIC_URETHRA_OF_MALE = "muscularis mucosa of pelvic urethra of male"
    SUBMUCOSA_OF_PELVIC_URETHRA_OF_MALE = "submucosa of pelvic urethra of male"
    ANTERIOR_MESENCHYMAL_PAD_OF_MALE = "anterior mesenchymal pad of male"
    DORSAL_MESENCHYMAL_PAD_OF_MALE = "dorsal mesenchymal pad of male"
    VENTRAL_MESENCHYMAL_PAD_OF_MALE = "ventral mesenchymal pad of male"
    ADVENTITIA_OF_PELVIC_URETHRA_OF_MALE = "adventitia of pelvic urethra of male"
    VENTRAL_EPITHELIAL_BUD_OF_FEMALE = "ventral epithelial bud of female"
    URETHRAL_GLAND_OF_PELVIC_URETHRA_OF_FEMALE = "urethral gland of pelvic urethra of female"
    URETHRAL_GLAND_OF_CLITORAL_URETHRA = "urethral gland of clitoral urethra"
    VENTRAL_EPITHELIAL_BUD_OF_MALE = "ventral epithelial bud of male"
    EPITHELIUM_OF_DORSAL_PROSTATIC_URETHRA = "epithelium of dorsal prostatic urethra"
    VERUMONTANUM = "verumontanum"
    REST_OF_EPITHELIUM_OF_DORSAL_PROSTATIC_URETHRA = "rest of epithelium of dorsal prostatic urethra"
    EPITHELIUM_OF_VENTRAL_PROSTATIC_URETHRA = "epithelium of ventral prostatic urethra"
    ADVENTITIA_OF_PROSTATIC_URETHRA = "adventitia of prostatic urethra"
    PROSTATE_BUD = "prostate bud"
    ANTERIOR_PROSTATE_BUD = "anterior prostate bud"
    DORSAL_PROSTATE_BUD = "dorsal prostate bud"
    VENTRAL_PROSTATE_BUD = "ventral prostate bud"
    LATERAL_PROSTATE_BUD = "lateral prostate bud"
    EPITHELIUM_OF_ANTERIOR_PROSTATE_BUD = "epithelium of anterior prostate bud"
    PERI_PROSTATIC_MESENCHYME_OF_ANTERIOR_PROSTATE_BUD = "peri-prostatic mesenchyme of anterior prostate bud"
    EPITHELIUM_OF_DORSAL_PROSTATE_BUD = "epithelium of dorsal prostate bud"
    PERI_PROSTATIC_MESENCHYME_OF_DORSAL_PROSTATE_BUD = "peri-prostatic mesenchyme of dorsal prostate bud"
    EPITHELIUM_OF_VENTRAL_PROSTATE_BUD = "epithelium of ventral prostate bud"
    PERI_PROSTATIC_MESENCHYME_OF_VENTRAL_PROSTATE_BUD = "peri-prostatic mesenchyme of ventral prostate bud"
    EPITHELIUM_OF_LATERAL_PROSTATE_BUD = "epithelium of lateral prostate bud"
    PERI_PROSTATIC_MESENCHYME_OF_LATERAL_PROSTATE_BUD = "peri-prostatic mesenchyme of lateral prostate bud"
    SHOULDER_MUSCLE = "shoulder muscle"
    TERES_MUSCLE = "teres muscle"
    TERES_MINOR = "teres minor"
    PSOAS_MUSCLE = "psoas muscle"
    METACARPAL_BONE = "metacarpal bone"
    METATARSAL_BONE = "metatarsal bone"
    FOOT_MUSCLE = "foot muscle"
    PARS_ACROMIALIS_OF_DELTOID = "pars acromialis of deltoid"
    PARS_CLAVICULARIS_OF_DELTOID = "pars clavicularis of deltoid"
    PARS_SCAPULARIS_OF_DELTOID = "pars scapularis of deltoid"
    PROXIMAL_CARPAL_BONE = "proximal carpal bone"
    PISIFORM = "pisiform"
    SCAPHOLUNATE = "scapholunate"
    TRIQUETRUM = "triquetrum"
    DISTAL_CARPAL_BONE = "distal carpal bone"
    CAPITATE = "capitate"
    CENTRAL_CARPAL_BONE = "central carpal bone"
    FALCIFORM = "falciform"
    HAMATE = "hamate"
    TRAPEZIUM = "trapezium"
    TRAPEZOID = "trapezoid"
    CUBOID = "cuboid"
    MEDIAL_CUNEIFORM = "medial cuneiform"
    INTERMEDIATE_CUNEIFORM = "intermediate cuneiform"
    LATERAL_CUNEIFORM = "lateral cuneiform"
    TIBIALE = "tibiale"
    NAVICULAR = "navicular"
    ANCONEUS = "anconeus"
    CORACOBRACHIALIS = "coracobrachialis"
    TENSOR_FASCIAE_ANTEBRACHII = "tensor fasciae antebrachii"
    LATERAL_HEAD_OF_TRICEPS_BRACHII = "lateral head of triceps brachii"
    LONG_HEAD_OF_TRICEPS_BRACHII = "long head of triceps brachii"
    MEDIAL_HEAD_OF_TRICEPS_BRACHII = "medial head of triceps brachii"
    EXTENSOR_CARPI_RADIALIS = "extensor carpi radialis"
    EXTENSOR_CARPI_RADIALIS_BREVIS = "extensor carpi radialis brevis"
    EXTENSOR_CARPI_RADIALIS_LONGUS = "extensor carpi radialis longus"
    EXTENSOR_CARPI_ULNARIS = "extensor carpi ulnaris"
    EXTENSOR_DIGITORUM_COMMUNIS = "extensor digitorum communis"
    EXTENSOR_DIGITORUM_LATERALIS = "extensor digitorum lateralis"
    EXTENSOR_INDICIS_PROPRIUS = "extensor indicis proprius"
    EXTENSOR_POLLICIS = "extensor pollicis"
    EXTENSOR_POLLICIS_BREVIS = "extensor pollicis brevis"
    EXTENSOR_POLLICIS_LONGUS = "extensor pollicis longus"
    FLEXOR_CARPI_RADIALIS = "flexor carpi radialis"
    FLEXOR_CARPI_ULNARIS = "flexor carpi ulnaris"
    HUMERAL_HEAD_OF_FLEXOR_DIGITORUM_PROFUNDUS = "humeral head of flexor digitorum profundus"
    RADIAL_HEAD_OF_FLEXOR_DIGITORUM_PROFUNDUS = "radial head of flexor digitorum profundus"
    ULNAR_HEAD_OF_FLEXOR_DIGITORUM_PROFUNDUS = "ulnar head of flexor digitorum profundus"
    FLEXOR_DIGITORUM_SUPERFICIALIS = "flexor digitorum superficialis"
    FLEXOR_DIGITORUM_SUPERFICIALIS_1 = "flexor digitorum superficialis 1"
    FLEXOR_DIGITORUM_SUPERFICIALIS_2 = "flexor digitorum superficialis 2"
    PALMARIS_LONGUS = "palmaris longus"
    PRONATOR_QUADRATUS = "pronator quadratus"
    PRONATOR_TERES = "pronator teres"
    SUPINATOR = "supinator"
    ABDUCTOR_POLLICIS = "abductor pollicis"
    ABDUCTOR_POLLICIS_LONGUS = "abductor pollicis longus"
    ABDUCTOR_POLLICIS_BREVIS = "abductor pollicis brevis"
    ABDUCTOR_DIGITI_QUINTI_OF_HAND = "abductor digiti quinti of hand"
    ADDUCTOR_POLLICIS = "adductor pollicis"
    FLEXOR_DIGITI_PRIMI_BREVIS_OF_HAND = "flexor digiti primi brevis of hand"
    FLEXOR_DIGITI_QUINTI_BREVIS = "flexor digiti quinti brevis"
    INTEROSSEUS_MUSCLE_OF_HAND = "interosseus muscle of hand"
    LUMBRICAL_MUSCLE_OF_HAND = "lumbrical muscle of hand"
    OPPONENS_DIGITI_QUINTI = "opponens digiti quinti"
    ADDUCTOR_BREVIS = "adductor brevis"
    ADDUCTOR_LONGUS = "adductor longus"
    ADDUCTOR_MAGNUS = "adductor magnus"
    ACCESSORY_HEAD_OF_BICEPS_FEMORIS = "accessory head of biceps femoris"
    ANTERIOR_HEAD_OF_BICEPS_FEMORIS = "anterior head of biceps femoris"
    POSTERIOR_HEAD_OF_BICEPS_FEMORIS = "posterior head of biceps femoris"
    CAUDOFEMORALIS = "caudofemoralis"
    GEMELLUS_MUSCLE = "gemellus muscle"
    GEMELLUS_INFERIOR = "gemellus inferior"
    GEMELLUS_SUPERIOR = "gemellus superior"
    GLUTEUS_MINIMUS = "gluteus minimus"
    GRACILIS_MUSCLE = "gracilis muscle"
    GRACILIS_ANTICUS = "gracilis anticus"
    GRACILIS_POSTICUS = "gracilis posticus"
    OBTURATOR_MUSCLE = "obturator muscle"
    OBTURATOR_EXTERNUS = "obturator externus"
    OBTURATOR_INTERNUS = "obturator internus"
    PECTINEUS = "pectineus"
    PIRIFORMIS = "piriformis"
    POPLITEUS = "popliteus"
    QUADRATUS_FEMORIS = "quadratus femoris"
    SEMIMEMBRANOSUS = "semimembranosus"
    VASTUS_INTERMEDIUS = "vastus intermedius"
    VASTUS_LATERALIS = "vastus lateralis"
    VASTUS_MEDIALIS = "vastus medialis"
    EXTENSOR_HALLUCIS_LONGUS = "extensor hallucis longus"
    FLEXOR_DIGITORUM_BREVIS = "flexor digitorum brevis"
    FLEXOR_DIGITORUM_LONGUS = "flexor digitorum longus"
    LATERAL_HEAD_OF_GASTROCNEMIUS = "lateral head of gastrocnemius"
    MEDIAL_HEAD_OF_GASTROCNEMIUS = "medial head of gastrocnemius"
    PERONEUS_MUSCLE = "peroneus muscle"
    PERONEUS_BREVIS = "peroneus brevis"
    PERONEUS_LONGUS = "peroneus longus"
    PLANTARIS = "plantaris"
    TIBIALIS_POSTERIOR = "tibialis posterior"
    ABDUCTOR_DIGITI_QUINTI_OF_FOOT = "abductor digiti quinti of foot"
    ADDUCTOR_INDICIS = "adductor indicis"
    EXTENSOR_DIGITORUM_BREVIS = "extensor digitorum brevis"
    FLEXOR_DIGITI_PRIMI_BREVIS_OF_FOOT = "flexor digiti primi brevis of foot"
    PERONEUS_DIGITI_QUARTI = "peroneus digiti quarti"
    PERONEUS_DIGITI_QUINTI = "peroneus digiti quinti"
    INTEROSSEUS_MUSCLE_OF_FOOT = "interosseus muscle of foot"
    LUMBRICAL_MUSCLE_OF_FOOT = "lumbrical muscle of foot"
    QUADRATUS_PLANTAE = "quadratus plantae"
    TENSOR_APONEUROSIS_PLANTARIS = "tensor aponeurosis plantaris"
    RECTUS_FEMORIS = "rectus femoris"
    TRICEPS_SURAE_MUSCLE = "triceps surae muscle"
    MEDIAN_SULCUS = "median sulcus"
    SERRATUS_DORSALIS = "serratus dorsalis"
    MEDULLARY_VELUM = "medullary velum"
    VERTEBRAL_VEIN = "vertebral vein"
    FORAMEN_TRANSVERSARIUM = "foramen transversarium"
    UTERUS_LUMEN = "uterus lumen"
    OTIC_GANGLION = "otic ganglion"
    SPHENOPALATINE_GANGLION = "sphenopalatine ganglion"
    SUBMANDIBULAR_GANGLION = "submandibular ganglion"
    RIGHT_LUNG_ACCESSORY_LOBE_VASCULAR_ELEMENT = "right lung accessory lobe vascular element"
    RIGHT_LUNG_CAUDAL_LOBE_VASCULAR_ELEMENT = "right lung caudal lobe vascular element"
    RIGHT_LUNG_CRANIAL_LOBE_VASCULAR_ELEMENT = "right lung cranial lobe vascular element"
    RIGHT_LUNG_MIDDLE_LOBE_VASCULAR_ELEMENT = "right lung middle lobe vascular element"
    LUNG_LOBE = "lung lobe"
    TERMINAL_BRONCHIOLE = "terminal bronchiole"
    TERMINAL_BRONCHIOLE_EPITHELIUM = "terminal bronchiole epithelium"
    BRONCHIAL_VEIN = "bronchial vein"
    ARTERY_SMOOTH_MUSCLE = "artery smooth muscle"
    ARTERIOLE_ENDOTHELIUM = "arteriole endothelium"
    ARTERIOLE_SMOOTH_MUSCLE = "arteriole smooth muscle"
    VEIN_ENDOTHELIUM = "vein endothelium"
    VEIN_SMOOTH_MUSCLE = "vein smooth muscle"
    VENULE_ENDOTHELIUM = "venule endothelium"
    CAPILLARY_BED = "capillary bed"
    CAPILLARY_ENDOTHELIUM = "capillary endothelium"
    ARTERY_TUNICA_INTIMA = "artery tunica intima"
    ARTERY_TUNICA_MEDIA = "artery tunica media"
    ARTERY_TUNICA_ADVENTITIA = "artery tunica adventitia"
    BLOOD_VESSEL_LAYER = "blood vessel layer"
    TUNICA_INTIMA = "tunica intima"
    TUNICA_MEDIA = "tunica media"
    TUNICA_ADVENTITIA = "tunica adventitia"
    BLOOD_VESSEL_ELASTIC_TISSUE = "blood vessel elastic tissue"
    INTERNAL_ELASTIC_LAMINA = "internal elastic lamina"
    EXTERNAL_ELASTIC_LAMINA = "external elastic lamina"
    ARTERY_INTERNAL_ELASTIC_LAMINA = "artery internal elastic lamina"
    ARTERY_EXTERNAL_ELASTIC_LAMINA = "artery external elastic lamina"
    ARTERIOLE_TUNICA_INTIMA = "arteriole tunica intima"
    ARTERIOLE_TUNICA_MEDIA = "arteriole tunica media"
    ARTERIOLE_TUNICA_ADVENTITIA = "arteriole tunica adventitia"
    VEIN_TUNICA_INTIMA = "vein tunica intima"
    VEIN_TUNICA_MEDIA = "vein tunica media"
    VEIN_TUNICA_ADVENTITIA = "vein tunica adventitia"
    LYMPHATIC_VESSEL_TUNICA_INTIMA = "lymphatic vessel tunica intima"
    LYMPHATIC_VESSEL_TUNICA_MEDIA = "lymphatic vessel tunica media"
    LYMPHATIC_VESSEL_SMOOTH_MUSCLE = "lymphatic vessel smooth muscle"
    LYMPHATIC_VESSEL_TUNICA_ADVENTITIA = "lymphatic vessel tunica adventitia"
    SURFACE_ECTODERM_OF_GENITAL_TUBERCLE = "surface ectoderm of genital tubercle"
    PALATINE_VASCULATURE = "palatine vasculature"
    REICHERT_S_CARTILAGE_PRE_CARTILAGE_CONDENSATION = "Reichert's cartilage pre-cartilage condensation"
    REICHERT_S_CARTILAGE_CONDENSATION = "Reichert's cartilage condensation"
    REICHERT_S_CARTILAGE = "Reichert's cartilage"
    STYLOID_PROCESS_OF_TEMPORAL_BONE = "styloid process of temporal bone"
    HYOID_BODY_UPPER_PART = "hyoid body upper part"
    CHORDAE_TENDINAE = "chordae tendinae"
    HEPATIC_VENOUS_PLEXUS = "hepatic venous plexus"
    LINGUAL_ARTERY = "lingual artery"
    THYMIC_VEIN = "thymic vein"
    LIVER_LOBAR_INTERZONE = "liver lobar interzone"
    VENOUS_PLEXUS = "venous plexus"
    VERTEBRAL_VENOUS_PLEXUS = "vertebral venous plexus"
    EXTERNAL_VERTEBRAL_VENOUS_PLEXUS = "external vertebral venous plexus"
    INTERNAL_VERTEBRAL_VENOUS_PLEXUS = "internal vertebral venous plexus"
    TAILGUT = "tailgut"
    BLADDER_URETHRA_TRANSITION_ZONE = "bladder-urethra transition zone"
    EPITHELIUM_OF_BLADDER_URETHRA_TRANSITION_ZONE = "epithelium of bladder-urethra transition zone"
    MESENCHYME_OF_BLADDER_URETHRA_TRANSITION_ZONE = "mesenchyme of bladder-urethra transition zone"
    REMNANT_OF_MESONEPHRIC_TUBULE_OF_MALE = "remnant of mesonephric tubule of male"
    FORELIMB_INTERPHALANGEAL_JOINT = "forelimb interphalangeal joint"
    HINDLIMB_INTERPHALANGEAL_JOINT = "hindlimb interphalangeal joint"
    METATARSAL_PHALANGEAL_JOINT = "metatarsal-phalangeal joint"
    BRONCHIOLE_SMOOTH_MUSCLE = "bronchiole smooth muscle"
    HAIR_FOLLICLE_OF_MALE_PREPUCE = "hair follicle of male prepuce"
    LAMINA_PROPRIA_OF_MALE_PREPUCE = "lamina propria of male prepuce"
    PREPUTIAL_SPACE_OF_MALE = "preputial space of male"
    NERVE_OF_UROTHELIUM_OF_BLADDER = "nerve of urothelium of bladder"
    NERVE_OF_LAMINA_PROPRIA_OF_BLADDER = "nerve of lamina propria of bladder"
    NERVE_OF_DETRUSOR_MUSCLE_OF_BLADDER = "nerve of detrusor muscle of bladder"
    NERVE_OF_OUTER_LAYER_OF_BLADDER = "nerve of outer layer of bladder"
    NERVE_OF_UROTHELIUM_OF_BLADDER_FUNDUS = "nerve of urothelium of bladder fundus"
    NERVE_OF_UROTHELIUM_OF_BLADDER_NECK = "nerve of urothelium of bladder neck"
    NERVE_OF_UROTHELIUM_OF_BLADDER_TRIGONE = "nerve of urothelium of bladder trigone"
    NERVE_OF_UROTHELIUM_OF_VENTRAL_BLADDER_NECK = "nerve of urothelium of ventral bladder neck"
    NERVE_OF_LAMINA_PROPRIA_OF_BLADDER_FUNDUS = "nerve of lamina propria of bladder fundus"
    NERVE_OF_LAMINA_PROPRIA_OF_BLADDER_NECK = "nerve of lamina propria of bladder neck"
    NERVE_OF_BLOOD_VESSEL_OF_LAMINA_PROPRIA_OF_BLADDER_FUNDUS = (
        "nerve of blood vessel of lamina propria of bladder fundus"
    )
    NERVE_OF_BLOOD_VESSEL_OF_LAMINA_PROPRIA_OF_BLADDER_NECK = "nerve of blood vessel of lamina propria of bladder neck"
    NERVE_OF_SUBUROTHELIAL_LAMINA_PROPRIA_OF_BLADDER_FUNDUS = "nerve of suburothelial lamina propria of bladder fundus"
    NERVE_OF_OUTER_LAMINA_PROPRIA_OF_BLADDER_FUNDUS = "nerve of outer lamina propria of bladder fundus"
    NERVE_OF_LAMINA_PROPRIA_OF_BLADDER_TRIGONE = "nerve of lamina propria of bladder trigone"
    NERVE_OF_LAMINA_PROPRIA_OF_VENTRAL_BLADDER_NECK = "nerve of lamina propria of ventral bladder neck"
    NERVE_OF_BLOOD_VESSEL_OF_SUBUROTHELIAL_LAMINA_PROPRIA_OF_BLADDER_FUNDUS = (
        "nerve of blood vessel of suburothelial lamina propria of bladder fundus"
    )
    NERVE_OF_BLOOD_VESSEL_OF_OUTER_LAMINA_PROPRIA_OF_BLADDER_FUNDUS = (
        "nerve of blood vessel of outer lamina propria of bladder fundus"
    )
    NERVE_OF_BLOOD_VESSEL_OF_LAMINA_PROPRIA_OF_TRIGONE = "nerve of blood vessel of lamina propria of trigone"
    NERVE_OF_BLOOD_VESSEL_OF_LAMINA_PROPRIA_OF_VENTRAL_BLADDER_NECK = (
        "nerve of blood vessel of lamina propria of ventral bladder neck"
    )
    NERVE_OF_NONVASCULAR_TISSUE_OF_DETRUSOR_OF_BLADDER = "nerve of nonvascular tissue of detrusor of bladder"
    NERVE_OF_BLOOD_VESSEL_OF_DETRUSOR_OF_BLADDER = "nerve of blood vessel of detrusor of bladder"
    NERVE_OF_DETRUSOR_MUSCLE_OF_BLADDER_FUNDUS = "nerve of detrusor muscle of bladder fundus"
    NERVE_OF_DETRUSOR_MUSCLE_OF_BLADDER_NECK = "nerve of detrusor muscle of bladder neck"
    NERVE_OF_DETRUSOR_MUSCLE_OF_BLADDER_TRIGONE = "nerve of detrusor muscle of bladder trigone"
    NERVE_OF_DETRUSOR_MUSCLE_OF_VENTRAL_BLADDER_NECK = "nerve of detrusor muscle of ventral bladder neck"
    NERVE_OF_BLOOD_VESSEL_OF_DETRUSOR_OF_BLADDER_FUNDUS = "nerve of blood vessel of detrusor of bladder fundus"
    NERVE_OF_BLOOD_VESSEL_OF_DETRUSOR_OF_BLADDER_NECK = "nerve of blood vessel of detrusor of bladder neck"
    NERVE_OF_BLOOD_VESSEL_OF_DETRUSOR_OF_TRIGONE = "nerve of blood vessel of detrusor of trigone"
    NERVE_OF_BLOOD_VESSEL_OF_DETRUSOR_OF_VENTRAL_BLADDER_NECK = (
        "nerve of blood vessel of detrusor of ventral bladder neck"
    )
    NERVE_OF_NONVASCULAR_TISSUE_OF_OUTER_LAYER_OF_BLADDER = "nerve of nonvascular tissue of outer layer of bladder"
    NERVE_OF_BLOOD_VESSEL_OF_OUTER_LAYER_OF_BLADDER = "nerve of blood vessel of outer layer of bladder"
    NERVE_OF_SEROSA_OF_BLADDER_FUNDUS = "nerve of serosa of bladder fundus"
    NERVE_OF_ADVENTITIA_OF_BLADDER_NECK = "nerve of adventitia of bladder neck"
    NERVE_OF_ADVENTITIA_OF_BLADDER_TRIGONE = "nerve of adventitia of bladder trigone"
    NERVE_OF_ADVENTITIA_OF_VENTRAL_BLADDER_NECK = "nerve of adventitia of ventral bladder neck"
    NERVE_OF_BLOOD_VESSEL_OF_SEROSA_OF_BLADDER_FUNDUS = "nerve of blood vessel of serosa of bladder fundus"
    NERVE_OF_BLOOD_VESSEL_OF_ADVENTITIA_OF_BLADDER_NECK = "nerve of blood vessel of adventitia of bladder neck"
    NERVE_OF_BLOOD_VESSEL_OF_ADVENTITIA_OF_TRIGONE = "nerve of blood vessel of adventitia of trigone"
    NERVE_OF_BLOOD_VESSEL_OF_ADVENTITIA_OF_VENTRAL_BLADDER_NECK = (
        "nerve of blood vessel of adventitia of ventral bladder neck"
    )
    DUODENUM_MESENCHYME = "duodenum mesenchyme"
    GENITAL_TUBERCLE_MESENCHYME = "genital tubercle mesenchyme"
    SMALL_INTESTINE_MESENCHYME = "small intestine mesenchyme"
    GENITAL_TUBERCLE_EPITHELIUM = "genital tubercle epithelium"
    DIGIT_DERMIS = "digit dermis"
    DIGIT_EPIDERMIS = "digit epidermis"
    TAIL_DERMATOME = "tail dermatome"
    STRIATED_MUSCLE_TISSUE = "striated muscle tissue"
    SIGMOID_COLON = "sigmoid colon"
    BACK_SKIN = "back skin"
    NERVE_OF_BLADDER_FUNDUS = "nerve of bladder fundus"
    NERVE_OF_BLADDER_NECK = "nerve of bladder neck"
    NERVE_OF_BLADDER_TRIGONE = "nerve of bladder trigone"
    NERVE_OF_VENTRAL_BLADDER_NECK = "nerve of ventral bladder neck"
    GLANDULAR_RIDGE = "glandular ridge"
    MALE_UROGENITAL_MATING_PROTUBERANCE = "male urogenital mating protuberance"
    BONE_OF_OS_PENIS = "bone of os penis"
    HYALINE_CARTILAGE_OF_OS_PENIS = "hyaline cartilage of os penis"
    MALE_UROGENITAL_MATING_PROTUBERANCE_CARTILAGE = "male urogenital mating protuberance cartilage"
    MALE_UROGENITAL_MATING_PROTUBERANCE_CORPUS_CAVERNOSUM = "male urogenital mating protuberance corpus cavernosum"
    MALE_UROGENITAL_MATING_PROTUBERANCE_SURFACE_EPITHELIUM = "male urogenital mating protuberance surface epithelium"
    MALE_UROGENITAL_MATING_PROTUBERANCE_RIDGE_GROOVE = "male urogenital mating protuberance ridge groove"
    MALE_UROGENITAL_MATING_PROTUBERANCE_SKIN = "male urogenital mating protuberance skin"
    CORPUS_CAVERNOSUM_GLANDIS = "corpus cavernosum glandis"
    CORPUS_CAVERNOSUM_URETHRA = "corpus cavernosum urethra"
    FEMALE_PREPUCE_HAIR_FOLLICLE = "female prepuce hair follicle"
    FEMALE_PREPUCE_LAMINA_PROPRIA = "female prepuce lamina propria"
    SCROTUM_SKIN = "scrotum skin"
    LAMINA_PROPRIA_OF_SCROTUM = "lamina propria of scrotum"
    DEVELOPING_VASCULATURE_OF_SCROTAL_SWELLING = "developing vasculature of scrotal swelling"
    SEPTUM_OF_SCROTAL_FOLD = "septum of scrotal fold"
    DEVELOPING_VASCULATURE_OF_SCROTAL_FOLD = "developing vasculature of scrotal fold"
    DEVELOPING_VASCULATURE_OF_LABIAL_FOLD = "developing vasculature of labial fold"
    DEVELOPING_VASCULATURE_OF_LABIAL_SWELLING = "developing vasculature of labial swelling"
    LAMINA_PROPRIA_OF_SEMINAL_VESICLE = "lamina propria of seminal vesicle"
    ADVENTITIA_OF_EJACULATORY_DUCT = "adventitia of ejaculatory duct"
    LAMINA_PROPRIA_OF_DUCTUS_DEFERENS = "lamina propria of ductus deferens"
    URETHRAL_SEAM = "urethral seam"
    MIGRATING_NEURAL_CREST = "migrating neural crest"
    URETHRAL_GLAND_OF_PELVIC_URETHRA_OF_MALE = "urethral gland of pelvic urethra of male"
    URETHRAL_GLAND_OF_PENILE_URETHRA = "urethral gland of penile urethra"
    MALE_URETHRAL_GLAND_BUD = "male urethral gland bud"
    ANTERIOR_MALE_URETHRAL_GLAND_BUD = "anterior male urethral gland bud"
    POSTERIOR_MALE_URETHRAL_GLAND_BUD = "posterior male urethral gland bud"
    FEMALE_ACCESSORY_GLAND = "female accessory gland"
    BULBOURETHRAL_GLAND_OF_FEMALE = "bulbourethral gland of female"
    EPITHELIUM_OF_BULBOURETHRAL_GLAND_OF_FEMALE = "epithelium of bulbourethral gland of female"
    LAMINA_PROPRIA_OF_BULBOURETHRAL_GLAND_OF_FEMALE = "lamina propria of bulbourethral gland of female"
    VASCULATURE_OF_BULBOURETHRAL_GLAND_OF_FEMALE = "vasculature of bulbourethral gland of female"
    PHALLIC_URETHRA_OF_FEMALE = "phallic urethra of female"
    URETHRAL_MEATUS_OF_PENILE_URETHRA = "urethral meatus of penile urethra"
    UROGENITAL_MEMBRANE_OF_FEMALE = "urogenital membrane of female"
    UROGENITAL_MEMBRANE_OF_MALE = "urogenital membrane of male"
    MESENCHYME_OF_METANEPHROS = "mesenchyme of metanephros"
    HEART_MESENCHYME = "heart mesenchyme"
    PREDENTIN = "predentin"
    TOOTH_BASEMENT_MEMBRANE = "tooth basement membrane"
    MESENCHYME_OF_GLANS_OF_FEMALE_GENITAL_TUBERCLE = "mesenchyme of glans of female genital tubercle"
    URETHRAL_PLATE_EPITHELIUM_OF_UROGENITAL_MEMBRANE_OF_FEMALE = (
        "urethral plate epithelium of urogenital membrane of female"
    )
    SKIN_OF_UROGENITAL_MEMBRANE_OF_FEMALE = "skin of urogenital membrane of female"
    URETHRAL_PLATE_EPITHELIUM_OF_UROGENITAL_MEMBRANE_OF_MALE = (
        "urethral plate epithelium of urogenital membrane of male"
    )
    SKIN_OF_UROGENITAL_MEMBRANE_OF_MALE = "skin of urogenital membrane of male"
    VISUAL_CORTEX = "visual cortex"
    PRELIMBIC_AREA = "prelimbic area"
    CA3_PYRAMIDAL_CELL_LAYER = "CA3 pyramidal cell layer"
    FOREBRAIN_MIDBRAIN_BOUNDARY_REGION = "forebrain-midbrain boundary region"
    SUBCOMMISSURAL_ORGAN_PRIMORDIUM = "subcommissural organ primordium"
    URETHRAL_TUBE_EPITHELIUM_OF_CLITORAL_URETHRA = "urethral tube epithelium of clitoral urethra"
    PROXIMAL_URETHRAL_MEATUS_OF_CLITORAL_URETHRA = "proximal urethral meatus of clitoral urethra"
    URETHRAL_MEATUS_OF_CLITORAL_URETHRA = "urethral meatus of clitoral urethra"
    URETHRAL_TUBE_EPITHELIUM_OF_PENILE_URETHRA = "urethral tube epithelium of penile urethra"
    GLAND_OF_VAGINAL_EPITHELIUM = "gland of vaginal epithelium"
    REST_OF_EPITHELIUM_OF_VAGINA = "rest of epithelium of vagina"
    HINDLIMB_NAIL = "hindlimb nail"
    MYOMETRIUM_INNER_CIRCULAR_SMOOTH_MUSCLE_LAYER = "myometrium inner circular smooth muscle layer"
    MYOMETRIUM_OUTER_LONGITUDINAL_SMOOTH_MUSCLE_LAYER = "myometrium outer longitudinal smooth muscle layer"
    GREAT_VESSEL_OF_HEART = "great vessel of heart"
    AORTIC_SAC = "aortic sac"
    OUTFLOW_TRACT_DISTAL_PART = "outflow tract distal part"
    ANTERIOR_AMNIOTIC_FOLD = "anterior amniotic fold"
    POSTERIOR_AMNIOTIC_FOLD = "posterior amniotic fold"
    PERIOPTIC_VASCULAR_PLEXUS = "perioptic vascular plexus"
    CEREBRAL_VENOUS_PLEXUS = "cerebral venous plexus"
    ANTERIOR_CEREBRAL_VEIN = "anterior cerebral vein"
    PARACARDIAC_GANGLION = "paracardiac ganglion"
    DENTATE_GYRUS_PRIMORDIUM = "dentate gyrus primordium"
    PROSPECTIVE_HIPPOCAMPAL_FISSURE = "prospective hippocampal fissure"
    HIPPOCAMPAL_FISSURE = "hippocampal fissure"
    MACULA_OF_VESTIBULAR_COMPONENT = "macula of vestibular component"
    EARLY_CONCEPTUS = "early conceptus"
    OVARY_FOLLICLE_CELL = "ovary follicle cell"
    MESENCHYME_OF_PREPUTIAL_SWELLING = "mesenchyme of preputial swelling"
    SKIN_OF_PREPUTIAL_SWELLING = "skin of preputial swelling"
    VIBRISSA_INNER_ROOT_SHEATH = "vibrissa inner root sheath"
    VIBRISSA_OUTER_ROOT_SHEATH = "vibrissa outer root sheath"
    HAIR_GERM = "hair germ"
    CA2_PYRAMIDAL_CELL_LAYER = "CA2 pyramidal cell layer"
    HAIR_EPIDERMAL_PLACODE = "hair epidermal placode"
    VIBRISSA_ROOT_SHEATH_MATRIX = "vibrissa root sheath matrix"
    VIBRISSA_DERMAL_PAPILLA = "vibrissa dermal papilla"
    HAIR_FOLLICLE_DERMAL_PAPILLA = "hair follicle dermal papilla"
    PELAGE_HAIR = "pelage hair"
    BULBUS_CORDIS_ENDOCARDIAL_TUBE = "bulbus cordis endocardial tube"
    BULBUS_CORDIS_ENDOCARDIAL_LINING = "bulbus cordis endocardial lining"
    BULBUS_CORDIS_CARDIAC_MUSCLE = "bulbus cordis cardiac muscle"
    BULBUS_CORDIS_CARDIAC_JELLY = "bulbus cordis cardiac jelly"
    COSTO_CHONDRAL_JOINT = "costo-chondral joint"
    VENTRAL_ECTODERMAL_RIDGE = "ventral ectodermal ridge"
    INTERSOMITIC_ARTERY = "intersomitic artery"
    UMBILICAL_BLOOD_VESSEL = "umbilical blood vessel"
    VITELLINE_BLOOD_VESSEL = "vitelline blood vessel"
    PERIRENAL_FAT = "perirenal fat"
    LARGE_INTESTINE_MESENCHYME = "large intestine mesenchyme"
    THORACIC_CAVITY = "thoracic cavity"
    HAIR_SHAFT = "hair shaft"
    ABDOMINAL_FAT_PAD = "abdominal fat pad"
    LEG_BLOOD_VESSEL = "leg blood vessel"
    LIMB_JOINT = "limb joint"
    TAIL_SKIN = "tail skin"
    ARTERIAL_SYSTEM_SMOOTH_MUSCLE = "arterial system smooth muscle"
    THYROID_GLAND_FOLLICLE = "thyroid gland follicle"
    ABDOMINAL_CAVITY = "abdominal cavity"
    PEYER_S_PATCH_EPITHELIUM = "Peyer's patch epithelium"
    ABDOMINAL_MAMMARY_GLAND = "abdominal mammary gland"
    NIPPLE = "nipple"
    DERMIS_CONNECTIVE_TISSUE = "dermis connective tissue"
    SUBSTANTIA_GELATINOSA = "substantia gelatinosa"
    COMMON_PERONEAL_NERVE = "common peroneal nerve"
    TIBIAL_NERVE = "tibial nerve"
    AURICULAR_NERVE = "auricular nerve"
    HARDERIAN_GLAND_EPITHELIUM = "Harderian gland epithelium"
    LACRIMAL_GLAND_EPITHELIUM = "lacrimal gland epithelium"
    MACULA_OF_RETINA = "macula of retina"
    RETINA_LIMITING_LAMINA = "retina limiting lamina"
    RETINA_EXTERNAL_LIMITING_LAMINA = "retina external limiting lamina"
    RETINA_INTERNAL_LIMITING_LAMINA = "retina internal limiting lamina"
    RETINA_BLOOD_VESSEL = "retina blood vessel"
    SQUAMOSAL_BONE = "squamosal bone"
    BRUNNER_S_GLAND = "Brunner's gland"
    ESOPHAGUS_SKELETAL_MUSCLE = "esophagus skeletal muscle"
    RENAL_TUBULE_EPITHELIUM = "renal tubule epithelium"
    EPIDIDYMAL_DUCT = "epididymal duct"
    XIPHOID_CARTILAGE = "xiphoid cartilage"
    AUCHENE_HAIR = "auchene hair"
    GUARD_HAIR = "guard hair"
    PANCREATICO_DUODENAL_ARTERY = "pancreatico-duodenal artery"
    TIBIAL_ARTERY = "tibial artery"
    LEFT_CORONARY_ARTERY = "left coronary artery"
    RIGHT_CORONARY_ARTERY = "right coronary artery"
    SEPTAL_CORONARY_ARTERY = "septal coronary artery"
    BILE_CANALICULUS = "bile canaliculus"
    EXOCRINE_GLAND_FLUID_OR_SECRETION = "exocrine gland fluid or secretion"
    SALIVA = "saliva"
    TEAR = "tear"
    MAMMARY_GLAND_FLUID_OR_SECRETION = "mammary gland fluid or secretion"
    COLOSTROM = "colostrom"
    MAMMARY_GLAND_MILK = "mammary gland milk"
    MALE_REPRODUCTIVE_SYSTEM_FLUID_OR_SECRETION = "male reproductive system fluid or secretion"
    SEMEN = "semen"
    SEMINAL_FLUID = "seminal fluid"
    BULBOURETHRAL_GLAND_SECRETION = "bulbourethral gland secretion"
    PROSTATE_GLAND_SECRETION = "prostate gland secretion"
    SEMINAL_VESICLE_GLAND_SECRETION = "seminal vesicle gland secretion"
    OTOLYMPH = "otolymph"
    ENDOLYMPH = "endolymph"
    PERILYMPH = "perilymph"
    RESPIRATORY_SYSTEM_FLUID_OR_SECRETION = "respiratory system fluid or secretion"
    LUNG_SURFACTANT = "lung surfactant"
    NASAL_MUCUS = "nasal mucus"
    SPUTUM = "sputum"
    URINE = "urine"
    INGUINAL_FAT_PAD = "inguinal fat pad"
    CORPORA_QUADRIGEMINA = "corpora quadrigemina"
    PRE_BOTZINGER_COMPLEX = "pre-botzinger complex"
    PERITUBULAR_CAPILLARY = "peritubular capillary"
    INNER_RENAL_MEDULLA_VASA_RECTA = "inner renal medulla vasa recta"
    OUTER_RENAL_MEDULLA_VASA_RECTA = "outer renal medulla vasa recta"
    MESANGIUM = "mesangium"
    GLOMERULAR_MESANGIUM = "glomerular mesangium"
    PAROTID_DUCT = "parotid duct"
    SUBLINGUAL_DUCT = "sublingual duct"
    SPINAL_CORD_INTERMEDIOLATERAL_NUCLEUS = "spinal cord intermediolateral nucleus"
    MAMMARY_GLAND_ALVEOLUS = "mammary gland alveolus"
    ALISPHENOID_BONE = "alisphenoid bone"
    PROSTATE_DUCT = "prostate duct"
    COCHLEAR_LABYRINTH = "cochlear labyrinth"
    ZYGOMATIC_ARCH = "zygomatic arch"
    TEMPORAL_BONE_ROUND_WINDOW = "temporal bone round window"
    MANDIBLE_CONDYLAR_PROCESS = "mandible condylar process"
    TRABECULAE_CARNEAE_CARDIAC_MUSCLE = "trabeculae carneae cardiac muscle"
    TRACHEA_GLAND = "trachea gland"
    DORSAL_STRIATUM = "dorsal striatum"
    VENTRAL_STRIATUM = "ventral striatum"
    EXORBITAL_LACRIMAL_GLAND = "exorbital lacrimal gland"
    INTRAORBITAL_LACRIMAL_GLAND = "intraorbital lacrimal gland"
    CARPAL_TARSAL_BONE = "carpal/tarsal bone"
    BONY_LABYRINTH = "bony labyrinth"
    CARTILAGINOUS_JOINT = "cartilaginous joint"
    PILOSEBACEOUS_GLAND = "pilosebaceous gland"
    TRUNK_BONE = "trunk bone"
    ACCESSORY_OLFACTORY_BULB_ANLAGE = "accessory olfactory bulb anlage"
    ACCESSORY_OLFACTORY_BULB_ANLAGE_MITRAL_CELL_LAYER = "accessory olfactory bulb anlage mitral cell layer"
    MAIN_OLFACTORY_BULB_VENTRICULAR_LAYER = "main olfactory bulb ventricular layer"
    MAIN_OLFACTORY_BULB_PROSPECTIVE_GLOMERULAR_LAYER = "main olfactory bulb prospective glomerular layer"
    VERTEBRAL_BODY = "vertebral body"
    LEFT_KIDNEY = "left kidney"
    RIGHT_KIDNEY = "right kidney"
    ALIMENTARY_SYSTEM_MUCOSA = "alimentary system mucosa"
    ALIMENTARY_SYSTEM_LAMINA_PROPRIA = "alimentary system lamina propria"
    STOMACH_LAMINA_PROPRIA = "stomach lamina propria"
    EYE_ANTERIOR_SEGMENT = "eye anterior segment"
    EYE_POSTERIOR_SEGMENT = "eye posterior segment"
    SENSORY_GANGLION = "sensory ganglion"
    BARORECEPTOR = "baroreceptor"
    LIMB_BLOOD_VESSEL = "limb blood vessel"
    MAIN_OLFACTORY_BULB_PROSPECTIVE_GRANULE_CELL_LAYER = "main olfactory bulb prospective granule cell layer"
    CORTICOSPINAL_TRACT = "corticospinal tract"
    CEREBELLUM_PURKINJE_CELL_CLUSTER = "cerebellum Purkinje cell cluster"
    HYPOTHALAMUS_SUBVENTRICULAR_ZONE = "hypothalamus subventricular zone"
    THALAMUS_SUBVENTRICULAR_ZONE = "thalamus subventricular zone"
    ZONA_LIMITANS_INTRATHALAMICA = "zona limitans intrathalamica"
    NERVE_FIBER_TRACT = "nerve fiber tract"
    FASCICULUS_RETROFLEXUS = "fasciculus retroflexus"
    STRIA_MEDULLARIS = "stria medullaris"
    CHOROID_PLEXUS_EPITHELIUM = "choroid plexus epithelium"
    CHOROID_PLEXUS_STROMA = "choroid plexus stroma"
    HEAD_BLOOD_VESSEL = "head blood vessel"
    CORNEA_BLOOD_VESSEL = "cornea blood vessel"
    SPIRAL_ARTERY = "spiral artery"
    VENOUS_SYSTEM_SMOOTH_MUSCLE = "venous system smooth muscle"
    VERTEBRAL_NEURAL_ARCH = "vertebral neural arch"
    INTRAMEMBRANOUS_BONE = "intramembranous bone"
    PRESPHENOID_BONE = "presphenoid bone"
    PTERYGOID_BONE = "pterygoid bone"
    SUPRAOCCIPITAL_BONE = "supraoccipital bone"
    PALATINE_BONE = "palatine bone"
    SESAMOID_BONE = "sesamoid bone"
    ULNAR_SESAMOID_BONE = "ulnar sesamoid bone"
    NUCLEUS_PREPOSITUS = "nucleus prepositus"
    LOWER_BACK = "lower back"
    LOWER_BACK_NERVE = "lower back nerve"
    SAPHENOUS_NERVE = "saphenous nerve"
    GENIOHYOID = "geniohyoid"
    LEVATOR_AURIS_LONGUS = "levator auris longus"
    PERINEUM = "perineum"
    PERINEAL_MUSCLE = "perineal muscle"
    SMOOTH_MUSCLE_OF_URETHRA = "smooth muscle of urethra"
    PUDENDUM_OF_FEMALE = "pudendum of female"
    CORPUS_CAVERNOSUM_OF_PENIS = "corpus cavernosum of penis"
    THORACIC_MAMMARY_GLAND = "thoracic mammary gland"
    MAMMARY_GLAND_LOBULE = "mammary gland lobule"
    LEFT_LUNG_RESPIRATORY_BRONCHIOLE = "left lung respiratory bronchiole"
    LEFT_LUNG_TERMINAL_BRONCHIOLE = "left lung terminal bronchiole"
    RIGHT_LUNG_RESPIRATORY_BRONCHIOLE = "right lung respiratory bronchiole"
    RIGHT_LUNG_TERMINAL_BRONCHIOLE = "right lung terminal bronchiole"
    AWL_HAIR = "awl hair"
    ZIGZAG_HAIR = "zigzag hair"
    RENAL_LYMPHATIC = "renal lymphatic"
    PONTINE_RETICULAR_NUCLEUS = "pontine reticular nucleus"
    LATERAL_VESTIBULAR_NUCLEUS = "lateral vestibular nucleus"
    CARDIAC_GANGLION = "cardiac ganglion"
    CARDIAC_FIBROUS_SKELETON = "cardiac fibrous skeleton"
    CARDIAC_INTERSTITIUM = "cardiac interstitium"
    PROSOMERE_1 = "prosomere 1"
    PROSOMERE_2 = "prosomere 2"
    PROSOMERE_3 = "prosomere 3"
    SPLEEN_MARGINAL_ZONE = "spleen marginal zone"
    EYELID_MUSCLE = "eyelid muscle"
    LEVATOR_PALPEBRAE_SUPERIORIS = "levator palpebrae superioris"
    PRIMORDIAL_GERM_CELL_OF_EXTRAEMBRYONIC_MESODERM = "primordial germ cell of extraembryonic mesoderm"
    VENTRAL_LATERAL_THALAMIC_NUCLEUS = "ventral lateral thalamic nucleus"
    PREOPTIC_AREA = "preoptic area"
    LATERAL_PREOPTIC_NUCLEUS = "lateral preoptic nucleus"
    FACE_SKIN = "face skin"
    HEAD_SKIN = "head skin"
    BASAL_MAGNOCELLULAR_COMPLEX = "basal magnocellular complex"
    ANTERIOR_ENTOPEDUNCULAR_AREA = "anterior entopeduncular area"
    VOMERONASAL_BRANCH_OF_OLFACTORY_I_NERVE = "vomeronasal branch of olfactory I nerve"
    VOMERONASAL_ORGAN_SENSORY_EPITHELIUM = "vomeronasal organ sensory epithelium"
    VOMERONASAL_ORGAN_NON_SENSORY_EPITHELIUM = "vomeronasal organ non-sensory epithelium"
    VOMERONASAL_ORGAN_VASCULAR_PUMP_ANLAGE = "vomeronasal organ vascular pump anlage"
    VOMERONASAL_ORGAN_VASCULAR_PUMP = "vomeronasal organ vascular pump"
    VOMERONASAL_ORGAN_GLANDULAR_TISSUE = "vomeronasal organ glandular tissue"
    RIB_CAGE = "rib cage"
    GONADAL_FAT_PAD = "gonadal fat pad"
    MESENTERIC_ADIPOSE_TISSUE = "mesenteric adipose tissue"
    LOWER_URINARY_TRACT = "lower urinary tract"
    FOLLICLE_CELL_OF_ANTRAL_FOLLICLE = "follicle cell of antral follicle"
    RHOMBIC_LIP = "rhombic lip"
    UPPER_RHOMBIC_LIP = "upper rhombic lip"
    LOWER_RHOMBIC_LIP = "lower rhombic lip"
    PROSOMERE = "prosomere"
    CERVICAL_DORSAL_ROOT_GANGLION = "cervical dorsal root ganglion"
    THORACIC_DORSAL_ROOT_GANGLION = "thoracic dorsal root ganglion"
    LUMBAR_DORSAL_ROOT_GANGLION = "lumbar dorsal root ganglion"
    SACRAL_DORSAL_ROOT_GANGLION = "sacral dorsal root ganglion"
    PROAMNIOTIC_CANAL = "proamniotic canal"
    ACROMIO_CLAVICULAR_JOINT = "acromio-clavicular joint"
    STERNO_COSTAL_JOINT = "sterno-costal joint"
    MESENCEPHALIC_VENOUS_PLEXUS = "mesencephalic venous plexus"
    PTERYGOID_VENOUS_PLEXUS = "pterygoid venous plexus"
    BRACHIAL_ARTERY = "brachial artery"
    INFERIOR_ALVEOLAR_ARTERY = "inferior alveolar artery"
    LATERAL_CAUDAL_ARTERY = "lateral caudal artery"
    SUPERIOR_VESICAL_ARTERY = "superior vesical artery"
    ADRENAL_VEIN = "adrenal vein"
    OCCIPITAL_VEIN = "occipital vein"
    SUPERIOR_ANASTOMOTIC_VEIN = "superior anastomotic vein"
    MULTIFIDUS = "multifidus"
    PHARYNGEAL_RAPHE = "pharyngeal raphe"
    QUADRICEPS_TENDON = "quadriceps tendon"
    INTESTINE_LAMINA_PROPRIA = "intestine lamina propria"
    LARGE_INTESTINE_LAMINA_PROPRIA = "large intestine lamina propria"
    OLFACTORY_MIGRATORY_MASS = "olfactory migratory mass"
    NERVE_TO_SOLEUS = "nerve to soleus"
    PRIMORDIAL_GERM_CELL = "primordial germ cell"
    SYNOVIAL_JOINT = "synovial joint"
    PRIMORDIAL_GERM_CELL_OF_BASE_OF_ALLANTOIS = "primordial germ cell of base of allantois"
    PRIMITIVE_HINDGUT = "primitive hindgut"
    PRIMORDIAL_GERM_CELL_OF_DORSAL_MESENTERY = "primordial germ cell of dorsal mesentery"
    MYSTACIAL_PAD = "mystacial pad"
    HEAD_SURFACE_ECTODERM = "head surface ectoderm"
    TRUNK_SURFACE_ECTODERM = "trunk surface ectoderm"
    HEART_VALVE_PRIMORDIUM = "heart valve primordium"
    ATRIOVENTRICULAR_VALVE_PRIMORDIUM = "atrioventricular valve primordium"
    SEMILUNAR_VALVE_PRIMORDIUM = "semilunar valve primordium"
    OUTFLOW_TRACT_ENDOCARDIAL_CUSHION = "outflow tract endocardial cushion"
    PRIMITIVE_MENINGES = "primitive meninges"
    LEPTOMENINGES = "leptomeninges"
    METENCEPHALON_MARGINAL_LAYER = "metencephalon marginal layer"
    MYELENCEPHALON_MARGINAL_LAYER = "myelencephalon marginal layer"
    METENCEPHALON_MANTLE_LAYER = "metencephalon mantle layer"
    TELENCEPHALON_SUBVENTRICULAR_ZONE = "telencephalon subventricular zone"
    DIENCEPHALON_SUBVENTRICULAR_ZONE = "diencephalon subventricular zone"
    SPINAL_NERVE_DORSAL_ROOT = "spinal nerve dorsal root"
    SPINAL_CORD_MIXED_SPINAL_NERVE = "spinal cord mixed spinal nerve"
    SPINAL_NERVE_VENTRAL_ROOT = "spinal nerve ventral root"
    TRABECULAR_MYOCARDIUM = "trabecular myocardium"
    COMPACT_MYOCARDIUM = "compact myocardium"
    HEART_PAPILLARY_MUSCLE = "heart papillary muscle"
    ATRIOVENTRICULAR_CANAL_ENDOCARDIAL_LINING = "atrioventricular canal endocardial lining"
    SUPERIOR_ATRIOVENTRICULAR_ENDOCARDIAL_CUSHION = "superior atrioventricular endocardial cushion"
    INFERIOR_ATRIOVENTRICULAR_ENDOCARDIAL_CUSHION = "inferior atrioventricular endocardial cushion"
    HEART_ENDOTHELIUM = "heart endothelium"
    OUTFLOW_TRACT_ENDOTHELIUM = "outflow tract endothelium"
    VASCULATURE = "vasculature"
    SECONDARY_HEART_FIELD = "secondary heart field"
    SOMATOSENSORY_CORTEX = "somatosensory cortex"
    PREFRONTAL_CORTEX = "prefrontal cortex"
    ORBITOFRONTAL_CORTEX = "orbitofrontal cortex"
    CA1_STRATUM_RADIATUM = "CA1 stratum radiatum"
    CONJUNCTIVA_EPITHELIUM = "conjunctiva epithelium"
    FAST_MUSCLE_TISSUE = "fast muscle tissue"
    SLOW_MUSCLE_TISSUE = "slow muscle tissue"
    JAW_PRIMORDIUM_EPITHELIUM = "jaw primordium epithelium"
    DENTAL_PLACODE = "dental placode"
    TOOTH_BUD = "tooth bud"
    CAP_STAGE_TOOTH = "cap stage tooth"
    BELL_STAGE_TOOTH = "bell stage tooth"
    EARLY_BELL_STAGE_TOOTH = "early bell stage tooth"
    MID_BELL_STAGE_TOOTH = "mid bell stage tooth"
    LATE_BELL_STAGE_TOOTH = "late bell stage tooth"
    LAMINA_PROPRIA = "lamina propria"
    IRIDOCORNEAL_ANGLE = "iridocorneal angle"
    EYE_TRABECULAR_MESHWORK = "eye trabecular meshwork"
    PRE_PLACODAL_REGION = "pre-placodal region"
    ECTODERMAL_PLACODE = "ectodermal placode"
    CRANIAL_PLACODE = "cranial placode"
    OPTIC_PLACODE = "optic placode"
    ADENOHYPOPHYSEAL_PLACODE = "adenohypophyseal placode"
    NEUROGENIC_PLACODE = "neurogenic placode"
    TRIGEMINAL_PLACODE = "trigeminal placode"
    EPIBRANCHIAL_PLACODE = "epibranchial placode"
    GENICULATE_PLACODE = "geniculate placode"
    PETROSAL_PLACODE = "petrosal placode"
    NODOSE_PLACODE = "nodose placode"
    NON_NEUROGENIC_PLACODE = "non-neurogenic placode"
    TONGUE_PAPILLARY_PLACODE = "tongue papillary placode"
    INTEGUMENTAL_SYSTEM_PLACODE = "integumental system placode"
    VIBRISSA_PLACODE = "vibrissa placode"
    OTIC_PRE_PLACODE = "otic pre-placode"
    ANTERIOR_PITUITARY_PRE_PLACODE = "anterior pituitary pre-placode"
    LENS_PRE_PLACODE = "lens pre-placode"
    EXTRAHEPATIC_BILE_DUCT = "extrahepatic bile duct"
    INTRAHEPATIC_BILE_DUCT = "intrahepatic bile duct"
    VENTRAL_COCHLEAR_NUCLEUS = "ventral cochlear nucleus"
    STOMACH_MUSCULARIS_EXTERNA = "stomach muscularis externa"
    AMNIOTIC_FLUID = "amniotic fluid"
    RETROSPLENIAL_CORTEX = "retrosplenial cortex"
    ATRIOVENTRICULAR_CANAL_CARDIAC_MUSCLE = "atrioventricular canal cardiac muscle"
    HYPOTHALAMUS_PERIVENTRICULAR_REGION = "hypothalamus periventricular region"
    HYPOTHALAMUS_SUBPARAVENTRICULAR_ZONE = "hypothalamus subparaventricular zone"
    BRAIN_ROOF_PLATE = "brain roof plate"
    CEREBELLUM_GRAY_MATTER = "cerebellum gray matter"
    MAMMILLARY_AREA = "mammillary area"
    CUNEATE_FASCICULUS = "cuneate fasciculus"
    GRACILE_FASCICULUS = "gracile fasciculus"
    SPINAL_CORD_LAMINA = "spinal cord lamina"
    ABDOMEN_ORGAN = "abdomen organ"
    ADRENAL_GLAND_CAPSULE = "adrenal gland capsule"
    ADRENAL_GLAND_X_ZONE = "adrenal gland X zone"
    AFFERENT_LYMPHATIC_VESSEL = "afferent lymphatic vessel"
    EFFERENT_LYMPHATIC_VESSEL = "efferent lymphatic vessel"
    ANTERIOR_BUCCAL_GLAND = "anterior buccal gland"
    ANTERIOR_LINGUAL_GLAND = "anterior lingual gland"
    MAJOR_SALIVARY_GLAND = "major salivary gland"
    CORNEA_ANTERIOR_LIMITING_LAMINA = "cornea anterior limiting lamina"
    ANUS = "anus"
    AORTA_ELASTIC_TISSUE = "aorta elastic tissue"
    HEART_APEX = "heart apex"
    AREOLA = "areola"
    CERVICAL_MAMMARY_GLAND = "cervical mammary gland"
    INGUINAL_MAMMARY_GLAND = "inguinal mammary gland"
    AXILLARY_LYMPH_NODE = "axillary lymph node"
    AURICULAR_LYMPH_NODE = "auricular lymph node"
    BRACHIAL_LYMPH_NODE = "brachial lymph node"
    BRONCHIAL_LYMPH_NODE = "bronchial lymph node"
    CELIAC_LYMPH_NODE = "celiac lymph node"
    CERVICAL_LYMPH_NODE = "cervical lymph node"
    GASTRIC_LYMPH_NODE = "gastric lymph node"
    HEPATIC_LYMPH_NODE = "hepatic lymph node"
    ILEOCOLIC_LYMPH_NODE = "ileocolic lymph node"
    ILIAC_LYMPH_NODE = "iliac lymph node"
    LUMBAR_LYMPH_NODE = "lumbar lymph node"
    MANDIBULAR_LYMPH_NODE = "mandibular lymph node"
    PANCREATIC_LYMPH_NODE = "pancreatic lymph node"
    PARAAORTIC_LYMPH_NODE = "paraaortic lymph node"
    PAROTID_LYMPH_NODE = "parotid lymph node"
    POPLITEAL_LYMPH_NODE = "popliteal lymph node"
    RENAL_LYMPH_NODE = "renal lymph node"
    SACRAL_LYMPH_NODE = "sacral lymph node"
    SCIATIC_LYMPH_NODE = "sciatic lymph node"
    SPLENIC_LYMPH_NODE = "splenic lymph node"
    SUBCUTANEOUS_LYMPH_NODE = "subcutaneous lymph node"
    SUBMANDIBULAR_LYMPH_NODE = "submandibular lymph node"
    THYMIC_LYMPH_NODE = "thymic lymph node"
    TRACHEOBRONCHIAL_LYMPH_NODE = "tracheobronchial lymph node"
    LYMPH_NODE_TRABECULUM = "lymph node trabeculum"
    LYMPH_NODE_B_CELL_DOMAIN = "lymph node B cell domain"
    LYMPHOID_TISSUE = "lymphoid tissue"
    BRONCHUS_ASSOCIATED_LYMPHOID_TISSUE = "bronchus-associated lymphoid tissue"
    GUT_ASSOCIATED_LYMPHOID_TISSUE = "gut-associated lymphoid tissue"
    MUCOSA_ASSOCIATED_LYMPHOID_TISSUE = "mucosa-associated lymphoid tissue"
    NASAL_ASSOCIATED_LYMPHOID_TISSUE = "nasal-associated lymphoid tissue"
    OROPHARYNGEAL_LYMPHOID_TISSUE = "oropharyngeal lymphoid tissue"
    BASICRANIUM = "basicranium"
    ALIMENTARY_SYSTEM_FLUID_OR_SECRETION = "alimentary system fluid or secretion"
    ESOPHAGUS_SECRETION = "esophagus secretion"
    STOMACH_SECRETION = "stomach secretion"
    INTESTINE_SECRETION = "intestine secretion"
    PANCREAS_SECRETION = "pancreas secretion"
    ENDOCRINE_PANCREAS_SECRETION = "endocrine pancreas secretion"
    EXOCRINE_PANCREAS_SECRETION = "exocrine pancreas secretion"
    BILE = "bile"
    BILE_DUCT_EPITHELIUM = "bile duct epithelium"
    EXTRAHEPATIC_BILE_DUCT_EPITHELIUM = "extrahepatic bile duct epithelium"
    INTRAHEPATIC_BILE_DUCT_EPITHELIUM = "intrahepatic bile duct epithelium"
    INTERLOBULAR_BILE_DUCT = "interlobular bile duct"
    INTRALOBULAR_BILE_DUCT = "intralobular bile duct"
    BONE_MARROW_CAVITY = "bone marrow cavity"
    CEREBELLUM_ANTERIOR_VERMIS = "cerebellum anterior vermis"
    CEREBELLUM_POSTERIOR_VERMIS = "cerebellum posterior vermis"
    CEREBELLUM_LOBULE = "cerebellum lobule"
    CEREBELLUM_HEMISPHERE_LOBULE = "cerebellum hemisphere lobule"
    FLOCCULONODULAR_LOBE = "flocculonodular lobe"
    CEREBROSPINAL_FLUID = "cerebrospinal fluid"
    CERUMEN = "cerumen"
    FECES = "feces"
    FEMALE_REPRODUCTIVE_SYSTEM_FLUID_OR_SECRETION = "female reproductive system fluid or secretion"
    OVARY_FOLLICLE_FLUID = "ovary follicle fluid"
    LYMPH = "lymph"
    OCULAR_REFRACTIVE_MEDIA = "ocular refractive media"
    PERICARDIAL_FLUID = "pericardial fluid"
    PLEURAL_FLUID = "pleural fluid"
    SKIN_FLUID_OR_SECRETION = "skin fluid or secretion"
    SEBUM = "sebum"
    SWEAT = "sweat"
    SYNOVIAL_FLUID = "synovial fluid"
    C1_VERTEBRA_VENTRAL_TUBERCLE = "C1 vertebra ventral tubercle"
    CERVIX_UTERI_SQUAMOUS_EPITHELIUM = "cervix uteri squamous epithelium"
    THORACIC_BODY_WALL = "thoracic body wall"
    CILIATED_COLUMNAR_OVIDUCT_EPITHELIUM = "ciliated columnar oviduct epithelium"
    CUBOIDAL_OVIDUCT_EPITHELIUM = "cuboidal oviduct epithelium"
    CRANIOCERVICAL_REGION = "craniocervical region"
    CRISTA_AMPULLARIS_NEUROEPITHELIUM = "crista ampullaris neuroepithelium"
    DENTATE_NUCLEUS = "dentate nucleus"
    FASTIGIAL_NUCLEUS = "fastigial nucleus"
    DERMIS_ADIPOSE_TISSUE = "dermis adipose tissue"
    HOLOCRINE_GLAND = "holocrine gland"
    MEROCRINE_GLAND = "merocrine gland"
    ECCRINE_SWEAT_GLAND = "eccrine sweat gland"
    ELASTIC_CARTILAGE = "elastic cartilage"
    ETHMOID_SINUS = "ethmoid sinus"
    FRONTAL_SINUS = "frontal sinus"
    MAXILLARY_SINUS = "maxillary sinus"
    SPHENOID_SINUS = "sphenoid sinus"
    EYELASH = "eyelash"
    FIBROCARTILAGE = "fibrocartilage"
    FONTANEL = "fontanel"
    GLOMERULAR_CAPILLARY_ENDOTHELIUM = "glomerular capillary endothelium"
    GLOTTIS = "glottis"
    GUSTATORY_SYSTEM = "gustatory system"
    ELASTIC_TISSUE = "elastic tissue"
    CARDIOVASCULAR_SYSTEM_ELASTIC_TISSUE = "cardiovascular system elastic tissue"
    HEART_ELASTIC_TISSUE = "heart elastic tissue"
    RESPIRATORY_SYSTEM_ELASTIC_TISSUE = "respiratory system elastic tissue"
    BRONCHUS_ELASTIC_TISSUE = "bronchus elastic tissue"
    LUNG_ELASTIC_TISSUE = "lung elastic tissue"
    TRACHEA_ELASTIC_TISSUE = "trachea elastic tissue"
    HEART_VENTRICLE_WALL = "heart ventricle wall"
    HELIX = "helix"
    HEPATIC_PORTAL_VEIN = "hepatic portal vein"
    IMMUNE_SYSTEM = "immune system"
    INFERIOR_CEREBELLAR_PEDUNCLE = "inferior cerebellar peduncle"
    MIDDLE_CEREBELLAR_PEDUNCLE = "middle cerebellar peduncle"
    SUPERIOR_CEREBELLAR_PEDUNCLE = "superior cerebellar peduncle"
    HAIR_FOLLICLE_INFUNDIBULUM = "hair follicle infundibulum"
    AXILLARY_FAT_PAD = "axillary fat pad"
    INTERSCAPULAR_FAT_PAD = "interscapular fat pad"
    MEDIASTINAL_FAT_PAD = "mediastinal fat pad"
    OMENTAL_FAT_PAD = "omental fat pad"
    PANCREATIC_FAT_PAD = "pancreatic fat pad"
    PARAMETRIAL_FAT_PAD = "parametrial fat pad"
    PERIOVARIAN_FAT_PAD = "periovarian fat pad"
    RENAL_FAT_PAD = "renal fat pad"
    RETROPERITONEAL_FAT_PAD = "retroperitoneal fat pad"
    JOINT_OF__RIB = "joint of  rib"
    INTERCHONDRAL_JOINT = "interchondral joint"
    JOINT_OF_VERTEBRAL_ARCH = "joint of vertebral arch"
    INTERVERTEBRAL_JOINT = "intervertebral joint"
    JOINT_OF_GIRDLE = "joint of girdle"
    RENAL_PYRAMID = "renal pyramid"
    LABIUM_MAJORUM = "labium majorum"
    LABIUM_MINORUM = "labium minorum"
    LACRIMAL_BONE = "lacrimal bone"
    LACRIMAL_SAC = "lacrimal sac"
    OCULOMOTOR_MUSCLE = "oculomotor muscle"
    RECTUS_EXTRAOCULAR_MUSCLE = "rectus extraocular muscle"
    INFERIOR_RECTUS_EXTRAOCULAR_MUSCLE = "inferior rectus extraocular muscle"
    LATERAL_RECTUS_EXTRAOCULAR_MUSCLE = "lateral rectus extraocular muscle"
    MEDIAL_RECTUS_EXTRAOCULAR_MUSCLE = "medial rectus extraocular muscle"
    SUPERIOR_RECTUS_EXTRAOCULAR_MUSCLE = "superior rectus extraocular muscle"
    OBLIQUE_EXTRAOCULAR_MUSCLE = "oblique extraocular muscle"
    INFERIOR_OBLIQUE_EXTRAOCULAR_MUSCLE = "inferior oblique extraocular muscle"
    SUPERIOR_OBLIQUE_EXTRAOCULAR_MUSCLE = "superior oblique extraocular muscle"
    LINGUAL_TONSIL = "lingual tonsil"
    PHARYNGEAL_TONSIL = "pharyngeal tonsil"
    TONSIL_CAPSULE = "tonsil capsule"
    TONSIL_CRYPT = "tonsil crypt"
    LIVER_PERISINUSOIDAL_SPACE = "liver perisinusoidal space"
    PINNA_LOBULE = "pinna lobule"
    MEMBRANOUS_URETHRA_OF_MALE = "membranous urethra of male"
    MALE_REPRODUCTIVE_GLAND_OR_ORGAN = "male reproductive gland or organ"
    MALLEUS_MANUBRIUM = "malleus manubrium"
    MALLEUS_PROCESSUS_BREVIS = "malleus processus brevis"
    MALLEUS_HEAD = "malleus head"
    MALLEUS_NECK = "malleus neck"
    MANDIBLE_ANGULAR_PROCESS = "mandible angular process"
    MANDIBLE_HEAD = "mandible head"
    MANDIBLE_NECK = "mandible neck"
    MANDIBLE_CORONOID_PROCESS = "mandible coronoid process"
    MANDIBLE_TEMPORAL_CREST = "mandible temporal crest"
    MANDIBLE_SYMPHYSIS = "mandible symphysis"
    MANDIBLE_PETROUS_PART = "mandible petrous part"
    MANDIBLE_SQUAMOUS_PART = "mandible squamous part"
    MANDIBLE_ALVEOLAR_PROCESS = "mandible alveolar process"
    MAXILLA_ALVEOLAR_PROCESS = "maxilla alveolar process"
    MASTICATORY_MUSCLE = "masticatory muscle"
    METAPODIAL_PAD = "metapodial pad"
    NASAL_BRIDGE = "nasal bridge"
    NEUROENDOCRINE_GLAND = "neuroendocrine gland"
    ORBIT_OF_SKULL = "orbit of skull"
    OSSEUS_LABYRINTH_VESTIBULE = "osseus labyrinth vestibule"
    VESTIBULAR_AQUEDUCT = "vestibular aqueduct"
    OSSEUS_SEMICIRCULAR_CANAL = "osseus semicircular canal"
    MEMBRANOUS_LABYRINTH = "membranous labyrinth"
    VESTIBULAR_LABYRINTH = "vestibular labyrinth"
    OTOLITHIC_MEMBRANE = "otolithic membrane"
    OTOLITH_ORGAN = "otolith organ"
    BULBAR_CONJUNCTIVA = "bulbar conjunctiva"
    PALPEBRAL_CONJUNCTIVA = "palpebral conjunctiva"
    PALPEBRAL_FISSURE = "palpebral fissure"
    HEART_VENTRICLE_PAPILLARY_MUSCLE = "heart ventricle papillary muscle"
    VERTEBRAL_PEDICLE = "vertebral pedicle"
    PELVIC_LIGAMENT = "pelvic ligament"
    PERIORBITAL_REGION = "periorbital region"
    PEYER_S_PATCH_FOLLICLE = "Peyer's patch follicle"
    PLACENTA_METRIAL_GLAND = "placenta metrial gland"
    PONTINE_RAPHE_NUCLEUS = "pontine raphe nucleus"
    PORTAL_LOBULE = "portal lobule"
    PORTAL_TRIAD = "portal triad"
    EYE_POSTERIOR_CHAMBER = "eye posterior chamber"
    PUPILLARY_MEMBRANE = "pupillary membrane"
    PUPIL = "pupil"
    PARAVERTEBRAL_GANGLION = "paravertebral ganglion"
    PREVERTEBRAL_GANGLION = "prevertebral ganglion"
    MESENTERIC_GANGLION = "mesenteric ganglion"
    INFERIOR_MESENTERIC_GANGLION = "inferior mesenteric ganglion"
    SUPERIOR_MESENTERIC_GANGLION = "superior mesenteric ganglion"
    RETROTRAPEZOID_NUCLEUS = "retrotrapezoid nucleus"
    SEPTUM_PELLUCIDUM = "septum pellucidum"
    REPRODUCTIVE_SYSTEM_GLAND = "reproductive system gland"
    SHOULDER_BONE = "shoulder bone"
    SOMATIC_NERVOUS_SYSTEM = "somatic nervous system"
    SPHENOID_BONE_PTERYGOID_PROCESS = "sphenoid bone pterygoid process"
    SPINAL_CORD_DORSAL_COLUMN = "spinal cord dorsal column"
    SPINAL_CORD_LATERAL_COLUMN = "spinal cord lateral column"
    SPINAL_CORD_LATERAL_MOTOR_COLUMN = "spinal cord lateral motor column"
    SPINAL_CORD_MEDIAL_MOTOR_COLUMN = "spinal cord medial motor column"
    SPINAL_CORD_MOTOR_COLUMN = "spinal cord motor column"
    SPINAL_CORD_VENTRAL_COLUMN = "spinal cord ventral column"
    SPIRAL_MODIOLAR_ARTERY = "spiral modiolar artery"
    SPLEEN_CENTRAL_ARTERIOLE = "spleen central arteriole"
    SPLEEN_MARGINAL_SINUS = "spleen marginal sinus"
    SPLEEN_TRABECULAR_ARTERY = "spleen trabecular artery"
    SPLEEN_TRABECULAR_VEIN = "spleen trabecular vein"
    STAPES_BASE = "stapes base"
    STOMACH_SMOOTH_MUSCLE_CIRCULAR_LAYER = "stomach smooth muscle circular layer"
    DUVET_HAIR = "duvet hair"
    NONSKELETAL_LIGAMENT = "nonskeletal ligament"
    SKELETAL_LIGAMENT = "skeletal ligament"
    STYLOHYOID_LIGAMENT = "stylohyoid ligament"
    SUBLINGUAL_GANGLION = "sublingual ganglion"
    SULCUS_AMPULLARIS = "sulcus ampullaris"
    VESTIBULAR_GANGLION_SUPERIOR_PART = "vestibular ganglion superior part"
    VESTIBULAR_GANGLION_INFERIOR_PART = "vestibular ganglion inferior part"
    EYELID_TARSUS = "eyelid tarsus"
    MEIBOMIAN_GLAND = "meibomian gland"
    OVARY_THECA_EXTERNA = "ovary theca externa"
    OVARY_THECA_INTERNA = "ovary theca interna"
    THYMUS_LOBULE = "thymus lobule"
    THYMUS_SUBCAPSULAR_EPITHELIUM = "thymus subcapsular epithelium"
    THYMUS_TRABECULA = "thymus trabecula"
    TONGUE_SQUAMOUS_EPITTHELIUM = "tongue squamous epitthelium"
    TONGUE_KERATINIZED_EPITHELIUM = "tongue keratinized epithelium"
    TONGUE_TASTE_BUD = "tongue taste bud"
    TOOTH_CAVITY = "tooth cavity"
    VERTEBRA_TRANSVERSE_PROCESS = "vertebra transverse process"
    TROCHANTER_OF_FEMUR = "trochanter of femur"
    UTERINE_ARTERY = "uterine artery"
    VENOUS_SINUS = "venous sinus"
    VERTEBRA_LAMINA = "vertebra lamina"
    VITREOUS_BODY = "vitreous body"
    VOMER_BONE = "vomer bone"
    _4TH_VENTRICLE_APERTURE = "4th ventricle aperture"
    _4TH_VENTRICLE_LATERAL_APERTURE = "4th ventricle lateral aperture"
    _4TH_VENTRICLE_MEDIAN_APERTURE = "4th ventricle median aperture"
    LIMB_SEGMENT = "limb segment"
    STYLOPOD = "stylopod"
    ZEUGOPOD = "zeugopod"
    HYPOPHARYNX = "hypopharynx"
    OVARIAN_BURSA = "ovarian bursa"
    PANCREAS_VASCULATURE = "pancreas vasculature"
    _4TH_VENTRICLE_CHOROID_PLEXUS_EPITHELIUM = "4th ventricle choroid plexus epithelium"
    _4TH_VENTRICLE_CHOROID_PLEXUS_STROMA = "4th ventricle choroid plexus stroma"
    LATERAL_VENTRICLE_CHOROID_PLEXUS_EPITHELIUM = "lateral ventricle choroid plexus epithelium"
    LATERAL_VENTRICLE_CHOROID_PLEXUS_STROMA = "lateral ventricle choroid plexus stroma"
    ABDOMEN_BLOOD_VESSEL = "abdomen blood vessel"
    ABDOMEN_CONNECTIVE_TISSUE = "abdomen connective tissue"
    ABDOMEN_NERVE = "abdomen nerve"
    ABDOMEN_SKIN = "abdomen skin"
    TAIL_DERMIS = "tail dermis"
    TAIL_EPIDERMIS = "tail epidermis"
    SPINAL_CORD_REGION = "spinal cord region"
    SPINAL_CORD_CERVICAL_REGION = "spinal cord cervical region"
    SPINAL_CORD_THORACIC_REGION = "spinal cord thoracic region"
    SPINAL_CORD_LUMBAR_REGION = "spinal cord lumbar region"
    SPINAL_CORD_SACRAL_REGION = "spinal cord sacral region"
    SPINAL_CORD_COCCYGEAL_REGION = "spinal cord coccygeal region"
    UNSEGMENTED_MESENCHYME = "unsegmented mesenchyme"
    _3RD_VENTRICLE_CHOROID_PLEXUS_EPITHEILUM = "3rd ventricle choroid plexus epitheilum"
    _3RD_VENTRICLE_CHOROID_PLEXUS_STROMA = "3rd ventricle choroid plexus stroma"
    ABDOMINAL_SEGMENT_BLOOD_VESSEL = "abdominal segment blood vessel"
    ABDOMINAL_SEGMENT_BONE = "abdominal segment bone"
    ABDOMINAL_SEGMENT_CONNECTIVE_TISSUE = "abdominal segment connective tissue"
    ABDOMINAL_SEGMENT_MUSCLE = "abdominal segment muscle"
    ABDOMINAL_SEGMENT_NERVE = "abdominal segment nerve"
    ABDOMINAL_SEGMENT_ORGAN = "abdominal segment organ"
    ABDOMINAL_SEGMENT_SKIN = "abdominal segment skin"
    ABDOMINAL_WALL = "abdominal wall"
    ACCESSORY_HEPATIC_VEIN = "accessory hepatic vein"
    CRANIAL_NERVE_NUCLEUS = "cranial nerve nucleus"
    ACCESSORY_XI_NERVE_NUCLEUS = "accessory XI nerve nucleus"
    ALIMENTARY_SYSTEM_MESENTERY = "alimentary system mesentery"
    ALVEOLAR_ARTERY = "alveolar artery"
    ANTERIOR_INFERIOR_CEREBELLAR_ARTERY = "anterior inferior cerebellar artery"
    ANTERIOR_TIBIAL_ARTERY = "anterior tibial artery"
    AURICULAR_ARTERY = "auricular artery"
    BUCCAL_ARTERY = "buccal artery"
    CEREBELLAR_ARTERY = "cerebellar artery"
    CERVICAL_ARTERY = "cervical artery"
    DIGITAL_ARTERY = "digital artery"
    DORSAL_ARTERY_OF_FOOT = "dorsal artery of foot"
    DORSAL_INTERCOSTAL_ARTERY = "dorsal intercostal artery"
    DORSAL_METACARPAL_ARTERY = "dorsal metacarpal artery"
    DORSAL_PENIS_ARTERY = "dorsal penis artery"
    ETHMOIDAL_ARTERY = "ethmoidal artery"
    ILEOCOLIC_ARTERY = "ileocolic artery"
    ILIAC_CIRCUMFLEX_ARTERY = "iliac circumflex artery"
    ILIO_LUMBAR_ARTERY = "ilio-lumbar artery"
    INFERIOR_PANCREATICO_DUODENAL_ARTERY = "inferior pancreatico-duodenal artery"
    INFERIOR_RECTAL_ARTERY = "inferior rectal artery"
    INFERIOR_SUPRARENAL_ARTERY = "inferior suprarenal artery"
    INFERIOR_VESICAL_ARTERY = "inferior vesical artery"
    INFRAORBITAL_ARTERY = "infraorbital artery"
    INTRALOBAR_ARTERY = "intralobar artery"
    LABIAL_ARTERY = "labial artery"
    LABYRINTHINE_ARTERY = "labyrinthine artery"
    LACRIMAL_ARTERY = "lacrimal artery"
    LATERAL_THORACIC_ARTERY = "lateral thoracic artery"
    LEFT_HEPATIC_ARTERY = "left hepatic artery"
    LEFT_PULMONARY_ARTERY = "left pulmonary artery"
    LEFT_RENAL_ARTERY = "left renal artery"
    LEFT_SPERMATIC_ARTERY = "left spermatic artery"
    LEFT_SUBCLAVIAN_ARTERY = "left subclavian artery"
    MEDIAL_PLANTAR_ARTERY = "medial plantar artery"
    MEDIAL_TARSAL_ARTERY = "medial tarsal artery"
    MEDIAN_ARTERY = "median artery"
    MENINGEAL_ARTERY = "meningeal artery"
    MENTAL_ARTERY = "mental artery"
    MIDDLE_COLIC_ARTERY = "middle colic artery"
    PERINEAL_ARTERY = "perineal artery"
    PHRENIC_ARTERY = "phrenic artery"
    POPLITEAL_ARTERY = "popliteal artery"
    POSTERIOR_TIBIAL_ARTERY = "posterior tibial artery"
    PUDENDAL_ARTERY = "pudendal artery"
    RADIAL_ARTERY = "radial artery"
    RETINAL_ARTERY = "retinal artery"
    SPERMATIC_ARTERY = "spermatic artery"
    SUBLINGUAL_ARTERY = "sublingual artery"
    SUBSCAPULAR_ARTERY = "subscapular artery"
    SUPERFICIAL_CERVICAL_ARTERY = "superficial cervical artery"
    SUPERIOR_TEMPORAL_ARTERY = "superior temporal artery"
    SUPERIOR_EXTERNAL_PUDENDAL_ARTERY = "superior external pudendal artery"
    SUPERIOR_GLUTEAL_ARTERY = "superior gluteal artery"
    SUPERIOR_PANCREATICO_DUODENAL_ARTERY = "superior pancreatico-duodenal artery"
    SUPERIOR_SUPRARENAL_ARTERY = "superior suprarenal artery"
    SUPRARENAL_ARTERY = "suprarenal artery"
    SURAL_ARTERY = "sural artery"
    SYSTEMIC_ARTERY = "systemic artery"
    TEMPORAL_ARTERY = "temporal artery"
    THORACO_ACROMIAL_ARTERY = "thoraco-acromial artery"
    THORACODORSAL_ARTERY = "thoracodorsal artery"
    TRANSVERSE_FACIAL_ARTERY = "transverse facial artery"
    VESICAL_ARTERY = "vesical artery"
    ANTERIOR_AURICULAR_VEIN = "anterior auricular vein"
    ANTERIOR_FACIAL_VEIN = "anterior facial vein"
    ANTERIOR_JUGULAR_VEIN = "anterior jugular vein"
    ANTERIOR_TIBIAL_VEIN = "anterior tibial vein"
    AURICULAR_VEIN = "auricular vein"
    CARDIAC_VEIN = "cardiac vein"
    CAUDAL_AURICULAR_VEIN = "caudal auricular vein"
    CEREBELLAR_VEIN = "cerebellar vein"
    CEREBRAL_VEIN = "cerebral vein"
    DIGITAL_VEIN = "digital vein"
    DORSAL_CEREBRAL_VEIN = "dorsal cerebral vein"
    DORSAL_METATARSAL_VEIN = "dorsal metatarsal vein"
    EXTERNAL_ILIAC_VEIN = "external iliac vein"
    EXTERNAL_THORACIC_VEIN = "external thoracic vein"
    GASTRIC_VEIN = "gastric vein"
    GREAT_SAPHENOUS_VEIN = "great saphenous vein"
    HYPOGASTRIC_VEIN = "hypogastric vein"
    ILEAL_VEIN = "ileal vein"
    ILEOCOLIC_VEIN = "ileocolic vein"
    ILIAC_VEIN = "iliac vein"
    ILIO_LUMBAR_VEIN = "ilio-lumbar vein"
    INFERIOR_CEREBELLAR_VEIN = "inferior cerebellar vein"
    INFERIOR_CEREBRAL_VEIN = "inferior cerebral vein"
    INFERIOR_EXTERNAL_PUDENDAL_VEIN = "inferior external pudendal vein"
    INFERIOR_PALPEBRAL_VEIN = "inferior palpebral vein"
    INFERIOR_PHRENIC_VEIN = "inferior phrenic vein"
    INFERIOR_VESICAL_VEIN = "inferior vesical vein"
    INTERNAL_MAMMARY_VEIN = "internal mammary vein"
    JEJUNAL_VEIN = "jejunal vein"
    LATERAL_MARGINAL_VEIN = "lateral marginal vein"
    LATERAL_SAPHENOUS_VEIN = "lateral saphenous vein"
    LATERAL_THORACIC_VEIN = "lateral thoracic vein"
    LEFT_HEPATIC_VEIN = "left hepatic vein"
    LEFT_PULMONARY_VEIN = "left pulmonary vein"
    LEFT_RENAL_VEIN = "left renal vein"
    LEFT_TESTICULAR_VEIN = "left testicular vein"
    MASSETERIC_VEIN = "masseteric vein"
    MAXILLARY_VEIN = "maxillary vein"
    MEDIAL_MARGINAL_VEIN = "medial marginal vein"
    MEDIAL_PLANTAR_VEIN = "medial plantar vein"
    MEDIAL_SAPHENOUS_VEIN = "medial saphenous vein"
    MEDIAN_CUBITAL_VEIN = "median cubital vein"
    MIDDLE_COLIC_VEIN = "middle colic vein"
    MIDDLE_HEPATIC_VEIN = "middle hepatic vein"
    MIDDLE_TEMPORAL_VEIN = "middle temporal vein"
    PANCREATICO_DUODENAL_VEIN = "pancreatico-duodenal vein"
    PERICARDIACOPHRENIC_VEIN = "pericardiacophrenic vein"
    PERINEAL_VEIN = "perineal vein"
    PHRENIC_VEIN = "phrenic vein"
    POPLITEAL_VEIN = "popliteal vein"
    POSTERIOR_FACIAL_VEIN = "posterior facial vein"
    POSTERIOR_TIBIAL_VEIN = "posterior tibial vein"
    PUDENDAL_VEIN = "pudendal vein"
    RETINAL_VEIN = "retinal vein"
    SAPHENOUS_VEIN = "saphenous vein"
    SMALL_SAPHENOUS_VEIN = "small saphenous vein"
    SUBMENTAL_VEIN = "submental vein"
    SUBCOSTAL_VEIN = "subcostal vein"
    SUBSCAPULAR_VEIN = "subscapular vein"
    SUPERIOR_CEREBELLAR_VEIN = "superior cerebellar vein"
    SUPERIOR_EXTERNAL_PUDENDAL_VEIN = "superior external pudendal vein"
    SUPERIOR_INTERCOSTAL_VEIN = "superior intercostal vein"
    SUPERIOR_PANCREATICO_DUODENAL_VEIN = "superior pancreatico-duodenal vein"
    SUPRAORBITAL_VEIN = "supraorbital vein"
    TEMPORAL_VEIN = "temporal vein"
    THORACIC_VEIN = "thoracic vein"
    THORACODORSAL_VEIN = "thoracodorsal vein"
    TIBIAL_VEIN = "tibial vein"
    TRANSVERSE_FACIAL_VEIN = "transverse facial vein"
    VEIN_OF_CLITORIS = "vein of clitoris"
    VEIN_OF_LOWER_LIP = "vein of lower lip"
    VEIN_OF_TRABECULAR_BONE = "vein of trabecular bone"
    VEIN_OF_UPPER_LIP = "vein of upper lip"
    VESICAL_VEIN = "vesical vein"
    COMMON_HEPATIC_ARTERY = "common hepatic artery"
    COMMON_PALMAR_DIGITAL_ARTERY = "common palmar digital artery"
    COMMON_PLANTAR_DIGITAL_ARTERY = "common plantar digital artery"
    FORELIMB_COMMON_DORSAL_DIGITAL_ARTERY = "forelimb common dorsal digital artery"
    FORELIMB_DIGITAL_ARTERY = "forelimb digital artery"
    FORELIMB_PROPER_DORSAL_DIGITAL_ARTERY = "forelimb proper dorsal digital artery"
    HINDLIMB_COMMON_DORSAL_DIGITAL_ARTERY = "hindlimb common dorsal digital artery"
    HINDLIMB_DIGITAL_ARTERY = "hindlimb digital artery"
    HINDLIMB_PROPER_DORSAL_DIGITAL_ARTERY = "hindlimb proper dorsal digital artery"
    PROFUNDA_BRACHII_ARTERY = "profunda brachii artery"
    PROFUNDA_FEMORIS_ARTERY = "profunda femoris artery"
    PROPER_HEPATIC_ARTERY = "proper hepatic artery"
    PROPER_PALMAR_DIGITAL_ARTERY = "proper palmar digital artery"
    PROPER_PLANTAR_DIGITAL_ARTERY = "proper plantar digital artery"
    SAPHENOUS_ARTERY = "saphenous artery"
    COMMON_PLANTAR_VEIN = "common plantar vein"
    COMMON_PALMAR_DIGITAL_VEIN = "common palmar digital vein"
    FORELIMB_DIGITAL_VEIN = "forelimb digital vein"
    HINDLIMB_DIGITAL_VEIN = "hindlimb digital vein"
    GREAT_VEIN_OF_HEART = "great vein of heart"
    LIENAL_VEIN = "lienal vein"
    NASO_FRONTAL_VEIN = "naso-frontal vein"
    PARAUMBILICAL_VEIN = "paraumbilical vein"
    PROPER_PALMAR_VEIN = "proper palmar vein"
    PROPER_PLANTAR_VEIN = "proper plantar vein"
    SURAL_VEIN = "sural vein"
    CHEST = "chest"
    CHEST_WALL = "chest wall"
    UPPER_BACK = "upper back"
    PELVIC_CAVITY = "pelvic cavity"
    NECK_BLOOD_VESSEL = "neck blood vessel"
    TRUNK_BLOOD_VESSEL = "trunk blood vessel"
    THORACIC_SEGMENT_BLOOD_VESSEL = "thoracic segment blood vessel"
    CHEST_BLOOD_VESSEL = "chest blood vessel"
    THORACIC_CAVITY_BLOOD_VESSEL = "thoracic cavity blood vessel"
    THORACIC_CAVITY_ARTERY = "thoracic cavity artery"
    THORACIC_CAVITY_VEIN = "thoracic cavity vein"
    BACK_BLOOD_VESSEL = "back blood vessel"
    PELVIS_BLOOD_VESSEL = "pelvis blood vessel"
    CRANIOCERVICAL_REGION_BONE = "craniocervical region bone"
    NECK_BONE = "neck bone"
    THORACIC_SEGMENT_BONE = "thoracic segment bone"
    BACK_BONE = "back bone"
    PELVIS_BONE = "pelvis bone"
    NECK_CONNECTIVE_TISSUE = "neck connective tissue"
    NECK_CARTILAGE = "neck cartilage"
    TRUNK_CONNECTIVE_TISSUE = "trunk connective tissue"
    THORACIC_SEGMENT_CONNECTIVE_TISSUE = "thoracic segment connective tissue"
    CHEST_CONNECTIVE_TISSUE = "chest connective tissue"
    THORACIC_CAVITY_CONNECTIVE_TISSUE = "thoracic cavity connective tissue"
    BACK_CONNECTIVE_TISSUE = "back connective tissue"
    PELVIS_CONNECTIVE_TISSUE = "pelvis connective tissue"
    AXIAL_MESODERM = "axial mesoderm"
    CRANIOCERVICAL_REGION_MUSCLE = "craniocervical region muscle"
    THORACIC_SEGMENT_MUSCLE = "thoracic segment muscle"
    UPPER_BACK_MUSCLE = "upper back muscle"
    LOWER_BACK_MUSCLE = "lower back muscle"
    NECK_NERVE = "neck nerve"
    TRUNK_NERVE = "trunk nerve"
    THORACIC_SEGMENT_NERVE = "thoracic segment nerve"
    CHEST_NERVE = "chest nerve"
    THORACIC_CAVITY_NERVE = "thoracic cavity nerve"
    BACK_NERVE = "back nerve"
    PELVIS_NERVE = "pelvis nerve"
    TRUNK_ORGAN = "trunk organ"
    THORACIC_SEGMENT_ORGAN = "thoracic segment organ"
    CHEST_ORGAN = "chest organ"
    THORACIC_CAVITY_ORGAN = "thoracic cavity organ"
    BACK_ORGAN = "back organ"
    PELVIS_ORGAN = "pelvis organ"
    CRANIOCERVICAL_REGION_SKIN = "craniocervical region skin"
    NECK_SKIN = "neck skin"
    TRUNK_SKIN = "trunk skin"
    CHEST_SKIN = "chest skin"
    UPPER_BACK_SKIN = "upper back skin"
    LOWER_BACK_SKIN = "lower back skin"
    PELVIS_SKIN = "pelvis skin"
    APPENDAGE = "appendage"
    METACARPAL_METATARSAL_BONE = "metacarpal/metatarsal bone"
    LIMB_DIGIT_1 = "limb digit 1"
    LIMB_DIGIT_2 = "limb digit 2"
    LIMB_DIGIT_3 = "limb digit 3"
    LIMB_DIGIT_4 = "limb digit 4"
    LIMB_DIGIT_5 = "limb digit 5"
    DIGIT_CONNECTIVE_TISSUE = "digit connective tissue"
    LIMB_NERVE = "limb nerve"
    FORELIMB_BLOOD_VESSEL = "forelimb blood vessel"
    ARM_BLOOD_VESSEL = "arm blood vessel"
    UPPER_ARM_BLOOD_VESSEL = "upper arm blood vessel"
    FOREARM_BLOOD_VESSEL = "forearm blood vessel"
    HAND_BLOOD_VESSEL = "hand blood vessel"
    FORELIMB_DIGIT_BLOOD_VESSEL = "forelimb digit blood vessel"
    HINDLIMB_BLOOD_VESSEL = "hindlimb blood vessel"
    UPPER_LEG_BLOOD_VESSEL = "upper leg blood vessel"
    LOWER_LEG_BLOOD_VESSEL = "lower leg blood vessel"
    FOOT_BLOOD_VESSEL = "foot blood vessel"
    HINDLIMB_DIGIT_BLOOD_VESSEL = "hindlimb digit blood vessel"
    ARM_BONE = "arm bone"
    LEG_BONE = "leg bone"
    FORELIMB_DIGIT_BONE = "forelimb digit bone"
    FORELIMB_MIDDLE_PHALANX = "forelimb middle phalanx"
    FORELIMB_PROXIMAL_PHALANX = "forelimb proximal phalanx"
    HIP_BONE = "hip bone"
    HINDLIMB_DIGIT_BONE = "hindlimb digit bone"
    HINDLIMB_DISTAL_PHALANX = "hindlimb distal phalanx"
    HINDLIMB_MIDDLE_PHALANX = "hindlimb middle phalanx"
    HINDLIMB_PROXIMAL_PHALANX = "hindlimb proximal phalanx"
    LIMB_CONNECTIVE_TISSUE = "limb connective tissue"
    FORELIMB_CONNECTIVE_TISSUE = "forelimb connective tissue"
    SHOULDER_CONNECTIVE_TISSUE = "shoulder connective tissue"
    ARM_CONNECTIVE_TISSUE = "arm connective tissue"
    UPPER_ARM_CONNECTIVE_TISSUE = "upper arm connective tissue"
    ELBOW_CONNECTIVE_TISSUE = "elbow connective tissue"
    FOREARM_CONNECTIVE_TISSUE = "forearm connective tissue"
    WRIST_CONNECTIVE_TISSUE = "wrist connective tissue"
    FORELIMB_DIGIT_CONNECTIVE_TISSUE = "forelimb digit connective tissue"
    HINDLIMB_CONNECTIVE_TISSUE = "hindlimb connective tissue"
    HIP_CONNECTIVE_TISSUE = "hip connective tissue"
    LEG_CONNECTIVE_TISSUE = "leg connective tissue"
    UPPER_LEG_CONNECTIVE_TISSUE = "upper leg connective tissue"
    KNEE_CONNECTIVE_TISSUE = "knee connective tissue"
    LOWER_LEG_CONNECTIVE_TISSUE = "lower leg connective tissue"
    ANKLE_CONNECTIVE_TISSUE = "ankle connective tissue"
    FOOT_CONNECTIVE_TISSUE = "foot connective tissue"
    HINDLIMB_DIGIT_CONNECTIVE_TISSUE = "hindlimb digit connective tissue"
    FORELIMB_DIGIT_MUSCLE = "forelimb digit muscle"
    HIP_MUSCLE = "hip muscle"
    HINDLIMB_DIGIT_MUSCLE = "hindlimb digit muscle"
    FORELIMB_NERVE = "forelimb nerve"
    SHOULDER_NERVE = "shoulder nerve"
    ARM_NERVE = "arm nerve"
    UPPER_ARM_NERVE = "upper arm nerve"
    FOREARM_NERVE = "forearm nerve"
    WRIST_NERVE = "wrist nerve"
    HAND_NERVE = "hand nerve"
    FORELIMB_DIGIT_NERVE = "forelimb digit nerve"
    HINDLIMB_NERVE = "hindlimb nerve"
    LEG_NERVE = "leg nerve"
    UPPER_LEG_NERVE = "upper leg nerve"
    LOWER_LEG_NERVE = "lower leg nerve"
    ANKLE_NERVE = "ankle nerve"
    FOOT_NERVE = "foot nerve"
    HINDLIMB_DIGIT_NERVE = "hindlimb digit nerve"
    AXILLA_SKIN = "axilla skin"
    ARM_SKIN = "arm skin"
    WRIST_SKIN = "wrist skin"
    LEG_SKIN = "leg skin"
    ANKLE_SKIN = "ankle skin"
    HEEL_SKIN = "heel skin"
    METAPODIAL_PAD_EPIDERMIS = "metapodial pad epidermis"
    LEFT_ADRENAL_GLAND = "left adrenal gland"
    RIGHT_ADRENAL_GLAND = "right adrenal gland"
    LEFT_BRANCH_OF_ATRIOVENTRICULAR_BUNDLE = "left branch of atrioventricular bundle"
    RIGHT_BRANCH_OF_ATRIOVENTRICULAR_BUNDLE = "right branch of atrioventricular bundle"
    LEFT_OUTER_EAR = "left outer ear"
    RIGHT_OUTER_EAR = "right outer ear"
    LEFT_EYE = "left eye"
    RIGHT_EYE = "right eye"
    LEFT_MAIN_BRONCHUS = "left main bronchus"
    RIGHT_MAIN_BRONCHUS = "right main bronchus"
    LEFT_OVARY = "left ovary"
    RIGHT_OVARY = "right ovary"
    LEFT_OVIDUCT = "left oviduct"
    RIGHT_OVIDUCT = "right oviduct"
    LEFT_TESTIS = "left testis"
    RIGHT_TESTIS = "right testis"
    LEFT_URETER = "left ureter"
    RIGHT_URETER = "right ureter"
    LEFT_UTERINE_HORN = "left uterine horn"
    RIGHT_UTERINE_HORN = "right uterine horn"
    RIGHT_HEPATIC_ARTERY = "right hepatic artery"
    RIGHT_HEPATIC_VEIN = "right hepatic vein"
    RIGHT_PULMONARY_ARTERY = "right pulmonary artery"
    RIGHT_PULMONARY_VEIN = "right pulmonary vein"
    RIGHT_RENAL_ARTERY = "right renal artery"
    RIGHT_RENAL_VEIN = "right renal vein"
    RIGHT_SPERMATIC_ARTERY = "right spermatic artery"
    RIGHT_TESTICULAR_VEIN = "right testicular vein"
    RIGHT_SUBCLAVIAN_ARTERY = "right subclavian artery"
    PAMPINIFORM_PLEXUS = "pampiniform plexus"
    AURICULAR_BLOOD_VESSEL = "auricular blood vessel"
    IRIS_BLOOD_VESSEL = "iris blood vessel"
    EYELID_BLOOD_VESSEL = "eyelid blood vessel"
    INTERSOMITIC_BLOOD_VESSEL = "intersomitic blood vessel"
    ALVEOLAR_ATRIUM = "alveolar atrium"
    ALVEOLAR_PORE = "alveolar pore"
    ANAL_FOLD = "anal fold"
    ANAL_REGION_SKELETAL_MUSCLE = "anal region skeletal muscle"
    ANAL_REGION_SMOOTH_MUSCLE = "anal region smooth muscle"
    ANAL_SAC = "anal sac"
    ANOCOCCYGEUS = "anococcygeus"
    ANSIFORM_LOBULE = "ansiform lobule"
    ANSIFORM_LOBULE_CRUS_I = "ansiform lobule crus I"
    ANSIFORM_LOBULE_CRUS_II = "ansiform lobule crus II"
    COPULA_PYRAMIDIS = "copula pyramidis"
    LOBULE_SIMPLEX = "lobule simplex"
    PARAFLOCCULUS = "paraflocculus"
    PARAMEDIAN_LOBULE = "paramedian lobule"
    ANTERIOR_COMMISSURE_ANTERIOR_PART = "anterior commissure anterior part"
    ANTERIOR_DIGASTRIC_MUSCLE = "anterior digastric muscle"
    POSTERIOR_DIGASTRIC_MUSCLE = "posterior digastric muscle"
    BED_NUCLEUS_OF_STRIA_TERMINALIS_ANTERIOR_PART = "bed nucleus of stria terminalis anterior part"
    BED_NUCLEUS_OF_STRIA_TERMINALIS_POSTERIOR_PART = "bed nucleus of stria terminalis posterior part"
    ANTERIOR_HYPOTHALAMIC_NUCLEUS = "anterior hypothalamic nucleus"
    ANTERIOR_COMMISSURE_POSTERIOR_PART = "anterior commissure posterior part"
    ANTERIOR_SEMICIRCULAR_DUCT = "anterior semicircular duct"
    LATERAL_SEMICIRCULAR_DUCT = "lateral semicircular duct"
    POSTERIOR_SEMICIRCULAR_DUCT = "posterior semicircular duct"
    CORNEA_ANTERIOR_STROMA = "cornea anterior stroma"
    CORNEA_POSTERIOR_STROMA = "cornea posterior stroma"
    ANTERIOR_TEGMENTAL_NUCLEUS = "anterior tegmental nucleus"
    ANTERIOR_UVEA = "anterior uvea"
    AORTIC_VALVE_ANULUS = "aortic valve anulus"
    MITRAL_VALVE_ANULUS = "mitral valve anulus"
    PULMONARY_VALVE_ANULUS = "pulmonary valve anulus"
    TRICUSPID_VALVE_ANULUS = "tricuspid valve anulus"
    HEART_ANULUS_FIBROSUS = "heart anulus fibrosus"
    APEX_OF_ARYTENOID = "apex of arytenoid"
    BASE_OF_ARYTENOID = "base of arytenoid"
    APEX_OF_HEART_VENTRICLE = "apex of heart ventricle"
    APOCRINE_SWEAT_GLAND = "apocrine sweat gland"
    AQUEOUS_DRAINAGE_SYSTEM = "aqueous drainage system"
    AQUEOUS_VEIN = "aqueous vein"
    INFERIOR_LABIAL_ARTERY = "inferior labial artery"
    SUPERIOR_LABIAL_ARTERY = "superior labial artery"
    ATRIOVENTRICULAR_SEPTAL_COMPLEX = "atrioventricular septal complex"
    ATRIAL_TRABECULA = "atrial trabecula"
    AUDITORY_BULLA = "auditory bulla"
    AURICULAR_LIGAMENT = "auricular ligament"
    BACHMANN_S_BUNDLE = "Bachmann's bundle"
    BARRINGTON_S_NUCLEUS = "Barrington's nucleus"
    TONGUE_ANTERIOR_PART = "tongue anterior part"
    TONGUE_POSTERIOR_PART = "tongue posterior part"
    BASILIC_VEIN = "basilic vein"
    BILIARY_DUCTULE = "biliary ductule"
    BODY_FLUID = "body fluid"
    BODY_CAVITY = "body cavity"
    BODY_OF_STERNUM = "body of sternum"
    HARD_PALATE_BONY_PART = "hard palate bony part"
    BRACHIAL_VEIN = "brachial vein"
    BRAIN_COMMISSURE = "brain commissure"
    BRAIN_VENTRICLE_AND_CHOROID_PLEXUS = "brain ventricle and choroid plexus"
    BRONCHOALVEOLAR_DUCT_JUNCTION = "bronchoalveolar duct junction"
    BRONCHUS_SUBMUCOSA = "bronchus submucosa"
    BRUCH_S_MEMBRANE = "Bruch's membrane"
    CHEEK = "cheek"
    BUCCAL_POUCH = "buccal pouch"
    BUCCINATOR = "buccinator"
    BULBOSPONGIOSUS = "bulbospongiosus"
    ISCHIOCAVERNOSUS = "ischiocavernosus"
    CANAL_OF_SCHLEMM = "canal of Schlemm"
    CANTHUS = "canthus"
    OUTER_CANTHUS = "outer canthus"
    CARDIAC_REGION_GASTRIC_GLAND = "cardiac region gastric gland"
    CARDIAC_NERVE_PLEXUS = "cardiac nerve plexus"
    CELIAC_NERVE_PLEXUS = "celiac nerve plexus"
    PULMONARY_NERVE_PLEXUS = "pulmonary nerve plexus"
    CARTILAGINOUS_JOINT_SUTURE = "cartilaginous joint suture"
    GOMPHOSIS = "gomphosis"
    SYNDESMOSIS = "syndesmosis"
    BONY_OTIC_CAPSULE = "bony otic capsule"
    CARTILAGINOUS_OTIC_CAPSULE = "cartilaginous otic capsule"
    CAUDA_EQUINA = "cauda equina"
    CAUDAL_GANGLIONIC_EMINENCE = "caudal ganglionic eminence"
    LATERAL_GANGLIONIC_EMINENCE = "lateral ganglionic eminence"
    MEDIAL_GANGLIONIC_EMINENCE = "medial ganglionic eminence"
    CAUDAL_HUMERAL_CIRCUMFLEX_VEIN = "caudal humeral circumflex vein"
    CELLULAR_CARTILAGE = "cellular cartilage"
    MUSCLE_ORGAN = "muscle organ"
    MYOCARDIUM_LAYER = "myocardium layer"
    SYNOVIUM = "synovium"
    CEREBRAL_HEMISPHERE_GRAY_MATTER = "cerebral hemisphere gray matter"
    CHOLEDOCHODUODENAL_JUNCTION = "choledochoduodenal junction"
    ENTERIC_NERVE = "enteric nerve"
    CHOLINERGIC_ENTERIC_NERVE = "cholinergic enteric nerve"
    PELVIS_MUSCLE = "pelvis muscle"
    COCCYGEUS = "coccygeus"
    CONSTRICTOR_VULVAE = "constrictor vulvae"
    CORNICULATE_CARTILAGE = "corniculate cartilage"
    KIDNEY_OUTER_CORTEX = "kidney outer cortex"
    KIDNEY_CORTICAL_ARCH = "kidney cortical arch"
    KIDNEY_COLUMN = "kidney column"
    KIDNEY_CORTEX_PERITUBULAR_CAPILLARY = "kidney cortex peritubular capillary"
    RENAL_STRAIGHT_TUBULE = "renal straight tubule"
    RENAL_TUBULE_BASEMENT_MEMBRANE = "renal tubule basement membrane"
    COSTAL_ARCH = "costal arch"
    DISTAL_RIB = "distal rib"
    FALSE_RIB = "false rib"
    FLOATING_RIB = "floating rib"
    PROXIMAL_RIB = "proximal rib"
    COSTO_CERVICAL_TRUNK = "costo-cervical trunk"
    CRANIAL_CAVITY = "cranial cavity"
    CRICOARYTENOID_MUSCLE = "cricoarytenoid muscle"
    CRUS_CEREBRI = "crus cerebri"
    CUTANEOUS_MUSCLE = "cutaneous muscle"
    CUTANEOUS_TRUNCI_MUSCLE = "cutaneous trunci muscle"
    LEVATOR_NASOLABIALIS = "levator nasolabialis"
    LEVATOR_LABII_SUPERIORIS = "levator labii superioris"
    INTEGUMENT = "integument"
    HYPODERMIS = "hypodermis"
    CUTANEOUS_VEIN = "cutaneous vein"
    EPONYCHIUM = "eponychium"
    HYPONYCHIUM = "hyponychium"
    PROXIMAL_NAIL_BED = "proximal nail bed"
    DEEP_CERVICAL_LYMPH_NODE = "deep cervical lymph node"
    DEEP_ILIAC_CIRCUMFLEX_VEIN = "deep iliac circumflex vein"
    DEEP_TEMPORAL_VEIN = "deep temporal vein"
    DEFERENT_DUCT_ARTERY = "deferent duct artery"
    DEFERENT_DUCT_VEIN = "deferent duct vein"
    DENTATE_GYRUS_PYRAMIDAL_LAYER = "dentate gyrus pyramidal layer"
    DEPRESSOR_LABII_INFERIORIS = "depressor labii inferioris"
    DORSAL_FOOT_INTEROSSEUS_MUSCLE = "dorsal foot interosseus muscle"
    PLANTAR_FOOT_INTEROSSEUS_MUSCLE = "plantar foot interosseus muscle"
    DORSAL_HAND_INTEROSSEUS_MUSCLE = "dorsal hand interosseus muscle"
    PALMAR_HAND_INTEROSSEUS_MUSCLE = "palmar hand interosseus muscle"
    DORSAL_VENOUS_ARCH = "dorsal venous arch"
    EPIGLOTTIC_CARTILAGE = "epiglottic cartilage"
    ESOPHAGUS_SMOOTH_MUSCLE_CIRCULAR_LAYER = "esophagus smooth muscle circular layer"
    ESOPHAGUS_SMOOTH_MUSCLE_LONGITUDINAL_LAYER = "esophagus smooth muscle longitudinal layer"
    EXTERNAL_AUDITORY_MEATUS_CARTILAGINOUS_PART = "external auditory meatus cartilaginous part"
    EXTERNAL_AUDITORY_MEATUS_OSSEUS_PART = "external auditory meatus osseus part"
    EXTRINSIC_AURICULAR_MUSCLE = "extrinsic auricular muscle"
    INTRINSIC_AURICULAR_MUSCLE = "intrinsic auricular muscle"
    EYE_CHAMBER = "eye chamber"
    EYE_SKIN_GLAND = "eye skin gland"
    EYE_SEBACEOUS_GLAND = "eye sebaceous gland"
    EYELID_CONNECTIVE_TISSUE = "eyelid connective tissue"
    EYELID_SUBCUTANEOUS_CONNECTIVE_TISSUE = "eyelid subcutaneous connective tissue"
    EYELID_SUBMUSCULAR_CONNECTIVE_TISSUE = "eyelid submuscular connective tissue"
    EYELID_NERVE = "eyelid nerve"
    EYELID_SKIN = "eyelid skin"
    MEMBRANOUS_URETHRA_OF_FEMALE = "membranous urethra of female"
    FEMUR_DIAPHYSIS = "femur diaphysis"
    FEMUR_EPIPHYSIS = "femur epiphysis"
    FEMUR_METAPHYSIS = "femur metaphysis"
    HEAD_OF_FEMUR = "head of femur"
    MEMBRANOUS_URETHRA = "membranous urethra"
    FOREBRAIN_MENINGES = "forebrain meninges"
    FOREBRAIN_ARACHNOID_MATER = "forebrain arachnoid mater"
    FOREBRAIN_DURA_MATER = "forebrain dura mater"
    FOREBRAIN_PIA_MATER = "forebrain pia mater"
    LOWER_RESPIRATORY_TRACT_CARTILAGE = "lower respiratory tract cartilage"
    LOWER_RESPIRATORY_TRACT_CONNECTIVE_TISSUE = "lower respiratory tract connective tissue"
    LOWER_RESPIRATORY_TRACT_EPITHELIUM = "lower respiratory tract epithelium"
    MAIN_BRONCHUS_CONNECTIVE_TISSUE = "main bronchus connective tissue"
    MAIN_BRONCHUS_SMOOTH_MUSCLE = "main bronchus smooth muscle"
    TRACHEA_BASEMENT_MEMBRANE = "trachea basement membrane"
    TRACHEA_BASAL_LAMINA = "trachea basal lamina"
    TRACHEA_RETICULAR_LAMINA = "trachea reticular lamina"
    TRACHEAL_BIFURCATION = "tracheal bifurcation"
    LARGE_INTESTINE_MUSCULARIS_MUCOSA = "large intestine muscularis mucosa"
    LARGE_INTESTINE_SEROSA = "large intestine serosa"
    LARGE_INTESTINE_SMOOTH_MUSCLE_CIRCULAR_LAYER = "large intestine smooth muscle circular layer"
    LARGE_INTESTINE_SMOOTH_MUSCLE_LONGITUDINAL_LAYER = "large intestine smooth muscle longitudinal layer"
    LARGE_INTESTINE_SUBMUCOSA = "large intestine submucosa"
    SMALL_INTESTINE_SEROSA = "small intestine serosa"
    SMALL_INTESTINE_SMOOTH_MUSCLE_CIRCULAR_LAYER = "small intestine smooth muscle circular layer"
    SMALL_INTESTINE_SMOOTH_MUSCLE_LONGITUDINAL_LAYER = "small intestine smooth muscle longitudinal layer"
    SMALL_INTESTINE_SUBMUCOSA = "small intestine submucosa"
    RESPIRATORY_SYSTEM_ARTERIAL_BLOOD_VESSEL = "respiratory system arterial blood vessel"
    RESPIRATORY_SYSTEM_ARTERIAL_ENDOTHELIUM = "respiratory system arterial endothelium"
    RESPIRATORY_SYSTEM_ARTERIAL_SMOOTH_MUSCLE = "respiratory system arterial smooth muscle"
    RESPIRATORY_SYSTEM_ARTERIOLE = "respiratory system arteriole"
    RESPIRATORY_SYSTEM_ARTERY = "respiratory system artery"
    RESPIRATORY_SYSTEM_BASEMENT_MEMBRANE = "respiratory system basement membrane"
    RESPIRATORY_SYSTEM_BASAL_LAMINA = "respiratory system basal lamina"
    RESPIRATORY_SYSTEM_RETICULAR_LAMINA = "respiratory system reticular lamina"
    RESPIRATORY_SYSTEM_BLOOD_VESSEL_SMOOTH_MUSCLE = "respiratory system blood vessel smooth muscle"
    RESPIRATORY_SYSTEM_CAPILLARY = "respiratory system capillary"
    RESPIRATORY_SYSTEM_CAPILLARY_ENDOTHELIUM = "respiratory system capillary endothelium"
    RESPIRATORY_SYSTEM_LAMINA_PROPRIA = "respiratory system lamina propria"
    RESPIRATORY_SYSTEM_MUCOSA = "respiratory system mucosa"
    RESPIRATORY_SYSTEM_SUBMUCOSA = "respiratory system submucosa"
    RESPIRATORY_SYSTEM_LYMPHATIC_VESSEL = "respiratory system lymphatic vessel"
    RESPIRATORY_SYSTEM_LYMPHATIC_VESSEL_ENDOTHELIUM = "respiratory system lymphatic vessel endothelium"
    RESPIRATORY_SYSTEM_LYMPHATIC_VESSEL_SMOOTH_MUSCLE = "respiratory system lymphatic vessel smooth muscle"
    RESPIRATORY_SYSTEM_SKELETAL_MUSCLE = "respiratory system skeletal muscle"
    RESPIRATORY_SYSTEM_VENOUS_BLOOD_VESSEL = "respiratory system venous blood vessel"
    RESPIRATORY_SYSTEM_VENOUS_ENDOTHELIUM = "respiratory system venous endothelium"
    RESPIRATORY_SYSTEM_VENOUS_SMOOTH_MUSCLE = "respiratory system venous smooth muscle"
    RESPIRATORY_SYSTEM_VENULE = "respiratory system venule"
    FLEXOR_HALLUCIS_LONGUS = "flexor hallucis longus"
    FOVEA_CENTRALIS = "fovea centralis"
    FRONTAL_ARTERY = "frontal artery"
    FUSIFORM_NUCLEUS = "fusiform nucleus"
    ALIMENTARY_SYSTEM_SEROSA = "alimentary system serosa"
    GRACILE_NUCLEUS = "gracile nucleus"
    GREATER_TROCHANTER = "greater trochanter"
    LESSER_TROCHANTER = "lesser trochanter"
    GRAY_MATTER = "gray matter"
    HEART_AND_PERICARDIUM = "heart and pericardium"
    HEPATIC_DUCT_SMOOTH_MUSCLE = "hepatic duct smooth muscle"
    HILAR_PART_OF_HEPATIC_DUCT = "hilar part of hepatic duct"
    HYOID_APPARATUS = "hyoid apparatus"
    ILEOCECAL_JUNCTION = "ileocecal junction"
    ILIAC_CREST = "iliac crest"
    ILIOCOSTALIS = "iliocostalis"
    ILIOCOSTALIS_THORACIS = "iliocostalis thoracis"
    ILIOTHORACIC = "iliothoracic"
    INFERIOR_EYELID_TARSUS = "inferior eyelid tarsus"
    SUPERIOR_EYELID_TARSUS = "superior eyelid tarsus"
    MEDIAL_PALPEBRAL_LIGAMENT = "medial palpebral ligament"
    ORBITAL_SEPTUM = "orbital septum"
    INFERIOR_OLIVARY_COMMISSURE = "inferior olivary commissure"
    INFERIOR_OLIVE_BETA_SUBNUCLEUS = "inferior olive beta subnucleus"
    INFERIOR_OLIVE_DORSAL_ACCESSORY_NUCLEUS = "inferior olive dorsal accessory nucleus"
    INFERIOR_OLIVE_MEDIAL_ACCESSORY_NUCLEUS = "inferior olive medial accessory nucleus"
    INFERIOR_OLIVE_MEDIAL_NUCLEUS = "inferior olive medial nucleus"
    INFERIOR_OLIVE_PRINCIPAL_NUCLEUS = "inferior olive principal nucleus"
    INFERIOR_PANCREATICO_DUODENAL_VEIN = "inferior pancreatico-duodenal vein"
    SYMPATHETIC_NERVE_PLEXUS = "sympathetic nerve plexus"
    INTERNAL_CAROTID_NERVE_PLEXUS = "internal carotid nerve plexus"
    INTESTINE_SEROSA = "intestine serosa"
    JOINT_ARTICULAR_SURFACE = "joint articular surface"
    JOINT_FIBROUS_CAPSULE = "joint fibrous capsule"
    JOINT_OF_VERTEBRAL_BODY = "joint of vertebral body"
    JUXTAMEDULLARY_CORTEX = "juxtamedullary cortex"
    LACRIMAL_CANALICULUS = "lacrimal canaliculus"
    LAMELLAR_BONE = "lamellar bone"
    LARYNX_LIGAMENT = "larynx ligament"
    LARYNGEAL_EXTRINSIC_LIGAMENT = "laryngeal extrinsic ligament"
    LARYNGEAL_INTRINSIC_LIGAMENT = "laryngeal intrinsic ligament"
    LARYNGEAL_EXTRINSIC_MUSCLE = "laryngeal extrinsic muscle"
    LARYNGEAL_INTRINSIC_MUSCLE = "laryngeal intrinsic muscle"
    LARYNX_MUCOUS_GLAND = "larynx mucous gland"
    SPINAL_CORD_BASAL_NUCLEUS = "spinal cord basal nucleus"
    LATERAL_CERVICAL_NUCLEUS = "lateral cervical nucleus"
    LATERAL_SPINAL_NUCLEUS = "lateral spinal nucleus"
    SPINAL_CORD_DORSAL_NUCLEUS = "spinal cord dorsal nucleus"
    LATERAL_DORSAL_DIGITAL_ARTERY_FOR_DIGIT_5 = "lateral dorsal digital artery for digit 5"
    MEDIAN_DORSAL_DIGITAL_ARTERY_FOR_DIGIT_1 = "median dorsal digital artery for digit 1"
    LATERAL_GLOBUS_PALLIDUS = "lateral globus pallidus"
    MEDIAL_GLOBUS_PALLIDUS = "medial globus pallidus"
    LIMITING_RIDGE_OF_STOMACH = "limiting ridge of stomach"
    LINGUAL_GLAND_DUCT = "lingual gland duct"
    LIP_SKELETAL_MUSCLE = "lip skeletal muscle"
    LIP_SKIN = "lip skin"
    LIVER_LEFT_LATERAL_LOBE = "liver left lateral lobe"
    LIVER_LEFT_MEDIAL_LOBE = "liver left medial lobe"
    LIVER_PAPILLARY_PROCESS = "liver papillary process"
    LOBAR_BRONCHUS_CONNECTIVE_TISSUE = "lobar bronchus connective tissue"
    LONGISSIMUS_MUSCLE = "longissimus muscle"
    LONGISSIMUS_ATLANTIS = "longissimus atlantis"
    LONGISSIMUS_CAPITIS = "longissimus capitis"
    LONGISSIMUS_CERVICIS = "longissimus cervicis"
    LONGISSIMUS_LUMBORUM = "longissimus lumborum"
    LONGISSIMUS_THORACIS = "longissimus thoracis"
    LONGUS_COLLI = "longus colli"
    LUMBAR_SYMPATHETIC_NERVE_TRUNK = "lumbar sympathetic nerve trunk"
    PELVIC_SYMPATHETIC_NERVE_TRUNK = "pelvic sympathetic nerve trunk"
    THORACIC_SYMPATHETIC_NERVE_TRUNK = "thoracic sympathetic nerve trunk"
    LUMBOSACROCAUDALIS = "lumbosacrocaudalis"
    LUNATE = "lunate"
    SCAPHOID = "scaphoid"
    LYMPH_NODE_FOLLICULAR_MANTLE = "lymph node follicular mantle"
    LYMPH_NODE_SUBCORTICAL_SINUS = "lymph node subcortical sinus"
    LYMPH_NODE_T_CELL_DOMAIN = "lymph node T cell domain"
    LYMPHOID_SYSTEM = "lymphoid system"
    LYMPH_ORGAN = "lymph organ"
    MAGNOCELLULAR_NUCLEUS = "magnocellular nucleus"
    OVAL_NUCLEUS = "oval nucleus"
    MAMMARY_SEBACEOUS_GLAND = "mammary sebaceous gland"
    MAMMARY_GLAND_SMOOTH_MUSCLE = "mammary gland smooth muscle"
    MEDIAL_FEMORAL_CIRCUMFLEX_ARTERY = "medial femoral circumflex artery"
    OBTURATOR_ARTERY = "obturator artery"
    OCCIPITAL_ARTERY = "occipital artery"
    OCULAR_ANGLE_ARTERY = "ocular angle artery"
    PERONEAL_ARTERY = "peroneal artery"
    ULNAR_ARTERY = "ulnar artery"
    URETERIC_ARTERY = "ureteric artery"
    OBTURATOR_VEIN = "obturator vein"
    PANCREATIC_VEIN = "pancreatic vein"
    URETERIC_VEIN = "ureteric vein"
    MODIFIED_SEBACEOUS_GLAND = "modified sebaceous gland"
    NASAL_CARTILAGE = "nasal cartilage"
    NASOPHARYNX_CONNECTIVE_TISSUE = "nasopharynx connective tissue"
    NERVE_PLEXUS = "nerve plexus"
    NOSE_SKIN = "nose skin"
    BRACHIOCEPHALIC_MUSCLE = "brachiocephalic muscle"
    CLEIDOBRACHIALIS_MUSCLE = "cleidobrachialis muscle"
    CLEIDOCEPHALICUS_MUSCLE = "cleidocephalicus muscle"
    CLEIDOOCCIPITAL_MUSCLE = "cleidooccipital muscle"
    OMOHYOID = "omohyoid"
    OMOTRANSVERSE = "omotransverse"
    ORBITALIS_MUSCLE = "orbitalis muscle"
    PAROTIDOAURICULAR = "parotidoauricular"
    SARTORIUS = "sartorius"
    SEMISPINALIS_CAPITIS = "semispinalis capitis"
    SEMISPINALIS_CERVICIS = "semispinalis cervicis"
    SEMISPINALIS_THORACIS = "semispinalis thoracis"
    SERRATUS_DORSALIS_CAUDALIS = "serratus dorsalis caudalis"
    SERRATUS_DORSALIS_CRANIALIS = "serratus dorsalis cranialis"
    SPHINCTER_COLLI_SUPERFICIALIS = "sphincter colli superficialis"
    SPINALIS_THORACIS = "spinalis thoracis"
    SPLENIUS = "splenius"
    STERNOOCCIPITAL = "sternooccipital"
    ZYGOMATICUS_MUSCLE = "zygomaticus muscle"
    OPHTHALMIC_PLEXUS = "ophthalmic plexus"
    OROPHARYNX_EPITHELIUM = "oropharynx epithelium"
    OUTER_EAR_CARTILAGE = "outer ear cartilage"
    OUTFLOW_TRACT_PERICARDIUM = "outflow tract pericardium"
    OUTFLOW_TRACT_SMOOTH_MUSCLE = "outflow tract smooth muscle"
    RECTUS_THORACIS = "rectus thoracis"
    PANCREAS_LOBE = "pancreas lobe"
    PANCREAS_LEFT_LOBE = "pancreas left lobe"
    PANCREAS_RIGHT_LOBE = "pancreas right lobe"
    PARARENAL_FAT = "pararenal fat"
    PARIETAL_MESOTHELIUM = "parietal mesothelium"
    PERIANAL_SKIN = "perianal skin"
    PERIFORNICAL_NUCLEUS = "perifornical nucleus"
    PERIPEDUNCULAR_NUCLEUS = "peripeduncular nucleus"
    PHRENIC_NERVE_TRUNK = "phrenic nerve trunk"
    AXILLARY_NERVE_TRUNK = "axillary nerve trunk"
    INTERCOSTAL_NERVE_TRUNK = "intercostal nerve trunk"
    POST_CRANIAL_AXIAL_SKELETON = "post-cranial axial skeleton"
    PRESACRAL_VERTEBRA = "presacral vertebra"
    PRESCAPULAR_LYMPH_NODE = "prescapular lymph node"
    RETROPHARYNGEAL_LYMPH_NODE = "retropharyngeal lymph node"
    SUPERFICIAL_CERVICAL_LYMPH_NODE = "superficial cervical lymph node"
    PRINCIPAL_NUCLEUS = "principal nucleus"
    TRANSVERSE_NUCLEUS = "transverse nucleus"
    PROXIMAL_SESAMOID_BONE_OF_FOOT = "proximal sesamoid bone of foot"
    PROXIMAL_SESAMOID_BONE_OF_HAND = "proximal sesamoid bone of hand"
    RAPHE_OBSCURUS_NUCLEUS = "raphe obscurus nucleus"
    REPRODUCTIVE_ORGAN = "reproductive organ"
    RETROCHIASMATIC_AREA = "retrochiasmatic area"
    RHOMBOID = "rhomboid"
    RIB_DORSAL_SEGMENT = "rib dorsal segment"
    HEAD_OF_RIB = "head of rib"
    TUBERCLE_OF_RIB = "tubercle of rib"
    SCALENUS_DORSALIS = "scalenus dorsalis"
    SCALENUS_MEDIUS = "scalenus medius"
    SEGMENTAL_BRONCHUS = "segmental bronchus"
    SEGMENTAL_BRONCHUS_EPITHELIUM = "segmental bronchus epithelium"
    SENSORY_CIRCUMVENTRICULAR_ORGAN = "sensory circumventricular organ"
    SUPRAOPTIC_CREST = "supraoptic crest"
    SKELETAL_MUSCLE_CONNECTIVE_TISSUE = "skeletal muscle connective tissue"
    SKELETAL_TISSUE = "skeletal tissue"
    SMALL_INTESTINE_MESENTERY = "small intestine mesentery"
    SNOUT_SKIN = "snout skin"
    SPINAL_CORD_CENTRAL_GRAY = "spinal cord central gray"
    SPINAL_CORD_DORSAL_COMMISSURAL_NUCLEUS = "spinal cord dorsal commissural nucleus"
    SPINAL_CORD_RETICULAR_NUCLEUS = "spinal cord reticular nucleus"
    SPLEEN_PERIARTERIOLAR_LYMPHATIC_SHEATH = "spleen periarteriolar lymphatic sheath"
    SPLEEN_VENOUS_SINUS = "spleen venous sinus"
    SPLENIC_CORD = "splenic cord"
    STOMACH_GREATER_CURVATURE = "stomach greater curvature"
    STOMACH_LESSER_CURVATURE = "stomach lesser curvature"
    SUPERFICIAL_PALMAR_ARCH = "superficial palmar arch"
    SUPERIOR_SALIVATORY_NUCLEUS = "superior salivatory nucleus"
    SURAL_NERVE = "sural nerve"
    SYMPHYSIS = "symphysis"
    TAIL_MUSCLE = "tail muscle"
    TAIL_TIP = "tail tip"
    TAIL_VEIN = "tail vein"
    INTERTARSAL_JOINT = "intertarsal joint"
    TERMINAL_BRONCHUS = "terminal bronchus"
    TERMINAL_BRONCHUS_EPITHELIUM = "terminal bronchus epithelium"
    THORACIC_SEGMENT_SKIN = "thoracic segment skin"
    THYMUS_LYMPHOID_TISSUE = "thymus lymphoid tissue"
    THYROID_GLAND_CAPSULE = "thyroid gland capsule"
    THYROID_GLAND_MEDULLA = "thyroid gland medulla"
    THYROID_GLAND_PARENCHYMA = "thyroid gland parenchyma"
    TIBIA_DIAPHYSIS = "tibia diaphysis"
    TIBIA_EPIPHYSIS = "tibia epiphysis"
    TIBIA_METAPHYSIS = "tibia metaphysis"
    TRIGEMINAL_V_PRINCIPAL_SENSORY_NUCLEUS = "trigeminal V principal sensory nucleus"
    TRUE_RIB = "true rib"
    TYMPANIC_CAVITY_BLOOD_VESSEL = "tympanic cavity blood vessel"
    TYMPANIC_CAVITY_MUSCLE = "tympanic cavity muscle"
    TYMPANIC_CAVITY_NERVE = "tympanic cavity nerve"
    TYMPANIC_CAVITY = "tympanic cavity"
    URETER_LUMINAL_UROTHELIUM = "ureter luminal urothelium"
    URETER_SUBLUMINAL_UROTHELIUM = "ureter subluminal urothelium"
    URETERAL_VALVE = "ureteral valve"
    URETEROPELVIC_JUNCTION = "ureteropelvic junction"
    URETEROVESICAL_JUNCTION = "ureterovesical junction"
    URETHRA_GLAND = "urethra gland"
    URETHRA_LAMINA_PROPRIA = "urethra lamina propria"
    URETHRA_MESENCHYMAL_LAYER = "urethra mesenchymal layer"
    URETHRA_MUSCLE = "urethra muscle"
    URETHRA_SKELETAL_MUSCLE = "urethra skeletal muscle"
    URETHRAL_SPHINCTER = "urethral sphincter"
    URETHRALIS = "urethralis"
    URINARY_BLADDER_MUSCLE = "urinary bladder muscle"
    UTERUS_SEROSA = "uterus serosa"
    VAGINA_SEBACEOUS_GLAND = "vagina sebaceous gland"
    VAGINA_SQUAMOUS_EPITHELIUM = "vagina squamous epithelium"
    VAGINAL_HYMEN = "vaginal hymen"
    VAGINAL_VEIN = "vaginal vein"
    VAGUS_NERVE = "vagus nerve"
    LEFT_VAGUS_NERVE_TRUNK = "left vagus nerve trunk"
    RIGHT_VAGUS_NERVE_TRUNK = "right vagus nerve trunk"
    VENA_CAVA_ENDOTHELIUM = "vena cava endothelium"
    VAGINAL_VENOUS_PLEXUS = "vaginal venous plexus"
    VERTEBRA_CRANIAL_ARTICULAR_PROCESS = "vertebra cranial articular process"
    VERTEBRA_NEURAL_CANAL = "vertebra neural canal"
    VISCERAL_MESOTHELIUM = "visceral mesothelium"
    VITREOUS_CHAMBER = "vitreous chamber"
    WHITE_FIBROCARTILAGE = "white fibrocartilage"
    ZYGOMATIC_BONE_TEMPORAL_PROCESS = "zygomatic bone temporal process"
    ZYMBAL_S_GLAND = "Zymbal's gland"
    BRONCHUS_BASEMENT_MEMBRANE = "bronchus basement membrane"
    BRONCHUS_BASAL_LAMINA = "bronchus basal lamina"
    BRONCHUS_RETICULAR_LAMINA = "bronchus reticular lamina"
    CEREBELLAR_GLOMERULUS = "cerebellar glomerulus"
    SUBARACHNOID_CISTERN = "subarachnoid cistern"
    CEREBELLOMEDULLARY_CISTERN = "cerebellomedullary cistern"
    LATERAL_CEREBELLOMEDULLARY_CISTERN = "lateral cerebellomedullary cistern"
    POSTERIOR_CEREBELLOMEDULLARY_CISTERN = "posterior cerebellomedullary cistern"
    QUADRIGEMINAL_CISTERN = "quadrigeminal cistern"
    CEREBELLUM_FISSURE = "cerebellum fissure"
    CEREBELLUM_INTERPOSITUS_NUCLEUS = "cerebellum interpositus nucleus"
    CEREBELLUM_GLOBOSE_NUCLEUS = "cerebellum globose nucleus"
    CEREBELLUM_EMBOLIFORM_NUCLEUS = "cerebellum emboliform nucleus"
    CEREBELLUM_INTERMEDIATE_ZONE = "cerebellum intermediate zone"
    CEREBELLUM_VERMIS_LOBULE_VIIA = "cerebellum vermis lobule VIIA"
    CEREBELLUM_VERMIS_LOBULE_VIIB = "cerebellum vermis lobule VIIB"
    CERVICAL_RIB = "cervical rib"
    NECK_OF_RIB = "neck of rib"
    CHIN = "chin"
    CINGULUM_BUNDLE = "cingulum bundle"
    CIRCLE_OF_WILLIS = "circle of Willis"
    COCHLEA_BASEMENT_MEMBRANE = "cochlea basement membrane"
    EXTERNAL_NOSE = "external nose"
    COLUMELLA_NASI = "columella nasi"
    VENTRICLE_COMPACT_MYOCARDIUM = "ventricle compact myocardium"
    CONJUNCTIVAL_FORNIX = "conjunctival fornix"
    CORNEO_SCLERAL_JUNCTION = "corneo-scleral junction"
    CORONARY_SINUS = "coronary sinus"
    CRISTA_TERMINALIS = "crista terminalis"
    CRYPT_OF_LIEBERKUHN_OF_DUODENUM = "crypt of Lieberkuhn of duodenum"
    CRYPT_OF_LIEBERKUHN_OF_ILEUM = "crypt of Lieberkuhn of ileum"
    CRYPT_OF_LIEBERKUHN_OF_JEJUNUM = "crypt of Lieberkuhn of jejunum"
    CUTANEOUS_ELASTIC_TISSUE = "cutaneous elastic tissue"
    DEPRESSOR_ANGULI_ORIS = "depressor anguli oris"
    ALIMENTARY_SYSTEM_ORGAN = "alimentary system organ"
    DORSAL_TELENCEPHALIC_COMMISSURE = "dorsal telencephalic commissure"
    VENTRAL_TELENCEPHALIC_COMMISSURE = "ventral telencephalic commissure"
    RETICULAR_TISSUE = "reticular tissue"
    ENDONEURIUM = "endoneurium"
    EPINEURIUM = "epineurium"
    PERINEURIUM = "perineurium"
    ENDONEURIAL_FLUID = "endoneurial fluid"
    EPAXIAL_MUSCULATURE = "epaxial musculature"
    HYPAXIAL_MUSCULATURE = "hypaxial musculature"
    BODY_WALL_MUSCULATURE = "body wall musculature"
    GIRDLE_REGION = "girdle region"
    GIRDLE_MUSCULATURE = "girdle musculature"
    PECTORAL_GIRDLE_REGION = "pectoral girdle region"
    PECTORAL_GIRDLE_MUSCULATURE = "pectoral girdle musculature"
    LIMB_GIRDLE_COMPLEX = "limb-girdle complex"
    LIMB_GIRDLE_COMPLEX_MUSCULATURE = "limb-girdle complex musculature"
    PECTORAL_COMPLEX = "pectoral complex"
    PECTORAL_COMPLEX_MUSCULATURE = "pectoral complex musculature"
    PELVIC_GIRDLE_REGION = "pelvic girdle region"
    PELVIC_COMPLEX = "pelvic complex"
    PELVIC_COMPLEX__MUSCULATURE = "pelvic complex  musculature"
    EPENDYMA = "ependyma"
    EPIDERMAL_DERMAL_JUNCTION = "epidermal-dermal junction"
    OVARY_FOLLICULAR_ANTRUM = "ovary follicular antrum"
    FORAMEN_MAGNUM = "foramen magnum"
    FOREHEAD = "forehead"
    GENU_OF_FACIAL_NERVE = "genu of facial nerve"
    NASAL_GLAND = "nasal gland"
    GLYMPHATIC_SYSTEM = "glymphatic system"
    GOLGI_TENDON_ORGAN = "golgi tendon organ"
    GONIAL_BONE = "gonial bone"
    HABENULAR_COMMISSURE = "habenular commissure"
    HAIR_FOLLICLE_BULGE = "hair follicle bulge"
    HAIR_FOLLICLE_ISTHMUS = "hair follicle isthmus"
    HAIR_FOLLICLE_MATRIX_REGION = "hair follicle matrix region"
    HEMOTRICHORIAL_PLACENTAL_MEMBRANE = "hemotrichorial placental membrane"
    HEPATIC_CORD = "hepatic cord"
    INGUINAL_CANAL = "inguinal canal"
    INNER_EAR_CANAL = "inner ear canal"
    INTEGUMENTAL_SYSTEM_ADNEXA = "integumental system adnexa"
    INTERPHALANGEAL_JOINT = "interphalangeal joint"
    COLLECTING_DUCT_EPITHELIUM = "collecting duct epithelium"
    METANEPHROS_CORTICOMEDULLARY_BOUNDARY = "metanephros corticomedullary boundary"
    DORSAL_CORTICOSPINAL_TRACT = "dorsal corticospinal tract"
    LATERAL_CORTICOSPINAL_TRACT = "lateral corticospinal tract"
    VENTRAL_CORTICOSPINAL_TRACT = "ventral corticospinal tract"
    HEAD_OF_FIBULA = "head of fibula"
    NECK_OF_FIBULA = "neck of fibula"
    BODY_OF_FIBULA = "body of fibula"
    LATERAL_MALLEOLUS_OF_FIBULA = "lateral malleolus of fibula"
    LATERAL_NASAL_GLAND = "lateral nasal gland"
    LINE_OF_SCHWALBE = "line of Schwalbe"
    LONG_BONE_EPIPHYSEAL_PLATE_OSSIFICATION_ZONE = "long bone epiphyseal plate ossification zone"
    LYMPH_NODE_MEDULLARY_CORD = "lymph node medullary cord"
    LYMPH_NODE_MEDULLARY_SINUS = "lymph node medullary sinus"
    MAJOR_SUBLINGUAL_DUCT = "major sublingual duct"
    MAMMARY_DUCT_TERMINAL_END_BUD = "mammary duct terminal end bud"
    MAMMARY_FAT_PAD = "mammary fat pad"
    MAMMARY_GLAND_LUMINAL_EPITHELIUM = "mammary gland luminal epithelium"
    MAMMARY_GLAND_MYOEPITHELIUM = "mammary gland myoepithelium"
    MAMMILLOTHALAMIC_TRACT_OF_HYPOTHALAMUS = "mammillothalamic tract of hypothalamus"
    EYELID_MARGIN = "eyelid margin"
    MEDULLARY_RAY = "medullary ray"
    MESANGIAL_MATRIX = "mesangial matrix"
    METOPTIC_PILLAR = "metoptic pillar"
    MINOR_SUBLINGUAL_DUCT = "minor sublingual duct"
    MOLAR_CROWN = "molar crown"
    FLOOR_OF_ORAL_CAVITY = "floor of oral cavity"
    URETHRA_MUCOSA = "urethra mucosa"
    MUCOUS_GLAND = "mucous gland"
    NASAL_CONCHA_OF_ETHMOID_BONE = "nasal concha of ethmoid bone"
    NOSE_TIP = "nose tip"
    OCULAR_FUNDUS = "ocular fundus"
    MAIN_OLFACTORY_BULB_LAYER = "main olfactory bulb layer"
    OLFACTORY_SEGMENT_OF_NASAL_MUCOSA = "olfactory segment of nasal mucosa"
    PRETECTAL_NUCLEUS = "pretectal nucleus"
    ANTERIOR_PRETECTAL_NUCLEUS = "anterior pretectal nucleus"
    OLIVARY_PRETECTAL_NUCLEUS = "olivary pretectal nucleus"
    POSTERIOR_PRETECTAL_NUCLEUS = "posterior pretectal nucleus"
    OSTEOID = "osteoid"
    OTOLITH = "otolith"
    LEFT_VENTRICLE_OUTFLOW_PART = "left ventricle outflow part"
    RIGHT_VENTRICLE_OUTFLOW_PART = "right ventricle outflow part"
    OVARY_CORTEX = "ovary cortex"
    OVARY_MEDULLA = "ovary medulla"
    PALATAL_MUSCLE = "palatal muscle"
    PALATAL_BONE = "palatal bone"
    PALATINE_BONE_HORIZONTAL_PLATE = "palatine bone horizontal plate"
    PALATINE_PROCESS_OF_MAXILLA = "palatine process of maxilla"
    PAROTID_GLAND_ACINUS = "parotid gland acinus"
    PAROTID_GLAND_MYOEPITHELIUM = "parotid gland myoepithelium"
    PECITINATE_MUSCLE = "pecitinate muscle"
    PERIANAL_GLAND = "perianal gland"
    PERIORBITAL_SKIN = "periorbital skin"
    PERIPHERAL_LYMPH_NODE = "peripheral lymph node"
    PEYER_S_PATCH_T_CELL_AREA = "Peyer's patch T cell area"
    PUBIC_SYMPHYSIS = "pubic symphysis"
    LUNG_ACINUS = "lung acinus"
    LUNG_ALVEOLAR_PARENCHYMA = "lung alveolar parenchyma"
    PULMONARY_NEUROENDOCRINE_BODY = "pulmonary neuroendocrine body"
    PYLORIC_CANAL = "pyloric canal"
    PYRAMIDAL_DECUSSATION = "pyramidal decussation"
    RESPIRATORY_CONDUCTING_TUBE = "respiratory conducting tube"
    RETROSPLENIAL_GRANULAR_CORTEX = "retrosplenial granular cortex"
    MOLAR_ROOT = "molar root"
    SCALP = "scalp"
    SEROUS_GLAND = "serous gland"
    SINOTUBULAR_JUNCTION = "sinotubular junction"
    FOREHEAD_SKIN = "forehead skin"
    SOMATIC_MOTOR_SYSTEM = "somatic motor system"
    SOMATIC_SENSORY_SYSTEM = "somatic sensory system"
    SPINAL_CORD_COMMISSURE = "spinal cord commissure"
    SPINAL_CORD_VENTRAL_COMMISSURE = "spinal cord ventral commissure"
    AUTONOMIC_NERVE = "autonomic nerve"
    SYMPATHETIC_NERVE = "sympathetic nerve"
    SPLANCHNIC_NERVE = "splanchnic nerve"
    SPLEEN_PRIMARY_B_FOLLICLE = "spleen primary B follicle"
    SPLEEN_SECONDARY_B_FOLLICLE = "spleen secondary B follicle"
    SPLEEN_B_CELL_CORONA = "spleen B cell corona"
    SPLEEN_FOLLICULAR_DENDRITIC_CELL_NETWORK = "spleen follicular dendritic cell network"
    SPLEEN_PERIFOLLICULAR_ZONE = "spleen perifollicular zone"
    STOMACH_SMOOTH_MUSCLE_INNER_OBLIQUE_LAYER = "stomach smooth muscle inner oblique layer"
    STOMACH_SMOOTH_MUSCLE_OUTER_LONGITUDINAL_LAYER = "stomach smooth muscle outer longitudinal layer"
    SUBARCUATE_FOSSA = "subarcuate fossa"
    SUBENDOCARDIUM = "subendocardium"
    SUBGLOTTIC_REGION = "subglottic region"
    SUBMUCOSAL_GLAND = "submucosal gland"
    TAIL_HAIR = "tail hair"
    MEIBOMIAN_GLAND_ACINUS = "meibomian gland acinus"
    THROAT = "throat"
    THYMUS_CORTICOMEDULLARY_BOUNDARY = "thymus corticomedullary boundary"
    TOOTH_ROOT = "tooth root"
    TUNNEL_OF_CORTI = "tunnel of Corti"
    URETHRAL_MEATUS = "urethral meatus"
    UROTHELIUM = "urothelium"
    UTERINE_FAT_PAD = "uterine fat pad"
    UTRICULOSACCULAR_DUCT = "utriculosaccular duct"
    UTRICLE_DUCT = "utricle duct"
    SACCULE_DUCT = "saccule duct"
    VAGINA_ORIFICE = "vagina orifice"
    VENTRAL_BODY_WALL = "ventral body wall"
    VESTIBULO_AUDITORY_SYSTEM = "vestibulo-auditory system"
    VESTIBULAR_SYSTEM = "vestibular system"
    BLOOD_VESSEL_WALL = "blood vessel wall"
    CHORDAMESODERM = "chordamesoderm"
    PRECHORDAL_MESODERM = "prechordal mesoderm"
    PRECHORDAL_PLATE = "prechordal plate"
    CARDIAC_NEURAL_CREST = "cardiac neural crest"
    CARDIOGENIC_SPLANCHNIC_MESODERM = "cardiogenic splanchnic mesoderm"
    PRIMARY_HEART_FIELD = "primary heart field"
    NEURAXIS_FLEXURE = "neuraxis flexure"
    CEPHALIC_MIDBRAIN_FLEXURE = "cephalic midbrain flexure"
    PONTINE_FLEXURE = "pontine flexure"
    CERVICAL_FLEXURE = "cervical flexure"
    NEPHROGENIC_CORD_ROSTRAL_PART = "nephrogenic cord rostral part"
    NEPHROGENIC_CORD_CAUDAL_PART = "nephrogenic cord caudal part"
    NEPHRON_EPITHELIUM = "nephron epithelium"
    CEREBELLAR_PLATE = "cerebellar plate"
    NEUROMERE = "neuromere"
    NEUROPORE = "neuropore"
    CRUCIATE_LIGAMENT_OF_KNEE = "cruciate ligament of knee"
    MENISCUS_OF_KNEE = "meniscus of knee"
    LATERAL_MENISCUS_OF_KNEE = "lateral meniscus of knee"
    MEDIAL_MENISCUS_OF_KNEE = "medial meniscus of knee"
    DEEP_PERONEAL_NERVE = "deep peroneal nerve"
    SUPERFICIAL_PERONEAL_NERVE = "superficial peroneal nerve"
    LATERAL_FEMORAL_CUTANEOUS_NERVE = "lateral femoral cutaneous nerve"
    UPPER_LIP_SKIN = "upper lip skin"
    LOWER_LIP_SKIN = "lower lip skin"
    PERIDERM = "periderm"
    THYMUS_CORTEX_EPITHELIUM = "thymus cortex epithelium"
    THYMUS_MEDULLA_EPITHELIUM = "thymus medulla epithelium"
    DUODENUM_VILLUS = "duodenum villus"
    JEJUNUM_VILLUS = "jejunum villus"
    ILEUM_VILLUS = "ileum villus"
    GUT_MUCOSA = "gut mucosa"
    GUT_SEROSA = "gut serosa"
    BLADDER_LUMEN = "bladder lumen"
    VERTEBROBASILAR_ARTERIAL_SYSTEM = "vertebrobasilar arterial system"
    CEREBRAL_HEMISPHERE = "cerebral hemisphere"
    LEFT_CEREBRAL_HEMISPHERE = "left cerebral hemisphere"
    RIGHT_CEREBRAL_HEMISPHERE = "right cerebral hemisphere"
    UPPER_JAW_PRIMORDIUM = "upper jaw primordium"
    LOWER_JAW_PRIMORDIUM = "lower jaw primordium"
    LONG_BONE_EPIPHYSEAL_PLATE_RESTING_ZONE = "long bone epiphyseal plate resting zone"
    EXTRACELLULAR_MATRIX = "extracellular matrix"
    BASEMENT_MEMBRANE = "basement membrane"
    BASEMENT_MEMBRANE_OF_EPITHELIUM = "basement membrane of epithelium"
    UTERUS_EPITHELIUM = "uterus epithelium"
    ATLANTO_OCCIPITAL_JOINT = "atlanto-occipital joint"
    OPTIC_CHOROID_PLUS_PIGMENTED_RETINAL_EPITHELIUM = "optic choroid plus pigmented retinal epithelium"
    LYMPH_SAC = "lymph sac"
    MAMMARY_GLAND_PRIMORDIUM = "mammary gland primordium"
    MAMMARY_GLAND_PRIMORDIUM_EPITHELIUM = "mammary gland primordium epithelium"
    MAMMARY_FAT_PAD_PRECURSOR = "mammary fat pad precursor"
    MAMMARY_SPROUT = "mammary sprout"
    CARPAL_METACARPAL_JOINT = "carpal-metacarpal joint"
    TARSAL_METATARSAL_JOINT = "tarsal-metatarsal joint"
    PROXIMAL_INTERPHALANGEAL_JOINT = "proximal interphalangeal joint"
    DISTAL_INTERPHALANGEAL_JOINT = "distal interphalangeal joint"
    FORELIMB_PROXIMAL_INTERPHALANGEAL_JOINT = "forelimb proximal interphalangeal joint"
    FORELIMB_DISTAL_INTERPHALANGEAL_JOINT = "forelimb distal interphalangeal joint"
    HINDLIMB_PROXIMAL_INTERPHALANGEAL_JOINT = "hindlimb proximal interphalangeal joint"
    HINDLIMB_DISTAL_INTERPHALANGEAL_JOINT = "hindlimb distal interphalangeal joint"
    JOINT_OF_AUDITORY_OSSICLE = "joint of auditory ossicle"
    INCUDOMALLEAL_JOINT = "incudomalleal joint"
    HEART_TRABECULA = "heart trabecula"
    ATRIAL_TRABECULA_CARDIAC_MUSCLE = "atrial trabecula cardiac muscle"
    ATRIOVENTRICULAR_SULCUS = "atrioventricular sulcus"
    LIMB_TENDON = "limb tendon"
    TENDON_PRIMORDIUM = "tendon primordium"
    LIMB_TENDON_PRIMORDIUM = "limb tendon primordium"
    FORELIMB_TENDON_PRIMORDIUM = "forelimb tendon primordium"
    HINDLIMB_TENDON_PRIMORDIUM = "hindlimb tendon primordium"
    FORELIMB_TENDON = "forelimb tendon"
    HINDLIMB_TENDON = "hindlimb tendon"
    FORELIMB_JOINT_PRIMORDIUM = "forelimb joint primordium"
    HINDLIMB_JOINT_PRIMORDIUM = "hindlimb joint primordium"
    HAND_JOINT_PRIMORDIUM = "hand joint primordium"
    FOOT_JOINT_PRIMORDIUM = "foot joint primordium"
    ANTERIOR_CRUCIATE_LIGAMENT = "anterior cruciate ligament"
    POSTERIOR_CRUCIATE_LIGAMENT = "posterior cruciate ligament"
    INTESTINE_SMOOTH_MUSCLE_CIRCULAR_LAYER = "intestine smooth muscle circular layer"
    INTESTINE_SMOOTH_MUSCLE_LONGITUDINAL_LAYER = "intestine smooth muscle longitudinal layer"
    SECONDARY_MOTOR_CORTEX = "secondary motor cortex"
    ECTORHINAL_AREA = "ectorhinal area"
    INFRALIMBIC_AREA = "infralimbic area"
    ORBITAL_AREA = "orbital area"
    ANTERIOR_MEDIAL_THALAMIC_NUCLEUS = "anterior medial thalamic nucleus"
    VESTIBULOCEREBELLAR_NUCLEUS = "vestibulocerebellar nucleus"
    LATERAL_DORSAL_THALAMIC_NUCLEUS = "lateral dorsal thalamic nucleus"
    LATERAL_POSTERIOR_THALAMIC_NUCLEUS = "lateral posterior thalamic nucleus"
    VENTRAL_MEDIAL_THALAMIC_NUCLEUS = "ventral medial thalamic nucleus"
    CRANIAL_CARTILAGE = "cranial cartilage"
    TECTUM_TRANSVERSUM = "tectum transversum"
    CERVICAL_LOOP = "cervical loop"
    STELLATE_RETICULUM = "stellate reticulum"
    STRATUM_INTERMEDIUM = "stratum intermedium"
    HERTWIG_S_EPITHELIAL_ROOT_SHEATH = "Hertwig's epithelial root sheath"
    ENAMEL_EPITHELIUM = "enamel epithelium"
    INNER_ENAMEL_EPITHELIUM = "inner enamel epithelium"
    OUTER_ENAMEL_EPITHELIUM = "outer enamel epithelium"
    DENTAL_ALVEOLUS = "dental alveolus"
    INTERALVEOLAR_BONE = "interalveolar bone"
    EARLY_BUD_STAGE_TOOTH = "early bud stage tooth"
    LATE_BUD_STAGE_TOOTH = "late bud stage tooth"
    ENAMEL_KNOT = "enamel knot"
    PRIMARY_ENAMEL_KNOT = "primary enamel knot"
    SECONDARY_ENAMEL_KNOT = "secondary enamel knot"
    TERTIARY_ENAMEL_KNOT = "tertiary enamel knot"
    TENSOR_VELI_PALATINI = "tensor veli palatini"
    LEVATOR_VELI_PALATINI = "levator veli palatini"
    PALATOPHARYNGEUS = "palatopharyngeus"
    LIMB_PRIMORDIUM = "limb primordium"
    SMALL_INTESTINE_VASCULAR_ELEMENT = "small intestine vascular element"
    SPLEEN_VASCULAR_ELEMENT = "spleen vascular element"
    INTESTINE_VASCULAR_ELEMENT = "intestine vascular element"
    LATERAL_NASAL_GLAND_2 = "lateral nasal gland 2"
    LATERAL_NASAL_GLAND_3 = "lateral nasal gland 3"
    LATERAL_NASAL_GLAND_4 = "lateral nasal gland 4"
    LATERAL_NASAL_GLAND_5 = "lateral nasal gland 5"
    LATERAL_NASAL_GLAND_DUCT = "lateral nasal gland duct"
    LATERAL_NASAL_GLAND_EPITHELIUM = "lateral nasal gland epithelium"
    LATERAL_NASAL_GLAND_MESENCHYME = "lateral nasal gland mesenchyme"
    MAXILLARY_SINUS_GLAND_PRIMORDIUM = "maxillary sinus gland primordium"
    MAXILLARY_SINUS_GLAND_PRIMORDIUM_EPITHELIUM = "maxillary sinus gland primordium epithelium"
    MAXILLARY_SINUS_GLAND_PRIMORDIUM_MESENCHYME = "maxillary sinus gland primordium mesenchyme"
    MAXILLARY_SINUS_GLAND = "maxillary sinus gland"
    MAXILLARY_SINUS_GLAND_EPITHELIUM = "maxillary sinus gland epithelium"
    MAXILLARY_SINUS_GLAND_MESENCHYME = "maxillary sinus gland mesenchyme"
    MEDIAL_NASAL_GLAND = "medial nasal gland"
    MEDIAL_NASAL_GLAND_EPITHELIUM = "medial nasal gland epithelium"
    MEDIAL_NASAL_GLAND_MESENCHYME = "medial nasal gland mesenchyme"
    STENO_S_GLAND = "Steno's gland"
    STENO_S_DUCT = "Steno's duct"
    STENO_S_GLAND_EPITHELIUM = "Steno's gland epithelium"
    STENO_S_GLAND_MESENCHYME = "Steno's gland mesenchyme"
    MAXILLARY_SINUS_EPITHELIUM = "maxillary sinus epithelium"
    MAXILLARY_SINUS_MESENCHYME = "maxillary sinus mesenchyme"
    NASAL_GLAND_EPITHELIUM = "nasal gland epithelium"
    NASAL_GLAND_MESENCHYME = "nasal gland mesenchyme"
    PARATHYROID_PRIMORDIUM = "parathyroid primordium"
    ADRENOGONADAL_PRIMORDIUM = "adrenogonadal primordium"
    ADRENAL_PRIMORDIUM = "adrenal primordium"
    TESTIS_EPITHELIUM = "testis epithelium"
    FOREGUT_GLAND_PRIMORDIUM = "foregut gland primordium"
    MANDIBLE_CONDYLAR_CONDENSATION = "mandible condylar condensation"
    MANDIBULAR_FOSSA = "mandibular fossa"
    EMBRYONIC_VISCERAL_ENDODERM = "embryonic visceral endoderm"
    EMBRYONIC_POSTERIOR_VISCERAL_ENDODERM = "embryonic posterior visceral endoderm"
    EXTRAEMBRYONIC_POSTERIOR_VISCERAL_ENDODERM = "extraembryonic posterior visceral endoderm"
    ALLANTOIS_ASSOCIATED_POSTERIOR_VISCERAL_ENDODERM = "allantois-associated posterior visceral endoderm"
    ALLANTOIC_CORE_DOMAIN = "Allantoic Core Domain"
    EXTRAEMBRYONIC_PRIMITIVE_STREAK = "extraembryonic primitive streak"
    LENS_PRIMORDIUM = "lens primordium"
    POSTERIOR_CROWN_OF_NODE = "posterior crown of node"
    MARGINAL_VEIN_OF_LIMB = "marginal vein of limb"
    CORTICAL_HEM = "cortical hem"
    NUCLEUS_OF_LATERAL_LEMNISCUS = "nucleus of lateral lemniscus"
    DORSAL_NUCLEUS_OF_LATERAL_LEMNISCUS = "dorsal nucleus of lateral lemniscus"
    INTERMEDIATE_NUCLEUS_OF_LATERAL_LEMNISCUS = "intermediate nucleus of lateral lemniscus"
    VENTRAL_NUCLEUS_OF_LATERAL_LEMNISCUS = "ventral nucleus of lateral lemniscus"
    GENITAL_TUBERCLE_SURFACE_EPITHELIUM = "genital tubercle surface epithelium"
    OVIDUCT_ISTHMUS = "oviduct isthmus"
    OVIDUCT_AMPULLA = "oviduct ampulla"
    PARAMEDIAN_RAPHE_NUCLEUS = "paramedian raphe nucleus"
    PARVICELLULAR_RETICULAR_NUCLEUS = "parvicellular reticular nucleus"
    CEREBRAL_PEDUNCLE = "cerebral peduncle"
    LOWER_UROGENITAL_TRACT = "lower urogenital tract"
    LOWER_UROGENITAL_TRACT_ADVENTITIA = "lower urogenital tract adventitia"
    LOWER_UROGENITAL_TRACT_EPITHELIUM = "lower urogenital tract epithelium"
    LOWER_UROGENITAL_TRACT_MESENCHYME = "lower urogenital tract mesenchyme"
    LOWER_UROGENITAL_TRACT_MUSCLE = "lower urogenital tract muscle"
    LOWER_UROGENITAL_TRACT_NERVE = "lower urogenital tract nerve"
    LOWER_UROGENITAL_TRACT_VASCULATURE = "lower urogenital tract vasculature"
    ROSTRAL_MIGRATORY_STREAM = "rostral migratory stream"
    LOWER_UROGENITAL_TRACT_OF_FEMALE = "lower urogenital tract of female"
    LOWER_UROGENITAL_TRACT_OF_MALE = "lower urogenital tract of male"
    FEMALE_LOWER_UROGENITAL_TRACT_EPITHELIUM = "female lower urogenital tract epithelium"
    FEMALE_LOWER_UROGENITAL_TRACT_MESENCHYME = "female lower urogenital tract mesenchyme"
    FEMALE_LOWER_UROGENITAL_TRACT_MUSCLE = "female lower urogenital tract muscle"
    MALE_LOWER_UROGENITAL_TRACT_EPITHELIUM = "male lower urogenital tract epithelium"
    MALE_LOWER_UROGENITAL_TRACT_MESENCHYME = "male lower urogenital tract mesenchyme"
    MALE_LOWER_UROGENITAL_TRACT_MUSCLE = "male lower urogenital tract muscle"
    CILIARY_ZONULE = "ciliary zonule"
    EXTERNAL_GENITALIA = "external genitalia"
    PREPUCE = "prepuce"
    SKELETAL_JOINT_PRIMORDIUM = "skeletal joint primordium"
    ANTERIOR_PRE_PLACODAL_REGION = "anterior pre-placodal region"
    POSTERIOR_PRE_PLACODAL_REGION = "posterior pre-placodal region"
    VENTRICLE_EPENDYMA = "ventricle ependyma"
    LOWER_URINARY_TRACT_EPITHELIUM = "lower urinary tract epithelium"
    LOWER_URINARY_TRACT_MESENCHYME = "lower urinary tract mesenchyme"
    DISTAL_GENITAL_TUBERCLE = "distal genital tubercle"
    GLANS_OF_GENITAL_TUBERCLE = "glans of genital tubercle"
    PROXIMAL_GENITAL_TUBERCLE = "proximal genital tubercle"
    NUCLEUS_INCERTUS = "nucleus incertus"
    HAIR_PEG = "hair peg"
    FACIAL_PROMINENCE = "facial prominence"
    FACIAL_PROMINENCE_ECTODERM = "facial prominence ectoderm"
    FACIAL_PROMINENCE_MESENCHYME = "facial prominence mesenchyme"
    MANDIBULAR_PROCESS_ECTODERM = "mandibular process ectoderm"
    MAXILLARY_PROCESS_ECTODERM = "maxillary process ectoderm"
    EPIDIDYMIS_INITIAL_SEGMENT = "epididymis initial segment"
    GESCHMACKSSTREIFEN = "geschmacksstreifen"
    PALLIUM = "pallium"
    LATERAL_PALLIUM = "lateral pallium"
    VENTRAL_PALLIUM = "ventral pallium"
    SUBPALLIUM = "subpallium"
    DIAGONAL_DOMAIN = "diagonal domain"
    PROXIMAL_STRAIGHT_TUBULE = "proximal straight tubule"
    OOCYTE_OF_PREANTRAL_FOLLICLE = "oocyte of preantral follicle"
    OOCYTE_OF_ANTRAL_FOLLICLE = "oocyte of antral follicle"
    FOLLICLE_CELL_OF_PREANTRAL_FOLLICLE = "follicle cell of preantral follicle"
    DORSAL_AORTA_ENDOTHELIUM = "dorsal aorta endothelium"
    EPIDERMIS_INTERMEDIATE_LAYER = "epidermis intermediate layer"
    AORTA_GONAD_MESONEPHROS = "aorta-gonad-mesonephros"
    DENTATE_NOTCH = "dentate notch"
    PARATENIAL_NUCLEUS = "paratenial nucleus"
    ANTERIOR_NEURAL_RIDGE = "anterior neural ridge"
    JUNCTIONAL_ZONE = "junctional zone"
    TEMPORO_MANDIBULAR_JOINT_DISC = "temporo-mandibular joint disc"
    ENTOPEDUNCULAR_NUCLEUS = "entopeduncular nucleus"
    CRANIOFACIAL_SUTURE = "craniofacial suture"
    FACIAL_SUTURE = "facial suture"
    FONTANELLE = "fontanelle"
    CORONAL_SUTURE_FRONTAL_OSTEOGENIC_FRONT = "coronal suture frontal osteogenic front"
    CORONAL_SUTURE_MESENCHYME = "coronal suture mesenchyme"
    CORONAL_SUTURE_PARIETAL_OSTEOGENIC_FRONT = "coronal suture parietal osteogenic front"
    FRONTAL_SUTURE_FRONTAL_OSTEOGENIC_FRONT = "frontal suture frontal osteogenic front"
    FRONTAL_SUTURE_MESENCHYME = "frontal suture mesenchyme"
    LAMBDOID_SUTURE_INTERPARIETAL_OSTEOGENIC_FRONT = "lambdoid suture interparietal osteogenic front"
    LAMBDOID_SUTURE_MESENCHYME = "lambdoid suture mesenchyme"
    LAMBDOID_SUTURE_PARIETAL_OSTEOGENIC_FRONT = "lambdoid suture parietal osteogenic front"
    SAGITTAL_SUTURE_MESENCHYME = "sagittal suture mesenchyme"
    SAGITTAL_SUTURE_PARIETAL_OSTEOGENIC_FRONT = "sagittal suture parietal osteogenic front"
    SALIVARY_GLAND_PRIMORDIUM = "salivary gland primordium"
    SALIVARY_GLAND_PRIMORDIUM_EPITHELIUM = "salivary gland primordium epithelium"
    SALIVARY_GLAND_PRIMORDIUM_MESENCHYME = "salivary gland primordium mesenchyme"
    HUMERUS_EPIPHYSIS = "humerus epiphysis"
    MUSCLE_BLOOD_VESSEL = "muscle blood vessel"
    SKIN_BLOOD_VESSEL = "skin blood vessel"
    THYMUS_BLOOD_VESSEL = "thymus blood vessel"
    BODY_OF_STOMACH_GASTRIC_GLAND = "body of stomach gastric gland"
    GUBERNACULUM = "gubernaculum"
    GUBERNACULUM_OF_FEMALE = "gubernaculum of female"
    VON_EBNER_S_GLAND = "von Ebner's gland"
    INTERFOLLICULAR_EPIDERMIS = "interfollicular epidermis"
    DERMO_EPIDERMAL_JUNCTION = "dermo-epidermal junction"
    CRANIAL_SUTURE_MESENCHYME = "cranial suture mesenchyme"
    CRANIAL_SUTURE_OSTEOGENIC_FRONT = "cranial suture osteogenic front"
    CORONAL_SUTURE_OSTEOGENIC_FRONT = "coronal suture osteogenic front"
    LAMBDOID_SUTURE_OSTEOGENIC_FRONT = "lambdoid suture osteogenic front"
    EXOCOELOMIC_CAVITY_FLUID = "exocoelomic cavity fluid"
    CORTICAL_ANTIHEM = "cortical antihem"
    BASAL_CELL_LAYER_OF_PELVIC_URETHRA = "basal cell layer of pelvic urethra"
    INTERMEDIATE_CELL_LAYER_OF_PELVIC_URETHRA = "intermediate cell layer of pelvic urethra"
    SUPERFICIAL_CELL_LAYER_OF_PELVIC_URETHRA = "superficial cell layer of pelvic urethra"
    SPINAL_NERVE_ROOT = "spinal nerve root"
    PALEOCORTEX = "paleocortex"


class MouseEmgMuscles(metaclass=MouseAnatomyMeta):
    """EMG muscles"""

    DELTOID = "deltoid"
    PECTORALIS_MAJOR = "pectoralis major"
    TRICEPS_BRACHII = "triceps brachii"
    LATERAL_HEAD_OF_TRICEPS_BRACHII = "lateral head of triceps brachii"
    LONG_HEAD_OF_TRICEPS_BRACHII = "long head of triceps brachii"
    MEDIAL_HEAD_OF_TRICEPS_BRACHII = "medial head of triceps brachii"
    BICEPS_BRACHII = "biceps brachii"
    LONG_HEAD_OF_BICEPS_BRACHII = "long head of biceps brachii"
    SHORT_HEAD_OF_BICEPS_BRACHII = "short head of biceps brachii"
    TENDON_OF_BICEPS_BRACHII = "tendon of biceps brachii"
    PARS_SCAPULARIS_OF_DELTOID = "pars scapularis of deltoid"
    EXTENSOR_CARPI_RADIALIS_LONGUS = "extensor carpi radialis longus"
    EXTENSOR_DIGITORUM_COMMUNIS = "extensor digitorum communis"
    EXTENSOR_DIGITORUM_LATERALIS = "extensor digitorum lateralis"
    EXTENSOR_CARPI_ULNARIS = "extensor carpi ulnaris"
    FLEXOR_CARPI_RADIALIS = "flexor carpi radialis"
    FLEXOR_CARPI_ULNARIS = "flexor carpi ulnaris"
    FLEXOR_DIGITORUM_PROFUNDUS = "flexor digitorum profundus"


class MouseBodyParts(metaclass=MouseAnatomyMeta):
    """Body parts"""

    FORELIMB = "forelimb"
    HEAD = "head"
    HINDLIMB = "hindlimb"
    NECK = "neck"
    TAIL = "tail"
    TRUNK = "trunk"


class GroundWireLocations(metaclass=MouseAnatomyMeta):
    """Ground wire locations"""

    FORELIMB = "forelimb"
    HEAD = "head"
    HINDLIMB = "hindlimb"
    NECK = "neck"
    TAIL = "tail"
    TRUNK = "trunk"
    BRAIN = "brain"
    CRANIUM = "cranium"


class MouseBloodVessels(metaclass=MouseAnatomyMeta):
    """Blood vessels"""

    CAROTID_ARTERY = "carotid artery"
    JUGULAR_VEIN = "jugular vein"


class InjectionTargets(metaclass=MouseAnatomyMeta):
    """Common injection targets"""

    VENOUS_SINUS = "venous sinus"
    PERITONEAL_CAVITY = "peritoneal cavity"
