import numpy as np
import pandas as pd


def read_pfam(path: str) -> pd.DataFrame:
    """Read the PFAM file.

    Args:
        path: The path to the PFAM file.
    """
    dtype = {
        'seq_id': object,
        'aln_start': np.uintc,
        'aln_end': np.uintc,
        'env_start': np.uintc,
        'env_end': np.uintc,
        'hmm_acc': object,
        'hmm_name': object,
        'type': object,
        'hmm_start': np.uintc,
        'hmm_end': np.uintc,
        'hmm_length': np.float64,
        'bit_score': np.float64,
        'e_value': np.float64,
        'significance': np.float64,
        'clan': object
    }
    lines = list()
    with open(path, 'r') as f:
        for line in f.readlines():
            line = line.strip()
            if line.startswith('#') or line == '':
                continue
            lines.append(line.split())
    return pd.DataFrame(lines, columns=dtype)


def read_pfam_tophit(path: str) -> pd.DataFrame:
    """Read the PFAM tophit file.

    Args:
        path: The path to the PFAM tophit file.
    """
    dtype = {
        'seq_id': object,
        'marker': object,
        'e_value': np.float64,
        'bit_score': np.float64,
    }
    lines = list()
    with open(path, 'r') as f:
        f.readline()
        for line in f.readlines():
            line = line.strip()
            gene_id, hits = line.split('\t')
            for hit in hits.split(';'):
                pfam_acc, e_val, bit_score = hit.split(',')
                lines.append([gene_id, pfam_acc, e_val, bit_score])
    return pd.DataFrame(lines, columns=dtype)


#: Markers used in the PFAM 27 release.
PFAM_27 = frozenset(
    {'PF00001.16', 'PF00002.19', 'PF00003.17', 'PF00004.24', 'PF00005.22', 'PF00006.20', 'PF00007.17', 'PF00008.22',
     'PF00009.22', 'PF00010.21', 'PF00011.16', 'PF00012.15', 'PF00013.24', 'PF00014.18', 'PF00015.16', 'PF00016.15',
     'PF00017.19', 'PF00018.23', 'PF00019.15', 'PF00020.13', 'PF00021.16', 'PF00022.14', 'PF00023.25', 'PF00024.21',
     'PF00025.16', 'PF00026.18', 'PF00027.24', 'PF00028.12', 'PF00029.14', 'PF00030.14', 'PF00031.16', 'PF00032.12',
     'PF00033.14', 'PF00034.16', 'PF00035.20', 'PF00036.27', 'PF00037.22', 'PF00038.16', 'PF00039.13', 'PF00040.14',
     'PF00041.16', 'PF00042.17', 'PF00043.20', 'PF00044.19', 'PF00045.14', 'PF00046.24', 'PF00047.20', 'PF00048.15',
     'PF00049.13', 'PF00050.16', 'PF00051.13', 'PF00052.13', 'PF00053.19', 'PF00054.18', 'PF00055.12', 'PF00056.18',
     'PF00057.13', 'PF00058.12', 'PF00059.16', 'PF00060.21', 'PF00061.18', 'PF00062.15', 'PF00063.16', 'PF00064.13',
     'PF00066.12', 'PF00067.17', 'PF00068.14', 'PF00069.20', 'PF00070.22', 'PF00071.17', 'PF00072.19', 'PF00073.15',
     'PF00074.15', 'PF00075.19', 'PF00076.17', 'PF00077.15', 'PF00078.22', 'PF00079.15', 'PF00080.15', 'PF00081.17',
     'PF00082.17', 'PF00083.19', 'PF00084.15', 'PF00085.15', 'PF00086.13', 'PF00087.16', 'PF00088.13', 'PF00089.21',
     'PF00090.14', 'PF00091.20', 'PF00092.23', 'PF00093.13', 'PF00094.20', 'PF00095.16', 'PF00096.21', 'PF00097.20',
     'PF00098.18', 'PF00100.18', 'PF00101.15', 'PF00102.22', 'PF00103.15', 'PF00104.25', 'PF00105.13', 'PF00106.20',
     'PF00107.21', 'PF00108.18', 'PF00109.21', 'PF00110.14', 'PF00111.22', 'PF00112.18', 'PF00113.17', 'PF00114.14',
     'PF00115.15', 'PF00116.15', 'PF00117.23', 'PF00118.19', 'PF00119.15', 'PF00120.19', 'PF00121.13', 'PF00122.15',
     'PF00123.15', 'PF00124.14', 'PF00125.19', 'PF00126.22', 'PF00127.15', 'PF00128.19', 'PF00129.13', 'PF00130.17',
     'PF00131.15', 'PF00132.19', 'PF00133.17', 'PF00134.18', 'PF00135.23', 'PF00136.16', 'PF00137.16', 'PF00139.14',
     'PF00140.15', 'PF00141.18', 'PF00142.13', 'PF00143.14', 'PF00144.19', 'PF00145.12', 'PF00146.16', 'PF00147.13',
     'PF00148.14', 'PF00149.23', 'PF00150.13', 'PF00151.14', 'PF00152.15', 'PF00153.22', 'PF00154.16', 'PF00155.16',
     'PF00156.22', 'PF00157.12', 'PF00158.21', 'PF00159.13', 'PF00160.16', 'PF00161.14', 'PF00162.14', 'PF00163.14',
     'PF00164.20', 'PF00165.18', 'PF00166.16', 'PF00167.13', 'PF00168.25', 'PF00169.24', 'PF00170.16', 'PF00171.17',
     'PF00172.13', 'PF00173.23', 'PF00174.14', 'PF00175.16', 'PF00176.18', 'PF00177.16', 'PF00178.17', 'PF00179.21',
     'PF00180.15', 'PF00181.18', 'PF00182.14', 'PF00183.13', 'PF00184.12', 'PF00185.19', 'PF00186.14', 'PF00187.14',
     'PF00188.21', 'PF00189.15', 'PF00190.17', 'PF00191.15', 'PF00193.12', 'PF00194.16', 'PF00195.14', 'PF00196.14',
     'PF00197.13', 'PF00198.18', 'PF00199.14', 'PF00200.18', 'PF00201.13', 'PF00202.16', 'PF00203.16', 'PF00204.20',
     'PF00205.17', 'PF00206.15', 'PF00207.17', 'PF00208.16', 'PF00209.13', 'PF00210.19', 'PF00211.15', 'PF00212.13',
     'PF00213.13', 'PF00214.14', 'PF00215.19', 'PF00216.16', 'PF00217.14', 'PF00218.16', 'PF00219.13', 'PF00220.12',
     'PF00221.14', 'PF00223.14', 'PF00224.16', 'PF00225.18', 'PF00226.26', 'PF00227.21', 'PF00228.15', 'PF00229.13',
     'PF00230.15', 'PF00231.14', 'PF00232.13', 'PF00233.14', 'PF00234.17', 'PF00235.14', 'PF00236.13', 'PF00237.14',
     'PF00238.14', 'PF00239.16', 'PF00240.18', 'PF00241.15', 'PF00242.12', 'PF00243.13', 'PF00244.15', 'PF00245.15',
     'PF00246.19', 'PF00248.16', 'PF00249.26', 'PF00250.13', 'PF00251.15', 'PF00252.13', 'PF00253.16', 'PF00254.23',
     'PF00255.14', 'PF00257.14', 'PF00258.20', 'PF00260.15', 'PF00261.15', 'PF00262.13', 'PF00263.16', 'PF00264.15',
     'PF00265.13', 'PF00266.14', 'PF00267.16', 'PF00268.16', 'PF00269.15', 'PF00270.24', 'PF00271.26', 'PF00272.14',
     'PF00273.15', 'PF00274.14', 'PF00275.15', 'PF00276.15', 'PF00277.13', 'PF00278.17', 'PF00280.13', 'PF00281.14',
     'PF00282.14', 'PF00283.14', 'PF00284.15', 'PF00285.16', 'PF00286.15', 'PF00287.13', 'PF00288.21', 'PF00289.17',
     'PF00290.15', 'PF00291.20', 'PF00292.13', 'PF00293.23', 'PF00294.19', 'PF00295.12', 'PF00296.15', 'PF00297.17',
     'PF00298.14', 'PF00299.13', 'PF00300.17', 'PF00301.15', 'PF00302.13', 'PF00303.14', 'PF00304.15', 'PF00305.14',
     'PF00306.22', 'PF00307.26', 'PF00308.13', 'PF00309.15', 'PF00310.16', 'PF00311.12', 'PF00312.17', 'PF00313.17',
     'PF00314.12', 'PF00316.15', 'PF00317.16', 'PF00318.15', 'PF00319.13', 'PF00320.22', 'PF00321.12', 'PF00322.12',
     'PF00323.14', 'PF00324.16', 'PF00325.15', 'PF00326.16', 'PF00327.15', 'PF00328.17', 'PF00329.14', 'PF00330.15',
     'PF00331.15', 'PF00332.13', 'PF00333.15', 'PF00334.14', 'PF00335.15', 'PF00336.13', 'PF00337.17', 'PF00338.17',
     'PF00339.24', 'PF00340.14', 'PF00341.12', 'PF00342.14', 'PF00343.15', 'PF00344.15', 'PF00345.15', 'PF00346.14',
     'PF00347.18', 'PF00348.12', 'PF00349.16', 'PF00350.18', 'PF00351.16', 'PF00352.16', 'PF00353.14', 'PF00354.12',
     'PF00355.21', 'PF00356.16', 'PF00357.15', 'PF00358.15', 'PF00359.17', 'PF00360.15', 'PF00361.15', 'PF00362.13',
     'PF00363.13', 'PF00364.17', 'PF00365.15', 'PF00366.15', 'PF00367.15', 'PF00368.13', 'PF00370.16', 'PF00372.14',
     'PF00373.13', 'PF00374.14', 'PF00375.13', 'PF00376.18', 'PF00377.13', 'PF00378.15', 'PF00379.18', 'PF00380.14',
     'PF00381.14', 'PF00382.14', 'PF00383.17', 'PF00384.17', 'PF00385.19', 'PF00386.16', 'PF00387.14', 'PF00388.14',
     'PF00389.25', 'PF00390.14', 'PF00391.18', 'PF00392.16', 'PF00393.14', 'PF00394.17', 'PF00395.15', 'PF00396.13',
     'PF00397.21', 'PF00398.15', 'PF00399.14', 'PF00400.27', 'PF00401.15', 'PF00402.13', 'PF00403.21', 'PF00404.13',
     'PF00405.12', 'PF00406.17', 'PF00407.14', 'PF00408.15', 'PF00410.14', 'PF00411.14', 'PF00412.17', 'PF00413.19',
     'PF00414.12', 'PF00415.13', 'PF00416.17', 'PF00418.14', 'PF00419.15', 'PF00420.19', 'PF00421.14', 'PF00423.14',
     'PF00424.13', 'PF00425.13', 'PF00426.13', 'PF00427.16', 'PF00428.14', 'PF00429.14', 'PF00430.13', 'PF00431.15',
     'PF00432.16', 'PF00433.19', 'PF00434.13', 'PF00435.16', 'PF00436.20', 'PF00437.15', 'PF00438.15', 'PF00439.20',
     'PF00440.18', 'PF00441.19', 'PF00443.24', 'PF00444.13', 'PF00445.13', 'PF00446.12', 'PF00447.12', 'PF00448.17',
     'PF00449.15', 'PF00450.17', 'PF00451.14', 'PF00452.14', 'PF00453.13', 'PF00454.22', 'PF00455.17', 'PF00456.16',
     'PF00457.12', 'PF00458.15', 'PF00459.20', 'PF00460.15', 'PF00462.19', 'PF00463.16', 'PF00464.14', 'PF00465.14',
     'PF00466.15', 'PF00467.24', 'PF00468.12', 'PF00469.15', 'PF00471.15', 'PF00472.15', 'PF00473.12', 'PF00474.12',
     'PF00475.13', 'PF00476.15', 'PF00477.12', 'PF00478.20', 'PF00479.17', 'PF00480.15', 'PF00481.16', 'PF00482.18',
     'PF00483.18', 'PF00484.14', 'PF00485.13', 'PF00486.23', 'PF00487.19', 'PF00488.16', 'PF00489.12', 'PF00490.16',
     'PF00491.16', 'PF00493.18', 'PF00494.14', 'PF00496.17', 'PF00497.15', 'PF00498.21', 'PF00499.15', 'PF00500.13',
     'PF00501.23', 'PF00502.14', 'PF00503.15', 'PF00504.16', 'PF00505.14', 'PF00506.13', 'PF00507.14', 'PF00508.12',
     'PF00509.13', 'PF00510.13', 'PF00511.12', 'PF00512.20', 'PF00513.13', 'PF00514.18', 'PF00515.23', 'PF00516.13',
     'PF00517.12', 'PF00518.12', 'PF00519.12', 'PF00520.26', 'PF00521.15', 'PF00522.13', 'PF00523.13', 'PF00524.13',
     'PF00525.13', 'PF00526.13', 'PF00527.13', 'PF00528.17', 'PF00529.15', 'PF00530.13', 'PF00531.17', 'PF00532.16',
     'PF00533.21', 'PF00534.15', 'PF00535.21', 'PF00536.25', 'PF00537.13', 'PF00538.14', 'PF00539.13', 'PF00540.13',
     'PF00541.12', 'PF00542.14', 'PF00543.17', 'PF00544.14', 'PF00545.15', 'PF00547.13', 'PF00548.15', 'PF00549.14',
     'PF00550.20', 'PF00551.14', 'PF00552.16', 'PF00553.14', 'PF00554.17', 'PF00555.14', 'PF00556.15', 'PF00557.19',
     'PF00558.14', 'PF00559.12', 'PF00560.28', 'PF00561.15', 'PF00562.23', 'PF00563.15', 'PF00564.19', 'PF00565.12',
     'PF00566.13', 'PF00567.19', 'PF00568.18', 'PF00569.12', 'PF00570.18', 'PF00571.23', 'PF00572.13', 'PF00573.17',
     'PF00574.18', 'PF00575.18', 'PF00576.16', 'PF00577.15', 'PF00578.16', 'PF00579.20', 'PF00580.16', 'PF00581.15',
     'PF00582.21', 'PF00583.19', 'PF00584.15', 'PF00585.13', 'PF00586.19', 'PF00587.20', 'PF00588.14', 'PF00589.17',
     'PF00590.15', 'PF00591.16', 'PF00593.19', 'PF00594.15', 'PF00595.19', 'PF00596.16', 'PF00598.14', 'PF00599.12',
     'PF00600.14', 'PF00601.14', 'PF00602.12', 'PF00603.12', 'PF00604.12', 'PF00605.12', 'PF00606.13', 'PF00607.15',
     'PF00608.12', 'PF00609.14', 'PF00610.16', 'PF00611.18', 'PF00612.22', 'PF00613.15', 'PF00614.17', 'PF00615.14',
     'PF00616.14', 'PF00617.14', 'PF00618.15', 'PF00619.16', 'PF00620.22', 'PF00621.15', 'PF00622.23', 'PF00623.15',
     'PF00624.13', 'PF00625.16', 'PF00626.17', 'PF00627.26', 'PF00628.24', 'PF00629.18', 'PF00630.14', 'PF00631.17',
     'PF00632.20', 'PF00633.18', 'PF00634.13', 'PF00635.21', 'PF00636.21', 'PF00637.15', 'PF00638.13', 'PF00639.16',
     'PF00640.18', 'PF00641.13', 'PF00642.19', 'PF00643.19', 'PF00644.15', 'PF00645.13', 'PF00646.28', 'PF00647.14',
     'PF00648.16', 'PF00649.13', 'PF00650.15', 'PF00651.26', 'PF00652.17', 'PF00653.16', 'PF00654.15', 'PF00656.17',
     'PF00657.17', 'PF00658.13', 'PF00659.13', 'PF00660.12', 'PF00661.16', 'PF00662.15', 'PF00664.18', 'PF00665.21',
     'PF00666.12', 'PF00667.15', 'PF00668.15', 'PF00669.15', 'PF00670.16', 'PF00672.20', 'PF00673.16', 'PF00674.13',
     'PF00675.15', 'PF00676.15', 'PF00677.12', 'PF00679.19', 'PF00680.15', 'PF00681.15', 'PF00682.14', 'PF00683.12',
     'PF00684.14', 'PF00685.22', 'PF00686.14', 'PF00687.16', 'PF00688.13', 'PF00689.16', 'PF00690.21', 'PF00691.15',
     'PF00692.14', 'PF00693.13', 'PF00694.14', 'PF00695.14', 'PF00696.23', 'PF00697.17', 'PF00698.16', 'PF00699.15',
     'PF00700.16', 'PF00701.17', 'PF00702.21', 'PF00703.16', 'PF00704.23', 'PF00705.13', 'PF00706.12', 'PF00707.17',
     'PF00708.13', 'PF00709.16', 'PF00710.15', 'PF00711.14', 'PF00712.14', 'PF00713.12', 'PF00714.12', 'PF00715.12',
     'PF00716.12', 'PF00717.18', 'PF00718.15', 'PF00719.14', 'PF00720.12', 'PF00721.16', 'PF00722.16', 'PF00723.16',
     'PF00724.15', 'PF00725.17', 'PF00726.12', 'PF00727.13', 'PF00728.17', 'PF00729.13', 'PF00730.20', 'PF00731.15',
     'PF00732.14', 'PF00733.16', 'PF00734.13', 'PF00735.13', 'PF00736.14', 'PF00737.15', 'PF00738.13', 'PF00739.14',
     'PF00740.13', 'PF00741.13', 'PF00742.14', 'PF00743.14', 'PF00745.15', 'PF00746.16', 'PF00747.12', 'PF00748.14',
     'PF00749.16', 'PF00750.14', 'PF00751.13', 'PF00752.12', 'PF00753.22', 'PF00754.20', 'PF00755.15', 'PF00756.15',
     'PF00757.15', 'PF00758.13', 'PF00759.14', 'PF00760.13', 'PF00761.15', 'PF00762.14', 'PF00763.18', 'PF00764.14',
     'PF00765.12', 'PF00766.14', 'PF00767.13', 'PF00768.15', 'PF00769.14', 'PF00770.13', 'PF00771.15', 'PF00772.16',
     'PF00773.14', 'PF00775.16', 'PF00777.13', 'PF00778.12', 'PF00779.14', 'PF00780.17', 'PF00781.19', 'PF00782.15',
     'PF00784.12', 'PF00786.23', 'PF00787.19', 'PF00788.18', 'PF00789.15', 'PF00790.14', 'PF00791.15', 'PF00792.19',
     'PF00793.15', 'PF00794.13', 'PF00795.17', 'PF00796.13', 'PF00797.12', 'PF00798.13', 'PF00799.15', 'PF00800.13',
     'PF00801.15', 'PF00802.14', 'PF00803.13', 'PF00804.20', 'PF00805.17', 'PF00806.14', 'PF00807.12', 'PF00808.18',
     'PF00809.17', 'PF00810.13', 'PF00811.13', 'PF00812.12', 'PF00813.15', 'PF00814.20', 'PF00815.15', 'PF00816.16',
     'PF00817.15', 'PF00818.12', 'PF00819.12', 'PF00820.14', 'PF00821.13', 'PF00822.15', 'PF00823.14', 'PF00825.13',
     'PF00827.12', 'PF00828.14', 'PF00829.16', 'PF00830.14', 'PF00831.18', 'PF00832.15', 'PF00833.13', 'PF00834.14',
     'PF00835.14', 'PF00836.14', 'PF00837.12', 'PF00838.12', 'PF00839.12', 'PF00840.15', 'PF00841.14', 'PF00842.16',
     'PF00843.12', 'PF00844.13', 'PF00845.14', 'PF00846.13', 'PF00847.15', 'PF00848.14', 'PF00849.17', 'PF00850.14',
     'PF00851.13', 'PF00852.14', 'PF00853.14', 'PF00854.16', 'PF00855.12', 'PF00856.23', 'PF00857.15', 'PF00858.19',
     'PF00859.13', 'PF00860.15', 'PF00861.17', 'PF00862.14', 'PF00863.14', 'PF00864.14', 'PF00865.13', 'PF00866.13',
     'PF00867.13', 'PF00868.15', 'PF00869.15', 'PF00870.13', 'PF00871.12', 'PF00872.13', 'PF00873.14', 'PF00874.15',
     'PF00875.13', 'PF00876.13', 'PF00877.14', 'PF00878.13', 'PF00879.13', 'PF00880.13', 'PF00881.19', 'PF00882.13',
     'PF00883.16', 'PF00884.18', 'PF00885.14', 'PF00886.14', 'PF00887.14', 'PF00888.17', 'PF00889.14', 'PF00890.19',
     'PF00891.13', 'PF00892.15', 'PF00893.14', 'PF00894.13', 'PF00895.15', 'PF00897.12', 'PF00898.12', 'PF00899.16',
     'PF00900.15', 'PF00901.12', 'PF00902.13', 'PF00903.20', 'PF00904.12', 'PF00905.17', 'PF00906.15', 'PF00907.17',
     'PF00908.12', 'PF00909.16', 'PF00910.17', 'PF00912.17', 'PF00913.14', 'PF00915.15', 'PF00916.15', 'PF00917.21',
     'PF00918.12', 'PF00919.15', 'PF00920.16', 'PF00921.12', 'PF00922.12', 'PF00923.14', 'PF00924.13', 'PF00925.15',
     'PF00926.14', 'PF00927.17', 'PF00928.16', 'PF00929.19', 'PF00930.16', 'PF00931.17', 'PF00932.14', 'PF00933.16',
     'PF00934.15', 'PF00935.14', 'PF00936.14', 'PF00937.13', 'PF00938.12', 'PF00939.14', 'PF00940.14', 'PF00941.16',
     'PF00942.13', 'PF00943.14', 'PF00944.14', 'PF00945.13', 'PF00946.14', 'PF00947.14', 'PF00948.16', 'PF00949.16',
     'PF00950.12', 'PF00951.13', 'PF00952.12', 'PF00953.16', 'PF00954.15', 'PF00955.16', 'PF00956.13', 'PF00957.16',
     'PF00958.17', 'PF00959.14', 'PF00960.13', 'PF00961.14', 'PF00962.17', 'PF00963.13', 'PF00964.12', 'PF00965.12',
     'PF00967.12', 'PF00969.14', 'PF00970.19', 'PF00971.13', 'PF00972.15', 'PF00973.14', 'PF00974.13', 'PF00975.15',
     'PF00976.13', 'PF00977.16', 'PF00978.16', 'PF00979.12', 'PF00980.12', 'PF00981.12', 'PF00982.16', 'PF00983.13',
     'PF00984.14', 'PF00985.12', 'PF00986.16', 'PF00988.17', 'PF00989.19', 'PF00990.16', 'PF00992.15', 'PF00993.15',
     'PF00994.19', 'PF00995.18', 'PF00996.13', 'PF00997.13', 'PF00998.18', 'PF00999.16', 'PF01000.21', 'PF01001.14',
     'PF01002.14', 'PF01003.14', 'PF01004.14', 'PF01005.14', 'PF01006.15', 'PF01007.15', 'PF01008.12', 'PF01010.14',
     'PF01011.16', 'PF01012.16', 'PF01014.13', 'PF01015.13', 'PF01016.14', 'PF01017.15', 'PF01018.17', 'PF01019.16',
     'PF01020.12', 'PF01021.14', 'PF01022.15', 'PF01023.14', 'PF01024.14', 'PF01025.14', 'PF01026.16', 'PF01027.15',
     'PF01028.15', 'PF01029.13', 'PF01030.19', 'PF01031.15', 'PF01032.13', 'PF01033.12', 'PF01034.15', 'PF01035.15',
     'PF01036.13', 'PF01037.16', 'PF01039.17', 'PF01040.13', 'PF01041.12', 'PF01042.16', 'PF01043.15', 'PF01044.14',
     'PF01047.17', 'PF01048.15', 'PF01049.12', 'PF01050.13', 'PF01051.16', 'PF01052.15', 'PF01053.15', 'PF01054.12',
     'PF01055.21', 'PF01056.13', 'PF01057.12', 'PF01058.17', 'PF01059.12', 'PF01060.18', 'PF01061.19', 'PF01062.16',
     'PF01063.14', 'PF01064.18', 'PF01065.14', 'PF01066.16', 'PF01067.17', 'PF01068.16', 'PF01070.13', 'PF01071.14',
     'PF01073.14', 'PF01074.17', 'PF01075.12', 'PF01076.14', 'PF01077.17', 'PF01078.16', 'PF01079.15', 'PF01080.12',
     'PF01081.14', 'PF01082.15', 'PF01083.17', 'PF01084.15', 'PF01085.13', 'PF01086.12', 'PF01087.17', 'PF01088.16',
     'PF01090.14', 'PF01091.13', 'PF01092.14', 'PF01093.12', 'PF01094.23', 'PF01095.14', 'PF01096.13', 'PF01097.13',
     'PF01098.14', 'PF01099.12', 'PF01101.13', 'PF01102.13', 'PF01103.18', 'PF01104.12', 'PF01105.19', 'PF01106.12',
     'PF01107.13', 'PF01108.12', 'PF01109.12', 'PF01110.12', 'PF01111.14', 'PF01112.13', 'PF01113.15', 'PF01114.13',
     'PF01115.12', 'PF01116.15', 'PF01117.15', 'PF01118.19', 'PF01119.14', 'PF01120.12', 'PF01121.15', 'PF01122.14',
     'PF01123.15', 'PF01124.13', 'PF01125.12', 'PF01126.15', 'PF01127.17', 'PF01128.14', 'PF01129.13', 'PF01130.16',
     'PF01131.15', 'PF01132.15', 'PF01133.12', 'PF01134.17', 'PF01135.14', 'PF01136.14', 'PF01137.16', 'PF01138.16',
     'PF01139.12', 'PF01140.14', 'PF01141.13', 'PF01142.13', 'PF01144.18', 'PF01145.20', 'PF01146.12', 'PF01147.12',
     'PF01148.15', 'PF01149.19', 'PF01150.12', 'PF01151.13', 'PF01152.16', 'PF01153.14', 'PF01154.12', 'PF01155.14',
     'PF01156.14', 'PF01157.13', 'PF01158.13', 'PF01159.14', 'PF01160.13', 'PF01161.15', 'PF01163.17', 'PF01165.15',
     'PF01166.13', 'PF01167.13', 'PF01168.15', 'PF01169.14', 'PF01170.13', 'PF01171.15', 'PF01172.13', 'PF01174.14',
     'PF01175.13', 'PF01176.14', 'PF01177.17', 'PF01179.15', 'PF01180.16', 'PF01182.15', 'PF01183.15', 'PF01184.14',
     'PF01185.13', 'PF01186.12', 'PF01187.13', 'PF01188.16', 'PF01189.12', 'PF01190.12', 'PF01191.14', 'PF01192.17',
     'PF01193.19', 'PF01194.12', 'PF01195.14', 'PF01196.14', 'PF01197.13', 'PF01198.14', 'PF01199.13', 'PF01200.13',
     'PF01201.17', 'PF01202.17', 'PF01203.14', 'PF01204.13', 'PF01205.14', 'PF01206.12', 'PF01207.12', 'PF01208.12',
     'PF01209.13', 'PF01210.18', 'PF01212.16', 'PF01213.14', 'PF01214.13', 'PF01215.14', 'PF01216.12', 'PF01217.15',
     'PF01218.13', 'PF01219.14', 'PF01220.14', 'PF01221.13', 'PF01222.12', 'PF01223.18', 'PF01225.20', 'PF01226.12',
     'PF01227.17', 'PF01228.16', 'PF01229.12', 'PF01230.18', 'PF01231.13', 'PF01232.18', 'PF01233.14', 'PF01234.12',
     'PF01235.12', 'PF01237.13', 'PF01238.16', 'PF01239.17', 'PF01241.13', 'PF01242.14', 'PF01243.15', 'PF01244.16',
     'PF01245.15', 'PF01246.15', 'PF01247.13', 'PF01248.21', 'PF01249.13', 'PF01250.12', 'PF01251.13', 'PF01252.13',
     'PF01253.17', 'PF01254.13', 'PF01255.14', 'PF01256.12', 'PF01257.14', 'PF01258.12', 'PF01259.13', 'PF01261.19',
     'PF01262.16', 'PF01263.15', 'PF01264.16', 'PF01265.12', 'PF01266.19', 'PF01267.12', 'PF01268.14', 'PF01269.12',
     'PF01270.12', 'PF01271.12', 'PF01272.14', 'PF01273.20', 'PF01274.17', 'PF01275.14', 'PF01276.15', 'PF01277.12',
     'PF01278.15', 'PF01279.12', 'PF01280.15', 'PF01281.14', 'PF01282.14', 'PF01283.14', 'PF01284.18', 'PF01285.13',
     'PF01286.13', 'PF01287.15', 'PF01288.15', 'PF01289.14', 'PF01290.15', 'PF01291.12', 'PF01292.15', 'PF01293.15',
     'PF01294.13', 'PF01295.13', 'PF01296.13', 'PF01297.12', 'PF01298.13', 'PF01299.12', 'PF01300.13', 'PF01301.14',
     'PF01302.20', 'PF01303.12', 'PF01304.12', 'PF01306.14', 'PF01307.12', 'PF01308.12', 'PF01309.13', 'PF01310.13',
     'PF01311.15', 'PF01312.14', 'PF01313.14', 'PF01314.13', 'PF01315.17', 'PF01316.16', 'PF01318.13', 'PF01320.13',
     'PF01321.13', 'PF01322.15', 'PF01323.15', 'PF01324.14', 'PF01325.14', 'PF01326.14', 'PF01327.16', 'PF01328.12',
     'PF01329.14', 'PF01330.16', 'PF01331.14', 'PF01333.14', 'PF01335.16', 'PF01336.20', 'PF01337.13', 'PF01338.13',
     'PF01339.12', 'PF01340.15', 'PF01341.12', 'PF01342.16', 'PF01343.13', 'PF01344.20', 'PF01345.13', 'PF01346.13',
     'PF01347.17', 'PF01348.16', 'PF01349.12', 'PF01350.12', 'PF01351.13', 'PF01352.22', 'PF01353.17', 'PF01355.12',
     'PF01356.14', 'PF01357.16', 'PF01358.13', 'PF01359.13', 'PF01361.16', 'PF01363.16', 'PF01364.13', 'PF01365.16',
     'PF01366.13', 'PF01367.15', 'PF01368.15', 'PF01369.15', 'PF01370.16', 'PF01371.14', 'PF01372.12', 'PF01373.12',
     'PF01374.13', 'PF01375.12', 'PF01376.13', 'PF01378.12', 'PF01379.15', 'PF01380.17', 'PF01381.17', 'PF01382.12',
     'PF01383.16', 'PF01384.15', 'PF01385.14', 'PF01386.14', 'PF01387.12', 'PF01388.16', 'PF01389.12', 'PF01390.15',
     'PF01391.13', 'PF01392.17', 'PF01393.14', 'PF01394.15', 'PF01395.17', 'PF01396.14', 'PF01397.16', 'PF01398.16',
     'PF01399.22', 'PF01400.19', 'PF01401.13', 'PF01402.16', 'PF01403.14', 'PF01404.14', 'PF01405.12', 'PF01406.14',
     'PF01407.12', 'PF01408.17', 'PF01409.15', 'PF01410.13', 'PF01411.14', 'PF01412.13', 'PF01413.14', 'PF01414.14',
     'PF01415.11', 'PF01416.15', 'PF01417.15', 'PF01418.12', 'PF01419.12', 'PF01420.14', 'PF01421.14', 'PF01422.12',
     'PF01423.17', 'PF01424.17', 'PF01425.16', 'PF01426.13', 'PF01427.12', 'PF01428.11', 'PF01429.14', 'PF01430.14',
     'PF01431.16', 'PF01432.15', 'PF01433.15', 'PF01434.13', 'PF01435.13', 'PF01436.16', 'PF01437.20', 'PF01439.13',
     'PF01440.11', 'PF01441.14', 'PF01442.13', 'PF01443.13', 'PF01445.12', 'PF01446.12', 'PF01447.13', 'PF01448.19',
     'PF01450.14', 'PF01451.16', 'PF01452.11', 'PF01453.19', 'PF01454.14', 'PF01455.13', 'PF01456.12', 'PF01457.11',
     'PF01458.12', 'PF01459.17', 'PF01462.13', 'PF01463.19', 'PF01464.15', 'PF01465.15', 'PF01466.14', 'PF01467.21',
     'PF01468.12', 'PF01469.13', 'PF01470.12', 'PF01471.13', 'PF01472.15', 'PF01473.15', 'PF01474.11', 'PF01475.14',
     'PF01476.15', 'PF01477.18', 'PF01478.13', 'PF01479.20', 'PF01480.12', 'PF01481.11', 'PF01483.15', 'PF01484.12',
     'PF01485.16', 'PF01486.12', 'PF01487.10', 'PF01488.15', 'PF01490.13', 'PF01491.11', 'PF01492.12', 'PF01493.14',
     'PF01494.14', 'PF01496.14', 'PF01497.13', 'PF01498.13', 'PF01499.11', 'PF01500.12', 'PF01501.15', 'PF01502.13',
     'PF01503.12', 'PF01504.13', 'PF01505.13', 'PF01506.14', 'PF01507.14', 'PF01508.11', 'PF01509.13', 'PF01510.20',
     'PF01512.12', 'PF01513.16', 'PF01514.12', 'PF01515.14', 'PF01516.11', 'PF01517.13', 'PF01518.11', 'PF01519.11',
     'PF01520.13', 'PF01521.15', 'PF01522.16', 'PF01523.11', 'PF01524.12', 'PF01525.11', 'PF01526.12', 'PF01527.15',
     'PF01528.11', 'PF01529.15', 'PF01530.13', 'PF01531.11', 'PF01532.15', 'PF01533.11', 'PF01534.12', 'PF01535.15',
     'PF01536.11', 'PF01537.12', 'PF01538.13', 'PF01539.12', 'PF01540.11', 'PF01541.19', 'PF01542.13', 'PF01543.12',
     'PF01544.13', 'PF01545.16', 'PF01546.23', 'PF01547.20', 'PF01548.12', 'PF01549.19', 'PF01551.17', 'PF01552.12',
     'PF01553.16', 'PF01554.13', 'PF01555.13', 'PF01556.13', 'PF01557.13', 'PF01558.13', 'PF01559.11', 'PF01560.12',
     'PF01561.11', 'PF01562.14', 'PF01563.11', 'PF01564.12', 'PF01565.18', 'PF01566.13', 'PF01567.11', 'PF01568.16',
     'PF01569.16', 'PF01570.12', 'PF01571.16', 'PF01573.11', 'PF01575.14', 'PF01576.14', 'PF01577.11', 'PF01578.15',
     'PF01579.13', 'PF01580.13', 'PF01581.11', 'PF01582.15', 'PF01583.15', 'PF01584.14', 'PF01585.18', 'PF01586.11',
     'PF01588.15', 'PF01589.11', 'PF01590.21', 'PF01591.13', 'PF01592.11', 'PF01593.19', 'PF01594.11', 'PF01595.15',
     'PF01596.12', 'PF01597.14', 'PF01599.14', 'PF01600.11', 'PF01601.11', 'PF01602.15', 'PF01603.15', 'PF01606.11',
     'PF01607.19', 'PF01608.12', 'PF01609.16', 'PF01610.12', 'PF01611.11', 'PF01612.15', 'PF01613.13', 'PF01614.13',
     'PF01616.11', 'PF01617.12', 'PF01618.11', 'PF01619.13', 'PF01620.11', 'PF01621.12', 'PF01623.12', 'PF01624.15',
     'PF01625.16', 'PF01627.18', 'PF01628.16', 'PF01629.11', 'PF01630.13', 'PF01632.14', 'PF01633.15', 'PF01634.13',
     'PF01635.13', 'PF01636.18', 'PF01637.13', 'PF01638.12', 'PF01639.12', 'PF01640.12', 'PF01641.13', 'PF01642.17',
     'PF01643.12', 'PF01644.12', 'PF01645.12', 'PF01646.11', 'PF01648.15', 'PF01649.13', 'PF01650.13', 'PF01652.13',
     'PF01653.13', 'PF01654.12', 'PF01655.13', 'PF01656.18', 'PF01657.12', 'PF01658.12', 'PF01659.11', 'PF01660.12',
     'PF01661.16', 'PF01663.17', 'PF01664.11', 'PF01665.11', 'PF01666.12', 'PF01667.12', 'PF01668.13', 'PF01669.12',
     'PF01670.11', 'PF01671.11', 'PF01672.11', 'PF01673.13', 'PF01674.13', 'PF01676.13', 'PF01677.11', 'PF01678.14',
     'PF01679.12', 'PF01680.12', 'PF01681.12', 'PF01682.14', 'PF01683.13', 'PF01684.11', 'PF01686.12', 'PF01687.12',
     'PF01688.12', 'PF01690.12', 'PF01691.11', 'PF01692.13', 'PF01693.11', 'PF01694.17', 'PF01695.12', 'PF01696.12',
     'PF01697.22', 'PF01698.11', 'PF01699.19', 'PF01700.11', 'PF01701.13', 'PF01702.13', 'PF01704.13', 'PF01705.14',
     'PF01706.11', 'PF01707.11', 'PF01708.11', 'PF01709.15', 'PF01710.11', 'PF01712.14', 'PF01713.16', 'PF01715.12',
     'PF01716.13', 'PF01717.13', 'PF01718.11', 'PF01719.12', 'PF01721.13', 'PF01722.13', 'PF01723.11', 'PF01724.11',
     'PF01725.11', 'PF01726.11', 'PF01728.14', 'PF01729.14', 'PF01730.11', 'PF01731.15', 'PF01732.11', 'PF01733.13',
     'PF01734.17', 'PF01735.13', 'PF01736.11', 'PF01737.12', 'PF01738.13', 'PF01739.13', 'PF01740.16', 'PF01741.13',
     'PF01742.12', 'PF01743.15', 'PF01744.15', 'PF01745.11', 'PF01746.16', 'PF01747.12', 'PF01749.15', 'PF01750.13',
     'PF01751.17', 'PF01752.12', 'PF01753.13', 'PF01754.11', 'PF01755.12', 'PF01756.14', 'PF01757.17', 'PF01758.11',
     'PF01759.16', 'PF01761.15', 'PF01762.16', 'PF01763.11', 'PF01764.20', 'PF01765.14', 'PF01766.12', 'PF01767.11',
     'PF01768.11', 'PF01769.11', 'PF01770.13', 'PF01771.12', 'PF01773.15', 'PF01774.12', 'PF01775.12', 'PF01776.12',
     'PF01777.13', 'PF01778.12', 'PF01779.12', 'PF01780.14', 'PF01781.13', 'PF01782.13', 'PF01783.18', 'PF01784.13',
     'PF01785.12', 'PF01786.12', 'PF01787.11', 'PF01788.12', 'PF01789.11', 'PF01790.13', 'PF01791.4', 'PF01793.11',
     'PF01794.14', 'PF01795.14', 'PF01796.12', 'PF01797.11', 'PF01798.13', 'PF01799.15', 'PF01801.11', 'PF01802.12',
     'PF01803.11', 'PF01804.13', 'PF01805.15', 'PF01806.12', 'PF01807.15', 'PF01808.13', 'PF01809.13', 'PF01810.13',
     'PF01812.15', 'PF01813.12', 'PF01814.18', 'PF01815.11', 'PF01816.12', 'PF01817.16', 'PF01818.12', 'PF01819.12',
     'PF01820.16', 'PF01821.13', 'PF01822.14', 'PF01823.14', 'PF01824.13', 'PF01825.16', 'PF01826.12', 'PF01827.22',
     'PF01828.12', 'PF01829.11', 'PF01830.13', 'PF01831.12', 'PF01832.15', 'PF01833.19', 'PF01834.11', 'PF01835.14',
     'PF01837.11', 'PF01839.18', 'PF01840.12', 'PF01841.14', 'PF01842.20', 'PF01843.14', 'PF01844.18', 'PF01845.12',
     'PF01846.14', 'PF01847.11', 'PF01848.11', 'PF01849.13', 'PF01850.16', 'PF01851.17', 'PF01852.14', 'PF01853.13',
     'PF01855.14', 'PF01856.12', 'PF01857.15', 'PF01858.12', 'PF01861.11', 'PF01862.11', 'PF01863.12', 'PF01864.12',
     'PF01865.11', 'PF01866.12', 'PF01867.11', 'PF01868.11', 'PF01869.15', 'PF01870.13', 'PF01871.12', 'PF01872.12',
     'PF01873.12', 'PF01874.11', 'PF01875.12', 'PF01876.11', 'PF01877.12', 'PF01878.13', 'PF01880.13', 'PF01881.11',
     'PF01882.13', 'PF01883.14', 'PF01884.12', 'PF01885.11', 'PF01886.11', 'PF01887.11', 'PF01888.12', 'PF01889.12',
     'PF01890.11', 'PF01891.11', 'PF01893.11', 'PF01894.12', 'PF01895.14', 'PF01896.14', 'PF01899.11', 'PF01900.14',
     'PF01901.11', 'PF01902.12', 'PF01903.12', 'PF01904.13', 'PF01905.11', 'PF01906.12', 'PF01907.14', 'PF01909.18',
     'PF01910.12', 'PF01912.13', 'PF01913.13', 'PF01914.12', 'PF01915.17', 'PF01916.12', 'PF01917.11', 'PF01918.16',
     'PF01920.15', 'PF01921.13', 'PF01922.12', 'PF01923.13', 'PF01924.11', 'PF01925.14', 'PF01926.18', 'PF01927.11',
     'PF01928.16', 'PF01929.12', 'PF01930.12', 'PF01931.13', 'PF01933.13', 'PF01934.12', 'PF01935.12', 'PF01936.13',
     'PF01937.14', 'PF01938.15', 'PF01939.11', 'PF01940.11', 'PF01941.14', 'PF01943.12', 'PF01944.12', 'PF01946.12',
     'PF01947.11', 'PF01948.13', 'PF01949.11', 'PF01950.11', 'PF01951.11', 'PF01954.11', 'PF01955.13', 'PF01956.11',
     'PF01957.13', 'PF01958.13', 'PF01959.11', 'PF01960.13', 'PF01963.12', 'PF01964.13', 'PF01965.19', 'PF01966.17',
     'PF01967.16', 'PF01968.13', 'PF01969.12', 'PF01970.11', 'PF01972.11', 'PF01973.13', 'PF01974.12', 'PF01975.12',
     'PF01976.12', 'PF01977.11', 'PF01978.14', 'PF01979.15', 'PF01980.11', 'PF01981.11', 'PF01982.11', 'PF01983.11',
     'PF01984.15', 'PF01985.16', 'PF01986.11', 'PF01987.12', 'PF01988.14', 'PF01989.11', 'PF01990.12', 'PF01991.13',
     'PF01992.11', 'PF01993.13', 'PF01994.11', 'PF01995.11', 'PF01996.11', 'PF01997.11', 'PF01998.12', 'PF02001.11',
     'PF02002.12', 'PF02005.11', 'PF02006.11', 'PF02007.13', 'PF02008.15', 'PF02009.11', 'PF02010.10', 'PF02011.10',
     'PF02012.15', 'PF02013.11', 'PF02014.11', 'PF02015.11', 'PF02016.10', 'PF02017.10', 'PF02018.12', 'PF02019.13',
     'PF02020.13', 'PF02021.12', 'PF02022.14', 'PF02023.12', 'PF02024.10', 'PF02025.10', 'PF02026.11', 'PF02027.12',
     'PF02028.12', 'PF02029.10', 'PF02030.10', 'PF02031.11', 'PF02033.13', 'PF02035.10', 'PF02036.12', 'PF02037.22',
     'PF02038.11', 'PF02040.10', 'PF02041.11', 'PF02042.10', 'PF02043.12', 'PF02044.12', 'PF02045.10', 'PF02046.10',
     'PF02048.11', 'PF02049.13', 'PF02050.11', 'PF02052.10', 'PF02053.10', 'PF02055.11', 'PF02056.11', 'PF02057.10',
     'PF02058.10', 'PF02059.10', 'PF02060.10', 'PF02061.11', 'PF02063.12', 'PF02064.10', 'PF02065.13', 'PF02066.10',
     'PF02067.10', 'PF02068.11', 'PF02069.11', 'PF02070.10', 'PF02071.15', 'PF02072.10', 'PF02073.10', 'PF02074.10',
     'PF02075.12', 'PF02076.10', 'PF02077.10', 'PF02078.11', 'PF02079.11', 'PF02080.16', 'PF02081.10', 'PF02082.15',
     'PF02083.10', 'PF02084.10', 'PF02085.11', 'PF02086.10', 'PF02087.10', 'PF02088.10', 'PF02089.10', 'PF02090.10',
     'PF02091.10', 'PF02092.12', 'PF02093.11', 'PF02095.10', 'PF02096.15', 'PF02097.10', 'PF02098.11', 'PF02099.12',
     'PF02100.12', 'PF02101.10', 'PF02102.10', 'PF02104.10', 'PF02106.10', 'PF02107.11', 'PF02108.11', 'PF02109.11',
     'PF02110.10', 'PF02112.10', 'PF02113.10', 'PF02114.11', 'PF02115.12', 'PF02116.10', 'PF02117.11', 'PF02118.16',
     'PF02119.11', 'PF02120.11', 'PF02121.13', 'PF02122.10', 'PF02123.11', 'PF02124.10', 'PF02126.13', 'PF02127.10',
     'PF02128.10', 'PF02129.13', 'PF02130.12', 'PF02132.10', 'PF02133.10', 'PF02134.16', 'PF02135.11', 'PF02136.15',
     'PF02137.13', 'PF02138.13', 'PF02140.13', 'PF02141.16', 'PF02142.17', 'PF02144.11', 'PF02145.10', 'PF02146.12',
     'PF02148.14', 'PF02149.14', 'PF02150.11', 'PF02151.14', 'PF02152.13', 'PF02153.12', 'PF02154.10', 'PF02155.10',
     'PF02156.10', 'PF02157.10', 'PF02158.10', 'PF02159.10', 'PF02160.10', 'PF02161.10', 'PF02162.12', 'PF02163.17',
     'PF02165.10', 'PF02166.11', 'PF02167.10', 'PF02169.11', 'PF02170.17', 'PF02171.12', 'PF02172.11', 'PF02173.12',
     'PF02174.12', 'PF02175.11', 'PF02176.13', 'PF02177.11', 'PF02178.14', 'PF02179.11', 'PF02180.12', 'PF02181.18',
     'PF02182.12', 'PF02183.13', 'PF02184.11', 'PF02185.11', 'PF02186.10', 'PF02187.12', 'PF02188.12', 'PF02189.10',
     'PF02190.11', 'PF02191.11', 'PF02192.11', 'PF02194.10', 'PF02195.13', 'PF02196.10', 'PF02197.12', 'PF02198.11',
     'PF02199.10', 'PF02200.11', 'PF02201.13', 'PF02202.11', 'PF02203.10', 'PF02204.13', 'PF02205.15', 'PF02206.13',
     'PF02207.15', 'PF02208.11', 'PF02209.14', 'PF02210.19', 'PF02211.10', 'PF02212.13', 'PF02213.11', 'PF02214.17',
     'PF02216.11', 'PF02217.11', 'PF02218.10', 'PF02219.12', 'PF02221.10', 'PF02222.17', 'PF02223.12', 'PF02224.13',
     'PF02225.17', 'PF02226.11', 'PF02228.11', 'PF02229.11', 'PF02230.11', 'PF02232.10', 'PF02233.11', 'PF02234.14',
     'PF02236.12', 'PF02237.12', 'PF02238.10', 'PF02239.11', 'PF02240.11', 'PF02241.13', 'PF02244.11', 'PF02245.11',
     'PF02246.10', 'PF02247.11', 'PF02248.11', 'PF02249.12', 'PF02250.10', 'PF02251.13', 'PF02252.13', 'PF02253.10',
     'PF02254.13', 'PF02255.11', 'PF02256.12', 'PF02257.10', 'PF02258.11', 'PF02259.18', 'PF02260.15', 'PF02261.11',
     'PF02262.11', 'PF02263.14', 'PF02264.10', 'PF02265.11', 'PF02267.12', 'PF02268.11', 'PF02269.11', 'PF02270.10',
     'PF02271.11', 'PF02272.14', 'PF02273.10', 'PF02274.12', 'PF02275.13', 'PF02276.13', 'PF02277.12', 'PF02278.13',
     'PF02281.11', 'PF02282.11', 'PF02283.11', 'PF02284.11', 'PF02285.10', 'PF02286.10', 'PF02287.10', 'PF02288.10',
     'PF02289.11', 'PF02290.10', 'PF02291.10', 'PF02293.10', 'PF02294.13', 'PF02295.12', 'PF02296.11', 'PF02297.12',
     'PF02298.12', 'PF02300.12', 'PF02301.13', 'PF02302.12', 'PF02303.12', 'PF02304.10', 'PF02305.12', 'PF02306.10',
     'PF02308.11', 'PF02309.11', 'PF02310.14', 'PF02311.14', 'PF02312.12', 'PF02313.12', 'PF02315.11', 'PF02316.11',
     'PF02317.12', 'PF02318.11', 'PF02319.15', 'PF02320.11', 'PF02321.13', 'PF02322.10', 'PF02323.10', 'PF02324.11',
     'PF02325.12', 'PF02326.10', 'PF02327.12', 'PF02329.11', 'PF02330.11', 'PF02331.10', 'PF02332.13', 'PF02333.10',
     'PF02334.11', 'PF02335.10', 'PF02336.10', 'PF02337.12', 'PF02338.14', 'PF02340.10', 'PF02341.10', 'PF02342.13',
     'PF02343.11', 'PF02344.10', 'PF02346.11', 'PF02347.11', 'PF02348.14', 'PF02349.10', 'PF02350.14', 'PF02351.11',
     'PF02352.10', 'PF02353.15', 'PF02354.11', 'PF02355.11', 'PF02357.14', 'PF02358.11', 'PF02359.13', 'PF02361.11',
     'PF02362.16', 'PF02363.14', 'PF02364.10', 'PF02365.10', 'PF02366.13', 'PF02367.12', 'PF02368.13', 'PF02369.11',
     'PF02370.11', 'PF02371.11', 'PF02372.10', 'PF02373.17', 'PF02374.10', 'PF02375.12', 'PF02376.10', 'PF02377.10',
     'PF02378.13', 'PF02380.10', 'PF02381.13', 'PF02382.10', 'PF02383.13', 'PF02384.11', 'PF02386.11', 'PF02387.10',
     'PF02388.11', 'PF02389.10', 'PF02390.12', 'PF02391.12', 'PF02392.11', 'PF02393.11', 'PF02394.11', 'PF02395.11',
     'PF02397.11', 'PF02398.11', 'PF02399.10', 'PF02401.13', 'PF02402.11', 'PF02403.17', 'PF02404.10', 'PF02405.11',
     'PF02406.12', 'PF02407.11', 'PF02408.15', 'PF02410.10', 'PF02411.10', 'PF02412.13', 'PF02413.12', 'PF02414.10',
     'PF02415.12', 'PF02416.11', 'PF02417.10', 'PF02419.12', 'PF02420.10', 'PF02421.13', 'PF02422.11', 'PF02423.10',
     'PF02424.10', 'PF02425.10', 'PF02426.11', 'PF02427.12', 'PF02428.10', 'PF02429.10', 'PF02430.10', 'PF02431.10',
     'PF02432.10', 'PF02433.10', 'PF02434.11', 'PF02435.11', 'PF02436.13', 'PF02437.12', 'PF02438.11', 'PF02439.10',
     'PF02440.10', 'PF02441.14', 'PF02442.12', 'PF02443.10', 'PF02444.11', 'PF02445.11', 'PF02446.12', 'PF02447.11',
     'PF02448.10', 'PF02449.10', 'PF02450.10', 'PF02451.10', 'PF02452.12', 'PF02453.12', 'PF02454.11', 'PF02455.11',
     'PF02456.10', 'PF02457.11', 'PF02458.10', 'PF02459.10', 'PF02460.13', 'PF02461.11', 'PF02462.10', 'PF02463.14',
     'PF02464.12', 'PF02465.13', 'PF02466.14', 'PF02467.11', 'PF02468.10', 'PF02469.17', 'PF02470.15', 'PF02471.12',
     'PF02472.11', 'PF02474.10', 'PF02475.11', 'PF02476.10', 'PF02477.10', 'PF02478.11', 'PF02479.11', 'PF02480.11',
     'PF02481.10', 'PF02482.14', 'PF02484.10', 'PF02485.16', 'PF02486.14', 'PF02487.12', 'PF02488.10', 'PF02489.11',
     'PF02491.15', 'PF02492.14', 'PF02493.15', 'PF02494.11', 'PF02495.12', 'PF02496.11', 'PF02497.10', 'PF02498.12',
     'PF02499.10', 'PF02500.10', 'PF02501.12', 'PF02502.13', 'PF02503.12', 'PF02504.10', 'PF02505.9', 'PF02507.10',
     'PF02508.9', 'PF02509.9', 'PF02510.9', 'PF02511.10', 'PF02512.9', 'PF02513.12', 'PF02514.11', 'PF02515.12',
     'PF02516.9', 'PF02517.11', 'PF02518.21', 'PF02519.9', 'PF02520.12', 'PF02521.9', 'PF02522.9', 'PF02524.9',
     'PF02525.12', 'PF02526.9', 'PF02527.10', 'PF02529.10', 'PF02530.9', 'PF02531.11', 'PF02532.9', 'PF02533.10',
     'PF02534.9', 'PF02535.17', 'PF02536.9', 'PF02537.10', 'PF02538.9', 'PF02540.12', 'PF02541.11', 'PF02542.11',
     'PF02543.10', 'PF02544.11', 'PF02545.9', 'PF02547.10', 'PF02548.10', 'PF02550.10', 'PF02551.10', 'PF02552.11',
     'PF02553.10', 'PF02554.9', 'PF02556.9', 'PF02557.12', 'PF02558.11', 'PF02559.11', 'PF02560.9', 'PF02561.9',
     'PF02562.11', 'PF02563.11', 'PF02565.10', 'PF02566.14', 'PF02567.11', 'PF02568.9', 'PF02569.10', 'PF02570.10',
     'PF02571.9', 'PF02572.10', 'PF02574.11', 'PF02575.11', 'PF02576.12', 'PF02577.9', 'PF02578.10', 'PF02579.12',
     'PF02580.11', 'PF02581.12', 'PF02582.9', 'PF02583.12', 'PF02585.12', 'PF02586.9', 'PF02588.10', 'PF02589.10',
     'PF02590.12', 'PF02591.10', 'PF02592.10', 'PF02593.9', 'PF02594.11', 'PF02595.10', 'PF02596.10', 'PF02597.15',
     'PF02598.12', 'PF02599.11', 'PF02600.11', 'PF02601.10', 'PF02602.10', 'PF02603.11', 'PF02604.14', 'PF02605.10',
     'PF02606.9', 'PF02607.12', 'PF02608.9', 'PF02609.11', 'PF02610.10', 'PF02611.10', 'PF02613.10', 'PF02614.9',
     'PF02615.9', 'PF02616.9', 'PF02617.12', 'PF02618.11', 'PF02620.12', 'PF02621.9', 'PF02622.10', 'PF02623.10',
     'PF02624.11', 'PF02625.11', 'PF02626.10', 'PF02627.15', 'PF02628.10', 'PF02629.14', 'PF02630.9', 'PF02631.11',
     'PF02632.9', 'PF02633.9', 'PF02634.10', 'PF02635.10', 'PF02636.12', 'PF02637.13', 'PF02638.10', 'PF02639.9',
     'PF02641.10', 'PF02643.10', 'PF02645.11', 'PF02646.11', 'PF02649.9', 'PF02650.9', 'PF02652.9', 'PF02653.11',
     'PF02654.10', 'PF02655.9', 'PF02656.10', 'PF02657.10', 'PF02659.10', 'PF02660.10', 'PF02661.13', 'PF02662.11',
     'PF02663.9', 'PF02664.10', 'PF02665.9', 'PF02666.10', 'PF02667.9', 'PF02668.11', 'PF02669.10', 'PF02670.11',
     'PF02671.16', 'PF02672.10', 'PF02673.13', 'PF02674.11', 'PF02675.10', 'PF02676.9', 'PF02677.9', 'PF02678.11',
     'PF02679.10', 'PF02680.9', 'PF02681.9', 'PF02682.11', 'PF02683.10', 'PF02684.10', 'PF02685.11', 'PF02686.10',
     'PF02687.16', 'PF02689.9', 'PF02690.10', 'PF02691.10', 'PF02694.10', 'PF02696.9', 'PF02697.9', 'PF02698.12',
     'PF02699.10', 'PF02700.9', 'PF02701.10', 'PF02702.12', 'PF02703.9', 'PF02704.9', 'PF02705.11', 'PF02706.10',
     'PF02707.11', 'PF02709.9', 'PF02710.9', 'PF02711.9', 'PF02713.9', 'PF02714.10', 'PF02718.9', 'PF02719.10',
     'PF02720.12', 'PF02721.9', 'PF02722.10', 'PF02723.9', 'PF02724.9', 'PF02725.9', 'PF02727.11', 'PF02728.11',
     'PF02729.16', 'PF02730.10', 'PF02731.10', 'PF02732.10', 'PF02733.12', 'PF02734.12', 'PF02735.11', 'PF02736.14',
     'PF02737.13', 'PF02738.13', 'PF02739.11', 'PF02740.9', 'PF02741.10', 'PF02742.10', 'PF02743.13', 'PF02744.12',
     'PF02745.10', 'PF02746.11', 'PF02747.10', 'PF02748.10', 'PF02749.11', 'PF02750.9', 'PF02751.9', 'PF02752.17',
     'PF02753.12', 'PF02754.11', 'PF02755.10', 'PF02756.9', 'PF02757.12', 'PF02758.11', 'PF02759.14', 'PF02760.10',
     'PF02761.9', 'PF02762.9', 'PF02763.9', 'PF02764.9', 'PF02765.12', 'PF02767.11', 'PF02768.10', 'PF02769.17',
     'PF02770.14', 'PF02771.11', 'PF02772.11', 'PF02773.11', 'PF02774.13', 'PF02775.16', 'PF02776.13', 'PF02777.13',
     'PF02778.9', 'PF02779.19', 'PF02780.15', 'PF02781.11', 'PF02782.11', 'PF02783.10', 'PF02784.11', 'PF02785.14',
     'PF02786.12', 'PF02787.14', 'PF02788.11', 'PF02789.12', 'PF02790.10', 'PF02791.12', 'PF02792.9', 'PF02793.17',
     'PF02794.11', 'PF02796.10', 'PF02797.10', 'PF02798.15', 'PF02799.10', 'PF02800.15', 'PF02801.17', 'PF02803.13',
     'PF02805.11', 'PF02806.13', 'PF02807.10', 'PF02809.15', 'PF02810.10', 'PF02811.14', 'PF02812.13', 'PF02813.9',
     'PF02814.10', 'PF02815.14', 'PF02816.13', 'PF02817.12', 'PF02818.10', 'PF02819.10', 'PF02820.13', 'PF02821.11',
     'PF02822.9', 'PF02823.11', 'PF02824.16', 'PF02825.15', 'PF02826.14', 'PF02827.11', 'PF02828.11', 'PF02829.9',
     'PF02830.13', 'PF02831.10', 'PF02832.11', 'PF02833.9', 'PF02834.11', 'PF02836.12', 'PF02837.13', 'PF02838.10',
     'PF02839.9', 'PF02840.10', 'PF02841.9', 'PF02843.11', 'PF02844.10', 'PF02845.11', 'PF02847.12', 'PF02852.17',
     'PF02854.14', 'PF02861.15', 'PF02862.12', 'PF02863.13', 'PF02864.10', 'PF02865.12', 'PF02866.13', 'PF02867.10',
     'PF02868.10', 'PF02870.10', 'PF02872.13', 'PF02873.11', 'PF02874.18', 'PF02875.16', 'PF02876.12', 'PF02877.9',
     'PF02878.11', 'PF02879.11', 'PF02880.11', 'PF02881.14', 'PF02882.14', 'PF02883.15', 'PF02884.12', 'PF02885.12',
     'PF02886.12', 'PF02887.11', 'PF02888.11', 'PF02889.11', 'PF02890.9', 'PF02891.15', 'PF02892.10', 'PF02893.15',
     'PF02894.12', 'PF02895.9', 'PF02896.13', 'PF02897.10', 'PF02898.10', 'PF02899.12', 'PF02900.13', 'PF02901.10',
     'PF02902.14', 'PF02903.9', 'PF02905.9', 'PF02906.9', 'PF02907.10', 'PF02909.12', 'PF02910.15', 'PF02911.13',
     'PF02912.13', 'PF02913.14', 'PF02914.10', 'PF02915.12', 'PF02916.10', 'PF02917.9', 'PF02918.10', 'PF02919.10',
     'PF02920.10', 'PF02921.9', 'PF02922.13', 'PF02923.10', 'PF02924.9', 'PF02925.11', 'PF02926.12', 'PF02927.9',
     'PF02928.11', 'PF02929.12', 'PF02931.18', 'PF02932.11', 'PF02933.12', 'PF02934.10', 'PF02935.11', 'PF02936.9',
     'PF02937.10', 'PF02938.9', 'PF02939.11', 'PF02940.10', 'PF02941.10', 'PF02942.9', 'PF02943.10', 'PF02944.15',
     'PF02945.10', 'PF02946.9', 'PF02947.9', 'PF02948.10', 'PF02949.15', 'PF02950.12', 'PF02951.9', 'PF02952.12',
     'PF02953.10', 'PF02954.14', 'PF02955.11', 'PF02956.9', 'PF02957.10', 'PF02958.15', 'PF02959.11', 'PF02960.9',
     'PF02961.9', 'PF02962.10', 'PF02963.11', 'PF02964.11', 'PF02965.12', 'PF02966.11', 'PF02969.12', 'PF02970.11',
     'PF02971.9', 'PF02972.9', 'PF02973.11', 'PF02974.9', 'PF02975.9', 'PF02976.10', 'PF02977.10', 'PF02978.14',
     'PF02979.11', 'PF02980.11', 'PF02981.10', 'PF02982.9', 'PF02983.9', 'PF02984.14', 'PF02985.17', 'PF02986.9',
     'PF02987.11', 'PF02988.10', 'PF02989.9', 'PF02990.11', 'PF02991.11', 'PF02992.9', 'PF02993.9', 'PF02994.9',
     'PF02995.12', 'PF02996.12', 'PF02998.9', 'PF02999.9', 'PF03000.9', 'PF03002.10', 'PF03003.11', 'PF03004.9',
     'PF03006.15', 'PF03007.11', 'PF03008.9', 'PF03009.12', 'PF03010.9', 'PF03011.10', 'PF03012.9', 'PF03013.9',
     'PF03014.9', 'PF03015.14', 'PF03016.10', 'PF03017.9', 'PF03018.9', 'PF03020.10', 'PF03021.9', 'PF03022.11',
     'PF03023.9', 'PF03024.9', 'PF03025.9', 'PF03026.10', 'PF03028.10', 'PF03029.12', 'PF03030.11', 'PF03031.13',
     'PF03032.10', 'PF03033.15', 'PF03034.10', 'PF03035.9', 'PF03036.11', 'PF03037.11', 'PF03038.9', 'PF03039.9',
     'PF03040.9', 'PF03041.9', 'PF03042.9', 'PF03043.9', 'PF03044.9', 'PF03045.10', 'PF03047.9', 'PF03048.9',
     'PF03049.9', 'PF03050.9', 'PF03051.10', 'PF03052.10', 'PF03053.9', 'PF03054.11', 'PF03055.10', 'PF03057.9',
     'PF03058.9', 'PF03059.11', 'PF03060.10', 'PF03061.17', 'PF03062.14', 'PF03063.15', 'PF03064.11', 'PF03065.10',
     'PF03066.10', 'PF03067.10', 'PF03068.10', 'PF03069.10', 'PF03070.11', 'PF03071.10', 'PF03072.9', 'PF03073.10',
     'PF03074.11', 'PF03076.9', 'PF03077.9', 'PF03078.10', 'PF03079.9', 'PF03080.10', 'PF03081.10', 'PF03082.9',
     'PF03083.11', 'PF03084.9', 'PF03085.10', 'PF03086.9', 'PF03087.9', 'PF03088.11', 'PF03089.9', 'PF03090.12',
     'PF03091.10', 'PF03092.11', 'PF03094.10', 'PF03095.10', 'PF03096.9', 'PF03097.13', 'PF03098.10', 'PF03099.14',
     'PF03100.10', 'PF03101.10', 'PF03102.9', 'PF03103.12', 'PF03104.14', 'PF03105.14', 'PF03106.10', 'PF03107.11',
     'PF03108.10', 'PF03109.11', 'PF03110.9', 'PF03112.9', 'PF03113.9', 'PF03114.13', 'PF03115.9', 'PF03116.10',
     'PF03117.9', 'PF03118.10', 'PF03119.11', 'PF03120.11', 'PF03121.10', 'PF03122.9', 'PF03123.10', 'PF03124.9',
     'PF03125.13', 'PF03126.13', 'PF03127.9', 'PF03128.9', 'PF03129.15', 'PF03130.11', 'PF03131.12', 'PF03133.10',
     'PF03134.14', 'PF03135.9', 'PF03136.10', 'PF03137.15', 'PF03139.10', 'PF03140.10', 'PF03141.11', 'PF03142.10',
     'PF03143.12', 'PF03144.20', 'PF03145.11', 'PF03146.10', 'PF03147.9', 'PF03148.9', 'PF03150.9', 'PF03151.11',
     'PF03152.9', 'PF03153.8', 'PF03154.10', 'PF03155.10', 'PF03157.8', 'PF03158.8', 'PF03159.13', 'PF03160.9',
     'PF03161.8', 'PF03162.8', 'PF03164.9', 'PF03165.11', 'PF03166.9', 'PF03167.14', 'PF03168.8', 'PF03169.10',
     'PF03170.8', 'PF03171.15', 'PF03172.8', 'PF03173.8', 'PF03174.8', 'PF03175.8', 'PF03176.10', 'PF03177.9',
     'PF03178.10', 'PF03179.10', 'PF03180.9', 'PF03181.10', 'PF03183.8', 'PF03184.14', 'PF03185.10', 'PF03186.8',
     'PF03187.9', 'PF03188.11', 'PF03189.8', 'PF03190.10', 'PF03192.8', 'PF03193.11', 'PF03194.10', 'PF03195.9',
     'PF03196.8', 'PF03197.8', 'PF03198.9', 'PF03199.10', 'PF03200.11', 'PF03201.11', 'PF03202.8', 'PF03203.9',
     'PF03205.9', 'PF03206.9', 'PF03207.8', 'PF03208.14', 'PF03209.10', 'PF03210.8', 'PF03211.8', 'PF03212.9',
     'PF03213.9', 'PF03214.8', 'PF03215.10', 'PF03216.8', 'PF03217.9', 'PF03219.9', 'PF03220.8', 'PF03221.11',
     'PF03222.8', 'PF03223.10', 'PF03224.9', 'PF03225.9', 'PF03226.9', 'PF03227.11', 'PF03228.9', 'PF03229.8',
     'PF03230.8', 'PF03231.8', 'PF03232.8', 'PF03233.8', 'PF03234.9', 'PF03235.9', 'PF03237.10', 'PF03238.8',
     'PF03239.9', 'PF03241.8', 'PF03242.8', 'PF03243.10', 'PF03244.9', 'PF03245.8', 'PF03246.8', 'PF03247.9',
     'PF03248.8', 'PF03249.8', 'PF03250.9', 'PF03251.8', 'PF03252.8', 'PF03253.9', 'PF03254.8', 'PF03255.9',
     'PF03256.11', 'PF03257.8', 'PF03258.9', 'PF03259.12', 'PF03260.8', 'PF03261.10', 'PF03262.8', 'PF03263.8',
     'PF03264.9', 'PF03265.10', 'PF03266.10', 'PF03268.9', 'PF03269.9', 'PF03270.8', 'PF03271.12', 'PF03272.8',
     'PF03273.8', 'PF03274.9', 'PF03275.8', 'PF03276.9', 'PF03277.8', 'PF03278.8', 'PF03279.8', 'PF03280.9',
     'PF03281.9', 'PF03283.8', 'PF03284.8', 'PF03285.10', 'PF03286.9', 'PF03287.9', 'PF03288.11', 'PF03289.8',
     'PF03290.8', 'PF03291.11', 'PF03292.9', 'PF03293.9', 'PF03294.9', 'PF03295.9', 'PF03296.8', 'PF03297.10',
     'PF03298.8', 'PF03299.9', 'PF03300.8', 'PF03301.8', 'PF03302.8', 'PF03303.8', 'PF03304.8', 'PF03305.8',
     'PF03306.8', 'PF03307.9', 'PF03308.11', 'PF03309.9', 'PF03310.8', 'PF03311.9', 'PF03312.10', 'PF03313.10',
     'PF03314.9', 'PF03315.10', 'PF03317.8', 'PF03318.8', 'PF03319.8', 'PF03320.8', 'PF03321.8', 'PF03323.8',
     'PF03324.9', 'PF03325.8', 'PF03326.8', 'PF03327.9', 'PF03328.9', 'PF03330.13', 'PF03331.8', 'PF03332.8',
     'PF03333.8', 'PF03334.9', 'PF03335.8', 'PF03336.8', 'PF03337.8', 'PF03338.10', 'PF03339.9', 'PF03340.8',
     'PF03341.8', 'PF03342.8', 'PF03343.8', 'PF03344.10', 'PF03345.9', 'PF03347.8', 'PF03348.10', 'PF03349.11',
     'PF03350.11', 'PF03351.12', 'PF03352.8', 'PF03353.10', 'PF03354.10', 'PF03355.9', 'PF03356.10', 'PF03357.16',
     'PF03358.10', 'PF03359.8', 'PF03360.11', 'PF03361.9', 'PF03362.8', 'PF03363.8', 'PF03364.15', 'PF03366.11',
     'PF03367.8', 'PF03368.9', 'PF03369.8', 'PF03370.8', 'PF03371.10', 'PF03372.18', 'PF03373.9', 'PF03374.9',
     'PF03376.9', 'PF03377.8', 'PF03378.10', 'PF03379.8', 'PF03380.9', 'PF03381.10', 'PF03382.9', 'PF03383.10',
     'PF03384.9', 'PF03385.12', 'PF03386.9', 'PF03387.9', 'PF03388.8', 'PF03389.10', 'PF03390.10', 'PF03391.10',
     'PF03392.8', 'PF03393.11', 'PF03394.8', 'PF03395.9', 'PF03396.10', 'PF03397.9', 'PF03398.9', 'PF03399.11',
     'PF03400.8', 'PF03401.9', 'PF03402.9', 'PF03403.8', 'PF03404.11', 'PF03405.9', 'PF03406.8', 'PF03407.11',
     'PF03408.9', 'PF03409.10', 'PF03410.8', 'PF03411.8', 'PF03412.10', 'PF03413.14', 'PF03414.8', 'PF03415.9',
     'PF03416.14', 'PF03417.11', 'PF03418.9', 'PF03419.8', 'PF03420.8', 'PF03421.11', 'PF03422.10', 'PF03423.8',
     'PF03424.9', 'PF03425.8', 'PF03426.9', 'PF03427.8', 'PF03428.8', 'PF03429.8', 'PF03430.8', 'PF03431.8',
     'PF03432.9', 'PF03433.8', 'PF03434.8', 'PF03435.13', 'PF03436.8', 'PF03437.10', 'PF03438.8', 'PF03439.8',
     'PF03440.9', 'PF03441.9', 'PF03442.9', 'PF03443.9', 'PF03444.10', 'PF03445.8', 'PF03446.10', 'PF03447.11',
     'PF03448.12', 'PF03449.10', 'PF03450.12', 'PF03451.9', 'PF03452.9', 'PF03453.12', 'PF03454.10', 'PF03455.14',
     'PF03456.13', 'PF03457.9', 'PF03458.8', 'PF03459.12', 'PF03460.12', 'PF03461.10', 'PF03462.13', 'PF03463.10',
     'PF03464.10', 'PF03465.10', 'PF03466.15', 'PF03467.10', 'PF03468.9', 'PF03469.9', 'PF03470.9', 'PF03471.12',
     'PF03472.10', 'PF03473.12', 'PF03474.9', 'PF03475.9', 'PF03476.11', 'PF03477.11', 'PF03478.13', 'PF03479.10',
     'PF03480.8', 'PF03481.8', 'PF03482.8', 'PF03483.12', 'PF03484.10', 'PF03485.11', 'PF03486.9', 'PF03487.8',
     'PF03488.9', 'PF03489.12', 'PF03490.8', 'PF03491.8', 'PF03492.10', 'PF03493.13', 'PF03494.8', 'PF03495.9',
     'PF03496.9', 'PF03497.11', 'PF03498.9', 'PF03500.8', 'PF03501.10', 'PF03502.8', 'PF03503.8', 'PF03504.8',
     'PF03505.9', 'PF03506.8', 'PF03507.8', 'PF03508.8', 'PF03509.9', 'PF03510.9', 'PF03511.9', 'PF03512.8',
     'PF03513.9', 'PF03514.9', 'PF03515.9', 'PF03516.8', 'PF03517.8', 'PF03519.9', 'PF03520.9', 'PF03521.9',
     'PF03522.10', 'PF03523.8', 'PF03524.10', 'PF03525.9', 'PF03526.8', 'PF03527.9', 'PF03528.10', 'PF03529.8',
     'PF03530.9', 'PF03531.9', 'PF03532.8', 'PF03533.9', 'PF03534.8', 'PF03535.8', 'PF03536.10', 'PF03537.8',
     'PF03538.9', 'PF03539.9', 'PF03540.8', 'PF03542.11', 'PF03543.9', 'PF03544.9', 'PF03545.8', 'PF03546.9',
     'PF03547.13', 'PF03548.10', 'PF03549.9', 'PF03550.9', 'PF03551.9', 'PF03552.9', 'PF03553.9', 'PF03554.8',
     'PF03555.9', 'PF03556.10', 'PF03557.10', 'PF03558.8', 'PF03559.9', 'PF03561.10', 'PF03562.12', 'PF03563.8',
     'PF03564.10', 'PF03566.8', 'PF03567.9', 'PF03568.12', 'PF03569.8', 'PF03571.10', 'PF03572.13', 'PF03573.8',
     'PF03574.10', 'PF03575.12', 'PF03576.9', 'PF03577.10', 'PF03578.10', 'PF03579.8', 'PF03580.9', 'PF03581.8',
     'PF03583.9', 'PF03584.10', 'PF03585.9', 'PF03586.8', 'PF03587.9', 'PF03588.9', 'PF03589.8', 'PF03590.10',
     'PF03591.9', 'PF03592.11', 'PF03594.8', 'PF03595.12', 'PF03596.8', 'PF03597.10', 'PF03598.10', 'PF03599.11',
     'PF03600.11', 'PF03601.9', 'PF03602.10', 'PF03603.8', 'PF03604.8', 'PF03605.9', 'PF03606.10', 'PF03607.12',
     'PF03608.8', 'PF03609.9', 'PF03610.11', 'PF03611.9', 'PF03612.9', 'PF03613.9', 'PF03614.8', 'PF03615.10',
     'PF03616.9', 'PF03617.8', 'PF03618.9', 'PF03619.11', 'PF03620.8', 'PF03621.8', 'PF03622.8', 'PF03623.8',
     'PF03625.9', 'PF03626.9', 'PF03627.8', 'PF03628.8', 'PF03629.13', 'PF03630.9', 'PF03631.10', 'PF03632.10',
     'PF03633.10', 'PF03634.8', 'PF03635.12', 'PF03636.10', 'PF03637.12', 'PF03638.10', 'PF03639.8', 'PF03640.10',
     'PF03641.9', 'PF03642.8', 'PF03643.10', 'PF03644.8', 'PF03645.8', 'PF03646.10', 'PF03647.8', 'PF03648.9',
     'PF03649.8', 'PF03650.8', 'PF03652.10', 'PF03653.8', 'PF03656.8', 'PF03657.8', 'PF03658.9', 'PF03659.9',
     'PF03660.9', 'PF03661.8', 'PF03662.9', 'PF03663.9', 'PF03664.8', 'PF03665.8', 'PF03666.8', 'PF03668.10',
     'PF03669.8', 'PF03670.8', 'PF03671.9', 'PF03672.8', 'PF03673.8', 'PF03676.9', 'PF03677.8', 'PF03678.9',
     'PF03681.10', 'PF03682.8', 'PF03683.8', 'PF03684.8', 'PF03685.8', 'PF03686.8', 'PF03687.8', 'PF03688.9',
     'PF03689.10', 'PF03690.8', 'PF03691.9', 'PF03692.10', 'PF03693.9', 'PF03694.8', 'PF03695.8', 'PF03698.8',
     'PF03699.8', 'PF03700.8', 'PF03701.9', 'PF03702.9', 'PF03703.9', 'PF03704.12', 'PF03705.10', 'PF03706.8',
     'PF03707.11', 'PF03708.9', 'PF03709.10', 'PF03710.10', 'PF03711.10', 'PF03712.10', 'PF03713.8', 'PF03714.9',
     'PF03715.8', 'PF03716.9', 'PF03717.10', 'PF03718.8', 'PF03719.10', 'PF03720.10', 'PF03721.9', 'PF03722.9',
     'PF03723.9', 'PF03724.11', 'PF03725.10', 'PF03726.9', 'PF03727.11', 'PF03728.8', 'PF03729.8', 'PF03730.9',
     'PF03731.10', 'PF03732.12', 'PF03733.8', 'PF03734.9', 'PF03735.9', 'PF03736.12', 'PF03737.10', 'PF03738.9',
     'PF03739.9', 'PF03740.8', 'PF03741.11', 'PF03742.9', 'PF03743.9', 'PF03744.8', 'PF03745.9', 'PF03746.11',
     'PF03747.9', 'PF03748.9', 'PF03749.8', 'PF03750.8', 'PF03752.8', 'PF03753.8', 'PF03754.8', 'PF03755.8',
     'PF03756.8', 'PF03759.8', 'PF03760.10', 'PF03761.10', 'PF03762.12', 'PF03763.8', 'PF03764.13', 'PF03765.10',
     'PF03766.8', 'PF03767.9', 'PF03768.10', 'PF03769.10', 'PF03770.11', 'PF03771.11', 'PF03772.11', 'PF03773.8',
     'PF03775.11', 'PF03776.9', 'PF03777.8', 'PF03778.8', 'PF03779.9', 'PF03780.8', 'PF03781.11', 'PF03782.12',
     'PF03783.9', 'PF03784.8', 'PF03785.9', 'PF03786.8', 'PF03787.10', 'PF03788.9', 'PF03789.8', 'PF03790.8',
     'PF03791.8', 'PF03792.8', 'PF03793.14', 'PF03795.9', 'PF03796.10', 'PF03797.14', 'PF03798.11', 'PF03799.10',
     'PF03800.9', 'PF03801.8', 'PF03802.9', 'PF03803.10', 'PF03804.8', 'PF03805.8', 'PF03806.8', 'PF03807.12',
     'PF03808.8', 'PF03810.14', 'PF03811.8', 'PF03812.8', 'PF03813.9', 'PF03814.10', 'PF03815.14', 'PF03816.9',
     'PF03817.8', 'PF03818.8', 'PF03819.12', 'PF03820.12', 'PF03821.11', 'PF03822.9', 'PF03823.9', 'PF03824.11',
     'PF03825.11', 'PF03826.12', 'PF03827.8', 'PF03828.14', 'PF03829.8', 'PF03830.10', 'PF03831.9', 'PF03832.8',
     'PF03833.8', 'PF03834.9', 'PF03835.10', 'PF03836.10', 'PF03837.9', 'PF03838.9', 'PF03839.11', 'PF03840.9',
     'PF03841.8', 'PF03842.8', 'PF03843.8', 'PF03845.8', 'PF03846.9', 'PF03847.8', 'PF03848.9', 'PF03849.9',
     'PF03850.9', 'PF03851.9', 'PF03852.10', 'PF03853.10', 'PF03854.9', 'PF03855.8', 'PF03856.8', 'PF03857.8',
     'PF03858.8', 'PF03859.11', 'PF03860.11', 'PF03861.9', 'PF03862.8', 'PF03863.8', 'PF03864.10', 'PF03865.8',
     'PF03866.8', 'PF03867.9', 'PF03868.10', 'PF03869.9', 'PF03870.10', 'PF03871.9', 'PF03872.8', 'PF03873.8',
     'PF03874.11', 'PF03875.8', 'PF03876.12', 'PF03878.10', 'PF03879.9', 'PF03880.10', 'PF03881.9', 'PF03882.9',
     'PF03883.9', 'PF03884.9', 'PF03885.8', 'PF03886.8', 'PF03887.9', 'PF03888.9', 'PF03889.8', 'PF03891.10',
     'PF03892.9', 'PF03893.11', 'PF03894.10', 'PF03895.10', 'PF03896.11', 'PF03898.8', 'PF03899.10', 'PF03900.10',
     'PF03901.12', 'PF03902.8', 'PF03903.8', 'PF03904.8', 'PF03905.8', 'PF03906.9', 'PF03907.8', 'PF03908.8',
     'PF03909.12', 'PF03910.8', 'PF03911.11', 'PF03912.9', 'PF03913.8', 'PF03914.12', 'PF03915.8', 'PF03916.9',
     'PF03917.12', 'PF03918.9', 'PF03919.10', 'PF03920.10', 'PF03921.9', 'PF03922.9', 'PF03923.8', 'PF03924.8',
     'PF03925.8', 'PF03927.8', 'PF03928.9', 'PF03929.11', 'PF03930.9', 'PF03931.10', 'PF03932.9', 'PF03934.8',
     'PF03935.10', 'PF03936.11', 'PF03937.11', 'PF03938.9', 'PF03939.8', 'PF03940.8', 'PF03941.10', 'PF03942.10',
     'PF03943.8', 'PF03944.9', 'PF03945.9', 'PF03946.9', 'PF03947.13', 'PF03948.9', 'PF03949.10', 'PF03950.13',
     'PF03951.14', 'PF03952.11', 'PF03953.12', 'PF03954.9', 'PF03955.9', 'PF03956.8', 'PF03957.8', 'PF03958.12',
     'PF03959.8', 'PF03960.10', 'PF03961.8', 'PF03962.10', 'PF03963.9', 'PF03964.10', 'PF03965.11', 'PF03966.11',
     'PF03967.8', 'PF03968.9', 'PF03969.11', 'PF03970.8', 'PF03971.9', 'PF03972.9', 'PF03973.8', 'PF03974.8',
     'PF03975.8', 'PF03976.9', 'PF03977.8', 'PF03978.8', 'PF03979.9', 'PF03980.9', 'PF03981.7', 'PF03982.8',
     'PF03983.7', 'PF03984.8', 'PF03985.8', 'PF03986.8', 'PF03987.10', 'PF03988.7', 'PF03989.8', 'PF03990.9',
     'PF03991.7', 'PF03992.11', 'PF03993.7', 'PF03994.9', 'PF03995.8', 'PF03996.10', 'PF03997.7', 'PF03998.8',
     'PF03999.7', 'PF04000.10', 'PF04001.8', 'PF04002.10', 'PF04003.7', 'PF04004.8', 'PF04005.7', 'PF04006.7',
     'PF04007.7', 'PF04008.9', 'PF04009.7', 'PF04010.8', 'PF04011.7', 'PF04012.7', 'PF04013.7', 'PF04014.13',
     'PF04015.7', 'PF04016.7', 'PF04017.7', 'PF04018.8', 'PF04019.7', 'PF04020.8', 'PF04021.7', 'PF04022.7',
     'PF04023.9', 'PF04024.7', 'PF04025.7', 'PF04026.7', 'PF04027.8', 'PF04028.8', 'PF04029.9', 'PF04030.9',
     'PF04031.8', 'PF04032.11', 'PF04033.7', 'PF04034.8', 'PF04036.7', 'PF04037.8', 'PF04038.7', 'PF04039.8',
     'PF04041.8', 'PF04042.11', 'PF04043.10', 'PF04045.9', 'PF04046.11', 'PF04048.9', 'PF04049.8', 'PF04050.9',
     'PF04051.11', 'PF04052.8', 'PF04053.9', 'PF04054.10', 'PF04055.16', 'PF04056.9', 'PF04057.7', 'PF04059.7',
     'PF04060.8', 'PF04061.9', 'PF04062.9', 'PF04063.9', 'PF04064.8', 'PF04065.10', 'PF04066.8', 'PF04068.10',
     'PF04069.7', 'PF04070.7', 'PF04071.7', 'PF04072.9', 'PF04073.10', 'PF04074.7', 'PF04075.9', 'PF04076.8',
     'PF04077.7', 'PF04078.8', 'PF04079.11', 'PF04080.8', 'PF04081.8', 'PF04082.13', 'PF04083.11', 'PF04084.9',
     'PF04085.9', 'PF04086.8', 'PF04087.9', 'PF04088.8', 'PF04089.9', 'PF04090.7', 'PF04091.7', 'PF04092.8',
     'PF04093.7', 'PF04094.9', 'PF04095.11', 'PF04096.9', 'PF04097.9', 'PF04098.10', 'PF04099.7', 'PF04100.7',
     'PF04101.11', 'PF04102.7', 'PF04103.10', 'PF04104.9', 'PF04106.7', 'PF04107.8', 'PF04108.7', 'PF04109.11',
     'PF04110.8', 'PF04111.7', 'PF04112.8', 'PF04113.9', 'PF04114.9', 'PF04115.7', 'PF04116.8', 'PF04117.7',
     'PF04118.9', 'PF04119.7', 'PF04120.7', 'PF04121.8', 'PF04122.7', 'PF04123.8', 'PF04124.7', 'PF04126.8',
     'PF04127.10', 'PF04129.7', 'PF04130.8', 'PF04131.9', 'PF04133.9', 'PF04134.7', 'PF04135.7', 'PF04136.10',
     'PF04137.10', 'PF04138.9', 'PF04139.8', 'PF04140.9', 'PF04142.10', 'PF04143.9', 'PF04144.8', 'PF04145.10',
     'PF04146.10', 'PF04147.7', 'PF04148.8', 'PF04149.7', 'PF04151.10', 'PF04152.9', 'PF04153.13', 'PF04155.13',
     'PF04156.9', 'PF04157.11', 'PF04158.9', 'PF04159.8', 'PF04160.7', 'PF04161.8', 'PF04162.7', 'PF04163.7',
     'PF04165.7', 'PF04166.7', 'PF04167.8', 'PF04168.7', 'PF04170.7', 'PF04172.11', 'PF04173.8', 'PF04174.8',
     'PF04175.7', 'PF04176.8', 'PF04177.7', 'PF04178.7', 'PF04179.7', 'PF04180.9', 'PF04181.8', 'PF04182.7',
     'PF04183.7', 'PF04184.7', 'PF04185.9', 'PF04186.8', 'PF04187.8', 'PF04188.8', 'PF04189.8', 'PF04190.8',
     'PF04191.8', 'PF04192.7', 'PF04193.9', 'PF04194.8', 'PF04195.7', 'PF04196.7', 'PF04197.7', 'PF04198.8',
     'PF04199.8', 'PF04200.7', 'PF04201.10', 'PF04202.8', 'PF04203.8', 'PF04204.11', 'PF04205.9', 'PF04206.7',
     'PF04207.7', 'PF04208.9', 'PF04209.8', 'PF04210.8', 'PF04211.8', 'PF04212.13', 'PF04213.8', 'PF04214.8',
     'PF04216.7', 'PF04217.8', 'PF04218.8', 'PF04219.7', 'PF04220.7', 'PF04221.7', 'PF04222.7', 'PF04223.7',
     'PF04224.7', 'PF04225.7', 'PF04226.8', 'PF04227.7', 'PF04228.8', 'PF04229.9', 'PF04230.8', 'PF04231.8',
     'PF04232.7', 'PF04233.9', 'PF04234.7', 'PF04235.7', 'PF04236.10', 'PF04237.8', 'PF04238.7', 'PF04239.7',
     'PF04240.7', 'PF04241.10', 'PF04242.8', 'PF04244.8', 'PF04245.8', 'PF04246.7', 'PF04247.7', 'PF04248.7',
     'PF04250.8', 'PF04252.8', 'PF04253.10', 'PF04254.8', 'PF04255.9', 'PF04256.7', 'PF04257.9', 'PF04258.8',
     'PF04259.9', 'PF04260.7', 'PF04261.7', 'PF04262.9', 'PF04263.11', 'PF04264.8', 'PF04265.9', 'PF04266.9',
     'PF04267.7', 'PF04268.7', 'PF04269.7', 'PF04270.8', 'PF04272.9', 'PF04273.8', 'PF04275.9', 'PF04276.7',
     'PF04277.8', 'PF04278.7', 'PF04279.10', 'PF04280.10', 'PF04281.8', 'PF04282.8', 'PF04283.7', 'PF04284.8',
     'PF04285.7', 'PF04286.7', 'PF04287.7', 'PF04288.8', 'PF04289.7', 'PF04290.7', 'PF04293.8', 'PF04294.8',
     'PF04295.8', 'PF04296.8', 'PF04297.9', 'PF04298.7', 'PF04299.7', 'PF04300.8', 'PF04301.8', 'PF04303.8',
     'PF04304.8', 'PF04305.9', 'PF04306.8', 'PF04307.9', 'PF04308.7', 'PF04309.7', 'PF04310.7', 'PF04311.8',
     'PF04312.8', 'PF04313.9', 'PF04314.8', 'PF04315.7', 'PF04316.8', 'PF04317.7', 'PF04318.7', 'PF04319.8',
     'PF04320.9', 'PF04321.12', 'PF04322.7', 'PF04324.10', 'PF04325.8', 'PF04326.9', 'PF04327.7', 'PF04328.8',
     'PF04332.10', 'PF04333.8', 'PF04334.7', 'PF04335.8', 'PF04336.7', 'PF04337.7', 'PF04338.7', 'PF04339.7',
     'PF04340.7', 'PF04341.7', 'PF04342.7', 'PF04343.8', 'PF04344.8', 'PF04345.8', 'PF04346.7', 'PF04347.8',
     'PF04348.8', 'PF04349.7', 'PF04350.8', 'PF04351.8', 'PF04352.8', 'PF04353.8', 'PF04354.8', 'PF04355.8',
     'PF04356.7', 'PF04357.8', 'PF04358.8', 'PF04359.9', 'PF04360.7', 'PF04361.8', 'PF04362.9', 'PF04363.7',
     'PF04364.8', 'PF04365.8', 'PF04366.7', 'PF04367.8', 'PF04368.8', 'PF04369.8', 'PF04370.7', 'PF04371.10',
     'PF04375.9', 'PF04376.8', 'PF04377.10', 'PF04378.8', 'PF04379.9', 'PF04380.8', 'PF04381.7', 'PF04382.8',
     'PF04383.8', 'PF04384.8', 'PF04385.10', 'PF04386.8', 'PF04387.9', 'PF04388.7', 'PF04389.12', 'PF04390.7',
     'PF04391.7', 'PF04392.7', 'PF04393.8', 'PF04394.9', 'PF04395.7', 'PF04397.10', 'PF04398.7', 'PF04399.8',
     'PF04400.8', 'PF04402.9', 'PF04403.8', 'PF04404.7', 'PF04405.9', 'PF04406.9', 'PF04407.7', 'PF04408.18',
     'PF04409.7', 'PF04410.9', 'PF04411.7', 'PF04412.8', 'PF04413.11', 'PF04414.7', 'PF04415.7', 'PF04417.7',
     'PF04418.7', 'PF04419.9', 'PF04420.9', 'PF04421.8', 'PF04422.8', 'PF04423.9', 'PF04424.8', 'PF04425.7',
     'PF04426.7', 'PF04427.13', 'PF04428.9', 'PF04430.9', 'PF04431.8', 'PF04432.8', 'PF04433.12', 'PF04434.12',
     'PF04435.13', 'PF04437.8', 'PF04438.11', 'PF04439.7', 'PF04440.11', 'PF04441.8', 'PF04442.9', 'PF04443.7',
     'PF04444.9', 'PF04445.8', 'PF04446.7', 'PF04447.7', 'PF04448.7', 'PF04449.7', 'PF04450.7', 'PF04451.7',
     'PF04452.9', 'PF04453.9', 'PF04454.7', 'PF04455.7', 'PF04456.7', 'PF04457.7', 'PF04458.7', 'PF04459.7',
     'PF04461.8', 'PF04463.7', 'PF04464.9', 'PF04465.7', 'PF04466.8', 'PF04467.7', 'PF04468.7', 'PF04471.7',
     'PF04472.7', 'PF04473.7', 'PF04474.7', 'PF04475.7', 'PF04476.8', 'PF04478.7', 'PF04479.8', 'PF04480.7',
     'PF04481.7', 'PF04483.7', 'PF04484.7', 'PF04485.7', 'PF04486.7', 'PF04487.7', 'PF04488.10', 'PF04489.8',
     'PF04490.7', 'PF04491.7', 'PF04492.8', 'PF04493.9', 'PF04494.10', 'PF04495.9', 'PF04496.7', 'PF04497.7',
     'PF04498.7', 'PF04499.10', 'PF04500.11', 'PF04501.7', 'PF04502.8', 'PF04503.8', 'PF04504.9', 'PF04505.7',
     'PF04506.8', 'PF04507.7', 'PF04508.7', 'PF04509.7', 'PF04510.7', 'PF04511.10', 'PF04512.7', 'PF04513.7',
     'PF04514.7', 'PF04515.7', 'PF04516.10', 'PF04517.7', 'PF04518.7', 'PF04519.8', 'PF04520.8', 'PF04521.8',
     'PF04522.7', 'PF04523.8', 'PF04525.7', 'PF04526.8', 'PF04527.7', 'PF04528.8', 'PF04529.7', 'PF04530.7',
     'PF04531.8', 'PF04532.7', 'PF04533.8', 'PF04534.7', 'PF04535.7', 'PF04536.9', 'PF04537.7', 'PF04538.7',
     'PF04539.11', 'PF04540.8', 'PF04541.8', 'PF04542.9', 'PF04544.7', 'PF04545.11', 'PF04546.8', 'PF04547.7',
     'PF04548.11', 'PF04549.9', 'PF04550.7', 'PF04551.9', 'PF04552.8', 'PF04553.7', 'PF04554.8', 'PF04555.8',
     'PF04556.7', 'PF04557.10', 'PF04558.10', 'PF04559.7', 'PF04560.15', 'PF04561.9', 'PF04562.7', 'PF04563.10',
     'PF04564.10', 'PF04565.11', 'PF04566.8', 'PF04567.12', 'PF04568.7', 'PF04569.9', 'PF04570.9', 'PF04571.9',
     'PF04572.7', 'PF04573.7', 'PF04574.8', 'PF04575.8', 'PF04576.10', 'PF04577.9', 'PF04578.8', 'PF04579.7',
     'PF04580.8', 'PF04582.7', 'PF04583.7', 'PF04584.9', 'PF04586.12', 'PF04587.10', 'PF04588.8', 'PF04589.8',
     'PF04591.7', 'PF04592.9', 'PF04593.9', 'PF04595.8', 'PF04596.7', 'PF04597.9', 'PF04598.7', 'PF04599.7',
     'PF04601.8', 'PF04602.7', 'PF04603.7', 'PF04604.8', 'PF04606.7', 'PF04607.12', 'PF04608.8', 'PF04609.7',
     'PF04610.9', 'PF04611.7', 'PF04612.7', 'PF04613.9', 'PF04614.7', 'PF04615.8', 'PF04616.9', 'PF04617.8',
     'PF04618.7', 'PF04619.7', 'PF04620.7', 'PF04621.8', 'PF04622.7', 'PF04623.7', 'PF04624.9', 'PF04625.8',
     'PF04626.8', 'PF04627.8', 'PF04628.8', 'PF04629.9', 'PF04630.7', 'PF04631.7', 'PF04632.7', 'PF04633.7',
     'PF04634.7', 'PF04636.8', 'PF04637.7', 'PF04639.7', 'PF04640.9', 'PF04641.7', 'PF04642.7', 'PF04643.7',
     'PF04644.7', 'PF04645.7', 'PF04646.7', 'PF04647.10', 'PF04648.7', 'PF04649.7', 'PF04650.12', 'PF04651.8',
     'PF04652.11', 'PF04654.7', 'PF04655.9', 'PF04656.7', 'PF04657.8', 'PF04658.8', 'PF04659.8', 'PF04660.7',
     'PF04661.7', 'PF04662.8', 'PF04663.7', 'PF04664.8', 'PF04665.7', 'PF04666.8', 'PF04667.12', 'PF04668.7',
     'PF04669.8', 'PF04670.7', 'PF04671.7', 'PF04672.7', 'PF04673.7', 'PF04674.7', 'PF04675.9', 'PF04676.9',
     'PF04677.10', 'PF04678.8', 'PF04679.10', 'PF04680.8', 'PF04681.7', 'PF04682.7', 'PF04683.8', 'PF04684.8',
     'PF04685.8', 'PF04686.7', 'PF04687.7', 'PF04688.8', 'PF04689.8', 'PF04690.8', 'PF04691.7', 'PF04692.8',
     'PF04693.7', 'PF04694.7', 'PF04695.8', 'PF04696.8', 'PF04697.8', 'PF04698.7', 'PF04699.9', 'PF04700.7',
     'PF04701.8', 'PF04702.7', 'PF04703.7', 'PF04704.8', 'PF04705.7', 'PF04706.7', 'PF04707.9', 'PF04708.7',
     'PF04709.7', 'PF04710.9', 'PF04711.8', 'PF04712.7', 'PF04713.7', 'PF04714.8', 'PF04715.8', 'PF04716.9',
     'PF04717.7', 'PF04718.10', 'PF04719.9', 'PF04720.7', 'PF04721.12', 'PF04722.8', 'PF04723.9', 'PF04724.8',
     'PF04725.7', 'PF04726.8', 'PF04727.8', 'PF04728.8', 'PF04729.8', 'PF04730.7', 'PF04731.7', 'PF04732.9',
     'PF04733.9', 'PF04734.8', 'PF04735.7', 'PF04736.7', 'PF04737.8', 'PF04738.8', 'PF04739.10', 'PF04740.7',
     'PF04741.7', 'PF04744.7', 'PF04745.7', 'PF04746.7', 'PF04747.7', 'PF04748.8', 'PF04749.12', 'PF04750.9',
     'PF04751.9', 'PF04752.7', 'PF04753.7', 'PF04754.7', 'PF04755.7', 'PF04756.8', 'PF04757.9', 'PF04758.9',
     'PF04759.8', 'PF04760.10', 'PF04761.7', 'PF04762.7', 'PF04763.7', 'PF04764.7', 'PF04765.8', 'PF04766.7',
     'PF04767.7', 'PF04768.8', 'PF04769.7', 'PF04770.7', 'PF04771.7', 'PF04772.7', 'PF04773.8', 'PF04774.10',
     'PF04775.9', 'PF04776.7', 'PF04777.8', 'PF04778.7', 'PF04780.7', 'PF04781.7', 'PF04782.7', 'PF04783.7',
     'PF04784.9', 'PF04785.7', 'PF04786.7', 'PF04787.7', 'PF04788.7', 'PF04789.10', 'PF04790.8', 'PF04791.11',
     'PF04792.7', 'PF04793.7', 'PF04794.7', 'PF04795.7', 'PF04796.7', 'PF04797.8', 'PF04798.7', 'PF04799.8',
     'PF04800.7', 'PF04801.8', 'PF04802.10', 'PF04803.7', 'PF04805.7', 'PF04806.7', 'PF04807.7', 'PF04808.7',
     'PF04809.8', 'PF04810.10', 'PF04811.10', 'PF04812.8', 'PF04813.7', 'PF04814.8', 'PF04815.10', 'PF04816.7',
     'PF04817.7', 'PF04818.8', 'PF04819.7', 'PF04820.9', 'PF04821.9', 'PF04822.8', 'PF04823.7', 'PF04824.11',
     'PF04825.8', 'PF04826.8', 'PF04827.9', 'PF04828.9', 'PF04829.8', 'PF04830.8', 'PF04831.8', 'PF04832.7',
     'PF04833.10', 'PF04834.7', 'PF04835.7', 'PF04836.7', 'PF04837.7', 'PF04838.7', 'PF04839.8', 'PF04840.7',
     'PF04841.8', 'PF04842.7', 'PF04843.7', 'PF04844.8', 'PF04845.8', 'PF04846.8', 'PF04847.7', 'PF04848.8',
     'PF04849.8', 'PF04850.7', 'PF04851.10', 'PF04852.7', 'PF04854.9', 'PF04855.7', 'PF04856.8', 'PF04857.15',
     'PF04858.8', 'PF04859.7', 'PF04860.7', 'PF04862.7', 'PF04863.8', 'PF04864.8', 'PF04865.9', 'PF04866.7',
     'PF04867.7', 'PF04868.7', 'PF04869.9', 'PF04870.11', 'PF04871.8', 'PF04872.8', 'PF04873.8', 'PF04874.9',
     'PF04875.7', 'PF04876.7', 'PF04877.7', 'PF04878.8', 'PF04879.11', 'PF04880.8', 'PF04881.8', 'PF04882.7',
     'PF04883.7', 'PF04884.9', 'PF04885.8', 'PF04886.7', 'PF04887.7', 'PF04888.7', 'PF04889.7', 'PF04890.8',
     'PF04891.7', 'PF04892.7', 'PF04893.12', 'PF04894.7', 'PF04895.7', 'PF04896.7', 'PF04898.9', 'PF04899.7',
     'PF04900.7', 'PF04901.8', 'PF04902.7', 'PF04904.8', 'PF04905.8', 'PF04906.8', 'PF04908.10', 'PF04909.9',
     'PF04910.9', 'PF04911.7', 'PF04912.9', 'PF04913.7', 'PF04914.7', 'PF04915.7', 'PF04916.8', 'PF04917.7',
     'PF04918.7', 'PF04919.7', 'PF04920.7', 'PF04921.9', 'PF04922.7', 'PF04923.7', 'PF04924.7', 'PF04925.10',
     'PF04926.10', 'PF04927.7', 'PF04928.12', 'PF04929.7', 'PF04930.10', 'PF04931.8', 'PF04932.10', 'PF04934.9',
     'PF04935.7', 'PF04936.7', 'PF04937.10', 'PF04938.7', 'PF04939.7', 'PF04940.7', 'PF04941.7', 'PF04942.9',
     'PF04943.7', 'PF04945.8', 'PF04947.9', 'PF04948.8', 'PF04949.8', 'PF04950.7', 'PF04951.8', 'PF04952.9',
     'PF04954.8', 'PF04955.7', 'PF04956.8', 'PF04957.7', 'PF04958.7', 'PF04959.8', 'PF04960.10', 'PF04961.7',
     'PF04962.7', 'PF04963.8', 'PF04964.9', 'PF04965.9', 'PF04966.7', 'PF04967.7', 'PF04968.7', 'PF04969.11',
     'PF04970.8', 'PF04971.7', 'PF04972.12', 'PF04973.7', 'PF04976.7', 'PF04977.10', 'PF04978.7', 'PF04979.9',
     'PF04981.8', 'PF04982.8', 'PF04983.13', 'PF04984.9', 'PF04985.9', 'PF04986.8', 'PF04987.9', 'PF04988.7',
     'PF04989.7', 'PF04990.7', 'PF04991.8', 'PF04992.9', 'PF04993.8', 'PF04994.8', 'PF04995.9', 'PF04996.7',
     'PF04997.7', 'PF04998.12', 'PF04999.8', 'PF05000.12', 'PF05001.8', 'PF05002.10', 'PF05003.7', 'PF05004.8',
     'PF05005.10', 'PF05006.7', 'PF05007.8', 'PF05008.10', 'PF05009.7', 'PF05010.9', 'PF05011.8', 'PF05013.7',
     'PF05014.10', 'PF05015.8', 'PF05016.9', 'PF05017.9', 'PF05018.8', 'PF05019.8', 'PF05020.10', 'PF05021.10',
     'PF05022.7', 'PF05023.9', 'PF05024.10', 'PF05025.8', 'PF05026.8', 'PF05028.9', 'PF05029.8', 'PF05030.7',
     'PF05031.7', 'PF05032.7', 'PF05033.11', 'PF05034.8', 'PF05035.7', 'PF05036.8', 'PF05037.8', 'PF05038.8',
     'PF05039.7', 'PF05041.10', 'PF05042.8', 'PF05043.8', 'PF05044.7', 'PF05045.7', 'PF05046.9', 'PF05047.11',
     'PF05048.8', 'PF05049.8', 'PF05050.7', 'PF05051.8', 'PF05052.7', 'PF05053.8', 'PF05054.7', 'PF05055.7',
     'PF05056.7', 'PF05057.9', 'PF05058.7', 'PF05059.7', 'PF05060.9', 'PF05061.8', 'PF05062.7', 'PF05063.9',
     'PF05064.8', 'PF05065.8', 'PF05066.8', 'PF05067.7', 'PF05068.7', 'PF05069.8', 'PF05071.11', 'PF05072.8',
     'PF05073.7', 'PF05075.9', 'PF05076.8', 'PF05077.7', 'PF05078.7', 'PF05079.7', 'PF05080.7', 'PF05081.7',
     'PF05082.8', 'PF05083.8', 'PF05084.8', 'PF05085.7', 'PF05086.7', 'PF05087.7', 'PF05088.7', 'PF05089.7',
     'PF05090.9', 'PF05091.7', 'PF05092.7', 'PF05093.8', 'PF05094.7', 'PF05095.7', 'PF05096.7', 'PF05097.7',
     'PF05098.7', 'PF05099.8', 'PF05100.7', 'PF05101.8', 'PF05102.7', 'PF05103.8', 'PF05104.7', 'PF05105.7',
     'PF05106.7', 'PF05107.7', 'PF05108.8', 'PF05109.8', 'PF05110.8', 'PF05111.7', 'PF05112.7', 'PF05113.8',
     'PF05114.8', 'PF05115.9', 'PF05116.8', 'PF05117.7', 'PF05118.10', 'PF05119.7', 'PF05120.7', 'PF05121.7',
     'PF05122.8', 'PF05123.7', 'PF05124.7', 'PF05125.7', 'PF05127.9', 'PF05128.7', 'PF05129.8', 'PF05130.7',
     'PF05131.9', 'PF05132.9', 'PF05133.9', 'PF05134.8', 'PF05135.8', 'PF05136.8', 'PF05137.8', 'PF05138.7',
     'PF05139.9', 'PF05140.9', 'PF05141.7', 'PF05142.7', 'PF05144.9', 'PF05145.7', 'PF05147.8', 'PF05148.10',
     'PF05149.7', 'PF05150.7', 'PF05151.7', 'PF05152.7', 'PF05153.10', 'PF05154.11', 'PF05155.10', 'PF05157.10',
     'PF05158.7', 'PF05159.9', 'PF05160.8', 'PF05161.8', 'PF05162.8', 'PF05163.7', 'PF05164.8', 'PF05165.7',
     'PF05166.8', 'PF05167.7', 'PF05168.9', 'PF05170.9', 'PF05171.7', 'PF05172.8', 'PF05173.9', 'PF05174.7',
     'PF05175.9', 'PF05176.9', 'PF05177.7', 'PF05178.7', 'PF05179.9', 'PF05180.7', 'PF05181.7', 'PF05182.8',
     'PF05183.7', 'PF05184.10', 'PF05185.11', 'PF05186.8', 'PF05187.8', 'PF05188.12', 'PF05189.8', 'PF05190.13',
     'PF05191.9', 'PF05192.13', 'PF05193.16', 'PF05194.7', 'PF05195.11', 'PF05196.8', 'PF05197.8', 'PF05198.11',
     'PF05199.8', 'PF05201.10', 'PF05202.7', 'PF05203.11', 'PF05204.9', 'PF05205.7', 'PF05206.9', 'PF05207.8',
     'PF05208.8', 'PF05209.8', 'PF05210.8', 'PF05211.7', 'PF05212.7', 'PF05213.7', 'PF05214.7', 'PF05215.8',
     'PF05216.8', 'PF05217.7', 'PF05218.9', 'PF05219.7', 'PF05220.7', 'PF05221.12', 'PF05222.10', 'PF05223.6',
     'PF05224.7', 'PF05225.11', 'PF05226.6', 'PF05227.8', 'PF05228.8', 'PF05229.10', 'PF05230.6', 'PF05231.9',
     'PF05232.7', 'PF05233.8', 'PF05234.6', 'PF05235.9', 'PF05236.9', 'PF05237.8', 'PF05238.8', 'PF05239.11',
     'PF05240.9', 'PF05241.7', 'PF05242.6', 'PF05244.6', 'PF05246.6', 'PF05247.8', 'PF05248.7', 'PF05250.6',
     'PF05251.7', 'PF05253.7', 'PF05254.7', 'PF05255.6', 'PF05256.7', 'PF05257.11', 'PF05258.7', 'PF05259.6',
     'PF05261.6', 'PF05262.6', 'PF05263.6', 'PF05264.6', 'PF05265.8', 'PF05266.9', 'PF05267.7', 'PF05268.6',
     'PF05269.6', 'PF05270.8', 'PF05271.6', 'PF05272.6', 'PF05273.8', 'PF05274.6', 'PF05275.6', 'PF05276.9',
     'PF05277.7', 'PF05278.7', 'PF05279.6', 'PF05280.6', 'PF05281.6', 'PF05282.6', 'PF05283.6', 'PF05284.7',
     'PF05285.7', 'PF05287.7', 'PF05288.6', 'PF05289.6', 'PF05290.6', 'PF05291.6', 'PF05292.6', 'PF05293.6',
     'PF05294.8', 'PF05295.6', 'PF05296.8', 'PF05297.6', 'PF05298.6', 'PF05299.7', 'PF05300.6', 'PF05301.6',
     'PF05302.6', 'PF05303.7', 'PF05304.7', 'PF05305.9', 'PF05306.6', 'PF05307.6', 'PF05308.6', 'PF05309.6',
     'PF05310.7', 'PF05311.6', 'PF05313.7', 'PF05314.6', 'PF05315.6', 'PF05316.7', 'PF05317.6', 'PF05318.7',
     'PF05320.7', 'PF05321.6', 'PF05322.6', 'PF05323.7', 'PF05324.8', 'PF05325.6', 'PF05326.6', 'PF05327.6',
     'PF05328.7', 'PF05331.6', 'PF05332.6', 'PF05334.8', 'PF05335.8', 'PF05336.8', 'PF05337.6', 'PF05338.7',
     'PF05339.6', 'PF05340.7', 'PF05341.6', 'PF05342.9', 'PF05343.9', 'PF05344.6', 'PF05345.7', 'PF05346.6',
     'PF05347.10', 'PF05348.6', 'PF05349.7', 'PF05350.7', 'PF05351.6', 'PF05352.7', 'PF05353.6', 'PF05354.6',
     'PF05355.6', 'PF05356.6', 'PF05357.8', 'PF05358.6', 'PF05359.6', 'PF05360.9', 'PF05361.7', 'PF05362.8',
     'PF05363.7', 'PF05364.9', 'PF05365.7', 'PF05366.6', 'PF05367.6', 'PF05368.8', 'PF05369.7', 'PF05370.6',
     'PF05371.7', 'PF05372.6', 'PF05373.6', 'PF05374.7', 'PF05375.8', 'PF05377.6', 'PF05378.8', 'PF05379.6',
     'PF05380.8', 'PF05381.7', 'PF05382.8', 'PF05383.12', 'PF05384.6', 'PF05385.6', 'PF05386.6', 'PF05387.6',
     'PF05388.6', 'PF05389.7', 'PF05390.6', 'PF05391.6', 'PF05392.6', 'PF05393.6', 'PF05394.6', 'PF05395.7',
     'PF05396.6', 'PF05397.7', 'PF05398.6', 'PF05399.6', 'PF05400.8', 'PF05401.6', 'PF05402.7', 'PF05403.6',
     'PF05404.7', 'PF05405.9', 'PF05406.10', 'PF05407.7', 'PF05408.6', 'PF05409.8', 'PF05410.8', 'PF05411.7',
     'PF05412.7', 'PF05413.6', 'PF05414.6', 'PF05415.6', 'PF05416.7', 'PF05417.6', 'PF05418.6', 'PF05419.7',
     'PF05420.6', 'PF05421.6', 'PF05422.7', 'PF05423.8', 'PF05424.6', 'PF05425.8', 'PF05426.7', 'PF05427.6',
     'PF05428.6', 'PF05430.6', 'PF05431.6', 'PF05432.6', 'PF05433.10', 'PF05434.6', 'PF05435.6', 'PF05436.6',
     'PF05437.7', 'PF05438.7', 'PF05439.7', 'PF05440.7', 'PF05443.6', 'PF05444.7', 'PF05445.6', 'PF05448.7',
     'PF05449.6', 'PF05450.10', 'PF05451.6', 'PF05452.6', 'PF05453.7', 'PF05454.6', 'PF05455.6', 'PF05456.6',
     'PF05458.7', 'PF05459.7', 'PF05460.8', 'PF05461.6', 'PF05462.6', 'PF05463.6', 'PF05464.6', 'PF05465.8',
     'PF05466.7', 'PF05467.6', 'PF05470.7', 'PF05472.6', 'PF05473.7', 'PF05474.6', 'PF05475.6', 'PF05476.6',
     'PF05477.6', 'PF05478.6', 'PF05479.6', 'PF05480.6', 'PF05481.7', 'PF05482.7', 'PF05483.7', 'PF05484.6',
     'PF05485.7', 'PF05486.7', 'PF05488.8', 'PF05489.7', 'PF05491.8', 'PF05493.8', 'PF05494.7', 'PF05495.7',
     'PF05496.7', 'PF05497.7', 'PF05498.6', 'PF05499.7', 'PF05501.6', 'PF05502.8', 'PF05503.7', 'PF05504.6',
     'PF05505.7', 'PF05506.7', 'PF05507.6', 'PF05508.6', 'PF05509.6', 'PF05510.8', 'PF05511.6', 'PF05512.6',
     'PF05513.6', 'PF05514.6', 'PF05515.6', 'PF05517.7', 'PF05518.6', 'PF05520.6', 'PF05521.6', 'PF05522.6',
     'PF05523.6', 'PF05524.8', 'PF05525.8', 'PF05526.6', 'PF05527.6', 'PF05528.6', 'PF05529.7', 'PF05531.7',
     'PF05532.7', 'PF05533.7', 'PF05534.7', 'PF05535.7', 'PF05536.6', 'PF05537.6', 'PF05538.6', 'PF05539.6',
     'PF05540.6', 'PF05541.7', 'PF05542.6', 'PF05543.8', 'PF05544.6', 'PF05545.6', 'PF05546.6', 'PF05547.6',
     'PF05548.6', 'PF05549.6', 'PF05550.6', 'PF05551.6', 'PF05552.7', 'PF05553.6', 'PF05554.6', 'PF05555.6',
     'PF05556.6', 'PF05557.8', 'PF05558.7', 'PF05559.6', 'PF05560.6', 'PF05561.6', 'PF05562.6', 'PF05563.7',
     'PF05564.7', 'PF05565.6', 'PF05566.7', 'PF05567.6', 'PF05568.6', 'PF05569.6', 'PF05570.6', 'PF05571.7',
     'PF05572.8', 'PF05573.7', 'PF05575.6', 'PF05576.6', 'PF05577.7', 'PF05578.7', 'PF05579.8', 'PF05580.7',
     'PF05581.7', 'PF05582.7', 'PF05584.6', 'PF05585.7', 'PF05586.6', 'PF05587.8', 'PF05588.6', 'PF05589.6',
     'PF05590.6', 'PF05591.7', 'PF05592.6', 'PF05593.9', 'PF05594.9', 'PF05595.6', 'PF05596.6', 'PF05597.6',
     'PF05598.6', 'PF05599.6', 'PF05600.7', 'PF05602.7', 'PF05603.7', 'PF05604.6', 'PF05605.7', 'PF05606.6',
     'PF05608.7', 'PF05609.7', 'PF05610.6', 'PF05611.6', 'PF05612.7', 'PF05613.6', 'PF05614.6', 'PF05615.8',
     'PF05616.8', 'PF05617.6', 'PF05618.6', 'PF05619.6', 'PF05620.6', 'PF05621.6', 'PF05622.7', 'PF05623.7',
     'PF05624.9', 'PF05625.6', 'PF05626.6', 'PF05627.6', 'PF05628.7', 'PF05629.6', 'PF05630.6', 'PF05631.9',
     'PF05632.6', 'PF05633.6', 'PF05634.6', 'PF05635.6', 'PF05636.6', 'PF05637.7', 'PF05638.7', 'PF05639.6',
     'PF05640.9', 'PF05641.7', 'PF05642.6', 'PF05643.6', 'PF05644.6', 'PF05645.8', 'PF05647.6', 'PF05648.9',
     'PF05649.8', 'PF05650.6', 'PF05651.8', 'PF05652.7', 'PF05653.9', 'PF05655.6', 'PF05656.9', 'PF05657.6',
     'PF05658.9', 'PF05659.6', 'PF05660.6', 'PF05661.7', 'PF05662.9', 'PF05663.6', 'PF05664.6', 'PF05666.6',
     'PF05667.6', 'PF05669.7', 'PF05670.8', 'PF05671.6', 'PF05672.6', 'PF05673.8', 'PF05674.7', 'PF05675.7',
     'PF05676.8', 'PF05677.7', 'PF05678.9', 'PF05679.11', 'PF05680.7', 'PF05681.9', 'PF05683.7', 'PF05684.7',
     'PF05685.7', 'PF05686.7', 'PF05687.8', 'PF05688.6', 'PF05689.6', 'PF05690.9', 'PF05691.7', 'PF05692.7',
     'PF05693.8', 'PF05694.6', 'PF05695.7', 'PF05696.6', 'PF05697.8', 'PF05698.9', 'PF05699.9', 'PF05700.6',
     'PF05701.6', 'PF05702.6', 'PF05703.6', 'PF05704.7', 'PF05705.9', 'PF05706.7', 'PF05707.7', 'PF05708.7',
     'PF05709.6', 'PF05710.7', 'PF05711.6', 'PF05712.8', 'PF05713.6', 'PF05714.6', 'PF05715.8', 'PF05716.8',
     'PF05717.8', 'PF05718.7', 'PF05719.6', 'PF05720.7', 'PF05721.8', 'PF05722.7', 'PF05724.6', 'PF05725.7',
     'PF05726.8', 'PF05727.6', 'PF05728.7', 'PF05729.7', 'PF05730.6', 'PF05731.6', 'PF05732.6', 'PF05733.6',
     'PF05734.6', 'PF05735.7', 'PF05736.6', 'PF05737.7', 'PF05738.8', 'PF05739.14', 'PF05741.8', 'PF05742.7',
     'PF05743.8', 'PF05744.6', 'PF05745.6', 'PF05746.10', 'PF05748.6', 'PF05749.6', 'PF05750.6', 'PF05751.6',
     'PF05752.6', 'PF05753.9', 'PF05754.9', 'PF05755.7', 'PF05756.6', 'PF05757.6', 'PF05758.7', 'PF05760.7',
     'PF05761.9', 'PF05762.9', 'PF05763.7', 'PF05764.8', 'PF05766.7', 'PF05767.7', 'PF05768.9', 'PF05769.6',
     'PF05770.6', 'PF05771.6', 'PF05772.7', 'PF05773.17', 'PF05774.6', 'PF05775.7', 'PF05776.7', 'PF05777.7',
     'PF05778.7', 'PF05781.7', 'PF05782.6', 'PF05783.6', 'PF05784.6', 'PF05785.7', 'PF05786.9', 'PF05787.8',
     'PF05788.7', 'PF05789.6', 'PF05790.10', 'PF05791.6', 'PF05792.8', 'PF05793.7', 'PF05794.8', 'PF05795.6',
     'PF05796.7', 'PF05797.6', 'PF05798.6', 'PF05800.6', 'PF05801.6', 'PF05802.6', 'PF05803.7', 'PF05804.7',
     'PF05805.7', 'PF05806.7', 'PF05808.6', 'PF05810.7', 'PF05811.8', 'PF05812.7', 'PF05813.7', 'PF05814.6',
     'PF05815.6', 'PF05816.6', 'PF05817.9', 'PF05818.7', 'PF05819.6', 'PF05820.6', 'PF05821.6', 'PF05822.7',
     'PF05823.7', 'PF05824.7', 'PF05825.6', 'PF05826.7', 'PF05827.7', 'PF05829.7', 'PF05830.6', 'PF05831.6',
     'PF05832.7', 'PF05833.6', 'PF05834.7', 'PF05835.7', 'PF05836.7', 'PF05837.7', 'PF05838.7', 'PF05839.6',
     'PF05840.8', 'PF05841.6', 'PF05842.6', 'PF05843.9', 'PF05844.7', 'PF05845.7', 'PF05846.7', 'PF05847.6',
     'PF05848.6', 'PF05849.6', 'PF05851.6', 'PF05852.6', 'PF05853.7', 'PF05854.6', 'PF05856.7', 'PF05857.6',
     'PF05858.7', 'PF05859.7', 'PF05860.8', 'PF05861.7', 'PF05862.6', 'PF05864.7', 'PF05865.6', 'PF05866.6',
     'PF05867.6', 'PF05868.6', 'PF05869.6', 'PF05870.6', 'PF05871.7', 'PF05872.7', 'PF05873.7', 'PF05874.6',
     'PF05875.7', 'PF05876.7', 'PF05878.6', 'PF05879.7', 'PF05880.6', 'PF05881.7', 'PF05883.6', 'PF05884.7',
     'PF05886.6', 'PF05887.6', 'PF05889.8', 'PF05890.7', 'PF05891.7', 'PF05892.6', 'PF05893.9', 'PF05894.7',
     'PF05895.7', 'PF05896.6', 'PF05899.7', 'PF05901.6', 'PF05902.8', 'PF05903.9', 'PF05904.6', 'PF05906.6',
     'PF05907.8', 'PF05908.6', 'PF05910.7', 'PF05911.6', 'PF05912.6', 'PF05913.6', 'PF05914.7', 'PF05915.7',
     'PF05916.6', 'PF05917.6', 'PF05918.6', 'PF05919.6', 'PF05920.6', 'PF05922.11', 'PF05923.7', 'PF05924.6',
     'PF05925.7', 'PF05926.6', 'PF05927.6', 'PF05928.6', 'PF05929.6', 'PF05930.7', 'PF05931.6', 'PF05932.8',
     'PF05933.8', 'PF05934.6', 'PF05935.6', 'PF05936.7', 'PF05937.6', 'PF05938.6', 'PF05939.8', 'PF05940.7',
     'PF05941.8', 'PF05942.6', 'PF05943.7', 'PF05944.7', 'PF05946.7', 'PF05947.7', 'PF05949.7', 'PF05950.6',
     'PF05951.8', 'PF05952.7', 'PF05953.7', 'PF05954.6', 'PF05955.6', 'PF05956.6', 'PF05957.8', 'PF05958.6',
     'PF05959.6', 'PF05960.6', 'PF05961.6', 'PF05962.6', 'PF05963.6', 'PF05964.9', 'PF05965.9', 'PF05966.7',
     'PF05968.6', 'PF05969.6', 'PF05970.9', 'PF05971.7', 'PF05972.6', 'PF05973.9', 'PF05974.7', 'PF05975.7',
     'PF05977.8', 'PF05978.11', 'PF05979.7', 'PF05980.7', 'PF05981.7', 'PF05982.7', 'PF05983.6', 'PF05984.7',
     'PF05985.6', 'PF05986.9', 'PF05987.8', 'PF05988.7', 'PF05989.7', 'PF05990.7', 'PF05991.6', 'PF05992.7',
     'PF05993.7', 'PF05994.6', 'PF05995.7', 'PF05996.7', 'PF05997.7', 'PF05999.6', 'PF06001.8', 'PF06002.7',
     'PF06003.7', 'PF06004.7', 'PF06005.7', 'PF06006.7', 'PF06007.6', 'PF06008.9', 'PF06009.7', 'PF06011.7',
     'PF06012.7', 'PF06013.7', 'PF06014.6', 'PF06015.7', 'PF06016.6', 'PF06017.8', 'PF06018.9', 'PF06019.6',
     'PF06020.6', 'PF06021.6', 'PF06022.6', 'PF06023.7', 'PF06024.7', 'PF06025.7', 'PF06026.9', 'PF06027.7',
     'PF06028.6', 'PF06029.6', 'PF06030.7', 'PF06031.8', 'PF06032.7', 'PF06033.6', 'PF06034.6', 'PF06035.6',
     'PF06037.6', 'PF06039.10', 'PF06040.6', 'PF06041.6', 'PF06042.6', 'PF06043.6', 'PF06044.7', 'PF06045.6',
     'PF06046.8', 'PF06047.6', 'PF06048.6', 'PF06049.7', 'PF06050.8', 'PF06051.7', 'PF06052.7', 'PF06053.6',
     'PF06054.6', 'PF06055.7', 'PF06056.7', 'PF06057.6', 'PF06058.8', 'PF06059.7', 'PF06060.7', 'PF06061.6',
     'PF06062.6', 'PF06064.6', 'PF06066.6', 'PF06067.6', 'PF06068.8', 'PF06069.6', 'PF06070.6', 'PF06071.8',
     'PF06072.6', 'PF06073.7', 'PF06074.7', 'PF06075.7', 'PF06076.7', 'PF06078.6', 'PF06079.6', 'PF06080.7',
     'PF06081.6', 'PF06082.6', 'PF06083.6', 'PF06084.6', 'PF06085.6', 'PF06086.7', 'PF06087.7', 'PF06088.6',
     'PF06089.7', 'PF06090.7', 'PF06092.7', 'PF06093.8', 'PF06094.7', 'PF06096.6', 'PF06097.6', 'PF06098.6',
     'PF06099.6', 'PF06100.6', 'PF06101.6', 'PF06102.7', 'PF06103.6', 'PF06105.7', 'PF06106.6', 'PF06107.6',
     'PF06108.7', 'PF06109.8', 'PF06110.6', 'PF06112.6', 'PF06113.7', 'PF06114.8', 'PF06115.6', 'PF06116.7',
     'PF06117.6', 'PF06119.9', 'PF06120.6', 'PF06121.9', 'PF06122.6', 'PF06123.7', 'PF06124.6', 'PF06125.6',
     'PF06126.6', 'PF06127.6', 'PF06128.6', 'PF06129.7', 'PF06130.7', 'PF06131.6', 'PF06133.6', 'PF06134.6',
     'PF06135.7', 'PF06136.8', 'PF06138.7', 'PF06139.7', 'PF06140.8', 'PF06141.6', 'PF06143.6', 'PF06144.8',
     'PF06145.6', 'PF06146.7', 'PF06147.6', 'PF06148.6', 'PF06149.7', 'PF06150.7', 'PF06151.8', 'PF06152.6',
     'PF06153.6', 'PF06154.6', 'PF06155.7', 'PF06156.8', 'PF06157.6', 'PF06159.8', 'PF06160.7', 'PF06161.6',
     'PF06162.7', 'PF06163.6', 'PF06165.6', 'PF06166.7', 'PF06167.7', 'PF06168.6', 'PF06169.7', 'PF06170.7',
     'PF06172.6', 'PF06173.7', 'PF06174.6', 'PF06175.6', 'PF06176.6', 'PF06177.6', 'PF06178.8', 'PF06179.7',
     'PF06180.6', 'PF06181.6', 'PF06182.6', 'PF06183.8', 'PF06184.6', 'PF06185.7', 'PF06186.6', 'PF06187.6',
     'PF06188.7', 'PF06189.7', 'PF06191.7', 'PF06193.6', 'PF06194.6', 'PF06195.8', 'PF06196.7', 'PF06197.8',
     'PF06198.6', 'PF06199.6', 'PF06200.9', 'PF06201.8', 'PF06202.9', 'PF06203.9', 'PF06204.6', 'PF06205.6',
     'PF06206.6', 'PF06207.6', 'PF06208.6', 'PF06209.8', 'PF06210.6', 'PF06211.7', 'PF06212.7', 'PF06213.7',
     'PF06214.6', 'PF06215.6', 'PF06216.6', 'PF06217.7', 'PF06218.6', 'PF06219.7', 'PF06220.7', 'PF06221.8',
     'PF06222.6', 'PF06223.7', 'PF06224.7', 'PF06226.8', 'PF06227.7', 'PF06228.8', 'PF06229.7', 'PF06230.6',
     'PF06231.6', 'PF06232.6', 'PF06233.7', 'PF06234.7', 'PF06235.6', 'PF06236.6', 'PF06237.7', 'PF06238.6',
     'PF06239.6', 'PF06240.8', 'PF06241.7', 'PF06242.6', 'PF06243.6', 'PF06244.7', 'PF06245.6', 'PF06246.7',
     'PF06247.6', 'PF06248.8', 'PF06249.7', 'PF06250.6', 'PF06251.6', 'PF06252.7', 'PF06253.6', 'PF06254.6',
     'PF06255.8', 'PF06256.6', 'PF06257.6', 'PF06258.6', 'PF06259.7', 'PF06260.7', 'PF06261.6', 'PF06262.6',
     'PF06265.6', 'PF06266.7', 'PF06267.7', 'PF06268.8', 'PF06269.7', 'PF06270.6', 'PF06271.7', 'PF06273.6',
     'PF06275.6', 'PF06276.7', 'PF06277.6', 'PF06278.6', 'PF06279.6', 'PF06280.7', 'PF06281.7', 'PF06282.6',
     'PF06283.6', 'PF06284.6', 'PF06286.6', 'PF06287.6', 'PF06288.8', 'PF06289.6', 'PF06290.6', 'PF06291.6',
     'PF06292.12', 'PF06293.9', 'PF06294.6', 'PF06295.7', 'PF06296.7', 'PF06297.9', 'PF06298.6', 'PF06299.7',
     'PF06300.7', 'PF06301.6', 'PF06303.7', 'PF06304.6', 'PF06305.6', 'PF06306.6', 'PF06307.7', 'PF06308.6',
     'PF06309.6', 'PF06311.7', 'PF06312.7', 'PF06313.6', 'PF06314.6', 'PF06315.6', 'PF06316.6', 'PF06317.6',
     'PF06319.7', 'PF06320.8', 'PF06321.6', 'PF06322.6', 'PF06323.6', 'PF06324.6', 'PF06325.8', 'PF06326.7',
     'PF06327.9', 'PF06328.6', 'PF06330.6', 'PF06331.7', 'PF06333.7', 'PF06334.6', 'PF06335.7', 'PF06336.6',
     'PF06337.7', 'PF06338.6', 'PF06339.7', 'PF06340.6', 'PF06341.6', 'PF06342.7', 'PF06344.6', 'PF06345.6',
     'PF06346.7', 'PF06347.8', 'PF06348.6', 'PF06350.7', 'PF06351.6', 'PF06353.7', 'PF06355.8', 'PF06356.6',
     'PF06357.6', 'PF06358.6', 'PF06360.6', 'PF06361.6', 'PF06362.6', 'PF06363.6', 'PF06364.7', 'PF06365.7',
     'PF06366.8', 'PF06367.11', 'PF06368.6', 'PF06369.7', 'PF06370.6', 'PF06371.8', 'PF06372.7', 'PF06373.6',
     'PF06374.6', 'PF06375.6', 'PF06376.7', 'PF06377.6', 'PF06378.6', 'PF06379.7', 'PF06380.6', 'PF06381.6',
     'PF06382.6', 'PF06384.6', 'PF06385.7', 'PF06386.6', 'PF06387.6', 'PF06388.6', 'PF06389.6', 'PF06390.7',
     'PF06391.8', 'PF06392.6', 'PF06393.6', 'PF06394.8', 'PF06395.6', 'PF06396.6', 'PF06397.7', 'PF06398.6',
     'PF06399.8', 'PF06400.6', 'PF06401.6', 'PF06403.6', 'PF06404.7', 'PF06405.6', 'PF06406.6', 'PF06407.6',
     'PF06409.6', 'PF06411.6', 'PF06412.6', 'PF06413.6', 'PF06414.7', 'PF06415.8', 'PF06416.7', 'PF06417.7',
     'PF06418.9', 'PF06419.6', 'PF06420.7', 'PF06421.7', 'PF06422.7', 'PF06423.7', 'PF06424.7', 'PF06426.9',
     'PF06427.6', 'PF06428.6', 'PF06429.8', 'PF06430.7', 'PF06431.6', 'PF06432.6', 'PF06433.6', 'PF06434.8',
     'PF06435.6', 'PF06436.6', 'PF06437.6', 'PF06438.7', 'PF06439.6', 'PF06440.6', 'PF06441.7', 'PF06442.6',
     'PF06443.6', 'PF06444.6', 'PF06445.10', 'PF06446.7', 'PF06448.6', 'PF06449.6', 'PF06450.7', 'PF06451.6',
     'PF06452.6', 'PF06453.6', 'PF06454.6', 'PF06455.6', 'PF06456.8', 'PF06457.6', 'PF06458.7', 'PF06459.7',
     'PF06460.7', 'PF06461.6', 'PF06462.7', 'PF06463.8', 'PF06464.6', 'PF06465.8', 'PF06466.6', 'PF06467.9',
     'PF06468.8', 'PF06469.6', 'PF06470.8', 'PF06471.7', 'PF06472.10', 'PF06473.7', 'PF06474.7', 'PF06475.6',
     'PF06476.7', 'PF06477.8', 'PF06478.8', 'PF06479.7', 'PF06480.10', 'PF06481.9', 'PF06482.6', 'PF06483.6',
     'PF06484.7', 'PF06485.6', 'PF06486.6', 'PF06487.7', 'PF06488.6', 'PF06489.6', 'PF06490.6', 'PF06491.6',
     'PF06493.6', 'PF06495.6', 'PF06496.6', 'PF06497.6', 'PF06500.6', 'PF06501.6', 'PF06502.6', 'PF06503.6',
     'PF06504.6', 'PF06505.6', 'PF06506.6', 'PF06507.8', 'PF06508.8', 'PF06510.6', 'PF06511.6', 'PF06512.8',
     'PF06513.6', 'PF06514.6', 'PF06515.6', 'PF06516.6', 'PF06517.6', 'PF06518.6', 'PF06519.6', 'PF06521.6',
     'PF06522.6', 'PF06523.6', 'PF06524.7', 'PF06525.6', 'PF06526.7', 'PF06527.6', 'PF06528.7', 'PF06529.6',
     'PF06530.7', 'PF06531.6', 'PF06532.6', 'PF06533.7', 'PF06534.8', 'PF06535.7', 'PF06536.6', 'PF06537.6',
     'PF06540.6', 'PF06541.6', 'PF06542.6', 'PF06543.7', 'PF06544.7', 'PF06545.6', 'PF06546.6', 'PF06547.7',
     'PF06548.6', 'PF06549.7', 'PF06550.6', 'PF06551.7', 'PF06552.7', 'PF06553.7', 'PF06554.7', 'PF06556.6',
     'PF06557.6', 'PF06558.7', 'PF06559.6', 'PF06560.6', 'PF06563.6', 'PF06564.7', 'PF06565.7', 'PF06566.6',
     'PF06567.6', 'PF06568.6', 'PF06569.6', 'PF06570.6', 'PF06572.7', 'PF06573.6', 'PF06574.7', 'PF06575.7',
     'PF06576.6', 'PF06577.7', 'PF06578.7', 'PF06579.7', 'PF06580.8', 'PF06581.7', 'PF06582.7', 'PF06583.7',
     'PF06584.8', 'PF06585.6', 'PF06586.6', 'PF06587.6', 'PF06588.6', 'PF06589.6', 'PF06590.6', 'PF06591.6',
     'PF06592.8', 'PF06593.6', 'PF06594.6', 'PF06595.6', 'PF06596.6', 'PF06597.6', 'PF06598.6', 'PF06599.6',
     'PF06600.6', 'PF06601.7', 'PF06602.9', 'PF06603.6', 'PF06605.6', 'PF06607.6', 'PF06608.6', 'PF06609.8',
     'PF06610.8', 'PF06611.7', 'PF06612.6', 'PF06613.6', 'PF06614.6', 'PF06615.6', 'PF06616.6', 'PF06617.8',
     'PF06618.6', 'PF06619.6', 'PF06620.6', 'PF06621.7', 'PF06622.6', 'PF06623.6', 'PF06624.7', 'PF06625.6',
     'PF06626.7', 'PF06627.6', 'PF06628.7', 'PF06629.7', 'PF06630.6', 'PF06631.6', 'PF06632.7', 'PF06633.6',
     'PF06634.7', 'PF06635.7', 'PF06636.6', 'PF06637.6', 'PF06638.6', 'PF06639.6', 'PF06640.6', 'PF06643.6',
     'PF06644.6', 'PF06645.8', 'PF06646.6', 'PF06648.6', 'PF06649.7', 'PF06650.7', 'PF06651.6', 'PF06652.7',
     'PF06653.6', 'PF06656.6', 'PF06657.8', 'PF06658.7', 'PF06661.6', 'PF06662.8', 'PF06663.8', 'PF06664.7',
     'PF06666.6', 'PF06667.7', 'PF06668.7', 'PF06669.6', 'PF06670.6', 'PF06671.6', 'PF06672.6', 'PF06673.6',
     'PF06674.6', 'PF06675.6', 'PF06676.6', 'PF06677.7', 'PF06678.6', 'PF06679.7', 'PF06680.6', 'PF06681.8',
     'PF06682.7', 'PF06683.6', 'PF06684.6', 'PF06685.6', 'PF06686.6', 'PF06687.7', 'PF06688.6', 'PF06689.8',
     'PF06690.6', 'PF06691.6', 'PF06692.6', 'PF06693.6', 'PF06694.6', 'PF06695.6', 'PF06696.6', 'PF06697.7',
     'PF06698.6', 'PF06699.6', 'PF06701.8', 'PF06702.7', 'PF06703.6', 'PF06705.6', 'PF06706.6', 'PF06707.6',
     'PF06708.6', 'PF06709.6', 'PF06711.6', 'PF06712.6', 'PF06713.6', 'PF06714.6', 'PF06715.7', 'PF06716.6',
     'PF06717.6', 'PF06718.6', 'PF06719.8', 'PF06720.6', 'PF06721.6', 'PF06722.7', 'PF06723.8', 'PF06724.6',
     'PF06725.6', 'PF06726.7', 'PF06727.6', 'PF06728.8', 'PF06729.7', 'PF06730.6', 'PF06732.6', 'PF06733.10',
     'PF06734.7', 'PF06736.6', 'PF06737.9', 'PF06738.7', 'PF06739.6', 'PF06740.7', 'PF06741.8', 'PF06742.6',
     'PF06743.10', 'PF06744.7', 'PF06745.8', 'PF06746.6', 'PF06747.8', 'PF06748.7', 'PF06749.7', 'PF06750.8',
     'PF06751.6', 'PF06752.7', 'PF06753.7', 'PF06754.7', 'PF06755.7', 'PF06756.6', 'PF06757.8', 'PF06758.8',
     'PF06760.6', 'PF06761.7', 'PF06762.9', 'PF06763.6', 'PF06764.6', 'PF06766.6', 'PF06767.6', 'PF06769.8',
     'PF06770.6', 'PF06771.6', 'PF06772.6', 'PF06773.6', 'PF06775.9', 'PF06776.7', 'PF06777.6', 'PF06778.7',
     'PF06779.9', 'PF06780.6', 'PF06781.7', 'PF06782.6', 'PF06783.6', 'PF06784.6', 'PF06785.6', 'PF06786.7',
     'PF06787.6', 'PF06788.8', 'PF06789.7', 'PF06790.6', 'PF06791.8', 'PF06792.6', 'PF06793.7', 'PF06794.7',
     'PF06795.6', 'PF06796.6', 'PF06797.6', 'PF06798.7', 'PF06799.6', 'PF06800.7', 'PF06802.6', 'PF06803.7',
     'PF06804.6', 'PF06805.7', 'PF06806.7', 'PF06807.9', 'PF06808.7', 'PF06809.6', 'PF06810.6', 'PF06812.7',
     'PF06813.8', 'PF06814.8', 'PF06815.8', 'PF06816.8', 'PF06817.9', 'PF06818.10', 'PF06819.6', 'PF06820.7',
     'PF06821.8', 'PF06822.7', 'PF06823.7', 'PF06824.6', 'PF06825.7', 'PF06826.7', 'PF06827.9', 'PF06830.6',
     'PF06831.9', 'PF06832.7', 'PF06833.6', 'PF06834.6', 'PF06835.8', 'PF06836.7', 'PF06837.6', 'PF06838.6',
     'PF06839.7', 'PF06840.6', 'PF06841.7', 'PF06842.7', 'PF06844.6', 'PF06847.6', 'PF06848.6', 'PF06849.7',
     'PF06850.6', 'PF06851.6', 'PF06852.7', 'PF06853.7', 'PF06854.6', 'PF06855.7', 'PF06856.6', 'PF06857.6',
     'PF06858.9', 'PF06859.7', 'PF06861.6', 'PF06862.7', 'PF06863.7', 'PF06864.7', 'PF06865.6', 'PF06866.6',
     'PF06868.6', 'PF06869.7', 'PF06870.7', 'PF06871.6', 'PF06872.6', 'PF06873.6', 'PF06874.6', 'PF06875.6',
     'PF06876.7', 'PF06877.6', 'PF06878.6', 'PF06880.6', 'PF06881.6', 'PF06882.7', 'PF06883.7', 'PF06884.6',
     'PF06886.6', 'PF06887.9', 'PF06888.7', 'PF06889.6', 'PF06890.7', 'PF06891.6', 'PF06892.6', 'PF06894.6',
     'PF06896.6', 'PF06897.7', 'PF06898.6', 'PF06899.6', 'PF06900.6', 'PF06901.7', 'PF06902.6', 'PF06903.7',
     'PF06904.7', 'PF06905.8', 'PF06906.6', 'PF06907.7', 'PF06908.6', 'PF06910.6', 'PF06911.7', 'PF06912.6',
     'PF06916.8', 'PF06917.7', 'PF06918.9', 'PF06919.6', 'PF06920.8', 'PF06922.6', 'PF06923.6', 'PF06924.6',
     'PF06925.6', 'PF06926.6', 'PF06929.6', 'PF06930.7', 'PF06931.6', 'PF06932.6', 'PF06933.6', 'PF06934.6',
     'PF06935.6', 'PF06936.6', 'PF06937.6', 'PF06938.6', 'PF06939.6', 'PF06940.6', 'PF06941.7', 'PF06942.7',
     'PF06943.7', 'PF06945.8', 'PF06946.6', 'PF06947.7', 'PF06949.6', 'PF06950.6', 'PF06951.6', 'PF06952.6',
     'PF06953.6', 'PF06954.6', 'PF06955.7', 'PF06956.6', 'PF06957.6', 'PF06958.7', 'PF06959.6', 'PF06961.8',
     'PF06962.7', 'PF06963.7', 'PF06964.7', 'PF06965.7', 'PF06966.7', 'PF06967.6', 'PF06968.8', 'PF06969.11',
     'PF06970.6', 'PF06971.8', 'PF06972.6', 'PF06973.7', 'PF06974.8', 'PF06975.6', 'PF06977.6', 'PF06978.6',
     'PF06979.7', 'PF06980.6', 'PF06983.8', 'PF06984.8', 'PF06985.6', 'PF06986.6', 'PF06988.6', 'PF06989.7',
     'PF06990.6', 'PF06991.6', 'PF06992.6', 'PF06993.7', 'PF06994.6', 'PF06995.6', 'PF06996.6', 'PF06998.6',
     'PF06999.7', 'PF07000.6', 'PF07001.6', 'PF07002.11', 'PF07004.7', 'PF07005.6', 'PF07006.6', 'PF07007.7',
     'PF07009.6', 'PF07010.7', 'PF07011.6', 'PF07012.7', 'PF07013.6', 'PF07014.7', 'PF07015.6', 'PF07016.6',
     'PF07017.6', 'PF07019.7', 'PF07020.6', 'PF07021.7', 'PF07022.8', 'PF07023.7', 'PF07024.8', 'PF07026.6',
     'PF07027.7', 'PF07028.6', 'PF07029.6', 'PF07030.7', 'PF07032.6', 'PF07033.6', 'PF07034.6', 'PF07035.7',
     'PF07037.6', 'PF07038.6', 'PF07039.6', 'PF07040.6', 'PF07041.6', 'PF07042.6', 'PF07043.8', 'PF07044.6',
     'PF07045.6', 'PF07046.6', 'PF07047.7', 'PF07048.6', 'PF07051.6', 'PF07052.6', 'PF07054.6', 'PF07055.7',
     'PF07056.6', 'PF07057.6', 'PF07058.6', 'PF07059.7', 'PF07061.6', 'PF07062.7', 'PF07063.8', 'PF07064.8',
     'PF07065.9', 'PF07066.6', 'PF07067.6', 'PF07068.6', 'PF07069.6', 'PF07070.6', 'PF07071.6', 'PF07072.6',
     'PF07073.7', 'PF07074.7', 'PF07075.6', 'PF07076.6', 'PF07077.6', 'PF07078.6', 'PF07079.6', 'PF07080.6',
     'PF07081.6', 'PF07082.6', 'PF07083.6', 'PF07084.7', 'PF07085.7', 'PF07086.7', 'PF07087.6', 'PF07088.6',
     'PF07090.6', 'PF07091.6', 'PF07092.7', 'PF07093.6', 'PF07094.6', 'PF07095.6', 'PF07096.6', 'PF07097.6',
     'PF07098.6', 'PF07099.6', 'PF07100.6', 'PF07101.6', 'PF07102.7', 'PF07103.6', 'PF07104.6', 'PF07105.6',
     'PF07106.8', 'PF07107.6', 'PF07108.6', 'PF07109.6', 'PF07110.6', 'PF07111.7', 'PF07112.6', 'PF07114.6',
     'PF07115.6', 'PF07116.6', 'PF07117.6', 'PF07118.6', 'PF07119.7', 'PF07120.6', 'PF07122.6', 'PF07123.7',
     'PF07124.6', 'PF07125.6', 'PF07126.7', 'PF07127.6', 'PF07128.7', 'PF07129.6', 'PF07130.7', 'PF07131.6',
     'PF07133.6', 'PF07134.6', 'PF07136.6', 'PF07137.6', 'PF07138.6', 'PF07139.6', 'PF07140.6', 'PF07141.6',
     'PF07142.7', 'PF07143.6', 'PF07145.10', 'PF07146.6', 'PF07147.7', 'PF07148.7', 'PF07149.6', 'PF07150.6',
     'PF07151.7', 'PF07152.7', 'PF07153.6', 'PF07154.6', 'PF07155.7', 'PF07156.9', 'PF07157.7', 'PF07158.6',
     'PF07159.7', 'PF07160.7', 'PF07161.8', 'PF07162.6', 'PF07163.7', 'PF07165.6', 'PF07166.6', 'PF07167.8',
     'PF07168.6', 'PF07171.7', 'PF07172.6', 'PF07173.7', 'PF07174.6', 'PF07175.6', 'PF07176.6', 'PF07177.7',
     'PF07178.6', 'PF07179.7', 'PF07180.6', 'PF07181.6', 'PF07182.6', 'PF07183.6', 'PF07184.6', 'PF07185.6',
     'PF07187.6', 'PF07188.6', 'PF07189.6', 'PF07190.6', 'PF07191.7', 'PF07192.6', 'PF07193.6', 'PF07194.6',
     'PF07195.7', 'PF07196.8', 'PF07197.7', 'PF07198.6', 'PF07199.6', 'PF07200.8', 'PF07201.6', 'PF07202.8',
     'PF07203.6', 'PF07204.6', 'PF07205.6', 'PF07206.6', 'PF07207.6', 'PF07208.6', 'PF07209.7', 'PF07210.7',
     'PF07212.6', 'PF07213.6', 'PF07214.7', 'PF07215.6', 'PF07216.7', 'PF07217.6', 'PF07218.6', 'PF07219.8',
     'PF07220.6', 'PF07221.6', 'PF07222.7', 'PF07223.6', 'PF07224.6', 'PF07225.7', 'PF07226.6', 'PF07227.6',
     'PF07228.7', 'PF07229.7', 'PF07230.6', 'PF07231.7', 'PF07232.6', 'PF07233.7', 'PF07234.6', 'PF07235.6',
     'PF07236.6', 'PF07237.6', 'PF07238.9', 'PF07239.6', 'PF07240.6', 'PF07242.6', 'PF07243.6', 'PF07244.10',
     'PF07245.6', 'PF07246.6', 'PF07247.7', 'PF07248.7', 'PF07249.7', 'PF07250.6', 'PF07252.6', 'PF07253.6',
     'PF07254.7', 'PF07255.6', 'PF07256.7', 'PF07258.9', 'PF07259.7', 'PF07260.6', 'PF07261.6', 'PF07262.6',
     'PF07263.6', 'PF07264.6', 'PF07265.6', 'PF07267.6', 'PF07268.6', 'PF07270.6', 'PF07271.6', 'PF07272.6',
     'PF07273.7', 'PF07274.7', 'PF07275.6', 'PF07276.6', 'PF07277.6', 'PF07278.6', 'PF07279.6', 'PF07280.6',
     'PF07281.7', 'PF07282.6', 'PF07283.6', 'PF07284.6', 'PF07285.6', 'PF07286.7', 'PF07287.6', 'PF07288.6',
     'PF07289.6', 'PF07290.6', 'PF07291.6', 'PF07292.8', 'PF07293.6', 'PF07294.6', 'PF07295.6', 'PF07296.6',
     'PF07297.7', 'PF07298.6', 'PF07299.6', 'PF07301.6', 'PF07302.6', 'PF07303.8', 'PF07304.6', 'PF07305.7',
     'PF07306.6', 'PF07307.6', 'PF07308.8', 'PF07309.6', 'PF07310.8', 'PF07311.7', 'PF07312.6', 'PF07313.7',
     'PF07314.6', 'PF07315.6', 'PF07316.6', 'PF07317.7', 'PF07318.7', 'PF07319.6', 'PF07321.7', 'PF07322.6',
     'PF07323.7', 'PF07324.6', 'PF07325.6', 'PF07326.6', 'PF07327.6', 'PF07328.6', 'PF07330.7', 'PF07331.6',
     'PF07332.6', 'PF07333.7', 'PF07334.8', 'PF07335.6', 'PF07336.6', 'PF07337.6', 'PF07338.8', 'PF07339.7',
     'PF07340.6', 'PF07341.6', 'PF07342.6', 'PF07343.6', 'PF07344.6', 'PF07345.6', 'PF07346.6', 'PF07347.7',
     'PF07348.7', 'PF07349.6', 'PF07350.7', 'PF07351.8', 'PF07352.7', 'PF07353.7', 'PF07354.7', 'PF07355.7',
     'PF07356.7', 'PF07357.6', 'PF07358.6', 'PF07359.6', 'PF07361.6', 'PF07362.7', 'PF07363.6', 'PF07364.7',
     'PF07365.7', 'PF07366.7', 'PF07367.6', 'PF07368.6', 'PF07369.6', 'PF07370.6', 'PF07371.7', 'PF07372.7',
     'PF07373.6', 'PF07374.6', 'PF07376.6', 'PF07377.7', 'PF07378.6', 'PF07379.6', 'PF07380.6', 'PF07381.6',
     'PF07382.6', 'PF07383.7', 'PF07384.6', 'PF07385.7', 'PF07386.6', 'PF07387.6', 'PF07388.6', 'PF07389.7',
     'PF07390.6', 'PF07391.6', 'PF07392.7', 'PF07393.6', 'PF07394.7', 'PF07395.6', 'PF07396.6', 'PF07397.6',
     'PF07398.6', 'PF07399.6', 'PF07400.6', 'PF07401.7', 'PF07402.6', 'PF07403.6', 'PF07404.6', 'PF07405.6',
     'PF07406.6', 'PF07407.6', 'PF07408.6', 'PF07409.7', 'PF07410.6', 'PF07411.7', 'PF07412.7', 'PF07413.6',
     'PF07415.6', 'PF07416.6', 'PF07417.7', 'PF07418.6', 'PF07419.7', 'PF07420.6', 'PF07421.6', 'PF07422.8',
     'PF07423.6', 'PF07424.6', 'PF07425.6', 'PF07426.6', 'PF07428.6', 'PF07429.6', 'PF07430.6', 'PF07431.7',
     'PF07432.8', 'PF07433.6', 'PF07434.6', 'PF07435.6', 'PF07436.6', 'PF07437.6', 'PF07438.6', 'PF07439.6',
     'PF07440.7', 'PF07441.6', 'PF07442.6', 'PF07443.8', 'PF07444.6', 'PF07445.7', 'PF07447.7', 'PF07448.6',
     'PF07449.6', 'PF07450.6', 'PF07451.6', 'PF07452.7', 'PF07453.8', 'PF07454.6', 'PF07455.6', 'PF07456.6',
     'PF07457.6', 'PF07458.7', 'PF07459.6', 'PF07460.6', 'PF07461.6', 'PF07462.6', 'PF07463.6', 'PF07464.6',
     'PF07465.8', 'PF07466.6', 'PF07467.6', 'PF07468.6', 'PF07469.7', 'PF07470.8', 'PF07471.7', 'PF07472.6',
     'PF07473.6', 'PF07474.7', 'PF07475.7', 'PF07476.6', 'PF07477.7', 'PF07478.8', 'PF07479.9', 'PF07481.6',
     'PF07482.6', 'PF07483.6', 'PF07484.7', 'PF07485.6', 'PF07486.7', 'PF07487.8', 'PF07488.7', 'PF07489.6',
     'PF07490.6', 'PF07491.6', 'PF07492.6', 'PF07494.6', 'PF07495.8', 'PF07496.10', 'PF07497.7', 'PF07498.7',
     'PF07499.8', 'PF07500.9', 'PF07501.7', 'PF07502.9', 'PF07503.7', 'PF07504.8', 'PF07505.6', 'PF07506.6',
     'PF07507.6', 'PF07508.8', 'PF07509.6', 'PF07510.6', 'PF07511.6', 'PF07514.6', 'PF07515.6', 'PF07516.8',
     'PF07517.9', 'PF07519.6', 'PF07520.6', 'PF07521.7', 'PF07522.9', 'PF07523.7', 'PF07524.8', 'PF07525.11',
     'PF07526.6', 'PF07527.8', 'PF07528.9', 'PF07529.8', 'PF07530.6', 'PF07531.9', 'PF07532.6', 'PF07533.11',
     'PF07534.11', 'PF07535.7', 'PF07536.9', 'PF07537.6', 'PF07538.6', 'PF07539.7', 'PF07540.6', 'PF07541.7',
     'PF07542.6', 'PF07543.7', 'PF07544.8', 'PF07545.9', 'PF07546.8', 'PF07547.8', 'PF07548.6', 'PF07549.9',
     'PF07550.6', 'PF07551.6', 'PF07552.6', 'PF07553.6', 'PF07554.8', 'PF07555.8', 'PF07556.6', 'PF07557.6',
     'PF07558.6', 'PF07559.9', 'PF07560.6', 'PF07561.6', 'PF07562.9', 'PF07563.6', 'PF07564.6', 'PF07565.8',
     'PF07566.7', 'PF07568.7', 'PF07569.6', 'PF07571.8', 'PF07572.7', 'PF07573.6', 'PF07574.8', 'PF07575.8',
     'PF07576.7', 'PF07577.6', 'PF07578.6', 'PF07579.6', 'PF07580.9', 'PF07581.7', 'PF07582.7', 'PF07583.6',
     'PF07584.6', 'PF07585.6', 'PF07586.6', 'PF07587.6', 'PF07588.6', 'PF07589.6', 'PF07590.6', 'PF07591.6',
     'PF07592.6', 'PF07593.7', 'PF07595.7', 'PF07596.6', 'PF07597.6', 'PF07598.6', 'PF07599.6', 'PF07600.6',
     'PF07602.6', 'PF07603.6', 'PF07606.6', 'PF07607.6', 'PF07608.6', 'PF07609.6', 'PF07610.6', 'PF07611.6',
     'PF07613.6', 'PF07614.6', 'PF07615.6', 'PF07617.6', 'PF07618.6', 'PF07619.6', 'PF07620.6', 'PF07621.6',
     'PF07622.6', 'PF07623.6', 'PF07624.6', 'PF07625.6', 'PF07626.6', 'PF07627.6', 'PF07628.6', 'PF07629.6',
     'PF07631.6', 'PF07632.6', 'PF07634.6', 'PF07635.6', 'PF07636.6', 'PF07637.6', 'PF07638.6', 'PF07639.6',
     'PF07640.6', 'PF07642.6', 'PF07643.6', 'PF07644.6', 'PF07645.10', 'PF07646.10', 'PF07647.12', 'PF07648.10',
     'PF07649.7', 'PF07650.12', 'PF07651.11', 'PF07652.9', 'PF07653.12', 'PF07654.10', 'PF07655.8', 'PF07657.8',
     'PF07659.6', 'PF07660.9', 'PF07661.8', 'PF07662.8', 'PF07663.6', 'PF07664.7', 'PF07666.6', 'PF07667.6',
     'PF07668.6', 'PF07669.6', 'PF07670.9', 'PF07671.6', 'PF07672.8', 'PF07673.9', 'PF07675.6', 'PF07676.7',
     'PF07677.9', 'PF07678.9', 'PF07679.11', 'PF07680.6', 'PF07681.7', 'PF07682.6', 'PF07683.9', 'PF07684.7',
     'PF07685.9', 'PF07686.12', 'PF07687.9', 'PF07688.7', 'PF07689.7', 'PF07690.11', 'PF07691.7', 'PF07692.6',
     'PF07693.9', 'PF07694.7', 'PF07695.6', 'PF07696.6', 'PF07697.6', 'PF07698.6', 'PF07699.8', 'PF07700.10',
     'PF07701.9', 'PF07702.8', 'PF07703.9', 'PF07704.6', 'PF07705.6', 'PF07706.7', 'PF07707.10', 'PF07708.6',
     'PF07709.6', 'PF07710.6', 'PF07711.6', 'PF07712.7', 'PF07713.8', 'PF07714.12', 'PF07715.10', 'PF07716.10',
     'PF07717.11', 'PF07718.7', 'PF07719.12', 'PF07720.7', 'PF07721.9', 'PF07722.8', 'PF07723.8', 'PF07724.9',
     'PF07725.7', 'PF07726.6', 'PF07727.9', 'PF07728.9', 'PF07729.7', 'PF07730.8', 'PF07731.9', 'PF07732.10',
     'PF07733.7', 'PF07734.8', 'PF07735.12', 'PF07736.6', 'PF07737.6', 'PF07738.8', 'PF07739.8', 'PF07740.7',
     'PF07741.8', 'PF07742.7', 'PF07743.8', 'PF07744.8', 'PF07745.8', 'PF07746.6', 'PF07747.6', 'PF07748.8',
     'PF07749.7', 'PF07750.6', 'PF07751.6', 'PF07752.6', 'PF07753.6', 'PF07754.6', 'PF07755.6', 'PF07756.7',
     'PF07757.8', 'PF07758.6', 'PF07759.7', 'PF07760.6', 'PF07761.7', 'PF07762.9', 'PF07763.8', 'PF07764.6',
     'PF07765.7', 'PF07766.8', 'PF07767.6', 'PF07768.6', 'PF07769.9', 'PF07771.6', 'PF07773.6', 'PF07774.8',
     'PF07775.6', 'PF07776.10', 'PF07777.6', 'PF07778.6', 'PF07779.7', 'PF07780.7', 'PF07781.6', 'PF07782.8',
     'PF07784.6', 'PF07785.6', 'PF07786.7', 'PF07787.7', 'PF07788.6', 'PF07789.7', 'PF07790.6', 'PF07791.6',
     'PF07792.7', 'PF07793.6', 'PF07794.6', 'PF07795.6', 'PF07796.6', 'PF07797.9', 'PF07798.6', 'PF07799.7',
     'PF07800.7', 'PF07801.6', 'PF07802.6', 'PF07803.6', 'PF07804.7', 'PF07805.7', 'PF07806.6', 'PF07807.6',
     'PF07808.8', 'PF07809.6', 'PF07810.8', 'PF07811.7', 'PF07812.7', 'PF07813.7', 'PF07814.8', 'PF07815.9',
     'PF07816.6', 'PF07817.8', 'PF07818.8', 'PF07819.8', 'PF07820.7', 'PF07821.7', 'PF07822.6', 'PF07823.6',
     'PF07824.7', 'PF07825.6', 'PF07826.6', 'PF07827.6', 'PF07828.7', 'PF07829.6', 'PF07830.8', 'PF07831.8',
     'PF07832.6', 'PF07833.6', 'PF07834.6', 'PF07835.7', 'PF07836.6', 'PF07837.7', 'PF07839.6', 'PF07840.7',
     'PF07841.8', 'PF07842.7', 'PF07843.6', 'PF07845.6', 'PF07846.6', 'PF07847.7', 'PF07848.7', 'PF07849.6',
     'PF07850.9', 'PF07851.8', 'PF07852.6', 'PF07853.6', 'PF07854.7', 'PF07855.7', 'PF07856.7', 'PF07857.7',
     'PF07858.7', 'PF07859.8', 'PF07860.6', 'PF07861.6', 'PF07862.6', 'PF07863.6', 'PF07864.6', 'PF07865.6',
     'PF07866.6', 'PF07867.6', 'PF07868.6', 'PF07869.7', 'PF07870.6', 'PF07871.6', 'PF07872.6', 'PF07873.6',
     'PF07874.6', 'PF07875.7', 'PF07876.7', 'PF07877.6', 'PF07878.6', 'PF07879.6', 'PF07880.6', 'PF07881.7',
     'PF07882.7', 'PF07883.6', 'PF07884.9', 'PF07885.11', 'PF07886.6', 'PF07887.6', 'PF07888.6', 'PF07889.7',
     'PF07890.7', 'PF07891.7', 'PF07892.6', 'PF07893.8', 'PF07894.7', 'PF07895.6', 'PF07896.7', 'PF07897.6',
     'PF07898.8', 'PF07899.6', 'PF07900.6', 'PF07901.6', 'PF07902.6', 'PF07903.6', 'PF07904.8', 'PF07905.6',
     'PF07906.8', 'PF07907.6', 'PF07908.7', 'PF07909.6', 'PF07910.8', 'PF07911.8', 'PF07912.8', 'PF07913.6',
     'PF07914.6', 'PF07915.8', 'PF07916.6', 'PF07918.6', 'PF07919.7', 'PF07920.6', 'PF07921.7', 'PF07922.6',
     'PF07923.8', 'PF07924.6', 'PF07925.6', 'PF07926.7', 'PF07927.7', 'PF07928.7', 'PF07929.6', 'PF07930.7',
     'PF07931.7', 'PF07932.7', 'PF07933.9', 'PF07934.7', 'PF07935.6', 'PF07936.7', 'PF07937.6', 'PF07938.7',
     'PF07939.6', 'PF07940.8', 'PF07941.6', 'PF07942.7', 'PF07943.8', 'PF07944.7', 'PF07945.6', 'PF07946.9',
     'PF07947.9', 'PF07948.6', 'PF07949.7', 'PF07950.6', 'PF07951.7', 'PF07952.7', 'PF07953.7', 'PF07954.6',
     'PF07955.6', 'PF07956.6', 'PF07957.6', 'PF07958.6', 'PF07959.7', 'PF07960.6', 'PF07961.6', 'PF07962.7',
     'PF07963.7', 'PF07964.6', 'PF07965.7', 'PF07966.7', 'PF07967.8', 'PF07968.7', 'PF07969.6', 'PF07970.7',
     'PF07971.7', 'PF07972.6', 'PF07973.9', 'PF07974.8', 'PF07975.7', 'PF07976.7', 'PF07977.8', 'PF07978.8',
     'PF07979.6', 'PF07980.6', 'PF07981.6', 'PF07982.7', 'PF07983.8', 'PF07984.7', 'PF07985.7', 'PF07986.7',
     'PF07987.6', 'PF07988.7', 'PF07989.6', 'PF07990.7', 'PF07991.7', 'PF07992.9', 'PF07993.7', 'PF07994.7',
     'PF07995.6', 'PF07996.6', 'PF07997.6', 'PF07998.6', 'PF07999.6', 'PF08000.6', 'PF08001.6', 'PF08002.6',
     'PF08003.6', 'PF08004.6', 'PF08005.7', 'PF08006.6', 'PF08007.7', 'PF08008.7', 'PF08009.6', 'PF08010.6',
     'PF08011.6', 'PF08012.6', 'PF08013.6', 'PF08014.6', 'PF08015.6', 'PF08016.7', 'PF08017.6', 'PF08018.6',
     'PF08019.7', 'PF08020.6', 'PF08021.6', 'PF08022.7', 'PF08023.7', 'PF08024.6', 'PF08025.6', 'PF08026.6',
     'PF08027.6', 'PF08028.6', 'PF08029.6', 'PF08030.7', 'PF08031.7', 'PF08032.7', 'PF08033.7', 'PF08034.6',
     'PF08035.6', 'PF08036.6', 'PF08037.6', 'PF08038.7', 'PF08039.6', 'PF08040.6', 'PF08041.6', 'PF08042.6',
     'PF08043.7', 'PF08044.6', 'PF08045.6', 'PF08046.6', 'PF08047.6', 'PF08048.7', 'PF08049.6', 'PF08050.7',
     'PF08051.6', 'PF08052.6', 'PF08053.6', 'PF08054.6', 'PF08055.6', 'PF08056.6', 'PF08057.6', 'PF08058.6',
     'PF08059.8', 'PF08060.8', 'PF08061.6', 'PF08062.6', 'PF08063.7', 'PF08064.8', 'PF08065.7', 'PF08066.7',
     'PF08067.6', 'PF08068.7', 'PF08069.7', 'PF08070.6', 'PF08071.7', 'PF08072.6', 'PF08073.7', 'PF08074.6',
     'PF08075.6', 'PF08076.6', 'PF08077.6', 'PF08078.7', 'PF08079.7', 'PF08080.7', 'PF08081.6', 'PF08082.6',
     'PF08083.6', 'PF08084.6', 'PF08085.6', 'PF08086.6', 'PF08087.6', 'PF08088.7', 'PF08089.6', 'PF08090.6',
     'PF08091.6', 'PF08092.6', 'PF08093.6', 'PF08094.6', 'PF08095.6', 'PF08096.6', 'PF08097.6', 'PF08098.6',
     'PF08099.6', 'PF08100.6', 'PF08101.6', 'PF08102.6', 'PF08103.6', 'PF08104.6', 'PF08105.6', 'PF08106.6',
     'PF08107.6', 'PF08108.6', 'PF08109.6', 'PF08110.7', 'PF08111.6', 'PF08112.6', 'PF08113.6', 'PF08114.6',
     'PF08115.6', 'PF08116.6', 'PF08117.6', 'PF08118.6', 'PF08119.6', 'PF08120.6', 'PF08121.6', 'PF08122.7',
     'PF08123.8', 'PF08124.6', 'PF08125.8', 'PF08126.6', 'PF08127.8', 'PF08129.6', 'PF08130.6', 'PF08131.6',
     'PF08132.6', 'PF08133.6', 'PF08134.6', 'PF08135.6', 'PF08136.6', 'PF08137.7', 'PF08138.6', 'PF08139.7',
     'PF08140.6', 'PF08141.7', 'PF08142.7', 'PF08143.6', 'PF08144.6', 'PF08145.7', 'PF08146.7', 'PF08147.7',
     'PF08148.7', 'PF08149.6', 'PF08150.7', 'PF08151.7', 'PF08152.7', 'PF08153.7', 'PF08154.7', 'PF08155.6',
     'PF08156.8', 'PF08157.6', 'PF08158.7', 'PF08159.7', 'PF08161.7', 'PF08163.7', 'PF08164.7', 'PF08165.6',
     'PF08166.7', 'PF08167.7', 'PF08168.6', 'PF08169.6', 'PF08170.7', 'PF08171.6', 'PF08172.7', 'PF08173.6',
     'PF08174.6', 'PF08175.7', 'PF08176.7', 'PF08177.6', 'PF08178.6', 'PF08179.7', 'PF08180.6', 'PF08181.6',
     'PF08182.6', 'PF08183.6', 'PF08184.6', 'PF08186.6', 'PF08187.6', 'PF08188.6', 'PF08189.6', 'PF08190.7',
     'PF08191.6', 'PF08192.6', 'PF08193.6', 'PF08194.7', 'PF08195.6', 'PF08196.6', 'PF08197.6', 'PF08198.6',
     'PF08199.6', 'PF08200.6', 'PF08201.6', 'PF08202.6', 'PF08203.6', 'PF08204.6', 'PF08205.7', 'PF08206.6',
     'PF08207.7', 'PF08208.6', 'PF08209.6', 'PF08210.6', 'PF08211.6', 'PF08212.7', 'PF08213.6', 'PF08214.6',
     'PF08216.6', 'PF08217.6', 'PF08218.6', 'PF08219.6', 'PF08220.7', 'PF08221.6', 'PF08222.6', 'PF08223.6',
     'PF08224.6', 'PF08225.6', 'PF08226.6', 'PF08227.6', 'PF08228.6', 'PF08229.6', 'PF08230.6', 'PF08231.7',
     'PF08232.7', 'PF08234.7', 'PF08235.8', 'PF08236.6', 'PF08237.6', 'PF08238.7', 'PF08239.6', 'PF08240.7',
     'PF08241.7', 'PF08242.7', 'PF08243.6', 'PF08244.7', 'PF08245.7', 'PF08246.7', 'PF08247.6', 'PF08248.6',
     'PF08249.6', 'PF08250.6', 'PF08251.6', 'PF08252.6', 'PF08253.6', 'PF08254.6', 'PF08255.6', 'PF08256.6',
     'PF08257.6', 'PF08258.6', 'PF08259.6', 'PF08260.6', 'PF08261.7', 'PF08262.6', 'PF08263.7', 'PF08264.8',
     'PF08265.6', 'PF08266.7', 'PF08267.7', 'PF08268.7', 'PF08269.6', 'PF08270.6', 'PF08271.7', 'PF08272.6',
     'PF08273.7', 'PF08274.7', 'PF08275.6', 'PF08276.6', 'PF08277.7', 'PF08278.6', 'PF08279.7', 'PF08280.6',
     'PF08281.7', 'PF08282.7', 'PF08283.6', 'PF08284.6', 'PF08285.6', 'PF08286.6', 'PF08287.6', 'PF08288.7',
     'PF08289.6', 'PF08290.6', 'PF08291.6', 'PF08292.7', 'PF08293.6', 'PF08294.6', 'PF08295.7', 'PF08297.6',
     'PF08298.6', 'PF08299.6', 'PF08300.8', 'PF08301.8', 'PF08302.6', 'PF08303.6', 'PF08305.6', 'PF08306.6',
     'PF08307.6', 'PF08308.6', 'PF08309.6', 'PF08310.6', 'PF08311.7', 'PF08312.7', 'PF08313.7', 'PF08314.6',
     'PF08315.7', 'PF08316.6', 'PF08317.6', 'PF08318.7', 'PF08320.7', 'PF08321.7', 'PF08323.6', 'PF08324.6',
     'PF08325.5', 'PF08326.7', 'PF08327.6', 'PF08328.6', 'PF08329.5', 'PF08331.5', 'PF08332.5', 'PF08333.6',
     'PF08334.6', 'PF08335.6', 'PF08336.6', 'PF08337.7', 'PF08338.6', 'PF08339.5', 'PF08340.6', 'PF08341.6',
     'PF08343.5', 'PF08344.6', 'PF08345.6', 'PF08346.7', 'PF08347.6', 'PF08348.6', 'PF08349.6', 'PF08350.5',
     'PF08351.6', 'PF08352.7', 'PF08353.5', 'PF08354.5', 'PF08355.7', 'PF08356.7', 'PF08357.6', 'PF08358.5',
     'PF08359.6', 'PF08360.6', 'PF08361.6', 'PF08362.6', 'PF08363.5', 'PF08364.6', 'PF08365.6', 'PF08366.8',
     'PF08367.6', 'PF08368.7', 'PF08369.5', 'PF08370.6', 'PF08372.5', 'PF08373.5', 'PF08374.6', 'PF08375.6',
     'PF08376.5', 'PF08377.5', 'PF08378.6', 'PF08379.5', 'PF08381.6', 'PF08383.6', 'PF08384.5', 'PF08385.7',
     'PF08386.5', 'PF08387.5', 'PF08388.6', 'PF08389.7', 'PF08390.6', 'PF08391.5', 'PF08392.7', 'PF08393.8',
     'PF08394.5', 'PF08395.7', 'PF08396.5', 'PF08397.6', 'PF08398.5', 'PF08399.6', 'PF08400.5', 'PF08401.6',
     'PF08402.5', 'PF08403.5', 'PF08404.5', 'PF08405.6', 'PF08406.5', 'PF08407.6', 'PF08408.5', 'PF08409.6',
     'PF08410.5', 'PF08411.5', 'PF08412.5', 'PF08414.5', 'PF08415.5', 'PF08416.8', 'PF08417.7', 'PF08418.5',
     'PF08421.6', 'PF08423.6', 'PF08424.5', 'PF08426.5', 'PF08427.5', 'PF08428.5', 'PF08429.6', 'PF08430.7',
     'PF08432.5', 'PF08433.5', 'PF08434.6', 'PF08435.6', 'PF08436.7', 'PF08437.5', 'PF08438.5', 'PF08439.5',
     'PF08440.5', 'PF08441.7', 'PF08442.5', 'PF08443.6', 'PF08444.5', 'PF08445.5', 'PF08446.6', 'PF08447.6',
     'PF08448.5', 'PF08449.6', 'PF08450.7', 'PF08451.6', 'PF08452.5', 'PF08453.5', 'PF08454.6', 'PF08455.5',
     'PF08456.5', 'PF08457.5', 'PF08458.5', 'PF08459.6', 'PF08460.5', 'PF08461.5', 'PF08462.5', 'PF08463.5',
     'PF08464.5', 'PF08465.5', 'PF08466.5', 'PF08467.5', 'PF08468.6', 'PF08469.5', 'PF08470.5', 'PF08471.5',
     'PF08472.5', 'PF08473.6', 'PF08474.6', 'PF08475.5', 'PF08476.5', 'PF08477.8', 'PF08478.5', 'PF08479.6',
     'PF08480.5', 'PF08481.5', 'PF08482.5', 'PF08483.6', 'PF08484.6', 'PF08485.5', 'PF08486.5', 'PF08487.5',
     'PF08488.6', 'PF08489.6', 'PF08490.7', 'PF08491.5', 'PF08492.7', 'PF08493.5', 'PF08494.6', 'PF08495.5',
     'PF08496.5', 'PF08497.5', 'PF08498.5', 'PF08499.7', 'PF08500.5', 'PF08501.6', 'PF08502.5', 'PF08503.5',
     'PF08504.6', 'PF08505.5', 'PF08506.5', 'PF08507.5', 'PF08508.5', 'PF08509.6', 'PF08510.7', 'PF08511.6',
     'PF08512.7', 'PF08513.6', 'PF08514.6', 'PF08515.7', 'PF08516.7', 'PF08517.7', 'PF08518.6', 'PF08519.7',
     'PF08520.5', 'PF08521.5', 'PF08522.5', 'PF08523.5', 'PF08524.6', 'PF08525.6', 'PF08526.5', 'PF08527.5',
     'PF08528.6', 'PF08529.6', 'PF08530.5', 'PF08531.5', 'PF08532.5', 'PF08533.5', 'PF08534.5', 'PF08535.5',
     'PF08536.6', 'PF08537.5', 'PF08538.5', 'PF08539.6', 'PF08540.5', 'PF08541.5', 'PF08542.6', 'PF08543.7',
     'PF08544.8', 'PF08545.5', 'PF08546.6', 'PF08547.7', 'PF08548.6', 'PF08549.5', 'PF08550.5', 'PF08551.5',
     'PF08552.6', 'PF08553.5', 'PF08555.5', 'PF08557.5', 'PF08558.5', 'PF08559.5', 'PF08560.5', 'PF08561.5',
     'PF08562.5', 'PF08563.6', 'PF08564.5', 'PF08565.6', 'PF08566.5', 'PF08567.6', 'PF08568.5', 'PF08569.6',
     'PF08570.5', 'PF08571.5', 'PF08572.5', 'PF08573.5', 'PF08574.5', 'PF08576.5', 'PF08577.6', 'PF08578.5',
     'PF08579.6', 'PF08580.5', 'PF08581.5', 'PF08583.5', 'PF08584.6', 'PF08585.7', 'PF08586.5', 'PF08587.6',
     'PF08588.5', 'PF08589.5', 'PF08590.5', 'PF08591.5', 'PF08592.6', 'PF08593.5', 'PF08594.5', 'PF08595.6',
     'PF08596.5', 'PF08597.5', 'PF08598.6', 'PF08599.5', 'PF08600.5', 'PF08601.5', 'PF08602.5', 'PF08603.6',
     'PF08604.5', 'PF08605.5', 'PF08606.6', 'PF08608.7', 'PF08609.5', 'PF08610.5', 'PF08611.5', 'PF08612.6',
     'PF08613.6', 'PF08614.6', 'PF08615.6', 'PF08616.5', 'PF08617.5', 'PF08618.5', 'PF08619.5', 'PF08620.5',
     'PF08621.5', 'PF08622.5', 'PF08623.5', 'PF08624.5', 'PF08625.6', 'PF08626.6', 'PF08627.5', 'PF08628.7',
     'PF08629.5', 'PF08630.5', 'PF08631.5', 'PF08632.5', 'PF08633.5', 'PF08634.5', 'PF08635.5', 'PF08636.5',
     'PF08637.5', 'PF08638.6', 'PF08639.5', 'PF08640.6', 'PF08641.7', 'PF08642.5', 'PF08643.5', 'PF08644.6',
     'PF08645.6', 'PF08646.5', 'PF08647.6', 'PF08648.7', 'PF08649.5', 'PF08650.5', 'PF08651.5', 'PF08652.6',
     'PF08653.5', 'PF08654.5', 'PF08655.5', 'PF08656.5', 'PF08657.5', 'PF08658.5', 'PF08659.5', 'PF08660.6',
     'PF08661.6', 'PF08662.6', 'PF08663.5', 'PF08664.5', 'PF08665.7', 'PF08666.7', 'PF08667.5', 'PF08668.7',
     'PF08669.6', 'PF08670.6', 'PF08671.5', 'PF08672.6', 'PF08673.5', 'PF08674.5', 'PF08675.6', 'PF08676.6',
     'PF08677.5', 'PF08678.5', 'PF08679.6', 'PF08680.5', 'PF08681.6', 'PF08682.5', 'PF08683.6', 'PF08684.5',
     'PF08685.6', 'PF08686.6', 'PF08687.6', 'PF08688.5', 'PF08689.5', 'PF08690.5', 'PF08691.5', 'PF08692.5',
     'PF08693.5', 'PF08694.6', 'PF08695.5', 'PF08696.6', 'PF08698.6', 'PF08699.5', 'PF08700.6', 'PF08701.6',
     'PF08702.5', 'PF08703.5', 'PF08704.5', 'PF08705.6', 'PF08706.6', 'PF08707.6', 'PF08708.6', 'PF08709.6',
     'PF08710.5', 'PF08711.6', 'PF08712.6', 'PF08713.6', 'PF08714.6', 'PF08715.5', 'PF08716.5', 'PF08717.5',
     'PF08718.6', 'PF08719.6', 'PF08720.5', 'PF08721.6', 'PF08722.6', 'PF08723.5', 'PF08724.5', 'PF08725.6',
     'PF08726.5', 'PF08727.6', 'PF08728.5', 'PF08729.5', 'PF08730.5', 'PF08731.6', 'PF08732.5', 'PF08733.5',
     'PF08734.6', 'PF08735.5', 'PF08736.6', 'PF08737.5', 'PF08738.5', 'PF08740.6', 'PF08741.5', 'PF08742.6',
     'PF08743.5', 'PF08744.5', 'PF08745.6', 'PF08746.6', 'PF08747.6', 'PF08748.6', 'PF08750.6', 'PF08751.6',
     'PF08752.5', 'PF08753.6', 'PF08755.6', 'PF08756.5', 'PF08757.6', 'PF08758.6', 'PF08759.6', 'PF08760.6',
     'PF08761.6', 'PF08762.5', 'PF08763.6', 'PF08764.5', 'PF08765.6', 'PF08766.6', 'PF08767.6', 'PF08768.6',
     'PF08769.6', 'PF08770.6', 'PF08771.6', 'PF08772.6', 'PF08773.6', 'PF08774.6', 'PF08775.5', 'PF08776.6',
     'PF08777.6', 'PF08778.5', 'PF08779.5', 'PF08780.6', 'PF08781.5', 'PF08782.5', 'PF08783.6', 'PF08784.6',
     'PF08785.6', 'PF08786.6', 'PF08787.6', 'PF08788.6', 'PF08789.5', 'PF08790.6', 'PF08792.5', 'PF08793.5',
     'PF08794.5', 'PF08795.5', 'PF08796.5', 'PF08797.6', 'PF08798.6', 'PF08799.6', 'PF08800.5', 'PF08801.6',
     'PF08802.5', 'PF08803.6', 'PF08804.5', 'PF08805.6', 'PF08806.6', 'PF08807.5', 'PF08808.6', 'PF08809.6',
     'PF08810.5', 'PF08811.6', 'PF08812.6', 'PF08813.6', 'PF08814.5', 'PF08815.5', 'PF08816.6', 'PF08817.5',
     'PF08818.6', 'PF08819.6', 'PF08820.5', 'PF08821.6', 'PF08822.6', 'PF08823.6', 'PF08824.5', 'PF08825.5',
     'PF08826.5', 'PF08827.6', 'PF08828.5', 'PF08829.5', 'PF08830.5', 'PF08831.5', 'PF08832.5', 'PF08833.5',
     'PF08837.6', 'PF08838.5', 'PF08839.6', 'PF08840.6', 'PF08841.5', 'PF08842.5', 'PF08843.6', 'PF08844.5',
     'PF08845.5', 'PF08846.5', 'PF08847.6', 'PF08848.6', 'PF08849.6', 'PF08850.6', 'PF08852.6', 'PF08853.6',
     'PF08854.5', 'PF08855.5', 'PF08856.6', 'PF08857.6', 'PF08858.5', 'PF08859.6', 'PF08860.5', 'PF08861.5',
     'PF08862.5', 'PF08863.5', 'PF08864.5', 'PF08865.6', 'PF08866.5', 'PF08867.6', 'PF08868.5', 'PF08869.6',
     'PF08870.6', 'PF08872.5', 'PF08873.6', 'PF08874.5', 'PF08875.6', 'PF08876.6', 'PF08877.5', 'PF08878.6',
     'PF08879.5', 'PF08880.6', 'PF08881.5', 'PF08882.6', 'PF08883.6', 'PF08884.6', 'PF08885.6', 'PF08886.6',
     'PF08887.6', 'PF08888.6', 'PF08889.6', 'PF08890.6', 'PF08891.6', 'PF08892.6', 'PF08893.5', 'PF08894.6',
     'PF08895.6', 'PF08896.5', 'PF08897.6', 'PF08898.5', 'PF08899.6', 'PF08900.6', 'PF08901.6', 'PF08902.6',
     'PF08903.6', 'PF08904.6', 'PF08905.6', 'PF08906.6', 'PF08907.6', 'PF08908.6', 'PF08909.6', 'PF08910.5',
     'PF08911.6', 'PF08912.6', 'PF08913.5', 'PF08914.6', 'PF08915.6', 'PF08916.6', 'PF08917.5', 'PF08918.5',
     'PF08919.5', 'PF08920.5', 'PF08921.6', 'PF08922.6', 'PF08923.5', 'PF08924.6', 'PF08925.6', 'PF08926.6',
     'PF08928.5', 'PF08929.5', 'PF08930.5', 'PF08931.5', 'PF08932.5', 'PF08933.6', 'PF08934.5', 'PF08935.5',
     'PF08936.5', 'PF08937.6', 'PF08938.5', 'PF08939.5', 'PF08940.6', 'PF08941.5', 'PF08942.5', 'PF08943.5',
     'PF08944.6', 'PF08945.5', 'PF08946.5', 'PF08947.5', 'PF08948.5', 'PF08949.5', 'PF08950.5', 'PF08951.5',
     'PF08952.6', 'PF08953.6', 'PF08954.6', 'PF08955.5', 'PF08956.5', 'PF08958.5', 'PF08960.5', 'PF08961.5',
     'PF08962.6', 'PF08963.5', 'PF08964.5', 'PF08965.5', 'PF08966.6', 'PF08967.5', 'PF08968.5', 'PF08969.6',
     'PF08970.5', 'PF08971.6', 'PF08972.6', 'PF08973.5', 'PF08974.5', 'PF08975.5', 'PF08976.6', 'PF08977.5',
     'PF08978.5', 'PF08979.6', 'PF08980.5', 'PF08982.6', 'PF08983.5', 'PF08984.6', 'PF08985.6', 'PF08986.5',
     'PF08987.5', 'PF08988.5', 'PF08989.5', 'PF08990.6', 'PF08991.5', 'PF08992.6', 'PF08993.5', 'PF08994.5',
     'PF08995.5', 'PF08996.5', 'PF08997.5', 'PF08998.6', 'PF08999.5', 'PF09000.5', 'PF09001.6', 'PF09002.6',
     'PF09003.5', 'PF09004.5', 'PF09005.5', 'PF09006.6', 'PF09007.6', 'PF09008.5', 'PF09009.5', 'PF09010.5',
     'PF09011.5', 'PF09012.5', 'PF09013.5', 'PF09014.5', 'PF09015.5', 'PF09016.5', 'PF09017.5', 'PF09018.6',
     'PF09019.6', 'PF09020.5', 'PF09021.6', 'PF09022.5', 'PF09023.5', 'PF09025.5', 'PF09026.5', 'PF09027.5',
     'PF09028.5', 'PF09029.5', 'PF09030.5', 'PF09032.6', 'PF09033.5', 'PF09034.5', 'PF09035.5', 'PF09036.5',
     'PF09037.5', 'PF09038.5', 'PF09039.6', 'PF09040.6', 'PF09041.5', 'PF09042.6', 'PF09043.6', 'PF09044.5',
     'PF09045.5', 'PF09046.5', 'PF09047.5', 'PF09048.5', 'PF09049.5', 'PF09050.5', 'PF09051.5', 'PF09052.5',
     'PF09053.5', 'PF09055.6', 'PF09056.6', 'PF09057.5', 'PF09058.5', 'PF09059.5', 'PF09060.5', 'PF09061.1',
     'PF09062.5', 'PF09063.5', 'PF09064.5', 'PF09065.5', 'PF09066.5', 'PF09067.5', 'PF09068.6', 'PF09069.6',
     'PF09070.6', 'PF09071.5', 'PF09072.5', 'PF09073.5', 'PF09074.5', 'PF09075.5', 'PF09076.5', 'PF09077.6',
     'PF09078.6', 'PF09079.6', 'PF09080.5', 'PF09081.5', 'PF09082.5', 'PF09083.5', 'PF09084.6', 'PF09085.5',
     'PF09086.6', 'PF09087.6', 'PF09088.6', 'PF09089.5', 'PF09090.6', 'PF09092.6', 'PF09093.6', 'PF09094.6',
     'PF09095.6', 'PF09096.5', 'PF09097.5', 'PF09098.5', 'PF09099.5', 'PF09100.5', 'PF09101.5', 'PF09102.5',
     'PF09103.5', 'PF09104.5', 'PF09105.5', 'PF09106.6', 'PF09107.6', 'PF09108.5', 'PF09109.5', 'PF09110.6',
     'PF09111.5', 'PF09112.5', 'PF09113.5', 'PF09114.5', 'PF09115.5', 'PF09116.5', 'PF09117.5', 'PF09118.6',
     'PF09119.5', 'PF09121.5', 'PF09122.5', 'PF09123.6', 'PF09124.5', 'PF09125.5', 'PF09126.5', 'PF09127.6',
     'PF09128.6', 'PF09129.6', 'PF09130.6', 'PF09131.5', 'PF09132.5', 'PF09133.5', 'PF09134.5', 'PF09135.6',
     'PF09136.5', 'PF09137.6', 'PF09138.6', 'PF09139.6', 'PF09140.6', 'PF09141.5', 'PF09142.6', 'PF09143.5',
     'PF09144.5', 'PF09145.5', 'PF09147.5', 'PF09148.5', 'PF09149.5', 'PF09150.5', 'PF09151.5', 'PF09152.5',
     'PF09153.5', 'PF09154.5', 'PF09155.5', 'PF09156.5', 'PF09157.6', 'PF09158.5', 'PF09159.5', 'PF09160.5',
     'PF09162.5', 'PF09163.6', 'PF09164.5', 'PF09165.5', 'PF09166.5', 'PF09167.6', 'PF09168.5', 'PF09169.5',
     'PF09170.5', 'PF09171.5', 'PF09172.6', 'PF09173.6', 'PF09174.5', 'PF09175.5', 'PF09176.6', 'PF09177.6',
     'PF09178.5', 'PF09179.6', 'PF09180.6', 'PF09181.5', 'PF09182.5', 'PF09183.5', 'PF09184.6', 'PF09185.5',
     'PF09186.6', 'PF09187.5', 'PF09188.5', 'PF09189.5', 'PF09190.6', 'PF09191.5', 'PF09192.5', 'PF09193.5',
     'PF09194.5', 'PF09195.6', 'PF09196.5', 'PF09197.5', 'PF09198.5', 'PF09199.5', 'PF09200.5', 'PF09201.5',
     'PF09202.6', 'PF09203.6', 'PF09204.5', 'PF09205.5', 'PF09206.6', 'PF09207.6', 'PF09208.5', 'PF09209.6',
     'PF09210.6', 'PF09211.5', 'PF09212.5', 'PF09213.5', 'PF09214.6', 'PF09215.5', 'PF09216.5', 'PF09217.5',
     'PF09218.5', 'PF09220.5', 'PF09221.5', 'PF09222.5', 'PF09223.6', 'PF09224.6', 'PF09225.5', 'PF09226.6',
     'PF09227.5', 'PF09228.5', 'PF09229.6', 'PF09230.5', 'PF09231.5', 'PF09232.5', 'PF09233.6', 'PF09234.5',
     'PF09235.5', 'PF09236.5', 'PF09237.6', 'PF09238.5', 'PF09239.6', 'PF09240.5', 'PF09241.5', 'PF09242.6',
     'PF09243.5', 'PF09244.5', 'PF09245.5', 'PF09246.5', 'PF09247.6', 'PF09248.5', 'PF09249.6', 'PF09250.6',
     'PF09251.5', 'PF09252.5', 'PF09253.5', 'PF09254.6', 'PF09255.5', 'PF09256.5', 'PF09257.5', 'PF09258.5',
     'PF09259.6', 'PF09260.6', 'PF09261.6', 'PF09262.6', 'PF09263.5', 'PF09264.5', 'PF09265.5', 'PF09266.5',
     'PF09267.5', 'PF09268.5', 'PF09269.6', 'PF09270.5', 'PF09271.6', 'PF09272.5', 'PF09273.6', 'PF09274.5',
     'PF09275.5', 'PF09276.5', 'PF09277.6', 'PF09278.6', 'PF09279.6', 'PF09280.6', 'PF09281.5', 'PF09282.5',
     'PF09284.5', 'PF09285.6', 'PF09286.6', 'PF09287.5', 'PF09288.5', 'PF09289.5', 'PF09290.6', 'PF09291.5',
     'PF09292.5', 'PF09293.5', 'PF09294.5', 'PF09295.5', 'PF09296.6', 'PF09297.6', 'PF09298.6', 'PF09299.6',
     'PF09300.5', 'PF09301.5', 'PF09302.6', 'PF09303.5', 'PF09304.5', 'PF09305.5', 'PF09306.5', 'PF09307.5',
     'PF09308.5', 'PF09309.5', 'PF09310.5', 'PF09311.6', 'PF09312.6', 'PF09313.6', 'PF09314.6', 'PF09315.6',
     'PF09316.5', 'PF09317.6', 'PF09318.5', 'PF09320.6', 'PF09321.5', 'PF09322.5', 'PF09323.5', 'PF09324.5',
     'PF09325.5', 'PF09326.6', 'PF09327.6', 'PF09328.5', 'PF09329.6', 'PF09330.6', 'PF09331.6', 'PF09332.6',
     'PF09333.6', 'PF09334.6', 'PF09335.6', 'PF09336.5', 'PF09337.5', 'PF09338.6', 'PF09339.5', 'PF09340.5',
     'PF09341.5', 'PF09342.6', 'PF09343.5', 'PF09344.5', 'PF09345.5', 'PF09346.5', 'PF09347.5', 'PF09348.5',
     'PF09349.5', 'PF09350.5', 'PF09351.5', 'PF09353.5', 'PF09354.5', 'PF09355.5', 'PF09356.5', 'PF09357.5',
     'PF09358.5', 'PF09359.5', 'PF09360.5', 'PF09361.5', 'PF09362.5', 'PF09363.5', 'PF09364.5', 'PF09365.5',
     'PF09366.5', 'PF09367.5', 'PF09368.5', 'PF09369.5', 'PF09370.5', 'PF09371.5', 'PF09372.5', 'PF09373.5',
     'PF09374.5', 'PF09375.5', 'PF09376.5', 'PF09377.5', 'PF09378.5', 'PF09379.5', 'PF09380.5', 'PF09381.5',
     'PF09382.5', 'PF09383.5', 'PF09384.5', 'PF09385.5', 'PF09386.5', 'PF09387.5', 'PF09388.5', 'PF09390.5',
     'PF09391.5', 'PF09392.5', 'PF09393.5', 'PF09394.5', 'PF09396.5', 'PF09397.5', 'PF09398.5', 'PF09399.5',
     'PF09400.5', 'PF09401.5', 'PF09402.5', 'PF09403.5', 'PF09404.5', 'PF09405.5', 'PF09406.5', 'PF09407.5',
     'PF09408.5', 'PF09409.5', 'PF09411.5', 'PF09412.5', 'PF09413.5', 'PF09414.5', 'PF09415.5', 'PF09416.5',
     'PF09418.5', 'PF09419.5', 'PF09420.5', 'PF09421.5', 'PF09422.5', 'PF09423.5', 'PF09424.5', 'PF09425.5',
     'PF09426.5', 'PF09427.5', 'PF09428.5', 'PF09429.5', 'PF09430.5', 'PF09431.5', 'PF09432.5', 'PF09435.5',
     'PF09436.5', 'PF09437.5', 'PF09438.5', 'PF09439.5', 'PF09440.5', 'PF09441.5', 'PF09442.5', 'PF09443.5',
     'PF09444.5', 'PF09445.5', 'PF09446.5', 'PF09447.5', 'PF09448.5', 'PF09449.5', 'PF09450.5', 'PF09451.5',
     'PF09452.5', 'PF09453.5', 'PF09454.5', 'PF09455.5', 'PF09456.5', 'PF09457.5', 'PF09458.5', 'PF09459.5',
     'PF09460.5', 'PF09461.5', 'PF09462.5', 'PF09463.5', 'PF09465.5', 'PF09466.5', 'PF09467.5', 'PF09468.5',
     'PF09469.5', 'PF09470.5', 'PF09471.5', 'PF09472.5', 'PF09474.5', 'PF09475.5', 'PF09476.5', 'PF09477.5',
     'PF09478.5', 'PF09479.5', 'PF09480.5', 'PF09481.5', 'PF09482.5', 'PF09483.5', 'PF09484.5', 'PF09485.5',
     'PF09486.5', 'PF09487.5', 'PF09488.5', 'PF09489.5', 'PF09490.5', 'PF09491.5', 'PF09492.5', 'PF09493.5',
     'PF09494.5', 'PF09495.5', 'PF09496.5', 'PF09497.5', 'PF09498.5', 'PF09499.5', 'PF09500.5', 'PF09501.5',
     'PF09502.5', 'PF09504.5', 'PF09505.5', 'PF09506.5', 'PF09507.5', 'PF09508.5', 'PF09509.5', 'PF09510.5',
     'PF09511.5', 'PF09512.5', 'PF09514.5', 'PF09515.5', 'PF09516.5', 'PF09517.5', 'PF09518.5', 'PF09519.5',
     'PF09520.5', 'PF09521.5', 'PF09522.5', 'PF09523.5', 'PF09524.5', 'PF09526.5', 'PF09527.5', 'PF09528.5',
     'PF09529.5', 'PF09531.5', 'PF09532.5', 'PF09533.5', 'PF09534.5', 'PF09535.5', 'PF09536.5', 'PF09537.5',
     'PF09538.5', 'PF09539.5', 'PF09543.5', 'PF09544.5', 'PF09545.5', 'PF09546.5', 'PF09547.5', 'PF09548.5',
     'PF09549.5', 'PF09550.5', 'PF09551.5', 'PF09552.5', 'PF09553.5', 'PF09554.5', 'PF09556.5', 'PF09557.5',
     'PF09558.5', 'PF09559.5', 'PF09560.5', 'PF09561.5', 'PF09562.5', 'PF09563.5', 'PF09564.5', 'PF09565.5',
     'PF09566.5', 'PF09567.5', 'PF09568.5', 'PF09569.5', 'PF09570.5', 'PF09571.5', 'PF09572.5', 'PF09573.5',
     'PF09574.5', 'PF09575.5', 'PF09577.5', 'PF09578.5', 'PF09579.5', 'PF09580.5', 'PF09581.5', 'PF09582.5',
     'PF09583.5', 'PF09584.5', 'PF09585.5', 'PF09586.5', 'PF09587.5', 'PF09588.5', 'PF09589.5', 'PF09590.5',
     'PF09591.5', 'PF09592.5', 'PF09593.5', 'PF09594.5', 'PF09595.5', 'PF09596.5', 'PF09597.5', 'PF09598.5',
     'PF09599.5', 'PF09600.5', 'PF09601.5', 'PF09602.5', 'PF09603.5', 'PF09604.5', 'PF09605.5', 'PF09606.5',
     'PF09607.5', 'PF09608.5', 'PF09609.5', 'PF09610.5', 'PF09611.5', 'PF09612.5', 'PF09613.5', 'PF09614.5',
     'PF09615.5', 'PF09617.5', 'PF09618.5', 'PF09619.5', 'PF09620.5', 'PF09621.5', 'PF09622.5', 'PF09623.5',
     'PF09624.5', 'PF09625.5', 'PF09626.5', 'PF09627.5', 'PF09628.5', 'PF09629.5', 'PF09630.5', 'PF09631.5',
     'PF09632.5', 'PF09633.5', 'PF09634.5', 'PF09635.5', 'PF09636.5', 'PF09637.5', 'PF09638.5', 'PF09639.5',
     'PF09640.5', 'PF09641.5', 'PF09642.5', 'PF09643.5', 'PF09644.5', 'PF09645.5', 'PF09646.5', 'PF09648.5',
     'PF09649.5', 'PF09650.5', 'PF09651.5', 'PF09652.5', 'PF09654.5', 'PF09655.5', 'PF09656.5', 'PF09657.5',
     'PF09658.5', 'PF09659.5', 'PF09660.5', 'PF09661.5', 'PF09662.5', 'PF09663.5', 'PF09664.5', 'PF09665.5',
     'PF09666.5', 'PF09667.5', 'PF09668.5', 'PF09669.5', 'PF09670.5', 'PF09671.5', 'PF09673.5', 'PF09674.5',
     'PF09675.5', 'PF09676.5', 'PF09677.5', 'PF09678.5', 'PF09679.5', 'PF09680.5', 'PF09681.5', 'PF09682.5',
     'PF09683.5', 'PF09684.5', 'PF09685.5', 'PF09686.5', 'PF09687.5', 'PF09688.5', 'PF09689.5', 'PF09690.5',
     'PF09691.5', 'PF09692.5', 'PF09693.5', 'PF09694.5', 'PF09695.5', 'PF09696.5', 'PF09697.5', 'PF09698.5',
     'PF09699.5', 'PF09700.5', 'PF09701.5', 'PF09702.5', 'PF09703.5', 'PF09704.5', 'PF09706.5', 'PF09707.5',
     'PF09709.5', 'PF09710.5', 'PF09711.5', 'PF09712.5', 'PF09713.5', 'PF09715.5', 'PF09716.5', 'PF09717.5',
     'PF09718.5', 'PF09719.5', 'PF09720.5', 'PF09721.5', 'PF09722.5', 'PF09723.5', 'PF09724.4', 'PF09725.4',
     'PF09726.4', 'PF09727.4', 'PF09728.4', 'PF09729.4', 'PF09730.4', 'PF09731.4', 'PF09732.4', 'PF09733.4',
     'PF09734.4', 'PF09735.4', 'PF09736.4', 'PF09737.4', 'PF09738.4', 'PF09739.4', 'PF09740.4', 'PF09741.4',
     'PF09742.4', 'PF09743.4', 'PF09744.4', 'PF09745.4', 'PF09746.4', 'PF09747.4', 'PF09748.4', 'PF09749.4',
     'PF09750.4', 'PF09751.4', 'PF09752.4', 'PF09753.4', 'PF09754.4', 'PF09755.4', 'PF09756.4', 'PF09757.4',
     'PF09758.4', 'PF09759.4', 'PF09762.4', 'PF09763.4', 'PF09764.4', 'PF09765.4', 'PF09766.4', 'PF09767.4',
     'PF09768.4', 'PF09769.4', 'PF09770.4', 'PF09771.4', 'PF09772.4', 'PF09773.4', 'PF09774.4', 'PF09775.4',
     'PF09776.4', 'PF09777.4', 'PF09778.4', 'PF09779.4', 'PF09781.4', 'PF09782.4', 'PF09783.4', 'PF09784.4',
     'PF09785.4', 'PF09786.4', 'PF09787.4', 'PF09788.4', 'PF09789.4', 'PF09790.4', 'PF09791.4', 'PF09792.4',
     'PF09793.4', 'PF09794.4', 'PF09795.4', 'PF09796.4', 'PF09797.4', 'PF09798.4', 'PF09799.4', 'PF09801.4',
     'PF09802.4', 'PF09803.4', 'PF09804.4', 'PF09805.4', 'PF09806.4', 'PF09807.4', 'PF09808.4', 'PF09809.4',
     'PF09810.4', 'PF09811.4', 'PF09812.4', 'PF09813.4', 'PF09814.4', 'PF09815.4', 'PF09816.4', 'PF09817.4',
     'PF09818.4', 'PF09819.4', 'PF09820.4', 'PF09821.4', 'PF09822.4', 'PF09823.4', 'PF09824.4', 'PF09825.4',
     'PF09826.4', 'PF09827.4', 'PF09828.4', 'PF09829.4', 'PF09830.4', 'PF09831.4', 'PF09832.4', 'PF09834.4',
     'PF09835.4', 'PF09836.4', 'PF09837.4', 'PF09838.4', 'PF09839.4', 'PF09840.4', 'PF09842.4', 'PF09843.4',
     'PF09844.4', 'PF09845.4', 'PF09846.4', 'PF09847.4', 'PF09848.4', 'PF09849.4', 'PF09850.4', 'PF09851.4',
     'PF09852.4', 'PF09853.4', 'PF09855.4', 'PF09856.4', 'PF09857.4', 'PF09858.4', 'PF09859.4', 'PF09860.4',
     'PF09861.4', 'PF09862.4', 'PF09863.4', 'PF09864.4', 'PF09865.4', 'PF09866.4', 'PF09867.4', 'PF09868.4',
     'PF09869.4', 'PF09870.4', 'PF09871.4', 'PF09872.4', 'PF09873.4', 'PF09874.4', 'PF09875.4', 'PF09876.4',
     'PF09877.4', 'PF09878.4', 'PF09879.4', 'PF09880.4', 'PF09881.4', 'PF09882.4', 'PF09883.4', 'PF09884.4',
     'PF09885.4', 'PF09886.4', 'PF09887.4', 'PF09888.4', 'PF09889.4', 'PF09890.4', 'PF09891.4', 'PF09892.4',
     'PF09893.4', 'PF09894.4', 'PF09895.4', 'PF09897.4', 'PF09898.4', 'PF09899.4', 'PF09900.4', 'PF09902.4',
     'PF09903.4', 'PF09904.4', 'PF09905.4', 'PF09906.4', 'PF09907.4', 'PF09909.4', 'PF09910.4', 'PF09911.4',
     'PF09912.4', 'PF09913.4', 'PF09916.4', 'PF09917.4', 'PF09918.4', 'PF09919.4', 'PF09920.4', 'PF09921.4',
     'PF09922.4', 'PF09923.4', 'PF09924.4', 'PF09925.4', 'PF09926.4', 'PF09928.4', 'PF09929.4', 'PF09930.4',
     'PF09931.4', 'PF09932.4', 'PF09933.4', 'PF09935.4', 'PF09936.4', 'PF09937.4', 'PF09938.4', 'PF09939.4',
     'PF09940.4', 'PF09941.4', 'PF09943.4', 'PF09945.4', 'PF09946.4', 'PF09947.4', 'PF09948.4', 'PF09949.4',
     'PF09950.4', 'PF09951.4', 'PF09952.4', 'PF09953.4', 'PF09954.4', 'PF09955.4', 'PF09956.4', 'PF09957.4',
     'PF09958.4', 'PF09959.4', 'PF09960.4', 'PF09961.4', 'PF09962.4', 'PF09963.4', 'PF09964.4', 'PF09965.4',
     'PF09966.4', 'PF09967.4', 'PF09968.4', 'PF09969.4', 'PF09970.4', 'PF09971.4', 'PF09972.4', 'PF09973.4',
     'PF09974.4', 'PF09976.4', 'PF09977.4', 'PF09979.4', 'PF09980.4', 'PF09981.4', 'PF09982.4', 'PF09983.4',
     'PF09984.4', 'PF09985.4', 'PF09986.4', 'PF09987.4', 'PF09988.4', 'PF09989.4', 'PF09990.4', 'PF09991.4',
     'PF09992.4', 'PF09994.4', 'PF09995.4', 'PF09996.4', 'PF09997.4', 'PF09998.4', 'PF09999.4', 'PF10000.4',
     'PF10001.4', 'PF10002.4', 'PF10003.4', 'PF10004.4', 'PF10005.4', 'PF10006.4', 'PF10007.4', 'PF10008.4',
     'PF10009.4', 'PF10011.4', 'PF10012.4', 'PF10013.4', 'PF10014.4', 'PF10015.4', 'PF10016.4', 'PF10017.4',
     'PF10018.4', 'PF10020.4', 'PF10021.4', 'PF10022.4', 'PF10023.4', 'PF10025.4', 'PF10026.4', 'PF10027.4',
     'PF10028.4', 'PF10029.4', 'PF10030.4', 'PF10031.4', 'PF10032.4', 'PF10033.4', 'PF10034.4', 'PF10035.4',
     'PF10036.4', 'PF10037.4', 'PF10038.4', 'PF10039.4', 'PF10040.4', 'PF10041.4', 'PF10042.4', 'PF10043.4',
     'PF10044.4', 'PF10045.4', 'PF10046.4', 'PF10047.4', 'PF10048.4', 'PF10049.4', 'PF10050.4', 'PF10051.4',
     'PF10052.4', 'PF10053.4', 'PF10054.4', 'PF10055.4', 'PF10056.4', 'PF10057.4', 'PF10058.4', 'PF10060.4',
     'PF10061.4', 'PF10062.4', 'PF10063.4', 'PF10065.4', 'PF10066.4', 'PF10067.4', 'PF10069.4', 'PF10070.4',
     'PF10071.4', 'PF10073.4', 'PF10074.4', 'PF10075.4', 'PF10076.4', 'PF10077.4', 'PF10078.4', 'PF10079.4',
     'PF10080.4', 'PF10081.4', 'PF10082.4', 'PF10083.4', 'PF10084.4', 'PF10086.4', 'PF10087.4', 'PF10088.4',
     'PF10090.4', 'PF10091.4', 'PF10092.4', 'PF10093.4', 'PF10094.4', 'PF10095.4', 'PF10096.4', 'PF10097.4',
     'PF10098.4', 'PF10099.4', 'PF10100.4', 'PF10101.4', 'PF10102.4', 'PF10103.4', 'PF10104.4', 'PF10105.4',
     'PF10106.4', 'PF10107.4', 'PF10108.4', 'PF10109.4', 'PF10110.4', 'PF10111.4', 'PF10112.4', 'PF10113.4',
     'PF10114.4', 'PF10115.4', 'PF10116.4', 'PF10117.4', 'PF10118.4', 'PF10119.4', 'PF10120.4', 'PF10122.4',
     'PF10123.4', 'PF10124.4', 'PF10125.4', 'PF10126.4', 'PF10127.4', 'PF10128.4', 'PF10129.4', 'PF10130.4',
     'PF10131.4', 'PF10133.4', 'PF10134.4', 'PF10135.4', 'PF10136.4', 'PF10137.4', 'PF10138.4', 'PF10139.4',
     'PF10140.4', 'PF10141.4', 'PF10142.4', 'PF10143.4', 'PF10144.4', 'PF10145.4', 'PF10146.4', 'PF10147.4',
     'PF10148.4', 'PF10149.4', 'PF10150.4', 'PF10151.4', 'PF10152.4', 'PF10153.4', 'PF10154.4', 'PF10155.4',
     'PF10156.4', 'PF10157.4', 'PF10158.4', 'PF10159.4', 'PF10160.4', 'PF10161.4', 'PF10162.4', 'PF10163.4',
     'PF10164.4', 'PF10165.4', 'PF10166.4', 'PF10167.4', 'PF10168.4', 'PF10169.4', 'PF10170.4', 'PF10171.4',
     'PF10172.4', 'PF10173.4', 'PF10174.4', 'PF10175.4', 'PF10176.4', 'PF10177.4', 'PF10178.4', 'PF10179.4',
     'PF10180.4', 'PF10181.4', 'PF10182.4', 'PF10183.4', 'PF10184.4', 'PF10185.4', 'PF10186.4', 'PF10187.4',
     'PF10188.4', 'PF10189.4', 'PF10190.4', 'PF10191.4', 'PF10192.4', 'PF10193.4', 'PF10195.4', 'PF10197.4',
     'PF10198.4', 'PF10199.4', 'PF10200.4', 'PF10203.4', 'PF10204.4', 'PF10205.4', 'PF10206.4', 'PF10208.4',
     'PF10209.4', 'PF10210.4', 'PF10211.4', 'PF10212.4', 'PF10213.4', 'PF10214.4', 'PF10215.4', 'PF10216.4',
     'PF10217.4', 'PF10218.4', 'PF10220.4', 'PF10221.4', 'PF10222.4', 'PF10223.4', 'PF10224.4', 'PF10225.4',
     'PF10226.4', 'PF10228.4', 'PF10229.4', 'PF10230.4', 'PF10231.4', 'PF10232.4', 'PF10233.4', 'PF10234.4',
     'PF10235.4', 'PF10236.4', 'PF10237.4', 'PF10238.4', 'PF10239.4', 'PF10240.4', 'PF10241.4', 'PF10242.4',
     'PF10243.4', 'PF10244.4', 'PF10245.4', 'PF10246.4', 'PF10247.4', 'PF10248.4', 'PF10249.4', 'PF10250.4',
     'PF10251.4', 'PF10252.4', 'PF10253.4', 'PF10254.4', 'PF10255.4', 'PF10256.4', 'PF10257.4', 'PF10258.4',
     'PF10259.4', 'PF10260.4', 'PF10261.4', 'PF10262.4', 'PF10263.4', 'PF10264.4', 'PF10265.4', 'PF10266.4',
     'PF10267.4', 'PF10268.4', 'PF10269.4', 'PF10270.4', 'PF10271.4', 'PF10272.4', 'PF10273.4', 'PF10274.4',
     'PF10275.4', 'PF10276.4', 'PF10277.4', 'PF10278.4', 'PF10279.4', 'PF10280.4', 'PF10281.4', 'PF10282.4',
     'PF10283.4', 'PF10284.4', 'PF10285.4', 'PF10287.4', 'PF10288.4', 'PF10290.4', 'PF10291.4', 'PF10292.4',
     'PF10293.4', 'PF10294.4', 'PF10295.4', 'PF10296.4', 'PF10297.4', 'PF10298.4', 'PF10300.4', 'PF10302.4',
     'PF10303.4', 'PF10304.4', 'PF10305.4', 'PF10306.4', 'PF10307.4', 'PF10309.4', 'PF10310.4', 'PF10311.4',
     'PF10312.4', 'PF10313.4', 'PF10315.4', 'PF10316.4', 'PF10317.4', 'PF10318.4', 'PF10319.4', 'PF10320.4',
     'PF10321.4', 'PF10322.4', 'PF10323.4', 'PF10324.4', 'PF10325.4', 'PF10326.4', 'PF10327.4', 'PF10328.4',
     'PF10329.4', 'PF10330.4', 'PF10332.4', 'PF10333.4', 'PF10334.4', 'PF10335.4', 'PF10336.4', 'PF10337.4',
     'PF10338.4', 'PF10339.4', 'PF10340.4', 'PF10341.4', 'PF10342.4', 'PF10343.4', 'PF10344.4', 'PF10345.4',
     'PF10346.4', 'PF10347.4', 'PF10348.4', 'PF10349.4', 'PF10350.4', 'PF10351.4', 'PF10353.4', 'PF10354.4',
     'PF10355.4', 'PF10356.4', 'PF10357.4', 'PF10358.4', 'PF10359.4', 'PF10360.4', 'PF10361.4', 'PF10363.4',
     'PF10364.4', 'PF10365.4', 'PF10366.4', 'PF10367.4', 'PF10368.4', 'PF10369.4', 'PF10370.4', 'PF10371.4',
     'PF10372.4', 'PF10373.4', 'PF10374.4', 'PF10375.4', 'PF10376.4', 'PF10377.4', 'PF10378.4', 'PF10379.4',
     'PF10380.4', 'PF10381.4', 'PF10382.4', 'PF10383.4', 'PF10384.4', 'PF10385.4', 'PF10386.4', 'PF10387.4',
     'PF10388.4', 'PF10389.4', 'PF10390.4', 'PF10391.4', 'PF10392.4', 'PF10393.4', 'PF10394.4', 'PF10395.4',
     'PF10396.4', 'PF10397.4', 'PF10398.4', 'PF10399.4', 'PF10400.4', 'PF10401.4', 'PF10403.4', 'PF10404.4',
     'PF10405.4', 'PF10406.4', 'PF10407.4', 'PF10408.4', 'PF10409.4', 'PF10410.4', 'PF10411.4', 'PF10412.4',
     'PF10413.4', 'PF10414.4', 'PF10415.4', 'PF10416.4', 'PF10417.4', 'PF10418.4', 'PF10419.4', 'PF10420.4',
     'PF10421.4', 'PF10422.4', 'PF10423.4', 'PF10425.4', 'PF10426.4', 'PF10427.4', 'PF10428.4', 'PF10429.4',
     'PF10430.4', 'PF10431.4', 'PF10432.4', 'PF10433.4', 'PF10434.4', 'PF10435.4', 'PF10436.4', 'PF10437.4',
     'PF10438.4', 'PF10439.4', 'PF10440.4', 'PF10441.4', 'PF10442.4', 'PF10443.4', 'PF10444.4', 'PF10445.4',
     'PF10446.4', 'PF10447.4', 'PF10448.4', 'PF10450.4', 'PF10451.4', 'PF10452.4', 'PF10453.4', 'PF10454.4',
     'PF10455.4', 'PF10456.4', 'PF10457.4', 'PF10458.4', 'PF10459.4', 'PF10460.4', 'PF10461.4', 'PF10462.4',
     'PF10463.4', 'PF10464.4', 'PF10465.4', 'PF10466.4', 'PF10467.4', 'PF10468.4', 'PF10469.4', 'PF10470.4',
     'PF10471.4', 'PF10472.4', 'PF10473.4', 'PF10474.4', 'PF10475.4', 'PF10476.4', 'PF10477.4', 'PF10479.4',
     'PF10480.4', 'PF10481.4', 'PF10482.4', 'PF10483.4', 'PF10484.4', 'PF10486.4', 'PF10487.4', 'PF10488.4',
     'PF10489.4', 'PF10490.4', 'PF10491.4', 'PF10492.4', 'PF10493.4', 'PF10494.4', 'PF10495.4', 'PF10496.4',
     'PF10497.4', 'PF10498.4', 'PF10500.4', 'PF10501.4', 'PF10502.4', 'PF10503.4', 'PF10504.4', 'PF10505.4',
     'PF10506.4', 'PF10507.4', 'PF10508.4', 'PF10509.4', 'PF10510.4', 'PF10511.4', 'PF10512.4', 'PF10513.4',
     'PF10514.4', 'PF10515.4', 'PF10516.4', 'PF10517.4', 'PF10518.4', 'PF10520.4', 'PF10521.4', 'PF10522.4',
     'PF10523.4', 'PF10524.4', 'PF10525.4', 'PF10528.4', 'PF10529.4', 'PF10530.4', 'PF10531.4', 'PF10532.4',
     'PF10533.4', 'PF10534.4', 'PF10536.4', 'PF10537.4', 'PF10538.4', 'PF10539.4', 'PF10540.4', 'PF10541.4',
     'PF10542.4', 'PF10543.4', 'PF10544.4', 'PF10545.4', 'PF10546.4', 'PF10547.4', 'PF10548.4', 'PF10549.4',
     'PF10550.4', 'PF10551.4', 'PF10552.4', 'PF10553.4', 'PF10554.4', 'PF10555.4', 'PF10557.4', 'PF10558.4',
     'PF10559.4', 'PF10561.4', 'PF10562.4', 'PF10563.4', 'PF10564.4', 'PF10565.4', 'PF10566.4', 'PF10567.4',
     'PF10568.4', 'PF10569.4', 'PF10570.4', 'PF10571.4', 'PF10572.4', 'PF10573.4', 'PF10574.4', 'PF10576.4',
     'PF10577.4', 'PF10578.4', 'PF10579.4', 'PF10580.4', 'PF10581.4', 'PF10582.4', 'PF10583.4', 'PF10584.4',
     'PF10585.4', 'PF10587.4', 'PF10588.4', 'PF10589.4', 'PF10590.4', 'PF10591.4', 'PF10592.4', 'PF10593.4',
     'PF10595.4', 'PF10596.4', 'PF10597.4', 'PF10598.4', 'PF10599.4', 'PF10600.4', 'PF10601.4', 'PF10602.4',
     'PF10604.4', 'PF10605.4', 'PF10606.4', 'PF10607.4', 'PF10608.4', 'PF10609.4', 'PF10610.4', 'PF10611.4',
     'PF10612.4', 'PF10613.4', 'PF10614.4', 'PF10615.4', 'PF10616.4', 'PF10617.4', 'PF10618.4', 'PF10620.4',
     'PF10621.4', 'PF10622.4', 'PF10623.4', 'PF10624.4', 'PF10625.4', 'PF10626.4', 'PF10627.4', 'PF10628.4',
     'PF10629.4', 'PF10630.4', 'PF10631.4', 'PF10632.4', 'PF10633.4', 'PF10634.4', 'PF10635.4', 'PF10636.4',
     'PF10637.4', 'PF10638.4', 'PF10639.4', 'PF10640.4', 'PF10642.4', 'PF10643.4', 'PF10644.4', 'PF10645.4',
     'PF10646.4', 'PF10647.4', 'PF10648.4', 'PF10649.4', 'PF10650.4', 'PF10651.4', 'PF10652.4', 'PF10653.4',
     'PF10654.4', 'PF10655.4', 'PF10656.4', 'PF10657.4', 'PF10658.4', 'PF10659.4', 'PF10660.4', 'PF10661.4',
     'PF10662.4', 'PF10664.4', 'PF10665.4', 'PF10666.4', 'PF10667.4', 'PF10668.4', 'PF10669.4', 'PF10670.4',
     'PF10671.4', 'PF10672.4', 'PF10673.4', 'PF10674.4', 'PF10675.4', 'PF10676.4', 'PF10677.4', 'PF10678.4',
     'PF10679.4', 'PF10680.4', 'PF10681.4', 'PF10682.4', 'PF10683.4', 'PF10684.4', 'PF10685.4', 'PF10686.4',
     'PF10688.4', 'PF10689.4', 'PF10690.4', 'PF10691.4', 'PF10692.4', 'PF10693.4', 'PF10694.4', 'PF10696.4',
     'PF10697.4', 'PF10698.4', 'PF10699.4', 'PF10702.4', 'PF10703.4', 'PF10704.4', 'PF10705.4', 'PF10706.4',
     'PF10707.4', 'PF10708.4', 'PF10709.4', 'PF10710.4', 'PF10711.4', 'PF10712.4', 'PF10713.4', 'PF10714.4',
     'PF10715.4', 'PF10716.4', 'PF10717.4', 'PF10718.4', 'PF10719.4', 'PF10720.4', 'PF10721.4', 'PF10722.4',
     'PF10723.4', 'PF10724.4', 'PF10725.4', 'PF10726.4', 'PF10727.4', 'PF10728.4', 'PF10729.4', 'PF10730.4',
     'PF10731.4', 'PF10732.4', 'PF10733.4', 'PF10734.4', 'PF10735.4', 'PF10736.4', 'PF10737.4', 'PF10738.4',
     'PF10739.4', 'PF10740.4', 'PF10741.4', 'PF10742.4', 'PF10743.4', 'PF10744.4', 'PF10745.4', 'PF10746.4',
     'PF10747.4', 'PF10748.4', 'PF10749.4', 'PF10750.4', 'PF10751.4', 'PF10752.4', 'PF10753.4', 'PF10754.4',
     'PF10755.4', 'PF10756.4', 'PF10757.4', 'PF10758.4', 'PF10759.4', 'PF10761.4', 'PF10762.4', 'PF10763.4',
     'PF10764.4', 'PF10765.4', 'PF10766.4', 'PF10767.4', 'PF10768.4', 'PF10769.4', 'PF10771.4', 'PF10772.4',
     'PF10774.4', 'PF10775.4', 'PF10776.4', 'PF10777.4', 'PF10778.4', 'PF10779.4', 'PF10780.4', 'PF10781.4',
     'PF10782.4', 'PF10783.4', 'PF10784.4', 'PF10785.4', 'PF10786.4', 'PF10787.4', 'PF10788.4', 'PF10789.4',
     'PF10790.4', 'PF10791.4', 'PF10792.4', 'PF10793.4', 'PF10794.4', 'PF10795.4', 'PF10796.4', 'PF10797.4',
     'PF10798.3', 'PF10799.3', 'PF10800.3', 'PF10801.3', 'PF10802.3', 'PF10803.3', 'PF10804.3', 'PF10805.3',
     'PF10806.3', 'PF10807.3', 'PF10808.3', 'PF10809.3', 'PF10810.3', 'PF10811.3', 'PF10812.3', 'PF10813.3',
     'PF10814.3', 'PF10815.3', 'PF10816.3', 'PF10817.3', 'PF10818.3', 'PF10819.3', 'PF10820.3', 'PF10821.3',
     'PF10823.3', 'PF10824.3', 'PF10825.3', 'PF10826.3', 'PF10827.3', 'PF10828.3', 'PF10829.3', 'PF10830.3',
     'PF10831.3', 'PF10832.3', 'PF10833.3', 'PF10834.3', 'PF10835.3', 'PF10836.3', 'PF10837.3', 'PF10838.3',
     'PF10839.3', 'PF10840.3', 'PF10841.3', 'PF10842.3', 'PF10843.3', 'PF10844.3', 'PF10845.3', 'PF10846.3',
     'PF10847.3', 'PF10848.3', 'PF10849.3', 'PF10850.3', 'PF10851.3', 'PF10852.3', 'PF10853.3', 'PF10854.3',
     'PF10855.3', 'PF10856.3', 'PF10857.3', 'PF10858.3', 'PF10859.3', 'PF10860.3', 'PF10861.3', 'PF10862.3',
     'PF10863.3', 'PF10864.3', 'PF10865.3', 'PF10866.3', 'PF10867.3', 'PF10868.3', 'PF10869.3', 'PF10870.3',
     'PF10871.3', 'PF10872.3', 'PF10873.3', 'PF10874.3', 'PF10875.3', 'PF10876.3', 'PF10877.3', 'PF10878.3',
     'PF10879.3', 'PF10880.3', 'PF10881.3', 'PF10882.3', 'PF10883.3', 'PF10884.3', 'PF10885.3', 'PF10886.3',
     'PF10887.3', 'PF10888.3', 'PF10890.3', 'PF10891.3', 'PF10892.3', 'PF10893.3', 'PF10894.3', 'PF10895.3',
     'PF10896.3', 'PF10897.3', 'PF10898.3', 'PF10899.3', 'PF10901.3', 'PF10902.3', 'PF10903.3', 'PF10904.3',
     'PF10905.3', 'PF10906.3', 'PF10907.3', 'PF10908.3', 'PF10909.3', 'PF10910.3', 'PF10911.3', 'PF10912.3',
     'PF10913.3', 'PF10914.3', 'PF10915.3', 'PF10916.3', 'PF10917.3', 'PF10918.3', 'PF10920.3', 'PF10921.3',
     'PF10922.3', 'PF10923.3', 'PF10924.3', 'PF10925.3', 'PF10926.3', 'PF10927.3', 'PF10928.3', 'PF10929.3',
     'PF10930.3', 'PF10931.3', 'PF10932.3', 'PF10933.3', 'PF10934.3', 'PF10935.3', 'PF10936.3', 'PF10937.3',
     'PF10938.3', 'PF10939.3', 'PF10940.3', 'PF10941.3', 'PF10942.3', 'PF10943.3', 'PF10944.3', 'PF10945.3',
     'PF10946.3', 'PF10947.3', 'PF10948.3', 'PF10949.3', 'PF10950.3', 'PF10951.3', 'PF10952.3', 'PF10953.3',
     'PF10954.3', 'PF10955.3', 'PF10956.3', 'PF10957.3', 'PF10958.3', 'PF10959.3', 'PF10960.3', 'PF10961.3',
     'PF10962.3', 'PF10963.3', 'PF10964.3', 'PF10965.3', 'PF10966.3', 'PF10967.3', 'PF10968.3', 'PF10969.3',
     'PF10970.3', 'PF10971.3', 'PF10972.3', 'PF10973.3', 'PF10974.3', 'PF10975.3', 'PF10976.3', 'PF10977.3',
     'PF10978.3', 'PF10979.3', 'PF10980.3', 'PF10981.3', 'PF10982.3', 'PF10983.3', 'PF10984.3', 'PF10985.3',
     'PF10986.3', 'PF10987.3', 'PF10988.3', 'PF10989.3', 'PF10990.3', 'PF10991.3', 'PF10992.3', 'PF10993.3',
     'PF10994.3', 'PF10995.3', 'PF10996.3', 'PF10997.3', 'PF10998.3', 'PF10999.3', 'PF11000.3', 'PF11001.3',
     'PF11002.3', 'PF11003.3', 'PF11004.3', 'PF11005.3', 'PF11006.3', 'PF11007.3', 'PF11008.3', 'PF11009.3',
     'PF11010.3', 'PF11011.3', 'PF11012.3', 'PF11013.3', 'PF11014.3', 'PF11015.3', 'PF11016.3', 'PF11017.3',
     'PF11018.3', 'PF11019.3', 'PF11020.3', 'PF11021.3', 'PF11022.3', 'PF11023.3', 'PF11024.3', 'PF11025.3',
     'PF11026.3', 'PF11027.3', 'PF11028.3', 'PF11029.3', 'PF11030.3', 'PF11031.3', 'PF11032.3', 'PF11033.3',
     'PF11034.3', 'PF11035.3', 'PF11036.3', 'PF11037.3', 'PF11038.3', 'PF11039.3', 'PF11040.3', 'PF11041.3',
     'PF11042.3', 'PF11043.3', 'PF11044.3', 'PF11045.3', 'PF11046.3', 'PF11047.3', 'PF11049.3', 'PF11050.3',
     'PF11051.3', 'PF11052.3', 'PF11053.3', 'PF11054.3', 'PF11055.3', 'PF11056.3', 'PF11057.3', 'PF11058.3',
     'PF11059.3', 'PF11060.3', 'PF11061.3', 'PF11062.3', 'PF11064.3', 'PF11065.3', 'PF11066.3', 'PF11067.3',
     'PF11068.3', 'PF11069.3', 'PF11070.3', 'PF11071.3', 'PF11072.3', 'PF11073.3', 'PF11074.3', 'PF11075.3',
     'PF11076.3', 'PF11077.3', 'PF11078.3', 'PF11079.3', 'PF11080.3', 'PF11081.3', 'PF11082.3', 'PF11083.3',
     'PF11084.3', 'PF11085.3', 'PF11086.3', 'PF11087.3', 'PF11088.3', 'PF11089.3', 'PF11090.3', 'PF11091.3',
     'PF11092.3', 'PF11093.3', 'PF11094.3', 'PF11095.3', 'PF11097.3', 'PF11098.3', 'PF11099.3', 'PF11100.3',
     'PF11101.3', 'PF11102.3', 'PF11103.3', 'PF11104.3', 'PF11105.3', 'PF11106.3', 'PF11107.3', 'PF11108.3',
     'PF11109.3', 'PF11110.3', 'PF11111.3', 'PF11112.3', 'PF11113.3', 'PF11114.3', 'PF11115.3', 'PF11116.3',
     'PF11117.3', 'PF11118.3', 'PF11119.3', 'PF11120.3', 'PF11121.3', 'PF11122.3', 'PF11123.3', 'PF11124.3',
     'PF11125.3', 'PF11126.3', 'PF11127.3', 'PF11128.3', 'PF11129.3', 'PF11130.3', 'PF11131.3', 'PF11132.3',
     'PF11133.3', 'PF11134.3', 'PF11135.3', 'PF11136.3', 'PF11137.3', 'PF11138.3', 'PF11139.3', 'PF11140.3',
     'PF11141.3', 'PF11142.3', 'PF11143.3', 'PF11144.3', 'PF11145.3', 'PF11146.3', 'PF11148.3', 'PF11149.3',
     'PF11150.3', 'PF11151.3', 'PF11152.3', 'PF11153.3', 'PF11154.3', 'PF11155.3', 'PF11157.3', 'PF11158.3',
     'PF11159.3', 'PF11160.3', 'PF11161.3', 'PF11162.3', 'PF11163.3', 'PF11164.3', 'PF11165.3', 'PF11166.3',
     'PF11167.3', 'PF11168.3', 'PF11169.3', 'PF11170.3', 'PF11171.3', 'PF11172.3', 'PF11173.3', 'PF11174.3',
     'PF11175.3', 'PF11176.3', 'PF11177.3', 'PF11178.3', 'PF11179.3', 'PF11180.3', 'PF11181.3', 'PF11182.3',
     'PF11183.3', 'PF11184.3', 'PF11185.3', 'PF11186.3', 'PF11187.3', 'PF11188.3', 'PF11189.3', 'PF11190.3',
     'PF11191.3', 'PF11192.3', 'PF11193.3', 'PF11195.3', 'PF11196.3', 'PF11197.3', 'PF11198.3', 'PF11199.3',
     'PF11200.3', 'PF11201.3', 'PF11202.3', 'PF11203.3', 'PF11204.3', 'PF11205.3', 'PF11207.3', 'PF11208.3',
     'PF11209.3', 'PF11210.3', 'PF11211.3', 'PF11212.3', 'PF11213.3', 'PF11214.3', 'PF11215.3', 'PF11216.3',
     'PF11217.3', 'PF11218.3', 'PF11219.3', 'PF11220.3', 'PF11221.3', 'PF11222.3', 'PF11223.3', 'PF11224.3',
     'PF11225.3', 'PF11226.3', 'PF11227.3', 'PF11228.3', 'PF11229.3', 'PF11230.3', 'PF11231.3', 'PF11232.3',
     'PF11233.3', 'PF11235.3', 'PF11236.3', 'PF11237.3', 'PF11238.3', 'PF11239.3', 'PF11240.3', 'PF11241.3',
     'PF11242.3', 'PF11243.3', 'PF11244.3', 'PF11245.3', 'PF11246.3', 'PF11247.3', 'PF11248.3', 'PF11249.3',
     'PF11250.3', 'PF11251.3', 'PF11252.3', 'PF11253.3', 'PF11254.3', 'PF11255.3', 'PF11256.3', 'PF11258.3',
     'PF11259.3', 'PF11260.3', 'PF11261.3', 'PF11262.3', 'PF11263.3', 'PF11264.3', 'PF11265.3', 'PF11266.3',
     'PF11267.3', 'PF11268.3', 'PF11269.3', 'PF11270.3', 'PF11271.3', 'PF11272.3', 'PF11273.3', 'PF11274.3',
     'PF11275.3', 'PF11276.3', 'PF11277.3', 'PF11278.3', 'PF11279.3', 'PF11280.3', 'PF11281.3', 'PF11282.3',
     'PF11283.3', 'PF11284.3', 'PF11285.3', 'PF11286.3', 'PF11287.3', 'PF11288.3', 'PF11289.3', 'PF11290.3',
     'PF11291.3', 'PF11292.3', 'PF11293.3', 'PF11294.3', 'PF11295.3', 'PF11296.3', 'PF11297.3', 'PF11298.3',
     'PF11299.3', 'PF11300.3', 'PF11301.3', 'PF11302.3', 'PF11303.3', 'PF11304.3', 'PF11305.3', 'PF11306.3',
     'PF11307.3', 'PF11308.3', 'PF11309.3', 'PF11310.3', 'PF11311.3', 'PF11312.3', 'PF11313.3', 'PF11314.3',
     'PF11315.3', 'PF11316.3', 'PF11317.3', 'PF11318.3', 'PF11319.3', 'PF11320.3', 'PF11321.3', 'PF11322.3',
     'PF11323.3', 'PF11324.3', 'PF11325.3', 'PF11326.3', 'PF11327.3', 'PF11328.3', 'PF11329.3', 'PF11330.3',
     'PF11331.3', 'PF11332.3', 'PF11333.3', 'PF11334.3', 'PF11335.3', 'PF11336.3', 'PF11337.3', 'PF11338.3',
     'PF11339.3', 'PF11340.3', 'PF11341.3', 'PF11342.3', 'PF11343.3', 'PF11344.3', 'PF11345.3', 'PF11346.3',
     'PF11347.3', 'PF11348.3', 'PF11349.3', 'PF11350.3', 'PF11351.3', 'PF11352.3', 'PF11353.3', 'PF11354.3',
     'PF11355.3', 'PF11356.3', 'PF11357.3', 'PF11358.3', 'PF11359.3', 'PF11360.3', 'PF11361.3', 'PF11362.3',
     'PF11363.3', 'PF11364.3', 'PF11365.3', 'PF11367.3', 'PF11368.3', 'PF11369.3', 'PF11371.3', 'PF11372.3',
     'PF11373.3', 'PF11374.3', 'PF11375.3', 'PF11376.3', 'PF11377.3', 'PF11378.3', 'PF11379.3', 'PF11380.3',
     'PF11381.3', 'PF11382.3', 'PF11383.3', 'PF11384.3', 'PF11385.3', 'PF11386.3', 'PF11387.3', 'PF11388.3',
     'PF11389.3', 'PF11390.3', 'PF11391.3', 'PF11392.3', 'PF11393.3', 'PF11394.3', 'PF11395.3', 'PF11396.3',
     'PF11397.3', 'PF11398.3', 'PF11399.3', 'PF11401.3', 'PF11402.3', 'PF11403.3', 'PF11404.3', 'PF11405.3',
     'PF11406.3', 'PF11407.3', 'PF11408.3', 'PF11409.3', 'PF11410.3', 'PF11411.3', 'PF11412.3', 'PF11413.3',
     'PF11414.3', 'PF11415.3', 'PF11416.3', 'PF11417.3', 'PF11418.3', 'PF11419.3', 'PF11420.3', 'PF11421.3',
     'PF11422.3', 'PF11423.3', 'PF11424.3', 'PF11426.3', 'PF11427.3', 'PF11428.3', 'PF11429.3', 'PF11430.3',
     'PF11431.3', 'PF11432.3', 'PF11433.3', 'PF11434.3', 'PF11435.3', 'PF11436.3', 'PF11437.3', 'PF11438.3',
     'PF11439.3', 'PF11440.3', 'PF11441.3', 'PF11442.3', 'PF11443.3', 'PF11444.3', 'PF11445.3', 'PF11446.3',
     'PF11447.3', 'PF11448.3', 'PF11449.3', 'PF11450.3', 'PF11452.3', 'PF11453.3', 'PF11454.3', 'PF11455.3',
     'PF11456.3', 'PF11457.3', 'PF11458.3', 'PF11459.3', 'PF11460.3', 'PF11461.3', 'PF11462.3', 'PF11463.3',
     'PF11464.3', 'PF11465.3', 'PF11466.3', 'PF11467.3', 'PF11468.3', 'PF11469.3', 'PF11470.3', 'PF11471.3',
     'PF11472.3', 'PF11473.3', 'PF11474.3', 'PF11475.3', 'PF11476.3', 'PF11477.3', 'PF11478.3', 'PF11479.3',
     'PF11480.3', 'PF11482.3', 'PF11483.3', 'PF11485.3', 'PF11486.3', 'PF11487.3', 'PF11488.3', 'PF11489.3',
     'PF11490.3', 'PF11491.3', 'PF11492.3', 'PF11493.3', 'PF11494.3', 'PF11495.3', 'PF11496.3', 'PF11497.3',
     'PF11498.3', 'PF11500.3', 'PF11501.3', 'PF11502.3', 'PF11503.3', 'PF11504.3', 'PF11505.3', 'PF11506.3',
     'PF11507.3', 'PF11508.3', 'PF11510.3', 'PF11511.3', 'PF11512.3', 'PF11513.3', 'PF11514.3', 'PF11515.3',
     'PF11516.3', 'PF11517.3', 'PF11518.3', 'PF11519.3', 'PF11520.3', 'PF11521.3', 'PF11522.3', 'PF11523.3',
     'PF11524.3', 'PF11525.3', 'PF11526.3', 'PF11527.3', 'PF11528.3', 'PF11529.3', 'PF11530.3', 'PF11531.3',
     'PF11532.3', 'PF11533.3', 'PF11534.3', 'PF11535.3', 'PF11536.3', 'PF11537.3', 'PF11538.3', 'PF11539.3',
     'PF11540.3', 'PF11542.3', 'PF11543.3', 'PF11544.3', 'PF11545.3', 'PF11546.3', 'PF11547.3', 'PF11548.3',
     'PF11549.3', 'PF11550.3', 'PF11551.3', 'PF11553.3', 'PF11554.3', 'PF11555.3', 'PF11556.3', 'PF11557.3',
     'PF11558.3', 'PF11559.3', 'PF11560.3', 'PF11561.3', 'PF11563.3', 'PF11564.3', 'PF11565.3', 'PF11566.3',
     'PF11567.3', 'PF11568.3', 'PF11569.3', 'PF11570.3', 'PF11571.3', 'PF11572.3', 'PF11573.3', 'PF11574.3',
     'PF11575.3', 'PF11576.3', 'PF11577.3', 'PF11578.3', 'PF11579.3', 'PF11580.3', 'PF11581.3', 'PF11582.3',
     'PF11583.3', 'PF11584.3', 'PF11585.3', 'PF11586.3', 'PF11587.3', 'PF11588.3', 'PF11589.3', 'PF11590.3',
     'PF11591.3', 'PF11592.3', 'PF11593.3', 'PF11594.3', 'PF11595.3', 'PF11596.3', 'PF11597.3', 'PF11598.3',
     'PF11599.3', 'PF11600.3', 'PF11601.3', 'PF11602.3', 'PF11603.3', 'PF11604.3', 'PF11605.3', 'PF11606.3',
     'PF11607.3', 'PF11608.3', 'PF11609.3', 'PF11610.3', 'PF11611.3', 'PF11612.3', 'PF11613.3', 'PF11614.3',
     'PF11615.3', 'PF11616.3', 'PF11617.3', 'PF11618.3', 'PF11619.3', 'PF11620.3', 'PF11621.3', 'PF11622.3',
     'PF11623.3', 'PF11624.3', 'PF11625.3', 'PF11626.3', 'PF11627.3', 'PF11628.3', 'PF11629.3', 'PF11630.3',
     'PF11631.3', 'PF11632.3', 'PF11633.3', 'PF11634.3', 'PF11635.3', 'PF11636.3', 'PF11637.3', 'PF11638.3',
     'PF11639.3', 'PF11640.3', 'PF11641.3', 'PF11642.3', 'PF11644.3', 'PF11645.3', 'PF11646.3', 'PF11647.3',
     'PF11648.3', 'PF11649.3', 'PF11650.3', 'PF11651.3', 'PF11652.3', 'PF11653.3', 'PF11654.3', 'PF11655.3',
     'PF11656.3', 'PF11657.3', 'PF11658.3', 'PF11659.3', 'PF11660.3', 'PF11661.3', 'PF11662.3', 'PF11663.3',
     'PF11665.3', 'PF11666.3', 'PF11667.3', 'PF11668.3', 'PF11669.3', 'PF11670.3', 'PF11671.3', 'PF11672.3',
     'PF11673.3', 'PF11674.3', 'PF11675.3', 'PF11676.3', 'PF11677.3', 'PF11678.3', 'PF11679.3', 'PF11680.3',
     'PF11681.3', 'PF11682.3', 'PF11683.3', 'PF11684.3', 'PF11685.3', 'PF11686.3', 'PF11687.3', 'PF11688.3',
     'PF11690.3', 'PF11691.3', 'PF11692.3', 'PF11693.3', 'PF11694.3', 'PF11695.3', 'PF11696.3', 'PF11697.3',
     'PF11698.3', 'PF11699.3', 'PF11700.3', 'PF11701.3', 'PF11702.3', 'PF11703.3', 'PF11704.3', 'PF11705.3',
     'PF11706.3', 'PF11707.3', 'PF11708.3', 'PF11709.3', 'PF11710.3', 'PF11711.3', 'PF11712.3', 'PF11713.3',
     'PF11714.3', 'PF11715.3', 'PF11716.3', 'PF11717.3', 'PF11718.3', 'PF11719.3', 'PF11720.3', 'PF11721.3',
     'PF11722.3', 'PF11723.3', 'PF11724.3', 'PF11725.3', 'PF11726.3', 'PF11727.3', 'PF11728.3', 'PF11729.3',
     'PF11730.3', 'PF11731.3', 'PF11732.3', 'PF11733.3', 'PF11734.3', 'PF11735.3', 'PF11736.3', 'PF11737.3',
     'PF11738.3', 'PF11739.3', 'PF11740.3', 'PF11741.3', 'PF11742.3', 'PF11743.3', 'PF11744.3', 'PF11745.3',
     'PF11746.3', 'PF11747.3', 'PF11748.3', 'PF11749.3', 'PF11750.3', 'PF11751.3', 'PF11752.3', 'PF11753.3',
     'PF11754.3', 'PF11755.3', 'PF11756.3', 'PF11757.3', 'PF11758.3', 'PF11759.3', 'PF11760.3', 'PF11761.3',
     'PF11762.3', 'PF11763.3', 'PF11764.3', 'PF11765.3', 'PF11766.3', 'PF11767.3', 'PF11768.3', 'PF11769.3',
     'PF11770.3', 'PF11771.3', 'PF11772.3', 'PF11773.3', 'PF11774.3', 'PF11775.3', 'PF11776.3', 'PF11777.3',
     'PF11778.3', 'PF11779.3', 'PF11780.3', 'PF11781.3', 'PF11782.3', 'PF11783.3', 'PF11784.3', 'PF11785.3',
     'PF11786.3', 'PF11787.3', 'PF11788.3', 'PF11789.3', 'PF11790.3', 'PF11791.3', 'PF11792.3', 'PF11793.3',
     'PF11794.3', 'PF11795.3', 'PF11796.3', 'PF11797.3', 'PF11798.3', 'PF11799.3', 'PF11800.3', 'PF11801.3',
     'PF11802.3', 'PF11803.3', 'PF11804.3', 'PF11805.3', 'PF11806.3', 'PF11807.3', 'PF11808.3', 'PF11809.3',
     'PF11810.3', 'PF11811.3', 'PF11812.3', 'PF11813.3', 'PF11814.3', 'PF11815.3', 'PF11816.3', 'PF11817.3',
     'PF11818.3', 'PF11819.3', 'PF11820.3', 'PF11821.3', 'PF11822.3', 'PF11823.3', 'PF11824.3', 'PF11825.3',
     'PF11826.3', 'PF11827.3', 'PF11828.3', 'PF11829.3', 'PF11830.3', 'PF11831.3', 'PF11832.3', 'PF11833.3',
     'PF11834.3', 'PF11835.3', 'PF11836.3', 'PF11837.3', 'PF11838.3', 'PF11839.3', 'PF11840.3', 'PF11841.3',
     'PF11842.3', 'PF11843.3', 'PF11844.3', 'PF11845.3', 'PF11846.3', 'PF11847.3', 'PF11848.3', 'PF11849.3',
     'PF11850.3', 'PF11851.3', 'PF11852.3', 'PF11853.3', 'PF11854.3', 'PF11855.3', 'PF11856.3', 'PF11857.3',
     'PF11858.3', 'PF11859.3', 'PF11860.3', 'PF11861.3', 'PF11862.3', 'PF11863.3', 'PF11864.3', 'PF11865.3',
     'PF11866.3', 'PF11867.3', 'PF11868.3', 'PF11869.3', 'PF11870.3', 'PF11871.3', 'PF11872.3', 'PF11873.3',
     'PF11874.3', 'PF11875.3', 'PF11876.3', 'PF11877.3', 'PF11878.3', 'PF11879.3', 'PF11880.3', 'PF11881.3',
     'PF11882.3', 'PF11883.3', 'PF11884.3', 'PF11885.3', 'PF11886.3', 'PF11887.3', 'PF11888.3', 'PF11889.3',
     'PF11890.3', 'PF11891.3', 'PF11892.3', 'PF11893.3', 'PF11894.3', 'PF11895.3', 'PF11896.3', 'PF11897.3',
     'PF11898.3', 'PF11899.3', 'PF11900.3', 'PF11901.3', 'PF11902.3', 'PF11903.3', 'PF11904.3', 'PF11905.3',
     'PF11906.3', 'PF11907.3', 'PF11909.3', 'PF11910.3', 'PF11911.3', 'PF11912.3', 'PF11913.3', 'PF11914.3',
     'PF11915.3', 'PF11916.3', 'PF11917.3', 'PF11918.3', 'PF11919.3', 'PF11920.3', 'PF11921.3', 'PF11922.3',
     'PF11923.3', 'PF11924.3', 'PF11925.3', 'PF11926.3', 'PF11927.3', 'PF11928.3', 'PF11929.3', 'PF11931.3',
     'PF11932.3', 'PF11933.3', 'PF11934.3', 'PF11935.3', 'PF11936.3', 'PF11937.3', 'PF11938.3', 'PF11939.3',
     'PF11940.3', 'PF11941.3', 'PF11942.3', 'PF11943.3', 'PF11944.3', 'PF11945.3', 'PF11946.3', 'PF11947.3',
     'PF11948.3', 'PF11949.3', 'PF11950.3', 'PF11951.3', 'PF11952.3', 'PF11953.3', 'PF11954.3', 'PF11955.3',
     'PF11956.3', 'PF11957.3', 'PF11958.3', 'PF11959.3', 'PF11960.3', 'PF11961.3', 'PF11962.3', 'PF11963.3',
     'PF11964.3', 'PF11965.3', 'PF11966.3', 'PF11967.3', 'PF11968.3', 'PF11969.3', 'PF11970.3', 'PF11971.3',
     'PF11972.3', 'PF11973.3', 'PF11974.3', 'PF11975.3', 'PF11976.3', 'PF11977.3', 'PF11978.3', 'PF11979.3',
     'PF11980.3', 'PF11981.3', 'PF11982.3', 'PF11983.3', 'PF11984.3', 'PF11985.3', 'PF11986.3', 'PF11987.3',
     'PF11988.3', 'PF11989.3', 'PF11990.3', 'PF11991.3', 'PF11992.3', 'PF11993.3', 'PF11994.3', 'PF11995.3',
     'PF11996.3', 'PF11997.3', 'PF11998.3', 'PF11999.3', 'PF12000.3', 'PF12001.3', 'PF12002.3', 'PF12003.3',
     'PF12004.3', 'PF12005.3', 'PF12006.3', 'PF12007.3', 'PF12008.3', 'PF12009.3', 'PF12010.3', 'PF12011.3',
     'PF12012.3', 'PF12013.3', 'PF12014.3', 'PF12015.3', 'PF12016.3', 'PF12017.3', 'PF12018.3', 'PF12019.3',
     'PF12020.3', 'PF12021.3', 'PF12022.3', 'PF12023.3', 'PF12024.3', 'PF12025.3', 'PF12026.3', 'PF12027.3',
     'PF12028.3', 'PF12029.3', 'PF12030.3', 'PF12031.3', 'PF12032.3', 'PF12033.3', 'PF12034.3', 'PF12035.3',
     'PF12036.3', 'PF12037.3', 'PF12038.3', 'PF12039.3', 'PF12040.3', 'PF12041.3', 'PF12042.3', 'PF12043.3',
     'PF12044.3', 'PF12045.3', 'PF12046.3', 'PF12047.3', 'PF12048.3', 'PF12049.3', 'PF12051.3', 'PF12052.3',
     'PF12053.3', 'PF12054.3', 'PF12055.3', 'PF12056.3', 'PF12057.3', 'PF12058.3', 'PF12059.3', 'PF12060.3',
     'PF12061.3', 'PF12062.3', 'PF12063.3', 'PF12064.3', 'PF12065.3', 'PF12066.3', 'PF12067.3', 'PF12068.3',
     'PF12069.3', 'PF12070.3', 'PF12071.3', 'PF12072.3', 'PF12073.3', 'PF12074.3', 'PF12075.3', 'PF12076.3',
     'PF12077.3', 'PF12078.3', 'PF12079.3', 'PF12080.3', 'PF12081.3', 'PF12083.3', 'PF12084.3', 'PF12085.3',
     'PF12086.3', 'PF12087.3', 'PF12088.3', 'PF12089.3', 'PF12090.3', 'PF12091.3', 'PF12092.3', 'PF12093.3',
     'PF12094.3', 'PF12095.3', 'PF12096.3', 'PF12097.3', 'PF12098.3', 'PF12099.3', 'PF12100.3', 'PF12101.3',
     'PF12102.3', 'PF12103.3', 'PF12104.3', 'PF12105.3', 'PF12106.3', 'PF12107.3', 'PF12108.3', 'PF12109.3',
     'PF12110.3', 'PF12111.3', 'PF12112.3', 'PF12113.3', 'PF12114.3', 'PF12115.3', 'PF12116.3', 'PF12117.3',
     'PF12118.3', 'PF12119.3', 'PF12120.3', 'PF12121.3', 'PF12122.3', 'PF12123.3', 'PF12124.3', 'PF12125.3',
     'PF12126.3', 'PF12127.3', 'PF12128.3', 'PF12129.3', 'PF12130.3', 'PF12131.3', 'PF12132.3', 'PF12133.3',
     'PF12134.3', 'PF12135.3', 'PF12136.3', 'PF12137.3', 'PF12138.3', 'PF12139.3', 'PF12140.3', 'PF12141.3',
     'PF12142.3', 'PF12143.3', 'PF12144.3', 'PF12145.3', 'PF12146.3', 'PF12147.3', 'PF12148.3', 'PF12149.3',
     'PF12150.3', 'PF12151.3', 'PF12152.3', 'PF12153.3', 'PF12154.3', 'PF12155.3', 'PF12156.3', 'PF12157.3',
     'PF12158.3', 'PF12159.3', 'PF12160.3', 'PF12161.3', 'PF12162.3', 'PF12163.3', 'PF12164.3', 'PF12165.3',
     'PF12166.3', 'PF12167.3', 'PF12168.3', 'PF12169.3', 'PF12170.3', 'PF12171.3', 'PF12172.3', 'PF12173.3',
     'PF12174.3', 'PF12175.3', 'PF12176.3', 'PF12177.3', 'PF12178.3', 'PF12179.3', 'PF12180.3', 'PF12181.3',
     'PF12182.3', 'PF12183.3', 'PF12185.3', 'PF12186.3', 'PF12187.3', 'PF12188.3', 'PF12189.3', 'PF12190.3',
     'PF12191.3', 'PF12192.3', 'PF12193.3', 'PF12194.3', 'PF12195.3', 'PF12196.3', 'PF12197.3', 'PF12198.3',
     'PF12199.3', 'PF12200.3', 'PF12201.3', 'PF12202.3', 'PF12203.3', 'PF12204.3', 'PF12205.3', 'PF12206.3',
     'PF12207.3', 'PF12208.3', 'PF12209.3', 'PF12210.3', 'PF12211.3', 'PF12212.3', 'PF12213.3', 'PF12214.3',
     'PF12215.3', 'PF12216.3', 'PF12217.3', 'PF12218.3', 'PF12219.3', 'PF12220.3', 'PF12221.3', 'PF12222.3',
     'PF12223.3', 'PF12224.3', 'PF12225.3', 'PF12226.3', 'PF12227.3', 'PF12228.3', 'PF12229.3', 'PF12230.3',
     'PF12231.3', 'PF12232.3', 'PF12233.3', 'PF12234.3', 'PF12235.3', 'PF12236.3', 'PF12237.3', 'PF12238.3',
     'PF12239.3', 'PF12240.3', 'PF12241.3', 'PF12242.3', 'PF12243.3', 'PF12244.3', 'PF12245.3', 'PF12246.3',
     'PF12247.3', 'PF12248.3', 'PF12249.3', 'PF12250.3', 'PF12251.3', 'PF12252.3', 'PF12253.3', 'PF12254.3',
     'PF12255.3', 'PF12256.3', 'PF12257.3', 'PF12258.3', 'PF12259.3', 'PF12260.3', 'PF12261.3', 'PF12262.3',
     'PF12263.3', 'PF12264.3', 'PF12265.3', 'PF12266.3', 'PF12267.3', 'PF12268.3', 'PF12269.3', 'PF12270.3',
     'PF12271.3', 'PF12272.3', 'PF12273.3', 'PF12274.3', 'PF12275.3', 'PF12276.3', 'PF12277.3', 'PF12278.3',
     'PF12279.3', 'PF12280.3', 'PF12281.3', 'PF12282.3', 'PF12283.3', 'PF12284.3', 'PF12285.3', 'PF12286.3',
     'PF12287.3', 'PF12288.3', 'PF12289.3', 'PF12290.3', 'PF12291.3', 'PF12292.3', 'PF12293.3', 'PF12294.3',
     'PF12295.3', 'PF12296.3', 'PF12297.3', 'PF12298.3', 'PF12299.3', 'PF12300.3', 'PF12301.3', 'PF12302.3',
     'PF12304.3', 'PF12305.3', 'PF12306.3', 'PF12307.3', 'PF12308.3', 'PF12309.3', 'PF12310.3', 'PF12311.3',
     'PF12312.3', 'PF12313.3', 'PF12314.3', 'PF12315.3', 'PF12316.3', 'PF12317.3', 'PF12318.3', 'PF12319.3',
     'PF12320.3', 'PF12321.3', 'PF12322.3', 'PF12323.3', 'PF12324.3', 'PF12325.3', 'PF12326.3', 'PF12327.3',
     'PF12328.3', 'PF12329.3', 'PF12330.3', 'PF12331.3', 'PF12333.3', 'PF12334.3', 'PF12335.3', 'PF12336.3',
     'PF12337.3', 'PF12338.3', 'PF12339.3', 'PF12340.3', 'PF12341.3', 'PF12342.3', 'PF12343.3', 'PF12344.3',
     'PF12345.3', 'PF12346.3', 'PF12347.3', 'PF12348.3', 'PF12349.3', 'PF12350.3', 'PF12351.3', 'PF12352.3',
     'PF12353.3', 'PF12354.3', 'PF12355.3', 'PF12356.3', 'PF12357.3', 'PF12358.3', 'PF12359.3', 'PF12360.3',
     'PF12361.3', 'PF12362.3', 'PF12363.3', 'PF12364.3', 'PF12365.3', 'PF12366.3', 'PF12367.3', 'PF12368.3',
     'PF12369.3', 'PF12371.3', 'PF12372.3', 'PF12373.3', 'PF12374.3', 'PF12375.3', 'PF12376.3', 'PF12377.3',
     'PF12378.3', 'PF12379.3', 'PF12380.3', 'PF12381.3', 'PF12382.3', 'PF12383.3', 'PF12384.3', 'PF12385.3',
     'PF12386.3', 'PF12387.3', 'PF12388.3', 'PF12389.3', 'PF12390.3', 'PF12391.3', 'PF12392.3', 'PF12393.3',
     'PF12394.3', 'PF12395.3', 'PF12396.3', 'PF12397.3', 'PF12398.3', 'PF12399.3', 'PF12400.3', 'PF12401.3',
     'PF12402.3', 'PF12403.3', 'PF12404.3', 'PF12406.3', 'PF12407.3', 'PF12408.3', 'PF12409.3', 'PF12410.3',
     'PF12411.3', 'PF12412.3', 'PF12413.3', 'PF12414.3', 'PF12415.3', 'PF12416.3', 'PF12417.3', 'PF12418.3',
     'PF12419.3', 'PF12420.3', 'PF12421.3', 'PF12422.3', 'PF12423.3', 'PF12424.3', 'PF12425.3', 'PF12426.3',
     'PF12427.3', 'PF12428.3', 'PF12429.3', 'PF12430.3', 'PF12431.3', 'PF12432.3', 'PF12433.3', 'PF12434.3',
     'PF12435.3', 'PF12436.3', 'PF12437.3', 'PF12438.3', 'PF12439.3', 'PF12440.3', 'PF12441.3', 'PF12442.3',
     'PF12443.3', 'PF12444.3', 'PF12445.3', 'PF12446.3', 'PF12447.3', 'PF12448.3', 'PF12449.3', 'PF12450.3',
     'PF12451.3', 'PF12452.3', 'PF12453.3', 'PF12454.3', 'PF12455.3', 'PF12456.3', 'PF12457.3', 'PF12458.3',
     'PF12459.3', 'PF12460.3', 'PF12461.3', 'PF12462.3', 'PF12463.3', 'PF12464.3', 'PF12465.3', 'PF12466.3',
     'PF12467.3', 'PF12468.3', 'PF12469.3', 'PF12470.3', 'PF12471.3', 'PF12472.3', 'PF12473.3', 'PF12474.3',
     'PF12475.3', 'PF12476.3', 'PF12477.3', 'PF12478.3', 'PF12479.3', 'PF12480.3', 'PF12481.3', 'PF12482.3',
     'PF12483.3', 'PF12484.3', 'PF12485.3', 'PF12486.3', 'PF12487.3', 'PF12488.3', 'PF12489.3', 'PF12490.3',
     'PF12491.3', 'PF12493.3', 'PF12494.3', 'PF12495.3', 'PF12496.3', 'PF12497.3', 'PF12498.3', 'PF12499.3',
     'PF12500.3', 'PF12501.3', 'PF12502.3', 'PF12503.3', 'PF12505.3', 'PF12506.3', 'PF12507.3', 'PF12508.3',
     'PF12509.3', 'PF12510.3', 'PF12511.3', 'PF12512.3', 'PF12513.3', 'PF12514.3', 'PF12515.3', 'PF12516.3',
     'PF12517.3', 'PF12518.3', 'PF12519.3', 'PF12520.3', 'PF12521.3', 'PF12522.3', 'PF12523.3', 'PF12524.3',
     'PF12525.3', 'PF12526.3', 'PF12527.3', 'PF12528.3', 'PF12529.3', 'PF12530.3', 'PF12531.3', 'PF12532.3',
     'PF12533.3', 'PF12534.3', 'PF12535.3', 'PF12536.3', 'PF12537.3', 'PF12538.3', 'PF12539.3', 'PF12540.3',
     'PF12541.3', 'PF12542.3', 'PF12543.3', 'PF12544.3', 'PF12545.3', 'PF12546.3', 'PF12547.3', 'PF12548.3',
     'PF12549.3', 'PF12550.3', 'PF12551.3', 'PF12552.3', 'PF12553.3', 'PF12554.3', 'PF12555.3', 'PF12556.3',
     'PF12557.3', 'PF12558.3', 'PF12559.3', 'PF12560.3', 'PF12561.3', 'PF12562.3', 'PF12563.3', 'PF12564.3',
     'PF12565.3', 'PF12566.3', 'PF12567.3', 'PF12568.3', 'PF12569.3', 'PF12570.3', 'PF12571.3', 'PF12572.3',
     'PF12573.3', 'PF12574.3', 'PF12575.3', 'PF12576.3', 'PF12577.3', 'PF12578.3', 'PF12579.3', 'PF12580.3',
     'PF12581.3', 'PF12582.3', 'PF12583.3', 'PF12584.3', 'PF12585.3', 'PF12586.3', 'PF12587.3', 'PF12588.3',
     'PF12589.3', 'PF12590.3', 'PF12591.3', 'PF12592.3', 'PF12593.3', 'PF12594.3', 'PF12595.3', 'PF12596.3',
     'PF12597.3', 'PF12598.3', 'PF12599.3', 'PF12600.3', 'PF12601.3', 'PF12602.3', 'PF12603.3', 'PF12604.3',
     'PF12605.3', 'PF12606.3', 'PF12607.3', 'PF12608.3', 'PF12609.3', 'PF12610.3', 'PF12611.3', 'PF12612.3',
     'PF12613.3', 'PF12614.3', 'PF12615.3', 'PF12616.3', 'PF12617.3', 'PF12618.3', 'PF12619.3', 'PF12620.3',
     'PF12621.3', 'PF12622.2', 'PF12623.2', 'PF12624.2', 'PF12625.2', 'PF12626.2', 'PF12627.2', 'PF12628.2',
     'PF12629.2', 'PF12630.2', 'PF12631.2', 'PF12632.2', 'PF12633.2', 'PF12634.2', 'PF12635.2', 'PF12636.2',
     'PF12637.2', 'PF12638.2', 'PF12639.2', 'PF12640.2', 'PF12641.2', 'PF12642.2', 'PF12643.2', 'PF12644.2',
     'PF12645.2', 'PF12646.2', 'PF12647.2', 'PF12648.2', 'PF12650.2', 'PF12651.2', 'PF12652.2', 'PF12653.2',
     'PF12654.2', 'PF12655.2', 'PF12656.2', 'PF12657.2', 'PF12658.2', 'PF12659.2', 'PF12660.2', 'PF12661.2',
     'PF12662.2', 'PF12663.2', 'PF12664.2', 'PF12666.2', 'PF12667.2', 'PF12668.2', 'PF12669.2', 'PF12670.2',
     'PF12671.2', 'PF12672.2', 'PF12673.2', 'PF12674.2', 'PF12675.2', 'PF12676.2', 'PF12677.2', 'PF12678.2',
     'PF12679.2', 'PF12680.2', 'PF12681.2', 'PF12682.2', 'PF12683.2', 'PF12684.2', 'PF12685.2', 'PF12686.2',
     'PF12687.2', 'PF12688.2', 'PF12689.2', 'PF12690.2', 'PF12691.2', 'PF12692.2', 'PF12693.2', 'PF12694.2',
     'PF12695.2', 'PF12696.2', 'PF12697.2', 'PF12698.2', 'PF12699.2', 'PF12700.2', 'PF12701.2', 'PF12702.2',
     'PF12703.2', 'PF12704.2', 'PF12705.2', 'PF12706.2', 'PF12707.2', 'PF12708.2', 'PF12709.2', 'PF12710.2',
     'PF12711.2', 'PF12712.2', 'PF12713.2', 'PF12714.2', 'PF12715.2', 'PF12716.2', 'PF12717.2', 'PF12718.2',
     'PF12719.2', 'PF12720.2', 'PF12721.2', 'PF12722.2', 'PF12723.2', 'PF12724.2', 'PF12725.2', 'PF12726.2',
     'PF12727.2', 'PF12728.2', 'PF12729.2', 'PF12730.2', 'PF12731.2', 'PF12732.2', 'PF12733.2', 'PF12734.2',
     'PF12735.2', 'PF12736.2', 'PF12737.2', 'PF12738.2', 'PF12739.2', 'PF12740.2', 'PF12741.2', 'PF12742.2',
     'PF12743.2', 'PF12744.2', 'PF12745.2', 'PF12746.2', 'PF12747.2', 'PF12749.2', 'PF12750.2', 'PF12751.2',
     'PF12752.2', 'PF12753.2', 'PF12754.2', 'PF12755.2', 'PF12756.2', 'PF12757.2', 'PF12758.2', 'PF12759.2',
     'PF12760.2', 'PF12761.2', 'PF12762.2', 'PF12763.2', 'PF12764.2', 'PF12765.2', 'PF12766.2', 'PF12767.2',
     'PF12768.2', 'PF12769.2', 'PF12770.2', 'PF12771.2', 'PF12772.2', 'PF12773.2', 'PF12774.2', 'PF12775.2',
     'PF12776.2', 'PF12777.2', 'PF12778.2', 'PF12779.2', 'PF12780.2', 'PF12781.2', 'PF12782.2', 'PF12783.2',
     'PF12784.2', 'PF12785.2', 'PF12786.2', 'PF12787.2', 'PF12788.2', 'PF12789.2', 'PF12790.2', 'PF12791.2',
     'PF12792.2', 'PF12793.2', 'PF12794.2', 'PF12795.2', 'PF12796.2', 'PF12797.2', 'PF12798.2', 'PF12799.2',
     'PF12800.2', 'PF12801.2', 'PF12802.2', 'PF12803.2', 'PF12804.2', 'PF12805.2', 'PF12806.2', 'PF12807.2',
     'PF12808.2', 'PF12809.2', 'PF12810.2', 'PF12811.2', 'PF12812.2', 'PF12813.2', 'PF12814.2', 'PF12815.2',
     'PF12816.2', 'PF12818.2', 'PF12819.2', 'PF12820.2', 'PF12821.2', 'PF12822.2', 'PF12823.2', 'PF12824.2',
     'PF12825.2', 'PF12826.2', 'PF12827.2', 'PF12828.2', 'PF12829.2', 'PF12830.2', 'PF12831.2', 'PF12832.2',
     'PF12833.2', 'PF12834.2', 'PF12835.2', 'PF12836.2', 'PF12837.2', 'PF12838.2', 'PF12840.2', 'PF12841.2',
     'PF12842.2', 'PF12843.2', 'PF12844.2', 'PF12845.2', 'PF12846.2', 'PF12847.2', 'PF12848.2', 'PF12849.2',
     'PF12850.2', 'PF12851.2', 'PF12852.2', 'PF12853.2', 'PF12854.2', 'PF12855.2', 'PF12856.2', 'PF12857.2',
     'PF12859.2', 'PF12860.2', 'PF12861.2', 'PF12862.2', 'PF12863.2', 'PF12864.2', 'PF12866.2', 'PF12867.2',
     'PF12868.2', 'PF12869.2', 'PF12870.2', 'PF12871.2', 'PF12872.2', 'PF12873.2', 'PF12874.2', 'PF12875.2',
     'PF12876.2', 'PF12877.2', 'PF12878.2', 'PF12879.2', 'PF12881.2', 'PF12882.2', 'PF12883.2', 'PF12884.2',
     'PF12885.2', 'PF12886.2', 'PF12887.2', 'PF12888.2', 'PF12889.2', 'PF12890.2', 'PF12891.2', 'PF12892.2',
     'PF12893.2', 'PF12894.2', 'PF12895.2', 'PF12896.2', 'PF12897.2', 'PF12898.2', 'PF12899.2', 'PF12900.2',
     'PF12901.2', 'PF12902.2', 'PF12903.2', 'PF12904.2', 'PF12905.2', 'PF12906.2', 'PF12907.2', 'PF12909.2',
     'PF12910.2', 'PF12911.2', 'PF12912.2', 'PF12913.2', 'PF12914.2', 'PF12915.2', 'PF12916.2', 'PF12917.2',
     'PF12918.2', 'PF12919.2', 'PF12920.2', 'PF12921.2', 'PF12922.2', 'PF12923.2', 'PF12924.2', 'PF12925.2',
     'PF12926.2', 'PF12927.2', 'PF12928.2', 'PF12929.2', 'PF12930.2', 'PF12931.2', 'PF12932.2', 'PF12933.2',
     'PF12934.2', 'PF12935.2', 'PF12936.2', 'PF12937.2', 'PF12938.2', 'PF12939.2', 'PF12940.2', 'PF12941.2',
     'PF12942.2', 'PF12943.2', 'PF12944.2', 'PF12945.2', 'PF12946.2', 'PF12947.2', 'PF12948.2', 'PF12949.2',
     'PF12950.2', 'PF12951.2', 'PF12952.2', 'PF12953.2', 'PF12954.2', 'PF12955.2', 'PF12956.2', 'PF12957.2',
     'PF12958.2', 'PF12959.2', 'PF12960.2', 'PF12961.2', 'PF12962.2', 'PF12963.2', 'PF12964.2', 'PF12965.2',
     'PF12966.2', 'PF12967.2', 'PF12968.2', 'PF12969.2', 'PF12970.2', 'PF12971.2', 'PF12972.2', 'PF12973.2',
     'PF12974.2', 'PF12975.2', 'PF12976.2', 'PF12977.2', 'PF12978.2', 'PF12979.2', 'PF12980.2', 'PF12981.2',
     'PF12982.2', 'PF12983.2', 'PF12984.2', 'PF12985.2', 'PF12986.2', 'PF12987.2', 'PF12988.2', 'PF12989.2',
     'PF12990.2', 'PF12991.2', 'PF12992.2', 'PF12993.2', 'PF12994.2', 'PF12995.2', 'PF12996.2', 'PF12997.2',
     'PF12998.2', 'PF12999.2', 'PF13000.2', 'PF13001.2', 'PF13002.2', 'PF13003.2', 'PF13004.2', 'PF13005.2',
     'PF13006.2', 'PF13007.2', 'PF13008.2', 'PF13009.1', 'PF13010.1', 'PF13011.1', 'PF13012.1', 'PF13013.1',
     'PF13014.1', 'PF13015.1', 'PF13016.1', 'PF13017.1', 'PF13018.1', 'PF13019.1', 'PF13020.1', 'PF13021.1',
     'PF13022.1', 'PF13023.1', 'PF13024.1', 'PF13025.1', 'PF13026.1', 'PF13027.1', 'PF13028.1', 'PF13029.1',
     'PF13030.1', 'PF13031.1', 'PF13032.1', 'PF13033.1', 'PF13034.1', 'PF13035.1', 'PF13036.1', 'PF13037.1',
     'PF13038.1', 'PF13039.1', 'PF13040.1', 'PF13041.1', 'PF13042.1', 'PF13043.1', 'PF13044.1', 'PF13045.1',
     'PF13046.1', 'PF13047.1', 'PF13048.1', 'PF13049.1', 'PF13050.1', 'PF13051.1', 'PF13052.1', 'PF13053.1',
     'PF13054.1', 'PF13055.1', 'PF13056.1', 'PF13057.1', 'PF13058.1', 'PF13059.1', 'PF13060.1', 'PF13061.1',
     'PF13062.1', 'PF13063.1', 'PF13064.1', 'PF13065.1', 'PF13066.1', 'PF13067.1', 'PF13068.1', 'PF13069.1',
     'PF13070.1', 'PF13071.1', 'PF13072.1', 'PF13073.1', 'PF13074.1', 'PF13075.1', 'PF13076.1', 'PF13077.1',
     'PF13078.1', 'PF13079.1', 'PF13080.1', 'PF13081.1', 'PF13082.1', 'PF13083.1', 'PF13084.1', 'PF13085.1',
     'PF13086.1', 'PF13087.1', 'PF13088.1', 'PF13089.1', 'PF13090.1', 'PF13091.1', 'PF13092.1', 'PF13093.1',
     'PF13094.1', 'PF13095.1', 'PF13096.1', 'PF13097.1', 'PF13098.1', 'PF13099.1', 'PF13100.1', 'PF13101.1',
     'PF13102.1', 'PF13103.1', 'PF13104.1', 'PF13105.1', 'PF13106.1', 'PF13107.1', 'PF13108.1', 'PF13109.1',
     'PF13110.1', 'PF13111.1', 'PF13112.1', 'PF13113.1', 'PF13114.1', 'PF13115.1', 'PF13116.1', 'PF13117.1',
     'PF13118.1', 'PF13119.1', 'PF13120.1', 'PF13121.1', 'PF13122.1', 'PF13123.1', 'PF13124.1', 'PF13125.1',
     'PF13126.1', 'PF13127.1', 'PF13128.1', 'PF13129.1', 'PF13130.1', 'PF13131.1', 'PF13132.1', 'PF13133.1',
     'PF13134.1', 'PF13135.1', 'PF13136.1', 'PF13137.1', 'PF13138.1', 'PF13139.1', 'PF13140.1', 'PF13141.1',
     'PF13142.1', 'PF13143.1', 'PF13144.1', 'PF13145.1', 'PF13146.1', 'PF13147.1', 'PF13148.1', 'PF13149.1',
     'PF13150.1', 'PF13151.1', 'PF13152.1', 'PF13153.1', 'PF13154.1', 'PF13155.1', 'PF13156.1', 'PF13157.1',
     'PF13158.1', 'PF13159.1', 'PF13160.1', 'PF13161.1', 'PF13162.1', 'PF13163.1', 'PF13164.1', 'PF13165.1',
     'PF13166.1', 'PF13167.1', 'PF13168.1', 'PF13169.1', 'PF13170.1', 'PF13171.1', 'PF13172.1', 'PF13173.1',
     'PF13174.1', 'PF13175.1', 'PF13176.1', 'PF13177.1', 'PF13178.1', 'PF13179.1', 'PF13180.1', 'PF13181.1',
     'PF13182.1', 'PF13183.1', 'PF13184.1', 'PF13185.1', 'PF13186.1', 'PF13187.1', 'PF13188.1', 'PF13189.1',
     'PF13190.1', 'PF13191.1', 'PF13192.1', 'PF13193.1', 'PF13194.1', 'PF13195.1', 'PF13196.1', 'PF13197.1',
     'PF13198.1', 'PF13199.1', 'PF13200.1', 'PF13201.1', 'PF13202.1', 'PF13203.1', 'PF13204.1', 'PF13205.1',
     'PF13206.1', 'PF13207.1', 'PF13208.1', 'PF13209.1', 'PF13210.1', 'PF13211.1', 'PF13212.1', 'PF13213.1',
     'PF13214.1', 'PF13215.1', 'PF13216.1', 'PF13217.1', 'PF13218.1', 'PF13219.1', 'PF13220.1', 'PF13221.1',
     'PF13222.1', 'PF13223.1', 'PF13224.1', 'PF13225.1', 'PF13226.1', 'PF13227.1', 'PF13228.1', 'PF13229.1',
     'PF13230.1', 'PF13231.1', 'PF13232.1', 'PF13233.1', 'PF13234.1', 'PF13236.1', 'PF13237.1', 'PF13238.1',
     'PF13239.1', 'PF13240.1', 'PF13241.1', 'PF13242.1', 'PF13243.1', 'PF13244.1', 'PF13245.1', 'PF13246.1',
     'PF13247.1', 'PF13248.1', 'PF13249.1', 'PF13250.1', 'PF13251.1', 'PF13252.1', 'PF13253.1', 'PF13254.1',
     'PF13255.1', 'PF13256.1', 'PF13257.1', 'PF13258.1', 'PF13259.1', 'PF13260.1', 'PF13261.1', 'PF13262.1',
     'PF13263.1', 'PF13264.1', 'PF13265.1', 'PF13266.1', 'PF13267.1', 'PF13268.1', 'PF13269.1', 'PF13270.1',
     'PF13271.1', 'PF13272.1', 'PF13273.1', 'PF13274.1', 'PF13275.1', 'PF13276.1', 'PF13277.1', 'PF13278.1',
     'PF13279.1', 'PF13280.1', 'PF13281.1', 'PF13282.1', 'PF13283.1', 'PF13284.1', 'PF13285.1', 'PF13286.1',
     'PF13287.1', 'PF13288.1', 'PF13289.1', 'PF13290.1', 'PF13291.1', 'PF13292.1', 'PF13293.1', 'PF13294.1',
     'PF13295.1', 'PF13296.1', 'PF13297.1', 'PF13298.1', 'PF13299.1', 'PF13300.1', 'PF13301.1', 'PF13302.1',
     'PF13303.1', 'PF13304.1', 'PF13305.1', 'PF13306.1', 'PF13307.1', 'PF13308.1', 'PF13309.1', 'PF13310.1',
     'PF13311.1', 'PF13312.1', 'PF13313.1', 'PF13314.1', 'PF13315.1', 'PF13316.1', 'PF13317.1', 'PF13318.1',
     'PF13319.1', 'PF13320.1', 'PF13321.1', 'PF13322.1', 'PF13323.1', 'PF13324.1', 'PF13325.1', 'PF13326.1',
     'PF13327.1', 'PF13328.1', 'PF13329.1', 'PF13330.1', 'PF13331.1', 'PF13332.1', 'PF13333.1', 'PF13334.1',
     'PF13335.1', 'PF13336.1', 'PF13337.1', 'PF13338.1', 'PF13339.1', 'PF13340.1', 'PF13341.1', 'PF13342.1',
     'PF13343.1', 'PF13344.1', 'PF13345.1', 'PF13346.1', 'PF13347.1', 'PF13348.1', 'PF13349.1', 'PF13350.1',
     'PF13351.1', 'PF13352.1', 'PF13353.1', 'PF13354.1', 'PF13355.1', 'PF13356.1', 'PF13358.1', 'PF13359.1',
     'PF13360.1', 'PF13361.1', 'PF13362.1', 'PF13363.1', 'PF13364.1', 'PF13365.1', 'PF13366.1', 'PF13367.1',
     'PF13368.1', 'PF13369.1', 'PF13370.1', 'PF13371.1', 'PF13372.1', 'PF13373.1', 'PF13374.1', 'PF13375.1',
     'PF13376.1', 'PF13377.1', 'PF13378.1', 'PF13379.1', 'PF13380.1', 'PF13382.1', 'PF13383.1', 'PF13384.1',
     'PF13385.1', 'PF13386.1', 'PF13387.1', 'PF13388.1', 'PF13389.1', 'PF13390.1', 'PF13391.1', 'PF13392.1',
     'PF13393.1', 'PF13394.1', 'PF13395.1', 'PF13396.1', 'PF13397.1', 'PF13398.1', 'PF13399.1', 'PF13400.1',
     'PF13401.1', 'PF13402.1', 'PF13403.1', 'PF13404.1', 'PF13405.1', 'PF13406.1', 'PF13407.1', 'PF13408.1',
     'PF13409.1', 'PF13410.1', 'PF13411.1', 'PF13412.1', 'PF13413.1', 'PF13414.1', 'PF13415.1', 'PF13416.1',
     'PF13417.1', 'PF13418.1', 'PF13419.1', 'PF13420.1', 'PF13421.1', 'PF13422.1', 'PF13423.1', 'PF13424.1',
     'PF13425.1', 'PF13426.1', 'PF13427.1', 'PF13428.1', 'PF13429.1', 'PF13430.1', 'PF13431.1', 'PF13432.1',
     'PF13433.1', 'PF13434.1', 'PF13435.1', 'PF13436.1', 'PF13437.1', 'PF13438.1', 'PF13439.1', 'PF13440.1',
     'PF13441.1', 'PF13442.1', 'PF13443.1', 'PF13444.1', 'PF13445.1', 'PF13446.1', 'PF13447.1', 'PF13448.1',
     'PF13449.1', 'PF13450.1', 'PF13451.1', 'PF13452.1', 'PF13453.1', 'PF13454.1', 'PF13455.1', 'PF13456.1',
     'PF13457.1', 'PF13458.1', 'PF13459.1', 'PF13460.1', 'PF13461.1', 'PF13462.1', 'PF13463.1', 'PF13464.1',
     'PF13465.1', 'PF13466.1', 'PF13467.1', 'PF13468.1', 'PF13469.1', 'PF13470.1', 'PF13471.1', 'PF13472.1',
     'PF13473.1', 'PF13474.1', 'PF13475.1', 'PF13476.1', 'PF13477.1', 'PF13478.1', 'PF13479.1', 'PF13480.1',
     'PF13481.1', 'PF13482.1', 'PF13483.1', 'PF13484.1', 'PF13485.1', 'PF13486.1', 'PF13487.1', 'PF13488.1',
     'PF13489.1', 'PF13490.1', 'PF13491.1', 'PF13492.1', 'PF13493.1', 'PF13494.1', 'PF13495.1', 'PF13496.1',
     'PF13497.1', 'PF13498.1', 'PF13499.1', 'PF13500.1', 'PF13501.1', 'PF13502.1', 'PF13503.1', 'PF13504.1',
     'PF13505.1', 'PF13506.1', 'PF13507.1', 'PF13508.1', 'PF13509.1', 'PF13510.1', 'PF13511.1', 'PF13512.1',
     'PF13513.1', 'PF13514.1', 'PF13515.1', 'PF13516.1', 'PF13517.1', 'PF13518.1', 'PF13519.1', 'PF13520.1',
     'PF13521.1', 'PF13522.1', 'PF13523.1', 'PF13524.1', 'PF13525.1', 'PF13526.1', 'PF13527.1', 'PF13528.1',
     'PF13529.1', 'PF13530.1', 'PF13531.1', 'PF13532.1', 'PF13533.1', 'PF13534.1', 'PF13535.1', 'PF13536.1',
     'PF13537.1', 'PF13538.1', 'PF13539.1', 'PF13540.1', 'PF13541.1', 'PF13542.1', 'PF13543.1', 'PF13544.1',
     'PF13545.1', 'PF13546.1', 'PF13547.1', 'PF13548.1', 'PF13549.1', 'PF13550.1', 'PF13551.1', 'PF13552.1',
     'PF13553.1', 'PF13554.1', 'PF13555.1', 'PF13556.1', 'PF13557.1', 'PF13558.1', 'PF13559.1', 'PF13560.1',
     'PF13561.1', 'PF13562.1', 'PF13563.1', 'PF13564.1', 'PF13565.1', 'PF13566.1', 'PF13567.1', 'PF13568.1',
     'PF13569.1', 'PF13570.1', 'PF13571.1', 'PF13572.1', 'PF13573.1', 'PF13574.1', 'PF13575.1', 'PF13576.1',
     'PF13577.1', 'PF13578.1', 'PF13579.1', 'PF13580.1', 'PF13581.1', 'PF13582.1', 'PF13583.1', 'PF13584.1',
     'PF13585.1', 'PF13586.1', 'PF13587.1', 'PF13588.1', 'PF13589.1', 'PF13590.1', 'PF13591.1', 'PF13592.1',
     'PF13593.1', 'PF13594.1', 'PF13595.1', 'PF13596.1', 'PF13597.1', 'PF13598.1', 'PF13599.1', 'PF13600.1',
     'PF13601.1', 'PF13602.1', 'PF13603.1', 'PF13604.1', 'PF13605.1', 'PF13606.1', 'PF13607.1', 'PF13608.1',
     'PF13609.1', 'PF13610.1', 'PF13611.1', 'PF13612.1', 'PF13613.1', 'PF13614.1', 'PF13615.1', 'PF13616.1',
     'PF13617.1', 'PF13618.1', 'PF13619.1', 'PF13620.1', 'PF13621.1', 'PF13622.1', 'PF13623.1', 'PF13624.1',
     'PF13625.1', 'PF13627.1', 'PF13628.1', 'PF13629.1', 'PF13630.1', 'PF13631.1', 'PF13632.1', 'PF13633.1',
     'PF13634.1', 'PF13635.1', 'PF13636.1', 'PF13637.1', 'PF13638.1', 'PF13639.1', 'PF13640.1', 'PF13641.1',
     'PF13642.1', 'PF13643.1', 'PF13644.1', 'PF13645.1', 'PF13646.1', 'PF13647.1', 'PF13648.1', 'PF13649.1',
     'PF13650.1', 'PF13651.1', 'PF13652.1', 'PF13653.1', 'PF13654.1', 'PF13655.1', 'PF13656.1', 'PF13657.1',
     'PF13659.1', 'PF13660.1', 'PF13661.1', 'PF13662.1', 'PF13663.1', 'PF13664.1', 'PF13665.1', 'PF13667.1',
     'PF13668.1', 'PF13669.1', 'PF13670.1', 'PF13671.1', 'PF13672.1', 'PF13673.1', 'PF13675.1', 'PF13676.1',
     'PF13677.1', 'PF13678.1', 'PF13679.1', 'PF13680.1', 'PF13681.1', 'PF13682.1', 'PF13683.1', 'PF13684.1',
     'PF13685.1', 'PF13686.1', 'PF13687.1', 'PF13688.1', 'PF13689.1', 'PF13690.1', 'PF13691.1', 'PF13692.1',
     'PF13693.1', 'PF13694.1', 'PF13695.1', 'PF13696.1', 'PF13698.1', 'PF13699.1', 'PF13700.1', 'PF13701.1',
     'PF13702.1', 'PF13703.1', 'PF13704.1', 'PF13705.1', 'PF13706.1', 'PF13707.1', 'PF13708.1', 'PF13709.1',
     'PF13710.1', 'PF13711.1', 'PF13712.1', 'PF13713.1', 'PF13714.1', 'PF13715.1', 'PF13716.1', 'PF13717.1',
     'PF13718.1', 'PF13719.1', 'PF13720.1', 'PF13721.1', 'PF13722.1', 'PF13723.1', 'PF13724.1', 'PF13725.1',
     'PF13726.1', 'PF13727.1', 'PF13728.1', 'PF13729.1', 'PF13730.1', 'PF13731.1', 'PF13732.1', 'PF13733.1',
     'PF13734.1', 'PF13735.1', 'PF13737.1', 'PF13738.1', 'PF13739.1', 'PF13740.1', 'PF13741.1', 'PF13742.1',
     'PF13743.1', 'PF13744.1', 'PF13745.1', 'PF13746.1', 'PF13747.1', 'PF13748.1', 'PF13749.1', 'PF13750.1',
     'PF13751.1', 'PF13752.1', 'PF13753.1', 'PF13754.1', 'PF13755.1', 'PF13756.1', 'PF13757.1', 'PF13758.1',
     'PF13759.1', 'PF13761.1', 'PF13762.1', 'PF13763.1', 'PF13764.1', 'PF13765.1', 'PF13766.1', 'PF13767.1',
     'PF13768.1', 'PF13769.1', 'PF13770.1', 'PF13771.1', 'PF13772.1', 'PF13773.1', 'PF13774.1', 'PF13775.1',
     'PF13776.1', 'PF13777.1', 'PF13778.1', 'PF13779.1', 'PF13780.1', 'PF13781.1', 'PF13782.1', 'PF13783.1',
     'PF13784.1', 'PF13785.1', 'PF13786.1', 'PF13787.1', 'PF13788.1', 'PF13789.1', 'PF13790.1', 'PF13791.1',
     'PF13792.1', 'PF13793.1', 'PF13794.1', 'PF13795.1', 'PF13796.1', 'PF13797.1', 'PF13798.1', 'PF13799.1',
     'PF13800.1', 'PF13801.1', 'PF13802.1', 'PF13803.1', 'PF13804.1', 'PF13805.1', 'PF13806.1', 'PF13807.1',
     'PF13808.1', 'PF13809.1', 'PF13810.1', 'PF13811.1', 'PF13812.1', 'PF13813.1', 'PF13814.1', 'PF13815.1',
     'PF13816.1', 'PF13817.1', 'PF13820.1', 'PF13821.1', 'PF13822.1', 'PF13823.1', 'PF13824.1', 'PF13825.1',
     'PF13826.1', 'PF13827.1', 'PF13828.1', 'PF13829.1', 'PF13830.1', 'PF13831.1', 'PF13832.1', 'PF13833.1',
     'PF13834.1', 'PF13835.1', 'PF13836.1', 'PF13837.1', 'PF13838.1', 'PF13839.1', 'PF13840.1', 'PF13841.1',
     'PF13842.1', 'PF13843.1', 'PF13844.1', 'PF13845.1', 'PF13846.1', 'PF13847.1', 'PF13848.1', 'PF13850.1',
     'PF13851.1', 'PF13852.1', 'PF13853.1', 'PF13854.1', 'PF13855.1', 'PF13856.1', 'PF13857.1', 'PF13858.1',
     'PF13859.1', 'PF13860.1', 'PF13861.1', 'PF13862.1', 'PF13863.1', 'PF13864.1', 'PF13865.1', 'PF13866.1',
     'PF13867.1', 'PF13868.1', 'PF13869.1', 'PF13870.1', 'PF13871.1', 'PF13872.1', 'PF13873.1', 'PF13874.1',
     'PF13875.1', 'PF13876.1', 'PF13877.1', 'PF13878.1', 'PF13879.1', 'PF13880.1', 'PF13881.1', 'PF13882.1',
     'PF13883.1', 'PF13884.1', 'PF13885.1', 'PF13886.1', 'PF13887.1', 'PF13888.1', 'PF13889.1', 'PF13890.1',
     'PF13891.1', 'PF13892.1', 'PF13893.1', 'PF13894.1', 'PF13895.1', 'PF13896.1', 'PF13897.1', 'PF13898.1',
     'PF13899.1', 'PF13900.1', 'PF13901.1', 'PF13902.1', 'PF13903.1', 'PF13904.1', 'PF13905.1', 'PF13906.1',
     'PF13907.1', 'PF13908.1', 'PF13909.1', 'PF13910.1', 'PF13911.1', 'PF13912.1', 'PF13913.1', 'PF13914.1',
     'PF13915.1', 'PF13916.1', 'PF13917.1', 'PF13918.1', 'PF13919.1', 'PF13920.1', 'PF13921.1', 'PF13922.1',
     'PF13923.1', 'PF13924.1', 'PF13925.1', 'PF13926.1', 'PF13927.1', 'PF13928.1', 'PF13929.1', 'PF13930.1',
     'PF13931.1', 'PF13932.1', 'PF13933.1', 'PF13934.1', 'PF13935.1', 'PF13936.1', 'PF13937.1', 'PF13938.1',
     'PF13939.1', 'PF13940.1', 'PF13941.1', 'PF13942.1', 'PF13943.1', 'PF13944.1', 'PF13945.1', 'PF13946.1',
     'PF13947.1', 'PF13948.1', 'PF13949.1', 'PF13950.1', 'PF13952.1', 'PF13953.1', 'PF13954.1', 'PF13955.1',
     'PF13956.1', 'PF13957.1', 'PF13958.1', 'PF13959.1', 'PF13960.1', 'PF13961.1', 'PF13962.1', 'PF13963.1',
     'PF13964.1', 'PF13965.1', 'PF13966.1', 'PF13967.1', 'PF13968.1', 'PF13969.1', 'PF13970.1', 'PF13971.1',
     'PF13972.1', 'PF13973.1', 'PF13974.1', 'PF13975.1', 'PF13976.1', 'PF13977.1', 'PF13978.1', 'PF13979.1',
     'PF13980.1', 'PF13981.1', 'PF13982.1', 'PF13983.1', 'PF13984.1', 'PF13985.1', 'PF13986.1', 'PF13987.1',
     'PF13988.1', 'PF13989.1', 'PF13990.1', 'PF13991.1', 'PF13992.1', 'PF13993.1', 'PF13994.1', 'PF13995.1',
     'PF13996.1', 'PF13997.1', 'PF13998.1', 'PF13999.1', 'PF14000.1', 'PF14001.1', 'PF14002.1', 'PF14003.1',
     'PF14004.1', 'PF14005.1', 'PF14006.1', 'PF14007.1', 'PF14008.1', 'PF14009.1', 'PF14010.1', 'PF14011.1',
     'PF14012.1', 'PF14013.1', 'PF14014.1', 'PF14015.1', 'PF14016.1', 'PF14017.1', 'PF14018.1', 'PF14019.1',
     'PF14020.1', 'PF14021.1', 'PF14022.1', 'PF14023.1', 'PF14024.1', 'PF14025.1', 'PF14026.1', 'PF14027.1',
     'PF14028.1', 'PF14029.1', 'PF14030.1', 'PF14031.1', 'PF14032.1', 'PF14033.1', 'PF14034.1', 'PF14035.1',
     'PF14036.1', 'PF14037.1', 'PF14038.1', 'PF14039.1', 'PF14040.1', 'PF14041.1', 'PF14042.1', 'PF14043.1',
     'PF14044.1', 'PF14045.1', 'PF14046.1', 'PF14047.1', 'PF14048.1', 'PF14049.1', 'PF14050.1', 'PF14051.1',
     'PF14052.1', 'PF14053.1', 'PF14054.1', 'PF14055.1', 'PF14056.1', 'PF14057.1', 'PF14058.1', 'PF14059.1',
     'PF14060.1', 'PF14061.1', 'PF14062.1', 'PF14063.1', 'PF14064.1', 'PF14065.1', 'PF14066.1', 'PF14067.1',
     'PF14068.1', 'PF14069.1', 'PF14070.1', 'PF14071.1', 'PF14072.1', 'PF14073.1', 'PF14074.1', 'PF14075.1',
     'PF14076.1', 'PF14077.1', 'PF14078.1', 'PF14079.1', 'PF14080.1', 'PF14081.1', 'PF14082.1', 'PF14083.1',
     'PF14084.1', 'PF14085.1', 'PF14086.1', 'PF14087.1', 'PF14088.1', 'PF14089.1', 'PF14090.1', 'PF14091.1',
     'PF14092.1', 'PF14093.1', 'PF14094.1', 'PF14096.1', 'PF14097.1', 'PF14098.1', 'PF14099.1', 'PF14100.1',
     'PF14101.1', 'PF14102.1', 'PF14103.1', 'PF14104.1', 'PF14105.1', 'PF14106.1', 'PF14107.1', 'PF14108.1',
     'PF14109.1', 'PF14110.1', 'PF14111.1', 'PF14112.1', 'PF14113.1', 'PF14114.1', 'PF14115.1', 'PF14116.1',
     'PF14117.1', 'PF14118.1', 'PF14119.1', 'PF14120.1', 'PF14121.1', 'PF14122.1', 'PF14123.1', 'PF14124.1',
     'PF14125.1', 'PF14126.1', 'PF14127.1', 'PF14128.1', 'PF14129.1', 'PF14130.1', 'PF14131.1', 'PF14132.1',
     'PF14133.1', 'PF14134.1', 'PF14135.1', 'PF14136.1', 'PF14137.1', 'PF14138.1', 'PF14139.1', 'PF14140.1',
     'PF14141.1', 'PF14142.1', 'PF14143.1', 'PF14144.1', 'PF14145.1', 'PF14146.1', 'PF14147.1', 'PF14148.1',
     'PF14149.1', 'PF14150.1', 'PF14151.1', 'PF14152.1', 'PF14153.1', 'PF14154.1', 'PF14155.1', 'PF14156.1',
     'PF14157.1', 'PF14158.1', 'PF14159.1', 'PF14160.1', 'PF14161.1', 'PF14162.1', 'PF14163.1', 'PF14164.1',
     'PF14165.1', 'PF14166.1', 'PF14167.1', 'PF14168.1', 'PF14169.1', 'PF14171.1', 'PF14172.1', 'PF14173.1',
     'PF14174.1', 'PF14175.1', 'PF14176.1', 'PF14177.1', 'PF14178.1', 'PF14179.1', 'PF14181.1', 'PF14182.1',
     'PF14183.1', 'PF14184.1', 'PF14185.1', 'PF14186.1', 'PF14187.1', 'PF14188.1', 'PF14189.1', 'PF14190.1',
     'PF14191.1', 'PF14192.1', 'PF14193.1', 'PF14194.1', 'PF14195.1', 'PF14196.1', 'PF14197.1', 'PF14198.1',
     'PF14199.1', 'PF14200.1', 'PF14201.1', 'PF14202.1', 'PF14203.1', 'PF14204.1', 'PF14205.1', 'PF14206.1',
     'PF14207.1', 'PF14208.1', 'PF14209.1', 'PF14210.1', 'PF14213.1', 'PF14214.1', 'PF14215.1', 'PF14216.1',
     'PF14217.1', 'PF14218.1', 'PF14219.1', 'PF14220.1', 'PF14221.1', 'PF14222.1', 'PF14223.1', 'PF14224.1',
     'PF14225.1', 'PF14226.1', 'PF14227.1', 'PF14228.1', 'PF14229.1', 'PF14230.1', 'PF14231.1', 'PF14232.1',
     'PF14233.1', 'PF14234.1', 'PF14235.1', 'PF14236.1', 'PF14237.1', 'PF14238.1', 'PF14239.1', 'PF14240.1',
     'PF14241.1', 'PF14242.1', 'PF14243.1', 'PF14244.1', 'PF14245.1', 'PF14246.1', 'PF14247.1', 'PF14248.1',
     'PF14249.1', 'PF14250.1', 'PF14251.1', 'PF14252.1', 'PF14253.1', 'PF14254.1', 'PF14255.1', 'PF14256.1',
     'PF14257.1', 'PF14258.1', 'PF14259.1', 'PF14260.1', 'PF14261.1', 'PF14262.1', 'PF14263.1', 'PF14264.1',
     'PF14265.1', 'PF14266.1', 'PF14267.1', 'PF14268.1', 'PF14269.1', 'PF14270.1', 'PF14271.1', 'PF14272.1',
     'PF14273.1', 'PF14274.1', 'PF14275.1', 'PF14276.1', 'PF14277.1', 'PF14278.1', 'PF14279.1', 'PF14280.1',
     'PF14281.1', 'PF14282.1', 'PF14283.1', 'PF14284.1', 'PF14285.1', 'PF14286.1', 'PF14287.1', 'PF14288.1',
     'PF14289.1', 'PF14290.1', 'PF14291.1', 'PF14292.1', 'PF14293.1', 'PF14294.1', 'PF14295.1', 'PF14296.1',
     'PF14297.1', 'PF14298.1', 'PF14299.1', 'PF14300.1', 'PF14301.1', 'PF14302.1', 'PF14303.1', 'PF14304.1',
     'PF14305.1', 'PF14306.1', 'PF14307.1', 'PF14308.1', 'PF14309.1', 'PF14310.1', 'PF14311.1', 'PF14312.1',
     'PF14313.1', 'PF14314.1', 'PF14315.1', 'PF14316.1', 'PF14317.1', 'PF14318.1', 'PF14319.1', 'PF14320.1',
     'PF14321.1', 'PF14322.1', 'PF14323.1', 'PF14324.1', 'PF14325.1', 'PF14326.1', 'PF14327.1', 'PF14328.1',
     'PF14329.1', 'PF14330.1', 'PF14331.1', 'PF14332.1', 'PF14333.1', 'PF14334.1', 'PF14335.1', 'PF14336.1',
     'PF14337.1', 'PF14338.1', 'PF14339.1', 'PF14340.1', 'PF14341.1', 'PF14342.1', 'PF14343.1', 'PF14344.1',
     'PF14345.1', 'PF14346.1', 'PF14347.1', 'PF14348.1', 'PF14349.1', 'PF14350.1', 'PF14351.1', 'PF14352.1',
     'PF14353.1', 'PF14354.1', 'PF14355.1', 'PF14356.1', 'PF14357.1', 'PF14358.1', 'PF14359.1', 'PF14360.1',
     'PF14361.1', 'PF14362.1', 'PF14363.1', 'PF14364.1', 'PF14365.1', 'PF14366.1', 'PF14367.1', 'PF14368.1',
     'PF14369.1', 'PF14370.1', 'PF14371.1', 'PF14372.1', 'PF14373.1', 'PF14374.1', 'PF14375.1', 'PF14376.1',
     'PF14377.1', 'PF14378.1', 'PF14379.1', 'PF14380.1', 'PF14381.1', 'PF14382.1', 'PF14383.1', 'PF14384.1',
     'PF14385.1', 'PF14386.1', 'PF14387.1', 'PF14388.1', 'PF14389.1', 'PF14390.1', 'PF14391.1', 'PF14392.1',
     'PF14393.1', 'PF14394.1', 'PF14395.1', 'PF14396.1', 'PF14397.1', 'PF14398.1', 'PF14399.1', 'PF14400.1',
     'PF14401.1', 'PF14402.1', 'PF14403.1', 'PF14404.1', 'PF14406.1', 'PF14407.1', 'PF14408.1', 'PF14409.1',
     'PF14410.1', 'PF14411.1', 'PF14412.1', 'PF14413.1', 'PF14414.1', 'PF14415.1', 'PF14416.1', 'PF14417.1',
     'PF14418.1', 'PF14419.1', 'PF14420.1', 'PF14421.1', 'PF14423.1', 'PF14424.1', 'PF14425.1', 'PF14426.1',
     'PF14427.1', 'PF14428.1', 'PF14429.1', 'PF14430.1', 'PF14431.1', 'PF14432.1', 'PF14433.1', 'PF14434.1',
     'PF14435.1', 'PF14436.1', 'PF14437.1', 'PF14438.1', 'PF14439.1', 'PF14440.1', 'PF14441.1', 'PF14442.1',
     'PF14443.1', 'PF14444.1', 'PF14445.1', 'PF14446.1', 'PF14447.1', 'PF14448.1', 'PF14449.1', 'PF14450.1',
     'PF14451.1', 'PF14452.1', 'PF14453.1', 'PF14454.1', 'PF14455.1', 'PF14456.1', 'PF14457.1', 'PF14459.1',
     'PF14460.1', 'PF14461.1', 'PF14462.1', 'PF14463.1', 'PF14464.1', 'PF14465.1', 'PF14466.1', 'PF14467.1',
     'PF14468.1', 'PF14469.1', 'PF14470.1', 'PF14471.1', 'PF14472.1', 'PF14473.1', 'PF14474.1', 'PF14475.1',
     'PF14476.1', 'PF14477.1', 'PF14478.1', 'PF14479.1', 'PF14480.1', 'PF14481.1', 'PF14482.1', 'PF14483.1',
     'PF14484.1', 'PF14485.1', 'PF14486.1', 'PF14487.1', 'PF14488.1', 'PF14489.1', 'PF14490.1', 'PF14491.1',
     'PF14492.1', 'PF14493.1', 'PF14494.1', 'PF14495.1', 'PF14496.1', 'PF14497.1', 'PF14498.1', 'PF14499.1',
     'PF14500.1', 'PF14501.1', 'PF14502.1', 'PF14503.1', 'PF14504.1', 'PF14505.1', 'PF14506.1', 'PF14507.1',
     'PF14508.1', 'PF14509.1', 'PF14510.1', 'PF14511.1', 'PF14512.1', 'PF14513.1', 'PF14514.1', 'PF14515.1',
     'PF14516.1', 'PF14517.1', 'PF14518.1', 'PF14519.1', 'PF14520.1', 'PF14521.1', 'PF14522.1', 'PF14523.1',
     'PF14524.1', 'PF14525.1', 'PF14526.1', 'PF14527.1', 'PF14528.1', 'PF14529.1', 'PF14530.1', 'PF14531.1',
     'PF14532.1', 'PF14533.1', 'PF14534.1', 'PF14535.1', 'PF14536.1', 'PF14537.1', 'PF14538.1', 'PF14539.1',
     'PF14540.1', 'PF14541.1', 'PF14542.1', 'PF14543.1', 'PF14544.1', 'PF14545.1', 'PF14547.1', 'PF14549.1',
     'PF14550.1', 'PF14551.1', 'PF14552.1', 'PF14553.1', 'PF14554.1', 'PF14555.1', 'PF14556.1', 'PF14557.1',
     'PF14558.1', 'PF14559.1', 'PF14560.1', 'PF14561.1', 'PF14562.1', 'PF14563.1', 'PF14564.1', 'PF14565.1',
     'PF14566.1', 'PF14567.1', 'PF14568.1', 'PF14569.1', 'PF14570.1', 'PF14571.1', 'PF14572.1', 'PF14573.1',
     'PF14574.1', 'PF14575.1', 'PF14576.1', 'PF14577.1', 'PF14578.1', 'PF14579.1', 'PF14580.1', 'PF14581.1',
     'PF14582.1', 'PF14583.1', 'PF14584.1', 'PF14585.1', 'PF14586.1', 'PF14587.1', 'PF14588.1', 'PF14589.1',
     'PF14590.1', 'PF14591.1', 'PF14592.1', 'PF14593.1', 'PF14594.1', 'PF14595.1', 'PF14596.1', 'PF14597.1',
     'PF14598.1', 'PF14599.1', 'PF14600.1', 'PF14601.1', 'PF14602.1', 'PF14603.1', 'PF14604.1', 'PF14605.1',
     'PF14606.1', 'PF14607.1', 'PF14608.1', 'PF14609.1', 'PF14610.1', 'PF14611.1', 'PF14612.1', 'PF14613.1',
     'PF14614.1', 'PF14615.1', 'PF14616.1', 'PF14617.1', 'PF14618.1', 'PF14619.1', 'PF14620.1', 'PF14621.1',
     'PF14622.1', 'PF14623.1', 'PF14624.1', 'PF14625.1', 'PF14626.1', 'PF14627.1', 'PF14628.1', 'PF14629.1',
     'PF14630.1', 'PF14631.1', 'PF14632.1', 'PF14633.1', 'PF14634.1', 'PF14635.1', 'PF14636.1', 'PF14637.1',
     'PF14638.1', 'PF14639.1', 'PF14640.1', 'PF14641.1', 'PF14642.1', 'PF14643.1', 'PF14644.1', 'PF14645.1',
     'PF14646.1', 'PF14647.1', 'PF14648.1', 'PF14649.1', 'PF14650.1', 'PF14651.1', 'PF14652.1', 'PF14653.1',
     'PF14654.1', 'PF14655.1', 'PF14656.1', 'PF14657.1', 'PF14658.1', 'PF14659.1', 'PF14660.1', 'PF14661.1',
     'PF14662.1', 'PF14663.1', 'PF14664.1', 'PF14665.1', 'PF14666.1', 'PF14667.1', 'PF14668.1', 'PF14669.1',
     'PF14670.1', 'PF14671.1', 'PF14672.1', 'PF14673.1', 'PF14674.1', 'PF14675.1', 'PF14676.1', 'PF14677.1',
     'PF14678.1', 'PF14679.1', 'PF14680.1', 'PF14681.1', 'PF14682.1', 'PF14683.1', 'PF14684.1', 'PF14685.1',
     'PF14686.1', 'PF14687.1', 'PF14688.1', 'PF14689.1', 'PF14690.1', 'PF14691.1', 'PF14692.1', 'PF14693.1',
     'PF14694.1', 'PF14695.1', 'PF14696.1', 'PF14697.1', 'PF14698.1', 'PF14699.1', 'PF14700.1', 'PF14701.1',
     'PF14702.1', 'PF14703.1', 'PF14704.1', 'PF14705.1', 'PF14706.1', 'PF14707.1', 'PF14709.1', 'PF14710.1',
     'PF14711.1', 'PF14712.1', 'PF14713.1', 'PF14714.1', 'PF14715.1', 'PF14716.1', 'PF14717.1', 'PF14718.1',
     'PF14719.1', 'PF14720.1', 'PF14721.1', 'PF14722.1', 'PF14723.1', 'PF14724.1', 'PF14725.1', 'PF14726.1',
     'PF14727.1', 'PF14728.1', 'PF14729.1', 'PF14730.1', 'PF14731.1', 'PF14732.1', 'PF14733.1', 'PF14734.1',
     'PF14735.1', 'PF14736.1', 'PF14737.1', 'PF14738.1', 'PF14739.1', 'PF14740.1', 'PF14741.1', 'PF14742.1',
     'PF14743.1', 'PF14744.1', 'PF14745.1', 'PF14746.1', 'PF14747.1', 'PF14748.1', 'PF14749.1', 'PF14750.1',
     'PF14751.1', 'PF14752.1', 'PF14753.1', 'PF14754.1', 'PF14755.1', 'PF14756.1', 'PF14757.1', 'PF14758.1',
     'PF14759.1', 'PF14760.1', 'PF14761.1', 'PF14762.1', 'PF14763.1', 'PF14764.1', 'PF14765.1', 'PF14766.1',
     'PF14767.1', 'PF14768.1', 'PF14769.1', 'PF14770.1', 'PF14771.1', 'PF14772.1', 'PF14773.1', 'PF14774.1',
     'PF14775.1', 'PF14776.1', 'PF14777.1', 'PF14778.1', 'PF14779.1', 'PF14780.1', 'PF14781.1', 'PF14782.1',
     'PF14783.1', 'PF14784.1', 'PF14785.1', 'PF14786.1', 'PF14787.1', 'PF14788.1', 'PF14789.1', 'PF14790.1',
     'PF14791.1', 'PF14792.1', 'PF14793.1', 'PF14794.1', 'PF14795.1', 'PF14796.1', 'PF14797.1', 'PF14798.1',
     'PF14799.1', 'PF14800.1', 'PF14801.1', 'PF14802.1', 'PF14803.1', 'PF14804.1', 'PF14805.1', 'PF14806.1',
     'PF14807.1', 'PF14808.1', 'PF14809.1', 'PF14810.1', 'PF14811.1', 'PF14812.1', 'PF14813.1', 'PF14814.1',
     'PF14815.1', 'PF14816.1', 'PF14817.1', 'PF14818.1', 'PF14819.1', 'PF14820.1', 'PF14821.1', 'PF14822.1',
     'PF14823.1', 'PF14824.1', 'PF14825.1', 'PF14826.1', 'PF14827.1', 'PF14828.1', 'PF14829.1', 'PF14830.1',
     'PF14831.1', 'PF14832.1', 'PF14833.1', 'PF14834.1', 'PF14835.1', 'PF14836.1', 'PF14837.1', 'PF14838.1',
     'PF14839.1', 'PF14840.1', 'PF14841.1', 'PF14842.1', 'PF14843.1', 'PF14844.1', 'PF14845.1', 'PF14846.1',
     'PF14847.1', 'PF14848.1', 'PF14849.1', 'PF14850.1', 'PF14851.1', 'PF14852.1', 'PF14853.1', 'PF14854.1',
     'PF14855.1', 'PF14856.1', 'PF14857.1', 'PF14858.1', 'PF14859.1', 'PF14860.1', 'PF14861.1', 'PF14862.1',
     'PF14863.1', 'PF14864.1', 'PF14865.1', 'PF14866.1', 'PF14867.1', 'PF14868.1', 'PF14869.1', 'PF14870.1',
     'PF14871.1', 'PF14872.1', 'PF14873.1', 'PF14874.1', 'PF14875.1', 'PF14876.1', 'PF14877.1', 'PF14878.1',
     'PF14879.1', 'PF14880.1', 'PF14881.1', 'PF14882.1', 'PF14883.1', 'PF14884.1', 'PF14885.1', 'PF14886.1',
     'PF14887.1', 'PF14888.1', 'PF14889.1', 'PF14890.1', 'PF14891.1', 'PF14892.1', 'PF14893.1', 'PF14894.1',
     'PF14895.1', 'PF14896.1', 'PF14897.1', 'PF14898.1', 'PF14899.1', 'PF14900.1', 'PF14901.1', 'PF14902.1',
     'PF14903.1', 'PF14904.1', 'PF14905.1', 'PF14906.1', 'PF14907.1', 'PF14908.1', 'PF14909.1', 'PF14910.1',
     'PF14911.1', 'PF14912.1', 'PF14913.1', 'PF14914.1', 'PF14915.1', 'PF14916.1', 'PF14917.1', 'PF14918.1',
     'PF14919.1', 'PF14920.1', 'PF14921.1', 'PF14922.1', 'PF14923.1', 'PF14924.1', 'PF14925.1', 'PF14926.1',
     'PF14927.1', 'PF14928.1', 'PF14929.1', 'PF14930.1', 'PF14931.1', 'PF14932.1', 'PF14933.1', 'PF14934.1',
     'PF14935.1', 'PF14936.1', 'PF14937.1', 'PF14938.1', 'PF14939.1', 'PF14940.1', 'PF14941.1', 'PF14942.1',
     'PF14943.1', 'PF14944.1', 'PF14945.1', 'PF14946.1', 'PF14947.1', 'PF14948.1', 'PF14949.1', 'PF14950.1',
     'PF14951.1', 'PF14952.1', 'PF14953.1', 'PF14954.1', 'PF14955.1', 'PF14956.1', 'PF14957.1', 'PF14958.1',
     'PF14959.1', 'PF14960.1', 'PF14961.1', 'PF14962.1', 'PF14963.1', 'PF14964.1', 'PF14965.1', 'PF14966.1',
     'PF14967.1', 'PF14968.1', 'PF14969.1', 'PF14970.1', 'PF14971.1', 'PF14972.1', 'PF14973.1', 'PF14974.1',
     'PF14975.1', 'PF14976.1', 'PF14977.1', 'PF14978.1', 'PF14979.1', 'PF14980.1', 'PF14981.1', 'PF14982.1',
     'PF14983.1', 'PF14984.1', 'PF14985.1', 'PF14986.1', 'PF14987.1', 'PF14988.1', 'PF14989.1', 'PF14990.1',
     'PF14991.1', 'PF14992.1', 'PF14993.1', 'PF14994.1', 'PF14995.1', 'PF14996.1', 'PF14997.1', 'PF14998.1',
     'PF14999.1', 'PF15000.1', 'PF15001.1', 'PF15002.1', 'PF15003.1', 'PF15004.1', 'PF15005.1', 'PF15006.1',
     'PF15007.1', 'PF15008.1', 'PF15009.1', 'PF15010.1', 'PF15011.1', 'PF15012.1', 'PF15013.1', 'PF15014.1',
     'PF15015.1', 'PF15016.1', 'PF15017.1', 'PF15018.1', 'PF15019.1', 'PF15020.1', 'PF15021.1', 'PF15022.1',
     'PF15023.1', 'PF15024.1', 'PF15025.1', 'PF15026.1', 'PF15027.1', 'PF15028.1', 'PF15029.1', 'PF15030.1',
     'PF15031.1', 'PF15032.1', 'PF15033.1', 'PF15034.1', 'PF15035.1', 'PF15036.1', 'PF15037.1', 'PF15038.1',
     'PF15039.1', 'PF15040.1', 'PF15041.1', 'PF15042.1', 'PF15043.1', 'PF15044.1', 'PF15045.1', 'PF15046.1',
     'PF15047.1', 'PF15048.1', 'PF15049.1', 'PF15050.1', 'PF15051.1', 'PF15052.1', 'PF15053.1', 'PF15054.1',
     'PF15055.1', 'PF15056.1', 'PF15057.1', 'PF15058.1', 'PF15059.1', 'PF15060.1', 'PF15061.1', 'PF15062.1',
     'PF15063.1', 'PF15064.1', 'PF15065.1', 'PF15066.1', 'PF15067.1', 'PF15068.1', 'PF15069.1', 'PF15070.1',
     'PF15071.1', 'PF15072.1', 'PF15073.1', 'PF15074.1', 'PF15075.1', 'PF15076.1', 'PF15077.1', 'PF15078.1',
     'PF15079.1', 'PF15080.1', 'PF15081.1', 'PF15082.1', 'PF15083.1', 'PF15084.1', 'PF15085.1', 'PF15086.1',
     'PF15087.1', 'PF15088.1', 'PF15089.1', 'PF15090.1', 'PF15091.1', 'PF15092.1', 'PF15093.1', 'PF15094.1',
     'PF15095.1', 'PF15096.1', 'PF15097.1', 'PF15098.1', 'PF15099.1', 'PF15100.1', 'PF15101.1', 'PF15102.1',
     'PF15103.1', 'PF15104.1', 'PF15105.1', 'PF15106.1', 'PF15107.1', 'PF15108.1', 'PF15109.1', 'PF15110.1',
     'PF15111.1', 'PF15112.1', 'PF15113.1', 'PF15114.1', 'PF15115.1', 'PF15116.1', 'PF15117.1', 'PF15118.1',
     'PF15119.1', 'PF15120.1', 'PF15121.1', 'PF15122.1', 'PF15123.1', 'PF15124.1', 'PF15125.1', 'PF15127.1',
     'PF15128.1', 'PF15129.1', 'PF15130.1', 'PF15131.1', 'PF15132.1', 'PF15133.1', 'PF15134.1', 'PF15135.1',
     'PF15136.1', 'PF15137.1', 'PF15138.1', 'PF15139.1', 'PF15140.1', 'PF15141.1', 'PF15142.1', 'PF15143.1',
     'PF15144.1', 'PF15145.1', 'PF15146.1', 'PF15147.1', 'PF15148.1', 'PF15149.1', 'PF15150.1', 'PF15151.1',
     'PF15152.1', 'PF15153.1', 'PF15155.1', 'PF15156.1', 'PF15157.1', 'PF15158.1', 'PF15159.1', 'PF15160.1',
     'PF15161.1', 'PF15162.1', 'PF15163.1', 'PF15164.1', 'PF15165.1', 'PF15167.1', 'PF15168.1', 'PF15169.1',
     'PF15170.1', 'PF15171.1', 'PF15172.1', 'PF15173.1', 'PF15174.1', 'PF15175.1', 'PF15176.1', 'PF15177.1',
     'PF15178.1', 'PF15179.1', 'PF15180.1', 'PF15181.1', 'PF15182.1', 'PF15183.1', 'PF15184.1', 'PF15185.1',
     'PF15186.1', 'PF15187.1', 'PF15188.1', 'PF15189.1', 'PF15190.1', 'PF15191.1', 'PF15192.1', 'PF15193.1',
     'PF15194.1', 'PF15195.1', 'PF15196.1', 'PF15197.1', 'PF15198.1', 'PF15199.1', 'PF15200.1', 'PF15201.1',
     'PF15202.1', 'PF15203.1', 'PF15204.1', 'PF15205.1', 'PF15206.1', 'PF15207.1', 'PF15208.1', 'PF15209.1',
     'PF15210.1', 'PF15211.1', 'PF15212.1', 'PF15213.1', 'PF15214.1', 'PF15215.1', 'PF15216.1', 'PF15217.1',
     'PF15218.1', 'PF15219.1', 'PF15220.1', 'PF15221.1', 'PF15222.1', 'PF15223.1', 'PF15224.1', 'PF15225.1',
     'PF15226.1', 'PF15227.1', 'PF15228.1', 'PF15229.1', 'PF15230.1', 'PF15231.1', 'PF15232.1', 'PF15233.1',
     'PF15234.1', 'PF15235.1', 'PF15236.1', 'PF15237.1', 'PF15238.1', 'PF15239.1', 'PF15240.1', 'PF15241.1',
     'PF15242.1', 'PF15243.1', 'PF15244.1', 'PF15245.1', 'PF15246.1', 'PF15247.1', 'PF15248.1', 'PF15249.1',
     'PF15250.1', 'PF15251.1', 'PF15252.1', 'PF15253.1', 'PF15254.1', 'PF15255.1', 'PF15256.1', 'PF15257.1',
     'PF15258.1', 'PF15259.1', 'PF15260.1', 'PF15261.1', 'PF15262.1', 'PF15263.1', 'PF15264.1', 'PF15265.1',
     'PF15266.1', 'PF15268.1', 'PF15269.1', 'PF15270.1', 'PF15271.1', 'PF15272.1', 'PF15273.1', 'PF15274.1',
     'PF15275.1', 'PF15276.1', 'PF15277.1', 'PF15278.1', 'PF15279.1', 'PF15280.1', 'PF15281.1', 'PF15282.1',
     'PF15283.1', 'PF15284.1', 'PF15285.1', 'PF15286.1', 'PF15287.1', 'PF15288.1', 'PF15289.1', 'PF15290.1',
     'PF15291.1', 'PF15292.1', 'PF15293.1', 'PF15294.1', 'PF15295.1', 'PF15296.1', 'PF15297.1', 'PF15298.1',
     'PF15299.1', 'PF15300.1', 'PF15301.1', 'PF15302.1', 'PF15303.1', 'PF15304.1', 'PF15305.1', 'PF15306.1',
     'PF15307.1', 'PF15308.1', 'PF15309.1', 'PF15310.1', 'PF15311.1', 'PF15312.1', 'PF15313.1', 'PF15314.1',
     'PF15315.1', 'PF15316.1', 'PF15317.1', 'PF15318.1', 'PF15319.1', 'PF15320.1', 'PF15321.1', 'PF15322.1',
     'PF15323.1', 'PF15324.1', 'PF15325.1', 'PF15326.1', 'PF15327.1', 'PF15328.1', 'PF15330.1', 'PF15331.1',
     'PF15332.1', 'PF15333.1', 'PF15334.1', 'PF15335.1', 'PF15336.1', 'PF15337.1', 'PF15338.1', 'PF15339.1',
     'PF15340.1', 'PF15341.1', 'PF15342.1', 'PF15343.1', 'PF15344.1', 'PF15345.1', 'PF15346.1', 'PF15347.1',
     'PF15348.1', 'PF15349.1', 'PF15350.1', 'PF15351.1', 'PF15352.1', 'PF15353.1', 'PF15354.1', 'PF15355.1',
     'PF15356.1', 'PF15357.1', 'PF15358.1', 'PF15359.1', 'PF15360.1', 'PF15361.1', 'PF15362.1', 'PF15363.1',
     'PF15364.1', 'PF15365.1', 'PF15366.1', 'PF15367.1', 'PF15368.1', 'PF15369.1', 'PF15370.1', 'PF15371.1',
     'PF15372.1', 'PF15373.1', 'PF15374.1', 'PF15375.1', 'PF15376.1', 'PF15377.1', 'PF15378.1', 'PF15379.1',
     'PF15380.1', 'PF15381.1', 'PF15382.1', 'PF15383.1', 'PF15384.1', 'PF15385.1', 'PF15386.1', 'PF15387.1',
     'PF15388.1', 'PF15389.1', 'PF15390.1', 'PF15391.1', 'PF15392.1', 'PF15393.1', 'PF15394.1', 'PF15395.1',
     'PF15396.1', 'PF15397.1', 'PF15398.1', 'PF15399.1', 'PF15400.1', 'PF15401.1', 'PF15402.1', 'PF15403.1',
     'PF15404.1', 'PF15405.1', 'PF15406.1', 'PF15407.1', 'PF15408.1', 'PF15409.1', 'PF15410.1', 'PF15411.1',
     'PF15412.1', 'PF15413.1', 'PF15414.1', 'PF15415.1', 'PF15416.1', 'PF15417.1', 'PF15418.1', 'PF15419.1',
     'PF15420.1', 'PF15421.1', 'PF15422.1', 'PF15423.1', 'PF15424.1', 'PF15425.1', 'PF15427.1', 'PF15428.1',
     'PF15429.1', 'PF15430.1', 'PF15431.1', 'PF15432.1', 'PF15433.1', 'PF15434.1', 'PF15435.1', 'PF15436.1',
     'PF15437.1', 'PF15438.1', 'PF15439.1', 'PF15440.1', 'PF15441.1', 'PF15442.1', 'PF15443.1', 'PF15444.1',
     'PF15445.1', 'PF15446.1', 'PF15447.1', 'PF15448.1', 'PF15449.1', 'PF15450.1', 'PF15451.1', 'PF15452.1',
     'PF15453.1', 'PF15454.1', 'PF15455.1', 'PF15456.1', 'PF15457.1', 'PF15458.1', 'PF15459.1', 'PF15460.1',
     'PF15461.1', 'PF15462.1', 'PF15463.1', 'PF15464.1', 'PF15465.1', 'PF15466.1', 'PF15467.1', 'PF15468.1',
     'PF15469.1', 'PF15470.1', 'PF15471.1', 'PF15472.1', 'PF15473.1', 'PF15474.1', 'PF15475.1', 'PF15476.1',
     'PF15477.1', 'PF15478.1', 'PF15479.1', 'PF15480.1', 'PF15481.1', 'PF15482.1', 'PF15483.1', 'PF15484.1',
     'PF15485.1', 'PF15486.1', 'PF15487.1', 'PF15488.1', 'PF15489.1', 'PF15490.1', 'PF15491.1', 'PF15492.1',
     'PF15493.1', 'PF15494.1', 'PF15495.1', 'PF15496.1', 'PF15497.1', 'PF15498.1', 'PF15499.1', 'PF15500.1',
     'PF15501.1', 'PF15502.1', 'PF15503.1', 'PF15504.1', 'PF15505.1', 'PF15506.1', 'PF15507.1', 'PF15508.1',
     'PF15509.1', 'PF15510.1', 'PF15511.1', 'PF15512.1', 'PF15513.1', 'PF15514.1', 'PF15515.1', 'PF15516.1',
     'PF15517.1', 'PF15518.1', 'PF15519.1', 'PF15520.1', 'PF15521.1', 'PF15522.1', 'PF15523.1', 'PF15524.1',
     'PF15525.1', 'PF15526.1', 'PF15527.1', 'PF15528.1', 'PF15529.1', 'PF15530.1', 'PF15531.1', 'PF15532.1',
     'PF15533.1', 'PF15534.1', 'PF15535.1', 'PF15536.1', 'PF15537.1', 'PF15538.1', 'PF15539.1', 'PF15540.1',
     'PF15541.1', 'PF15542.1', 'PF15543.1', 'PF15544.1', 'PF15545.1', 'PF15546.1', 'PF15547.1', 'PF15548.1',
     'PF15549.1', 'PF15550.1', 'PF15551.1', 'PF15552.1', 'PF15553.1', 'PF15554.1', 'PF15555.1', 'PF15556.1',
     'PF15557.1', 'PF15558.1', 'PF15559.1', 'PF15560.1', 'PF15561.1', 'PF15562.1', 'PF15563.1', 'PF15564.1',
     'PF15565.1', 'PF15566.1', 'PF15567.1', 'PF15568.1', 'PF15569.1', 'PF15570.1', 'PF15571.1', 'PF15572.1',
     'PF15573.1', 'PF15574.1', 'PF15575.1', 'PF15576.1', 'PF15577.1', 'PF15578.1', 'PF15579.1', 'PF15580.1',
     'PF15581.1', 'PF15582.1', 'PF15583.1', 'PF15584.1', 'PF15585.1', 'PF15586.1', 'PF15587.1', 'PF15588.1',
     'PF15589.1', 'PF15590.1', 'PF15591.1', 'PF15592.1', 'PF15593.1', 'PF15594.1', 'PF15595.1', 'PF15596.1',
     'PF15597.1', 'PF15598.1', 'PF15599.1', 'PF15600.1', 'PF15601.1', 'PF15602.1', 'PF15603.1', 'PF15604.1',
     'PF15605.1', 'PF15606.1', 'PF15607.1', 'PF15608.1', 'PF15609.1', 'PF15610.1', 'PF15611.1', 'PF15612.1',
     'PF15613.1', 'PF15614.1', 'PF15615.1', 'PF15616.1', 'PF15617.1', 'PF15619.1', 'PF15620.1', 'PF15621.1',
     'PF15622.1', 'PF15623.1', 'PF15624.1', 'PF15625.1', 'PF15626.1', 'PF15627.1', 'PF15628.1', 'PF15629.1',
     'PF15630.1', 'PF15631.1', 'PF15632.1', 'PF15633.1', 'PF15634.1', 'PF15635.1', 'PF15636.1', 'PF15637.1',
     'PF15638.1', 'PF15639.1', 'PF15640.1', 'PF15641.1', 'PF15642.1', 'PF15643.1', 'PF15644.1', 'PF15645.1',
     'PF15646.1', 'PF15647.1', 'PF15648.1', 'PF15649.1', 'PF15650.1', 'PF15651.1', 'PF15652.1', 'PF15653.1',
     'PF15654.1', 'PF15655.1', 'PF15656.1', 'PF15657.1', 'PF15658.1', 'PF15659.1', 'PF15660.1'})
