# Tutorial: obtain a sample from a population as a matrix type.

Version 0.5.4 added fwdpp::data_matrix, which represents a sample from a population as a matrix (a one-dimensional vector plus the number of rows/columns).

There are two types of matrix:

* Haplotype.  This consists of two rows/individual.  Data are encoded as 0 = ancetral, 1 = derived.
* Genotype.  There is one row per individual.  Data are encoded as 0,1, or 2 copies of the derived allele.

These objects are generated by fwdpp::haplotype_matrix and fwdpp::genotype_matrix, respectively, where the input is a list of indexes of individuals and **keys** to the mutations present in the sample.  A list of keys is generated by fwdpp::mutation_keys.  Please see the documentation of that function for details about its return value.  The return value of fwdpp::mutation_keys may be filtered manually to further fine-tune the sample (remove fixed mutations, apply frequency filters, etc.).

The type fwdpp::data_matrix stores the data in a C++ vector of type char, which is space-efficient while still allowing easy conversion to matrix types from other libraries.  The data types are easily used in combination with the GSL matrix types, with no need for additional memory allocation.

The memory layout of the matrix is "C-style", or row-major, to make the objects compatible with GSL types, NumPy array types, etc.

\code{.cpp}
#include <fwdpp/sugar/matrix.hpp>
#include <gsl/gsl_matrix_char.h>
//assume poptype represents some single-deme object
poptype p;
//Evolve the population...(step skipped)

//Generate keys for the mutations found in individuals
//0,1, and 2, for both neutral and selected mutations
keys = fwdpp::mutation_keys(p,{0,1,2},true,true);
hm = fwdpp::haplotype_matrix(pop,{0,1,2},keys.first,keys.second);
//Create a GSL matrix type for the neutral mutations
//neutral_matrix will have the type gsl_matrix_char_const_view
auto neutral_matrix =  gsl_matrix_char_const_view_array(hm.neutral.data(),
                                                        hm.nrow, hm.neutral_positions.size());
\endcode

To get the matrix for selected mutations, replace neutral_ with selected_ in the last line of the above example.

The above code sketch gives you a GSL matrix that you can use in the normal fashion for that library.

Sometimes, you'll need a matrix to be based on a floating-point type.  In such cases, you'll need to make a copy of the data:

\code{.cpp}
//This is the main GSL matrix header.
//GSL functions on matrix/vector types
//assume double is the data.
//See GSL manual for more info.
#include <gsl/gsl_matrix.h>
hm = fwdpp::haplotype_matrix(pop,{0,1,2},keys.first,keys.second);
//Copy the vector of chars to a vector of doubles
std::vector<double> neutral(hm.neutral.begin(),hm.neutral.end());
//neutral_matrix is gsl_matrix_const_view
auto neutral_matrix =  gsl_matrix_const_view_array(hm.neutral.data(),
                                                        hm.nrow, hm.neutral_positions.size());
\endcode

I've made **constant** objects to view the data in GSL matrix form.  You could use the non-const versions, but you would need to be careful.  There are other data types in fwdpp::data_matrix that could become inconsistent in the event that the main data matrix is modified.

For more examples of using these functions, see the source code of the unit test file sugar_matrixTest.cc.
