"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NestedStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const constructs_1 = require("constructs");
const assets_1 = require("./assets");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cloudformation_generated_1 = require("./cloudformation.generated");
const lazy_1 = require("./lazy");
const names_1 = require("./names");
const removal_policy_1 = require("./removal-policy");
const stack_1 = require("./stack");
const stack_synthesizers_1 = require("./stack-synthesizers");
const token_1 = require("./token");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const NESTED_STACK_SYMBOL = Symbol.for('@aws-cdk/core.NestedStack');
/**
 * A CloudFormation nested stack.
 *
 * When you apply template changes to update a top-level stack, CloudFormation
 * updates the top-level stack and initiates an update to its nested stacks.
 * CloudFormation updates the resources of modified nested stacks, but does not
 * update the resources of unmodified nested stacks.
 *
 * Furthermore, this stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 * Cross references of resource attributes between the parent stack and the
 * nested stack will automatically be translated to stack parameters and
 * outputs.
 *
 * @stability stable
 */
class NestedStack extends stack_1.Stack {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b;
        const parentStack = findParentStack(scope);
        super(scope, id, {
            env: { account: parentStack.account, region: parentStack.region },
            synthesizer: new stack_synthesizers_1.NestedStackSynthesizer(parentStack.synthesizer),
        });
        this._parentStack = parentStack;
        // @deprecate: remove this in v2.0 (redundent)
        const parentScope = new construct_compat_1.Construct(scope, id + '.NestedStack');
        Object.defineProperty(this, NESTED_STACK_SYMBOL, { value: true });
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${names_1.Names.uniqueId(this)}.nested.template.json`;
        this.parameters = props.parameters || {};
        this.resource = new cloudformation_generated_1.CfnStack(parentScope, `${id}.NestedStackResource`, {
            // This value cannot be cached since it changes during the synthesis phase
            templateUrl: lazy_1.Lazy.uncachedString({ produce: () => this._templateUrl || '<unresolved>' }),
            parameters: lazy_1.Lazy.any({ produce: () => Object.keys(this.parameters).length > 0 ? this.parameters : undefined }),
            notificationArns: props.notificationArns,
            timeoutInMinutes: props.timeout ? props.timeout.toMinutes() : undefined,
        });
        this.resource.applyRemovalPolicy((_b = props.removalPolicy) !== null && _b !== void 0 ? _b : removal_policy_1.RemovalPolicy.DESTROY);
        this.nestedStackResource = this.resource;
        // context-aware stack name: if resolved from within this stack, return AWS::StackName
        // if resolved from the outer stack, use the { Ref } of the AWS::CloudFormation::Stack resource
        // which resolves the ARN of the stack. We need to extract the stack name, which is the second
        // component after splitting by "/"
        this._contextualStackName = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_NAME, cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split('/', this.resource.ref)));
        this._contextualStackId = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_ID, this.resource.ref);
    }
    /**
     * Checks if `x` is an object of type `NestedStack`.
     *
     * @stability stable
     */
    static isNestedStack(x) {
        return x != null && typeof (x) === 'object' && NESTED_STACK_SYMBOL in x;
    }
    /**
     * An attribute that represents the name of the nested stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return a token that parses the name from the stack ID.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackName" }`
     *
     * @stability stable
     * @attribute true
     * @example
     *
     * mystack-mynestedstack-sggfrhxhum7w
     */
    get stackName() {
        return this._contextualStackName;
    }
    /**
     * An attribute that represents the ID of the stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return `{ "Ref": "LogicalIdOfNestedStackResource" }`.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackId" }`
     *
     * @stability stable
     * @attribute true
     * @example
     *
     * "arn:aws:cloudformation:us-east-2:123456789012:stack/mystack-mynestedstack-sggfrhxhum7w/f449b250-b969-11e0-a185-5081d0136786"
     */
    get stackId() {
        return this._contextualStackId;
    }
    /**
     * Assign a value to one of the nested stack parameters.
     *
     * @param name The parameter name (ID).
     * @param value The value to assign.
     * @stability stable
     */
    setParameter(name, value) {
        this.parameters[name] = value;
    }
    /**
     * Defines an asset at the parent stack which represents the template of this
     * nested stack.
     *
     * This private API is used by `App.prepare()` within a loop that rectifies
     * references every time an asset is added. This is because (at the moment)
     * assets are addressed using CloudFormation parameters.
     *
     * @returns `true` if a new asset was added or `false` if an asset was
     * previously added. When this returns `true`, App will do another reference
     * rectification cycle.
     *
     * @internal
     */
    _prepareTemplateAsset() {
        if (this._templateUrl) {
            return false;
        }
        const cfn = JSON.stringify(this._toCloudFormation());
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        const templateLocation = this._parentStack.addFileAsset({
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        });
        // if bucketName/objectKey are cfn parameters from a stack other than the parent stack, they will
        // be resolved as cross-stack references like any other (see "multi" tests).
        this._templateUrl = `https://s3.${this._parentStack.region}.${this._parentStack.urlSuffix}/${templateLocation.bucketName}/${templateLocation.objectKey}`;
        return true;
    }
    contextualAttribute(innerValue, outerValue) {
        return token_1.Token.asString({
            resolve: (context) => {
                if (stack_1.Stack.of(context.scope) === this) {
                    return innerValue;
                }
                else {
                    return outerValue;
                }
            },
        });
    }
}
exports.NestedStack = NestedStack;
_a = JSII_RTTI_SYMBOL_1;
NestedStack[_a] = { fqn: "@aws-cdk/core.NestedStack", version: "1.123.0" };
/**
 * Validates the scope for a nested stack. Nested stacks must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    if (!scope) {
        throw new Error('Nested stacks cannot be defined as a root construct');
    }
    const parentStack = constructs_1.Node.of(scope).scopes.reverse().find(p => stack_1.Stack.isStack(p));
    if (!parentStack) {
        throw new Error('Nested stacks must be defined within scope of another non-nested stack');
    }
    return parentStack;
}
//# sourceMappingURL=data:application/json;base64,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