The Eikon Data API for Python allows your Python applications to access data 
directly from Eikon or Refinitv Workspace, powering in-house or thirdparty 
desktop apps with Refinitiv data. It provides seamless workflow with the same 
data across all applications running on the desktop. It leverages Eikon data 
and entitlements to simplify market data management and reporting.
The Eikon Data API for Python is a software library that works in conjunction 
with the [Eikon](http://solutions.refinitiv.com/eikon-trading-software) 
desktop application and 
[Refinitiv Workspace](https://www.refinitiv.com/en/products/refinitiv-workspace-wealth).

# Some examples

## Import Eikon and set your App Key

```python
import eikon as ek

ek.set_app_key('8e9bxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx1b035d')
```

## Get real-time Snapshots

```python
df, err = ek.get_data(
    instruments = ['GOOG.O','MSFT.O', 'FB.O'], 
    fields = ['BID','ASK']
)
display(df)
```

|   | Instrument | BID | ASK |
| --- | ---------- | --- | --- |
| 0 | GOOG.O | 1350.48 | 1352.19 |
| 1 | MSFT.O | 152.38 | 152.40 |
| 2 | FB.O | 203.08 | 203.15 |

## Get Real-time Streaming data

```python
streaming_prices = ek.StreamingPrices(
    instruments = ['EUR=', 'GBP=', 'JPY=', 'CAD='], 
    fields = ['DSPLY_NAME', 'BID', 'ASK'],
    on_update = lambda streaming_price, instrument_name, fields : 
        print("Update received for {}: {}".format(instrument_name, fields))
)

streaming_prices.open()
```
Output:

    <StreamState.Open: 3>

    Update received for JPY=: {'DSPLY_NAME': 'BARCLAYS     LON', 'BID': 109.59, 'ASK': 109.62}
    Update received for GBP=: {'DSPLY_NAME': 'ASANPACIFIBK MOW', 'BID': 1.341, 'ASK': 1.3411}
    Update received for EUR=: {'DSPLY_NAME': 'UBS          ZUR', 'BID': 1.117, 'ASK': 1.1174}
    Update received for CAD=: {'DSPLY_NAME': 'HSBC         LON', 'BID': 1.3165, 'ASK': 1.3167}
    Update received for JPY=: {'DSPLY_NAME': 'ASANPACIFIBK MOW', 'BID': 109.59, 'ASK': 109.61}
    Update received for GBP=: {'DSPLY_NAME': 'INTERPROMBAN MOW', 'BID': 1.341, 'ASK': 1.3412}
    Update received for EUR=: {'DSPLY_NAME': 'RBS          LON', 'BID': 1.117, 'ASK': 1.1174}
    Update received for CAD=: {'DSPLY_NAME': 'CIBC         TOR', 'BID': 1.316, 'ASK': 1.3164}
    Update received for JPY=: {'DSPLY_NAME': 'BARCLAYS     LON', 'BID': 109.59, 'ASK': 109.62}
    Update received for GBP=: {'DSPLY_NAME': 'INTERPROMBAN MOW', 'BID': 1.341, 'ASK': 1.3413}
    Update received for EUR=: {'DSPLY_NAME': 'BARCLAYS     LON', 'BID': 1.117, 'ASK': 1.1174}
    Update received for CAD=: {'DSPLY_NAME': 'CIBC         TOR', 'BID': 1.316, 'ASK': 1.3164}
    Update received for JPY=: {'DSPLY_NAME': 'ASANPACIFIBK MOW', 'BID': 109.59, 'ASK': 109.61}
    Update received for GBP=: {'DSPLY_NAME': 'BARCLAYS     LON', 'BID': 1.341, 'ASK': 1.3414}


## Get Fundamental & Reference data 

```python
df, err = ek.get_data(
    instruments = ['GOOG.O','MSFT.O', 'FB.O'], 
    fields = ['TR.LegalAddressCity','TR.LegalAddressLine1','TR.Employees']
)
display(df)
```
|   | Instrument | Legal Address City | Legal Address Line 1 | Full-Time Employees |
| --- | ---------- | ------------------ | -------------------- | ------------------- |
| 0 | GOOG.O | WILMINGTON | 251 Little Falls Dr | 98771 |
| 1 | MSFT.O | TUMWATER | 300 Deschutes Way SW Ste 304 | 144000 |
| 2 | FB.O | WILMINGTON | 251 Little Falls Dr | 35587 |

## Get TimeSeries

```python
ek.get_timeseries('AAPL.O', interval='minute')
```
| AAPL.O | HIGH | LOW | OPEN | CLOSE | COUNT | VOLUME |
| ---- | --- | --- | --- | --- | --- | --- |
| **Date** |   |   |   |   |   |   |
| 2019-09-12 19:57:00 | 223.2000 | 222.8600 | 223.1800 | 222.9600 | 3387.0 | 267258.0 |
| 2019-09-12 19:58:00 | 223.1800 | 222.8900 | 222.9700 | 223.1700 | 1925.0 | 210251.0 |
| 2019-09-12 19:59:00 | 223.2800 | 223.0800 | 223.1700 | 223.1500 | 2106.0 | 223191.0 |
| ... | ... | ... | ... | ... | ... | ... |
| 2019-12-12 14:31:00 | 268.3000 | 267.3200 | 267.8200 | 267.9350 | 2974.0 | 724278.0 |
| 2019-12-12 14:32:00 | 268.3600 | 267.6000 | 267.9500 | 268.3000 | 1721.0 | 193413.0 |

50000 rows by 6 columns

## Get News HeadLines


```python
ek.get_news_headlines('IBM.N', count=100)
```
|   | versionCreated | text | storyId | sourceCode |
| --- | -------------- | ---- | ------- | ---------- |
| 2019-12-12 12:45:10.958 | 2019-12-12 12:45:10.958 | IBM India calls for balance between protecting... | urn:newsml:reuters.com:20191212:nNRAafsi86:1 | NS:ASNEWS |
| 2019-12-12 12:03:54.056 | 2019-12-12 12:03:54.056 | Red Hat announces renewal of FIPS 140-2 securi... | urn:newsml:reuters.com:20191212:nNRAafs2g9:1 | NS:DATMTR |
| 2019-12-12 08:07:44.753 | 2019-12-12 08:07:44.753 | Engineer forever changed retail with creation ... | urn:newsml:reuters.com:20191212:nNRAafpj8v:1 | NS:GLOBML |
| ... | ... | ... | ... | ... |
| 2019-12-03 16:18:50.532 | 2019-12-03 16:18:50.532 | United States : IBM Watson Health Demonstrates... | urn:newsml:reuters.com:20191203:nNRAad1a5r:1 | NS:ECLPCM |
| 2019-12-03 13:00:10.642 | 2019-12-03 13:00:10.642 | Nozomi Networks Works with IBM to Secure Indus... | urn:newsml:reuters.com:20191203:nGNX8Yr8Hy:1 | NS:GNW |

100 rows by 4 columns

## Get Symbology

```python
ek.get_symbology(['MSFT.O', 'GOOG.O', 'IBM.N'])
```
|   | CUSIP | ISIN | OAPermID | RIC | SEDOL | ticker |
| --- | ----- | ---- | -------- | --- | ----- | ------ |
| MSFT.O | 594918104 | US5949181045 | 4295907168 | MSFT.O | NaN | MSFT |
| GOOG.O | 02079K107 | US02079K1079 | 5030853586 | GOOG.O | NaN | GOOG |
| IBM.N | 459200101 | US4592001014 | 4295904307 | IBM.N | 2005973 | IBM |

# Learning materials

 To learn more about the Eikon Data API Python library just connect to the Refinitiv Developer Community. By [registering](https://developers.refinitiv.com/iam/register) and [login](https://developers.refinitiv.com/content/devportal/en_us/initCookie.html) to the Refinitiv Developer Community portal you will get free access to a number of learning materials like [Quick Start guides](https://developers.refinitiv.com/eikon-apis/eikon-data-api/quick-start), [Tutorials](https://developers.refinitiv.com/eikon-apis/eikon-data-api/learning), [Documentation](https://developers.refinitiv.com/eikon-apis/eikon-data-api/docs) and much more.  

# Help and Support

If you have any questions regarding the API usage, please post them on the [Eikon Data API Q&A Forum](https://community.developers.refinitiv.com/spaces/92/index.html). The Refinitiv Developer Community will be very pleased to help you. 


