"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile = exports.DependenciesFile || (exports.DependenciesFile = {}));
class Packaging {
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv lock -r > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else {
            return this.withNoPackaging();
        }
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGFja2FnaW5nLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicGFja2FnaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFFN0IsSUFBWSxnQkFLWDtBQUxELFdBQVksZ0JBQWdCO0lBQzFCLDRDQUF3QixDQUFBO0lBQ3hCLDBDQUFzQixDQUFBO0lBQ3RCLDJDQUF1QixDQUFBO0lBQ3ZCLDZCQUFTLENBQUE7QUFDWCxDQUFDLEVBTFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFLM0I7QUEwQkQsTUFBYSxTQUFTO0lBNkRwQixZQUFZLEtBQXFCO1FBQy9CLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7UUFDL0MsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO0tBQzFDO0lBOUREOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE9BQU87UUFDbkIsT0FBTyxJQUFJLFNBQVMsQ0FBQztZQUNuQixnQkFBZ0IsRUFBRSxnQkFBZ0IsQ0FBQyxHQUFHO1NBQ3ZDLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsVUFBVTtRQUN0QixPQUFPLElBQUksU0FBUyxDQUFDO1lBQ25CLGdCQUFnQixFQUFFLGdCQUFnQixDQUFDLE1BQU07WUFDekMsK0dBQStHO1lBQy9HLGlHQUFpRztZQUNqRyxhQUFhLEVBQUUsNkNBQTZDLGdCQUFnQixDQUFDLEdBQUcsa0JBQWtCO1NBQ25HLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQTRCO1FBQ25ELE9BQU8sSUFBSSxTQUFTLENBQUM7WUFDbkIsZ0JBQWdCLEVBQUUsZ0JBQWdCLENBQUMsTUFBTTtZQUN6Qyx3RUFBd0U7WUFDeEUsYUFBYSxFQUFFO2dCQUNoQixRQUFRLEVBQUUsUUFBUTtnQkFDZixHQUFHLEtBQUssRUFBRSxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDO2dCQUN6RCxvQkFBb0I7Z0JBQ3BCLFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxHQUFHO2dCQUNoQyxVQUFVLEVBQUUsZ0JBQWdCLENBQUMsR0FBRzthQUNwQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7U0FDVCxDQUFDLENBQUM7S0FDSjtJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGVBQWU7UUFDM0IsT0FBTyxJQUFJLFNBQVMsQ0FBQyxFQUFFLGdCQUFnQixFQUFFLGdCQUFnQixDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7S0FDbkU7SUFFTSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQWEsRUFBRSxtQkFBNkI7UUFDbEUsSUFBSSxFQUFFLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUU7WUFDNUQsT0FBTyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7U0FDMUI7UUFBQyxJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRTtZQUM5RCxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRSxtQkFBbUIsRUFBRSxDQUFDLENBQUM7U0FDakQ7YUFBTSxJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRTtZQUNoRSxPQUFPLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztTQUN2QjthQUFNO1lBQ0wsT0FBTyxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7U0FDL0I7S0FDRjtDQVFGO0FBakVELDhCQWlFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5cbmV4cG9ydCBlbnVtIERlcGVuZGVuY2llc0ZpbGUge1xuICBQSVAgPSAncmVxdWlyZW1lbnRzLnR4dCcsXG4gIFBPRVRSWSA9ICdwb2V0cnkubG9jaycsXG4gIFBJUEVOViA9ICdQaXBmaWxlLmxvY2snLFxuICBOT05FID0gJydcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQYWNrYWdpbmdQcm9wcyB7XG4gIC8qKlxuICAgKiBEZXBlbmRlbmN5IGZpbGUgZm9yIHRoZSB0eXBlIG9mIHBhY2thZ2luZy5cbiAgICovXG4gIHJlYWRvbmx5IGRlcGVuZGVuY2llc0ZpbGU6IERlcGVuZGVuY2llc0ZpbGU7XG4gIC8qKlxuICAgKiBDb21tYW5kIHRvIGV4cG9ydCB0aGUgZGVwZW5kZW5jaWVzIGludG8gYSBwaXAtY29tcGF0aWJsZSBgcmVxdWlyZW1lbnRzLnR4dGAgZm9ybWF0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGRlcGVuZGVuY2llcyBhcmUgZXhwb3J0ZWQuXG4gICAqL1xuICByZWFkb25seSBleHBvcnRDb21tYW5kPzogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFBvZXRyeVBhY2thZ2luZ1Byb3BzIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gZXhwb3J0IFBvZXRyeSBkZXBlbmRlbmNpZXMgd2l0aCBoYXNoZXMuIE5vdGUgdGhhdCB0aGlzIGNhbiBjYXVzZSBidWlsZHMgdG8gZmFpbCBpZiBub3QgYWxsIGRlcGVuZGVuY2llc1xuICAgKiBleHBvcnQgd2l0aCBhIGhhc2guXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy8xOTIzMlxuICAgKiBAZGVmYXVsdCBIYXNoZXMgYXJlIE5PVCBpbmNsdWRlZCBpbiB0aGUgZXhwb3J0ZWQgYHJlcXVpcmVtZW50cy50eHRgIGZpbGVcbiAgICovXG4gIHJlYWRvbmx5IHBvZXRyeUluY2x1ZGVIYXNoZXM/OiBib29sZWFuO1xufVxuXG5leHBvcnQgY2xhc3MgUGFja2FnaW5nIHtcblxuICAvKipcbiAgICogU3RhbmRhcmQgcGFja2FnaW5nIHdpdGggYHBpcGAuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHdpdGhQaXAoKTogUGFja2FnaW5nIHtcbiAgICByZXR1cm4gbmV3IFBhY2thZ2luZyh7XG4gICAgICBkZXBlbmRlbmNpZXNGaWxlOiBEZXBlbmRlbmNpZXNGaWxlLlBJUCxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQYWNrYWdpbmcgd2l0aCBgcGlwZW52YC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgd2l0aFBpcGVudigpOiBQYWNrYWdpbmcge1xuICAgIHJldHVybiBuZXcgUGFja2FnaW5nKHtcbiAgICAgIGRlcGVuZGVuY2llc0ZpbGU6IERlcGVuZGVuY2llc0ZpbGUuUElQRU5WLFxuICAgICAgLy8gQnkgZGVmYXVsdCwgcGlwZW52IGNyZWF0ZXMgYSB2aXJ0dWFsZW52IGluIGAvLmxvY2FsYCwgc28gd2UgZm9yY2UgaXQgdG8gY3JlYXRlIG9uZSBpbiB0aGUgcGFja2FnZSBkaXJlY3RvcnkuXG4gICAgICAvLyBBdCB0aGUgZW5kLCB3ZSByZW1vdmUgdGhlIHZpcnR1YWxlbnYgdG8gYXZvaWQgY3JlYXRpbmcgYSBkdXBsaWNhdGUgY29weSBpbiB0aGUgTGFtYmRhIHBhY2thZ2UuXG4gICAgICBleHBvcnRDb21tYW5kOiBgUElQRU5WX1ZFTlZfSU5fUFJPSkVDVD0xIHBpcGVudiBsb2NrIC1yID4gJHtEZXBlbmRlbmNpZXNGaWxlLlBJUH0gJiYgcm0gLXJmIC52ZW52YCxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQYWNrYWdpbmcgd2l0aCBgcG9ldHJ5YC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgd2l0aFBvZXRyeShwcm9wcz86IFBvZXRyeVBhY2thZ2luZ1Byb3BzKSB7XG4gICAgcmV0dXJuIG5ldyBQYWNrYWdpbmcoe1xuICAgICAgZGVwZW5kZW5jaWVzRmlsZTogRGVwZW5kZW5jaWVzRmlsZS5QT0VUUlksXG4gICAgICAvLyBFeHBvcnQgZGVwZW5kZW5jaWVzIHdpdGggY3JlZGVudGlhbHMgYXZhaWxhYmxlIGluIHRoZSBidW5kbGluZyBpbWFnZS5cbiAgICAgIGV4cG9ydENvbW1hbmQ6IFtcblx0ICAgICdwb2V0cnknLCAnZXhwb3J0JyxcbiAgICAgICAgLi4ucHJvcHM/LnBvZXRyeUluY2x1ZGVIYXNoZXMgPyBbXSA6IFsnLS13aXRob3V0LWhhc2hlcyddLFxuICAgICAgICAnLS13aXRoLWNyZWRlbnRpYWxzJyxcbiAgICAgICAgJy0tZm9ybWF0JywgRGVwZW5kZW5jaWVzRmlsZS5QSVAsXG4gICAgICAgICctLW91dHB1dCcsIERlcGVuZGVuY2llc0ZpbGUuUElQLFxuXHQgIF0uam9pbignICcpLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE5vIGRlcGVuZGVuY2llcyBvciBwYWNrYWdpbmcuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHdpdGhOb1BhY2thZ2luZygpOiBQYWNrYWdpbmcge1xuICAgIHJldHVybiBuZXcgUGFja2FnaW5nKHsgZGVwZW5kZW5jaWVzRmlsZTogRGVwZW5kZW5jaWVzRmlsZS5OT05FIH0pO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tRW50cnkoZW50cnk6IHN0cmluZywgcG9ldHJ5SW5jbHVkZUhhc2hlcz86IGJvb2xlYW4pOiBQYWNrYWdpbmcge1xuICAgIGlmIChmcy5leGlzdHNTeW5jKHBhdGguam9pbihlbnRyeSwgRGVwZW5kZW5jaWVzRmlsZS5QSVBFTlYpKSkge1xuICAgICAgcmV0dXJuIHRoaXMud2l0aFBpcGVudigpO1xuICAgIH0gaWYgKGZzLmV4aXN0c1N5bmMocGF0aC5qb2luKGVudHJ5LCBEZXBlbmRlbmNpZXNGaWxlLlBPRVRSWSkpKSB7XG4gICAgICByZXR1cm4gdGhpcy53aXRoUG9ldHJ5KHsgcG9ldHJ5SW5jbHVkZUhhc2hlcyB9KTtcbiAgICB9IGVsc2UgaWYgKGZzLmV4aXN0c1N5bmMocGF0aC5qb2luKGVudHJ5LCBEZXBlbmRlbmNpZXNGaWxlLlBJUCkpKSB7XG4gICAgICByZXR1cm4gdGhpcy53aXRoUGlwKCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiB0aGlzLndpdGhOb1BhY2thZ2luZygpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBkZXBlbmRlbmNpZXNGaWxlOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBleHBvcnRDb21tYW5kPzogc3RyaW5nO1xuICBjb25zdHJ1Y3Rvcihwcm9wczogUGFja2FnaW5nUHJvcHMpIHtcbiAgICB0aGlzLmRlcGVuZGVuY2llc0ZpbGUgPSBwcm9wcy5kZXBlbmRlbmNpZXNGaWxlO1xuICAgIHRoaXMuZXhwb3J0Q29tbWFuZCA9IHByb3BzLmV4cG9ydENvbW1hbmQ7XG4gIH1cbn0iXX0=