"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SnsToSqs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const subscriptions = require("aws-cdk-lib/aws-sns-subscriptions");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
const core_1 = require("@aws-solutions-constructs/core");
const core_2 = require("aws-cdk-lib/core");
const cx_api_1 = require("aws-cdk-lib/cx-api");
/**
 * @summary The SnsToSqs class.
 */
class SnsToSqs extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the SnsToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {SnsToSqsProps} props - user provided props for the construct.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckSnsProps(props);
        defaults.CheckSqsProps(props);
        this.uniquePropChecks(props);
        const activeKeys = SnsToSqs.createRequiredKeys(scope, id, props);
        if (!activeKeys.useDeprecatedInterface) {
            this.queueEncryptionKey = activeKeys.queueKey;
            this.topicEncryptionKey = activeKeys.topicKey;
        }
        // Setup the SNS topic
        if (!props.existingTopicObj) {
            // If an existingTopicObj was not specified create new topic
            const buildTopicResponse = defaults.buildTopic(this, id, {
                topicProps: props.topicProps,
                encryptionKey: activeKeys.topicKey,
                enableEncryptionWithCustomerManagedKey: activeKeys.encryptTopicWithCmk,
            });
            if (activeKeys.useDeprecatedInterface) {
                this.encryptionKey = buildTopicResponse.key;
            }
            this.snsTopic = buildTopicResponse.topic;
        }
        else {
            // If an existingTopicObj was specified utilize the provided topic
            this.snsTopic = props.existingTopicObj;
        }
        // Setup the queue
        const buildQueueResponse = defaults.buildQueue(this, 'queue', {
            existingQueueObj: props.existingQueueObj,
            queueProps: props.queueProps,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount,
            enableEncryptionWithCustomerManagedKey: activeKeys.encryptQueueWithCmk,
            encryptionKey: activeKeys.queueKey,
        });
        this.sqsQueue = buildQueueResponse.queue;
        this.deadLetterQueue = buildQueueResponse.dlq;
        // Setup the SQS queue subscription to the SNS topic
        this.snsTopic.addSubscription(new subscriptions.SqsSubscription(this.sqsQueue, props.sqsSubscriptionProps));
    }
    /*
    *
    * With the old and new interfaces both supported, it got complex figuring out
    * what props to use for what keys. This complexity has all been extracted to this
    * function to keep the constructor itself more straightforward
    *
    * This function will interpret what info the client provided (throw an error if there are conflicts),
    * determine from the data and it's own internal logic whether to implement the deprecated or new interface,
    * CREATE ANY KEYS NEEDED (not passing a create key flag to the buildTopic and buildQueue functions), then
    * return the keys in a structure.
    *
    * Odd things to know:
    * -If it decides to implement the deprecated interface it will still fill in the topicKey and queueKey values - topicKey
    *      and queueKey will ALWAYS be set, for the old interface they will be set to the same key as singleKey
    * -If the client provides no key info, this function will use the FeatureFlag to determine which interface to use
    */
    static createRequiredKeys(scope, id, props) {
        let topicKey;
        let encryptTopicWithCmk = false;
        let queueKey;
        let encryptQueueWithCmk = false;
        let singleKey;
        // First - confirm that only 1 interface is being used
        let useDeprecatedInterface = false;
        let useCurrentInterface = false;
        if (props.enableEncryptionWithCustomerManagedKey || props.enableEncryptionWithCustomerManagedKey || props.encryptionKeyProps) {
            useDeprecatedInterface = true;
            defaults.printWarning('The enableEncryptionWithCustomerManagedKey, enableEncryptionWithCustomerManagedKey and encryptionKeyProps props values for SnsToSqsProps ' +
                'are deprecated. Consider moving to encryptQueueWithCmk, queueEncryptionKeyProps, existingQueueEncryptionKey, encryptTopicWithCmk, ' +
                'topicEncryptionKeyProps and existingTopicEncryptionKey.');
        }
        if (props.encryptQueueWithCmk ||
            props.queueEncryptionKeyProps ||
            props.existingQueueEncryptionKey ||
            props.encryptTopicWithCmk ||
            props.topicEncryptionKeyProps ||
            props.existingTopicEncryptionKey) {
            useCurrentInterface = true;
        }
        if (useCurrentInterface && useDeprecatedInterface) {
            throw new Error('Cannot specify both deprecated key props and new key props');
        }
        // If neither are set, use the feature flag choose the functionality
        if (!useCurrentInterface && !useDeprecatedInterface) {
            defaults.printWarning('Unable to determine whether to use current or deprecated Key functionality, using Feature Flag SNS_SUBSCRIPTIONS_SQS_DECRYPTION_POLICY to decide');
            if (core_2.FeatureFlags.of(scope).isEnabled(cx_api_1.SNS_SUBSCRIPTIONS_SQS_DECRYPTION_POLICY)) {
                useCurrentInterface = true;
            }
            else {
                useDeprecatedInterface = true;
            }
        }
        // If Deprecated functionality
        // Use code from above to create single key
        if (useDeprecatedInterface) {
            const queueKeyNeeded = DoWeNeedACmk(props.existingQueueObj, props.queueProps?.encryptionMasterKey, props.enableEncryptionWithCustomerManagedKey);
            const topicKeyNeeded = DoWeNeedACmk(props.existingTopicObj, props.topicProps?.masterKey, props.enableEncryptionWithCustomerManagedKey);
            if (queueKeyNeeded || topicKeyNeeded) {
                // We need to encrypt the resources with a single key
                singleKey = props.encryptionKey ?? (0, core_1.buildEncryptionKey)(scope, id, props.encryptionKeyProps);
                topicKey = topicKeyNeeded ? singleKey : undefined;
                queueKey = queueKeyNeeded ? singleKey : undefined;
                encryptQueueWithCmk = queueKeyNeeded;
                encryptTopicWithCmk = topicKeyNeeded;
            }
        }
        if (useCurrentInterface) {
            if (props.queueProps?.encryption) {
                throw new Error('The new interface of SnsToSqs does not support managing encryption using the queueProps.encryption setting. ' +
                    'To use a totally unencrypted queue (not recommended), create the queue separately and pass in the existingQueueObj prop');
            }
            if (DoWeNeedACmk(props.existingTopicObj, props.topicProps?.masterKey, props.encryptTopicWithCmk)) {
                topicKey = props.existingTopicEncryptionKey ?? (0, core_1.buildEncryptionKey)(scope, `${id}topic`, props.topicEncryptionKeyProps);
                encryptTopicWithCmk = true;
            }
            if (DoWeNeedACmk(props.existingQueueObj, props.queueProps?.encryptionMasterKey, props.encryptQueueWithCmk)) {
                queueKey = props.existingQueueEncryptionKey ?? (0, core_1.buildEncryptionKey)(scope, `${id}queue`, props.queueEncryptionKeyProps);
                encryptQueueWithCmk = true;
            }
        }
        return {
            useDeprecatedInterface,
            singleKey,
            topicKey,
            queueKey,
            encryptQueueWithCmk,
            encryptTopicWithCmk
        };
    }
    /*
     * Because the props for the queue and topic are unique on this construct, we need
     * to check for props issues that the standard checks won't catch
     */
    uniquePropChecks(props) {
        let errorMessages = '';
        let errorFound = false;
        if (props.topicProps?.masterKey && props.existingTopicEncryptionKey) {
            errorMessages += 'Error - Either provide topicProps.masterKey or existingTopicEncryptionKey, but not both.\n';
            errorFound = true;
        }
        if (props.topicProps?.masterKey && props.topicEncryptionKeyProps) {
            errorMessages += 'Error - Either provide topicProps.masterKey or topicEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if (props.existingTopicEncryptionKey && props.topicEncryptionKeyProps) {
            errorMessages += 'Error - Either provide existingTopicEncryptionKey or topicEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if (props.queueProps?.encryptionMasterKey && props.queueEncryptionKeyProps) {
            errorMessages += 'Error - Either provide queueProps.encryptionMasterKey or queueEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if (props.queueProps?.encryptionMasterKey && props.existingQueueEncryptionKey) {
            errorMessages += 'Error - Either provide queueProps.encryptionMasterKey or existingQueueEncryptionKey, but not both.\n';
            errorFound = true;
        }
        if (props.existingQueueEncryptionKey && props.queueEncryptionKeyProps) {
            errorMessages += 'Error - Either provide existingQueueEncryptionKey or queueEncryptionKeyProps, but not both.\n';
            errorFound = true;
        }
        if ((props.encryptQueueWithCmk === false) && (props.queueEncryptionKeyProps)) {
            errorMessages += 'Error - if encryptQueueWithCmk is false, submitting queueEncryptionKeyProps is invalid\n';
            errorFound = true;
        }
        if ((props.encryptQueueWithCmk === false) && (props.existingQueueEncryptionKey)) {
            errorMessages += 'Error - if encryptQueueWithCmk is false, submitting existingQueueEncryptionKey is invalid\n';
            errorFound = true;
        }
        if ((props.encryptTopicWithCmk === false) && (props.topicEncryptionKeyProps)) {
            errorMessages += 'Error - if encryptTopicWithCmk is false, submitting topicEncryptionKeyProps is invalid\n';
            errorFound = true;
        }
        if ((props.encryptTopicWithCmk === false) && (props.existingTopicEncryptionKey)) {
            errorMessages += 'Error - if encryptTopicWithCmk is false, submitting existingTopicEncryptionKey is invalid\n';
            errorFound = true;
        }
        if ((props.enableEncryptionWithCustomerManagedKey === false) && (props.encryptionKey || props.encryptionKeyProps)) {
            errorMessages += 'Error - if enableEncryptionWithCustomerManagedKey is false, submitting encryptionKey or encryptionKeyProps is invalid\n';
            errorFound = true;
        }
        if (errorFound) {
            throw new Error(errorMessages);
        }
    }
}
exports.SnsToSqs = SnsToSqs;
_a = JSII_RTTI_SYMBOL_1;
SnsToSqs[_a] = { fqn: "@aws-solutions-constructs/aws-sns-sqs.SnsToSqs", version: "2.85.2" };
function DoWeNeedACmk(existingResource, keyInResourceProps, encryptionFlag) {
    // We only need tp create a CMK if
    //   -The client did not supply an existing resource
    //   -The client did not specify an existing key in the resource properties
    //   -The client did not explicitly turn off encryption by setting the flag to false.
    if (!existingResource &&
        !keyInResourceProps &&
        defaults.CheckBooleanWithDefault(encryptionFlag, true)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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