"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../../lib");
const iam = require("aws-cdk-lib/aws-iam");
const sns = require("aws-cdk-lib/aws-sns");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-sns-sqs';
stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
const dlqEncryptionKey = new kms.Key(stack, 'ImportedSQSEncryptionKey', {
    enableKeyRotation: true,
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
});
const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {
    encryptionMasterKey: dlqEncryptionKey
});
// Definitions
const props = {
    sqsSubscriptionProps: {
        rawMessageDelivery: true,
        // To test this stack, when publishing the SNS message, add a message attribute:
        // Type: string, Name: store, Value: (any value)
        filterPolicy: {
            store: sns.SubscriptionFilter.existsFilter(),
        },
        deadLetterQueue: dlq,
    },
    topicEncryptionKeyProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
    },
    queueEncryptionKeyProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
    }
};
const snsToSqsStack = new lib_1.SnsToSqs(stack, 'test-sns-sqs-stack', props);
// Grant yourself permissions to use the Customer Managed KMS Key
const policyStatement = new iam.PolicyStatement({
    actions: ["kms:Encrypt", "kms:Decrypt"],
    effect: iam.Effect.ALLOW,
    principals: [new iam.AccountRootPrincipal()],
    resources: ["*"]
});
snsToSqsStack.queueEncryptionKey?.addToResourcePolicy(policyStatement);
snsToSqsStack.topicEncryptionKey?.addToResourcePolicy(policyStatement);
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuc25zc3FzLXJhd01lc3NhZ2VEZWxpdmVyeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImludGVnLnNuc3Nxcy1yYXdNZXNzYWdlRGVsaXZlcnkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILFVBQVU7QUFDViw2Q0FBd0Q7QUFDeEQsbUNBQW9EO0FBQ3BELDJDQUEyQztBQUMzQywyQ0FBMkM7QUFDM0MsMkNBQTJDO0FBQzNDLDJDQUEyQztBQUMzQyx5REFBd0U7QUFDeEUsa0VBQXVEO0FBRXZELFFBQVE7QUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLGlCQUFHLEVBQUUsQ0FBQztBQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLENBQUMsR0FBRyxFQUFFLElBQUEsNkJBQXNCLEVBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztBQUNqRSxLQUFLLENBQUMsZUFBZSxDQUFDLFdBQVcsR0FBRyxrQ0FBa0MsQ0FBQztBQUN2RSxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUVyRixNQUFNLGdCQUFnQixHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsMEJBQTBCLEVBQUU7SUFDdEUsaUJBQWlCLEVBQUUsSUFBSTtJQUN2QixhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO0NBQ3JDLENBQUMsQ0FBQztBQUNILE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7SUFDbkQsbUJBQW1CLEVBQUUsZ0JBQWdCO0NBQ3RDLENBQUMsQ0FBQztBQUVILGNBQWM7QUFDZCxNQUFNLEtBQUssR0FBa0I7SUFDM0Isb0JBQW9CLEVBQUU7UUFDcEIsa0JBQWtCLEVBQUUsSUFBSTtRQUN4QixnRkFBZ0Y7UUFDaEYsZ0RBQWdEO1FBQ2hELFlBQVksRUFBRTtZQUNaLEtBQUssRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsWUFBWSxFQUFFO1NBQzdDO1FBQ0QsZUFBZSxFQUFFLEdBQUc7S0FDckI7SUFDRCx1QkFBdUIsRUFBRTtRQUN2QixhQUFhLEVBQUcsMkJBQWEsQ0FBQyxPQUFPO0tBQ3RDO0lBQ0QsdUJBQXVCLEVBQUU7UUFDdkIsYUFBYSxFQUFHLDJCQUFhLENBQUMsT0FBTztLQUN0QztDQUNGLENBQUM7QUFFRixNQUFNLGFBQWEsR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFFdkUsaUVBQWlFO0FBQ2pFLE1BQU0sZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztJQUM5QyxPQUFPLEVBQUUsQ0FBQyxhQUFhLEVBQUUsYUFBYSxDQUFDO0lBQ3ZDLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7SUFDeEIsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztJQUM1QyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7Q0FDakIsQ0FBQyxDQUFDO0FBRUgsYUFBYSxDQUFDLGtCQUFrQixFQUFFLG1CQUFtQixDQUFDLGVBQWUsQ0FBQyxDQUFDO0FBQ3ZFLGFBQWEsQ0FBQyxrQkFBa0IsRUFBRSxtQkFBbUIsQ0FBQyxlQUFlLENBQUMsQ0FBQztBQUV2RSxJQUFJLDZCQUFTLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLFNBQVMsRUFBRTtRQUN6QyxLQUFLO0tBQ04sRUFBRSxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG4vLyBJbXBvcnRzXG5pbXBvcnQgeyBBcHAsIFJlbW92YWxQb2xpY3ksIFN0YWNrIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5pbXBvcnQgeyBTbnNUb1NxcywgU25zVG9TcXNQcm9wcyB9IGZyb20gXCIuLi8uLi9saWJcIjtcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHNucyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc25zJztcbmltcG9ydCAqIGFzIHNxcyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3FzJztcbmltcG9ydCAqIGFzIGttcyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mta21zJztcbmltcG9ydCB7IGdlbmVyYXRlSW50ZWdTdGFja05hbWUgfSBmcm9tICdAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmUnO1xuaW1wb3J0IHsgSW50ZWdUZXN0IH0gZnJvbSAnQGF3cy1jZGsvaW50ZWctdGVzdHMtYWxwaGEnO1xuXG4vLyBTZXR1cFxuY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCBnZW5lcmF0ZUludGVnU3RhY2tOYW1lKF9fZmlsZW5hbWUpKTtcbnN0YWNrLnRlbXBsYXRlT3B0aW9ucy5kZXNjcmlwdGlvbiA9ICdJbnRlZ3JhdGlvbiBUZXN0IGZvciBhd3Mtc25zLXNxcyc7XG5zdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCB0cnVlKTtcblxuY29uc3QgZGxxRW5jcnlwdGlvbktleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAnSW1wb3J0ZWRTUVNFbmNyeXB0aW9uS2V5Jywge1xuICBlbmFibGVLZXlSb3RhdGlvbjogdHJ1ZSxcbiAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZXG59KTtcbmNvbnN0IGRscSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdleGlzdGluZy1kbHEtb2JqJywge1xuICBlbmNyeXB0aW9uTWFzdGVyS2V5OiBkbHFFbmNyeXB0aW9uS2V5XG59KTtcblxuLy8gRGVmaW5pdGlvbnNcbmNvbnN0IHByb3BzOiBTbnNUb1Nxc1Byb3BzID0ge1xuICBzcXNTdWJzY3JpcHRpb25Qcm9wczoge1xuICAgIHJhd01lc3NhZ2VEZWxpdmVyeTogdHJ1ZSxcbiAgICAvLyBUbyB0ZXN0IHRoaXMgc3RhY2ssIHdoZW4gcHVibGlzaGluZyB0aGUgU05TIG1lc3NhZ2UsIGFkZCBhIG1lc3NhZ2UgYXR0cmlidXRlOlxuICAgIC8vIFR5cGU6IHN0cmluZywgTmFtZTogc3RvcmUsIFZhbHVlOiAoYW55IHZhbHVlKVxuICAgIGZpbHRlclBvbGljeToge1xuICAgICAgc3RvcmU6IHNucy5TdWJzY3JpcHRpb25GaWx0ZXIuZXhpc3RzRmlsdGVyKCksXG4gICAgfSxcbiAgICBkZWFkTGV0dGVyUXVldWU6IGRscSxcbiAgfSxcbiAgdG9waWNFbmNyeXB0aW9uS2V5UHJvcHM6IHtcbiAgICByZW1vdmFsUG9saWN5OiAgUmVtb3ZhbFBvbGljeS5ERVNUUk9ZXG4gIH0sXG4gIHF1ZXVlRW5jcnlwdGlvbktleVByb3BzOiB7XG4gICAgcmVtb3ZhbFBvbGljeTogIFJlbW92YWxQb2xpY3kuREVTVFJPWVxuICB9XG59O1xuXG5jb25zdCBzbnNUb1Nxc1N0YWNrID0gbmV3IFNuc1RvU3FzKHN0YWNrLCAndGVzdC1zbnMtc3FzLXN0YWNrJywgcHJvcHMpO1xuXG4vLyBHcmFudCB5b3Vyc2VsZiBwZXJtaXNzaW9ucyB0byB1c2UgdGhlIEN1c3RvbWVyIE1hbmFnZWQgS01TIEtleVxuY29uc3QgcG9saWN5U3RhdGVtZW50ID0gbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICBhY3Rpb25zOiBbXCJrbXM6RW5jcnlwdFwiLCBcImttczpEZWNyeXB0XCJdLFxuICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gIHByaW5jaXBhbHM6IFtuZXcgaWFtLkFjY291bnRSb290UHJpbmNpcGFsKCldLFxuICByZXNvdXJjZXM6IFtcIipcIl1cbn0pO1xuXG5zbnNUb1Nxc1N0YWNrLnF1ZXVlRW5jcnlwdGlvbktleT8uYWRkVG9SZXNvdXJjZVBvbGljeShwb2xpY3lTdGF0ZW1lbnQpO1xuc25zVG9TcXNTdGFjay50b3BpY0VuY3J5cHRpb25LZXk/LmFkZFRvUmVzb3VyY2VQb2xpY3kocG9saWN5U3RhdGVtZW50KTtcblxubmV3IEludGVnVGVzdChzdGFjaywgJ0ludGVnJywgeyB0ZXN0Q2FzZXM6IFtcbiAgc3RhY2tcbl0gfSk7XG4iXX0=