"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-sns-sqs Standard Queue';
stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
// Definitions
const snsManagedKey = kms.Alias.fromAliasName(stack, 'sns-managed-key', 'alias/aws/sns');
const topic = new sns.Topic(stack, 'TestTopic', {
    displayName: 'Customer subscription topic',
    fifo: false,
    masterKey: snsManagedKey
});
const encryptionKeyProps = {
    enableKeyRotation: true,
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
};
const key = new kms.Key(stack, 'ImportedEncryptionKey', encryptionKeyProps);
const deadLetterQueueKey = new kms.Key(stack, 'ImportedDLQEncryptionKey', encryptionKeyProps);
const props = {
    enableEncryptionWithCustomerManagedKey: true,
    existingTopicObj: topic,
    queueProps: {
        fifo: false,
    },
    deadLetterQueueProps: {
        encryption: sqs.QueueEncryption.KMS,
        fifo: false,
        encryptionMasterKey: deadLetterQueueKey
    },
    encryptionKey: key
};
new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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