"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const assertions_1 = require("aws-cdk-lib/assertions");
const utils_1 = require("./utils");
// This set of tests targets the legacy functionality, so each text MUST
// start with:
//        stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
test('Pattern deployment w/ new Topic, new Queue and default props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {};
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
        TopicArn: {
            Ref: "testsnssqsSnsTopic2CD0065B"
        },
        Endpoint: {
            "Fn::GetAtt": [
                "testsnssqsqueueB02504BF",
                "Arn"
            ]
        }
    });
});
test('Pattern deployment w/ new topic, new queue, and overridden props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {
        topicProps: {
            topicName: "new-topic",
        },
        queueProps: {
            queueName: "new-queue.fifo",
            fifo: true
        },
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKeyProps: {
            enableKeyRotation: false
        },
        deployDeadLetterQueue: false,
        maxReceiveCount: 0
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Topic", {
        TopicName: "new-topic",
    });
    // Assertion 2
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "new-queue.fifo",
        FifoQueue: true
    });
    // Assertion 3
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
test('Test getter methods, old interface', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        enableEncryptionWithCustomerManagedKey: true,
        deployDeadLetterQueue: true,
        maxReceiveCount: 0
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.resourceCountIs("AWS::KMS::Key", 1);
});
test('Test deployment w/ existing queue, and topic', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        existingTopicObj: topic,
        existingQueueObj: queue
    });
    expect(app.snsTopic).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SNS::Topic", {
        TopicName: "existing-topic-obj"
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "existing-queue-obj"
    });
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
});
test('Test deployment with imported encryption key, old interface', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false
    });
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: kmsKey
    });
    // Assertion 2
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    // Assertion 3
    template.hasResourceProperties("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "importedkey38675D68",
                "Arn"
            ]
        }
    });
});
test('Test deployment with SNS managed KMS key, old interface', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: kms.Alias.fromAliasName(stack, 'sns-managed-key', 'alias/aws/sns')
        },
        queueProps: {
            encryption: sqs.QueueEncryption.KMS
        },
        enableEncryptionWithCustomerManagedKey: false
    });
    // Assertion 2
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.sse);
});
test('Test deployment with CMK encrypted SNS Topic (avoids interface)', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const testDescription = 'someString-llasdj';
    const cmk = new kms.Key(stack, 'cmk', { description: testDescription });
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: cmk
        }
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::KMS::Key", {
        Description: testDescription
    });
    // A key is still created for the SQS queue, so there are 2 keys in the stack
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
});
test('Pattern deployment w/ existing topic and FIFO queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'TestTopic', {
        contentBasedDeduplication: true,
        displayName: 'Customer subscription topic',
        fifo: true,
        topicName: 'customerTopic',
    });
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        existingTopicObj: topic,
        queueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: true,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: true,
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    template.resourceCountIs("AWS::KMS::Key", 0);
    template.hasResourceProperties("AWS::SQS::Queue", {
        FifoQueue: true,
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testsnssqsqueuedlq3331312B",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
});
test('Pattern deployment w/ existing topic and Standard queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'TestTopic', {
        displayName: 'Customer subscription topic',
        fifo: false,
        topicName: 'customerTopic',
    });
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        existingTopicObj: topic,
        queueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: false,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: false,
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    template.hasResourceProperties("AWS::SQS::Queue", {
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testsnssqsqueuedlq3331312B",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
});
test('Check raw message delivery is true', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {
        sqsSubscriptionProps: {
            rawMessageDelivery: true
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
        TopicArn: {
            Ref: "testsnssqsSnsTopic2CD0065B"
        },
        Endpoint: {
            "Fn::GetAtt": [
                "testsnssqsqueueB02504BF",
                "Arn"
            ]
        },
        RawMessageDelivery: true
    });
});
test('Check for filter policy', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const props = {
        sqsSubscriptionProps: {
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    allowlist: ['red', 'orange'],
                    matchPrefixes: ['bl'],
                })
            }
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        FilterPolicy: {
            color: [
                "red",
                "orange",
                {
                    prefix: "bl"
                }
            ]
        }
    });
});
test('Check SNS dead letter queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {});
    const props = {
        sqsSubscriptionProps: {
            deadLetterQueue: dlq
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "existingdlqobj784C5542",
                    "Arn"
                ]
            }
        }
    });
});
test('Construct uses topicProps.masterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const cmk = new kms.Key(stack, 'cmkfortopic');
    const props = {
        topicProps: {
            masterKey: cmk
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmkfortopic0904647B",
                "Arn"
            ]
        }
    });
});
test('Construct uses queueProps.encryptionMasterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const cmk = new kms.Key(stack, 'cmkforqueue');
    const props = {
        queueProps: {
            encryptionMasterKey: cmk
        }
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmkforqueue4E093476",
                "Arn"
            ]
        }
    });
});
test('Construct does not override unencrypted topic when passed in existingTopicObj prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    const props = {
        existingTopicObj,
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(testConstruct.snsTopic).toBeDefined();
    // Ensure that the existing topic above is the only topic in the stack
    template.resourceCountIs("AWS::SNS::Topic", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
});
test('Confirm that CheckSnsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicObj: topic,
            topicProps: {
                topicName: 'topic-name'
            },
            existingQueueObj: queue
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps or existingTopicObj, but not both.\n/);
});
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that CheckSnsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicObj: topic,
            topicProps: {
                topicName: 'topic-name'
            },
            existingQueueObj: queue
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps or existingTopicObj, but not both.\n/);
});
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that Construct checks for mixed deprecated and active props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            enableEncryptionWithCustomerManagedKey: true,
            encryptQueueWithCmk: true
        });
    };
    expect(app).toThrowError(/Cannot specify both deprecated key props and new key props/);
});
test('Error if enableEncryption is false and encryption settings are provided', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            enableEncryptionWithCustomerManagedKey: false,
            encryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError("Error - if enableEncryptionWithCustomerManagedKey is false, submitting encryptionKey or encryptionKeyProps is invalid\n");
});
test('test CreateRequiredKeys for no arguments', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {});
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Topic is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingTopicObj: {}
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Queue is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingQueueObj: {}
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Topic props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        topicProps: {
            masterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).toBeDefined();
});
test('test CreateRequiredKeys when Queue props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        queueProps: {
            encryptionMasterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeTruthy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).toBeDefined();
});
// ***************************
//
// Over the course of implementing the new interface and keeping the old interface capability,
// it became clear that the old interface had some irregular behavior that we did not want to alter.
// So we made a table of behavior for different inputs, then ran a test for each row of that table
// to capture the current behavior. These are those tests that protect existing behavior. Some
// repeat tests already implemented above, we chose not to worry about that.
//
// ***************************
test('1 Legacy Behavior - Queue Props, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        queueProps: {
            encryptionMasterKey: existingKey,
        }
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.sse);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.sse);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('2 Legacy Behavior - Queue Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        queueProps: {
            encryptionMasterKey: existingKey,
        }
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('4 Legacy Behavior - Topic Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        topicProps: {
            masterKey: existingKey,
        }
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('5 Legacy Behavior - Topic Props, Existing Key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
            encryptionKey: existingKey,
            topicProps: {
                masterKey: existingKey,
            }
        });
    };
    expect(app).toThrowError();
});
test('6 Legacy Behavior - Queue Props, Existing Key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
            encryptionKey: existingKey,
            queueProps: {
                encryptionMasterKey: existingKey,
            }
        });
    };
    expect(app).toThrowError();
});
test('7 Legacy Behavior - Queue Props, Key Props provided', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
            encryptionKeyProps: {},
            queueProps: {
                encryptionMasterKey: existingKey,
            }
        });
    };
    expect(app).toThrowError();
});
test('8 Legacy Behavior - Topic Props, Queue Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const existingKeyTwo = new kms.Key(stack, 'test-existing-key-two', {
        enableKeyRotation: true,
        alias: 'existingKeyTwo'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        topicProps: {
            masterKey: existingKey,
        },
        queueProps: {
            encryptionMasterKey: existingKeyTwo
        }
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    // Legacy code created an extra, unneeded key. New
    // code returns this to 2
    template.resourceCountIs("AWS::KMS::Key", 2);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('9 Legacy Behavior - Topic Props, Queue Props, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const existingKeyTwo = new kms.Key(stack, 'test-existing-key-two', {
        enableKeyRotation: true,
        alias: 'existingKeyTwo'
    });
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        topicProps: {
            masterKey: existingKey,
        },
        queueProps: {
            encryptionMasterKey: existingKeyTwo
        }
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('10 Legacy Behavior - Existing Topic and Queue, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        existingQueueObj: queue,
        existingTopicObj: topic
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 0);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
});
test('11 Legacy Behavior - Existing Topic and Queue, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        existingQueueObj: queue,
        existingTopicObj: topic
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    // Legacy code created an unused key here, new code
    // does not so this is 0
    template.resourceCountIs("AWS::KMS::Key", 0);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
});
test('13 Legacy Behavior - Existing Topic, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        existingTopicObj: topic
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('14 Legacy Behavior - Existing Topic, Key Props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const topic = new sns.Topic(stack, 'test-topic', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: existingKey,
        existingTopicObj: topic
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('15 Legacy Behavior - Existing Topic, Existing Key, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const topic = new sns.Topic(stack, 'test-topic', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        encryptionKeyProps: { enableKeyRotation: false },
        existingTopicObj: topic
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
});
test('16 Legacy Behavior - Existing Queue, EncryptFlag False', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: false,
        existingQueueObj: queue
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.sse);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 0);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.sse);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
});
test('17 Legacy Behavior - Existing Queue, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        existingQueueObj: queue
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
});
test('18 Legacy Behavior - Existing Queue, Existing Key, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const existingKey = new kms.Key(stack, 'test-existing-key', {
        enableKeyRotation: true,
        alias: 'existingKey'
    });
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: existingKey,
        existingQueueObj: queue
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
});
test('19 Legacy Behavior - Existing Queue, Key Props, EncryptFlag True', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
    const queue = new sqs.Queue(stack, 'test-queue', {});
    const snsToSqsStack = new lib_1.SnsToSqs(stack, 'SnsToSqsPattern', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKeyProps: { enableKeyRotation: false },
        existingQueueObj: queue
    });
    (0, utils_1.CheckKeyProperty)(snsToSqsStack.encryptionKey, utils_1.keyType.cmk);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic25zLXNxcy1vbGQudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNucy1zcXMtb2xkLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILFVBQVU7QUFDViw2Q0FBbUQ7QUFDbkQsZ0NBQWlEO0FBQ2pELDJDQUEyQztBQUMzQywyQ0FBMkM7QUFDM0MsMkNBQTJDO0FBQzNDLHVEQUFrRDtBQUNsRCxtQ0FBMEY7QUFFMUYsd0VBQXdFO0FBQ3hFLGNBQWM7QUFDZCxnR0FBZ0c7QUFFaEcsSUFBSSxDQUFDLDhEQUE4RCxFQUFFLEdBQUcsRUFBRTtJQUN4RSxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxLQUFLLEdBQWtCLEVBQUUsQ0FBQztJQUNoQyxNQUFNLGFBQWEsR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRWpFLE1BQU0sQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDN0MsTUFBTSxDQUFDLGFBQWEsQ0FBQyxhQUFhLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUVsRCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRXpDLFFBQVEsQ0FBQyxlQUFlLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQzdDLFFBQVEsQ0FBQyxlQUFlLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDL0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUUvQyxRQUFRLENBQUMscUJBQXFCLENBQUMsd0JBQXdCLEVBQUU7UUFDdkQsUUFBUSxFQUFFLEtBQUs7UUFDZixRQUFRLEVBQUU7WUFDUixHQUFHLEVBQUUsNEJBQTRCO1NBQ2xDO1FBQ0QsUUFBUSxFQUFFO1lBQ1IsWUFBWSxFQUFFO2dCQUNaLHlCQUF5QjtnQkFDekIsS0FBSzthQUNOO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxrRUFBa0UsRUFBRSxHQUFHLEVBQUU7SUFDNUUsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sS0FBSyxHQUFrQjtRQUMzQixVQUFVLEVBQUU7WUFDVixTQUFTLEVBQUUsV0FBVztTQUN2QjtRQUNELFVBQVUsRUFBRTtZQUNWLFNBQVMsRUFBRSxnQkFBZ0I7WUFDM0IsSUFBSSxFQUFFLElBQUk7U0FDWDtRQUNELHNDQUFzQyxFQUFFLElBQUk7UUFDNUMsa0JBQWtCLEVBQUU7WUFDbEIsaUJBQWlCLEVBQUUsS0FBSztTQUN6QjtRQUNELHFCQUFxQixFQUFFLEtBQUs7UUFDNUIsZUFBZSxFQUFFLENBQUM7S0FDbkIsQ0FBQztJQUNGLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDM0MsY0FBYztJQUNkLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFekMsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGlCQUFpQixFQUFFO1FBQ2hELFNBQVMsRUFBRSxXQUFXO0tBQ3ZCLENBQUMsQ0FBQztJQUNILGNBQWM7SUFDZCxRQUFRLENBQUMscUJBQXFCLENBQUMsaUJBQWlCLEVBQUU7UUFDaEQsU0FBUyxFQUFFLGdCQUFnQjtRQUMzQixTQUFTLEVBQUUsSUFBSTtLQUNoQixDQUFDLENBQUM7SUFDSCxjQUFjO0lBQ2QsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGVBQWUsRUFBRTtRQUM5QyxpQkFBaUIsRUFBRSxLQUFLO0tBQ3pCLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtJQUM5QyxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxLQUFLLEdBQWtCO1FBQzNCLHNDQUFzQyxFQUFFLElBQUk7UUFDNUMscUJBQXFCLEVBQUUsSUFBSTtRQUMzQixlQUFlLEVBQUUsQ0FBQztLQUNuQixDQUFDO0lBQ0YsTUFBTSxhQUFhLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUVqRSxNQUFNLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQzdDLE1BQU0sQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDbEQsTUFBTSxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUM3QyxNQUFNLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBRXBELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFekMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFFL0MsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsOENBQThDLEVBQUUsR0FBRyxFQUFFO0lBQ3hELFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixxQkFBcUI7SUFDckIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRTtRQUN2RCxTQUFTLEVBQUUsb0JBQW9CO0tBQ2hDLENBQUMsQ0FBQztJQUNILE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsb0JBQW9CLEVBQUU7UUFDdkQsU0FBUyxFQUFFLG9CQUFvQjtLQUNoQyxDQUFDLENBQUM7SUFDSCxNQUFNLEdBQUcsR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7UUFDbEQsZ0JBQWdCLEVBQUUsS0FBSztRQUN2QixnQkFBZ0IsRUFBRSxLQUFLO0tBQ3hCLENBQUMsQ0FBQztJQUNILE1BQU0sQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFFbkMsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGlCQUFpQixFQUFFO1FBQ2hELFNBQVMsRUFBRSxvQkFBb0I7S0FDaEMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGlCQUFpQixFQUFFO1FBQ2hELFNBQVMsRUFBRSxvQkFBb0I7S0FDaEMsQ0FBQyxDQUFDO0lBRUgsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUM1QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2REFBNkQsRUFBRSxHQUFHLEVBQUU7SUFDdkUsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLFFBQVE7SUFDUixNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtRQUNoRCxpQkFBaUIsRUFBRSxLQUFLO0tBQ3pCLENBQUMsQ0FBQztJQUNILHFCQUFxQjtJQUNyQixJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7UUFDdEMsc0NBQXNDLEVBQUUsSUFBSTtRQUM1QyxhQUFhLEVBQUUsTUFBTTtLQUN0QixDQUFDLENBQUM7SUFDSCxjQUFjO0lBQ2QsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUV6QyxRQUFRLENBQUMscUJBQXFCLENBQUMsZUFBZSxFQUFFO1FBQzlDLGlCQUFpQixFQUFFLEtBQUs7S0FDekIsQ0FBQyxDQUFDO0lBQ0gsY0FBYztJQUNkLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxpQkFBaUIsRUFBRTtRQUNoRCxjQUFjLEVBQUU7WUFDZCxZQUFZLEVBQUU7Z0JBQ1oscUJBQXFCO2dCQUNyQixLQUFLO2FBQ047U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHlEQUF5RCxFQUFFLEdBQUcsRUFBRTtJQUNuRSxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYscUJBQXFCO0lBQ3JCLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtRQUN0QyxVQUFVLEVBQUU7WUFDVixTQUFTLEVBQUUsR0FBRyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFLGVBQWUsQ0FBQztTQUM5RTtRQUNELFVBQVUsRUFBRTtZQUNWLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLEdBQUc7U0FDcEM7UUFDRCxzQ0FBc0MsRUFBRSxLQUFLO0tBQzlDLENBQUMsQ0FBQztJQUNILGNBQWM7SUFDZCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRTNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlFQUFpRSxFQUFFLEdBQUcsRUFBRTtJQUMzRSxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxlQUFlLEdBQUcsbUJBQW1CLENBQUM7SUFDNUMsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsRUFBRSxXQUFXLEVBQUUsZUFBZSxFQUFFLENBQUMsQ0FBQztJQUN4RSxxQkFBcUI7SUFDckIsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO1FBQ3RDLFVBQVUsRUFBRTtZQUNWLFNBQVMsRUFBRSxHQUFHO1NBQ2Y7S0FDRixDQUFDLENBQUM7SUFDSCxjQUFjO0lBQ2QsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUV6QyxRQUFRLENBQUMscUJBQXFCLENBQUMsZUFBZSxFQUFFO1FBQzlDLFdBQVcsRUFBRSxlQUFlO0tBQzdCLENBQUMsQ0FBQztJQUNILDZFQUE2RTtJQUM3RSxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUM3QyxRQUFRLENBQUMsZUFBZSxDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ2pELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFEQUFxRCxFQUFFLEdBQUcsRUFBRTtJQUMvRCxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7UUFDOUMseUJBQXlCLEVBQUUsSUFBSTtRQUMvQixXQUFXLEVBQUUsNkJBQTZCO1FBQzFDLElBQUksRUFBRSxJQUFJO1FBQ1YsU0FBUyxFQUFFLGVBQWU7S0FDM0IsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQWtCO1FBQzNCLHNDQUFzQyxFQUFFLEtBQUs7UUFDN0MsZ0JBQWdCLEVBQUUsS0FBSztRQUN2QixVQUFVLEVBQUU7WUFDVixVQUFVLEVBQUUsR0FBRyxDQUFDLGVBQWUsQ0FBQyxXQUFXO1lBQzNDLElBQUksRUFBRSxJQUFJO1NBQ1g7UUFDRCxvQkFBb0IsRUFBRTtZQUNwQixVQUFVLEVBQUUsR0FBRyxDQUFDLGVBQWUsQ0FBQyxXQUFXO1lBQzNDLElBQUksRUFBRSxJQUFJO1NBQ1g7S0FDRixDQUFDO0lBRUYsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUUzQyxjQUFjO0lBQ2QsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUUxQyxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUU3QyxRQUFRLENBQUMscUJBQXFCLENBQUMsaUJBQWlCLEVBQUU7UUFDaEQsU0FBUyxFQUFFLElBQUk7UUFDZixhQUFhLEVBQUU7WUFDYixtQkFBbUIsRUFBRTtnQkFDbkIsWUFBWSxFQUFFO29CQUNaLDRCQUE0QjtvQkFDNUIsS0FBSztpQkFDTjthQUNGO1lBQ0QsZUFBZSxFQUFFLEVBQUU7U0FDcEI7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5REFBeUQsRUFBRSxHQUFHLEVBQUU7SUFDbkUsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1FBQzlDLFdBQVcsRUFBRSw2QkFBNkI7UUFDMUMsSUFBSSxFQUFFLEtBQUs7UUFDWCxTQUFTLEVBQUUsZUFBZTtLQUMzQixDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBa0I7UUFDM0Isc0NBQXNDLEVBQUUsS0FBSztRQUM3QyxnQkFBZ0IsRUFBRSxLQUFLO1FBQ3ZCLFVBQVUsRUFBRTtZQUNWLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLFdBQVc7WUFDM0MsSUFBSSxFQUFFLEtBQUs7U0FDWjtRQUNELG9CQUFvQixFQUFFO1lBQ3BCLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLFdBQVc7WUFDM0MsSUFBSSxFQUFFLEtBQUs7U0FDWjtLQUNGLENBQUM7SUFFRixJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTNDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMxQyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFFMUMsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGlCQUFpQixFQUFFO1FBQ2hELGFBQWEsRUFBRTtZQUNiLG1CQUFtQixFQUFFO2dCQUNuQixZQUFZLEVBQUU7b0JBQ1osNEJBQTRCO29CQUM1QixLQUFLO2lCQUNOO2FBQ0Y7WUFDRCxlQUFlLEVBQUUsRUFBRTtTQUNwQjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtJQUM5QyxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxLQUFLLEdBQWtCO1FBQzNCLG9CQUFvQixFQUFFO1lBQ3BCLGtCQUFrQixFQUFFLElBQUk7U0FDekI7S0FDRixDQUFDO0lBQ0YsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUUzQyxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRXpDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx3QkFBd0IsRUFBRTtRQUN2RCxRQUFRLEVBQUUsS0FBSztRQUNmLFFBQVEsRUFBRTtZQUNSLEdBQUcsRUFBRSw0QkFBNEI7U0FDbEM7UUFDRCxRQUFRLEVBQUU7WUFDUixZQUFZLEVBQUU7Z0JBQ1oseUJBQXlCO2dCQUN6QixLQUFLO2FBQ047U0FDRjtRQUNELGtCQUFrQixFQUFFLElBQUk7S0FDekIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO0lBQ25DLGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLEtBQUssR0FBa0I7UUFDM0Isb0JBQW9CLEVBQUU7WUFDcEIsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsWUFBWSxDQUFDO29CQUN6QyxTQUFTLEVBQUUsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDO29CQUM1QixhQUFhLEVBQUUsQ0FBQyxJQUFJLENBQUM7aUJBQ3RCLENBQUM7YUFDSDtTQUNGO0tBQ0YsQ0FBQztJQUVGLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFM0MsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUV6QyxRQUFRLENBQUMscUJBQXFCLENBQUMsd0JBQXdCLEVBQUU7UUFDdkQsWUFBWSxFQUFFO1lBQ1osS0FBSyxFQUFFO2dCQUNMLEtBQUs7Z0JBQ0wsUUFBUTtnQkFDUjtvQkFDRSxNQUFNLEVBQUUsSUFBSTtpQkFDYjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDekQsTUFBTSxLQUFLLEdBQWtCO1FBQzNCLG9CQUFvQixFQUFFO1lBQ3BCLGVBQWUsRUFBRSxHQUFHO1NBQ3JCO0tBQ0YsQ0FBQztJQUVGLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFM0MsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUV6QyxRQUFRLENBQUMscUJBQXFCLENBQUMsd0JBQXdCLEVBQUU7UUFDdkQsYUFBYSxFQUFFO1lBQ2IsbUJBQW1CLEVBQUU7Z0JBQ25CLFlBQVksRUFBRTtvQkFDWix3QkFBd0I7b0JBQ3hCLEtBQUs7aUJBQ047YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdUVBQXVFLEVBQUUsR0FBRyxFQUFFO0lBQ2pGLGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQzlDLE1BQU0sS0FBSyxHQUFrQjtRQUMzQixVQUFVLEVBQUU7WUFDVixTQUFTLEVBQUUsR0FBRztTQUNmO0tBQ0YsQ0FBQztJQUVGLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFM0MsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGlCQUFpQixFQUFFO1FBQ2hELGNBQWMsRUFBRTtZQUNkLFlBQVksRUFBRTtnQkFDWixxQkFBcUI7Z0JBQ3JCLEtBQUs7YUFDTjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaUZBQWlGLEVBQUUsR0FBRyxFQUFFO0lBQzNGLGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQzlDLE1BQU0sS0FBSyxHQUFrQjtRQUMzQixVQUFVLEVBQUU7WUFDVixtQkFBbUIsRUFBRSxHQUFHO1NBQ3pCO0tBQ0YsQ0FBQztJQUVGLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFM0MsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGlCQUFpQixFQUFFO1FBQ2hELGNBQWMsRUFBRTtZQUNkLFlBQVksRUFBRTtnQkFDWixxQkFBcUI7Z0JBQ3JCLEtBQUs7YUFDTjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0ZBQW9GLEVBQUUsR0FBRyxFQUFFO0lBQzlGLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7UUFDckQsU0FBUyxFQUFFLHFCQUFxQjtLQUNqQyxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBa0I7UUFDM0IsZ0JBQWdCO0tBQ2pCLENBQUM7SUFFRixNQUFNLGFBQWEsR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRWpFLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLE1BQU0sQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDN0Msc0VBQXNFO0lBQ3RFLFFBQVEsQ0FBQyxlQUFlLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFL0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUU1QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7SUFDaEQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLHFCQUFxQjtJQUNyQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLG9CQUFvQixFQUFFO1FBQ3ZELFNBQVMsRUFBRSxvQkFBb0I7S0FDaEMsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRTtRQUN2RCxTQUFTLEVBQUUsb0JBQW9CO0tBQ2hDLENBQUMsQ0FBQztJQUVILE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtZQUN0QyxnQkFBZ0IsRUFBRSxLQUFLO1lBQ3ZCLFVBQVUsRUFBRTtnQkFDVixTQUFTLEVBQUUsWUFBWTthQUN4QjtZQUNELGdCQUFnQixFQUFFLEtBQUs7U0FDeEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBRUYsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsd0VBQXdFLENBQUMsQ0FBQztBQUNyRyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7SUFDaEQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtZQUN0QyxVQUFVLEVBQUU7Z0JBQ1YsYUFBYSxFQUFFLDJCQUFhLENBQUMsT0FBTzthQUNyQztZQUNELGdCQUFnQixFQUFFLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsQ0FBQztTQUNuRCxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUM7SUFFRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyx3RUFBd0UsQ0FBQyxDQUFDO0FBQ3JHLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIscUJBQXFCO0lBQ3JCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsb0JBQW9CLEVBQUU7UUFDdkQsU0FBUyxFQUFFLG9CQUFvQjtLQUNoQyxDQUFDLENBQUM7SUFDSCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLG9CQUFvQixFQUFFO1FBQ3ZELFNBQVMsRUFBRSxvQkFBb0I7S0FDaEMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO1lBQ3RDLGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsVUFBVSxFQUFFO2dCQUNWLFNBQVMsRUFBRSxZQUFZO2FBQ3hCO1lBQ0QsZ0JBQWdCLEVBQUUsS0FBSztTQUN4QixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUM7SUFFRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyx3RUFBd0UsQ0FBQyxDQUFDO0FBQ3JHLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO1lBQ3RDLFVBQVUsRUFBRTtnQkFDVixhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO2FBQ3JDO1lBQ0QsZ0JBQWdCLEVBQUUsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDO1NBQ25ELENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztJQUVGLFlBQVk7SUFDWixNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxDQUFDLHdFQUF3RSxDQUFDLENBQUM7QUFDckcsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscUVBQXFFLEVBQUUsR0FBRyxFQUFFO0lBQy9FLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7WUFDdEMsVUFBVSxFQUFFO2dCQUNWLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87YUFDckM7WUFDRCxzQ0FBc0MsRUFBRSxJQUFJO1lBQzVDLG1CQUFtQixFQUFFLElBQUk7U0FDMUIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBRUYsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyw0REFBNEQsQ0FBQyxDQUFDO0FBQ3pGLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHlFQUF5RSxFQUFFLEdBQUcsRUFBRTtJQUNuRixRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO1lBQ3RDLHNDQUFzQyxFQUFFLEtBQUs7WUFDN0Msa0JBQWtCLEVBQUUsRUFBRTtTQUN2QixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUM7SUFFRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyx5SEFBeUgsQ0FBQyxDQUFDO0FBQ3RKLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDBDQUEwQyxFQUFFLEdBQUcsRUFBRTtJQUNwRCxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLE1BQU0sR0FBRyxjQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztJQUU5RCxNQUFNLENBQUMsTUFBTSxDQUFDLHNCQUFzQixDQUFDLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDbkQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ2hELE1BQU0sQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUNoRCxNQUFNLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDdEMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztBQUN6QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7SUFDMUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxNQUFNLEdBQUcsY0FBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDeEQsZ0JBQWdCLEVBQUUsRUFBZTtLQUNsQyxDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsTUFBTSxDQUFDLHNCQUFzQixDQUFDLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDbkQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ2hELE1BQU0sQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUMvQyxNQUFNLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQzFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7QUFDekMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO0lBQzFELE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sTUFBTSxHQUFHLGNBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1FBQ3hELGdCQUFnQixFQUFFLEVBQWU7S0FDbEMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ25ELE1BQU0sQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUMvQyxNQUFNLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDaEQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDMUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUN0QyxNQUFNLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0FBQ3pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFEQUFxRCxFQUFFLEdBQUcsRUFBRTtJQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLE1BQU0sR0FBRyxjQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUN4RCxVQUFVLEVBQUU7WUFDVixTQUFTLEVBQUUsRUFBYTtTQUN6QjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxNQUFNLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUNuRCxNQUFNLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDaEQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFNBQVMsRUFBRSxDQUFDO0lBQy9DLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDdEMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDMUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztBQUN6QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxREFBcUQsRUFBRSxHQUFHLEVBQUU7SUFDL0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxNQUFNLEdBQUcsY0FBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDeEQsVUFBVSxFQUFFO1lBQ1YsbUJBQW1CLEVBQUUsRUFBYTtTQUNuQztLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxNQUFNLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUNuRCxNQUFNLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLENBQUMsU0FBUyxFQUFFLENBQUM7SUFDL0MsTUFBTSxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ2hELE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQzFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDdEMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztBQUN6QyxDQUFDLENBQUMsQ0FBQztBQUVILDhCQUE4QjtBQUM5QixFQUFFO0FBQ0YsOEZBQThGO0FBQzlGLG9HQUFvRztBQUNwRyxrR0FBa0c7QUFDbEcsOEZBQThGO0FBQzlGLDRFQUE0RTtBQUM1RSxFQUFFO0FBQ0YsOEJBQThCO0FBRTlCLElBQUksQ0FBQyxvREFBb0QsRUFBRSxHQUFHLEVBQUU7SUFFOUQsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDMUQsaUJBQWlCLEVBQUUsSUFBSTtRQUN2QixLQUFLLEVBQUUsYUFBYTtLQUNyQixDQUFDLENBQUM7SUFFSCxNQUFNLGFBQWEsR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7UUFFM0Qsc0NBQXNDLEVBQUUsS0FBSztRQUU3QyxVQUFVLEVBQUU7WUFDVixtQkFBbUIsRUFBRSxXQUFXO1NBQ2pDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsSUFBQSx3QkFBZ0IsRUFBQyxhQUFhLENBQUMsYUFBYSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUUzRCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUU3QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRTNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG1EQUFtRCxFQUFFLEdBQUcsRUFBRTtJQUU3RCxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtRQUMxRCxpQkFBaUIsRUFBRSxJQUFJO1FBQ3ZCLEtBQUssRUFBRSxhQUFhO0tBQ3JCLENBQUMsQ0FBQztJQUVILE1BQU0sYUFBYSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtRQUUzRCxzQ0FBc0MsRUFBRSxJQUFJO1FBRTVDLFVBQVUsRUFBRTtZQUNWLG1CQUFtQixFQUFFLFdBQVc7U0FDakM7S0FDRixDQUFDLENBQUM7SUFFSCxJQUFBLHdCQUFnQixFQUFDLGFBQWEsQ0FBQyxhQUFhLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRTNELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBRTdDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7QUFFM0MsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbURBQW1ELEVBQUUsR0FBRyxFQUFFO0lBRTdELGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQzFELGlCQUFpQixFQUFFLElBQUk7UUFDdkIsS0FBSyxFQUFFLGFBQWE7S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxhQUFhLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBRTNELHNDQUFzQyxFQUFFLElBQUk7UUFFNUMsVUFBVSxFQUFFO1lBQ1YsU0FBUyxFQUFFLFdBQVc7U0FDdkI7S0FDRixDQUFDLENBQUM7SUFFSCxJQUFBLHdCQUFnQixFQUFDLGFBQWEsQ0FBQyxhQUFhLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRTNELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBRTdDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7QUFFM0MsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsK0NBQStDLEVBQUUsR0FBRyxFQUFFO0lBRXpELGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQzFELGlCQUFpQixFQUFFLElBQUk7UUFDdkIsS0FBSyxFQUFFLGFBQWE7S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1lBRXJDLGFBQWEsRUFBRSxXQUFXO1lBRTFCLFVBQVUsRUFBRTtnQkFDVixTQUFTLEVBQUUsV0FBVzthQUN2QjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztJQUVGLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrQ0FBK0MsRUFBRSxHQUFHLEVBQUU7SUFFekQsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDMUQsaUJBQWlCLEVBQUUsSUFBSTtRQUN2QixLQUFLLEVBQUUsYUFBYTtLQUNyQixDQUFDLENBQUM7SUFFSCxNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7WUFFckMsYUFBYSxFQUFFLFdBQVc7WUFFMUIsVUFBVSxFQUFFO2dCQUNWLG1CQUFtQixFQUFFLFdBQVc7YUFDakM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUM7SUFFRixNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxFQUFFLENBQUM7QUFDN0IsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscURBQXFELEVBQUUsR0FBRyxFQUFFO0lBRS9ELGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQzFELGlCQUFpQixFQUFFLElBQUk7UUFDdkIsS0FBSyxFQUFFLGFBQWE7S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1lBRXJDLGtCQUFrQixFQUFFLEVBQUU7WUFFdEIsVUFBVSxFQUFFO2dCQUNWLG1CQUFtQixFQUFFLFdBQVc7YUFDakM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUM7SUFFRixNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxFQUFFLENBQUM7QUFDN0IsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsZ0VBQWdFLEVBQUUsR0FBRyxFQUFFO0lBRTFFLGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQzFELGlCQUFpQixFQUFFLElBQUk7UUFDdkIsS0FBSyxFQUFFLGFBQWE7S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSx1QkFBdUIsRUFBRTtRQUNqRSxpQkFBaUIsRUFBRSxJQUFJO1FBQ3ZCLEtBQUssRUFBRSxnQkFBZ0I7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxhQUFhLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBRTNELHNDQUFzQyxFQUFFLElBQUk7UUFFNUMsVUFBVSxFQUFFO1lBQ1YsU0FBUyxFQUFFLFdBQVc7U0FDdkI7UUFDRCxVQUFVLEVBQUU7WUFDVixtQkFBbUIsRUFBRSxjQUFjO1NBQ3BDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsSUFBQSx3QkFBZ0IsRUFBQyxhQUFhLENBQUMsYUFBYSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUUzRCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxrREFBa0Q7SUFDbEQseUJBQXlCO0lBQ3pCLFFBQVEsQ0FBQyxlQUFlLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBRTdDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7QUFFM0MsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaUVBQWlFLEVBQUUsR0FBRyxFQUFFO0lBRTNFLGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQzFELGlCQUFpQixFQUFFLElBQUk7UUFDdkIsS0FBSyxFQUFFLGFBQWE7S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSx1QkFBdUIsRUFBRTtRQUNqRSxpQkFBaUIsRUFBRSxJQUFJO1FBQ3ZCLEtBQUssRUFBRSxnQkFBZ0I7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxhQUFhLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBRTNELHNDQUFzQyxFQUFFLEtBQUs7UUFFN0MsVUFBVSxFQUFFO1lBQ1YsU0FBUyxFQUFFLFdBQVc7U0FDdkI7UUFDRCxVQUFVLEVBQUU7WUFDVixtQkFBbUIsRUFBRSxjQUFjO1NBQ3BDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsSUFBQSx3QkFBZ0IsRUFBQyxhQUFhLENBQUMsYUFBYSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUUzRCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUU3QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRTNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtFQUFrRSxFQUFFLEdBQUcsRUFBRTtJQUU1RSxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDckQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFckQsTUFBTSxhQUFhLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBRTNELHNDQUFzQyxFQUFFLEtBQUs7UUFFN0MsZ0JBQWdCLEVBQUUsS0FBSztRQUN2QixnQkFBZ0IsRUFBRSxLQUFLO0tBQ3hCLENBQUMsQ0FBQztJQUVILElBQUEsd0JBQWdCLEVBQUMsYUFBYSxDQUFDLGFBQWEsRUFBRSxlQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFFNUQsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFN0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUU1QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxpRUFBaUUsRUFBRSxHQUFHLEVBQUU7SUFFM0UsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBRXJELE1BQU0sYUFBYSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtRQUUzRCxzQ0FBc0MsRUFBRSxJQUFJO1FBRTVDLGdCQUFnQixFQUFFLEtBQUs7UUFDdkIsZ0JBQWdCLEVBQUUsS0FBSztLQUN4QixDQUFDLENBQUM7SUFFSCxJQUFBLHdCQUFnQixFQUFDLGFBQWEsQ0FBQyxhQUFhLEVBQUUsZUFBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBRTVELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLG1EQUFtRDtJQUNuRCx3QkFBd0I7SUFDeEIsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFN0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUU1QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx1REFBdUQsRUFBRSxHQUFHLEVBQUU7SUFFakUsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBRXJELE1BQU0sYUFBYSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtRQUUzRCxzQ0FBc0MsRUFBRSxJQUFJO1FBRTVDLGdCQUFnQixFQUFFLEtBQUs7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsSUFBQSx3QkFBZ0IsRUFBQyxhQUFhLENBQUMsYUFBYSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUU1RCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUU3QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDMUMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRTNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGdEQUFnRCxFQUFFLEdBQUcsRUFBRTtJQUUxRCxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtRQUMxRCxpQkFBaUIsRUFBRSxJQUFJO1FBQ3ZCLEtBQUssRUFBRSxhQUFhO0tBQ3JCLENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBRXJELE1BQU0sYUFBYSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtRQUMzRCxzQ0FBc0MsRUFBRSxJQUFJO1FBQzVDLGFBQWEsRUFBRSxXQUFXO1FBRTFCLGdCQUFnQixFQUFFLEtBQUs7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsSUFBQSx3QkFBZ0IsRUFBQyxhQUFhLENBQUMsYUFBYSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUU1RCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUU3QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDMUMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRTNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFFQUFxRSxFQUFFLEdBQUcsRUFBRTtJQUUvRSxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFckQsTUFBTSxhQUFhLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBQzNELGtCQUFrQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsS0FBSyxFQUFFO1FBQ2hELGdCQUFnQixFQUFFLEtBQUs7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsSUFBQSx3QkFBZ0IsRUFBQyxhQUFhLENBQUMsYUFBYSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUU1RCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUU3QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDMUMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRTNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHdEQUF3RCxFQUFFLEdBQUcsRUFBRTtJQUVsRSxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsdURBQXVELEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFckQsTUFBTSxhQUFhLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBQzNELHNDQUFzQyxFQUFFLEtBQUs7UUFDN0MsZ0JBQWdCLEVBQUUsS0FBSztLQUN4QixDQUFDLENBQUM7SUFFSCxJQUFBLHdCQUFnQixFQUFDLGFBQWEsQ0FBQyxhQUFhLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRTNELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBRTdDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDNUMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO0lBRWpFLGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx1REFBdUQsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUV0RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztJQUVyRCxNQUFNLGFBQWEsR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7UUFDM0Qsc0NBQXNDLEVBQUUsSUFBSTtRQUM1QyxnQkFBZ0IsRUFBRSxLQUFLO0tBQ3hCLENBQUMsQ0FBQztJQUVILElBQUEsd0JBQWdCLEVBQUMsYUFBYSxDQUFDLGFBQWEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFM0QsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFN0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUM1QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxRUFBcUUsRUFBRSxHQUFHLEVBQUU7SUFFL0UsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDMUQsaUJBQWlCLEVBQUUsSUFBSTtRQUN2QixLQUFLLEVBQUUsYUFBYTtLQUNyQixDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztJQUVyRCxNQUFNLGFBQWEsR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7UUFDM0Qsc0NBQXNDLEVBQUUsSUFBSTtRQUM1QyxhQUFhLEVBQUUsV0FBVztRQUMxQixnQkFBZ0IsRUFBRSxLQUFLO0tBQ3hCLENBQUMsQ0FBQztJQUVILElBQUEsd0JBQWdCLEVBQUMsYUFBYSxDQUFDLGFBQWEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFM0QsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFM0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFN0MsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pDLElBQUEseUJBQWlCLEVBQUMsUUFBUSxFQUFFLGVBQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUM1QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxrRUFBa0UsRUFBRSxHQUFHLEVBQUU7SUFFNUUsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHVEQUF1RCxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXRGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBRXJELE1BQU0sYUFBYSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtRQUMzRCxzQ0FBc0MsRUFBRSxJQUFJO1FBQzVDLGtCQUFrQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsS0FBSyxFQUFDO1FBQy9DLGdCQUFnQixFQUFFLEtBQUs7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsSUFBQSx3QkFBZ0IsRUFBQyxhQUFhLENBQUMsYUFBYSxFQUFFLGVBQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUUzRCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUU3QyxJQUFBLHlCQUFpQixFQUFDLFFBQVEsRUFBRSxlQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekMsSUFBQSx5QkFBaUIsRUFBQyxRQUFRLEVBQUUsZUFBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQzVDLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0IHsgU3RhY2ssIFJlbW92YWxQb2xpY3kgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IFNuc1RvU3FzLCBTbnNUb1Nxc1Byb3BzIH0gZnJvbSBcIi4uL2xpYlwiO1xuaW1wb3J0ICogYXMgc3FzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zcXMnO1xuaW1wb3J0ICogYXMgc25zIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zbnMnO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ2F3cy1jZGstbGliL2F3cy1rbXMnO1xuaW1wb3J0IHsgVGVtcGxhdGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hc3NlcnRpb25zJztcbmltcG9ydCB7IENoZWNrS2V5UHJvcGVydHksIENoZWNrUXVldWVLZXlUeXBlLCBDaGVja1RvcGljS2V5VHlwZSwga2V5VHlwZSB9IGZyb20gJy4vdXRpbHMnO1xuXG4vLyBUaGlzIHNldCBvZiB0ZXN0cyB0YXJnZXRzIHRoZSBsZWdhY3kgZnVuY3Rpb25hbGl0eSwgc28gZWFjaCB0ZXh0IE1VU1Rcbi8vIHN0YXJ0IHdpdGg6XG4vLyAgICAgICAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG50ZXN0KCdQYXR0ZXJuIGRlcGxveW1lbnQgdy8gbmV3IFRvcGljLCBuZXcgUXVldWUgYW5kIGRlZmF1bHQgcHJvcHMnLCAoKSA9PiB7XG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHByb3BzOiBTbnNUb1Nxc1Byb3BzID0ge307XG4gIGNvbnN0IHRlc3RDb25zdHJ1Y3QgPSBuZXcgU25zVG9TcXMoc3RhY2ssICd0ZXN0LXNucy1zcXMnLCBwcm9wcyk7XG5cbiAgZXhwZWN0KHRlc3RDb25zdHJ1Y3Quc25zVG9waWMpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdCh0ZXN0Q29uc3RydWN0LmVuY3J5cHRpb25LZXkpLnRvQmVEZWZpbmVkKCk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICBDaGVja1F1ZXVlS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuICBDaGVja1RvcGljS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMSk7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6U1FTOjpRdWV1ZVwiLCAyKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpTTlM6OlRvcGljXCIsIDEpO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6U05TOjpTdWJzY3JpcHRpb25cIiwge1xuICAgIFByb3RvY29sOiBcInNxc1wiLFxuICAgIFRvcGljQXJuOiB7XG4gICAgICBSZWY6IFwidGVzdHNuc3Nxc1Nuc1RvcGljMkNEMDA2NUJcIlxuICAgIH0sXG4gICAgRW5kcG9pbnQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwidGVzdHNuc3Nxc3F1ZXVlQjAyNTA0QkZcIixcbiAgICAgICAgXCJBcm5cIlxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnUGF0dGVybiBkZXBsb3ltZW50IHcvIG5ldyB0b3BpYywgbmV3IHF1ZXVlLCBhbmQgb3ZlcnJpZGRlbiBwcm9wcycsICgpID0+IHtcbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgcHJvcHM6IFNuc1RvU3FzUHJvcHMgPSB7XG4gICAgdG9waWNQcm9wczoge1xuICAgICAgdG9waWNOYW1lOiBcIm5ldy10b3BpY1wiLFxuICAgIH0sXG4gICAgcXVldWVQcm9wczoge1xuICAgICAgcXVldWVOYW1lOiBcIm5ldy1xdWV1ZS5maWZvXCIsXG4gICAgICBmaWZvOiB0cnVlXG4gICAgfSxcbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZSxcbiAgICBlbmNyeXB0aW9uS2V5UHJvcHM6IHtcbiAgICAgIGVuYWJsZUtleVJvdGF0aW9uOiBmYWxzZVxuICAgIH0sXG4gICAgZGVwbG95RGVhZExldHRlclF1ZXVlOiBmYWxzZSxcbiAgICBtYXhSZWNlaXZlQ291bnQ6IDBcbiAgfTtcbiAgbmV3IFNuc1RvU3FzKHN0YWNrLCAndGVzdC1zbnMtc3FzJywgcHJvcHMpO1xuICAvLyBBc3NlcnRpb24gMVxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG5cbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OlNOUzo6VG9waWNcIiwge1xuICAgIFRvcGljTmFtZTogXCJuZXctdG9waWNcIixcbiAgfSk7XG4gIC8vIEFzc2VydGlvbiAyXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6U1FTOjpRdWV1ZVwiLCB7XG4gICAgUXVldWVOYW1lOiBcIm5ldy1xdWV1ZS5maWZvXCIsXG4gICAgRmlmb1F1ZXVlOiB0cnVlXG4gIH0pO1xuICAvLyBBc3NlcnRpb24gM1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OktNUzo6S2V5XCIsIHtcbiAgICBFbmFibGVLZXlSb3RhdGlvbjogZmFsc2VcbiAgfSk7XG59KTtcblxudGVzdCgnVGVzdCBnZXR0ZXIgbWV0aG9kcywgb2xkIGludGVyZmFjZScsICgpID0+IHtcbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBwcm9wczogU25zVG9TcXNQcm9wcyA9IHtcbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZSxcbiAgICBkZXBsb3lEZWFkTGV0dGVyUXVldWU6IHRydWUsXG4gICAgbWF4UmVjZWl2ZUNvdW50OiAwXG4gIH07XG4gIGNvbnN0IHRlc3RDb25zdHJ1Y3QgPSBuZXcgU25zVG9TcXMoc3RhY2ssICd0ZXN0LXNucy1zcXMnLCBwcm9wcyk7XG5cbiAgZXhwZWN0KHRlc3RDb25zdHJ1Y3Quc25zVG9waWMpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdCh0ZXN0Q29uc3RydWN0LmVuY3J5cHRpb25LZXkpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdCh0ZXN0Q29uc3RydWN0LnNxc1F1ZXVlKS50b0JlRGVmaW5lZCgpO1xuICBleHBlY3QodGVzdENvbnN0cnVjdC5kZWFkTGV0dGVyUXVldWUpLnRvQmVEZWZpbmVkKCk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICBDaGVja1F1ZXVlS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuICBDaGVja1RvcGljS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMSk7XG5cbn0pO1xuXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gZXhpc3RpbmcgcXVldWUsIGFuZCB0b3BpYycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBjb25zdCB0b3BpYyA9IG5ldyBzbnMuVG9waWMoc3RhY2ssIFwiZXhpc3RpbmctdG9waWMtb2JqXCIsIHtcbiAgICB0b3BpY05hbWU6ICdleGlzdGluZy10b3BpYy1vYmonXG4gIH0pO1xuICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdleGlzdGluZy1xdWV1ZS1vYmonLCB7XG4gICAgcXVldWVOYW1lOiAnZXhpc3RpbmctcXVldWUtb2JqJ1xuICB9KTtcbiAgY29uc3QgYXBwID0gbmV3IFNuc1RvU3FzKHN0YWNrLCAnc25zLXRvLXNxcy1zdGFjaycsIHtcbiAgICBleGlzdGluZ1RvcGljT2JqOiB0b3BpYyxcbiAgICBleGlzdGluZ1F1ZXVlT2JqOiBxdWV1ZVxuICB9KTtcbiAgZXhwZWN0KGFwcC5zbnNUb3BpYykudG9CZURlZmluZWQoKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6U05TOjpUb3BpY1wiLCB7XG4gICAgVG9waWNOYW1lOiBcImV4aXN0aW5nLXRvcGljLW9ialwiXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6U1FTOjpRdWV1ZVwiLCB7XG4gICAgUXVldWVOYW1lOiBcImV4aXN0aW5nLXF1ZXVlLW9ialwiXG4gIH0pO1xuXG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLm5vbmUpO1xuICBDaGVja1RvcGljS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5ub25lKTtcbn0pO1xuXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgd2l0aCBpbXBvcnRlZCBlbmNyeXB0aW9uIGtleSwgb2xkIGludGVyZmFjZScsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIC8vIFNldHVwXG4gIGNvbnN0IGttc0tleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAnaW1wb3J0ZWQta2V5Jywge1xuICAgIGVuYWJsZUtleVJvdGF0aW9uOiBmYWxzZVxuICB9KTtcbiAgLy8gSGVscGVyIGRlY2xhcmF0aW9uXG4gIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Nucy10by1zcXMtc3RhY2snLCB7XG4gICAgZW5hYmxlRW5jcnlwdGlvbldpdGhDdXN0b21lck1hbmFnZWRLZXk6IHRydWUsXG4gICAgZW5jcnlwdGlvbktleToga21zS2V5XG4gIH0pO1xuICAvLyBBc3NlcnRpb24gMlxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG4gIENoZWNrVG9waWNLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpLTVM6OktleVwiLCB7XG4gICAgRW5hYmxlS2V5Um90YXRpb246IGZhbHNlXG4gIH0pO1xuICAvLyBBc3NlcnRpb24gM1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OlNOUzo6VG9waWNcIiwge1xuICAgIEttc01hc3RlcktleUlkOiB7XG4gICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcImltcG9ydGVka2V5Mzg2NzVENjhcIixcbiAgICAgICAgXCJBcm5cIlxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnVGVzdCBkZXBsb3ltZW50IHdpdGggU05TIG1hbmFnZWQgS01TIGtleSwgb2xkIGludGVyZmFjZScsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBuZXcgU25zVG9TcXMoc3RhY2ssICdzbnMtdG8tc3FzLXN0YWNrJywge1xuICAgIHRvcGljUHJvcHM6IHtcbiAgICAgIG1hc3RlcktleToga21zLkFsaWFzLmZyb21BbGlhc05hbWUoc3RhY2ssICdzbnMtbWFuYWdlZC1rZXknLCAnYWxpYXMvYXdzL3NucycpXG4gICAgfSxcbiAgICBxdWV1ZVByb3BzOiB7XG4gICAgICBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLktNU1xuICAgIH0sXG4gICAgZW5hYmxlRW5jcnlwdGlvbldpdGhDdXN0b21lck1hbmFnZWRLZXk6IGZhbHNlXG4gIH0pO1xuICAvLyBBc3NlcnRpb24gMlxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG5cbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuc3NlKTtcblxufSk7XG5cbnRlc3QoJ1Rlc3QgZGVwbG95bWVudCB3aXRoIENNSyBlbmNyeXB0ZWQgU05TIFRvcGljIChhdm9pZHMgaW50ZXJmYWNlKScsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHRlc3REZXNjcmlwdGlvbiA9ICdzb21lU3RyaW5nLWxsYXNkaic7XG4gIGNvbnN0IGNtayA9IG5ldyBrbXMuS2V5KHN0YWNrLCAnY21rJywgeyBkZXNjcmlwdGlvbjogdGVzdERlc2NyaXB0aW9uIH0pO1xuICAvLyBIZWxwZXIgZGVjbGFyYXRpb25cbiAgbmV3IFNuc1RvU3FzKHN0YWNrLCAnc25zLXRvLXNxcy1zdGFjaycsIHtcbiAgICB0b3BpY1Byb3BzOiB7XG4gICAgICBtYXN0ZXJLZXk6IGNta1xuICAgIH1cbiAgfSk7XG4gIC8vIEFzc2VydGlvbiAxXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICBDaGVja1F1ZXVlS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuICBDaGVja1RvcGljS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6S01TOjpLZXlcIiwge1xuICAgIERlc2NyaXB0aW9uOiB0ZXN0RGVzY3JpcHRpb25cbiAgfSk7XG4gIC8vIEEga2V5IGlzIHN0aWxsIGNyZWF0ZWQgZm9yIHRoZSBTUVMgcXVldWUsIHNvIHRoZXJlIGFyZSAyIGtleXMgaW4gdGhlIHN0YWNrXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMik7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6U05TOjpUb3BpY1wiLCAxKTtcbn0pO1xuXG50ZXN0KCdQYXR0ZXJuIGRlcGxveW1lbnQgdy8gZXhpc3RpbmcgdG9waWMgYW5kIEZJRk8gcXVldWUnLCAoKSA9PiB7XG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHRvcGljID0gbmV3IHNucy5Ub3BpYyhzdGFjaywgJ1Rlc3RUb3BpYycsIHtcbiAgICBjb250ZW50QmFzZWREZWR1cGxpY2F0aW9uOiB0cnVlLFxuICAgIGRpc3BsYXlOYW1lOiAnQ3VzdG9tZXIgc3Vic2NyaXB0aW9uIHRvcGljJyxcbiAgICBmaWZvOiB0cnVlLFxuICAgIHRvcGljTmFtZTogJ2N1c3RvbWVyVG9waWMnLFxuICB9KTtcblxuICBjb25zdCBwcm9wczogU25zVG9TcXNQcm9wcyA9IHtcbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogZmFsc2UsXG4gICAgZXhpc3RpbmdUb3BpY09iajogdG9waWMsXG4gICAgcXVldWVQcm9wczoge1xuICAgICAgZW5jcnlwdGlvbjogc3FzLlF1ZXVlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCxcbiAgICAgIGZpZm86IHRydWUsXG4gICAgfSxcbiAgICBkZWFkTGV0dGVyUXVldWVQcm9wczoge1xuICAgICAgZW5jcnlwdGlvbjogc3FzLlF1ZXVlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCxcbiAgICAgIGZpZm86IHRydWUsXG4gICAgfVxuICB9O1xuXG4gIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Rlc3Qtc25zLXNxcycsIHByb3BzKTtcblxuICAvLyBBc3NlcnRpb24gMVxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG5cbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG4gIENoZWNrVG9waWNLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLm5vbmUpO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMCk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpTUVM6OlF1ZXVlXCIsIHtcbiAgICBGaWZvUXVldWU6IHRydWUsXG4gICAgUmVkcml2ZVBvbGljeToge1xuICAgICAgZGVhZExldHRlclRhcmdldEFybjoge1xuICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgIFwidGVzdHNuc3Nxc3F1ZXVlZGxxMzMzMTMxMkJcIixcbiAgICAgICAgICBcIkFyblwiXG4gICAgICAgIF1cbiAgICAgIH0sXG4gICAgICBtYXhSZWNlaXZlQ291bnQ6IDE1XG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KCdQYXR0ZXJuIGRlcGxveW1lbnQgdy8gZXhpc3RpbmcgdG9waWMgYW5kIFN0YW5kYXJkIHF1ZXVlJywgKCkgPT4ge1xuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCB0b3BpYyA9IG5ldyBzbnMuVG9waWMoc3RhY2ssICdUZXN0VG9waWMnLCB7XG4gICAgZGlzcGxheU5hbWU6ICdDdXN0b21lciBzdWJzY3JpcHRpb24gdG9waWMnLFxuICAgIGZpZm86IGZhbHNlLFxuICAgIHRvcGljTmFtZTogJ2N1c3RvbWVyVG9waWMnLFxuICB9KTtcblxuICBjb25zdCBwcm9wczogU25zVG9TcXNQcm9wcyA9IHtcbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogZmFsc2UsXG4gICAgZXhpc3RpbmdUb3BpY09iajogdG9waWMsXG4gICAgcXVldWVQcm9wczoge1xuICAgICAgZW5jcnlwdGlvbjogc3FzLlF1ZXVlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCxcbiAgICAgIGZpZm86IGZhbHNlLFxuICAgIH0sXG4gICAgZGVhZExldHRlclF1ZXVlUHJvcHM6IHtcbiAgICAgIGVuY3J5cHRpb246IHNxcy5RdWV1ZUVuY3J5cHRpb24uVU5FTkNSWVBURUQsXG4gICAgICBmaWZvOiBmYWxzZSxcbiAgICB9XG4gIH07XG5cbiAgbmV3IFNuc1RvU3FzKHN0YWNrLCAndGVzdC1zbnMtc3FzJywgcHJvcHMpO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICBDaGVja1F1ZXVlS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5ub25lKTtcbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpTUVM6OlF1ZXVlXCIsIHtcbiAgICBSZWRyaXZlUG9saWN5OiB7XG4gICAgICBkZWFkTGV0dGVyVGFyZ2V0QXJuOiB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgXCJ0ZXN0c25zc3FzcXVldWVkbHEzMzMxMzEyQlwiLFxuICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgXVxuICAgICAgfSxcbiAgICAgIG1heFJlY2VpdmVDb3VudDogMTVcbiAgICB9XG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NoZWNrIHJhdyBtZXNzYWdlIGRlbGl2ZXJ5IGlzIHRydWUnLCAoKSA9PiB7XG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHByb3BzOiBTbnNUb1Nxc1Byb3BzID0ge1xuICAgIHNxc1N1YnNjcmlwdGlvblByb3BzOiB7XG4gICAgICByYXdNZXNzYWdlRGVsaXZlcnk6IHRydWVcbiAgICB9XG4gIH07XG4gIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Rlc3Qtc25zLXNxcycsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG5cbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OlNOUzo6U3Vic2NyaXB0aW9uXCIsIHtcbiAgICBQcm90b2NvbDogXCJzcXNcIixcbiAgICBUb3BpY0Fybjoge1xuICAgICAgUmVmOiBcInRlc3RzbnNzcXNTbnNUb3BpYzJDRDAwNjVCXCJcbiAgICB9LFxuICAgIEVuZHBvaW50OiB7XG4gICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcInRlc3RzbnNzcXNxdWV1ZUIwMjUwNEJGXCIsXG4gICAgICAgIFwiQXJuXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFJhd01lc3NhZ2VEZWxpdmVyeTogdHJ1ZVxuICB9KTtcbn0pO1xuXG50ZXN0KCdDaGVjayBmb3IgZmlsdGVyIHBvbGljeScsICgpID0+IHtcbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgcHJvcHM6IFNuc1RvU3FzUHJvcHMgPSB7XG4gICAgc3FzU3Vic2NyaXB0aW9uUHJvcHM6IHtcbiAgICAgIGZpbHRlclBvbGljeToge1xuICAgICAgICBjb2xvcjogc25zLlN1YnNjcmlwdGlvbkZpbHRlci5zdHJpbmdGaWx0ZXIoe1xuICAgICAgICAgIGFsbG93bGlzdDogWydyZWQnLCAnb3JhbmdlJ10sXG4gICAgICAgICAgbWF0Y2hQcmVmaXhlczogWydibCddLFxuICAgICAgICB9KVxuICAgICAgfVxuICAgIH1cbiAgfTtcblxuICBuZXcgU25zVG9TcXMoc3RhY2ssICd0ZXN0LXNucy1zcXMnLCBwcm9wcyk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG4gIENoZWNrVG9waWNLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpTTlM6OlN1YnNjcmlwdGlvblwiLCB7XG4gICAgRmlsdGVyUG9saWN5OiB7XG4gICAgICBjb2xvcjogW1xuICAgICAgICBcInJlZFwiLFxuICAgICAgICBcIm9yYW5nZVwiLFxuICAgICAgICB7XG4gICAgICAgICAgcHJlZml4OiBcImJsXCJcbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnQ2hlY2sgU05TIGRlYWQgbGV0dGVyIHF1ZXVlJywgKCkgPT4ge1xuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCBkbHEgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnZXhpc3RpbmctZGxxLW9iaicsIHt9KTtcbiAgY29uc3QgcHJvcHM6IFNuc1RvU3FzUHJvcHMgPSB7XG4gICAgc3FzU3Vic2NyaXB0aW9uUHJvcHM6IHtcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZTogZGxxXG4gICAgfVxuICB9O1xuXG4gIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Rlc3Qtc25zLXNxcycsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG5cbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OlNOUzo6U3Vic2NyaXB0aW9uXCIsIHtcbiAgICBSZWRyaXZlUG9saWN5OiB7XG4gICAgICBkZWFkTGV0dGVyVGFyZ2V0QXJuOiB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgXCJleGlzdGluZ2RscW9iajc4NEM1NTQyXCIsXG4gICAgICAgICAgXCJBcm5cIlxuICAgICAgICBdXG4gICAgICB9XG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KCdDb25zdHJ1Y3QgdXNlcyB0b3BpY1Byb3BzLm1hc3RlcktleSB3aGVuIHNwZWNpZmllZCAoYXZvaWRzIGludGVyZmFjZSknLCAoKSA9PiB7XG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IGNtayA9IG5ldyBrbXMuS2V5KHN0YWNrLCAnY21rZm9ydG9waWMnKTtcbiAgY29uc3QgcHJvcHM6IFNuc1RvU3FzUHJvcHMgPSB7XG4gICAgdG9waWNQcm9wczoge1xuICAgICAgbWFzdGVyS2V5OiBjbWtcbiAgICB9XG4gIH07XG5cbiAgbmV3IFNuc1RvU3FzKHN0YWNrLCAndGVzdC1zbnMtc3FzJywgcHJvcHMpO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpTTlM6OlRvcGljXCIsIHtcbiAgICBLbXNNYXN0ZXJLZXlJZDoge1xuICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgXCJjbWtmb3J0b3BpYzA5MDQ2NDdCXCIsXG4gICAgICAgIFwiQXJuXCJcbiAgICAgIF1cbiAgICB9XG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NvbnN0cnVjdCB1c2VzIHF1ZXVlUHJvcHMuZW5jcnlwdGlvbk1hc3RlcktleSB3aGVuIHNwZWNpZmllZCAoYXZvaWRzIGludGVyZmFjZSknLCAoKSA9PiB7XG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IGNtayA9IG5ldyBrbXMuS2V5KHN0YWNrLCAnY21rZm9ycXVldWUnKTtcbiAgY29uc3QgcHJvcHM6IFNuc1RvU3FzUHJvcHMgPSB7XG4gICAgcXVldWVQcm9wczoge1xuICAgICAgZW5jcnlwdGlvbk1hc3RlcktleTogY21rXG4gICAgfVxuICB9O1xuXG4gIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Rlc3Qtc25zLXNxcycsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6U1FTOjpRdWV1ZVwiLCB7XG4gICAgS21zTWFzdGVyS2V5SWQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwiY21rZm9ycXVldWU0RTA5MzQ3NlwiLFxuICAgICAgICBcIkFyblwiXG4gICAgICBdXG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KCdDb25zdHJ1Y3QgZG9lcyBub3Qgb3ZlcnJpZGUgdW5lbmNyeXB0ZWQgdG9waWMgd2hlbiBwYXNzZWQgaW4gZXhpc3RpbmdUb3BpY09iaiBwcm9wJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgZXhpc3RpbmdUb3BpY09iaiA9IG5ldyBzbnMuVG9waWMoc3RhY2ssICdUb3BpYycsIHtcbiAgICB0b3BpY05hbWU6ICdleGlzdGluZy10b3BpYy1uYW1lJ1xuICB9KTtcblxuICBjb25zdCBwcm9wczogU25zVG9TcXNQcm9wcyA9IHtcbiAgICBleGlzdGluZ1RvcGljT2JqLFxuICB9O1xuXG4gIGNvbnN0IHRlc3RDb25zdHJ1Y3QgPSBuZXcgU25zVG9TcXMoc3RhY2ssICd0ZXN0LXNucy1zcXMnLCBwcm9wcyk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICBleHBlY3QodGVzdENvbnN0cnVjdC5zbnNUb3BpYykudG9CZURlZmluZWQoKTtcbiAgLy8gRW5zdXJlIHRoYXQgdGhlIGV4aXN0aW5nIHRvcGljIGFib3ZlIGlzIHRoZSBvbmx5IHRvcGljIGluIHRoZSBzdGFja1xuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OlNOUzo6VG9waWNcIiwgMSk7XG5cbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG5cbn0pO1xuXG50ZXN0KCdDb25maXJtIHRoYXQgQ2hlY2tTbnNQcm9wcyBpcyBjYWxsZWQnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBjb25zdCB0b3BpYyA9IG5ldyBzbnMuVG9waWMoc3RhY2ssIFwiZXhpc3RpbmctdG9waWMtb2JqXCIsIHtcbiAgICB0b3BpY05hbWU6ICdleGlzdGluZy10b3BpYy1vYmonXG4gIH0pO1xuICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdleGlzdGluZy1xdWV1ZS1vYmonLCB7XG4gICAgcXVldWVOYW1lOiAnZXhpc3RpbmctcXVldWUtb2JqJ1xuICB9KTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgbmV3IFNuc1RvU3FzKHN0YWNrLCAnc25zLXRvLXNxcy1zdGFjaycsIHtcbiAgICAgIGV4aXN0aW5nVG9waWNPYmo6IHRvcGljLFxuICAgICAgdG9waWNQcm9wczoge1xuICAgICAgICB0b3BpY05hbWU6ICd0b3BpYy1uYW1lJ1xuICAgICAgfSxcbiAgICAgIGV4aXN0aW5nUXVldWVPYmo6IHF1ZXVlXG4gICAgfSk7XG4gIH07XG5cbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigvRXJyb3IgLSBFaXRoZXIgcHJvdmlkZSB0b3BpY1Byb3BzIG9yIGV4aXN0aW5nVG9waWNPYmosIGJ1dCBub3QgYm90aC5cXG4vKTtcbn0pO1xuXG50ZXN0KCdDb25maXJtIHRoYXQgQ2hlY2tTcXNQcm9wcyBpcyBjYWxsZWQnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Nucy10by1zcXMtc3RhY2snLCB7XG4gICAgICBxdWV1ZVByb3BzOiB7XG4gICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIH0sXG4gICAgICBleGlzdGluZ1F1ZXVlT2JqOiBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAndGVzdCcsIHt9KVxuICAgIH0pO1xuICB9O1xuXG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoXCJFcnJvciAtIEVpdGhlciBwcm92aWRlIHF1ZXVlUHJvcHMgb3IgZXhpc3RpbmdRdWV1ZU9iaiwgYnV0IG5vdCBib3RoLlxcblwiKTtcbn0pO1xuXG50ZXN0KCdDb25maXJtIHRoYXQgQ2hlY2tTbnNQcm9wcyBpcyBjYWxsZWQnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBjb25zdCB0b3BpYyA9IG5ldyBzbnMuVG9waWMoc3RhY2ssIFwiZXhpc3RpbmctdG9waWMtb2JqXCIsIHtcbiAgICB0b3BpY05hbWU6ICdleGlzdGluZy10b3BpYy1vYmonXG4gIH0pO1xuICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdleGlzdGluZy1xdWV1ZS1vYmonLCB7XG4gICAgcXVldWVOYW1lOiAnZXhpc3RpbmctcXVldWUtb2JqJ1xuICB9KTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgbmV3IFNuc1RvU3FzKHN0YWNrLCAnc25zLXRvLXNxcy1zdGFjaycsIHtcbiAgICAgIGV4aXN0aW5nVG9waWNPYmo6IHRvcGljLFxuICAgICAgdG9waWNQcm9wczoge1xuICAgICAgICB0b3BpY05hbWU6ICd0b3BpYy1uYW1lJ1xuICAgICAgfSxcbiAgICAgIGV4aXN0aW5nUXVldWVPYmo6IHF1ZXVlXG4gICAgfSk7XG4gIH07XG5cbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigvRXJyb3IgLSBFaXRoZXIgcHJvdmlkZSB0b3BpY1Byb3BzIG9yIGV4aXN0aW5nVG9waWNPYmosIGJ1dCBub3QgYm90aC5cXG4vKTtcbn0pO1xuXG50ZXN0KCdDb25maXJtIHRoYXQgQ2hlY2tTcXNQcm9wcyBpcyBjYWxsZWQnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Nucy10by1zcXMtc3RhY2snLCB7XG4gICAgICBxdWV1ZVByb3BzOiB7XG4gICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIH0sXG4gICAgICBleGlzdGluZ1F1ZXVlT2JqOiBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAndGVzdCcsIHt9KVxuICAgIH0pO1xuICB9O1xuXG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoXCJFcnJvciAtIEVpdGhlciBwcm92aWRlIHF1ZXVlUHJvcHMgb3IgZXhpc3RpbmdRdWV1ZU9iaiwgYnV0IG5vdCBib3RoLlxcblwiKTtcbn0pO1xuXG50ZXN0KCdDb25maXJtIHRoYXQgQ29uc3RydWN0IGNoZWNrcyBmb3IgbWl4ZWQgZGVwcmVjYXRlZCBhbmQgYWN0aXZlIHByb3BzJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBTbnNUb1NxcyhzdGFjaywgJ3Nucy10by1zcXMtc3RhY2snLCB7XG4gICAgICBxdWV1ZVByb3BzOiB7XG4gICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIH0sXG4gICAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZSxcbiAgICAgIGVuY3J5cHRRdWV1ZVdpdGhDbWs6IHRydWVcbiAgICB9KTtcbiAgfTtcblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoL0Nhbm5vdCBzcGVjaWZ5IGJvdGggZGVwcmVjYXRlZCBrZXkgcHJvcHMgYW5kIG5ldyBrZXkgcHJvcHMvKTtcbn0pO1xuXG50ZXN0KCdFcnJvciBpZiBlbmFibGVFbmNyeXB0aW9uIGlzIGZhbHNlIGFuZCBlbmNyeXB0aW9uIHNldHRpbmdzIGFyZSBwcm92aWRlZCcsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgbmV3IFNuc1RvU3FzKHN0YWNrLCAnc25zLXRvLXNxcy1zdGFjaycsIHtcbiAgICAgIGVuYWJsZUVuY3J5cHRpb25XaXRoQ3VzdG9tZXJNYW5hZ2VkS2V5OiBmYWxzZSxcbiAgICAgIGVuY3J5cHRpb25LZXlQcm9wczoge31cbiAgICB9KTtcbiAgfTtcblxuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKFwiRXJyb3IgLSBpZiBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleSBpcyBmYWxzZSwgc3VibWl0dGluZyBlbmNyeXB0aW9uS2V5IG9yIGVuY3J5cHRpb25LZXlQcm9wcyBpcyBpbnZhbGlkXFxuXCIpO1xufSk7XG5cbnRlc3QoJ3Rlc3QgQ3JlYXRlUmVxdWlyZWRLZXlzIGZvciBubyBhcmd1bWVudHMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCByZXN1bHQgPSBTbnNUb1Nxcy5jcmVhdGVSZXF1aXJlZEtleXMoc3RhY2ssICd0ZXN0Jywge30pO1xuXG4gIGV4cGVjdChyZXN1bHQudXNlRGVwcmVjYXRlZEludGVyZmFjZSkudG9CZVRydXRoeSgpO1xuICBleHBlY3QocmVzdWx0LmVuY3J5cHRRdWV1ZVdpdGhDbWspLnRvQmVUcnV0aHkoKTtcbiAgZXhwZWN0KHJlc3VsdC5lbmNyeXB0VG9waWNXaXRoQ21rKS50b0JlVHJ1dGh5KCk7XG4gIGV4cGVjdChyZXN1bHQucXVldWVLZXkpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChyZXN1bHQudG9waWNLZXkpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChyZXN1bHQuc2luZ2xlS2V5KS50b0JlRGVmaW5lZCgpO1xufSk7XG5cbnRlc3QoJ3Rlc3QgQ3JlYXRlUmVxdWlyZWRLZXlzIHdoZW4gVG9waWMgaXMgcHJvdmlkZWQnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCByZXN1bHQgPSBTbnNUb1Nxcy5jcmVhdGVSZXF1aXJlZEtleXMoc3RhY2ssICd0ZXN0Jywge1xuICAgIGV4aXN0aW5nVG9waWNPYmo6IHt9IGFzIHNucy5Ub3BpY1xuICB9KTtcblxuICBleHBlY3QocmVzdWx0LnVzZURlcHJlY2F0ZWRJbnRlcmZhY2UpLnRvQmVUcnV0aHkoKTtcbiAgZXhwZWN0KHJlc3VsdC5lbmNyeXB0UXVldWVXaXRoQ21rKS50b0JlVHJ1dGh5KCk7XG4gIGV4cGVjdChyZXN1bHQuZW5jcnlwdFRvcGljV2l0aENtaykudG9CZUZhbHN5KCk7XG4gIGV4cGVjdChyZXN1bHQucXVldWVLZXkpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChyZXN1bHQudG9waWNLZXkpLm5vdC50b0JlRGVmaW5lZCgpO1xuICBleHBlY3QocmVzdWx0LnNpbmdsZUtleSkudG9CZURlZmluZWQoKTtcbn0pO1xuXG50ZXN0KCd0ZXN0IENyZWF0ZVJlcXVpcmVkS2V5cyB3aGVuIFF1ZXVlIGlzIHByb3ZpZGVkJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgcmVzdWx0ID0gU25zVG9TcXMuY3JlYXRlUmVxdWlyZWRLZXlzKHN0YWNrLCAndGVzdCcsIHtcbiAgICBleGlzdGluZ1F1ZXVlT2JqOiB7fSBhcyBzcXMuUXVldWVcbiAgfSk7XG5cbiAgZXhwZWN0KHJlc3VsdC51c2VEZXByZWNhdGVkSW50ZXJmYWNlKS50b0JlVHJ1dGh5KCk7XG4gIGV4cGVjdChyZXN1bHQuZW5jcnlwdFF1ZXVlV2l0aENtaykudG9CZUZhbHN5KCk7XG4gIGV4cGVjdChyZXN1bHQuZW5jcnlwdFRvcGljV2l0aENtaykudG9CZVRydXRoeSgpO1xuICBleHBlY3QocmVzdWx0LnF1ZXVlS2V5KS5ub3QudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHJlc3VsdC50b3BpY0tleSkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHJlc3VsdC5zaW5nbGVLZXkpLnRvQmVEZWZpbmVkKCk7XG59KTtcblxudGVzdCgndGVzdCBDcmVhdGVSZXF1aXJlZEtleXMgd2hlbiBUb3BpYyBwcm9wcyBoYXZlIGEga2V5JywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgcmVzdWx0ID0gU25zVG9TcXMuY3JlYXRlUmVxdWlyZWRLZXlzKHN0YWNrLCAndGVzdCcsIHtcbiAgICB0b3BpY1Byb3BzOiB7XG4gICAgICBtYXN0ZXJLZXk6IHt9IGFzIGttcy5LZXlcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChyZXN1bHQudXNlRGVwcmVjYXRlZEludGVyZmFjZSkudG9CZVRydXRoeSgpO1xuICBleHBlY3QocmVzdWx0LmVuY3J5cHRRdWV1ZVdpdGhDbWspLnRvQmVUcnV0aHkoKTtcbiAgZXhwZWN0KHJlc3VsdC5lbmNyeXB0VG9waWNXaXRoQ21rKS50b0JlRmFsc3koKTtcbiAgZXhwZWN0KHJlc3VsdC5xdWV1ZUtleSkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHJlc3VsdC50b3BpY0tleSkubm90LnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChyZXN1bHQuc2luZ2xlS2V5KS50b0JlRGVmaW5lZCgpO1xufSk7XG5cbnRlc3QoJ3Rlc3QgQ3JlYXRlUmVxdWlyZWRLZXlzIHdoZW4gUXVldWUgcHJvcHMgaGF2ZSBhIGtleScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHJlc3VsdCA9IFNuc1RvU3FzLmNyZWF0ZVJlcXVpcmVkS2V5cyhzdGFjaywgJ3Rlc3QnLCB7XG4gICAgcXVldWVQcm9wczoge1xuICAgICAgZW5jcnlwdGlvbk1hc3RlcktleToge30gYXMga21zLktleVxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KHJlc3VsdC51c2VEZXByZWNhdGVkSW50ZXJmYWNlKS50b0JlVHJ1dGh5KCk7XG4gIGV4cGVjdChyZXN1bHQuZW5jcnlwdFF1ZXVlV2l0aENtaykudG9CZUZhbHN5KCk7XG4gIGV4cGVjdChyZXN1bHQuZW5jcnlwdFRvcGljV2l0aENtaykudG9CZVRydXRoeSgpO1xuICBleHBlY3QocmVzdWx0LnF1ZXVlS2V5KS5ub3QudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHJlc3VsdC50b3BpY0tleSkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHJlc3VsdC5zaW5nbGVLZXkpLnRvQmVEZWZpbmVkKCk7XG59KTtcblxuLy8gKioqKioqKioqKioqKioqKioqKioqKioqKioqXG4vL1xuLy8gT3ZlciB0aGUgY291cnNlIG9mIGltcGxlbWVudGluZyB0aGUgbmV3IGludGVyZmFjZSBhbmQga2VlcGluZyB0aGUgb2xkIGludGVyZmFjZSBjYXBhYmlsaXR5LFxuLy8gaXQgYmVjYW1lIGNsZWFyIHRoYXQgdGhlIG9sZCBpbnRlcmZhY2UgaGFkIHNvbWUgaXJyZWd1bGFyIGJlaGF2aW9yIHRoYXQgd2UgZGlkIG5vdCB3YW50IHRvIGFsdGVyLlxuLy8gU28gd2UgbWFkZSBhIHRhYmxlIG9mIGJlaGF2aW9yIGZvciBkaWZmZXJlbnQgaW5wdXRzLCB0aGVuIHJhbiBhIHRlc3QgZm9yIGVhY2ggcm93IG9mIHRoYXQgdGFibGVcbi8vIHRvIGNhcHR1cmUgdGhlIGN1cnJlbnQgYmVoYXZpb3IuIFRoZXNlIGFyZSB0aG9zZSB0ZXN0cyB0aGF0IHByb3RlY3QgZXhpc3RpbmcgYmVoYXZpb3IuIFNvbWVcbi8vIHJlcGVhdCB0ZXN0cyBhbHJlYWR5IGltcGxlbWVudGVkIGFib3ZlLCB3ZSBjaG9zZSBub3QgdG8gd29ycnkgYWJvdXQgdGhhdC5cbi8vXG4vLyAqKioqKioqKioqKioqKioqKioqKioqKioqKipcblxudGVzdCgnMSBMZWdhY3kgQmVoYXZpb3IgLSBRdWV1ZSBQcm9wcywgRW5jcnlwdEZsYWcgRmFsc2UnLCAoKSA9PiB7XG5cbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgZXhpc3RpbmdLZXkgPSBuZXcga21zLktleShzdGFjaywgJ3Rlc3QtZXhpc3Rpbmcta2V5Jywge1xuICAgIGVuYWJsZUtleVJvdGF0aW9uOiB0cnVlLFxuICAgIGFsaWFzOiAnZXhpc3RpbmdLZXknXG4gIH0pO1xuXG4gIGNvbnN0IHNuc1RvU3FzU3RhY2sgPSBuZXcgU25zVG9TcXMoc3RhY2ssICdTbnNUb1Nxc1BhdHRlcm4nLCB7XG5cbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogZmFsc2UsXG5cbiAgICBxdWV1ZVByb3BzOiB7XG4gICAgICBlbmNyeXB0aW9uTWFzdGVyS2V5OiBleGlzdGluZ0tleSxcbiAgICB9XG4gIH0pO1xuXG4gIENoZWNrS2V5UHJvcGVydHkoc25zVG9TcXNTdGFjay5lbmNyeXB0aW9uS2V5LCBrZXlUeXBlLnNzZSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMSk7XG5cbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuc3NlKTtcbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcblxufSk7XG5cbnRlc3QoJzIgTGVnYWN5IEJlaGF2aW9yIC0gUXVldWUgUHJvcHMsIEVuY3J5cHRGbGFnIFRydWUnLCAoKSA9PiB7XG5cbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgZXhpc3RpbmdLZXkgPSBuZXcga21zLktleShzdGFjaywgJ3Rlc3QtZXhpc3Rpbmcta2V5Jywge1xuICAgIGVuYWJsZUtleVJvdGF0aW9uOiB0cnVlLFxuICAgIGFsaWFzOiAnZXhpc3RpbmdLZXknXG4gIH0pO1xuXG4gIGNvbnN0IHNuc1RvU3FzU3RhY2sgPSBuZXcgU25zVG9TcXMoc3RhY2ssICdTbnNUb1Nxc1BhdHRlcm4nLCB7XG5cbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZSxcblxuICAgIHF1ZXVlUHJvcHM6IHtcbiAgICAgIGVuY3J5cHRpb25NYXN0ZXJLZXk6IGV4aXN0aW5nS2V5LFxuICAgIH1cbiAgfSk7XG5cbiAgQ2hlY2tLZXlQcm9wZXJ0eShzbnNUb1Nxc1N0YWNrLmVuY3J5cHRpb25LZXksIGtleVR5cGUuY21rKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG5cbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpLTVM6OktleVwiLCAyKTtcblxuICBDaGVja1RvcGljS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuICBDaGVja1F1ZXVlS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuXG59KTtcblxudGVzdCgnNCBMZWdhY3kgQmVoYXZpb3IgLSBUb3BpYyBQcm9wcywgRW5jcnlwdEZsYWcgVHJ1ZScsICgpID0+IHtcblxuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCBleGlzdGluZ0tleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAndGVzdC1leGlzdGluZy1rZXknLCB7XG4gICAgZW5hYmxlS2V5Um90YXRpb246IHRydWUsXG4gICAgYWxpYXM6ICdleGlzdGluZ0tleSdcbiAgfSk7XG5cbiAgY29uc3Qgc25zVG9TcXNTdGFjayA9IG5ldyBTbnNUb1NxcyhzdGFjaywgJ1Nuc1RvU3FzUGF0dGVybicsIHtcblxuICAgIGVuYWJsZUVuY3J5cHRpb25XaXRoQ3VzdG9tZXJNYW5hZ2VkS2V5OiB0cnVlLFxuXG4gICAgdG9waWNQcm9wczoge1xuICAgICAgbWFzdGVyS2V5OiBleGlzdGluZ0tleSxcbiAgICB9XG4gIH0pO1xuXG4gIENoZWNrS2V5UHJvcGVydHkoc25zVG9TcXNTdGFjay5lbmNyeXB0aW9uS2V5LCBrZXlUeXBlLmNtayk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMik7XG5cbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcblxufSk7XG5cbnRlc3QoJzUgTGVnYWN5IEJlaGF2aW9yIC0gVG9waWMgUHJvcHMsIEV4aXN0aW5nIEtleScsICgpID0+IHtcblxuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCBleGlzdGluZ0tleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAndGVzdC1leGlzdGluZy1rZXknLCB7XG4gICAgZW5hYmxlS2V5Um90YXRpb246IHRydWUsXG4gICAgYWxpYXM6ICdleGlzdGluZ0tleSdcbiAgfSk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBTbnNUb1NxcyhzdGFjaywgJ1Nuc1RvU3FzUGF0dGVybicsIHtcblxuICAgICAgZW5jcnlwdGlvbktleTogZXhpc3RpbmdLZXksXG5cbiAgICAgIHRvcGljUHJvcHM6IHtcbiAgICAgICAgbWFzdGVyS2V5OiBleGlzdGluZ0tleSxcbiAgICAgIH1cbiAgICB9KTtcbiAgfTtcblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG50ZXN0KCc2IExlZ2FjeSBCZWhhdmlvciAtIFF1ZXVlIFByb3BzLCBFeGlzdGluZyBLZXknLCAoKSA9PiB7XG5cbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgZXhpc3RpbmdLZXkgPSBuZXcga21zLktleShzdGFjaywgJ3Rlc3QtZXhpc3Rpbmcta2V5Jywge1xuICAgIGVuYWJsZUtleVJvdGF0aW9uOiB0cnVlLFxuICAgIGFsaWFzOiAnZXhpc3RpbmdLZXknXG4gIH0pO1xuXG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgU25zVG9TcXMoc3RhY2ssICdTbnNUb1Nxc1BhdHRlcm4nLCB7XG5cbiAgICAgIGVuY3J5cHRpb25LZXk6IGV4aXN0aW5nS2V5LFxuXG4gICAgICBxdWV1ZVByb3BzOiB7XG4gICAgICAgIGVuY3J5cHRpb25NYXN0ZXJLZXk6IGV4aXN0aW5nS2V5LFxuICAgICAgfVxuICAgIH0pO1xuICB9O1xuXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigpO1xufSk7XG5cbnRlc3QoJzcgTGVnYWN5IEJlaGF2aW9yIC0gUXVldWUgUHJvcHMsIEtleSBQcm9wcyBwcm92aWRlZCcsICgpID0+IHtcblxuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCBleGlzdGluZ0tleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAndGVzdC1leGlzdGluZy1rZXknLCB7XG4gICAgZW5hYmxlS2V5Um90YXRpb246IHRydWUsXG4gICAgYWxpYXM6ICdleGlzdGluZ0tleSdcbiAgfSk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBTbnNUb1NxcyhzdGFjaywgJ1Nuc1RvU3FzUGF0dGVybicsIHtcblxuICAgICAgZW5jcnlwdGlvbktleVByb3BzOiB7fSxcblxuICAgICAgcXVldWVQcm9wczoge1xuICAgICAgICBlbmNyeXB0aW9uTWFzdGVyS2V5OiBleGlzdGluZ0tleSxcbiAgICAgIH1cbiAgICB9KTtcbiAgfTtcblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG50ZXN0KCc4IExlZ2FjeSBCZWhhdmlvciAtIFRvcGljIFByb3BzLCBRdWV1ZSBQcm9wcywgRW5jcnlwdEZsYWcgVHJ1ZScsICgpID0+IHtcblxuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCBleGlzdGluZ0tleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAndGVzdC1leGlzdGluZy1rZXknLCB7XG4gICAgZW5hYmxlS2V5Um90YXRpb246IHRydWUsXG4gICAgYWxpYXM6ICdleGlzdGluZ0tleSdcbiAgfSk7XG5cbiAgY29uc3QgZXhpc3RpbmdLZXlUd28gPSBuZXcga21zLktleShzdGFjaywgJ3Rlc3QtZXhpc3Rpbmcta2V5LXR3bycsIHtcbiAgICBlbmFibGVLZXlSb3RhdGlvbjogdHJ1ZSxcbiAgICBhbGlhczogJ2V4aXN0aW5nS2V5VHdvJ1xuICB9KTtcblxuICBjb25zdCBzbnNUb1Nxc1N0YWNrID0gbmV3IFNuc1RvU3FzKHN0YWNrLCAnU25zVG9TcXNQYXR0ZXJuJywge1xuXG4gICAgZW5hYmxlRW5jcnlwdGlvbldpdGhDdXN0b21lck1hbmFnZWRLZXk6IHRydWUsXG5cbiAgICB0b3BpY1Byb3BzOiB7XG4gICAgICBtYXN0ZXJLZXk6IGV4aXN0aW5nS2V5LFxuICAgIH0sXG4gICAgcXVldWVQcm9wczoge1xuICAgICAgZW5jcnlwdGlvbk1hc3RlcktleTogZXhpc3RpbmdLZXlUd29cbiAgICB9XG4gIH0pO1xuXG4gIENoZWNrS2V5UHJvcGVydHkoc25zVG9TcXNTdGFjay5lbmNyeXB0aW9uS2V5LCBrZXlUeXBlLmNtayk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIC8vIExlZ2FjeSBjb2RlIGNyZWF0ZWQgYW4gZXh0cmEsIHVubmVlZGVkIGtleS4gTmV3XG4gIC8vIGNvZGUgcmV0dXJucyB0aGlzIHRvIDJcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpLTVM6OktleVwiLCAyKTtcblxuICBDaGVja1RvcGljS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuICBDaGVja1F1ZXVlS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuXG59KTtcblxudGVzdCgnOSBMZWdhY3kgQmVoYXZpb3IgLSBUb3BpYyBQcm9wcywgUXVldWUgUHJvcHMsIEVuY3J5cHRGbGFnIEZhbHNlJywgKCkgPT4ge1xuXG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IGV4aXN0aW5nS2V5ID0gbmV3IGttcy5LZXkoc3RhY2ssICd0ZXN0LWV4aXN0aW5nLWtleScsIHtcbiAgICBlbmFibGVLZXlSb3RhdGlvbjogdHJ1ZSxcbiAgICBhbGlhczogJ2V4aXN0aW5nS2V5J1xuICB9KTtcblxuICBjb25zdCBleGlzdGluZ0tleVR3byA9IG5ldyBrbXMuS2V5KHN0YWNrLCAndGVzdC1leGlzdGluZy1rZXktdHdvJywge1xuICAgIGVuYWJsZUtleVJvdGF0aW9uOiB0cnVlLFxuICAgIGFsaWFzOiAnZXhpc3RpbmdLZXlUd28nXG4gIH0pO1xuXG4gIGNvbnN0IHNuc1RvU3FzU3RhY2sgPSBuZXcgU25zVG9TcXMoc3RhY2ssICdTbnNUb1Nxc1BhdHRlcm4nLCB7XG5cbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogZmFsc2UsXG5cbiAgICB0b3BpY1Byb3BzOiB7XG4gICAgICBtYXN0ZXJLZXk6IGV4aXN0aW5nS2V5LFxuICAgIH0sXG4gICAgcXVldWVQcm9wczoge1xuICAgICAgZW5jcnlwdGlvbk1hc3RlcktleTogZXhpc3RpbmdLZXlUd29cbiAgICB9XG4gIH0pO1xuXG4gIENoZWNrS2V5UHJvcGVydHkoc25zVG9TcXNTdGFjay5lbmNyeXB0aW9uS2V5LCBrZXlUeXBlLmNtayk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMik7XG5cbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcblxufSk7XG5cbnRlc3QoJzEwIExlZ2FjeSBCZWhhdmlvciAtIEV4aXN0aW5nIFRvcGljIGFuZCBRdWV1ZSwgRW5jcnlwdEZsYWcgRmFsc2UnLCAoKSA9PiB7XG5cbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgdG9waWMgPSBuZXcgc25zLlRvcGljKHN0YWNrLCAndGVzdC10b3BpYycsIHt9KTtcbiAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAndGVzdC1xdWV1ZScsIHt9KTtcblxuICBjb25zdCBzbnNUb1Nxc1N0YWNrID0gbmV3IFNuc1RvU3FzKHN0YWNrLCAnU25zVG9TcXNQYXR0ZXJuJywge1xuXG4gICAgZW5hYmxlRW5jcnlwdGlvbldpdGhDdXN0b21lck1hbmFnZWRLZXk6IGZhbHNlLFxuXG4gICAgZXhpc3RpbmdRdWV1ZU9iajogcXVldWUsXG4gICAgZXhpc3RpbmdUb3BpY09iajogdG9waWNcbiAgfSk7XG5cbiAgQ2hlY2tLZXlQcm9wZXJ0eShzbnNUb1Nxc1N0YWNrLmVuY3J5cHRpb25LZXksIGtleVR5cGUubm9uZSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMCk7XG5cbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLm5vbmUpO1xuXG59KTtcblxudGVzdCgnMTEgTGVnYWN5IEJlaGF2aW9yIC0gRXhpc3RpbmcgVG9waWMgYW5kIFF1ZXVlLCBFbmNyeXB0RmxhZyBUcnVlJywgKCkgPT4ge1xuXG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHRvcGljID0gbmV3IHNucy5Ub3BpYyhzdGFjaywgJ3Rlc3QtdG9waWMnLCB7fSk7XG4gIGNvbnN0IHF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ3Rlc3QtcXVldWUnLCB7fSk7XG5cbiAgY29uc3Qgc25zVG9TcXNTdGFjayA9IG5ldyBTbnNUb1NxcyhzdGFjaywgJ1Nuc1RvU3FzUGF0dGVybicsIHtcblxuICAgIGVuYWJsZUVuY3J5cHRpb25XaXRoQ3VzdG9tZXJNYW5hZ2VkS2V5OiB0cnVlLFxuXG4gICAgZXhpc3RpbmdRdWV1ZU9iajogcXVldWUsXG4gICAgZXhpc3RpbmdUb3BpY09iajogdG9waWNcbiAgfSk7XG5cbiAgQ2hlY2tLZXlQcm9wZXJ0eShzbnNUb1Nxc1N0YWNrLmVuY3J5cHRpb25LZXksIGtleVR5cGUubm9uZSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIC8vIExlZ2FjeSBjb2RlIGNyZWF0ZWQgYW4gdW51c2VkIGtleSBoZXJlLCBuZXcgY29kZVxuICAvLyBkb2VzIG5vdCBzbyB0aGlzIGlzIDBcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpLTVM6OktleVwiLCAwKTtcblxuICBDaGVja1RvcGljS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5ub25lKTtcbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG5cbn0pO1xuXG50ZXN0KCcxMyBMZWdhY3kgQmVoYXZpb3IgLSBFeGlzdGluZyBUb3BpYywgRW5jcnlwdEZsYWcgVHJ1ZScsICgpID0+IHtcblxuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCB0b3BpYyA9IG5ldyBzbnMuVG9waWMoc3RhY2ssICd0ZXN0LXRvcGljJywge30pO1xuXG4gIGNvbnN0IHNuc1RvU3FzU3RhY2sgPSBuZXcgU25zVG9TcXMoc3RhY2ssICdTbnNUb1Nxc1BhdHRlcm4nLCB7XG5cbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZSxcblxuICAgIGV4aXN0aW5nVG9waWNPYmo6IHRvcGljXG4gIH0pO1xuXG4gIENoZWNrS2V5UHJvcGVydHkoc25zVG9TcXNTdGFjay5lbmNyeXB0aW9uS2V5LCBrZXlUeXBlLm5vbmUpO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OktNUzo6S2V5XCIsIDEpO1xuXG4gIENoZWNrVG9waWNLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLm5vbmUpO1xuICBDaGVja1F1ZXVlS2V5VHlwZSh0ZW1wbGF0ZSwga2V5VHlwZS5jbWspO1xuXG59KTtcblxudGVzdCgnMTQgTGVnYWN5IEJlaGF2aW9yIC0gRXhpc3RpbmcgVG9waWMsIEtleSBQcm9wcycsICgpID0+IHtcblxuICAvLyBJbml0aWFsIFNldHVwXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIHN0YWNrLm5vZGUuc2V0Q29udGV4dChcIkBhd3MtY2RrL2F3cy1zbnMtc3Vic2NyaXB0aW9uczpyZXN0cmljdFNxc0Rlc2NyeXB0aW9uXCIsIGZhbHNlKTtcblxuICBjb25zdCBleGlzdGluZ0tleSA9IG5ldyBrbXMuS2V5KHN0YWNrLCAndGVzdC1leGlzdGluZy1rZXknLCB7XG4gICAgZW5hYmxlS2V5Um90YXRpb246IHRydWUsXG4gICAgYWxpYXM6ICdleGlzdGluZ0tleSdcbiAgfSk7XG5cbiAgY29uc3QgdG9waWMgPSBuZXcgc25zLlRvcGljKHN0YWNrLCAndGVzdC10b3BpYycsIHt9KTtcblxuICBjb25zdCBzbnNUb1Nxc1N0YWNrID0gbmV3IFNuc1RvU3FzKHN0YWNrLCAnU25zVG9TcXNQYXR0ZXJuJywge1xuICAgIGVuYWJsZUVuY3J5cHRpb25XaXRoQ3VzdG9tZXJNYW5hZ2VkS2V5OiB0cnVlLFxuICAgIGVuY3J5cHRpb25LZXk6IGV4aXN0aW5nS2V5LFxuXG4gICAgZXhpc3RpbmdUb3BpY09iajogdG9waWNcbiAgfSk7XG5cbiAgQ2hlY2tLZXlQcm9wZXJ0eShzbnNUb1Nxc1N0YWNrLmVuY3J5cHRpb25LZXksIGtleVR5cGUubm9uZSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMSk7XG5cbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG5cbn0pO1xuXG50ZXN0KCcxNSBMZWdhY3kgQmVoYXZpb3IgLSBFeGlzdGluZyBUb3BpYywgRXhpc3RpbmcgS2V5LCBFbmNyeXB0RmxhZyBUcnVlJywgKCkgPT4ge1xuXG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHRvcGljID0gbmV3IHNucy5Ub3BpYyhzdGFjaywgJ3Rlc3QtdG9waWMnLCB7fSk7XG5cbiAgY29uc3Qgc25zVG9TcXNTdGFjayA9IG5ldyBTbnNUb1NxcyhzdGFjaywgJ1Nuc1RvU3FzUGF0dGVybicsIHtcbiAgICBlbmNyeXB0aW9uS2V5UHJvcHM6IHsgZW5hYmxlS2V5Um90YXRpb246IGZhbHNlIH0sXG4gICAgZXhpc3RpbmdUb3BpY09iajogdG9waWNcbiAgfSk7XG5cbiAgQ2hlY2tLZXlQcm9wZXJ0eShzbnNUb1Nxc1N0YWNrLmVuY3J5cHRpb25LZXksIGtleVR5cGUubm9uZSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMSk7XG5cbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG5cbn0pO1xuXG50ZXN0KCcxNiBMZWdhY3kgQmVoYXZpb3IgLSBFeGlzdGluZyBRdWV1ZSwgRW5jcnlwdEZsYWcgRmFsc2UnLCAoKSA9PiB7XG5cbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAndGVzdC1xdWV1ZScsIHt9KTtcblxuICBjb25zdCBzbnNUb1Nxc1N0YWNrID0gbmV3IFNuc1RvU3FzKHN0YWNrLCAnU25zVG9TcXNQYXR0ZXJuJywge1xuICAgIGVuYWJsZUVuY3J5cHRpb25XaXRoQ3VzdG9tZXJNYW5hZ2VkS2V5OiBmYWxzZSxcbiAgICBleGlzdGluZ1F1ZXVlT2JqOiBxdWV1ZVxuICB9KTtcblxuICBDaGVja0tleVByb3BlcnR5KHNuc1RvU3FzU3RhY2suZW5jcnlwdGlvbktleSwga2V5VHlwZS5zc2UpO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OktNUzo6S2V5XCIsIDApO1xuXG4gIENoZWNrVG9waWNLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLnNzZSk7XG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLm5vbmUpO1xufSk7XG5cbnRlc3QoJzE3IExlZ2FjeSBCZWhhdmlvciAtIEV4aXN0aW5nIFF1ZXVlLCBFbmNyeXB0RmxhZyBUcnVlJywgKCkgPT4ge1xuXG4gIC8vIEluaXRpYWwgU2V0dXBcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgc3RhY2subm9kZS5zZXRDb250ZXh0KFwiQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zOnJlc3RyaWN0U3FzRGVzY3J5cHRpb25cIiwgZmFsc2UpO1xuXG4gIGNvbnN0IHF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ3Rlc3QtcXVldWUnLCB7fSk7XG5cbiAgY29uc3Qgc25zVG9TcXNTdGFjayA9IG5ldyBTbnNUb1NxcyhzdGFjaywgJ1Nuc1RvU3FzUGF0dGVybicsIHtcbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZSxcbiAgICBleGlzdGluZ1F1ZXVlT2JqOiBxdWV1ZVxuICB9KTtcblxuICBDaGVja0tleVByb3BlcnR5KHNuc1RvU3FzU3RhY2suZW5jcnlwdGlvbktleSwga2V5VHlwZS5jbWspO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OktNUzo6S2V5XCIsIDEpO1xuXG4gIENoZWNrVG9waWNLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLm5vbmUpO1xufSk7XG5cbnRlc3QoJzE4IExlZ2FjeSBCZWhhdmlvciAtIEV4aXN0aW5nIFF1ZXVlLCBFeGlzdGluZyBLZXksIEVuY3J5cHRGbGFnIFRydWUnLCAoKSA9PiB7XG5cbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgZXhpc3RpbmdLZXkgPSBuZXcga21zLktleShzdGFjaywgJ3Rlc3QtZXhpc3Rpbmcta2V5Jywge1xuICAgIGVuYWJsZUtleVJvdGF0aW9uOiB0cnVlLFxuICAgIGFsaWFzOiAnZXhpc3RpbmdLZXknXG4gIH0pO1xuXG4gIGNvbnN0IHF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ3Rlc3QtcXVldWUnLCB7fSk7XG5cbiAgY29uc3Qgc25zVG9TcXNTdGFjayA9IG5ldyBTbnNUb1NxcyhzdGFjaywgJ1Nuc1RvU3FzUGF0dGVybicsIHtcbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZSxcbiAgICBlbmNyeXB0aW9uS2V5OiBleGlzdGluZ0tleSxcbiAgICBleGlzdGluZ1F1ZXVlT2JqOiBxdWV1ZVxuICB9KTtcblxuICBDaGVja0tleVByb3BlcnR5KHNuc1RvU3FzU3RhY2suZW5jcnlwdGlvbktleSwga2V5VHlwZS5jbWspO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OktNUzo6S2V5XCIsIDEpO1xuXG4gIENoZWNrVG9waWNLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLmNtayk7XG4gIENoZWNrUXVldWVLZXlUeXBlKHRlbXBsYXRlLCBrZXlUeXBlLm5vbmUpO1xufSk7XG5cbnRlc3QoJzE5IExlZ2FjeSBCZWhhdmlvciAtIEV4aXN0aW5nIFF1ZXVlLCBLZXkgUHJvcHMsIEVuY3J5cHRGbGFnIFRydWUnLCAoKSA9PiB7XG5cbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBzdGFjay5ub2RlLnNldENvbnRleHQoXCJAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnM6cmVzdHJpY3RTcXNEZXNjcnlwdGlvblwiLCBmYWxzZSk7XG5cbiAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAndGVzdC1xdWV1ZScsIHt9KTtcblxuICBjb25zdCBzbnNUb1Nxc1N0YWNrID0gbmV3IFNuc1RvU3FzKHN0YWNrLCAnU25zVG9TcXNQYXR0ZXJuJywge1xuICAgIGVuYWJsZUVuY3J5cHRpb25XaXRoQ3VzdG9tZXJNYW5hZ2VkS2V5OiB0cnVlLFxuICAgIGVuY3J5cHRpb25LZXlQcm9wczogeyBlbmFibGVLZXlSb3RhdGlvbjogZmFsc2V9LFxuICAgIGV4aXN0aW5nUXVldWVPYmo6IHF1ZXVlXG4gIH0pO1xuXG4gIENoZWNrS2V5UHJvcGVydHkoc25zVG9TcXNTdGFjay5lbmNyeXB0aW9uS2V5LCBrZXlUeXBlLmNtayk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6S01TOjpLZXlcIiwgMSk7XG5cbiAgQ2hlY2tUb3BpY0tleVR5cGUodGVtcGxhdGUsIGtleVR5cGUuY21rKTtcbiAgQ2hlY2tRdWV1ZUtleVR5cGUodGVtcGxhdGUsIGtleVR5cGUubm9uZSk7XG59KTtcbiJdfQ==