import click
import json
import sys
import shlex

from globus_sdk import TransferClient, RefreshTokenAuthorizer

from globus_cli import version
from globus_cli.config import get_transfer_refresh_token, internal_auth_client
from globus_cli.safeio import safeprint
from globus_cli.helpers import print_table


def get_client():
    transfer_rt = get_transfer_refresh_token()
    authorizer = None

    # if there's a refresh token, use it to build the authorizer
    if transfer_rt is not None:
        authorizer = RefreshTokenAuthorizer(
            transfer_rt, internal_auth_client())

    return TransferClient(authorizer=authorizer, app_name=version.app_name)


def display_name_or_cname(ep_doc):
    return ep_doc['display_name'] or ep_doc['canonical_name']


def print_json_from_iterator(iterator):
    json_output_dict = {'DATA': []}
    for item in iterator:
        dat = item
        try:
            dat = item.data
        except AttributeError:
            pass
        json_output_dict['DATA'].append(dat)
    safeprint(json.dumps(json_output_dict, indent=2))


def endpoint_list_to_text(iterator):
    print_table(iterator, [('Owner', 'owner_string'), ('ID', 'id'),
                           ('Display Name', display_name_or_cname)])


def assemble_generic_doc(datatype, **kwargs):
    doc = {'DATA_TYPE': datatype}
    for argname in kwargs:
        if kwargs[argname] is not None:
            doc[argname] = kwargs[argname]
    return doc


def submission_id_option(f):
    """
    Simple decorator that attaches an option to a command for consuming a
    submission ID, i.e. --submission-id
    This is useful for task submission, in conjunction with the
        globus transfer task generate-submission-id
    command.
    """
    f = click.option(
        '--submission-id', help=(
            'Task submission ID, as generated by `globus transfer task '
            'generate-submission-id`. Used for safe resubmission in the '
            'presence of network failures.'))(f)
    return f


def endpoint_id_option(*args, **kwargs):
    def decorate(f, **kwargs):
        """
        Work of actually decorating a function -- wrapped in here because we
        want to dispatch depending on how this is invoked
        """
        helptext = kwargs.get('help', 'ID of the Endpoint')
        f = click.option('--endpoint-id', required=True, help=helptext)(f)
        return f

    # special behavior when invoked with only one non-keyword argument: act as
    # a normal decorator, decorating and returning that argument with
    # click.option
    if len(args) == 1 and len(kwargs) == 0:
        return decorate(args[0])

    # if we're not doing that, we should see no positional args
    # the alternative behavior is to fall through and discard *args, but this
    # will probably confuse someone in the future when their arguments are
    # silently discarded
    elif len(args) != 0:
        raise ValueError(
                'endpoint_id_option() cannot take positional args')

    # final case: got 0 or more kwargs, no positionals
    # do the function-which-returns-a-decorator dance to produce a
    # new decorator based on the arguments given
    else:
        def inner_decorator(f):
            return decorate(f, **kwargs)
        return inner_decorator


def shlex_process_stdin(process_command, helptext):
    """
    Use shlex to process stdin line-by-line.
    Also prints help text.

    Requires that @process_command be a Click command object, used for
    processing single lines of input. helptext is prepended to the standard
    message printed to interactive sessions.
    """
    # if input is interactive, print help to stderr
    if sys.stdin.isatty():
        safeprint(
            ('{}\n'.format(helptext) +
             'Lines are split with shlex in POSIX mode: '
             'https://docs.python.org/library/shlex.html#parsing-rules\n'
             'Terminate input with Ctrl+D or <EOF>\n'), write_to_stderr=True)

    # use readlines() rather than implicit file read line looping to force
    # python to properly capture EOF (otherwise, EOF acts as a flush and
    # things get weird)
    for line in sys.stdin.readlines():
        # get the argument vector:
        # do a shlex split to handle quoted paths with spaces in them
        # also lets us have comments with #
        argv = shlex.split(line, comments=True)
        if argv:
            try:
                process_command.main(args=argv)
            except SystemExit:
                pass
