"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const ingress = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'services';
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = props.type ?? ServiceType.CLUSTER_IP;
        }
        this._externalIPs = props.externalIPs ?? [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        if (props.selector) {
            this.select(props.selector);
        }
        for (const port of props.ports ?? []) {
            this.bind(port.port, port);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        const ingr = options.ingress ?? new ingress.Ingress(this, 'Ingress');
        ingr.addRule(path, ingress.IngressBackend.fromService(this), options.pathType);
        return ingr;
    }
    /**
     * Ports for this service.
     *
     * Use `bind()` to bind additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Return the first port of the service.
     */
    get port() {
        return [...this._ports][0].port;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    bind(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * Require this service to select pods matching the selector.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    select(selector) {
        const labels = selector.toPodSelectorConfig().labelSelector._toKube().matchLabels ?? {};
        for (const [key, value] of Object.entries(labels)) {
            this._selector[key] = value;
        }
    }
    /**
     * Require this service to select pods with this label.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    selectLabel(key, value) {
        this._selector[key] = value;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol ? this._portProtocolToKube(port.protocol) : undefined,
            });
        }
        const serviceType = this._serviceTypeToKube(this.type);
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: serviceType,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: serviceType,
            externalName: this.externalName,
        };
    }
    _portProtocolToKube(protocol) {
        switch (protocol) {
            case container.Protocol.SCTP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.SCTP;
            case container.Protocol.TCP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.TCP;
            case container.Protocol.UDP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.UDP;
            default:
                throw new Error(`Unsupported port protocol: ${protocol}`);
        }
    }
    _serviceTypeToKube(serviceType) {
        switch (serviceType) {
            case ServiceType.CLUSTER_IP:
                return k8s.IoK8SApiCoreV1ServiceSpecType.CLUSTER_IP;
            case ServiceType.EXTERNAL_NAME:
                return k8s.IoK8SApiCoreV1ServiceSpecType.EXTERNAL_NAME;
            case ServiceType.LOAD_BALANCER:
                return k8s.IoK8SApiCoreV1ServiceSpecType.LOAD_BALANCER;
            case ServiceType.NODE_PORT:
                return k8s.IoK8SApiCoreV1ServiceSpecType.NODE_PORT;
            default:
                throw new Error(`Unsupported service type: ${serviceType}`);
        }
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-23.Service", version: "2.3.30" };
//# sourceMappingURL=data:application/json;base64,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