from __future__ import annotations as _annotations

"""
Profile: http://hl7.org/fhir/StructureDefinition/MedicinalProductDefinition
Release: R4B
Version: 4.3.0
Build ID: c475c22
Last updated: 2022-05-28T12:47:40.239+10:00
"""
import typing

from pydantic import Field

from . import backboneelement, domainresource, fhirtypes


class MedicinalProductDefinition(domainresource.DomainResource):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    Detailed definition of a medicinal product.
    A medicinal product, being a substance or combination of substances that is
    intended to treat, prevent or diagnose a disease, or to restore, correct or
    modify physiological functions by exerting a pharmacological, immunological
    or metabolic action. This resource is intended to define and detail such
    products and their properties, for uses other than direct patient care
    (e.g. regulatory use, or drug catalogs).
    """

    __resource_type__ = "MedicinalProductDefinition"

    additionalMonitoringIndicator: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="additionalMonitoringIndicator",
        title=(
            "Whether the Medicinal Product is subject to additional monitoring for "
            "regulatory reasons"
        ),
        description=(
            "Whether the Medicinal Product is subject to additional monitoring for "
            "regulatory reasons, such as heightened reporting requirements."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    attachedDocument: typing.List[fhirtypes.ReferenceType] | None = Field(  # type: ignore
        default=None,
        alias="attachedDocument",
        title="Additional documentation about the medicinal product",
        description=(
            "Additional information or supporting documentation about the medicinal"
            " product."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["DocumentReference"],
        },
    )

    characteristic: typing.List[fhirtypes.MedicinalProductDefinitionCharacteristicType] | None = Field(  # type: ignore
        default=None,
        alias="characteristic",
        title='Key product features such as "sugar free", "modified release"',
        description=(
            'Allows the key product features to be recorded, such as "sugar free", '
            '"modified release", "parallel import".'
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    classification: typing.List[fhirtypes.CodeableConceptType] | None = Field(  # type: ignore
        default=None,
        alias="classification",
        title="Allows the product to be classified by various systems",
        description=(
            "Allows the product to be classified by various systems, commonly WHO "
            "ATC."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    clinicalTrial: typing.List[fhirtypes.ReferenceType] | None = Field(  # type: ignore
        default=None,
        alias="clinicalTrial",
        title="Clinical trials or studies that this product is involved in",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["ResearchStudy"],
        },
    )

    code: typing.List[fhirtypes.CodingType] | None = Field(  # type: ignore
        default=None,
        alias="code",
        title="A code that this product is known by, within some formal terminology",
        description=(
            "A code that this product is known by, usually within some formal "
            "terminology, perhaps assigned by a third party (i.e. not the "
            "manufacturer or regulator). Products (types of medications) tend to be"
            " known by identifiers during development and within regulatory "
            "process. However when they are prescribed they tend to be identified "
            "by codes. The same product may be have multiple codes, applied to it "
            "by multiple organizations."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    combinedPharmaceuticalDoseForm: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="combinedPharmaceuticalDoseForm",
        title=(
            "The dose form for a single part product, or combined form of a "
            "multiple part product"
        ),
        description=(
            "The dose form for a single part product, or combined form of a "
            "multiple part product. This is one concept that describes all the "
            "components. It does not represent the form with components physically "
            "mixed, if that might be necessary, for which see "
            "(AdministrableProductDefinition.administrableDoseForm)."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    contact: typing.List[fhirtypes.MedicinalProductDefinitionContactType] | None = Field(  # type: ignore
        default=None,
        alias="contact",
        title="A product specific contact, person (in a role), or an organization",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    crossReference: typing.List[fhirtypes.MedicinalProductDefinitionCrossReferenceType] | None = Field(  # type: ignore
        default=None,
        alias="crossReference",
        title=(
            "Reference to another product, e.g. for linking authorised to "
            "investigational product"
        ),
        description=(
            "Reference to another product, e.g. for linking authorised to "
            "investigational product, or a virtual product."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    description: fhirtypes.MarkdownType | None = Field(  # type: ignore
        default=None,
        alias="description",
        title="General description of this product",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )
    description__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_description", title="Extension field for ``description``."
    )

    domain: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="domain",
        title="If this medicine applies to human or veterinary uses",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    identifier: typing.List[fhirtypes.IdentifierType] | None = Field(  # type: ignore
        default=None,
        alias="identifier",
        title="Business identifier for this product. Could be an MPID",
        description=(
            "Business identifier for this product. Could be an MPID. When in "
            "development or being regulated, products are typically referenced by "
            "official identifiers, assigned by a manufacturer or regulator, and "
            "unique to a product (which, when compared to a product instance being "
            "prescribed, is actually a product type). See also "
            "MedicinalProductDefinition.code."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    impurity: typing.List[fhirtypes.CodeableReferenceType] | None = Field(  # type: ignore
        default=None,
        alias="impurity",
        title=(
            "Any component of the drug product which is not the chemical entity "
            "defined as the drug substance, or an excipient in the drug product"
        ),
        description=(
            "Any component of the drug product which is not the chemical entity "
            "defined as the drug substance, or an excipient in the drug product. "
            "This includes process-related impurities and contaminants, product-"
            "related impurities including degradation products."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["SubstanceDefinition"],
        },
    )

    indication: fhirtypes.MarkdownType | None = Field(  # type: ignore
        default=None,
        alias="indication",
        title=(
            "Description of indication(s) for this product, used when structured "
            "indications are not required"
        ),
        description=(
            "Description of indication(s) for this product, used when structured "
            "indications are not required. In cases where structured indications "
            "are required, they are captured using the ClinicalUseDefinition "
            "resource. An indication is a medical situation for which using the "
            "product is appropriate."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )
    indication__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_indication", title="Extension field for ``indication``."
    )

    ingredient: typing.List[fhirtypes.CodeableConceptType] | None = Field(  # type: ignore
        default=None,
        alias="ingredient",
        title=(
            "The ingredients of this medicinal product - when not detailed in other"
            " resources"
        ),
        description=(
            "The ingredients of this medicinal product - when not detailed in other"
            " resources. This is only needed if the ingredients are not specified "
            "by incoming references from the Ingredient resource, or indirectly via"
            " incoming AdministrableProductDefinition, PackagedProductDefinition or"
            " ManufacturedItemDefinition references. In cases where those levels of"
            " detail are not used, the ingredients may be specified directly here "
            "as codes."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    legalStatusOfSupply: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="legalStatusOfSupply",
        title=(
            "The legal status of supply of the medicinal product as classified by "
            "the regulator"
        ),
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    marketingStatus: typing.List[fhirtypes.MarketingStatusType] | None = Field(  # type: ignore
        default=None,
        alias="marketingStatus",
        title=(
            "Marketing status of the medicinal product, in contrast to marketing "
            "authorization"
        ),
        description=(
            "Marketing status of the medicinal product, in contrast to marketing "
            "authorization. This refers to the product being actually 'on the "
            "market' as opposed to being allowed to be on the market (which is an "
            "authorization)."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    masterFile: typing.List[fhirtypes.ReferenceType] | None = Field(  # type: ignore
        default=None,
        alias="masterFile",
        title=(
            "A master file for the medicinal product (e.g. Pharmacovigilance System"
            " Master File)"
        ),
        description=(
            "A master file for the medicinal product (e.g. Pharmacovigilance System"
            " Master File). Drug master files (DMFs) are documents submitted to "
            "regulatory agencies to provide confidential detailed information about"
            " facilities, processes or articles used in the manufacturing, "
            "processing, packaging and storing of drug products."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["DocumentReference"],
        },
    )

    name: typing.List[fhirtypes.MedicinalProductDefinitionNameType] = Field(  # type: ignore
        default=...,
        alias="name",
        title="The product's name, including full name and possibly coded parts",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    operation: typing.List[fhirtypes.MedicinalProductDefinitionOperationType] | None = Field(  # type: ignore
        default=None,
        alias="operation",
        title="A manufacturing or administrative process for the medicinal product",
        description=(
            "A manufacturing or administrative process or step associated with (or "
            "performed on) the medicinal product."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    packagedMedicinalProduct: typing.List[fhirtypes.CodeableConceptType] | None = Field(  # type: ignore
        default=None,
        alias="packagedMedicinalProduct",
        title="Package type for the product",
        description=(
            "Package type for the product. See also the PackagedProductDefinition "
            "resource."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    pediatricUseIndicator: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="pediatricUseIndicator",
        title="If authorised for use in children",
        description="If authorised for use in children, or infants, neonates etc.",
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    route: typing.List[fhirtypes.CodeableConceptType] | None = Field(  # type: ignore
        default=None,
        alias="route",
        title=(
            "The path by which the product is taken into or makes contact with the "
            "body"
        ),
        description=(
            "The path by which the product is taken into or makes contact with the "
            "body. In some regions this is referred to as the licenced or approved "
            "route. See also AdministrableProductDefinition resource. "
            "MedicinalProductDefinition.route is the same concept as "
            "AdministrableProductDefinition.routeOfAdministration.code, and they "
            "cannot be used together."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    specialMeasures: typing.List[fhirtypes.CodeableConceptType] | None = Field(  # type: ignore
        default=None,
        alias="specialMeasures",
        title=(
            "Whether the Medicinal Product is subject to special measures for "
            "regulatory reasons"
        ),
        description=(
            "Whether the Medicinal Product is subject to special measures for "
            "regulatory reasons, such as a requirement to conduct post-"
            "authorisation studies."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    status: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="status",
        title="The status within the lifecycle of this product record",
        description=(
            "The status within the lifecycle of this product record. A high-level "
            "status, this is not intended to duplicate details carried elsewhere "
            "such as legal status, or authorization status."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    statusDate: fhirtypes.DateTimeType | None = Field(  # type: ignore
        default=None,
        alias="statusDate",
        title="The date at which the given status became applicable",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )
    statusDate__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_statusDate", title="Extension field for ``statusDate``."
    )

    type: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="type",
        title="Regulatory type, e.g. Investigational or Authorized",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    version: fhirtypes.StringType | None = Field(  # type: ignore
        default=None,
        alias="version",
        title="A business identifier relating to a specific version of the product",
        description=(
            "A business identifier relating to a specific version of the product, "
            "this is commonly used to support revisions to an existing product."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )
    version__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_version", title="Extension field for ``version``."
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinition`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "meta",
            "implicitRules",
            "language",
            "text",
            "contained",
            "extension",
            "modifierExtension",
            "identifier",
            "type",
            "domain",
            "version",
            "status",
            "statusDate",
            "description",
            "combinedPharmaceuticalDoseForm",
            "route",
            "indication",
            "legalStatusOfSupply",
            "additionalMonitoringIndicator",
            "specialMeasures",
            "pediatricUseIndicator",
            "classification",
            "marketingStatus",
            "packagedMedicinalProduct",
            "ingredient",
            "impurity",
            "attachedDocument",
            "masterFile",
            "contact",
            "clinicalTrial",
            "code",
            "name",
            "crossReference",
            "operation",
            "characteristic",
        ]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinition`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "meta",
            "implicitRules",
            "identifier",
            "type",
            "domain",
            "version",
            "status",
            "statusDate",
            "description",
            "combinedPharmaceuticalDoseForm",
            "route",
            "indication",
            "legalStatusOfSupply",
            "additionalMonitoringIndicator",
            "specialMeasures",
            "pediatricUseIndicator",
            "classification",
            "marketingStatus",
            "packagedMedicinalProduct",
            "ingredient",
            "impurity",
            "attachedDocument",
            "masterFile",
            "contact",
            "clinicalTrial",
            "code",
            "name",
            "crossReference",
            "operation",
            "characteristic",
        ]


class MedicinalProductDefinitionCharacteristic(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    Key product features such as "sugar free", "modified release".
    Allows the key product features to be recorded, such as "sugar free",
    "modified release", "parallel import".
    """

    __resource_type__ = "MedicinalProductDefinitionCharacteristic"

    type: fhirtypes.CodeableConceptType = Field(  # type: ignore
        default=...,
        alias="type",
        title="A code expressing the type of characteristic",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    valueAttachment: fhirtypes.AttachmentType | None = Field(  # type: ignore
        default=None,
        alias="valueAttachment",
        title="A value for the characteristic",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # Choice of Data Types. i.e value[x]
            "one_of_many": "value",
            "one_of_many_required": False,
        },
    )

    valueBoolean: bool | None = Field(  # type: ignore
        default=None,
        alias="valueBoolean",
        title="A value for the characteristic",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # Choice of Data Types. i.e value[x]
            "one_of_many": "value",
            "one_of_many_required": False,
        },
    )
    valueBoolean__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None,
        alias="_valueBoolean",
        title="Extension field for ``valueBoolean``.",
    )

    valueCodeableConcept: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="valueCodeableConcept",
        title="A value for the characteristic",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # Choice of Data Types. i.e value[x]
            "one_of_many": "value",
            "one_of_many_required": False,
        },
    )

    valueDate: fhirtypes.DateType | None = Field(  # type: ignore
        default=None,
        alias="valueDate",
        title="A value for the characteristic",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # Choice of Data Types. i.e value[x]
            "one_of_many": "value",
            "one_of_many_required": False,
        },
    )
    valueDate__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_valueDate", title="Extension field for ``valueDate``."
    )

    valueQuantity: fhirtypes.QuantityType | None = Field(  # type: ignore
        default=None,
        alias="valueQuantity",
        title="A value for the characteristic",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # Choice of Data Types. i.e value[x]
            "one_of_many": "value",
            "one_of_many_required": False,
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinitionCharacteristic`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "extension",
            "modifierExtension",
            "type",
            "valueCodeableConcept",
            "valueQuantity",
            "valueDate",
            "valueBoolean",
            "valueAttachment",
        ]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinitionCharacteristic`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "modifierExtension",
            "type",
            "valueCodeableConcept",
            "valueQuantity",
            "valueDate",
            "valueBoolean",
            "valueAttachment",
        ]

    def get_one_of_many_fields(self) -> typing.Dict[str, typing.List[str]]:
        """https://www.hl7.org/fhir/formats.html#choice
        A few elements have a choice of more than one data type for their content.
        All such elements have a name that takes the form nnn[x].
        The "nnn" part of the name is constant, and the "[x]" is replaced with
        the title-cased name of the type that is actually used.
        The table view shows each of these names explicitly.

        Elements that have a choice of data type cannot repeat - they must have a
        maximum cardinality of 1. When constructing an instance of an element with a
        choice of types, the authoring system must create a single element with a
        data type chosen from among the list of permitted data types.
        """
        one_of_many_fields = {
            "value": [
                "valueAttachment",
                "valueBoolean",
                "valueCodeableConcept",
                "valueDate",
                "valueQuantity",
            ]
        }
        return one_of_many_fields


class MedicinalProductDefinitionContact(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    A product specific contact, person (in a role), or an organization.
    """

    __resource_type__ = "MedicinalProductDefinitionContact"

    contact: fhirtypes.ReferenceType = Field(  # type: ignore
        default=...,
        alias="contact",
        title="A product specific contact, person (in a role), or an organization",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["Organization", "PractitionerRole"],
        },
    )

    type: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="type",
        title=(
            "Allows the contact to be classified, for example QPPV, "
            "Pharmacovigilance Enquiry Information"
        ),
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinitionContact`` according to specification,
        with preserving the original sequence order.
        """
        return ["id", "extension", "modifierExtension", "type", "contact"]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinitionContact`` according to specification,
        with preserving the original sequence order.
        """
        return ["modifierExtension", "type", "contact"]


class MedicinalProductDefinitionCrossReference(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    Reference to another product, e.g. for linking authorised to
    investigational product.
    Reference to another product, e.g. for linking authorised to
    investigational product, or a virtual product.
    """

    __resource_type__ = "MedicinalProductDefinitionCrossReference"

    product: fhirtypes.CodeableReferenceType = Field(  # type: ignore
        default=...,
        alias="product",
        title=(
            "Reference to another product, e.g. for linking authorised to "
            "investigational product"
        ),
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["MedicinalProductDefinition"],
        },
    )

    type: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="type",
        title=(
            "The type of relationship, for instance branded to generic or virtual "
            "to actual product"
        ),
        description=(
            "The type of relationship, for instance branded to generic, virtual to "
            "actual product, product to development product (investigational), "
            "parallel import version."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinitionCrossReference`` according to specification,
        with preserving the original sequence order.
        """
        return ["id", "extension", "modifierExtension", "product", "type"]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinitionCrossReference`` according to specification,
        with preserving the original sequence order.
        """
        return ["modifierExtension", "product", "type"]


class MedicinalProductDefinitionName(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    The product's name, including full name and possibly coded parts.
    """

    __resource_type__ = "MedicinalProductDefinitionName"

    countryLanguage: typing.List[fhirtypes.MedicinalProductDefinitionNameCountryLanguageType] | None = Field(  # type: ignore
        default=None,
        alias="countryLanguage",
        title="Country and jurisdiction where the name applies",
        description=(
            "Country and jurisdiction where the name applies, and associated "
            "language."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    namePart: typing.List[fhirtypes.MedicinalProductDefinitionNameNamePartType] | None = Field(  # type: ignore
        default=None,
        alias="namePart",
        title="Coding words or phrases of the name",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    productName: fhirtypes.StringType | None = Field(  # type: ignore
        default=None,
        alias="productName",
        title="The full product name",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            "element_required": True,
        },
    )
    productName__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_productName", title="Extension field for ``productName``."
    )

    type: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="type",
        title="Type of product name, such as rINN, BAN, Proprietary, Non-Proprietary",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinitionName`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "extension",
            "modifierExtension",
            "productName",
            "type",
            "namePart",
            "countryLanguage",
        ]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinitionName`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "modifierExtension",
            "productName",
            "type",
            "namePart",
            "countryLanguage",
        ]

    def get_required_fields(self) -> typing.List[typing.Tuple[str, str]]:
        """https://www.hl7.org/fhir/extensibility.html#Special-Case
        In some cases, implementers might find that they do not have appropriate data for
        an element with minimum cardinality = 1. In this case, the element must be present,
        but unless the resource or a profile on it has made the actual value of the primitive
        data type mandatory, it is possible to provide an extension that explains why
        the primitive value is not present.
        """
        required_fields = [("productName", "productName__ext")]
        return required_fields


class MedicinalProductDefinitionNameCountryLanguage(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    Country and jurisdiction where the name applies.
    Country and jurisdiction where the name applies, and associated language.
    """

    __resource_type__ = "MedicinalProductDefinitionNameCountryLanguage"

    country: fhirtypes.CodeableConceptType = Field(  # type: ignore
        default=...,
        alias="country",
        title="Country code for where this name applies",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    jurisdiction: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="jurisdiction",
        title="Jurisdiction code for where this name applies",
        description=(
            "Jurisdiction code for where this name applies. A jurisdiction may be a"
            " sub- or supra-national entity (e.g. a state or a geographic region)."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    language: fhirtypes.CodeableConceptType = Field(  # type: ignore
        default=...,
        alias="language",
        title="Language code for this name",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinitionNameCountryLanguage`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "extension",
            "modifierExtension",
            "country",
            "jurisdiction",
            "language",
        ]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinitionNameCountryLanguage`` according to specification,
        with preserving the original sequence order.
        """
        return ["modifierExtension", "country", "jurisdiction", "language"]


class MedicinalProductDefinitionNameNamePart(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    Coding words or phrases of the name.
    """

    __resource_type__ = "MedicinalProductDefinitionNameNamePart"

    part: fhirtypes.StringType | None = Field(  # type: ignore
        default=None,
        alias="part",
        title="A fragment of a product name",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            "element_required": True,
        },
    )
    part__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_part", title="Extension field for ``part``."
    )

    type: fhirtypes.CodeableConceptType = Field(  # type: ignore
        default=...,
        alias="type",
        title="Identifying type for this part of the name (e.g. strength part)",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinitionNameNamePart`` according to specification,
        with preserving the original sequence order.
        """
        return ["id", "extension", "modifierExtension", "part", "type"]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinitionNameNamePart`` according to specification,
        with preserving the original sequence order.
        """
        return ["modifierExtension", "part", "type"]

    def get_required_fields(self) -> typing.List[typing.Tuple[str, str]]:
        """https://www.hl7.org/fhir/extensibility.html#Special-Case
        In some cases, implementers might find that they do not have appropriate data for
        an element with minimum cardinality = 1. In this case, the element must be present,
        but unless the resource or a profile on it has made the actual value of the primitive
        data type mandatory, it is possible to provide an extension that explains why
        the primitive value is not present.
        """
        required_fields = [("part", "part__ext")]
        return required_fields


class MedicinalProductDefinitionOperation(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    A manufacturing or administrative process for the medicinal product.
    A manufacturing or administrative process or step associated with (or
    performed on) the medicinal product.
    """

    __resource_type__ = "MedicinalProductDefinitionOperation"

    confidentialityIndicator: fhirtypes.CodeableConceptType | None = Field(  # type: ignore
        default=None,
        alias="confidentialityIndicator",
        title=(
            "Specifies whether this process is considered proprietary or "
            "confidential"
        ),
        description=(
            "Specifies whether this particular business or manufacturing process is"
            " considered proprietary or confidential."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    effectiveDate: fhirtypes.PeriodType | None = Field(  # type: ignore
        default=None,
        alias="effectiveDate",
        title="Date range of applicability",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    organization: typing.List[fhirtypes.ReferenceType] | None = Field(  # type: ignore
        default=None,
        alias="organization",
        title=(
            "The organization responsible for the particular process, e.g. the "
            "manufacturer or importer"
        ),
        description=(
            "The organization or establishment responsible for (or associated with)"
            " the particular process or step, examples include the manufacturer, "
            "importer, agent."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["Organization"],
        },
    )

    type: fhirtypes.CodeableReferenceType | None = Field(  # type: ignore
        default=None,
        alias="type",
        title=(
            "The type of manufacturing operation e.g. manufacturing itself, re-"
            "packaging"
        ),
        description=(
            "The type of manufacturing operation e.g. manufacturing itself, re-"
            "packaging. For the authorization of this, a RegulatedAuthorization "
            "would point to the same plan or activity referenced here."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["ActivityDefinition", "PlanDefinition"],
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``MedicinalProductDefinitionOperation`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "extension",
            "modifierExtension",
            "type",
            "effectiveDate",
            "organization",
            "confidentialityIndicator",
        ]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``MedicinalProductDefinitionOperation`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "modifierExtension",
            "type",
            "effectiveDate",
            "organization",
            "confidentialityIndicator",
        ]
