from __future__ import annotations as _annotations

"""
Profile: http://hl7.org/fhir/StructureDefinition/Person
Release: STU3
Version: 3.0.2
Revision: 11917
Last updated: 2019-10-24T11:53:00+11:00
"""
import typing

from pydantic import Field

from . import backboneelement, domainresource, fhirtypes


class Person(domainresource.DomainResource):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    A generic person record.
    Demographics and administrative information about a person independent of a
    specific health-related context.
    """

    __resource_type__ = "Person"

    active: bool | None = Field(  # type: ignore
        default=None,
        alias="active",
        title="This person's record is in active use",
        description="Whether this person's record is in active use.",
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )
    active__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_active", title="Extension field for ``active``."
    )

    address: typing.List[fhirtypes.AddressType] | None = Field(  # type: ignore
        default=None,
        alias="address",
        title="One or more addresses for the person",
        description=None,
        json_schema_extra={
            "element_property": True,
        },
    )

    birthDate: fhirtypes.DateType | None = Field(  # type: ignore
        default=None,
        alias="birthDate",
        title="The date on which the person was born",
        description="The birth date for the person.",
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )
    birthDate__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_birthDate", title="Extension field for ``birthDate``."
    )

    gender: fhirtypes.CodeType | None = Field(  # type: ignore
        default=None,
        alias="gender",
        title="male | female | other | unknown",
        description="Administrative Gender.",
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Enum values can be used in validation,
            # but use in your own responsibilities, read official FHIR documentation.
            "enum_values": ["male", "female", "other", "unknown"],
        },
    )
    gender__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_gender", title="Extension field for ``gender``."
    )

    identifier: typing.List[fhirtypes.IdentifierType] | None = Field(  # type: ignore
        default=None,
        alias="identifier",
        title="A human identifier for this person",
        description="Identifier for a person within a particular scope.",
        json_schema_extra={
            "element_property": True,
        },
    )

    link: typing.List[fhirtypes.PersonLinkType] | None = Field(  # type: ignore
        default=None,
        alias="link",
        title="Link to a resource that concerns the same actual person",
        description=None,
        json_schema_extra={
            "element_property": True,
        },
    )

    managingOrganization: fhirtypes.ReferenceType | None = Field(  # type: ignore
        default=None,
        alias="managingOrganization",
        title="The organization that is the custodian of the person record",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": ["Organization"],
        },
    )

    name: typing.List[fhirtypes.HumanNameType] | None = Field(  # type: ignore
        default=None,
        alias="name",
        title="A name associated with the person",
        description=None,
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    photo: fhirtypes.AttachmentType | None = Field(  # type: ignore
        default=None,
        alias="photo",
        title="Image of the person",
        description=(
            "An image that can be displayed as a thumbnail of the person to enhance"
            " the identification of the individual."
        ),
        json_schema_extra={
            "element_property": True,
        },
    )

    telecom: typing.List[fhirtypes.ContactPointType] | None = Field(  # type: ignore
        default=None,
        alias="telecom",
        title="A contact detail for the person",
        description=(
            "A contact detail for the person, e.g. a telephone number or an email "
            "address."
        ),
        json_schema_extra={
            "element_property": True,
            "summary_element_property": True,
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``Person`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "meta",
            "implicitRules",
            "language",
            "text",
            "contained",
            "extension",
            "modifierExtension",
            "identifier",
            "name",
            "telecom",
            "gender",
            "birthDate",
            "address",
            "photo",
            "managingOrganization",
            "active",
            "link",
        ]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``Person`` according to specification,
        with preserving the original sequence order.
        """
        return [
            "id",
            "meta",
            "implicitRules",
            "name",
            "telecom",
            "gender",
            "birthDate",
            "managingOrganization",
            "active",
        ]


class PersonLink(backboneelement.BackboneElement):
    """Disclaimer: Any field name ends with ``__ext`` doesn't part of
    Resource StructureDefinition, instead used to enable Extensibility feature
    for FHIR Primitive Data Types.

    Link to a resource that concerns the same actual person.
    """

    __resource_type__ = "PersonLink"

    assurance: fhirtypes.CodeType | None = Field(  # type: ignore
        default=None,
        alias="assurance",
        title="level1 | level2 | level3 | level4",
        description=(
            "Level of assurance that this link is actually associated with the "
            "target resource."
        ),
        json_schema_extra={
            "element_property": True,
            # note: Enum values can be used in validation,
            # but use in your own responsibilities, read official FHIR documentation.
            "enum_values": ["level1", "level2", "level3", "level4"],
        },
    )
    assurance__ext: fhirtypes.FHIRPrimitiveExtensionType | None = Field(  # type: ignore
        default=None, alias="_assurance", title="Extension field for ``assurance``."
    )

    target: fhirtypes.ReferenceType = Field(  # type: ignore
        default=...,
        alias="target",
        title="The resource to which this actual person is associated",
        description=None,
        json_schema_extra={
            "element_property": True,
            # note: Listed Resource Type(s) should be allowed as Reference.
            "enum_reference_types": [
                "Patient",
                "Practitioner",
                "RelatedPerson",
                "Person",
            ],
        },
    )

    @classmethod
    def elements_sequence(cls):
        """returning all element names from
        ``PersonLink`` according to specification,
        with preserving the original sequence order.
        """
        return ["id", "extension", "modifierExtension", "target", "assurance"]

    @classmethod
    def summary_elements_sequence(cls):
        """returning all element names (those have summary mode are enabled) from ``PersonLink`` according to specification,
        with preserving the original sequence order.
        """
        return ["modifierExtension"]
