import numpy as np
import scipy.interpolate
from scipy import constants

# ---------------------------------------------------------------------------------
const = {}
const['mass'] = 6.64647641E-27 # kg
const['Tlambda'] = 2.1768 # K
const['Tλ'] = const['Tlambda']
const['kappa'] = 6.626E-34/6.65E-27 # m^2/s
const['xi0'] = 3.45E-10 # m zero temperature correlation length
const['ξ0'] = const['xi0']
const['nu'] = 0.6717 # correlation length critical exponent
const['ν'] = const['nu'] 
# ---------------------------------------------------------------------------------


# ---------------------------------------------------------------------------------
def convert_P(P,unit1,unit2):
    '''Convert pressures between units.'''
    conv = {}
    conv['atm->Pa'] = 101325.0
    conv['psi->Pa'] = 6894.7572
    conv['bar->Pa'] = 100000.0
    conv['mbar->Pa'] = 100.0
    conv['Pa->Pa'] = 1.0
    conv['torr->Pa'] = 133.322
    
    # convert the first unit to pascals
    P *= conv['%s->Pa'%unit1]
    
    # convert to the unit of choice
    P /= conv['%s->Pa'%unit2]
    
    return P

# ---------------------------------------------------------------------------------
def pressure_SVP(T):
    '''Pressure in Pa at saturated vapor pressure.
    
       Works for 0.0 <= T <= 5.0 K
       Source: http://pages.uoregon.edu/rjd/vapor18.htm (R.J. Donnelly) for 0.65 <= T <= 5.0
               
       Notes: for T < 0.65 we performed a fit to P(T) = (17.565)*T^(11.405)
    '''
    
    if np.min(T) < 0.0 or np.max(T) > 5.001:
        raise ValueError('0.0 <= T <= 5.0 K')
        
    T90 = np.arange(0.0,5.01,0.05)
    P = np.array([0.0, 2.54872922e-14, 6.91176244e-11, 7.04563507e-09, 
                  1.87436389e-07, 2.38846237e-06, 1.91066809e-05, 
                  1.10845327e-04, 5.08298718e-04, 1.94767546e-03, 
                  6.47714335e-03, 1.92074433e-02, 5.18143860e-02])
    P = np.append(P,[1.101E-01, 2.923E-01, 6.893E-01, 1.475E+00, 2.914E+00, 
                     5.380E+00, 9.381E+00, 1.558E+01, 2.479E+01, 3.802E+01, 
                     5.647E+01, 8.152E+01, 1.147E+02, 1.579E+02, 2.129E+02, 
                     2.819E+02, 3.673E+02, 4.715E+02, 5.971E+02, 7.465E+02, 
                     9.226E+02, 1.128E+03, 1.366E+03, 1.638E+03, 1.949E+03, 
                     2.299E+03, 2.692E+03, 3.130E+03, 3.613E+03, 4.141E+03, 
                     4.716E+03, 5.335E+03, 6.005E+03, 6.730E+03, 7.512E+03, 
                     8.354E+03, 9.258E+03, 1.023E+04, 1.127E+04, 1.237E+04, 
                     1.355E+04, 1.481E+04, 1.614E+04, 1.755E+04, 1.905E+04,     
                     2.063E+04, 2.229E+04, 2.405E+04, 2.589E+04, 2.783E+04, 
                     2.987E+04, 3.201E+04, 3.425E+04, 3.659E+04, 3.904E+04, 
                     4.160E+04, 4.426E+04, 4.705E+04, 4.994E+04, 5.296E+04, 
                     5.609E+04, 5.935E+04, 6.273E+04, 6.625E+04, 6.989E+04, 
                     7.366E+04, 7.757E+04, 8.162E+04, 8.580E+04, 9.013E+04, 
                     9.461E+04, 9.923E+04, 1.040E+05, 1.089E+05, 1.140E+05, 
                     1.193E+05, 1.247E+05, 1.303E+05, 1.360E+05, 1.419E+05, 
                     1.480E+05, 1.543E+05, 1.608E+05, 1.674E+05, 1.743E+05, 
                     1.813E+05, 1.886E+05, 1.960E+05])
    return scipy.interpolate.interp1d(T90,P,kind='cubic')(T)

# ---------------------------------------------------------------------------------
def density_SVP(T):
    '''The mass density of helium-4 in kg/m^3.
    
       Works for 0 <= T <= 5.0 K
       Source: http://pages.uoregon.edu/rjd/vapor2.htm (R.J. Donnelly)
    '''
    
    if np.min(T) < 0.0 or np.max(T) > 5.001:
        raise ValueError('0 <= T <= 5.0 K')
    
    T90 = np.arange(0.0,5.01,0.05)
    rho = np.array([1.451397E-1, 1.451397E-1, 1.451397E-1, 1.451396E-1, 1.451395E-1, 1.451395E-1, 1.451393E-1,
                    1.451391E-1, 1.451388E-1, 1.451384E-1, 1.451377E-1, 1.451368E-1, 1.451356E-1, 1.451342E-1,
                    1.451324E-1, 1.451304E-1, 1.451281E-1, 1.451257E-1, 1.451232E-1, 1.451207E-1, 1.451183E-1,
                    1.451163E-1, 1.451150E-1, 1.451144E-1, 1.451151E-1, 1.451173E-1, 1.451215E-1, 1.451281E-1,
                    1.451373E-1, 1.451493E-1, 1.451646E-1, 1.451837E-1, 1.452071E-1, 1.452352E-1, 1.452686E-1,
                    1.453079E-1, 1.453538E-1, 1.454070E-1, 1.454684E-1, 1.455394E-1, 1.456217E-1, 1.457181E-1,
                    1.458340E-1, 1.459840E-1, 1.461049E-1, 1.459877E-1, 1.458148E-1, 1.456071E-1, 1.453727E-1,
                    1.451162E-1, 1.448402E-1, 1.445467E-1, 1.442368E-1, 1.439114E-1, 1.435712E-1, 1.432164E-1,
                    1.428472E-1, 1.424638E-1, 1.420661E-1, 1.416538E-1, 1.412269E-1, 1.407850E-1, 1.403279E-1,
                    1.398551E-1, 1.393663E-1, 1.388611E-1, 1.383390E-1, 1.377997E-1, 1.372427E-1, 1.366675E-1,
                    1.360736E-1, 1.354605E-1, 1.348278E-1, 1.341748E-1, 1.335009E-1, 1.328054E-1, 1.320877E-1,
                    1.313467E-1, 1.305817E-1, 1.297914E-1, 1.289745E-1, 1.281296E-1, 1.272549E-1, 1.263483E-1,
                    1.254075E-1, 1.244297E-1, 1.234117E-1, 1.223498E-1, 1.212398E-1, 1.200768E-1, 1.188552E-1,
                    1.175686E-1, 1.162098E-1, 1.147706E-1, 1.132419E-1, 1.116131E-1, 1.098727E-1, 1.080076E-1,
                    1.060033E-1, 1.035300E-1, 1.008300E-1])
    rho *= 1.0E3
    return scipy.interpolate.interp1d(T90,rho,kind='cubic')(T)

# ---------------------------------------------------------------------------------
def superfluid_fraction_SVP(T):
    '''Superfluid fraction at saturated vapor pressure.
    
       Data taken from: http://pages.uoregon.edu/rjd/vapor3.htm
    '''
    
    if np.min(T) < 0.0 or np.max(T) > 5.001:
        raise ValueError('0 <= T <= 5.0 K')
        
    T90 = np.append(np.arange(0.0,2.101,0.05),[2.11,2.12,2.13,2.14,2.15,2.16,2.17,2.171,2.172,2.173,                                               2.174,2.175,2.176,2.1768])
    sfrac = np.array([1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000,                        1.000, 1.000, 1.000, 1.000, 0.999, 0.998, 0.997, 0.995, 0.993, 0.990, 0.986, 0.981,                        0.974, 0.966, 0.955, 0.942, 0.92346,0.909,0.889, 0.865, 0.838, 0.807, 0.771, 0.732, 0.687,                        0.636, 0.580, 0.518, 0.447, 0.362, 0.259, 0.236, 0.212, 0.186, 0.158, 0.128, 0.093,                        0.050, 0.045, 0.039, 0.033, 0.027, 0.021, 0.012, 0.000])
    fsvp = scipy.interpolate.interp1d(T90,sfrac,kind='cubic')
    
    def get_frac(T):
        if T > const['Tlambda']:
            return 0.0
        else:
            return fsvp(T)
    vf = np.vectorize(get_frac)
    return vf(T)

# ---------------------------------------------------------------------------------
def superfluid_density_SVP(T):
    '''The superfluid density at saturated vapor pressure in kg/m^3.
    
       Data taken from: http://pages.uoregon.edu/rjd/vapor3.htm
    '''
    
    if np.min(T) < 0.0 or np.max(T) > 5.001:
        raise ValueError('0 <= T <= 5.0 K')
    
    return superfluid_fraction_SVP(T)*density_SVP(T)


# ---------------------------------------------------------------------------------
def normal_density_SVP(T):
    '''The normal density at saturated vapor pressure in kg/m^3.
    
       Data taken from: http://pages.uoregon.edu/rjd/vapor3.htm
    '''
    
    if np.min(T) < 0.0 or np.max(T) > 5.001:
        raise ValueError('0 <= T <= 5.0 K')
    
    return (1.0-superfluid_fraction_SVP(T))*density_SVP(T)


# ---------------------------------------------------------------------------------
def density_data(T,P):
    '''The density in kg/m^3 of helium at P = 2.5 and P = 5.0 atm.
    
       Source http://pages.uoregon.edu/rjd/table.htm (Russel J.Donnelly) up to 2.1K
              http://webbook.nist.gov/chemistry/fluid/ from 2.1K 
              
        Notes: the data point for P=5atm and T=2.1K is added manually
    '''
    
    if np.min(T) < 0.0 or np.max(T) > 5.001:
        raise ValueError('0.0 <= T <= 5.0 K')
    
    T90 = np.array([0.0,0.1000,0.1500,0.2000,0.2500,0.3000,0.3500,0.4000,0.4500,0.5000,0.5500,0.6000,                    0.6500,0.7000,0.7500,0.8000,0.8500,0.9000,0.9500,1.0000,1.0500,1.1000,                    1.1500,1.2000,1.2500,1.3000,1.3500,1.4000,1.4500,1.5000,1.5500,1.6000,                    1.6500,1.7000,1.7500,1.8000,1.8500,1.9000,1.9500,2.0000,2.0500,2.1000,                    2.1768,2.1909,2.2050,2.2191,2.2333,2.2474,2.2615,2.2756,2.2897,2.3038,                    2.3180,2.3321,2.3462,2.3603,2.3744,2.3885,2.4027,2.4168,2.4309,2.4450,                    2.4591,2.4732,2.4874,2.5015,2.5156,2.5297,2.5438,2.5579,2.5720,2.5862,                    2.6003,2.6144,2.6285,2.6426,2.6567,2.6709,2.6850,2.6991,2.7132,2.7273,                    2.7414,2.7556,2.7697,2.7838,2.7979,2.8120,2.8261,2.8403,2.8544,2.8685,                    2.8826,2.8967,2.9108,2.9249,2.9391,2.9532,2.9673,2.9814,2.9955,3.0096,                    3.0238,3.0379,3.0520,3.0661,3.0802,3.0943,3.1085,3.1226,3.1367,3.1508,                    3.1649,3.1790,3.1932,3.2073,3.2214,3.2355,3.2496,3.2637,3.2778,3.2920,                    3.3061,3.3202,3.3343,3.3484,3.3625,3.3767,3.3908,3.4049,3.4190,3.4331,                    3.4472,3.4614,3.4755,3.4896,3.5037,3.5178,3.5319,3.5461,3.5602,3.5743,                    3.5884,3.6025,3.6166,3.6307,3.6449,3.6590,3.6731,3.6872,3.7013,3.7154,                    3.7296,3.7437,3.7578,3.7719,3.7860,3.8001,3.8143,3.8284,3.8425,3.8566,                    3.8707,3.8848,3.8990,3.9131,3.9272,3.9413,3.9554,3.9695,3.9836,3.9978,                    4.0119,4.0260,4.0401,4.0542,4.0683,4.0825,4.0966,4.1107,4.1248,4.1389,                    4.1530,4.1672,4.1813,4.1954,4.2095,4.2236,4.2377,4.2519,4.2660,4.2801,                    4.2942,4.3083,4.3224,4.3365,4.3507,4.3648,4.3789,4.3930,4.4071,4.4212,                    4.4354,4.4495,4.4636,4.4777,4.4918,4.5059,4.5201,4.5342,4.5483,4.5624,                    4.5765,4.5906,4.6048,4.6189,4.6330,4.6471,4.6612,4.6753,4.6894,4.7036,                    4.7177,4.7318,4.7459,4.7600,4.7741,4.7883,4.8024,4.8165,4.8306,4.8447,                    4.8588,4.8730,4.8871,4.9012,4.9153,4.9294,4.9435,4.9577,4.9718,4.9859,                    5.0000])
    
    rho = np.zeros([len(T90),2])
    rho[:,0] = np.array([149.2500,149.2500,149.2500,149.2500,149.2500,149.2500,149.2500,
                        149.2500,149.2500,149.2500,149.2500,149.2500,\
                        149.2500,149.2500,149.2500,149.2400,149.2400,149.2400,149.2400,149.2500,149.2500,149.2500,\
                        149.2600,149.2600,149.2700,149.2900,149.3000,149.3200,149.3400,149.3700,149.4100,149.4400,\
                        149.4900,149.5400,149.6000,149.6700,149.7500,149.8400,149.9600,150.0900,150.2400,150.4100,\
                        151.0100,150.9800,150.9500,150.9200,150.8800,150.8400,150.8100,150.7700,150.7200,150.6800,\
                        150.6400,150.5900,150.5400,150.4900,150.4400,150.3800,150.3300,150.2700,150.2100,150.1600,\
                        150.0900,150.0300,149.9700,149.9000,149.8400,149.7700,149.7000,149.6300,149.5600,149.4800,\
                        149.4100,149.3300,149.2500,149.1800,149.1000,149.0200,148.9300,148.8500,148.7700,148.6800,\
                        148.5900,148.5000,148.4200,148.3200,148.2300,148.1400,148.0500,147.9500,147.8500,147.7600,\
                        147.6600,147.5600,147.4600,147.3600,147.2500,147.1500,147.0400,146.9400,146.8300,146.7200,\
                        146.6100,146.5000,146.3900,146.2700,146.1600,146.0400,145.9300,145.8100,145.6900,145.5700,\
                        145.4500,145.3300,145.2000,145.0800,144.9500,144.8300,144.7000,144.5700,144.4400,144.3100,\
                        144.1700,144.0400,143.9000,143.7700,143.6300,143.4900,143.3500,143.2100,143.0700,142.9200,\
                        142.7800,142.6300,142.4800,142.3400,142.1900,142.0400,141.8800,141.7300,141.5700,141.4200,\
                        141.2600,141.1000,140.9400,140.7800,140.6100,140.4500,140.2800,140.1200,139.9500,139.7800,\
                        139.6100,139.4300,139.2600,139.0800,138.9000,138.7200,138.5400,138.3600,138.1800,137.9900,\
                        137.8000,137.6100,137.4200,137.2300,137.0400,136.8400,136.6400,136.4400,136.2400,136.0400,\
                        135.8400,135.6300,135.4200,135.2100,135.0000,134.7800,134.5700,134.3500,134.1300,133.9000,\
                        133.6800,133.4500,133.2200,132.9900,132.7500,132.5200,132.2800,132.0300,131.7900,131.5400,\
                        131.2900,131.0400,130.7900,130.5300,130.2700,130.0000,129.7400,129.4700,129.1900,128.9200,\
                        128.6400,128.3500,128.0700,127.7800,127.4800,127.1800,126.8800,126.5800,126.2700,125.9500,\
                        125.6300,125.3100,124.9800,124.6500,124.3100,123.9700,123.6200,123.2600,122.9000,122.5400,\
                        122.1600,121.7800,121.4000,121.0100,120.6100,120.2000,119.7800,119.3600,118.9200,118.4800,\
                        118.0300,117.5600,117.0900,116.6000,116.1000,115.5900,115.0700,114.5300,113.9700,113.4000,\
                        112.8100])
    rho[:,1] = np.array([152.8400,152.8400,152.8400,152.8400,152.8400,152.8400,152.8400,152.8400,152.8300,                        152.8300,152.8300,152.8300,                        152.8300,152.8300,152.8300,152.8300,152.8300,152.8300,152.8300,152.8400,152.8400,152.8500,                        152.8600,152.8700,152.8900,152.9100,152.9300,152.9600,152.9900,153.0300,153.0300,153.1300,                        153.1900,153.2700,153.3500,153.4500,153.5700,153.7000,153.8700,154.0500,154.2700,154.8000,                        155.0400,155.0200,154.9900,154.9600,154.9300,154.9000,154.8700,154.8400,154.8000,154.7600,                        154.7200,154.6800,154.6400,154.6000,154.5600,154.5100,154.4600,154.4200,154.3700,154.3200,                        154.2600,154.2100,154.1600,154.1000,154.0500,153.9900,153.9300,153.8700,153.8100,153.7500,                        153.6900,153.6200,153.5600,153.4900,153.4300,153.3600,153.2900,153.2200,153.1500,153.0800,                        153.0100,152.9300,152.8600,152.7800,152.7100,152.6300,152.5500,152.4700,152.3900,152.3100,                        152.2300,152.1500,152.0700,151.9800,151.9000,151.8100,151.7200,151.6400,151.5500,151.4600,                        151.3700,151.2800,151.1800,151.0900,151.0000,150.9000,150.8100,150.7100,150.6100,150.5200,                        150.4200,150.3200,150.2200,150.1200,150.0100,149.9100,149.8100,149.7000,149.6000,149.4900,                        149.3800,149.2800,149.1700,149.0600,148.9500,148.8400,148.7200,148.6100,148.5000,148.3800,                        148.2700,148.1500,148.0300,147.9100,147.7900,147.6800,147.5500,147.4300,147.3100,147.1900,                        147.0600,146.9400,146.8100,146.6900,146.5600,146.4300,146.3000,146.1700,146.0400,145.9100,                        145.7700,145.6400,145.5000,145.3700,145.2300,145.0900,144.9600,144.8200,144.6800,144.5300,                        144.3900,144.2500,144.1000,143.9600,143.8100,143.6700,143.5200,143.3700,143.2200,143.0700,                        142.9200,142.7600,142.6100,142.4600,142.3000,142.1400,141.9800,141.8300,141.6700,141.5000,                        141.3400,141.1800,141.0100,140.8500,140.6800,140.5200,140.3500,140.1800,140.0100,139.8300,                        139.6600,139.4900,139.3100,139.1300,138.9600,138.7800,138.6000,138.4200,138.2300,138.0500,                        137.8600,137.6800,137.4900,137.3000,137.1100,136.9200,136.7300,136.5300,136.3400,136.1400,                        135.9400,135.7400,135.5400,135.3400,135.1300,134.9300,134.7200,134.5100,134.3000,134.0900,                        133.8800,133.6600,133.4400,133.2300,133.0100,132.7900,132.5600,132.3400,132.1100,131.8800,                        131.6500,131.4200,131.1900,130.9500,130.7200,130.4800,130.2400,129.9900,129.7500,129.5000,                        129.2500])
    

    if np.abs(P-convert_P(2.5,'atm','Pa')) < 1.0E-6:
        i = 0
    elif np.abs(P-convert_P(5.0,'atm','Pa')) < 1.0E-6:
        i = 1
    else:
        raise ValueError('Only have P = 2.5 atm and 5.0 atm data at this time.')

    return scipy.interpolate.interp1d(T90,rho[:,i],kind='cubic')(T)


# ---------------------------------------------------------------------------------
def density_fit(T,P):
    '''The mass density in kg/m^3 as a function of temperature T in kelvin and pressure P in pascals. 
       Works for T >= 0.1 K and SVP <= P <= 5 atm
       Data from: http://pages.uoregon.edu/rjd/table3.htm
    
       Notes: The last data point for P = 5.0 atm was interpolated.
    '''
    
    if np.min(T) < 0.0 or np.max(T) > 5.001:
        raise ValueError('0.0 <= T <= 5.0 K')

    Psvp = pressure_SVP(T)
    
    if P < Psvp:
        raise ValueError('P < Psvp for T = %3.2f'%T)
    elif convert_P(P,'Pa','atm') > 5.0:
        raise ValueError('P > 5 atm')
        
    fP = convert_P(np.array([0.0,2.5,5]),'atm','Pa')
    fP[0] = Psvp
    frho = np.array([density_SVP(T),density_data(T,convert_P(2.5,'atm','Pa')),density_data(T,convert_P(5.0,'atm','Pa'))])
   
    f = scipy.interpolate.interp1d(fP,frho,kind='linear')
    return f(P)

density = np.vectorize(density_fit)


# ---------------------------------------------------------------------------------
def superfluid_fraction_fit(T,P):
    '''The superfluid fraction as a function of temperature T in kelvin and pressure P in pascals. 

       Works for T >= 0.1 K and SVP <= P <= 5 atm
       Data from: http://pages.uoregon.edu/rjd/table4.htm
    
       Notes: The last data point for P = 5.0 atm was interpolated.
    '''
    if np.min(T) < 0.0 or np.max(T) > 5.0:
        raise ValueError('0.0 <= T <= 5.0 K')
        
    if T > const['Tlambda']:
        return 0.0
    
    Psvp = pressure_SVP(T)
    
    if P < Psvp:
        raise ValueError('P < Psvp for T = %3.2f'%T)
    elif convert_P(P,'Pa','atm') > 5.0:
        raise ValueError('P > 5.0 atm')
        
    fP = convert_P(np.array([0.0,2.5,5]),'atm','Pa')
    fP[0] = Psvp
    
    T90 = np.append(np.arange(0.0,2.101,0.05),const['Tlambda'])
    sfrac = np.array([
        [1.0,1.0,1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.99999,\
         0.99998, 0.99996, 0.99991, 0.99980, 0.99958, 0.99917, 0.99849, 0.99741, 0.99577,\
         0.99339, 0.99003, 0.98550, 0.97955, 0.97215, 0.96306, 0.95208, 0.93927, 0.92346,\
         0.90410, 0.88235, 0.85880, 0.83261, 0.80187, 0.76546, 0.72272, 0.67564, 0.62301,\
         0.56383, 0.49884, 0.41842, 0.32603, 0.23033, 0.00000],
        [1.0,1.0,1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.99999,\
         0.99998, 0.99996, 0.99990, 0.99977, 0.99951, 0.99904, 0.99825, 0.99702, 0.99517,\
         0.99252, 0.98880, 0.98000, 0.97738, 0.96934, 0.95940, 0.94765, 0.93362, 0.91662,\
         0.89572, 0.87242, 0.84716, 0.81900, 0.78605, 0.74670, 0.70194, 0.65106, 0.59352,\
         0.52922, 0.45845, 0.37204, 0.27155, 0.16761, 0.00000],
        [1.0,1.0,1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 1.0000, 9.9999E-1, 9.9998E-1,\
         9.9995E-1, 9.9987E-1, 9.9971E-1, 9.9938E-1, 9.9880E-1, 9.9785E-1, 9.9637E-1, 9.9421E-1,\
         9.9116E-1, 9.8698E-1, 9.8147E-1, 9.7450E-1, 9.6577E-1, 9.5491E-1, 9.4235E-1, 9.2695E-1,\
         9.0852E-1, 8.8608E-1, 8.6099E-1, 8.3380E-1, 8.0345E-1, 7.6782E-1, 7.2543E-1, 6.7820E-1,\
         6.2303E-1, 5.5966E-1, 4.8891E-1, 4.1119E-1, 3.1719E-1, 2.0819E-1,1.0146417E-1,0.0]])
    
    fsvp = scipy.interpolate.interp1d(T90,sfrac[0,:],kind='cubic')
    f25 = scipy.interpolate.interp1d(T90,sfrac[1,:],kind='cubic')
    f50 = scipy.interpolate.interp1d(T90,sfrac[2,],kind='cubic')
    
    fratio = np.array([fsvp(T),f25(T),f50(T)])
    
    f = scipy.interpolate.interp1d(fP,fratio,kind='linear')
    return f(P)

superfluid_fraction = np.vectorize(superfluid_fraction_fit)

# ---------------------------------------------------------------------------------
def superfluid_density(T,P):
    '''The superfluid mass density in kg/m^3 as a function of temperature T in 
       kelvin and pressure P in pascals. 
       
       Data from: http://pages.uoregon.edu/rjd/table4.htm
    '''
    return superfluid_fraction(T,P)*density(T,P)

# ---------------------------------------------------------------------------------
def normal_density(T,P):
    '''The normal mass density in kg/m^3 as a function of temperature T in kelvin and pressure P in pascals. 
       
       Data from: http://pages.uoregon.edu/rjd/table4.htm
    '''
    return (1.0-superfluid_fraction(T,P))*density(T,P)

# ---------------------------------------------------------------------------------
def lambda_dB(T, mass=const['mass']):
    ''' Thermal de Broglie wavelength in meters.'''
    return constants.h / np.sqrt(2.0*constants.pi*mass*constants.k*T)

def ΛdB(T,mass=const['mass']):
    ''' Thermal de Broglie wavelength in meters.'''
    return labmda_dB(T,mass)

# ---------------------------------------------------------------------------------
def B2(T,use_donnelly = True, use_DCGT1=False):
    '''Second virial coefficient for helium in m^3/mol (see https://doi.org/10.1088/0026-1394/46/5/017)'''
    if use_donnelly:
        return B2_donnelly(T)
    
    b1 = 18.5604
    b2 = -2.22170
    b3 = -412.863
    b4 = -0.03094
    B = b1 + b2/np.sqrt(T**3) + b3/T + b4*T
    
    if use_DCGT1:
        b1 = 18.2665
        b2 = -10.2405
        b3 = -407.939
        b4 = -0.02612
        B = b1 + b2/np.sqrt(T**3) + b3/T + b4*T
    return B/(100**3)

# ---------------------------------------------------------------------------------
def B2_donnelly(T):
    '''Second virial coefficient for helium in m^3/mol'''
    a = 2.305E-5 #[m^3/mol]
    b = 4.2177E-4 #[m^3 K/mol]
    return a - b/T

# ---------------------------------------------------------------------------------
def molar_volume(P,T):
    ''' The molar gas volume in m^3 as a function of pressure [Pa] and
        temperature [K].
    '''
    return ((constants.R * T)/(2.0 * P))*(1.0 + 1.0*np.sqrt(np.abs(1.0 + 4.0*B2(T)*P/(constants.R * T))))

# ---------------------------------------------------------------------------------
def chemical_potential(P,T):
    '''Chemical potential in K as a function of pressure [Pa] and temperature [K].
    '''
    t1 = - T * np.log(molar_volume(P,T)/(lambda_dB(T)**3 * constants.N_A))
    t2 = 3.0*T*B2(T)/molar_volume(P,T)
    return t1+t2

μ = lambda P,T: chemical_potential(P,T)

# ---------------------------------------------------------------------------------
def pressure_fit(μ_,T):
    '''Pressure at SVP in Pa as a function of chemical potential (in K) and
    temperature.
    '''

    import pynverse
    P0 = pressure_SVP(T)
    Pμ = pynverse.inversefunc(chemical_potential, args=(T),domain=[1E-200,None], 
                               accuracy=8)
    return Pμ(μ_)

# ---------------------------------------------------------------------------------
def pressure(μ_,T):
    ''' Pressure at SVP in Pa as a function of chemical potential (in K) and
    temperature.'''
    λ = np.exp(μ_/T)
    Q_1_over_V = 1/lambda_dB(T)**3
    z = λ*Q_1_over_V
    b_2 = -B2(T) / constants.N_A
    P = z + b_2*z**2
    return constants.k*T*P

# ---------------------------------------------------------------------------------
def xi(T):
    '''The temperature dependent correlation length in m.'''
    return const['xi0']/abs(1 - T/const['Tlambda'])**(const['nu'])

ξ = lambda T: xi(T)
# ---------------------------------------------------------------------------------
helium_dispersion_svp_knots = np.array([0.0894,
                  0.0894,
                  0.0894,
                  0.0894,
                  0.15,
                  0.510,
                  1.60,
                  2.023,
                  2.42,
                  2.665,
                  3.60,
                  3.60,
                  3.60,
                  3.60
                 ])
helium_dispersion_svp_coefficients = np.array([1.53895,
                         1.932,
                         4.8,
                         14.85,
                         14.88,
                         5.9384,
                         16.5014,
                         17.72455,
                         18.43656,
                         18.43545
                        ])
_helium_dispersion_svp = scipy.interpolate.BSpline(helium_dispersion_svp_knots, helium_dispersion_svp_coefficients, 3)
def helium_dispersion_svp(x):
    """
        Evaluate helium dispersion at SVP below T_lambda.
        Data for spline fits from Donnelly 1998.
        See https://doi.org/10.1063/1.556028 for more details.

        Parameters
        ----------
        x : array_like
            points to evaluate the helium dispersion at in reciprocal angstrom

        Returns
        -------
        y : array_like
            helium dispersion at SVP below T_lambda in kelvin in shape of `x`


    """
    return _helium_dispersion_svp(x)
# ---------------------------------------------------------------------------------
