import * as iam from '@aws-cdk/aws-iam';
import { Construct, IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Alias } from './alias';
/**
 * A KMS Key, either managed by this CDK app, or imported.
 */
export interface IKey extends IResource {
    /**
     * The ARN of the key.
     *
     * @attribute
     */
    readonly keyArn: string;
    /**
     * The ID of the key
     * (the part that looks something like: 1234abcd-12ab-34cd-56ef-1234567890ab).
     *
     * @attribute
     */
    readonly keyId: string;
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias: string): Alias;
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement: iam.PolicyStatement, allowNoOp?: boolean): iam.AddToResourcePolicyResult;
    /**
     * Grant the indicated permissions on this key to the given principal
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee: iam.IGrantable): iam.Grant;
}
declare abstract class KeyBase extends Resource implements IKey {
    /**
     * The ARN of the key.
     */
    abstract readonly keyArn: string;
    abstract readonly keyId: string;
    /**
     * Optional policy document that represents the resource policy of this key.
     *
     * If specified, addToResourcePolicy can be used to edit this policy.
     * Otherwise this method will no-op.
     */
    protected abstract readonly policy?: iam.PolicyDocument;
    /**
     * Optional property to control trusting account identities.
     *
     * If specified grants will default identity policies instead of to both
     * resource and identity policies.
     */
    protected abstract readonly trustAccountIdentities: boolean;
    /**
     * Collection of aliases added to the key
     *
     * Tracked to determine whether or not the aliasName should be added to the end of its ID
     */
    private readonly aliases;
    /**
     * Defines a new alias for the key.
     */
    addAlias(aliasName: string): Alias;
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement: iam.PolicyStatement, allowNoOp?: boolean): iam.AddToResourcePolicyResult;
    protected validate(): string[];
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    private granteeStackDependsOnKeyStack;
    private principalIsANewlyCreatedResource;
    private isGranteeFromAnotherRegion;
    private isGranteeFromAnotherAccount;
}
/**
 * Construction properties for a KMS Key object
 */
export interface KeyProps {
    /**
     * A description of the key. Use a description that helps your users decide
     * whether the key is appropriate for a particular task.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * Initial alias to add to the key
     *
     * More aliases can be added later by calling `addAlias`.
     *
     * @default - No alias is added for the key.
     */
    readonly alias?: string;
    /**
     * Indicates whether AWS KMS rotates the key.
     *
     * @default false
     */
    readonly enableKeyRotation?: boolean;
    /**
     * Indicates whether the key is available for use.
     *
     * @default - Key is enabled.
     */
    readonly enabled?: boolean;
    /**
     * Custom policy document to attach to the KMS key.
     *
     * @default - A policy document with permissions for the account root to
     * administer the key will be created.
     */
    readonly policy?: iam.PolicyDocument;
    /**
     * Whether the encryption key should be retained when it is removed from the Stack. This is useful when one wants to
     * retain access to data that was encrypted with a key that is being retired.
     *
     * @default RemovalPolicy.Retain
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Whether the key usage can be granted by IAM policies
     *
     * Setting this to true adds a default statement which delegates key
     * access control completely to the identity's IAM policy (similar
     * to how it works for other AWS resources).
     *
     * @default false
     * @see https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam
     */
    readonly trustAccountIdentities?: boolean;
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
export declare class Key extends KeyBase {
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope: Construct, id: string, keyArn: string): IKey;
    readonly keyArn: string;
    readonly keyId: string;
    protected readonly policy?: iam.PolicyDocument;
    protected readonly trustAccountIdentities: boolean;
    constructor(scope: Construct, id: string, props?: KeyProps);
    private allowAccountIdentitiesToControl;
    /**
     * Let users or IAM policies from this account admin this key.
     * @link https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    private allowAccountToAdmin;
}
export {};
