"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Key = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
class KeyBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Collection of aliases added to the key
         *
         * Tracked to determine whether or not the aliasName should be added to the end of its ID
         */
        this.aliases = [];
    }
    /**
     * Defines a new alias for the key.
     */
    addAlias(aliasName) {
        const aliasId = this.aliases.length > 0 ? `Alias${aliasName}` : 'Alias';
        const alias = new alias_1.Alias(this, aliasId, { aliasName, targetKey: this });
        this.aliases.push(alias);
        return alias;
    }
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = core_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return { statementAdded: false };
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee, ...actions) {
        // KMS verifies whether the principals included in its key policy actually exist.
        // This is a problem if the stack the grantee is part of depends on the key stack
        // (as it won't exist before the key policy is attempted to be created).
        // In that case, make the account the resource policy principal
        const granteeStackDependsOnKeyStack = this.granteeStackDependsOnKeyStack(grantee);
        const principal = granteeStackDependsOnKeyStack
            ? new iam.AccountPrincipal(granteeStackDependsOnKeyStack)
            : grantee.grantPrincipal;
        const crossAccountAccess = this.isGranteeFromAnotherAccount(grantee);
        const crossRegionAccess = this.isGranteeFromAnotherRegion(grantee);
        const crossEnvironment = crossAccountAccess || crossRegionAccess;
        const grantOptions = {
            grantee,
            actions,
            resource: this,
            resourceArns: [this.keyArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        };
        if (this.trustAccountIdentities) {
            return iam.Grant.addToPrincipalOrResource(grantOptions);
        }
        else {
            return iam.Grant.addToPrincipalAndResource({
                ...grantOptions,
                // if the key is used in a cross-environment matter,
                // we can't access the Key ARN (they don't have physical names),
                // so fall back to using '*'. ToDo we need to make this better... somehow
                resourceArns: crossEnvironment ? ['*'] : [this.keyArn],
                resourcePolicyPrincipal: principal,
            });
        }
    }
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt');
    }
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    granteeStackDependsOnKeyStack(grantee) {
        const grantPrincipal = grantee.grantPrincipal;
        if (!(core_1.Construct.isConstruct(grantPrincipal))) {
            return undefined;
        }
        // this logic should only apply to newly created
        // (= not imported) resources
        if (!this.principalIsANewlyCreatedResource(grantPrincipal)) {
            return undefined;
        }
        // return undefined;
        const keyStack = core_1.Stack.of(this);
        const granteeStack = core_1.Stack.of(grantPrincipal);
        if (keyStack === granteeStack) {
            return undefined;
        }
        return granteeStack.dependencies.includes(keyStack)
            ? granteeStack.account
            : undefined;
    }
    principalIsANewlyCreatedResource(principal) {
        // yes, this sucks
        // this is just a temporary stopgap to stem the bleeding while we work on a proper fix
        return principal instanceof iam.Role ||
            principal instanceof iam.User ||
            principal instanceof iam.Group;
    }
    isGranteeFromAnotherRegion(grantee) {
        if (!(core_1.Construct.isConstruct(grantee))) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.region !== identityStack.region;
    }
    isGranteeFromAnotherAccount(grantee) {
        if (!(core_1.Construct.isConstruct(grantee))) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.account !== identityStack.account;
    }
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.policy = props.policy || new iam.PolicyDocument();
        this.trustAccountIdentities = props.trustAccountIdentities || false;
        if (this.trustAccountIdentities) {
            this.allowAccountIdentitiesToControl();
        }
        else {
            this.allowAccountToAdmin();
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
        });
        this.keyArn = resource.attrArn;
        this.keyId = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy);
        if (props.alias !== undefined) {
            this.addAlias(props.alias);
        }
    }
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor(keyId) {
                super(scope, id);
                this.keyArn = keyArn;
                this.policy = undefined;
                // defaulting true: if we are importing the key the key policy is
                // undefined and impossible to change here; this means updating identity
                // policies is really the only option
                this.trustAccountIdentities = true;
                this.keyId = keyId;
            }
        }
        const keyResourceName = core_1.Stack.of(scope).parseArn(keyArn).resourceName;
        if (!keyResourceName) {
            throw new Error(`KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key/<keyId>', got: '${keyArn}'`);
        }
        return new Import(keyResourceName);
    }
    allowAccountIdentitiesToControl() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['kms:*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Let users or IAM policies from this account admin this key.
     * @link https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    allowAccountToAdmin() {
        const actions = [
            'kms:Create*',
            'kms:Describe*',
            'kms:Enable*',
            'kms:List*',
            'kms:Put*',
            'kms:Update*',
            'kms:Revoke*',
            'kms:Disable*',
            'kms:Get*',
            'kms:Delete*',
            'kms:ScheduleKeyDeletion',
            'kms:CancelKeyDeletion',
            'kms:GenerateDataKey',
            'kms:TagResource',
            'kms:UntagResource',
        ];
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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