"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk_1 = require("@aws-cdk/cdk");
const os_1 = require("os");
const bucket_policy_1 = require("./bucket-policy");
const notifications_resource_1 = require("./notifications-resource");
const perms = require("./perms");
const s3_generated_1 = require("./s3.generated");
const util_1 = require("./util");
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
class BucketBase extends cdk_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Indicates if a bucket resource policy should automatically created upon
         * the first call to `addToResourcePolicy`.
         */
        this.autoCreatePolicy = false;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.s3'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                resources: {
                    ARN: options.paths ? options.paths.map(p => this.arnForObjects(p)) : [this.bucketArn],
                },
            }
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailPutObject(id, options) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutObject'],
            },
        });
        return rule;
    }
    /**
     * Adds a statement to the resource policy for a principal (i.e.
     * account/role/service) to perform actions on this bucket and/or it's
     * contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for
     * this bucket or objects.
     */
    addToResourcePolicy(permission) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new bucket_policy_1.BucketPolicy(this, 'Policy', { bucket: this });
        }
        if (this.policy) {
            this.policy.document.addStatement(permission);
        }
    }
    /**
     * The https URL of an S3 object. For example:
     * @example https://s3.us-west-1.amazonaws.com/onlybucket
     * @example https://s3.us-west-1.amazonaws.com/bucket/key
     * @example https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    urlForObject(key) {
        const components = [`https://s3.${this.node.stack.region}.${this.node.stack.urlSuffix}/${this.bucketName}`];
        if (key) {
            // trim prepending '/'
            if (typeof key === 'string' && key.startsWith('/')) {
                key = key.substr(1);
            }
            components.push('/');
            components.push(key);
        }
        return components.join('');
    }
    /**
     * Returns an ARN that represents all objects within the bucket that match
     * the key pattern specified. To represent all keys, specify ``"*"``.
     *
     * If you specify multiple components for keyPattern, they will be concatenated::
     *
     *   arnForObjects('home/', team, '/', user, '/*')
     *
     */
    arnForObjects(keyPattern) {
        return `${this.bucketArn}/${keyPattern}`;
    }
    /**
     * Grant read permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantRead(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_READ_ACTIONS, perms.KEY_READ_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantWrite(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_WRITE_ACTIONS, perms.KEY_WRITE_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantPut(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_PUT_ACTIONS, perms.KEY_WRITE_ACTIONS, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects
     * in this bucket.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantDelete(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_DELETE_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantReadWrite(identity, objectsKeyPattern = '*') {
        const bucketActions = perms.BUCKET_READ_ACTIONS.concat(perms.BUCKET_WRITE_ACTIONS);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.grant(identity, bucketActions, keyActions, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *     const grant = bucket.grantPublicAccess();
     *     grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow. Default is "s3:GetObject".
     */
    grantPublicAccess(keyPrefix = '*', ...allowedActions) {
        if (this.disallowPublicAccess) {
            throw new Error("Cannot grant public access when 'blockPublicPolicy' is enabled");
        }
        allowedActions = allowedActions.length > 0 ? allowedActions : ['s3:GetObject'];
        return iam.Grant.addToPrincipalOrResource({
            actions: allowedActions,
            resourceArns: [this.arnForObjects(keyPrefix)],
            grantee: new iam.Anyone(),
            resource: this,
        });
    }
    grant(grantee, bucketActions, keyActions, resourceArn, ...otherResourceArns) {
        const resources = [resourceArn, ...otherResourceArns];
        const ret = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: bucketActions,
            resourceArns: resources,
            resource: this,
        });
        if (this.encryptionKey) {
            this.encryptionKey.grant(grantee, ...keyActions);
        }
        return ret;
    }
}
class BlockPublicAccess {
    constructor(options) {
        this.blockPublicAcls = options.blockPublicAcls;
        this.blockPublicPolicy = options.blockPublicPolicy;
        this.ignorePublicAcls = options.ignorePublicAcls;
        this.restrictPublicBuckets = options.restrictPublicBuckets;
    }
}
BlockPublicAccess.BlockAll = new BlockPublicAccess({
    blockPublicAcls: true,
    blockPublicPolicy: true,
    ignorePublicAcls: true,
    restrictPublicBuckets: true
});
BlockPublicAccess.BlockAcls = new BlockPublicAccess({
    blockPublicAcls: true,
    ignorePublicAcls: true
});
exports.BlockPublicAccess = BlockPublicAccess;
/**
 * An S3 bucket with associated policy objects
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 */
class Bucket extends BucketBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.autoCreatePolicy = true;
        this.lifecycleRules = [];
        this.metrics = [];
        const { bucketEncryption, encryptionKey } = this.parseEncryption(props);
        if (props.bucketName && !cdk_1.Token.isToken(props.bucketName)) {
            this.validateBucketName(props.bucketName);
        }
        const resource = new s3_generated_1.CfnBucket(this, 'Resource', {
            bucketName: props && props.bucketName,
            bucketEncryption,
            versioningConfiguration: props.versioned ? { status: 'Enabled' } : undefined,
            lifecycleConfiguration: new cdk_1.Token(() => this.parseLifecycleConfiguration()),
            websiteConfiguration: this.renderWebsiteConfiguration(props),
            publicAccessBlockConfiguration: props.blockPublicAccess,
            metricsConfigurations: new cdk_1.Token(() => this.parseMetricConfiguration())
        });
        cdk_1.applyRemovalPolicy(resource, props.removalPolicy !== undefined ? props.removalPolicy : cdk_1.RemovalPolicy.Orphan);
        this.versioned = props.versioned;
        this.encryptionKey = encryptionKey;
        this.bucketArn = resource.bucketArn;
        this.bucketName = resource.bucketName;
        this.bucketDomainName = resource.bucketDomainName;
        this.bucketWebsiteUrl = resource.bucketWebsiteUrl;
        this.bucketDualStackDomainName = resource.bucketDualStackDomainName;
        this.bucketRegionalDomainName = resource.bucketRegionalDomainName;
        this.disallowPublicAccess = props.blockPublicAccess && props.blockPublicAccess.blockPublicPolicy;
        // Add all bucket metric configurations rules
        (props.metrics || []).forEach(this.addMetric.bind(this));
        // Add all lifecycle rules
        (props.lifecycleRules || []).forEach(this.addLifecycleRule.bind(this));
        // defines a BucketNotifications construct. Notice that an actual resource will only
        // be added if there are notifications added, so we don't need to condition this.
        this.notifications = new notifications_resource_1.BucketNotifications(this, 'Notifications', { bucket: this });
        if (props.publicReadAccess) {
            this.grantPublicAccess();
        }
    }
    static fromBucketArn(scope, id, bucketArn) {
        return Bucket.fromBucketAttributes(scope, id, { bucketArn });
    }
    static fromBucketName(scope, id, bucketName) {
        return Bucket.fromBucketAttributes(scope, id, { bucketName });
    }
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object. Can be obtained from a call to
     * `bucket.export()` or manually created.
     */
    static fromBucketAttributes(scope, id, attrs) {
        const region = scope.node.stack.region;
        const urlSuffix = scope.node.stack.urlSuffix;
        const bucketName = util_1.parseBucketName(scope, attrs);
        if (!bucketName) {
            throw new Error('Bucket name is required');
        }
        const newUrlFormat = attrs.bucketWebsiteNewUrlFormat === undefined
            ? false
            : attrs.bucketWebsiteNewUrlFormat;
        const websiteUrl = newUrlFormat
            ? `${bucketName}.s3-website.${region}.${urlSuffix}`
            : `${bucketName}.s3-website-${region}.${urlSuffix}`;
        class Import extends BucketBase {
            constructor() {
                super(...arguments);
                this.bucketName = bucketName;
                this.bucketArn = util_1.parseBucketArn(scope, attrs);
                this.bucketDomainName = attrs.bucketDomainName || `${bucketName}.s3.${urlSuffix}`;
                this.bucketWebsiteUrl = attrs.bucketWebsiteUrl || websiteUrl;
                this.bucketRegionalDomainName = attrs.bucketRegionalDomainName || `${bucketName}.s3.${region}.${urlSuffix}`;
                this.bucketDualStackDomainName = attrs.bucketDualStackDomainName || `${bucketName}.s3.dualstack.${region}.${urlSuffix}`;
                this.bucketWebsiteNewUrlFormat = newUrlFormat;
                this.policy = undefined;
                this.autoCreatePolicy = false;
                this.disallowPublicAccess = false;
            }
            /**
             * Exports this bucket from the stack.
             */
            export() {
                return attrs;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a lifecycle rule to the bucket
     *
     * @param rule The rule to add
     */
    addLifecycleRule(rule) {
        if ((rule.noncurrentVersionExpirationInDays !== undefined
            || (rule.noncurrentVersionTransitions && rule.noncurrentVersionTransitions.length > 0))
            && !this.versioned) {
            throw new Error("Cannot use 'noncurrent' rules on a nonversioned bucket");
        }
        this.lifecycleRules.push(rule);
    }
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add
     */
    addMetric(metric) {
        this.metrics.push(metric);
    }
    /**
     * Adds a bucket notification event destination.
     * @param event The event to trigger the notification
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue)
     *
     * @param filters S3 object key filter rules to determine which objects
     * trigger this event. Each filter must include a `prefix` and/or `suffix`
     * that will be matched against the s3 object key. Refer to the S3 Developer Guide
     * for details about allowed filter rules.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-filtering
     *
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     */
    addEventNotification(event, dest, ...filters) {
        this.notifications.addNotification(event, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * created in the bucket. This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectCreatedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.ObjectCreated, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * removed from the bucket. This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectRemovedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.ObjectRemoved, dest, ...filters);
    }
    validateBucketName(bucketName) {
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
        if (bucketName.length < 3 || bucketName.length > 63) {
            errors.push('Bucket name must be at least 3 and no more than 63 characters');
        }
        const charsetMatch = bucketName.match(/[^a-z0-9.-]/);
        if (charsetMatch) {
            errors.push('Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) '
                + `(offset: ${charsetMatch.index})`);
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(0))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + '(offset: 0)');
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(bucketName.length - 1))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + `(offset: ${bucketName.length - 1})`);
        }
        const consecSymbolMatch = bucketName.match(/\.-|-\.|\.\./);
        if (consecSymbolMatch) {
            errors.push('Bucket name must not have dash next to period, or period next to dash, or consecutive periods '
                + `(offset: ${consecSymbolMatch.index})`);
        }
        if (/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(bucketName)) {
            errors.push('Bucket name must not resemble an IP address');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid S3 bucket name (value: ${bucketName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default to unencrypted.
        const encryptionType = props.encryption || BucketEncryption.Unencrypted;
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== BucketEncryption.Kms && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === BucketEncryption.Unencrypted) {
            return { bucketEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === BucketEncryption.Kms) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`
            });
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        serverSideEncryptionByDefault: {
                            sseAlgorithm: 'aws:kms',
                            kmsMasterKeyId: encryptionKey.keyArn
                        }
                    }
                ]
            };
            return { encryptionKey, bucketEncryption };
        }
        if (encryptionType === BucketEncryption.S3Managed) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'AES256' } }
                ]
            };
            return { bucketEncryption };
        }
        if (encryptionType === BucketEncryption.KmsManaged) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'aws:kms' } }
                ]
            };
            return { bucketEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    /**
     * Parse the lifecycle configuration out of the uucket props
     * @param props Par
     */
    parseLifecycleConfiguration() {
        if (!this.lifecycleRules || this.lifecycleRules.length === 0) {
            return undefined;
        }
        const self = this;
        return { rules: this.lifecycleRules.map(parseLifecycleRule) };
        function parseLifecycleRule(rule) {
            const enabled = rule.enabled !== undefined ? rule.enabled : true;
            const x = {
                // tslint:disable-next-line:max-line-length
                abortIncompleteMultipartUpload: rule.abortIncompleteMultipartUploadAfterDays !== undefined ? { daysAfterInitiation: rule.abortIncompleteMultipartUploadAfterDays } : undefined,
                expirationDate: rule.expirationDate,
                expirationInDays: rule.expirationInDays,
                id: rule.id,
                noncurrentVersionExpirationInDays: rule.noncurrentVersionExpirationInDays,
                noncurrentVersionTransitions: rule.noncurrentVersionTransitions,
                prefix: rule.prefix,
                status: enabled ? 'Enabled' : 'Disabled',
                transitions: rule.transitions,
                tagFilters: self.parseTagFilters(rule.tagFilters)
            };
            return x;
        }
    }
    parseMetricConfiguration() {
        if (!this.metrics || this.metrics.length === 0) {
            return undefined;
        }
        const self = this;
        return this.metrics.map(parseMetric);
        function parseMetric(metric) {
            return {
                id: metric.id,
                prefix: metric.prefix,
                tagFilters: self.parseTagFilters(metric.tagFilters)
            };
        }
    }
    parseTagFilters(tagFilters) {
        if (!tagFilters || tagFilters.length === 0) {
            return undefined;
        }
        return Object.keys(tagFilters).map(tag => ({
            key: tag,
            value: tagFilters[tag]
        }));
    }
    renderWebsiteConfiguration(props) {
        if (!props.websiteErrorDocument && !props.websiteIndexDocument) {
            return undefined;
        }
        if (props.websiteErrorDocument && !props.websiteIndexDocument) {
            throw new Error(`"websiteIndexDocument" is required if "websiteErrorDocument" is set`);
        }
        return {
            indexDocument: props.websiteIndexDocument,
            errorDocument: props.websiteErrorDocument
        };
    }
}
exports.Bucket = Bucket;
/**
 * What kind of server-side encryption to apply to this bucket
 */
var BucketEncryption;
(function (BucketEncryption) {
    /**
     * Objects in the bucket are not encrypted.
     */
    BucketEncryption["Unencrypted"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by KMS.
     */
    BucketEncryption["KmsManaged"] = "MANAGED";
    /**
     * Server-side encryption with a master key managed by S3.
     */
    BucketEncryption["S3Managed"] = "S3MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    BucketEncryption["Kms"] = "KMS";
})(BucketEncryption = exports.BucketEncryption || (exports.BucketEncryption = {}));
/**
 * Notification event types.
 */
var EventType;
(function (EventType) {
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["ObjectCreated"] = "s3:ObjectCreated:*";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["ObjectCreatedPut"] = "s3:ObjectCreated:Put";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["ObjectCreatedPost"] = "s3:ObjectCreated:Post";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["ObjectCreatedCopy"] = "s3:ObjectCreated:Copy";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["ObjectCreatedCompleteMultipartUpload"] = "s3:ObjectCreated:CompleteMultipartUpload";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["ObjectRemoved"] = "s3:ObjectRemoved:*";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["ObjectRemovedDelete"] = "s3:ObjectRemoved:Delete";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["ObjectRemovedDeleteMarkerCreated"] = "s3:ObjectRemoved:DeleteMarkerCreated";
    /**
     * You can use this event type to request Amazon S3 to send a notification
     * message when Amazon S3 detects that an object of the RRS storage class is
     * lost.
     */
    EventType["ReducedRedundancyLostObject"] = "s3:ReducedRedundancyLostObject";
})(EventType = exports.EventType || (exports.EventType = {}));
//# sourceMappingURL=data:application/json;base64,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