"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const os_1 = require("os");
const s3 = require("../lib");
const lib_1 = require("../lib");
module.exports = {
    'default bucket'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'CFN properties are type-validated during resolution'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            bucketName: new cdk.Token(() => 5).toString() // Oh no
        });
        test.throws(() => {
            assert_1.SynthUtils.toCloudFormation(stack);
        }, /bucketName: 5 should be a string/);
        test.done();
    },
    'bucket without encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.Unencrypted
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with managed encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KmsManaged
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'valid bucket names'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'abc.xyz-34ab'
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '124.pp--33'
        }));
        test.done();
    },
    'bucket validation skips tokenized values'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: new cdk.Token(() => '_BUCKET').toString()
        }));
        test.done();
    },
    'fails with message on invalid bucket names'(test) {
        const stack = new cdk.Stack();
        const bucket = `-buckEt.-${new Array(65).join('$')}`;
        const expectedErrors = [
            `Invalid S3 bucket name (value: ${bucket})`,
            'Bucket name must be at least 3 and no more than 63 characters',
            'Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) (offset: 5)',
            'Bucket name must start and end with a lowercase character or number (offset: 0)',
            `Bucket name must start and end with a lowercase character or number (offset: ${bucket.length - 1})`,
            'Bucket name must not have dash next to period, or period next to dash, or consecutive periods (offset: 7)',
        ].join(os_1.EOL);
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: bucket
            // tslint:disable-next-line:only-arrow-functions
        }), function (err) {
            return expectedErrors === err.message;
        });
        test.done();
    },
    'fails if bucket name has less than 3 or more than 63 characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'a'
        }), /at least 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: new Array(65).join('x')
        }), /no more than 63/);
        test.done();
    },
    'fails if bucket name has invalid characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'b@cket'
        }), /offset: 1/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucKet'
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'bučket'
        }), /offset: 2/);
        test.done();
    },
    'fails if bucket name does not start or end with lowercase character or number'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '-ucket'
        }), /offset: 0/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucke.'
        }), /offset: 5/);
        test.done();
    },
    'fails only if bucket name has the consecutive symbols (..), (.-), (-.)'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'buc..ket'
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'buck.-et'
        }), /offset: 4/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'b-.ucket'
        }), /offset: 1/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: 'bu--cket'
        }));
        test.done();
    },
    'fails only if bucket name resembles IP address'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '1.2.3.4'
        }), /must not resemble an IP address/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '1.2.3'
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: '1.2.3.a'
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: '1000.2.3.4'
        }));
        test.done();
    },
    'fails if encryption key is used with managed encryption'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KmsManaged,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'fails if encryption key is used with encryption set to unencrypted'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.Unencrypted,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'encryptionKey can specify kms key'(test) {
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'MyKey', { description: 'hello, world' });
        new s3.Bucket(stack, 'MyBucket', { encryptionKey, encryption: s3.BucketEncryption.Kms });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyKey6AB29FA6": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "hello, world",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain"
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "KMSMasterKeyID": {
                                            "Fn::GetAtt": [
                                                "MyKey6AB29FA6",
                                                "Arn"
                                            ]
                                        },
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with versioning turned on'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            versioned: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "VersioningConfiguration": {
                            "Status": "Enabled"
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAll'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BlockAll,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "BlockPublicPolicy": true,
                            "IgnorePublicAcls": true,
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAcls'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BlockAcls,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "IgnorePublicAcls": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with custom block public access setting'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: new s3.BlockPublicAccess({ restrictPublicBuckets: true })
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'permissions': {
        'addPermission creates a bucket policy'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            bucket.addToResourcePolicy(new iam.PolicyStatement().addResource('foo').addAction('bar'));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                    },
                    "MyBucketPolicyE7FBAC7B": {
                        "Type": "AWS::S3::BucketPolicy",
                        "Properties": {
                            "Bucket": {
                                "Ref": "MyBucketF68F3FF0"
                            },
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "bar",
                                        "Effect": "Allow",
                                        "Resource": "foo"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                    }
                }
            });
            test.done();
        },
        'forBucket returns a permission statement associated with the bucket\'s ARN'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const x = new iam.PolicyStatement().addResource(bucket.bucketArn).addAction('s3:ListBucket');
            test.deepEqual(bucket.node.resolve(x), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }
            });
            test.done();
        },
        'arnForObjects returns a permission statement associated with objects in the bucket'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const p = new iam.PolicyStatement().addResource(bucket.arnForObjects('hello/world')).addAction('s3:GetObject');
            test.deepEqual(bucket.node.resolve(p), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [{ 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }, '/hello/world']
                    ]
                }
            });
            test.done();
        },
        'arnForObjects accepts multiple arguments and FnConcats them'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const user = new iam.User(stack, 'MyUser');
            const team = new iam.Group(stack, 'MyTeam');
            const resource = bucket.arnForObjects(`home/${team.groupName}/${user.userName}/*`);
            const p = new iam.PolicyStatement().addResource(resource).addAction('s3:GetObject');
            test.deepEqual(bucket.node.resolve(p), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [
                            { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                            '/home/',
                            { Ref: 'MyTeam01DD6685' },
                            '/',
                            { Ref: 'MyUserDC45028B' },
                            '/*'
                        ]
                    ]
                }
            });
            test.done();
        }
    },
    'removal policy can be used to specify behavior upon delete'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', { removalPolicy: cdk.RemovalPolicy.Orphan, encryption: s3.BucketEncryption.Unencrypted });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyBucketF68F3FF0: {
                    Type: 'AWS::S3::Bucket',
                    DeletionPolicy: 'Retain'
                }
            }
        });
        test.done();
    },
    'import/export': {
        'static import(ref) allows importing an external/existing bucket'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn });
            // this is a no-op since the bucket is external
            bucket.addToResourcePolicy(new iam.PolicyStatement().addResource('foo').addAction('bar'));
            const p = new iam.PolicyStatement().addResource(bucket.bucketArn).addAction('s3:ListBucket');
            // it is possible to obtain a permission statement for a ref
            test.deepEqual(bucket.node.resolve(p), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: 'arn:aws:s3:::my-bucket'
            });
            test.deepEqual(bucket.bucketArn, bucketArn);
            test.deepEqual(bucket.node.resolve(bucket.bucketName), 'my-bucket');
            test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack), {}, 'the ref is not a real resource');
            test.done();
        },
        'import can also be used to import arbitrary ARNs'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn: 'arn:aws:s3:::my-bucket' });
            bucket.addToResourcePolicy(new iam.PolicyStatement().addAllResources().addAction('*'));
            // at this point we technically didn't create any resources in the consuming stack.
            assert_1.expect(stack).toMatch({});
            // but now we can reference the bucket
            // you can even use the bucket name, which will be extracted from the arn provided.
            const user = new iam.User(stack, 'MyUser');
            user.addToPolicy(new iam.PolicyStatement()
                .addResource(bucket.arnForObjects(`my/folder/${bucket.bucketName}`))
                .addAction('s3:*'));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "s3:*",
                                        "Effect": "Allow",
                                        "Resource": "arn:aws:s3:::my-bucket/my/folder/my-bucket"
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        },
                    }
                }
            });
            test.done();
        },
    },
    'grantRead'(test) {
        const stack = new cdk.Stack();
        const reader = new iam.User(stack, 'Reader');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.grantRead(reader);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "ReaderF7BF189D": {
                    "Type": "AWS::IAM::User"
                },
                "ReaderDefaultPolicy151F3818": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": [
                                        "s3:GetObject*",
                                        "s3:GetBucket*",
                                        "s3:List*"
                                    ],
                                    "Effect": "Allow",
                                    "Resource": [
                                        {
                                            "Fn::GetAtt": [
                                                "MyBucketF68F3FF0",
                                                "Arn"
                                            ]
                                        },
                                        {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    {
                                                        "Fn::GetAtt": [
                                                            "MyBucketF68F3FF0",
                                                            "Arn"
                                                        ]
                                                    },
                                                    "/*"
                                                ]
                                            ]
                                        }
                                    ]
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "ReaderDefaultPolicy151F3818",
                        "Users": [
                            {
                                "Ref": "ReaderF7BF189D"
                            }
                        ]
                    }
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                },
            }
        });
        test.done();
    },
    'grantReadWrite': {
        'can be used to grant reciprocal permissions to an identity'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'grant permissions to non-identity principal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
            // WHEN
            bucket.grantRead(new iam.OrganizationPrincipal('o-1234'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                PolicyDocument: {
                    "Version": "2012-10-17",
                    "Statement": [
                        {
                            "Action": ["s3:GetObject*", "s3:GetBucket*", "s3:List*"],
                            "Condition": { "StringEquals": { "aws:PrincipalOrgID": "o-1234" } },
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": [
                                { "Fn::GetAtt": ["MyBucketF68F3FF0", "Arn"] },
                                { "Fn::Join": ["", [{ "Fn::GetAtt": ["MyBucketF68F3FF0", "Arn"] }, "/*"]] }
                            ]
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                "KeyPolicy": {
                    "Statement": [
                        {
                            "Action": ["kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*", "kms:Update*",
                                "kms:Revoke*", "kms:Disable*", "kms:Get*", "kms:Delete*", "kms:ScheduleKeyDeletion", "kms:CancelKeyDeletion"],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": ["", [
                                            "arn:", { "Ref": "AWS::Partition" }, ":iam::", { "Ref": "AWS::AccountId" }, ":root"
                                        ]]
                                }
                            },
                            "Resource": "*"
                        },
                        {
                            "Action": ["kms:Decrypt", "kms:DescribeKey"],
                            "Effect": "Allow",
                            "Resource": "*",
                            "Principal": "*",
                            "Condition": { "StringEquals": { "aws:PrincipalOrgID": "o-1234" } },
                        }
                    ],
                    "Version": "2012-10-17"
                },
            }));
            test.done();
        },
        'if an encryption key is included, encrypt/decrypt permissions are also added both ways'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketKeyC17130CF": {
                        "Type": "AWS::KMS::Key",
                        "Properties": {
                            "Description": "Created by MyBucket",
                            "KeyPolicy": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "kms:Create*",
                                            "kms:Describe*",
                                            "kms:Enable*",
                                            "kms:List*",
                                            "kms:Put*",
                                            "kms:Update*",
                                            "kms:Revoke*",
                                            "kms:Disable*",
                                            "kms:Get*",
                                            "kms:Delete*",
                                            "kms:ScheduleKeyDeletion",
                                            "kms:CancelKeyDeletion"
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":iam::",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":root"
                                                    ]
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::GetAtt": [
                                                    "MyUserDC45028B",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                        "DeletionPolicy": "Retain"
                    },
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                        "Properties": {
                            "BucketEncryption": {
                                "ServerSideEncryptionConfiguration": [
                                    {
                                        "ServerSideEncryptionByDefault": {
                                            "KMSMasterKeyID": {
                                                "Fn::GetAtt": [
                                                    "MyBucketKeyC17130CF",
                                                    "Arn"
                                                ]
                                            },
                                            "SSEAlgorithm": "aws:kms"
                                        }
                                    }
                                ]
                            }
                        }
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::GetAtt": [
                                                "MyBucketKeyC17130CF",
                                                "Arn"
                                            ]
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
    },
    'more grants'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
        const putter = new iam.User(stack, 'Putter');
        const writer = new iam.User(stack, 'Writer');
        const deleter = new iam.User(stack, 'Deleter');
        bucket.grantPut(putter);
        bucket.grantWrite(writer);
        bucket.grantDelete(deleter);
        const resources = assert_1.SynthUtils.toCloudFormation(stack).Resources;
        const actions = (id) => resources[id].Properties.PolicyDocument.Statement[0].Action;
        test.deepEqual(actions('WriterDefaultPolicyDC585BCE'), ['s3:DeleteObject*', 's3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('PutterDefaultPolicyAB138DD3'), ['s3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('DeleterDefaultPolicyCD33B8A0'), 's3:DeleteObject*');
        test.done();
    },
    'cross-stack permissions'(test) {
        const stackA = new cdk.Stack();
        const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket');
        const stackB = new cdk.Stack();
        const user = new iam.User(stackB, 'UserWhoNeedsAccess');
        bucketFromStackA.grantRead(user);
        assert_1.expect(stackA).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            }
        });
        assert_1.expect(stackB).toMatch({
            "Resources": {
                "UserWhoNeedsAccessF8959C3D": {
                    "Type": "AWS::IAM::User"
                },
                "UserWhoNeedsAccessDefaultPolicy6A9EB530": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": [
                                        "s3:GetObject*",
                                        "s3:GetBucket*",
                                        "s3:List*"
                                    ],
                                    "Effect": "Allow",
                                    "Resource": [
                                        {
                                            "Fn::ImportValue": "Stack:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                                        },
                                        {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    {
                                                        "Fn::ImportValue": "Stack:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                                                    },
                                                    "/*"
                                                ]
                                            ]
                                        }
                                    ]
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "UserWhoNeedsAccessDefaultPolicy6A9EB530",
                        "Users": [
                            {
                                "Ref": "UserWhoNeedsAccessF8959C3D"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'urlForObject returns a token with the S3 URL of the token'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        new cdk.CfnOutput(stack, 'BucketURL', { value: bucket.urlForObject() });
        new cdk.CfnOutput(stack, 'MyFileURL', { value: bucket.urlForObject('my/file.txt') });
        new cdk.CfnOutput(stack, 'YourFileURL', { value: bucket.urlForObject('/your/file.txt') }); // "/" is optional
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                }
            },
            "Outputs": {
                "BucketURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                }
                            ]
                        ]
                    },
                },
                "MyFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/my/file.txt"
                            ]
                        ]
                    }
                },
                "YourFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/your/file.txt"
                            ]
                        ]
                    },
                }
            }
        });
        test.done();
    },
    'grantPublicAccess': {
        'by default, grants s3:GetObject to all objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"keyPrefix" can be used to only grant access to certain objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('only/access/these/*');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/only/access/these/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"allowedActions" can be used to specify actions explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('*', 's3:GetObject', 's3:PutObject');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": ["s3:GetObject", "s3:PutObject"],
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'returns the PolicyStatement which can be then customized'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            const result = bucket.grantPublicAccess();
            result.resourceStatement.addCondition('IpAddress', { "aws:SourceIp": "54.240.143.0/24" });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] },
                            "Condition": {
                                "IpAddress": { "aws:SourceIp": "54.240.143.0/24" }
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'throws when blockPublicPolicy is set to true'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', {
                blockPublicAccess: new s3.BlockPublicAccess({ blockPublicPolicy: true })
            });
            // THEN
            test.throws(() => bucket.grantPublicAccess(), /blockPublicPolicy/);
            test.done();
        }
    },
    'website configuration': {
        'only index doc'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html'
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html"
                }
            }));
            test.done();
        },
        'fails if only error doc is specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteErrorDocument: 'error.html'
                });
            }, /"websiteIndexDocument" is required if "websiteErrorDocument" is set/);
            test.done();
        },
        'error and index docs'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html',
                websiteErrorDocument: 'error.html',
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html",
                    ErrorDocument: "error.html"
                }
            }));
            test.done();
        },
        'exports the WebsiteURL'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index.html'
            });
            test.deepEqual(bucket.node.resolve(bucket.bucketWebsiteUrl), { 'Fn::GetAtt': ['Website32962D0B', 'WebsiteURL'] });
            test.done();
        }
    },
    'Bucket.fromBucketArn'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        const bucket = lib_1.Bucket.fromBucketArn(stack, 'my-bucket', 'arn:aws:s3:::my_corporate_bucket');
        // THEN
        test.deepEqual(bucket.bucketName, 'my_corporate_bucket');
        test.deepEqual(bucket.bucketArn, 'arn:aws:s3:::my_corporate_bucket');
        test.done();
    },
    'Bucket.fromBucketName'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        const bucket = lib_1.Bucket.fromBucketName(stack, 'imported-bucket', 'my-bucket-name');
        // THEN
        test.deepEqual(bucket.bucketName, 'my-bucket-name');
        test.deepEqual(stack.node.resolve(bucket.bucketArn), {
            'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket-name']]
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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