"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3n = require("@aws-cdk/aws-s3-notifications");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const s3 = require("../lib");
const notification_dests_1 = require("./notification-dests");
module.exports = {
    'bucket without notifications'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                }
            }
        });
        test.done();
    },
    'when notification are added, a custom resource is provisioned + a lambda handler for it'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        const topic = new notification_dests_1.Topic(stack, 'MyTopic');
        bucket.addEventNotification(s3.EventType.ObjectCreated, topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', { Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)' }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications'));
        test.done();
    },
    'when notification are added, you can tag the lambda'(test) {
        const stack = new cdk.Stack();
        stack.node.apply(new cdk.Tag('Lambda', 'AreTagged'));
        const bucket = new s3.Bucket(stack, 'MyBucket');
        const topic = new notification_dests_1.Topic(stack, 'MyTopic');
        bucket.addEventNotification(s3.EventType.ObjectCreated, topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Tags: [{ Key: 'Lambda', Value: 'AreTagged' }],
            Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications'));
        test.done();
    },
    'bucketNotificationTarget is not called during synthesis'(test) {
        const stack = new cdk.Stack();
        // notice the order here - topic is defined before bucket
        // but this shouldn't impact the fact that the topic policy includes
        // the bucket information
        const topic = new notification_dests_1.Topic(stack, 'Topic');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.addObjectCreatedNotification(topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::TopicPolicy', {
            "Topics": [
                {
                    "Ref": "TopicBFC7AF6E"
                }
            ],
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "sns:Publish",
                        "Condition": {
                            "ArnLike": {
                                "aws:SourceArn": {
                                    "Fn::GetAtt": [
                                        "MyBucketF68F3FF0",
                                        "Arn"
                                    ]
                                }
                            }
                        },
                        "Effect": "Allow",
                        "Principal": {
                            "Service": { "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]] }
                        },
                        "Resource": {
                            "Ref": "TopicBFC7AF6E"
                        },
                        "Sid": "sid0"
                    }
                ],
                "Version": "2012-10-17"
            }
        }));
        test.done();
    },
    'subscription types'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        const queueTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...'
            })
        };
        const lambdaTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Lambda,
                arn: 'arn:aws:lambda:...'
            })
        };
        const topicTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Topic,
                arn: 'arn:aws:sns:...'
            })
        };
        bucket.addEventNotification(s3.EventType.ObjectCreated, queueTarget);
        bucket.addEventNotification(s3.EventType.ObjectCreated, lambdaTarget);
        bucket.addObjectRemovedNotification(topicTarget, { prefix: 'prefix' });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "LambdaFunctionConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectCreated:*"
                        ],
                        "LambdaFunctionArn": "arn:aws:lambda:..."
                    }
                ],
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectCreated:*"
                        ],
                        "QueueArn": "arn:aws:sqs:..."
                    }
                ],
                "TopicConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:*"
                        ],
                        "TopicArn": "arn:aws:sns:...",
                        "Filter": {
                            "Key": {
                                "FilterRules": [
                                    {
                                        "Name": "prefix",
                                        "Value": "prefix"
                                    }
                                ]
                            }
                        }
                    }
                ]
            }
        }));
        test.done();
    },
    'multiple subscriptions of the same type'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        bucket.addEventNotification(s3.EventType.ObjectRemovedDelete, {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...:queue1'
            })
        });
        bucket.addEventNotification(s3.EventType.ObjectRemovedDelete, {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...:queue2'
            })
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "QueueArn": "arn:aws:sqs:...:queue1"
                    },
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "QueueArn": "arn:aws:sqs:...:queue2"
                    }
                ]
            }
        }));
        test.done();
    },
    'prefix/suffix filters'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        const bucketNotificationTarget = {
            type: s3n.BucketNotificationDestinationType.Queue,
            arn: 'arn:aws:sqs:...'
        };
        bucket.addEventNotification(s3.EventType.ObjectRemovedDelete, { asBucketNotificationDestination: _ => bucketNotificationTarget }, { prefix: 'images/', suffix: '.jpg' });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "Filter": {
                            "Key": {
                                "FilterRules": [
                                    {
                                        "Name": "suffix",
                                        "Value": ".jpg"
                                    },
                                    {
                                        "Name": "prefix",
                                        "Value": "images/"
                                    }
                                ]
                            }
                        },
                        "QueueArn": "arn:aws:sqs:..."
                    }
                ]
            }
        }));
        test.done();
    },
    'a notification destination can specify a set of dependencies that must be resolved before the notifications resource is created'(test) {
        const stack = new cdk_1.Stack();
        const bucket = new s3.Bucket(stack, 'Bucket');
        const dependent = new cdk.CfnResource(stack, 'Dependent', { type: 'DependOnMe' });
        const dest = {
            asBucketNotificationDestination: () => ({
                arn: 'arn',
                type: s3n.BucketNotificationDestinationType.Queue,
                dependencies: [dependent]
            })
        };
        bucket.addObjectCreatedNotification(dest);
        stack.node.prepareTree();
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Resources.BucketNotifications8F2E257D, {
            Type: 'Custom::S3BucketNotifications',
            Properties: {
                ServiceToken: { 'Fn::GetAtt': ['BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691', 'Arn'] },
                BucketName: { Ref: 'Bucket83908E77' },
                NotificationConfiguration: { QueueConfigurations: [{ Events: ['s3:ObjectCreated:*'], QueueArn: 'arn' }] }
            },
            DependsOn: ['Dependent']
        });
        test.done();
    },
    'CloudWatch Events': {
        'onPutItem contains the Bucket ARN itself when path is undefined'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
                bucketName: 'MyBucket',
            });
            bucket.onCloudTrailPutObject('PutRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: 'ID' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.s3",
                    ],
                    "detail": {
                        "eventName": [
                            "PutObject",
                        ],
                        "resources": {
                            "ARN": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::MyBucket",
                                        ],
                                    ],
                                },
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        "onPutItem contains the path when it's provided"(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
                bucketName: 'MyBucket',
            });
            bucket.onCloudTrailPutObject('PutRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: 'ID' })
                },
                paths: ['my/path.zip']
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.s3",
                    ],
                    "detail": {
                        "eventName": [
                            "PutObject",
                        ],
                        "resources": {
                            "ARN": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::MyBucket/my/path.zip"
                                        ],
                                    ],
                                },
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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