"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const encoding_1 = require("./private/encoding");
const token_map_1 = require("./private/token-map");
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
class StringConcat {
    join(left, right) {
        if (left === undefined) {
            return right !== undefined ? `${right}` : undefined;
        }
        if (right === undefined) {
            return `${left}`;
        }
        return `${left}${right}`;
    }
}
exports.StringConcat = StringConcat;
/**
 * Default resolver implementation
 *
 * @experimental
 */
class DefaultTokenResolver {
    constructor(concat) {
        this.concat = concat;
    }
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t, context, postProcessor) {
        try {
            let resolved = t.resolve(context);
            // The token might have returned more values that need resolving, recurse
            resolved = context.resolve(resolved);
            resolved = postProcessor.postProcess(resolved, context);
            return resolved;
        }
        catch (e) {
            let message = `Resolution error: ${e.message}.`;
            if (t.creationStack && t.creationStack.length > 0) {
                message += `\nObject creation stack:\n  at ${t.creationStack.join('\n  at ')}`;
            }
            e.message = message;
            throw e;
        }
    }
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments, context) {
        return fragments.mapTokens({ mapToken: context.resolve }).join(this.concat);
    }
    resolveList(xs, context) {
        // Must be a singleton list token, because concatenation is not allowed.
        if (xs.length !== 1) {
            throw new Error(`Cannot add elements to list token, got: ${xs}`);
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const tokenMap = token_map_1.TokenMap.instance();
        const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
        if (fragments.length !== 1) {
            throw new Error(`Cannot concatenate strings in a tokenized string array, got: ${xs[0]}`);
        }
        return fragments.mapTokens({ mapToken: context.resolve }).firstValue;
    }
}
exports.DefaultTokenResolver = DefaultTokenResolver;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb2x2YWJsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlc29sdmFibGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSxpREFBaUQ7QUFDakQsbURBQStDO0FBZ0cvQzs7OztHQUlHO0FBQ0gsTUFBYSxZQUFZO0lBQ2hCLElBQUksQ0FBQyxJQUFxQixFQUFFLEtBQXNCO1FBQ3ZELElBQUksSUFBSSxLQUFLLFNBQVMsRUFBRTtZQUFFLE9BQU8sS0FBSyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1NBQUU7UUFDaEYsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQUUsT0FBTyxHQUFHLElBQUksRUFBRSxDQUFDO1NBQUU7UUFDOUMsT0FBTyxHQUFHLElBQUksR0FBRyxLQUFLLEVBQUUsQ0FBQztJQUMzQixDQUFDO0NBQ0Y7QUFORCxvQ0FNQztBQUVEOzs7O0dBSUc7QUFDSCxNQUFhLG9CQUFvQjtJQUMvQixZQUE2QixNQUE2QjtRQUE3QixXQUFNLEdBQU4sTUFBTSxDQUF1QjtJQUMxRCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxZQUFZLENBQUMsQ0FBYyxFQUFFLE9BQXdCLEVBQUUsYUFBNkI7UUFDekYsSUFBSTtZQUNGLElBQUksUUFBUSxHQUFHLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7WUFFbEMseUVBQXlFO1lBQ3pFLFFBQVEsR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3JDLFFBQVEsR0FBRyxhQUFhLENBQUMsV0FBVyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUN4RCxPQUFPLFFBQVEsQ0FBQztTQUNqQjtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsSUFBSSxPQUFPLEdBQUcscUJBQXFCLENBQUMsQ0FBQyxPQUFPLEdBQUcsQ0FBQztZQUNoRCxJQUFJLENBQUMsQ0FBQyxhQUFhLElBQUksQ0FBQyxDQUFDLGFBQWEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO2dCQUNqRCxPQUFPLElBQUksa0NBQWtDLENBQUMsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUM7YUFDaEY7WUFFRCxDQUFDLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztZQUNwQixNQUFNLENBQUMsQ0FBQztTQUNUO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ksYUFBYSxDQUFDLFNBQW1DLEVBQUUsT0FBd0I7UUFDaEYsT0FBTyxTQUFTLENBQUMsU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLE9BQU8sQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDOUUsQ0FBQztJQUVNLFdBQVcsQ0FBQyxFQUFZLEVBQUUsT0FBd0I7UUFDdkQsd0VBQXdFO1FBQ3hFLElBQUksRUFBRSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDbkIsTUFBTSxJQUFJLEtBQUssQ0FBQywyQ0FBMkMsRUFBRSxFQUFFLENBQUMsQ0FBQztTQUNsRTtRQUVELE1BQU0sR0FBRyxHQUFHLHNCQUFXLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzVDLE1BQU0sUUFBUSxHQUFHLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDckMsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBQ2pFLElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDMUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUMxRjtRQUVELE9BQU8sU0FBUyxDQUFDLFNBQVMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxVQUFVLENBQUM7SUFDdkUsQ0FBQztDQUNGO0FBbkRELG9EQW1EQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElDb25zdHJ1Y3QgfSBmcm9tIFwiLi9jb25zdHJ1Y3RcIjtcbmltcG9ydCB7IFRva2VuU3RyaW5nIH0gZnJvbSBcIi4vcHJpdmF0ZS9lbmNvZGluZ1wiO1xuaW1wb3J0IHsgVG9rZW5NYXAgfSBmcm9tIFwiLi9wcml2YXRlL3Rva2VuLW1hcFwiO1xuaW1wb3J0IHsgVG9rZW5pemVkU3RyaW5nRnJhZ21lbnRzIH0gZnJvbSBcIi4vc3RyaW5nLWZyYWdtZW50c1wiO1xuXG4vKipcbiAqIEN1cnJlbnQgcmVzb2x1dGlvbiBjb250ZXh0IGZvciB0b2tlbnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUmVzb2x2ZUNvbnRleHQge1xuICAvKipcbiAgICogVGhlIHNjb3BlIGZyb20gd2hpY2ggcmVzb2x1dGlvbiBoYXMgYmVlbiBpbml0aWF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHNjb3BlOiBJQ29uc3RydWN0O1xuXG4gIC8qKlxuICAgKiBSZXNvbHZlIGFuIGlubmVyIG9iamVjdFxuICAgKi9cbiAgcmVzb2x2ZSh4OiBhbnkpOiBhbnk7XG5cbiAgLyoqXG4gICAqIFVzZSB0aGlzIHBvc3Rwcm9jZXNzb3IgYWZ0ZXIgdGhlIGVudGlyZSB0b2tlbiBzdHJ1Y3R1cmUgaGFzIGJlZW4gcmVzb2x2ZWRcbiAgICovXG4gIHJlZ2lzdGVyUG9zdFByb2Nlc3Nvcihwb3N0UHJvY2Vzc29yOiBJUG9zdFByb2Nlc3Nvcik6IHZvaWQ7XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciB2YWx1ZXMgdGhhdCBjYW4gYmUgcmVzb2x2YWJsZSBsYXRlclxuICpcbiAqIFRva2VucyBhcmUgc3BlY2lhbCBvYmplY3RzIHRoYXQgcGFydGljaXBhdGUgaW4gc3ludGhlc2lzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSZXNvbHZhYmxlIHtcbiAgLyoqXG4gICAqIFRoZSBjcmVhdGlvbiBzdGFjayBvZiB0aGlzIHJlc29sdmFibGUgd2hpY2ggd2lsbCBiZSBhcHBlbmRlZCB0byBlcnJvcnNcbiAgICogdGhyb3duIGR1cmluZyByZXNvbHV0aW9uLlxuICAgKlxuICAgKiBJZiB0aGlzIHJldHVybnMgYW4gZW1wdHkgYXJyYXkgdGhlIHN0YWNrIHdpbGwgbm90IGJlIGF0dGFjaGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgY3JlYXRpb25TdGFjazogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFByb2R1Y2UgdGhlIFRva2VuJ3MgdmFsdWUgYXQgcmVzb2x1dGlvbiB0aW1lXG4gICAqL1xuICByZXNvbHZlKGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCk6IGFueTtcblxuICAvKipcbiAgICogUmV0dXJuIGEgc3RyaW5nIHJlcHJlc2VudGF0aW9uIG9mIHRoaXMgcmVzb2x2YWJsZSBvYmplY3QuXG4gICAqXG4gICAqIFJldHVybnMgYSByZXZlcnNpYmxlIHN0cmluZyByZXByZXNlbnRhdGlvbi5cbiAgICovXG4gIHRvU3RyaW5nKCk6IHN0cmluZztcbn1cblxuLyoqXG4gKiBBIFRva2VuIHRoYXQgY2FuIHBvc3QtcHJvY2VzcyB0aGUgY29tcGxldGUgcmVzb2x2ZWQgdmFsdWUsIGFmdGVyIHJlc29sdmUoKSBoYXMgcmVjdXJzZWQgb3ZlciBpdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElQb3N0UHJvY2Vzc29yICB7XG4gIC8qKlxuICAgKiBQcm9jZXNzIHRoZSBjb21wbGV0ZWx5IHJlc29sdmVkIHZhbHVlLCBhZnRlciBmdWxsIHJlY3Vyc2lvbi9yZXNvbHV0aW9uIGhhcyBoYXBwZW5lZFxuICAgKi9cbiAgcG9zdFByb2Nlc3MoaW5wdXQ6IGFueSwgY29udGV4dDogSVJlc29sdmVDb250ZXh0KTogYW55O1xufVxuXG4vKipcbiAqIEhvdyB0byByZXNvbHZlIHRva2Vuc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIElUb2tlblJlc29sdmVyIHtcbiAgLyoqXG4gICAqIFJlc29sdmUgYSBzaW5nbGUgdG9rZW5cbiAgICovXG4gIHJlc29sdmVUb2tlbih0OiBJUmVzb2x2YWJsZSwgY29udGV4dDogSVJlc29sdmVDb250ZXh0LCBwb3N0UHJvY2Vzc29yOiBJUG9zdFByb2Nlc3Nvcik6IGFueTtcblxuICAvKipcbiAgICogUmVzb2x2ZSBhIHN0cmluZyB3aXRoIGF0IGxlYXN0IG9uZSBzdHJpbmdpZmllZCB0b2tlbiBpbiBpdFxuICAgKlxuICAgKiAoTWF5IHVzZSBjb25jYXRlbmF0aW9uKVxuICAgKi9cbiAgcmVzb2x2ZVN0cmluZyhzOiBUb2tlbml6ZWRTdHJpbmdGcmFnbWVudHMsIGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCk6IGFueTtcblxuICAvKipcbiAgICogUmVzb2x2ZSBhIHRva2VuaXplZCBsaXN0XG4gICAqL1xuICByZXNvbHZlTGlzdChsOiBzdHJpbmdbXSwgY29udGV4dDogSVJlc29sdmVDb250ZXh0KTogYW55O1xufVxuXG4vKipcbiAqIEZ1bmN0aW9uIHVzZWQgdG8gY29uY2F0ZW5hdGUgc3ltYm9scyBpbiB0aGUgdGFyZ2V0IGRvY3VtZW50IGxhbmd1YWdlXG4gKlxuICogSW50ZXJmYWNlIHNvIGl0IGNvdWxkIHBvdGVudGlhbGx5IGJlIGV4cG9zZWQgb3ZlciBqc2lpLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJRnJhZ21lbnRDb25jYXRlbmF0b3Ige1xuICAvKipcbiAgICogSm9pbiB0aGUgZnJhZ21lbnQgb24gdGhlIGxlZnQgYW5kIG9uIHRoZSByaWdodFxuICAgKi9cbiAgam9pbihsZWZ0OiBhbnkgfCB1bmRlZmluZWQsIHJpZ2h0OiBhbnkgfCB1bmRlZmluZWQpOiBhbnk7XG59XG5cbi8qKlxuICogQ29udmVydHMgYWxsIGZyYWdtZW50cyB0byBzdHJpbmdzIGFuZCBjb25jYXRzIHRob3NlXG4gKlxuICogRHJvcHMgJ3VuZGVmaW5lZCdzLlxuICovXG5leHBvcnQgY2xhc3MgU3RyaW5nQ29uY2F0IGltcGxlbWVudHMgSUZyYWdtZW50Q29uY2F0ZW5hdG9yIHtcbiAgcHVibGljIGpvaW4obGVmdDogYW55IHwgdW5kZWZpbmVkLCByaWdodDogYW55IHwgdW5kZWZpbmVkKTogYW55IHtcbiAgICBpZiAobGVmdCA9PT0gdW5kZWZpbmVkKSB7IHJldHVybiByaWdodCAhPT0gdW5kZWZpbmVkID8gYCR7cmlnaHR9YCA6IHVuZGVmaW5lZDsgfVxuICAgIGlmIChyaWdodCA9PT0gdW5kZWZpbmVkKSB7IHJldHVybiBgJHtsZWZ0fWA7IH1cbiAgICByZXR1cm4gYCR7bGVmdH0ke3JpZ2h0fWA7XG4gIH1cbn1cblxuLyoqXG4gKiBEZWZhdWx0IHJlc29sdmVyIGltcGxlbWVudGF0aW9uXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgRGVmYXVsdFRva2VuUmVzb2x2ZXIgaW1wbGVtZW50cyBJVG9rZW5SZXNvbHZlciB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgY29uY2F0OiBJRnJhZ21lbnRDb25jYXRlbmF0b3IpIHtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZhdWx0IFRva2VuIHJlc29sdXRpb25cbiAgICpcbiAgICogUmVzb2x2ZSB0aGUgVG9rZW4sIHJlY3Vyc2UgaW50byB3aGF0ZXZlciBpdCByZXR1cm5zLFxuICAgKiB0aGVuIGZpbmFsbHkgcG9zdC1wcm9jZXNzIGl0LlxuICAgKi9cbiAgcHVibGljIHJlc29sdmVUb2tlbih0OiBJUmVzb2x2YWJsZSwgY29udGV4dDogSVJlc29sdmVDb250ZXh0LCBwb3N0UHJvY2Vzc29yOiBJUG9zdFByb2Nlc3Nvcikge1xuICAgIHRyeSB7XG4gICAgICBsZXQgcmVzb2x2ZWQgPSB0LnJlc29sdmUoY29udGV4dCk7XG5cbiAgICAgIC8vIFRoZSB0b2tlbiBtaWdodCBoYXZlIHJldHVybmVkIG1vcmUgdmFsdWVzIHRoYXQgbmVlZCByZXNvbHZpbmcsIHJlY3Vyc2VcbiAgICAgIHJlc29sdmVkID0gY29udGV4dC5yZXNvbHZlKHJlc29sdmVkKTtcbiAgICAgIHJlc29sdmVkID0gcG9zdFByb2Nlc3Nvci5wb3N0UHJvY2VzcyhyZXNvbHZlZCwgY29udGV4dCk7XG4gICAgICByZXR1cm4gcmVzb2x2ZWQ7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgbGV0IG1lc3NhZ2UgPSBgUmVzb2x1dGlvbiBlcnJvcjogJHtlLm1lc3NhZ2V9LmA7XG4gICAgICBpZiAodC5jcmVhdGlvblN0YWNrICYmIHQuY3JlYXRpb25TdGFjay5sZW5ndGggPiAwKSB7XG4gICAgICAgIG1lc3NhZ2UgKz0gYFxcbk9iamVjdCBjcmVhdGlvbiBzdGFjazpcXG4gIGF0ICR7dC5jcmVhdGlvblN0YWNrLmpvaW4oJ1xcbiAgYXQgJyl9YDtcbiAgICAgIH1cblxuICAgICAgZS5tZXNzYWdlID0gbWVzc2FnZTtcbiAgICAgIHRocm93IGU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJlc29sdmUgc3RyaW5nIGZyYWdtZW50cyB0byBUb2tlbnNcbiAgICovXG4gIHB1YmxpYyByZXNvbHZlU3RyaW5nKGZyYWdtZW50czogVG9rZW5pemVkU3RyaW5nRnJhZ21lbnRzLCBjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpIHtcbiAgICByZXR1cm4gZnJhZ21lbnRzLm1hcFRva2Vucyh7IG1hcFRva2VuOiBjb250ZXh0LnJlc29sdmUgfSkuam9pbih0aGlzLmNvbmNhdCk7XG4gIH1cblxuICBwdWJsaWMgcmVzb2x2ZUxpc3QoeHM6IHN0cmluZ1tdLCBjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpIHtcbiAgICAvLyBNdXN0IGJlIGEgc2luZ2xldG9uIGxpc3QgdG9rZW4sIGJlY2F1c2UgY29uY2F0ZW5hdGlvbiBpcyBub3QgYWxsb3dlZC5cbiAgICBpZiAoeHMubGVuZ3RoICE9PSAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBhZGQgZWxlbWVudHMgdG8gbGlzdCB0b2tlbiwgZ290OiAke3hzfWApO1xuICAgIH1cblxuICAgIGNvbnN0IHN0ciA9IFRva2VuU3RyaW5nLmZvckxpc3RUb2tlbih4c1swXSk7XG4gICAgY29uc3QgdG9rZW5NYXAgPSBUb2tlbk1hcC5pbnN0YW5jZSgpO1xuICAgIGNvbnN0IGZyYWdtZW50cyA9IHN0ci5zcGxpdCh0b2tlbk1hcC5sb29rdXBUb2tlbi5iaW5kKHRva2VuTWFwKSk7XG4gICAgaWYgKGZyYWdtZW50cy5sZW5ndGggIT09IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IGNvbmNhdGVuYXRlIHN0cmluZ3MgaW4gYSB0b2tlbml6ZWQgc3RyaW5nIGFycmF5LCBnb3Q6ICR7eHNbMF19YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGZyYWdtZW50cy5tYXBUb2tlbnMoeyBtYXBUb2tlbjogY29udGV4dC5yZXNvbHZlIH0pLmZpcnN0VmFsdWU7XG4gIH1cbn1cbiJdfQ==