"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStore = exports.ValidationSettingsMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
const policy_1 = require("./policy");
const permissions_1 = require("./private/permissions");
const RELEVANT_HTTP_METHODS = ['get', 'post', 'put', 'patch', 'delete', 'head'];
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
var ValidationSettingsMode;
(function (ValidationSettingsMode) {
    ValidationSettingsMode["OFF"] = "OFF";
    ValidationSettingsMode["STRICT"] = "STRICT";
})(ValidationSettingsMode || (exports.ValidationSettingsMode = ValidationSettingsMode = {}));
class PolicyStoreBase extends core_1.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.policyStoreArn],
            scope: this,
        });
    }
    grantAuth(grantee) {
        return this.grant(grantee, ...permissions_1.AUTH_ACTIONS);
    }
    grantRead(grantee) {
        return this.grant(grantee, ...permissions_1.READ_ACTIONS);
    }
    grantWrite(grantee) {
        return this.grant(grantee, ...permissions_1.WRITE_ACTIONS.concat(permissions_1.READ_ACTIONS));
    }
}
class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope, id, policyStoreId) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreId });
    }
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope, id, policyStoreArn) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreArn });
    }
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope, id, attrs) {
        class Import extends PolicyStoreBase {
            constructor(policyStoreArn, policyStoreId) {
                super(scope, id);
                this.policyStoreArn = policyStoreArn;
                this.policyStoreId = policyStoreId;
            }
        }
        let policyStoreId;
        let policyStoreArn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.policyStoreId) {
            if (!attrs.policyStoreArn) {
                throw new Error('One of policyStoreId or policyStoreArn is required!');
            }
            policyStoreArn = attrs.policyStoreArn;
            const maybeId = stack.splitArn(attrs.policyStoreArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for PolicyStore must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            policyStoreId = maybeId;
        }
        else {
            if (attrs.policyStoreArn) {
                throw new Error('Only one of policyStoreArn or policyStoreId can be provided');
            }
            policyStoreId = attrs.policyStoreId;
            policyStoreArn = stack.formatArn({
                resource: 'policy-store',
                resourceName: attrs.policyStoreId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(policyStoreArn, policyStoreId);
    }
    /**
     * This method generates a schema based on an swagger file. It makes the same assumptions and decisions
     * made in the Amazon Verified Permissions console. This feature is built for swagger files generated from an Amazon API Gateway
     * export. It's possible that some swagger files generated by other tools will not work. In that case, please
     * file an issue.
     * @param swaggerFilePath absolute path to a swagger file in the local directory structure, in json format
     * @param groupEntityTypeName optional parameter to specify the group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromOpenApiSpec(swaggerFilePath, groupEntityTypeName) {
        const openApiSpecString = fs.readFileSync(swaggerFilePath, 'utf-8');
        const openApiSpec = JSON.parse(openApiSpecString);
        if (!openApiSpec.paths) {
            throw new Error('Invalid OpenAPI spec - missing paths object');
        }
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(openApiSpec.info.title);
        const pathUrls = Object.keys(openApiSpec.paths);
        const actionNames = [];
        for (const pathUrl of pathUrls) {
            const pathDef = openApiSpec.paths[pathUrl];
            if (!pathDef) {
                continue;
            }
            let pathVerbs = Object.keys(pathDef);
            if (pathVerbs.includes('x-amazon-apigateway-any-method')) {
                pathVerbs = RELEVANT_HTTP_METHODS;
            }
            for (const httpVerb of pathVerbs) {
                if (!RELEVANT_HTTP_METHODS.includes(httpVerb)) {
                    continue;
                }
                const actionName = `${httpVerb} ${pathUrl}`;
                actionNames.push(actionName);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    /**
     * This method generates a schema based on an AWS CDK RestApi construct. It makes the same assumptions
     * and decisions made in the Amazon Verified Permissions console.
     *
     * @param restApi The RestApi construct instance from which to generate the schema.
     * @param groupEntityTypeName Specifies a group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromRestApi(restApi, groupEntityTypeName) {
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(restApi.restApiName);
        const actionNames = [];
        for (const method of restApi.methods) {
            const pathVerb = method.httpMethod.toLowerCase();
            const pathUrl = method.resource.path;
            if (pathVerb === 'any') {
                for (const verb of RELEVANT_HTTP_METHODS) {
                    actionNames.push(`${verb} ${pathUrl}`);
                }
            }
            if (RELEVANT_HTTP_METHODS.includes(pathVerb)) {
                actionNames.push(`${pathVerb} ${pathUrl}`);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    constructor(scope, id, props = {
        validationSettings: {
            mode: ValidationSettingsMode.OFF,
        },
    }) {
        super(scope, id);
        if (props.schema) {
            (0, cedar_helpers_1.checkParseSchema)(props.schema.cedarJson);
        }
        this.policyStore = new aws_verifiedpermissions_1.CfnPolicyStore(this, id, {
            schema: props.schema
                ? {
                    cedarJson: props.schema.cedarJson,
                }
                : undefined,
            validationSettings: {
                mode: props.validationSettings.mode,
            },
            description: props.description,
        });
        this.policyStoreArn = this.getResourceArnAttribute(this.policyStore.attrArn, {
            resource: 'policy-store',
            resourceName: this.physicalName,
            service: 'verifiedpermissions',
        });
        this.policyStoreName = this.getResourceNameAttribute(this.policyStore.ref);
        this.policyStoreId = this.policyStore.attrPolicyStoreId;
        this.schema = props.schema;
        this.validationSettings = props.validationSettings;
        this.description = props.description;
    }
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions) {
        let policies = policyDefinitions.map((policyOption) => {
            return new policy_1.Policy(this, policyOption.policyId, {
                policyStore: this,
                definition: policyOption.policyConfiguration,
            });
        });
        return policies;
    }
    /**
     * Takes in an absolute path to a directory containing .cedar files and adds the contents of each
     * .cedar file as policies to this policy store. Parses the policies with cedar-wasm and, if the policy store has a schema,
     * performs semantic validation of the policies as well.
     * @param absolutePath a string representing an absolute path to the directory containing your policies
     * @returns An array of created Policy constructs.
     */
    addPoliciesFromPath(absolutePath) {
        if (!fs.statSync(absolutePath).isDirectory()) {
            throw new Error(`The path ${absolutePath} does not appear to be a directory`);
        }
        const policyFileNames = fs
            .readdirSync(absolutePath)
            .map((f) => path.join(absolutePath, f))
            .filter((f) => !fs.statSync(f).isDirectory() && f.endsWith('.cedar'));
        if (this.validationSettings.mode === ValidationSettingsMode.STRICT) {
            if (!this.schema) {
                throw new Error('A schema must exist when adding policies to a policy store with strict validation mode.');
            }
            for (const policyFile of policyFileNames) {
                const policyStatement = fs.readFileSync(policyFile, 'utf-8');
                (0, cedar_helpers_1.validatePolicy)(policyStatement, this.schema.cedarJson);
            }
        }
        const policies = policyFileNames.map((cedarFile) => policy_1.Policy.fromFile(this, cedarFile, {
            path: cedarFile,
            policyStore: this,
        }));
        return policies;
    }
}
exports.PolicyStore = PolicyStore;
_a = JSII_RTTI_SYMBOL_1;
PolicyStore[_a] = { fqn: "@cdklabs/cdk-verified-permissions.PolicyStore", version: "0.1.6" };
//# sourceMappingURL=data:application/json;base64,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