"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const constructs_1 = require("constructs");
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     *
     * @param scope The scope.
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     *
     * The toolkit will fail synthesis when errors are reported.
     *
     * @param message The error message.
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property` (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about alternatives.
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${constructs_1.Node.of(this.scope).path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        constructs_1.Node.of(this.scope).addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
//# sourceMappingURL=data:application/json;base64,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