"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerVolumeConsistency = exports.BundlingDockerImage = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("./fs");
/**
 * A Docker image used for asset bundling.
 */
class BundlingDockerImage {
    /** @param image The Docker image */
    constructor(image, _imageHash) {
        this.image = image;
        this._imageHash = _imageHash;
    }
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name.
     */
    static fromRegistry(image) {
        return new BundlingDockerImage(image);
    }
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     */
    static fromAsset(path, options = {}) {
        const buildArgs = options.buildArgs || {};
        const dockerArgs = [
            'build', '-q',
            ...(options.file ? ['-f', options.file] : []),
            ...flatten(Object.entries(buildArgs).map(([k, v]) => ['--build-arg', `${k}=${v}`])),
            path,
        ];
        const docker = dockerExec(dockerArgs);
        const match = docker.stdout.toString().match(/sha256:[a-z0-9]+/);
        if (!match) {
            throw new Error('Failed to extract image ID from Docker build output');
        }
        // Fingerprints the directory containing the Dockerfile we're building and
        // differentiates the fingerprint based on build arguments. We do this so
        // we can provide a stable image hash. Otherwise, the image ID will be
        // different every time the Docker layer cache is cleared, due primarily to
        // timestamps.
        const hash = fs_1.FileSystem.fingerprint(path, { extraHash: JSON.stringify(options) });
        return new BundlingDockerImage(match[0], hash);
    }
    /**
     * Provides a stable representation of this image for JSON serialization.
     *
     * @returns The overridden image name if set or image hash name in that order
     */
    toJSON() {
        var _a;
        return (_a = this._imageHash) !== null && _a !== void 0 ? _a : this.image;
    }
    /**
     * Runs a Docker image.
     */
    run(options = {}) {
        const volumes = options.volumes || [];
        const environment = options.environment || {};
        const command = options.command || [];
        const dockerArgs = [
            'run', '--rm',
            ...options.user
                ? ['-u', options.user]
                : [],
            ...flatten(volumes.map(v => { var _a; return ['-v', `${v.hostPath}:${v.containerPath}:${(_a = v.consistency) !== null && _a !== void 0 ? _a : DockerVolumeConsistency.DELEGATED}`]; })),
            ...flatten(Object.entries(environment).map(([k, v]) => ['--env', `${k}=${v}`])),
            ...options.workingDirectory
                ? ['-w', options.workingDirectory]
                : [],
            this.image,
            ...command,
        ];
        dockerExec(dockerArgs, {
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
        });
    }
    /**
     * Copies a file or directory out of the Docker image to the local filesystem.
     */
    cp(imagePath, outputPath) {
        const { stdout } = dockerExec(['create', this.image]);
        const match = stdout.toString().match(/([0-9a-f]{16,})/);
        if (!match) {
            throw new Error('Failed to extract container ID from Docker create output');
        }
        const containerId = match[1];
        const containerPath = `${containerId}:${imagePath}`;
        try {
            dockerExec(['cp', containerPath, outputPath]);
        }
        catch (err) {
            throw new Error(`Failed to copy files from ${containerPath} to ${outputPath}: ${err}`);
        }
        finally {
            dockerExec(['rm', '-v', containerId]);
        }
    }
}
exports.BundlingDockerImage = BundlingDockerImage;
/**
 * Supported Docker volume consistency types.
 *
 * Only valid on macOS due to the way file storage works on Mac
 */
var DockerVolumeConsistency;
(function (DockerVolumeConsistency) {
    DockerVolumeConsistency["CONSISTENT"] = "consistent";
    DockerVolumeConsistency["DELEGATED"] = "delegated";
    DockerVolumeConsistency["CACHED"] = "cached";
})(DockerVolumeConsistency = exports.DockerVolumeConsistency || (exports.DockerVolumeConsistency = {}));
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
function dockerExec(args, options) {
    var _a, _b, _c;
    const prog = (_a = process.env.CDK_DOCKER) !== null && _a !== void 0 ? _a : 'docker';
    const proc = child_process_1.spawnSync(prog, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_b = proc.stdout) === null || _b === void 0 ? void 0 : _b.toString().trim()}\n\n\nstderr: ${(_c = proc.stderr) === null || _c === void 0 ? void 0 : _c.toString().trim()}`);
        }
        throw new Error(`${prog} exited with status ${proc.status}`);
    }
    return proc;
}
//# sourceMappingURL=data:application/json;base64,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