"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnJson = void 0;
const custom_resource_1 = require("./custom-resource");
const cfn_utils_provider_1 = require("./private/cfn-utils-provider");
const stack_1 = require("./stack");
const stack_trace_1 = require("./stack-trace");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
/**
 * Captures a synthesis-time JSON object a CloudFormation reference which resolves during deployment to the resolved values of the JSON object.
 *
 * The main use case for this is to overcome a limitation in CloudFormation that
 * does not allow using intrinsic functions as dictionary keys (because
 * dictionary keys in JSON must be strings). Specifically this is common in IAM
 * conditions such as `StringEquals: { lhs: "rhs" }` where you want "lhs" to be
 * a reference.
 *
 * This object is resolvable, so it can be used as a value.
 *
 * This construct is backed by a custom resource.
 */
class CfnJson extends construct_compat_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The creation stack of this resolvable which will be appended to errors thrown during resolution.
         *
         * This may return an array with a single informational element indicating how
         * to get this property populated, if it was skipped for performance reasons.
         */
        this.creationStack = [];
        this.creationStack = stack_trace_1.captureStackTrace();
        // stringify the JSON object in a token-aware way.
        this.jsonString = stack_1.Stack.of(this).toJsonString(props.value);
        const resource = new custom_resource_1.CustomResource(this, 'Resource', {
            serviceToken: cfn_utils_provider_1.CfnUtilsProvider.getOrCreate(this),
            resourceType: "Custom::AWSCDKCfnJson" /* CFN_JSON */,
            properties: {
                Value: this.jsonString,
            },
        });
        this.value = resource.getAtt('Value');
    }
    /**
     * This is required in case someone JSON.stringifys an object which refrences this object. Otherwise, we'll get a cyclic JSON reference.
     */
    toJSON() {
        return this.jsonString;
    }
    /**
     * Produce the Token's value at resolution time.
     */
    resolve(_) {
        return this.value;
    }
}
exports.CfnJson = CfnJson;
//# sourceMappingURL=data:application/json;base64,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