"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Simple function to evaluate CloudFormation intrinsics.
 *
 * Note that this function is not production quality, it exists to support tests.
 */
const cloudformation_lang_1 = require("../lib/private/cloudformation-lang");
function evaluateCFN(object, context = {}) {
    const intrinsics = {
        'Fn::Join'(separator, args) {
            return args.map(evaluate).join(separator);
        },
        'Ref'(logicalId) {
            if (!(logicalId in context)) {
                throw new Error(`Trying to evaluate Ref of '${logicalId}' but not in context!`);
            }
            return context[logicalId];
        },
        'Fn::GetAtt'(logicalId, attributeName) {
            const key = `${logicalId}.${attributeName}`;
            if (!(key in context)) {
                throw new Error(`Trying to evaluate Fn::GetAtt of '${logicalId}.${attributeName}' but not in context!`);
            }
            return context[key];
        },
        'Fn::Sub'(argument) {
            const template = evaluate(Array.isArray(argument) ? argument[0] : argument);
            const placeholders = Array.isArray(argument) ? evaluate(argument[1]) : context;
            return template.replace(/\$\{([a-zA-Z0-9.:-]*)\}/g, (_, key) => {
                if (key in placeholders) {
                    return placeholders[key];
                }
                throw new Error(`Unknown placeholder in Fn::Sub: ${key}`);
            });
        },
    };
    return evaluate(object);
    function evaluate(obj) {
        if (Array.isArray(obj)) {
            return obj.map(evaluate);
        }
        if (typeof obj === 'object') {
            const keys = Object.keys(obj);
            if (keys.length === 1 && (cloudformation_lang_1.isNameOfCloudFormationIntrinsic(keys[0]) || keys[0] === 'Ref')) {
                return evaluateIntrinsic(keys[0], obj[keys[0]]);
            }
            const ret = {};
            for (const key of keys) {
                ret[key] = evaluateCFN(obj[key]);
            }
            return ret;
        }
        return obj;
    }
    function evaluateIntrinsic(name, args) {
        if (!(name in intrinsics)) {
            throw new Error(`Intrinsic ${name} not supported here`);
        }
        if (!Array.isArray(args)) {
            args = [args];
        }
        return intrinsics[name].apply(intrinsics, args);
    }
}
exports.evaluateCFN = evaluateCFN;
//# sourceMappingURL=data:application/json;base64,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