import logging
import os

import runez

from pickley import abort, PICKLEY

LOG = logging.getLogger(__name__)

WRAPPER_MARK = "# Wrapper generated by https://pypi.org/project/pickley/"

GENERIC_WRAPPER = (
    """
#!/bin/bash

# pypi-package: {name}
%s

if [[ -x {pickley} ]]; then
    {hook}nohup {pickley} auto-upgrade {name}{bg}
fi
if [[ -x {source} ]]; then
    {hook}exec {source} "$@"
else
    echo "{source} is not available anymore"
    echo ""
    echo "Please reinstall with:"
    echo "{pickley} install -f {name}"
    exit 1
fi
"""
    % WRAPPER_MARK
)

# Specific wrapper for pickley itself (avoid calling ourselves back recursively for auto-upgrade)
PICKLEY_WRAPPER = (
    """
#!/bin/bash

# pypi-package: pickley
%s

if [[ -x {source} ]]; then
    if [[ "$*" != *"auto-"* ]]; then
        {hook}nohup {source} auto-upgrade {name}{bg}
    fi
    {hook}exec {source} "$@"
else
    echo "{source} is not available anymore"
    echo ""
    echo "Please reinstall with:"
    echo '/bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/codrsquad/pickley/main/get-pickley)"'
    exit 1
fi
"""
    % WRAPPER_MARK
)


class DeliveryMethod:
    """
    Various implementation of delivering the actual executables
    """

    action = "Delivered"
    short_name = "deliver"
    ping = True  # Touch .ping file when done?

    @classmethod
    def delivery_method_by_name(cls, name):
        """
        Args:
            name (str): Name of delivery method

        Returns:
            (DeliveryMethod): Associated delivery method
        """
        if name == "wrap":
            return DeliveryMethodWrap()

        if name == "symlink":
            return DeliveryMethodSymlink()

        return abort(f"Unknown delivery method '{runez.red(name)}'")

    def install(self, venv, entry_points):
        """
        Args:
            venv (pickley.package.PythonVenv): Virtual env where executables reside (DOT_META/<package>/...)
            entry_points (dict | list | tuple): Full path of executable to deliver (<base>/<entry_point>)
        """
        if not venv.pspec.is_clear_for_installation():
            abort(f"{runez.short(venv.pspec.exe_path(venv.pspec.dashed))} exists and was not installed by pickley")

        try:
            prev_manifest = venv.pspec.manifest
            for name in entry_points:
                src = os.path.join(venv.folder, "bin", name)
                dest = venv.pspec.exe_path(name)
                ssrc = runez.short(src)
                sdest = runez.short(dest)
                if runez.DRYRUN:
                    print(f"Would {self.short_name} {sdest} -> {ssrc}")
                    continue

                if not os.path.exists(src):
                    abort(f"Can't {self.short_name} {sdest} -> {runez.red(ssrc)}: source does not exist")

                LOG.debug("%s %s -> %s", self.action, sdest, ssrc)
                self._install(venv.pspec, dest, src)

            manifest = venv.pspec.save_manifest(entry_points)
            if not runez.DRYRUN and prev_manifest and prev_manifest.entrypoints:
                for old_ep in prev_manifest.entrypoints:
                    if old_ep and old_ep not in entry_points:
                        # Remove old entry points that are not in new manifest any more
                        runez.delete(venv.pspec.exe_path(old_ep))

            if self.ping:
                # Touch the .ping file since this is a fresh install (no need to check for upgrades right away)
                runez.touch(venv.pspec.ping_path)

            return manifest

        except Exception as e:
            abort(f"Failed to {self.short_name} {venv.pspec}: {runez.red(e)}")

    def _install(self, pspec, target, source):
        raise NotImplementedError(f"{self.__class__.__name__} is not implemented")


class DeliveryMethodSymlink(DeliveryMethod):
    """
    Deliver via symlink
    """

    action = "Symlinked"
    short_name = "symlink"

    def _install(self, pspec, target, source):
        runez.delete(target, logger=False)
        if os.path.isabs(source) and os.path.isabs(target):
            parent = runez.parent_folder(target)
            if runez.parent_folder(source).startswith(parent):
                # Use relative path if source is under target
                source = os.path.relpath(source, parent)

        os.symlink(source, target)


class DeliveryMethodWrap(DeliveryMethod):
    """
    Deliver via a small wrap that ensures target executable is up-to-date
    """

    action = "Wrapped"
    short_name = "wrap"

    # Can be set in tests to make wrapper a no-op
    hook = ""
    bg = " &> /dev/null &"

    def _install(self, pspec, target, source):
        pickley = pspec.cfg.base.full_path(PICKLEY)
        wrapper = PICKLEY_WRAPPER if pspec.dashed == PICKLEY else GENERIC_WRAPPER
        contents = wrapper.lstrip().format(
            hook=self.hook,
            bg=self.bg,
            name=runez.quoted(pspec.dashed, adapter=None),
            pickley=runez.quoted(pickley, adapter=None),
            source=runez.quoted(source, adapter=None),
        )
        runez.delete(target, logger=False)
        runez.write(target, contents, logger=False)
        runez.make_executable(target, logger=False)
