from dataclasses import dataclass
from typing import Dict, List, Optional
import json
from .file_analyzer import FileAnalyzer

@dataclass
class ProjectInsights:
    """Rich project insights generated by LLM analysis"""
    overview: str
    key_features: List[Dict[str, str]]  # Changed to structured feature objects
    architecture: Dict[str, any]  # Changed to contain all architecture info
    tech_stack: Dict[str, List[str]]  # Changed to structured tech stack info
    code_patterns: List[Dict[str, str]]  # Added code patterns

class ManifestEnricher:
    def __init__(self, llm_provider):
        self.llm = llm_provider
        self.file_analyzer = FileAnalyzer(llm_provider)

    async def enrich_manifest(self, manifest: Dict) -> Dict:
        """Enrich manifest with LLM-generated insights"""
        # Generate project insights
        insights = await self._analyze_project(manifest)
        
        # Add enriched data to manifest
        enriched_manifest = manifest.copy()
        enriched_manifest["project_insights"] = {
            "overview": insights.overview,
            "key_features": insights.key_features,
            "architecture": insights.architecture,
            "tech_stack": insights.tech_stack,
            "code_patterns": insights.code_patterns
        }
        
        return enriched_manifest

    async def _analyze_project(self, manifest: Dict) -> ProjectInsights:
        """Generate comprehensive project insights using LLM"""
        
        # Prepare context for LLM
        project_context = self._prepare_project_context(manifest)
        
        # Generate overview and features
        overview_prompt = f"""
        Analyze this project and provide a JSON response with the project overview and key features.
        
        Project Context:
        {project_context}
        
        Respond with JSON in the following format:
        {{
            "overview": "Comprehensive overview paragraph with markdown formatting",
            "key_features": [
                {{
                    "name": "Feature name",
                    "description": "Feature description with markdown formatting",
                    "category": "Feature category (e.g., Core, Integration, UI)",
                    "status": "Status (e.g., Implemented, Planned, In Progress)"
                }}
            ]
        }}
        """
        
        overview_response = await self.llm.generate(overview_prompt)
        overview_data = self.file_analyzer.parse_json(overview_response)
        
        # Analyze architecture
        architecture_prompt = f"""
        Analyze the architectural aspects of this project and provide a JSON response.
        
        Project Context:
        {project_context}
        
        Respond with JSON in the following format:
        {{
            "architecture": {{
                "summary": "High-level architecture summary with markdown formatting",
                "design_patterns": [
                    {{
                        "name": "Pattern name",
                        "purpose": "How/why it's used",
                        "location": "Where it's implemented"
                    }}
                ],
                "primary_components": [
                    {{
                        "name": "Component name",
                        "responsibility": "Component's main responsibility",
                        "dependencies": ["List of dependencies"],
                        "key_files": ["List of key files"]
                    }}
                ],
                "data_flow": {{
                    "description": "Description of data flow",
                    "key_processes": ["List of key processes"]
                }}
            }}
        }}
        """
        
        arch_response = await self.llm.generate(architecture_prompt)
        arch_data = self.file_analyzer.parse_json(arch_response)
        
        # Analyze tech stack
        tech_prompt = f"""
        Analyze the technology choices and patterns in this project and provide a JSON response.
        
        Project Context:
        {project_context}
        
        Respond with JSON in the following format:
        {{
            "tech_stack": {{
                "languages": [
                    {{
                        "name": "Language name",
                        "purpose": "How it's used",
                        "key_features_used": ["List of key features used"]
                    }}
                ],
                "frameworks": [
                    {{
                        "name": "Framework name",
                        "purpose": "How it's used",
                        "key_features_used": ["List of key features used"]
                    }}
                ],
                "tools": [
                    {{
                        "name": "Tool name",
                        "purpose": "How it's used"
                    }}
                ]
            }},
            "code_patterns": [
                {{
                    "name": "Pattern name",
                    "description": "Pattern description",
                    "example_location": "Where to find examples"
                }}
            ]
        }}
        """
        
        tech_response = await self.llm.generate(tech_prompt)
        tech_data = self.file_analyzer.parse_json(tech_response)
        
        return ProjectInsights(
            overview=overview_data.get("overview", ""),
            key_features=overview_data.get("key_features", []),
            architecture=arch_data.get("architecture", {}),
            tech_stack=tech_data.get("tech_stack", {}),
            code_patterns=tech_data.get("code_patterns", [])
        )

    def _prepare_project_context(self, manifest: Dict) -> str:
        """Prepare relevant context from manifest for LLM analysis"""
        context = []
        
        # Basic project info
        context.append(f"Project Name: {manifest.get('name', 'Unknown')}")
        context.append(f"Primary Language: {manifest.get('primary_language', 'Unknown')}")
        
        # Frameworks and dependencies
        if manifest.get('frameworks'):
            context.append("\nFrameworks:")
            context.extend([f"- {fw}" for fw in manifest['frameworks']])
        
        # Dependencies
        if manifest.get('dependencies'):
            context.append("\nDependencies:")
            for dep_name, dep_info in manifest['dependencies'].items():
                context.append(f"- {dep_name} ({dep_info.get('type', 'unknown')})")
        
        # Files and their purposes
        if manifest.get('files'):
            context.append("\nKey Files and Purposes:")
            for file_path, file_info in manifest['files'].items():
                if isinstance(file_info, dict) and file_info.get('purpose'):
                    context.append(f"- {file_path}: {file_info['purpose']}")
        
        return "\n".join(context)
