"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const destinations = require("../lib");
/*
 * Stack verification steps:
 * * aws lambda invoke --function-name <deployed fn name> --invocation-type Event --payload '"OK"' response.json
 * * aws lambda invoke --function-name <deployed fn name> --invocation-type Event --payload '"NOT OK"' response.json
 */
class TestStack extends core_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const topic = new sns.Topic(this, 'Topic');
        const queue = new sqs.Queue(this, 'Queue');
        const fn = new lambda.Function(this, 'SnsSqs', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = async (event) => {
        if (event === 'OK') return 'success';
        throw new Error('failure');
      };`),
            onFailure: new destinations.SnsDestination(topic),
            onSuccess: new destinations.SqsDestination(queue),
            maxEventAge: core_1.Duration.hours(3),
            retryAttempts: 1,
        });
        const onSuccessLambda = new lambda.Function(this, 'OnSucces', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = async (event) => {
        console.log(event);
      };`),
        });
        new lambda.Function(this, 'EventBusLambda', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = async (event) => {
        if (event === 'OK') return 'success';
        throw new Error('failure');
      };`),
            onFailure: new destinations.EventBridgeDestination(),
            onSuccess: new destinations.LambdaDestination(onSuccessLambda),
        });
        const version = fn.addVersion('MySpecialVersion');
        new lambda.Alias(this, 'MySpecialAlias', {
            aliasName: 'MySpecialAlias',
            version,
            onSuccess: new destinations.SqsDestination(queue),
            onFailure: new destinations.SnsDestination(topic),
            maxEventAge: core_1.Duration.hours(2),
            retryAttempts: 0,
        });
    }
}
const app = new core_1.App();
new TestStack(app, 'aws-cdk-lambda-destinations');
app.synth();
//# sourceMappingURL=data:application/json;base64,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