# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core.ipynb.

# %% auto 0
__all__ = ['logger', 'coerce_primitive_values', 'extract_declared_items', 'ConfigValidatorException', 'ConfigValidator']

# %% ../nbs/00_core.ipynb 2
import json
import logging
import os
import os.path
from typing import Union

import dotenv
import jsonschema
from fs.base import FS
from fs.osfs import OSFS
from jsonschema import ValidationError, validate

logger = logging.getLogger(__name__)

# %% ../nbs/00_core.ipynb 4
def coerce_primitive_values(json_schema: dict, data: dict) -> dict:
    """
    Given a JSON schema dictionary, return a dictionary where the values that have
    primitive types ('string', 'integer', 'number', 'boolean') 
    as described in the schema are converted to the corresponding types from `str`.
   
    Args:
        - `json_schema` (dict): JSON schema used for extraction. The function expects but does not validate this 
                                to be a JSON schema.
        - `data` (dict): The data (e.g., dotenv, os.environ) to extract from.

    Returns:
        dict: A coerced dictionary with values converted to the corresponding primitive types.
    """
    if not isinstance(data, dict):
        return data
    out = data.copy()
    # use the json schema to convert types on known properties
    for property_name, property_schema in json_schema['properties'].items():
        property_type = property_schema.get('type')
        property_value = data.get(property_name)
        if property_value is None:
            continue
        # property_value should be a string at this point
        try:
            if property_type == 'integer':
                out[property_name] = int(data[property_name])
            elif property_type == 'number':
                out[property_name] = float(data[property_name])
            elif property_type == 'boolean':
                parsed_boolean = None
                if property_value.lower() in (
                    'true', 'yes', 'y', '1', 'on',
                ):
                    parsed_boolean = True
                elif property_value.lower() in (
                    'false', 'no', 'n', '0', 'off',
                ):
                    parsed_boolean = False
                out[property_name] = parsed_boolean
        except:
            # leave any validation error descriptions to json schema
            continue
    return out

# %% ../nbs/00_core.ipynb 7
def extract_declared_items(json_schema: dict, data: dict) -> dict:
    """
    Given a JSON schema dict, return a dict following specified rules:
    
    1. All keys not declared in the schema are removed.
    2. All keys declared in the schema are present. If a key declared in the schema with a default 
       is not present in the original data, it's added with the 'default' value.
    
    Args:
        - `json_schema` (dict): JSON schema used for extraction. The function expects but does not validate this 
                                to be a JSON schema.
        - `data` (dict): The data (for example, dotenv, os.environ) to extract from.

    Returns:
        dict: A dictionary that has been processed according to the rules specified above.
    """
    properties = json_schema['properties']
    out = {key: value for (key, value) in data.items() if key in properties}
    for required_property, property_schema in properties.items():
        if required_property not in out and 'default' in property_schema:
            out[required_property] = property_schema['default']
    return out

# %% ../nbs/00_core.ipynb 9
class ConfigValidatorException(Exception):
    
    def __init__(self, errors):
        super().__init__('''config failed to validate against JSON schema\n{}'''.format(
            '\n'.join(f'{error.json_path}: {error.message}' for error in errors)
        ))
        self.errors = errors


class ConfigValidator(object):

    # this is a risky value: it gets reused across instances;
    # the idea is to maybe set it once and use it multiple times.
    # but in testing this smells bad
    DEFAULT_STORAGE_DRIVER: FS = None  # defaults to OSFS
    
    CONFIG_VALIDATOR_JSON_SCHEMA_ENVVAR_NAME = 'CONFIG_VALIDATOR_JSON_SCHEMA'
    
    @classmethod
    def get_default_storage_driver(cls):
        if cls.DEFAULT_STORAGE_DRIVER is None:
            cls.DEFAULT_STORAGE_DRIVER = OSFS(os.getcwd())
        return cls.DEFAULT_STORAGE_DRIVER

    
    @classmethod
    def _get_maybe_abspath_driver(cls, maybe_abspath: str):
        if os.path.isabs(maybe_abspath):  # special case
            return OSFS('/')
        else:
            return cls.get_default_storage_driver()
        
    @classmethod
    def load_json(cls, json_source: Union[str, dict]=None, storage_driver: FS = None) -> dict:
        """
        Convenience method to return a dictionary from either a file path or an already-loaded dictionary.

        Args:
            - `json_source` (Union[str, dict], optional): The JSON source to load.
                                This can be a file path (str) 
                                or an already loaded dictionary (dict). 
            - `storage_driver` (FS, optional): An instance of the storage driver used to load the JSON file. 
                                               If not provided, OSFS from the current working dir is used.

        Returns:
            dict: A dictionary that was loaded from the provided `json_source`.
        """
        if isinstance(json_source, dict):
            return json_source
        
        if storage_driver is None:
            storage_driver = cls._get_maybe_abspath_driver(json_source)
        with storage_driver.open(json_source) as ifile:
            return json.load(ifile)

    @classmethod
    def get_default_json_schema(cls, storage_driver: FS = None) -> dict:
        if cls.CONFIG_VALIDATOR_JSON_SCHEMA_ENVVAR_NAME in os.environ:
            expected_json_schema_path = \
                os.environ[cls.CONFIG_VALIDATOR_JSON_SCHEMA_ENVVAR_NAME]
            return cls.load_json(expected_json_schema_path, storage_driver)
        return None

    def __init__(self, json_schema: Union[str, dict]=None, storage_driver: FS=None):
        """
        Initialize the instance with a JSON schema and a storage driver.

        Args:
            - `json_schema` (Union[str, dict], optional): A string path to a JSON schema file, or a schema in dictionary form. 
                                                          If no value is provided, it will fall back to looking for an environment 
                                                          variable corresponding to the class variable 
                                                          `CONFIG_VALIDATOR_JSON_SCHEMA_ENVVAR_NAME` to find a JSON schema file.
            - `storage_driver` (FS, optional): The storage driver to use. If no value is provided, 
                                               `self.__class__.DEFAULT_STORAGE_DRIVER` is used.

        Raises:
            Exception: An exception is raised if no valid JSON schema is provided or found.
        """
        if isinstance(json_schema, (str, dict)):
            self._json_schema = self.__class__.load_json(json_schema, storage_driver=storage_driver)
        elif (default_schema := self.__class__.get_default_json_schema(storage_driver=storage_driver)):
            self._json_schema = default_schema
        else:
            raise Exception(
                'did not receive or find a JSON schema (after falling back to os.environ["{}"])'.format(
                    self.__class__.CONFIG_VALIDATOR_JSON_SCHEMA_ENVVAR_NAME,
                ))

    def load_config(self, config: dict):
        extracted_config = extract_declared_items(self._json_schema, config)
        coerced_config = coerce_primitive_values(self._json_schema, extracted_config)
        validator = jsonschema.Draft4Validator(self._json_schema)
        errors = list(validator.iter_errors(coerced_config))
        if errors:
            raise ConfigValidatorException(errors)
        return coerced_config
    
    @classmethod
    def load_validated_config(cls, json_schema: Union[str, dict], config: dict, **kwargs):
        return cls(json_schema, **kwargs).load_config(config)

    @classmethod
    def load_validated_environment(cls, json_schema: Union[str, dict]=None, **kwargs):
        return cls.load_validated_config(json_schema, dict(os.environ), **kwargs)
        
    @classmethod
    def load_dotenv(cls,
                    json_schema: Union[str, dict]=None,
                    dotenv_path: str=None,
                    storage_driver: FS=None,
                    override: bool=False,
                   ):
        """
        Loads environment variables from a .env file or JSON schema defaults, where applicable, into `os.environ`.

        Args:
            - `json_schema` (Union[str, dict], optional):
                                            A JSON schema used for extraction. It can be a string path to 
                                            a JSON schema file or a dictionary. If not provided, another method 
                                            (such as an environment variable or default schema) is used.
            - `dotenv_path` (str, optional): Path to the .env file to load the variables from.
                                             If not provided, loads an empty dict to start.
            - `storage_driver` (FS, optional): The storage driver to use for loading files. If not given,
                                               ".env" will be attempted from the current working directory;
                                               if that does not exist, an empty dict will be used.
            - `override` (bool, optional): If True, variables from the .env file or schema default override existing
                                           `os.environ` variables.
        """

        # WARN this sidesteps storage_driver!
        # it will cause breakage if storage_driver != OSFS AND `.env` exists in PWD
        if dotenv_path is None:
            maybe_dotenv_path = dotenv.find_dotenv()  # '' if not exist; else abspath
            if maybe_dotenv_path:
                logger.debug(f'using detected dotenv path; {maybe_dotenv_path}')
                dotenv_path = maybe_dotenv_path
        
        config = None
        
        if dotenv_path:
            dotenv_storage_driver = storage_driver or cls._get_maybe_abspath_driver(dotenv_path)
            with dotenv_storage_driver.open(dotenv_path) as ifile:
                config = dotenv.dotenv_values(stream=ifile)
                
        if config is None:
            dotenv_storage_driver = storage_driver or cls.get_default_storage_driver()
            if dotenv_storage_driver.exists('.env'):  # unlike dotenv.find_dotenv, stay relative!
                with dotenv_storage_driver.open('.env') as ifile:
                    config = dotenv.dotenv_values(stream=ifile)
        
        if config is None:
            config = {}
        
        if json_schema is not None:
            json_schema_dict = cls.load_json(json_schema, storage_driver=storage_driver) or {}
        else: 
            json_schema_dict = cls.get_default_json_schema(storage_driver)
            
        for key in json_schema_dict.get('properties', []):
            if key not in os.environ:
                continue
            if key in config and config[key] != os.environ[key]:
                logger.debug(f'os.environ key "{key}" overriding value present in {dotenv_path}')
            config[key] = os.environ[key]
        validated_config = cls.load_validated_config(
            json_schema or cls.get_default_json_schema(storage_driver=storage_driver),
            config, storage_driver=storage_driver)
        
        if override:
            for key, value in validated_config.items():
                if key in os.environ:
                    continue
                os.environ[key] = str(value)
                
        return validated_config
