"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Effect = exports.PolicyStatement = void 0;
const cdk = require("@aws-cdk/core");
const principals_1 = require("./principals");
const util_1 = require("./util");
const ensureArrayOrUndefined = (field) => {
    if (field === undefined) {
        return undefined;
    }
    if (typeof (field) !== 'string' && !Array.isArray(field)) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    if (Array.isArray(field) && !!field.find((f) => typeof (f) !== 'string')) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    return Array.isArray(field) ? field : [field];
};
/**
 * Represents a statement in an IAM policy document.
 *
 * @stability stable
 */
class PolicyStatement {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.notPrincipal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        // Validate actions
        for (const action of [...props.actions || [], ...props.notActions || []]) {
            if (!/^(\*|[a-zA-Z0-9-]+:[a-zA-Z0-9*]+)$/.test(action)) {
                throw new Error(`Action '${action}' is invalid. An action string consists of a service namespace, a colon, and the name of an action. Action names can include wildcards.`);
            }
        }
        this.sid = props.sid;
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    /**
     * Creates a new PolicyStatement based on the object provided.
     *
     * This will accept an object created from the `.toJSON()` call
     *
     * @param obj the PolicyStatement in object form.
     * @stability stable
     */
    static fromJson(obj) {
        return new PolicyStatement({
            sid: obj.Sid,
            actions: ensureArrayOrUndefined(obj.Action),
            resources: ensureArrayOrUndefined(obj.Resource),
            conditions: obj.Condition,
            effect: obj.Effect,
            notActions: ensureArrayOrUndefined(obj.NotAction),
            notResources: ensureArrayOrUndefined(obj.NotResource),
            principals: obj.Principal ? [new JsonPrincipal(obj.Principal)] : undefined,
            notPrincipals: obj.NotPrincipal ? [new JsonPrincipal(obj.NotPrincipal)] : undefined,
        });
    }
    //
    // Actions
    //
    /**
     * Specify allowed actions into the "Action" section of the policy statement.
     *
     * @param actions actions that will be allowed.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html
     * @stability stable
     */
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error('Cannot add \'Actions\' to policy statement if \'NotActions\' have been added');
        }
        this.action.push(...actions);
    }
    /**
     * Explicitly allow all actions except the specified list of actions into the "NotAction" section of the policy document.
     *
     * @param notActions actions that will be denied.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html
     * @stability stable
     */
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error('Cannot add \'NotActions\' to policy statement if \'Actions\' have been added');
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     *
     * @stability stable
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0;
    }
    /**
     * Adds principals to the "Principal" section of a policy statement.
     *
     * @param principals IAM principals that will be added.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
     * @stability stable
     */
    addPrincipals(...principals) {
        if (Object.keys(principals).length > 0 && Object.keys(this.notPrincipal).length > 0) {
            throw new Error('Cannot add \'Principals\' to policy statement if \'NotPrincipals\' have been added');
        }
        for (const principal of principals) {
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    /**
     * Specify principals that is not allowed or denied access to the "NotPrincipal" section of a policy statement.
     *
     * @param notPrincipals IAM principals that will be denied access.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html
     * @stability stable
     */
    addNotPrincipals(...notPrincipals) {
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this.principal).length > 0) {
            throw new Error('Cannot add \'NotPrincipals\' to policy statement if \'Principals\' have been added');
        }
        for (const notPrincipal of notPrincipals) {
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this.notPrincipal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    /**
     * Specify AWS account ID as the principal entity to the "Principal" section of a policy statement.
     *
     * @stability stable
     */
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    /**
     * Specify a principal using the ARN  identifier of the principal.
     *
     * You cannot specify IAM groups and instance profiles as principals.
     *
     * @param arn ARN identifier of AWS account, IAM user, or IAM role (i.e. arn:aws:iam::123456789012:user/user-name).
     * @stability stable
     */
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts options for adding the service principal (such as specifying a principal in a different region).
     * @stability stable
     */
    addServicePrincipal(service, opts) {
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    /**
     * Adds a federated identity provider such as Amazon Cognito to this policy statement.
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com').
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    /**
     * Adds an AWS account root user principal to this policy statement.
     *
     * @stability stable
     */
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    /**
     * Adds a canonical user ID principal to this policy document.
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     * @stability stable
     */
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    /**
     * Adds all identities in all accounts ("*") to this policy statement.
     *
     * @stability stable
     */
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.Anyone());
    }
    //
    // Resources
    //
    /**
     * Specify resources that this policy statement applies into the "Resource" section of this policy statement.
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement applies to.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_resource.html
     * @stability stable
     */
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error('Cannot add \'Resources\' to policy statement if \'NotResources\' have been added');
        }
        this.resource.push(...arns);
    }
    /**
     * Specify resources that this policy statement will not apply to in the "NotResource" section of this policy statement.
     *
     * All resources except the specified list will be matched.
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement does not apply to.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html
     * @stability stable
     */
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error('Cannot add \'NotResources\' to policy statement if \'Resources\' have been added');
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     *
     * @stability stable
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     *
     * @stability stable
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy.
     *
     * @stability stable
     */
    addCondition(key, value) {
        const existingValue = this.condition[key];
        this.condition[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Add multiple conditions to the Policy.
     *
     * @stability stable
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account.
     *
     * @stability stable
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    /**
     * JSON-ify the policy statement.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toStatementJson() {
        return noUndef({
            Action: _norm(this.action),
            NotAction: _norm(this.notAction),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            NotPrincipal: _normPrincipal(this.notPrincipal),
            Resource: _norm(this.resource),
            NotResource: _norm(this.notResource),
            Sid: _norm(this.sid),
        });
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isUnresolved(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
    /**
     * String representation of this policy statement.
     *
     * @stability stable
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement',
        });
    }
    /**
     * JSON-ify the statement.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON() {
        return this.toStatementJson();
    }
    /**
     * Add a principal's conditions
     *
     * For convenience, principals have been modeled as both a principal
     * and a set of conditions. This makes it possible to have a single
     * object represent e.g. an "SNS Topic" (SNS service principal + aws:SourcArn
     * condition) or an Organization member (* + aws:OrgId condition).
     *
     * However, when using multiple principals in the same policy statement,
     * they must all have the same conditions or the OR samentics
     * implied by a list of principals cannot be guaranteed (user needs to
     * add multiple statements in that case).
     */
    addPrincipalConditions(conditions) {
        // Stringifying the conditions is an easy way to do deep equality
        const theseConditions = JSON.stringify(conditions);
        if (this.principalConditionsJson === undefined) {
            // First principal, anything goes
            this.principalConditionsJson = theseConditions;
        }
        else {
            if (this.principalConditionsJson !== theseConditions) {
                throw new Error(`All principals in a PolicyStatement must have the same Conditions (got '${this.principalConditionsJson}' and '${theseConditions}'). Use multiple statements instead.`);
            }
        }
        this.addConditions(conditions);
    }
    /**
     * Validate that the policy statement satisfies base requirements for a policy.
     *
     * @stability stable
     */
    validateForAnyPolicy() {
        const errors = new Array();
        if (this.action.length === 0 && this.notAction.length === 0) {
            errors.push('A PolicyStatement must specify at least one \'action\' or \'notAction\'.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for a resource-based policy.
     *
     * @stability stable
     */
    validateForResourcePolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length === 0 && Object.keys(this.notPrincipal).length === 0) {
            errors.push('A PolicyStatement used in a resource-based policy must specify at least one IAM principal.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for an identity-based policy.
     *
     * @stability stable
     */
    validateForIdentityPolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0) {
            errors.push('A PolicyStatement used in an identity-based policy cannot specify any IAM principals.');
        }
        if (Object.keys(this.resource).length === 0 && Object.keys(this.notResource).length === 0) {
            errors.push('A PolicyStatement used in an identity-based policy must specify at least one resource.');
        }
        return errors;
    }
}
exports.PolicyStatement = PolicyStatement;
/**
 * The Effect element of an IAM policy.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_effect.html
 * @stability stable
 */
var Effect;
(function (Effect) {
    Effect["ALLOW"] = "Allow";
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function noUndef(x) {
    const ret = {};
    for (const [key, value] of Object.entries(x)) {
        if (value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
class JsonPrincipal extends principals_1.PrincipalBase {
    constructor(json = {}) {
        super();
        // special case: if principal is a string, turn it into an "AWS" principal
        if (typeof (json) === 'string') {
            json = { AWS: json };
        }
        this.policyFragment = {
            principalJson: json,
            conditions: {},
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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