"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositePrincipal = exports.Anyone = exports.AnyPrincipal = exports.AccountRootPrincipal = exports.OpenIdConnectPrincipal = exports.WebIdentityPrincipal = exports.FederatedPrincipal = exports.CanonicalUserPrincipal = exports.OrganizationPrincipal = exports.ServicePrincipal = exports.AccountPrincipal = exports.ArnPrincipal = exports.PrincipalPolicyFragment = exports.PrincipalWithConditions = exports.PrincipalBase = void 0;
const cdk = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals.
 *
 * @stability stable
 */
class PrincipalBase {
    constructor() {
        /**
         * The principal to grant permissions to.
         *
         * @stability stable
         */
        this.grantPrincipal = this;
        /**
         * The AWS account ID of this principal.
         *
         * Can be undefined when the account is not known
         * (for example, for service principals).
         * Can be a Token - in that case,
         * it's assumed to be AWS::AccountId.
         *
         * @stability stable
         */
        this.principalAccount = undefined;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         *
         * @stability stable
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPrincipalPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return { statementAdded: false };
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    /**
     * JSON-ify the principal.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     * @stability stable
     */
    withConditions(conditions) {
        return new PrincipalWithConditions(this, conditions);
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 *
 * @stability stable
 */
class PrincipalWithConditions {
    /**
     * @stability stable
     */
    constructor(principal, conditions) {
        this.principal = principal;
        /**
         * The principal to grant permissions to.
         *
         * @stability stable
         */
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         *
         * @stability stable
         */
        this.assumeRoleAction = this.principal.assumeRoleAction;
        this.additionalConditions = conditions;
    }
    /**
     * Add a condition to the principal.
     *
     * @stability stable
     */
    addCondition(key, value) {
        const existingValue = this.conditions[key];
        this.conditions[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Adds multiple conditions to the principal.
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     *
     * @stability stable
     */
    addConditions(conditions) {
        Object.entries(conditions).forEach(([key, value]) => {
            this.addCondition(key, value);
        });
    }
    /**
     * The conditions under which the policy is in effect.
     *
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     *
     * @stability stable
     */
    get conditions() {
        return this.mergeConditions(this.principal.policyFragment.conditions, this.additionalConditions);
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment(this.principal.policyFragment.principalJson, this.conditions);
    }
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPrincipalPolicy(statement) {
        return this.principal.addToPrincipalPolicy(statement);
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return this.principal.toString();
    }
    /**
     * JSON-ify the principal.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    mergeConditions(principalConditions, additionalConditions) {
        const mergedConditions = {};
        Object.entries(principalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = condition;
        });
        Object.entries(additionalConditions).forEach(([operator, condition]) => {
            // merge the conditions if one of the additional conditions uses an
            // operator that's already used by the principal's conditions merge the
            // inner structure.
            const existing = mergedConditions[operator];
            if (!existing) {
                mergedConditions[operator] = condition;
                return; // continue
            }
            // if either the existing condition or the new one contain unresolved
            // tokens, fail the merge. this is as far as we go at this point.
            if (cdk.Token.isUnresolved(condition) || cdk.Token.isUnresolved(existing)) {
                throw new Error(`multiple "${operator}" conditions cannot be merged if one of them contains an unresolved token`);
            }
            mergedConditions[operator] = { ...existing, ...condition };
        });
        return mergedConditions;
    }
}
exports.PrincipalWithConditions = PrincipalWithConditions;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 *
 * @stability stable
 */
class PrincipalPolicyFragment {
    /**
     * @param principalJson JSON of the "Principal" section in a policy statement.
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(principalJson, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
/**
 * Specify a principal by the Amazon Resource Name (ARN).
 *
 * You can specify AWS accounts, IAM users, Federated SAML users, IAM roles, and specific assumed-role sessions.
 * You cannot specify IAM groups or instance profiles as principals
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
 * @stability stable
 */
class ArnPrincipal extends PrincipalBase {
    /**
     * @param arn Amazon Resource Name (ARN) of the principal entity (i.e. arn:aws:iam::123456789012:user/user-name).
     * @stability stable
     */
    constructor(arn) {
        super();
        this.arn = arn;
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
/**
 * Specify AWS account ID as the principal entity in a policy to delegate authority to the account.
 *
 * @stability stable
 */
class AccountPrincipal extends ArnPrincipal {
    /**
     * @param accountId AWS account ID (i.e. 123456789012).
     * @stability stable
     */
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 *
 * @stability stable
 */
class ServicePrincipal extends PrincipalBase {
    /**
     * @param service AWS service (i.e. sqs.amazonaws.com).
     * @stability stable
     */
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString(),
            ],
        }, this.opts.conditions);
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization.
 *
 * @stability stable
 */
class OrganizationPrincipal extends PrincipalBase {
    /**
     * @param organizationId The unique identifier (ID) of an organization (i.e. o-12345abcde).
     * @stability stable
     */
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy principal for canonicalUserIds - useful for S3 bucket policies that use Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 * @stability stable
 */
class CanonicalUserPrincipal extends PrincipalBase {
    /**
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     * @stability stable
     */
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
/**
 * Principal entity that represents a federated identity provider such as Amazon Cognito, that can be used to provide temporary security credentials to users who have been authenticated.
 *
 * Additional condition keys are available when the temporary security credentials are used to make a request.
 * You can use these keys to write policies that limit the access of federated users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#condition-keys-wif
 * @stability stable
 */
class FederatedPrincipal extends PrincipalBase {
    /**
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito).
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
/**
 * A principal that represents a federated identity provider as Web Identity such as Cognito, Amazon, Facebook, Google, etc.
 *
 * @stability stable
 */
class WebIdentityPrincipal extends FederatedPrincipal {
    /**
     * @param identityProvider identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito).
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(identityProvider, conditions = {}) {
        super(identityProvider, conditions !== null && conditions !== void 0 ? conditions : {}, 'sts:AssumeRoleWithWebIdentity');
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `WebIdentityPrincipal(${this.federated})`;
    }
}
exports.WebIdentityPrincipal = WebIdentityPrincipal;
/**
 * A principal that represents a federated identity provider as from a OpenID Connect provider.
 *
 * @stability stable
 */
class OpenIdConnectPrincipal extends WebIdentityPrincipal {
    /**
     * @param openIdConnectProvider OpenID Connect provider.
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(openIdConnectProvider, conditions = {}) {
        super(openIdConnectProvider.openIdConnectProviderArn, conditions !== null && conditions !== void 0 ? conditions : {});
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `OpenIdConnectPrincipal(${this.federated})`;
    }
}
exports.OpenIdConnectPrincipal = OpenIdConnectPrincipal;
/**
 * Use the AWS account into which a stack is deployed as the principal entity in a policy.
 *
 * @stability stable
 */
class AccountRootPrincipal extends AccountPrincipal {
    /**
     * @stability stable
     */
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return 'AccountRootPrincipal()';
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts.
 *
 * @stability stable
 */
class AnyPrincipal extends ArnPrincipal {
    /**
     * @stability stable
     */
    constructor() {
        super('*');
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return 'AnyPrincipal()';
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * (deprecated) A principal representing all identities in all accounts.
 *
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
/**
 * Represents a principal that has multiple types of principals.
 *
 * A composite principal cannot
 * have conditions. i.e. multiple ServicePrincipals that form a composite principal
 *
 * @stability stable
 */
class CompositePrincipal extends PrincipalBase {
    /**
     * @stability stable
     */
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    /**
     * Adds IAM principals to the composite principal.
     *
     * Composite principals cannot have
     * conditions.
     *
     * @param principals IAM principals that will be added to the composite principal.
     * @stability stable
     */
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error('Cannot add multiple principals with different "assumeRoleAction". ' +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error('Components of a CompositePrincipal must not have conditions. ' +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return '<unresolved-token>';
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service,
        });
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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