"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expiration = void 0;
/**
 * Represents a date of expiration.
 *
 * The amount can be specified either as a Date object, timestamp, Duration or string.
 *
 * @stability stable
 */
class Expiration {
    constructor(date) {
        this.date = date;
    }
    /**
     * Expire at the specified date.
     *
     * @param d date to expire at.
     * @stability stable
     */
    static atDate(d) { return new Expiration(d); }
    /**
     * Expire at the specified timestamp.
     *
     * @param t timestamp in unix milliseconds.
     * @stability stable
     */
    static atTimestamp(t) { return Expiration.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time.
     *
     * @param t the duration to wait before expiring.
     * @stability stable
     */
    static after(t) { return Expiration.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Expire at specified date, represented as a string.
     *
     * @param s the string that represents date to expire at.
     * @stability stable
     */
    static fromString(s) { return new Expiration(new Date(s)); }
    /**
     * Exipration Value in a formatted Unix Epoch Time in seconds.
     *
     * @stability stable
     */
    toEpoch() {
        return Math.round(this.date.getTime() / 1000);
    }
    /**
     * Check if Exipiration expires before input.
     *
     * @param t the duration to check against.
     * @stability stable
     */
    isBefore(t) {
        return this.date < new Date(Date.now() + t.toMilliseconds());
    }
    /**
     * Check if Exipiration expires after input.
     *
     * @param t the duration to check against.
     * @stability stable
     */
    isAfter(t) {
        return this.date > new Date(Date.now() + t.toMilliseconds());
    }
}
exports.Expiration = Expiration;
//# sourceMappingURL=data:application/json;base64,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