import base64
import os
import hashlib
from typing import Optional, Union
from enum import Enum
from time import time
from logly import logger


class CacheBackend(Enum):
    """Доступные типы кэша"""
    MEMORY = "memory"
    REDIS = "redis"


class PKCEManager:
    """
    Proof Key for Code Exchange (PKCE) менеджер с поддержкой гибридного кэша.
    
    Сохраняет: state → code_verifier
    
    Поддерживает:
    - In-memory кэш (оперативная память)
    - Redis кэш
    - Комбинированный подход (оба одновременно)
    """

    def __init__(
        self,
        ttl: int = 300,
        backend: CacheBackend = CacheBackend.MEMORY,
        aio_redis_client=None,
    ):
        """
        Args:
            ttl: Time to live для кэша в секундах
            backend: CacheBackend.MEMORY или CacheBackend.REDIS
            aio_redis_client: Redis клиент (требуется если backend=REDIS)
        """
        self.ttl = ttl
        self.backend = backend
        self.redis = aio_redis_client
        
        # In-memory хранилище: {state: (code_verifier, expiry_time)}
        self.memory_cache: dict[str, tuple[str, float]] = {}

    @staticmethod
    def generate_state() -> str:
        """Генерирует state"""
        return base64.urlsafe_b64encode(os.urandom(16)).rstrip(b"=").decode()

    @staticmethod
    def generate_pkce_pair() -> tuple[str, str]:
        """Генерирует пару code_verifier и code_challenge"""
        code_verifier = base64.urlsafe_b64encode(os.urandom(40)).rstrip(b"=").decode()
        digest = hashlib.sha256(code_verifier.encode()).digest()
        code_challenge = base64.urlsafe_b64encode(digest).rstrip(b"=").decode()
        return code_verifier, code_challenge

    def _is_expired(self, expiry_time: float) -> bool:
        """Проверяет истек ли срок действия"""
        return time() >= expiry_time

    async def store(self, state: str, code_verifier: str) -> None:
        """
        Сохраняет state → code_verifier в кэш
        
        Args:
            state: PKCE state
            code_verifier: PKCE code verifier
        """
        if self.backend == CacheBackend.MEMORY:
            await self._store_memory(state, code_verifier)
        elif self.backend == CacheBackend.REDIS:
            await self._store_redis(state, code_verifier)

    async def _store_memory(self, state: str, code_verifier: str) -> None:
        """Сохраняет в in-memory кэш"""
        expiry_time = time() + self.ttl
        self.memory_cache[state] = (code_verifier, expiry_time)

    async def _store_redis(self, state: str, code_verifier: str) -> None:
        """Сохраняет в Redis"""
        if not self.redis:
            raise RuntimeError("Redis client not configured")
        key = f"pkce:{state}"
        await self.redis.set(key, code_verifier, ex=self.ttl)

    async def pop(self, state: str) -> Optional[str]:
        """
        Извлекает и удаляет code_verifier из кэша
        
        Args:
            state: PKCE state
            
        Returns:
            code_verifier или None если не найден или истек
        """
        if self.backend == CacheBackend.MEMORY:
            return await self._pop_memory(state)
        elif self.backend == CacheBackend.REDIS:
            return await self._pop_redis(state)

    async def _pop_memory(self, state: str) -> Optional[str]:
        """Извлекает из in-memory кэша"""
        if state not in self.memory_cache:
            return None

        code_verifier, expiry_time = self.memory_cache[state]
        
        # Удаляем в любом случае
        del self.memory_cache[state]
        
        # Проверяем срок действия
        if self._is_expired(expiry_time):
            return None

        return code_verifier

    async def _pop_redis(self, state: str) -> Optional[str]:
        """Извлекает из Redis"""
        if not self.redis:
            raise RuntimeError("Redis client not configured")
        
        key = f"pkce:{state}"
        verifier = await self.redis.get(key)
        if verifier:
            await self.redis.delete(key)
            return verifier
        return None

    async def cleanup_memory(self) -> int:
        """
        Удаляет истекшие записи из in-memory кэша
        
        Returns:
            Количество удаленных записей
        """
        current_time = time()
        expired_keys = [
            state for state, (_, expiry_time) in self.memory_cache.items()
            if current_time >= expiry_time
        ]
        
        for state in expired_keys:
            del self.memory_cache[state]
        
        return len(expired_keys)

    async def get_memory_stats(self) -> dict:
        """Возвращает статистику in-memory кэша"""
        return {
            "total_entries": len(self.memory_cache),
            "backend": self.backend.value,
            "ttl": self.ttl,
        }


class HybridPKCEManager(PKCEManager):
    """
    Расширенный PKCE менеджер с поддержкой одновременного использования
    обоих типов кэша (memory и Redis).
    
    Стратегия:
    - Записи идут в оба хранилища одновременно
    - При чтении сначала пытаемся memory, затем Redis
    - Оба хранилища синхронизируются
    """

    def __init__(self, ttl: int = 300, aio_redis_client=None):
        """
        Args:
            ttl: Time to live для кэша в секундах
            aio_redis_client: Redis клиент
        """
        super().__init__(ttl=ttl, backend=CacheBackend.MEMORY, aio_redis_client=aio_redis_client)
        self.use_redis = aio_redis_client is not None


    async def store(self, state: str, code_verifier: str) -> None:
        """
        Сохраняет в оба кэша одновременно
        
        Args:
            state: PKCE state
            code_verifier: PKCE code verifier
        """
        # Сохраняем в memory
        await self._store_memory(state, code_verifier)
        
        # Сохраняем в Redis, если доступен
        if self.use_redis:
            try:
                await self._store_redis(state, code_verifier)
            except Exception as e:
                # Логируем ошибку, но продолжаем работу с memory
                logger.error(f"Redis store error: {e.__str__()}")


    async def pop(self, state: str) -> Optional[str]:
        """
        Извлекает из кэша (сначала из memory, затем из Redis)
        
        Args:
            state: PKCE state
            
        Returns:
            code_verifier или None
        """
        # Пытаемся в memory

        verifier = await self._pop_memory(state)

        if verifier:
            # Удаляем из Redis если была найдена в memory
            if self.use_redis:
                try:
                    await self._pop_redis(state)
                except Exception as e:
                    logger.error(f"Redis pop error: {e.__str__()}")
            return verifier

        # Если не в memory, пытаемся в Redis
        if self.use_redis:
            try:
                verifier = await self._pop_redis(state)
                return verifier
            except Exception as e:
                logger.error(f"Redis pop error: {e.__str__()}")

        return None

    async def get_stats(self) -> dict:
        """Возвращает объединенную статистику"""
        memory_stats = await self.get_memory_stats()
        return {
            **memory_stats,
            "hybrid_mode": True,
            "redis_enabled": self.use_redis,
        }

