"""

--------------------

This script demonstrates basic operations on object like creating new
object, placing it into scene, selecting it and making it active.

```../examples/bpy.types.Object.py```


--------------------

PropertyGroups are the base class for dynamically defined sets of properties.

They can be used to extend existing blender data with your own types which can
be animated, accessed from the user interface and from python.

[NOTE]
The values assigned to blender data are saved to disk but the class
definitions are not, this means whenever you load blender the class needs
to be registered too.
This is best done by creating an addon which loads on startup and registers
your properties.

[NOTE]
PropertyGroups must be registered before assigning them to blender data.

Property types used in class declarations are all in bpy.props



```../examples/bpy.types.PropertyGroup.py```


--------------------

Here is an example of a simple menu. Menus differ from panels in that they must
reference from a header, panel or another menu.

Notice the 'CATEGORY_MT_name' in  Menu.bl_idname, this is a naming
convention for menus.

[NOTE]
Menu subclasses must be registered before referencing them from blender.

[NOTE]
Menus have their Layout.operator_context initialized as
'EXEC_REGION_WIN' rather than 'INVOKE_DEFAULT' (see Execution Context <operator-execution_context>).
If the operator context needs to initialize inputs from the
Operator.invoke function, then this needs to be explicitly set.

```../examples/bpy.types.Menu.py```


--------------------

This menu demonstrates some different functions.

```../examples/bpy.types.Menu.1.py```


--------------------

When creating menus for addons you can't reference menus in Blender's default
scripts.
Instead, the addon can add menu items to existing menus.

The function menu_draw acts like Menu.draw.

```../examples/bpy.types.Menu.2.py```


--------------------

Preset menus are simply a convention that uses a menu sub-class
to perform the common task of managing presets.

This example shows how you can add a preset menu.

This example uses the object draw options,
however you can use properties defined by your own scripts too.

```../examples/bpy.types.Menu.3.py```


--------------------

The mesh data is accessed in object mode and intended for compact storage,
for more flexible mesh editing from python see bmesh.

Blender stores 4 main arrays to define mesh geometry.

* Mesh.vertices (3 points in space)
* Mesh.edges (reference 2 vertices)
* Mesh.loops (reference a single vertex and edge)
* Mesh.polygons: (reference a range of loops)

Each polygon reference a slice in the loop array, this way, polygons do not store vertices or corner data such as UV's directly,
only a reference to loops that the polygon uses.

Mesh.loops, Mesh.uv_layers Mesh.vertex_colors are all aligned so the same polygon loop
indices can be used to find the UV's and vertex colors as with as the vertices.

To compare mesh API options see: NGons and Tessellation Faces <info_gotcha_mesh_faces>

This example script prints the vertices and UV's for each polygon, assumes the active object is a mesh with UVs.

```../examples/bpy.types.Mesh.py```

```../examples/bpy.types.AddonPreferences.1.py```


--------------------

This script is the UIList subclass used to show material slots, with a bunch of additional commentaries.

Notice the name of the class, this naming convention is similar as the one for panels or menus.

[NOTE]
UIList subclasses must be registered for blender to use them.

```../examples/bpy.types.UIList.1.py```


--------------------

This script is an extended version of the UIList

 subclass used to show vertex groups. It is not used 'as is',
because iterating over all vertices in a 'draw' function is a very bad idea for UI performances! However, it's a good
example of how to create/use filtering/reordering callbacks.

```../examples/bpy.types.UIList.2.py```


--------------------

The NodeTree.poll function determines if a node tree is visible
in the given context (similar to how Panel.poll
and Menu.poll define visibility). If it returns False,
the node tree type will not be selectable in the node editor.

A typical condition for shader nodes would be to check the active render engine
of the scene and only show nodes of the renderer they are designed for.

```../examples/bpy.types.NodeTree.py```


--------------------

This script shows simple operator which prints a message.

Since the operator only has an Operator.execute function it takes no
user input.

[NOTE]
Operator subclasses must be registered before accessing them from blender.

```../examples/bpy.types.Operator.py```


--------------------

Operator.invoke is used to initialize the operator from the context
at the moment the operator is called.
invoke() is typically used to assign properties which are then used by
execute().
Some operators don't have an execute() function, removing the ability to be
repeated from a script or macro.

This example shows how to define an operator which gets mouse input to
execute a function and that this operator can be invoked or executed from
the python api.

Also notice this operator defines its own properties, these are different
to typical class properties because blender registers them with the
operator, to use as arguments when called, saved for operator undo/redo and
automatically added into the user interface.

```../examples/bpy.types.Operator.1.py```


--------------------

This example shows how an operator can use the file selector.

Notice the invoke function calls a window manager method and returns
{'RUNNING_MODAL'}

, this means the file selector stays open and the operator does not
exit immediately after invoke finishes.

The file selector runs the operator, calling Operator.execute when the
user confirms.

The Operator.poll function is optional, used to check if the operator
can run.

```../examples/bpy.types.Operator.2.py```


--------------------

This operator uses its Operator.invoke function to call a popup.

```../examples/bpy.types.Operator.3.py```


--------------------

By default operator properties use an automatic user interface layout.
If you need more control you can create your own layout with a
Operator.draw function.

This works like the Panel and Menu draw functions, its used
for dialogs and file selectors.

```../examples/bpy.types.Operator.4.py```


--------------------

This operator defines a Operator.modal function which running,
handling events until it returns {'FINISHED'}

 or {'CANCELLED'}

.

Grab, Rotate, Scale and Fly-Mode are examples of modal operators.
They are especially useful for interactive tools,
your operator can have its own state where keys toggle options as the operator
runs.

Operator.invoke is used to initialize the operator as being by
returning {'RUNNING_MODAL'}

, initializing the modal loop.

Notice __init__()

 and __del__()

 are declared.
For other operator types they are not useful but for modal operators they will
be called before the Operator.invoke and after the operator finishes.

```../examples/bpy.types.Operator.5.py```


--------------------

```../examples/bpy.types.RenderEngine.py```


--------------------

This script is a simple panel which will draw into the object properties
section.

Notice the 'CATEGORY_PT_name' Panel.bl_idname, this is a naming
convention for panels.

[NOTE]
Panel subclasses must be registered for blender to use them.

```../examples/bpy.types.Panel.py```


--------------------

This panel has a Panel.poll and Panel.draw_header function,
even though the contents is basic this closely resembles blenders panels.

```../examples/bpy.types.Panel.1.py```


--------------------

A mix-in parent class can be used to share common properties and
Menu.poll function.

```../examples/bpy.types.Panel.2.py```

"""

import typing
import collections.abc
import bl_operators.anim
import bl_operators.clip
import bl_operators.file
import bl_operators.freestyle
import bl_operators.mask
import bl_operators.node
import bl_operators.presets
import bl_operators.view3d
import bl_operators.wm
import bl_ui
import bl_ui.properties_constraint
import bl_ui.properties_data_armature
import bl_ui.properties_data_bone
import bl_ui.properties_data_camera
import bl_ui.properties_data_curve
import bl_ui.properties_data_empty
import bl_ui.properties_data_lamp
import bl_ui.properties_data_lattice
import bl_ui.properties_data_mesh
import bl_ui.properties_data_metaball
import bl_ui.properties_data_modifier
import bl_ui.properties_data_speaker
import bl_ui.properties_freestyle
import bl_ui.properties_game
import bl_ui.properties_grease_pencil_common
import bl_ui.properties_mask_common
import bl_ui.properties_material
import bl_ui.properties_object
import bl_ui.properties_paint_common
import bl_ui.properties_particle
import bl_ui.properties_physics_cloth
import bl_ui.properties_physics_common
import bl_ui.properties_physics_dynamicpaint
import bl_ui.properties_physics_field
import bl_ui.properties_physics_fluid
import bl_ui.properties_physics_rigidbody
import bl_ui.properties_physics_rigidbody_constraint
import bl_ui.properties_physics_smoke
import bl_ui.properties_physics_softbody
import bl_ui.properties_render
import bl_ui.properties_render_layer
import bl_ui.properties_scene
import bl_ui.properties_texture
import bl_ui.properties_world
import bl_ui.space_clip
import bl_ui.space_console
import bl_ui.space_dopesheet
import bl_ui.space_filebrowser
import bl_ui.space_graph
import bl_ui.space_image
import bl_ui.space_info
import bl_ui.space_logic
import bl_ui.space_nla
import bl_ui.space_node
import bl_ui.space_outliner
import bl_ui.space_properties
import bl_ui.space_sequencer
import bl_ui.space_text
import bl_ui.space_time
import bl_ui.space_userpref
import bl_ui.space_view3d
import bl_ui.space_view3d_toolbar
import keyingsets_builtins
import mathutils

GenericType1 = typing.TypeVar("GenericType1")
GenericType2 = typing.TypeVar("GenericType2")

class bpy_prop_collection(typing.Generic[GenericType1]):
    """built-in class used for all collections."""

    def find(self, key: str | None) -> int:
        """Returns the index of a key in a collection or -1 when not found
        (matches pythons string find function of the same name).

                :param key: The identifier for the collection member.
                :type key: str | None
                :return: index of the key.
                :rtype: int
        """
        ...

    def foreach_get(
        self,
        attr: str,
        seq: collections.abc.MutableSequence[bool]
        | collections.abc.MutableSequence[int]
        | collections.abc.MutableSequence[float],
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        Multi-dimensional arrays (like array of vectors) will be flattened into seq.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.MutableSequence[bool] | collections.abc.MutableSequence[int] | collections.abc.MutableSequence[float]
        """
        ...

    def foreach_set(
        self,
        attr: str,
        seq: collections.abc.Sequence[bool]
        | collections.abc.Sequence[int]
        | collections.abc.Sequence[float],
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        seq must be uni-dimensional, multi-dimensional arrays (like array of vectors) will be re-created from it.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.Sequence[bool] | collections.abc.Sequence[int] | collections.abc.Sequence[float]
        """
        ...

    def get(
        self, key: str | None, default: GenericType2 = None
    ) -> GenericType1 | GenericType2:
        """Returns the value of the item assigned to key or default when not found
        (matches pythons dictionary function of the same name).

                :param key: The identifier for the collection member.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
                :type default: GenericType2
                :return:
                :rtype: GenericType1 | GenericType2
        """
        ...

    def items(self) -> list[tuple[str, GenericType1]]:
        """Return the identifiers of collection members
        (matching pythons dict.items() functionality).

                :return:
                :rtype: list[tuple[str, GenericType1]]
        """
        ...

    def keys(self) -> list[str]:
        """Return the identifiers of collection members
        (matching pythons dict.keys() functionality).

                :return: the identifiers for each member of this collection.
                :rtype: list[str]
        """
        ...

    def values(self) -> list[GenericType1]:
        """Return the values of collection
        (matching pythons dict.values() functionality).

                :return:
                :rtype: list[GenericType1]
        """
        ...

    @typing.overload
    def __getitem__(self, key: int | str) -> GenericType1:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: GenericType1
        """
        ...

    @typing.overload
    def __getitem__(self, key: slice) -> tuple[GenericType1]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: tuple[GenericType1]
        """
        ...

    def __getitem__(self, key: int | str | slice) -> GenericType1 | tuple[GenericType1]:
        """

        :param key:
        :type key: int | str | slice
        :return:
        :rtype: GenericType1 | tuple[GenericType1]
        """
        ...

    @typing.overload
    def __setitem__(self, key: int | str, value: GenericType1):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: GenericType1
        """
        ...

    @typing.overload
    def __setitem__(self, key: slice, value: tuple[GenericType1]):
        """

        :param key:
        :type key: slice
        :param value:
        :type value: tuple[GenericType1]
        """
        ...

    def __setitem__(
        self, key: int | str | slice, value: GenericType1 | tuple[GenericType1]
    ):
        """

        :param key:
        :type key: int | str | slice
        :param value:
        :type value: GenericType1 | tuple[GenericType1]
        """
        ...

    def __delitem__(self, key: int | str | slice):
        """

        :param key:
        :type key: int | str | slice
        """
        ...

    def __iter__(self) -> collections.abc.Iterator[GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[GenericType1]
        """
        ...

    def __next__(self) -> GenericType1:
        """

        :return:
        :rtype: GenericType1
        """
        ...

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """
        ...

    def __contains__(self, key: str | tuple[str, ...]) -> bool:
        """

        :param key:
        :type key: str | tuple[str, ...]
        :return:
        :rtype: bool
        """
        ...

class bpy_struct(typing.Generic[GenericType1]):
    """built-in base class for all classes in bpy.types."""

    id_data: typing.Any
    """ The `bpy.types.ID` object this datablock is from or None, (not available for all data types)"""

    def as_pointer(self) -> int:
        """Returns the memory address which holds a pointer to blenders internal data

        :return: int (memory address).
        :rtype: int
        """
        ...

    def driver_add(self, path: str | None, index: int | None = -1) -> FCurve:
        """Adds driver(s) to the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: The driver(s) added.
        :rtype: FCurve
        """
        ...

    def driver_remove(self, path: str | None, index: int | None = -1) -> bool:
        """Remove driver(s) from the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: Success of driver removal.
        :rtype: bool
        """
        ...

    def get(self, key: str | None, default=None):
        """Returns the value of the custom property assigned to key or default
        when not found (matches pythons dictionary function of the same name).

                :param key: The key associated with the custom property.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
        """
        ...

    def is_property_hidden(self, property) -> bool:
        """Check if a property is hidden.

        :param property:
        :return: True when the property is hidden.
        :rtype: bool
        """
        ...

    def is_property_readonly(self, property) -> bool:
        """Check if a property is readonly.

        :param property:
        :return: True when the property is readonly (not writable).
        :rtype: bool
        """
        ...

    def is_property_set(self, property) -> bool:
        """Check if a property is set, use for testing operator properties.

        :param property:
        :return: True when the property has been set.
        :rtype: bool
        """
        ...

    def items(self):
        """Returns the items of this objects custom properties (matches pythons
        dictionary function of the same name).

                :return: custom property key, value pairs.
        """
        ...

    def keyframe_delete(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = None,
        group: str | None = "",
    ) -> bool:
        """Remove a keyframe from this properties fcurve.

        :param data_path: path to the property to remove a key, analogous to the fcurve's data path.
        :type data_path: str | None
        :param index: array index of the property to remove a key. Defaults to -1 removing all indices or a single channel if the property is not an array.
        :type index: int | None
        :param frame: The frame on which the keyframe is deleted, defaulting to the current frame.
        :type frame: float | None
        :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
        :type group: str | None
        :return: Success of keyframe deleation.
        :rtype: bool
        """
        ...

    def keyframe_insert(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = None,
        group: str | None = "",
    ) -> bool:
        """Insert a keyframe on the property given, adding fcurves and animation data when necessary.This is the most simple example of inserting a keyframe from python.Note that when keying data paths which contain nested properties this must be
        done from the `ID` subclass, in this case the `Armature` rather
        than the bone.

                :param data_path: path to the property to key, analogous to the fcurve's data path.
                :type data_path: str | None
                :param index: array index of the property to key.
        Defaults to -1 which will key all indices or a single channel if the property is not an array.
                :type index: int | None
                :param frame: The frame on which the keyframe is inserted, defaulting to the current frame.
                :type frame: float | None
                :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
                :type group: str | None
                :return: Success of keyframe insertion.
                :rtype: bool
        """
        ...

    def keys(self) -> list[str]:
        """Returns the keys of this objects custom properties (matches pythons
        dictionary function of the same name).

                :return: custom property keys.
                :rtype: list[str]
        """
        ...

    def path_from_id(self, property: str | None = "") -> str:
        """Returns the data path from the ID to this object (string).

                :param property: Optional property name which can be used if the path is
        to a property of this object.
                :type property: str | None
                :return: The path from `bpy.types.bpy_struct.id_data`
        to this struct and property (when given).
                :rtype: str
        """
        ...

    def path_resolve(self, path: str | None, coerce: bool | None = True):
        """Returns the property from the path, raise an exception when not found.

                :param path: path which this property resolves.
                :type path: str | None
                :param coerce: optional argument, when True, the property will be converted
        into its python representation.
                :type coerce: bool | None
        """
        ...

    def property_unset(self, property):
        """Unset a property, will use default value afterward.

        :param property:
        """
        ...

    def type_recast(self):
        """Return a new instance, this is needed because types
        such as textures can be changed at runtime.

                :return: a new instance of this object with the type initialized again.
        """
        ...

    def values(self) -> list:
        """Returns the values of this objects custom properties (matches pythons
        dictionary function of the same name).

                :return: custom property values.
                :rtype: list
        """
        ...

    def __getitem__(self, key: int | str) -> typing.Any:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: typing.Any
        """
        ...

    def __setitem__(self, key: int | str, value: typing.Any):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: typing.Any
        """
        ...

    def __delitem__(self, key: int | str):
        """

        :param key:
        :type key: int | str
        """
        ...

class bpy_prop_array(typing.Generic[GenericType1]):
    def __get__(self, instance, owner) -> bpy_prop_array[GenericType1]:
        """

        :param instance:
        :param owner:
        :return:
        :rtype: bpy_prop_array[GenericType1]
        """
        ...

    def __set__(self, instance, value: collections.abc.Iterable[GenericType1]):
        """

        :param instance:
        :param value:
        :type value: collections.abc.Iterable[GenericType1]
        """
        ...

    def foreach_get(self, seq: collections.abc.MutableSequence[GenericType1]):
        """

        :param seq:
        :type seq: collections.abc.MutableSequence[GenericType1]
        """
        ...

    def foreach_set(self, seq: typing.Sequence[GenericType1]):
        """

        :param seq:
        :type seq: typing.Sequence[GenericType1]
        """
        ...

    @typing.overload
    def __getitem__(self, key: int) -> GenericType1:
        """

        :param key:
        :type key: int
        :return:
        :rtype: GenericType1
        """
        ...

    @typing.overload
    def __getitem__(self, key: slice) -> tuple[GenericType1]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: tuple[GenericType1]
        """
        ...

    def __getitem__(self, key: int | slice) -> GenericType1 | tuple[GenericType1]:
        """

        :param key:
        :type key: int | slice
        :return:
        :rtype: GenericType1 | tuple[GenericType1]
        """
        ...

    @typing.overload
    def __setitem__(self, key: int, value: GenericType1):
        """

        :param key:
        :type key: int
        :param value:
        :type value: GenericType1
        """
        ...

    @typing.overload
    def __setitem__(self, key: slice, value: tuple[GenericType1]):
        """

        :param key:
        :type key: slice
        :param value:
        :type value: tuple[GenericType1]
        """
        ...

    def __setitem__(self, key: int | slice, value: GenericType1 | tuple[GenericType1]):
        """

        :param key:
        :type key: int | slice
        :param value:
        :type value: GenericType1 | tuple[GenericType1]
        """
        ...

    def __delitem__(self, key: int | slice):
        """

        :param key:
        :type key: int | slice
        """
        ...

    def __iter__(self) -> collections.abc.Iterator[GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[GenericType1]
        """
        ...

    def __next__(self) -> GenericType1:
        """

        :return:
        :rtype: GenericType1
        """
        ...

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """
        ...

class ActionFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of action F-Curves"""

    def new(
        self,
        data_path: str | typing.Any,
        index: typing.Any | None = 0,
        action_group: str | typing.Any = "",
    ) -> FCurve:
        """Add an F-Curve to the action

        :param data_path: Data Path, F-Curve data path to use
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :param action_group: Action Group, Acton group to add this F-Curve into
        :type action_group: str | typing.Any
        :return: Newly created F-Curve
        :rtype: FCurve
        """
        ...

    def find(self, data_path: str | typing.Any, index: typing.Any | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the action.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """
        ...

    def remove(self, fcurve: FCurve):
        """Remove action group

        :param fcurve: F-Curve to remove
        :type fcurve: FCurve
        """
        ...

class ActionGroup(bpy_struct):
    """Groups of F-Curves"""

    channels: bpy_prop_collection[FCurve]
    """ F-Curves in this group

    :type: bpy_prop_collection[FCurve]
    """

    color_set: str
    """ Custom color set to use

    :type: str
    """

    colors: ThemeBoneColorSet
    """ Copy of the colors associated with the group's color set

    :type: ThemeBoneColorSet
    """

    is_custom_color_set: bool
    """ Color set is user-defined instead of a fixed theme color set

    :type: bool
    """

    lock: bool
    """ Action group is locked

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ Action group is selected

    :type: bool
    """

    show_expanded: bool
    """ Action group is expanded

    :type: bool
    """

class ActionGroups(bpy_prop_collection[ActionGroup], bpy_struct):
    """Collection of action groups"""

    def new(self, name: str | typing.Any) -> ActionGroup:
        """Create a new action group and add it to the action

        :param name: New name for the action group
        :type name: str | typing.Any
        :return: Newly created action group
        :rtype: ActionGroup
        """
        ...

    def remove(self, action_group: ActionGroup):
        """Remove action group

        :param action_group: Action group to remove
        :type action_group: ActionGroup
        """
        ...

class ActionPoseMarkers(bpy_prop_collection[TimelineMarker], bpy_struct):
    """Collection of timeline markers"""

    active: TimelineMarker | None
    """ Active pose marker for this action

    :type: TimelineMarker | None
    """

    active_index: int | None
    """ Index of active pose marker

    :type: int | None
    """

    def new(self, name: str | typing.Any) -> TimelineMarker:
        """Add a pose marker to the action

        :param name: New name for the marker (not unique)
        :type name: str | typing.Any
        :return: Newly created marker
        :rtype: TimelineMarker
        """
        ...

    def remove(self, marker: TimelineMarker):
        """Remove a timeline marker

        :param marker: Timeline marker to remove
        :type marker: TimelineMarker
        """
        ...

class Actuator(bpy_struct):
    """Actuator to apply actions in the game engine"""

    active: bool | None
    """ Set the active state of the actuator

    :type: bool | None
    """

    name: str
    """ 

    :type: str
    """

    pin: bool
    """ Display when not linked to a visible states controller

    :type: bool
    """

    show_expanded: bool
    """ Set actuator expanded in the user interface

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

    def link(self, controller: Controller | None):
        """Link the actuator to a controller

        :param controller: Controller to link to
        :type controller: Controller | None
        """
        ...

    def unlink(self, controller: Controller | None):
        """Unlink the actuator from a controller

        :param controller: Controller to unlink from
        :type controller: Controller | None
        """
        ...

class Addon(bpy_struct):
    """Python add-ons to be loaded automatically"""

    module: str
    """ Module name

    :type: str
    """

    preferences: AddonPreferences
    """ 

    :type: AddonPreferences
    """

class AddonPreferences(bpy_struct):
    bl_idname: str
    """ 

    :type: str
    """

    layout: UILayout
    """ 

    :type: UILayout
    """

class Addons(bpy_prop_collection[Addon], bpy_struct):
    """Collection of add-ons"""

    @classmethod
    def new(cls) -> Addon:
        """Add a new add-on

        :return: Add-on data
        :rtype: Addon
        """
        ...

    @classmethod
    def remove(cls, addon: Addon):
        """Remove add-on

        :param addon: Add-on to remove
        :type addon: Addon
        """
        ...

class AlembicObjectPath(bpy_struct):
    """Path of an object inside of an Alembic archive"""

    path: str
    """ Object path

    :type: str
    """

class AlembicObjectPaths(bpy_prop_collection[AlembicObjectPath], bpy_struct):
    """Collection of object paths"""

    ...

class AnimData(bpy_struct):
    """Animation data for data-block"""

    action: Action
    """ Active Action for this data-block

    :type: Action
    """

    action_blend_type: str
    """ Method used for combining Active Action's result with result of NLA stack

    :type: str
    """

    action_extrapolation: str
    """ Action to take for gaps past the Active Action's range (when evaluating with NLA)

    :type: str
    """

    action_influence: float
    """ Amount the Active Action contributes to the result of the NLA stack

    :type: float
    """

    drivers: AnimDataDrivers
    """ The Drivers/Expressions for this data-block

    :type: AnimDataDrivers
    """

    nla_tracks: NlaTracks
    """ NLA Tracks (i.e. Animation Layers)

    :type: NlaTracks
    """

    use_nla: bool
    """ NLA stack is evaluated when evaluating this block

    :type: bool
    """

    use_tweak_mode: bool
    """ Whether to enable or disable tweak mode in NLA

    :type: bool
    """

class AnimDataDrivers(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of Driver F-Curves"""

    def from_existing(self, src_driver: FCurve | None = None) -> FCurve:
        """Add a new driver given an existing one

        :param src_driver: Existing Driver F-Curve to use as template for a new one
        :type src_driver: FCurve | None
        :return: New Driver F-Curve
        :rtype: FCurve
        """
        ...

    def find(self, data_path: str | typing.Any, index: typing.Any | None = 0) -> FCurve:
        """Find a driver F-Curve. Note that this function performs a linear scan of all driver F-Curves.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """
        ...

class AnimViz(bpy_struct):
    """Settings for the visualization of motion"""

    motion_path: AnimVizMotionPaths
    """ Motion Path settings for visualization

    :type: AnimVizMotionPaths
    """

    onion_skin_frames: AnimVizOnionSkinning
    """ Onion Skinning (ghosting) settings for visualization

    :type: AnimVizOnionSkinning
    """

class AnimVizMotionPaths(bpy_struct):
    """Motion Path settings for animation visualization"""

    bake_location: str
    """ When calculating Bone Paths, use Head or Tips

    :type: str
    """

    frame_after: int
    """ Number of frames to show after the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_before: int
    """ Number of frames to show before the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_end: int
    """ End frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_start: int
    """ Starting frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_step: int
    """ Number of frames between paths shown (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    show_frame_numbers: bool
    """ Show frame numbers on Motion Paths

    :type: bool
    """

    show_keyframe_action_all: bool
    """ For bone motion paths, search whole Action for keyframes instead of in group with matching name only (is slower)

    :type: bool
    """

    show_keyframe_highlight: bool
    """ Emphasize position of keyframes on Motion Paths

    :type: bool
    """

    show_keyframe_numbers: bool
    """ Show frame numbers of Keyframes on Motion Paths

    :type: bool
    """

    type: str
    """ Type of range to show for Motion Paths

    :type: str
    """

class AnimVizOnionSkinning(bpy_struct):
    """Onion Skinning settings for animation visualization"""

    frame_after: int
    """ Number of frames to show after the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_before: int
    """ Number of frames to show before the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_end: int
    """ End frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_start: int
    """ Starting frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_step: int
    """ Number of frames between ghosts shown (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    show_only_selected: bool
    """ For Pose-Mode drawing, only draw ghosts for selected bones

    :type: bool
    """

    type: str
    """ Method used for determining what ghosts get drawn

    :type: str
    """

class AnyType(bpy_struct):
    """RNA type used for pointers to any possible data"""

    ...

class Area(bpy_struct):
    """Area in a subdivided screen, containing an editor"""

    height: int
    """ Area height

    :type: int
    """

    regions: bpy_prop_collection[Region]
    """ Regions this area is subdivided in

    :type: bpy_prop_collection[Region]
    """

    show_menus: bool
    """ Show menus in the header

    :type: bool
    """

    spaces: AreaSpaces
    """ Spaces contained in this area, the first being the active space (NOTE: Useful for example to restore a previously used 3D view space in a certain area to get the old view orientation)

    :type: AreaSpaces
    """

    type: str
    """ Current editor type for this area

    :type: str
    """

    width: int
    """ Area width

    :type: int
    """

    x: int
    """ The window relative vertical location of the area

    :type: int
    """

    y: int
    """ The window relative horizontal location of the area

    :type: int
    """

    def tag_redraw(self):
        """tag_redraw"""
        ...

    def header_text_set(self, text: str | typing.Any = ""):
        """Set the header text

        :param text: Text, New string for the header, no argument clears the text
        :type text: str | typing.Any
        """
        ...

class AreaSpaces(bpy_prop_collection[Space], bpy_struct):
    """Collection of spaces"""

    active: Space
    """ Space currently being displayed in this area

    :type: Space
    """

class ArmatureBones(bpy_prop_collection[Bone], bpy_struct):
    """Collection of armature bones"""

    active: Bone | None
    """ Armature's active bone

    :type: Bone | None
    """

class ArmatureEditBones(bpy_prop_collection[EditBone], bpy_struct):
    """Collection of armature edit bones"""

    active: EditBone | None
    """ Armatures active edit bone

    :type: EditBone | None
    """

    def new(self, name: str | typing.Any) -> EditBone:
        """Add a new bone

        :param name: New name for the bone
        :type name: str | typing.Any
        :return: Newly created edit bone
        :rtype: EditBone
        """
        ...

    def remove(self, bone: EditBone):
        """Remove an existing bone from the armature

        :param bone: EditBone to remove
        :type bone: EditBone
        """
        ...

class BackgroundImage(bpy_struct):
    """Image and settings for display in the 3D View background"""

    clip: MovieClip
    """ Movie clip displayed and edited in this space

    :type: MovieClip
    """

    clip_user: MovieClipUser
    """ Parameters defining which frame of the movie clip is displayed

    :type: MovieClipUser
    """

    draw_depth: str
    """ Draw under or over everything

    :type: str
    """

    frame_method: str
    """ How the image fits in the camera frame

    :type: str
    """

    image: Image
    """ Image displayed and edited in this space

    :type: Image
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    offset_x: float
    """ Offset image horizontally from the world origin

    :type: float
    """

    offset_y: float
    """ Offset image vertically from the world origin

    :type: float
    """

    opacity: float
    """ Image opacity to blend the image against the background color

    :type: float
    """

    rotation: float
    """ Rotation for the background image (ortho view only)

    :type: float
    """

    show_background_image: bool
    """ Show this image as background

    :type: bool
    """

    show_expanded: bool
    """ Show the expanded in the user interface

    :type: bool
    """

    show_on_foreground: bool
    """ Show this image in front of objects in viewport

    :type: bool
    """

    size: float
    """ Size of the background image (ortho view only)

    :type: float
    """

    source: str
    """ Data source used for background

    :type: str
    """

    use_camera_clip: bool
    """ Use movie clip from active scene camera

    :type: bool
    """

    use_flip_x: bool
    """ Flip the background image horizontally

    :type: bool
    """

    use_flip_y: bool
    """ Flip the background image vertically

    :type: bool
    """

    view_axis: str
    """ The axis to display the image on

    :type: str
    """

class BackgroundImages(bpy_prop_collection[BackgroundImage], bpy_struct):
    """Collection of background images"""

    def new(self) -> BackgroundImage:
        """Add new background image

        :return: Image displayed as viewport background
        :rtype: BackgroundImage
        """
        ...

    def remove(self, image: BackgroundImage):
        """Remove background image

        :param image: Image displayed as viewport background
        :type image: BackgroundImage
        """
        ...

    def clear(self):
        """Remove all background images"""
        ...

class BakePixel(bpy_struct):
    du_dx: float
    """ 

    :type: float
    """

    du_dy: float
    """ 

    :type: float
    """

    dv_dx: float
    """ 

    :type: float
    """

    dv_dy: float
    """ 

    :type: float
    """

    next: BakePixel
    """ 

    :type: BakePixel
    """

    object_id: int
    """ 

    :type: int
    """

    primitive_id: int
    """ 

    :type: int
    """

    uv: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class BakeSettings(bpy_struct):
    """Bake data for a Scene data-block"""

    cage_extrusion: float
    """ Distance to use for the inward ray cast when using selected to active

    :type: float
    """

    cage_object: str
    """ Object to use as cage instead of calculating the cage from the active object with cage extrusion

    :type: str
    """

    filepath: str
    """ Image filepath to use when saving externally

    :type: str
    """

    height: int
    """ Vertical dimension of the baking map

    :type: int
    """

    image_settings: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    margin: int
    """ Extends the baked result as a post process filter

    :type: int
    """

    normal_b: str
    """ Axis to bake in blue channel

    :type: str
    """

    normal_g: str
    """ Axis to bake in green channel

    :type: str
    """

    normal_r: str
    """ Axis to bake in red channel

    :type: str
    """

    normal_space: str
    """ Choose normal space for baking

    :type: str
    """

    pass_filter: set[str]
    """ Passes to include in the active baking pass

    :type: set[str]
    """

    save_mode: str
    """ Choose how to save the baking map

    :type: str
    """

    use_automatic_name: bool
    """ Automatically name the output file with the pass type (external only)

    :type: bool
    """

    use_cage: bool
    """ Cast rays to active object from a cage

    :type: bool
    """

    use_clear: bool
    """ Clear Images before baking (internal only)

    :type: bool
    """

    use_pass_ambient_occlusion: bool
    """ Add ambient occlusion contribution

    :type: bool
    """

    use_pass_color: bool
    """ Color the pass

    :type: bool
    """

    use_pass_diffuse: bool
    """ Add diffuse contribution

    :type: bool
    """

    use_pass_direct: bool
    """ Add direct lighting contribution

    :type: bool
    """

    use_pass_emit: bool
    """ Add emission contribution

    :type: bool
    """

    use_pass_glossy: bool
    """ Add glossy contribution

    :type: bool
    """

    use_pass_indirect: bool
    """ Add indirect lighting contribution

    :type: bool
    """

    use_pass_subsurface: bool
    """ Add subsurface contribution

    :type: bool
    """

    use_pass_transmission: bool
    """ Add transmission contribution

    :type: bool
    """

    use_selected_to_active: bool
    """ Bake shading on the surface of selected objects to the active object

    :type: bool
    """

    use_split_materials: bool
    """ Split external images per material (external only)

    :type: bool
    """

    width: int
    """ Horizontal dimension of the baking map

    :type: int
    """

class BezierSplinePoint(bpy_struct):
    """Bezier curve point with two handles"""

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the first handle

    :type: bpy_prop_array[float]
    """

    handle_left_type: str
    """ Handle types

    :type: str
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the second handle

    :type: bpy_prop_array[float]
    """

    handle_right_type: str
    """ Handle types

    :type: str
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Handle 1 selection status

    :type: bool
    """

    select_right_handle: bool
    """ Handle 2 selection status

    :type: bool
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

class BlendData(bpy_struct):
    """Main data structure representing a .blend file and all its datablocks"""

    actions: BlendDataActions
    """ Action datablocks

    :type: BlendDataActions
    """

    armatures: BlendDataArmatures
    """ Armature datablocks

    :type: BlendDataArmatures
    """

    brushes: BlendDataBrushes
    """ Brush datablocks

    :type: BlendDataBrushes
    """

    cache_files: BlendDataCacheFiles
    """ Cache Files datablocks

    :type: BlendDataCacheFiles
    """

    cameras: BlendDataCameras
    """ Camera datablocks

    :type: BlendDataCameras
    """

    curves: BlendDataCurves
    """ Curve datablocks

    :type: BlendDataCurves
    """

    filepath: str
    """ Path to the .blend file

    :type: str
    """

    fonts: BlendDataFonts
    """ Vector font datablocks

    :type: BlendDataFonts
    """

    grease_pencil: BlendDataGreasePencils
    """ Grease Pencil datablocks

    :type: BlendDataGreasePencils
    """

    groups: BlendDataGroups
    """ Group datablocks

    :type: BlendDataGroups
    """

    images: BlendDataImages
    """ Image datablocks

    :type: BlendDataImages
    """

    is_dirty: bool
    """ Have recent edits been saved to disk

    :type: bool
    """

    is_saved: bool
    """ Has the current session been saved to disk as a .blend file

    :type: bool
    """

    lamps: BlendDataLamps
    """ Lamp datablocks

    :type: BlendDataLamps
    """

    lattices: BlendDataLattices
    """ Lattice datablocks

    :type: BlendDataLattices
    """

    libraries: BlendDataLibraries
    """ Library datablocks

    :type: BlendDataLibraries
    """

    linestyles: BlendDataLineStyles
    """ Line Style datablocks

    :type: BlendDataLineStyles
    """

    masks: BlendDataMasks
    """ Masks datablocks

    :type: BlendDataMasks
    """

    materials: BlendDataMaterials
    """ Material datablocks

    :type: BlendDataMaterials
    """

    meshes: BlendDataMeshes
    """ Mesh datablocks

    :type: BlendDataMeshes
    """

    metaballs: BlendDataMetaBalls
    """ Metaball datablocks

    :type: BlendDataMetaBalls
    """

    movieclips: BlendDataMovieClips
    """ Movie Clip datablocks

    :type: BlendDataMovieClips
    """

    node_groups: BlendDataNodeTrees
    """ Node group datablocks

    :type: BlendDataNodeTrees
    """

    objects: BlendDataObjects
    """ Object datablocks

    :type: BlendDataObjects
    """

    paint_curves: BlendDataPaintCurves
    """ Paint Curves datablocks

    :type: BlendDataPaintCurves
    """

    palettes: BlendDataPalettes
    """ Palette datablocks

    :type: BlendDataPalettes
    """

    particles: BlendDataParticles
    """ Particle datablocks

    :type: BlendDataParticles
    """

    scenes: BlendDataScenes
    """ Scene datablocks

    :type: BlendDataScenes
    """

    screens: BlendDataScreens
    """ Screen datablocks

    :type: BlendDataScreens
    """

    shape_keys: bpy_prop_collection[Key]
    """ Shape Key datablocks

    :type: bpy_prop_collection[Key]
    """

    sounds: BlendDataSounds
    """ Sound datablocks

    :type: BlendDataSounds
    """

    speakers: BlendDataSpeakers
    """ Speaker datablocks

    :type: BlendDataSpeakers
    """

    texts: BlendDataTexts
    """ Text datablocks

    :type: BlendDataTexts
    """

    textures: BlendDataTextures
    """ Texture datablocks

    :type: BlendDataTextures
    """

    use_autopack: bool
    """ Automatically pack all external data into .blend file

    :type: bool
    """

    version: bpy_prop_array[int]
    """ Version of Blender the .blend was saved with

    :type: bpy_prop_array[int]
    """

    window_managers: BlendDataWindowManagers
    """ Window manager datablocks

    :type: BlendDataWindowManagers
    """

    worlds: BlendDataWorlds
    """ World datablocks

    :type: BlendDataWorlds
    """

    def user_map(
        self,
        subset: collections.abc.Sequence | None,
        key_types: set[str] | None,
        value_types: set[str] | None,
    ) -> dict:
        """Returns a mapping of all ID datablocks in current bpy.data to a set of all datablocks using them.For list of valid set members for key_types & value_types, see: `bpy.types.KeyingSetPath.id_type`.

        :param subset: When passed, only these data-blocks and their users will be included as keys/values in the map.
        :type subset: collections.abc.Sequence | None
        :param key_types: Filter the keys mapped by ID types.
        :type key_types: set[str] | None
        :param value_types: Filter the values in the set by ID types.
        :type value_types: set[str] | None
        :return: dictionary of `bpy.types.ID` instances, with sets of ID's as their values.
        :rtype: dict
        """
        ...

class BlendDataActions(bpy_prop_collection[Action], bpy_struct):
    """Collection of actions"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Action:
        """Add a new action to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New action data-block
        :rtype: Action
        """
        ...

    def remove(self, action: Action, do_unlink: bool | typing.Any | None = False):
        """Remove a action from the current blendfile

        :param action: Action to remove
        :type action: Action
        :param do_unlink: Unlink all usages of this action before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataArmatures(bpy_prop_collection[Armature], bpy_struct):
    """Collection of armatures"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Armature:
        """Add a new armature to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New armature data-block
        :rtype: Armature
        """
        ...

    def remove(self, armature: Armature, do_unlink: bool | typing.Any | None = False):
        """Remove a armature from the current blendfile

        :param armature: Armature to remove
        :type armature: Armature
        :param do_unlink: Unlink all usages of this armature before deleting it (WARNING: will also delete objects instancing that armature data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataBrushes(bpy_prop_collection[Brush], bpy_struct):
    """Collection of brushes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any, mode: str | None = "TEXTURE_PAINT") -> Brush:
        """Add a new brush to the main database

                :param name: New name for the data-block
                :type name: str | typing.Any
                :param mode: Paint Mode for the new brush

        OBJECT Object Mode.

        EDIT Edit Mode.

        POSE Pose Mode.

        SCULPT Sculpt Mode.

        VERTEX_PAINT Vertex Paint.

        WEIGHT_PAINT Weight Paint.

        TEXTURE_PAINT Texture Paint.

        PARTICLE_EDIT Particle Edit.

        GPENCIL_EDIT Edit Strokes, Edit Grease Pencil Strokes.
                :type mode: str | None
                :return: New brush data-block
                :rtype: Brush
        """
        ...

    def remove(self, brush: Brush, do_unlink: bool | typing.Any | None = False):
        """Remove a brush from the current blendfile

        :param brush: Brush to remove
        :type brush: Brush
        :param do_unlink: Unlink all usages of this brush before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataCacheFiles(bpy_prop_collection[CacheFile], bpy_struct):
    """Collection of cache files"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataCameras(bpy_prop_collection[Camera], bpy_struct):
    """Collection of cameras"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Camera:
        """Add a new camera to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New camera data-block
        :rtype: Camera
        """
        ...

    def remove(self, camera: Camera, do_unlink: bool | typing.Any | None = False):
        """Remove a camera from the current blendfile

        :param camera: Camera to remove
        :type camera: Camera
        :param do_unlink: Unlink all usages of this camera before deleting it (WARNING: will also delete objects instancing that camera data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataCurves(bpy_prop_collection[Curve], bpy_struct):
    """Collection of curves"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any, type: str | None) -> Curve:
        """Add a new curve to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param type: Type, The type of curve to add
        :type type: str | None
        :return: New curve data-block
        :rtype: Curve
        """
        ...

    def remove(self, curve: Curve, do_unlink: bool | typing.Any | None = False):
        """Remove a curve from the current blendfile

        :param curve: Curve to remove
        :type curve: Curve
        :param do_unlink: Unlink all usages of this curve before deleting it (WARNING: will also delete objects instancing that curve data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataFonts(bpy_prop_collection[VectorFont], bpy_struct):
    """Collection of fonts"""

    is_updated: bool
    """ 

    :type: bool
    """

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> VectorFont:
        """Load a new font into the main database

        :param filepath: path of the font to load
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New font data-block
        :rtype: VectorFont
        """
        ...

    def remove(self, vfont: VectorFont, do_unlink: bool | typing.Any | None = False):
        """Remove a font from the current blendfile

        :param vfont: Font to remove
        :type vfont: VectorFont
        :param do_unlink: Unlink all usages of this font before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataGreasePencils(bpy_prop_collection[GreasePencil], bpy_struct):
    """Collection of grease pencils"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    @classmethod
    def new(cls, name: str | typing.Any) -> GreasePencil:
        """new

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New grease pencil data-block
        :rtype: GreasePencil
        """
        ...

    def remove(
        self, grease_pencil: GreasePencil, do_unlink: bool | typing.Any | None = False
    ):
        """Remove a grease pencil instance from the current blendfile

        :param grease_pencil: Grease Pencil to remove
        :type grease_pencil: GreasePencil
        :param do_unlink: Unlink all usages of this grease pencil before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

class BlendDataGroups(bpy_prop_collection[Group], bpy_struct):
    """Collection of groups"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Group:
        """Add a new group to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New group data-block
        :rtype: Group
        """
        ...

    def remove(self, group: Group, do_unlink: bool | typing.Any | None = False):
        """Remove a group from the current blendfile

        :param group: Group to remove
        :type group: Group
        :param do_unlink: Unlink all usages of this group before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataImages(bpy_prop_collection[Image], bpy_struct):
    """Collection of images"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(
        self,
        name: str | typing.Any,
        width: int | None,
        height: int | None,
        alpha: bool | typing.Any | None = False,
        float_buffer: bool | typing.Any | None = False,
        stereo3d: bool | typing.Any | None = False,
    ) -> Image:
        """Add a new image to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param width: Width of the image
        :type width: int | None
        :param height: Height of the image
        :type height: int | None
        :param alpha: Alpha, Use alpha channel
        :type alpha: bool | typing.Any | None
        :param float_buffer: Float Buffer, Create an image with floating point color
        :type float_buffer: bool | typing.Any | None
        :param stereo3d: Stereo 3D, Create left and right views
        :type stereo3d: bool | typing.Any | None
        :return: New image data-block
        :rtype: Image
        """
        ...

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> Image:
        """Load a new image into the main database

        :param filepath: path of the file to load
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New image data-block
        :rtype: Image
        """
        ...

    def remove(self, image: Image, do_unlink: bool | typing.Any | None = False):
        """Remove an image from the current blendfile

        :param image: Image to remove
        :type image: Image
        :param do_unlink: Unlink all usages of this image before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataLamps(bpy_prop_collection[Lamp], bpy_struct):
    """Collection of lamps"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any, type: str | None) -> Lamp:
        """Add a new lamp to the main database

                :param name: New name for the data-block
                :type name: str | typing.Any
                :param type: Type, The type of texture to add

        POINT Point, Omnidirectional point light source.

        SUN Sun, Constant direction parallel ray light source.

        SPOT Spot, Directional cone light source.

        HEMI Hemi, 180 degree constant light source.

        AREA Area, Directional area light source.
                :type type: str | None
                :return: New lamp data-block
                :rtype: Lamp
        """
        ...

    def remove(self, lamp: Lamp, do_unlink: bool | typing.Any | None = False):
        """Remove a lamp from the current blendfile

        :param lamp: Lamp to remove
        :type lamp: Lamp
        :param do_unlink: Unlink all usages of this lamp before deleting it (WARNING: will also delete objects instancing that lamp data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataLattices(bpy_prop_collection[Lattice], bpy_struct):
    """Collection of lattices"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Lattice:
        """Add a new lattice to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New lattices data-block
        :rtype: Lattice
        """
        ...

    def remove(self, lattice: Lattice, do_unlink: bool | typing.Any | None = False):
        """Remove a lattice from the current blendfile

        :param lattice: Lattice to remove
        :type lattice: Lattice
        :param do_unlink: Unlink all usages of this lattice before deleting it (WARNING: will also delete objects instancing that lattice data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataLibraries(bpy_prop_collection[Library], bpy_struct):
    """Collection of libraries"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def load(
        self,
        filepath: str | None,
        link: bool | None = False,
        relative: bool | None = False,
    ):
        """Returns a context manager which exposes 2 library objects on entering.
        Each object has attributes matching bpy.data which are lists of strings to be linked.

                :param filepath: The path to a blend file.
                :type filepath: str | None
                :param link: When False reference to the original file is lost.
                :type link: bool | None
                :param relative: When True the path is stored relative to the open blend file.
                :type relative: bool | None
        """
        ...

    def write(
        self,
        filepath: str | None,
        datablocks: set | None,
        relative_remap: bool | None = False,
        fake_user: bool | None = False,
        compress: bool | None = False,
    ):
        """Write data-blocks into a blend file.

        :param filepath: The path to write the blend-file.
        :type filepath: str | None
        :param datablocks: set of data-blocks (`bpy.types.ID` instances).
        :type datablocks: set | None
        :param relative_remap: When True, remap the paths relative to the current blend-file.
        :type relative_remap: bool | None
        :param fake_user: When True, data-blocks will be written with fake-user flag enabled.
        :type fake_user: bool | None
        :param compress: When True, write a compressed blend file.
        :type compress: bool | None
        """
        ...

class BlendDataLineStyles(bpy_prop_collection[FreestyleLineStyle], bpy_struct):
    """Collection of line styles"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def new(self, name: str | typing.Any) -> FreestyleLineStyle:
        """Add a new line style instance to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New line style data-block
        :rtype: FreestyleLineStyle
        """
        ...

    def remove(
        self, linestyle: FreestyleLineStyle, do_unlink: bool | typing.Any | None = False
    ):
        """Remove a line style instance from the current blendfile

        :param linestyle: Line style to remove
        :type linestyle: FreestyleLineStyle
        :param do_unlink: Unlink all usages of this line style before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

class BlendDataMasks(bpy_prop_collection[Mask], bpy_struct):
    """Collection of masks"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def new(self, name: str | typing.Any = "") -> Mask:
        """Add a new mask with a given name to the main database

        :param name: Mask, Name of new mask data-block
        :type name: str | typing.Any
        :return: New mask data-block
        :rtype: Mask
        """
        ...

    def remove(self, mask: Mask, do_unlink: bool | typing.Any | None = False):
        """Remove a masks from the current blendfile.

        :param mask: Mask to remove
        :type mask: Mask
        :param do_unlink: Unlink all usages of this mask before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

class BlendDataMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Material:
        """Add a new material to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New material data-block
        :rtype: Material
        """
        ...

    def remove(self, material: Material, do_unlink: bool | typing.Any | None = False):
        """Remove a material from the current blendfile

        :param material: Material to remove
        :type material: Material
        :param do_unlink: Unlink all usages of this material before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataMeshes(bpy_prop_collection[Mesh], bpy_struct):
    """Collection of meshes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Mesh:
        """Add a new mesh to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New mesh data-block
        :rtype: Mesh
        """
        ...

    def new_from_object(
        self,
        scene: Scene,
        object: Object,
        apply_modifiers: bool | None,
        settings: str | None,
        calc_tessface: bool | typing.Any | None = True,
        calc_undeformed: bool | typing.Any | None = False,
    ) -> Mesh:
        """Add a new mesh created from object with modifiers applied

                :param scene: Scene within which to evaluate modifiers
                :type scene: Scene
                :param object: Object to create mesh from
                :type object: Object
                :param apply_modifiers: Apply modifiers
                :type apply_modifiers: bool | None
                :param settings: Modifier settings to apply

        PREVIEW Preview, Apply modifier preview settings.

        RENDER Render, Apply modifier render settings.
                :type settings: str | None
                :param calc_tessface: Calculate Tessellation, Calculate tessellation faces
                :type calc_tessface: bool | typing.Any | None
                :param calc_undeformed: Calculate Undeformed, Calculate undeformed vertex coordinates
                :type calc_undeformed: bool | typing.Any | None
                :return: Mesh created from object, remove it if it is only used for export
                :rtype: Mesh
        """
        ...

    def remove(self, mesh: Mesh, do_unlink: bool | typing.Any | None = False):
        """Remove a mesh from the current blendfile

        :param mesh: Mesh to remove
        :type mesh: Mesh
        :param do_unlink: Unlink all usages of this mesh before deleting it (WARNING: will also delete objects instancing that mesh data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataMetaBalls(bpy_prop_collection[MetaBall], bpy_struct):
    """Collection of metaballs"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> MetaBall:
        """Add a new metaball to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New metaball data-block
        :rtype: MetaBall
        """
        ...

    def remove(self, metaball: MetaBall, do_unlink: bool | typing.Any | None = False):
        """Remove a metaball from the current blendfile

        :param metaball: Metaball to remove
        :type metaball: MetaBall
        :param do_unlink: Unlink all usages of this metaball before deleting it (WARNING: will also delete objects instancing that metaball data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataMovieClips(bpy_prop_collection[MovieClip], bpy_struct):
    """Collection of movie clips"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

    def remove(self, clip: MovieClip, do_unlink: bool | typing.Any | None = False):
        """Remove a movie clip from the current blendfile.

        :param clip: Movie clip to remove
        :type clip: MovieClip
        :param do_unlink: Unlink all usages of this movie clip before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> MovieClip:
        """Add a new movie clip to the main database from a file (while check_existing is disabled for consistency with other load functions, behavior with multiple movie-clips using the same file may incorrectly generate proxies)

        :param filepath: path for the data-block
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New movie clip data-block
        :rtype: MovieClip
        """
        ...

class BlendDataNodeTrees(bpy_prop_collection[NodeTree], bpy_struct):
    """Collection of node trees"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any, type: str | None) -> NodeTree:
        """Add a new node tree to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param type: Type, The type of node_group to add
        :type type: str | None
        :return: New node tree data-block
        :rtype: NodeTree
        """
        ...

    def remove(self, tree: NodeTree, do_unlink: bool | typing.Any | None = False):
        """Remove a node tree from the current blendfile

        :param tree: Node tree to remove
        :type tree: NodeTree
        :param do_unlink: Unlink all usages of this node tree before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of objects"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any, object_data: ID | None) -> Object:
        """Add a new object to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :param object_data: Object data or None for an empty object
        :type object_data: ID | None
        :return: New object data-block
        :rtype: Object
        """
        ...

    def remove(self, object: Object, do_unlink: bool | typing.Any | None = False):
        """Remove a object from the current blendfile

        :param object: Object to remove
        :type object: Object
        :param do_unlink: Unlink all usages of this object before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataPaintCurves(bpy_prop_collection[PaintCurve], bpy_struct):
    """Collection of paint curves"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataPalettes(bpy_prop_collection[Palette], bpy_struct):
    """Collection of palettes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Palette:
        """Add a new palette to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New palette data-block
        :rtype: Palette
        """
        ...

    def remove(self, palette: Palette, do_unlink: bool | typing.Any | None = False):
        """Remove a palette from the current blendfile

        :param palette: Palette to remove
        :type palette: Palette
        :param do_unlink: Unlink all usages of this palette before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataParticles(bpy_prop_collection[ParticleSettings], bpy_struct):
    """Collection of particle settings"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> ParticleSettings:
        """Add a new particle settings instance to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New particle settings data-block
        :rtype: ParticleSettings
        """
        ...

    def remove(
        self, particle: ParticleSettings, do_unlink: bool | typing.Any | None = False
    ):
        """Remove a particle settings instance from the current blendfile

        :param particle: Particle Settings to remove
        :type particle: ParticleSettings
        :param do_unlink: Unlink all usages of those particle settings before deleting them
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataScenes(bpy_prop_collection[Scene], bpy_struct):
    """Collection of scenes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Scene:
        """Add a new scene to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New scene data-block
        :rtype: Scene
        """
        ...

    def remove(self, scene: Scene, do_unlink: bool | typing.Any | None = False):
        """Remove a scene from the current blendfile

        :param scene: Scene to remove
        :type scene: Scene
        :param do_unlink: Unlink all usages of this scene before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataScreens(bpy_prop_collection[Screen], bpy_struct):
    """Collection of screens"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataSounds(bpy_prop_collection[Sound], bpy_struct):
    """Collection of sounds"""

    is_updated: bool
    """ 

    :type: bool
    """

    def load(
        self,
        filepath: str | typing.Any,
        check_existing: bool | typing.Any | None = False,
    ) -> Sound:
        """Add a new sound to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str | typing.Any
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | typing.Any | None
        :return: New text data-block
        :rtype: Sound
        """
        ...

    def remove(self, sound: Sound, do_unlink: bool | typing.Any | None = False):
        """Remove a sound from the current blendfile

        :param sound: Sound to remove
        :type sound: Sound
        :param do_unlink: Unlink all usages of this sound before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataSpeakers(bpy_prop_collection[Speaker], bpy_struct):
    """Collection of speakers"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Speaker:
        """Add a new speaker to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New speaker data-block
        :rtype: Speaker
        """
        ...

    def remove(self, speaker: Speaker, do_unlink: bool | typing.Any | None = False):
        """Remove a speaker from the current blendfile

        :param speaker: Speaker to remove
        :type speaker: Speaker
        :param do_unlink: Unlink all usages of this speaker before deleting it (WARNING: will also delete objects instancing that speaker data)
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataTexts(bpy_prop_collection[Text], bpy_struct):
    """Collection of texts"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> Text:
        """Add a new text to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New text data-block
        :rtype: Text
        """
        ...

    def remove(self, text: Text, do_unlink: bool | typing.Any | None = False):
        """Remove a text from the current blendfile

        :param text: Text to remove
        :type text: Text
        :param do_unlink: Unlink all usages of this text before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def load(
        self, filepath: str | typing.Any, internal: bool | typing.Any | None = False
    ) -> Text:
        """Add a new text to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str | typing.Any
        :param internal: Make internal, Make text file internal after loading
        :type internal: bool | typing.Any | None
        :return: New text data-block
        :rtype: Text
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataTextures(bpy_prop_collection[Texture], bpy_struct):
    """Collection of groups"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any, type: str | None) -> Texture:
        """Add a new texture to the main database

                :param name: New name for the data-block
                :type name: str | typing.Any
                :param type: Type, The type of texture to add

        NONE None.

        BLEND Blend, Procedural - create a ramp texture.

        CLOUDS Clouds, Procedural - create a cloud-like fractal noise texture.

        DISTORTED_NOISE Distorted Noise, Procedural - noise texture distorted by two noise algorithms.

        ENVIRONMENT_MAP Environment Map, Create a render of the environment mapped to a texture.

        IMAGE Image or Movie, Allow for images or movies to be used as textures.

        MAGIC Magic, Procedural - color texture based on trigonometric functions.

        MARBLE Marble, Procedural - marble-like noise texture with wave generated bands.

        MUSGRAVE Musgrave, Procedural - highly flexible fractal noise texture.

        NOISE Noise, Procedural - random noise, gives a different result every time, for every frame, for every pixel.

        OCEAN Ocean, Use a texture generated by an Ocean modifier.

        POINT_DENSITY Point Density.

        STUCCI Stucci, Procedural - create a fractal noise texture.

        VORONOI Voronoi, Procedural - create cell-like patterns based on Worley noise.

        VOXEL_DATA Voxel Data, Create a 3D texture based on volumetric data.

        WOOD Wood, Procedural - wave generated bands or rings, with optional noise.
                :type type: str | None
                :return: New texture data-block
                :rtype: Texture
        """
        ...

    def remove(self, texture: Texture, do_unlink: bool | typing.Any | None = False):
        """Remove a texture from the current blendfile

        :param texture: Texture to remove
        :type texture: Texture
        :param do_unlink: Unlink all usages of this texture before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataWindowManagers(bpy_prop_collection[WindowManager], bpy_struct):
    """Collection of window managers"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlendDataWorlds(bpy_prop_collection[World], bpy_struct):
    """Collection of worlds"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str | typing.Any) -> World:
        """Add a new world to the main database

        :param name: New name for the data-block
        :type name: str | typing.Any
        :return: New world data-block
        :rtype: World
        """
        ...

    def remove(self, world: World, do_unlink: bool | typing.Any | None = False):
        """Remove a world from the current blendfile

        :param world: World to remove
        :type world: World
        :param do_unlink: Unlink all usages of this world before deleting it
        :type do_unlink: bool | typing.Any | None
        """
        ...

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """
        ...

class BlenderRNA(bpy_struct):
    """Blender RNA structure definitions"""

    structs: bpy_prop_collection[Struct]
    """ 

    :type: bpy_prop_collection[Struct]
    """

class BoidRule(bpy_struct):
    name: str
    """ Boid rule name

    :type: str
    """

    type: str
    """ 

    :type: str
    """

    use_in_air: bool
    """ Use rule when boid is flying

    :type: bool
    """

    use_on_land: bool
    """ Use rule when boid is on land

    :type: bool
    """

class BoidSettings(bpy_struct):
    """Settings for boid physics"""

    accuracy: float
    """ Accuracy of attack

    :type: float
    """

    active_boid_state: BoidRule
    """ 

    :type: BoidRule
    """

    active_boid_state_index: int | None
    """ 

    :type: int | None
    """

    aggression: float
    """ Boid will fight this times stronger enemy

    :type: float
    """

    air_acc_max: float
    """ Maximum acceleration in air (relative to maximum speed)

    :type: float
    """

    air_ave_max: float
    """ Maximum angular velocity in air (relative to 180 degrees)

    :type: float
    """

    air_personal_space: float
    """ Radius of boids personal space in air (% of particle size)

    :type: float
    """

    air_speed_max: float
    """ Maximum speed in air

    :type: float
    """

    air_speed_min: float
    """ Minimum speed in air (relative to maximum speed)

    :type: float
    """

    bank: float
    """ Amount of rotation around velocity vector on turns

    :type: float
    """

    health: float
    """ Initial boid health when born

    :type: float
    """

    height: float
    """ Boid height relative to particle size

    :type: float
    """

    land_acc_max: float
    """ Maximum acceleration on land (relative to maximum speed)

    :type: float
    """

    land_ave_max: float
    """ Maximum angular velocity on land (relative to 180 degrees)

    :type: float
    """

    land_jump_speed: float
    """ Maximum speed for jumping

    :type: float
    """

    land_personal_space: float
    """ Radius of boids personal space on land (% of particle size)

    :type: float
    """

    land_smooth: float
    """ How smoothly the boids land

    :type: float
    """

    land_speed_max: float
    """ Maximum speed on land

    :type: float
    """

    land_stick_force: float
    """ How strong a force must be to start effecting a boid on land

    :type: float
    """

    pitch: float
    """ Amount of rotation around side vector

    :type: float
    """

    range: float
    """ Maximum distance from which a boid can attack

    :type: float
    """

    states: bpy_prop_collection[BoidState]
    """ 

    :type: bpy_prop_collection[BoidState]
    """

    strength: float
    """ Maximum caused damage on attack per second

    :type: float
    """

    use_climb: bool
    """ Allow boids to climb goal objects

    :type: bool
    """

    use_flight: bool
    """ Allow boids to move in air

    :type: bool
    """

    use_land: bool
    """ Allow boids to move on land

    :type: bool
    """

class BoidState(bpy_struct):
    """Boid state for boid physics"""

    active_boid_rule: BoidRule
    """ 

    :type: BoidRule
    """

    active_boid_rule_index: int | None
    """ 

    :type: int | None
    """

    falloff: float
    """ 

    :type: float
    """

    name: str
    """ Boid state name

    :type: str
    """

    rule_fuzzy: float
    """ 

    :type: float
    """

    rules: bpy_prop_collection[BoidRule]
    """ 

    :type: bpy_prop_collection[BoidRule]
    """

    ruleset_type: str
    """ How the rules in the list are evaluated

    :type: str
    """

    volume: float
    """ 

    :type: float
    """

class Bone(bpy_struct):
    """Bone in an Armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveiny: float
    """ Y-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveouty: float
    """ Y-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_in: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_out: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: float
    """ Scale factor for start of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_scaleout: float
    """ Scale factor for end of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    children: bpy_prop_collection[Bone]
    """ Bones which are children of this bone

    :type: bpy_prop_collection[Bone]
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: bpy_prop_array[float]
    """ Location of head end of the bone relative to its parent

    :type: bpy_prop_array[float]
    """

    head_local: bpy_prop_array[float]
    """ Location of head end of the bone relative to armature

    :type: bpy_prop_array[float]
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when it is not in Edit Mode (i.e. in Object or Pose Modes)

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    layers: list[bool]
    """ Layers bone exists in

    :type: list[bool]
    """

    matrix: bpy_prop_array[float]
    """ 3x3 bone matrix

    :type: bpy_prop_array[float]
    """

    matrix_local: bpy_prop_array[float]
    """ 4x4 bone matrix relative to armature

    :type: bpy_prop_array[float]
    """

    name: str
    """ 

    :type: str
    """

    parent: Bone
    """ Parent bone (in same Armature)

    :type: Bone
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always drawn as Wireframe regardless of viewport draw mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: bpy_prop_array[float]
    """ Location of tail end of the bone

    :type: bpy_prop_array[float]
    """

    tail_local: bpy_prop_array[float]
    """ Location of tail end of the bone relative to armature

    :type: bpy_prop_array[float]
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Use Roll Out of parent bone as Roll In of its children

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_inherit_scale: bool
    """ Bone inherits scaling from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character
(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.
(readonly)"""

    children: typing.Any
    """ A list of all the bones children.
(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.
(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.
(readonly)"""

    length: typing.Any
    """ The distance from head to tail,
when set the head is moved to fit the length."""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent
(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)
(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.
(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.
(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.
(readonly)"""

    def evaluate_envelope(self, point: collections.abc.Iterable[float] | None) -> float:
        """Calculate bone envelope at given point

        :param point: Point, Position in 3d space to evaluate
        :type point: collections.abc.Iterable[float] | None
        :return: Factor, Envelope factor
        :rtype: float
        """
        ...

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """
        ...

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """
        ...

class BoneGroup(bpy_struct):
    """Groups of Pose Channels (Bones)"""

    color_set: str
    """ Custom color set to use

    :type: str
    """

    colors: ThemeBoneColorSet
    """ Copy of the colors associated with the group's color set

    :type: ThemeBoneColorSet
    """

    is_custom_color_set: bool
    """ Color set is user-defined instead of a fixed theme color set

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

class BoneGroups(bpy_prop_collection[BoneGroup], bpy_struct):
    """Collection of bone groups"""

    active: BoneGroup | None
    """ Active bone group for this pose

    :type: BoneGroup | None
    """

    active_index: int | None
    """ Active index in bone groups array

    :type: int | None
    """

    def new(self, name: str | typing.Any = "Group") -> BoneGroup:
        """Add a new bone group to the object

        :param name: Name of the new group
        :type name: str | typing.Any
        :return: New bone group
        :rtype: BoneGroup
        """
        ...

    def remove(self, group: BoneGroup):
        """Remove a bone group from this object

        :param group: Removed bone group
        :type group: BoneGroup
        """
        ...

class BrushCapabilities(bpy_struct):
    """Read-only indications of which brush operations are supported by the current brush"""

    has_overlay: bool
    """ 

    :type: bool
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_smooth_stroke: bool
    """ 

    :type: bool
    """

    has_spacing: bool
    """ 

    :type: bool
    """

class CameraStereoData(bpy_struct):
    """Stereoscopy settings for a Camera data-block"""

    convergence_distance: float
    """ The converge point for the stereo cameras (often the distance between a projector and the projection screen)

    :type: float
    """

    convergence_mode: str
    """ 

    :type: str
    """

    interocular_distance: float
    """ Set the distance between the eyes - the stereo plane distance / 30 should be fine

    :type: float
    """

    pivot: str
    """ 

    :type: str
    """

    pole_merge_angle_from: float
    """ Angle at which interocular distance starts to fade to 0

    :type: float
    """

    pole_merge_angle_to: float
    """ Angle at which interocular distance is 0

    :type: float
    """

    use_pole_merge: bool
    """ Fade interocular distance to 0 after the given cutoff angle

    :type: bool
    """

    use_spherical_stereo: bool
    """ Render every pixel rotating the camera around the middle of the interocular distance

    :type: bool
    """

class ChannelDriverVariables(bpy_prop_collection[DriverVariable], bpy_struct):
    """Collection of channel driver Variables"""

    def new(self) -> DriverVariable:
        """Add a new variable for the driver

        :return: Newly created Driver Variable
        :rtype: DriverVariable
        """
        ...

    def remove(self, variable: DriverVariable):
        """Remove an existing variable from the driver

        :param variable: Variable to remove from the driver
        :type variable: DriverVariable
        """
        ...

class ChildParticle(bpy_struct):
    """Child particle interpolated from simulated or edited particles"""

    ...

class ClothCollisionSettings(bpy_struct):
    """Cloth simulation settings for self collision and collision with other objects"""

    collision_quality: int
    """ How many collision iterations should be done. (higher is better quality but slower)

    :type: int
    """

    damping: float
    """ Amount of velocity lost on collision

    :type: float
    """

    distance_min: float
    """ Minimum distance between collision objects before collision response takes in

    :type: float
    """

    distance_repel: float
    """ Maximum distance to apply repulsion force, must be greater than minimum distance

    :type: float
    """

    friction: float
    """ Friction force if a collision happened (higher = less movement)

    :type: float
    """

    group: Group
    """ Limit colliders to this Group

    :type: Group
    """

    repel_force: float
    """ Repulsion force to apply on cloth when close to colliding

    :type: float
    """

    self_collision_quality: int
    """ How many self collision iterations should be done (higher is better quality but slower)

    :type: int
    """

    self_distance_min: float
    """ 0.5 means no distance at all, 1.0 is maximum distance

    :type: float
    """

    self_friction: float
    """ Friction/damping with self contact

    :type: float
    """

    use_collision: bool
    """ Enable collisions with other objects

    :type: bool
    """

    use_self_collision: bool
    """ Enable self collisions

    :type: bool
    """

    vertex_group_self_collisions: str
    """ Vertex group to define vertices which are not used during self collisions

    :type: str
    """

class ClothSettings(bpy_struct):
    """Cloth simulation settings for an object"""

    air_damping: float
    """ Air has normally some thickness which slows falling things down

    :type: float
    """

    bending_damping: float
    """ Damping of bending motion

    :type: float
    """

    bending_stiffness: float
    """ Wrinkle coefficient (higher = less smaller but more big wrinkles)

    :type: float
    """

    bending_stiffness_max: float
    """ Maximum bending stiffness value

    :type: float
    """

    collider_friction: float
    """ 

    :type: float
    """

    density_strength: float
    """ Influence of target density on the simulation

    :type: float
    """

    density_target: float
    """ Maximum density of hair

    :type: float
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    goal_default: float
    """ Default Goal (vertex target position) value, when no Vertex Group used

    :type: float
    """

    goal_friction: float
    """ Goal (vertex target position) friction

    :type: float
    """

    goal_max: float
    """ Goal maximum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_min: float
    """ Goal minimum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_spring: float
    """ Goal (vertex target position) spring stiffness

    :type: float
    """

    gravity: bpy_prop_array[float]
    """ Gravity or external force vector

    :type: bpy_prop_array[float]
    """

    internal_friction: float
    """ 

    :type: float
    """

    mass: float
    """ Mass of cloth material

    :type: float
    """

    pin_stiffness: float
    """ Pin (vertex target position) spring stiffness

    :type: float
    """

    quality: int
    """ Quality of the simulation in steps per frame (higher is better quality but slower)

    :type: int
    """

    rest_shape_key: ShapeKey
    """ Shape key to use the rest spring lengths from

    :type: ShapeKey
    """

    sewing_force_max: float
    """ Maximum sewing force

    :type: float
    """

    shrink_max: float
    """ Max amount to shrink cloth by

    :type: float
    """

    shrink_min: float
    """ Min amount to shrink cloth by

    :type: float
    """

    spring_damping: float
    """ Damping of cloth velocity (higher = more smooth, less jiggling)

    :type: float
    """

    structural_stiffness: float
    """ Overall stiffness of structure

    :type: float
    """

    structural_stiffness_max: float
    """ Maximum structural stiffness value

    :type: float
    """

    time_scale: float
    """ Cloth speed is multiplied by this value

    :type: float
    """

    use_dynamic_mesh: bool
    """ Make simulation respect deformations in the base mesh

    :type: bool
    """

    use_pin_cloth: bool
    """ Enable pinning of cloth vertices to other objects/positions

    :type: bool
    """

    use_sewing_springs: bool
    """ Pulls loose edges together

    :type: bool
    """

    use_stiffness_scale: bool
    """ If enabled, stiffness can be scaled along a weight painted vertex group

    :type: bool
    """

    vel_damping: float
    """ Damp velocity to help cloth reach the resting position faster (1.0 = no damping, 0.0 = fully dampened)

    :type: float
    """

    vertex_group_bending: str
    """ Vertex group for fine control over bending stiffness

    :type: str
    """

    vertex_group_mass: str
    """ Vertex Group for pinning of vertices

    :type: str
    """

    vertex_group_shrink: str
    """ Vertex Group for shrinking cloth

    :type: str
    """

    vertex_group_structural_stiffness: str
    """ Vertex group for fine control over structural stiffness

    :type: str
    """

    voxel_cell_size: float
    """ Size of the voxel grid cells for interaction effects

    :type: float
    """

class ClothSolverResult(bpy_struct):
    """Result of cloth solver iteration"""

    avg_error: float
    """ Average error during substeps

    :type: float
    """

    avg_iterations: float
    """ Average iterations during substeps

    :type: float
    """

    max_error: float
    """ Maximum error during substeps

    :type: float
    """

    max_iterations: int
    """ Maximum iterations during substeps

    :type: int
    """

    min_error: float
    """ Minimum error during substeps

    :type: float
    """

    min_iterations: int
    """ Minimum iterations during substeps

    :type: int
    """

    status: set[str]
    """ Status of the solver iteration

    :type: set[str]
    """

class CollisionSettings(bpy_struct):
    """Collision settings for object in physics simulation"""

    absorption: float
    """ How much of effector force gets lost during collision with this object (in percent)

    :type: float
    """

    damping: float
    """ Amount of damping during collision

    :type: float
    """

    damping_factor: float
    """ Amount of damping during particle collision

    :type: float
    """

    damping_random: float
    """ Random variation of damping

    :type: float
    """

    friction_factor: float
    """ Amount of friction during particle collision

    :type: float
    """

    friction_random: float
    """ Random variation of friction

    :type: float
    """

    permeability: float
    """ Chance that the particle will pass through the mesh

    :type: float
    """

    stickiness: float
    """ Amount of stickiness to surface collision

    :type: float
    """

    thickness_inner: float
    """ Inner face thickness (only used by softbodies)

    :type: float
    """

    thickness_outer: float
    """ Outer face thickness

    :type: float
    """

    use: bool
    """ Enable this objects as a collider for physics systems

    :type: bool
    """

    use_particle_kill: bool
    """ Kill collided particles

    :type: bool
    """

class ColorManagedDisplaySettings(bpy_struct):
    """Color management specific to display device"""

    display_device: str
    """ Display device name

    :type: str
    """

class ColorManagedInputColorspaceSettings(bpy_struct):
    """Input color space settings"""

    name: str
    """ Color space of the image or movie on disk

    :type: str
    """

class ColorManagedSequencerColorspaceSettings(bpy_struct):
    """Input color space settings"""

    name: str
    """ Color space that the sequencer operates in

    :type: str
    """

class ColorManagedViewSettings(bpy_struct):
    """Color management settings used for displaying images on the display"""

    curve_mapping: CurveMapping
    """ Color curve mapping applied before display transform

    :type: CurveMapping
    """

    exposure: float
    """ Exposure (stops) applied before display transform

    :type: float
    """

    gamma: float
    """ Amount of gamma modification applied after display transform

    :type: float
    """

    look: str
    """ Additional transform applied before view transform for an artistic needs

    :type: str
    """

    use_curve_mapping: bool
    """ Use RGB curved for pre-display transformation

    :type: bool
    """

    view_transform: str
    """ View used when converting image to a display space

    :type: str
    """

class ColorMapping(bpy_struct):
    """Color mapping settings"""

    blend_color: bpy_prop_array[float]
    """ Blend color to mix with texture output color

    :type: bpy_prop_array[float]
    """

    blend_factor: float
    """ 

    :type: float
    """

    blend_type: str
    """ Mode used to mix with texture output color

    :type: str
    """

    brightness: float
    """ Adjust the brightness of the texture

    :type: float
    """

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    contrast: float
    """ Adjust the contrast of the texture

    :type: float
    """

    saturation: float
    """ Adjust the saturation of colors in the texture

    :type: float
    """

    use_color_ramp: bool
    """ Toggle color ramp operations

    :type: bool
    """

class ColorRamp(bpy_struct):
    """Color ramp mapping a scalar value to a color"""

    color_mode: str
    """ Set color mode to use for interpolation

    :type: str
    """

    elements: ColorRampElements
    """ 

    :type: ColorRampElements
    """

    hue_interpolation: str
    """ Set color interpolation

    :type: str
    """

    interpolation: str
    """ Set interpolation between color stops

    :type: str
    """

    def evaluate(self, position: float | None) -> bpy_prop_array[float]:
        """Evaluate ColorRamp

        :param position: Position, Evaluate ColorRamp at position
        :type position: float | None
        :return: Color, Color at given position
        :rtype: bpy_prop_array[float]
        """
        ...

class ColorRampElement(bpy_struct):
    """Element defining a color at a position in the color ramp"""

    alpha: float
    """ Set alpha of selected color stop

    :type: float
    """

    color: bpy_prop_array[float]
    """ Set color of selected color stop

    :type: bpy_prop_array[float]
    """

    position: float
    """ Set position of selected color stop

    :type: float
    """

class ColorRampElements(bpy_prop_collection[ColorRampElement], bpy_struct):
    """Collection of Color Ramp Elements"""

    def new(self, position: float | None) -> ColorRampElement:
        """Add element to ColorRamp

        :param position: Position, Position to add element
        :type position: float | None
        :return: New element
        :rtype: ColorRampElement
        """
        ...

    def remove(self, element: ColorRampElement):
        """Delete element from ColorRamp

        :param element: Element to remove
        :type element: ColorRampElement
        """
        ...

class CompositorNodeOutputFileFileSlots(
    bpy_prop_collection[NodeOutputFileSlotFile], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str | typing.Any) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all file slots from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

class CompositorNodeOutputFileLayerSlots(
    bpy_prop_collection[NodeOutputFileSlotLayer], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str | typing.Any) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all file slots from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

class ConsoleLine(bpy_struct):
    """Input line for the interactive console"""

    body: str
    """ Text in the line

    :type: str
    """

    current_character: int
    """ 

    :type: int
    """

    type: str
    """ Console line type when used in scrollback

    :type: str
    """

class Constraint(bpy_struct):
    """Constraint modifying the transformation of objects and bones"""

    active: bool | None
    """ Constraint is the one being edited

    :type: bool | None
    """

    error_location: float
    """ Amount of residual error in Blender space unit for constraints that work on position

    :type: float
    """

    error_rotation: float
    """ Amount of residual error in radians for constraints that work on orientation

    :type: float
    """

    influence: float
    """ Amount of influence constraint will have on the final solution

    :type: float
    """

    is_proxy_local: bool
    """ Constraint was added in this proxy instance (i.e. did not belong to source Armature)

    :type: bool
    """

    is_valid: bool
    """ Constraint has valid settings and can be evaluated

    :type: bool
    """

    mute: bool
    """ Enable/Disable Constraint

    :type: bool
    """

    name: str
    """ Constraint name

    :type: str
    """

    owner_space: str
    """ Space that owner is evaluated in

    :type: str
    """

    show_expanded: bool
    """ Constraint's panel is expanded in UI

    :type: bool
    """

    target_space: str
    """ Space that target is evaluated in

    :type: str
    """

    type: str
    """ 

    :type: str
    """

class ConstraintTarget(bpy_struct):
    """Target object for multi-target constraints"""

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

class Context(bpy_struct):
    """Current windowmanager and data context"""

    area: Area
    """ 

    :type: Area
    """

    blend_data: BlendData
    """ 

    :type: BlendData
    """

    mode: str
    """ 

    :type: str
    """

    region: Region
    """ 

    :type: Region
    """

    region_data: RegionView3D
    """ 

    :type: RegionView3D
    """

    scene: Scene
    """ 

    :type: Scene
    """

    screen: Screen
    """ 

    :type: Screen
    """

    space_data: Space
    """ 

    :type: Space
    """

    tool_settings: ToolSettings
    """ 

    :type: ToolSettings
    """

    user_preferences: UserPreferences
    """ 

    :type: UserPreferences
    """

    window: Window
    """ 

    :type: Window
    """

    window_manager: WindowManager
    """ 

    :type: WindowManager
    """

    area: Area
    """ 

    :type: Area
    """

    blend_data: BlendData
    """ 

    :type: BlendData
    """

    mode: str
    """ 

    :type: str
    """

    region: Region
    """ 

    :type: Region
    """

    region_data: RegionView3D
    """ 

    :type: RegionView3D
    """

    scene: Scene
    """ 

    :type: Scene
    """

    screen: Screen
    """ 

    :type: Screen
    """

    space_data: Space
    """ 

    :type: Space
    """

    tool_settings: ToolSettings
    """ 

    :type: ToolSettings
    """

    user_preferences: UserPreferences
    """ 

    :type: UserPreferences
    """

    window: Window
    """ 

    :type: Window
    """

    window_manager: WindowManager
    """ 

    :type: WindowManager
    """

    scene: Scene
    """ 

    :type: Scene
    """

    visible_objects: list[Object]
    """ 

    :type: list[Object]
    """

    visible_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selectable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selectable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selected_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    editable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selected_editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_editable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    visible_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    visible_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    selected_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    active_bone: EditBone | None
    """ 

    :type: EditBone | None
    """

    active_pose_bone: PoseBone | None
    """ 

    :type: PoseBone | None
    """

    active_base: ObjectBase | None
    """ 

    :type: ObjectBase | None
    """

    active_object: Object | None
    """ 

    :type: Object | None
    """

    object: Object
    """ 

    :type: Object
    """

    edit_object: Object
    """ 

    :type: Object
    """

    sculpt_object: Object
    """ 

    :type: Object
    """

    vertex_paint_object: Object
    """ 

    :type: Object
    """

    weight_paint_object: Object
    """ 

    :type: Object
    """

    image_paint_object: Object
    """ 

    :type: Object
    """

    particle_edit_object: Object
    """ 

    :type: Object
    """

    sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_editable_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    gpencil_data: typing.Any
    gpencil_data_owner: ID
    """ 

    :type: ID
    """

    visible_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_strokes: list[GPencilStroke]
    """ 

    :type: list[GPencilStroke]
    """

    active_gpencil_layer: list[GPencilLayer] | None
    """ 

    :type: list[GPencilLayer] | None
    """

    active_gpencil_frame: typing.Any
    active_gpencil_palette: list[GPencilPalette] | None
    """ 

    :type: list[GPencilPalette] | None
    """

    active_gpencil_palettecolor: list[GPencilPaletteColor] | None
    """ 

    :type: list[GPencilPaletteColor] | None
    """

    active_gpencil_brush: GPencilSculptBrush | None
    """ 

    :type: GPencilSculptBrush | None
    """

    active_operator: Operator | None
    """ 

    :type: Operator | None
    """

    selected_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selected_editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_editable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    visible_objects: list[Object]
    """ 

    :type: list[Object]
    """

    visible_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selectable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selectable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    active_base: ObjectBase | None
    """ 

    :type: ObjectBase | None
    """

    active_object: Object | None
    """ 

    :type: Object | None
    """

    texture_slot: MaterialTextureSlot
    """ 

    :type: MaterialTextureSlot
    """

    scene: Scene
    """ 

    :type: Scene
    """

    world: World
    """ 

    :type: World
    """

    object: Object
    """ 

    :type: Object
    """

    mesh: Mesh
    """ 

    :type: Mesh
    """

    armature: Armature
    """ 

    :type: Armature
    """

    lattice: Lattice
    """ 

    :type: Lattice
    """

    curve: Curve
    """ 

    :type: Curve
    """

    meta_ball: MetaBall
    """ 

    :type: MetaBall
    """

    lamp: Lamp
    """ 

    :type: Lamp
    """

    speaker: Speaker
    """ 

    :type: Speaker
    """

    camera: Camera
    """ 

    :type: Camera
    """

    material: Material
    """ 

    :type: Material
    """

    material_slot: MaterialSlot
    """ 

    :type: MaterialSlot
    """

    texture: Texture
    """ 

    :type: Texture
    """

    texture_user: ID
    """ 

    :type: ID
    """

    texture_user_property: Property
    """ 

    :type: Property
    """

    bone: Bone
    """ 

    :type: Bone
    """

    edit_bone: EditBone
    """ 

    :type: EditBone
    """

    pose_bone: PoseBone
    """ 

    :type: PoseBone
    """

    particle_system: ParticleSystem
    """ 

    :type: ParticleSystem
    """

    particle_system_editable: ParticleSystem
    """ 

    :type: ParticleSystem
    """

    particle_settings: ParticleSettings
    """ 

    :type: ParticleSettings
    """

    cloth: ClothModifier
    """ 

    :type: ClothModifier
    """

    soft_body: SoftBodyModifier
    """ 

    :type: SoftBodyModifier
    """

    fluid: FluidSimulationModifier
    """ 

    :type: FluidSimulationModifier
    """

    smoke: SmokeModifier
    """ 

    :type: SmokeModifier
    """

    collision: CollisionModifier
    """ 

    :type: CollisionModifier
    """

    brush: Brush
    """ 

    :type: Brush
    """

    dynamic_paint: DynamicPaintModifier
    """ 

    :type: DynamicPaintModifier
    """

    line_style: FreestyleLineStyle
    """ 

    :type: FreestyleLineStyle
    """

    edit_image: Image
    """ 

    :type: Image
    """

    edit_mask: Mask
    """ 

    :type: Mask
    """

    selected_nodes: list[Node]
    """ 

    :type: list[Node]
    """

    active_node: Node | None
    """ 

    :type: Node | None
    """

    edit_text: Text
    """ 

    :type: Text
    """

    edit_movieclip: MovieClip
    """ 

    :type: MovieClip
    """

    edit_mask: Mask
    """ 

    :type: Mask
    """

    edit_mask: Mask
    """ 

    :type: Mask
    """

    def copy(self): ...

class Controller(bpy_struct):
    """Game engine logic brick to process events, connecting sensors to actuators"""

    active: bool | None
    """ Set the active state of the controller

    :type: bool | None
    """

    actuators: bpy_prop_collection[Actuator]
    """ The list containing the actuators connected to the controller

    :type: bpy_prop_collection[Actuator]
    """

    name: str
    """ 

    :type: str
    """

    show_expanded: bool
    """ Set controller expanded in the user interface

    :type: bool
    """

    states: int
    """ Set Controller state index (1 to 30)

    :type: int
    """

    type: str
    """ 

    :type: str
    """

    use_priority: bool
    """ Mark controller for execution before all non-marked controllers (good for startup scripts)

    :type: bool
    """

    def link(self, sensor: Sensor | None = None, actuator: Actuator | None = None):
        """Link the controller with a sensor/actuator

        :param sensor: Sensor to link the controller to
        :type sensor: Sensor | None
        :param actuator: Actuator to link the controller to
        :type actuator: Actuator | None
        """
        ...

    def unlink(self, sensor: Sensor | None = None, actuator: Actuator | None = None):
        """Unlink the controller from a sensor/actuator

        :param sensor: Sensor to unlink the controller from
        :type sensor: Sensor | None
        :param actuator: Actuator to unlink the controller from
        :type actuator: Actuator | None
        """
        ...

class CurveMap(bpy_struct):
    """Curve in a curve mapping"""

    extend: str
    """ Extrapolate the curve or extend it horizontally

    :type: str
    """

    points: CurveMapPoints
    """ 

    :type: CurveMapPoints
    """

    def evaluate(self, position: float | None) -> float:
        """Evaluate curve at given location

        :param position: Position, Position to evaluate curve at
        :type position: float | None
        :return: Value, Value of curve at given location
        :rtype: float
        """
        ...

class CurveMapPoint(bpy_struct):
    """Point of a curve used for a curve mapping"""

    handle_type: str
    """ Curve interpolation at this point: Bezier or vector

    :type: str
    """

    location: bpy_prop_array[float]
    """ X/Y coordinates of the curve point

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Selection state of the curve point

    :type: bool
    """

class CurveMapPoints(bpy_prop_collection[CurveMapPoint], bpy_struct):
    """Collection of Curve Map Points"""

    def new(self, position: float | None, value: float | None) -> CurveMapPoint:
        """Add point to CurveMap

        :param position: Position, Position to add point
        :type position: float | None
        :param value: Value, Value of point
        :type value: float | None
        :return: New point
        :rtype: CurveMapPoint
        """
        ...

    def remove(self, point: CurveMapPoint):
        """Delete point from CurveMap

        :param point: PointElement to remove
        :type point: CurveMapPoint
        """
        ...

class CurveMapping(bpy_struct):
    """Curve mapping to map color, vector and scalar values to other values using a user defined curve"""

    black_level: bpy_prop_array[float]
    """ For RGB curves, the color that black is mapped to

    :type: bpy_prop_array[float]
    """

    clip_max_x: float
    """ 

    :type: float
    """

    clip_max_y: float
    """ 

    :type: float
    """

    clip_min_x: float
    """ 

    :type: float
    """

    clip_min_y: float
    """ 

    :type: float
    """

    curves: bpy_prop_collection[CurveMap]
    """ 

    :type: bpy_prop_collection[CurveMap]
    """

    use_clip: bool
    """ Force the curve view to fit a defined boundary

    :type: bool
    """

    white_level: bpy_prop_array[float]
    """ For RGB curves, the color that white is mapped to

    :type: bpy_prop_array[float]
    """

    def update(self):
        """Update curve mapping after making changes"""
        ...

    def initialize(self):
        """Initialize curve"""
        ...

class CurvePaintSettings(bpy_struct):
    corner_angle: float
    """ Angles above this are considered corners

    :type: float
    """

    curve_type: str
    """ Type of curve to use for new strokes

    :type: str
    """

    depth_mode: str
    """ Method of projecting depth

    :type: str
    """

    error_threshold: int
    """ Allow deviation for a smoother, less precise line

    :type: int
    """

    fit_method: str
    """ Curve fitting method

    :type: str
    """

    radius_max: float
    """ Radius to use when the maximum pressure is applied (or when a tablet isn't used)

    :type: float
    """

    radius_min: float
    """ Minimum radius when the minimum pressure is applied (also the minimum when tapering)

    :type: float
    """

    radius_taper_end: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    radius_taper_start: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    surface_offset: float
    """ Offset the stroke from the surface

    :type: float
    """

    surface_plane: str
    """ Plane for projected stroke

    :type: str
    """

    use_corners_detect: bool
    """ Detect corners and use non-aligned handles

    :type: bool
    """

    use_offset_absolute: bool
    """ Apply a fixed offset (don't scale by the radius)

    :type: bool
    """

    use_pressure_radius: bool
    """ Map tablet pressure to curve radius

    :type: bool
    """

    use_stroke_endpoints: bool
    """ Use the start of the stroke for the depth

    :type: bool
    """

class CurveSplines(bpy_prop_collection[Spline], bpy_struct):
    """Collection of curve splines"""

    active: Spline | None
    """ Active curve spline

    :type: Spline | None
    """

    def new(self, type: str | None) -> Spline:
        """Add a new spline to the curve

        :param type: type for the new spline
        :type type: str | None
        :return: The newly created spline
        :rtype: Spline
        """
        ...

    def remove(self, spline: Spline):
        """Remove a spline from a curve

        :param spline: The spline to remove
        :type spline: Spline
        """
        ...

    def clear(self):
        """Remove all splines from a curve"""
        ...

class Depsgraph(bpy_struct):
    def debug_graphviz(self, filename: str | typing.Any):
        """debug_graphviz

        :param filename: File Name, File in which to store graphviz debug output
        :type filename: str | typing.Any
        """
        ...

    def debug_rebuild(self):
        """debug_rebuild"""
        ...

    def debug_stats(self):
        """Report the number of elements in the Dependency Graph"""
        ...

class DisplaySafeAreas(bpy_struct):
    """Safe Areas used in 3D view and the VSE"""

    action: bpy_prop_array[float]
    """ Safe area for general elements

    :type: bpy_prop_array[float]
    """

    action_center: bpy_prop_array[float]
    """ Safe area for general elements in a different aspect ratio

    :type: bpy_prop_array[float]
    """

    title: bpy_prop_array[float]
    """ Safe area for text and graphics

    :type: bpy_prop_array[float]
    """

    title_center: bpy_prop_array[float]
    """ Safe area for text and graphics in a different aspect ratio

    :type: bpy_prop_array[float]
    """

class DopeSheet(bpy_struct):
    """Settings for filtering the channels shown in animation editors"""

    filter_fcurve_name: str
    """ F-Curve live filtering string

    :type: str
    """

    filter_group: Group
    """ Group that included object should be a member of

    :type: Group
    """

    filter_text: str
    """ Live filtering string

    :type: str
    """

    show_armatures: bool
    """ Include visualization of armature related animation data

    :type: bool
    """

    show_cameras: bool
    """ Include visualization of camera related animation data

    :type: bool
    """

    show_curves: bool
    """ Include visualization of curve related animation data

    :type: bool
    """

    show_datablock_filters: bool
    """ Show options for whether channels related to certain types of data are included

    :type: bool
    """

    show_expanded_summary: bool
    """ Collapse summary when shown, so all other channels get hidden (Dope Sheet editors only)

    :type: bool
    """

    show_gpencil: bool
    """ Include visualization of Grease Pencil related animation data and frames

    :type: bool
    """

    show_gpencil_3d_only: bool
    """ Only show Grease Pencil datablocks used as part of the active scene

    :type: bool
    """

    show_hidden: bool
    """ Include channels from objects/bone that are not visible

    :type: bool
    """

    show_lamps: bool
    """ Include visualization of lamp related animation data

    :type: bool
    """

    show_lattices: bool
    """ Include visualization of lattice related animation data

    :type: bool
    """

    show_linestyles: bool
    """ Include visualization of Line Style related Animation data

    :type: bool
    """

    show_materials: bool
    """ Include visualization of material related animation data

    :type: bool
    """

    show_meshes: bool
    """ Include visualization of mesh related animation data

    :type: bool
    """

    show_metaballs: bool
    """ Include visualization of metaball related animation data

    :type: bool
    """

    show_missing_nla: bool
    """ Include animation data blocks with no NLA data (NLA editor only)

    :type: bool
    """

    show_modifiers: bool
    """ Include visualization of animation data related to data-blocks linked to modifiers

    :type: bool
    """

    show_nodes: bool
    """ Include visualization of node related animation data

    :type: bool
    """

    show_only_errors: bool
    """ Only include F-Curves and drivers that are disabled or have errors

    :type: bool
    """

    show_only_group_objects: bool
    """ Only include channels from objects in the specified group

    :type: bool
    """

    show_only_matching_fcurves: bool
    """ Only include F-Curves with names containing search text

    :type: bool
    """

    show_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    show_particles: bool
    """ Include visualization of particle related animation data

    :type: bool
    """

    show_scenes: bool
    """ Include visualization of scene related animation data

    :type: bool
    """

    show_shapekeys: bool
    """ Include visualization of shape key related animation data

    :type: bool
    """

    show_speakers: bool
    """ Include visualization of speaker related animation data

    :type: bool
    """

    show_summary: bool
    """ Display an additional 'summary' line (Dope Sheet editors only)

    :type: bool
    """

    show_textures: bool
    """ Include visualization of texture related animation data

    :type: bool
    """

    show_transforms: bool
    """ Include visualization of object-level animation data (mostly transforms)

    :type: bool
    """

    show_worlds: bool
    """ Include visualization of world related animation data

    :type: bool
    """

    source: ID
    """ ID-Block representing source data, usually ID_SCE (i.e. Scene)

    :type: ID
    """

    use_datablock_sort: bool
    """ Alphabetically sorts datablocks - mainly objects in the scene (disable to increase viewport speed)

    :type: bool
    """

    use_filter_text: bool
    """ Only include channels with names containing search text

    :type: bool
    """

    use_multi_word_filter: bool
    """ Perform fuzzy/multi-word matching (WARNING: May be slow)

    :type: bool
    """

class Driver(bpy_struct):
    """Driver for the value of a setting based on an external value"""

    expression: str
    """ Expression to use for Scripted Expression

    :type: str
    """

    is_valid: bool
    """ Driver could not be evaluated in past, so should be skipped

    :type: bool
    """

    show_debug_info: bool
    """ Show intermediate values for the driver calculations to allow debugging of drivers

    :type: bool
    """

    type: str
    """ Driver type

    :type: str
    """

    use_self: bool
    """ Include a 'self' variable in the name-space, so drivers can easily reference the data being modified (object, bone, etc...)

    :type: bool
    """

    variables: ChannelDriverVariables
    """ Properties acting as inputs for this driver

    :type: ChannelDriverVariables
    """

class DriverTarget(bpy_struct):
    """Source of input values for driver variables"""

    bone_target: str
    """ Name of PoseBone to use as target

    :type: str
    """

    data_path: str
    """ RNA Path (from ID-block) to property used

    :type: str
    """

    id: ID
    """ ID-block that the specific property used can be found from (id_type property must be set first)

    :type: ID
    """

    id_type: str
    """ Type of ID-block that can be used

    :type: str
    """

    transform_space: str
    """ Space in which transforms are used

    :type: str
    """

    transform_type: str
    """ Driver variable type

    :type: str
    """

class DriverVariable(bpy_struct):
    """Variable from some source/target for driver relationship"""

    is_name_valid: bool
    """ Is this a valid name for a driver variable

    :type: bool
    """

    name: str
    """ Name to use in scripted expressions/functions (no spaces or dots are allowed, and must start with a letter)

    :type: str
    """

    targets: bpy_prop_collection[DriverTarget]
    """ Sources of input data for evaluating this variable

    :type: bpy_prop_collection[DriverTarget]
    """

    type: str
    """ Driver variable type

    :type: str
    """

class DupliObject(bpy_struct):
    """An object duplicate"""

    hide: bool
    """ Don't show dupli object in viewport or render

    :type: bool
    """

    index: int
    """ Index in the lowest-level dupli list

    :type: int
    """

    matrix: bpy_prop_array[float]
    """ Object duplicate transformation matrix

    :type: bpy_prop_array[float]
    """

    object: Object
    """ Object being duplicated

    :type: Object
    """

    orco: bpy_prop_array[float]
    """ Generated coordinates in parent object space

    :type: bpy_prop_array[float]
    """

    particle_system: ParticleSystem
    """ Particle system that this dupli object was instanced from

    :type: ParticleSystem
    """

    persistent_id: bpy_prop_array[int]
    """ Persistent identifier for inter-frame matching of objects with motion blur

    :type: bpy_prop_array[int]
    """

    type: str
    """ Duplicator type that generated this dupli object

    :type: str
    """

    uv: bpy_prop_array[float]
    """ UV coordinates in parent object space

    :type: bpy_prop_array[float]
    """

class DynamicPaintBrushSettings(bpy_struct):
    """Brush settings"""

    invert_proximity: bool
    """ Proximity falloff is applied inside the volume

    :type: bool
    """

    material: Material
    """ Material to use (if not defined, material linked to the mesh is used)

    :type: Material
    """

    paint_alpha: float
    """ Paint alpha

    :type: float
    """

    paint_color: bpy_prop_array[float]
    """ Color of the paint

    :type: bpy_prop_array[float]
    """

    paint_distance: float
    """ Maximum distance from brush to mesh surface to affect paint

    :type: float
    """

    paint_ramp: ColorRamp
    """ Color ramp used to define proximity falloff

    :type: ColorRamp
    """

    paint_source: str
    """ 

    :type: str
    """

    paint_wetness: float
    """ Paint wetness, visible in wetmap (some effects only affect wet paint)

    :type: float
    """

    particle_system: ParticleSystem
    """ The particle system to paint with

    :type: ParticleSystem
    """

    proximity_falloff: str
    """ Proximity falloff type

    :type: str
    """

    ray_direction: str
    """ Ray direction to use for projection (if brush object is located in that direction it's painted)

    :type: str
    """

    smooth_radius: float
    """ Smooth falloff added after solid radius

    :type: float
    """

    smudge_strength: float
    """ Smudge effect strength

    :type: float
    """

    solid_radius: float
    """ Radius that will be painted solid

    :type: float
    """

    use_absolute_alpha: bool
    """ Only increase alpha value if paint alpha is higher than existing

    :type: bool
    """

    use_material: bool
    """ Use object material to define color and influence

    :type: bool
    """

    use_negative_volume: bool
    """ Negate influence inside the volume

    :type: bool
    """

    use_paint_erase: bool
    """ Erase / remove paint instead of adding it

    :type: bool
    """

    use_particle_radius: bool
    """ Use radius from particle settings

    :type: bool
    """

    use_proximity_project: bool
    """ Brush is projected to canvas from defined direction within brush proximity

    :type: bool
    """

    use_proximity_ramp_alpha: bool
    """ Only read color ramp alpha

    :type: bool
    """

    use_smudge: bool
    """ Make this brush to smudge existing paint as it moves

    :type: bool
    """

    use_velocity_alpha: bool
    """ Multiply brush influence by velocity color ramp alpha

    :type: bool
    """

    use_velocity_color: bool
    """ Replace brush color by velocity color ramp

    :type: bool
    """

    use_velocity_depth: bool
    """ Multiply brush intersection depth (displace, waves) by velocity ramp alpha

    :type: bool
    """

    velocity_max: float
    """ Velocity considered as maximum influence (Blender units per frame)

    :type: float
    """

    velocity_ramp: ColorRamp
    """ Color ramp used to define brush velocity effect

    :type: ColorRamp
    """

    wave_clamp: float
    """ Maximum level of surface intersection used to influence waves (use 0.0 to disable)

    :type: float
    """

    wave_factor: float
    """ Multiplier for wave influence of this brush

    :type: float
    """

    wave_type: str
    """ 

    :type: str
    """

class DynamicPaintCanvasSettings(bpy_struct):
    """Dynamic Paint canvas settings"""

    canvas_surfaces: DynamicPaintSurfaces
    """ Paint surface list

    :type: DynamicPaintSurfaces
    """

class DynamicPaintSurface(bpy_struct):
    """A canvas surface layer"""

    brush_group: Group
    """ Only use brush objects from this group

    :type: Group
    """

    brush_influence_scale: float
    """ Adjust influence brush objects have on this surface

    :type: float
    """

    brush_radius_scale: float
    """ Adjust radius of proximity brushes or particles for this surface

    :type: float
    """

    color_dry_threshold: float
    """ The wetness level when colors start to shift to the background

    :type: float
    """

    color_spread_speed: float
    """ How fast colors get mixed within wet paint

    :type: float
    """

    depth_clamp: float
    """ Maximum level of depth intersection in object space (use 0.0 to disable)

    :type: float
    """

    displace_factor: float
    """ Strength of displace when applied to the mesh

    :type: float
    """

    displace_type: str
    """ 

    :type: str
    """

    dissolve_speed: int
    """ Approximately in how many frames should dissolve happen

    :type: int
    """

    drip_acceleration: float
    """ How much surface acceleration affects dripping

    :type: float
    """

    drip_velocity: float
    """ How much surface velocity affects dripping

    :type: float
    """

    dry_speed: int
    """ Approximately in how many frames should drying happen

    :type: int
    """

    effect_ui: str
    """ 

    :type: str
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    frame_end: int
    """ Simulation end frame

    :type: int
    """

    frame_start: int
    """ Simulation start frame

    :type: int
    """

    frame_substeps: int
    """ Do extra frames between scene frames to ensure smooth motion

    :type: int
    """

    image_fileformat: str
    """ 

    :type: str
    """

    image_output_path: str
    """ Directory to save the textures

    :type: str
    """

    image_resolution: int
    """ Output image resolution

    :type: int
    """

    init_color: bpy_prop_array[float]
    """ Initial color of the surface

    :type: bpy_prop_array[float]
    """

    init_color_type: str
    """ 

    :type: str
    """

    init_layername: str
    """ 

    :type: str
    """

    init_texture: Texture
    """ 

    :type: Texture
    """

    is_active: bool
    """ Toggle whether surface is processed or ignored

    :type: bool
    """

    is_cache_user: bool
    """ 

    :type: bool
    """

    name: str
    """ Surface name

    :type: str
    """

    output_name_a: str
    """ Name used to save output from this surface

    :type: str
    """

    output_name_b: str
    """ Name used to save output from this surface

    :type: str
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    preview_id: str
    """ 

    :type: str
    """

    show_preview: bool
    """ Display surface preview in 3D-views

    :type: bool
    """

    shrink_speed: float
    """ How fast shrink effect moves on the canvas surface

    :type: float
    """

    spread_speed: float
    """ How fast spread effect moves on the canvas surface

    :type: float
    """

    surface_format: str
    """ Surface Format

    :type: str
    """

    surface_type: str
    """ Surface Type

    :type: str
    """

    use_antialiasing: bool
    """ Use 5x multisampling to smooth paint edges

    :type: bool
    """

    use_color_preview: bool
    """ Whether this surface has some color preview for 3D view

    :type: bool
    """

    use_dissolve: bool
    """ Enable to make surface changes disappear over time

    :type: bool
    """

    use_dissolve_log: bool
    """ Use logarithmic dissolve (makes high values to fade faster than low values)

    :type: bool
    """

    use_drip: bool
    """ Process drip effect (drip wet paint to gravity direction)

    :type: bool
    """

    use_dry_log: bool
    """ Use logarithmic drying (makes high values to dry faster than low values)

    :type: bool
    """

    use_drying: bool
    """ Enable to make surface wetness dry over time

    :type: bool
    """

    use_incremental_displace: bool
    """ New displace is added cumulatively on top of existing

    :type: bool
    """

    use_output_a: bool
    """ Save this output layer

    :type: bool
    """

    use_output_b: bool
    """ Save this output layer

    :type: bool
    """

    use_premultiply: bool
    """ Multiply color by alpha (recommended for Blender input)

    :type: bool
    """

    use_shrink: bool
    """ Process shrink effect (shrink paint areas)

    :type: bool
    """

    use_spread: bool
    """ Process spread effect (spread wet paint around surface)

    :type: bool
    """

    use_wave_open_border: bool
    """ Pass waves through mesh edges

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    wave_damping: float
    """ Wave damping factor

    :type: float
    """

    wave_smoothness: float
    """ Limit maximum steepness of wave slope between simulation points (use higher values for smoother waves at expense of reduced detail)

    :type: float
    """

    wave_speed: float
    """ Wave propagation speed

    :type: float
    """

    wave_spring: float
    """ Spring force that pulls water level back to zero

    :type: float
    """

    wave_timescale: float
    """ Wave time scaling factor

    :type: float
    """

    def output_exists(self, object: Object, index: int | None) -> bool:
        """Checks if surface output layer of given name exists

        :param object:
        :type object: Object
        :param index: Index
        :type index: int | None
        :return:
        :rtype: bool
        """
        ...

class DynamicPaintSurfaces(bpy_prop_collection[DynamicPaintSurface], bpy_struct):
    """Collection of Dynamic Paint Canvas surfaces"""

    active: DynamicPaintSurface
    """ Active Dynamic Paint surface being displayed

    :type: DynamicPaintSurface
    """

    active_index: int | None
    """ 

    :type: int | None
    """

class EditBone(bpy_struct):
    """Editmode bone in an Armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveiny: float
    """ Y-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveouty: float
    """ Y-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_in: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_out: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: float
    """ Scale factor for start of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_scaleout: float
    """ Scale factor for end of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: bpy_prop_array[float]
    """ Location of head end of the bone

    :type: bpy_prop_array[float]
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when in Edit Mode

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    layers: list[bool]
    """ Layers bone exists in

    :type: list[bool]
    """

    lock: bool
    """ Bone is not able to be transformed when in Edit Mode

    :type: bool
    """

    matrix: bpy_prop_array[float]
    """ Matrix combining loc/rot of the bone (head position, direction and roll), in armature space (WARNING: does not include/support bone's length/size)

    :type: bpy_prop_array[float]
    """

    name: str
    """ 

    :type: str
    """

    parent: EditBone
    """ Parent edit bone (in same Armature)

    :type: EditBone
    """

    roll: float
    """ Bone rotation around head-tail axis

    :type: float
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always drawn as Wireframe regardless of viewport draw mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: bpy_prop_array[float]
    """ Location of tail end of the bone

    :type: bpy_prop_array[float]
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Use Roll Out of parent bone as Roll In of its children

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_inherit_scale: bool
    """ Bone inherits scaling from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character
(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.
(readonly)"""

    children: typing.Any
    """ A list of all the bones children.
(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.
(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.
(readonly)"""

    length: typing.Any
    """ The distance from head to tail,
when set the head is moved to fit the length."""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent
(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)
(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.
(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.
(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.
(readonly)"""

    def align_roll(self, vector: collections.abc.Iterable[float] | None):
        """Align the bone to a localspace roll so the Z axis points in the direction of the vector given

        :param vector: Vector
        :type vector: collections.abc.Iterable[float] | None
        """
        ...

    def align_orientation(self, other):
        """Align this bone to another by moving its tail and settings its roll
        the length of the other bone is not used.

                :param other:
        """
        ...

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """
        ...

    def transform(
        self,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
        scale: bool | None = True,
        roll: bool | None = True,
    ):
        """Transform the the bones head, tail, roll and envelope
        (when the matrix has a scale component).

                :param matrix: 3x3 or 4x4 transformation matrix.
                :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param scale: Scale the bone envelope by the matrix.
                :type scale: bool | None
                :param roll: Correct the roll to point in the same relative
        direction to the head and tail.
                :type roll: bool | None
        """
        ...

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """
        ...

class EffectorWeights(bpy_struct):
    """Effector weights for physics simulation"""

    all: float
    """ All effector's weight

    :type: float
    """

    apply_to_hair_growing: bool
    """ Use force fields when growing hair

    :type: bool
    """

    boid: float
    """ Boid effector weight

    :type: float
    """

    charge: float
    """ Charge effector weight

    :type: float
    """

    curve_guide: float
    """ Curve guide effector weight

    :type: float
    """

    drag: float
    """ Drag effector weight

    :type: float
    """

    force: float
    """ Force effector weight

    :type: float
    """

    gravity: float
    """ Global gravity weight

    :type: float
    """

    group: Group
    """ Limit effectors to this Group

    :type: Group
    """

    harmonic: float
    """ Harmonic effector weight

    :type: float
    """

    lennardjones: float
    """ Lennard-Jones effector weight

    :type: float
    """

    magnetic: float
    """ Magnetic effector weight

    :type: float
    """

    smokeflow: float
    """ Smoke Flow effector weight

    :type: float
    """

    texture: float
    """ Texture effector weight

    :type: float
    """

    turbulence: float
    """ Turbulence effector weight

    :type: float
    """

    vortex: float
    """ Vortex effector weight

    :type: float
    """

    wind: float
    """ Wind effector weight

    :type: float
    """

class EnumPropertyItem(bpy_struct):
    """Definition of a choice in an RNA enum property"""

    description: str
    """ Description of the item's purpose

    :type: str
    """

    icon: str
    """ Icon of the item

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    name: str
    """ Human readable name

    :type: str
    """

    value: int
    """ Value of the item

    :type: int
    """

class EnvironmentMap(bpy_struct):
    """Environment map created by the renderer and cached for subsequent renders"""

    clip_end: float
    """ Objects further than this are not visible to map

    :type: float
    """

    clip_start: float
    """ Objects nearer than this are not visible to map

    :type: float
    """

    depth: int
    """ Number of times a map will be rendered recursively (mirror effects)

    :type: int
    """

    is_valid: bool
    """ True if this map is ready for use, False if it needs rendering

    :type: bool
    """

    layers_ignore: list[bool]
    """ Hide objects on these layers when generating the Environment Map

    :type: list[bool]
    """

    mapping: str
    """ 

    :type: str
    """

    resolution: int
    """ Pixel resolution of the rendered environment map

    :type: int
    """

    source: str
    """ 

    :type: str
    """

    viewpoint_object: Object
    """ Object to use as the environment map's viewpoint location

    :type: Object
    """

    zoom: float
    """ 

    :type: float
    """

    def clear(self):
        """Discard the environment map and free it from memory"""
        ...

    def save(
        self,
        filepath: str | typing.Any,
        scene: Scene | None = None,
        layout: typing.Any | None = (
            0.0,
            0.0,
            1.0,
            0.0,
            2.0,
            0.0,
            0.0,
            1.0,
            1.0,
            1.0,
            2.0,
            1.0,
        ),
    ):
        """Save the environment map to disc using the scene render settings

        :param filepath: File path, Location of the output file
        :type filepath: str | typing.Any
        :param scene: Overrides the scene from which image parameters are taken
        :type scene: Scene | None
        :param layout: File layout, Flat array describing the X,Y position of each cube face in the output image, where 1 is the size of a face - order is [+Z -Z +Y -X -Y +X] (use -1 to skip a face)
        :type layout: typing.Any | None
        """
        ...

class Event(bpy_struct):
    """Window Manager Event"""

    alt: bool
    """ True when the Alt/Option key is held

    :type: bool
    """

    ascii: str
    """ Single ASCII character for this event

    :type: str
    """

    ctrl: bool
    """ True when the Ctrl key is held

    :type: bool
    """

    is_tablet: bool
    """ The pressure of the tablet or 1.0 if no tablet present

    :type: bool
    """

    mouse_prev_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_prev_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    mouse_region_x: int
    """ The region relative horizontal location of the mouse

    :type: int
    """

    mouse_region_y: int
    """ The region relative vertical location of the mouse

    :type: int
    """

    mouse_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    oskey: bool
    """ True when the Cmd key is held

    :type: bool
    """

    pressure: float
    """ The pressure of the tablet or 1.0 if no tablet present

    :type: float
    """

    shift: bool
    """ True when the Shift key is held

    :type: bool
    """

    tilt: bpy_prop_array[float]
    """ The pressure of the tablet or zeroes if no tablet present

    :type: bpy_prop_array[float]
    """

    type: str
    """ 

    :type: str
    """

    unicode: str
    """ Single unicode character for this event

    :type: str
    """

    value: str
    """ The type of event, only applies to some

    :type: str
    """

class FCurve(bpy_struct):
    """F-Curve defining values of a period of time"""

    array_index: int
    """ Index to the specific property affected by F-Curve if applicable

    :type: int
    """

    color: bpy_prop_array[float]
    """ Color of the F-Curve in the Graph Editor

    :type: bpy_prop_array[float]
    """

    color_mode: str
    """ Method used to determine color of F-Curve in Graph Editor

    :type: str
    """

    data_path: str
    """ RNA Path to property affected by F-Curve

    :type: str
    """

    driver: Driver
    """ Channel Driver (only set for Driver F-Curves)

    :type: Driver
    """

    extrapolation: str
    """ Method used for evaluating value of F-Curve outside first and last keyframes

    :type: str
    """

    group: ActionGroup
    """ Action Group that this F-Curve belongs to

    :type: ActionGroup
    """

    hide: bool
    """ F-Curve and its keyframes are hidden in the Graph Editor graphs

    :type: bool
    """

    is_valid: bool
    """ False when F-Curve could not be evaluated in past, so should be skipped when evaluating

    :type: bool
    """

    keyframe_points: FCurveKeyframePoints
    """ User-editable keyframes

    :type: FCurveKeyframePoints
    """

    lock: bool
    """ F-Curve's settings cannot be edited

    :type: bool
    """

    modifiers: FCurveModifiers
    """ Modifiers affecting the shape of the F-Curve

    :type: FCurveModifiers
    """

    mute: bool
    """ F-Curve is not evaluated

    :type: bool
    """

    sampled_points: bpy_prop_collection[FCurveSample]
    """ Sampled animation data

    :type: bpy_prop_collection[FCurveSample]
    """

    select: bool
    """ F-Curve is selected for editing

    :type: bool
    """

    def evaluate(self, frame: float | None) -> float:
        """Evaluate F-Curve

        :param frame: Frame, Evaluate F-Curve at given frame
        :type frame: float | None
        :return: Value, Value of F-Curve specific frame
        :rtype: float
        """
        ...

    def update(self):
        """Ensure keyframes are sorted in chronological order and handles are set correctly"""
        ...

    def range(self) -> bpy_prop_array[float]:
        """Get the time extents for F-Curve

        :return: Range, Min/Max values
        :rtype: bpy_prop_array[float]
        """
        ...

    def update_autoflags(self, data: typing.Any):
        """Update FCurve flags set automatically from affected property (currently, integer/discrete flags set when the property is not a float)

        :param data: Data, Data containing the property controlled by given FCurve
        :type data: typing.Any
        """
        ...

    def convert_to_samples(self, start: int | None, end: int | None):
        """Convert current FCurve from keyframes to sample points, if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """
        ...

    def convert_to_keyframes(self, start: int | None, end: int | None):
        """Convert current FCurve from sample points to keyframes (linear interpolation), if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """
        ...

class FCurveKeyframePoints(bpy_prop_collection[Keyframe], bpy_struct):
    """Collection of keyframe points"""

    def insert(
        self,
        frame: float | None,
        value: float | None,
        options: set[str] | None = {},
        keyframe_type: str | None = "KEYFRAME",
    ) -> Keyframe:
        """Add a keyframe point to a F-Curve

                :param frame: X Value of this keyframe point
                :type frame: float | None
                :param value: Y Value of this keyframe point
                :type value: float | None
                :param options: Keyframe options

        REPLACE Replace, Don't add any new keyframes, but just replace existing ones.

        NEEDED Needed, Only adds keyframes that are needed.

        FAST Fast, Fast keyframe insertion to avoid recalculating the curve each time.
                :type options: set[str] | None
                :param keyframe_type: Type of keyframe to insert

        KEYFRAME Keyframe, Normal keyframe - e.g. for key poses.

        BREAKDOWN Breakdown, A breakdown pose - e.g. for transitions between key poses.

        MOVING_HOLD Moving Hold, A keyframe that is part of a moving hold.

        EXTREME Extreme, An 'extreme' pose, or some other purpose as needed.

        JITTER Jitter, A filler or baked keyframe for keying on ones, or some other purpose as needed.
                :type keyframe_type: str | None
                :return: Newly created keyframe
                :rtype: Keyframe
        """
        ...

    def add(self, count: typing.Any | None = 1):
        """Add a keyframe point to a F-Curve

        :param count: Number, Number of points to add to the spline
        :type count: typing.Any | None
        """
        ...

    def remove(self, keyframe: Keyframe, fast: bool | typing.Any | None = False):
        """Remove keyframe from an F-Curve

        :param keyframe: Keyframe to remove
        :type keyframe: Keyframe
        :param fast: Fast, Fast keyframe removal to avoid recalculating the curve each time
        :type fast: bool | typing.Any | None
        """
        ...

class FCurveModifiers(bpy_prop_collection[FModifier], bpy_struct):
    """Collection of F-Curve Modifiers"""

    active: FModifier | None
    """ Active F-Curve Modifier

    :type: FModifier | None
    """

    def new(self, type: str | None) -> FModifier:
        """Add a constraint to this object

                :param type: Constraint type to add

        NULL Invalid.

        GENERATOR Generator, Generate a curve using a factorized or expanded polynomial.

        FNGENERATOR Built-In Function, Generate a curve using standard math functions such as sin and cos.

        ENVELOPE Envelope, Reshape F-Curve values - e.g. change amplitude of movements.

        CYCLES Cycles, Cyclic extend/repeat keyframe sequence.

        NOISE Noise, Add pseudo-random noise on top of F-Curves.

        LIMITS Limits, Restrict maximum and minimum values of F-Curve.

        STEPPED Stepped Interpolation, Snap values to nearest grid-step - e.g. for a stop-motion look.
                :type type: str | None
                :return: New fmodifier
                :rtype: FModifier
        """
        ...

    def remove(self, modifier: FModifier):
        """Remove a modifier from this F-Curve

        :param modifier: Removed modifier
        :type modifier: FModifier
        """
        ...

class FCurveSample(bpy_struct):
    """Sample point for F-Curve"""

    co: bpy_prop_array[float]
    """ Point coordinates

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Selection status

    :type: bool
    """

class FFmpegSettings(bpy_struct):
    """FFmpeg related settings for the scene"""

    audio_bitrate: int
    """ Audio bitrate (kb/s)

    :type: int
    """

    audio_channels: str
    """ Audio channel count

    :type: str
    """

    audio_codec: str
    """ FFmpeg audio codec to use

    :type: str
    """

    audio_mixrate: int
    """ Audio samplerate(samples/s)

    :type: int
    """

    audio_volume: float
    """ Audio volume

    :type: float
    """

    buffersize: int
    """ Rate control: buffer size (kb)

    :type: int
    """

    codec: str
    """ FFmpeg codec to use

    :type: str
    """

    format: str
    """ Output file format

    :type: str
    """

    gopsize: int
    """ Distance between key frames

    :type: int
    """

    maxrate: int
    """ Rate control: max rate (kb/s)

    :type: int
    """

    minrate: int
    """ Rate control: min rate (kb/s)

    :type: int
    """

    muxrate: int
    """ Mux rate (bits/s(!))

    :type: int
    """

    packetsize: int
    """ Mux packet size (byte)

    :type: int
    """

    use_autosplit: bool
    """ Autosplit output at 2GB boundary

    :type: bool
    """

    use_lossless_output: bool
    """ Use lossless output for video streams

    :type: bool
    """

    video_bitrate: int
    """ Video bitrate (kb/s)

    :type: int
    """

class FModifier(bpy_struct):
    """Modifier for values of F-Curve"""

    active: bool | None
    """ F-Curve Modifier is the one being edited

    :type: bool | None
    """

    blend_in: float
    """ Number of frames from start frame for influence to take effect

    :type: float
    """

    blend_out: float
    """ Number of frames from end frame for influence to fade out

    :type: float
    """

    frame_end: float
    """ Frame that modifier's influence ends (if Restrict Frame Range is in use)

    :type: float
    """

    frame_start: float
    """ Frame that modifier's influence starts (if Restrict Frame Range is in use)

    :type: float
    """

    influence: float
    """ Amount of influence F-Curve Modifier will have when not fading in/out

    :type: float
    """

    is_valid: bool
    """ F-Curve Modifier has invalid settings and will not be evaluated

    :type: bool
    """

    mute: bool
    """ F-Curve Modifier will not be evaluated

    :type: bool
    """

    show_expanded: bool
    """ F-Curve Modifier's panel is expanded in UI

    :type: bool
    """

    type: str
    """ F-Curve Modifier Type

    :type: str
    """

    use_influence: bool
    """ F-Curve Modifier's effects will be tempered by a default factor

    :type: bool
    """

    use_restricted_range: bool
    """ F-Curve Modifier is only applied for the specified frame range to help mask off effects in order to chain them

    :type: bool
    """

class FModifierEnvelopeControlPoint(bpy_struct):
    """Control point for envelope F-Modifier"""

    frame: float
    """ Frame this control-point occurs on

    :type: float
    """

    max: float
    """ Upper bound of envelope at this control-point

    :type: float
    """

    min: float
    """ Lower bound of envelope at this control-point

    :type: float
    """

class FModifierEnvelopeControlPoints(
    bpy_prop_collection[FModifierEnvelopeControlPoint], bpy_struct
):
    """Control points defining the shape of the envelope"""

    def add(self, frame: float | None) -> FModifierEnvelopeControlPoint:
        """Add a control point to a FModifierEnvelope

        :param frame: Frame to add this control-point
        :type frame: float | None
        :return: Newly created control-point
        :rtype: FModifierEnvelopeControlPoint
        """
        ...

    def remove(self, point: FModifierEnvelopeControlPoint):
        """Remove a control-point from an FModifierEnvelope

        :param point: Control-point to remove
        :type point: FModifierEnvelopeControlPoint
        """
        ...

class FieldSettings(bpy_struct):
    """Field settings for an object in physics simulation"""

    apply_to_location: bool
    """ Effect particles' location

    :type: bool
    """

    apply_to_rotation: bool
    """ Effect particles' dynamic rotation

    :type: bool
    """

    distance_max: float
    """ Maximum distance for the field to work

    :type: float
    """

    distance_min: float
    """ Minimum distance for the field's fall-off

    :type: float
    """

    falloff_power: float
    """ Falloff power (real gravitational falloff = 2)

    :type: float
    """

    falloff_type: str
    """ 

    :type: str
    """

    flow: float
    """ Convert effector force into air flow velocity

    :type: float
    """

    guide_clump_amount: float
    """ Amount of clumping

    :type: float
    """

    guide_clump_shape: float
    """ Shape of clumping

    :type: float
    """

    guide_free: float
    """ Guide-free time from particle life's end

    :type: float
    """

    guide_kink_amplitude: float
    """ The amplitude of the offset

    :type: float
    """

    guide_kink_axis: str
    """ Which axis to use for offset

    :type: str
    """

    guide_kink_frequency: float
    """ The frequency of the offset (1/total length)

    :type: float
    """

    guide_kink_shape: float
    """ Adjust the offset to the beginning/end

    :type: float
    """

    guide_kink_type: str
    """ Type of periodic offset on the curve

    :type: str
    """

    guide_minimum: float
    """ The distance from which particles are affected fully

    :type: float
    """

    harmonic_damping: float
    """ Damping of the harmonic force

    :type: float
    """

    inflow: float
    """ Inwards component of the vortex force

    :type: float
    """

    linear_drag: float
    """ Drag component proportional to velocity

    :type: float
    """

    noise: float
    """ Amount of noise for the force strength

    :type: float
    """

    quadratic_drag: float
    """ Drag component proportional to the square of velocity

    :type: float
    """

    radial_falloff: float
    """ Radial falloff power (real gravitational falloff = 2)

    :type: float
    """

    radial_max: float
    """ Maximum radial distance for the field to work

    :type: float
    """

    radial_min: float
    """ Minimum radial distance for the field's fall-off

    :type: float
    """

    rest_length: float
    """ Rest length of the harmonic force

    :type: float
    """

    seed: int
    """ Seed of the noise

    :type: int
    """

    shape: str
    """ Which direction is used to calculate the effector force

    :type: str
    """

    size: float
    """ Size of the turbulence

    :type: float
    """

    source_object: Object
    """ Select domain object of the smoke simulation

    :type: Object
    """

    strength: float
    """ Strength of force field

    :type: float
    """

    texture: Texture
    """ Texture to use as force

    :type: Texture
    """

    texture_mode: str
    """ How the texture effect is calculated (RGB & Curl need a RGB texture, else Gradient will be used instead)

    :type: str
    """

    texture_nabla: float
    """ Defines size of derivative offset used for calculating gradient and curl

    :type: float
    """

    type: str
    """ Type of field

    :type: str
    """

    use_2d_force: bool
    """ Apply force only in 2D

    :type: bool
    """

    use_absorption: bool
    """ Force gets absorbed by collision objects

    :type: bool
    """

    use_global_coords: bool
    """ Use effector/global coordinates for turbulence

    :type: bool
    """

    use_guide_path_add: bool
    """ Based on distance/falloff it adds a portion of the entire path

    :type: bool
    """

    use_guide_path_weight: bool
    """ Use curve weights to influence the particle influence along the curve

    :type: bool
    """

    use_max_distance: bool
    """ Use a maximum distance for the field to work

    :type: bool
    """

    use_min_distance: bool
    """ Use a minimum distance for the field's fall-off

    :type: bool
    """

    use_multiple_springs: bool
    """ Every point is effected by multiple springs

    :type: bool
    """

    use_object_coords: bool
    """ Use object/global coordinates for texture

    :type: bool
    """

    use_radial_max: bool
    """ Use a maximum radial distance for the field to work

    :type: bool
    """

    use_radial_min: bool
    """ Use a minimum radial distance for the field's fall-off

    :type: bool
    """

    use_root_coords: bool
    """ Texture coordinates from root particle locations

    :type: bool
    """

    use_smoke_density: bool
    """ Adjust force strength based on smoke density

    :type: bool
    """

    z_direction: str
    """ Effect in full or only positive/negative Z direction

    :type: str
    """

class FileBrowserFSMenuEntry(bpy_struct):
    """File Select Parameters"""

    is_valid: bool
    """ Whether this path is currently reachable

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    path: str
    """ 

    :type: str
    """

    use_save: bool
    """ Whether this path is saved in bookmarks, or generated from OS

    :type: bool
    """

class FileSelectParams(bpy_struct):
    """File Select Parameters"""

    directory: str
    """ Directory displayed in the file browser

    :type: str
    """

    display_size: str
    """ Change the size of the display (width of columns or thumbnails size)

    :type: str
    """

    display_type: str
    """ Display mode for the file list

    :type: str
    """

    filename: str
    """ Active file in the file browser

    :type: str
    """

    filter_glob: str
    """ 

    :type: str
    """

    filter_id: set[str]
    """ Which ID types to show/hide, when browsing a library

    :type: set[str]
    """

    filter_id_category: set[str]
    """ Which ID categories to show/hide, when browsing a library

    :type: set[str]
    """

    filter_search: str
    """ Filter by name, supports '*' wildcard

    :type: str
    """

    recursion_level: str
    """ Numbers of dirtree levels to show simultaneously

    :type: str
    """

    show_hidden: bool
    """ Show hidden dot files

    :type: bool
    """

    sort_method: str
    """ 

    :type: str
    """

    title: str
    """ Title for the file browser

    :type: str
    """

    use_filter: bool
    """ Enable filtering of files

    :type: bool
    """

    use_filter_backup: bool
    """ Show .blend1, .blend2, etc. files

    :type: bool
    """

    use_filter_blender: bool
    """ Show .blend files

    :type: bool
    """

    use_filter_blendid: bool
    """ Show .blend files items (objects, materials, etc.)

    :type: bool
    """

    use_filter_folder: bool
    """ Show folders

    :type: bool
    """

    use_filter_font: bool
    """ Show font files

    :type: bool
    """

    use_filter_image: bool
    """ Show image files

    :type: bool
    """

    use_filter_movie: bool
    """ Show movie files

    :type: bool
    """

    use_filter_script: bool
    """ Show script files

    :type: bool
    """

    use_filter_sound: bool
    """ Show sound files

    :type: bool
    """

    use_filter_text: bool
    """ Show text files

    :type: bool
    """

    use_library_browsing: bool
    """ Whether we may browse blender files' content or not

    :type: bool
    """

class FluidSettings(bpy_struct):
    """Fluid simulation settings for an object taking part in the simulation"""

    type: str
    """ Type of participation in the fluid simulation

    :type: str
    """

class FluidVertexVelocity(bpy_struct):
    """Velocity of a simulated fluid mesh"""

    velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class FreestyleLineSet(bpy_struct):
    """Line set for associating lines and style parameters"""

    edge_type_combination: str
    """ Specify a logical combination of selection conditions on feature edge types

    :type: str
    """

    edge_type_negation: str
    """ Specify either inclusion or exclusion of feature edges selected by edge types

    :type: str
    """

    exclude_border: bool
    """ Exclude border edges

    :type: bool
    """

    exclude_contour: bool
    """ Exclude contours

    :type: bool
    """

    exclude_crease: bool
    """ Exclude crease edges

    :type: bool
    """

    exclude_edge_mark: bool
    """ Exclude edge marks

    :type: bool
    """

    exclude_external_contour: bool
    """ Exclude external contours

    :type: bool
    """

    exclude_material_boundary: bool
    """ Exclude edges at material boundaries

    :type: bool
    """

    exclude_ridge_valley: bool
    """ Exclude ridges and valleys

    :type: bool
    """

    exclude_silhouette: bool
    """ Exclude silhouette edges

    :type: bool
    """

    exclude_suggestive_contour: bool
    """ Exclude suggestive contours

    :type: bool
    """

    face_mark_condition: str
    """ Specify a feature edge selection condition based on face marks

    :type: str
    """

    face_mark_negation: str
    """ Specify either inclusion or exclusion of feature edges selected by face marks

    :type: str
    """

    group: Group
    """ A group of objects based on which feature edges are selected

    :type: Group
    """

    group_negation: str
    """ Specify either inclusion or exclusion of feature edges belonging to a group of objects

    :type: str
    """

    linestyle: FreestyleLineStyle
    """ Line style settings

    :type: FreestyleLineStyle
    """

    name: str
    """ Line set name

    :type: str
    """

    qi_end: int
    """ Last QI value of the QI range

    :type: int
    """

    qi_start: int
    """ First QI value of the QI range

    :type: int
    """

    select_border: bool
    """ Select border edges (open mesh edges)

    :type: bool
    """

    select_by_edge_types: bool
    """ Select feature edges based on edge types

    :type: bool
    """

    select_by_face_marks: bool
    """ Select feature edges by face marks

    :type: bool
    """

    select_by_group: bool
    """ Select feature edges based on a group of objects

    :type: bool
    """

    select_by_image_border: bool
    """ Select feature edges by image border (less memory consumption)

    :type: bool
    """

    select_by_visibility: bool
    """ Select feature edges based on visibility

    :type: bool
    """

    select_contour: bool
    """ Select contours (outer silhouettes of each object)

    :type: bool
    """

    select_crease: bool
    """ Select crease edges (those between two faces making an angle smaller than the Crease Angle)

    :type: bool
    """

    select_edge_mark: bool
    """ Select edge marks (edges annotated by Freestyle edge marks)

    :type: bool
    """

    select_external_contour: bool
    """ Select external contours (outer silhouettes of occluding and occluded objects)

    :type: bool
    """

    select_material_boundary: bool
    """ Select edges at material boundaries

    :type: bool
    """

    select_ridge_valley: bool
    """ Select ridges and valleys (boundary lines between convex and concave areas of surface)

    :type: bool
    """

    select_silhouette: bool
    """ Select silhouettes (edges at the boundary of visible and hidden faces)

    :type: bool
    """

    select_suggestive_contour: bool
    """ Select suggestive contours (almost silhouette/contour edges)

    :type: bool
    """

    show_render: bool
    """ Enable or disable this line set during stroke rendering

    :type: bool
    """

    visibility: str
    """ Determine how to use visibility for feature edge selection

    :type: str
    """

class FreestyleModuleSettings(bpy_struct):
    """Style module configuration for specifying a style module"""

    script: Text
    """ Python script to define a style module

    :type: Text
    """

    use: bool
    """ Enable or disable this style module during stroke rendering

    :type: bool
    """

class FreestyleModules(bpy_prop_collection[FreestyleModuleSettings], bpy_struct):
    """A list of style modules (to be applied from top to bottom)"""

    def new(self) -> FreestyleModuleSettings:
        """Add a style module to scene render layer Freestyle settings

        :return: Newly created style module
        :rtype: FreestyleModuleSettings
        """
        ...

    def remove(self, module: FreestyleModuleSettings):
        """Remove a style module from scene render layer Freestyle settings

        :param module: Style module to remove
        :type module: FreestyleModuleSettings
        """
        ...

class FreestyleSettings(bpy_struct):
    """Freestyle settings for a SceneRenderLayer data-block"""

    crease_angle: float
    """ Angular threshold for detecting crease edges

    :type: float
    """

    kr_derivative_epsilon: float
    """ Kr derivative epsilon for computing suggestive contours

    :type: float
    """

    linesets: Linesets
    """ 

    :type: Linesets
    """

    mode: str
    """ Select the Freestyle control mode

    :type: str
    """

    modules: FreestyleModules
    """ A list of style modules (to be applied from top to bottom)

    :type: FreestyleModules
    """

    sphere_radius: float
    """ Sphere radius for computing curvatures

    :type: float
    """

    use_advanced_options: bool
    """ Enable advanced edge detection options (sphere radius and Kr derivative epsilon)

    :type: bool
    """

    use_culling: bool
    """ If enabled, out-of-view edges are ignored

    :type: bool
    """

    use_material_boundaries: bool
    """ Enable material boundaries

    :type: bool
    """

    use_ridges_and_valleys: bool
    """ Enable ridges and valleys

    :type: bool
    """

    use_smoothness: bool
    """ Take face smoothness into account in view map calculation

    :type: bool
    """

    use_suggestive_contours: bool
    """ Enable suggestive contours

    :type: bool
    """

    use_view_map_cache: bool
    """ Keep the computed view map and avoid re-calculating it if mesh geometry is unchanged

    :type: bool
    """

class Function(bpy_struct):
    """RNA function definition"""

    description: str
    """ Description of the Function's purpose

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    is_registered: bool
    """ Function is registered as callback as part of type registration

    :type: bool
    """

    is_registered_optional: bool
    """ Function is optionally registered as callback part of type registration

    :type: bool
    """

    parameters: bpy_prop_collection[Property]
    """ Parameters for the function

    :type: bpy_prop_collection[Property]
    """

    use_self: bool
    """ Function does not pass its self as an argument (becomes a static method in python)

    :type: bool
    """

    use_self_type: bool
    """ Function passes its self type as an argument (becomes a class method in python if use_self is false)

    :type: bool
    """

class GPUDOFSettings(bpy_struct):
    """Settings for GPU based depth of field"""

    blades: int
    """ Blades for dof effect

    :type: int
    """

    focal_length: float
    """ Focal length for dof effect

    :type: float
    """

    focus_distance: float
    """ Viewport depth of field focus distance

    :type: float
    """

    fstop: float
    """ F-stop for dof effect

    :type: float
    """

    is_hq_supported: bool
    """ Use high quality depth of field

    :type: bool
    """

    sensor: float
    """ Size of sensor

    :type: float
    """

    use_high_quality: bool
    """ Use high quality depth of field

    :type: bool
    """

class GPUFXSettings(bpy_struct):
    """Settings for GPU based compositing"""

    dof: GPUDOFSettings
    """ 

    :type: GPUDOFSettings
    """

    ssao: GPUSSAOSettings
    """ 

    :type: GPUSSAOSettings
    """

    use_dof: bool
    """ Use depth of field on viewport using the values from active camera

    :type: bool
    """

    use_ssao: bool
    """ Use screen space ambient occlusion of field on viewport

    :type: bool
    """

class GPUSSAOSettings(bpy_struct):
    """Settings for GPU based screen space ambient occlusion"""

    attenuation: float
    """ Attenuation constant

    :type: float
    """

    color: bpy_prop_array[float]
    """ Color for screen space ambient occlusion effect

    :type: bpy_prop_array[float]
    """

    distance_max: float
    """ Distance of object that contribute to the SSAO effect

    :type: float
    """

    factor: float
    """ Strength of the SSAO effect

    :type: float
    """

    samples: int
    """ Number of samples

    :type: int
    """

class GPencilBrush(bpy_struct):
    """Collection of brushes being used to control the line style of new strokes"""

    angle: float
    """ Direction of the stroke at which brush gives maximal thickness (0° for horizontal)

    :type: float
    """

    angle_factor: float
    """ Reduce brush thickness by this factor when stroke is perpendicular to 'Angle' direction

    :type: float
    """

    curve_jitter: CurveMapping
    """ Curve used for the jitter effect

    :type: CurveMapping
    """

    curve_sensitivity: CurveMapping
    """ Curve used for the sensitivity

    :type: CurveMapping
    """

    curve_strength: CurveMapping
    """ Curve used for the strength

    :type: CurveMapping
    """

    jitter: float
    """ Jitter factor for new strokes

    :type: float
    """

    line_width: int
    """ Thickness of strokes (in pixels)

    :type: int
    """

    name: str
    """ Brush name

    :type: str
    """

    pen_sensitivity_factor: float
    """ Pressure sensitivity factor for new strokes

    :type: float
    """

    pen_smooth_factor: float
    """ Amount of smoothing to apply to newly created strokes, to reduce jitter/noise

    :type: float
    """

    pen_smooth_steps: int
    """ Number of times to smooth newly created strokes

    :type: int
    """

    pen_subdivision_steps: int
    """ Number of times to subdivide newly created strokes, for less jagged strokes

    :type: int
    """

    random_press: float
    """ Randomness factor for pressure and strength in new strokes

    :type: float
    """

    random_subdiv: float
    """ Randomness factor for new strokes after subdivision

    :type: float
    """

    strength: float
    """ Color strength for new strokes (affect alpha factor of color)

    :type: float
    """

    use_jitter_pressure: bool
    """ Use tablet pressure for jitter

    :type: bool
    """

    use_pressure: bool
    """ Use tablet pressure

    :type: bool
    """

    use_random_pressure: bool
    """ Use random value for pressure

    :type: bool
    """

    use_random_strength: bool
    """ Use random value for strength

    :type: bool
    """

    use_strength_pressure: bool
    """ Use tablet pressure for color strength

    :type: bool
    """

class GPencilFrame(bpy_struct):
    """Collection of related sketches on a particular frame"""

    frame_number: int
    """ The frame on which this sketch appears

    :type: int
    """

    is_edited: bool
    """ Frame is being edited (painted on)

    :type: bool
    """

    select: bool
    """ Frame is selected for editing in the Dope Sheet

    :type: bool
    """

    strokes: GPencilStrokes
    """ Freehand curves defining the sketch on this frame

    :type: GPencilStrokes
    """

    def clear(self):
        """Remove all the grease pencil frame data"""
        ...

class GPencilFrames(bpy_prop_collection[GPencilFrame], bpy_struct):
    """Collection of grease pencil frames"""

    def new(self, frame_number: int | None) -> GPencilFrame:
        """Add a new grease pencil frame

        :param frame_number: Frame Number, The frame on which this sketch appears
        :type frame_number: int | None
        :return: The newly created frame
        :rtype: GPencilFrame
        """
        ...

    def remove(self, frame: GPencilFrame):
        """Remove a grease pencil frame

        :param frame: Frame, The frame to remove
        :type frame: GPencilFrame
        """
        ...

    def copy(self, source: GPencilFrame) -> GPencilFrame:
        """Copy a grease pencil frame

        :param source: Source, The source frame
        :type source: GPencilFrame
        :return: The newly copied frame
        :rtype: GPencilFrame
        """
        ...

class GPencilLayer(bpy_struct):
    """Collection of related sketches"""

    active_frame: GPencilFrame
    """ Frame currently being displayed for this layer

    :type: GPencilFrame
    """

    after_color: bpy_prop_array[float]
    """ Base color for ghosts after the active frame

    :type: bpy_prop_array[float]
    """

    before_color: bpy_prop_array[float]
    """ Base color for ghosts before the active frame

    :type: bpy_prop_array[float]
    """

    frames: GPencilFrames
    """ Sketches for this layer on different frames

    :type: GPencilFrames
    """

    ghost_after_range: int
    """ Maximum number of frames to show after current frame (0 = show only the next sketch, -1 = don't show any frames after current)

    :type: int
    """

    ghost_before_range: int
    """ Maximum number of frames to show before current frame (0 = show only the previous sketch, -1 = don't show any frames before current)

    :type: int
    """

    hide: bool
    """ Set layer Visibility

    :type: bool
    """

    info: str
    """ Layer name

    :type: str
    """

    is_parented: bool
    """ True when the layer parent object is set

    :type: bool
    """

    line_change: int
    """ Thickness change to apply to current strokes (in pixels)

    :type: int
    """

    lock: bool
    """ Protect layer from further editing and/or frame changes

    :type: bool
    """

    lock_frame: bool
    """ Lock current frame displayed by layer

    :type: bool
    """

    matrix_inverse: bpy_prop_array[float]
    """ Parent inverse transformation matrix

    :type: bpy_prop_array[float]
    """

    opacity: float
    """ Layer Opacity

    :type: float
    """

    parent: Object
    """ Parent Object

    :type: Object
    """

    parent_bone: str
    """ Name of parent bone in case of a bone parenting relation

    :type: str
    """

    parent_type: str
    """ Type of parent relation

    :type: str
    """

    select: bool
    """ Layer is selected for editing in the Dope Sheet

    :type: bool
    """

    show_points: bool
    """ Draw the points which make up the strokes (for debugging purposes)

    :type: bool
    """

    show_x_ray: bool
    """ Make the layer draw in front of objects

    :type: bool
    """

    tint_color: bpy_prop_array[float]
    """ Color for tinting stroke colors

    :type: bpy_prop_array[float]
    """

    tint_factor: float
    """ Factor of tinting color

    :type: float
    """

    unlock_color: bool
    """ Unprotect selected colors from further editing and/or frame changes

    :type: bool
    """

    use_ghost_custom_colors: bool
    """ Use custom colors for ghost frames

    :type: bool
    """

    use_onion_skinning: bool
    """ Ghost frames on either side of frame

    :type: bool
    """

    use_volumetric_strokes: bool
    """ Draw strokes as a series of circular blobs, resulting in a volumetric effect

    :type: bool
    """

    def clear(self):
        """Remove all the grease pencil layer data"""
        ...

class GPencilPalette(bpy_struct):
    """Collection of related palettes"""

    colors: GPencilPaletteColors
    """ Colors of the palette

    :type: GPencilPaletteColors
    """

    name: str
    """ Palette name

    :type: str
    """

class GPencilPaletteColor(bpy_struct):
    """Collection of related colors"""

    alpha: float
    """ Color Opacity

    :type: float
    """

    color: bpy_prop_array[float]
    """ Color for strokes

    :type: bpy_prop_array[float]
    """

    fill_alpha: float
    """ Opacity for filling region bounded by each stroke

    :type: float
    """

    fill_color: bpy_prop_array[float]
    """ Color for filling region bounded by each stroke

    :type: bpy_prop_array[float]
    """

    ghost: bool
    """ Display the color in onion skinning

    :type: bool
    """

    hide: bool
    """ Set color Visibility

    :type: bool
    """

    is_fill_visible: bool
    """ True when opacity of fill is set high enough to be visible

    :type: bool
    """

    is_stroke_visible: bool
    """ True when opacity of stroke is set high enough to be visible

    :type: bool
    """

    lock: bool
    """ Protect color from further editing and/or frame changes

    :type: bool
    """

    name: str
    """ Color name

    :type: str
    """

    use_hq_fill: bool
    """ Fill strokes using high quality to avoid glitches (slower fps during animation play)

    :type: bool
    """

    use_volumetric_strokes: bool
    """ Draw strokes as a series of circular blobs, resulting in a volumetric effect

    :type: bool
    """

class GPencilPaletteColors(bpy_prop_collection[GPencilPaletteColor], bpy_struct):
    """Collection of palette colors"""

    active: GPencilPaletteColor | None
    """ Current active color

    :type: GPencilPaletteColor | None
    """

    active_index: int | None
    """ Index of active palette color

    :type: int | None
    """

    def new(self) -> GPencilPaletteColor:
        """Add a new color to the palette

        :return: The newly created color
        :rtype: GPencilPaletteColor
        """
        ...

    def remove(self, color: GPencilPaletteColor):
        """Remove a color from the palette

        :param color: The color to remove
        :type color: GPencilPaletteColor
        """
        ...

class GPencilSculptBrush(bpy_struct):
    """Stroke editing brush"""

    affect_pressure: bool
    """ Affect pressure values as well when smoothing strokes

    :type: bool
    """

    direction: str
    """ 

    :type: str
    """

    size: int
    """ Radius of the brush in pixels

    :type: int
    """

    strength: float
    """ Brush strength

    :type: float
    """

    use_falloff: bool
    """ Strength of brush decays with distance from cursor

    :type: bool
    """

    use_pressure_strength: bool
    """ Enable tablet pressure sensitivity for strength

    :type: bool
    """

class GPencilSculptSettings(bpy_struct):
    """Properties for Grease Pencil stroke sculpting tool"""

    affect_position: bool
    """ The brush affects the position of the point

    :type: bool
    """

    affect_strength: bool
    """ The brush affects the color strength of the point

    :type: bool
    """

    affect_thickness: bool
    """ The brush affects the thickness of the point

    :type: bool
    """

    brush: GPencilSculptBrush
    """ 

    :type: GPencilSculptBrush
    """

    selection_alpha: float
    """ Alpha value for selected vertices

    :type: float
    """

    tool: str
    """ 

    :type: str
    """

    use_select_mask: bool
    """ Only sculpt selected stroke points

    :type: bool
    """

class GPencilStroke(bpy_struct):
    """Freehand curve defining part of a sketch"""

    color: GPencilPaletteColor
    """ Color from palette used in Stroke

    :type: GPencilPaletteColor
    """

    colorname: str
    """ Palette color name

    :type: str
    """

    draw_cyclic: bool
    """ Enable cyclic drawing, closing the stroke

    :type: bool
    """

    draw_mode: str
    """ 

    :type: str
    """

    line_width: int
    """ Thickness of stroke (in pixels)

    :type: int
    """

    points: GPencilStrokePoints
    """ Stroke data points

    :type: GPencilStrokePoints
    """

    select: bool
    """ Stroke is selected for viewport editing

    :type: bool
    """

    triangles: bpy_prop_collection[GPencilTriangle]
    """ Triangulation data for HQ fill

    :type: bpy_prop_collection[GPencilTriangle]
    """

class GPencilStrokePoint(bpy_struct):
    """Data point for freehand stroke curve"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    pressure: float
    """ Pressure of tablet at point when drawing it

    :type: float
    """

    select: bool
    """ Point is selected for viewport editing

    :type: bool
    """

    strength: float
    """ Color intensity (alpha factor)

    :type: float
    """

class GPencilStrokePoints(bpy_prop_collection[GPencilStrokePoint], bpy_struct):
    """Collection of grease pencil stroke points"""

    def add(
        self,
        count: typing.Any | None = 1,
        pressure: typing.Any | None = 1.0,
        strength: typing.Any | None = 1.0,
    ):
        """Add a new grease pencil stroke point

        :param count: Number, Number of points to add to the stroke
        :type count: typing.Any | None
        :param pressure: Pressure, Pressure for newly created points
        :type pressure: typing.Any | None
        :param strength: Strength, Color intensity (alpha factor) for newly created points
        :type strength: typing.Any | None
        """
        ...

    def pop(self, index: typing.Any | None = -1):
        """Remove a grease pencil stroke point

        :param index: Index, point index
        :type index: typing.Any | None
        """
        ...

class GPencilStrokes(bpy_prop_collection[GPencilStroke], bpy_struct):
    """Collection of grease pencil stroke"""

    def new(self, colorname: str | typing.Any = "") -> GPencilStroke:
        """Add a new grease pencil stroke

        :param colorname: Color, Name of the color
        :type colorname: str | typing.Any
        :return: The newly created stroke
        :rtype: GPencilStroke
        """
        ...

    def remove(self, stroke: GPencilStroke):
        """Remove a grease pencil stroke

        :param stroke: Stroke, The stroke to remove
        :type stroke: GPencilStroke
        """
        ...

class GPencilTriangle(bpy_struct):
    """Triangulation data for HQ fill"""

    v1: int
    """ First triangle vertex index

    :type: int
    """

    v2: int
    """ Second triangle vertex index

    :type: int
    """

    v3: int
    """ Third triangle vertex index

    :type: int
    """

class GameObjectSettings(bpy_struct):
    """Game engine related settings for the object"""

    actuators: bpy_prop_collection[Actuator]
    """ Game engine actuators to act on events

    :type: bpy_prop_collection[Actuator]
    """

    angular_velocity_max: float
    """ Clamp angular velocity to this maximum speed, in angle per second

    :type: float
    """

    angular_velocity_min: float
    """ Clamp angular velocity to this minimum speed (except when totally still), in angle per second

    :type: float
    """

    collision_bounds_type: str
    """ Select the collision shape that better fits the object

    :type: str
    """

    collision_group: list[bool]
    """ The collision group of the object

    :type: list[bool]
    """

    collision_margin: float
    """ Extra margin around object for collision detection, small amount required for stability

    :type: float
    """

    collision_mask: list[bool]
    """ The groups this object can collide with

    :type: list[bool]
    """

    controllers: bpy_prop_collection[Controller]
    """ Game engine controllers to process events, connecting sensors to actuators

    :type: bpy_prop_collection[Controller]
    """

    damping: float
    """ General movement damping

    :type: float
    """

    fall_speed: float
    """ Maximum speed at which the character will fall

    :type: float
    """

    form_factor: float
    """ Form factor scales the inertia tensor

    :type: float
    """

    friction_coefficients: bpy_prop_array[float]
    """ Relative friction coefficients in the in the X, Y and Z directions, when anisotropic friction is enabled

    :type: bpy_prop_array[float]
    """

    jump_max: int
    """ The maximum number of jumps the character can make before it hits the ground

    :type: int
    """

    jump_speed: float
    """ Upward velocity applied to the character when jumping

    :type: float
    """

    lock_location_x: bool
    """ Disable simulation of linear motion along the X axis

    :type: bool
    """

    lock_location_y: bool
    """ Disable simulation of linear motion along the Y axis

    :type: bool
    """

    lock_location_z: bool
    """ Disable simulation of linear motion along the Z axis

    :type: bool
    """

    lock_rotation_x: bool
    """ Disable simulation of angular motion along the X axis

    :type: bool
    """

    lock_rotation_y: bool
    """ Disable simulation of angular motion along the Y axis

    :type: bool
    """

    lock_rotation_z: bool
    """ Disable simulation of angular motion along the Z axis

    :type: bool
    """

    mass: float
    """ Mass of the object

    :type: float
    """

    obstacle_radius: float
    """ Radius of object representation in obstacle simulation

    :type: float
    """

    physics_type: str
    """ Select the type of physical representation

    :type: str
    """

    properties: bpy_prop_collection[GameProperty]
    """ Game engine properties

    :type: bpy_prop_collection[GameProperty]
    """

    radius: float
    """ Radius of bounding sphere and material physics

    :type: float
    """

    rotation_damping: float
    """ General rotation damping

    :type: float
    """

    sensors: bpy_prop_collection[Sensor]
    """ Game engine sensor to detect events

    :type: bpy_prop_collection[Sensor]
    """

    show_actuators: bool
    """ Shows actuators for this object in the user interface

    :type: bool
    """

    show_controllers: bool
    """ Shows controllers for this object in the user interface

    :type: bool
    """

    show_debug_state: bool
    """ Print state debug info in the game engine

    :type: bool
    """

    show_sensors: bool
    """ Shows sensors for this object in the user interface

    :type: bool
    """

    show_state_panel: bool
    """ Show state panel

    :type: bool
    """

    soft_body: GameSoftBodySettings
    """ Settings for Bullet soft body simulation

    :type: GameSoftBodySettings
    """

    states_initial: list[bool]
    """ Initial state when the game starts

    :type: list[bool]
    """

    states_visible: list[bool]
    """ State determining which controllers are displayed

    :type: list[bool]
    """

    step_height: float
    """ Maximum height of steps the character can run over

    :type: float
    """

    use_activity_culling: bool
    """ Disable simulation of angular motion along the Z axis

    :type: bool
    """

    use_actor: bool
    """ Object is detected by the Near and Radar sensor

    :type: bool
    """

    use_all_states: bool
    """ Set all state bits

    :type: bool
    """

    use_anisotropic_friction: bool
    """ Enable anisotropic friction

    :type: bool
    """

    use_collision_bounds: bool
    """ Specify a collision bounds type other than the default

    :type: bool
    """

    use_collision_compound: bool
    """ Add children to form a compound collision object

    :type: bool
    """

    use_ghost: bool
    """ Object does not react to collisions, like a ghost

    :type: bool
    """

    use_material_physics_fh: bool
    """ React to force field physics settings in materials

    :type: bool
    """

    use_obstacle_create: bool
    """ Create representation for obstacle simulation

    :type: bool
    """

    use_record_animation: bool
    """ Record animation objects without physics

    :type: bool
    """

    use_rotate_from_normal: bool
    """ Use face normal to rotate object, so that it points away from the surface

    :type: bool
    """

    use_sleep: bool
    """ Disable auto (de)activation in physics simulation

    :type: bool
    """

    used_states: list[bool]
    """ States which are being used by controllers

    :type: list[bool]
    """

    velocity_max: float
    """ Clamp velocity to this maximum speed, in distance per second

    :type: float
    """

    velocity_min: float
    """ Clamp velocity to this minimum speed (except when totally still), in distance per second

    :type: float
    """

class GameProperty(bpy_struct):
    """Game engine user defined object property"""

    name: str
    """ Available as GameObject attributes in the game engine's python API

    :type: str
    """

    show_debug: bool
    """ Print debug information for this property

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

class GameSoftBodySettings(bpy_struct):
    """Soft body simulation settings for an object in the game engine"""

    cluster_iterations: int
    """ Number of cluster iterations

    :type: int
    """

    collision_margin: float
    """ Collision margin for soft body. Small value makes the algorithm unstable

    :type: float
    """

    dynamic_friction: float
    """ Dynamic Friction

    :type: float
    """

    linear_stiffness: float
    """ Linear stiffness of the soft body links

    :type: float
    """

    location_iterations: int
    """ Position solver iterations

    :type: int
    """

    shape_threshold: float
    """ Shape matching threshold

    :type: float
    """

    use_bending_constraints: bool
    """ Enable bending constraints

    :type: bool
    """

    use_cluster_rigid_to_softbody: bool
    """ Enable cluster collision between soft and rigid body

    :type: bool
    """

    use_cluster_soft_to_softbody: bool
    """ Enable cluster collision between soft and soft body

    :type: bool
    """

    use_shape_match: bool
    """ Enable soft body shape matching goal

    :type: bool
    """

    weld_threshold: float
    """ Welding threshold: distance between nearby vertices to be considered equal => set to 0.0 to disable welding test and speed up scene loading (ok if the mesh has no duplicates)

    :type: float
    """

class GreasePencilBrushes(bpy_prop_collection[GPencilBrush], bpy_struct):
    """Collection of grease pencil brushes"""

    active: GPencilBrush | None
    """ Current active brush

    :type: GPencilBrush | None
    """

    active_index: int | None
    """ Index of active brush

    :type: int | None
    """

    def new(
        self, name: str | typing.Any, set_active: bool | typing.Any | None = False
    ) -> GPencilBrush:
        """Add a new grease pencil brush

        :param name: Name, Name of the brush
        :type name: str | typing.Any
        :param set_active: Set Active, Set the newly created brush to the active brush
        :type set_active: bool | typing.Any | None
        :return: The newly created brush
        :rtype: GPencilBrush
        """
        ...

    def remove(self, brush: GPencilBrush):
        """Remove a grease pencil brush

        :param brush: The brush to remove
        :type brush: GPencilBrush
        """
        ...

class GreasePencilLayers(bpy_prop_collection[GPencilLayer], bpy_struct):
    """Collection of grease pencil layers"""

    active: GPencilLayer | None
    """ Active grease pencil layer

    :type: GPencilLayer | None
    """

    active_index: int | None
    """ Index of active grease pencil layer

    :type: int | None
    """

    def new(
        self, name: str | typing.Any, set_active: bool | typing.Any | None = False
    ) -> GPencilLayer:
        """Add a new grease pencil layer

        :param name: Name, Name of the layer
        :type name: str | typing.Any
        :param set_active: Set Active, Set the newly created layer to the active layer
        :type set_active: bool | typing.Any | None
        :return: The newly created layer
        :rtype: GPencilLayer
        """
        ...

    def remove(self, layer: GPencilLayer):
        """Remove a grease pencil layer

        :param layer: The layer to remove
        :type layer: GPencilLayer
        """
        ...

class GreasePencilPalettes(bpy_prop_collection[GPencilPalette], bpy_struct):
    """Collection of grease pencil palettes"""

    active: GPencilPalette | None
    """ Current active palette

    :type: GPencilPalette | None
    """

    active_index: int | None
    """ Index of active palette

    :type: int | None
    """

    def new(
        self, name: str | typing.Any, set_active: bool | typing.Any | None = False
    ) -> GPencilPalette:
        """Add a new grease pencil palette

        :param name: Name, Name of the palette
        :type name: str | typing.Any
        :param set_active: Set Active, Activate the newly created palette
        :type set_active: bool | typing.Any | None
        :return: The newly created palette
        :rtype: GPencilPalette
        """
        ...

    def remove(self, palette: GPencilPalette):
        """Remove a grease pencil palette

        :param palette: The palette to remove
        :type palette: GPencilPalette
        """
        ...

class GroupObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of group objects"""

    def link(self, object: Object):
        """Add this object to a group

        :param object: Object to add
        :type object: Object
        """
        ...

    def unlink(self, object: Object | None):
        """Remove this object to a group

        :param object: Object to remove
        :type object: Object | None
        """
        ...

class Header(bpy_struct):
    """Editor header containing UI elements"""

    bl_idname: str
    """ If this is set, the header gets a custom ID, otherwise it takes the name of the class used to define the panel; for example, if the class name is "OBJECT_HT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_HT_hello"

    :type: str
    """

    bl_space_type: str
    """ The space where the header is going to be used in

    :type: str
    """

    layout: UILayout
    """ Structure of the header in the UI

    :type: UILayout
    """

    def draw(self, context: Context | None):
        """Draw UI elements into the header UI layout

        :param context:
        :type context: Context | None
        """
        ...

class Histogram(bpy_struct):
    """Statistical view of the levels of color in an image"""

    mode: str
    """ Channels to display when drawing the histogram

    :type: str
    """

    show_line: bool
    """ Display lines rather than filled shapes

    :type: bool
    """

class ID(bpy_struct):
    """Base type for data-blocks, defining a unique name, linking from other libraries and garbage collection"""

    is_library_indirect: bool
    """ Is this ID block linked indirectly

    :type: bool
    """

    is_updated: bool
    """ Datablock is tagged for recalculation

    :type: bool
    """

    is_updated_data: bool
    """ Datablock data is tagged for recalculation

    :type: bool
    """

    library: Library
    """ Library file the data-block is linked from

    :type: Library
    """

    name: str
    """ Unique data-block ID name

    :type: str
    """

    preview: ImagePreview
    """ Preview image and icon of this data-block (None if not supported for this type of data)

    :type: ImagePreview
    """

    tag: bool
    """ Tools can use this to tag data for their own purposes (initial state is undefined)

    :type: bool
    """

    use_fake_user: bool
    """ Save this data-block even if it has no users

    :type: bool
    """

    users: int
    """ Number of times this data-block is referenced

    :type: int
    """

    def copy(self) -> ID:
        """Create a copy of this data-block (not supported for all data-blocks)

        :return: New copy of the ID
        :rtype: ID
        """
        ...

    def user_clear(self):
        """Clear the user count of a data-block so its not saved, on reload the data will be removedThis function is for advanced use only, misuse can crash blender since the user
        count is used to prevent data being removed when it is used.

        """
        ...

    def user_remap(self, new_id: ID):
        """Replace all usage in the .blend file of this ID by new given one

        :param new_id: New ID to use
        :type new_id: ID
        """
        ...

    def user_of_id(self, id: ID) -> int:
        """Count the number of times that ID uses/references given one

        :param id: ID to count usages
        :type id: ID
        :return: Number of usages/references of given id by current datablock
        :rtype: int
        """
        ...

    def animation_data_create(self) -> AnimData:
        """Create animation data to this ID, note that not all ID types support this

        :return: New animation data or NULL
        :rtype: AnimData
        """
        ...

    def animation_data_clear(self):
        """Clear animation on this this ID"""
        ...

    def update_tag(self, refresh: set[str] | None = {}):
        """Tag the ID to update its display data, e.g. when calling `bpy.types.Scene.update`

        :param refresh: Type of updates to perform
        :type refresh: set[str] | None
        """
        ...

class IDMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    def append(self, material: Material | None):
        """Add a new material to the data block

        :param material: Material to add
        :type material: Material | None
        """
        ...

    def pop(
        self,
        index: typing.Any | None = -1,
        update_data: bool | typing.Any | None = False,
    ) -> Material:
        """Remove a material from the data block

        :param index: Index of material to remove
        :type index: typing.Any | None
        :param update_data: Update data by re-adjusting the material slots assigned
        :type update_data: bool | typing.Any | None
        :return: Material to remove
        :rtype: Material
        """
        ...

    def clear(self, update_data: bool | typing.Any | None = False):
        """Remove all materials from the data block

        :param update_data: Update data by re-adjusting the material slots assigned
        :type update_data: bool | typing.Any | None
        """
        ...

class IKParam(bpy_struct):
    """Base type for IK solver parameters"""

    ik_solver: str
    """ IK solver for which these parameters are defined

    :type: str
    """

class ImageFormatSettings(bpy_struct):
    """Settings for image formats"""

    cineon_black: int
    """ Log conversion reference blackpoint

    :type: int
    """

    cineon_gamma: float
    """ Log conversion gamma

    :type: float
    """

    cineon_white: int
    """ Log conversion reference whitepoint

    :type: int
    """

    color_depth: str
    """ Bit depth per channel

    :type: str
    """

    color_mode: str
    """ Choose BW for saving grayscale images, RGB for saving red, green and blue channels, and RGBA for saving red, green, blue and alpha channels

    :type: str
    """

    compression: int
    """ Amount of time to determine best compression: 0 = no compression with fast file output, 100 = maximum lossless compression with slow file output

    :type: int
    """

    display_settings: ColorManagedDisplaySettings
    """ Settings of device saved image would be displayed on

    :type: ColorManagedDisplaySettings
    """

    exr_codec: str
    """ Codec settings for OpenEXR

    :type: str
    """

    file_format: str
    """ File format to save the rendered images as

    :type: str
    """

    jpeg2k_codec: str
    """ Codec settings for Jpek2000

    :type: str
    """

    quality: int
    """ Quality for image formats that support lossy compression

    :type: int
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    tiff_codec: str
    """ Compression mode for TIFF

    :type: str
    """

    use_cineon_log: bool
    """ Convert to logarithmic color space

    :type: bool
    """

    use_jpeg2k_cinema_48: bool
    """ Use Openjpeg Cinema Preset (48fps)

    :type: bool
    """

    use_jpeg2k_cinema_preset: bool
    """ Use Openjpeg Cinema Preset

    :type: bool
    """

    use_jpeg2k_ycc: bool
    """ Save luminance-chrominance-chrominance channels instead of RGB colors

    :type: bool
    """

    use_preview: bool
    """ When rendering animations, save JPG preview images in same directory

    :type: bool
    """

    use_zbuffer: bool
    """ Save the z-depth per pixel (32 bit unsigned int z-buffer)

    :type: bool
    """

    view_settings: ColorManagedViewSettings
    """ Color management settings applied on image before saving

    :type: ColorManagedViewSettings
    """

    views_format: str
    """ Format of multiview media

    :type: str
    """

class ImagePackedFile(bpy_struct):
    filepath: str
    """ 

    :type: str
    """

    packed_file: PackedFile
    """ 

    :type: PackedFile
    """

    def save(self):
        """Save the packed file to its filepath"""
        ...

class ImagePreview(bpy_struct):
    """Preview image and icon"""

    icon_id: int
    """ Unique integer identifying this preview as an icon (zero means invalid)

    :type: int
    """

    icon_pixels: int
    """ Icon pixels, as bytes (always RGBA 32bits)

    :type: int
    """

    icon_pixels_float: float
    """ Icon pixels components, as floats (RGBA concatenated values)

    :type: float
    """

    icon_size: bpy_prop_array[int]
    """ Width and height in pixels

    :type: bpy_prop_array[int]
    """

    image_pixels: int
    """ Image pixels, as bytes (always RGBA 32bits)

    :type: int
    """

    image_pixels_float: float
    """ Image pixels components, as floats (RGBA concatenated values)

    :type: float
    """

    image_size: bpy_prop_array[int]
    """ Width and height in pixels

    :type: bpy_prop_array[int]
    """

    is_icon_custom: bool
    """ True if this preview icon has been modified by py script,and is no more auto-generated by Blender

    :type: bool
    """

    is_image_custom: bool
    """ True if this preview image has been modified by py script,and is no more auto-generated by Blender

    :type: bool
    """

    def reload(self):
        """Reload the preview from its source path"""
        ...

class ImageUser(bpy_struct):
    """Parameters defining how an Image data-block is used by another data-block"""

    fields_per_frame: int
    """ Number of fields per rendered frame (2 fields is 1 image)

    :type: int
    """

    frame_current: int
    """ Current frame number in image sequence or movie

    :type: int
    """

    frame_duration: int
    """ Number of images of a movie to use

    :type: int
    """

    frame_offset: int
    """ Offset the number of the frame to use in the animation

    :type: int
    """

    frame_start: int
    """ Global starting frame of the movie/sequence, assuming first picture has a #1

    :type: int
    """

    multilayer_layer: int
    """ Layer in multilayer image

    :type: int
    """

    multilayer_pass: int
    """ Pass in multilayer image

    :type: int
    """

    multilayer_view: int
    """ View in multilayer image

    :type: int
    """

    use_auto_refresh: bool
    """ Always refresh image on frame changes

    :type: bool
    """

    use_cyclic: bool
    """ Cycle the images in the movie

    :type: bool
    """

class ImapaintToolCapabilities(bpy_struct):
    """Read-only indications of which brush operations are supported by the current image paint brush"""

    has_accumulate: bool
    """ 

    :type: bool
    """

    has_radius: bool
    """ 

    :type: bool
    """

    has_space_attenuation: bool
    """ 

    :type: bool
    """

class KeyConfig(bpy_struct):
    """Input configuration, including keymaps"""

    is_user_defined: bool
    """ Indicates that a keyconfig was defined by the user

    :type: bool
    """

    keymaps: KeyMaps
    """ Key maps configured as part of this configuration

    :type: KeyMaps
    """

    name: str
    """ Name of the key configuration

    :type: str
    """

class KeyConfigurations(bpy_prop_collection[KeyConfig], bpy_struct):
    """Collection of KeyConfigs"""

    active: KeyConfig | None
    """ Active key configuration (preset)

    :type: KeyConfig | None
    """

    addon: KeyConfig
    """ Key configuration that can be extended by add-ons, and is added to the active configuration when handling events

    :type: KeyConfig
    """

    default: KeyConfig
    """ Default builtin key configuration

    :type: KeyConfig
    """

    user: KeyConfig
    """ Final key configuration that combines keymaps from the active and add-on configurations, and can be edited by the user

    :type: KeyConfig
    """

    def new(self, name: str | typing.Any) -> KeyConfig:
        """new

        :param name: Name
        :type name: str | typing.Any
        :return: Key Configuration, Added key configuration
        :rtype: KeyConfig
        """
        ...

    def remove(self, keyconfig: KeyConfig):
        """remove

        :param keyconfig: Key Configuration, Removed key configuration
        :type keyconfig: KeyConfig
        """
        ...

class KeyMap(bpy_struct):
    """Input configuration, including keymaps"""

    is_modal: bool
    """ Indicates that a keymap is used for translate modal events for an operator

    :type: bool
    """

    is_user_modified: bool
    """ Keymap is defined by the user

    :type: bool
    """

    keymap_items: KeyMapItems
    """ Items in the keymap, linking an operator to an input event

    :type: KeyMapItems
    """

    name: str
    """ Name of the key map

    :type: str
    """

    region_type: str
    """ Optional region type keymap is associated with

    :type: str
    """

    show_expanded_children: bool
    """ Children expanded in the user interface

    :type: bool
    """

    show_expanded_items: bool
    """ Expanded in the user interface

    :type: bool
    """

    space_type: str
    """ Optional space type keymap is associated with

    :type: str
    """

    def active(self) -> KeyMap:
        """active

        :return: Key Map, Active key map
        :rtype: KeyMap
        """
        ...

    def restore_to_default(self):
        """restore_to_default"""
        ...

    def restore_item_to_default(self, item: KeyMapItem):
        """restore_item_to_default

        :param item: Item
        :type item: KeyMapItem
        """
        ...

class KeyMapItem(bpy_struct):
    """Item in a Key Map"""

    active: bool | None
    """ Activate or deactivate item

    :type: bool | None
    """

    alt: bool
    """ Alt key pressed

    :type: bool
    """

    any: bool
    """ Any modifier keys pressed

    :type: bool
    """

    ctrl: bool
    """ Control key pressed

    :type: bool
    """

    id: int
    """ ID of the item

    :type: int
    """

    idname: str
    """ Identifier of operator to call on input event

    :type: str
    """

    is_user_defined: bool
    """ Is this keymap item user defined (doesn't just replace a builtin item)

    :type: bool
    """

    is_user_modified: bool
    """ Is this keymap item modified by the user

    :type: bool
    """

    key_modifier: str
    """ Regular key pressed as a modifier

    :type: str
    """

    map_type: str
    """ Type of event mapping

    :type: str
    """

    name: str
    """ Name of operator (translated) to call on input event

    :type: str
    """

    oskey: bool
    """ Operating system key pressed

    :type: bool
    """

    properties: OperatorProperties
    """ Properties to set when the operator is called

    :type: OperatorProperties
    """

    propvalue: str
    """ The value this event translates to in a modal keymap

    :type: str
    """

    shift: bool
    """ Shift key pressed

    :type: bool
    """

    show_expanded: bool
    """ Show key map event and property details in the user interface

    :type: bool
    """

    type: str
    """ Type of event

    :type: str
    """

    value: str
    """ 

    :type: str
    """

    def compare(self, item: KeyMapItem | None) -> bool:
        """compare

        :param item: Item
        :type item: KeyMapItem | None
        :return: Comparison result
        :rtype: bool
        """
        ...

class KeyMapItems(bpy_prop_collection[KeyMapItem], bpy_struct):
    """Collection of keymap items"""

    def new(
        self,
        idname: str | typing.Any,
        type: str | None,
        value: str | None,
        any: bool | typing.Any | None = False,
        shift: bool | typing.Any | None = False,
        ctrl: bool | typing.Any | None = False,
        alt: bool | typing.Any | None = False,
        oskey: bool | typing.Any | None = False,
        key_modifier: str | None = "NONE",
        head: bool | typing.Any | None = False,
    ) -> KeyMapItem:
        """new

                :param idname: Operator Identifier
                :type idname: str | typing.Any
                :param type: Type

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type type: str | None
                :param value: Value
                :type value: str | None
                :param any: Any
                :type any: bool | typing.Any | None
                :param shift: Shift
                :type shift: bool | typing.Any | None
                :param ctrl: Ctrl
                :type ctrl: bool | typing.Any | None
                :param alt: Alt
                :type alt: bool | typing.Any | None
                :param oskey: OS Key
                :type oskey: bool | typing.Any | None
                :param key_modifier: Key Modifier

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type key_modifier: str | None
                :param head: At Head, Force item to be added at start (not end) of key map so that it doesn't get blocked by an existing key map item
                :type head: bool | typing.Any | None
                :return: Item, Added key map item
                :rtype: KeyMapItem
        """
        ...

    def new_modal(
        self,
        propvalue: str | typing.Any,
        type: str | None,
        value: str | None,
        any: bool | typing.Any | None = False,
        shift: bool | typing.Any | None = False,
        ctrl: bool | typing.Any | None = False,
        alt: bool | typing.Any | None = False,
        oskey: bool | typing.Any | None = False,
        key_modifier: str | None = "NONE",
    ) -> KeyMapItem:
        """new_modal

                :param propvalue: Property Value
                :type propvalue: str | typing.Any
                :param type: Type

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type type: str | None
                :param value: Value
                :type value: str | None
                :param any: Any
                :type any: bool | typing.Any | None
                :param shift: Shift
                :type shift: bool | typing.Any | None
                :param ctrl: Ctrl
                :type ctrl: bool | typing.Any | None
                :param alt: Alt
                :type alt: bool | typing.Any | None
                :param oskey: OS Key
                :type oskey: bool | typing.Any | None
                :param key_modifier: Key Modifier

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type key_modifier: str | None
                :return: Item, Added key map item
                :rtype: KeyMapItem
        """
        ...

    def remove(self, item: KeyMapItem):
        """remove

        :param item: Item
        :type item: KeyMapItem
        """
        ...

    def from_id(self, id: int | None) -> KeyMapItem:
        """from_id

        :param id: id, ID of the item
        :type id: int | None
        :return: Item
        :rtype: KeyMapItem
        """
        ...

class KeyMaps(bpy_prop_collection[KeyMap], bpy_struct):
    """Collection of keymaps"""

    def new(
        self,
        name: str | typing.Any,
        space_type: str | None = "EMPTY",
        region_type: str | None = "WINDOW",
        modal: bool | typing.Any | None = False,
    ) -> KeyMap:
        """new

                :param name: Name
                :type name: str | typing.Any
                :param space_type: Space Type

        EMPTY Empty.

        VIEW_3D 3D View, 3D viewport.

        TIMELINE Timeline, Timeline and playback controls.

        GRAPH_EDITOR Graph Editor, Edit drivers and keyframe interpolation.

        DOPESHEET_EDITOR Dope Sheet, Adjust timing of keyframes.

        NLA_EDITOR NLA Editor, Combine and layer Actions.

        IMAGE_EDITOR UV/Image Editor, View and edit images and UV Maps.

        SEQUENCE_EDITOR Video Sequence Editor, Video editing tools.

        CLIP_EDITOR Movie Clip Editor, Motion tracking tools.

        TEXT_EDITOR Text Editor, Edit scripts and in-file documentation.

        NODE_EDITOR Node Editor, Editor for node-based shading and compositing tools.

        LOGIC_EDITOR Logic Editor, Game logic editing.

        PROPERTIES Properties, Edit properties of active object and related data-blocks.

        OUTLINER Outliner, Overview of scene graph and all available data-blocks.

        USER_PREFERENCES User Preferences, Edit persistent configuration settings.

        INFO Info, Main menu bar and list of error messages (drag down to expand and display).

        FILE_BROWSER File Browser, Browse for files and assets.

        CONSOLE Python Console, Interactive programmatic console for advanced editing and script development.
                :type space_type: str | None
                :param region_type: Region Type
                :type region_type: str | None
                :param modal: Modal
                :type modal: bool | typing.Any | None
                :return: Key Map, Added key map
                :rtype: KeyMap
        """
        ...

    def remove(self, keymap: KeyMap):
        """remove

        :param keymap: Key Map, Removed key map
        :type keymap: KeyMap
        """
        ...

    def find(
        self,
        name: str | typing.Any,
        space_type: str | None = "EMPTY",
        region_type: str | None = "WINDOW",
    ) -> KeyMap:
        """find

                :param name: Name
                :type name: str | typing.Any
                :param space_type: Space Type

        EMPTY Empty.

        VIEW_3D 3D View, 3D viewport.

        TIMELINE Timeline, Timeline and playback controls.

        GRAPH_EDITOR Graph Editor, Edit drivers and keyframe interpolation.

        DOPESHEET_EDITOR Dope Sheet, Adjust timing of keyframes.

        NLA_EDITOR NLA Editor, Combine and layer Actions.

        IMAGE_EDITOR UV/Image Editor, View and edit images and UV Maps.

        SEQUENCE_EDITOR Video Sequence Editor, Video editing tools.

        CLIP_EDITOR Movie Clip Editor, Motion tracking tools.

        TEXT_EDITOR Text Editor, Edit scripts and in-file documentation.

        NODE_EDITOR Node Editor, Editor for node-based shading and compositing tools.

        LOGIC_EDITOR Logic Editor, Game logic editing.

        PROPERTIES Properties, Edit properties of active object and related data-blocks.

        OUTLINER Outliner, Overview of scene graph and all available data-blocks.

        USER_PREFERENCES User Preferences, Edit persistent configuration settings.

        INFO Info, Main menu bar and list of error messages (drag down to expand and display).

        FILE_BROWSER File Browser, Browse for files and assets.

        CONSOLE Python Console, Interactive programmatic console for advanced editing and script development.
                :type space_type: str | None
                :param region_type: Region Type
                :type region_type: str | None
                :return: Key Map, Corresponding key map
                :rtype: KeyMap
        """
        ...

    def find_modal(self, name: str | typing.Any) -> KeyMap:
        """find_modal

        :param name: Operator Name
        :type name: str | typing.Any
        :return: Key Map, Corresponding key map
        :rtype: KeyMap
        """
        ...

class Keyframe(bpy_struct):
    """Bezier curve point with two handles defining a Keyframe on an F-Curve"""

    amplitude: float
    """ Amount to boost elastic bounces for 'elastic' easing

    :type: float
    """

    back: float
    """ Amount of overshoot for 'back' easing

    :type: float
    """

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    easing: str
    """ Which ends of the segment between this and the next keyframe easing interpolation is applied to

    :type: str
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the left handle (before the control point)

    :type: bpy_prop_array[float]
    """

    handle_left_type: str
    """ Handle types

    :type: str
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the right handle (after the control point)

    :type: bpy_prop_array[float]
    """

    handle_right_type: str
    """ Handle types

    :type: str
    """

    interpolation: str
    """ Interpolation method to use for segment of the F-Curve from this Keyframe until the next Keyframe

    :type: str
    """

    period: float
    """ Time between bounces for elastic easing

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Left handle selection status

    :type: bool
    """

    select_right_handle: bool
    """ Right handle selection status

    :type: bool
    """

    type: str
    """ Type of keyframe (for visual purposes only)

    :type: str
    """

class KeyingSet(bpy_struct):
    """Settings that should be keyframed together"""

    bl_description: str
    """ A short description of the keying set

    :type: str
    """

    bl_idname: str
    """ If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is "BUILTIN_KSI_location", and bl_idname is not set by the script, then bl_idname = "BUILTIN_KSI_location")

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    is_path_absolute: bool
    """ Keying Set defines specific paths/settings to be keyframed (i.e. is not reliant on context info)

    :type: bool
    """

    paths: KeyingSetPaths
    """ Keying Set Paths to define settings that get keyframed together

    :type: KeyingSetPaths
    """

    type_info: KeyingSetInfo
    """ Callback function defines for built-in Keying Sets

    :type: KeyingSetInfo
    """

    use_insertkey_needed: bool
    """ Only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_needed: bool
    """ Override default setting to only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_visual: bool
    """ Override default setting to insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_override_xyz_to_rgb: bool
    """ Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis

    :type: bool
    """

    use_insertkey_visual: bool
    """ Insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis

    :type: bool
    """

    def refresh(self):
        """Refresh Keying Set to ensure that it is valid for the current context (call before each use of one)"""
        ...

class KeyingSetInfo(bpy_struct):
    """Callback function defines for builtin Keying Sets"""

    bl_description: str
    """ A short description of the keying set

    :type: str
    """

    bl_idname: str
    """ If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is "BUILTIN_KSI_location", and bl_idname is not set by the script, then bl_idname = "BUILTIN_KSI_location")

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[str]
    """ Keying Set options to use when inserting keyframes

    :type: set[str]
    """

    def poll(self, context: Context | None) -> bool:
        """Test if Keying Set can be used or not

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """
        ...

    def iterator(self, context: Context | None, ks: KeyingSet | None):
        """Call generate() on the structs which have properties to be keyframed

        :param context:
        :type context: Context | None
        :param ks:
        :type ks: KeyingSet | None
        """
        ...

    def generate(self, context: Context | None, ks: KeyingSet | None, data: typing.Any):
        """Add Paths to the Keying Set to keyframe the properties of the given data

        :param context:
        :type context: Context | None
        :param ks:
        :type ks: KeyingSet | None
        :param data:
        :type data: typing.Any
        """
        ...

class KeyingSetPath(bpy_struct):
    """Path to a setting for use in a Keying Set"""

    array_index: int
    """ Index to the specific setting if applicable

    :type: int
    """

    data_path: str
    """ Path to property setting

    :type: str
    """

    group: str
    """ Name of Action Group to assign setting(s) for this path to

    :type: str
    """

    group_method: str
    """ Method used to define which Group-name to use

    :type: str
    """

    id: ID
    """ ID-Block that keyframes for Keying Set should be added to (for Absolute Keying Sets only)

    :type: ID
    """

    id_type: str
    """ Type of ID-block that can be used

    :type: str
    """

    use_entire_array: bool
    """ When an 'array/vector' type is chosen (Location, Rotation, Color, etc.), entire array is to be used

    :type: bool
    """

    use_insertkey_needed: bool
    """ Only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_needed: bool
    """ Override default setting to only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_visual: bool
    """ Override default setting to insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_override_xyz_to_rgb: bool
    """ Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis

    :type: bool
    """

    use_insertkey_visual: bool
    """ Insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis

    :type: bool
    """

class KeyingSetPaths(bpy_prop_collection[KeyingSetPath], bpy_struct):
    """Collection of keying set paths"""

    active: KeyingSetPath | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSetPath | None
    """

    active_index: int | None
    """ Current Keying Set index

    :type: int | None
    """

    def add(
        self,
        target_id: ID | None,
        data_path: str | typing.Any,
        index: typing.Any | None = -1,
        group_method: str | None = "KEYINGSET",
        group_name: str | typing.Any = "",
    ) -> KeyingSetPath:
        """Add a new path for the Keying Set

        :param target_id: Target ID, ID-Datablock for the destination
        :type target_id: ID | None
        :param data_path: Data-Path, RNA-Path to destination property
        :type data_path: str | typing.Any
        :param index: Index, The index of the destination property (i.e. axis of Location/Rotation/etc.), or -1 for the entire array
        :type index: typing.Any | None
        :param group_method: Grouping Method, Method used to define which Group-name to use
        :type group_method: str | None
        :param group_name: Group Name, Name of Action Group to assign destination to (only if grouping mode is to use this name)
        :type group_name: str | typing.Any
        :return: New Path, Path created and added to the Keying Set
        :rtype: KeyingSetPath
        """
        ...

    def remove(self, path: KeyingSetPath):
        """Remove the given path from the Keying Set

        :param path: Path
        :type path: KeyingSetPath
        """
        ...

    def clear(self):
        """Remove all the paths from the Keying Set"""
        ...

class KeyingSets(bpy_prop_collection[KeyingSet], bpy_struct):
    """Scene keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

    def new(
        self,
        idname: str | typing.Any = "KeyingSet",
        name: str | typing.Any = "KeyingSet",
    ) -> KeyingSet:
        """Add a new Keying Set to Scene

        :param idname: IDName, Internal identifier of Keying Set
        :type idname: str | typing.Any
        :param name: Name, User visible name of Keying Set
        :type name: str | typing.Any
        :return: Newly created Keying Set
        :rtype: KeyingSet
        """
        ...

class KeyingSetsAll(bpy_prop_collection[KeyingSet], bpy_struct):
    """All available keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

class LampSkySettings(bpy_struct):
    """Sky related settings for a sun lamp"""

    atmosphere_distance_factor: float
    """ Multiplier to convert blender units to physical distance

    :type: float
    """

    atmosphere_extinction: float
    """ Extinction scattering contribution factor

    :type: float
    """

    atmosphere_inscattering: float
    """ Scatter contribution factor

    :type: float
    """

    atmosphere_turbidity: float
    """ Sky turbidity

    :type: float
    """

    backscattered_light: float
    """ Backscattered light

    :type: float
    """

    horizon_brightness: float
    """ Horizon brightness

    :type: float
    """

    sky_blend: float
    """ Blend factor with sky

    :type: float
    """

    sky_blend_type: str
    """ Blend mode for combining sun sky with world sky

    :type: str
    """

    sky_color_space: str
    """ Color space to use for internal XYZ->RGB color conversion

    :type: str
    """

    sky_exposure: float
    """ Strength of sky shading exponential exposure correction

    :type: float
    """

    spread: float
    """ Horizon Spread

    :type: float
    """

    sun_brightness: float
    """ Sun brightness

    :type: float
    """

    sun_intensity: float
    """ Sun intensity

    :type: float
    """

    sun_size: float
    """ Sun size

    :type: float
    """

    use_atmosphere: bool
    """ Apply sun effect on atmosphere

    :type: bool
    """

    use_sky: bool
    """ Apply sun effect on sky

    :type: bool
    """

class LampTextureSlots(bpy_prop_collection[LampTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> LampTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: LampTextureSlot
        """
        ...

    @classmethod
    def create(cls, index: int | None) -> LampTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: LampTextureSlot
        """
        ...

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """
        ...

class LatticePoint(bpy_struct):
    """Point in the lattice grid"""

    co: bpy_prop_array[float]
    """ Original undeformed location used to calculate the strength of the deform effect (edit/animate the Deformed Location instead)

    :type: bpy_prop_array[float]
    """

    co_deform: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    groups: bpy_prop_collection[VertexGroupElement]
    """ Weights for the vertex groups this point is member of

    :type: bpy_prop_collection[VertexGroupElement]
    """

    select: bool
    """ Selection status

    :type: bool
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

class LineStyleAlphaModifiers(bpy_prop_collection[LineStyleAlphaModifier], bpy_struct):
    """Alpha modifiers for changing line alphas"""

    def new(self, name: str | typing.Any, type: str | None) -> LineStyleAlphaModifier:
        """Add a alpha modifier to line style

        :param name: New name for the alpha modifier (not unique)
        :type name: str | typing.Any
        :param type: Alpha modifier type to add
        :type type: str | None
        :return: Newly added alpha modifier
        :rtype: LineStyleAlphaModifier
        """
        ...

    def remove(self, modifier: LineStyleAlphaModifier):
        """Remove a alpha modifier from line style

        :param modifier: Alpha modifier to remove
        :type modifier: LineStyleAlphaModifier
        """
        ...

class LineStyleColorModifiers(bpy_prop_collection[LineStyleColorModifier], bpy_struct):
    """Color modifiers for changing line colors"""

    def new(self, name: str | typing.Any, type: str | None) -> LineStyleColorModifier:
        """Add a color modifier to line style

        :param name: New name for the color modifier (not unique)
        :type name: str | typing.Any
        :param type: Color modifier type to add
        :type type: str | None
        :return: Newly added color modifier
        :rtype: LineStyleColorModifier
        """
        ...

    def remove(self, modifier: LineStyleColorModifier):
        """Remove a color modifier from line style

        :param modifier: Color modifier to remove
        :type modifier: LineStyleColorModifier
        """
        ...

class LineStyleGeometryModifiers(
    bpy_prop_collection[LineStyleGeometryModifier], bpy_struct
):
    """Geometry modifiers for changing line geometries"""

    def new(
        self, name: str | typing.Any, type: str | None
    ) -> LineStyleGeometryModifier:
        """Add a geometry modifier to line style

        :param name: New name for the geometry modifier (not unique)
        :type name: str | typing.Any
        :param type: Geometry modifier type to add
        :type type: str | None
        :return: Newly added geometry modifier
        :rtype: LineStyleGeometryModifier
        """
        ...

    def remove(self, modifier: LineStyleGeometryModifier):
        """Remove a geometry modifier from line style

        :param modifier: Geometry modifier to remove
        :type modifier: LineStyleGeometryModifier
        """
        ...

class LineStyleModifier(bpy_struct):
    """Base type to define modifiers"""

    ...

class LineStyleTextureSlots(bpy_prop_collection[LineStyleTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> LineStyleTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """
        ...

    @classmethod
    def create(cls, index: int | None) -> LineStyleTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """
        ...

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """
        ...

class LineStyleThicknessModifiers(
    bpy_prop_collection[LineStyleThicknessModifier], bpy_struct
):
    """Thickness modifiers for changing line thickness"""

    def new(
        self, name: str | typing.Any, type: str | None
    ) -> LineStyleThicknessModifier:
        """Add a thickness modifier to line style

        :param name: New name for the thickness modifier (not unique)
        :type name: str | typing.Any
        :param type: Thickness modifier type to add
        :type type: str | None
        :return: Newly added thickness modifier
        :rtype: LineStyleThicknessModifier
        """
        ...

    def remove(self, modifier: LineStyleThicknessModifier):
        """Remove a thickness modifier from line style

        :param modifier: Thickness modifier to remove
        :type modifier: LineStyleThicknessModifier
        """
        ...

class Linesets(bpy_prop_collection[FreestyleLineSet], bpy_struct):
    """Line sets for associating lines and style parameters"""

    active: FreestyleLineSet
    """ Active line set being displayed

    :type: FreestyleLineSet
    """

    active_index: int | None
    """ Index of active line set slot

    :type: int | None
    """

    def new(self, name: str | typing.Any) -> FreestyleLineSet:
        """Add a line set to scene render layer Freestyle settings

        :param name: New name for the line set (not unique)
        :type name: str | typing.Any
        :return: Newly created line set
        :rtype: FreestyleLineSet
        """
        ...

    def remove(self, lineset: FreestyleLineSet):
        """Remove a line set from scene render layer Freestyle settings

        :param lineset: Line set to remove
        :type lineset: FreestyleLineSet
        """
        ...

class LodLevel(bpy_struct):
    distance: float
    """ Distance to begin using this level of detail

    :type: float
    """

    object: Object
    """ Object to use for this level of detail

    :type: Object
    """

    object_hysteresis_percentage: int
    """ Minimum distance change required to transition to the previous level of detail

    :type: int
    """

    use_material: bool
    """ Use the material from this object at this level of detail

    :type: bool
    """

    use_mesh: bool
    """ Use the mesh from this object at this level of detail

    :type: bool
    """

    use_object_hysteresis: bool
    """ Override LoD Hysteresis scene setting for this LoD level

    :type: bool
    """

class LoopColors(bpy_prop_collection[MeshLoopColorLayer], bpy_struct):
    """Collection of vertex colors"""

    active: MeshLoopColorLayer | None
    """ Active vertex color layer

    :type: MeshLoopColorLayer | None
    """

    active_index: int | None
    """ Active vertex color index

    :type: int | None
    """

    def new(self, name: str | typing.Any = "Col") -> MeshLoopColorLayer:
        """Add a vertex color layer to Mesh

        :param name: Vertex color name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshLoopColorLayer
        """
        ...

    def remove(self, layer: MeshLoopColorLayer):
        """Remove a vertex color layer

        :param layer: The layer to remove
        :type layer: MeshLoopColorLayer
        """
        ...

class Macro(bpy_struct):
    """Storage of a macro operator being executed, or registered after execution"""

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[str]
    """ Options for this operator type

    :type: set[str]
    """

    bl_translation_context: str
    """ 

    :type: str
    """

    name: str
    """ 

    :type: str
    """

    properties: OperatorProperties
    """ 

    :type: OperatorProperties
    """

    def report(self, type: set[str] | None, message: str | typing.Any):
        """report

        :param type: Type
        :type type: set[str] | None
        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    def draw(self, context: Context):
        """Draw function for the operator

        :param context:
        :type context: Context
        """
        ...

class MaskLayer(bpy_struct):
    """Single layer used for masking pixels"""

    alpha: float
    """ Render Opacity

    :type: float
    """

    blend: str
    """ Method of blending mask layers

    :type: str
    """

    falloff: str
    """ Falloff type the feather

    :type: str
    """

    hide: bool
    """ Restrict visibility in the viewport

    :type: bool
    """

    hide_render: bool
    """ Restrict renderability

    :type: bool
    """

    hide_select: bool
    """ Restrict selection in the viewport

    :type: bool
    """

    invert: bool
    """ Invert the mask black/white

    :type: bool
    """

    name: str
    """ Unique name of layer

    :type: str
    """

    select: bool
    """ Layer is selected for editing in the Dope Sheet

    :type: bool
    """

    splines: MaskSplines
    """ Collection of splines which defines this layer

    :type: MaskSplines
    """

    use_fill_holes: bool
    """ Calculate holes when filling overlapping curves

    :type: bool
    """

    use_fill_overlap: bool
    """ Calculate self intersections and overlap before filling

    :type: bool
    """

class MaskLayers(bpy_prop_collection[MaskLayer], bpy_struct):
    """Collection of layers used by mask"""

    active: MaskLayer | None
    """ Active layer in this mask

    :type: MaskLayer | None
    """

    def new(self, name: str | typing.Any = "") -> MaskLayer:
        """Add layer to this mask

        :param name: Name, Name of new layer
        :type name: str | typing.Any
        :return: New mask layer
        :rtype: MaskLayer
        """
        ...

    def remove(self, layer: MaskLayer):
        """Remove layer from this mask

        :param layer: Shape to be removed
        :type layer: MaskLayer
        """
        ...

    def clear(self):
        """Remove all mask layers"""
        ...

class MaskParent(bpy_struct):
    """Parenting settings for masking element"""

    id: ID
    """ ID-block to which masking element would be parented to or to it's property

    :type: ID
    """

    id_type: str
    """ Type of ID-block that can be used

    :type: str
    """

    parent: str
    """ Name of parent object in specified data block to which parenting happens

    :type: str
    """

    sub_parent: str
    """ Name of parent sub-object in specified data block to which parenting happens

    :type: str
    """

    type: str
    """ Parent Type

    :type: str
    """

class MaskSpline(bpy_struct):
    """Single spline used for defining mask shape"""

    offset_mode: str
    """ The method used for calculating the feather offset

    :type: str
    """

    points: MaskSplinePoints
    """ Collection of points

    :type: MaskSplinePoints
    """

    use_cyclic: bool
    """ Make this spline a closed loop

    :type: bool
    """

    use_fill: bool
    """ Make this spline filled

    :type: bool
    """

    use_self_intersection_check: bool
    """ Prevent feather from self-intersections

    :type: bool
    """

    weight_interpolation: str
    """ The type of weight interpolation for spline

    :type: str
    """

class MaskSplinePoint(bpy_struct):
    """Single point in spline used for defining mask"""

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    feather_points: bpy_prop_collection[MaskSplinePointUW]
    """ Points defining feather

    :type: bpy_prop_collection[MaskSplinePointUW]
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the first handle

    :type: bpy_prop_array[float]
    """

    handle_left_type: str
    """ Handle type

    :type: str
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the second handle

    :type: bpy_prop_array[float]
    """

    handle_right_type: str
    """ Handle type

    :type: str
    """

    handle_type: str
    """ Handle type

    :type: str
    """

    parent: MaskParent
    """ 

    :type: MaskParent
    """

    select: bool
    """ Selection status

    :type: bool
    """

    weight: float
    """ Weight of the point

    :type: float
    """

class MaskSplinePointUW(bpy_struct):
    """Single point in spline segment defining feather"""

    select: bool
    """ Selection status

    :type: bool
    """

    u: float
    """ U coordinate of point along spline segment

    :type: float
    """

    weight: float
    """ Weight of feather point

    :type: float
    """

class MaskSplinePoints(bpy_prop_collection[MaskSplinePoint], bpy_struct):
    """Collection of masking spline points"""

    def add(self, count: typing.Any | None = 1):
        """Add a number of point to this spline

        :param count: Number, Number of points to add to the spline
        :type count: typing.Any | None
        """
        ...

    def remove(self, point: MaskSplinePoint):
        """Remove a point from a spline

        :param point: The point to remove
        :type point: MaskSplinePoint
        """
        ...

class MaskSplines(bpy_prop_collection[MaskSpline], bpy_struct):
    """Collection of masking splines"""

    active: MaskSpline | None
    """ Active spline of masking layer

    :type: MaskSpline | None
    """

    active_point: MaskSplinePoint | None
    """ Active spline of masking layer

    :type: MaskSplinePoint | None
    """

    def new(self) -> MaskSpline:
        """Add a new spline to the layer

        :return: The newly created spline
        :rtype: MaskSpline
        """
        ...

    def remove(self, spline: MaskSpline):
        """Remove a spline from a layer

        :param spline: The spline to remove
        :type spline: MaskSpline
        """
        ...

class MaterialGameSettings(bpy_struct):
    """Game Engine settings for a Material data-block"""

    alpha_blend: str
    """ Blend Mode for Transparent Faces

    :type: str
    """

    face_orientation: str
    """ Especial face orientation options

    :type: str
    """

    invisible: bool
    """ Make face invisible

    :type: bool
    """

    physics: bool
    """ Use physics properties of materials

    :type: bool
    """

    text: bool
    """ Use material as text in Game Engine

    :type: bool
    """

    use_backface_culling: bool
    """ Hide Back of the face in Game Engine

    :type: bool
    """

class MaterialHalo(bpy_struct):
    """Halo particle effect settings for a Material data-block"""

    add: float
    """ Strength of the add effect

    :type: float
    """

    flare_boost: float
    """ Give the flare extra strength

    :type: float
    """

    flare_seed: int
    """ Offset in the flare seed table

    :type: int
    """

    flare_size: float
    """ Factor by which the flare is larger than the halo

    :type: float
    """

    flare_subflare_count: int
    """ Number of sub-flares

    :type: int
    """

    flare_subflare_size: float
    """ Dimension of the sub-flares, dots and circles

    :type: float
    """

    hardness: int
    """ Hardness of the halo

    :type: int
    """

    line_count: int
    """ Number of star shaped lines rendered over the halo

    :type: int
    """

    ring_count: int
    """ Number of rings rendered over the halo

    :type: int
    """

    seed: int
    """ Randomize ring dimension and line location

    :type: int
    """

    size: float
    """ Dimension of the halo

    :type: float
    """

    star_tip_count: int
    """ Number of points on the star shaped halo

    :type: int
    """

    use_extreme_alpha: bool
    """ Use extreme alpha

    :type: bool
    """

    use_flare_mode: bool
    """ Render halo as a lens flare

    :type: bool
    """

    use_lines: bool
    """ Render star shaped lines over halo

    :type: bool
    """

    use_ring: bool
    """ Render rings over halo

    :type: bool
    """

    use_shaded: bool
    """ Let halo receive light and shadows from external objects

    :type: bool
    """

    use_soft: bool
    """ Soften the edges of halos at intersections with other geometry

    :type: bool
    """

    use_star: bool
    """ Render halo as a star

    :type: bool
    """

    use_texture: bool
    """ Give halo a texture

    :type: bool
    """

    use_vertex_normal: bool
    """ Use the vertex normal to specify the dimension of the halo

    :type: bool
    """

class MaterialPhysics(bpy_struct):
    """Physics settings for a Material data-block"""

    elasticity: float
    """ Elasticity of collisions

    :type: float
    """

    fh_damping: float
    """ Damping of the spring force, when inside the physics distance area

    :type: float
    """

    fh_distance: float
    """ Distance of the physics area

    :type: float
    """

    fh_force: float
    """ Upward spring force, when inside the physics distance area

    :type: float
    """

    friction: float
    """ Coulomb friction coefficient, when inside the physics distance area

    :type: float
    """

    use_fh_normal: bool
    """ Align dynamic game objects along the surface normal, when inside the physics distance area

    :type: bool
    """

class MaterialRaytraceMirror(bpy_struct):
    """Raytraced reflection settings for a Material data-block"""

    depth: int
    """ Maximum allowed number of light inter-reflections

    :type: int
    """

    distance: float
    """ Maximum distance of reflected rays (reflections further than this range fade to sky color or material color)

    :type: float
    """

    fade_to: str
    """ The color that rays with no intersection within the Max Distance take (material color can be best for indoor scenes, sky color for outdoor)

    :type: str
    """

    fresnel: float
    """ Power of Fresnel for mirror reflection

    :type: float
    """

    fresnel_factor: float
    """ Blending factor for Fresnel

    :type: float
    """

    gloss_anisotropic: float
    """ The shape of the reflection, from 0.0 (circular) to 1.0 (fully stretched along the tangent

    :type: float
    """

    gloss_factor: float
    """ The shininess of the reflection (values < 1.0 give diffuse, blurry reflections)

    :type: float
    """

    gloss_samples: int
    """ Number of cone samples averaged for blurry reflections

    :type: int
    """

    gloss_threshold: float
    """ Threshold for adaptive sampling (if a sample contributes less than this amount [as a percentage], sampling is stopped)

    :type: float
    """

    reflect_factor: float
    """ Amount of mirror reflection for raytrace

    :type: float
    """

    use: bool
    """ Enable raytraced reflections

    :type: bool
    """

class MaterialRaytraceTransparency(bpy_struct):
    """Raytraced refraction settings for a Material data-block"""

    depth: int
    """ Maximum allowed number of light inter-refractions

    :type: int
    """

    depth_max: float
    """ Maximum depth for light to travel through the transparent material before becoming fully filtered (0.0 is disabled)

    :type: float
    """

    falloff: float
    """ Falloff power for transmissivity filter effect (1.0 is linear)

    :type: float
    """

    filter: float
    """ Amount to blend in the material's diffuse color in raytraced transparency (simulating absorption)

    :type: float
    """

    fresnel: float
    """ Power of Fresnel for transparency (Ray or ZTransp)

    :type: float
    """

    fresnel_factor: float
    """ Blending factor for Fresnel

    :type: float
    """

    gloss_factor: float
    """ The clarity of the refraction. Values < 1.0 give diffuse, blurry refractions

    :type: float
    """

    gloss_samples: int
    """ Number of cone samples averaged for blurry refractions

    :type: int
    """

    gloss_threshold: float
    """ Threshold for adaptive sampling. If a sample contributes less than this amount (as a percentage), sampling is stopped

    :type: float
    """

    ior: float
    """ Angular index of refraction for raytraced refraction

    :type: float
    """

class MaterialSlot(bpy_struct):
    """Material slot in an object"""

    link: str
    """ Link material to object or the object's data

    :type: str
    """

    material: Material
    """ Material data-block used by this material slot

    :type: Material
    """

    name: str
    """ Material slot name

    :type: str
    """

class MaterialStrand(bpy_struct):
    """Strand settings for a Material data-block"""

    blend_distance: float
    """ Worldspace distance over which to blend in the surface normal

    :type: float
    """

    root_size: float
    """ Start size of strands in pixels or Blender units

    :type: float
    """

    shape: float
    """ Positive values make strands rounder, negative ones make strands spiky

    :type: float
    """

    size_min: float
    """ Minimum size of strands in pixels

    :type: float
    """

    tip_size: float
    """ End size of strands in pixels or Blender units

    :type: float
    """

    use_blender_units: bool
    """ Use Blender units for widths instead of pixels

    :type: bool
    """

    use_surface_diffuse: bool
    """ Make diffuse shading more similar to shading the surface

    :type: bool
    """

    use_tangent_shading: bool
    """ Use direction of strands as normal for tangent-shading

    :type: bool
    """

    uv_layer: str
    """ Name of UV map to override

    :type: str
    """

    width_fade: float
    """ Transparency along the width of the strand

    :type: float
    """

class MaterialSubsurfaceScattering(bpy_struct):
    """Diffuse subsurface scattering settings for a Material data-block"""

    back: float
    """ Back scattering weight

    :type: float
    """

    color: bpy_prop_array[float]
    """ Scattering color

    :type: bpy_prop_array[float]
    """

    color_factor: float
    """ Blend factor for SSS colors

    :type: float
    """

    error_threshold: float
    """ Error tolerance (low values are slower and higher quality)

    :type: float
    """

    front: float
    """ Front scattering weight

    :type: float
    """

    ior: float
    """ Index of refraction (higher values are denser)

    :type: float
    """

    radius: bpy_prop_array[float]
    """ Mean red/green/blue scattering path length

    :type: bpy_prop_array[float]
    """

    scale: float
    """ Object scale factor

    :type: float
    """

    texture_factor: float
    """ Texture scattering blend factor

    :type: float
    """

    use: bool
    """ Enable diffuse subsurface scattering effects in a material

    :type: bool
    """

class MaterialTextureSlots(bpy_prop_collection[MaterialTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> MaterialTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: MaterialTextureSlot
        """
        ...

    @classmethod
    def create(cls, index: int | None) -> MaterialTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: MaterialTextureSlot
        """
        ...

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """
        ...

class MaterialVolume(bpy_struct):
    """Volume rendering settings for a Material data-block"""

    asymmetry: float
    """ Back scattering (-1.0) to Forward scattering (1.0) and the range in between

    :type: float
    """

    cache_resolution: int
    """ Resolution of the voxel grid, low resolutions are faster, high resolutions use more memory

    :type: int
    """

    density: float
    """ The base density of the volume

    :type: float
    """

    density_scale: float
    """ Multiplier for the material's density

    :type: float
    """

    depth_threshold: float
    """ Stop ray marching early if transmission drops below this luminance - higher values give speedups in dense volumes at the expense of accuracy

    :type: float
    """

    emission: float
    """ Amount of light that gets emitted by the volume

    :type: float
    """

    emission_color: bpy_prop_array[float]
    """ Color of emitted light

    :type: bpy_prop_array[float]
    """

    light_method: str
    """ Method of shading, attenuating, and scattering light through the volume

    :type: str
    """

    ms_diffusion: float
    """ Diffusion factor, the strength of the blurring effect

    :type: float
    """

    ms_intensity: float
    """ Multiplier for multiple scattered light energy

    :type: float
    """

    ms_spread: float
    """ Proportional distance over which the light is diffused

    :type: float
    """

    reflection: float
    """ Multiplier to make out-scattered light brighter or darker (non-physically correct)

    :type: float
    """

    reflection_color: bpy_prop_array[float]
    """ Color of light scattered out of the volume (does not affect transmission)

    :type: bpy_prop_array[float]
    """

    scattering: float
    """ Amount of light that gets scattered out by the volume - the more out-scattering, the shallower the light will penetrate

    :type: float
    """

    step_method: str
    """ Method of calculating the steps through the volume

    :type: str
    """

    step_size: float
    """ Distance between subsequent volume depth samples

    :type: float
    """

    transmission_color: bpy_prop_array[float]
    """ Result color of the volume, after other light has been scattered/absorbed

    :type: bpy_prop_array[float]
    """

    use_external_shadows: bool
    """ Receive shadows from sources outside the volume (temporary)

    :type: bool
    """

    use_light_cache: bool
    """ Pre-calculate the shading information into a voxel grid, speeds up shading at slightly less accuracy

    :type: bool
    """

class Menu(bpy_struct):
    """Editor menu containing buttons"""

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ If this is set, the menu gets a custom ID, otherwise it takes the name of the class used to define the menu (for example, if the class name is "OBJECT_MT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_MT_hello")

    :type: str
    """

    bl_label: str
    """ The menu label

    :type: str
    """

    bl_translation_context: str | typing.Any
    """ 

    :type: str | typing.Any
    """

    layout: UILayout
    """ Defines the structure of the menu in the UI

    :type: UILayout
    """

    @classmethod
    def poll(cls, context: Context | None) -> bool:
        """If this method returns a non-null output, then the menu can be drawn

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """
        ...

    def draw(self, context: Context | None):
        """Draw UI elements into the menu UI layout

        :param context:
        :type context: Context | None
        """
        ...

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """
        ...

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """
        ...

    @classmethod
    def append(cls, draw_func):
        """

        :param draw_func:
        """
        ...

    @classmethod
    def prepend(cls, draw_func):
        """

        :param draw_func:
        """
        ...

    @classmethod
    def remove(cls, draw_func):
        """

        :param draw_func:
        """
        ...

class MeshColor(bpy_struct):
    """Vertex colors for a face in a Mesh"""

    color1: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color2: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color3: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color4: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class MeshColorLayer(bpy_struct):
    """Layer of vertex colors in a Mesh data-block"""

    active: bool | None
    """ Sets the layer as active for display and editing

    :type: bool | None
    """

    active_render: bool | None
    """ Sets the layer as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshColor]
    """ 

    :type: bpy_prop_collection[MeshColor]
    """

    name: str
    """ Name of Vertex color layer

    :type: str
    """

class MeshEdge(bpy_struct):
    """Edge in a Mesh data-block"""

    bevel_weight: float
    """ Weight used by the Bevel modifier

    :type: float
    """

    crease: float
    """ Weight used by the Subsurf modifier for creasing

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this edge

    :type: int
    """

    is_loose: bool
    """ Loose edge

    :type: bool
    """

    select: bool
    """ 

    :type: bool
    """

    use_edge_sharp: bool
    """ Sharp edge for the EdgeSplit modifier

    :type: bool
    """

    use_freestyle_mark: bool
    """ Edge mark for Freestyle line rendering

    :type: bool
    """

    use_seam: bool
    """ Seam edge for UV unwrapping

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    key: typing.Any
    """ (readonly)"""

class MeshEdges(bpy_prop_collection[MeshEdge], bpy_struct):
    """Collection of mesh edges"""

    def add(self, count: typing.Any | None = 0):
        """add

        :param count: Count, Number of edges to add
        :type count: typing.Any | None
        """
        ...

class MeshLoop(bpy_struct):
    """Loop in a Mesh data-block"""

    bitangent: bpy_prop_array[float]
    """ Bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, use it only if really needed, slower access than bitangent_sign)

    :type: bpy_prop_array[float]
    """

    bitangent_sign: float
    """ Sign of the bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, bitangent = bitangent_sign * cross(normal, tangent))

    :type: float
    """

    edge_index: int
    """ Edge index

    :type: int
    """

    index: int
    """ Index of this loop

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length split normal vector of this vertex for this polygon (must be computed beforehand using calc_normals_split or calc_tangents)

    :type: bpy_prop_array[float]
    """

    tangent: bpy_prop_array[float]
    """ Local space unit length tangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents)

    :type: bpy_prop_array[float]
    """

    vertex_index: int
    """ Vertex index

    :type: int
    """

class MeshLoopColor(bpy_struct):
    """Vertex loop colors in a Mesh"""

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class MeshLoopColorLayer(bpy_struct):
    """Layer of vertex colors in a Mesh data-block"""

    active: bool | None
    """ Sets the layer as active for display and editing

    :type: bool | None
    """

    active_render: bool | None
    """ Sets the layer as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshLoopColor]
    """ 

    :type: bpy_prop_collection[MeshLoopColor]
    """

    name: str
    """ Name of Vertex color layer

    :type: str
    """

class MeshLoops(bpy_prop_collection[MeshLoop], bpy_struct):
    """Collection of mesh loops"""

    def add(self, count: typing.Any | None = 0):
        """add

        :param count: Count, Number of loops to add
        :type count: typing.Any | None
        """
        ...

class MeshPaintMaskLayer(bpy_struct):
    """Per-vertex paint mask data"""

    data: bpy_prop_collection[MeshPaintMaskProperty]
    """ 

    :type: bpy_prop_collection[MeshPaintMaskProperty]
    """

class MeshPaintMaskProperty(bpy_struct):
    """Floating point paint mask value"""

    value: float
    """ 

    :type: float
    """

class MeshPolygon(bpy_struct):
    """Polygon in a Mesh data-block"""

    area: float
    """ Read only area of this polygon

    :type: float
    """

    center: bpy_prop_array[float]
    """ Center of this polygon

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this polygon

    :type: int
    """

    loop_start: int
    """ Index of the first loop of this polygon

    :type: int
    """

    loop_total: int
    """ Number of loops used by this polygon

    :type: int
    """

    material_index: int
    """ 

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length normal vector for this polygon

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    use_freestyle_mark: bool
    """ Face mark for Freestyle line rendering

    :type: bool
    """

    use_smooth: bool
    """ 

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    edge_keys: typing.Any
    """ (readonly)"""

    loop_indices: typing.Any
    """ (readonly)"""

    def flip(self):
        """Invert winding of this polygon (flip its normal)"""
        ...

class MeshPolygonFloatProperty(bpy_struct):
    """User defined floating point number value in a float properties layer"""

    value: float
    """ 

    :type: float
    """

class MeshPolygonFloatPropertyLayer(bpy_struct):
    """User defined layer of floating point number values"""

    data: bpy_prop_collection[MeshPolygonFloatProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonFloatProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshPolygonIntProperty(bpy_struct):
    """User defined integer number value in an integer properties layer"""

    value: int
    """ 

    :type: int
    """

class MeshPolygonIntPropertyLayer(bpy_struct):
    """User defined layer of integer number values"""

    data: bpy_prop_collection[MeshPolygonIntProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonIntProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshPolygonStringProperty(bpy_struct):
    """User defined string text value in a string properties layer"""

    value: str
    """ 

    :type: str
    """

class MeshPolygonStringPropertyLayer(bpy_struct):
    """User defined layer of string text values"""

    data: bpy_prop_collection[MeshPolygonStringProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonStringProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshPolygons(bpy_prop_collection[MeshPolygon], bpy_struct):
    """Collection of mesh polygons"""

    active: int | None
    """ The active polygon for this mesh

    :type: int | None
    """

    def add(self, count: typing.Any | None = 0):
        """add

        :param count: Count, Number of polygons to add
        :type count: typing.Any | None
        """
        ...

class MeshSkinVertex(bpy_struct):
    """Per-vertex skin data for use with the Skin modifier"""

    radius: bpy_prop_array[float]
    """ Radius of the skin

    :type: bpy_prop_array[float]
    """

    use_loose: bool
    """ If vertex has multiple adjacent edges, it is hulled to them directly

    :type: bool
    """

    use_root: bool
    """ Vertex is a root for rotation calculations and armature generation

    :type: bool
    """

class MeshSkinVertexLayer(bpy_struct):
    """Per-vertex skin data for use with the Skin modifier"""

    data: bpy_prop_collection[MeshSkinVertex]
    """ 

    :type: bpy_prop_collection[MeshSkinVertex]
    """

    name: str
    """ Name of skin layer

    :type: str
    """

class MeshStatVis(bpy_struct):
    distort_max: float
    """ Maximum angle to display

    :type: float
    """

    distort_min: float
    """ Minimum angle to display

    :type: float
    """

    overhang_axis: str
    """ 

    :type: str
    """

    overhang_max: float
    """ Maximum angle to display

    :type: float
    """

    overhang_min: float
    """ Minimum angle to display

    :type: float
    """

    sharp_max: float
    """ Maximum angle to display

    :type: float
    """

    sharp_min: float
    """ Minimum angle to display

    :type: float
    """

    thickness_max: float
    """ Maximum for measuring thickness

    :type: float
    """

    thickness_min: float
    """ Minimum for measuring thickness

    :type: float
    """

    thickness_samples: int
    """ Number of samples to test per face

    :type: int
    """

    type: str
    """ Type of data to visualize/check

    :type: str
    """

class MeshTessFace(bpy_struct):
    """TessFace in a Mesh data-block"""

    area: float
    """ Read only area of this face

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this face

    :type: int
    """

    material_index: int
    """ 

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length normal vector for this face

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    split_normals: bpy_prop_array[float]
    """ Local space unit length split normals vectors of the vertices of this face (must be computed beforehand using calc_normals_split or calc_tangents, and then calc_tessface)

    :type: bpy_prop_array[float]
    """

    use_smooth: bool
    """ 

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    vertices_raw: bpy_prop_array[int]
    """ Fixed size vertex indices array

    :type: bpy_prop_array[int]
    """

    center: typing.Any
    """ The midpoint of the face.
(readonly)"""

    edge_keys: typing.Any
    """ (readonly)"""

class MeshTessFaces(bpy_prop_collection[MeshTessFace], bpy_struct):
    """Collection of mesh faces"""

    active: int | None
    """ The active face for this mesh

    :type: int | None
    """

    def add(self, count: typing.Any | None = 0):
        """add

        :param count: Count, Number of faces to add
        :type count: typing.Any | None
        """
        ...

class MeshTextureFace(bpy_struct):
    """UV map and image texture for a face"""

    image: Image
    """ 

    :type: Image
    """

    uv: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv1: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv2: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv3: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv4: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv_raw: bpy_prop_array[float]
    """ Fixed size UV coordinates array

    :type: bpy_prop_array[float]
    """

class MeshTextureFaceLayer(bpy_struct):
    """UV map with assigned image textures in a Mesh data-block"""

    active: bool | None
    """ Set the map as active for display and editing

    :type: bool | None
    """

    active_clone: bool | None
    """ Set the map as active for cloning

    :type: bool | None
    """

    active_render: bool | None
    """ Set the map as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshTextureFace]
    """ 

    :type: bpy_prop_collection[MeshTextureFace]
    """

    name: str
    """ Name of UV map

    :type: str
    """

class MeshTexturePoly(bpy_struct):
    """UV map and image texture for a face"""

    image: Image
    """ 

    :type: Image
    """

class MeshTexturePolyLayer(bpy_struct):
    """UV map with assigned image textures in a Mesh data-block"""

    active: bool | None
    """ Set the map as active for display and editing

    :type: bool | None
    """

    active_clone: bool | None
    """ Set the map as active for cloning

    :type: bool | None
    """

    active_render: bool | None
    """ Set the map as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshTexturePoly]
    """ 

    :type: bpy_prop_collection[MeshTexturePoly]
    """

    name: str
    """ Name of UV map

    :type: str
    """

class MeshUVLoop(bpy_struct):
    pin_uv: bool
    """ 

    :type: bool
    """

    select: bool
    """ 

    :type: bool
    """

    select_edge: bool
    """ 

    :type: bool
    """

    uv: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class MeshUVLoopLayer(bpy_struct):
    data: bpy_prop_collection[MeshUVLoop]
    """ 

    :type: bpy_prop_collection[MeshUVLoop]
    """

    name: str
    """ Name of UV map

    :type: str
    """

class MeshVertex(bpy_struct):
    """Vertex in a Mesh data-block"""

    bevel_weight: float
    """ Weight used by the Bevel modifier 'Only Vertices' option

    :type: float
    """

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    groups: bpy_prop_collection[VertexGroupElement]
    """ Weights for the vertex groups this vertex is member of

    :type: bpy_prop_collection[VertexGroupElement]
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this vertex

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Vertex Normal

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    undeformed_co: bpy_prop_array[float]
    """ For meshes with modifiers applied, the coordinate of the vertex with no deforming modifiers applied, as used for generated texture coordinates

    :type: bpy_prop_array[float]
    """

class MeshVertexFloatProperty(bpy_struct):
    """User defined floating point number value in a float properties layer"""

    value: float
    """ 

    :type: float
    """

class MeshVertexFloatPropertyLayer(bpy_struct):
    """User defined layer of floating point number values"""

    data: bpy_prop_collection[MeshVertexFloatProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexFloatProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshVertexIntProperty(bpy_struct):
    """User defined integer number value in an integer properties layer"""

    value: int
    """ 

    :type: int
    """

class MeshVertexIntPropertyLayer(bpy_struct):
    """User defined layer of integer number values"""

    data: bpy_prop_collection[MeshVertexIntProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexIntProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshVertexStringProperty(bpy_struct):
    """User defined string text value in a string properties layer"""

    value: str
    """ 

    :type: str
    """

class MeshVertexStringPropertyLayer(bpy_struct):
    """User defined layer of string text values"""

    data: bpy_prop_collection[MeshVertexStringProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexStringProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshVertices(bpy_prop_collection[MeshVertex], bpy_struct):
    """Collection of mesh vertices"""

    def add(self, count: typing.Any | None = 0):
        """add

        :param count: Count, Number of vertices to add
        :type count: typing.Any | None
        """
        ...

class MetaBallElements(bpy_prop_collection[MetaElement], bpy_struct):
    """Collection of metaball elements"""

    active: MetaElement
    """ Last selected element

    :type: MetaElement
    """

    def new(self, type: str | None = "BALL") -> MetaElement:
        """Add a new element to the metaball

        :param type: type for the new meta-element
        :type type: str | None
        :return: The newly created meta-element
        :rtype: MetaElement
        """
        ...

    def remove(self, element: MetaElement):
        """Remove an element from the metaball

        :param element: The element to remove
        :type element: MetaElement
        """
        ...

    def clear(self):
        """Remove all elements from the metaball"""
        ...

class MetaElement(bpy_struct):
    """Blobby element in a Metaball data-block"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ Hide element

    :type: bool
    """

    radius: float
    """ 

    :type: float
    """

    rotation: bpy_prop_array[float]
    """ Normalized quaternion rotation

    :type: bpy_prop_array[float]
    """

    size_x: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    size_y: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    size_z: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    stiffness: float
    """ Stiffness defines how much of the element to fill

    :type: float
    """

    type: str
    """ Metaball types

    :type: str
    """

    use_negative: bool
    """ Set metaball as negative one

    :type: bool
    """

class Modifier(bpy_struct):
    """Modifier affecting the geometry data of an object"""

    name: str
    """ Modifier name

    :type: str
    """

    show_expanded: bool
    """ Set modifier expanded in the user interface

    :type: bool
    """

    show_in_editmode: bool
    """ Display modifier in Edit mode

    :type: bool
    """

    show_on_cage: bool
    """ Adjust edit cage to modifier result

    :type: bool
    """

    show_render: bool
    """ Use modifier during render

    :type: bool
    """

    show_viewport: bool
    """ Display modifier in viewport

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

    use_apply_on_spline: bool
    """ Apply this and all preceding deformation modifiers on splines' points rather than on filled curve/surface

    :type: bool
    """

class MotionPath(bpy_struct):
    """Cache of the worldspace positions of an element over a frame range"""

    frame_end: int
    """ End frame of the stored range

    :type: int
    """

    frame_start: int
    """ Starting frame of the stored range

    :type: int
    """

    is_modified: bool
    """ Path is being edited

    :type: bool
    """

    length: int
    """ Number of frames cached

    :type: int
    """

    points: bpy_prop_collection[MotionPathVert]
    """ Cached positions per frame

    :type: bpy_prop_collection[MotionPathVert]
    """

    use_bone_head: bool
    """ For PoseBone paths, use the bone head location when calculating this path

    :type: bool
    """

class MotionPathVert(bpy_struct):
    """Cached location on path"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Path point is selected for editing

    :type: bool
    """

class MovieClipProxy(bpy_struct):
    """Proxy parameters for a movie clip"""

    build_100: bool
    """ Build proxy resolution 100% of the original footage dimension

    :type: bool
    """

    build_25: bool
    """ Build proxy resolution 25% of the original footage dimension

    :type: bool
    """

    build_50: bool
    """ Build proxy resolution 50% of the original footage dimension

    :type: bool
    """

    build_75: bool
    """ Build proxy resolution 75% of the original footage dimension

    :type: bool
    """

    build_free_run: bool
    """ Build free run time code index

    :type: bool
    """

    build_free_run_rec_date: bool
    """ Build free run time code index using Record Date/Time

    :type: bool
    """

    build_record_run: bool
    """ Build record run time code index

    :type: bool
    """

    build_undistorted_100: bool
    """ Build proxy resolution 100% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_25: bool
    """ Build proxy resolution 25% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_50: bool
    """ Build proxy resolution 50% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_75: bool
    """ Build proxy resolution 75% of the original undistorted footage dimension

    :type: bool
    """

    directory: str
    """ Location to store the proxy files

    :type: str
    """

    quality: int
    """ JPEG quality of proxy images

    :type: int
    """

    timecode: str
    """ 

    :type: str
    """

class MovieClipScopes(bpy_struct):
    """Scopes for statistical view of a movie clip"""

    ...

class MovieClipUser(bpy_struct):
    """Parameters defining how a MovieClip data-block is used by another data-block"""

    frame_current: int
    """ Current frame number in movie or image sequence

    :type: int
    """

    proxy_render_size: str
    """ Draw preview using full resolution or different proxy resolutions

    :type: str
    """

    use_render_undistorted: bool
    """ Render preview using undistorted proxy

    :type: bool
    """

class MovieReconstructedCamera(bpy_struct):
    """Match-moving reconstructed camera data from tracker"""

    average_error: float
    """ Average error of reconstruction

    :type: float
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    matrix: bpy_prop_array[float]
    """ Worldspace transformation matrix

    :type: bpy_prop_array[float]
    """

class MovieTracking(bpy_struct):
    """Match-moving data for tracking"""

    active_object_index: int | None
    """ Index of active object

    :type: int | None
    """

    camera: MovieTrackingCamera
    """ 

    :type: MovieTrackingCamera
    """

    dopesheet: MovieTrackingDopesheet
    """ 

    :type: MovieTrackingDopesheet
    """

    objects: MovieTrackingObjects
    """ Collection of objects in this tracking data object

    :type: MovieTrackingObjects
    """

    plane_tracks: MovieTrackingPlaneTracks
    """ Collection of plane tracks in this tracking data object

    :type: MovieTrackingPlaneTracks
    """

    reconstruction: MovieTrackingReconstruction
    """ 

    :type: MovieTrackingReconstruction
    """

    settings: MovieTrackingSettings
    """ 

    :type: MovieTrackingSettings
    """

    stabilization: MovieTrackingStabilization
    """ 

    :type: MovieTrackingStabilization
    """

    tracks: MovieTrackingTracks
    """ Collection of tracks in this tracking data object

    :type: MovieTrackingTracks
    """

class MovieTrackingCamera(bpy_struct):
    """Match-moving camera data for tracking"""

    distortion_model: str
    """ Distortion model used for camera lenses

    :type: str
    """

    division_k1: float
    """ First coefficient of second order division distortion

    :type: float
    """

    division_k2: float
    """ First coefficient of second order division distortion

    :type: float
    """

    focal_length: float
    """ Camera's focal length

    :type: float
    """

    focal_length_pixels: float
    """ Camera's focal length

    :type: float
    """

    k1: float
    """ First coefficient of third order polynomial radial distortion

    :type: float
    """

    k2: float
    """ Second coefficient of third order polynomial radial distortion

    :type: float
    """

    k3: float
    """ Third coefficient of third order polynomial radial distortion

    :type: float
    """

    pixel_aspect: float
    """ Pixel aspect ratio

    :type: float
    """

    principal: bpy_prop_array[float]
    """ Optical center of lens

    :type: bpy_prop_array[float]
    """

    sensor_width: float
    """ Width of CCD sensor in millimeters

    :type: float
    """

    units: str
    """ Units used for camera focal length

    :type: str
    """

class MovieTrackingDopesheet(bpy_struct):
    """Match-moving dopesheet data"""

    show_hidden: bool
    """ Include channels from objects/bone that aren't visible

    :type: bool
    """

    show_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    sort_method: str
    """ Method to be used to sort channels in dopesheet view

    :type: str
    """

    use_invert_sort: bool
    """ Invert sort order of dopesheet channels

    :type: bool
    """

class MovieTrackingMarker(bpy_struct):
    """Match-moving marker data for tracking"""

    co: bpy_prop_array[float]
    """ Marker position at frame in normalized coordinates

    :type: bpy_prop_array[float]
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    is_keyed: bool
    """ Whether the position of the marker is keyframed or tracked

    :type: bool
    """

    mute: bool
    """ Is marker muted for current frame

    :type: bool
    """

    pattern_bound_box: bpy_prop_array[float]
    """ Pattern area bounding box in normalized coordinates

    :type: bpy_prop_array[float]
    """

    pattern_corners: bpy_prop_array[float]
    """ Array of coordinates which represents pattern's corners in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

    search_max: bpy_prop_array[float]
    """ Right-bottom corner of search area in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

    search_min: bpy_prop_array[float]
    """ Left-bottom corner of search area in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

class MovieTrackingMarkers(bpy_prop_collection[MovieTrackingMarker], bpy_struct):
    """Collection of markers for movie tracking track"""

    def find_frame(
        self, frame: int | None, exact: bool | typing.Any | None = True
    ) -> MovieTrackingMarker:
        """Get marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get marker at exact frame number rather than get estimated marker
        :type exact: bool | typing.Any | None
        :return: Marker for specified frame
        :rtype: MovieTrackingMarker
        """
        ...

    def insert_frame(
        self, frame: int | None, co: typing.Any | None = (0.0, 0.0)
    ) -> MovieTrackingMarker:
        """Insert a new marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :param co: Coordinate, Place new marker at the given frame using specified in normalized space coordinates
        :type co: typing.Any | None
        :return: Newly created marker
        :rtype: MovieTrackingMarker
        """
        ...

    def delete_frame(self, frame: int | None):
        """Delete marker at specified frame

        :param frame: Frame, Frame number to delete marker from
        :type frame: int | None
        """
        ...

class MovieTrackingObject(bpy_struct):
    """Match-moving object tracking and reconstruction data"""

    is_camera: bool
    """ Object is used for camera tracking

    :type: bool
    """

    keyframe_a: int
    """ First keyframe used for reconstruction initialization

    :type: int
    """

    keyframe_b: int
    """ Second keyframe used for reconstruction initialization

    :type: int
    """

    name: str
    """ Unique name of object

    :type: str
    """

    plane_tracks: MovieTrackingObjectPlaneTracks
    """ Collection of plane tracks in this tracking data object

    :type: MovieTrackingObjectPlaneTracks
    """

    reconstruction: MovieTrackingReconstruction
    """ 

    :type: MovieTrackingReconstruction
    """

    scale: float
    """ Scale of object solution in camera space

    :type: float
    """

    tracks: MovieTrackingObjectTracks
    """ Collection of tracks in this tracking data object

    :type: MovieTrackingObjectTracks
    """

class MovieTrackingObjectPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of tracking plane tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

class MovieTrackingObjectTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    def new(
        self, name: str | typing.Any = "", frame: typing.Any | None = 1
    ) -> MovieTrackingTrack:
        """create new motion track in this movie clip

        :param name: Name of new track
        :type name: str | typing.Any
        :param frame: Frame, Frame number to add tracks on
        :type frame: typing.Any | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """
        ...

class MovieTrackingObjects(bpy_prop_collection[MovieTrackingObject], bpy_struct):
    """Collection of movie tracking objects"""

    active: MovieTrackingObject | None
    """ Active object in this tracking data object

    :type: MovieTrackingObject | None
    """

    def new(self, name: str | typing.Any) -> MovieTrackingObject:
        """Add tracking object to this movie clip

        :param name: Name of new object
        :type name: str | typing.Any
        :return: New motion tracking object
        :rtype: MovieTrackingObject
        """
        ...

    def remove(self, object: MovieTrackingObject):
        """Remove tracking object from this movie clip

        :param object: Motion tracking object to be removed
        :type object: MovieTrackingObject
        """
        ...

class MovieTrackingPlaneMarker(bpy_struct):
    """Match-moving plane marker data for tracking"""

    corners: bpy_prop_array[float]
    """ Array of coordinates which represents UI rectangle corners in frame normalized coordinates

    :type: bpy_prop_array[float]
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    mute: bool
    """ Is marker muted for current frame

    :type: bool
    """

class MovieTrackingPlaneMarkers(
    bpy_prop_collection[MovieTrackingPlaneMarker], bpy_struct
):
    """Collection of markers for movie tracking plane track"""

    def find_frame(
        self, frame: int | None, exact: bool | typing.Any | None = True
    ) -> MovieTrackingPlaneMarker:
        """Get plane marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get plane marker at exact frame number rather than get estimated marker
        :type exact: bool | typing.Any | None
        :return: Plane marker for specified frame
        :rtype: MovieTrackingPlaneMarker
        """
        ...

    def insert_frame(self, frame: int | None) -> MovieTrackingPlaneMarker:
        """Insert a new plane marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :return: Newly created plane marker
        :rtype: MovieTrackingPlaneMarker
        """
        ...

    def delete_frame(self, frame: int | None):
        """Delete plane marker at specified frame

        :param frame: Frame, Frame number to delete plane marker from
        :type frame: int | None
        """
        ...

class MovieTrackingPlaneTrack(bpy_struct):
    """Match-moving plane track data for tracking"""

    image: Image
    """ Image displayed in the track during editing in clip editor

    :type: Image
    """

    image_opacity: float
    """ Opacity of the image

    :type: float
    """

    markers: MovieTrackingPlaneMarkers
    """ Collection of markers in track

    :type: MovieTrackingPlaneMarkers
    """

    name: str
    """ Unique name of track

    :type: str
    """

    select: bool
    """ Plane track is selected

    :type: bool
    """

    use_auto_keying: bool
    """ Automatic keyframe insertion when moving plane corners

    :type: bool
    """

class MovieTrackingPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of movie tracking plane tracks"""

    active: MovieTrackingPlaneTrack | None
    """ Active plane track in this tracking data object

    :type: MovieTrackingPlaneTrack | None
    """

class MovieTrackingReconstructedCameras(
    bpy_prop_collection[MovieReconstructedCamera], bpy_struct
):
    """Collection of solved cameras"""

    def find_frame(self, frame: typing.Any | None = 1) -> MovieReconstructedCamera:
        """Find a reconstructed camera for a give frame number

        :param frame: Frame, Frame number to find camera for
        :type frame: typing.Any | None
        :return: Camera for a given frame
        :rtype: MovieReconstructedCamera
        """
        ...

    def matrix_from_frame(self, frame: typing.Any | None = 1) -> bpy_prop_array[float]:
        """Return interpolated camera matrix for a given frame

        :param frame: Frame, Frame number to find camera for
        :type frame: typing.Any | None
        :return: Matrix, Interpolated camera matrix for a given frame
        :rtype: bpy_prop_array[float]
        """
        ...

class MovieTrackingReconstruction(bpy_struct):
    """Match-moving reconstruction data from tracker"""

    average_error: float
    """ Average error of reconstruction

    :type: float
    """

    cameras: MovieTrackingReconstructedCameras
    """ Collection of solved cameras

    :type: MovieTrackingReconstructedCameras
    """

    is_valid: bool
    """ Is tracking data contains valid reconstruction information

    :type: bool
    """

class MovieTrackingSettings(bpy_struct):
    """Match moving settings"""

    clean_action: str
    """ Cleanup action to execute

    :type: str
    """

    clean_error: float
    """ Effect on tracks which have a larger re-projection error

    :type: float
    """

    clean_frames: int
    """ Effect on tracks which are tracked less than the specified amount of frames

    :type: int
    """

    default_correlation_min: float
    """ Default minimum value of correlation between matched pattern and reference that is still treated as successful tracking

    :type: float
    """

    default_frames_limit: int
    """ Every tracking cycle, this number of frames are tracked

    :type: int
    """

    default_margin: int
    """ Default distance from image boundary at which marker stops tracking

    :type: int
    """

    default_motion_model: str
    """ Default motion model to use for tracking

    :type: str
    """

    default_pattern_match: str
    """ Track pattern from given frame when tracking marker to next frame

    :type: str
    """

    default_pattern_size: int
    """ Size of pattern area for newly created tracks

    :type: int
    """

    default_search_size: int
    """ Size of search area for newly created tracks

    :type: int
    """

    default_weight: float
    """ Influence of newly created track on a final solution

    :type: float
    """

    distance: float
    """ Distance between two bundles used for scene scaling

    :type: float
    """

    object_distance: float
    """ Distance between two bundles used for object scaling

    :type: float
    """

    refine_intrinsics: str
    """ Refine intrinsics during camera solving

    :type: str
    """

    show_default_expanded: bool
    """ Show default options expanded in the user interface

    :type: bool
    """

    show_extra_expanded: bool
    """ Show extra options expanded in the user interface

    :type: bool
    """

    speed: str
    """ Limit speed of tracking to make visual feedback easier (this does not affect the tracking quality)

    :type: str
    """

    use_default_blue_channel: bool
    """ Use blue channel from footage for tracking

    :type: bool
    """

    use_default_brute: bool
    """ Use a brute-force translation-only initialization when tracking

    :type: bool
    """

    use_default_green_channel: bool
    """ Use green channel from footage for tracking

    :type: bool
    """

    use_default_mask: bool
    """ Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking

    :type: bool
    """

    use_default_normalization: bool
    """ Normalize light intensities while tracking (slower)

    :type: bool
    """

    use_default_red_channel: bool
    """ Use red channel from footage for tracking

    :type: bool
    """

    use_keyframe_selection: bool
    """ Automatically select keyframes when solving camera/object motion

    :type: bool
    """

    use_tripod_solver: bool
    """ Use special solver to track a stable camera position, such as a tripod

    :type: bool
    """

class MovieTrackingStabilization(bpy_struct):
    """2D stabilization based on tracking markers"""

    active_rotation_track_index: int | None
    """ Index of active track in rotation stabilization tracks list

    :type: int | None
    """

    active_track_index: int | None
    """ Index of active track in translation stabilization tracks list

    :type: int | None
    """

    anchor_frame: int
    """ Reference point to anchor stabilization (other frames will be adjusted relative to this frame's position)

    :type: int
    """

    filter_type: str
    """ Interpolation to use for sub-pixel shifts and rotations due to stabilization

    :type: str
    """

    influence_location: float
    """ Influence of stabilization algorithm on footage location

    :type: float
    """

    influence_rotation: float
    """ Influence of stabilization algorithm on footage rotation

    :type: float
    """

    influence_scale: float
    """ Influence of stabilization algorithm on footage scale

    :type: float
    """

    rotation_tracks: bpy_prop_collection[MovieTrackingTrack]
    """ Collection of tracks used for 2D stabilization (translation)

    :type: bpy_prop_collection[MovieTrackingTrack]
    """

    scale_max: float
    """ Limit the amount of automatic scaling

    :type: float
    """

    show_tracks_expanded: bool
    """ Show UI list of tracks participating in stabilization

    :type: bool
    """

    target_position: bpy_prop_array[float]
    """ Known relative offset of original shot, will be subtracted (e.g. for panning shot, can be animated)

    :type: bpy_prop_array[float]
    """

    target_rotation: float
    """ Rotation present on original shot, will be compensated (e.g. for deliberate tilting)

    :type: float
    """

    target_scale: float
    """ Explicitly scale resulting frame to compensate zoom of original shot

    :type: float
    """

    tracks: bpy_prop_collection[MovieTrackingTrack]
    """ Collection of tracks used for 2D stabilization (translation)

    :type: bpy_prop_collection[MovieTrackingTrack]
    """

    use_2d_stabilization: bool
    """ Use 2D stabilization for footage

    :type: bool
    """

    use_autoscale: bool
    """ Automatically scale footage to cover unfilled areas when stabilizing

    :type: bool
    """

    use_stabilize_rotation: bool
    """ Stabilize detected rotation around center of frame

    :type: bool
    """

    use_stabilize_scale: bool
    """ Compensate any scale changes relative to center of rotation

    :type: bool
    """

class MovieTrackingTrack(bpy_struct):
    """Match-moving track data for tracking"""

    average_error: float
    """ Average error of re-projection

    :type: float
    """

    bundle: bpy_prop_array[float]
    """ Position of bundle reconstructed from this track

    :type: bpy_prop_array[float]
    """

    color: bpy_prop_array[float]
    """ Color of the track in the Movie Clip Editor and the 3D viewport after a solve

    :type: bpy_prop_array[float]
    """

    correlation_min: float
    """ Minimal value of correlation between matched pattern and reference that is still treated as successful tracking

    :type: float
    """

    frames_limit: int
    """ Every tracking cycle, this number of frames are tracked

    :type: int
    """

    grease_pencil: GreasePencil
    """ Grease pencil data for this track

    :type: GreasePencil
    """

    has_bundle: bool
    """ True if track has a valid bundle

    :type: bool
    """

    hide: bool
    """ Track is hidden

    :type: bool
    """

    lock: bool
    """ Track is locked and all changes to it are disabled

    :type: bool
    """

    margin: int
    """ Distance from image boundary at which marker stops tracking

    :type: int
    """

    markers: MovieTrackingMarkers
    """ Collection of markers in track

    :type: MovieTrackingMarkers
    """

    motion_model: str
    """ Default motion model to use for tracking

    :type: str
    """

    name: str
    """ Unique name of track

    :type: str
    """

    offset: bpy_prop_array[float]
    """ Offset of track from the parenting point

    :type: bpy_prop_array[float]
    """

    pattern_match: str
    """ Track pattern from given frame when tracking marker to next frame

    :type: str
    """

    select: bool
    """ Track is selected

    :type: bool
    """

    select_anchor: bool
    """ Track's anchor point is selected

    :type: bool
    """

    select_pattern: bool
    """ Track's pattern area is selected

    :type: bool
    """

    select_search: bool
    """ Track's search area is selected

    :type: bool
    """

    use_alpha_preview: bool
    """ Apply track's mask on displaying preview

    :type: bool
    """

    use_blue_channel: bool
    """ Use blue channel from footage for tracking

    :type: bool
    """

    use_brute: bool
    """ Use a brute-force translation only pre-track before refinement

    :type: bool
    """

    use_custom_color: bool
    """ Use custom color instead of theme-defined

    :type: bool
    """

    use_grayscale_preview: bool
    """ Display what the tracking algorithm sees in the preview

    :type: bool
    """

    use_green_channel: bool
    """ Use green channel from footage for tracking

    :type: bool
    """

    use_mask: bool
    """ Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking

    :type: bool
    """

    use_normalization: bool
    """ Normalize light intensities while tracking. Slower

    :type: bool
    """

    use_red_channel: bool
    """ Use red channel from footage for tracking

    :type: bool
    """

    weight: float
    """ Influence of this track on a final solution

    :type: float
    """

    weight_stab: float
    """ Influence of this track on 2D stabilization

    :type: float
    """

class MovieTrackingTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    def new(
        self, name: str | typing.Any = "", frame: typing.Any | None = 1
    ) -> MovieTrackingTrack:
        """Create new motion track in this movie clip

        :param name: Name of new track
        :type name: str | typing.Any
        :param frame: Frame, Frame number to add track on
        :type frame: typing.Any | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """
        ...

class NlaStrip(bpy_struct):
    """A container referencing an existing Action"""

    action: Action
    """ Action referenced by this strip

    :type: Action
    """

    action_frame_end: float
    """ Last frame from action to use

    :type: float
    """

    action_frame_start: float
    """ First frame from action to use

    :type: float
    """

    active: bool
    """ NLA Strip is active

    :type: bool
    """

    blend_in: float
    """ Number of frames at start of strip to fade in influence

    :type: float
    """

    blend_out: float
    """ 

    :type: float
    """

    blend_type: str
    """ Method used for combining strip's result with accumulated result

    :type: str
    """

    extrapolation: str
    """ Action to take for gaps past the strip extents

    :type: str
    """

    fcurves: NlaStripFCurves
    """ F-Curves for controlling the strip's influence and timing

    :type: NlaStripFCurves
    """

    frame_end: float
    """ 

    :type: float
    """

    frame_start: float
    """ 

    :type: float
    """

    influence: float
    """ Amount the strip contributes to the current result

    :type: float
    """

    modifiers: bpy_prop_collection[FModifier]
    """ Modifiers affecting all the F-Curves in the referenced Action

    :type: bpy_prop_collection[FModifier]
    """

    mute: bool
    """ NLA Strip is not evaluated

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    repeat: float
    """ Number of times to repeat the action range

    :type: float
    """

    scale: float
    """ Scaling factor for action

    :type: float
    """

    select: bool
    """ NLA Strip is selected

    :type: bool
    """

    strip_time: float
    """ Frame of referenced Action to evaluate

    :type: float
    """

    strips: bpy_prop_collection[NlaStrip]
    """ NLA Strips that this strip acts as a container for (if it is of type Meta)

    :type: bpy_prop_collection[NlaStrip]
    """

    type: str
    """ Type of NLA Strip

    :type: str
    """

    use_animated_influence: bool
    """ Influence setting is controlled by an F-Curve rather than automatically determined

    :type: bool
    """

    use_animated_time: bool
    """ Strip time is controlled by an F-Curve rather than automatically determined

    :type: bool
    """

    use_animated_time_cyclic: bool
    """ Cycle the animated time within the action start & end

    :type: bool
    """

    use_auto_blend: bool
    """ Number of frames for Blending In/Out is automatically determined from overlapping strips

    :type: bool
    """

    use_reverse: bool
    """ NLA Strip is played back in reverse order (only when timing is automatically determined)

    :type: bool
    """

    use_sync_length: bool
    """ Update range of frames referenced from action after tweaking strip and its keyframes

    :type: bool
    """

class NlaStripFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of NLA strip F-Curves"""

    def find(self, data_path: str | typing.Any, index: typing.Any | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the NLA strip.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str | typing.Any
        :param index: Index, Array index
        :type index: typing.Any | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """
        ...

class NlaStrips(bpy_prop_collection[NlaStrip], bpy_struct):
    """Collection of Nla Strips"""

    def new(
        self, name: str | typing.Any, start: int | None, action: Action
    ) -> NlaStrip:
        """Add a new Action-Clip strip to the track

        :param name: Name for the NLA Strips
        :type name: str | typing.Any
        :param start: Start Frame, Start frame for this strip
        :type start: int | None
        :param action: Action to assign to this strip
        :type action: Action
        :return: New NLA Strip
        :rtype: NlaStrip
        """
        ...

    def remove(self, strip: NlaStrip):
        """Remove a NLA Strip

        :param strip: NLA Strip to remove
        :type strip: NlaStrip
        """
        ...

class NlaTrack(bpy_struct):
    """A animation layer containing Actions referenced as NLA strips"""

    active: bool
    """ NLA Track is active

    :type: bool
    """

    is_solo: bool
    """ NLA Track is evaluated itself (i.e. active Action and all other NLA Tracks in the same AnimData block are disabled)

    :type: bool
    """

    lock: bool
    """ NLA Track is locked

    :type: bool
    """

    mute: bool
    """ NLA Track is not evaluated

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ NLA Track is selected

    :type: bool
    """

    strips: NlaStrips
    """ NLA Strips on this NLA-track

    :type: NlaStrips
    """

class NlaTracks(bpy_prop_collection[NlaTrack], bpy_struct):
    """Collection of NLA Tracks"""

    active: NlaTrack | None
    """ Active Object constraint

    :type: NlaTrack | None
    """

    def new(self, prev: NlaTrack | None = None) -> NlaTrack:
        """Add a new NLA Track

        :param prev: NLA Track to add the new one after
        :type prev: NlaTrack | None
        :return: New NLA Track
        :rtype: NlaTrack
        """
        ...

    def remove(self, track: NlaTrack):
        """Remove a NLA Track

        :param track: NLA Track to remove
        :type track: NlaTrack
        """
        ...

class Node(bpy_struct):
    """Node in a node tree"""

    bl_description: str
    """ 

    :type: str
    """

    bl_height_default: float
    """ 

    :type: float
    """

    bl_height_max: float
    """ 

    :type: float
    """

    bl_height_min: float
    """ 

    :type: float
    """

    bl_icon: str
    """ The node icon

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ The node label

    :type: str
    """

    bl_static_type: str
    """ Node type (deprecated, use with care)

    :type: str
    """

    bl_width_default: float
    """ 

    :type: float
    """

    bl_width_max: float
    """ 

    :type: float
    """

    bl_width_min: float
    """ 

    :type: float
    """

    color: bpy_prop_array[float]
    """ Custom color of the node body

    :type: bpy_prop_array[float]
    """

    dimensions: bpy_prop_array[float]
    """ Absolute bounding box dimensions of the node

    :type: bpy_prop_array[float]
    """

    height: float
    """ Height of the node

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    inputs: NodeInputs
    """ 

    :type: NodeInputs
    """

    internal_links: bpy_prop_collection[NodeLink]
    """ Internal input-to-output connections for muting

    :type: bpy_prop_collection[NodeLink]
    """

    label: str
    """ Optional custom node label

    :type: str
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    mute: bool
    """ 

    :type: bool
    """

    name: str
    """ Unique node identifier

    :type: str
    """

    outputs: NodeOutputs
    """ 

    :type: NodeOutputs
    """

    parent: Node
    """ Parent this node is attached to

    :type: Node
    """

    select: bool
    """ Node selection state

    :type: bool
    """

    shading_compatibility: set[str]
    """ 

    :type: set[str]
    """

    show_options: bool
    """ 

    :type: bool
    """

    show_preview: bool
    """ 

    :type: bool
    """

    show_texture: bool
    """ Draw node in viewport textured draw mode

    :type: bool
    """

    type: str
    """ Node type (deprecated, use bl_static_type or bl_idname for the actual identifier string)

    :type: str
    """

    use_custom_color: bool
    """ Use custom color for the node

    :type: bool
    """

    width: float
    """ Width of the node

    :type: float
    """

    width_hidden: float
    """ Width of the node in hidden state

    :type: float
    """

    def socket_value_update(self, context: Context):
        """Update after property changes

        :param context:
        :type context: Context
        """
        ...

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """
        ...

    @classmethod
    def poll(cls, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node type can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """
        ...

    def poll_instance(self, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """
        ...

    def update(self):
        """Update on editor changes"""
        ...

    def insert_link(self, link: NodeLink):
        """Handle creation of a link to or from the node

        :param link: Link, Node link that will be inserted
        :type link: NodeLink
        """
        ...

    def init(self, context: Context):
        """Initialize a new instance of this node

        :param context:
        :type context: Context
        """
        ...

    def copy(self, node: Node):
        """Initialize a new instance of this node from an existing node

        :param node: Node, Existing node to copy
        :type node: Node
        """
        ...

    def free(self):
        """Clean up node on removal"""
        ...

    def draw_buttons(self, context: Context, layout: UILayout):
        """Draw node buttons

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_buttons_ext(self, context: Context, layout: UILayout):
        """Draw node buttons in the sidebar

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_label(self) -> str | typing.Any:
        """Returns a dynamic label string

        :return: Label
        :rtype: str | typing.Any
        """
        ...

class NodeInputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(
        self,
        type: str | typing.Any,
        name: str | typing.Any,
        identifier: str | typing.Any = "",
    ) -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

class NodeInstanceHash(bpy_struct):
    """Hash table containing node instance data"""

    ...

class NodeInternalSocketTemplate(bpy_struct):
    """Type and default value of a node socket"""

    identifier: str
    """ Identifier of the socket

    :type: str
    """

    name: str
    """ Name of the socket

    :type: str
    """

    type: str
    """ Data type of the socket

    :type: str
    """

class NodeLink(bpy_struct):
    """Link is valid"""

    from_node: Node
    """ 

    :type: Node
    """

    from_socket: NodeSocket
    """ 

    :type: NodeSocket
    """

    is_hidden: bool
    """ Link is hidden due to invisible sockets

    :type: bool
    """

    is_valid: bool
    """ 

    :type: bool
    """

    to_node: Node
    """ 

    :type: Node
    """

    to_socket: NodeSocket
    """ 

    :type: NodeSocket
    """

class NodeLinks(bpy_prop_collection[NodeLink], bpy_struct):
    """Collection of Node Links"""

    def new(
        self,
        input: NodeSocket,
        output: NodeSocket,
        verify_limits: bool | typing.Any | None = True,
    ) -> NodeLink:
        """Add a node link to this node tree

        :param input: The input socket
        :type input: NodeSocket
        :param output: The output socket
        :type output: NodeSocket
        :param verify_limits: Verify Limits, Remove existing links if connection limit is exceeded
        :type verify_limits: bool | typing.Any | None
        :return: New node link
        :rtype: NodeLink
        """
        ...

    def remove(self, link: NodeLink):
        """remove a node link from the node tree

        :param link: The node link to remove
        :type link: NodeLink
        """
        ...

    def clear(self):
        """remove all node links from the node tree"""
        ...

class NodeOutputFileSlotFile(bpy_struct):
    """Single layer file slot of the file output node"""

    format: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    path: str
    """ Subpath used for this slot

    :type: str
    """

    use_node_format: bool
    """ 

    :type: bool
    """

class NodeOutputFileSlotLayer(bpy_struct):
    """Multilayer slot of the file output node"""

    name: str
    """ OpenEXR layer name used for this slot

    :type: str
    """

class NodeOutputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(
        self,
        type: str | typing.Any,
        name: str | typing.Any,
        identifier: str | typing.Any = "",
    ) -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str | typing.Any
        :return: New socket
        :rtype: NodeSocket
        """
        ...

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

class NodeSocket(bpy_struct):
    """Input or output socket of a node"""

    bl_idname: str
    """ 

    :type: str
    """

    enabled: bool
    """ Enable the socket

    :type: bool
    """

    hide: bool
    """ Hide the socket

    :type: bool
    """

    hide_value: bool
    """ Hide the socket input value

    :type: bool
    """

    identifier: str
    """ Unique identifier for mapping sockets

    :type: str
    """

    is_linked: bool
    """ True if the socket is connected

    :type: bool
    """

    is_output: bool
    """ True if the socket is an output, otherwise input

    :type: bool
    """

    link_limit: int
    """ Max number of links allowed for this socket

    :type: int
    """

    name: str
    """ Socket name

    :type: str
    """

    node: Node
    """ Node owning this socket

    :type: Node
    """

    show_expanded: bool
    """ Socket links are expanded in the user interface

    :type: bool
    """

    type: str
    """ Data type

    :type: str
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

    def draw(
        self, context: Context, layout: UILayout, node: Node, text: str | typing.Any
    ):
        """Draw socket

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        :param node: Node, Node the socket belongs to
        :type node: Node
        :param text: Text, Text label to draw alongside properties
        :type text: str | typing.Any
        """
        ...

    def draw_color(self, context: Context, node: Node) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context
        :param node: Node, Node the socket belongs to
        :type node: Node
        :return: Color
        :rtype: bpy_prop_array[float]
        """
        ...

class NodeSocketInterface(bpy_struct):
    """Parameters to define node sockets"""

    bl_socket_idname: str
    """ 

    :type: str
    """

    identifier: str
    """ Unique identifier for mapping sockets

    :type: str
    """

    is_output: bool
    """ True if the socket is an output, otherwise input

    :type: bool
    """

    name: str
    """ Socket name

    :type: str
    """

    def draw(self, context: Context, layout: UILayout):
        """Draw template settings

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_color(self, context: Context) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context
        :return: Color
        :rtype: bpy_prop_array[float]
        """
        ...

    def register_properties(self, data_rna_type: Struct | None):
        """Define RNA properties of a socket

        :param data_rna_type: Data RNA Type, RNA type for special socket properties
        :type data_rna_type: Struct | None
        """
        ...

    def init_socket(self, node: Node, socket: NodeSocket, data_path: str | typing.Any):
        """Initialize a node socket instance

        :param node: Node, Node of the socket to initialize
        :type node: Node
        :param socket: Socket, Socket to initialize
        :type socket: NodeSocket
        :param data_path: Data Path, Path to specialized socket data
        :type data_path: str | typing.Any
        """
        ...

    def from_socket(self, node: Node, socket: NodeSocket):
        """Setup template parameters from an existing socket

        :param node: Node, Node of the original socket
        :type node: Node
        :param socket: Socket, Original socket
        :type socket: NodeSocket
        """
        ...

class NodeTreeInputs(bpy_prop_collection[NodeSocketInterface], bpy_struct):
    """Collection of Node Tree Sockets"""

    def new(
        self, type: str | typing.Any, name: str | typing.Any
    ) -> NodeSocketInterface:
        """Add a socket to this node tree

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocketInterface
        """
        ...

    def remove(self, socket: NodeSocketInterface | None):
        """Remove a socket from this node tree

        :param socket: The socket to remove
        :type socket: NodeSocketInterface | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node tree"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

class NodeTreeOutputs(bpy_prop_collection[NodeSocketInterface], bpy_struct):
    """Collection of Node Tree Sockets"""

    def new(
        self, type: str | typing.Any, name: str | typing.Any
    ) -> NodeSocketInterface:
        """Add a socket to this node tree

        :param type: Type, Data type
        :type type: str | typing.Any
        :param name: Name
        :type name: str | typing.Any
        :return: New socket
        :rtype: NodeSocketInterface
        """
        ...

    def remove(self, socket: NodeSocketInterface | None):
        """Remove a socket from this node tree

        :param socket: The socket to remove
        :type socket: NodeSocketInterface | None
        """
        ...

    def clear(self):
        """Remove all sockets from this node tree"""
        ...

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """
        ...

class NodeTreePath(bpy_struct):
    """Element of the node space tree path"""

    node_tree: NodeTree
    """ Base node tree from context

    :type: NodeTree
    """

class Nodes(bpy_prop_collection[Node], bpy_struct):
    """Collection of Nodes"""

    active: Node | None
    """ Active node in this tree

    :type: Node | None
    """

    def new(self, type: str | typing.Any) -> Node:
        """Add a node to this node tree

        :param type: Type, Type of node to add (Warning: should be same as node.bl_idname, not node.type!)
        :type type: str | typing.Any
        :return: New node
        :rtype: Node
        """
        ...

    def remove(self, node: Node):
        """Remove a node from this node tree

        :param node: The node to remove
        :type node: Node
        """
        ...

    def clear(self):
        """Remove all nodes from this node tree"""
        ...

class ObjectBase(bpy_struct):
    """An object instance in a scene"""

    layers: list[bool]
    """ Layers the object base is on

    :type: list[bool]
    """

    layers_local_view: list[bool]
    """ 3D local view layers the object base is on

    :type: list[bool]
    """

    object: Object
    """ Object this base links to

    :type: Object
    """

    select: bool
    """ Object base selection state

    :type: bool
    """

    def layers_from_view(self, view: SpaceView3D):
        """Sets the object layers from a 3D View (use when adding an object in local view)

        :param view:
        :type view: SpaceView3D
        """
        ...

class ObjectConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of object constraints"""

    active: Constraint | None
    """ Active Object constraint

    :type: Constraint | None
    """

    def new(self, type: str | None) -> Constraint:
        """Add a new constraint to this object

                :param type: Constraint type to add

        CAMERA_SOLVER Camera Solver.

        FOLLOW_TRACK Follow Track.

        OBJECT_SOLVER Object Solver.

        COPY_LOCATION Copy Location, Copy the location of a target (with an optional offset), so that they move together.

        COPY_ROTATION Copy Rotation, Copy the rotation of a target (with an optional offset), so that they rotate together.

        COPY_SCALE Copy Scale, Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.

        COPY_TRANSFORMS Copy Transforms, Copy all the transformations of a target, so that they move together.

        LIMIT_DISTANCE Limit Distance, Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).

        LIMIT_LOCATION Limit Location, Restrict movement along each axis within given ranges.

        LIMIT_ROTATION Limit Rotation, Restrict rotation along each axis within given ranges.

        LIMIT_SCALE Limit Scale, Restrict scaling along each axis with given ranges.

        MAINTAIN_VOLUME Maintain Volume, Compensate for scaling one axis by applying suitable scaling to the other two axes.

        TRANSFORM Transformation, Use one transform property from target to control another (or same) property on owner.

        TRANSFORM_CACHE Transform Cache, Look up the transformation matrix from an external file.

        CLAMP_TO Clamp To, Restrict movements to lie along a curve by remapping location along curve's longest axis.

        DAMPED_TRACK Damped Track, Point towards a target by performing the smallest rotation necessary.

        IK Inverse Kinematics, Control a chain of bones by specifying the endpoint target (Bones only).

        LOCKED_TRACK Locked Track, Rotate around the specified ('locked') axis to point towards a target.

        SPLINE_IK Spline IK, Align chain of bones along a curve (Bones only).

        STRETCH_TO Stretch To, Stretch along Y-Axis to point towards a target.

        TRACK_TO Track To, Legacy tracking constraint prone to twisting artifacts.

        ACTION Action, Use transform property of target to look up pose for owner from an Action.

        CHILD_OF Child Of, Make target the 'detachable' parent of owner.

        FLOOR Floor, Use position (and optionally rotation) of target to define a 'wall' or 'floor' that the owner can not cross.

        FOLLOW_PATH Follow Path, Use to animate an object/bone following a path.

        PIVOT Pivot, Change pivot point for transforms (buggy).

        RIGID_BODY_JOINT Rigid Body Joint, Use to define a Rigid Body Constraint (for Game Engine use only).

        SHRINKWRAP Shrinkwrap, Restrict movements to surface of target mesh.
                :type type: str | None
                :return: New constraint
                :rtype: Constraint
        """
        ...

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """
        ...

    def clear(self):
        """Remove all constraint from this object"""
        ...

class ObjectModifiers(bpy_prop_collection[Modifier], bpy_struct):
    """Collection of object modifiers"""

    def new(self, name: str | typing.Any, type: str | None) -> Modifier:
        """Add a new modifier

                :param name: New name for the modifier
                :type name: str | typing.Any
                :param type: Modifier type to add

        DATA_TRANSFER Data Transfer.

        MESH_CACHE Mesh Cache.

        MESH_SEQUENCE_CACHE Mesh Sequence Cache.

        NORMAL_EDIT Normal Edit.

        UV_PROJECT UV Project.

        UV_WARP UV Warp.

        VERTEX_WEIGHT_EDIT Vertex Weight Edit.

        VERTEX_WEIGHT_MIX Vertex Weight Mix.

        VERTEX_WEIGHT_PROXIMITY Vertex Weight Proximity.

        ARRAY Array.

        BEVEL Bevel.

        BOOLEAN Boolean.

        BUILD Build.

        DECIMATE Decimate.

        EDGE_SPLIT Edge Split.

        MASK Mask.

        MIRROR Mirror.

        MULTIRES Multiresolution.

        REMESH Remesh.

        SCREW Screw.

        SKIN Skin.

        SOLIDIFY Solidify.

        SUBSURF Subdivision Surface.

        TRIANGULATE Triangulate.

        WIREFRAME Wireframe, Generate a wireframe on the edges of a mesh.

        ARMATURE Armature.

        CAST Cast.

        CORRECTIVE_SMOOTH Corrective Smooth.

        CURVE Curve.

        DISPLACE Displace.

        HOOK Hook.

        LAPLACIANSMOOTH Laplacian Smooth.

        LAPLACIANDEFORM Laplacian Deform.

        LATTICE Lattice.

        MESH_DEFORM Mesh Deform.

        SHRINKWRAP Shrinkwrap.

        SIMPLE_DEFORM Simple Deform.

        SMOOTH Smooth.

        WARP Warp.

        WAVE Wave.

        CLOTH Cloth.

        COLLISION Collision.

        DYNAMIC_PAINT Dynamic Paint.

        EXPLODE Explode.

        FLUID_SIMULATION Fluid Simulation.

        OCEAN Ocean.

        PARTICLE_INSTANCE Particle Instance.

        PARTICLE_SYSTEM Particle System.

        SMOKE Smoke.

        SOFT_BODY Soft Body.

        SURFACE Surface.
                :type type: str | None
                :return: Newly created modifier
                :rtype: Modifier
        """
        ...

    def remove(self, modifier: Modifier):
        """Remove an existing modifier from the object

        :param modifier: Modifier to remove
        :type modifier: Modifier
        """
        ...

    def clear(self):
        """Remove all modifiers from the object"""
        ...

class OceanTexData(bpy_struct):
    """Ocean Texture settings"""

    ocean_object: Object
    """ Object containing the ocean modifier

    :type: Object
    """

    output: str
    """ The data that is output by the texture

    :type: str
    """

class Operator(bpy_struct):
    """Storage of an operator being executed, or registered after execution"""

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[str]
    """ Options for this operator type

    :type: set[str]
    """

    bl_translation_context: str
    """ 

    :type: str
    """

    has_reports: bool
    """ Operator has a set of reports (warnings and errors) from last execution

    :type: bool
    """

    layout: UILayout
    """ 

    :type: UILayout
    """

    macros: bpy_prop_collection[Macro]
    """ 

    :type: bpy_prop_collection[Macro]
    """

    name: str
    """ 

    :type: str
    """

    options: OperatorOptions
    """ Runtime options

    :type: OperatorOptions
    """

    properties: OperatorProperties
    """ 

    :type: OperatorProperties
    """

    bl_property: typing.Any
    """ The name of a property to use as this operators primary property.
Currently this is only used to select the default property when
expanding an operator into a menu.
:type: string"""

    is_registered: bool
    """ 

    :type: bool
    """

    def report(self, type: set[str] | None, message: str | typing.Any):
        """report

        :param type: Type
        :type type: set[str] | None
        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    def execute(self, context: Context) -> set[str]:
        """Execute the operator

                :param context:
                :type context: Context
                :return: result

        RUNNING_MODAL Running Modal, Keep the operator running with blender.

        CANCELLED Cancelled, When no action has been taken, operator exits.

        FINISHED Finished, When the operator is complete, operator exits.

        PASS_THROUGH Pass Through, Do nothing and pass the event on.

        INTERFACE Interface, Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def check(self, context: Context) -> bool:
        """Check the operator settings, return True to signal a change to redraw

        :param context:
        :type context: Context
        :return: result
        :rtype: bool
        """
        ...

    def invoke(self, context: Context, event: Event) -> set[str]:
        """Invoke the operator

                :param context:
                :type context: Context
                :param event:
                :type event: Event
                :return: result

        RUNNING_MODAL Running Modal, Keep the operator running with blender.

        CANCELLED Cancelled, When no action has been taken, operator exits.

        FINISHED Finished, When the operator is complete, operator exits.

        PASS_THROUGH Pass Through, Do nothing and pass the event on.

        INTERFACE Interface, Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def modal(self, context: Context, event: Event) -> set[str]:
        """Modal operator function

                :param context:
                :type context: Context
                :param event:
                :type event: Event
                :return: result

        RUNNING_MODAL Running Modal, Keep the operator running with blender.

        CANCELLED Cancelled, When no action has been taken, operator exits.

        FINISHED Finished, When the operator is complete, operator exits.

        PASS_THROUGH Pass Through, Do nothing and pass the event on.

        INTERFACE Interface, Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    def draw(self, context: Context):
        """Draw function for the operator

        :param context:
        :type context: Context
        """
        ...

    def cancel(self, context: Context):
        """Called when the operator is canceled

        :param context:
        :type context: Context
        """
        ...

    def as_keywords(self, ignore=()):
        """Return a copy of the properties as a dictionary

        :param ignore:
        """
        ...

class OperatorMacro(bpy_struct):
    """Storage of a sub operator in a macro after it has been added"""

    properties: OperatorProperties
    """ 

    :type: OperatorProperties
    """

class OperatorOptions(bpy_struct):
    """Runtime options"""

    is_grab_cursor: bool
    """ True when the cursor is grabbed

    :type: bool
    """

    is_invoke: bool
    """ True when invoked (even if only the execute callbacks available)

    :type: bool
    """

    use_cursor_region: bool
    """ Enable to use the region under the cursor for modal execution

    :type: bool
    """

class OperatorProperties(bpy_struct):
    """Input properties of an Operator"""

    ...

class PackedFile(bpy_struct):
    """External file packed into the .blend file"""

    data: str
    """ Raw data (bytes, exact content of the embedded file)

    :type: str
    """

    size: int
    """ Size of packed file in bytes

    :type: int
    """

class Paint(bpy_struct):
    brush: Brush
    """ Active Brush

    :type: Brush
    """

    cavity_curve: CurveMapping
    """ Editable cavity curve

    :type: CurveMapping
    """

    input_samples: int
    """ Average multiple input samples together to smooth the brush stroke

    :type: int
    """

    palette: Palette
    """ Active Palette

    :type: Palette
    """

    show_brush: bool
    """ 

    :type: bool
    """

    show_brush_on_surface: bool
    """ 

    :type: bool
    """

    show_low_resolution: bool
    """ For multires, show low resolution while navigating the view

    :type: bool
    """

    tile_offset: bpy_prop_array[float]
    """ Stride at which tiled strokes are copied

    :type: bpy_prop_array[float]
    """

    tile_x: bool
    """ Tile along X axis

    :type: bool
    """

    tile_y: bool
    """ Tile along Y axis

    :type: bool
    """

    tile_z: bool
    """ Tile along Z axis

    :type: bool
    """

    use_cavity: bool
    """ Mask painting according to mesh geometry cavity

    :type: bool
    """

    use_symmetry_feather: bool
    """ Reduce the strength of the brush where it overlaps symmetrical daubs

    :type: bool
    """

    use_symmetry_x: bool
    """ Mirror brush across the X axis

    :type: bool
    """

    use_symmetry_y: bool
    """ Mirror brush across the Y axis

    :type: bool
    """

    use_symmetry_z: bool
    """ Mirror brush across the Z axis

    :type: bool
    """

class PaletteColor(bpy_struct):
    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    strength: float
    """ 

    :type: float
    """

    weight: float
    """ 

    :type: float
    """

class PaletteColors(bpy_prop_collection[PaletteColor], bpy_struct):
    """Collection of palette colors"""

    active: PaletteColor | None
    """ 

    :type: PaletteColor | None
    """

    def new(self) -> PaletteColor:
        """Add a new color to the palette

        :return: The newly created color
        :rtype: PaletteColor
        """
        ...

    def remove(self, color: PaletteColor):
        """Remove a color from the palette

        :param color: The color to remove
        :type color: PaletteColor
        """
        ...

    def clear(self):
        """Remove all colors from the palette"""
        ...

class Panel(bpy_struct):
    """Panel containing UI elements"""

    bl_category: str
    """ 

    :type: str
    """

    bl_context: str
    """ The context in which the panel belongs to. (TODO: explain the possible combinations bl_context/bl_region_type/bl_space_type)

    :type: str
    """

    bl_idname: str
    """ If this is set, the panel gets a custom ID, otherwise it takes the name of the class used to define the panel. For example, if the class name is "OBJECT_PT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_PT_hello"

    :type: str
    """

    bl_label: str
    """ The panel label, shows up in the panel header at the right of the triangle used to collapse the panel

    :type: str
    """

    bl_options: set[str]
    """ Options for this panel type

    :type: set[str]
    """

    bl_region_type: str
    """ The region where the panel is going to be used in

    :type: str
    """

    bl_space_type: str
    """ The space where the panel is going to be used in

    :type: str
    """

    bl_translation_context: str | typing.Any
    """ 

    :type: str | typing.Any
    """

    layout: UILayout
    """ Defines the structure of the panel in the UI

    :type: UILayout
    """

    text: str
    """ XXX todo

    :type: str
    """

    use_pin: bool
    """ 

    :type: bool
    """

    is_registered: bool
    """ 

    :type: bool
    """

    @classmethod
    def poll(cls, context: Context) -> bool:
        """If this method returns a non-null output, then the panel can be drawn

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    def draw(self, context: Context):
        """Draw UI elements into the panel UI layout

        :param context:
        :type context: Context
        """
        ...

    def draw_header(self, context: Context):
        """Draw UI elements into the panel's header UI layout

        :param context:
        :type context: Context
        """
        ...

class Particle(bpy_struct):
    """Particle in a particle system"""

    alive_state: str
    """ 

    :type: str
    """

    angular_velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    birth_time: float
    """ 

    :type: float
    """

    die_time: float
    """ 

    :type: float
    """

    hair_keys: bpy_prop_collection[ParticleHairKey]
    """ 

    :type: bpy_prop_collection[ParticleHairKey]
    """

    is_exist: bool
    """ 

    :type: bool
    """

    is_visible: bool
    """ 

    :type: bool
    """

    lifetime: float
    """ 

    :type: float
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    particle_keys: bpy_prop_collection[ParticleKey]
    """ 

    :type: bpy_prop_collection[ParticleKey]
    """

    prev_angular_velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    prev_location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    prev_rotation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    prev_velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    rotation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    size: float
    """ 

    :type: float
    """

    velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    def uv_on_emitter(self, modifier: ParticleSystemModifier) -> bpy_prop_array[float]:
        """Obtain uv for particle on derived mesh

        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :return: uv
        :rtype: bpy_prop_array[float]
        """
        ...

class ParticleBrush(bpy_struct):
    """Particle editing brush"""

    count: int
    """ Particle count

    :type: int
    """

    curve: CurveMapping
    """ 

    :type: CurveMapping
    """

    length_mode: str
    """ 

    :type: str
    """

    puff_mode: str
    """ 

    :type: str
    """

    size: int
    """ Radius of the brush in pixels

    :type: int
    """

    steps: int
    """ Brush steps

    :type: int
    """

    strength: float
    """ Brush strength

    :type: float
    """

    use_puff_volume: bool
    """ Apply puff to unselected end-points (helps maintain hair volume when puffing root)

    :type: bool
    """

class ParticleDupliWeight(bpy_struct):
    """Weight of a particle dupliobject in a group"""

    count: int
    """ The number of times this object is repeated with respect to other objects

    :type: int
    """

    name: str
    """ Particle dupliobject name

    :type: str
    """

class ParticleEdit(bpy_struct):
    """Properties of particle editing mode"""

    brush: ParticleBrush
    """ 

    :type: ParticleBrush
    """

    default_key_count: int
    """ How many keys to make new particles with

    :type: int
    """

    draw_step: int
    """ How many steps to draw the path with

    :type: int
    """

    emitter_distance: float
    """ Distance to keep particles away from the emitter

    :type: float
    """

    fade_frames: int
    """ How many frames to fade

    :type: int
    """

    is_editable: bool
    """ A valid edit mode exists

    :type: bool
    """

    is_hair: bool
    """ Editing hair

    :type: bool
    """

    object: Object
    """ The edited object

    :type: Object
    """

    select_mode: str
    """ Particle select and display mode

    :type: str
    """

    shape_object: Object
    """ Outer shape to use for tools

    :type: Object
    """

    show_particles: bool
    """ Draw actual particles

    :type: bool
    """

    tool: str
    """ 

    :type: str
    """

    type: str
    """ 

    :type: str
    """

    use_auto_velocity: bool
    """ Calculate point velocities automatically

    :type: bool
    """

    use_default_interpolate: bool
    """ Interpolate new particles from the existing ones

    :type: bool
    """

    use_emitter_deflect: bool
    """ Keep paths from intersecting the emitter

    :type: bool
    """

    use_fade_time: bool
    """ Fade paths and keys further away from current frame

    :type: bool
    """

    use_preserve_length: bool
    """ Keep path lengths constant

    :type: bool
    """

    use_preserve_root: bool
    """ Keep root keys unmodified

    :type: bool
    """

class ParticleHairKey(bpy_struct):
    """Particle key for hair particle system"""

    co: bpy_prop_array[float]
    """ Location of the hair key in object space

    :type: bpy_prop_array[float]
    """

    co_local: bpy_prop_array[float]
    """ Location of the hair key in its local coordinate system, relative to the emitting face

    :type: bpy_prop_array[float]
    """

    time: float
    """ Relative time of key over hair length

    :type: float
    """

    weight: float
    """ Weight for cloth simulation

    :type: float
    """

    def co_object(
        self, object: Object, modifier: ParticleSystemModifier, particle: Particle
    ) -> bpy_prop_array[float]:
        """Obtain hairkey location with particle and modifier data

        :param object: Object
        :type object: Object
        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :param particle: hair particle
        :type particle: Particle
        :return: Co, Exported hairkey location
        :rtype: bpy_prop_array[float]
        """
        ...

class ParticleKey(bpy_struct):
    """Key location for a particle over time"""

    angular_velocity: bpy_prop_array[float]
    """ Key angular velocity

    :type: bpy_prop_array[float]
    """

    location: bpy_prop_array[float]
    """ Key location

    :type: bpy_prop_array[float]
    """

    rotation: bpy_prop_array[float]
    """ Key rotation quaternion

    :type: bpy_prop_array[float]
    """

    time: float
    """ Time of key over the simulation

    :type: float
    """

    velocity: bpy_prop_array[float]
    """ Key velocity

    :type: bpy_prop_array[float]
    """

class ParticleSettingsTextureSlots(
    bpy_prop_collection[ParticleSettingsTextureSlot], bpy_struct
):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> ParticleSettingsTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """
        ...

    @classmethod
    def create(cls, index: int | None) -> ParticleSettingsTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """
        ...

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """
        ...

class ParticleSystem(bpy_struct):
    """Particle system in an object"""

    active_particle_target: ParticleTarget
    """ 

    :type: ParticleTarget
    """

    active_particle_target_index: int | None
    """ 

    :type: int | None
    """

    billboard_normal_uv: str
    """ UV map to control billboard normals

    :type: str
    """

    billboard_split_uv: str
    """ UV map to control billboard splitting

    :type: str
    """

    billboard_time_index_uv: str
    """ UV map to control billboard time index (X-Y)

    :type: str
    """

    child_particles: bpy_prop_collection[ChildParticle]
    """ Child particles generated by the particle system

    :type: bpy_prop_collection[ChildParticle]
    """

    child_seed: int
    """ Offset in the random number table for child particles, to get a different randomized result

    :type: int
    """

    cloth: ClothModifier
    """ Cloth dynamics for hair

    :type: ClothModifier
    """

    dt_frac: float
    """ The current simulation time step size, as a fraction of a frame

    :type: float
    """

    has_multiple_caches: bool
    """ Particle system has multiple point caches

    :type: bool
    """

    invert_vertex_group_clump: bool
    """ Negate the effect of the clump vertex group

    :type: bool
    """

    invert_vertex_group_density: bool
    """ Negate the effect of the density vertex group

    :type: bool
    """

    invert_vertex_group_field: bool
    """ Negate the effect of the field vertex group

    :type: bool
    """

    invert_vertex_group_kink: bool
    """ Negate the effect of the kink vertex group

    :type: bool
    """

    invert_vertex_group_length: bool
    """ Negate the effect of the length vertex group

    :type: bool
    """

    invert_vertex_group_rotation: bool
    """ Negate the effect of the rotation vertex group

    :type: bool
    """

    invert_vertex_group_roughness_1: bool
    """ Negate the effect of the roughness 1 vertex group

    :type: bool
    """

    invert_vertex_group_roughness_2: bool
    """ Negate the effect of the roughness 2 vertex group

    :type: bool
    """

    invert_vertex_group_roughness_end: bool
    """ Negate the effect of the roughness end vertex group

    :type: bool
    """

    invert_vertex_group_size: bool
    """ Negate the effect of the size vertex group

    :type: bool
    """

    invert_vertex_group_tangent: bool
    """ Negate the effect of the tangent vertex group

    :type: bool
    """

    invert_vertex_group_velocity: bool
    """ Negate the effect of the velocity vertex group

    :type: bool
    """

    is_editable: bool
    """ Particle system can be edited in particle mode

    :type: bool
    """

    is_edited: bool
    """ Particle system has been edited in particle mode

    :type: bool
    """

    is_global_hair: bool
    """ Hair keys are in global coordinate space

    :type: bool
    """

    name: str
    """ Particle system name

    :type: str
    """

    parent: Object
    """ Use this object's coordinate system instead of global coordinate system

    :type: Object
    """

    particles: bpy_prop_collection[Particle]
    """ Particles generated by the particle system

    :type: bpy_prop_collection[Particle]
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    reactor_target_object: Object
    """ For reactor systems, the object that has the target particle system (empty if same object)

    :type: Object
    """

    reactor_target_particle_system: int
    """ For reactor systems, index of particle system on the target object

    :type: int
    """

    seed: int
    """ Offset in the random number table, to get a different randomized result

    :type: int
    """

    settings: ParticleSettings
    """ Particle system settings

    :type: ParticleSettings
    """

    targets: bpy_prop_collection[ParticleTarget]
    """ Target particle systems

    :type: bpy_prop_collection[ParticleTarget]
    """

    use_hair_dynamics: bool
    """ Enable hair dynamics using cloth simulation

    :type: bool
    """

    use_keyed_timing: bool
    """ Use key times

    :type: bool
    """

    vertex_group_clump: str
    """ Vertex group to control clump

    :type: str
    """

    vertex_group_density: str
    """ Vertex group to control density

    :type: str
    """

    vertex_group_field: str
    """ Vertex group to control field

    :type: str
    """

    vertex_group_kink: str
    """ Vertex group to control kink

    :type: str
    """

    vertex_group_length: str
    """ Vertex group to control length

    :type: str
    """

    vertex_group_rotation: str
    """ Vertex group to control rotation

    :type: str
    """

    vertex_group_roughness_1: str
    """ Vertex group to control roughness 1

    :type: str
    """

    vertex_group_roughness_2: str
    """ Vertex group to control roughness 2

    :type: str
    """

    vertex_group_roughness_end: str
    """ Vertex group to control roughness end

    :type: str
    """

    vertex_group_size: str
    """ Vertex group to control size

    :type: str
    """

    vertex_group_tangent: str
    """ Vertex group to control tangent

    :type: str
    """

    vertex_group_velocity: str
    """ Vertex group to control velocity

    :type: str
    """

    def set_resolution(
        self,
        scene: Scene | None = None,
        object: Object | None = None,
        resolution: str | None = "PREVIEW",
    ):
        """Set the resolution to use for the number of particles

                :param scene: Scene
                :type scene: Scene | None
                :param object: Object
                :type object: Object | None
                :param resolution: Resolution settings to apply

        PREVIEW Preview, Apply modifier preview settings.

        RENDER Render, Apply modifier render settings.
                :type resolution: str | None
        """
        ...

    def co_hair(
        self,
        object: Object,
        particle_no: typing.Any | None = 0,
        step: typing.Any | None = 0,
    ) -> bpy_prop_array[float]:
        """Obtain cache hair data

        :param object: Object
        :type object: Object
        :param particle_no: Particle no
        :type particle_no: typing.Any | None
        :param step: step no
        :type step: typing.Any | None
        :return: Co, Exported hairkey location
        :rtype: bpy_prop_array[float]
        """
        ...

    def uv_on_emitter(
        self,
        modifier: ParticleSystemModifier,
        particle: Particle,
        particle_no: typing.Any | None = 0,
        uv_no: typing.Any | None = 0,
    ) -> bpy_prop_array[float]:
        """Obtain uv for all particles

        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :param particle: Particle
        :type particle: Particle
        :param particle_no: Particle no
        :type particle_no: typing.Any | None
        :param uv_no: UV no
        :type uv_no: typing.Any | None
        :return: uv
        :rtype: bpy_prop_array[float]
        """
        ...

    def mcol_on_emitter(
        self,
        modifier: ParticleSystemModifier,
        particle: Particle,
        particle_no: typing.Any | None = 0,
        vcol_no: typing.Any | None = 0,
    ) -> bpy_prop_array[float]:
        """Obtain mcol for all particles

        :param modifier: Particle modifier
        :type modifier: ParticleSystemModifier
        :param particle: Particle
        :type particle: Particle
        :param particle_no: Particle no
        :type particle_no: typing.Any | None
        :param vcol_no: vcol no
        :type vcol_no: typing.Any | None
        :return: mcol
        :rtype: bpy_prop_array[float]
        """
        ...

class ParticleSystems(bpy_prop_collection[ParticleSystem], bpy_struct):
    """Collection of particle systems"""

    active: ParticleSystem
    """ Active particle system being displayed

    :type: ParticleSystem
    """

    active_index: int | None
    """ Index of active particle system slot

    :type: int | None
    """

class ParticleTarget(bpy_struct):
    """Target particle system"""

    alliance: str
    """ 

    :type: str
    """

    duration: float
    """ 

    :type: float
    """

    is_valid: bool
    """ Keyed particles target is valid

    :type: bool
    """

    name: str
    """ Particle target name

    :type: str
    """

    object: Object
    """ The object that has the target particle system (empty if same object)

    :type: Object
    """

    system: int
    """ The index of particle system on the target object

    :type: int
    """

    time: float
    """ 

    :type: float
    """

class PathCompare(bpy_struct):
    """Match paths against this value"""

    path: str
    """ 

    :type: str
    """

    use_glob: bool
    """ Enable wildcard globbing

    :type: bool
    """

class PathCompareCollection(bpy_prop_collection[PathCompare], bpy_struct):
    """Collection of paths"""

    @classmethod
    def new(cls) -> PathCompare:
        """Add a new path

        :return:
        :rtype: PathCompare
        """
        ...

    @classmethod
    def remove(cls, pathcmp: PathCompare):
        """Remove path

        :param pathcmp:
        :type pathcmp: PathCompare
        """
        ...

class PointCache(bpy_struct):
    """Point cache for physics simulations"""

    compression: str
    """ Compression method to be used

    :type: str
    """

    filepath: str
    """ Cache file path

    :type: str
    """

    frame_end: int
    """ Frame on which the simulation stops

    :type: int
    """

    frame_start: int
    """ Frame on which the simulation starts

    :type: int
    """

    frame_step: int
    """ Number of frames between cached frames

    :type: int
    """

    index: int
    """ Index number of cache files

    :type: int
    """

    info: str
    """ Info on current cache status

    :type: str
    """

    is_baked: bool
    """ 

    :type: bool
    """

    is_baking: bool
    """ 

    :type: bool
    """

    is_frame_skip: bool
    """ 

    :type: bool
    """

    is_outdated: bool
    """ 

    :type: bool
    """

    name: str
    """ Cache name

    :type: str
    """

    point_caches: PointCaches
    """ Point cache list

    :type: PointCaches
    """

    use_disk_cache: bool
    """ Save cache files to disk (.blend file must be saved first)

    :type: bool
    """

    use_external: bool
    """ Read cache from an external location

    :type: bool
    """

    use_library_path: bool
    """ Use this file's path for the disk cache when library linked into another file (for local bakes per scene file, disable this option)

    :type: bool
    """

class PointCaches(bpy_prop_collection[PointCache], bpy_struct):
    """Collection of point caches"""

    active_index: int | None
    """ 

    :type: int | None
    """

class PointDensity(bpy_struct):
    """Point density settings"""

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    falloff: str
    """ Method of attenuating density by distance from the point

    :type: str
    """

    falloff_curve: CurveMapping
    """ 

    :type: CurveMapping
    """

    falloff_soft: float
    """ Softness of the 'soft' falloff option

    :type: float
    """

    falloff_speed_scale: float
    """ Multiplier to bring particle speed within an acceptable range

    :type: float
    """

    noise_basis: str
    """ Noise formula used for turbulence

    :type: str
    """

    object: Object
    """ Object to take point data from

    :type: Object
    """

    particle_cache_space: str
    """ Coordinate system to cache particles in

    :type: str
    """

    particle_color_source: str
    """ Data to derive color results from

    :type: str
    """

    particle_system: ParticleSystem
    """ Particle System to render as points

    :type: ParticleSystem
    """

    point_source: str
    """ Point data to use as renderable point density

    :type: str
    """

    radius: float
    """ Radius from the shaded sample to look for points within

    :type: float
    """

    speed_scale: float
    """ Multiplier to bring particle speed within an acceptable range

    :type: float
    """

    turbulence_depth: int
    """ Level of detail in the added turbulent noise

    :type: int
    """

    turbulence_influence: str
    """ Method for driving added turbulent noise

    :type: str
    """

    turbulence_scale: float
    """ Scale of the added turbulent noise

    :type: float
    """

    turbulence_strength: float
    """ Strength of the added turbulent noise

    :type: float
    """

    use_falloff_curve: bool
    """ Use a custom falloff curve

    :type: bool
    """

    use_turbulence: bool
    """ Add directed noise to the density at render-time

    :type: bool
    """

    vertex_attribute_name: str
    """ Vertex attribute to use for color

    :type: str
    """

    vertex_cache_space: str
    """ Coordinate system to cache vertices in

    :type: str
    """

    vertex_color_source: str
    """ Data to derive color results from

    :type: str
    """

class PolygonFloatProperties(
    bpy_prop_collection[MeshPolygonFloatPropertyLayer], bpy_struct
):
    """Collection of float properties"""

    def new(
        self, name: str | typing.Any = "Float Prop"
    ) -> MeshPolygonFloatPropertyLayer:
        """Add a float property layer to Mesh

        :param name: Float property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshPolygonFloatPropertyLayer
        """
        ...

class PolygonIntProperties(
    bpy_prop_collection[MeshPolygonIntPropertyLayer], bpy_struct
):
    """Collection of int properties"""

    def new(self, name: str | typing.Any = "Int Prop") -> MeshPolygonIntPropertyLayer:
        """Add a integer property layer to Mesh

        :param name: Int property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshPolygonIntPropertyLayer
        """
        ...

class PolygonStringProperties(
    bpy_prop_collection[MeshPolygonStringPropertyLayer], bpy_struct
):
    """Collection of string properties"""

    def new(
        self, name: str | typing.Any = "String Prop"
    ) -> MeshPolygonStringPropertyLayer:
        """Add a string property layer to Mesh

        :param name: String property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshPolygonStringPropertyLayer
        """
        ...

class Pose(bpy_struct):
    """A collection of pose channels, including settings for animating bones"""

    animation_visualization: AnimViz
    """ Animation data for this data-block

    :type: AnimViz
    """

    bone_groups: BoneGroups
    """ Groups of the bones

    :type: BoneGroups
    """

    bones: bpy_prop_collection[PoseBone]
    """ Individual pose bones for the armature

    :type: bpy_prop_collection[PoseBone]
    """

    ik_param: IKParam
    """ Parameters for IK solver

    :type: IKParam
    """

    ik_solver: str
    """ Selection of IK solver for IK chain

    :type: str
    """

class PoseBone(bpy_struct):
    """Channel defining pose data for a bone in a Pose"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveiny: float
    """ Y-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveouty: float
    """ Y-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_custom_handle_end: PoseBone
    """ Bone that serves as the end handle for the B-Bone curve

    :type: PoseBone
    """

    bbone_custom_handle_start: PoseBone
    """ Bone that serves as the start handle for the B-Bone curve

    :type: PoseBone
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: float
    """ Scale factor for start of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_scaleout: float
    """ Scale factor for end of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bone: Bone
    """ Bone associated with this PoseBone

    :type: Bone
    """

    bone_group: BoneGroup
    """ Bone Group this pose channel belongs to

    :type: BoneGroup
    """

    bone_group_index: int
    """ Bone Group this pose channel belongs to (0=no group)

    :type: int
    """

    child: PoseBone
    """ Child of this pose bone

    :type: PoseBone
    """

    constraints: PoseBoneConstraints
    """ Constraints that act on this PoseChannel

    :type: PoseBoneConstraints
    """

    custom_shape: Object
    """ Object that defines custom draw type for this bone

    :type: Object
    """

    custom_shape_scale: float
    """ Adjust the size of the custom shape

    :type: float
    """

    custom_shape_transform: PoseBone
    """ Bone that defines the display transform of this custom shape

    :type: PoseBone
    """

    head: bpy_prop_array[float]
    """ Location of head of the channel's bone

    :type: bpy_prop_array[float]
    """

    ik_linear_weight: float
    """ Weight of scale constraint for IK

    :type: float
    """

    ik_max_x: float
    """ Maximum angles for IK Limit

    :type: float
    """

    ik_max_y: float
    """ Maximum angles for IK Limit

    :type: float
    """

    ik_max_z: float
    """ Maximum angles for IK Limit

    :type: float
    """

    ik_min_x: float
    """ Minimum angles for IK Limit

    :type: float
    """

    ik_min_y: float
    """ Minimum angles for IK Limit

    :type: float
    """

    ik_min_z: float
    """ Minimum angles for IK Limit

    :type: float
    """

    ik_rotation_weight: float
    """ Weight of rotation constraint for IK

    :type: float
    """

    ik_stiffness_x: float
    """ IK stiffness around the X axis

    :type: float
    """

    ik_stiffness_y: float
    """ IK stiffness around the Y axis

    :type: float
    """

    ik_stiffness_z: float
    """ IK stiffness around the Z axis

    :type: float
    """

    ik_stretch: float
    """ Allow scaling of the bone for IK

    :type: float
    """

    is_in_ik_chain: bool
    """ Is part of an IK chain

    :type: bool
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    lock_ik_x: bool
    """ Disallow movement around the X axis

    :type: bool
    """

    lock_ik_y: bool
    """ Disallow movement around the Y axis

    :type: bool
    """

    lock_ik_z: bool
    """ Disallow movement around the Z axis

    :type: bool
    """

    lock_location: list[bool]
    """ Lock editing of location in the interface

    :type: list[bool]
    """

    lock_rotation: list[bool]
    """ Lock editing of rotation in the interface

    :type: list[bool]
    """

    lock_rotation_w: bool
    """ Lock editing of 'angle' component of four-component rotations in the interface

    :type: bool
    """

    lock_rotations_4d: bool
    """ Lock editing of four component rotations by components (instead of as Eulers)

    :type: bool
    """

    lock_scale: list[bool]
    """ Lock editing of scale in the interface

    :type: list[bool]
    """

    matrix: bpy_prop_array[float]
    """ Final 4x4 matrix after constraints and drivers are applied (object space)

    :type: bpy_prop_array[float]
    """

    matrix_basis: bpy_prop_array[float]
    """ Alternative access to location/scale/rotation relative to the parent and own rest bone

    :type: bpy_prop_array[float]
    """

    matrix_channel: bpy_prop_array[float]
    """ 4x4 matrix, before constraints

    :type: bpy_prop_array[float]
    """

    motion_path: MotionPath
    """ Motion Path for this element

    :type: MotionPath
    """

    name: str
    """ 

    :type: str
    """

    parent: PoseBone
    """ Parent of this pose bone

    :type: PoseBone
    """

    rotation_axis_angle: bpy_prop_array[float]
    """ Angle of Rotation for Axis-Angle rotation representation

    :type: bpy_prop_array[float]
    """

    rotation_euler: bpy_prop_array[float]
    """ Rotation in Eulers

    :type: bpy_prop_array[float]
    """

    rotation_mode: str
    """ 

    :type: str
    """

    rotation_quaternion: bpy_prop_array[float]
    """ Rotation in Quaternions

    :type: bpy_prop_array[float]
    """

    scale: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tail: bpy_prop_array[float]
    """ Location of tail of the channel's bone

    :type: bpy_prop_array[float]
    """

    use_bbone_custom_handles: bool
    """ Use custom reference bones as handles for B-Bones instead of next/previous bones, leave these blank to use only B-Bone offset properties to control the shape

    :type: bool
    """

    use_bbone_relative_end_handle: bool
    """ Treat custom end handle position as a relative value

    :type: bool
    """

    use_bbone_relative_start_handle: bool
    """ Treat custom start handle position as a relative value

    :type: bool
    """

    use_custom_shape_bone_size: bool
    """ Scale the custom object by the bone length

    :type: bool
    """

    use_ik_limit_x: bool
    """ Limit movement around the X axis

    :type: bool
    """

    use_ik_limit_y: bool
    """ Limit movement around the Y axis

    :type: bool
    """

    use_ik_limit_z: bool
    """ Limit movement around the Z axis

    :type: bool
    """

    use_ik_linear_control: bool
    """ Apply channel size as IK constraint if stretching is enabled

    :type: bool
    """

    use_ik_rotation_control: bool
    """ Apply channel rotation as IK constraint

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character
(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.
(readonly)"""

    children: typing.Any
    """ (readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.
(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.
(readonly)"""

    length: typing.Any
    """ The distance from head to tail,
when set the head is moved to fit the length."""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent
(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)
(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.
(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.
(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.
(readonly)"""

    def evaluate_envelope(self, point: collections.abc.Iterable[float] | None) -> float:
        """Calculate bone envelope at given point

        :param point: Point, Position in 3d space to evaluate
        :type point: collections.abc.Iterable[float] | None
        :return: Factor, Envelope factor
        :rtype: float
        """
        ...

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """
        ...

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """
        ...

class PoseBoneConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of pose bone constraints"""

    active: Constraint | None
    """ Active PoseChannel constraint

    :type: Constraint | None
    """

    def new(self, type: str | None) -> Constraint:
        """Add a constraint to this object

                :param type: Constraint type to add

        CAMERA_SOLVER Camera Solver.

        FOLLOW_TRACK Follow Track.

        OBJECT_SOLVER Object Solver.

        COPY_LOCATION Copy Location, Copy the location of a target (with an optional offset), so that they move together.

        COPY_ROTATION Copy Rotation, Copy the rotation of a target (with an optional offset), so that they rotate together.

        COPY_SCALE Copy Scale, Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.

        COPY_TRANSFORMS Copy Transforms, Copy all the transformations of a target, so that they move together.

        LIMIT_DISTANCE Limit Distance, Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).

        LIMIT_LOCATION Limit Location, Restrict movement along each axis within given ranges.

        LIMIT_ROTATION Limit Rotation, Restrict rotation along each axis within given ranges.

        LIMIT_SCALE Limit Scale, Restrict scaling along each axis with given ranges.

        MAINTAIN_VOLUME Maintain Volume, Compensate for scaling one axis by applying suitable scaling to the other two axes.

        TRANSFORM Transformation, Use one transform property from target to control another (or same) property on owner.

        TRANSFORM_CACHE Transform Cache, Look up the transformation matrix from an external file.

        CLAMP_TO Clamp To, Restrict movements to lie along a curve by remapping location along curve's longest axis.

        DAMPED_TRACK Damped Track, Point towards a target by performing the smallest rotation necessary.

        IK Inverse Kinematics, Control a chain of bones by specifying the endpoint target (Bones only).

        LOCKED_TRACK Locked Track, Rotate around the specified ('locked') axis to point towards a target.

        SPLINE_IK Spline IK, Align chain of bones along a curve (Bones only).

        STRETCH_TO Stretch To, Stretch along Y-Axis to point towards a target.

        TRACK_TO Track To, Legacy tracking constraint prone to twisting artifacts.

        ACTION Action, Use transform property of target to look up pose for owner from an Action.

        CHILD_OF Child Of, Make target the 'detachable' parent of owner.

        FLOOR Floor, Use position (and optionally rotation) of target to define a 'wall' or 'floor' that the owner can not cross.

        FOLLOW_PATH Follow Path, Use to animate an object/bone following a path.

        PIVOT Pivot, Change pivot point for transforms (buggy).

        RIGID_BODY_JOINT Rigid Body Joint, Use to define a Rigid Body Constraint (for Game Engine use only).

        SHRINKWRAP Shrinkwrap, Restrict movements to surface of target mesh.
                :type type: str | None
                :return: New constraint
                :rtype: Constraint
        """
        ...

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """
        ...

class Property(bpy_struct):
    """RNA property definition"""

    description: str
    """ Description of the property for tooltips

    :type: str
    """

    icon: str
    """ Icon of the item

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    is_animatable: bool
    """ Property is animatable through RNA

    :type: bool
    """

    is_argument_optional: bool
    """ True when the property is optional in a Python function implementing an RNA function

    :type: bool
    """

    is_enum_flag: bool
    """ True when multiple enums

    :type: bool
    """

    is_hidden: bool
    """ True when the property is hidden

    :type: bool
    """

    is_library_editable: bool
    """ Property is editable from linked instances (changes not saved)

    :type: bool
    """

    is_never_none: bool
    """ True when this value can't be set to None

    :type: bool
    """

    is_output: bool
    """ True when this property is an output value from an RNA function

    :type: bool
    """

    is_readonly: bool
    """ Property is editable through RNA

    :type: bool
    """

    is_registered: bool
    """ Property is registered as part of type registration

    :type: bool
    """

    is_registered_optional: bool
    """ Property is optionally registered as part of type registration

    :type: bool
    """

    is_required: bool
    """ False when this property is an optional argument in an RNA function

    :type: bool
    """

    is_runtime: bool
    """ Property has been dynamically created at runtime

    :type: bool
    """

    is_skip_save: bool
    """ True when the property is not saved in presets

    :type: bool
    """

    name: str
    """ Human readable name

    :type: str
    """

    srna: Struct
    """ Struct definition used for properties assigned to this item

    :type: Struct
    """

    subtype: str
    """ Semantic interpretation of the property

    :type: str
    """

    translation_context: str
    """ Translation context of the property's name

    :type: str
    """

    type: str
    """ Data type of the property

    :type: str
    """

    unit: str
    """ Type of units for this property

    :type: str
    """

class PropertyGroup(bpy_struct):
    """Group of ID properties"""

    name: str
    """ Unique name used in the code and scripting

    :type: str
    """

class PropertyGroupItem(bpy_struct):
    """Property that stores arbitrary, user defined properties"""

    collection: bpy_prop_collection[PropertyGroup]
    """ 

    :type: bpy_prop_collection[PropertyGroup]
    """

    double: float
    """ 

    :type: float
    """

    double_array: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    float: float
    """ 

    :type: float
    """

    float_array: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    group: PropertyGroup
    """ 

    :type: PropertyGroup
    """

    idp_array: bpy_prop_collection[PropertyGroup]
    """ 

    :type: bpy_prop_collection[PropertyGroup]
    """

    int: int
    """ 

    :type: int
    """

    int_array: bpy_prop_array[int]
    """ 

    :type: bpy_prop_array[int]
    """

    string: str
    """ 

    :type: str
    """

class Region(bpy_struct):
    """Region in a subdivided screen area"""

    height: int
    """ Region height

    :type: int
    """

    id: int
    """ Unique ID for this region

    :type: int
    """

    type: str
    """ Type of this region

    :type: str
    """

    view2d: View2D
    """ 2D view of the region

    :type: View2D
    """

    width: int
    """ Region width

    :type: int
    """

    x: int
    """ The window relative vertical location of the region

    :type: int
    """

    y: int
    """ The window relative horizontal location of the region

    :type: int
    """

    def tag_redraw(self):
        """tag_redraw"""
        ...

class RegionView3D(bpy_struct):
    """3D View region data"""

    is_perspective: bool
    """ 

    :type: bool
    """

    lock_rotation: bool
    """ Lock view rotation in side views

    :type: bool
    """

    perspective_matrix: bpy_prop_array[float]
    """ Current perspective matrix (window_matrix * view_matrix)

    :type: bpy_prop_array[float]
    """

    show_sync_view: bool
    """ Sync view position between side views

    :type: bool
    """

    use_box_clip: bool
    """ Clip objects based on what's visible in other side views

    :type: bool
    """

    view_camera_offset: bpy_prop_array[float]
    """ View shift in camera view

    :type: bpy_prop_array[float]
    """

    view_camera_zoom: float
    """ Zoom factor in camera view

    :type: float
    """

    view_distance: float
    """ Distance to the view location

    :type: float
    """

    view_location: bpy_prop_array[float]
    """ View pivot location

    :type: bpy_prop_array[float]
    """

    view_matrix: bpy_prop_array[float]
    """ Current view matrix

    :type: bpy_prop_array[float]
    """

    view_perspective: str
    """ View Perspective

    :type: str
    """

    view_rotation: bpy_prop_array[float]
    """ Rotation in quaternions (keep normalized)

    :type: bpy_prop_array[float]
    """

    window_matrix: bpy_prop_array[float]
    """ Current window matrix

    :type: bpy_prop_array[float]
    """

    def update(self):
        """Recalculate the view matrices"""
        ...

class RenderEngine(bpy_struct):
    """Render engine"""

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_use_exclude_layers: bool
    """ 

    :type: bool
    """

    bl_use_postprocess: bool
    """ 

    :type: bool
    """

    bl_use_preview: bool
    """ 

    :type: bool
    """

    bl_use_save_buffers: bool
    """ 

    :type: bool
    """

    bl_use_shading_nodes: bool
    """ 

    :type: bool
    """

    bl_use_shading_nodes_custom: bool
    """ 

    :type: bool
    """

    bl_use_spherical_stereo: bool
    """ 

    :type: bool
    """

    bl_use_texture_preview: bool
    """ 

    :type: bool
    """

    camera_override: Object
    """ 

    :type: Object
    """

    is_animation: bool
    """ 

    :type: bool
    """

    is_preview: bool
    """ 

    :type: bool
    """

    layer_override: list[bool]
    """ 

    :type: list[bool]
    """

    render: RenderSettings
    """ 

    :type: RenderSettings
    """

    resolution_x: int
    """ 

    :type: int
    """

    resolution_y: int
    """ 

    :type: int
    """

    tile_x: int
    """ 

    :type: int
    """

    tile_y: int
    """ 

    :type: int
    """

    use_highlight_tiles: bool
    """ 

    :type: bool
    """

    def update(self, data: BlendData | None = None, scene: Scene | None = None):
        """Export scene data for render

        :param data:
        :type data: BlendData | None
        :param scene:
        :type scene: Scene | None
        """
        ...

    def render(self, scene: Scene | None = None):
        """Render scene into an image

        :param scene:
        :type scene: Scene | None
        """
        ...

    def bake(
        self,
        scene: Scene | None,
        object: Object | None,
        pass_type: str | None,
        pass_filter: int | None,
        object_id: int | None,
        pixel_array: BakePixel | None,
        num_pixels: int | None,
        depth: int | None,
        result: typing.Any | None,
    ):
        """Bake passes

        :param scene:
        :type scene: Scene | None
        :param object:
        :type object: Object | None
        :param pass_type: Pass, Pass to bake
        :type pass_type: str | None
        :param pass_filter: Pass Filter, Filter to combined, diffuse, glossy, transmission and subsurface passes
        :type pass_filter: int | None
        :param object_id: Object Id, Id of the current object being baked in relation to the others
        :type object_id: int | None
        :param pixel_array:
        :type pixel_array: BakePixel | None
        :param num_pixels: Number of Pixels, Size of the baking batch
        :type num_pixels: int | None
        :param depth: Pixels depth, Number of channels
        :type depth: int | None
        :param result:
        :type result: typing.Any | None
        """
        ...

    def view_update(self, context: Context | None = None):
        """Update on data changes for viewport render

        :param context:
        :type context: Context | None
        """
        ...

    def view_draw(self, context: Context | None = None):
        """Draw viewport render

        :param context:
        :type context: Context | None
        """
        ...

    def update_script_node(self, node: Node | None = None):
        """Compile shader script node

        :param node:
        :type node: Node | None
        """
        ...

    def tag_redraw(self):
        """Request redraw for viewport rendering"""
        ...

    def tag_update(self):
        """Request update call for viewport rendering"""
        ...

    def begin_result(
        self,
        x: int | None,
        y: int | None,
        w: int | None,
        h: int | None,
        layer: str | typing.Any = "",
        view: str | typing.Any = "",
    ) -> RenderResult:
        """Create render result to write linear floating point render layers and passes

        :param x: X
        :type x: int | None
        :param y: Y
        :type y: int | None
        :param w: Width
        :type w: int | None
        :param h: Height
        :type h: int | None
        :param layer: Layer, Single layer to get render result for
        :type layer: str | typing.Any
        :param view: View, Single view to get render result for
        :type view: str | typing.Any
        :return: Result
        :rtype: RenderResult
        """
        ...

    def update_result(self, result: RenderResult | None):
        """Signal that pixels have been updated and can be redrawn in the user interface

        :param result: Result
        :type result: RenderResult | None
        """
        ...

    def end_result(
        self,
        result: RenderResult | None,
        cancel: bool | typing.Any | None = False,
        do_merge_results: bool | typing.Any | None = False,
    ):
        """All pixels in the render result have been set and are final

        :param result: Result
        :type result: RenderResult | None
        :param cancel: Cancel, Don't mark tile as done, don't merge results unless forced
        :type cancel: bool | typing.Any | None
        :param do_merge_results: Merge Results, Merge results even if cancel=true
        :type do_merge_results: bool | typing.Any | None
        """
        ...

    def test_break(self) -> bool:
        """Test if the render operation should been canceled, this is a fast call that should be used regularly for responsiveness

        :return: Break
        :rtype: bool
        """
        ...

    def active_view_get(self) -> str | typing.Any:
        """active_view_get

        :return: View, Single view active
        :rtype: str | typing.Any
        """
        ...

    def active_view_set(self, view: str | typing.Any = ""):
        """active_view_set

        :param view: View, Single view to set as active
        :type view: str | typing.Any
        """
        ...

    def camera_shift_x(
        self,
        camera: Object | None,
        use_spherical_stereo: bool | typing.Any | None = False,
    ) -> float:
        """camera_shift_x

        :param camera:
        :type camera: Object | None
        :param use_spherical_stereo: Spherical Stereo
        :type use_spherical_stereo: bool | typing.Any | None
        :return: Shift X
        :rtype: float
        """
        ...

    def camera_model_matrix(
        self,
        camera: Object | None,
        use_spherical_stereo: bool | typing.Any | None = False,
        r_model_matrix: collections.abc.Iterable[float] | None = None,
    ):
        """camera_model_matrix

        :param camera:
        :type camera: Object | None
        :param use_spherical_stereo: Spherical Stereo
        :type use_spherical_stereo: bool | typing.Any | None
        :param r_model_matrix: Model Matrix, Normalized camera model matrix
        :type r_model_matrix: collections.abc.Iterable[float] | None
        """
        ...

    def use_spherical_stereo(self, camera: Object | None) -> bool:
        """use_spherical_stereo

        :param camera:
        :type camera: Object | None
        :return: Spherical Stereo
        :rtype: bool
        """
        ...

    def update_stats(self, stats: str | typing.Any, info: str | typing.Any):
        """Update and signal to redraw render status text

        :param stats: Stats
        :type stats: str | typing.Any
        :param info: Info
        :type info: str | typing.Any
        """
        ...

    def frame_set(self, frame: int | None, subframe: float | None):
        """Evaluate scene at a different frame (for motion blur)

        :param frame: Frame
        :type frame: int | None
        :param subframe: Subframe
        :type subframe: float | None
        """
        ...

    def update_progress(self, progress: float | None):
        """Update progress percentage of render

        :param progress: Percentage of render that's done
        :type progress: float | None
        """
        ...

    def update_memory_stats(
        self, memory_used: typing.Any | None = 0.0, memory_peak: typing.Any | None = 0.0
    ):
        """Update memory usage statistics

        :param memory_used: Current memory usage in megabytes
        :type memory_used: typing.Any | None
        :param memory_peak: Peak memory usage in megabytes
        :type memory_peak: typing.Any | None
        """
        ...

    def report(self, type: set[str] | None, message: str | typing.Any):
        """Report info, warning or error messages

        :param type: Type
        :type type: set[str] | None
        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    def error_set(self, message: str | typing.Any):
        """Set error message displaying after the render is finished

        :param message: Report Message
        :type message: str | typing.Any
        """
        ...

    def bind_display_space_shader(self, scene: Scene | None):
        """Bind GLSL fragment shader that converts linear colors to display space colors using scene color management settings

        :param scene:
        :type scene: Scene | None
        """
        ...

    def unbind_display_space_shader(self):
        """Unbind GLSL display space shader, must always be called after binding the shader"""
        ...

    def support_display_space_shader(self, scene: Scene | None) -> bool:
        """Test if GLSL display space shader is supported for the combination of graphics card and scene settings

        :param scene:
        :type scene: Scene | None
        :return: Supported
        :rtype: bool
        """
        ...

class RenderLayer(bpy_struct):
    exclude_ambient_occlusion: bool
    """ Exclude AO pass from combined

    :type: bool
    """

    exclude_emit: bool
    """ Exclude emission pass from combined

    :type: bool
    """

    exclude_environment: bool
    """ Exclude environment pass from combined

    :type: bool
    """

    exclude_indirect: bool
    """ Exclude indirect pass from combined

    :type: bool
    """

    exclude_reflection: bool
    """ Exclude raytraced reflection pass from combined

    :type: bool
    """

    exclude_refraction: bool
    """ Exclude raytraced refraction pass from combined

    :type: bool
    """

    exclude_shadow: bool
    """ Exclude shadow pass from combined

    :type: bool
    """

    exclude_specular: bool
    """ Exclude specular pass from combined

    :type: bool
    """

    invert_zmask: bool
    """ For Zmask, only render what is behind solid z values instead of in front

    :type: bool
    """

    layers: list[bool]
    """ Scene layers included in this render layer

    :type: list[bool]
    """

    layers_exclude: list[bool]
    """ Exclude scene layers from having any influence

    :type: list[bool]
    """

    layers_zmask: list[bool]
    """ Zmask scene layers for solid faces

    :type: list[bool]
    """

    light_override: Group
    """ Group to override all other lights in this render layer

    :type: Group
    """

    material_override: Material
    """ Material to override all other materials in this render layer

    :type: Material
    """

    name: str
    """ Render layer name

    :type: str
    """

    passes: RenderPasses
    """ 

    :type: RenderPasses
    """

    use: bool
    """ Disable or enable the render layer

    :type: bool
    """

    use_all_z: bool
    """ Fill in Z values for solid faces in invisible layers, for masking

    :type: bool
    """

    use_ao: bool
    """ Render AO in this Layer

    :type: bool
    """

    use_edge_enhance: bool
    """ Render Edge-enhance in this Layer (only works for Solid faces)

    :type: bool
    """

    use_freestyle: bool
    """ Render stylized strokes in this Layer

    :type: bool
    """

    use_halo: bool
    """ Render Halos in this Layer (on top of Solid)

    :type: bool
    """

    use_pass_ambient_occlusion: bool
    """ Deliver AO pass

    :type: bool
    """

    use_pass_color: bool
    """ Deliver shade-less color pass

    :type: bool
    """

    use_pass_combined: bool
    """ Deliver full combined RGBA buffer

    :type: bool
    """

    use_pass_diffuse: bool
    """ Deliver diffuse pass

    :type: bool
    """

    use_pass_diffuse_color: bool
    """ Deliver diffuse color pass

    :type: bool
    """

    use_pass_diffuse_direct: bool
    """ Deliver diffuse direct pass

    :type: bool
    """

    use_pass_diffuse_indirect: bool
    """ Deliver diffuse indirect pass

    :type: bool
    """

    use_pass_emit: bool
    """ Deliver emission pass

    :type: bool
    """

    use_pass_environment: bool
    """ Deliver environment lighting pass

    :type: bool
    """

    use_pass_glossy_color: bool
    """ Deliver glossy color pass

    :type: bool
    """

    use_pass_glossy_direct: bool
    """ Deliver glossy direct pass

    :type: bool
    """

    use_pass_glossy_indirect: bool
    """ Deliver glossy indirect pass

    :type: bool
    """

    use_pass_indirect: bool
    """ Deliver indirect lighting pass

    :type: bool
    """

    use_pass_material_index: bool
    """ Deliver material index pass

    :type: bool
    """

    use_pass_mist: bool
    """ Deliver mist factor pass (0.0-1.0)

    :type: bool
    """

    use_pass_normal: bool
    """ Deliver normal pass

    :type: bool
    """

    use_pass_object_index: bool
    """ Deliver object index pass

    :type: bool
    """

    use_pass_reflection: bool
    """ Deliver raytraced reflection pass

    :type: bool
    """

    use_pass_refraction: bool
    """ Deliver raytraced refraction pass

    :type: bool
    """

    use_pass_shadow: bool
    """ Deliver shadow pass

    :type: bool
    """

    use_pass_specular: bool
    """ Deliver specular pass

    :type: bool
    """

    use_pass_subsurface_color: bool
    """ Deliver subsurface color pass

    :type: bool
    """

    use_pass_subsurface_direct: bool
    """ Deliver subsurface direct pass

    :type: bool
    """

    use_pass_subsurface_indirect: bool
    """ Deliver subsurface indirect pass

    :type: bool
    """

    use_pass_transmission_color: bool
    """ Deliver transmission color pass

    :type: bool
    """

    use_pass_transmission_direct: bool
    """ Deliver transmission direct pass

    :type: bool
    """

    use_pass_transmission_indirect: bool
    """ Deliver transmission indirect pass

    :type: bool
    """

    use_pass_uv: bool
    """ Deliver texture UV pass

    :type: bool
    """

    use_pass_vector: bool
    """ Deliver speed vector pass

    :type: bool
    """

    use_pass_z: bool
    """ Deliver Z values pass

    :type: bool
    """

    use_sky: bool
    """ Render Sky in this Layer

    :type: bool
    """

    use_solid: bool
    """ Render Solid faces in this Layer

    :type: bool
    """

    use_strand: bool
    """ Render Strands in this Layer

    :type: bool
    """

    use_zmask: bool
    """ Only render what's in front of the solid z values

    :type: bool
    """

    use_ztransp: bool
    """ Render Z-Transparent faces in this Layer (on top of Solid and Halos)

    :type: bool
    """

    def load_from_file(
        self,
        filename: str | typing.Any,
        x: typing.Any | None = 0,
        y: typing.Any | None = 0,
    ):
        """Copies the pixels of this renderlayer from an image file

        :param filename: Filename, Filename to load into this render tile, must be no smaller than the renderlayer
        :type filename: str | typing.Any
        :param x: Offset X, Offset the position to copy from if the image is larger than the render layer
        :type x: typing.Any | None
        :param y: Offset Y, Offset the position to copy from if the image is larger than the render layer
        :type y: typing.Any | None
        """
        ...

class RenderLayers(bpy_prop_collection[SceneRenderLayer], bpy_struct):
    """Collection of render layers"""

    active: SceneRenderLayer
    """ Active Render Layer

    :type: SceneRenderLayer
    """

    active_index: int | None
    """ Active index in render layer array

    :type: int | None
    """

    def new(self, name: str | typing.Any) -> SceneRenderLayer:
        """Add a render layer to scene

        :param name: New name for the render layer (not unique)
        :type name: str | typing.Any
        :return: Newly created render layer
        :rtype: SceneRenderLayer
        """
        ...

    def remove(self, layer: SceneRenderLayer):
        """Remove a render layer

        :param layer: Render layer to remove
        :type layer: SceneRenderLayer
        """
        ...

class RenderPass(bpy_struct):
    channel_id: str
    """ 

    :type: str
    """

    channels: int
    """ 

    :type: int
    """

    debug_type: str
    """ 

    :type: str
    """

    name: str
    """ 

    :type: str
    """

    rect: float
    """ 

    :type: float
    """

    type: str
    """ 

    :type: str
    """

    view_id: int
    """ 

    :type: int
    """

class RenderPasses(bpy_prop_collection[RenderPass], bpy_struct):
    """Collection of render passes"""

    def find_by_type(self, pass_type: str | None, view: str | typing.Any) -> RenderPass:
        """Get the render pass for a given type and view

        :param pass_type: Pass
        :type pass_type: str | None
        :param view: View, Render view to get pass from
        :type view: str | typing.Any
        :return: The matching render pass
        :rtype: RenderPass
        """
        ...

class RenderResult(bpy_struct):
    """Result of rendering, including all layers and passes"""

    layers: bpy_prop_collection[RenderLayer]
    """ 

    :type: bpy_prop_collection[RenderLayer]
    """

    resolution_x: int
    """ 

    :type: int
    """

    resolution_y: int
    """ 

    :type: int
    """

    views: bpy_prop_collection[RenderView]
    """ 

    :type: bpy_prop_collection[RenderView]
    """

    def load_from_file(self, filename: str | typing.Any):
        """Copies the pixels of this render result from an image file

        :param filename: File Name, Filename to load into this render tile, must be no smaller than the render result
        :type filename: str | typing.Any
        """
        ...

class RenderSettings(bpy_struct):
    """Rendering settings for a Scene data-block"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    antialiasing_samples: str
    """ Amount of anti-aliasing samples per pixel

    :type: str
    """

    bake: BakeSettings
    """ 

    :type: BakeSettings
    """

    bake_aa_mode: str
    """ 

    :type: str
    """

    bake_bias: float
    """ Bias towards faces further away from the object (in blender units)

    :type: float
    """

    bake_distance: float
    """ Maximum distance from active object to other object (in blender units)

    :type: float
    """

    bake_margin: int
    """ Extends the baked result as a post process filter

    :type: int
    """

    bake_normal_space: str
    """ Choose normal space for baking

    :type: str
    """

    bake_quad_split: str
    """ Choose the method used to split a quad into 2 triangles for baking

    :type: str
    """

    bake_samples: int
    """ Number of samples used for ambient occlusion baking from multires

    :type: int
    """

    bake_type: str
    """ Choose shading information to bake into the image

    :type: str
    """

    bake_user_scale: float
    """ Instead of automatically normalizing to 0..1, apply a user scale to the derivative map

    :type: float
    """

    border_max_x: float
    """ Maximum X value for the render border

    :type: float
    """

    border_max_y: float
    """ Maximum Y value for the render border

    :type: float
    """

    border_min_x: float
    """ Minimum X value for the render border

    :type: float
    """

    border_min_y: float
    """ Minimum Y value for the render border

    :type: float
    """

    display_mode: str
    """ Select where rendered images will be displayed

    :type: str
    """

    dither_intensity: float
    """ Amount of dithering noise added to the rendered image to break up banding

    :type: float
    """

    edge_color: bpy_prop_array[float]
    """ Edge color

    :type: bpy_prop_array[float]
    """

    edge_threshold: int
    """ Threshold for drawing outlines on geometry edges

    :type: int
    """

    engine: str
    """ Engine to use for rendering

    :type: str
    """

    ffmpeg: FFmpegSettings
    """ FFmpeg related settings for the scene

    :type: FFmpegSettings
    """

    field_order: str
    """ Order of video fields (select which lines get rendered first, to create smooth motion for TV output)

    :type: str
    """

    file_extension: str
    """ The file extension used for saving renders

    :type: str
    """

    filepath: str
    """ Directory/name to save animations, # characters defines the position and length of frame numbers

    :type: str
    """

    filter_size: float
    """ Width over which the reconstruction filter combines samples

    :type: float
    """

    fps: int
    """ Framerate, expressed in frames per second

    :type: int
    """

    fps_base: float
    """ Framerate base

    :type: float
    """

    frame_map_new: int
    """ How many frames the Map Old will last

    :type: int
    """

    frame_map_old: int
    """ Old mapping value in frames

    :type: int
    """

    has_multiple_engines: bool
    """ More than one rendering engine is available

    :type: bool
    """

    image_settings: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    is_movie_format: bool
    """ When true the format is a movie

    :type: bool
    """

    layers: RenderLayers
    """ 

    :type: RenderLayers
    """

    line_thickness: float
    """ Line thickness in pixels

    :type: float
    """

    line_thickness_mode: str
    """ Line thickness mode for Freestyle line drawing

    :type: str
    """

    motion_blur_samples: int
    """ Number of scene samples to take with motion blur

    :type: int
    """

    motion_blur_shutter: float
    """ Time taken in frames between shutter open and close

    :type: float
    """

    motion_blur_shutter_curve: CurveMapping
    """ Curve defining the shutter's openness over time

    :type: CurveMapping
    """

    octree_resolution: str
    """ Resolution of raytrace accelerator, use higher resolutions for larger scenes

    :type: str
    """

    pixel_aspect_x: float
    """ Horizontal aspect ratio - for anamorphic or non-square pixel output

    :type: float
    """

    pixel_aspect_y: float
    """ Vertical aspect ratio - for anamorphic or non-square pixel output

    :type: float
    """

    pixel_filter_type: str
    """ Reconstruction filter used for combining anti-aliasing samples

    :type: str
    """

    preview_start_resolution: int
    """ Resolution to start rendering preview at, progressively increasing it to the full viewport size

    :type: int
    """

    raytrace_method: str
    """ Type of raytrace accelerator structure

    :type: str
    """

    resolution_percentage: int
    """ Percentage scale for render resolution

    :type: int
    """

    resolution_x: int
    """ Number of horizontal pixels in the rendered image

    :type: int
    """

    resolution_y: int
    """ Number of vertical pixels in the rendered image

    :type: int
    """

    sequencer_gl_preview: str
    """ Method to draw in the sequencer view

    :type: str
    """

    sequencer_gl_render: str
    """ Method to draw in the sequencer view

    :type: str
    """

    simplify_ao_sss: float
    """ Global approximate AO and SSS quality factor

    :type: float
    """

    simplify_child_particles: float
    """ Global child particles percentage

    :type: float
    """

    simplify_child_particles_render: float
    """ Global child particles percentage during rendering

    :type: float
    """

    simplify_shadow_samples: int
    """ Global maximum shadow samples

    :type: int
    """

    simplify_subdivision: int
    """ Global maximum subdivision level

    :type: int
    """

    simplify_subdivision_render: int
    """ Global maximum subdivision level during rendering

    :type: int
    """

    stamp_background: bpy_prop_array[float]
    """ Color to use behind stamp text

    :type: bpy_prop_array[float]
    """

    stamp_font_size: int
    """ Size of the font used when rendering stamp text

    :type: int
    """

    stamp_foreground: bpy_prop_array[float]
    """ Color to use for stamp text

    :type: bpy_prop_array[float]
    """

    stamp_note_text: str
    """ Custom text to appear in the stamp note

    :type: str
    """

    stereo_views: bpy_prop_collection[SceneRenderView]
    """ 

    :type: bpy_prop_collection[SceneRenderView]
    """

    threads: int
    """ Number of CPU threads to use simultaneously while rendering (for multi-core/CPU systems)

    :type: int
    """

    threads_mode: str
    """ Determine the amount of render threads used

    :type: str
    """

    tile_x: int
    """ Horizontal tile size to use while rendering

    :type: int
    """

    tile_y: int
    """ Vertical tile size to use while rendering

    :type: int
    """

    use_antialiasing: bool
    """ Render and combine multiple samples per pixel to prevent jagged edges

    :type: bool
    """

    use_bake_antialiasing: bool
    """ Enables Anti-aliasing

    :type: bool
    """

    use_bake_clear: bool
    """ Clear Images before baking

    :type: bool
    """

    use_bake_lores_mesh: bool
    """ Calculate heights against unsubdivided low resolution mesh

    :type: bool
    """

    use_bake_multires: bool
    """ Bake directly from multires object

    :type: bool
    """

    use_bake_normalize: bool
    """ With displacement normalize to the distance, with ambient occlusion normalize without using material settings

    :type: bool
    """

    use_bake_selected_to_active: bool
    """ Bake shading on the surface of selected objects to the active object

    :type: bool
    """

    use_bake_to_vertex_color: bool
    """ Bake to vertex colors instead of to a UV-mapped image

    :type: bool
    """

    use_bake_user_scale: bool
    """ Use a user scale for the derivative map

    :type: bool
    """

    use_border: bool
    """ Render a user-defined border region, within the frame size (note that this disables save_buffers and full_sample)

    :type: bool
    """

    use_compositing: bool
    """ Process the render result through the compositing pipeline, if compositing nodes are enabled

    :type: bool
    """

    use_crop_to_border: bool
    """ Crop the rendered frame to the defined border size

    :type: bool
    """

    use_edge_enhance: bool
    """ Create a toon outline around the edges of geometry

    :type: bool
    """

    use_envmaps: bool
    """ Calculate environment maps while rendering

    :type: bool
    """

    use_fields: bool
    """ Render image to two fields per frame, for interlaced TV output

    :type: bool
    """

    use_fields_still: bool
    """ Disable the time difference between fields

    :type: bool
    """

    use_file_extension: bool
    """ Add the file format extensions to the rendered file name (eg: filename + .jpg)

    :type: bool
    """

    use_free_image_textures: bool
    """ Free all image textures from memory after render, to save memory before compositing

    :type: bool
    """

    use_freestyle: bool
    """ Draw stylized strokes using Freestyle

    :type: bool
    """

    use_full_sample: bool
    """ Save for every anti-aliasing sample the entire RenderLayer results (this solves anti-aliasing issues with compositing)

    :type: bool
    """

    use_game_engine: bool
    """ Current rendering engine is a game engine

    :type: bool
    """

    use_instances: bool
    """ Instance support leads to effective memory reduction when using duplicates

    :type: bool
    """

    use_local_coords: bool
    """ Vertex coordinates are stored locally on each primitive (increases memory usage, but may have impact on speed)

    :type: bool
    """

    use_lock_interface: bool
    """ Lock interface during rendering in favor of giving more memory to the renderer

    :type: bool
    """

    use_motion_blur: bool
    """ Use multi-sampled 3D scene motion blur

    :type: bool
    """

    use_multiview: bool
    """ Use multiple views in the scene

    :type: bool
    """

    use_overwrite: bool
    """ Overwrite existing files while rendering

    :type: bool
    """

    use_persistent_data: bool
    """ Keep render data around for faster re-renders

    :type: bool
    """

    use_placeholder: bool
    """ Create empty placeholder files while rendering frames (similar to Unix 'touch')

    :type: bool
    """

    use_raytrace: bool
    """ Pre-calculate the raytrace accelerator and render raytracing effects

    :type: bool
    """

    use_render_cache: bool
    """ Save render cache to EXR files (useful for heavy compositing, Note: affects indirectly rendered scenes)

    :type: bool
    """

    use_save_buffers: bool
    """ Save tiles for all RenderLayers and SceneNodes to files in the temp directory (saves memory, required for Full Sample)

    :type: bool
    """

    use_sequencer: bool
    """ Process the render (and composited) result through the video sequence editor pipeline, if sequencer strips exist

    :type: bool
    """

    use_sequencer_gl_preview: bool
    """ 

    :type: bool
    """

    use_sequencer_gl_textured_solid: bool
    """ Draw face-assigned textures in solid draw method

    :type: bool
    """

    use_shading_nodes: bool
    """ Active render engine uses new shading nodes system

    :type: bool
    """

    use_shadows: bool
    """ Calculate shadows while rendering

    :type: bool
    """

    use_simplify: bool
    """ Enable simplification of scene for quicker preview renders

    :type: bool
    """

    use_simplify_triangulate: bool
    """ Disable non-planar quads being triangulated

    :type: bool
    """

    use_single_layer: bool
    """ Only render the active layer

    :type: bool
    """

    use_spherical_stereo: bool
    """ Active render engine supports spherical stereo rendering

    :type: bool
    """

    use_sss: bool
    """ Calculate sub-surface scattering in materials rendering

    :type: bool
    """

    use_stamp: bool
    """ Render the stamp info text in the rendered image

    :type: bool
    """

    use_stamp_camera: bool
    """ Include the name of the active camera in image metadata

    :type: bool
    """

    use_stamp_date: bool
    """ Include the current date in image metadata

    :type: bool
    """

    use_stamp_filename: bool
    """ Include the .blend filename in image metadata

    :type: bool
    """

    use_stamp_frame: bool
    """ Include the frame number in image metadata

    :type: bool
    """

    use_stamp_labels: bool
    """ Draw stamp labels ("Camera" in front of camera name, etc.)

    :type: bool
    """

    use_stamp_lens: bool
    """ Include the active camera's lens in image metadata

    :type: bool
    """

    use_stamp_marker: bool
    """ Include the name of the last marker in image metadata

    :type: bool
    """

    use_stamp_memory: bool
    """ Include the peak memory usage in image metadata

    :type: bool
    """

    use_stamp_note: bool
    """ Include a custom note in image metadata

    :type: bool
    """

    use_stamp_render_time: bool
    """ Include the render time in image metadata

    :type: bool
    """

    use_stamp_scene: bool
    """ Include the name of the active scene in image metadata

    :type: bool
    """

    use_stamp_sequencer_strip: bool
    """ Include the name of the foreground sequence strip in image metadata

    :type: bool
    """

    use_stamp_strip_meta: bool
    """ Use metadata from the strips in the sequencer

    :type: bool
    """

    use_stamp_time: bool
    """ Include the rendered frame timecode as HH:MM:SS.FF in image metadata

    :type: bool
    """

    use_textures: bool
    """ Use textures to affect material properties

    :type: bool
    """

    use_world_space_shading: bool
    """ Use world space interpretation of lighting data for node materials

    :type: bool
    """

    views: RenderViews
    """ 

    :type: RenderViews
    """

    views_format: str
    """ 

    :type: str
    """

    def frame_path(
        self,
        frame: typing.Any | None = -2147483648,
        preview: bool | typing.Any | None = False,
        view: str | typing.Any = "",
    ) -> str | typing.Any:
        """Return the absolute path to the filename to be written for a given frame

        :param frame: Frame number to use, if unset the current frame will be used
        :type frame: typing.Any | None
        :param preview: Preview, Use preview range
        :type preview: bool | typing.Any | None
        :param view: View, The name of the view to use to replace the "%" chars
        :type view: str | typing.Any
        :return: File Path, The resulting filepath from the scenes render settings
        :rtype: str | typing.Any
        """
        ...

class RenderSlot(bpy_struct):
    """Parameters defining the render slot"""

    name: str
    """ Render slot name

    :type: str
    """

class RenderSlots(bpy_prop_collection[RenderSlot], bpy_struct):
    """Collection of the render slots"""

    active: RenderSlot | None
    """ Active render slot of the image

    :type: RenderSlot | None
    """

    active_index: int | None
    """ Index of an active render slot of the image

    :type: int | None
    """

class RenderView(bpy_struct):
    name: str
    """ 

    :type: str
    """

class RenderViews(bpy_prop_collection[SceneRenderView], bpy_struct):
    """Collection of render views"""

    active: SceneRenderView
    """ Active Render View

    :type: SceneRenderView
    """

    active_index: int | None
    """ Active index in render view array

    :type: int | None
    """

    def new(self, name: str | typing.Any) -> SceneRenderView:
        """Add a render view to scene

        :param name: New name for the marker (not unique)
        :type name: str | typing.Any
        :return: Newly created render view
        :rtype: SceneRenderView
        """
        ...

    def remove(self, view: SceneRenderView):
        """Remove a render view

        :param view: Render view to remove
        :type view: SceneRenderView
        """
        ...

class RigidBodyConstraint(bpy_struct):
    """Constraint influencing Objects inside Rigid Body Simulation"""

    breaking_threshold: float
    """ Impulse threshold that must be reached for the constraint to break

    :type: float
    """

    disable_collisions: bool
    """ Disable collisions between constrained rigid bodies

    :type: bool
    """

    enabled: bool
    """ Enable this constraint

    :type: bool
    """

    limit_ang_x_lower: float
    """ Lower limit of X axis rotation

    :type: float
    """

    limit_ang_x_upper: float
    """ Upper limit of X axis rotation

    :type: float
    """

    limit_ang_y_lower: float
    """ Lower limit of Y axis rotation

    :type: float
    """

    limit_ang_y_upper: float
    """ Upper limit of Y axis rotation

    :type: float
    """

    limit_ang_z_lower: float
    """ Lower limit of Z axis rotation

    :type: float
    """

    limit_ang_z_upper: float
    """ Upper limit of Z axis rotation

    :type: float
    """

    limit_lin_x_lower: float
    """ Lower limit of X axis translation

    :type: float
    """

    limit_lin_x_upper: float
    """ Upper limit of X axis translation

    :type: float
    """

    limit_lin_y_lower: float
    """ Lower limit of Y axis translation

    :type: float
    """

    limit_lin_y_upper: float
    """ Upper limit of Y axis translation

    :type: float
    """

    limit_lin_z_lower: float
    """ Lower limit of Z axis translation

    :type: float
    """

    limit_lin_z_upper: float
    """ Upper limit of Z axis translation

    :type: float
    """

    motor_ang_max_impulse: float
    """ Maximum angular motor impulse

    :type: float
    """

    motor_ang_target_velocity: float
    """ Target angular motor velocity

    :type: float
    """

    motor_lin_max_impulse: float
    """ Maximum linear motor impulse

    :type: float
    """

    motor_lin_target_velocity: float
    """ Target linear motor velocity

    :type: float
    """

    object1: Object
    """ First Rigid Body Object to be constrained

    :type: Object
    """

    object2: Object
    """ Second Rigid Body Object to be constrained

    :type: Object
    """

    solver_iterations: int
    """ Number of constraint solver iterations made per simulation step (higher values are more accurate but slower)

    :type: int
    """

    spring_damping_x: float
    """ Damping on the X axis

    :type: float
    """

    spring_damping_y: float
    """ Damping on the Y axis

    :type: float
    """

    spring_damping_z: float
    """ Damping on the Z axis

    :type: float
    """

    spring_stiffness_x: float
    """ Stiffness on the X axis

    :type: float
    """

    spring_stiffness_y: float
    """ Stiffness on the Y axis

    :type: float
    """

    spring_stiffness_z: float
    """ Stiffness on the Z axis

    :type: float
    """

    type: str
    """ Type of Rigid Body Constraint

    :type: str
    """

    use_breaking: bool
    """ Constraint can be broken if it receives an impulse above the threshold

    :type: bool
    """

    use_limit_ang_x: bool
    """ Limit rotation around X axis

    :type: bool
    """

    use_limit_ang_y: bool
    """ Limit rotation around Y axis

    :type: bool
    """

    use_limit_ang_z: bool
    """ Limit rotation around Z axis

    :type: bool
    """

    use_limit_lin_x: bool
    """ Limit translation on X axis

    :type: bool
    """

    use_limit_lin_y: bool
    """ Limit translation on Y axis

    :type: bool
    """

    use_limit_lin_z: bool
    """ Limit translation on Z axis

    :type: bool
    """

    use_motor_ang: bool
    """ Enable angular motor

    :type: bool
    """

    use_motor_lin: bool
    """ Enable linear motor

    :type: bool
    """

    use_override_solver_iterations: bool
    """ Override the number of solver iterations for this constraint

    :type: bool
    """

    use_spring_x: bool
    """ Enable spring on X axis

    :type: bool
    """

    use_spring_y: bool
    """ Enable spring on Y axis

    :type: bool
    """

    use_spring_z: bool
    """ Enable spring on Z axis

    :type: bool
    """

class RigidBodyObject(bpy_struct):
    """Settings for object participating in Rigid Body Simulation"""

    angular_damping: float
    """ Amount of angular velocity that is lost over time

    :type: float
    """

    collision_groups: list[bool]
    """ Collision Groups Rigid Body belongs to

    :type: list[bool]
    """

    collision_margin: float
    """ Threshold of distance near surface where collisions are still considered (best results when non-zero)

    :type: float
    """

    collision_shape: str
    """ Collision Shape of object in Rigid Body Simulations

    :type: str
    """

    deactivate_angular_velocity: float
    """ Angular Velocity below which simulation stops simulating object

    :type: float
    """

    deactivate_linear_velocity: float
    """ Linear Velocity below which simulation stops simulating object

    :type: float
    """

    enabled: bool
    """ Rigid Body actively participates to the simulation

    :type: bool
    """

    friction: float
    """ Resistance of object to movement

    :type: float
    """

    kinematic: bool
    """ Allow rigid body to be controlled by the animation system

    :type: bool
    """

    linear_damping: float
    """ Amount of linear velocity that is lost over time

    :type: float
    """

    mass: float
    """ How much the object 'weighs' irrespective of gravity

    :type: float
    """

    mesh_source: str
    """ Source of the mesh used to create collision shape

    :type: str
    """

    restitution: float
    """ Tendency of object to bounce after colliding with another (0 = stays still, 1 = perfectly elastic)

    :type: float
    """

    type: str
    """ Role of object in Rigid Body Simulations

    :type: str
    """

    use_deactivation: bool
    """ Enable deactivation of resting rigid bodies (increases performance and stability but can cause glitches)

    :type: bool
    """

    use_deform: bool
    """ Rigid body deforms during simulation

    :type: bool
    """

    use_margin: bool
    """ Use custom collision margin (some shapes will have a visible gap around them)

    :type: bool
    """

    use_start_deactivated: bool
    """ Deactivate rigid body at the start of the simulation

    :type: bool
    """

class RigidBodyWorld(bpy_struct):
    """Self-contained rigid body simulation environment and settings"""

    constraints: Group
    """ Group containing rigid body constraint objects

    :type: Group
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    enabled: bool
    """ Simulation will be evaluated

    :type: bool
    """

    group: Group
    """ Group containing objects participating in this simulation

    :type: Group
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    solver_iterations: int
    """ Number of constraint solver iterations made per simulation step (higher values are more accurate but slower)

    :type: int
    """

    steps_per_second: int
    """ Number of simulation steps taken per second (higher values are more accurate but slower)

    :type: int
    """

    time_scale: float
    """ Change the speed of the simulation

    :type: float
    """

    use_split_impulse: bool
    """ Reduce extra velocity that can build up when objects collide (lowers simulation stability a little so use only when necessary)

    :type: bool
    """

    def convex_sweep_test(
        self,
        object: Object,
        start: collections.abc.Iterable[float] | None,
        end: collections.abc.Iterable[float] | None,
    ):
        """Sweep test convex rigidbody against the current rigidbody world

                :param object: Rigidbody object with a convex collision shape
                :type object: Object
                :param start:
                :type start: collections.abc.Iterable[float] | None
                :param end:
                :type end: collections.abc.Iterable[float] | None
                :return: object_location, The hit location of this sweep test, float array of 3 items in [-inf, inf]

        hitpoint, The hit location of this sweep test, float array of 3 items in [-inf, inf]

        normal, The face normal at the sweep test hit location, float array of 3 items in [-inf, inf]

        has_hit, If the function has found collision point, value is 1, otherwise 0, int in [-inf, inf]
        """
        ...

class SPHFluidSettings(bpy_struct):
    """Settings for particle fluids physics"""

    buoyancy: float
    """ Artificial buoyancy force in negative gravity direction based on pressure differences inside the fluid

    :type: float
    """

    factor_radius: bool
    """ Interaction radius is a factor of 4 * particle size

    :type: bool
    """

    factor_repulsion: bool
    """ Repulsion is a factor of stiffness

    :type: bool
    """

    factor_rest_length: bool
    """ Spring rest length is a factor of 2 * particle size

    :type: bool
    """

    factor_stiff_viscosity: bool
    """ Stiff viscosity is a factor of normal viscosity

    :type: bool
    """

    fluid_radius: float
    """ Fluid interaction radius

    :type: float
    """

    linear_viscosity: float
    """ Linear viscosity

    :type: float
    """

    plasticity: float
    """ How much the spring rest length can change after the elastic limit is crossed

    :type: float
    """

    repulsion: float
    """ How strongly the fluid tries to keep from clustering (factor of stiffness)

    :type: float
    """

    rest_density: float
    """ Fluid rest density

    :type: float
    """

    rest_length: float
    """ Spring rest length (factor of particle radius)

    :type: float
    """

    solver: str
    """ The code used to calculate internal forces on particles

    :type: str
    """

    spring_force: float
    """ Spring force

    :type: float
    """

    spring_frames: int
    """ Create springs for this number of frames since particles birth (0 is always)

    :type: int
    """

    stiff_viscosity: float
    """ Creates viscosity for expanding fluid

    :type: float
    """

    stiffness: float
    """ How incompressible the fluid is (speed of sound)

    :type: float
    """

    use_factor_density: bool
    """ Density is calculated as a factor of default density (depends on particle size)

    :type: bool
    """

    use_initial_rest_length: bool
    """ Use the initial length as spring rest length instead of 2 * particle size

    :type: bool
    """

    use_viscoelastic_springs: bool
    """ Use viscoelastic springs instead of Hooke's springs

    :type: bool
    """

    yield_ratio: float
    """ How much the spring has to be stretched/compressed in order to change it's rest length

    :type: float
    """

class SceneBases(bpy_prop_collection[ObjectBase], bpy_struct):
    """Collection of scene bases"""

    active: ObjectBase | None
    """ Active object base in the scene

    :type: ObjectBase | None
    """

class SceneGameData(bpy_struct):
    """Game data for a Scene data-block"""

    activity_culling_box_radius: float
    """ Radius of the activity bubble, in Manhattan length (objects outside the box are activity-culled)

    :type: float
    """

    deactivation_angular_threshold: float
    """ Angular velocity that an object must be below before the deactivation timer can start

    :type: float
    """

    deactivation_linear_threshold: float
    """ Linear velocity that an object must be below before the deactivation timer can start

    :type: float
    """

    deactivation_time: float
    """ Amount of time (in seconds) after which objects with a velocity less than the given threshold will deactivate (0.0 means no deactivation)

    :type: float
    """

    depth: int
    """ Display bit depth of full screen display

    :type: int
    """

    dome_angle: int
    """ Field of View of the Dome - it only works in mode Fisheye and Truncated

    :type: int
    """

    dome_buffer_resolution: float
    """ Buffer Resolution - decrease it to increase speed

    :type: float
    """

    dome_mode: str
    """ Dome physical configurations

    :type: str
    """

    dome_tessellation: int
    """ Tessellation level - check the generated mesh in wireframe mode

    :type: int
    """

    dome_text: Text
    """ Custom Warp Mesh data file

    :type: Text
    """

    dome_tilt: int
    """ Camera rotation in horizontal axis

    :type: int
    """

    exit_key: str
    """ The key that exits the Game Engine

    :type: str
    """

    fps: int
    """ Nominal number of game frames per second (physics fixed timestep = 1/fps, independently of actual frame rate)

    :type: int
    """

    frame_color: bpy_prop_array[float]
    """ Set color of the bars

    :type: bpy_prop_array[float]
    """

    frame_type: str
    """ Select the type of Framing you want

    :type: str
    """

    frequency: int
    """ Display clock frequency of fullscreen display

    :type: int
    """

    level_height: float
    """ Max difference in heights of obstacles to enable their interaction

    :type: float
    """

    logic_step_max: int
    """ Maximum number of logic frame per game frame if graphics slows down the game, higher value allows better synchronization with physics

    :type: int
    """

    material_mode: str
    """ Material mode to use for rendering

    :type: str
    """

    obstacle_simulation: str
    """ Simulation used for obstacle avoidance in the game engine

    :type: str
    """

    occlusion_culling_resolution: int
    """ Size of the occlusion buffer, use higher value for better precision (slower)

    :type: int
    """

    physics_engine: str
    """ Physics engine used for physics simulation in the game engine

    :type: str
    """

    physics_gravity: float
    """ Gravitational constant used for physics simulation in the game engine

    :type: float
    """

    physics_step_max: int
    """ Maximum number of physics step per game frame if graphics slows down the game, higher value allows physics to keep up with realtime

    :type: int
    """

    physics_step_sub: int
    """ Number of simulation substep per physic timestep, higher value give better physics precision

    :type: int
    """

    raster_storage: str
    """ Set the storage mode used by the rasterizer

    :type: str
    """

    recast_data: SceneGameRecastData
    """ 

    :type: SceneGameRecastData
    """

    resolution_x: int
    """ Number of horizontal pixels in the screen

    :type: int
    """

    resolution_y: int
    """ Number of vertical pixels in the screen

    :type: int
    """

    samples: str
    """ The number of AA Samples to use for MSAA

    :type: str
    """

    scene_hysteresis_percentage: int
    """ Minimum distance change required to transition to the previous level of detail

    :type: int
    """

    show_debug_properties: bool
    """ Show properties marked for debugging while the game runs

    :type: bool
    """

    show_framerate_profile: bool
    """ Show framerate and profiling information while the game runs

    :type: bool
    """

    show_fullscreen: bool
    """ Start player in a new fullscreen display

    :type: bool
    """

    show_mouse: bool
    """ Start player with a visible mouse cursor

    :type: bool
    """

    show_obstacle_simulation: bool
    """ Enable debug visualization for obstacle simulation

    :type: bool
    """

    show_physics_visualization: bool
    """ Show a visualization of physics bounds and interactions

    :type: bool
    """

    stereo: str
    """ 

    :type: str
    """

    stereo_eye_separation: float
    """ Set the distance between the eyes - the camera focal distance/30 should be fine

    :type: float
    """

    stereo_mode: str
    """ Stereographic techniques

    :type: str
    """

    use_activity_culling: bool
    """ Activity culling is enabled

    :type: bool
    """

    use_animation_record: bool
    """ Record animation to F-Curves

    :type: bool
    """

    use_auto_start: bool
    """ Automatically start game at load time

    :type: bool
    """

    use_deprecation_warnings: bool
    """ Print warnings when using deprecated features in the python API

    :type: bool
    """

    use_desktop: bool
    """ Use the current desktop resolution in fullscreen mode

    :type: bool
    """

    use_display_lists: bool
    """ Use display lists to speed up rendering by keeping geometry on the GPU

    :type: bool
    """

    use_frame_rate: bool
    """ Respect the frame rate from the Physics panel in the world properties rather than rendering as many frames as possible

    :type: bool
    """

    use_glsl_color_management: bool
    """ Use color management for GLSL rendering

    :type: bool
    """

    use_glsl_environment_lighting: bool
    """ Use environment lighting for GLSL rendering

    :type: bool
    """

    use_glsl_extra_textures: bool
    """ Use extra textures like normal or specular maps for GLSL rendering

    :type: bool
    """

    use_glsl_lights: bool
    """ Use lights for GLSL rendering

    :type: bool
    """

    use_glsl_nodes: bool
    """ Use nodes for GLSL rendering

    :type: bool
    """

    use_glsl_ramps: bool
    """ Use ramps for GLSL rendering

    :type: bool
    """

    use_glsl_shaders: bool
    """ Use shaders for GLSL rendering

    :type: bool
    """

    use_glsl_shadows: bool
    """ Use shadows for GLSL rendering

    :type: bool
    """

    use_material_caching: bool
    """ Cache materials in the converter (this is faster, but can cause problems with older Singletexture and Multitexture games)

    :type: bool
    """

    use_occlusion_culling: bool
    """ Use optimized Bullet DBVT tree for view frustum and occlusion culling (more efficient, but it can waste unnecessary CPU if the scene doesn't have occluder objects)

    :type: bool
    """

    use_restrict_animation_updates: bool
    """ Restrict the number of animation updates to the animation FPS (this is better for performance, but can cause issues with smooth playback)

    :type: bool
    """

    use_scene_hysteresis: bool
    """ Use LoD Hysteresis setting for the scene

    :type: bool
    """

    vsync: str
    """ Change vsync settings

    :type: str
    """

class SceneGameRecastData(bpy_struct):
    """Recast data for a Game data-block"""

    agent_height: float
    """ Minimum height where the agent can still walk

    :type: float
    """

    agent_radius: float
    """ Radius of the agent

    :type: float
    """

    cell_height: float
    """ Rasterized cell height

    :type: float
    """

    cell_size: float
    """ Rasterized cell size

    :type: float
    """

    climb_max: float
    """ Maximum height between grid cells the agent can climb

    :type: float
    """

    edge_max_error: float
    """ Maximum distance error from contour to cells

    :type: float
    """

    edge_max_len: float
    """ Maximum contour edge length

    :type: float
    """

    partitioning: str
    """ Choose partitioning method

    :type: str
    """

    region_merge_size: float
    """ Minimum regions size (smaller regions will be merged)

    :type: float
    """

    region_min_size: float
    """ Minimum regions size (smaller regions will be deleted)

    :type: float
    """

    sample_dist: float
    """ Detail mesh sample spacing

    :type: float
    """

    sample_max_error: float
    """ Detail mesh simplification max sample error

    :type: float
    """

    slope_max: float
    """ Maximum walkable slope angle

    :type: float
    """

    verts_per_poly: int
    """ Max number of vertices per polygon

    :type: int
    """

class SceneObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of scene objects"""

    active: Object | None
    """ Active object for this scene

    :type: Object | None
    """

    def link(self, object: Object) -> ObjectBase:
        """Link object to scene, run scene.update() after

        :param object: Object to add to scene
        :type object: Object
        :return: The newly created base
        :rtype: ObjectBase
        """
        ...

    def unlink(self, object: Object):
        """Unlink object from scene

        :param object: Object to remove from scene
        :type object: Object
        """
        ...

class SceneRenderLayer(bpy_struct):
    """Render layer"""

    exclude_ambient_occlusion: bool
    """ Exclude AO pass from combined

    :type: bool
    """

    exclude_emit: bool
    """ Exclude emission pass from combined

    :type: bool
    """

    exclude_environment: bool
    """ Exclude environment pass from combined

    :type: bool
    """

    exclude_indirect: bool
    """ Exclude indirect pass from combined

    :type: bool
    """

    exclude_reflection: bool
    """ Exclude raytraced reflection pass from combined

    :type: bool
    """

    exclude_refraction: bool
    """ Exclude raytraced refraction pass from combined

    :type: bool
    """

    exclude_shadow: bool
    """ Exclude shadow pass from combined

    :type: bool
    """

    exclude_specular: bool
    """ Exclude specular pass from combined

    :type: bool
    """

    freestyle_settings: FreestyleSettings
    """ 

    :type: FreestyleSettings
    """

    invert_zmask: bool
    """ For Zmask, only render what is behind solid z values instead of in front

    :type: bool
    """

    layers: list[bool]
    """ Scene layers included in this render layer

    :type: list[bool]
    """

    layers_exclude: list[bool]
    """ Exclude scene layers from having any influence

    :type: list[bool]
    """

    layers_zmask: list[bool]
    """ Zmask scene layers for solid faces

    :type: list[bool]
    """

    light_override: Group
    """ Group to override all other lights in this render layer

    :type: Group
    """

    material_override: Material
    """ Material to override all other materials in this render layer

    :type: Material
    """

    name: str
    """ Render layer name

    :type: str
    """

    pass_alpha_threshold: float
    """ Z, Index, normal, UV and vector passes are only affected by surfaces with alpha transparency equal to or higher than this threshold

    :type: float
    """

    samples: int
    """ Override number of render samples for this render layer, 0 will use the scene setting

    :type: int
    """

    use: bool
    """ Disable or enable the render layer

    :type: bool
    """

    use_all_z: bool
    """ Fill in Z values for solid faces in invisible layers, for masking

    :type: bool
    """

    use_ao: bool
    """ Render AO in this Layer

    :type: bool
    """

    use_edge_enhance: bool
    """ Render Edge-enhance in this Layer (only works for Solid faces)

    :type: bool
    """

    use_freestyle: bool
    """ Render stylized strokes in this Layer

    :type: bool
    """

    use_halo: bool
    """ Render Halos in this Layer (on top of Solid)

    :type: bool
    """

    use_pass_ambient_occlusion: bool
    """ Deliver AO pass

    :type: bool
    """

    use_pass_color: bool
    """ Deliver shade-less color pass

    :type: bool
    """

    use_pass_combined: bool
    """ Deliver full combined RGBA buffer

    :type: bool
    """

    use_pass_diffuse: bool
    """ Deliver diffuse pass

    :type: bool
    """

    use_pass_diffuse_color: bool
    """ Deliver diffuse color pass

    :type: bool
    """

    use_pass_diffuse_direct: bool
    """ Deliver diffuse direct pass

    :type: bool
    """

    use_pass_diffuse_indirect: bool
    """ Deliver diffuse indirect pass

    :type: bool
    """

    use_pass_emit: bool
    """ Deliver emission pass

    :type: bool
    """

    use_pass_environment: bool
    """ Deliver environment lighting pass

    :type: bool
    """

    use_pass_glossy_color: bool
    """ Deliver glossy color pass

    :type: bool
    """

    use_pass_glossy_direct: bool
    """ Deliver glossy direct pass

    :type: bool
    """

    use_pass_glossy_indirect: bool
    """ Deliver glossy indirect pass

    :type: bool
    """

    use_pass_indirect: bool
    """ Deliver indirect lighting pass

    :type: bool
    """

    use_pass_material_index: bool
    """ Deliver material index pass

    :type: bool
    """

    use_pass_mist: bool
    """ Deliver mist factor pass (0.0-1.0)

    :type: bool
    """

    use_pass_normal: bool
    """ Deliver normal pass

    :type: bool
    """

    use_pass_object_index: bool
    """ Deliver object index pass

    :type: bool
    """

    use_pass_reflection: bool
    """ Deliver raytraced reflection pass

    :type: bool
    """

    use_pass_refraction: bool
    """ Deliver raytraced refraction pass

    :type: bool
    """

    use_pass_shadow: bool
    """ Deliver shadow pass

    :type: bool
    """

    use_pass_specular: bool
    """ Deliver specular pass

    :type: bool
    """

    use_pass_subsurface_color: bool
    """ Deliver subsurface color pass

    :type: bool
    """

    use_pass_subsurface_direct: bool
    """ Deliver subsurface direct pass

    :type: bool
    """

    use_pass_subsurface_indirect: bool
    """ Deliver subsurface indirect pass

    :type: bool
    """

    use_pass_transmission_color: bool
    """ Deliver transmission color pass

    :type: bool
    """

    use_pass_transmission_direct: bool
    """ Deliver transmission direct pass

    :type: bool
    """

    use_pass_transmission_indirect: bool
    """ Deliver transmission indirect pass

    :type: bool
    """

    use_pass_uv: bool
    """ Deliver texture UV pass

    :type: bool
    """

    use_pass_vector: bool
    """ Deliver speed vector pass

    :type: bool
    """

    use_pass_z: bool
    """ Deliver Z values pass

    :type: bool
    """

    use_sky: bool
    """ Render Sky in this Layer

    :type: bool
    """

    use_solid: bool
    """ Render Solid faces in this Layer

    :type: bool
    """

    use_strand: bool
    """ Render Strands in this Layer

    :type: bool
    """

    use_zmask: bool
    """ Only render what's in front of the solid z values

    :type: bool
    """

    use_ztransp: bool
    """ Render Z-Transparent faces in this Layer (on top of Solid and Halos)

    :type: bool
    """

class SceneRenderView(bpy_struct):
    """Render viewpoint for 3D stereo and multiview rendering"""

    camera_suffix: str
    """ Suffix to identify the cameras to use, and added to the render images for this view

    :type: str
    """

    file_suffix: str
    """ Suffix added to the render images for this view

    :type: str
    """

    name: str
    """ Render view name

    :type: str
    """

    use: bool
    """ Disable or enable the render view

    :type: bool
    """

class Scopes(bpy_struct):
    """Scopes for statistical view of an image"""

    accuracy: float
    """ Proportion of original image source pixel lines to sample

    :type: float
    """

    histogram: Histogram
    """ Histogram for viewing image statistics

    :type: Histogram
    """

    use_full_resolution: bool
    """ Sample every pixel of the image

    :type: bool
    """

    vectorscope_alpha: float
    """ Opacity of the points

    :type: float
    """

    waveform_alpha: float
    """ Opacity of the points

    :type: float
    """

    waveform_mode: str
    """ 

    :type: str
    """

class SculptToolCapabilities(bpy_struct):
    """Read-only indications of which brush operations are supported by the current sculpt tool"""

    has_accumulate: bool
    """ 

    :type: bool
    """

    has_auto_smooth: bool
    """ 

    :type: bool
    """

    has_gravity: bool
    """ 

    :type: bool
    """

    has_height: bool
    """ 

    :type: bool
    """

    has_jitter: bool
    """ 

    :type: bool
    """

    has_normal_weight: bool
    """ 

    :type: bool
    """

    has_persistence: bool
    """ 

    :type: bool
    """

    has_pinch_factor: bool
    """ 

    :type: bool
    """

    has_plane_offset: bool
    """ 

    :type: bool
    """

    has_rake_factor: bool
    """ 

    :type: bool
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_sculpt_plane: bool
    """ 

    :type: bool
    """

    has_secondary_color: bool
    """ 

    :type: bool
    """

    has_smooth_stroke: bool
    """ 

    :type: bool
    """

    has_space_attenuation: bool
    """ 

    :type: bool
    """

    has_strength_pressure: bool
    """ 

    :type: bool
    """

class Sensor(bpy_struct):
    """Game engine logic brick to detect events"""

    active: bool | None
    """ Set active state of the sensor

    :type: bool | None
    """

    controllers: bpy_prop_collection[Controller]
    """ The list containing the controllers connected to the sensor

    :type: bpy_prop_collection[Controller]
    """

    invert: bool
    """ Invert the level(output) of this sensor

    :type: bool
    """

    name: str
    """ Sensor name

    :type: str
    """

    pin: bool
    """ Display when not linked to a visible states controller

    :type: bool
    """

    show_expanded: bool
    """ Set sensor expanded in the user interface

    :type: bool
    """

    tick_skip: int
    """ Number of logic ticks skipped between 2 active pulses (0 = pulse every logic tick, 1 = skip 1 logic tick between pulses, etc.)

    :type: int
    """

    type: str
    """ 

    :type: str
    """

    use_level: bool
    """ Level detector, trigger controllers of new states (only applicable upon logic state transition)

    :type: bool
    """

    use_pulse_false_level: bool
    """ Activate FALSE level triggering (pulse mode)

    :type: bool
    """

    use_pulse_true_level: bool
    """ Activate TRUE level triggering (pulse mode)

    :type: bool
    """

    use_tap: bool
    """ Trigger controllers only for an instant, even while the sensor remains true

    :type: bool
    """

    def link(self, controller: Controller | None):
        """Link the sensor to a controller

        :param controller: Controller to link to
        :type controller: Controller | None
        """
        ...

    def unlink(self, controller: Controller | None):
        """Unlink the sensor from a controller

        :param controller: Controller to unlink from
        :type controller: Controller | None
        """
        ...

class Sequence(bpy_struct):
    """Sequence strip in the sequence editor"""

    blend_alpha: float
    """ 

    :type: float
    """

    blend_type: str
    """ 

    :type: str
    """

    channel: int
    """ Y position of the sequence strip

    :type: int
    """

    effect_fader: float
    """ 

    :type: float
    """

    frame_duration: int
    """ The length of the contents of this strip before the handles are applied

    :type: int
    """

    frame_final_duration: int
    """ The length of the contents of this strip after the handles are applied

    :type: int
    """

    frame_final_end: int
    """ End frame displayed in the sequence editor after offsets are applied

    :type: int
    """

    frame_final_start: int
    """ Start frame displayed in the sequence editor after offsets are applied, setting this is equivalent to moving the handle, not the actual start frame

    :type: int
    """

    frame_offset_end: int
    """ 

    :type: int
    """

    frame_offset_start: int
    """ 

    :type: int
    """

    frame_start: int
    """ 

    :type: int
    """

    frame_still_end: int
    """ 

    :type: int
    """

    frame_still_start: int
    """ 

    :type: int
    """

    lock: bool
    """ Lock strip so that it can't be transformed

    :type: bool
    """

    modifiers: SequenceModifiers
    """ Modifiers affecting this strip

    :type: SequenceModifiers
    """

    mute: bool
    """ 

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ 

    :type: bool
    """

    select_left_handle: bool
    """ 

    :type: bool
    """

    select_right_handle: bool
    """ 

    :type: bool
    """

    speed_factor: float
    """ Multiply the current speed of the sequence with this number or remap current frame to this frame

    :type: float
    """

    type: str
    """ 

    :type: str
    """

    use_default_fade: bool
    """ Fade effect using the built-in default (usually make transition as long as effect strip)

    :type: bool
    """

    use_linear_modifiers: bool
    """ Calculate modifiers in linear space instead of sequencer's space

    :type: bool
    """

    def update(self, data: bool | typing.Any | None = False):
        """Update the strip dimensions

        :param data: Data, Update strip data
        :type data: bool | typing.Any | None
        """
        ...

    def strip_elem_from_frame(self, frame: int | None) -> SequenceElement:
        """Return the strip element from a given frame or None

        :param frame: Frame, The frame to get the strip element from
        :type frame: int | None
        :return: strip element of the current frame
        :rtype: SequenceElement
        """
        ...

    def swap(self, other: Sequence):
        """swap

        :param other: Other
        :type other: Sequence
        """
        ...

class SequenceColorBalanceData(bpy_struct):
    """Color balance parameters for a sequence strip and it's modifiers"""

    gain: bpy_prop_array[float]
    """ Color balance gain (highlights)

    :type: bpy_prop_array[float]
    """

    gamma: bpy_prop_array[float]
    """ Color balance gamma (midtones)

    :type: bpy_prop_array[float]
    """

    invert_gain: bool
    """ 

    :type: bool
    """

    invert_gamma: bool
    """ 

    :type: bool
    """

    invert_lift: bool
    """ 

    :type: bool
    """

    lift: bpy_prop_array[float]
    """ Color balance lift (shadows)

    :type: bpy_prop_array[float]
    """

class SequenceCrop(bpy_struct):
    """Cropping parameters for a sequence strip"""

    max_x: int
    """ 

    :type: int
    """

    max_y: int
    """ 

    :type: int
    """

    min_x: int
    """ 

    :type: int
    """

    min_y: int
    """ 

    :type: int
    """

class SequenceEditor(bpy_struct):
    """Sequence editing data for a Scene data-block"""

    active_strip: Sequence | None
    """ Sequencer's active strip

    :type: Sequence | None
    """

    meta_stack: bpy_prop_collection[Sequence]
    """ Meta strip stack, last is currently edited meta strip

    :type: bpy_prop_collection[Sequence]
    """

    overlay_frame: int
    """ 

    :type: int
    """

    proxy_dir: str
    """ 

    :type: str
    """

    proxy_storage: str
    """ How to store proxies for this project

    :type: str
    """

    sequences: Sequences
    """ Top-level strips only

    :type: Sequences
    """

    sequences_all: bpy_prop_collection[Sequence]
    """ All strips, recursively including those inside metastrips

    :type: bpy_prop_collection[Sequence]
    """

    show_overlay: bool
    """ Partial overlay on top of the sequencer

    :type: bool
    """

    use_overlay_lock: bool
    """ 

    :type: bool
    """

class SequenceElement(bpy_struct):
    """Sequence strip data for a single frame"""

    filename: str
    """ 

    :type: str
    """

    orig_height: int
    """ Original image height

    :type: int
    """

    orig_width: int
    """ Original image width

    :type: int
    """

class SequenceElements(bpy_prop_collection[SequenceElement], bpy_struct):
    """Collection of SequenceElement"""

    def append(self, filename: str | typing.Any) -> SequenceElement:
        """Push an image from ImageSequence.directory

        :param filename: Filepath to image
        :type filename: str | typing.Any
        :return: New SequenceElement
        :rtype: SequenceElement
        """
        ...

    def pop(self, index: int | None):
        """Pop an image off the collection

        :param index: Index of image to remove
        :type index: int | None
        """
        ...

class SequenceModifier(bpy_struct):
    """Modifier for sequence strip"""

    input_mask_id: Mask
    """ Mask ID used as mask input for the modifier

    :type: Mask
    """

    input_mask_strip: Sequence
    """ Strip used as mask input for the modifier

    :type: Sequence
    """

    input_mask_type: str
    """ Type of input data used for mask

    :type: str
    """

    mask_time: str
    """ Time to use for the Mask animation

    :type: str
    """

    mute: bool
    """ Mute this modifier

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    show_expanded: bool
    """ Mute expanded settings for the modifier

    :type: bool
    """

    type: str
    """ 

    :type: str
    """

class SequenceModifiers(bpy_prop_collection[SequenceModifier], bpy_struct):
    """Collection of strip modifiers"""

    def new(self, name: str | typing.Any, type: str | None) -> SequenceModifier:
        """Add a new modifier

        :param name: New name for the modifier
        :type name: str | typing.Any
        :param type: Modifier type to add
        :type type: str | None
        :return: Newly created modifier
        :rtype: SequenceModifier
        """
        ...

    def remove(self, modifier: SequenceModifier):
        """Remove an existing modifier from the sequence

        :param modifier: Modifier to remove
        :type modifier: SequenceModifier
        """
        ...

    def clear(self):
        """Remove all modifiers from the sequence"""
        ...

class SequenceProxy(bpy_struct):
    """Proxy parameters for a sequence strip"""

    build_100: bool
    """ Build 100% proxy resolution

    :type: bool
    """

    build_25: bool
    """ Build 25% proxy resolution

    :type: bool
    """

    build_50: bool
    """ Build 50% proxy resolution

    :type: bool
    """

    build_75: bool
    """ Build 75% proxy resolution

    :type: bool
    """

    build_free_run: bool
    """ Build free run time code index

    :type: bool
    """

    build_free_run_rec_date: bool
    """ Build free run time code index using Record Date/Time

    :type: bool
    """

    build_record_run: bool
    """ Build record run time code index

    :type: bool
    """

    directory: str
    """ Location to store the proxy files

    :type: str
    """

    filepath: str
    """ Location of custom proxy file

    :type: str
    """

    quality: int
    """ JPEG Quality of proxies to build

    :type: int
    """

    timecode: str
    """ 

    :type: str
    """

    use_overwrite: bool
    """ Overwrite existing proxy files when building

    :type: bool
    """

    use_proxy_custom_directory: bool
    """ Use a custom directory to store data

    :type: bool
    """

    use_proxy_custom_file: bool
    """ Use a custom file to read proxy data from

    :type: bool
    """

class SequenceTransform(bpy_struct):
    """Transform parameters for a sequence strip"""

    offset_x: int
    """ 

    :type: int
    """

    offset_y: int
    """ 

    :type: int
    """

class Sequences(bpy_prop_collection[Sequence], bpy_struct):
    """Collection of Sequences"""

    def new_clip(
        self,
        name: str | typing.Any,
        clip: MovieClip,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new movie clip sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param clip: Movie clip to add
        :type clip: MovieClip
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_mask(
        self,
        name: str | typing.Any,
        mask: Mask,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new mask sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param mask: Mask to add
        :type mask: Mask
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_scene(
        self,
        name: str | typing.Any,
        scene: Scene,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new scene sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param scene: Scene to add
        :type scene: Scene
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_image(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new image sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param filepath: Filepath to image
        :type filepath: str | typing.Any
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_movie(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new movie sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param filepath: Filepath to movie
        :type filepath: str | typing.Any
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_sound(
        self,
        name: str | typing.Any,
        filepath: str | typing.Any,
        channel: int | None,
        frame_start: int | None,
    ) -> Sequence:
        """Add a new sound sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param filepath: Filepath to movie
        :type filepath: str | typing.Any
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def new_effect(
        self,
        name: str | typing.Any,
        type: str | None,
        channel: int | None,
        frame_start: int | None,
        frame_end: typing.Any | None = 0,
        seq1: Sequence | None = None,
        seq2: Sequence | None = None,
        seq3: Sequence | None = None,
    ) -> Sequence:
        """Add a new effect sequence

        :param name: Name for the new sequence
        :type name: str | typing.Any
        :param type: Type, type for the new sequence
        :type type: str | None
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :param frame_end: The end frame for the new sequence
        :type frame_end: typing.Any | None
        :param seq1: Sequence 1 for effect
        :type seq1: Sequence | None
        :param seq2: Sequence 2 for effect
        :type seq2: Sequence | None
        :param seq3: Sequence 3 for effect
        :type seq3: Sequence | None
        :return: New Sequence
        :rtype: Sequence
        """
        ...

    def remove(self, sequence: Sequence):
        """Remove a Sequence

        :param sequence: Sequence to remove
        :type sequence: Sequence
        """
        ...

class ShapeKey(bpy_struct):
    """Shape key in a shape keys data-block"""

    data: bpy_prop_collection[UnknownType]
    """ 

    :type: bpy_prop_collection[UnknownType]
    """

    frame: float
    """ Frame for absolute keys

    :type: float
    """

    interpolation: str
    """ Interpolation type for absolute shape keys

    :type: str
    """

    mute: bool
    """ Mute this shape key

    :type: bool
    """

    name: str
    """ Name of Shape Key

    :type: str
    """

    relative_key: ShapeKey
    """ Shape used as a relative key

    :type: ShapeKey
    """

    slider_max: float
    """ Maximum for slider

    :type: float
    """

    slider_min: float
    """ Minimum for slider

    :type: float
    """

    value: float
    """ Value of shape key at the current frame

    :type: float
    """

    vertex_group: str
    """ Vertex weight group, to blend with basis shape

    :type: str
    """

    def normals_vertex_get(self) -> float:
        """Compute local space vertices' normals for this shape key

        :return: normals
        :rtype: float
        """
        ...

    def normals_polygon_get(self) -> float:
        """Compute local space faces' normals for this shape key

        :return: normals
        :rtype: float
        """
        ...

    def normals_split_get(self) -> float:
        """Compute local space face corners' normals for this shape key

        :return: normals
        :rtype: float
        """
        ...

class ShapeKeyBezierPoint(bpy_struct):
    """Point in a shape key for Bezier curves"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_left: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_right: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ShapeKeyCurvePoint(bpy_struct):
    """Point in a shape key for curves"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tilt: float
    """ 

    :type: float
    """

class ShapeKeyPoint(bpy_struct):
    """Point in a shape key"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class SmokeCollSettings(bpy_struct):
    """Smoke collision settings"""

    collision_type: str
    """ Collision type

    :type: str
    """

class SmokeDomainSettings(bpy_struct):
    """Smoke domain settings"""

    adapt_margin: int
    """ Margin added around fluid to minimize boundary interference

    :type: int
    """

    adapt_threshold: float
    """ Maximum amount of fluid cell can contain before it is considered empty

    :type: float
    """

    additional_res: int
    """ Maximum number of additional cells

    :type: int
    """

    alpha: float
    """ How much density affects smoke motion (higher value results in faster rising smoke)

    :type: float
    """

    amplify: int
    """ Enhance the resolution of smoke by this factor using noise

    :type: int
    """

    beta: float
    """ How much heat affects smoke motion (higher value results in faster rising smoke)

    :type: float
    """

    burning_rate: float
    """ Speed of the burning reaction (use larger values for smaller flame)

    :type: float
    """

    cache_file_format: str
    """ Select the file format to be used for caching

    :type: str
    """

    cell_size: bpy_prop_array[float]
    """ Cell Size

    :type: bpy_prop_array[float]
    """

    collision_extents: str
    """ Select which domain border will be treated as collision object

    :type: str
    """

    collision_group: Group
    """ Limit collisions to this group

    :type: Group
    """

    color_grid: bpy_prop_array[float]
    """ Smoke color grid

    :type: bpy_prop_array[float]
    """

    data_depth: str
    """ Bit depth for writing all scalar (including vector) lower values reduce file size

    :type: str
    """

    density_grid: bpy_prop_array[float]
    """ Smoke density grid

    :type: bpy_prop_array[float]
    """

    dissolve_speed: int
    """ Dissolve Speed

    :type: int
    """

    domain_resolution: bpy_prop_array[int]
    """ Smoke Grid Resolution

    :type: bpy_prop_array[int]
    """

    effector_group: Group
    """ Limit effectors to this group

    :type: Group
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    flame_grid: bpy_prop_array[float]
    """ Smoke flame grid

    :type: bpy_prop_array[float]
    """

    flame_ignition: float
    """ Minimum temperature of flames

    :type: float
    """

    flame_max_temp: float
    """ Maximum temperature of flames

    :type: float
    """

    flame_smoke: float
    """ Amount of smoke created by burning fuel

    :type: float
    """

    flame_smoke_color: bpy_prop_array[float]
    """ Color of smoke emitted from burning fuel

    :type: bpy_prop_array[float]
    """

    flame_vorticity: float
    """ Additional vorticity for the flames

    :type: float
    """

    fluid_group: Group
    """ Limit fluid objects to this group

    :type: Group
    """

    heat_grid: bpy_prop_array[float]
    """ Smoke heat grid

    :type: bpy_prop_array[float]
    """

    highres_sampling: str
    """ Method for sampling the high resolution flow

    :type: str
    """

    noise_type: str
    """ Noise method which is used for creating the high resolution

    :type: str
    """

    openvdb_cache_compress_type: str
    """ Compression method to be used

    :type: str
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    point_cache_compress_type: str
    """ Compression method to be used

    :type: str
    """

    resolution_max: int
    """ Maximal resolution used in the fluid domain

    :type: int
    """

    show_high_resolution: bool
    """ Show high resolution (using amplification)

    :type: bool
    """

    start_point: bpy_prop_array[float]
    """ Start point

    :type: bpy_prop_array[float]
    """

    strength: float
    """ Strength of noise

    :type: float
    """

    time_scale: float
    """ Adjust simulation speed

    :type: float
    """

    use_adaptive_domain: bool
    """ Adapt simulation resolution and size to fluid

    :type: bool
    """

    use_dissolve_smoke: bool
    """ Enable smoke to disappear over time

    :type: bool
    """

    use_dissolve_smoke_log: bool
    """ Using 1/x

    :type: bool
    """

    use_high_resolution: bool
    """ Enable high resolution (using amplification)

    :type: bool
    """

    velocity_grid: bpy_prop_array[float]
    """ Smoke velocity grid

    :type: bpy_prop_array[float]
    """

    vorticity: float
    """ Amount of turbulence/rotation in fluid

    :type: float
    """

class SmokeFlowSettings(bpy_struct):
    """Smoke flow settings"""

    density: float
    """ 

    :type: float
    """

    density_vertex_group: str
    """ Name of vertex group which determines surface emission rate

    :type: str
    """

    fuel_amount: float
    """ 

    :type: float
    """

    noise_texture: Texture
    """ Texture that controls emission strength

    :type: Texture
    """

    particle_size: float
    """ Particle size in simulation cells

    :type: float
    """

    particle_system: ParticleSystem
    """ Particle systems emitted from the object

    :type: ParticleSystem
    """

    smoke_color: bpy_prop_array[float]
    """ Color of smoke

    :type: bpy_prop_array[float]
    """

    smoke_flow_source: str
    """ Change how smoke is emitted

    :type: str
    """

    smoke_flow_type: str
    """ Change how flow affects the simulation

    :type: str
    """

    subframes: int
    """ Number of additional samples to take between frames to improve quality of fast moving flows

    :type: int
    """

    surface_distance: float
    """ Maximum distance from mesh surface to emit smoke

    :type: float
    """

    temperature: float
    """ Temperature difference to ambient temperature

    :type: float
    """

    texture_map_type: str
    """ Texture mapping type

    :type: str
    """

    texture_offset: float
    """ Z-offset of texture mapping

    :type: float
    """

    texture_size: float
    """ Size of texture mapping

    :type: float
    """

    use_absolute: bool
    """ Only allow given density value in emitter area

    :type: bool
    """

    use_initial_velocity: bool
    """ Smoke has some initial velocity when it is emitted

    :type: bool
    """

    use_particle_size: bool
    """ Set particle size in simulation cells or use nearest cell

    :type: bool
    """

    use_texture: bool
    """ Use a texture to control emission strength

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    velocity_factor: float
    """ Multiplier of source velocity passed to smoke

    :type: float
    """

    velocity_normal: float
    """ Amount of normal directional velocity

    :type: float
    """

    velocity_random: float
    """ Amount of random velocity

    :type: float
    """

    volume_density: float
    """ Factor for smoke emitted from inside the mesh volume

    :type: float
    """

class SoftBodySettings(bpy_struct):
    """Soft body simulation settings for an object"""

    aero: int
    """ Make edges 'sail'

    :type: int
    """

    aerodynamics_type: str
    """ Method of calculating aerodynamic interaction

    :type: str
    """

    ball_damp: float
    """ Blending to inelastic collision

    :type: float
    """

    ball_size: float
    """ Absolute ball size or factor if not manually adjusted

    :type: float
    """

    ball_stiff: float
    """ Ball inflating pressure

    :type: float
    """

    bend: float
    """ Bending Stiffness

    :type: float
    """

    choke: int
    """ 'Viscosity' inside collision target

    :type: int
    """

    collision_group: Group
    """ Limit colliders to this Group

    :type: Group
    """

    collision_type: str
    """ Choose Collision Type

    :type: str
    """

    damping: float
    """ Edge spring friction

    :type: float
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    error_threshold: float
    """ The Runge-Kutta ODE solver error limit, low value gives more precision, high values speed

    :type: float
    """

    friction: float
    """ General media friction for point movements

    :type: float
    """

    fuzzy: int
    """ Fuzziness while on collision, high values make collision handling faster but less stable

    :type: int
    """

    goal_default: float
    """ Default Goal (vertex target position) value

    :type: float
    """

    goal_friction: float
    """ Goal (vertex target position) friction

    :type: float
    """

    goal_max: float
    """ Goal maximum, vertex weights are scaled to match this range

    :type: float
    """

    goal_min: float
    """ Goal minimum, vertex weights are scaled to match this range

    :type: float
    """

    goal_spring: float
    """ Goal (vertex target position) spring stiffness

    :type: float
    """

    gravity: float
    """ Apply gravitation to point movement

    :type: float
    """

    location_mass_center: bpy_prop_array[float]
    """ Location of Center of mass

    :type: bpy_prop_array[float]
    """

    mass: float
    """ General Mass value

    :type: float
    """

    plastic: int
    """ Permanent deform

    :type: int
    """

    pull: float
    """ Edge spring stiffness when longer than rest length

    :type: float
    """

    push: float
    """ Edge spring stiffness when shorter than rest length

    :type: float
    """

    rotation_estimate: bpy_prop_array[float]
    """ Estimated rotation matrix

    :type: bpy_prop_array[float]
    """

    scale_estimate: bpy_prop_array[float]
    """ Estimated scale matrix

    :type: bpy_prop_array[float]
    """

    shear: float
    """ Shear Stiffness

    :type: float
    """

    speed: float
    """ Tweak timing for physics to control frequency and speed

    :type: float
    """

    spring_length: int
    """ Alter spring length to shrink/blow up (unit %) 0 to disable

    :type: int
    """

    step_max: int
    """ Maximal # solver steps/frame

    :type: int
    """

    step_min: int
    """ Minimal # solver steps/frame

    :type: int
    """

    use_auto_step: bool
    """ Use velocities for automagic step sizes

    :type: bool
    """

    use_diagnose: bool
    """ Turn on SB diagnose console prints

    :type: bool
    """

    use_edge_collision: bool
    """ Edges collide too

    :type: bool
    """

    use_edges: bool
    """ Use Edges as springs

    :type: bool
    """

    use_estimate_matrix: bool
    """ Estimate matrix... split to COM, ROT, SCALE

    :type: bool
    """

    use_face_collision: bool
    """ Faces collide too, can be very slow

    :type: bool
    """

    use_goal: bool
    """ Define forces for vertices to stick to animated position

    :type: bool
    """

    use_self_collision: bool
    """ Enable naive vertex ball self collision

    :type: bool
    """

    use_stiff_quads: bool
    """ Add diagonal springs on 4-gons

    :type: bool
    """

    vertex_group_goal: str
    """ Control point weight values

    :type: str
    """

    vertex_group_mass: str
    """ Control point mass values

    :type: str
    """

    vertex_group_spring: str
    """ Control point spring strength values

    :type: str
    """

class Space(bpy_struct):
    """Space data for a screen area"""

    show_locked_time: bool
    """ 

    :type: bool
    """

    type: str
    """ Space data type

    :type: str
    """

    def draw_handler_add(self):
        """Undocumented"""
        ...

    def draw_handler_remove(self):
        """Undocumented"""
        ...

class SpaceNodeEditorPath(bpy_prop_collection[NodeTreePath], bpy_struct):
    """Get the node tree path as a string"""

    to_string: str
    """ 

    :type: str
    """

    def clear(self):
        """Reset the node tree path"""
        ...

    def start(self, node_tree: NodeTree | None):
        """Set the root node tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        """
        ...

    def append(self, node_tree: NodeTree | None, node: Node | None = None):
        """Append a node group tree to the path

        :param node_tree: Node Tree, Node tree to append to the node editor path
        :type node_tree: NodeTree | None
        :param node: Node, Group node linking to this node tree
        :type node: Node | None
        """
        ...

    def pop(self):
        """Remove the last node tree from the path"""
        ...

class SpaceUVEditor(bpy_struct):
    """UV editor data for the image editor space"""

    draw_stretch_type: str
    """ Type of stretch to draw

    :type: str
    """

    edge_draw_type: str
    """ Draw type for drawing UV edges

    :type: str
    """

    lock_bounds: bool
    """ Constraint to stay within the image bounds while editing

    :type: bool
    """

    show_faces: bool
    """ Draw faces over the image

    :type: bool
    """

    show_metadata: bool
    """ Draw metadata properties of the image

    :type: bool
    """

    show_modified_edges: bool
    """ Draw edges after modifiers are applied

    :type: bool
    """

    show_normalized_coords: bool
    """ Display UV coordinates from 0.0 to 1.0 rather than in pixels

    :type: bool
    """

    show_other_objects: bool
    """ Draw other selected objects that share the same image

    :type: bool
    """

    show_smooth_edges: bool
    """ Draw UV edges anti-aliased

    :type: bool
    """

    show_stretch: bool
    """ Draw faces colored according to the difference in shape between UVs and their 3D coordinates (blue for low distortion, red for high distortion)

    :type: bool
    """

    show_texpaint: bool
    """ Draw overlay of texture paint uv layer

    :type: bool
    """

    sticky_select_mode: str
    """ Automatically select also UVs sharing the same vertex as the ones being selected

    :type: str
    """

    use_live_unwrap: bool
    """ Continuously unwrap the selected UV island while transforming pinned vertices

    :type: bool
    """

    use_snap_to_pixels: bool
    """ Snap UVs to pixel locations while editing

    :type: bool
    """

class Spline(bpy_struct):
    """Element of a curve, either NURBS, Bezier or Polyline or a character with text objects"""

    bezier_points: SplineBezierPoints
    """ Collection of points for Bezier curves only

    :type: SplineBezierPoints
    """

    character_index: int
    """ Location of this character in the text data (only for text curves)

    :type: int
    """

    hide: bool
    """ Hide this curve in Edit mode

    :type: bool
    """

    material_index: int
    """ 

    :type: int
    """

    order_u: int
    """ NURBS order in the U direction (for splines and surfaces, higher values let points influence a greater area)

    :type: int
    """

    order_v: int
    """ NURBS order in the V direction (for surfaces only, higher values let points influence a greater area)

    :type: int
    """

    point_count_u: int
    """ Total number points for the curve or surface in the U direction

    :type: int
    """

    point_count_v: int
    """ Total number points for the surface on the V direction

    :type: int
    """

    points: SplinePoints
    """ Collection of points that make up this poly or nurbs spline

    :type: SplinePoints
    """

    radius_interpolation: str
    """ The type of radius interpolation for Bezier curves

    :type: str
    """

    resolution_u: int
    """ Curve or Surface subdivisions per segment

    :type: int
    """

    resolution_v: int
    """ Surface subdivisions per segment

    :type: int
    """

    tilt_interpolation: str
    """ The type of tilt interpolation for 3D, Bezier curves

    :type: str
    """

    type: str
    """ The interpolation type for this curve element

    :type: str
    """

    use_bezier_u: bool
    """ Make this nurbs curve or surface act like a Bezier spline in the U direction (Order U must be 3 or 4, Cyclic U must be disabled)

    :type: bool
    """

    use_bezier_v: bool
    """ Make this nurbs surface act like a Bezier spline in the V direction (Order V must be 3 or 4, Cyclic V must be disabled)

    :type: bool
    """

    use_cyclic_u: bool
    """ Make this curve or surface a closed loop in the U direction

    :type: bool
    """

    use_cyclic_v: bool
    """ Make this surface a closed loop in the V direction

    :type: bool
    """

    use_endpoint_u: bool
    """ Make this nurbs curve or surface meet the endpoints in the U direction (Cyclic U must be disabled)

    :type: bool
    """

    use_endpoint_v: bool
    """ Make this nurbs surface meet the endpoints in the V direction (Cyclic V must be disabled)

    :type: bool
    """

    use_smooth: bool
    """ Smooth the normals of the surface or beveled curve

    :type: bool
    """

class SplineBezierPoints(bpy_prop_collection[BezierSplinePoint], bpy_struct):
    """Collection of spline Bezier points"""

    def add(self, count: typing.Any | None = 1):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: typing.Any | None
        """
        ...

class SplinePoint(bpy_struct):
    """Spline point without handles"""

    co: bpy_prop_array[float]
    """ Point coordinates

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    select: bool
    """ Selection status

    :type: bool
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    weight: float
    """ NURBS weight

    :type: float
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

class SplinePoints(bpy_prop_collection[SplinePoint], bpy_struct):
    """Collection of spline points"""

    def add(self, count: typing.Any | None = 1):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: typing.Any | None
        """
        ...

class Stereo3dDisplay(bpy_struct):
    """Settings for stereo 3D display"""

    anaglyph_type: str
    """ 

    :type: str
    """

    display_mode: str
    """ 

    :type: str
    """

    interlace_type: str
    """ 

    :type: str
    """

    use_interlace_swap: bool
    """ Swap left and right stereo channels

    :type: bool
    """

    use_sidebyside_crosseyed: bool
    """ Right eye should see left image and vice-versa

    :type: bool
    """

class Stereo3dFormat(bpy_struct):
    """Settings for stereo output"""

    anaglyph_type: str
    """ 

    :type: str
    """

    display_mode: str
    """ 

    :type: str
    """

    interlace_type: str
    """ 

    :type: str
    """

    use_interlace_swap: bool
    """ Swap left and right stereo channels

    :type: bool
    """

    use_sidebyside_crosseyed: bool
    """ Right eye should see left image and vice-versa

    :type: bool
    """

    use_squeezed_frame: bool
    """ Combine both views in a squeezed image

    :type: bool
    """

class Struct(bpy_struct):
    """RNA structure definition"""

    base: Struct
    """ Struct definition this is derived from

    :type: Struct
    """

    description: str
    """ Description of the Struct's purpose

    :type: str
    """

    functions: bpy_prop_collection[Function]
    """ 

    :type: bpy_prop_collection[Function]
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    name: str
    """ Human readable name

    :type: str
    """

    name_property: StringProperty
    """ Property that gives the name of the struct

    :type: StringProperty
    """

    nested: Struct
    """ Struct in which this struct is always nested, and to which it logically belongs

    :type: Struct
    """

    properties: bpy_prop_collection[Property]
    """ Properties in the struct

    :type: bpy_prop_collection[Property]
    """

    translation_context: str
    """ Translation context of the struct's name

    :type: str
    """

class TessfaceUVTextures(bpy_prop_collection[MeshTextureFaceLayer], bpy_struct):
    """Collection of UV maps for tessellated faces"""

    active: MeshTextureFaceLayer
    """ Active UV Map

    :type: MeshTextureFaceLayer
    """

    active_index: int | None
    """ Active UV Map index

    :type: int | None
    """

    def new(self, name: str | typing.Any = "UVMap") -> MeshTextureFaceLayer:
        """Add a UV tessface-texture layer to Mesh (only for meshes with no polygons)

        :param name: UV map name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshTextureFaceLayer
        """
        ...

class TexMapping(bpy_struct):
    """Texture coordinate mapping settings"""

    mapping: str
    """ 

    :type: str
    """

    mapping_x: str
    """ 

    :type: str
    """

    mapping_y: str
    """ 

    :type: str
    """

    mapping_z: str
    """ 

    :type: str
    """

    max: bpy_prop_array[float]
    """ Maximum value for clipping

    :type: bpy_prop_array[float]
    """

    min: bpy_prop_array[float]
    """ Minimum value for clipping

    :type: bpy_prop_array[float]
    """

    rotation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    scale: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    translation: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    use_max: bool
    """ Whether to use maximum clipping value

    :type: bool
    """

    use_min: bool
    """ Whether to use minimum clipping value

    :type: bool
    """

    vector_type: str
    """ Type of vector that the mapping transforms

    :type: str
    """

class TexPaintSlot(bpy_struct):
    """Slot that contains information about texture painting"""

    index: int
    """ Index of MTex slot in the material

    :type: int
    """

    uv_layer: str
    """ Name of UV map

    :type: str
    """

class TextBox(bpy_struct):
    """Text bounding box for layout"""

    height: float
    """ 

    :type: float
    """

    width: float
    """ 

    :type: float
    """

    x: float
    """ 

    :type: float
    """

    y: float
    """ 

    :type: float
    """

class TextCharacterFormat(bpy_struct):
    """Text character formatting settings"""

    material_index: int
    """ 

    :type: int
    """

    use_bold: bool
    """ 

    :type: bool
    """

    use_italic: bool
    """ 

    :type: bool
    """

    use_small_caps: bool
    """ 

    :type: bool
    """

    use_underline: bool
    """ 

    :type: bool
    """

class TextLine(bpy_struct):
    """Line of text in a Text data-block"""

    body: str
    """ Text in the line

    :type: str
    """

class TextureSlot(bpy_struct):
    """Texture slot defining the mapping and influence of a texture"""

    blend_type: str
    """ Mode used to apply the texture

    :type: str
    """

    color: bpy_prop_array[float]
    """ Default color for textures that don't return RGB or when RGB to intensity is enabled

    :type: bpy_prop_array[float]
    """

    default_value: float
    """ Value to use for Ref, Spec, Amb, Emit, Alpha, RayMir, TransLu and Hard

    :type: float
    """

    invert: bool
    """ Invert the values of the texture to reverse its effect

    :type: bool
    """

    name: str
    """ Texture slot name

    :type: str
    """

    offset: bpy_prop_array[float]
    """ Fine tune of the texture mapping X, Y and Z locations

    :type: bpy_prop_array[float]
    """

    output_node: str
    """ Which output node to use, for node-based textures

    :type: str
    """

    scale: bpy_prop_array[float]
    """ Set scaling for the texture's X, Y and Z sizes

    :type: bpy_prop_array[float]
    """

    texture: Texture
    """ Texture data-block used by this texture slot

    :type: Texture
    """

    use_rgb_to_intensity: bool
    """ Convert texture RGB values to intensity (gray) values

    :type: bool
    """

    use_stencil: bool
    """ Use this texture as a blending value on the next texture

    :type: bool
    """

class Theme(bpy_struct):
    """Theme settings defining draw style and colors in the user interface"""

    bone_color_sets: bpy_prop_collection[ThemeBoneColorSet]
    """ 

    :type: bpy_prop_collection[ThemeBoneColorSet]
    """

    clip_editor: ThemeClipEditor
    """ 

    :type: ThemeClipEditor
    """

    console: ThemeConsole
    """ 

    :type: ThemeConsole
    """

    dopesheet_editor: ThemeDopeSheet
    """ 

    :type: ThemeDopeSheet
    """

    file_browser: ThemeFileBrowser
    """ 

    :type: ThemeFileBrowser
    """

    graph_editor: ThemeGraphEditor
    """ 

    :type: ThemeGraphEditor
    """

    image_editor: ThemeImageEditor
    """ 

    :type: ThemeImageEditor
    """

    info: ThemeInfo
    """ 

    :type: ThemeInfo
    """

    logic_editor: ThemeLogicEditor
    """ 

    :type: ThemeLogicEditor
    """

    name: str
    """ Name of the theme

    :type: str
    """

    nla_editor: ThemeNLAEditor
    """ 

    :type: ThemeNLAEditor
    """

    node_editor: ThemeNodeEditor
    """ 

    :type: ThemeNodeEditor
    """

    outliner: ThemeOutliner
    """ 

    :type: ThemeOutliner
    """

    properties: ThemeProperties
    """ 

    :type: ThemeProperties
    """

    sequence_editor: ThemeSequenceEditor
    """ 

    :type: ThemeSequenceEditor
    """

    text_editor: ThemeTextEditor
    """ 

    :type: ThemeTextEditor
    """

    theme_area: str
    """ 

    :type: str
    """

    timeline: ThemeTimeline
    """ 

    :type: ThemeTimeline
    """

    user_interface: ThemeUserInterface
    """ 

    :type: ThemeUserInterface
    """

    user_preferences: ThemeUserPreferences
    """ 

    :type: ThemeUserPreferences
    """

    view_3d: ThemeView3D
    """ 

    :type: ThemeView3D
    """

class ThemeBoneColorSet(bpy_struct):
    """Theme settings for bone color sets"""

    active: bpy_prop_array[float] | None
    """ Color used for active bones

    :type: bpy_prop_array[float] | None
    """

    normal: bpy_prop_array[float]
    """ Color used for the surface of bones

    :type: bpy_prop_array[float]
    """

    select: bpy_prop_array[float]
    """ Color used for selected bones

    :type: bpy_prop_array[float]
    """

    show_colored_constraints: bool
    """ Allow the use of colors indicating constraints/keyed status

    :type: bool
    """

class ThemeClipEditor(bpy_struct):
    """Theme settings for the Movie Clip Editor"""

    active_marker: bpy_prop_array[float] | None
    """ Color of active marker

    :type: bpy_prop_array[float] | None
    """

    disabled_marker: bpy_prop_array[float]
    """ Color of disabled marker

    :type: bpy_prop_array[float]
    """

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_size: int
    """ 

    :type: int
    """

    handle_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_auto_clamped: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_auto_clamped: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex_size: int
    """ 

    :type: int
    """

    locked_marker: bpy_prop_array[float]
    """ Color of locked marker

    :type: bpy_prop_array[float]
    """

    marker: bpy_prop_array[float]
    """ Color of marker

    :type: bpy_prop_array[float]
    """

    marker_outline: bpy_prop_array[float]
    """ Color of marker's outline

    :type: bpy_prop_array[float]
    """

    path_after: bpy_prop_array[float]
    """ Color of path after current frame

    :type: bpy_prop_array[float]
    """

    path_before: bpy_prop_array[float]
    """ Color of path before current frame

    :type: bpy_prop_array[float]
    """

    selected_marker: bpy_prop_array[float]
    """ Color of selected marker

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    space_list: ThemeSpaceListGeneric
    """ Settings for space list

    :type: ThemeSpaceListGeneric
    """

    strips: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    strips_selected: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeConsole(bpy_struct):
    """Theme settings for the Console"""

    cursor: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    line_error: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    line_info: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    line_input: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    line_output: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

class ThemeDopeSheet(bpy_struct):
    """Theme settings for the Dope Sheet"""

    active_channels_group: bpy_prop_array[float] | None
    """ 

    :type: bpy_prop_array[float] | None
    """

    channel_group: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    channels: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    channels_selected: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    dopesheet_channel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    dopesheet_subchannel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    grid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    keyframe: bpy_prop_array[float]
    """ Color of Keyframe

    :type: bpy_prop_array[float]
    """

    keyframe_border: bpy_prop_array[float]
    """ Color of keyframe border

    :type: bpy_prop_array[float]
    """

    keyframe_border_selected: bpy_prop_array[float]
    """ Color of selected keyframe border

    :type: bpy_prop_array[float]
    """

    keyframe_breakdown: bpy_prop_array[float]
    """ Color of breakdown keyframe

    :type: bpy_prop_array[float]
    """

    keyframe_breakdown_selected: bpy_prop_array[float]
    """ Color of selected breakdown keyframe

    :type: bpy_prop_array[float]
    """

    keyframe_extreme: bpy_prop_array[float]
    """ Color of extreme keyframe

    :type: bpy_prop_array[float]
    """

    keyframe_extreme_selected: bpy_prop_array[float]
    """ Color of selected extreme keyframe

    :type: bpy_prop_array[float]
    """

    keyframe_jitter: bpy_prop_array[float]
    """ Color of jitter keyframe

    :type: bpy_prop_array[float]
    """

    keyframe_jitter_selected: bpy_prop_array[float]
    """ Color of selected jitter keyframe

    :type: bpy_prop_array[float]
    """

    keyframe_scale_factor: float
    """ Scale factor for adjusting the height of keyframes

    :type: float
    """

    keyframe_selected: bpy_prop_array[float]
    """ Color of selected keyframe

    :type: bpy_prop_array[float]
    """

    long_key: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    long_key_selected: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    space_list: ThemeSpaceListGeneric
    """ Settings for space list

    :type: ThemeSpaceListGeneric
    """

    summary: bpy_prop_array[float]
    """ Color of summary channel

    :type: bpy_prop_array[float]
    """

    value_sliders: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    view_sliders: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeFileBrowser(bpy_struct):
    """Theme settings for the File Browser"""

    selected_file: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

class ThemeFontStyle(bpy_struct):
    """Theme settings for Font"""

    font_kerning_style: str
    """ Which style to use for font kerning

    :type: str
    """

    points: int
    """ 

    :type: int
    """

    shadow: int
    """ Shadow size (0, 3 and 5 supported)

    :type: int
    """

    shadow_alpha: float
    """ 

    :type: float
    """

    shadow_offset_x: int
    """ Shadow offset in pixels

    :type: int
    """

    shadow_offset_y: int
    """ Shadow offset in pixels

    :type: int
    """

    shadow_value: float
    """ Shadow color in gray value

    :type: float
    """

class ThemeGradientColors(bpy_struct):
    """Theme settings for background colors and gradient"""

    gradient: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    high_gradient: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    show_grad: bool
    """ Do a gradient for the background of the viewport working area

    :type: bool
    """

class ThemeGraphEditor(bpy_struct):
    """Theme settings for the graph editor"""

    active_channels_group: bpy_prop_array[float] | None
    """ 

    :type: bpy_prop_array[float] | None
    """

    channel_group: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    channels_region: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    dopesheet_channel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    dopesheet_subchannel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    grid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_auto_clamped: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_auto_clamped: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_vect: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vect: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex_size: int
    """ 

    :type: int
    """

    lastsel_point: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    space_list: ThemeSpaceListGeneric
    """ Settings for space list

    :type: ThemeSpaceListGeneric
    """

    vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex_size: int
    """ 

    :type: int
    """

    vertex_unreferenced: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    window_sliders: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeImageEditor(bpy_struct):
    """Theme settings for the Image Editor"""

    edge_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    editmesh_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    face: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    face_dot: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    face_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    facedot_size: int
    """ 

    :type: int
    """

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    freestyle_face_mark: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_size: int
    """ 

    :type: int
    """

    handle_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_auto_clamped: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_auto_clamped: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vertex_size: int
    """ 

    :type: int
    """

    metadatabg: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    metadatatext: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    paint_curve_handle: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    paint_curve_pivot: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    preview_stitch_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    preview_stitch_edge: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    preview_stitch_face: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    preview_stitch_stitchable: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    preview_stitch_unstitchable: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    preview_stitch_vert: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    scope_back: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    uv_others: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv_shadow: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex_size: int
    """ 

    :type: int
    """

    vertex_unreferenced: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    wire_edit: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeInfo(bpy_struct):
    """Theme settings for Info"""

    info_debug: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_debug_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_error: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_error_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_info: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_info_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_selected: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_selected_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_warning: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    info_warning_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

class ThemeLogicEditor(bpy_struct):
    """Theme settings for the Logic Editor"""

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

class ThemeNLAEditor(bpy_struct):
    """Theme settings for the NLA Editor"""

    active_action: bpy_prop_array[float] | None
    """ Animation data block has active action

    :type: bpy_prop_array[float] | None
    """

    active_action_unset: bpy_prop_array[float] | None
    """ Animation data block doesn't have active action

    :type: bpy_prop_array[float] | None
    """

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    grid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    keyframe_border: bpy_prop_array[float]
    """ Color of keyframe border

    :type: bpy_prop_array[float]
    """

    keyframe_border_selected: bpy_prop_array[float]
    """ Color of selected keyframe border

    :type: bpy_prop_array[float]
    """

    meta_strips: bpy_prop_array[float]
    """ Meta Strip - Unselected (for grouping related strips)

    :type: bpy_prop_array[float]
    """

    meta_strips_selected: bpy_prop_array[float]
    """ Meta Strip - Selected (for grouping related strips)

    :type: bpy_prop_array[float]
    """

    sound_strips: bpy_prop_array[float]
    """ Sound Strip - Unselected (for timing speaker sounds)

    :type: bpy_prop_array[float]
    """

    sound_strips_selected: bpy_prop_array[float]
    """ Sound Strip - Selected (for timing speaker sounds)

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    space_list: ThemeSpaceListGeneric
    """ Settings for space list

    :type: ThemeSpaceListGeneric
    """

    strips: bpy_prop_array[float]
    """ Action-Clip Strip - Unselected

    :type: bpy_prop_array[float]
    """

    strips_selected: bpy_prop_array[float]
    """ Action-Clip Strip - Selected

    :type: bpy_prop_array[float]
    """

    transition_strips: bpy_prop_array[float]
    """ Transition Strip - Unselected

    :type: bpy_prop_array[float]
    """

    transition_strips_selected: bpy_prop_array[float]
    """ Transition Strip - Selected

    :type: bpy_prop_array[float]
    """

    tweak: bpy_prop_array[float]
    """ Color for strip/action being 'tweaked' or edited

    :type: bpy_prop_array[float]
    """

    tweak_duplicate: bpy_prop_array[float]
    """ Warning/error indicator color for strips referencing the strip being tweaked

    :type: bpy_prop_array[float]
    """

    view_sliders: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeNodeEditor(bpy_struct):
    """Theme settings for the Node Editor"""

    color_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    converter_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    distor_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    filter_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    frame_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_size: int
    """ 

    :type: int
    """

    group_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    group_socket_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    input_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    layout_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    matte_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    node_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    node_backdrop: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    node_selected: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    noodle_curving: int
    """ Curving of the noodle

    :type: int
    """

    output_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    pattern_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    script_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    selected_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    shader_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    space_list: ThemeSpaceListGeneric
    """ Settings for space list

    :type: ThemeSpaceListGeneric
    """

    texture_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vector_node: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    wire: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    wire_inner: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    wire_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeOutliner(bpy_struct):
    """Theme settings for the Outliner"""

    match: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    selected_highlight: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

class ThemePanelColors(bpy_struct):
    """Theme settings for panel colors"""

    back: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    header: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    show_back: bool
    """ 

    :type: bool
    """

    show_header: bool
    """ 

    :type: bool
    """

class ThemeProperties(bpy_struct):
    """Theme settings for the Properties"""

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

class ThemeSequenceEditor(bpy_struct):
    """Theme settings for the Sequence Editor"""

    audio_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    draw_action: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    effect_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_size: int
    """ 

    :type: int
    """

    grid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    image_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    keyframe: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    meta_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    metadatabg: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    metadatatext: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    movie_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    movieclip_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    preview_back: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    scene_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    text_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    transition_strip: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    window_sliders: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeSpaceGeneric(bpy_struct):
    back: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    button: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    button_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    button_text_hi: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    button_title: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    header: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    header_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    header_text_hi: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    panelcolors: ThemePanelColors
    """ 

    :type: ThemePanelColors
    """

    tab_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tab_back: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tab_inactive: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tab_outline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    text_hi: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    title: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeSpaceGradient(bpy_struct):
    button: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    button_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    button_text_hi: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    button_title: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gradients: ThemeGradientColors
    """ 

    :type: ThemeGradientColors
    """

    header: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    header_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    header_text_hi: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    panelcolors: ThemePanelColors
    """ 

    :type: ThemePanelColors
    """

    tab_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tab_back: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tab_inactive: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    tab_outline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    text_hi: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    title: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeSpaceListGeneric(bpy_struct):
    list: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    list_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    list_text_hi: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    list_title: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeStyle(bpy_struct):
    """Theme settings for style sets"""

    panel_title: ThemeFontStyle
    """ 

    :type: ThemeFontStyle
    """

    widget: ThemeFontStyle
    """ 

    :type: ThemeFontStyle
    """

    widget_label: ThemeFontStyle
    """ 

    :type: ThemeFontStyle
    """

class ThemeTextEditor(bpy_struct):
    """Theme settings for the Text Editor"""

    cursor: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    line_numbers_background: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    selected_text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    syntax_builtin: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    syntax_comment: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    syntax_numbers: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    syntax_preprocessor: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    syntax_reserved: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    syntax_special: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    syntax_string: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    syntax_symbols: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeTimeline(bpy_struct):
    """Theme settings for the Timeline"""

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    grid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

    time_grease_pencil: bpy_prop_array[float]
    """ Color of Grease Pencil keyframes

    :type: bpy_prop_array[float]
    """

    time_keyframe: bpy_prop_array[float]
    """ Base color for keyframe indicator lines

    :type: bpy_prop_array[float]
    """

class ThemeUserInterface(bpy_struct):
    """Theme settings for user interface elements"""

    axis_x: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    axis_y: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    axis_z: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    icon_alpha: float
    """ Transparency of icons in the interface, to reduce contrast

    :type: float
    """

    icon_file: str
    """ 

    :type: str
    """

    menu_shadow_fac: float
    """ Blending factor for menu shadows

    :type: float
    """

    menu_shadow_width: int
    """ Width of menu shadows, set to zero to disable

    :type: int
    """

    wcol_box: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_list_item: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_menu: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_menu_back: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_menu_item: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_num: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_numslider: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_option: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_pie_menu: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_progress: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_pulldown: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_radio: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_regular: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_scroll: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_state: ThemeWidgetStateColors
    """ 

    :type: ThemeWidgetStateColors
    """

    wcol_text: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_toggle: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_tool: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    wcol_tooltip: ThemeWidgetColors
    """ 

    :type: ThemeWidgetColors
    """

    widget_emboss: bpy_prop_array[float]
    """ Color of the 1px shadow line underlying widgets

    :type: bpy_prop_array[float]
    """

class ThemeUserPreferences(bpy_struct):
    """Theme settings for the User Preferences"""

    space: ThemeSpaceGeneric
    """ Settings for space

    :type: ThemeSpaceGeneric
    """

class ThemeView3D(bpy_struct):
    """Theme settings for the 3D View"""

    act_spline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    bone_pose: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    bone_pose_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    bone_solid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    bundle_solid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    camera: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    camera_path: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    clipping_border_3d: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    edge_crease: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    edge_facesel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    edge_seam: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    edge_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    edge_sharp: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    editmesh_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    empty: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    extra_edge_angle: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    extra_edge_len: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    extra_face_angle: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    extra_face_area: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    face: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    face_dot: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    face_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    facedot_size: int
    """ 

    :type: int
    """

    frame_current: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    freestyle_edge_mark: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    freestyle_face_mark: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    gp_vertex_size: int
    """ 

    :type: int
    """

    grid: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_align: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_auto: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_free: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_sel_vect: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    handle_vect: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    lamp: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    lastsel_point: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    normal: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    nurb_sel_uline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    nurb_sel_vline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    nurb_uline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    nurb_vline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    object_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    object_grouped: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    object_grouped_active: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    object_selected: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    outline_width: int
    """ 

    :type: int
    """

    paint_curve_handle: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    paint_curve_pivot: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    skin_root: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    space: ThemeSpaceGradient
    """ Settings for space

    :type: ThemeSpaceGradient
    """

    speaker: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    split_normal: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    text_grease_pencil: bpy_prop_array[float]
    """ Color for indicating Grease Pencil keyframes

    :type: bpy_prop_array[float]
    """

    text_keyframe: bpy_prop_array[float]
    """ Color for indicating Object keyframes

    :type: bpy_prop_array[float]
    """

    transform: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex_normal: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex_select: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    vertex_size: int
    """ 

    :type: int
    """

    vertex_unreferenced: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    view_overlay: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    wire: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    wire_edit: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeWidgetColors(bpy_struct):
    """Theme settings for widget color sets"""

    inner: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    inner_sel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    item: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    outline: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    shadedown: int
    """ 

    :type: int
    """

    shadetop: int
    """ 

    :type: int
    """

    show_shaded: bool
    """ 

    :type: bool
    """

    text: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    text_sel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class ThemeWidgetStateColors(bpy_struct):
    """Theme settings for widget state colors"""

    blend: float
    """ 

    :type: float
    """

    inner_anim: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    inner_anim_sel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    inner_driven: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    inner_driven_sel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    inner_key: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    inner_key_sel: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class TimelineMarker(bpy_struct):
    """Marker for noting points in the timeline"""

    camera: Object
    """ Camera this timeline sets to active

    :type: Object
    """

    frame: int
    """ The frame on which the timeline marker appears

    :type: int
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ Marker selection state

    :type: bool
    """

class TimelineMarkers(bpy_prop_collection[TimelineMarker], bpy_struct):
    """Collection of timeline markers"""

    def new(
        self, name: str | typing.Any, frame: typing.Any | None = 1
    ) -> TimelineMarker:
        """Add a keyframe to the curve

        :param name: New name for the marker (not unique)
        :type name: str | typing.Any
        :param frame: The frame for the new marker
        :type frame: typing.Any | None
        :return: Newly created timeline marker
        :rtype: TimelineMarker
        """
        ...

    def remove(self, marker: TimelineMarker):
        """Remove a timeline marker

        :param marker: Timeline marker to remove
        :type marker: TimelineMarker
        """
        ...

    def clear(self):
        """Remove all timeline markers"""
        ...

class Timer(bpy_struct):
    """Window event timer"""

    time_delta: float
    """ Time since last step in seconds

    :type: float
    """

    time_duration: float
    """ Time since last step in seconds

    :type: float
    """

    time_step: float
    """ 

    :type: float
    """

class ToolSettings(bpy_struct):
    auto_keying_mode: str
    """ Mode of automatic keyframe insertion for Objects and Bones

    :type: str
    """

    curve_paint_settings: CurvePaintSettings
    """ 

    :type: CurvePaintSettings
    """

    double_threshold: float
    """ Limit for removing duplicates and 'Auto Merge'

    :type: float
    """

    edge_path_live_unwrap: bool
    """ Changing edges seam re-calculates UV unwrap

    :type: bool
    """

    edge_path_mode: str
    """ The edge flag to tag when selecting the shortest path

    :type: str
    """

    etch_adaptive_limit: float
    """ Correlation threshold for number of bones in the subdivided stroke

    :type: float
    """

    etch_convert_mode: str
    """ Method used to convert stroke to bones

    :type: str
    """

    etch_length_limit: float
    """ Maximum length of the subdivided bones

    :type: float
    """

    etch_number: str
    """ Text to replace &N with (e.g. 'Finger.&N' -> 'Finger.1' or 'Finger.One')

    :type: str
    """

    etch_roll_mode: str
    """ Method used to adjust the roll of bones when retargeting

    :type: str
    """

    etch_side: str
    """ Text to replace &S with (e.g. 'Arm.&S' -> 'Arm.R' or 'Arm.Right')

    :type: str
    """

    etch_subdivision_number: int
    """ Number of bones in the subdivided stroke

    :type: int
    """

    etch_template: Object
    """ Template armature that will be retargeted to the stroke

    :type: Object
    """

    gpencil_brushes: GreasePencilBrushes
    """ Grease Pencil drawing brushes

    :type: GreasePencilBrushes
    """

    gpencil_sculpt: GPencilSculptSettings
    """ 

    :type: GPencilSculptSettings
    """

    gpencil_stroke_placement_image_editor: str
    """ 

    :type: str
    """

    gpencil_stroke_placement_sequencer_preview: str
    """ 

    :type: str
    """

    gpencil_stroke_placement_view2d: str
    """ 

    :type: str
    """

    gpencil_stroke_placement_view3d: str
    """ 

    :type: str
    """

    grease_pencil_source: str
    """ Datablock where active Grease Pencil data is found from

    :type: str
    """

    image_paint: ImagePaint
    """ 

    :type: ImagePaint
    """

    keyframe_type: str
    """ Type of keyframes to create when inserting keyframes

    :type: str
    """

    lock_markers: bool
    """ Prevent marker editing

    :type: bool
    """

    mesh_select_mode: list[bool]
    """ Which mesh elements selection works on

    :type: list[bool]
    """

    normal_size: float
    """ Display size for normals in the 3D view

    :type: float
    """

    particle_edit: ParticleEdit
    """ 

    :type: ParticleEdit
    """

    proportional_edit: str
    """ Proportional Editing mode, allows transforms with distance fall-off

    :type: str
    """

    proportional_edit_falloff: str
    """ Falloff type for proportional editing mode

    :type: str
    """

    proportional_size: float
    """ Display size for proportional editing circle

    :type: float
    """

    sculpt: Sculpt
    """ 

    :type: Sculpt
    """

    show_uv_local_view: bool
    """ Draw only faces with the currently displayed image assigned

    :type: bool
    """

    snap_element: str
    """ Type of element to snap to

    :type: str
    """

    snap_node_element: str
    """ Type of element to snap to

    :type: str
    """

    snap_target: str
    """ Which part to snap onto the target

    :type: str
    """

    snap_uv_element: str
    """ Type of element to snap to

    :type: str
    """

    statvis: MeshStatVis
    """ 

    :type: MeshStatVis
    """

    unified_paint_settings: UnifiedPaintSettings
    """ 

    :type: UnifiedPaintSettings
    """

    use_auto_normalize: bool
    """ Ensure all bone-deforming vertex groups add up to 1.0 while weight painting

    :type: bool
    """

    use_bone_sketching: bool
    """ Use sketching to create and edit bones

    :type: bool
    """

    use_etch_autoname: bool
    """ Automatically generate values to replace &N and &S suffix placeholders in template names

    :type: bool
    """

    use_etch_overdraw: bool
    """ Adjust strokes by drawing near them

    :type: bool
    """

    use_etch_quick: bool
    """ Automatically convert and delete on stroke end

    :type: bool
    """

    use_gpencil_additive_drawing: bool
    """ When creating new frames, the strokes from the previous/active frame are included as the basis for the new one

    :type: bool
    """

    use_gpencil_continuous_drawing: bool
    """ Allow drawing multiple strokes at a time with Grease Pencil

    :type: bool
    """

    use_gpencil_draw_onback: bool
    """ When draw new strokes, the new stroke is drawn below of all strokes in the layer

    :type: bool
    """

    use_gpencil_stroke_endpoints: bool
    """ Only use the first and last parts of the stroke for snapping

    :type: bool
    """

    use_keyframe_insert_auto: bool
    """ Automatic keyframe insertion for Objects and Bones

    :type: bool
    """

    use_keyframe_insert_keyingset: bool
    """ Automatic keyframe insertion using active Keying Set only

    :type: bool
    """

    use_mesh_automerge: bool
    """ Automatically merge vertices moved to the same location

    :type: bool
    """

    use_multipaint: bool
    """ Paint across the weights of all selected bones, maintaining their relative influence

    :type: bool
    """

    use_proportional_action: bool
    """ Proportional editing in action editor

    :type: bool
    """

    use_proportional_edit_mask: bool
    """ Proportional editing mask mode

    :type: bool
    """

    use_proportional_edit_objects: bool
    """ Proportional editing object mode

    :type: bool
    """

    use_proportional_fcurve: bool
    """ Proportional editing in FCurve editor

    :type: bool
    """

    use_record_with_nla: bool
    """ Add a new NLA Track + Strip for every loop/pass made over the animation to allow non-destructive tweaking

    :type: bool
    """

    use_snap: bool
    """ Snap during transform

    :type: bool
    """

    use_snap_align_rotation: bool
    """ Align rotation with the snapping target

    :type: bool
    """

    use_snap_grid_absolute: bool
    """ Absolute grid alignment while translating (based on the pivot center)

    :type: bool
    """

    use_snap_peel_object: bool
    """ Consider objects as whole when finding volume center

    :type: bool
    """

    use_snap_project: bool
    """ Project individual elements on the surface of other objects

    :type: bool
    """

    use_snap_self: bool
    """ Snap onto itself (editmode)

    :type: bool
    """

    use_uv_sculpt: bool
    """ Enable brush for UV sculpting

    :type: bool
    """

    use_uv_select_sync: bool
    """ Keep UV and edit mode mesh selection in sync

    :type: bool
    """

    uv_relax_method: str
    """ Algorithm used for UV relaxation

    :type: str
    """

    uv_sculpt: UvSculpt
    """ 

    :type: UvSculpt
    """

    uv_sculpt_all_islands: bool
    """ Brush operates on all islands

    :type: bool
    """

    uv_sculpt_lock_borders: bool
    """ Disable editing of boundary edges

    :type: bool
    """

    uv_sculpt_tool: str
    """ Select Tools for the UV sculpt brushes

    :type: str
    """

    uv_select_mode: str
    """ UV selection and display mode

    :type: str
    """

    vertex_group_subset: str
    """ Filter Vertex groups for Display

    :type: str
    """

    vertex_group_user: str
    """ Display unweighted vertices

    :type: str
    """

    vertex_group_weight: float
    """ Weight to assign in vertex groups

    :type: float
    """

    vertex_paint: VertexPaint
    """ 

    :type: VertexPaint
    """

    weight_paint: VertexPaint
    """ 

    :type: VertexPaint
    """

class TransformOrientation(bpy_struct):
    matrix: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    name: str
    """ Name of the custom transform orientation

    :type: str
    """

class UILayout(bpy_struct):
    """User interface layout in a panel or header"""

    active: bool | None
    """ 

    :type: bool | None
    """

    alert: bool
    """ 

    :type: bool
    """

    alignment: str
    """ 

    :type: str
    """

    enabled: bool
    """ When false, this (sub)layout is grayed out

    :type: bool
    """

    operator_context: str
    """ 

    :type: str
    """

    scale_x: float
    """ Scale factor along the X for items in this (sub)layout

    :type: float
    """

    scale_y: float
    """ Scale factor along the Y for items in this (sub)layout

    :type: float
    """

    def row(self, align: bool | typing.Any | None = False) -> UILayout:
        """Sub-layout. Items placed in this sublayout are placed next to each other in a row

        :param align: Align buttons to each other
        :type align: bool | typing.Any | None
        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    def column(self, align: bool | typing.Any | None = False) -> UILayout:
        """Sub-layout. Items placed in this sublayout are placed under each other in a column

        :param align: Align buttons to each other
        :type align: bool | typing.Any | None
        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    def column_flow(
        self, columns: typing.Any | None = 0, align: bool | typing.Any | None = False
    ) -> UILayout:
        """column_flow

        :param columns: Number of columns, 0 is automatic
        :type columns: typing.Any | None
        :param align: Align buttons to each other
        :type align: bool | typing.Any | None
        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    def box(self) -> UILayout:
        """Sublayout (items placed in this sublayout are placed under each other in a column and are surrounded by a box)

        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    def split(
        self,
        percentage: typing.Any | None = 0.0,
        align: bool | typing.Any | None = False,
    ) -> UILayout:
        """split

        :param percentage: Percentage, Percentage of width to split at
        :type percentage: typing.Any | None
        :param align: Align buttons to each other
        :type align: bool | typing.Any | None
        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    def menu_pie(self) -> UILayout:
        """Sublayout. Items placed in this sublayout are placed in a radial fashion around the menu center)

        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    @classmethod
    def icon(cls, data: typing.Any) -> int:
        """Return the custom icon for this data, use it e.g. to get materials or texture icons

        :param data: Data from which to take the icon
        :type data: typing.Any
        :return: Icon identifier
        :rtype: int
        """
        ...

    @classmethod
    def enum_item_name(
        cls, data: typing.Any, property: str | typing.Any, identifier: str | typing.Any
    ) -> str | typing.Any:
        """Return the UI name for this enum item

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param identifier: Identifier of the enum item
        :type identifier: str | typing.Any
        :return: UI name of the enum item
        :rtype: str | typing.Any
        """
        ...

    @classmethod
    def enum_item_description(
        cls, data: typing.Any, property: str | typing.Any, identifier: str | typing.Any
    ) -> str | typing.Any:
        """Return the UI description for this enum item

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param identifier: Identifier of the enum item
        :type identifier: str | typing.Any
        :return: UI description of the enum item
        :rtype: str | typing.Any
        """
        ...

    @classmethod
    def enum_item_icon(
        cls, data: typing.Any, property: str | typing.Any, identifier: str | typing.Any
    ) -> int:
        """Return the icon for this enum item

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param identifier: Identifier of the enum item
        :type identifier: str | typing.Any
        :return: Icon identifier
        :rtype: int
        """
        ...

    def prop(
        self,
        data: typing.Any,
        property: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
        expand: bool | typing.Any | None = False,
        slider: bool | typing.Any | None = False,
        toggle: bool | typing.Any | None = False,
        icon_only: bool | typing.Any | None = False,
        event: bool | typing.Any | None = False,
        full_event: bool | typing.Any | None = False,
        emboss: bool | typing.Any | None = True,
        index: typing.Any | None = -1,
        icon_value: typing.Any | None = 0,
    ):
        """Item. Exposes an RNA item and places it into the layout

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        :param expand: Expand button to show more detail
        :type expand: bool | typing.Any | None
        :param slider: Use slider widget for numeric values
        :type slider: bool | typing.Any | None
        :param toggle: Use toggle widget for boolean values
        :type toggle: bool | typing.Any | None
        :param icon_only: Draw only icons in buttons, no text
        :type icon_only: bool | typing.Any | None
        :param event: Use button to input key events
        :type event: bool | typing.Any | None
        :param full_event: Use button to input full events including modifiers
        :type full_event: bool | typing.Any | None
        :param emboss: Draw the button itself, just the icon/text
        :type emboss: bool | typing.Any | None
        :param index: The index of this button, when set a single member of an array can be accessed, when set to -1 all array members are used
        :type index: typing.Any | None
        :param icon_value: Icon Value, Override automatic icon of the item
        :type icon_value: typing.Any | None
        """
        ...

    def props_enum(self, data: typing.Any, property: str | typing.Any):
        """props_enum

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

    def prop_menu_enum(
        self,
        data: typing.Any,
        property: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
    ):
        """prop_menu_enum

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        """
        ...

    def prop_enum(
        self,
        data: typing.Any,
        property: str | typing.Any,
        value: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
    ):
        """prop_enum

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param value: Enum property value
        :type value: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        """
        ...

    def prop_search(
        self,
        data: typing.Any,
        property: str | typing.Any,
        search_data: typing.Any,
        search_property: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
    ):
        """prop_search

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param search_data: Data from which to take collection to search in
        :type search_data: typing.Any
        :param search_property: Identifier of search collection property
        :type search_property: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        """
        ...

    def operator(
        self,
        operator: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
        emboss: bool | typing.Any | None = True,
        icon_value: typing.Any | None = 0,
    ) -> OperatorProperties:
        """Item. Places a button into the layout to call an Operator

        :param operator: Identifier of the operator
        :type operator: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        :param emboss: Draw the button itself, just the icon/text
        :type emboss: bool | typing.Any | None
        :param icon_value: Icon Value, Override automatic icon of the item
        :type icon_value: typing.Any | None
        :return: Operator properties to fill in, return when 'properties' is set to true
        :rtype: OperatorProperties
        """
        ...

    def operator_enum(self, operator: str | typing.Any, property: str | typing.Any):
        """operator_enum

        :param operator: Identifier of the operator
        :type operator: str | typing.Any
        :param property: Identifier of property in operator
        :type property: str | typing.Any
        """
        ...

    def operator_menu_enum(
        self,
        operator: str | typing.Any,
        property: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
    ):
        """operator_menu_enum

        :param operator: Identifier of the operator
        :type operator: str | typing.Any
        :param property: Identifier of property in operator
        :type property: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        """
        ...

    def label(
        self,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
        icon_value: typing.Any | None = 0,
    ):
        """Item. Displays text and/or icon in the layout

        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        :param icon_value: Icon Value, Override automatic icon of the item
        :type icon_value: typing.Any | None
        """
        ...

    def menu(
        self,
        menu: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
        icon: str | None = "NONE",
        icon_value: typing.Any | None = 0,
    ):
        """menu

        :param menu: Identifier of the menu
        :type menu: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        :param icon: Icon, Override automatic icon of the item
        :type icon: str | None
        :param icon_value: Icon Value, Override automatic icon of the item
        :type icon_value: typing.Any | None
        """
        ...

    def separator(self):
        """Item. Inserts empty space into the layout between items"""
        ...

    def context_pointer_set(self, name: str | typing.Any, data: typing.Any | None):
        """context_pointer_set

        :param name: Name, Name of entry in the context
        :type name: str | typing.Any
        :param data: Pointer to put in context
        :type data: typing.Any | None
        """
        ...

    def template_header(self):
        """Inserts common Space header UI (editor type selector)"""
        ...

    def template_ID(
        self,
        data: typing.Any,
        property: str | typing.Any,
        new: str | typing.Any = "",
        open: str | typing.Any = "",
        unlink: str | typing.Any = "",
    ):
        """template_ID

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param new: Operator identifier to create a new ID block
        :type new: str | typing.Any
        :param open: Operator identifier to open a file for creating a new ID block
        :type open: str | typing.Any
        :param unlink: Operator identifier to unlink the ID block
        :type unlink: str | typing.Any
        """
        ...

    def template_ID_preview(
        self,
        data: typing.Any,
        property: str | typing.Any,
        new: str | typing.Any = "",
        open: str | typing.Any = "",
        unlink: str | typing.Any = "",
        rows: typing.Any | None = 0,
        cols: typing.Any | None = 0,
    ):
        """template_ID_preview

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param new: Operator identifier to create a new ID block
        :type new: str | typing.Any
        :param open: Operator identifier to open a file for creating a new ID block
        :type open: str | typing.Any
        :param unlink: Operator identifier to unlink the ID block
        :type unlink: str | typing.Any
        :param rows: Number of thumbnail preview rows to display
        :type rows: typing.Any | None
        :param cols: Number of thumbnail preview columns to display
        :type cols: typing.Any | None
        """
        ...

    def template_any_ID(
        self,
        data: typing.Any,
        property: str | typing.Any,
        type_property: str | typing.Any,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
    ):
        """template_any_ID

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param type_property: Identifier of property in data giving the type of the ID-blocks to use
        :type type_property: str | typing.Any
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        """
        ...

    def template_path_builder(
        self,
        data: typing.Any,
        property: str | typing.Any,
        root: ID | None,
        text: str | typing.Any = "",
        text_ctxt: str | typing.Any = "",
        translate: bool | typing.Any | None = True,
    ):
        """template_path_builder

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param root: ID-block from which path is evaluated from
        :type root: ID | None
        :param text: Override automatic text of the item
        :type text: str | typing.Any
        :param text_ctxt: Override automatic translation context of the given text
        :type text_ctxt: str | typing.Any
        :param translate: Translate the given text, when UI translation is enabled
        :type translate: bool | typing.Any | None
        """
        ...

    def template_modifier(self, data: Modifier) -> UILayout:
        """Generates the UI layout for modifiers

        :param data: Modifier data
        :type data: Modifier
        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    def template_constraint(self, data: Constraint) -> UILayout:
        """Generates the UI layout for constraints

        :param data: Constraint data
        :type data: Constraint
        :return: Sub-layout to put items in
        :rtype: UILayout
        """
        ...

    def template_preview(
        self,
        id: ID | None,
        show_buttons: bool | typing.Any | None = True,
        parent: ID | None = None,
        slot: TextureSlot | None = None,
        preview_id: str | typing.Any = "",
    ):
        """Item. A preview window for materials, textures, lamps or worlds

        :param id: ID data-block
        :type id: ID | None
        :param show_buttons: Show preview buttons?
        :type show_buttons: bool | typing.Any | None
        :param parent: ID data-block
        :type parent: ID | None
        :param slot: Texture slot
        :type slot: TextureSlot | None
        :param preview_id: Identifier of this preview widget, if not set the ID type will be used (i.e. all previews of materials without explicit ID will have the same size...)
        :type preview_id: str | typing.Any
        """
        ...

    def template_curve_mapping(
        self,
        data: typing.Any,
        property: str | typing.Any,
        type: str | None = "NONE",
        levels: bool | typing.Any | None = False,
        brush: bool | typing.Any | None = False,
        use_negative_slope: bool | typing.Any | None = False,
    ):
        """Item. A curve mapping widget used for e.g falloff curves for lamps

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param type: Type, Type of curves to display
        :type type: str | None
        :param levels: Show black/white levels
        :type levels: bool | typing.Any | None
        :param brush: Show brush options
        :type brush: bool | typing.Any | None
        :param use_negative_slope: Use a negative slope by default
        :type use_negative_slope: bool | typing.Any | None
        """
        ...

    def template_color_ramp(
        self,
        data: typing.Any,
        property: str | typing.Any,
        expand: bool | typing.Any | None = False,
    ):
        """Item. A color ramp widget

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param expand: Expand button to show more detail
        :type expand: bool | typing.Any | None
        """
        ...

    def template_icon_view(
        self,
        data: typing.Any,
        property: str | typing.Any,
        show_labels: bool | typing.Any | None = False,
        scale: typing.Any | None = 5.0,
    ):
        """Enum. Large widget showing Icon previews

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param show_labels: Show enum label in preview buttons
        :type show_labels: bool | typing.Any | None
        :param scale: Scale, Scale the icon size (by the button size)
        :type scale: typing.Any | None
        """
        ...

    def template_histogram(self, data: typing.Any, property: str | typing.Any):
        """Item. A histogramm widget to analyze imaga data

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

    def template_waveform(self, data: typing.Any, property: str | typing.Any):
        """Item. A waveform widget to analyze imaga data

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

    def template_vectorscope(self, data: typing.Any, property: str | typing.Any):
        """Item. A vectorscope widget to analyze imaga data

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

    def template_layers(
        self,
        data: typing.Any,
        property: str | typing.Any,
        used_layers_data: typing.Any | None,
        used_layers_property: str | typing.Any,
        active_layer: int | None,
    ):
        """template_layers

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param used_layers_data: Data from which to take property
        :type used_layers_data: typing.Any | None
        :param used_layers_property: Identifier of property in data
        :type used_layers_property: str | typing.Any
        :param active_layer: Active Layer
        :type active_layer: int | None
        """
        ...

    def template_color_picker(
        self,
        data: typing.Any,
        property: str | typing.Any,
        value_slider: bool | typing.Any | None = False,
        lock: bool | typing.Any | None = False,
        lock_luminosity: bool | typing.Any | None = False,
        cubic: bool | typing.Any | None = False,
    ):
        """Item. A color wheel widget to pick colors

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param value_slider: Display the value slider to the right of the color wheel
        :type value_slider: bool | typing.Any | None
        :param lock: Lock the color wheel display to value 1.0 regardless of actual color
        :type lock: bool | typing.Any | None
        :param lock_luminosity: Keep the color at its original vector length
        :type lock_luminosity: bool | typing.Any | None
        :param cubic: Cubic saturation for picking values close to white
        :type cubic: bool | typing.Any | None
        """
        ...

    def template_palette(
        self,
        data: typing.Any,
        property: str | typing.Any,
        color: bool | typing.Any | None = False,
    ):
        """Item. A palette used to pick colors

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param color: Display the colors as colors or values
        :type color: bool | typing.Any | None
        """
        ...

    def template_image_layers(self, image: Image | None, image_user: ImageUser | None):
        """template_image_layers

        :param image:
        :type image: Image | None
        :param image_user:
        :type image_user: ImageUser | None
        """
        ...

    def template_image(
        self,
        data: typing.Any,
        property: str | typing.Any,
        image_user: ImageUser,
        compact: bool | typing.Any | None = False,
        multiview: bool | typing.Any | None = False,
    ):
        """Item(s). User interface for selecting images and their source paths

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param image_user:
        :type image_user: ImageUser
        :param compact: Use more compact layout
        :type compact: bool | typing.Any | None
        :param multiview: Expose Multi-View options
        :type multiview: bool | typing.Any | None
        """
        ...

    def template_image_settings(
        self,
        image_settings: ImageFormatSettings,
        color_management: bool | typing.Any | None = False,
    ):
        """User interface for setting image format options

        :param image_settings:
        :type image_settings: ImageFormatSettings
        :param color_management: Show color management settings
        :type color_management: bool | typing.Any | None
        """
        ...

    def template_image_stereo_3d(self, stereo_3d_format: Stereo3dFormat):
        """User interface for setting image stereo 3d options

        :param stereo_3d_format:
        :type stereo_3d_format: Stereo3dFormat
        """
        ...

    def template_image_views(self, image_settings: ImageFormatSettings):
        """User interface for setting image views output options

        :param image_settings:
        :type image_settings: ImageFormatSettings
        """
        ...

    def template_movieclip(
        self,
        data: typing.Any,
        property: str | typing.Any,
        compact: bool | typing.Any | None = False,
    ):
        """Item(s). User interface for selecting movie clips and their source paths

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param compact: Use more compact layout
        :type compact: bool | typing.Any | None
        """
        ...

    def template_track(self, data: typing.Any, property: str | typing.Any):
        """Item. A movie-track widget to preview tracking image.

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

    def template_marker(
        self,
        data: typing.Any,
        property: str | typing.Any,
        clip_user: MovieClipUser,
        track: MovieTrackingTrack,
        compact: bool | typing.Any | None = False,
    ):
        """Item. A widget to control single marker settings.

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param clip_user:
        :type clip_user: MovieClipUser
        :param track:
        :type track: MovieTrackingTrack
        :param compact: Use more compact layout
        :type compact: bool | typing.Any | None
        """
        ...

    def template_movieclip_information(
        self, data: typing.Any, property: str | typing.Any, clip_user: MovieClipUser
    ):
        """Item. Movie clip information data.

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param clip_user:
        :type clip_user: MovieClipUser
        """
        ...

    def template_list(
        self,
        listtype_name: str | typing.Any,
        list_id: str | typing.Any = "",
        dataptr: typing.Any | None = None,
        propname: str | typing.Any = None,
        active_dataptr: typing.Any = None,
        active_propname: str | typing.Any = None,
        item_dyntip_propname: str | typing.Any = "",
        rows: typing.Any | None = 5,
        maxrows: typing.Any | None = 5,
        type: str | None = "DEFAULT",
        columns: typing.Any | None = 9,
    ):
        """Item. A list widget to display data, e.g. vertexgroups.

                :param listtype_name: Identifier of the list type to use
                :type listtype_name: str | typing.Any
                :param list_id: Identifier of this list widget (mandatory when using default "UI_UL_list" class). If this is set, the uilist gets a custom ID, otherwise it takes the name of the class used to define the uilist (for example, if the class name is "OBJECT_UL_vgroups", and list_id is not set by the script, then bl_idname = "OBJECT_UL_vgroups")
                :type list_id: str | typing.Any
                :param dataptr: Data from which to take the Collection property
                :type dataptr: typing.Any | None
                :param propname: Identifier of the Collection property in data
                :type propname: str | typing.Any
                :param active_dataptr: Data from which to take the integer property, index of the active item
                :type active_dataptr: typing.Any
                :param active_propname: Identifier of the integer property in active_data, index of the active item
                :type active_propname: str | typing.Any
                :param item_dyntip_propname: Identifier of a string property in items, to use as tooltip content
                :type item_dyntip_propname: str | typing.Any
                :param rows: Default and minimum number of rows to display
                :type rows: typing.Any | None
                :param maxrows: Default maximum number of rows to display
                :type maxrows: typing.Any | None
                :param type: Type, Type of layout to use

        DEFAULT Default Layout, Use the default, multi-rows layout.

        COMPACT Compact Layout, Use the compact, single-row layout.

        GRID Grid Layout, Use the grid-based layout.
                :type type: str | None
                :param columns: Number of items to display per row, for GRID layout
                :type columns: typing.Any | None
        """
        ...

    def template_running_jobs(self):
        """template_running_jobs"""
        ...

    def template_operator_search(self):
        """template_operator_search"""
        ...

    def template_header_3D(self):
        """Inserts common 3DView header UI (selectors for context mode, shading, etc.)"""
        ...

    def template_edit_mode_selection(self):
        """Inserts common 3DView Edit modes header UI (selector for selection mode)"""
        ...

    def template_reports_banner(self):
        """template_reports_banner"""
        ...

    def template_node_link(
        self, ntree: NodeTree | None, node: Node | None, socket: NodeSocket | None
    ):
        """template_node_link

        :param ntree:
        :type ntree: NodeTree | None
        :param node:
        :type node: Node | None
        :param socket:
        :type socket: NodeSocket | None
        """
        ...

    def template_node_view(
        self, ntree: NodeTree | None, node: Node | None, socket: NodeSocket | None
    ):
        """template_node_view

        :param ntree:
        :type ntree: NodeTree | None
        :param node:
        :type node: Node | None
        :param socket:
        :type socket: NodeSocket | None
        """
        ...

    def template_texture_user(self):
        """template_texture_user"""
        ...

    def template_keymap_item_properties(self, item: KeyMapItem):
        """template_keymap_item_properties

        :param item:
        :type item: KeyMapItem
        """
        ...

    def template_component_menu(
        self,
        data: typing.Any | None,
        property: str | typing.Any,
        name: str | typing.Any = "",
    ):
        """Item. Display expanded property in a popup menu

        :param data: Data from which to take property
        :type data: typing.Any | None
        :param property: Identifier of property in data
        :type property: str | typing.Any
        :param name:
        :type name: str | typing.Any
        """
        ...

    def introspect(self) -> str | typing.Any:
        """introspect

        :return: Descr, DESCR
        :rtype: str | typing.Any
        """
        ...

    def template_colorspace_settings(
        self, data: typing.Any, property: str | typing.Any
    ):
        """Item. A widget to control input color space settings.

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

    def template_colormanaged_view_settings(
        self, data: typing.Any, property: str | typing.Any
    ):
        """Item. A widget to control color managed view settings settings.

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

    def template_node_socket(self, color: typing.Any | None = (0.0, 0.0, 0.0, 1.0)):
        """Node Socket Icon

        :param color: Color
        :type color: typing.Any | None
        """
        ...

    def template_cache_file(self, data: typing.Any, property: str | typing.Any):
        """Item(s). User interface for selecting cache files and their source paths

        :param data: Data from which to take property
        :type data: typing.Any
        :param property: Identifier of property in data
        :type property: str | typing.Any
        """
        ...

class UIList(bpy_struct):
    """UI list containing the elements of a collection"""

    bitflag_filter_item: int
    """ The value of the reserved bitflag 'FILTER_ITEM' (in filter_flags values)

    :type: int
    """

    bl_idname: str
    """ If this is set, the uilist gets a custom ID, otherwise it takes the name of the class used to define the uilist (for example, if the class name is "OBJECT_UL_vgroups", and bl_idname is not set by the script, then bl_idname = "OBJECT_UL_vgroups")

    :type: str
    """

    filter_name: str
    """ Only show items matching this name (use '*' as wildcard)

    :type: str
    """

    layout_type: str
    """ 

    :type: str
    """

    use_filter_invert: bool
    """ Invert filtering (show hidden items, and vice-versa)

    :type: bool
    """

    use_filter_show: bool
    """ Show filtering options

    :type: bool
    """

    use_filter_sort_alpha: bool
    """ Sort items by their name

    :type: bool
    """

    use_filter_sort_reverse: bool
    """ Invert the order of shown items

    :type: bool
    """

    def draw_item(
        self,
        context: Context | None,
        layout: UILayout,
        data: typing.Any | None,
        item: typing.Any | None,
        icon: int | None,
        active_data: typing.Any,
        active_property: str | None,
        index: typing.Any | None = 0,
        flt_flag: typing.Any | None = 0,
    ):
        """Draw an item in the list (NOTE: when you define your own draw_item function, you may want to check given 'item' is of the right type...)

        :param context:
        :type context: Context | None
        :param layout: Layout to draw the item
        :type layout: UILayout
        :param data: Data from which to take Collection property
        :type data: typing.Any | None
        :param item: Item of the collection property
        :type item: typing.Any | None
        :param icon: Icon of the item in the collection
        :type icon: int | None
        :param active_data: Data from which to take property for the active element
        :type active_data: typing.Any
        :param active_property: Identifier of property in active_data, for the active element
        :type active_property: str | None
        :param index: Index of the item in the collection
        :type index: typing.Any | None
        :param flt_flag: The filter-flag result for this item
        :type flt_flag: typing.Any | None
        """
        ...

    def draw_filter(self, context: Context | None, layout: UILayout):
        """Draw filtering options

        :param context:
        :type context: Context | None
        :param layout: Layout to draw the item
        :type layout: UILayout
        """
        ...

    def filter_items(
        self,
        context: Context | None,
        data: typing.Any | None,
        property: str | typing.Any,
    ):
        """Filter and/or re-order items of the collection (output filter results in filter_flags, and reorder results in filter_neworder arrays)

                :param context:
                :type context: Context | None
                :param data: Data from which to take Collection property
                :type data: typing.Any | None
                :param property: Identifier of property in data, for the collection
                :type property: str | typing.Any
                :return: filter_flags, An array of filter flags, one for each item in the collection (NOTE: FILTER_ITEM bit is reserved, it defines whether the item is shown or not), int array of 1 items in [0, inf]

        filter_neworder, An array of indices, one for each item in the collection, mapping the org index to the new one, int array of 1 items in [0, inf]
        """
        ...

class UIPieMenu(bpy_struct):
    layout: UILayout
    """ 

    :type: UILayout
    """

class UIPopupMenu(bpy_struct):
    layout: UILayout
    """ 

    :type: UILayout
    """

class UVLoopLayers(bpy_prop_collection[MeshUVLoopLayer], bpy_struct):
    """Collection of uv loop layers"""

    active: MeshUVLoopLayer | None
    """ Active UV loop layer

    :type: MeshUVLoopLayer | None
    """

    active_index: int | None
    """ Active UV loop layer index

    :type: int | None
    """

class UVProjector(bpy_struct):
    """UV projector used by the UV project modifier"""

    object: Object
    """ Object to use as projector transform

    :type: Object
    """

class UVTextures(bpy_prop_collection[MeshTexturePolyLayer], bpy_struct):
    """Collection of UV maps"""

    active: MeshTexturePolyLayer | None
    """ Active UV Map

    :type: MeshTexturePolyLayer | None
    """

    active_index: int | None
    """ Active UV Map index

    :type: int | None
    """

    def new(self, name: str | typing.Any = "UVMap") -> MeshTexturePolyLayer:
        """Add a UV map layer to Mesh

        :param name: UV map name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshTexturePolyLayer
        """
        ...

    def remove(self, layer: MeshTexturePolyLayer):
        """Remove a vertex color layer

        :param layer: The layer to remove
        :type layer: MeshTexturePolyLayer
        """
        ...

class UnifiedPaintSettings(bpy_struct):
    """Overrides for some of the active brush's settings"""

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    secondary_color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    size: int
    """ Radius of the brush

    :type: int
    """

    strength: float
    """ How powerful the effect of the brush is when applied

    :type: float
    """

    unprojected_radius: float
    """ Radius of brush in Blender units

    :type: float
    """

    use_locked_size: bool
    """ When locked brush stays same size relative to object; when unlocked brush size is given in pixels

    :type: bool
    """

    use_pressure_size: bool
    """ Enable tablet pressure sensitivity for size

    :type: bool
    """

    use_pressure_strength: bool
    """ Enable tablet pressure sensitivity for strength

    :type: bool
    """

    use_unified_color: bool
    """ Instead of per-brush color, the color is shared across brushes

    :type: bool
    """

    use_unified_size: bool
    """ Instead of per-brush radius, the radius is shared across brushes

    :type: bool
    """

    use_unified_strength: bool
    """ Instead of per-brush strength, the strength is shared across brushes

    :type: bool
    """

    use_unified_weight: bool
    """ Instead of per-brush weight, the weight is shared across brushes

    :type: bool
    """

    weight: float
    """ Weight to assign in vertex groups

    :type: float
    """

class UnitSettings(bpy_struct):
    scale_length: float
    """ Scale to use when converting between blender units and dimensions

    :type: float
    """

    system: str
    """ The unit system to use for button display

    :type: str
    """

    system_rotation: str
    """ Unit to use for displaying/editing rotation values

    :type: str
    """

    use_separate: bool
    """ Display units in pairs (e.g. 1m 0cm)

    :type: bool
    """

class UnknownType(bpy_struct):
    """Stub RNA type used for pointers to unknown or internal data"""

    ...

class UserPreferences(bpy_struct):
    """Global user preferences"""

    active_section: str | None
    """ Active section of the user preferences shown in the user interface

    :type: str | None
    """

    addons: Addons
    """ 

    :type: Addons
    """

    autoexec_paths: PathCompareCollection
    """ 

    :type: PathCompareCollection
    """

    edit: UserPreferencesEdit
    """ Settings for interacting with Blender data

    :type: UserPreferencesEdit
    """

    filepaths: UserPreferencesFilePaths
    """ Default paths for external files

    :type: UserPreferencesFilePaths
    """

    inputs: UserPreferencesInput
    """ Settings for input devices

    :type: UserPreferencesInput
    """

    system: UserPreferencesSystem
    """ Graphics driver and operating system settings

    :type: UserPreferencesSystem
    """

    themes: bpy_prop_collection[Theme]
    """ 

    :type: bpy_prop_collection[Theme]
    """

    ui_styles: bpy_prop_collection[ThemeStyle]
    """ 

    :type: bpy_prop_collection[ThemeStyle]
    """

    version: bpy_prop_array[int]
    """ Version of Blender the userpref.blend was saved with

    :type: bpy_prop_array[int]
    """

    view: UserPreferencesView
    """ Preferences related to viewing data

    :type: UserPreferencesView
    """

class UserPreferencesEdit(bpy_struct):
    """Settings for interacting with Blender data"""

    auto_keying_mode: str
    """ Mode of automatic keyframe insertion for Objects and Bones (default setting used for new Scenes)

    :type: str
    """

    fcurve_unselected_alpha: float
    """ Amount that unselected F-Curves stand out from the background (Graph Editor)

    :type: float
    """

    grease_pencil_default_color: bpy_prop_array[float]
    """ Color of new Grease Pencil layers

    :type: bpy_prop_array[float]
    """

    grease_pencil_eraser_radius: int
    """ Radius of eraser 'brush'

    :type: int
    """

    grease_pencil_euclidean_distance: int
    """ Distance moved by mouse when drawing stroke to include

    :type: int
    """

    grease_pencil_manhattan_distance: int
    """ Pixels moved by mouse per axis when drawing stroke

    :type: int
    """

    keyframe_new_handle_type: str
    """ Handle type for handles of new keyframes

    :type: str
    """

    keyframe_new_interpolation_type: str
    """ Interpolation mode used for first keyframe on newly added F-Curves (subsequent keyframes take interpolation from preceding keyframe)

    :type: str
    """

    material_link: str
    """ Toggle whether the material is linked to object data or the object block

    :type: str
    """

    node_margin: int
    """ Minimum distance between nodes for Auto-offsetting nodes

    :type: int
    """

    object_align: str
    """ When adding objects from a 3D View menu, either align them with that view or with the world

    :type: str
    """

    sculpt_paint_overlay_color: bpy_prop_array[float]
    """ Color of texture overlay

    :type: bpy_prop_array[float]
    """

    undo_memory_limit: int
    """ Maximum memory usage in megabytes (0 means unlimited)

    :type: int
    """

    undo_steps: int
    """ Number of undo steps available (smaller values conserve memory)

    :type: int
    """

    use_auto_keying: bool
    """ Automatic keyframe insertion for Objects and Bones (default setting used for new Scenes)

    :type: bool
    """

    use_auto_keying_warning: bool
    """ Show warning indicators when transforming objects and bones if auto keying is enabled

    :type: bool
    """

    use_drag_immediately: bool
    """ Moving things with a mouse drag confirms when releasing the button

    :type: bool
    """

    use_duplicate_action: bool
    """ Causes actions to be duplicated with the object

    :type: bool
    """

    use_duplicate_armature: bool
    """ Causes armature data to be duplicated with the object

    :type: bool
    """

    use_duplicate_curve: bool
    """ Causes curve data to be duplicated with the object

    :type: bool
    """

    use_duplicate_fcurve: bool
    """ Causes F-curve data to be duplicated with the object

    :type: bool
    """

    use_duplicate_lamp: bool
    """ Causes lamp data to be duplicated with the object

    :type: bool
    """

    use_duplicate_material: bool
    """ Causes material data to be duplicated with the object

    :type: bool
    """

    use_duplicate_mesh: bool
    """ Causes mesh data to be duplicated with the object

    :type: bool
    """

    use_duplicate_metaball: bool
    """ Causes metaball data to be duplicated with the object

    :type: bool
    """

    use_duplicate_particle: bool
    """ Causes particle systems to be duplicated with the object

    :type: bool
    """

    use_duplicate_surface: bool
    """ Causes surface data to be duplicated with the object

    :type: bool
    """

    use_duplicate_text: bool
    """ Causes text data to be duplicated with the object

    :type: bool
    """

    use_duplicate_texture: bool
    """ Causes texture data to be duplicated with the object

    :type: bool
    """

    use_enter_edit_mode: bool
    """ Enter Edit Mode automatically after adding a new object

    :type: bool
    """

    use_global_undo: bool
    """ Global undo works by keeping a full copy of the file itself in memory, so takes extra memory

    :type: bool
    """

    use_grease_pencil_simplify_stroke: bool
    """ Simplify the final stroke

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) and also Color is based on the transform axis

    :type: bool
    """

    use_keyframe_insert_available: bool
    """ Automatic keyframe insertion in available F-Curves

    :type: bool
    """

    use_keyframe_insert_needed: bool
    """ Keyframe insertion only when keyframe needed

    :type: bool
    """

    use_negative_frames: bool
    """ Current frame number can be manually set to a negative value

    :type: bool
    """

    use_visual_keying: bool
    """ Use Visual keying automatically for constrained objects

    :type: bool
    """

class UserPreferencesFilePaths(bpy_struct):
    """Default paths for external files"""

    animation_player: str
    """ Path to a custom animation/frame sequence player

    :type: str
    """

    animation_player_preset: str
    """ Preset configs for external animation players

    :type: str
    """

    auto_save_time: int
    """ The time (in minutes) to wait between automatic temporary saves

    :type: int
    """

    font_directory: str
    """ The default directory to search for loading fonts

    :type: str
    """

    hide_recent_locations: bool
    """ Hide recent locations in the file selector

    :type: bool
    """

    hide_system_bookmarks: bool
    """ Hide system bookmarks in the file selector

    :type: bool
    """

    i18n_branches_directory: str
    """ The path to the '/branches' directory of your local svn-translation copy, to allow translating from the UI

    :type: str
    """

    image_editor: str
    """ Path to an image editor

    :type: str
    """

    recent_files: int
    """ Maximum number of recently opened files to remember

    :type: int
    """

    render_cache_directory: str
    """ Where to cache raw render results

    :type: str
    """

    render_output_directory: str
    """ The default directory for rendering output, for new scenes

    :type: str
    """

    save_version: int
    """ The number of old versions to maintain in the current directory, when manually saving

    :type: int
    """

    script_directory: str
    """ Alternate script path, matching the default layout with subdirs: startup, add-ons & modules (requires restart)

    :type: str
    """

    show_hidden_files_datablocks: bool
    """ Hide files/data-blocks that start with a dot (.*)

    :type: bool
    """

    show_thumbnails: bool
    """ Open in thumbnail view for images and movies

    :type: bool
    """

    sound_directory: str
    """ The default directory to search for sounds

    :type: str
    """

    temporary_directory: str
    """ The directory for storing temporary save files

    :type: str
    """

    texture_directory: str
    """ The default directory to search for textures

    :type: str
    """

    use_auto_save_temporary_files: bool
    """ Automatic saving of temporary files in temp directory, uses process ID (Sculpt or edit mode data won't be saved!')

    :type: bool
    """

    use_file_compression: bool
    """ Enable file compression when saving .blend files

    :type: bool
    """

    use_filter_files: bool
    """ Display only files with extensions in the image select window

    :type: bool
    """

    use_keep_session: bool
    """ Always load session recovery and save it after quitting Blender

    :type: bool
    """

    use_load_ui: bool
    """ Load user interface setup when loading .blend files

    :type: bool
    """

    use_relative_paths: bool
    """ Default relative path option for the file selector

    :type: bool
    """

    use_save_preview_images: bool
    """ Enables automatic saving of preview images in the .blend file

    :type: bool
    """

class UserPreferencesInput(bpy_struct):
    """Settings for input devices"""

    active_keyconfig: str
    """ The name of the active key configuration

    :type: str
    """

    drag_threshold: int
    """ Amount of pixels you have to drag before dragging UI items happens

    :type: int
    """

    invert_mouse_zoom: bool
    """ Invert the axis of mouse movement for zooming

    :type: bool
    """

    invert_zoom_wheel: bool
    """ Swap the Mouse Wheel zoom direction

    :type: bool
    """

    mouse_double_click_time: int
    """ Time/delay (in ms) for a double click

    :type: int
    """

    navigation_mode: str
    """ Which method to use for viewport navigation

    :type: str
    """

    ndof_deadzone: float
    """ Deadzone of the 3D Mouse

    :type: float
    """

    ndof_fly_helicopter: bool
    """ Device up/down directly controls your Z position

    :type: bool
    """

    ndof_lock_horizon: bool
    """ Keep horizon level while flying with 3D Mouse

    :type: bool
    """

    ndof_orbit_sensitivity: float
    """ Overall sensitivity of the 3D Mouse for orbiting

    :type: float
    """

    ndof_pan_yz_swap_axis: bool
    """ Pan using up/down on the device (otherwise forward/backward)

    :type: bool
    """

    ndof_panx_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_pany_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_panz_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_rotx_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_roty_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_rotz_invert_axis: bool
    """ 

    :type: bool
    """

    ndof_sensitivity: float
    """ Overall sensitivity of the 3D Mouse for panning

    :type: float
    """

    ndof_show_guide: bool
    """ Display the center and axis during rotation

    :type: bool
    """

    ndof_view_navigate_method: str
    """ Navigation style in the viewport

    :type: str
    """

    ndof_view_rotate_method: str
    """ Rotation style in the viewport

    :type: str
    """

    ndof_zoom_invert: bool
    """ Zoom using opposite direction

    :type: bool
    """

    select_mouse: str
    """ Mouse button used for selection

    :type: str
    """

    tweak_threshold: int
    """ Number of pixels you have to drag before tweak event is triggered

    :type: int
    """

    use_emulate_numpad: bool
    """ Main 1 to 0 keys act as the numpad ones (useful for laptops)

    :type: bool
    """

    use_mouse_continuous: bool
    """ Allow moving the mouse outside the view on some manipulations (transform, ui control drag)

    :type: bool
    """

    use_mouse_emulate_3_button: bool
    """ Emulate Middle Mouse with Alt+Left Mouse (doesn't work with Left Mouse Select option)

    :type: bool
    """

    use_mouse_mmb_paste: bool
    """ In text window, paste with middle mouse button instead of panning

    :type: bool
    """

    use_ndof: bool
    """ 

    :type: bool
    """

    use_trackpad_natural: bool
    """ If your system uses 'natural' scrolling, this option keeps consistent trackpad usage throughout the UI

    :type: bool
    """

    view_rotate_method: str
    """ Rotation style in the viewport

    :type: str
    """

    view_zoom_axis: str
    """ Axis of mouse movement to zoom in or out on

    :type: str
    """

    view_zoom_method: str
    """ Which style to use for viewport scaling

    :type: str
    """

    walk_navigation: WalkNavigation
    """ Settings for walk navigation mode

    :type: WalkNavigation
    """

    wheel_scroll_lines: int
    """ Number of lines scrolled at a time with the mouse wheel

    :type: int
    """

class UserPreferencesSystem(bpy_struct):
    """Graphics driver and operating system settings"""

    anisotropic_filter: str
    """ Quality of the anisotropic filtering (values greater than 1.0 enable anisotropic filtering)

    :type: str
    """

    audio_channels: str
    """ Audio channel count

    :type: str
    """

    audio_device: str
    """ Audio output device

    :type: str
    """

    audio_mixing_buffer: str
    """ Number of samples used by the audio mixing buffer

    :type: str
    """

    audio_sample_format: str
    """ Audio sample format

    :type: str
    """

    audio_sample_rate: str
    """ Audio sample rate

    :type: str
    """

    author: str
    """ Name that will be used in exported files when format supports such feature

    :type: str
    """

    color_picker_type: str
    """ Different styles of displaying the color picker widget

    :type: str
    """

    dpi: int
    """ Font size and resolution for display

    :type: int
    """

    font_path_ui: str
    """ Path to interface font

    :type: str
    """

    font_path_ui_mono: str
    """ Path to interface mono-space Font

    :type: str
    """

    frame_server_port: int
    """ Frameserver Port for Frameserver Rendering

    :type: int
    """

    gl_clip_alpha: float
    """ Clip alpha below this threshold in the 3D textured view

    :type: float
    """

    gl_texture_limit: str
    """ Limit the texture size to save graphics memory

    :type: str
    """

    image_draw_method: str
    """ Method used for displaying images on the screen

    :type: str
    """

    legacy_compute_device_type: int
    """ For backwards compatibility only

    :type: int
    """

    memory_cache_limit: int
    """ Memory cache limit (in megabytes)

    :type: int
    """

    multi_sample: str
    """ Enable OpenGL multi-sampling, only for systems that support it, requires restart

    :type: str
    """

    opensubdiv_compute_type: str
    """ Type of computer back-end used with OpenSubdiv

    :type: str
    """

    pixel_size: float
    """ 

    :type: float
    """

    prefetch_frames: int
    """ Number of frames to render ahead during playback (sequencer only)

    :type: int
    """

    screencast_fps: int
    """ Frame rate for the screencast to be played back

    :type: int
    """

    screencast_wait_time: int
    """ Time in milliseconds between each frame recorded for screencast

    :type: int
    """

    scrollback: int
    """ Maximum number of lines to store for the console buffer

    :type: int
    """

    select_method: str
    """ Use OpenGL occlusion queries or selection render mode to accelerate selection

    :type: str
    """

    solid_lights: bpy_prop_collection[UserSolidLight]
    """ Lights user to display objects in solid draw mode

    :type: bpy_prop_collection[UserSolidLight]
    """

    texture_collection_rate: int
    """ Number of seconds between each run of the GL texture garbage collector

    :type: int
    """

    texture_time_out: int
    """ Time since last access of a GL texture in seconds after which it is freed (set to 0 to keep textures allocated)

    :type: int
    """

    use_16bit_textures: bool
    """ Use 16 bit per component texture for float images

    :type: bool
    """

    use_gpu_mipmap: bool
    """ Generate Image Mipmaps on the GPU

    :type: bool
    """

    use_international_fonts: bool
    """ Use international fonts

    :type: bool
    """

    use_mipmaps: bool
    """ Scale textures for the 3D View (looks nicer but uses more memory and slows image reloading)

    :type: bool
    """

    use_preview_images: bool
    """ Allow user to choose any codec (Windows only, might generate instability)

    :type: bool
    """

    use_region_overlap: bool
    """ Draw tool/property regions over the main region, when using Triple Buffer

    :type: bool
    """

    use_scripts_auto_execute: bool
    """ Allow any .blend file to run scripts automatically (unsafe with blend files from an untrusted source)

    :type: bool
    """

    use_tabs_as_spaces: bool
    """ Automatically convert all new tabs into spaces for new and loaded text files

    :type: bool
    """

    use_text_antialiasing: bool
    """ Draw user interface text anti-aliased

    :type: bool
    """

    use_textured_fonts: bool
    """ Use textures for drawing international fonts

    :type: bool
    """

    use_translate_interface: bool
    """ Translate interface

    :type: bool
    """

    use_translate_new_dataname: bool
    """ Translate new data names (when adding/creating some)

    :type: bool
    """

    use_translate_tooltips: bool
    """ Translate tooltips

    :type: bool
    """

    use_weight_color_range: bool
    """ Enable color range used for weight visualization in weight painting mode

    :type: bool
    """

    virtual_pixel_mode: str
    """ Modify the pixel size for hi-res devices

    :type: str
    """

    weight_color_range: ColorRamp
    """ Color range used for weight visualization in weight painting mode

    :type: ColorRamp
    """

    window_draw_method: str
    """ Drawing method used by the window manager

    :type: str
    """

class UserPreferencesView(bpy_struct):
    """Preferences related to viewing data"""

    manipulator_handle_size: int
    """ Size of manipulator handles as percentage of the radius

    :type: int
    """

    manipulator_hotspot: int
    """ Distance around the handles to accept mouse clicks

    :type: int
    """

    manipulator_size: int
    """ Diameter of the manipulator

    :type: int
    """

    mini_axis_brightness: int
    """ Brightness of the icon

    :type: int
    """

    mini_axis_size: int
    """ The axes icon's size

    :type: int
    """

    object_origin_size: int
    """ Diameter in Pixels for Object/Lamp origin display

    :type: int
    """

    open_left_mouse_delay: int
    """ Time in 1/10 seconds to hold the Left Mouse Button before opening the toolbox

    :type: int
    """

    open_right_mouse_delay: int
    """ Time in 1/10 seconds to hold the Right Mouse Button before opening the toolbox

    :type: int
    """

    open_sublevel_delay: int
    """ Time delay in 1/10 seconds before automatically opening sub level menus

    :type: int
    """

    open_toplevel_delay: int
    """ Time delay in 1/10 seconds before automatically opening top level menus

    :type: int
    """

    pie_animation_timeout: int
    """ Time needed to fully animate the pie to unfolded state (in 1/100ths of sec)

    :type: int
    """

    pie_initial_timeout: int
    """ Pie menus will use the initial mouse position as center for this amount of time (in 1/100ths of sec)

    :type: int
    """

    pie_menu_confirm: int
    """ Distance threshold after which selection is made (zero to disable)

    :type: int
    """

    pie_menu_radius: int
    """ Pie menu size in pixels

    :type: int
    """

    pie_menu_threshold: int
    """ Distance from center needed before a selection can be made

    :type: int
    """

    rotation_angle: float
    """ Rotation step for numerical pad keys (2 4 6 8)

    :type: float
    """

    show_column_layout: bool
    """ Use a column layout for toolbox

    :type: bool
    """

    show_large_cursors: bool
    """ Use large mouse cursors when available

    :type: bool
    """

    show_manipulator: bool
    """ Use 3D transform manipulator

    :type: bool
    """

    show_mini_axis: bool
    """ Show a small rotating 3D axes in the bottom left corner of the 3D View

    :type: bool
    """

    show_object_info: bool
    """ Display objects name and frame number in 3D view

    :type: bool
    """

    show_playback_fps: bool
    """ Show the frames per second screen refresh rate, while animation is played back

    :type: bool
    """

    show_splash: bool
    """ Display splash screen on startup

    :type: bool
    """

    show_tooltips: bool
    """ Display tooltips (when off hold Alt to force display)

    :type: bool
    """

    show_tooltips_python: bool
    """ Show Python references in tooltips

    :type: bool
    """

    show_view_name: bool
    """ Show the name of the view's direction in each 3D View

    :type: bool
    """

    smooth_view: int
    """ Time to animate the view in milliseconds, zero to disable

    :type: int
    """

    timecode_style: str
    """ Format of Time Codes displayed when not displaying timing in terms of frames

    :type: str
    """

    use_auto_perspective: bool
    """ Automatically switch between orthographic and perspective when changing from top/front/side views

    :type: bool
    """

    use_camera_lock_parent: bool
    """ When the camera is locked to the view and in fly mode, transform the parent rather than the camera

    :type: bool
    """

    use_directional_menus: bool
    """ Otherwise menus, etc will always be top to bottom, left to right, no matter opening direction

    :type: bool
    """

    use_global_pivot: bool
    """ Lock the same rotation/scaling pivot in all 3D Views

    :type: bool
    """

    use_global_scene: bool
    """ Force the current Scene to be displayed in all Screens

    :type: bool
    """

    use_mouse_depth_cursor: bool
    """ Use the depth under the mouse when placing the cursor

    :type: bool
    """

    use_mouse_depth_navigate: bool
    """ Use the depth under the mouse to improve view pan/rotate/zoom functionality

    :type: bool
    """

    use_mouse_over_open: bool
    """ Open menu buttons and pulldowns automatically when the mouse is hovering

    :type: bool
    """

    use_quit_dialog: bool
    """ Ask for confirmation when quitting through the window close button

    :type: bool
    """

    use_rotate_around_active: bool
    """ Use selection as the pivot point

    :type: bool
    """

    use_zoom_to_mouse: bool
    """ Zoom in towards the mouse pointer's position in the 3D view, rather than the 2D window center

    :type: bool
    """

    view2d_grid_spacing_min: int
    """ Minimum number of pixels between each gridline in 2D Viewports

    :type: int
    """

    view_frame_keyframes: int
    """ Keyframes around cursor that we zoom around

    :type: int
    """

    view_frame_seconds: float
    """ Seconds around cursor that we zoom around

    :type: float
    """

    view_frame_type: str
    """ How zooming to frame focuses around current frame

    :type: str
    """

class UserSolidLight(bpy_struct):
    """Light used for OpenGL lighting in solid draw mode"""

    diffuse_color: bpy_prop_array[float]
    """ Diffuse color of the OpenGL light

    :type: bpy_prop_array[float]
    """

    direction: bpy_prop_array[float]
    """ Direction that the OpenGL light is shining

    :type: bpy_prop_array[float]
    """

    specular_color: bpy_prop_array[float]
    """ Color of the light's specular highlight

    :type: bpy_prop_array[float]
    """

    use: bool
    """ Enable this OpenGL light in solid draw mode

    :type: bool
    """

class VertexColors(bpy_prop_collection[MeshColorLayer], bpy_struct):
    """Collection of vertex colors"""

    active: MeshColorLayer
    """ Active vertex color layer

    :type: MeshColorLayer
    """

    active_index: int | None
    """ Active vertex color index

    :type: int | None
    """

    def new(self, name: str | typing.Any = "Col") -> MeshColorLayer:
        """Add a vertex color layer to Mesh

        :param name: Vertex color name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshColorLayer
        """
        ...

class VertexFloatProperties(
    bpy_prop_collection[MeshVertexFloatPropertyLayer], bpy_struct
):
    """Collection of float properties"""

    def new(
        self, name: str | typing.Any = "Float Prop"
    ) -> MeshVertexFloatPropertyLayer:
        """Add a float property layer to Mesh

        :param name: Float property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshVertexFloatPropertyLayer
        """
        ...

class VertexGroup(bpy_struct):
    """Group of vertices, used for armature deform and other purposes"""

    index: int
    """ Index number of the vertex group

    :type: int
    """

    lock_weight: bool
    """ Maintain the relative weights for the group

    :type: bool
    """

    name: str
    """ Vertex group name

    :type: str
    """

    def add(
        self,
        index: collections.abc.Iterable[int] | None,
        weight: float | None,
        type: str | None,
    ):
        """Add vertices to the group

                :param index: Index List
                :type index: collections.abc.Iterable[int] | None
                :param weight: Vertex weight
                :type weight: float | None
                :param type: Vertex assign mode

        REPLACE Replace, Replace.

        ADD Add, Add.

        SUBTRACT Subtract, Subtract.
                :type type: str | None
        """
        ...

    def remove(self, index: collections.abc.Iterable[int] | None):
        """Remove a vertex from the group

        :param index: Index List
        :type index: collections.abc.Iterable[int] | None
        """
        ...

    def weight(self, index: int | None) -> float:
        """Get a vertex weight from the group

        :param index: Index, The index of the vertex
        :type index: int | None
        :return: Vertex weight
        :rtype: float
        """
        ...

class VertexGroupElement(bpy_struct):
    """Weight value of a vertex in a vertex group"""

    group: int
    """ 

    :type: int
    """

    weight: float
    """ Vertex Weight

    :type: float
    """

class VertexGroups(bpy_prop_collection[VertexGroup], bpy_struct):
    """Collection of vertex groups"""

    active: VertexGroup
    """ Vertex groups of the object

    :type: VertexGroup
    """

    active_index: int | None
    """ Active index in vertex group array

    :type: int | None
    """

    def new(self, name: str | typing.Any = "Group") -> VertexGroup:
        """Add vertex group to object

        :param name: Vertex group name
        :type name: str | typing.Any
        :return: New vertex group
        :rtype: VertexGroup
        """
        ...

    def remove(self, group: VertexGroup):
        """Delete vertex group from object

        :param group: Vertex group to remove
        :type group: VertexGroup
        """
        ...

    def clear(self):
        """Delete all vertex groups from object"""
        ...

class VertexIntProperties(bpy_prop_collection[MeshVertexIntPropertyLayer], bpy_struct):
    """Collection of int properties"""

    def new(self, name: str | typing.Any = "Int Prop") -> MeshVertexIntPropertyLayer:
        """Add a integer property layer to Mesh

        :param name: Int property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshVertexIntPropertyLayer
        """
        ...

class VertexStringProperties(
    bpy_prop_collection[MeshVertexStringPropertyLayer], bpy_struct
):
    """Collection of string properties"""

    def new(
        self, name: str | typing.Any = "String Prop"
    ) -> MeshVertexStringPropertyLayer:
        """Add a string property layer to Mesh

        :param name: String property name
        :type name: str | typing.Any
        :return: The newly created layer
        :rtype: MeshVertexStringPropertyLayer
        """
        ...

class View2D(bpy_struct):
    """Scroll and zoom for a 2D region"""

    def region_to_view(self, x: int | None, y: int | None) -> bpy_prop_array[float]:
        """Transform region coordinates to 2D view

        :param x: x, Region x coordinate
        :type x: int | None
        :param y: y, Region y coordinate
        :type y: int | None
        :return: Result, View coordinates
        :rtype: bpy_prop_array[float]
        """
        ...

    def view_to_region(
        self, x: float | None, y: float | None, clip: bool | typing.Any | None = True
    ) -> bpy_prop_array[int]:
        """Transform 2D view coordinates to region

        :param x: x, 2D View x coordinate
        :type x: float | None
        :param y: y, 2D View y coordinate
        :type y: float | None
        :param clip: Clip, Clip coordinates to the visible region
        :type clip: bool | typing.Any | None
        :return: Result, Region coordinates
        :rtype: bpy_prop_array[int]
        """
        ...

class VoxelData(bpy_struct):
    """Voxel data settings"""

    domain_object: Object
    """ Object used as the smoke simulation domain

    :type: Object
    """

    extension: str
    """ How the texture is extrapolated past its original bounds

    :type: str
    """

    file_format: str
    """ Format of the source data set to render

    :type: str
    """

    filepath: str
    """ The external source data file to use

    :type: str
    """

    hair_data_type: str
    """ Simulation value to be used as a texture

    :type: str
    """

    intensity: float
    """ Multiplier for intensity values

    :type: float
    """

    interpolation: str
    """ Method to interpolate/smooth values between voxel cells

    :type: str
    """

    resolution: bpy_prop_array[int]
    """ Resolution of the voxel grid

    :type: bpy_prop_array[int]
    """

    smoke_data_type: str
    """ Simulation value to be used as a texture

    :type: str
    """

    still_frame: int
    """ The frame number to always use

    :type: int
    """

    use_still_frame: bool
    """ Always render a still frame from the voxel data sequence

    :type: bool
    """

class WalkNavigation(bpy_struct):
    """Walk navigation settings"""

    jump_height: float
    """ Maximum height of a jump

    :type: float
    """

    mouse_speed: float
    """ Speed factor for when looking around, high values mean faster mouse movement

    :type: float
    """

    teleport_time: float
    """ Interval of time warp when teleporting in navigation mode

    :type: float
    """

    use_gravity: bool
    """ Walk with gravity, or free navigate

    :type: bool
    """

    use_mouse_reverse: bool
    """ Reverse the mouse look

    :type: bool
    """

    view_height: float
    """ View distance from the floor when walking

    :type: float
    """

    walk_speed: float
    """ Base speed for walking and flying

    :type: float
    """

    walk_speed_factor: float
    """ Multiplication factor when using the fast or slow modifiers

    :type: float
    """

class Window(bpy_struct):
    """Open window"""

    height: int
    """ Window height

    :type: int
    """

    screen: Screen
    """ Active screen showing in the window

    :type: Screen
    """

    stereo_3d_display: Stereo3dDisplay
    """ Settings for stereo 3d display

    :type: Stereo3dDisplay
    """

    width: int
    """ Window width

    :type: int
    """

    x: int
    """ Horizontal location of the window

    :type: int
    """

    y: int
    """ Vertical location of the window

    :type: int
    """

    def cursor_warp(self, x: int | None, y: int | None):
        """Set the cursor position

        :param x:
        :type x: int | None
        :param y:
        :type y: int | None
        """
        ...

    def cursor_set(self, cursor: str | None):
        """Set the cursor

        :param cursor: cursor
        :type cursor: str | None
        """
        ...

    def cursor_modal_set(self, cursor: str | None):
        """Restore the previous cursor after calling cursor_modal_set

        :param cursor: cursor
        :type cursor: str | None
        """
        ...

    def cursor_modal_restore(self):
        """cursor_modal_restore"""
        ...

class WorldLighting(bpy_struct):
    """Lighting for a World data-block"""

    adapt_to_speed: float
    """ Use the speed vector pass to reduce AO samples in fast moving pixels - higher values result in more aggressive sample reduction (requires Vec pass enabled, for Raytrace Adaptive QMC)

    :type: float
    """

    ao_blend_type: str
    """ Defines how AO mixes with material shading

    :type: str
    """

    ao_factor: float
    """ Factor for ambient occlusion blending

    :type: float
    """

    bias: float
    """ Bias (in radians) to prevent smoothed faces from showing banding (for Raytrace Constant Jittered)

    :type: float
    """

    correction: float
    """ Ad-hoc correction for over-occlusion due to the approximation

    :type: float
    """

    distance: float
    """ Length of rays, defines how far away other faces give occlusion effect

    :type: float
    """

    environment_color: str
    """ Defines where the color of the environment light comes from

    :type: str
    """

    environment_energy: float
    """ Defines the strength of environment light

    :type: float
    """

    error_threshold: float
    """ Low values are slower and higher quality

    :type: float
    """

    falloff_strength: float
    """ Attenuation falloff strength, the higher, the less influence distant objects have

    :type: float
    """

    gather_method: str
    """ 

    :type: str
    """

    indirect_bounces: int
    """ Number of indirect diffuse light bounces

    :type: int
    """

    indirect_factor: float
    """ Factor for how much surrounding objects contribute to light

    :type: float
    """

    passes: int
    """ Number of preprocessing passes to reduce over-occlusion

    :type: int
    """

    sample_method: str
    """ Method for generating shadow samples (for Raytrace)

    :type: str
    """

    samples: int
    """ Amount of ray samples. Higher values give smoother results and longer rendering times

    :type: int
    """

    threshold: float
    """ Samples below this threshold will be considered fully shadowed/unshadowed and skipped (for Raytrace Adaptive QMC)

    :type: float
    """

    use_ambient_occlusion: bool
    """ Use Ambient Occlusion to add shadowing based on distance between objects

    :type: bool
    """

    use_cache: bool
    """ Cache AO results in pixels and interpolate over neighboring pixels for speedup

    :type: bool
    """

    use_environment_light: bool
    """ Add light coming from the environment

    :type: bool
    """

    use_falloff: bool
    """ Distance will be used to attenuate shadows

    :type: bool
    """

    use_indirect_light: bool
    """ Add indirect light bouncing of surrounding objects

    :type: bool
    """

class WorldMistSettings(bpy_struct):
    """Mist settings for a World data-block"""

    depth: float
    """ Distance over which the mist effect fades in

    :type: float
    """

    falloff: str
    """ Type of transition used to fade mist

    :type: str
    """

    height: float
    """ Control how much mist density decreases with height

    :type: float
    """

    intensity: float
    """ Overall minimum intensity of the mist effect

    :type: float
    """

    start: float
    """ Starting distance of the mist, measured from the camera

    :type: float
    """

    use_mist: bool
    """ Occlude objects with the environment color as they are further away

    :type: bool
    """

class WorldTextureSlots(bpy_prop_collection[WorldTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> WorldTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: WorldTextureSlot
        """
        ...

    @classmethod
    def create(cls, index: int | None) -> WorldTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: WorldTextureSlot
        """
        ...

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """
        ...

class ActionActuator(Actuator, bpy_struct):
    """Actuator to control the object movement"""

    action: Action
    """ 

    :type: Action
    """

    apply_to_children: bool
    """ Update Action on all children Objects as well

    :type: bool
    """

    blend_mode: str
    """ How this layer is blended with previous layers

    :type: str
    """

    frame_blend_in: int
    """ Number of frames of motion blending

    :type: int
    """

    frame_end: float
    """ 

    :type: float
    """

    frame_property: str
    """ Assign the action's current frame number to this property

    :type: str
    """

    frame_start: float
    """ 

    :type: float
    """

    layer: int
    """ The animation layer to play the action on

    :type: int
    """

    layer_weight: float
    """ How much of the previous layer to blend into this one

    :type: float
    """

    play_mode: str
    """ Action playback type

    :type: str
    """

    priority: int
    """ Execution priority - lower numbers will override actions with higher numbers (with 2 or more actions at once, the overriding channels must be lower in the stack)

    :type: int
    """

    property: str
    """ Use this property to define the Action position

    :type: str
    """

    use_additive: bool
    """ Action is added to the current loc/rot/scale in global or local coordinate according to Local flag

    :type: bool
    """

    use_continue_last_frame: bool
    """ Restore last frame when switching on/off, otherwise play from the start each time

    :type: bool
    """

    use_force: bool
    """ Apply Action as a global or local force depending on the local option (dynamic objects only)

    :type: bool
    """

    use_local: bool
    """ Let the Action act in local coordinates, used in Force and Add mode

    :type: bool
    """

class ArmatureActuator(Actuator, bpy_struct):
    bone: str
    """ Bone on which the constraint is defined

    :type: str
    """

    constraint: str
    """ Name of the constraint to control

    :type: str
    """

    influence: float
    """ Influence of this constraint

    :type: float
    """

    mode: str
    """ 

    :type: str
    """

    secondary_target: Object
    """ Set this object as the secondary target of the constraint (only IK polar target at the moment)

    :type: Object
    """

    target: Object
    """ Set this object as the target of the constraint

    :type: Object
    """

    weight: float
    """ Weight of this constraint

    :type: float
    """

class CameraActuator(Actuator, bpy_struct):
    axis: str
    """ Axis the Camera will try to get behind

    :type: str
    """

    damping: float
    """ Strength of the constraint that drives the camera behind the target

    :type: float
    """

    height: float
    """ 

    :type: float
    """

    max: float
    """ 

    :type: float
    """

    min: float
    """ 

    :type: float
    """

    object: Object
    """ Look at this Object

    :type: Object
    """

class ConstraintActuator(Actuator, bpy_struct):
    """Actuator to handle Constraints"""

    angle_max: float
    """ Maximum angle allowed with target direction (no correction is done if angle with target direction is between min and max)

    :type: float
    """

    angle_min: float
    """ Minimum angle to maintain with target direction (no correction is done if angle with target direction is between min and max)

    :type: float
    """

    damping: int
    """ Damping factor: time constant (in frame) of low pass filter

    :type: int
    """

    damping_rotation: int
    """ Use a different damping for orientation

    :type: int
    """

    direction: str
    """ Direction of the ray

    :type: str
    """

    direction_axis: str
    """ Select the axis to be aligned along the reference direction

    :type: str
    """

    direction_axis_pos: str
    """ Select the axis to be aligned along the reference direction

    :type: str
    """

    distance: float
    """ Keep this distance to target

    :type: float
    """

    fh_damping: float
    """ Damping factor of the force field spring

    :type: float
    """

    fh_force: float
    """ Spring force within the force field area

    :type: float
    """

    fh_height: float
    """ Height of the force field area

    :type: float
    """

    limit: str
    """ 

    :type: str
    """

    limit_max: float
    """ 

    :type: float
    """

    limit_min: float
    """ 

    :type: float
    """

    material: str
    """ Ray detects only Objects with this material

    :type: str
    """

    mode: str
    """ The type of the constraint

    :type: str
    """

    property: str
    """ Ray detects only Objects with this property

    :type: str
    """

    range: float
    """ Maximum length of ray

    :type: float
    """

    rotation_max: bpy_prop_array[float]
    """ Reference Direction

    :type: bpy_prop_array[float]
    """

    time: int
    """ Maximum activation time in frame, 0 for unlimited

    :type: int
    """

    use_fh_normal: bool
    """ Add a horizontal spring force on slopes

    :type: bool
    """

    use_fh_paralel_axis: bool
    """ Keep object axis parallel to normal

    :type: bool
    """

    use_force_distance: bool
    """ Force distance of object to point of impact of ray

    :type: bool
    """

    use_local: bool
    """ Set ray along object's axis or global axis

    :type: bool
    """

    use_material_detect: bool
    """ Detect material instead of property

    :type: bool
    """

    use_normal: bool
    """ Set object axis along (local axis) or parallel (global axis) to the normal at hit position

    :type: bool
    """

    use_persistent: bool
    """ Persistent actuator: stays active even if ray does not reach target

    :type: bool
    """

class EditObjectActuator(Actuator, bpy_struct):
    """Actuator used to edit objects"""

    angular_velocity: bpy_prop_array[float]
    """ Angular velocity upon creation

    :type: bpy_prop_array[float]
    """

    dynamic_operation: str
    """ 

    :type: str
    """

    linear_velocity: bpy_prop_array[float]
    """ Velocity upon creation

    :type: bpy_prop_array[float]
    """

    mass: float
    """ The mass of the object

    :type: float
    """

    mesh: Mesh
    """ Replace the existing, when left blank 'Phys' will remake the existing physics mesh

    :type: Mesh
    """

    mode: str
    """ The mode of the actuator

    :type: str
    """

    object: Object
    """ Add this Object and all its children (can't be on a visible layer)

    :type: Object
    """

    time: int
    """ Duration the new Object lives or the track takes

    :type: int
    """

    track_axis: str
    """ The axis that points to the target object

    :type: str
    """

    track_object: Object
    """ Track to this Object

    :type: Object
    """

    up_axis: str
    """ The axis that points upward

    :type: str
    """

    use_3d_tracking: bool
    """ Enable 3D tracking

    :type: bool
    """

    use_local_angular_velocity: bool
    """ Apply the rotation locally

    :type: bool
    """

    use_local_linear_velocity: bool
    """ Apply the transformation locally

    :type: bool
    """

    use_replace_display_mesh: bool
    """ Replace the display mesh

    :type: bool
    """

    use_replace_physics_mesh: bool
    """ Replace the physics mesh (triangle bounds only - compound shapes not supported)

    :type: bool
    """

class Filter2DActuator(Actuator, bpy_struct):
    """Actuator to apply screen graphic effects"""

    filter_pass: int
    """ Set filter order

    :type: int
    """

    glsl_shader: Text
    """ 

    :type: Text
    """

    mode: str
    """ 

    :type: str
    """

    motion_blur_factor: float
    """ Motion blur factor

    :type: float
    """

    use_motion_blur: bool
    """ Enable/Disable Motion Blur

    :type: bool
    """

class GameActuator(Actuator, bpy_struct):
    filename: str
    """ The file to use, depending on the mode (e.g. the blend file to load or a destination for saving a screenshot) - use the "//" prefix for a relative path

    :type: str
    """

    mode: str
    """ 

    :type: str
    """

class MessageActuator(Actuator, bpy_struct):
    body_message: str
    """ Optional, message body Text

    :type: str
    """

    body_property: str
    """ The message body will be set by the Property Value

    :type: str
    """

    body_type: str
    """ Toggle message type: either Text or a PropertyName

    :type: str
    """

    subject: str
    """ Optional, message subject (this is what can be filtered on)

    :type: str
    """

    to_property: str
    """ Optional, send message to objects with this name only, or empty to broadcast

    :type: str
    """

class MouseActuator(Actuator, bpy_struct):
    local_x: bool
    """ Apply rotation locally

    :type: bool
    """

    local_y: bool
    """ Apply rotation locally

    :type: bool
    """

    max_x: float
    """ Maximum positive rotation allowed by X mouse movement (0 for infinite)

    :type: float
    """

    max_y: float
    """ Maximum positive rotation allowed by Y mouse movement (0 for infinite)

    :type: float
    """

    min_x: float
    """ Maximum negative rotation allowed by X mouse movement (0 for infinite)

    :type: float
    """

    min_y: float
    """ Maximum negative rotation allowed by Y mouse movement (0 for infinite)

    :type: float
    """

    mode: str
    """ 

    :type: str
    """

    object_axis_x: str
    """ Local object axis mouse movement in the X direction will apply to

    :type: str
    """

    object_axis_y: str
    """ Local object axis mouse movement in the Y direction will apply to

    :type: str
    """

    reset_x: bool
    """ Reset the cursor's X position to the center of the screen space after calculating

    :type: bool
    """

    reset_y: bool
    """ Reset the cursor's Y position to the center of the screen space after calculating

    :type: bool
    """

    sensitivity_x: float
    """ Sensitivity of the X axis

    :type: float
    """

    sensitivity_y: float
    """ Sensitivity of the Y axis

    :type: float
    """

    threshold_x: float
    """ Amount of X motion before mouse movement will register

    :type: float
    """

    threshold_y: float
    """ Amount of Y motion before mouse movement will register

    :type: float
    """

    use_axis_x: bool
    """ Calculate mouse movement on the X axis

    :type: bool
    """

    use_axis_y: bool
    """ Calculate mouse movement on the Y axis

    :type: bool
    """

    visible: bool
    """ Make mouse cursor visible

    :type: bool
    """

class ObjectActuator(Actuator, bpy_struct):
    """Actuator to control the object movement"""

    angular_velocity: bpy_prop_array[float]
    """ Angular velocity

    :type: bpy_prop_array[float]
    """

    damping: int
    """ Number of frames to reach the target velocity

    :type: int
    """

    derivate_coefficient: float
    """ Not required, high values can cause instability

    :type: float
    """

    force: bpy_prop_array[float]
    """ Force

    :type: bpy_prop_array[float]
    """

    force_max_x: float
    """ Upper limit for X force

    :type: float
    """

    force_max_y: float
    """ Upper limit for Y force

    :type: float
    """

    force_max_z: float
    """ Upper limit for Z force

    :type: float
    """

    force_min_x: float
    """ Lower limit for X force

    :type: float
    """

    force_min_y: float
    """ Lower limit for Y force

    :type: float
    """

    force_min_z: float
    """ Lower limit for Z force

    :type: float
    """

    integral_coefficient: float
    """ Low value (0.01) for slow response, high value (0.5) for fast response

    :type: float
    """

    linear_velocity: bpy_prop_array[float]
    """ Linear velocity (in Servo mode it sets the target relative linear velocity, it will be achieved by automatic application of force - Null velocity is a valid target)

    :type: bpy_prop_array[float]
    """

    mode: str
    """ Specify the motion system

    :type: str
    """

    offset_location: bpy_prop_array[float]
    """ Location

    :type: bpy_prop_array[float]
    """

    offset_rotation: bpy_prop_array[float]
    """ Rotation

    :type: bpy_prop_array[float]
    """

    proportional_coefficient: float
    """ Typical value is 60x integral coefficient

    :type: float
    """

    reference_object: Object
    """ Reference object for velocity calculation, leave empty for world reference

    :type: Object
    """

    torque: bpy_prop_array[float]
    """ Torque

    :type: bpy_prop_array[float]
    """

    use_add_character_location: bool
    """ Toggle between ADD and SET character location

    :type: bool
    """

    use_add_linear_velocity: bool
    """ Toggles between ADD and SET linV

    :type: bool
    """

    use_character_jump: bool
    """ Make the character jump using the settings in the physics properties

    :type: bool
    """

    use_local_angular_velocity: bool
    """ Angular velocity is defined in local coordinates

    :type: bool
    """

    use_local_force: bool
    """ Force is defined in local coordinates

    :type: bool
    """

    use_local_linear_velocity: bool
    """ Velocity is defined in local coordinates

    :type: bool
    """

    use_local_location: bool
    """ Location is defined in local coordinates

    :type: bool
    """

    use_local_rotation: bool
    """ Rotation is defined in local coordinates

    :type: bool
    """

    use_local_torque: bool
    """ Torque is defined in local coordinates

    :type: bool
    """

    use_servo_limit_x: bool
    """ Set limit to force along the X axis

    :type: bool
    """

    use_servo_limit_y: bool
    """ Set limit to force along the Y axis

    :type: bool
    """

    use_servo_limit_z: bool
    """ Set limit to force along the Z axis

    :type: bool
    """

class ParentActuator(Actuator, bpy_struct):
    mode: str
    """ 

    :type: str
    """

    object: Object
    """ Set this object as parent

    :type: Object
    """

    use_compound: bool
    """ Add this object shape to the parent shape (only if the parent shape is already compound)

    :type: bool
    """

    use_ghost: bool
    """ Make this object ghost while parented

    :type: bool
    """

class PropertyActuator(Actuator, bpy_struct):
    """Actuator to handle properties"""

    mode: str
    """ 

    :type: str
    """

    object: Object
    """ Copy from this Object

    :type: Object
    """

    object_property: str
    """ Copy this property

    :type: str
    """

    property: str
    """ The name of the property

    :type: str
    """

    value: str
    """ The name of the property or the value to use (use "" around strings)

    :type: str
    """

class RandomActuator(Actuator, bpy_struct):
    chance: float
    """ Pick a number between 0 and 1, success if it's below this value

    :type: float
    """

    distribution: str
    """ Choose the type of distribution

    :type: str
    """

    float_max: float
    """ Choose a number from a range: upper boundary of the range

    :type: float
    """

    float_mean: float
    """ A normal distribution: mean of the distribution

    :type: float
    """

    float_min: float
    """ Choose a number from a range: lower boundary of the range

    :type: float
    """

    float_value: float
    """ Always return this number

    :type: float
    """

    half_life_time: float
    """ Negative exponential dropoff

    :type: float
    """

    int_max: int
    """ Choose a number from a range: upper boundary of the range

    :type: int
    """

    int_mean: float
    """ Expected mean value of the distribution

    :type: float
    """

    int_min: int
    """ Choose a number from a range: lower boundary of the range

    :type: int
    """

    int_value: int
    """ Always return this number

    :type: int
    """

    property: str
    """ Assign the random value to this property

    :type: str
    """

    seed: int
    """ Initial seed of the random generator, use Python for more freedom (choose 0 for not random)

    :type: int
    """

    standard_derivation: float
    """ A normal distribution: standard deviation of the distribution

    :type: float
    """

    use_always_true: bool
    """ Always false or always true

    :type: bool
    """

class SceneActuator(Actuator, bpy_struct):
    camera: Object
    """ Set this Camera (leave empty to refer to self object)

    :type: Object
    """

    mode: str
    """ 

    :type: str
    """

    scene: Scene
    """ Scene to be added/removed/paused/resumed

    :type: Scene
    """

class SoundActuator(Actuator, bpy_struct):
    """Sound file"""

    cone_inner_angle_3d: float
    """ The angle of the inner cone

    :type: float
    """

    cone_outer_angle_3d: float
    """ The angle of the outer cone

    :type: float
    """

    cone_outer_gain_3d: float
    """ The gain outside the outer cone (the gain in the outer cone will be interpolated between this value and the normal gain in the inner cone)

    :type: float
    """

    distance_3d_max: float
    """ The maximum distance at which you can hear the sound

    :type: float
    """

    distance_3d_reference: float
    """ The distance where the sound has a gain of 1.0

    :type: float
    """

    gain_3d_max: float
    """ The maximum gain of the sound, no matter how near it is

    :type: float
    """

    gain_3d_min: float
    """ The minimum gain of the sound, no matter how far it is away

    :type: float
    """

    mode: str
    """ 

    :type: str
    """

    pitch: float
    """ Pitch of the sound

    :type: float
    """

    rolloff_factor_3d: float
    """ The influence factor on volume depending on distance

    :type: float
    """

    sound: Sound
    """ 

    :type: Sound
    """

    use_sound_3d: bool
    """ Enable/Disable 3D Sound

    :type: bool
    """

    volume: float
    """ Initial volume of the sound

    :type: float
    """

class StateActuator(Actuator, bpy_struct):
    """Actuator to handle states"""

    operation: str
    """ Select the bit operation on object state mask

    :type: str
    """

    states: list[bool]
    """ 

    :type: list[bool]
    """

class SteeringActuator(Actuator, bpy_struct):
    acceleration: float
    """ Max acceleration

    :type: float
    """

    distance: float
    """ Relax distance

    :type: float
    """

    facing: bool
    """ Enable automatic facing

    :type: bool
    """

    facing_axis: str
    """ Axis for automatic facing

    :type: str
    """

    lock_z_velocity: bool
    """ Disable simulation of linear motion along Z axis

    :type: bool
    """

    mode: str
    """ 

    :type: str
    """

    navmesh: Object
    """ Navigation mesh

    :type: Object
    """

    normal_up: bool
    """ Use normal of the navmesh to set "UP" vector

    :type: bool
    """

    self_terminated: bool
    """ Terminate when target is reached

    :type: bool
    """

    show_visualization: bool
    """ Enable debug visualization for 'Path following'

    :type: bool
    """

    target: Object
    """ Target object

    :type: Object
    """

    turn_speed: float
    """ Max turn speed

    :type: float
    """

    update_period: int
    """ Path update period

    :type: int
    """

    velocity: float
    """ Velocity magnitude

    :type: float
    """

class VisibilityActuator(Actuator, bpy_struct):
    """Actuator to set visibility and occlusion of the object"""

    apply_to_children: bool
    """ Set all the children of this object to the same visibility/occlusion recursively

    :type: bool
    """

    use_occlusion: bool
    """ Set the object to occlude objects behind it (initialized from the object type in physics button)

    :type: bool
    """

    use_visible: bool
    """ Set the objects visible (initialized from the object render restriction toggle in physics button)

    :type: bool
    """

class CyclesPreferences(AddonPreferences, bpy_struct):
    compute_device_type: str
    """ Device to use for computation (rendering with Cycles)

    :type: str
    """

    devices: bpy_prop_collection[CyclesDeviceSettings]
    """ 

    :type: bpy_prop_collection[CyclesDeviceSettings]
    """

    def draw(self, context):
        """

        :param context:
        """
        ...

    def draw_impl(self, layout, context):
        """

        :param layout:
        :param context:
        """
        ...

    def get_device_types(self, context):
        """

        :param context:
        """
        ...

    def get_devices(self): ...
    def get_num_gpu_devices(self): ...
    def has_active_device(self): ...

class BoidRuleAverageSpeed(BoidRule, bpy_struct):
    level: float
    """ How much velocity's z-component is kept constant

    :type: float
    """

    speed: float
    """ Percentage of maximum speed

    :type: float
    """

    wander: float
    """ How fast velocity's direction is randomized

    :type: float
    """

class BoidRuleAvoid(BoidRule, bpy_struct):
    fear_factor: float
    """ Avoid object if danger from it is above this threshold

    :type: float
    """

    object: Object
    """ Object to avoid

    :type: Object
    """

    use_predict: bool
    """ Predict target movement

    :type: bool
    """

class BoidRuleAvoidCollision(BoidRule, bpy_struct):
    look_ahead: float
    """ Time to look ahead in seconds

    :type: float
    """

    use_avoid: bool
    """ Avoid collision with other boids

    :type: bool
    """

    use_avoid_collision: bool
    """ Avoid collision with deflector objects

    :type: bool
    """

class BoidRuleFight(BoidRule, bpy_struct):
    distance: float
    """ Attack boids at max this distance

    :type: float
    """

    flee_distance: float
    """ Flee to this distance

    :type: float
    """

class BoidRuleFollowLeader(BoidRule, bpy_struct):
    distance: float
    """ Distance behind leader to follow

    :type: float
    """

    object: Object
    """ Follow this object instead of a boid

    :type: Object
    """

    queue_count: int
    """ How many boids in a line

    :type: int
    """

    use_line: bool
    """ Follow leader in a line

    :type: bool
    """

class BoidRuleGoal(BoidRule, bpy_struct):
    object: Object
    """ Goal object

    :type: Object
    """

    use_predict: bool
    """ Predict target movement

    :type: bool
    """

class ActionConstraint(Constraint, bpy_struct):
    """Map an action to the transform axes of a bone"""

    action: Action
    """ The constraining action

    :type: Action
    """

    frame_end: int
    """ Last frame of the Action to use

    :type: int
    """

    frame_start: int
    """ First frame of the Action to use

    :type: int
    """

    max: float
    """ Maximum value for target channel range

    :type: float
    """

    min: float
    """ Minimum value for target channel range

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    transform_channel: str
    """ Transformation channel from the target that is used to key the Action

    :type: str
    """

    use_bone_object_action: bool
    """ Bones only: apply the object's transformation channels of the action to the constrained bone, instead of bone's channels

    :type: bool
    """

class CameraSolverConstraint(Constraint, bpy_struct):
    """Lock motion to the reconstructed camera movement"""

    clip: MovieClip
    """ Movie Clip to get tracking data from

    :type: MovieClip
    """

    use_active_clip: bool
    """ Use active clip defined in scene

    :type: bool
    """

class ChildOfConstraint(Constraint, bpy_struct):
    """Create constraint-based parent-child relationship"""

    inverse_matrix: bpy_prop_array[float]
    """ Transformation matrix to apply before

    :type: bpy_prop_array[float]
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_location_x: bool
    """ Use X Location of Parent

    :type: bool
    """

    use_location_y: bool
    """ Use Y Location of Parent

    :type: bool
    """

    use_location_z: bool
    """ Use Z Location of Parent

    :type: bool
    """

    use_rotation_x: bool
    """ Use X Rotation of Parent

    :type: bool
    """

    use_rotation_y: bool
    """ Use Y Rotation of Parent

    :type: bool
    """

    use_rotation_z: bool
    """ Use Z Rotation of Parent

    :type: bool
    """

    use_scale_x: bool
    """ Use X Scale of Parent

    :type: bool
    """

    use_scale_y: bool
    """ Use Y Scale of Parent

    :type: bool
    """

    use_scale_z: bool
    """ Use Z Scale of Parent

    :type: bool
    """

class ClampToConstraint(Constraint, bpy_struct):
    """Constrain an object's location to the nearest point along the target path"""

    main_axis: str
    """ Main axis of movement

    :type: str
    """

    target: Object
    """ Target Object (Curves only)

    :type: Object
    """

    use_cyclic: bool
    """ Treat curve as cyclic curve (no clamping to curve bounding box)

    :type: bool
    """

class CopyLocationConstraint(Constraint, bpy_struct):
    """Copy the location of the target"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    invert_x: bool
    """ Invert the X location

    :type: bool
    """

    invert_y: bool
    """ Invert the Y location

    :type: bool
    """

    invert_z: bool
    """ Invert the Z location

    :type: bool
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_offset: bool
    """ Add original location into copied location

    :type: bool
    """

    use_x: bool
    """ Copy the target's X location

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y location

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z location

    :type: bool
    """

class CopyRotationConstraint(Constraint, bpy_struct):
    """Copy the rotation of the target"""

    invert_x: bool
    """ Invert the X rotation

    :type: bool
    """

    invert_y: bool
    """ Invert the Y rotation

    :type: bool
    """

    invert_z: bool
    """ Invert the Z rotation

    :type: bool
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_offset: bool
    """ Add original rotation into copied rotation

    :type: bool
    """

    use_x: bool
    """ Copy the target's X rotation

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y rotation

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z rotation

    :type: bool
    """

class CopyScaleConstraint(Constraint, bpy_struct):
    """Copy the scale of the target"""

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_offset: bool
    """ Add original scale into copied scale

    :type: bool
    """

    use_x: bool
    """ Copy the target's X scale

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y scale

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z scale

    :type: bool
    """

class CopyTransformsConstraint(Constraint, bpy_struct):
    """Copy all the transforms of the target"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

class DampedTrackConstraint(Constraint, bpy_struct):
    """Point toward target by taking the shortest rotation path"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    track_axis: str
    """ Axis that points to the target object

    :type: str
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

class FloorConstraint(Constraint, bpy_struct):
    """Use the target object for location limitation"""

    floor_location: str
    """ Location of target that object will not pass through

    :type: str
    """

    offset: float
    """ Offset of floor from object origin

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_rotation: bool
    """ Use the target's rotation to determine floor

    :type: bool
    """

    use_sticky: bool
    """ Immobilize object while constrained

    :type: bool
    """

class FollowPathConstraint(Constraint, bpy_struct):
    """Lock motion to the target path"""

    forward_axis: str
    """ Axis that points forward along the path

    :type: str
    """

    offset: float
    """ Offset from the position corresponding to the time frame

    :type: float
    """

    offset_factor: float
    """ Percentage value defining target position along length of curve

    :type: float
    """

    target: Object
    """ Target Object

    :type: Object
    """

    up_axis: str
    """ Axis that points upward

    :type: str
    """

    use_curve_follow: bool
    """ Object will follow the heading and banking of the curve

    :type: bool
    """

    use_curve_radius: bool
    """ Object is scaled by the curve radius

    :type: bool
    """

    use_fixed_location: bool
    """ Object will stay locked to a single point somewhere along the length of the curve regardless of time

    :type: bool
    """

class FollowTrackConstraint(Constraint, bpy_struct):
    """Lock motion to the target motion track"""

    camera: Object
    """ Camera to which motion is parented (if empty active scene camera is used)

    :type: Object
    """

    clip: MovieClip
    """ Movie Clip to get tracking data from

    :type: MovieClip
    """

    depth_object: Object
    """ Object used to define depth in camera space by projecting onto surface of this object

    :type: Object
    """

    frame_method: str
    """ How the footage fits in the camera frame

    :type: str
    """

    object: str
    """ Movie tracking object to follow (if empty, camera object is used)

    :type: str
    """

    track: str
    """ Movie tracking track to follow

    :type: str
    """

    use_3d_position: bool
    """ Use 3D position of track to parent to

    :type: bool
    """

    use_active_clip: bool
    """ Use active clip defined in scene

    :type: bool
    """

    use_undistorted_position: bool
    """ Parent to undistorted position of 2D track

    :type: bool
    """

class KinematicConstraint(Constraint, bpy_struct):
    """Inverse Kinematics"""

    chain_count: int
    """ How many bones are included in the IK effect - 0 uses all bones

    :type: int
    """

    distance: float
    """ Radius of limiting sphere

    :type: float
    """

    ik_type: str
    """ 

    :type: str
    """

    iterations: int
    """ Maximum number of solving iterations

    :type: int
    """

    limit_mode: str
    """ Distances in relation to sphere of influence to allow

    :type: str
    """

    lock_location_x: bool
    """ Constraint position along X axis

    :type: bool
    """

    lock_location_y: bool
    """ Constraint position along Y axis

    :type: bool
    """

    lock_location_z: bool
    """ Constraint position along Z axis

    :type: bool
    """

    lock_rotation_x: bool
    """ Constraint rotation along X axis

    :type: bool
    """

    lock_rotation_y: bool
    """ Constraint rotation along Y axis

    :type: bool
    """

    lock_rotation_z: bool
    """ Constraint rotation along Z axis

    :type: bool
    """

    orient_weight: float
    """ For Tree-IK: Weight of orientation control for this target

    :type: float
    """

    pole_angle: float
    """ Pole rotation offset

    :type: float
    """

    pole_subtarget: str
    """ 

    :type: str
    """

    pole_target: Object
    """ Object for pole rotation

    :type: Object
    """

    reference_axis: str
    """ Constraint axis Lock options relative to Bone or Target reference

    :type: str
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_location: bool
    """ Chain follows position of target

    :type: bool
    """

    use_rotation: bool
    """ Chain follows rotation of target

    :type: bool
    """

    use_stretch: bool
    """ Enable IK Stretching

    :type: bool
    """

    use_tail: bool
    """ Include bone's tail as last element in chain

    :type: bool
    """

    weight: float
    """ For Tree-IK: Weight of position control for this target

    :type: float
    """

class LimitDistanceConstraint(Constraint, bpy_struct):
    """Limit the distance from target object"""

    distance: float
    """ Radius of limiting sphere

    :type: float
    """

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    limit_mode: str
    """ Distances in relation to sphere of influence to allow

    :type: str
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LimitLocationConstraint(Constraint, bpy_struct):
    """Limit the location of the constrained object"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    max_z: float
    """ Highest Z value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    min_z: float
    """ Lowest Z value to allow

    :type: float
    """

    use_max_x: bool
    """ Use the maximum X value

    :type: bool
    """

    use_max_y: bool
    """ Use the maximum Y value

    :type: bool
    """

    use_max_z: bool
    """ Use the maximum Z value

    :type: bool
    """

    use_min_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_min_y: bool
    """ Use the minimum Y value

    :type: bool
    """

    use_min_z: bool
    """ Use the minimum Z value

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LimitRotationConstraint(Constraint, bpy_struct):
    """Limit the rotation of the constrained object"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    max_z: float
    """ Highest Z value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    min_z: float
    """ Lowest Z value to allow

    :type: float
    """

    use_limit_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_limit_y: bool
    """ Use the minimum Y value

    :type: bool
    """

    use_limit_z: bool
    """ Use the minimum Z value

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LimitScaleConstraint(Constraint, bpy_struct):
    """Limit the scaling of the constrained object"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    max_z: float
    """ Highest Z value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    min_z: float
    """ Lowest Z value to allow

    :type: float
    """

    use_max_x: bool
    """ Use the maximum X value

    :type: bool
    """

    use_max_y: bool
    """ Use the maximum Y value

    :type: bool
    """

    use_max_z: bool
    """ Use the maximum Z value

    :type: bool
    """

    use_min_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_min_y: bool
    """ Use the minimum Y value

    :type: bool
    """

    use_min_z: bool
    """ Use the minimum Z value

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LockedTrackConstraint(Constraint, bpy_struct):
    """Point toward the target along the track axis, while locking the other axis"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    lock_axis: str
    """ Axis that points upward

    :type: str
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    track_axis: str
    """ Axis that points to the target object

    :type: str
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

class MaintainVolumeConstraint(Constraint, bpy_struct):
    """Maintain a constant volume along a single scaling axis"""

    free_axis: str
    """ The free scaling axis of the object

    :type: str
    """

    volume: float
    """ Volume of the bone at rest

    :type: float
    """

class ObjectSolverConstraint(Constraint, bpy_struct):
    """Lock motion to the reconstructed object movement"""

    camera: Object
    """ Camera to which motion is parented (if empty active scene camera is used)

    :type: Object
    """

    clip: MovieClip
    """ Movie Clip to get tracking data from

    :type: MovieClip
    """

    object: str
    """ Movie tracking object to follow

    :type: str
    """

    use_active_clip: bool
    """ Use active clip defined in scene

    :type: bool
    """

class PivotConstraint(Constraint, bpy_struct):
    """Rotate around a different point"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    offset: bpy_prop_array[float]
    """ Offset of pivot from target (when set), or from owner's location (when Fixed Position is off), or the absolute pivot point

    :type: bpy_prop_array[float]
    """

    rotation_range: str
    """ Rotation range on which pivoting should occur

    :type: str
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object, defining the position of the pivot when defined

    :type: Object
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_relative_location: bool
    """ Offset will be an absolute point in space instead of relative to the target

    :type: bool
    """

class PythonConstraint(Constraint, bpy_struct):
    """Use Python script for constraint evaluation"""

    has_script_error: bool
    """ The linked Python script has thrown an error

    :type: bool
    """

    target_count: int
    """ Usually only 1-3 are needed

    :type: int
    """

    targets: bpy_prop_collection[ConstraintTarget]
    """ Target Objects

    :type: bpy_prop_collection[ConstraintTarget]
    """

    text: Text
    """ The text object that contains the Python script

    :type: Text
    """

    use_targets: bool
    """ Use the targets indicated in the constraint panel

    :type: bool
    """

class RigidBodyJointConstraint(Constraint, bpy_struct):
    """For use with the Game Engine"""

    axis_x: float
    """ Rotate pivot on X axis

    :type: float
    """

    axis_y: float
    """ Rotate pivot on Y axis

    :type: float
    """

    axis_z: float
    """ Rotate pivot on Z axis

    :type: float
    """

    child: Object
    """ Child object

    :type: Object
    """

    limit_angle_max_x: float
    """ 

    :type: float
    """

    limit_angle_max_y: float
    """ 

    :type: float
    """

    limit_angle_max_z: float
    """ 

    :type: float
    """

    limit_angle_min_x: float
    """ 

    :type: float
    """

    limit_angle_min_y: float
    """ 

    :type: float
    """

    limit_angle_min_z: float
    """ 

    :type: float
    """

    limit_max_x: float
    """ 

    :type: float
    """

    limit_max_y: float
    """ 

    :type: float
    """

    limit_max_z: float
    """ 

    :type: float
    """

    limit_min_x: float
    """ 

    :type: float
    """

    limit_min_y: float
    """ 

    :type: float
    """

    limit_min_z: float
    """ 

    :type: float
    """

    pivot_type: str
    """ 

    :type: str
    """

    pivot_x: float
    """ Offset pivot on X

    :type: float
    """

    pivot_y: float
    """ Offset pivot on Y

    :type: float
    """

    pivot_z: float
    """ Offset pivot on Z

    :type: float
    """

    show_pivot: bool
    """ Display the pivot point and rotation in 3D view

    :type: bool
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_angular_limit_x: bool
    """ Use minimum/maximum X angular limit

    :type: bool
    """

    use_angular_limit_y: bool
    """ Use minimum/maximum Y angular limit

    :type: bool
    """

    use_angular_limit_z: bool
    """ Use minimum/maximum Z angular limit

    :type: bool
    """

    use_limit_x: bool
    """ Use minimum/maximum X limit

    :type: bool
    """

    use_limit_y: bool
    """ Use minimum/maximum y limit

    :type: bool
    """

    use_limit_z: bool
    """ Use minimum/maximum z limit

    :type: bool
    """

    use_linked_collision: bool
    """ Disable collision between linked bodies

    :type: bool
    """

class ShrinkwrapConstraint(Constraint, bpy_struct):
    """Create constraint-based shrinkwrap relationship"""

    distance: float
    """ Distance to Target

    :type: float
    """

    project_axis: str
    """ Axis constrain to

    :type: str
    """

    project_axis_space: str
    """ Space for the projection axis

    :type: str
    """

    project_limit: float
    """ Limit the distance used for projection (zero disables)

    :type: float
    """

    shrinkwrap_type: str
    """ Select type of shrinkwrap algorithm for target position

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

class SplineIKConstraint(Constraint, bpy_struct):
    """Align 'n' bones along a curve"""

    bulge: float
    """ Factor between volume variation and stretching

    :type: float
    """

    bulge_max: float
    """ Maximum volume stretching factor

    :type: float
    """

    bulge_min: float
    """ Minimum volume stretching factor

    :type: float
    """

    bulge_smooth: float
    """ Strength of volume stretching clamping

    :type: float
    """

    chain_count: int
    """ How many bones are included in the chain

    :type: int
    """

    joint_bindings: bpy_prop_array[float]
    """ (EXPERIENCED USERS ONLY) The relative positions of the joints along the chain, as percentages

    :type: bpy_prop_array[float]
    """

    target: Object
    """ Curve that controls this relationship

    :type: Object
    """

    use_bulge_max: bool
    """ Use upper limit for volume variation

    :type: bool
    """

    use_bulge_min: bool
    """ Use lower limit for volume variation

    :type: bool
    """

    use_chain_offset: bool
    """ Offset the entire chain relative to the root joint

    :type: bool
    """

    use_curve_radius: bool
    """ Average radius of the endpoints is used to tweak the X and Z Scaling of the bones, on top of XZ Scale mode

    :type: bool
    """

    use_even_divisions: bool
    """ Ignore the relative lengths of the bones when fitting to the curve

    :type: bool
    """

    use_y_stretch: bool
    """ Stretch the Y axis of the bones to fit the curve

    :type: bool
    """

    xz_scale_mode: str
    """ Method used for determining the scaling of the X and Z axes of the bones

    :type: str
    """

class StretchToConstraint(Constraint, bpy_struct):
    """Stretch to meet the target object"""

    bulge: float
    """ Factor between volume variation and stretching

    :type: float
    """

    bulge_max: float
    """ Maximum volume stretching factor

    :type: float
    """

    bulge_min: float
    """ Minimum volume stretching factor

    :type: float
    """

    bulge_smooth: float
    """ Strength of volume stretching clamping

    :type: float
    """

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    keep_axis: str
    """ Axis to maintain during stretch

    :type: str
    """

    rest_length: float
    """ Length at rest position

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_bulge_max: bool
    """ Use upper limit for volume variation

    :type: bool
    """

    use_bulge_min: bool
    """ Use lower limit for volume variation

    :type: bool
    """

    volume: str
    """ Maintain the object's volume as it stretches

    :type: str
    """

class TrackToConstraint(Constraint, bpy_struct):
    """Aim the constrained object toward the target"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    track_axis: str
    """ Axis that points to the target object

    :type: str
    """

    up_axis: str
    """ Axis that points upward

    :type: str
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_target_z: bool
    """ Target's Z axis, not World Z axis, will constraint the Up direction

    :type: bool
    """

class TransformCacheConstraint(Constraint, bpy_struct):
    """Look up transformation from an external file"""

    cache_file: CacheFile
    """ 

    :type: CacheFile
    """

    object_path: str
    """ Path to the object in the Alembic archive used to lookup the transform matrix

    :type: str
    """

class TransformConstraint(Constraint, bpy_struct):
    """Map transformations of the target to the object"""

    from_max_x: float
    """ Top range of X axis source motion

    :type: float
    """

    from_max_x_rot: float
    """ Top range of X axis source motion

    :type: float
    """

    from_max_x_scale: float
    """ Top range of X axis source motion

    :type: float
    """

    from_max_y: float
    """ Top range of Y axis source motion

    :type: float
    """

    from_max_y_rot: float
    """ Top range of Y axis source motion

    :type: float
    """

    from_max_y_scale: float
    """ Top range of Y axis source motion

    :type: float
    """

    from_max_z: float
    """ Top range of Z axis source motion

    :type: float
    """

    from_max_z_rot: float
    """ Top range of Z axis source motion

    :type: float
    """

    from_max_z_scale: float
    """ Top range of Z axis source motion

    :type: float
    """

    from_min_x: float
    """ Bottom range of X axis source motion

    :type: float
    """

    from_min_x_rot: float
    """ Bottom range of X axis source motion

    :type: float
    """

    from_min_x_scale: float
    """ Bottom range of X axis source motion

    :type: float
    """

    from_min_y: float
    """ Bottom range of Y axis source motion

    :type: float
    """

    from_min_y_rot: float
    """ Bottom range of Y axis source motion

    :type: float
    """

    from_min_y_scale: float
    """ Bottom range of Y axis source motion

    :type: float
    """

    from_min_z: float
    """ Bottom range of Z axis source motion

    :type: float
    """

    from_min_z_rot: float
    """ Bottom range of Z axis source motion

    :type: float
    """

    from_min_z_scale: float
    """ Bottom range of Z axis source motion

    :type: float
    """

    map_from: str
    """ The transformation type to use from the target

    :type: str
    """

    map_to: str
    """ The transformation type to affect of the constrained object

    :type: str
    """

    map_to_x_from: str
    """ The source axis constrained object's X axis uses

    :type: str
    """

    map_to_y_from: str
    """ The source axis constrained object's Y axis uses

    :type: str
    """

    map_to_z_from: str
    """ The source axis constrained object's Z axis uses

    :type: str
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object
    """ Target Object

    :type: Object
    """

    to_max_x: float
    """ Top range of X axis destination motion

    :type: float
    """

    to_max_x_rot: float
    """ Top range of X axis destination motion

    :type: float
    """

    to_max_x_scale: float
    """ Top range of X axis destination motion

    :type: float
    """

    to_max_y: float
    """ Top range of Y axis destination motion

    :type: float
    """

    to_max_y_rot: float
    """ Top range of Y axis destination motion

    :type: float
    """

    to_max_y_scale: float
    """ Top range of Y axis destination motion

    :type: float
    """

    to_max_z: float
    """ Top range of Z axis destination motion

    :type: float
    """

    to_max_z_rot: float
    """ Top range of Z axis destination motion

    :type: float
    """

    to_max_z_scale: float
    """ Top range of Z axis destination motion

    :type: float
    """

    to_min_x: float
    """ Bottom range of X axis destination motion

    :type: float
    """

    to_min_x_rot: float
    """ Bottom range of X axis destination motion

    :type: float
    """

    to_min_x_scale: float
    """ Bottom range of X axis destination motion

    :type: float
    """

    to_min_y: float
    """ Bottom range of Y axis destination motion

    :type: float
    """

    to_min_y_rot: float
    """ Bottom range of Y axis destination motion

    :type: float
    """

    to_min_y_scale: float
    """ Bottom range of Y axis destination motion

    :type: float
    """

    to_min_z: float
    """ Bottom range of Z axis destination motion

    :type: float
    """

    to_min_z_rot: float
    """ Bottom range of Z axis destination motion

    :type: float
    """

    to_min_z_scale: float
    """ Bottom range of Z axis destination motion

    :type: float
    """

    use_motion_extrapolate: bool
    """ Extrapolate ranges

    :type: bool
    """

class AndController(Controller, bpy_struct):
    """Controller passing on events based on a logical AND operation"""

    ...

class ExpressionController(Controller, bpy_struct):
    """Controller passing on events based on the evaluation of an expression"""

    expression: str
    """ 

    :type: str
    """

class NandController(Controller, bpy_struct):
    """Controller passing on events based on a logical NAND operation"""

    ...

class NorController(Controller, bpy_struct):
    """Controller passing on events based on a logical NOR operation"""

    ...

class OrController(Controller, bpy_struct):
    """Controller passing on events based on a logical OR operation"""

    ...

class PythonController(Controller, bpy_struct):
    """Controller executing a python script"""

    mode: str
    """ Python script type (textblock or module - faster)

    :type: str
    """

    module: str
    """ Module name and function to run, e.g. "someModule.main" (internal texts and external python files can be used)

    :type: str
    """

    text: Text
    """ Text data-block with the python script

    :type: Text
    """

    use_debug: bool
    """ Continuously reload the module from disk for editing external modules without restarting

    :type: bool
    """

class XnorController(Controller, bpy_struct):
    """Controller passing on events based on a logical XNOR operation"""

    ...

class XorController(Controller, bpy_struct):
    """Controller passing on events based on a logical XOR operation"""

    ...

class FModifierCycles(FModifier, bpy_struct):
    """Repeat the values of the modified F-Curve"""

    cycles_after: int
    """ Maximum number of cycles to allow after last keyframe (0 = infinite)

    :type: int
    """

    cycles_before: int
    """ Maximum number of cycles to allow before first keyframe (0 = infinite)

    :type: int
    """

    mode_after: str
    """ Cycling mode to use after last keyframe

    :type: str
    """

    mode_before: str
    """ Cycling mode to use before first keyframe

    :type: str
    """

class FModifierEnvelope(FModifier, bpy_struct):
    """Scale the values of the modified F-Curve"""

    control_points: FModifierEnvelopeControlPoints
    """ Control points defining the shape of the envelope

    :type: FModifierEnvelopeControlPoints
    """

    default_max: float
    """ Upper distance from Reference Value for 1:1 default influence

    :type: float
    """

    default_min: float
    """ Lower distance from Reference Value for 1:1 default influence

    :type: float
    """

    reference_value: float
    """ Value that envelope's influence is centered around / based on

    :type: float
    """

class FModifierFunctionGenerator(FModifier, bpy_struct):
    """Generate values using a Built-In Function"""

    amplitude: float
    """ Scale factor determining the maximum/minimum values

    :type: float
    """

    function_type: str
    """ Type of built-in function to use

    :type: str
    """

    phase_multiplier: float
    """ Scale factor determining the 'speed' of the function

    :type: float
    """

    phase_offset: float
    """ Constant factor to offset time by for function

    :type: float
    """

    use_additive: bool
    """ Values generated by this modifier are applied on top of the existing values instead of overwriting them

    :type: bool
    """

    value_offset: float
    """ Constant factor to offset values by

    :type: float
    """

class FModifierGenerator(FModifier, bpy_struct):
    """Deterministically generate values for the modified F-Curve"""

    coefficients: bpy_prop_array[float]
    """ Coefficients for 'x' (starting from lowest power of x^0)

    :type: bpy_prop_array[float]
    """

    mode: str
    """ Type of generator to use

    :type: str
    """

    poly_order: int
    """ The highest power of 'x' for this polynomial (number of coefficients - 1)

    :type: int
    """

    use_additive: bool
    """ Values generated by this modifier are applied on top of the existing values instead of overwriting them

    :type: bool
    """

class FModifierLimits(FModifier, bpy_struct):
    """Limit the time/value ranges of the modified F-Curve"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    use_max_x: bool
    """ Use the maximum X value

    :type: bool
    """

    use_max_y: bool
    """ Use the maximum Y value

    :type: bool
    """

    use_min_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_min_y: bool
    """ Use the minimum Y value

    :type: bool
    """

class FModifierNoise(FModifier, bpy_struct):
    """Give randomness to the modified F-Curve"""

    blend_type: str
    """ Method of modifying the existing F-Curve

    :type: str
    """

    depth: int
    """ Amount of fine level detail present in the noise

    :type: int
    """

    offset: float
    """ Time offset for the noise effect

    :type: float
    """

    phase: float
    """ A random seed for the noise effect

    :type: float
    """

    scale: float
    """ Scaling (in time) of the noise

    :type: float
    """

    strength: float
    """ Amplitude of the noise - the amount that it modifies the underlying curve

    :type: float
    """

class FModifierPython(FModifier, bpy_struct):
    """Perform user-defined operation on the modified F-Curve"""

    ...

class FModifierStepped(FModifier, bpy_struct):
    """Hold each interpolated value from the F-Curve for several frames without changing the timing"""

    frame_end: float
    """ Frame that modifier's influence ends (if applicable)

    :type: float
    """

    frame_offset: float
    """ Reference number of frames before frames get held (use to get hold for '1-3' vs '5-7' holding patterns)

    :type: float
    """

    frame_start: float
    """ Frame that modifier's influence starts (if applicable)

    :type: float
    """

    frame_step: float
    """ Number of frames to hold each value

    :type: float
    """

    use_frame_end: bool
    """ Restrict modifier to only act before its 'end' frame

    :type: bool
    """

    use_frame_start: bool
    """ Restrict modifier to only act after its 'start' frame

    :type: bool
    """

class ControlFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for objects controlling the motion of fluid in the simulation"""

    attraction_radius: float
    """ Force field radius around the control object

    :type: float
    """

    attraction_strength: float
    """ Force strength for directional attraction towards the control object

    :type: float
    """

    end_time: float
    """ Time when the control particles are deactivated

    :type: float
    """

    quality: float
    """ Quality which is used for object sampling (higher = better but slower)

    :type: float
    """

    start_time: float
    """ Time when the control particles are activated

    :type: float
    """

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse control object movement

    :type: bool
    """

    velocity_radius: float
    """ Force field radius around the control object

    :type: float
    """

    velocity_strength: float
    """ Force strength of how much of the control object's velocity is influencing the fluid velocity

    :type: float
    """

class DomainFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for the domain of a fluid simulation"""

    compressibility: float
    """ Allowed compressibility due to gravitational force for standing fluid (directly affects simulation step size)

    :type: float
    """

    end_time: float
    """ Simulation time of the last blender frame (in seconds)

    :type: float
    """

    filepath: str
    """ Directory (and/or filename prefix) to store baked fluid simulation files in

    :type: str
    """

    fluid_mesh_vertices: bpy_prop_collection[FluidVertexVelocity]
    """ Vertices of the fluid mesh generated by simulation

    :type: bpy_prop_collection[FluidVertexVelocity]
    """

    frame_offset: int
    """ Offset when reading baked cache

    :type: int
    """

    generate_particles: float
    """ Amount of particles to generate (0=off, 1=normal, >1=more)

    :type: float
    """

    gravity: bpy_prop_array[float]
    """ Gravity in X, Y and Z direction

    :type: bpy_prop_array[float]
    """

    grid_levels: int
    """ Number of coarsened grids to use (-1 for automatic)

    :type: int
    """

    memory_estimate: str
    """ Estimated amount of memory needed for baking the domain

    :type: str
    """

    partial_slip_factor: float
    """ Amount of mixing between no- and free-slip, 0 is no slip and 1 is free slip

    :type: float
    """

    preview_resolution: int
    """ Preview resolution in X,Y and Z direction

    :type: int
    """

    render_display_mode: str
    """ How to display the mesh for rendering

    :type: str
    """

    resolution: int
    """ Domain resolution in X,Y and Z direction

    :type: int
    """

    simulation_rate: float
    """ Fluid motion rate (0 = stationary, 1 = normal speed)

    :type: float
    """

    simulation_scale: float
    """ Size of the simulation domain in meters

    :type: float
    """

    slip_type: str
    """ 

    :type: str
    """

    start_time: float
    """ Simulation time of the first blender frame (in seconds)

    :type: float
    """

    surface_smooth: float
    """ Amount of surface smoothing (a value of 0 is off, 1 is normal smoothing and more than 1 is extra smoothing)

    :type: float
    """

    surface_subdivisions: int
    """ Number of isosurface subdivisions (this is necessary for the inclusion of particles into the surface generation - WARNING: can lead to longer computation times !)

    :type: int
    """

    threads: int
    """ Override number of threads for the simulation, 0 is automatic

    :type: int
    """

    tracer_particles: int
    """ Number of tracer particles to generate

    :type: int
    """

    use_reverse_frames: bool
    """ Reverse fluid frames

    :type: bool
    """

    use_speed_vectors: bool
    """ Generate speed vectors for vector blur

    :type: bool
    """

    use_surface_noobs: bool
    """ Removes the air gap between fluid surface and obstacles - WARNING: Can result in a dissolving surface in other areas

    :type: bool
    """

    use_time_override: bool
    """ Use a custom start and end time (in seconds) instead of the scene's timeline

    :type: bool
    """

    viewport_display_mode: str
    """ How to display the mesh in the viewport

    :type: str
    """

    viscosity_base: float
    """ Viscosity setting: value that is multiplied by 10 to the power of (exponent*-1)

    :type: float
    """

    viscosity_exponent: int
    """ Negative exponent for the viscosity value (to simplify entering small values e.g. 5*10^-6)

    :type: int
    """

class FluidFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for the fluid in the simulation"""

    initial_velocity: bpy_prop_array[float]
    """ Initial velocity of fluid

    :type: bpy_prop_array[float]
    """

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_animated_mesh: bool
    """ Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it)

    :type: bool
    """

    volume_initialization: str
    """ Volume initialization type (WARNING: complex volumes might require too much memory and break simulation)

    :type: str
    """

class InflowFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for objects adding fluids in the simulation"""

    inflow_velocity: bpy_prop_array[float]
    """ Initial velocity of fluid

    :type: bpy_prop_array[float]
    """

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_animated_mesh: bool
    """ Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it)

    :type: bool
    """

    use_local_coords: bool
    """ Use local coordinates for inflow (e.g. for rotating objects)

    :type: bool
    """

    volume_initialization: str
    """ Volume initialization type (WARNING: complex volumes might require too much memory and break simulation)

    :type: str
    """

class ObstacleFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for obstacles in the simulation"""

    impact_factor: float
    """ This is an unphysical value for moving objects - it controls the impact an obstacle has on the fluid, =0 behaves a bit like outflow (deleting fluid), =1 is default, while >1 results in high forces (can be used to tweak total mass)

    :type: float
    """

    partial_slip_factor: float
    """ Amount of mixing between no- and free-slip, 0 is no slip and 1 is free slip

    :type: float
    """

    slip_type: str
    """ 

    :type: str
    """

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_animated_mesh: bool
    """ Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it)

    :type: bool
    """

    volume_initialization: str
    """ Volume initialization type (WARNING: complex volumes might require too much memory and break simulation)

    :type: str
    """

class OutflowFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for objects removing fluids from the simulation"""

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_animated_mesh: bool
    """ Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it)

    :type: bool
    """

    volume_initialization: str
    """ Volume initialization type (WARNING: complex volumes might require too much memory and break simulation)

    :type: str
    """

class ParticleFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for objects storing fluid particles generated by the simulation"""

    alpha_influence: float
    """ Amount of particle alpha change, inverse of size influence: 0=off (all same alpha), 1=full (larger particles get lower alphas, smaller ones higher values)

    :type: float
    """

    filepath: str
    """ Directory (and/or filename prefix) to store and load particles from

    :type: str
    """

    particle_influence: float
    """ Amount of particle size scaling: 0=off (all same size), 1=full (range 0.2-2.0), >1=stronger

    :type: float
    """

    show_tracer: bool
    """ Show tracer particles

    :type: bool
    """

    use_drops: bool
    """ Show drop particles

    :type: bool
    """

    use_floats: bool
    """ Show floating foam particles

    :type: bool
    """

class GameBooleanProperty(GameProperty, bpy_struct):
    """Game engine user defined Boolean property"""

    value: bool
    """ Property value

    :type: bool
    """

class GameFloatProperty(GameProperty, bpy_struct):
    """Game engine user defined floating point number property"""

    value: float
    """ Property value

    :type: float
    """

class GameIntProperty(GameProperty, bpy_struct):
    """Game engine user defined integer number property"""

    value: int
    """ Property value

    :type: int
    """

class GameStringProperty(GameProperty, bpy_struct):
    """Game engine user defined text string property"""

    value: str
    """ Property value

    :type: str
    """

class GameTimerProperty(GameProperty, bpy_struct):
    """Game engine user defined timer property"""

    value: float
    """ Property value

    :type: float
    """

class Action(ID, bpy_struct):
    """A collection of F-Curves for animation"""

    fcurves: ActionFCurves
    """ The individual F-Curves that make up the action

    :type: ActionFCurves
    """

    frame_range: bpy_prop_array[float]
    """ The final frame range of all F-Curves within this action

    :type: bpy_prop_array[float]
    """

    groups: ActionGroups
    """ Convenient groupings of F-Curves

    :type: ActionGroups
    """

    id_root: str
    """ Type of ID block that action can be used on - DO NOT CHANGE UNLESS YOU KNOW WHAT YOU ARE DOING

    :type: str
    """

    pose_markers: ActionPoseMarkers
    """ Markers specific to this action, for labeling poses

    :type: ActionPoseMarkers
    """

class Armature(ID, bpy_struct):
    """Armature data-block containing a hierarchy of bones, usually used for rigging characters"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    bones: ArmatureBones
    """ 

    :type: ArmatureBones
    """

    deform_method: str
    """ Vertex Deformer Method (Game Engine only)

    :type: str
    """

    draw_type: str
    """ 

    :type: str
    """

    edit_bones: ArmatureEditBones
    """ 

    :type: ArmatureEditBones
    """

    ghost_frame_end: int
    """ End frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    ghost_frame_start: int
    """ Starting frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    ghost_size: int
    """ Frame step for Ghosts (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    ghost_step: int
    """ Number of frame steps on either side of current frame to show as ghosts (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    ghost_type: str
    """ Method of Onion-skinning for active Action

    :type: str
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    layers: list[bool]
    """ Armature layer visibility

    :type: list[bool]
    """

    layers_protected: list[bool]
    """ Protected layers in Proxy Instances are restored to Proxy settings on file reload and undo

    :type: list[bool]
    """

    pose_position: str
    """ Show armature in binding pose or final posed state

    :type: str
    """

    show_axes: bool
    """ Draw bone axes

    :type: bool
    """

    show_bone_custom_shapes: bool
    """ Draw bones with their custom shapes

    :type: bool
    """

    show_group_colors: bool
    """ Draw bone group colors

    :type: bool
    """

    show_names: bool
    """ Draw bone names

    :type: bool
    """

    show_only_ghost_selected: bool
    """ 

    :type: bool
    """

    use_auto_ik: bool
    """ Add temporary IK constraints while grabbing bones in Pose Mode

    :type: bool
    """

    use_deform_delay: bool
    """ Don't deform children when manipulating bones in Pose Mode

    :type: bool
    """

    use_mirror_x: bool
    """ Apply changes to matching bone on opposite side of X-Axis

    :type: bool
    """

    def transform(self, matrix: collections.abc.Iterable[float] | None):
        """Transform armature bones by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        """
        ...

class Brush(ID, bpy_struct):
    """Brush data-block for storing brush settings for painting and sculpting"""

    auto_smooth_factor: float
    """ Amount of smoothing to automatically apply to each stroke

    :type: float
    """

    blend: str
    """ Brush blending mode

    :type: str
    """

    blur_kernel_radius: int
    """ Radius of kernel used for soften and sharpen in pixels

    :type: int
    """

    blur_mode: str
    """ 

    :type: str
    """

    brush_capabilities: BrushCapabilities
    """ Brush's capabilities

    :type: BrushCapabilities
    """

    clone_alpha: float
    """ Opacity of clone image display

    :type: float
    """

    clone_image: Image
    """ Image for clone tool

    :type: Image
    """

    clone_offset: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    crease_pinch_factor: float
    """ How much the crease brush pinches

    :type: float
    """

    cursor_color_add: bpy_prop_array[float]
    """ Color of cursor when adding

    :type: bpy_prop_array[float]
    """

    cursor_color_subtract: bpy_prop_array[float]
    """ Color of cursor when subtracting

    :type: bpy_prop_array[float]
    """

    cursor_overlay_alpha: int
    """ 

    :type: int
    """

    curve: CurveMapping
    """ Editable falloff curve

    :type: CurveMapping
    """

    direction: str
    """ 

    :type: str
    """

    fill_threshold: float
    """ Threshold above which filling is not propagated

    :type: float
    """

    grad_spacing: int
    """ Spacing before brush gradient goes full circle

    :type: int
    """

    gradient: ColorRamp
    """ 

    :type: ColorRamp
    """

    gradient_fill_mode: str
    """ 

    :type: str
    """

    gradient_stroke_mode: str
    """ 

    :type: str
    """

    height: float
    """ Affectable height of brush (layer height for layer tool, i.e.)

    :type: float
    """

    icon_filepath: str
    """ File path to brush icon

    :type: str
    """

    image_paint_capabilities: ImapaintToolCapabilities
    """ Brush's capabilities in image paint mode

    :type: ImapaintToolCapabilities
    """

    image_tool: str
    """ 

    :type: str
    """

    jitter: float
    """ Jitter the position of the brush while painting

    :type: float
    """

    jitter_absolute: int
    """ Jitter the position of the brush in pixels while painting

    :type: int
    """

    mask_overlay_alpha: int
    """ 

    :type: int
    """

    mask_stencil_dimension: bpy_prop_array[float]
    """ Dimensions of mask stencil in viewport

    :type: bpy_prop_array[float]
    """

    mask_stencil_pos: bpy_prop_array[float]
    """ Position of mask stencil in viewport

    :type: bpy_prop_array[float]
    """

    mask_texture: Texture
    """ 

    :type: Texture
    """

    mask_texture_slot: BrushTextureSlot
    """ 

    :type: BrushTextureSlot
    """

    mask_tool: str
    """ 

    :type: str
    """

    normal_weight: float
    """ How much grab will pull vertexes out of surface during a grab

    :type: float
    """

    paint_curve: PaintCurve
    """ Active Paint Curve

    :type: PaintCurve
    """

    plane_offset: float
    """ Adjust plane on which the brush acts towards or away from the object surface

    :type: float
    """

    plane_trim: float
    """ If a vertex is further away from offset plane than this, then it is not affected

    :type: float
    """

    rake_factor: float
    """ How much grab will follow cursor rotation

    :type: float
    """

    rate: float
    """ Interval between paints for Airbrush

    :type: float
    """

    sculpt_capabilities: SculptToolCapabilities
    """ Brush's capabilities in sculpt mode

    :type: SculptToolCapabilities
    """

    sculpt_plane: str
    """ 

    :type: str
    """

    sculpt_tool: str
    """ 

    :type: str
    """

    secondary_color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    sharp_threshold: float
    """ Threshold below which, no sharpening is done

    :type: float
    """

    size: int
    """ Radius of the brush in pixels

    :type: int
    """

    smooth_stroke_factor: float
    """ Higher values give a smoother stroke

    :type: float
    """

    smooth_stroke_radius: int
    """ Minimum distance from last point before stroke continues

    :type: int
    """

    spacing: int
    """ Spacing between brush daubs as a percentage of brush diameter

    :type: int
    """

    stencil_dimension: bpy_prop_array[float]
    """ Dimensions of stencil in viewport

    :type: bpy_prop_array[float]
    """

    stencil_pos: bpy_prop_array[float]
    """ Position of stencil in viewport

    :type: bpy_prop_array[float]
    """

    strength: float
    """ How powerful the effect of the brush is when applied

    :type: float
    """

    stroke_method: str
    """ 

    :type: str
    """

    texture: Texture
    """ 

    :type: Texture
    """

    texture_overlay_alpha: int
    """ 

    :type: int
    """

    texture_sample_bias: float
    """ Value added to texture samples

    :type: float
    """

    texture_slot: BrushTextureSlot
    """ 

    :type: BrushTextureSlot
    """

    unprojected_radius: float
    """ Radius of brush in Blender units

    :type: float
    """

    use_accumulate: bool
    """ Accumulate stroke daubs on top of each other

    :type: bool
    """

    use_adaptive_space: bool
    """ Space daubs according to surface orientation instead of screen space

    :type: bool
    """

    use_airbrush: bool
    """ Keep applying paint effect while holding mouse (spray)

    :type: bool
    """

    use_alpha: bool
    """ When this is disabled, lock alpha while painting

    :type: bool
    """

    use_anchor: bool
    """ Keep the brush anchored to the initial location

    :type: bool
    """

    use_cursor_overlay: bool
    """ Show cursor in viewport

    :type: bool
    """

    use_cursor_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_curve: bool
    """ Define the stroke curve with a bezier curve. Dabs are separated according to spacing

    :type: bool
    """

    use_custom_icon: bool
    """ Set the brush icon from an image file

    :type: bool
    """

    use_edge_to_edge: bool
    """ Drag anchor brush from edge-to-edge

    :type: bool
    """

    use_frontface: bool
    """ Brush only affects vertexes that face the viewer

    :type: bool
    """

    use_gradient: bool
    """ Use Gradient by utilizing a sampling method

    :type: bool
    """

    use_inverse_smooth_pressure: bool
    """ Lighter pressure causes more smoothing to be applied

    :type: bool
    """

    use_line: bool
    """ Draw a line with dabs separated according to spacing

    :type: bool
    """

    use_locked_size: bool
    """ When locked brush stays same size relative to object; when unlocked brush size is given in pixels

    :type: bool
    """

    use_offset_pressure: bool
    """ Enable tablet pressure sensitivity for offset

    :type: bool
    """

    use_original_normal: bool
    """ When locked keep using normal of surface where stroke was initiated

    :type: bool
    """

    use_paint_image: bool
    """ Use this brush in texture paint mode

    :type: bool
    """

    use_paint_sculpt: bool
    """ Use this brush in sculpt mode

    :type: bool
    """

    use_paint_vertex: bool
    """ Use this brush in vertex paint mode

    :type: bool
    """

    use_paint_weight: bool
    """ Use this brush in weight paint mode

    :type: bool
    """

    use_persistent: bool
    """ Sculpt on a persistent layer of the mesh

    :type: bool
    """

    use_plane_trim: bool
    """ Enable Plane Trim

    :type: bool
    """

    use_pressure_jitter: bool
    """ Enable tablet pressure sensitivity for jitter

    :type: bool
    """

    use_pressure_masking: str
    """ Pen pressure makes texture influence smaller

    :type: str
    """

    use_pressure_size: bool
    """ Enable tablet pressure sensitivity for size

    :type: bool
    """

    use_pressure_spacing: bool
    """ Enable tablet pressure sensitivity for spacing

    :type: bool
    """

    use_pressure_strength: bool
    """ Enable tablet pressure sensitivity for strength

    :type: bool
    """

    use_primary_overlay: bool
    """ Show texture in viewport

    :type: bool
    """

    use_primary_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_relative_jitter: bool
    """ Jittering happens in screen space, not relative to brush size

    :type: bool
    """

    use_restore_mesh: bool
    """ Allow a single dot to be carefully positioned

    :type: bool
    """

    use_secondary_overlay: bool
    """ Show texture in viewport

    :type: bool
    """

    use_secondary_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_smooth_stroke: bool
    """ Brush lags behind mouse and follows a smoother path

    :type: bool
    """

    use_space: bool
    """ Limit brush application to the distance specified by spacing

    :type: bool
    """

    use_space_attenuation: bool
    """ Automatically adjust strength to give consistent results for different spacings

    :type: bool
    """

    vertex_tool: str
    """ Brush blending mode

    :type: str
    """

    weight: float
    """ Vertex weight when brush is applied

    :type: float
    """

class CacheFile(ID, bpy_struct):
    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    filepath: str
    """ Path to external displacements file

    :type: str
    """

    forward_axis: str
    """ 

    :type: str
    """

    frame: float
    """ The time to use for looking up the data in the cache file, or to determine which file to use in a file sequence

    :type: float
    """

    is_sequence: bool
    """ Whether the cache is separated in a series of files

    :type: bool
    """

    object_paths: AlembicObjectPaths
    """ Paths of the objects inside the Alembic archive

    :type: AlembicObjectPaths
    """

    override_frame: bool
    """ Whether to use a custom frame for looking up data in the cache file, instead of using the current scene frame

    :type: bool
    """

    scale: float
    """ Value by which to enlarge or shrink the object with respect to the world's origin (only applicable through a Transform Cache constraint)

    :type: float
    """

    up_axis: str
    """ 

    :type: str
    """

class Camera(ID, bpy_struct):
    """Camera data-block for storing camera settings"""

    angle: float
    """ Camera lens field of view

    :type: float
    """

    angle_x: float
    """ Camera lens horizontal field of view

    :type: float
    """

    angle_y: float
    """ Camera lens vertical field of view

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    clip_end: float
    """ Camera far clipping distance

    :type: float
    """

    clip_start: float
    """ Camera near clipping distance

    :type: float
    """

    cycles: CyclesCameraSettings
    """ Cycles camera settings

    :type: CyclesCameraSettings
    """

    dof_distance: float
    """ Distance to the focus point for depth of field

    :type: float
    """

    dof_object: Object
    """ Use this object to define the depth of field focal point

    :type: Object
    """

    draw_size: float
    """ Apparent size of the Camera object in the 3D View

    :type: float
    """

    gpu_dof: GPUDOFSettings
    """ 

    :type: GPUDOFSettings
    """

    lens: float
    """ Perspective Camera lens value in millimeters

    :type: float
    """

    lens_unit: str
    """ Unit to edit lens in for the user interface

    :type: str
    """

    ortho_scale: float
    """ Orthographic Camera scale (similar to zoom)

    :type: float
    """

    passepartout_alpha: float
    """ Opacity (alpha) of the darkened overlay in Camera view

    :type: float
    """

    sensor_fit: str
    """ Method to fit image and field of view angle inside the sensor

    :type: str
    """

    sensor_height: float
    """ Vertical size of the image sensor area in millimeters

    :type: float
    """

    sensor_width: float
    """ Horizontal size of the image sensor area in millimeters

    :type: float
    """

    shift_x: float
    """ Camera horizontal shift

    :type: float
    """

    shift_y: float
    """ Camera vertical shift

    :type: float
    """

    show_guide: set[str]
    """ Draw overlay

    :type: set[str]
    """

    show_limits: bool
    """ Draw the clipping range and focus point on the camera

    :type: bool
    """

    show_mist: bool
    """ Draw a line from the Camera to indicate the mist area

    :type: bool
    """

    show_name: bool
    """ Show the active Camera's name in Camera view

    :type: bool
    """

    show_passepartout: bool
    """ Show a darkened overlay outside the image area in Camera view

    :type: bool
    """

    show_safe_areas: bool
    """ Show TV title safe and action safe areas in Camera view

    :type: bool
    """

    show_safe_center: bool
    """ Show safe areas to fit content in a different aspect ratio

    :type: bool
    """

    show_sensor: bool
    """ Show sensor size (film gate) in Camera view

    :type: bool
    """

    stereo: CameraStereoData
    """ 

    :type: CameraStereoData
    """

    type: str
    """ Camera types

    :type: str
    """

    def view_frame(self, scene: Scene | None = None):
        """Return 4 points for the cameras frame (before object transformation)

                :param scene: Scene to use for aspect calculation, when omitted 1:1 aspect is used
                :type scene: Scene | None
                :return: result_1, Result, float array of 3 items in [-inf, inf]

        result_2, Result, float array of 3 items in [-inf, inf]

        result_3, Result, float array of 3 items in [-inf, inf]

        result_4, Result, float array of 3 items in [-inf, inf]
        """
        ...

class Curve(ID, bpy_struct):
    """Curve data-block storing curves, splines and NURBS"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    bevel_depth: float
    """ Bevel depth when not using a bevel object

    :type: float
    """

    bevel_factor_end: float
    """ Factor that defines to where beveling of spline happens (0=to the very beginning, 1=to the very end)

    :type: float
    """

    bevel_factor_mapping_end: str
    """ Determines how the end bevel factor is mapped to a spline

    :type: str
    """

    bevel_factor_mapping_start: str
    """ Determines how the start bevel factor is mapped to a spline

    :type: str
    """

    bevel_factor_start: float
    """ Factor that defines from where beveling of spline happens (0=from the very beginning, 1=from the very end)

    :type: float
    """

    bevel_object: Object
    """ Curve object name that defines the bevel shape

    :type: Object
    """

    bevel_resolution: int
    """ Bevel resolution when depth is non-zero and no specific bevel object has been defined

    :type: int
    """

    cycles: CyclesMeshSettings
    """ Cycles mesh settings

    :type: CyclesMeshSettings
    """

    dimensions: str
    """ Select 2D or 3D curve type

    :type: str
    """

    eval_time: float
    """ Parametric position along the length of the curve that Objects 'following' it should be at (position is evaluated by dividing by the 'Path Length' value)

    :type: float
    """

    extrude: float
    """ Amount of curve extrusion when not using a bevel object

    :type: float
    """

    fill_mode: str
    """ Mode of filling curve

    :type: str
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    offset: float
    """ Offset the curve to adjust the width of a text

    :type: float
    """

    path_duration: int
    """ The number of frames that are needed to traverse the path, defining the maximum value for the 'Evaluation Time' setting

    :type: int
    """

    render_resolution_u: int
    """ Surface resolution in U direction used while rendering (zero uses preview resolution)

    :type: int
    """

    render_resolution_v: int
    """ Surface resolution in V direction used while rendering (zero uses preview resolution)

    :type: int
    """

    resolution_u: int
    """ Surface resolution in U direction

    :type: int
    """

    resolution_v: int
    """ Surface resolution in V direction

    :type: int
    """

    shape_keys: Key
    """ 

    :type: Key
    """

    show_handles: bool
    """ Display Bezier handles in editmode

    :type: bool
    """

    show_normal_face: bool
    """ Display 3D curve normals in editmode

    :type: bool
    """

    splines: CurveSplines
    """ Collection of splines in this curve data object

    :type: CurveSplines
    """

    taper_object: Object
    """ Curve object name that defines the taper (width)

    :type: Object
    """

    texspace_location: bpy_prop_array[float]
    """ Texture space location

    :type: bpy_prop_array[float]
    """

    texspace_size: bpy_prop_array[float]
    """ Texture space size

    :type: bpy_prop_array[float]
    """

    twist_mode: str
    """ The type of tilt calculation for 3D Curves

    :type: str
    """

    twist_smooth: float
    """ Smoothing iteration for tangents

    :type: float
    """

    use_auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    use_deform_bounds: bool
    """ Option for curve-deform: Use the mesh bounds to clamp the deformation

    :type: bool
    """

    use_fill_caps: bool
    """ Fill caps for beveled curves

    :type: bool
    """

    use_fill_deform: bool
    """ Fill curve after applying shape keys and all modifiers

    :type: bool
    """

    use_map_taper: bool
    """ Map effect of taper object on actually beveled curve

    :type: bool
    """

    use_path: bool
    """ Enable the curve to become a translation path

    :type: bool
    """

    use_path_follow: bool
    """ Make curve path children to rotate along the path

    :type: bool
    """

    use_radius: bool
    """ Option for paths and curve-deform: apply the curve radius with path following it and deforming

    :type: bool
    """

    use_stretch: bool
    """ Option for curve-deform: make deformed child to stretch along entire path

    :type: bool
    """

    use_uv_as_generated: bool
    """ Uses the UV values as Generated textured coordinates

    :type: bool
    """

    def transform(
        self,
        matrix: collections.abc.Iterable[float] | None,
        shape_keys: bool | typing.Any | None = False,
    ):
        """Transform curve by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        :param shape_keys: Transform Shape Keys
        :type shape_keys: bool | typing.Any | None
        """
        ...

    def validate_material_indices(self) -> bool:
        """Validate material indices of splines or letters, return True when the curve has had invalid indices corrected (to default 0)

        :return: Result
        :rtype: bool
        """
        ...

class FreestyleLineStyle(ID, bpy_struct):
    """Freestyle line style, reusable by multiple line sets"""

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    alpha: float
    """ Base alpha transparency, possibly modified by alpha transparency modifiers

    :type: float
    """

    alpha_modifiers: LineStyleAlphaModifiers
    """ List of alpha transparency modifiers

    :type: LineStyleAlphaModifiers
    """

    angle_max: float
    """ Maximum 2D angle for splitting chains

    :type: float
    """

    angle_min: float
    """ Minimum 2D angle for splitting chains

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    caps: str
    """ Select the shape of both ends of strokes

    :type: str
    """

    chain_count: int
    """ Chain count for the selection of first N chains

    :type: int
    """

    chaining: str
    """ Select the way how feature edges are jointed to form chains

    :type: str
    """

    color: bpy_prop_array[float]
    """ Base line color, possibly modified by line color modifiers

    :type: bpy_prop_array[float]
    """

    color_modifiers: LineStyleColorModifiers
    """ List of line color modifiers

    :type: LineStyleColorModifiers
    """

    dash1: int
    """ Length of the 1st dash for dashed lines

    :type: int
    """

    dash2: int
    """ Length of the 2nd dash for dashed lines

    :type: int
    """

    dash3: int
    """ Length of the 3rd dash for dashed lines

    :type: int
    """

    gap1: int
    """ Length of the 1st gap for dashed lines

    :type: int
    """

    gap2: int
    """ Length of the 2nd gap for dashed lines

    :type: int
    """

    gap3: int
    """ Length of the 3rd gap for dashed lines

    :type: int
    """

    geometry_modifiers: LineStyleGeometryModifiers
    """ List of stroke geometry modifiers

    :type: LineStyleGeometryModifiers
    """

    integration_type: str
    """ Select the way how the sort key is computed for each chain

    :type: str
    """

    length_max: float
    """ Maximum curvilinear 2D length for the selection of chains

    :type: float
    """

    length_min: float
    """ Minimum curvilinear 2D length for the selection of chains

    :type: float
    """

    material_boundary: bool
    """ If true, chains of feature edges are split at material boundaries

    :type: bool
    """

    node_tree: NodeTree
    """ Node tree for node-based shaders

    :type: NodeTree
    """

    panel: str
    """ Select the property panel to be shown

    :type: str
    """

    rounds: int
    """ Number of rounds in a sketchy multiple touch

    :type: int
    """

    sort_key: str
    """ Select the sort key to determine the stacking order of chains

    :type: str
    """

    sort_order: str
    """ Select the sort order

    :type: str
    """

    split_dash1: int
    """ Length of the 1st dash for splitting

    :type: int
    """

    split_dash2: int
    """ Length of the 2nd dash for splitting

    :type: int
    """

    split_dash3: int
    """ Length of the 3rd dash for splitting

    :type: int
    """

    split_gap1: int
    """ Length of the 1st gap for splitting

    :type: int
    """

    split_gap2: int
    """ Length of the 2nd gap for splitting

    :type: int
    """

    split_gap3: int
    """ Length of the 3rd gap for splitting

    :type: int
    """

    split_length: float
    """ Curvilinear 2D length for chain splitting

    :type: float
    """

    texture_slots: LineStyleTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: LineStyleTextureSlots
    """

    texture_spacing: float
    """ Spacing for textures along stroke length

    :type: float
    """

    thickness: float
    """ Base line thickness, possibly modified by line thickness modifiers

    :type: float
    """

    thickness_modifiers: LineStyleThicknessModifiers
    """ List of line thickness modifiers

    :type: LineStyleThicknessModifiers
    """

    thickness_position: str
    """ Thickness position of silhouettes and border edges (applicable when plain chaining is used with the Same Object option)

    :type: str
    """

    thickness_ratio: float
    """ A number between 0 (inside) and 1 (outside) specifying the relative position of stroke thickness

    :type: float
    """

    use_angle_max: bool
    """ Split chains at points with angles larger than the maximum 2D angle

    :type: bool
    """

    use_angle_min: bool
    """ Split chains at points with angles smaller than the minimum 2D angle

    :type: bool
    """

    use_chain_count: bool
    """ Enable the selection of first N chains

    :type: bool
    """

    use_chaining: bool
    """ Enable chaining of feature edges

    :type: bool
    """

    use_dashed_line: bool
    """ Enable or disable dashed line

    :type: bool
    """

    use_length_max: bool
    """ Enable the selection of chains by a maximum 2D length

    :type: bool
    """

    use_length_min: bool
    """ Enable the selection of chains by a minimum 2D length

    :type: bool
    """

    use_nodes: bool
    """ Use shader nodes for the line style

    :type: bool
    """

    use_same_object: bool
    """ If true, only feature edges of the same object are joined

    :type: bool
    """

    use_sorting: bool
    """ Arrange the stacking order of strokes

    :type: bool
    """

    use_split_length: bool
    """ Enable chain splitting by curvilinear 2D length

    :type: bool
    """

    use_split_pattern: bool
    """ Enable chain splitting by dashed line patterns

    :type: bool
    """

    use_texture: bool
    """ Enable or disable textured strokes

    :type: bool
    """

class GreasePencil(ID, bpy_struct):
    """Freehand annotation sketchbook"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    layers: GreasePencilLayers
    """ 

    :type: GreasePencilLayers
    """

    palettes: GreasePencilPalettes
    """ 

    :type: GreasePencilPalettes
    """

    show_stroke_direction: bool
    """ Show stroke drawing direction with a bigger green dot (start) and smaller red dot (end) points

    :type: bool
    """

    use_onion_skinning: bool
    """ Show ghosts of the frames before and after the current frame, toggle to enable on active layer or disable all

    :type: bool
    """

    use_stroke_edit_mode: bool
    """ Edit Grease Pencil strokes instead of viewport data

    :type: bool
    """

    def clear(self):
        """Remove all the grease pencil data"""
        ...

class Group(ID, bpy_struct):
    """Group of Object data-blocks"""

    dupli_offset: bpy_prop_array[float]
    """ Offset from the origin to use when instancing as DupliGroup

    :type: bpy_prop_array[float]
    """

    layers: list[bool]
    """ Layers visible when this group is instanced as a dupli

    :type: list[bool]
    """

    objects: GroupObjects
    """ A collection of this groups objects

    :type: GroupObjects
    """

    users_dupli_group: typing.Any
    """ The dupli group this group is used in
(readonly)"""

class Image(ID, bpy_struct):
    """Image data-block referencing an external or packed image"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    bindcode: bpy_prop_array[int]
    """ OpenGL bindcode

    :type: bpy_prop_array[int]
    """

    channels: int
    """ Number of channels in pixels buffer

    :type: int
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    depth: int
    """ Image bit depth

    :type: int
    """

    display_aspect: bpy_prop_array[float]
    """ Display Aspect for this image, does not affect rendering

    :type: bpy_prop_array[float]
    """

    field_order: str
    """ Order of video fields (select which lines are displayed first)

    :type: str
    """

    file_format: str
    """ Format used for re-saving this file

    :type: str
    """

    filepath: str
    """ Image/Movie file name

    :type: str
    """

    filepath_raw: str
    """ Image/Movie file name (without data refreshing)

    :type: str
    """

    fps: int
    """ Speed of the animation in frames per second

    :type: int
    """

    frame_duration: int
    """ Duration (in frames) of the image (1 when not a video/sequence)

    :type: int
    """

    frame_end: int
    """ End frame of an animated texture

    :type: int
    """

    frame_start: int
    """ Start frame of an animated texture

    :type: int
    """

    generated_color: bpy_prop_array[float]
    """ Fill color for the generated image

    :type: bpy_prop_array[float]
    """

    generated_height: int
    """ Generated image height

    :type: int
    """

    generated_type: str
    """ Generated image type

    :type: str
    """

    generated_width: int
    """ Generated image width

    :type: int
    """

    has_data: bool
    """ True if this image has data

    :type: bool
    """

    is_dirty: bool
    """ Image has changed and is not saved

    :type: bool
    """

    is_float: bool
    """ True if this image is stored in float buffer

    :type: bool
    """

    is_multiview: bool
    """ Image has more than one view

    :type: bool
    """

    is_stereo_3d: bool
    """ Image has left and right views

    :type: bool
    """

    mapping: str
    """ Mapping type to use for this image in the game engine

    :type: str
    """

    packed_file: PackedFile
    """ First packed file of the image

    :type: PackedFile
    """

    packed_files: bpy_prop_collection[ImagePackedFile]
    """ Collection of packed images

    :type: bpy_prop_collection[ImagePackedFile]
    """

    pixels: float
    """ Image pixels in floating point values

    :type: float
    """

    render_slots: RenderSlots
    """ Render slots of the image

    :type: RenderSlots
    """

    resolution: bpy_prop_array[float]
    """ X/Y pixels per meter

    :type: bpy_prop_array[float]
    """

    size: bpy_prop_array[int]
    """ Width and height in pixels, zero when image data cant be loaded

    :type: bpy_prop_array[int]
    """

    source: str
    """ Where the image comes from

    :type: str
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    tiles_x: int
    """ Degree of repetition in the X direction

    :type: int
    """

    tiles_y: int
    """ Degree of repetition in the Y direction

    :type: int
    """

    type: str
    """ How to generate the image

    :type: str
    """

    use_alpha: bool
    """ Use the alpha channel information from the image or make image fully opaque

    :type: bool
    """

    use_animation: bool
    """ Use as animated texture in the game engine

    :type: bool
    """

    use_clamp_x: bool
    """ Disable texture repeating horizontally

    :type: bool
    """

    use_clamp_y: bool
    """ Disable texture repeating vertically

    :type: bool
    """

    use_deinterlace: bool
    """ Deinterlace movie file on load

    :type: bool
    """

    use_fields: bool
    """ Use fields of the image

    :type: bool
    """

    use_generated_float: bool
    """ Generate floating point buffer

    :type: bool
    """

    use_multiview: bool
    """ Use Multiple Views (when available)

    :type: bool
    """

    use_tiles: bool
    """ Use of tilemode for faces (default shift-LMB to pick the tile for selected faces)

    :type: bool
    """

    use_view_as_render: bool
    """ Apply render part of display transformation when displaying this image on the screen

    :type: bool
    """

    views_format: str
    """ Mode to load image views

    :type: str
    """

    def save_render(self, filepath: str | typing.Any, scene: Scene | None = None):
        """Save image to a specific path using a scenes render settings

        :param filepath: Save path
        :type filepath: str | typing.Any
        :param scene: Scene to take image parameters from
        :type scene: Scene | None
        """
        ...

    def save(self):
        """Save image to its source path"""
        ...

    def pack(
        self,
        as_png: bool | typing.Any | None = False,
        data: str | typing.Any = "",
        data_len: typing.Any | None = 0,
    ):
        """Pack an image as embedded data into the .blend file

        :param as_png: as_png, Pack the image as PNG (needed for generated/dirty images)
        :type as_png: bool | typing.Any | None
        :param data: data, Raw data (bytes, exact content of the embedded file)
        :type data: str | typing.Any
        :param data_len: data_len, length of given data (mandatory if data is provided)
        :type data_len: typing.Any | None
        """
        ...

    def unpack(self, method: str | None = "USE_LOCAL"):
        """Save an image packed in the .blend file to disk

        :param method: method, How to unpack
        :type method: str | None
        """
        ...

    def reload(self):
        """Reload the image from its source path"""
        ...

    def update(self):
        """Update the display image from the floating point buffer"""
        ...

    def scale(self, width: int | None, height: int | None):
        """Scale the image in pixels

        :param width: Width
        :type width: int | None
        :param height: Height
        :type height: int | None
        """
        ...

    def gl_touch(
        self,
        frame: typing.Any | None = 0,
        filter: typing.Any | None = 9985,
        mag: typing.Any | None = 9729,
    ) -> int:
        """Delay the image from being cleaned from the cache due inactivity

        :param frame: Frame, Frame of image sequence or movie
        :type frame: typing.Any | None
        :param filter: Filter, The texture minifying function to use if the image wasn't loaded
        :type filter: typing.Any | None
        :param mag: Magnification, The texture magnification function to use if the image wasn't loaded
        :type mag: typing.Any | None
        :return: Error, OpenGL error value
        :rtype: int
        """
        ...

    def gl_load(
        self,
        frame: typing.Any | None = 0,
        filter: typing.Any | None = 9985,
        mag: typing.Any | None = 9729,
    ) -> int:
        """Load the image into OpenGL graphics memory

        :param frame: Frame, Frame of image sequence or movie
        :type frame: typing.Any | None
        :param filter: Filter, The texture minifying function
        :type filter: typing.Any | None
        :param mag: Magnification, The texture magnification function
        :type mag: typing.Any | None
        :return: Error, OpenGL error value
        :rtype: int
        """
        ...

    def gl_free(self):
        """Free the image from OpenGL graphics memory"""
        ...

    def filepath_from_user(
        self, image_user: ImageUser | None = None
    ) -> str | typing.Any:
        """Return the absolute path to the filepath of an image frame specified by the image user

        :param image_user: Image user of the image to get filepath for
        :type image_user: ImageUser | None
        :return: File Path, The resulting filepath from the image and it's user
        :rtype: str | typing.Any
        """
        ...

    def buffers_free(self):
        """Free the image buffers from memory"""
        ...

class Key(ID, bpy_struct):
    """Shape keys data-block containing different shapes of geometric data-blocks"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    eval_time: float
    """ Evaluation time for absolute shape keys

    :type: float
    """

    key_blocks: bpy_prop_collection[ShapeKey]
    """ Shape keys

    :type: bpy_prop_collection[ShapeKey]
    """

    reference_key: ShapeKey
    """ 

    :type: ShapeKey
    """

    use_relative: bool
    """ Make shape keys relative, otherwise play through shapes as a sequence using the evaluation time

    :type: bool
    """

    user: ID
    """ Datablock using these shape keys

    :type: ID
    """

class Lamp(ID, bpy_struct):
    """Lamp data-block for lighting a scene"""

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    color: bpy_prop_array[float]
    """ Light color

    :type: bpy_prop_array[float]
    """

    cycles: CyclesLampSettings
    """ Cycles lamp settings

    :type: CyclesLampSettings
    """

    distance: float
    """ Falloff distance - the light is at half the original intensity at this point

    :type: float
    """

    energy: float
    """ Amount of light that the lamp emits

    :type: float
    """

    node_tree: NodeTree
    """ Node tree for node based lamps

    :type: NodeTree
    """

    texture_slots: LampTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: LampTextureSlots
    """

    type: str
    """ Type of Lamp

    :type: str
    """

    use_diffuse: bool
    """ Do diffuse shading

    :type: bool
    """

    use_negative: bool
    """ Cast negative light

    :type: bool
    """

    use_nodes: bool
    """ Use shader nodes to render the lamp

    :type: bool
    """

    use_own_layer: bool
    """ Illuminate objects only on the same layers the lamp is on

    :type: bool
    """

    use_specular: bool
    """ Create specular highlights

    :type: bool
    """

class Lattice(ID, bpy_struct):
    """Lattice data-block defining a grid for deforming other objects"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    interpolation_type_u: str
    """ 

    :type: str
    """

    interpolation_type_v: str
    """ 

    :type: str
    """

    interpolation_type_w: str
    """ 

    :type: str
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    points: bpy_prop_collection[LatticePoint]
    """ Points of the lattice

    :type: bpy_prop_collection[LatticePoint]
    """

    points_u: int
    """ Point in U direction (can't be changed when there are shape keys)

    :type: int
    """

    points_v: int
    """ Point in V direction (can't be changed when there are shape keys)

    :type: int
    """

    points_w: int
    """ Point in W direction (can't be changed when there are shape keys)

    :type: int
    """

    shape_keys: Key
    """ 

    :type: Key
    """

    use_outside: bool
    """ Only draw, and take into account, the outer vertices

    :type: bool
    """

    vertex_group: str
    """ Vertex group to apply the influence of the lattice

    :type: str
    """

    def transform(
        self,
        matrix: collections.abc.Iterable[float] | None,
        shape_keys: bool | typing.Any | None = False,
    ):
        """Transform lattice by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        :param shape_keys: Transform Shape Keys
        :type shape_keys: bool | typing.Any | None
        """
        ...

class Library(ID, bpy_struct):
    """External .blend file from which data is linked"""

    filepath: str
    """ Path to the library .blend file

    :type: str
    """

    packed_file: PackedFile
    """ 

    :type: PackedFile
    """

    parent: Library
    """ 

    :type: Library
    """

    users_id: typing.Any
    """ ID data blocks which use this library
(readonly)"""

    def reload(self):
        """Reload this library and all its linked datablocks"""
        ...

class Mask(ID, bpy_struct):
    """Mask data-block defining mask for compositing"""

    active_layer_index: int | None
    """ Index of active layer in list of all mask's layers

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    frame_end: int
    """ Final frame of the mask (used for sequencer)

    :type: int
    """

    frame_start: int
    """ First frame of the mask (used for sequencer)

    :type: int
    """

    layers: MaskLayers
    """ Collection of layers which defines this mask

    :type: MaskLayers
    """

class Material(ID, bpy_struct):
    """Material data-block to define the appearance of geometric objects for rendering"""

    active_node_material: Material | None
    """ Active node material

    :type: Material | None
    """

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    alpha: float
    """ Alpha transparency of the material

    :type: float
    """

    ambient: float
    """ Amount of global ambient color the material receives

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    cycles: CyclesMaterialSettings
    """ Cycles material settings

    :type: CyclesMaterialSettings
    """

    darkness: float
    """ Minnaert darkness

    :type: float
    """

    diffuse_color: bpy_prop_array[float]
    """ Diffuse color of the material

    :type: bpy_prop_array[float]
    """

    diffuse_fresnel: float
    """ Power of Fresnel

    :type: float
    """

    diffuse_fresnel_factor: float
    """ Blending factor of Fresnel

    :type: float
    """

    diffuse_intensity: float
    """ Amount of diffuse reflection

    :type: float
    """

    diffuse_ramp: ColorRamp
    """ Color ramp used to affect diffuse shading

    :type: ColorRamp
    """

    diffuse_ramp_blend: str
    """ Blending method of the ramp and the diffuse color

    :type: str
    """

    diffuse_ramp_factor: float
    """ Blending factor (also uses alpha in Colorband)

    :type: float
    """

    diffuse_ramp_input: str
    """ How the ramp maps on the surface

    :type: str
    """

    diffuse_shader: str
    """ 

    :type: str
    """

    diffuse_toon_size: float
    """ Size of diffuse toon area

    :type: float
    """

    diffuse_toon_smooth: float
    """ Smoothness of diffuse toon area

    :type: float
    """

    emit: float
    """ Amount of light to emit

    :type: float
    """

    game_settings: MaterialGameSettings
    """ Game material settings

    :type: MaterialGameSettings
    """

    halo: MaterialHalo
    """ Halo settings for the material

    :type: MaterialHalo
    """

    invert_z: bool
    """ Render material's faces with an inverted Z buffer (scanline only)

    :type: bool
    """

    light_group: Group
    """ Limit lighting to lamps in this Group

    :type: Group
    """

    line_color: bpy_prop_array[float]
    """ Line color used for Freestyle line rendering

    :type: bpy_prop_array[float]
    """

    line_priority: int
    """ The line color of a higher priority is used at material boundaries

    :type: int
    """

    mirror_color: bpy_prop_array[float]
    """ Mirror color of the material

    :type: bpy_prop_array[float]
    """

    node_tree: NodeTree
    """ Node tree for node based materials

    :type: NodeTree
    """

    offset_z: float
    """ Give faces an artificial offset in the Z buffer for Z transparency

    :type: float
    """

    paint_active_slot: int
    """ Index of active texture paint slot

    :type: int
    """

    paint_clone_slot: int
    """ Index of clone texture paint slot

    :type: int
    """

    pass_index: int
    """ Index number for the "Material Index" render pass

    :type: int
    """

    physics: MaterialPhysics
    """ Game physics settings

    :type: MaterialPhysics
    """

    preview_render_type: str
    """ Type of preview render

    :type: str
    """

    raytrace_mirror: MaterialRaytraceMirror
    """ Raytraced reflection settings for the material

    :type: MaterialRaytraceMirror
    """

    raytrace_transparency: MaterialRaytraceTransparency
    """ Raytraced transparency settings for the material

    :type: MaterialRaytraceTransparency
    """

    roughness: float
    """ Oren-Nayar Roughness

    :type: float
    """

    shadow_buffer_bias: float
    """ Factor to multiply shadow buffer bias with (0 is ignore)

    :type: float
    """

    shadow_cast_alpha: float
    """ Shadow casting alpha, in use for Irregular and Deep shadow buffer

    :type: float
    """

    shadow_only_type: str
    """ How to draw shadows

    :type: str
    """

    shadow_ray_bias: float
    """ Shadow raytracing bias to prevent terminator problems on shadow boundary

    :type: float
    """

    specular_alpha: float
    """ Alpha transparency for specular areas

    :type: float
    """

    specular_color: bpy_prop_array[float]
    """ Specular color of the material

    :type: bpy_prop_array[float]
    """

    specular_hardness: int
    """ How hard (sharp) the specular reflection is

    :type: int
    """

    specular_intensity: float
    """ How intense (bright) the specular reflection is

    :type: float
    """

    specular_ior: float
    """ Specular index of refraction

    :type: float
    """

    specular_ramp: ColorRamp
    """ Color ramp used to affect specular shading

    :type: ColorRamp
    """

    specular_ramp_blend: str
    """ Blending method of the ramp and the specular color

    :type: str
    """

    specular_ramp_factor: float
    """ Blending factor (also uses alpha in Colorband)

    :type: float
    """

    specular_ramp_input: str
    """ How the ramp maps on the surface

    :type: str
    """

    specular_shader: str
    """ 

    :type: str
    """

    specular_slope: float
    """ The standard deviation of surface slope

    :type: float
    """

    specular_toon_size: float
    """ Size of specular toon area

    :type: float
    """

    specular_toon_smooth: float
    """ Smoothness of specular toon area

    :type: float
    """

    strand: MaterialStrand
    """ Strand settings for the material

    :type: MaterialStrand
    """

    subsurface_scattering: MaterialSubsurfaceScattering
    """ Subsurface scattering settings for the material

    :type: MaterialSubsurfaceScattering
    """

    texture_paint_images: bpy_prop_collection[Image]
    """ Texture images used for texture painting

    :type: bpy_prop_collection[Image]
    """

    texture_paint_slots: bpy_prop_collection[TexPaintSlot]
    """ Texture slots defining the mapping and influence of textures

    :type: bpy_prop_collection[TexPaintSlot]
    """

    texture_slots: MaterialTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: MaterialTextureSlots
    """

    translucency: float
    """ Amount of diffuse shading on the back side

    :type: float
    """

    transparency_method: str
    """ Method to use for rendering transparency

    :type: str
    """

    type: str
    """ Material type defining how the object is rendered

    :type: str
    """

    use_cast_approximate: bool
    """ Allow this material to cast shadows when using approximate ambient occlusion

    :type: bool
    """

    use_cast_buffer_shadows: bool
    """ Allow this material to cast shadows from shadow buffer lamps

    :type: bool
    """

    use_cast_shadows: bool
    """ Allow this material to cast shadows

    :type: bool
    """

    use_cast_shadows_only: bool
    """ Make objects with this material appear invisible (not rendered), only casting shadows

    :type: bool
    """

    use_cubic: bool
    """ Use cubic interpolation for diffuse values, for smoother transitions

    :type: bool
    """

    use_diffuse_ramp: bool
    """ Toggle diffuse ramp operations

    :type: bool
    """

    use_face_texture: bool
    """ Replace the object's base color with color from UV map image textures

    :type: bool
    """

    use_face_texture_alpha: bool
    """ Replace the object's base alpha value with alpha from UV map image textures

    :type: bool
    """

    use_full_oversampling: bool
    """ Force this material to render full shading/textures for all anti-aliasing samples

    :type: bool
    """

    use_light_group_exclusive: bool
    """ Material uses the light group exclusively - these lamps are excluded from other scene lighting

    :type: bool
    """

    use_light_group_local: bool
    """ When linked in, material uses local light group with the same name

    :type: bool
    """

    use_mist: bool
    """ Use mist with this material (in world settings)

    :type: bool
    """

    use_nodes: bool
    """ Use shader nodes to render the material

    :type: bool
    """

    use_object_color: bool
    """ Modulate the result with a per-object color

    :type: bool
    """

    use_only_shadow: bool
    """ Render shadows as the material's alpha value, making the material transparent except for shadowed areas

    :type: bool
    """

    use_ray_shadow_bias: bool
    """ Prevent raytraced shadow errors on surfaces with smooth shaded normals (terminator problem)

    :type: bool
    """

    use_raytrace: bool
    """ Include this material and geometry that uses it in raytracing calculations

    :type: bool
    """

    use_shadeless: bool
    """ Make this material insensitive to light or shadow

    :type: bool
    """

    use_shadows: bool
    """ Allow this material to receive shadows

    :type: bool
    """

    use_sky: bool
    """ Render this material with zero alpha, with sky background in place (scanline only)

    :type: bool
    """

    use_specular_ramp: bool
    """ Toggle specular ramp operations

    :type: bool
    """

    use_tangent_shading: bool
    """ Use the material's tangent vector instead of the normal for shading - for anisotropic shading effects

    :type: bool
    """

    use_textures: list[bool]
    """ Enable/Disable each texture

    :type: list[bool]
    """

    use_transparency: bool
    """ Render material as transparent

    :type: bool
    """

    use_transparent_shadows: bool
    """ Allow this object to receive transparent shadows cast through other objects

    :type: bool
    """

    use_uv_project: bool
    """ Use to ensure UV interpolation is correct for camera projections (use with UV project modifier)

    :type: bool
    """

    use_vertex_color_light: bool
    """ Add vertex colors as additional lighting

    :type: bool
    """

    use_vertex_color_paint: bool
    """ Replace object base color with vertex colors (multiply with 'texture face' face assigned textures)

    :type: bool
    """

    volume: MaterialVolume
    """ Volume settings for the material

    :type: MaterialVolume
    """

class Mesh(ID, bpy_struct):
    """Mesh data-block defining geometric surfaces"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    auto_smooth_angle: float
    """ Maximum angle between face normals that will be considered as smooth (unused if custom split normals data are available)

    :type: float
    """

    auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    cycles: CyclesMeshSettings
    """ Cycles mesh settings

    :type: CyclesMeshSettings
    """

    edges: MeshEdges
    """ Edges of the mesh

    :type: MeshEdges
    """

    has_custom_normals: bool
    """ True if there are custom split normals data in this mesh

    :type: bool
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    loops: MeshLoops
    """ Loops of the mesh (polygon corners)

    :type: MeshLoops
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    polygon_layers_float: PolygonFloatProperties
    """ 

    :type: PolygonFloatProperties
    """

    polygon_layers_int: PolygonIntProperties
    """ 

    :type: PolygonIntProperties
    """

    polygon_layers_string: PolygonStringProperties
    """ 

    :type: PolygonStringProperties
    """

    polygons: MeshPolygons
    """ Polygons of the mesh

    :type: MeshPolygons
    """

    shape_keys: Key
    """ 

    :type: Key
    """

    show_double_sided: bool
    """ Display the mesh with double or single sided lighting (OpenGL only)

    :type: bool
    """

    show_edge_bevel_weight: bool
    """ Display weights created for the Bevel modifier

    :type: bool
    """

    show_edge_crease: bool
    """ Display creases created for subsurf weighting

    :type: bool
    """

    show_edge_seams: bool
    """ Display UV unwrapping seams

    :type: bool
    """

    show_edge_sharp: bool
    """ Display sharp edges, used with the EdgeSplit modifier

    :type: bool
    """

    show_edges: bool
    """ Display selected edges using highlights in the 3D view and UV editor

    :type: bool
    """

    show_extra_edge_angle: bool
    """ Display selected edge angle, using global values when set in the transform panel

    :type: bool
    """

    show_extra_edge_length: bool
    """ Display selected edge lengths, using global values when set in the transform panel

    :type: bool
    """

    show_extra_face_angle: bool
    """ Display the angles in the selected edges, using global values when set in the transform panel

    :type: bool
    """

    show_extra_face_area: bool
    """ Display the area of selected faces, using global values when set in the transform panel

    :type: bool
    """

    show_extra_indices: bool
    """ Display the index numbers of selected vertices, edges, and faces

    :type: bool
    """

    show_faces: bool
    """ Display all faces as shades in the 3D view and UV editor

    :type: bool
    """

    show_freestyle_edge_marks: bool
    """ Display Freestyle edge marks, used with the Freestyle renderer

    :type: bool
    """

    show_freestyle_face_marks: bool
    """ Display Freestyle face marks, used with the Freestyle renderer

    :type: bool
    """

    show_normal_face: bool
    """ Display face normals as lines

    :type: bool
    """

    show_normal_loop: bool
    """ Display vertex-per-face normals as lines

    :type: bool
    """

    show_normal_vertex: bool
    """ Display vertex normals as lines

    :type: bool
    """

    show_statvis: bool
    """ Display statistical information about the mesh

    :type: bool
    """

    show_weight: bool
    """ Draw weights in editmode

    :type: bool
    """

    skin_vertices: bpy_prop_collection[MeshSkinVertexLayer]
    """ All skin vertices

    :type: bpy_prop_collection[MeshSkinVertexLayer]
    """

    tessface_uv_textures: TessfaceUVTextures
    """ All UV maps for tessellated faces (read-only, for use by renderers)

    :type: TessfaceUVTextures
    """

    tessface_vertex_colors: VertexColors
    """ All tessellated face colors (read-only, for use by renderers)

    :type: VertexColors
    """

    tessfaces: MeshTessFaces
    """ Tessellated faces of the mesh (derived from polygons)

    :type: MeshTessFaces
    """

    texco_mesh: Mesh
    """ Derive texture coordinates from another mesh

    :type: Mesh
    """

    texspace_location: bpy_prop_array[float]
    """ Texture space location

    :type: bpy_prop_array[float]
    """

    texspace_size: bpy_prop_array[float]
    """ Texture space size

    :type: bpy_prop_array[float]
    """

    texture_mesh: Mesh
    """ Use another mesh for texture indices (vertex indices must be aligned)

    :type: Mesh
    """

    total_edge_sel: int
    """ Selected edge count in editmode

    :type: int
    """

    total_face_sel: int
    """ Selected face count in editmode

    :type: int
    """

    total_vert_sel: int
    """ Selected vertex count in editmode

    :type: int
    """

    use_auto_smooth: bool
    """ Auto smooth (based on smooth/sharp faces/edges and angle between faces), or use custom split normals data if available

    :type: bool
    """

    use_auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    use_customdata_edge_bevel: bool
    """ 

    :type: bool
    """

    use_customdata_edge_crease: bool
    """ 

    :type: bool
    """

    use_customdata_vertex_bevel: bool
    """ 

    :type: bool
    """

    use_mirror_topology: bool
    """ Use topology based mirroring (for when both sides of mesh have matching, unique topology)

    :type: bool
    """

    use_mirror_x: bool
    """ X Axis mirror editing

    :type: bool
    """

    use_paint_mask: bool
    """ Face selection masking for painting

    :type: bool
    """

    use_paint_mask_vertex: bool
    """ Vertex selection masking for painting (weight paint only)

    :type: bool
    """

    uv_layer_clone: MeshUVLoopLayer
    """ UV loop layer to be used as cloning source

    :type: MeshUVLoopLayer
    """

    uv_layer_clone_index: int
    """ Clone UV loop layer index

    :type: int
    """

    uv_layer_stencil: MeshUVLoopLayer
    """ UV loop layer to mask the painted area

    :type: MeshUVLoopLayer
    """

    uv_layer_stencil_index: int
    """ Mask UV loop layer index

    :type: int
    """

    uv_layers: UVLoopLayers
    """ All UV loop layers

    :type: UVLoopLayers
    """

    uv_texture_clone: MeshTexturePolyLayer
    """ UV map to be used as cloning source

    :type: MeshTexturePolyLayer
    """

    uv_texture_clone_index: int
    """ Clone UV map index

    :type: int
    """

    uv_texture_stencil: MeshTexturePolyLayer
    """ UV map to mask the painted area

    :type: MeshTexturePolyLayer
    """

    uv_texture_stencil_index: int
    """ Mask UV map index

    :type: int
    """

    uv_textures: UVTextures
    """ All UV maps

    :type: UVTextures
    """

    vertex_colors: LoopColors
    """ All vertex colors

    :type: LoopColors
    """

    vertex_layers_float: VertexFloatProperties
    """ 

    :type: VertexFloatProperties
    """

    vertex_layers_int: VertexIntProperties
    """ 

    :type: VertexIntProperties
    """

    vertex_layers_string: VertexStringProperties
    """ 

    :type: VertexStringProperties
    """

    vertex_paint_masks: bpy_prop_collection[MeshPaintMaskLayer]
    """ Vertex paint mask

    :type: bpy_prop_collection[MeshPaintMaskLayer]
    """

    vertices: MeshVertices
    """ Vertices of the mesh

    :type: MeshVertices
    """

    edge_keys: typing.Any
    """ (readonly)"""

    def transform(
        self,
        matrix: collections.abc.Iterable[float] | None,
        shape_keys: bool | typing.Any | None = False,
    ):
        """Transform mesh vertices by a matrix (Warning: inverts normals if matrix is negative)

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        :param shape_keys: Transform Shape Keys
        :type shape_keys: bool | typing.Any | None
        """
        ...

    def flip_normals(self):
        """Invert winding of all polygons (clears tessellation, does not handle custom normals)"""
        ...

    def calc_normals(self):
        """Calculate vertex normals"""
        ...

    def create_normals_split(self):
        """Empty split vertex normals"""
        ...

    def calc_normals_split(self):
        """Calculate split vertex normals, which preserve sharp edges"""
        ...

    def free_normals_split(self):
        """Free split vertex normals"""
        ...

    def split_faces(self):
        """Spli faces based on the edge angle"""
        ...

    def calc_tangents(self, uvmap: str | typing.Any = ""):
        """Compute tangents and bitangent signs, to be used together with the split normals to get a complete tangent space for normal mapping (split normals are also computed if not yet present)

        :param uvmap: Name of the UV map to use for tangent space computation
        :type uvmap: str | typing.Any
        """
        ...

    def free_tangents(self):
        """Free tangents"""
        ...

    def calc_tessface(self, free_mpoly: bool | typing.Any | None = False):
        """Calculate face tessellation (supports editmode too)

        :param free_mpoly: Free MPoly, Free data used by polygons and loops. WARNING: This destructive operation removes regular faces, only used on temporary mesh data-blocks to reduce memory footprint of render engines and export scripts
        :type free_mpoly: bool | typing.Any | None
        """
        ...

    def calc_smooth_groups(self, use_bitflags: bool | typing.Any | None = False):
        """Calculate smooth groups from sharp edges

                :param use_bitflags: Produce bitflags groups instead of simple numeric values
                :type use_bitflags: bool | typing.Any | None
                :return: poly_groups, Smooth Groups, int array of 1 items in [-inf, inf]

        groups, Total number of groups, int in [0, inf]
        """
        ...

    def normals_split_custom_set(self, normals: collections.abc.Iterable[float] | None):
        """Define custom split normals of this mesh (use zero-vectors to keep auto ones)

        :param normals: Normals
        :type normals: collections.abc.Iterable[float] | None
        """
        ...

    def normals_split_custom_set_from_vertices(
        self, normals: collections.abc.Iterable[float] | None
    ):
        """Define custom split normals of this mesh, from vertices' normals (use zero-vectors to keep auto ones)

        :param normals: Normals
        :type normals: collections.abc.Iterable[float] | None
        """
        ...

    def update(
        self,
        calc_edges: bool | typing.Any | None = False,
        calc_tessface: bool | typing.Any | None = False,
    ):
        """update

        :param calc_edges: Calculate Edges, Force recalculation of edges
        :type calc_edges: bool | typing.Any | None
        :param calc_tessface: Calculate Tessellation, Force recalculation of tessellation faces
        :type calc_tessface: bool | typing.Any | None
        """
        ...

    def unit_test_compare(self, mesh: Mesh | None = None) -> str | typing.Any:
        """unit_test_compare

        :param mesh: Mesh to compare to
        :type mesh: Mesh | None
        :return: Return value, String description of result of comparison
        :rtype: str | typing.Any
        """
        ...

    def validate(
        self,
        verbose: bool | typing.Any | None = False,
        clean_customdata: bool | typing.Any | None = True,
    ) -> bool:
        """Validate geometry, return True when the mesh has had invalid geometry corrected/removed

        :param verbose: Verbose, Output information about the errors found
        :type verbose: bool | typing.Any | None
        :param clean_customdata: Clean Custom Data, Remove temp/cached custom-data layers, like e.g. normals...
        :type clean_customdata: bool | typing.Any | None
        :return: Result
        :rtype: bool
        """
        ...

    def validate_material_indices(self) -> bool:
        """Validate material indices of polygons, return True when the mesh has had invalid indices corrected (to default 0)

        :return: Result
        :rtype: bool
        """
        ...

    def from_pydata(
        self, vertices: list | None, edges: list | None, faces: list | None
    ):
        """Make a mesh from a list of vertices/edges/faces
        Until we have a nicer way to make geometry, use this.

                :param vertices: float triplets each representing (X, Y, Z)
        eg: [(0.0, 1.0, 0.5), ...].
                :type vertices: list | None
                :param edges: int pairs, each pair contains two indices to the
        vertices argument. eg: [(1, 2), ...]
                :type edges: list | None
                :param faces: iterator of faces, each faces contains three or more indices to
        the vertices argument. eg: [(5, 6, 8, 9), (1, 2, 3), ...]
                :type faces: list | None
        """
        ...

class MetaBall(ID, bpy_struct):
    """Metaball data-block to defined blobby surfaces"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    cycles: CyclesMeshSettings
    """ Cycles mesh settings

    :type: CyclesMeshSettings
    """

    elements: MetaBallElements
    """ Meta elements

    :type: MetaBallElements
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    render_resolution: float
    """ Polygonization resolution in rendering

    :type: float
    """

    resolution: float
    """ Polygonization resolution in the 3D viewport

    :type: float
    """

    texspace_location: bpy_prop_array[float]
    """ Texture space location

    :type: bpy_prop_array[float]
    """

    texspace_size: bpy_prop_array[float]
    """ Texture space size

    :type: bpy_prop_array[float]
    """

    threshold: float
    """ Influence of meta elements

    :type: float
    """

    update_method: str
    """ Metaball edit update behavior

    :type: str
    """

    use_auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    def transform(self, matrix: collections.abc.Iterable[float] | None):
        """Transform meta elements by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        """
        ...

class MovieClip(ID, bpy_struct):
    """MovieClip data-block referencing an external movie file"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    display_aspect: bpy_prop_array[float]
    """ Display Aspect for this clip, does not affect rendering

    :type: bpy_prop_array[float]
    """

    filepath: str
    """ Filename of the movie or sequence file

    :type: str
    """

    frame_duration: int
    """ Detected duration of movie clip in frames

    :type: int
    """

    frame_offset: int
    """ Offset of footage first frame relative to it's file name (affects only how footage is loading, does not change data associated with a clip)

    :type: int
    """

    frame_start: int
    """ Global scene frame number at which this movie starts playing (affects all data associated with a clip)

    :type: int
    """

    grease_pencil: GreasePencil
    """ Grease pencil data for this movie clip

    :type: GreasePencil
    """

    proxy: MovieClipProxy
    """ 

    :type: MovieClipProxy
    """

    size: bpy_prop_array[int]
    """ Width and height in pixels, zero when image data cant be loaded

    :type: bpy_prop_array[int]
    """

    source: str
    """ Where the clip comes from

    :type: str
    """

    tracking: MovieTracking
    """ 

    :type: MovieTracking
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this clip

    :type: bool
    """

    use_proxy_custom_directory: bool
    """ Create proxy images in a custom directory (default is movie location)

    :type: bool
    """

class NodeTree(ID, bpy_struct):
    """Node tree consisting of linked nodes used for shading, textures and compositing"""

    active_input: int | None
    """ Index of the active input

    :type: int | None
    """

    active_output: int | None
    """ Index of the active output

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    bl_description: str
    """ 

    :type: str
    """

    bl_icon: str
    """ The node tree icon

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ The node tree label

    :type: str
    """

    grease_pencil: GreasePencil
    """ Grease Pencil data-block

    :type: GreasePencil
    """

    inputs: NodeTreeInputs
    """ Node tree inputs

    :type: NodeTreeInputs
    """

    links: NodeLinks
    """ 

    :type: NodeLinks
    """

    nodes: Nodes
    """ 

    :type: Nodes
    """

    outputs: NodeTreeOutputs
    """ Node tree outputs

    :type: NodeTreeOutputs
    """

    type: str
    """ Node Tree type (deprecated, bl_idname is the actual node tree type identifier)

    :type: str
    """

    view_center: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    def interface_update(self, context: Context):
        """Updated node group interface

        :param context:
        :type context: Context
        """
        ...

    @classmethod
    def poll(cls, context: Context) -> bool:
        """Check visibility in the editor

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        ...

    def update(self):
        """Update on editor changes"""
        ...

    @classmethod
    def get_from_context(cls, context: Context):
        """Get a node tree from the context

                :param context:
                :type context: Context
                :return: result_1, Active node tree from context, `NodeTree`

        result_2, ID data block that owns the node tree, `ID`

        result_3, Original ID data block selected from the context, `ID`
        """
        ...

class Object(ID, bpy_struct):
    """Object data-block defining an object in a scene"""

    active_material: Material | None
    """ Active material being displayed

    :type: Material | None
    """

    active_material_index: int | None
    """ Index of active material slot

    :type: int | None
    """

    active_shape_key: ShapeKey
    """ Current shape key

    :type: ShapeKey
    """

    active_shape_key_index: int | None
    """ Current shape key index

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    animation_visualization: AnimViz
    """ Animation data for this data-block

    :type: AnimViz
    """

    bound_box: bpy_prop_array[float]
    """ Object's bounding box in object-space coordinates, all values are -1.0 when not available

    :type: bpy_prop_array[float]
    """

    collision: CollisionSettings
    """ Settings for using the object as a collider in physics simulation

    :type: CollisionSettings
    """

    color: bpy_prop_array[float]
    """ Object color and alpha, used when faces have the ObColor mode enabled

    :type: bpy_prop_array[float]
    """

    constraints: ObjectConstraints
    """ Constraints affecting the transformation of the object

    :type: ObjectConstraints
    """

    cycles: CyclesObjectSettings
    """ Cycles object settings

    :type: CyclesObjectSettings
    """

    cycles_visibility: CyclesVisibilitySettings
    """ Cycles visibility settings

    :type: CyclesVisibilitySettings
    """

    data: ID
    """ Object data

    :type: ID
    """

    delta_location: bpy_prop_array[float]
    """ Extra translation added to the location of the object

    :type: bpy_prop_array[float]
    """

    delta_rotation_euler: bpy_prop_array[float]
    """ Extra rotation added to the rotation of the object (when using Euler rotations)

    :type: bpy_prop_array[float]
    """

    delta_rotation_quaternion: bpy_prop_array[float]
    """ Extra rotation added to the rotation of the object (when using Quaternion rotations)

    :type: bpy_prop_array[float]
    """

    delta_scale: bpy_prop_array[float]
    """ Extra scaling added to the scale of the object

    :type: bpy_prop_array[float]
    """

    dimensions: bpy_prop_array[float]
    """ Absolute bounding box dimensions of the object

    :type: bpy_prop_array[float]
    """

    draw_bounds_type: str
    """ Object boundary display type

    :type: str
    """

    draw_type: str
    """ Maximum draw type to display object with in viewport

    :type: str
    """

    dupli_faces_scale: float
    """ Scale the DupliFace objects

    :type: float
    """

    dupli_frames_end: int
    """ End frame for DupliFrames

    :type: int
    """

    dupli_frames_off: int
    """ Recurring frames to exclude from the Dupliframes

    :type: int
    """

    dupli_frames_on: int
    """ Number of frames to use between DupOff frames

    :type: int
    """

    dupli_frames_start: int
    """ Start frame for DupliFrames

    :type: int
    """

    dupli_group: Group
    """ Instance an existing group

    :type: Group
    """

    dupli_list: bpy_prop_collection[DupliObject]
    """ Object duplis

    :type: bpy_prop_collection[DupliObject]
    """

    dupli_type: str
    """ If not None, object duplication method to use

    :type: str
    """

    empty_draw_size: float
    """ Size of display for empties in the viewport

    :type: float
    """

    empty_draw_type: str
    """ Viewport display style for empties

    :type: str
    """

    empty_image_offset: bpy_prop_array[float]
    """ Origin offset distance

    :type: bpy_prop_array[float]
    """

    field: FieldSettings
    """ Settings for using the object as a field in physics simulation

    :type: FieldSettings
    """

    game: GameObjectSettings
    """ Game engine related settings for the object

    :type: GameObjectSettings
    """

    grease_pencil: GreasePencil
    """ Grease Pencil data-block

    :type: GreasePencil
    """

    hide: bool
    """ Restrict visibility in the viewport

    :type: bool
    """

    hide_render: bool
    """ Restrict renderability

    :type: bool
    """

    hide_select: bool
    """ Restrict selection in the viewport

    :type: bool
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    is_duplicator: bool
    """ 

    :type: bool
    """

    layers: list[bool]
    """ Layers the object is on

    :type: list[bool]
    """

    layers_local_view: list[bool]
    """ 3D local view layers the object is on

    :type: list[bool]
    """

    location: bpy_prop_array[float]
    """ Location of the object

    :type: bpy_prop_array[float]
    """

    lock_location: list[bool]
    """ Lock editing of location in the interface

    :type: list[bool]
    """

    lock_rotation: list[bool]
    """ Lock editing of rotation in the interface

    :type: list[bool]
    """

    lock_rotation_w: bool
    """ Lock editing of 'angle' component of four-component rotations in the interface

    :type: bool
    """

    lock_rotations_4d: bool
    """ Lock editing of four component rotations by components (instead of as Eulers)

    :type: bool
    """

    lock_scale: list[bool]
    """ Lock editing of scale in the interface

    :type: list[bool]
    """

    lod_levels: bpy_prop_collection[LodLevel]
    """ A collection of detail levels to automatically switch between

    :type: bpy_prop_collection[LodLevel]
    """

    material_slots: bpy_prop_collection[MaterialSlot]
    """ Material slots in the object

    :type: bpy_prop_collection[MaterialSlot]
    """

    matrix_basis: bpy_prop_array[float]
    """ Matrix access to location, rotation and scale (including deltas), before constraints and parenting are applied

    :type: bpy_prop_array[float]
    """

    matrix_local: bpy_prop_array[float]
    """ Parent relative transformation matrix - WARNING: Only takes into account 'Object' parenting, so e.g. in case of bone parenting you get a matrix relative to the Armature object, not to the actual parent bone

    :type: bpy_prop_array[float]
    """

    matrix_parent_inverse: bpy_prop_array[float]
    """ Inverse of object's parent matrix at time of parenting

    :type: bpy_prop_array[float]
    """

    matrix_world: bpy_prop_array[float]
    """ Worldspace transformation matrix

    :type: bpy_prop_array[float]
    """

    mode: str
    """ Object interaction mode

    :type: str
    """

    modifiers: ObjectModifiers
    """ Modifiers affecting the geometric data of the object

    :type: ObjectModifiers
    """

    motion_path: MotionPath
    """ Motion Path for this element

    :type: MotionPath
    """

    parent: Object
    """ Parent Object

    :type: Object
    """

    parent_bone: str
    """ Name of parent bone in case of a bone parenting relation

    :type: str
    """

    parent_type: str
    """ Type of parent relation

    :type: str
    """

    parent_vertices: bpy_prop_array[int]
    """ Indices of vertices in case of a vertex parenting relation

    :type: bpy_prop_array[int]
    """

    particle_systems: ParticleSystems
    """ Particle systems emitted from the object

    :type: ParticleSystems
    """

    pass_index: int
    """ Index number for the "Object Index" render pass

    :type: int
    """

    pose: Pose
    """ Current pose for armatures

    :type: Pose
    """

    pose_library: Action
    """ Action used as a pose library for armatures

    :type: Action
    """

    proxy: Object
    """ Library object this proxy object controls

    :type: Object
    """

    proxy_group: Object
    """ Library group duplicator object this proxy object controls

    :type: Object
    """

    rigid_body: RigidBodyObject
    """ Settings for rigid body simulation

    :type: RigidBodyObject
    """

    rigid_body_constraint: RigidBodyConstraint
    """ Constraint constraining rigid bodies

    :type: RigidBodyConstraint
    """

    rotation_axis_angle: bpy_prop_array[float]
    """ Angle of Rotation for Axis-Angle rotation representation

    :type: bpy_prop_array[float]
    """

    rotation_euler: bpy_prop_array[float]
    """ Rotation in Eulers

    :type: bpy_prop_array[float]
    """

    rotation_mode: str
    """ 

    :type: str
    """

    rotation_quaternion: bpy_prop_array[float]
    """ Rotation in Quaternions

    :type: bpy_prop_array[float]
    """

    scale: bpy_prop_array[float]
    """ Scaling of the object

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Object selection state

    :type: bool
    """

    show_all_edges: bool
    """ Display all edges for mesh objects

    :type: bool
    """

    show_axis: bool
    """ Display the object's origin and axes

    :type: bool
    """

    show_bounds: bool
    """ Display the object's bounds

    :type: bool
    """

    show_name: bool
    """ Display the object's name

    :type: bool
    """

    show_only_shape_key: bool
    """ Always show the current Shape for this Object

    :type: bool
    """

    show_texture_space: bool
    """ Display the object's texture space

    :type: bool
    """

    show_transparent: bool
    """ Display material transparency in the object (unsupported for duplicator drawing)

    :type: bool
    """

    show_wire: bool
    """ Add the object's wireframe over solid drawing

    :type: bool
    """

    show_x_ray: bool
    """ Make the object draw in front of others (unsupported for duplicator drawing)

    :type: bool
    """

    slow_parent_offset: float
    """ Delay in the parent relationship

    :type: float
    """

    soft_body: SoftBodySettings
    """ Settings for soft body simulation

    :type: SoftBodySettings
    """

    track_axis: str
    """ Axis that points in 'forward' direction (applies to DupliFrame when parent 'Follow' is enabled)

    :type: str
    """

    type: str
    """ Type of Object

    :type: str
    """

    up_axis: str
    """ Axis that points in the upward direction (applies to DupliFrame when parent 'Follow' is enabled)

    :type: str
    """

    use_dupli_faces_scale: bool
    """ Scale dupli based on face size

    :type: bool
    """

    use_dupli_frames_speed: bool
    """ Set dupliframes to use the current frame instead of parent curve's evaluation time

    :type: bool
    """

    use_dupli_vertices_rotation: bool
    """ Rotate dupli according to vertex normal

    :type: bool
    """

    use_dynamic_topology_sculpting: bool
    """ 

    :type: bool
    """

    use_extra_recalc_data: bool
    """ Refresh this object's data again on frame changes, dependency graph hack

    :type: bool
    """

    use_extra_recalc_object: bool
    """ Refresh this object again on frame changes, dependency graph hack

    :type: bool
    """

    use_shape_key_edit_mode: bool
    """ Apply shape keys in edit mode (for Meshes only)

    :type: bool
    """

    use_slow_parent: bool
    """ Create a delay in the parent relationship (beware: this isn't renderfarm safe and may be invalid after jumping around the timeline)

    :type: bool
    """

    vertex_groups: VertexGroups
    """ Vertex groups of the object

    :type: VertexGroups
    """

    children: typing.Any
    """ All the children of this object
(readonly)"""

    users_group: typing.Any
    """ The groups this object is in
(readonly)"""

    users_scene: typing.Any
    """ The scenes this object is in
(readonly)"""

    def convert_space(
        self,
        pose_bone: PoseBone | None = None,
        matrix: typing.Any | None = (
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
            0.0,
        ),
        from_space: str | None = "WORLD",
        to_space: str | None = "WORLD",
    ) -> bpy_prop_array[float]:
        """Convert (transform) the given matrix from one space to another

                :param pose_bone: Bone to use to define spaces (may be None, in which case only the two 'WORLD' and 'LOCAL' spaces are usable)
                :type pose_bone: PoseBone | None
                :param matrix: The matrix to transform
                :type matrix: typing.Any | None
                :param from_space: The space in which 'matrix' is currently

        WORLD World Space, The most gobal space in Blender.

        POSE Pose Space, The pose space of a bone (its armature's object space).

        LOCAL_WITH_PARENT Local With Parent, The local space of a bone's parent bone.

        LOCAL Local Space, The local space of an object/bone.
                :type from_space: str | None
                :param to_space: The space to which you want to transform 'matrix'

        WORLD World Space, The most gobal space in Blender.

        POSE Pose Space, The pose space of a bone (its armature's object space).

        LOCAL_WITH_PARENT Local With Parent, The local space of a bone's parent bone.

        LOCAL Local Space, The local space of an object/bone.
                :type to_space: str | None
                :return: The transformed matrix
                :rtype: bpy_prop_array[float]
        """
        ...

    def calc_matrix_camera(
        self,
        x: typing.Any | None = 1,
        y: typing.Any | None = 1,
        scale_x: typing.Any | None = 1.0,
        scale_y: typing.Any | None = 1.0,
    ) -> bpy_prop_array[float]:
        """Generate the camera projection matrix of this object (mostly useful for Camera and Lamp types)

        :param x: Width of the render area
        :type x: typing.Any | None
        :param y: Height of the render area
        :type y: typing.Any | None
        :param scale_x: Width scaling factor
        :type scale_x: typing.Any | None
        :param scale_y: height scaling factor
        :type scale_y: typing.Any | None
        :return: The camera projection matrix
        :rtype: bpy_prop_array[float]
        """
        ...

    def camera_fit_coords(self, scene: Scene | None, coordinates: typing.Any):
        """Compute the coordinate (and scale for ortho cameras) given object should be to 'see' all given coordinates

                :param scene: Scene to get render size information from, if available
                :type scene: Scene | None
                :param coordinates: Coordinates to fit in
                :type coordinates: typing.Any
                :return: co_return, The location to aim to be able to see all given points, float array of 3 items in [-inf, inf]

        scale_return, The ortho scale to aim to be able to see all given points (if relevant), float in [-inf, inf]
        """
        ...

    def to_mesh(
        self,
        scene: Scene,
        apply_modifiers: bool | None,
        settings: str | None,
        calc_tessface: bool | typing.Any | None = True,
        calc_undeformed: bool | typing.Any | None = False,
    ) -> Mesh:
        """Create a Mesh data-block with modifiers applied

                :param scene: Scene within which to evaluate modifiers
                :type scene: Scene
                :param apply_modifiers: Apply modifiers
                :type apply_modifiers: bool | None
                :param settings: Modifier settings to apply

        PREVIEW Preview, Apply modifier preview settings.

        RENDER Render, Apply modifier render settings.
                :type settings: str | None
                :param calc_tessface: Calculate Tessellation, Calculate tessellation faces
                :type calc_tessface: bool | typing.Any | None
                :param calc_undeformed: Calculate Undeformed, Calculate undeformed vertex coordinates
                :type calc_undeformed: bool | typing.Any | None
                :return: Mesh created from object, remove it if it is only used for export
                :rtype: Mesh
        """
        ...

    def dupli_list_create(self, scene: Scene, settings: str | None = "VIEWPORT"):
        """Create a list of dupli objects for this object, needs to be freed manually with free_dupli_list to restore the objects real matrix and layers

                :param scene: Scene within which to evaluate duplis
                :type scene: Scene
                :param settings: Generate texture coordinates for rendering

        VIEWPORT Viewport, Generate duplis using viewport settings.

        PREVIEW Preview, Generate duplis using preview settings.

        RENDER Render, Generate duplis using render settings.
                :type settings: str | None
        """
        ...

    def dupli_list_clear(self):
        """Free the list of dupli objects"""
        ...

    def find_armature(self) -> Object:
        """Find armature influencing this object as a parent or via a modifier

        :return: Armature object influencing this object or NULL
        :rtype: Object
        """
        ...

    def shape_key_add(
        self, name: str | typing.Any = "Key", from_mix: bool | typing.Any | None = True
    ) -> ShapeKey:
        """Add shape key to this object

        :param name: Unique name for the new keyblock
        :type name: str | typing.Any
        :param from_mix: Create new shape from existing mix of shapes
        :type from_mix: bool | typing.Any | None
        :return: New shape keyblock
        :rtype: ShapeKey
        """
        ...

    def shape_key_remove(self, key: ShapeKey):
        """Remove a Shape Key from this object

        :param key: Keyblock to be removed
        :type key: ShapeKey
        """
        ...

    def ray_cast(
        self,
        origin: collections.abc.Iterable[float] | None,
        direction: collections.abc.Iterable[float] | None,
        distance: typing.Any | None = 1.70141e38,
    ):
        """Cast a ray onto in object space

                :param origin:
                :type origin: collections.abc.Iterable[float] | None
                :param direction:
                :type direction: collections.abc.Iterable[float] | None
                :param distance: Maximum distance
                :type distance: typing.Any | None
                :return: result, boolean

        location, The hit location of this ray cast, float array of 3 items in [-inf, inf]

        normal, The face normal at the ray cast hit location, float array of 3 items in [-inf, inf]

        index, The face index, -1 when original data isn't available, int in [-inf, inf]
        """
        ...

    def closest_point_on_mesh(
        self,
        origin: collections.abc.Iterable[float] | None,
        distance: typing.Any | None = 1.84467e19,
    ):
        """Find the nearest point in object space

                :param origin:
                :type origin: collections.abc.Iterable[float] | None
                :param distance: Maximum distance
                :type distance: typing.Any | None
                :return: result, boolean

        location, The location on the object closest to the point, float array of 3 items in [-inf, inf]

        normal, The face normal at the closest point, float array of 3 items in [-inf, inf]

        index, The face index, -1 when original data isn't available, int in [-inf, inf]
        """
        ...

    def is_visible(self, scene: Scene) -> bool:
        """Determine if object is visible in a given scene

        :param scene:
        :type scene: Scene
        :return: Object visibility
        :rtype: bool
        """
        ...

    def is_modified(self, scene: Scene, settings: str | None) -> bool:
        """Determine if this object is modified from the base mesh data

                :param scene:
                :type scene: Scene
                :param settings: Modifier settings to apply

        PREVIEW Preview, Apply modifier preview settings.

        RENDER Render, Apply modifier render settings.
                :type settings: str | None
                :return: Object visibility
                :rtype: bool
        """
        ...

    def is_deform_modified(self, scene: Scene, settings: str | None) -> bool:
        """Determine if this object is modified by a deformation from the base mesh data

                :param scene:
                :type scene: Scene
                :param settings: Modifier settings to apply

        PREVIEW Preview, Apply modifier preview settings.

        RENDER Render, Apply modifier render settings.
                :type settings: str | None
                :return: Object visibility
                :rtype: bool
        """
        ...

    def update_from_editmode(self) -> bool:
        """Load the objects edit-mode data into the object data

        :return: Success
        :rtype: bool
        """
        ...

    def cache_release(self):
        """Release memory used by caches associated with this object. Intended to be used by render engines only"""
        ...

class PaintCurve(ID, bpy_struct): ...

class Palette(ID, bpy_struct):
    colors: PaletteColors
    """ 

    :type: PaletteColors
    """

class ParticleSettings(ID, bpy_struct):
    """Particle settings, reusable by multiple particle systems"""

    active_dupliweight: ParticleDupliWeight
    """ 

    :type: ParticleDupliWeight
    """

    active_dupliweight_index: int | None
    """ 

    :type: int | None
    """

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    adaptive_angle: int
    """ How many degrees path has to curve to make another render segment

    :type: int
    """

    adaptive_pixel: int
    """ How many pixels path has to cover to make another render segment

    :type: int
    """

    angular_velocity_factor: float
    """ Angular velocity amount (in radians per second)

    :type: float
    """

    angular_velocity_mode: str
    """ What axis is used to change particle rotation with time

    :type: str
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    apply_effector_to_children: bool
    """ Apply effectors to children

    :type: bool
    """

    apply_guide_to_children: bool
    """ 

    :type: bool
    """

    bending_random: float
    """ Random stiffness of hairs

    :type: float
    """

    billboard_align: str
    """ In respect to what the billboards are aligned

    :type: str
    """

    billboard_animation: str
    """ How to animate billboard textures

    :type: str
    """

    billboard_object: Object
    """ Billboards face this object (default is active camera)

    :type: Object
    """

    billboard_offset: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    billboard_offset_split: str
    """ How to offset billboard textures

    :type: str
    """

    billboard_size: bpy_prop_array[float]
    """ Scale billboards relative to particle size

    :type: bpy_prop_array[float]
    """

    billboard_tilt: float
    """ Tilt of the billboards

    :type: float
    """

    billboard_tilt_random: float
    """ Random tilt of the billboards

    :type: float
    """

    billboard_uv_split: int
    """ Number of rows/columns to split UV coordinates for billboards

    :type: int
    """

    billboard_velocity_head: float
    """ Scale billboards by velocity

    :type: float
    """

    billboard_velocity_tail: float
    """ Scale billboards by velocity

    :type: float
    """

    boids: BoidSettings
    """ 

    :type: BoidSettings
    """

    branch_threshold: float
    """ Threshold of branching

    :type: float
    """

    brownian_factor: float
    """ Amount of random, erratic particle movement

    :type: float
    """

    child_length: float
    """ Length of child paths

    :type: float
    """

    child_length_threshold: float
    """ Amount of particles left untouched by child path length

    :type: float
    """

    child_nbr: int
    """ Number of children/parent

    :type: int
    """

    child_parting_factor: float
    """ Create parting in the children based on parent strands

    :type: float
    """

    child_parting_max: float
    """ Maximum root to tip angle (tip distance/root distance for long hair)

    :type: float
    """

    child_parting_min: float
    """ Minimum root to tip angle (tip distance/root distance for long hair)

    :type: float
    """

    child_radius: float
    """ Radius of children around parent

    :type: float
    """

    child_roundness: float
    """ Roundness of children around parent

    :type: float
    """

    child_size: float
    """ A multiplier for the child particle size

    :type: float
    """

    child_size_random: float
    """ Random variation to the size of the child particles

    :type: float
    """

    child_type: str
    """ Create child particles

    :type: str
    """

    clump_curve: CurveMapping
    """ Curve defining clump tapering

    :type: CurveMapping
    """

    clump_factor: float
    """ Amount of clumping

    :type: float
    """

    clump_noise_size: float
    """ Size of clump noise

    :type: float
    """

    clump_shape: float
    """ Shape of clumping

    :type: float
    """

    collision_group: Group
    """ Limit colliders to this Group

    :type: Group
    """

    color_maximum: float
    """ Maximum length of the particle color vector

    :type: float
    """

    count: int
    """ Total number of particles

    :type: int
    """

    courant_target: float
    """ The relative distance a particle can move before requiring more subframes (target Courant number); 0.01-0.3 is the recommended range

    :type: float
    """

    create_long_hair_children: bool
    """ Calculate children that suit long hair well

    :type: bool
    """

    cycles: CyclesCurveSettings
    """ Cycles hair settings

    :type: CyclesCurveSettings
    """

    damping: float
    """ Amount of damping

    :type: float
    """

    distribution: str
    """ How to distribute particles on selected element

    :type: str
    """

    drag_factor: float
    """ Amount of air-drag

    :type: float
    """

    draw_color: str
    """ Draw additional particle data as a color

    :type: str
    """

    draw_method: str
    """ How particles are drawn in viewport

    :type: str
    """

    draw_percentage: int
    """ Percentage of particles to display in 3D view

    :type: int
    """

    draw_size: int
    """ Size of particles on viewport in pixels (0=default)

    :type: int
    """

    draw_step: int
    """ How many steps paths are drawn with (power of 2)

    :type: int
    """

    dupli_group: Group
    """ Show Objects in this Group in place of particles

    :type: Group
    """

    dupli_object: Object
    """ Show this Object in place of particles

    :type: Object
    """

    dupli_weights: bpy_prop_collection[ParticleDupliWeight]
    """ Weights for all of the objects in the dupli group

    :type: bpy_prop_collection[ParticleDupliWeight]
    """

    effect_hair: float
    """ Hair stiffness for effectors

    :type: float
    """

    effector_amount: int
    """ How many particles are effectors (0 is all particles)

    :type: int
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    emit_from: str
    """ Where to emit particles from

    :type: str
    """

    factor_random: float
    """ Give the starting velocity a random variation

    :type: float
    """

    fluid: SPHFluidSettings
    """ 

    :type: SPHFluidSettings
    """

    force_field_1: FieldSettings
    """ 

    :type: FieldSettings
    """

    force_field_2: FieldSettings
    """ 

    :type: FieldSettings
    """

    frame_end: float
    """ Frame number to stop emitting particles

    :type: float
    """

    frame_start: float
    """ Frame number to start emitting particles

    :type: float
    """

    grid_random: float
    """ Add random offset to the grid locations

    :type: float
    """

    grid_resolution: int
    """ The resolution of the particle grid

    :type: int
    """

    hair_length: float
    """ Length of the hair

    :type: float
    """

    hair_step: int
    """ Number of hair segments

    :type: int
    """

    hexagonal_grid: bool
    """ Create the grid in a hexagonal pattern

    :type: bool
    """

    integrator: str
    """ Algorithm used to calculate physics, from the fastest to the most stable/accurate: Midpoint, Euler, Verlet, RK4 (Old)

    :type: str
    """

    invert_grid: bool
    """ Invert what is considered object and what is not

    :type: bool
    """

    is_fluid: bool
    """ Particles were created by a fluid simulation

    :type: bool
    """

    jitter_factor: float
    """ Amount of jitter applied to the sampling

    :type: float
    """

    keyed_loops: int
    """ Number of times the keys are looped

    :type: int
    """

    keys_step: int
    """ 

    :type: int
    """

    kink: str
    """ Type of periodic offset on the path

    :type: str
    """

    kink_amplitude: float
    """ The amplitude of the offset

    :type: float
    """

    kink_amplitude_clump: float
    """ How much clump affects kink amplitude

    :type: float
    """

    kink_amplitude_random: float
    """ Random variation of the amplitude

    :type: float
    """

    kink_axis: str
    """ Which axis to use for offset

    :type: str
    """

    kink_axis_random: float
    """ Random variation of the orientation

    :type: float
    """

    kink_extra_steps: int
    """ Extra steps for resolution of special kink features

    :type: int
    """

    kink_flat: float
    """ How flat the hairs are

    :type: float
    """

    kink_frequency: float
    """ The frequency of the offset (1/total length)

    :type: float
    """

    kink_shape: float
    """ Adjust the offset to the beginning/end

    :type: float
    """

    length_random: float
    """ Give path length a random variation

    :type: float
    """

    lifetime: float
    """ Life span of the particles

    :type: float
    """

    lifetime_random: float
    """ Give the particle life a random variation

    :type: float
    """

    line_length_head: float
    """ Length of the line's head

    :type: float
    """

    line_length_tail: float
    """ Length of the line's tail

    :type: float
    """

    lock_billboard: bool
    """ Lock the billboards align axis

    :type: bool
    """

    lock_boids_to_surface: bool
    """ Constrain boids to a surface

    :type: bool
    """

    mass: float
    """ Mass of the particles

    :type: float
    """

    material: int
    """ Index of material slot used for rendering particles

    :type: int
    """

    material_slot: str
    """ Material slot used for rendering particles

    :type: str
    """

    normal_factor: float
    """ Let the surface normal give the particle a starting velocity

    :type: float
    """

    object_align_factor: bpy_prop_array[float]
    """ Let the emitter object orientation give the particle a starting velocity

    :type: bpy_prop_array[float]
    """

    object_factor: float
    """ Let the object give the particle a starting velocity

    :type: float
    """

    particle_factor: float
    """ Let the target particle give the particle a starting velocity

    :type: float
    """

    particle_size: float
    """ The size of the particles

    :type: float
    """

    path_end: float
    """ End time of drawn path

    :type: float
    """

    path_start: float
    """ Starting time of drawn path

    :type: float
    """

    phase_factor: float
    """ Rotation around the chosen orientation axis

    :type: float
    """

    phase_factor_random: float
    """ Randomize rotation around the chosen orientation axis

    :type: float
    """

    physics_type: str
    """ Particle physics type

    :type: str
    """

    react_event: str
    """ The event of target particles to react on

    :type: str
    """

    reactor_factor: float
    """ Let the vector away from the target particle's location give the particle a starting velocity

    :type: float
    """

    regrow_hair: bool
    """ Regrow hair for each frame

    :type: bool
    """

    render_step: int
    """ How many steps paths are rendered with (power of 2)

    :type: int
    """

    render_type: str
    """ How particles are rendered

    :type: str
    """

    rendered_child_count: int
    """ Number of children/parent for rendering

    :type: int
    """

    rotation_factor_random: float
    """ Randomize particle orientation

    :type: float
    """

    rotation_mode: str
    """ Particle orientation axis (does not affect Explode modifier's results)

    :type: str
    """

    roughness_1: float
    """ Amount of location dependent rough

    :type: float
    """

    roughness_1_size: float
    """ Size of location dependent rough

    :type: float
    """

    roughness_2: float
    """ Amount of random rough

    :type: float
    """

    roughness_2_size: float
    """ Size of random rough

    :type: float
    """

    roughness_2_threshold: float
    """ Amount of particles left untouched by random rough

    :type: float
    """

    roughness_curve: CurveMapping
    """ Curve defining roughness

    :type: CurveMapping
    """

    roughness_end_shape: float
    """ Shape of end point rough

    :type: float
    """

    roughness_endpoint: float
    """ Amount of end point rough

    :type: float
    """

    show_guide_hairs: bool
    """ Show guide hairs

    :type: bool
    """

    show_hair_grid: bool
    """ Show hair simulation grid

    :type: bool
    """

    show_health: bool
    """ Draw boid health

    :type: bool
    """

    show_number: bool
    """ Show particle number

    :type: bool
    """

    show_size: bool
    """ Show particle size

    :type: bool
    """

    show_unborn: bool
    """ Show particles before they are emitted

    :type: bool
    """

    show_velocity: bool
    """ Show particle velocity

    :type: bool
    """

    simplify_rate: float
    """ Speed of simplification

    :type: float
    """

    simplify_refsize: int
    """ Reference size in pixels, after which simplification begins

    :type: int
    """

    simplify_transition: float
    """ Transition period for fading out strands

    :type: float
    """

    simplify_viewport: float
    """ Speed of Simplification

    :type: float
    """

    size_random: float
    """ Give the particle size a random variation

    :type: float
    """

    subframes: int
    """ Subframes to simulate for improved stability and finer granularity simulations (dt = timestep / (subframes + 1))

    :type: int
    """

    tangent_factor: float
    """ Let the surface tangent give the particle a starting velocity

    :type: float
    """

    tangent_phase: float
    """ Rotate the surface tangent

    :type: float
    """

    texture_slots: ParticleSettingsTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: ParticleSettingsTextureSlots
    """

    time_tweak: float
    """ A multiplier for physics timestep (1.0 means one frame = 1/25 seconds)

    :type: float
    """

    timestep: float
    """ The simulation timestep per frame (seconds per frame)

    :type: float
    """

    trail_count: int
    """ Number of trail particles

    :type: int
    """

    type: str
    """ Particle Type

    :type: str
    """

    use_absolute_path_time: bool
    """ Path timing is in absolute frames

    :type: bool
    """

    use_adaptive_subframes: bool
    """ Automatically set the number of subframes

    :type: bool
    """

    use_advanced_hair: bool
    """ Use full physics calculations for growing hair

    :type: bool
    """

    use_clump_curve: bool
    """ Use a curve to define clump tapering

    :type: bool
    """

    use_clump_noise: bool
    """ Create random clumps around the parent

    :type: bool
    """

    use_dead: bool
    """ Show particles after they have died

    :type: bool
    """

    use_die_on_collision: bool
    """ Particles die when they collide with a deflector object

    :type: bool
    """

    use_dynamic_rotation: bool
    """ Particle rotations are affected by collisions and effectors

    :type: bool
    """

    use_emit_random: bool
    """ Emit in random order of elements

    :type: bool
    """

    use_even_distribution: bool
    """ Use even distribution from faces based on face areas or edge lengths

    :type: bool
    """

    use_global_dupli: bool
    """ Use object's global coordinates for duplication

    :type: bool
    """

    use_group_count: bool
    """ Use object multiple times in the same group

    :type: bool
    """

    use_group_pick_random: bool
    """ Pick objects from group randomly

    :type: bool
    """

    use_hair_bspline: bool
    """ Interpolate hair using B-Splines

    :type: bool
    """

    use_modifier_stack: bool
    """ Emit particles from mesh with modifiers applied (must use same subsurf level for viewport and render for correct results)

    :type: bool
    """

    use_multiply_size_mass: bool
    """ Multiply mass by particle size

    :type: bool
    """

    use_parent_particles: bool
    """ Render parent particles

    :type: bool
    """

    use_react_multiple: bool
    """ React multiple times

    :type: bool
    """

    use_react_start_end: bool
    """ Give birth to unreacted particles eventually

    :type: bool
    """

    use_render_adaptive: bool
    """ Draw steps of the particle path

    :type: bool
    """

    use_render_emitter: bool
    """ Render emitter Object also

    :type: bool
    """

    use_rotation_dupli: bool
    """ Use object's rotation for duplication (global x-axis is aligned particle rotation axis)

    :type: bool
    """

    use_rotations: bool
    """ Calculate particle rotations

    :type: bool
    """

    use_roughness_curve: bool
    """ Use a curve to define roughness

    :type: bool
    """

    use_scale_dupli: bool
    """ Use object's scale for duplication

    :type: bool
    """

    use_self_effect: bool
    """ Particle effectors affect themselves

    :type: bool
    """

    use_simplify: bool
    """ Remove child strands as the object becomes smaller on the screen

    :type: bool
    """

    use_simplify_viewport: bool
    """ 

    :type: bool
    """

    use_size_deflect: bool
    """ Use particle's size in deflection

    :type: bool
    """

    use_strand_primitive: bool
    """ Use the strand primitive for rendering

    :type: bool
    """

    use_velocity_length: bool
    """ Multiply line length by particle speed

    :type: bool
    """

    use_whole_group: bool
    """ Use whole group at once

    :type: bool
    """

    userjit: int
    """ Emission locations / face (0 = automatic)

    :type: int
    """

    virtual_parents: float
    """ Relative amount of virtual parents

    :type: float
    """

class Scene(ID, bpy_struct):
    """Scene data block, consisting in objects and defining time and render related settings"""

    active_clip: MovieClip | None
    """ Active movie clip used for constraints and viewport drawing

    :type: MovieClip | None
    """

    active_layer: int
    """ Active scene layer index

    :type: int
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    audio_distance_model: str
    """ Distance model for distance attenuation calculation

    :type: str
    """

    audio_doppler_factor: float
    """ Pitch factor for Doppler effect calculation

    :type: float
    """

    audio_doppler_speed: float
    """ Speed of sound for Doppler effect calculation

    :type: float
    """

    audio_volume: float
    """ Audio volume

    :type: float
    """

    background_set: Scene
    """ Background set scene

    :type: Scene
    """

    camera: Object
    """ Active camera, used for rendering the scene

    :type: Object
    """

    cursor_location: bpy_prop_array[float]
    """ 3D cursor location

    :type: bpy_prop_array[float]
    """

    cycles: CyclesRenderSettings
    """ Cycles render settings

    :type: CyclesRenderSettings
    """

    cycles_curves: CyclesCurveRenderSettings
    """ Cycles hair rendering settings

    :type: CyclesCurveRenderSettings
    """

    depsgraph: Depsgraph
    """ Dependencies in the scene data

    :type: Depsgraph
    """

    display_settings: ColorManagedDisplaySettings
    """ Settings of device saved image would be displayed on

    :type: ColorManagedDisplaySettings
    """

    frame_current: int
    """ Current Frame, to update animation data from python frame_set() instead

    :type: int
    """

    frame_current_final: float
    """ Current frame with subframe and time remapping applied

    :type: float
    """

    frame_end: int
    """ Final frame of the playback/rendering range

    :type: int
    """

    frame_preview_end: int
    """ Alternative end frame for UI playback

    :type: int
    """

    frame_preview_start: int
    """ Alternative start frame for UI playback

    :type: int
    """

    frame_start: int
    """ First frame of the playback/rendering range

    :type: int
    """

    frame_step: int
    """ Number of frames to skip forward while rendering/playing back each frame

    :type: int
    """

    frame_subframe: float
    """ 

    :type: float
    """

    game_settings: SceneGameData
    """ 

    :type: SceneGameData
    """

    gravity: bpy_prop_array[float]
    """ Constant acceleration in a given direction

    :type: bpy_prop_array[float]
    """

    grease_pencil: GreasePencil
    """ Grease Pencil data-block

    :type: GreasePencil
    """

    is_nla_tweakmode: bool
    """ Whether there is any action referenced by NLA being edited (strictly read-only)

    :type: bool
    """

    keying_sets: KeyingSets
    """ Absolute Keying Sets for this Scene

    :type: KeyingSets
    """

    keying_sets_all: KeyingSetsAll
    """ All Keying Sets available for use (Builtins and Absolute Keying Sets for this Scene)

    :type: KeyingSetsAll
    """

    layers: list[bool]
    """ Visible layers - Shift-Click/Drag to select multiple layers

    :type: list[bool]
    """

    lock_frame_selection_to_range: bool
    """ Don't allow frame to be selected with mouse outside of frame range

    :type: bool
    """

    node_tree: NodeTree
    """ Compositing node tree

    :type: NodeTree
    """

    object_bases: SceneBases
    """ 

    :type: SceneBases
    """

    objects: SceneObjects
    """ 

    :type: SceneObjects
    """

    orientations: bpy_prop_collection[TransformOrientation]
    """ 

    :type: bpy_prop_collection[TransformOrientation]
    """

    render: RenderSettings
    """ 

    :type: RenderSettings
    """

    rigidbody_world: RigidBodyWorld
    """ 

    :type: RigidBodyWorld
    """

    safe_areas: DisplaySafeAreas
    """ 

    :type: DisplaySafeAreas
    """

    sequence_editor: SequenceEditor
    """ 

    :type: SequenceEditor
    """

    sequencer_colorspace_settings: ColorManagedSequencerColorspaceSettings
    """ Settings of color space sequencer is working in

    :type: ColorManagedSequencerColorspaceSettings
    """

    show_keys_from_selected_only: bool
    """ Consider keyframes for active Object and/or its selected bones only (in timeline and when jumping between keyframes)

    :type: bool
    """

    sync_mode: str
    """ How to sync playback

    :type: str
    """

    timeline_markers: TimelineMarkers
    """ Markers used in all timelines for the current scene

    :type: TimelineMarkers
    """

    tool_settings: ToolSettings
    """ 

    :type: ToolSettings
    """

    unit_settings: UnitSettings
    """ Unit editing settings

    :type: UnitSettings
    """

    use_audio: bool
    """ Play back of audio from Sequence Editor will be muted

    :type: bool
    """

    use_audio_scrub: bool
    """ Play audio from Sequence Editor while scrubbing

    :type: bool
    """

    use_audio_sync: bool
    """ Play back and sync with audio clock, dropping frames if frame display is too slow

    :type: bool
    """

    use_frame_drop: bool
    """ Play back dropping frames if frame display is too slow

    :type: bool
    """

    use_gravity: bool
    """ Use global gravity for all dynamics

    :type: bool
    """

    use_nodes: bool
    """ Enable the compositing node tree

    :type: bool
    """

    use_preview_range: bool
    """ Use an alternative start/end frame range for animation playback and OpenGL renders instead of the Render properties start/end frame range

    :type: bool
    """

    use_stamp_note: str
    """ User defined note for the render stamping

    :type: str
    """

    view_settings: ColorManagedViewSettings
    """ Color management settings applied on image before saving

    :type: ColorManagedViewSettings
    """

    world: World
    """ World used for rendering the scene

    :type: World
    """

    def sequence_editor_create(self) -> SequenceEditor:
        """Ensure sequence editor is valid in this scene

        :return: New sequence editor data or NULL
        :rtype: SequenceEditor
        """
        ...

    def sequence_editor_clear(self):
        """Clear sequence editor in this scene"""
        ...

    def statistics(self) -> str | typing.Any:
        """statistics

        :return: Statistics
        :rtype: str | typing.Any
        """
        ...

    def frame_set(self, frame: int | None, subframe: typing.Any | None = 0.0):
        """Set scene frame updating all objects immediately

        :param frame: Frame number to set
        :type frame: int | None
        :param subframe: Sub-frame time, between 0.0 and 1.0
        :type subframe: typing.Any | None
        """
        ...

    def update(self):
        """Update data tagged to be updated from previous access to data or operators"""
        ...

    def uvedit_aspect(self, object: Object) -> bpy_prop_array[float]:
        """Get uv aspect for current object

        :param object: Object
        :type object: Object
        :return: aspect
        :rtype: bpy_prop_array[float]
        """
        ...

    def ray_cast(
        self,
        origin: collections.abc.Iterable[float] | None,
        direction: collections.abc.Iterable[float] | None,
        distance: typing.Any | None = 1.70141e38,
    ):
        """Cast a ray onto in object space

                :param origin:
                :type origin: collections.abc.Iterable[float] | None
                :param direction:
                :type direction: collections.abc.Iterable[float] | None
                :param distance: Maximum distance
                :type distance: typing.Any | None
                :return: result, boolean

        location, The hit location of this ray cast, float array of 3 items in [-inf, inf]

        normal, The face normal at the ray cast hit location, float array of 3 items in [-inf, inf]

        index, The face index, -1 when original data isn't available, int in [-inf, inf]

        object, Ray cast object, `Object`

        matrix, Matrix, float array of 16 items in [-inf, inf]
        """
        ...

    def collada_export(
        self,
        filepath: str | typing.Any,
        apply_modifiers: bool | typing.Any | None = False,
        export_mesh_type: typing.Any | None = 0,
        selected: bool | typing.Any | None = False,
        include_children: bool | typing.Any | None = False,
        include_armatures: bool | typing.Any | None = False,
        include_shapekeys: bool | typing.Any | None = False,
        deform_bones_only: bool | typing.Any | None = False,
        active_uv_only: bool | typing.Any | None = False,
        include_uv_textures: bool | typing.Any | None = False,
        include_material_textures: bool | typing.Any | None = False,
        use_texture_copies: bool | typing.Any | None = False,
        use_ngons: bool | typing.Any | None = True,
        use_object_instantiation: bool | typing.Any | None = True,
        use_blender_profile: bool | typing.Any | None = True,
        sort_by_name: bool | typing.Any | None = False,
        open_sim: bool | typing.Any | None = False,
        export_transformation_type: typing.Any | None = 0,
    ):
        """Export to collada file

        :param filepath: File Path, File path to write Collada file
        :type filepath: str | typing.Any
        :param apply_modifiers: Apply Modifiers, Apply modifiers
        :type apply_modifiers: bool | typing.Any | None
        :param export_mesh_type: Resolution, Modifier resolution for export
        :type export_mesh_type: typing.Any | None
        :param selected: Selection Only, Export only selected elements
        :type selected: bool | typing.Any | None
        :param include_children: Include Children, Export all children of selected objects (even if not selected)
        :type include_children: bool | typing.Any | None
        :param include_armatures: Include Armatures, Export related armatures (even if not selected)
        :type include_armatures: bool | typing.Any | None
        :param include_shapekeys: Include Shape Keys, Export all Shape Keys from Mesh Objects
        :type include_shapekeys: bool | typing.Any | None
        :param deform_bones_only: Deform Bones only, Only export deforming bones with armatures
        :type deform_bones_only: bool | typing.Any | None
        :param active_uv_only: Active UV Layer only, Export only the active UV Layer
        :type active_uv_only: bool | typing.Any | None
        :param include_uv_textures: Include UV Textures, Export textures assigned to the object UV maps
        :type include_uv_textures: bool | typing.Any | None
        :param include_material_textures: Include Material Textures, Export textures assigned to the object Materials
        :type include_material_textures: bool | typing.Any | None
        :param use_texture_copies: copy, Copy textures to same folder where the .dae file is exported
        :type use_texture_copies: bool | typing.Any | None
        :param use_ngons: Use NGons, Keep NGons in Export
        :type use_ngons: bool | typing.Any | None
        :param use_object_instantiation: Use Object Instances, Instantiate multiple Objects from same Data
        :type use_object_instantiation: bool | typing.Any | None
        :param use_blender_profile: Use Blender Profile, Export additional Blender specific information (for material, shaders, bones, etc.)
        :type use_blender_profile: bool | typing.Any | None
        :param sort_by_name: Sort by Object name, Sort exported data by Object name
        :type sort_by_name: bool | typing.Any | None
        :param open_sim: Export for SL/OpenSim, Compatibility mode for SL, OpenSim and similar online worlds
        :type open_sim: bool | typing.Any | None
        :param export_transformation_type: Transformation, Transformation type for translation, scale and rotation
        :type export_transformation_type: typing.Any | None
        """
        ...

    def alembic_export(
        self,
        filepath: str | typing.Any,
        frame_start: typing.Any | None = 1,
        frame_end: typing.Any | None = 1,
        xform_samples: typing.Any | None = 1,
        geom_samples: typing.Any | None = 1,
        shutter_open: typing.Any | None = 0.0,
        shutter_close: typing.Any | None = 1.0,
        selected_only: bool | typing.Any | None = False,
        uvs: bool | typing.Any | None = True,
        normals: bool | typing.Any | None = True,
        vcolors: bool | typing.Any | None = False,
        apply_subdiv: bool | typing.Any | None = True,
        flatten: bool | typing.Any | None = False,
        visible_layers_only: bool | typing.Any | None = False,
        renderable_only: bool | typing.Any | None = False,
        face_sets: bool | typing.Any | None = False,
        subdiv_schema: bool | typing.Any | None = False,
        compression_type: str | None = "OGAWA",
        packuv: bool | typing.Any | None = False,
        scale: typing.Any | None = 1.0,
    ):
        """Export to Alembic file

        :param filepath: File Path, File path to write Alembic file
        :type filepath: str | typing.Any
        :param frame_start: Start, Start Frame
        :type frame_start: typing.Any | None
        :param frame_end: End, End Frame
        :type frame_end: typing.Any | None
        :param xform_samples: Xform samples, Transform samples per frame
        :type xform_samples: typing.Any | None
        :param geom_samples: Geom samples, Geometry samples per frame
        :type geom_samples: typing.Any | None
        :param shutter_open: Shutter open
        :type shutter_open: typing.Any | None
        :param shutter_close: Shutter close
        :type shutter_close: typing.Any | None
        :param selected_only: Selected only, Export only selected objects
        :type selected_only: bool | typing.Any | None
        :param uvs: UVs, Export UVs
        :type uvs: bool | typing.Any | None
        :param normals: Normals, Export cormals
        :type normals: bool | typing.Any | None
        :param vcolors: Vertex colors, Export vertex colors
        :type vcolors: bool | typing.Any | None
        :param apply_subdiv: Subsurfs as meshes, Export subdivision surfaces as meshes
        :type apply_subdiv: bool | typing.Any | None
        :param flatten: Flatten hierarchy, Flatten hierarchy
        :type flatten: bool | typing.Any | None
        :param visible_layers_only: Visible layers only, Export only objects in visible layers
        :type visible_layers_only: bool | typing.Any | None
        :param renderable_only: Renderable objects only, Export only objects marked renderable in the outliner
        :type renderable_only: bool | typing.Any | None
        :param face_sets: Facesets, Export face sets
        :type face_sets: bool | typing.Any | None
        :param subdiv_schema: Use Alembic subdivision Schema, Use Alembic subdivision Schema
        :type subdiv_schema: bool | typing.Any | None
        :param compression_type: Compression
        :type compression_type: str | None
        :param packuv: Export with packed UV islands, Export with packed UV islands
        :type packuv: bool | typing.Any | None
        :param scale: Scale, Value by which to enlarge or shrink the objects with respect to the world's origin
        :type scale: typing.Any | None
        """
        ...

class Screen(ID, bpy_struct):
    """Screen data-block, defining the layout of areas in a window"""

    areas: bpy_prop_collection[Area]
    """ Areas the screen is subdivided into

    :type: bpy_prop_collection[Area]
    """

    is_animation_playing: bool
    """ Animation playback is active

    :type: bool
    """

    scene: Scene
    """ Active scene to be edited in the screen

    :type: Scene
    """

    show_fullscreen: bool
    """ An area is maximized, filling this screen

    :type: bool
    """

    use_follow: bool
    """ Follow current frame in editors

    :type: bool
    """

    use_play_3d_editors: bool
    """ 

    :type: bool
    """

    use_play_animation_editors: bool
    """ 

    :type: bool
    """

    use_play_clip_editors: bool
    """ 

    :type: bool
    """

    use_play_image_editors: bool
    """ 

    :type: bool
    """

    use_play_node_editors: bool
    """ 

    :type: bool
    """

    use_play_properties_editors: bool
    """ 

    :type: bool
    """

    use_play_sequence_editors: bool
    """ 

    :type: bool
    """

    use_play_top_left_3d_editor: bool
    """ 

    :type: bool
    """

class Sound(ID, bpy_struct):
    """Sound data-block referencing an external or packed sound file"""

    filepath: str
    """ Sound sample file used by this Sound data-block

    :type: str
    """

    packed_file: PackedFile
    """ 

    :type: PackedFile
    """

    use_memory_cache: bool
    """ The sound file is decoded and loaded into RAM

    :type: bool
    """

    use_mono: bool
    """ If the file contains multiple audio channels they are rendered to a single one

    :type: bool
    """

    factory: typing.Any
    """ The aud.Factory object of the sound.
(readonly)"""

    def pack(self):
        """Pack the sound into the current blend file"""
        ...

    def unpack(self, method: str | None = "USE_LOCAL"):
        """Unpack the sound to the samples filename

        :param method: method, How to unpack
        :type method: str | None
        """
        ...

class Speaker(ID, bpy_struct):
    """Speaker data-block for 3D audio speaker objects"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    attenuation: float
    """ How strong the distance affects volume, depending on distance model

    :type: float
    """

    cone_angle_inner: float
    """ Angle of the inner cone, in degrees, inside the cone the volume is 100 %

    :type: float
    """

    cone_angle_outer: float
    """ Angle of the outer cone, in degrees, outside this cone the volume is the outer cone volume, between inner and outer cone the volume is interpolated

    :type: float
    """

    cone_volume_outer: float
    """ Volume outside the outer cone

    :type: float
    """

    distance_max: float
    """ Maximum distance for volume calculation, no matter how far away the object is

    :type: float
    """

    distance_reference: float
    """ Reference distance at which volume is 100 %

    :type: float
    """

    muted: bool
    """ Mute the speaker

    :type: bool
    """

    pitch: float
    """ Playback pitch of the sound

    :type: float
    """

    relative: bool
    """ Whether the source is relative to the camera or not

    :type: bool
    """

    sound: Sound
    """ Sound data-block used by this speaker

    :type: Sound
    """

    volume: float
    """ How loud the sound is

    :type: float
    """

    volume_max: float
    """ Maximum volume, no matter how near the object is

    :type: float
    """

    volume_min: float
    """ Minimum volume, no matter how far away the object is

    :type: float
    """

class Text(ID, bpy_struct):
    """Text data-block referencing an external or packed text file"""

    current_character: int
    """ Index of current character in current line, and also start index of character in selection if one exists

    :type: int
    """

    current_line: TextLine
    """ Current line, and start line of selection if one exists

    :type: TextLine
    """

    current_line_index: int
    """ Index of current TextLine in TextLine collection

    :type: int
    """

    filepath: str
    """ Filename of the text file

    :type: str
    """

    is_dirty: bool
    """ Text file has been edited since last save

    :type: bool
    """

    is_in_memory: bool
    """ Text file is in memory, without a corresponding file on disk

    :type: bool
    """

    is_modified: bool
    """ Text file on disk is different than the one in memory

    :type: bool
    """

    lines: bpy_prop_collection[TextLine]
    """ Lines of text

    :type: bpy_prop_collection[TextLine]
    """

    select_end_character: int
    """ Index of character after end of selection in the selection end line

    :type: int
    """

    select_end_line: TextLine
    """ End line of selection

    :type: TextLine
    """

    use_module: bool
    """ Register this text as a module on loading, Text name must end with ".py"

    :type: bool
    """

    use_tabs_as_spaces: bool
    """ Automatically converts all new tabs into spaces

    :type: bool
    """

    users_logic: typing.Any
    """ Logic bricks that use this text
(readonly)"""

    def clear(self):
        """clear the text block"""
        ...

    def write(self, text: str | typing.Any):
        """write text at the cursor location and advance to the end of the text block

        :param text: New text for this data-block
        :type text: str | typing.Any
        """
        ...

    def as_string(self):
        """Return the text as a string."""
        ...

    def from_string(self, string):
        """Replace text with this string.

        :param string:
        """
        ...

class Texture(ID, bpy_struct):
    """Texture data-block used by materials, lamps, worlds and brushes"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    contrast: float
    """ Adjust the contrast of the texture

    :type: float
    """

    factor_blue: float
    """ 

    :type: float
    """

    factor_green: float
    """ 

    :type: float
    """

    factor_red: float
    """ 

    :type: float
    """

    intensity: float
    """ Adjust the brightness of the texture

    :type: float
    """

    node_tree: NodeTree
    """ Node tree for node-based textures

    :type: NodeTree
    """

    saturation: float
    """ Adjust the saturation of colors in the texture

    :type: float
    """

    type: str
    """ 

    :type: str
    """

    use_clamp: bool
    """ Set negative texture RGB and intensity values to zero, for some uses like displacement this option can be disabled to get the full range

    :type: bool
    """

    use_color_ramp: bool
    """ Toggle color ramp operations

    :type: bool
    """

    use_nodes: bool
    """ Make this a node-based texture

    :type: bool
    """

    use_preview_alpha: bool
    """ Show Alpha in Preview Render

    :type: bool
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

    def evaluate(
        self, value: collections.abc.Iterable[float] | None
    ) -> bpy_prop_array[float]:
        """Evaluate the texture at the coordinates given

        :param value:
        :type value: collections.abc.Iterable[float] | None
        :return: Result
        :rtype: bpy_prop_array[float]
        """
        ...

class VectorFont(ID, bpy_struct):
    """Vector font for Text objects"""

    filepath: str
    """ 

    :type: str
    """

    packed_file: PackedFile
    """ 

    :type: PackedFile
    """

    def pack(self):
        """Pack the font into the current blend file"""
        ...

    def unpack(self, method: str | None = "USE_LOCAL"):
        """Unpack the font to the samples filename

        :param method: method, How to unpack
        :type method: str | None
        """
        ...

class WindowManager(ID, bpy_struct):
    """Window manager data-block defining open windows and other user interface data"""

    addon_filter: str
    """ Filter add-ons by category

    :type: str
    """

    addon_search: str
    """ Search within the selected filter

    :type: str
    """

    addon_support: set[str]
    """ Display support level

    :type: set[str]
    """

    clipboard: str
    """ 

    :type: str
    """

    keyconfigs: KeyConfigurations
    """ Registered key configurations

    :type: KeyConfigurations
    """

    operators: bpy_prop_collection[Operator]
    """ Operator registry

    :type: bpy_prop_collection[Operator]
    """

    windows: bpy_prop_collection[Window]
    """ Open windows

    :type: bpy_prop_collection[Window]
    """

    @classmethod
    def fileselect_add(cls, operator: Operator | None):
        """Opens a file selector with an operator. The string properties 'filepath', 'filename', 'directory' and a 'files' collection are assigned when present in the operator

        :param operator: Operator to call
        :type operator: Operator | None
        """
        ...

    @classmethod
    def modal_handler_add(cls, operator: Operator | None) -> bool:
        """Add a modal handler to the window manager, for the given modal operator (called by invoke() with self, just before returning {'RUNNING_MODAL'})

        :param operator: Operator to call
        :type operator: Operator | None
        :return: Whether adding the handler was successful
        :rtype: bool
        """
        ...

    def event_timer_add(
        self, time_step: float | None, window: Window | None = None
    ) -> Timer:
        """Add a timer to the given window, to generate periodic 'TIMER' events

        :param time_step: Time Step, Interval in seconds between timer events
        :type time_step: float | None
        :param window: Window to attach the timer to, or None
        :type window: Window | None
        :return:
        :rtype: Timer
        """
        ...

    def event_timer_remove(self, timer: Timer):
        """event_timer_remove

        :param timer:
        :type timer: Timer
        """
        ...

    def progress_begin(self, min: float | None, max: float | None):
        """Start progress report

        :param min: min, any value in range [0,9999]
        :type min: float | None
        :param max: max, any value in range [min+1,9998]
        :type max: float | None
        """
        ...

    def progress_update(self, value: float | None):
        """Update the progress feedback

        :param value: value, Any value between min and max as set in progress_begin()
        :type value: float | None
        """
        ...

    def progress_end(self):
        """Terminate progress report"""
        ...

    @classmethod
    def invoke_props_popup(
        cls, operator: Operator | None, event: Event | None
    ) -> set[str]:
        """Operator popup invoke (show operator properties and execute it automatically on changes)

                :param operator: Operator to call
                :type operator: Operator | None
                :param event: Event
                :type event: Event | None
                :return: result

        RUNNING_MODAL Running Modal, Keep the operator running with blender.

        CANCELLED Cancelled, When no action has been taken, operator exits.

        FINISHED Finished, When the operator is complete, operator exits.

        PASS_THROUGH Pass Through, Do nothing and pass the event on.

        INTERFACE Interface, Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    @classmethod
    def invoke_props_dialog(
        cls,
        operator: Operator | None,
        width: typing.Any | None = 300,
        height: typing.Any | None = 20,
    ) -> set[str]:
        """Operator dialog (non-autoexec popup) invoke (show operator properties and only execute it on click on OK button)

                :param operator: Operator to call
                :type operator: Operator | None
                :param width: Width of the popup
                :type width: typing.Any | None
                :param height: Height of the popup
                :type height: typing.Any | None
                :return: result

        RUNNING_MODAL Running Modal, Keep the operator running with blender.

        CANCELLED Cancelled, When no action has been taken, operator exits.

        FINISHED Finished, When the operator is complete, operator exits.

        PASS_THROUGH Pass Through, Do nothing and pass the event on.

        INTERFACE Interface, Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    @classmethod
    def invoke_search_popup(cls, operator: Operator | None):
        """Operator search popup invoke (search in values of operator's type 'prop' EnumProperty, and execute it on confirmation)

        :param operator: Operator to call
        :type operator: Operator | None
        """
        ...

    @classmethod
    def invoke_popup(
        cls,
        operator: Operator | None,
        width: typing.Any | None = 300,
        height: typing.Any | None = 20,
    ) -> set[str]:
        """Operator popup invoke (only shows operator's properties, without executing it)

                :param operator: Operator to call
                :type operator: Operator | None
                :param width: Width of the popup
                :type width: typing.Any | None
                :param height: Height of the popup
                :type height: typing.Any | None
                :return: result

        RUNNING_MODAL Running Modal, Keep the operator running with blender.

        CANCELLED Cancelled, When no action has been taken, operator exits.

        FINISHED Finished, When the operator is complete, operator exits.

        PASS_THROUGH Pass Through, Do nothing and pass the event on.

        INTERFACE Interface, Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    @classmethod
    def invoke_confirm(cls, operator: Operator | None, event: Event | None) -> set[str]:
        """Operator confirmation popup (only to let user confirm the execution, no operator properties shown)

                :param operator: Operator to call
                :type operator: Operator | None
                :param event: Event
                :type event: Event | None
                :return: result

        RUNNING_MODAL Running Modal, Keep the operator running with blender.

        CANCELLED Cancelled, When no action has been taken, operator exits.

        FINISHED Finished, When the operator is complete, operator exits.

        PASS_THROUGH Pass Through, Do nothing and pass the event on.

        INTERFACE Interface, Handled but not executed (popup menus).
                :rtype: set[str]
        """
        ...

    @classmethod
    def pupmenu_begin__internal(
        cls, title: str | typing.Any, icon: str | None = "NONE"
    ) -> UIPopupMenu:
        """pupmenu_begin__internal

        :param title:
        :type title: str | typing.Any
        :param icon: icon
        :type icon: str | None
        :return:
        :rtype: UIPopupMenu
        """
        ...

    @classmethod
    def pupmenu_end__internal(cls, menu: UIPopupMenu = None):
        """pupmenu_end__internal

        :param menu:
        :type menu: UIPopupMenu
        """
        ...

    @classmethod
    def piemenu_begin__internal(
        cls, title: str | typing.Any, icon: str | None = "NONE", event: Event = None
    ) -> UIPieMenu:
        """piemenu_begin__internal

        :param title:
        :type title: str | typing.Any
        :param icon: icon
        :type icon: str | None
        :param event:
        :type event: Event
        :return:
        :rtype: UIPieMenu
        """
        ...

    @classmethod
    def piemenu_end__internal(cls, menu: UIPieMenu = None):
        """piemenu_end__internal

        :param menu:
        :type menu: UIPieMenu
        """
        ...

    def popup_menu(self, draw_func, title="", icon="NONE"):
        """Popup menus can be useful for creating menus without having to register menu classes.Note that they will not block the scripts execution, so the caller can't wait for user input.

        :param draw_func:
        :param title:
        :param icon:
        """
        ...

    def popup_menu_pie(self, event, draw_func, title="", icon="NONE"):
        """

        :param event:
        :param draw_func:
        :param title:
        :param icon:
        """
        ...

class World(ID, bpy_struct):
    """World data-block describing the environment and ambient lighting of a scene"""

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    ambient_color: bpy_prop_array[float]
    """ Ambient color of the world

    :type: bpy_prop_array[float]
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    color_range: float
    """ The color range that will be mapped to 0-1

    :type: float
    """

    cycles: CyclesWorldSettings
    """ Cycles world settings

    :type: CyclesWorldSettings
    """

    cycles_visibility: CyclesVisibilitySettings
    """ Cycles visibility settings

    :type: CyclesVisibilitySettings
    """

    exposure: float
    """ Amount of exponential color correction for light

    :type: float
    """

    horizon_color: bpy_prop_array[float]
    """ Color at the horizon

    :type: bpy_prop_array[float]
    """

    light_settings: WorldLighting
    """ World lighting settings

    :type: WorldLighting
    """

    mist_settings: WorldMistSettings
    """ World mist settings

    :type: WorldMistSettings
    """

    node_tree: NodeTree
    """ Node tree for node based worlds

    :type: NodeTree
    """

    texture_slots: WorldTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: WorldTextureSlots
    """

    use_nodes: bool
    """ Use shader nodes to render the world

    :type: bool
    """

    use_sky_blend: bool
    """ Render background with natural progression from horizon to zenith

    :type: bool
    """

    use_sky_paper: bool
    """ Flatten blend or texture coordinates

    :type: bool
    """

    use_sky_real: bool
    """ Render background with a real horizon, relative to the camera angle

    :type: bool
    """

    zenith_color: bpy_prop_array[float]
    """ Color at the zenith

    :type: bpy_prop_array[float]
    """

class Itasc(IKParam, bpy_struct):
    """Parameters for the iTaSC IK solver"""

    damping_epsilon: float
    """ Singular value under which damping is progressively applied (higher values=more stability, less reactivity - default=0.1)

    :type: float
    """

    damping_max: float
    """ Maximum damping coefficient when singular value is nearly 0 (higher values=more stability, less reactivity - default=0.5)

    :type: float
    """

    feedback: float
    """ Feedback coefficient for error correction, average response time is 1/feedback (default=20)

    :type: float
    """

    iterations: int
    """ Maximum number of iterations for convergence in case of reiteration

    :type: int
    """

    mode: str
    """ 

    :type: str
    """

    precision: float
    """ Precision of convergence in case of reiteration

    :type: float
    """

    reiteration_method: str
    """ Defines if the solver is allowed to reiterate (converge until precision is met) on none, first or all frames

    :type: str
    """

    solver: str
    """ Solving method selection: automatic damping or manual damping

    :type: str
    """

    step_count: int
    """ Divide the frame interval into this many steps

    :type: int
    """

    step_max: float
    """ Higher bound for timestep in second in case of automatic substeps

    :type: float
    """

    step_min: float
    """ Lower bound for timestep in second in case of automatic substeps

    :type: float
    """

    use_auto_step: bool
    """ Automatically determine the optimal number of steps for best performance/accuracy trade off

    :type: bool
    """

    velocity_max: float
    """ Maximum joint velocity in rad/s (default=50)

    :type: float
    """

class Available(KeyingSetInfo, bpy_struct):
    @staticmethod
    def generate(ksi, context, ks, data):
        """

        :param ksi:
        :param context:
        :param ks:
        :param data:
        """
        ...

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """
        ...

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """
        ...

class LocRotScale(KeyingSetInfo, bpy_struct):
    def generate(self, context, ks, data):
        """

        :param context:
        :param ks:
        :param data:
        """
        ...

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """
        ...

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """
        ...

class Location(KeyingSetInfo, bpy_struct):
    @staticmethod
    def generate(ksi, context, ks, data):
        """

        :param ksi:
        :param context:
        :param ks:
        :param data:
        """
        ...

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """
        ...

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """
        ...

class Rotation(KeyingSetInfo, bpy_struct):
    @staticmethod
    def generate(ksi, context, ks, data):
        """

        :param ksi:
        :param context:
        :param ks:
        :param data:
        """
        ...

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """
        ...

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """
        ...

class Scaling(KeyingSetInfo, bpy_struct):
    @staticmethod
    def generate(ksi, context, ks, data):
        """

        :param ksi:
        :param context:
        :param ks:
        :param data:
        """
        ...

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """
        ...

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """
        ...

class WholeCharacter(KeyingSetInfo, bpy_struct):
    @staticmethod
    def addProp(ksi, ks, bone, prop, index=-1, use_groups=True):
        """

        :param ksi:
        :param ks:
        :param bone:
        :param prop:
        :param index:
        :param use_groups:
        """
        ...

    @staticmethod
    def doBBone(ksi, context, ks, pchan):
        """

        :param ksi:
        :param context:
        :param ks:
        :param pchan:
        """
        ...

    @staticmethod
    def doCustomProps(ksi, ks, bone):
        """

        :param ksi:
        :param ks:
        :param bone:
        """
        ...

    @staticmethod
    def doLoc(ksi, ks, bone):
        """

        :param ksi:
        :param ks:
        :param bone:
        """
        ...

    @staticmethod
    def doRot3d(ksi, ks, bone):
        """

        :param ksi:
        :param ks:
        :param bone:
        """
        ...

    @staticmethod
    def doRot4d(ksi, ks, bone):
        """

        :param ksi:
        :param ks:
        :param bone:
        """
        ...

    @staticmethod
    def doScale(ksi, ks, bone):
        """

        :param ksi:
        :param ks:
        :param bone:
        """
        ...

    @staticmethod
    def generate(ksi, context, ks, bone):
        """

        :param ksi:
        :param context:
        :param ks:
        :param bone:
        """
        ...

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """
        ...

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """
        ...

class LineStyleAlphaModifier(LineStyleModifier, bpy_struct):
    """Base type to define alpha transparency modifiers"""

    ...

class LineStyleColorModifier(LineStyleModifier, bpy_struct):
    """Base type to define line color modifiers"""

    ...

class LineStyleGeometryModifier(LineStyleModifier, bpy_struct):
    """Base type to define stroke geometry modifiers"""

    ...

class LineStyleThicknessModifier(LineStyleModifier, bpy_struct):
    """Base type to define line thickness modifiers"""

    ...

class GPENCIL_PIE_sculpt(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """
        ...

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """
        ...

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """
        ...

class GPENCIL_PIE_settings_palette(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """
        ...

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """
        ...

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """
        ...

class GPENCIL_PIE_tool_palette(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """
        ...

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """
        ...

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """
        ...

class GPENCIL_PIE_tools_more(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """
        ...

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """
        ...

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """
        ...

class ArmatureModifier(Modifier, bpy_struct):
    """Armature deformation modifier"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    object: Object
    """ Armature object to deform with

    :type: Object
    """

    use_bone_envelopes: bool
    """ Bind Bone envelopes to armature modifier

    :type: bool
    """

    use_deform_preserve_volume: bool
    """ Deform rotation interpolation with quaternions

    :type: bool
    """

    use_multi_modifier: bool
    """ Use same input as previous modifier, and mix results using overall vgroup

    :type: bool
    """

    use_vertex_groups: bool
    """ Bind vertex groups to armature modifier

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class ArrayModifier(Modifier, bpy_struct):
    """Array duplication modifier"""

    constant_offset_displace: bpy_prop_array[float]
    """ Value for the distance between arrayed items

    :type: bpy_prop_array[float]
    """

    count: int
    """ Number of duplicates to make

    :type: int
    """

    curve: Object
    """ Curve object to fit array length to

    :type: Object
    """

    end_cap: Object
    """ Mesh object to use as an end cap

    :type: Object
    """

    fit_length: float
    """ Length to fit array within

    :type: float
    """

    fit_type: str
    """ Array length calculation method

    :type: str
    """

    merge_threshold: float
    """ Limit below which to merge vertices

    :type: float
    """

    offset_object: Object
    """ Use the location and rotation of another object to determine the distance and rotational change between arrayed items

    :type: Object
    """

    relative_offset_displace: bpy_prop_array[float]
    """ The size of the geometry will determine the distance between arrayed items

    :type: bpy_prop_array[float]
    """

    start_cap: Object
    """ Mesh object to use as a start cap

    :type: Object
    """

    use_constant_offset: bool
    """ Add a constant offset

    :type: bool
    """

    use_merge_vertices: bool
    """ Merge vertices in adjacent duplicates

    :type: bool
    """

    use_merge_vertices_cap: bool
    """ Merge vertices in first and last duplicates

    :type: bool
    """

    use_object_offset: bool
    """ Add another object's transformation to the total offset

    :type: bool
    """

    use_relative_offset: bool
    """ Add an offset relative to the object's bounding box

    :type: bool
    """

class BevelModifier(Modifier, bpy_struct):
    """Bevel modifier to make edges and vertices more rounded"""

    angle_limit: float
    """ Angle above which to bevel edges

    :type: float
    """

    edge_weight_method: str
    """ What edge weight to use for weighting a vertex

    :type: str
    """

    limit_method: str
    """ 

    :type: str
    """

    loop_slide: bool
    """ Prefer sliding along edges to having even widths

    :type: bool
    """

    material: int
    """ Material index of generated faces, -1 for automatic

    :type: int
    """

    offset_type: str
    """ What distance Width measures

    :type: str
    """

    profile: float
    """ The profile shape (0.5 = round)

    :type: float
    """

    segments: int
    """ Number of segments for round edges/verts

    :type: int
    """

    use_clamp_overlap: bool
    """ Clamp the width to avoid overlap

    :type: bool
    """

    use_only_vertices: bool
    """ Bevel verts/corners, not edges

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

    width: float
    """ Bevel value/amount

    :type: float
    """

class BooleanModifier(Modifier, bpy_struct):
    """Boolean operations modifier"""

    double_threshold: float
    """ Threshold for checking overlapping geometry

    :type: float
    """

    object: Object
    """ Mesh object to use for Boolean operation

    :type: Object
    """

    operation: str
    """ 

    :type: str
    """

    solver: str
    """ 

    :type: str
    """

class BuildModifier(Modifier, bpy_struct):
    """Build effect modifier"""

    frame_duration: float
    """ Total time the build effect requires

    :type: float
    """

    frame_start: float
    """ Start frame of the effect

    :type: float
    """

    seed: int
    """ Seed for random if used

    :type: int
    """

    use_random_order: bool
    """ Randomize the faces or edges during build

    :type: bool
    """

    use_reverse: bool
    """ Deconstruct the mesh instead of building it

    :type: bool
    """

class CastModifier(Modifier, bpy_struct):
    """Modifier to cast to other shapes"""

    cast_type: str
    """ Target object shape

    :type: str
    """

    factor: float
    """ 

    :type: float
    """

    object: Object
    """ Control object: if available, its location determines the center of the effect

    :type: Object
    """

    radius: float
    """ Only deform vertices within this distance from the center of the effect (leave as 0 for infinite.)

    :type: float
    """

    size: float
    """ Size of projection shape (leave as 0 for auto)

    :type: float
    """

    use_radius_as_size: bool
    """ Use radius as size of projection shape (0 = auto)

    :type: bool
    """

    use_transform: bool
    """ Use object transform to control projection shape

    :type: bool
    """

    use_x: bool
    """ 

    :type: bool
    """

    use_y: bool
    """ 

    :type: bool
    """

    use_z: bool
    """ 

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class ClothModifier(Modifier, bpy_struct):
    """Cloth simulation modifier"""

    collision_settings: ClothCollisionSettings
    """ 

    :type: ClothCollisionSettings
    """

    hair_grid_max: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hair_grid_min: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hair_grid_resolution: bpy_prop_array[int]
    """ 

    :type: bpy_prop_array[int]
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    settings: ClothSettings
    """ 

    :type: ClothSettings
    """

    solver_result: ClothSolverResult
    """ 

    :type: ClothSolverResult
    """

class CollisionModifier(Modifier, bpy_struct):
    """Collision modifier defining modifier stack position used for collision"""

    settings: CollisionSettings
    """ 

    :type: CollisionSettings
    """

class CorrectiveSmoothModifier(Modifier, bpy_struct):
    """Correct distortion caused by deformation"""

    factor: float
    """ Smooth factor effect

    :type: float
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    is_bind: bool
    """ 

    :type: bool
    """

    iterations: int
    """ 

    :type: int
    """

    rest_source: str
    """ Select the source of rest positions

    :type: str
    """

    smooth_type: str
    """ Method used for smoothing

    :type: str
    """

    use_only_smooth: bool
    """ Apply smoothing without reconstructing the surface

    :type: bool
    """

    use_pin_boundary: bool
    """ Excludes boundary vertices from being smoothed

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class CurveModifier(Modifier, bpy_struct):
    """Curve deformation modifier"""

    deform_axis: str
    """ The axis that the curve deforms along

    :type: str
    """

    object: Object
    """ Curve object to deform with

    :type: Object
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class DataTransferModifier(Modifier, bpy_struct):
    """Modifier transferring some data from a source mesh"""

    data_types_edges: set[str]
    """ Which edge data layers to transfer

    :type: set[str]
    """

    data_types_loops: set[str]
    """ Which face corner data layers to transfer

    :type: set[str]
    """

    data_types_polys: set[str]
    """ Which poly data layers to transfer

    :type: set[str]
    """

    data_types_verts: set[str]
    """ Which vertex data layers to transfer

    :type: set[str]
    """

    edge_mapping: str
    """ Method used to map source edges to destination ones

    :type: str
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    islands_precision: float
    """ Factor controlling precision of islands handling (typically, 0.1 should be enough, higher values can make things really slow)

    :type: float
    """

    layers_uv_select_dst: str
    """ How to match source and destination layers

    :type: str
    """

    layers_uv_select_src: str
    """ Which layers to transfer, in case of multi-layers types

    :type: str
    """

    layers_vcol_select_dst: str
    """ How to match source and destination layers

    :type: str
    """

    layers_vcol_select_src: str
    """ Which layers to transfer, in case of multi-layers types

    :type: str
    """

    layers_vgroup_select_dst: str
    """ How to match source and destination layers

    :type: str
    """

    layers_vgroup_select_src: str
    """ Which layers to transfer, in case of multi-layers types

    :type: str
    """

    loop_mapping: str
    """ Method used to map source faces' corners to destination ones

    :type: str
    """

    max_distance: float
    """ Maximum allowed distance between source and destination element, for non-topology mappings

    :type: float
    """

    mix_factor: float
    """ Factor to use when applying data to destination (exact behavior depends on mix mode)

    :type: float
    """

    mix_mode: str
    """ How to affect destination elements with source values

    :type: str
    """

    object: Object
    """ Object to transfer data from

    :type: Object
    """

    poly_mapping: str
    """ Method used to map source faces to destination ones

    :type: str
    """

    ray_radius: float
    """ 'Width' of rays (especially useful when raycasting against vertices or edges)

    :type: float
    """

    use_edge_data: bool
    """ Enable edge data transfer

    :type: bool
    """

    use_loop_data: bool
    """ Enable face corner data transfer

    :type: bool
    """

    use_max_distance: bool
    """ Source elements must be closer than given distance from destination one

    :type: bool
    """

    use_object_transform: bool
    """ Evaluate source and destination meshes in global space

    :type: bool
    """

    use_poly_data: bool
    """ Enable face data transfer

    :type: bool
    """

    use_vert_data: bool
    """ Enable vertex data transfer

    :type: bool
    """

    vert_mapping: str
    """ Method used to map source vertices to destination ones

    :type: str
    """

    vertex_group: str
    """ Vertex group name for selecting the affected areas

    :type: str
    """

class DecimateModifier(Modifier, bpy_struct):
    """Decimation modifier"""

    angle_limit: float
    """ Only dissolve angles below this (planar only)

    :type: float
    """

    decimate_type: str
    """ 

    :type: str
    """

    delimit: set[str]
    """ Limit merging geometry

    :type: set[str]
    """

    face_count: int
    """ The current number of faces in the decimated mesh

    :type: int
    """

    invert_vertex_group: bool
    """ Invert vertex group influence (collapse only)

    :type: bool
    """

    iterations: int
    """ Number of times reduce the geometry (unsubdivide only)

    :type: int
    """

    ratio: float
    """ Ratio of triangles to reduce to (collapse only)

    :type: float
    """

    symmetry_axis: str
    """ Axis of symmetry

    :type: str
    """

    use_collapse_triangulate: bool
    """ Keep triangulated faces resulting from decimation (collapse only)

    :type: bool
    """

    use_dissolve_boundaries: bool
    """ Dissolve all vertices inbetween face boundaries (planar only)

    :type: bool
    """

    use_symmetry: bool
    """ Maintain symmetry on an axis

    :type: bool
    """

    vertex_group: str
    """ Vertex group name (collapse only)

    :type: str
    """

    vertex_group_factor: float
    """ Vertex group strength

    :type: float
    """

class DisplaceModifier(Modifier, bpy_struct):
    """Displacement modifier"""

    direction: str
    """ 

    :type: str
    """

    mid_level: float
    """ Material value that gives no displacement

    :type: float
    """

    strength: float
    """ Amount to displace geometry

    :type: float
    """

    texture: Texture
    """ 

    :type: Texture
    """

    texture_coords: str
    """ 

    :type: str
    """

    texture_coords_object: Object
    """ Object to set the texture coordinates

    :type: Object
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class DynamicPaintModifier(Modifier, bpy_struct):
    """Dynamic Paint modifier"""

    brush_settings: DynamicPaintBrushSettings
    """ 

    :type: DynamicPaintBrushSettings
    """

    canvas_settings: DynamicPaintCanvasSettings
    """ 

    :type: DynamicPaintCanvasSettings
    """

    ui_type: str
    """ 

    :type: str
    """

class EdgeSplitModifier(Modifier, bpy_struct):
    """Edge splitting modifier to create sharp edges"""

    split_angle: float
    """ Angle above which to split edges

    :type: float
    """

    use_edge_angle: bool
    """ Split edges with high angle between faces

    :type: bool
    """

    use_edge_sharp: bool
    """ Split edges that are marked as sharp

    :type: bool
    """

class ExplodeModifier(Modifier, bpy_struct):
    """Explosion effect modifier based on a particle system"""

    particle_uv: str
    """ UV map to change with particle age

    :type: str
    """

    protect: float
    """ Clean vertex group edges

    :type: float
    """

    show_alive: bool
    """ Show mesh when particles are alive

    :type: bool
    """

    show_dead: bool
    """ Show mesh when particles are dead

    :type: bool
    """

    show_unborn: bool
    """ Show mesh when particles are unborn

    :type: bool
    """

    use_edge_cut: bool
    """ Cut face edges for nicer shrapnel

    :type: bool
    """

    use_size: bool
    """ Use particle size for the shrapnel

    :type: bool
    """

    vertex_group: str
    """ 

    :type: str
    """

class FluidSimulationModifier(Modifier, bpy_struct):
    """Fluid simulation modifier"""

    settings: FluidSettings
    """ Settings for how this object is used in the fluid simulation

    :type: FluidSettings
    """

class HookModifier(Modifier, bpy_struct):
    """Hook modifier to modify the location of vertices"""

    center: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    falloff_curve: CurveMapping
    """ Custom Lamp Falloff Curve

    :type: CurveMapping
    """

    falloff_radius: float
    """ If not zero, the distance from the hook where influence ends

    :type: float
    """

    falloff_type: str
    """ 

    :type: str
    """

    matrix_inverse: bpy_prop_array[float]
    """ Reverse the transformation between this object and its target

    :type: bpy_prop_array[float]
    """

    object: Object
    """ Parent Object for hook, also recalculates and clears offset

    :type: Object
    """

    strength: float
    """ Relative force of the hook

    :type: float
    """

    subtarget: str
    """ Name of Parent Bone for hook (if applicable), also recalculates and clears offset

    :type: str
    """

    use_falloff_uniform: bool
    """ Compensate for non-uniform object scale

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class LaplacianDeformModifier(Modifier, bpy_struct):
    """Mesh deform modifier"""

    is_bind: bool
    """ Whether geometry has been bound to anchors

    :type: bool
    """

    iterations: int
    """ 

    :type: int
    """

    vertex_group: str
    """ Name of Vertex Group which determines Anchors

    :type: str
    """

class LaplacianSmoothModifier(Modifier, bpy_struct):
    """Smoothing effect modifier"""

    iterations: int
    """ 

    :type: int
    """

    lambda_border: float
    """ Lambda factor in border

    :type: float
    """

    lambda_factor: float
    """ Smooth factor effect

    :type: float
    """

    use_normalized: bool
    """ Improve and stabilize the enhanced shape

    :type: bool
    """

    use_volume_preserve: bool
    """ Apply volume preservation after smooth

    :type: bool
    """

    use_x: bool
    """ Smooth object along X axis

    :type: bool
    """

    use_y: bool
    """ Smooth object along Y axis

    :type: bool
    """

    use_z: bool
    """ Smooth object along Z axis

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class LatticeModifier(Modifier, bpy_struct):
    """Lattice deformation modifier"""

    object: Object
    """ Lattice object to deform with

    :type: Object
    """

    strength: float
    """ Strength of modifier effect

    :type: float
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class MaskModifier(Modifier, bpy_struct):
    """Mask modifier to hide parts of the mesh"""

    armature: Object
    """ Armature to use as source of bones to mask

    :type: Object
    """

    invert_vertex_group: bool
    """ Use vertices that are not part of region defined

    :type: bool
    """

    mode: str
    """ 

    :type: str
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class MeshCacheModifier(Modifier, bpy_struct):
    """Cache Mesh"""

    cache_format: str
    """ 

    :type: str
    """

    deform_mode: str
    """ 

    :type: str
    """

    eval_factor: float
    """ Evaluation time in seconds

    :type: float
    """

    eval_frame: float
    """ The frame to evaluate (starting at 0)

    :type: float
    """

    eval_time: float
    """ Evaluation time in seconds

    :type: float
    """

    factor: float
    """ Influence of the deformation

    :type: float
    """

    filepath: str
    """ Path to external displacements file

    :type: str
    """

    flip_axis: set[str]
    """ 

    :type: set[str]
    """

    forward_axis: str
    """ 

    :type: str
    """

    frame_scale: float
    """ Evaluation time in seconds

    :type: float
    """

    frame_start: float
    """ Add this to the start frame

    :type: float
    """

    interpolation: str
    """ 

    :type: str
    """

    play_mode: str
    """ 

    :type: str
    """

    time_mode: str
    """ Method to control playback time

    :type: str
    """

    up_axis: str
    """ 

    :type: str
    """

class MeshDeformModifier(Modifier, bpy_struct):
    """Mesh deformation modifier to deform with other meshes"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    is_bound: bool
    """ Whether geometry has been bound to control cage

    :type: bool
    """

    object: Object
    """ Mesh object to deform with

    :type: Object
    """

    precision: int
    """ The grid size for binding

    :type: int
    """

    use_dynamic_bind: bool
    """ Recompute binding dynamically on top of other deformers (slower and more memory consuming)

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class MeshSequenceCacheModifier(Modifier, bpy_struct):
    """Cache Mesh"""

    cache_file: CacheFile
    """ 

    :type: CacheFile
    """

    object_path: str
    """ Path to the object in the Alembic archive used to lookup geometric data

    :type: str
    """

    read_data: set[str]
    """ 

    :type: set[str]
    """

class MirrorModifier(Modifier, bpy_struct):
    """Mirroring modifier"""

    merge_threshold: float
    """ Distance within which mirrored vertices are merged

    :type: float
    """

    mirror_object: Object
    """ Object to use as mirror

    :type: Object
    """

    use_clip: bool
    """ Prevent vertices from going through the mirror during transform

    :type: bool
    """

    use_mirror_merge: bool
    """ Merge vertices within the merge threshold

    :type: bool
    """

    use_mirror_u: bool
    """ Mirror the U texture coordinate around the 0.5 point

    :type: bool
    """

    use_mirror_v: bool
    """ Mirror the V texture coordinate around the 0.5 point

    :type: bool
    """

    use_mirror_vertex_groups: bool
    """ Mirror vertex groups (e.g. .R->.L)

    :type: bool
    """

    use_x: bool
    """ Enable X axis mirror

    :type: bool
    """

    use_y: bool
    """ Enable Y axis mirror

    :type: bool
    """

    use_z: bool
    """ Enable Z axis mirror

    :type: bool
    """

class MultiresModifier(Modifier, bpy_struct):
    """Multiresolution mesh modifier"""

    filepath: str
    """ Path to external displacements file

    :type: str
    """

    is_external: bool
    """ Store multires displacements outside the .blend file, to save memory

    :type: bool
    """

    levels: int
    """ Number of subdivisions to use in the viewport

    :type: int
    """

    render_levels: int
    """ The subdivision level visible at render time

    :type: int
    """

    sculpt_levels: int
    """ Number of subdivisions to use in sculpt mode

    :type: int
    """

    show_only_control_edges: bool
    """ Skip drawing/rendering of interior subdivided edges

    :type: bool
    """

    subdivision_type: str
    """ Select type of subdivision algorithm

    :type: str
    """

    total_levels: int
    """ Number of subdivisions for which displacements are stored

    :type: int
    """

    use_subsurf_uv: bool
    """ Use subsurf to subdivide UVs

    :type: bool
    """

class NormalEditModifier(Modifier, bpy_struct):
    """Modifier affecting/generating custom normals"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    mix_factor: float
    """ How much of generated normals to mix with exiting ones

    :type: float
    """

    mix_limit: float
    """ Maximum angle between old and new normals

    :type: float
    """

    mix_mode: str
    """ How to mix generated normals with existing ones

    :type: str
    """

    mode: str
    """ How to affect (generate) normals

    :type: str
    """

    offset: bpy_prop_array[float]
    """ Offset from object's center

    :type: bpy_prop_array[float]
    """

    target: Object
    """ Target object used to affect normals

    :type: Object
    """

    use_direction_parallel: bool
    """ Use same direction for all normals, from origin to target's center (Directional mode only)

    :type: bool
    """

    vertex_group: str
    """ Vertex group name for selecting/weighting the affected areas

    :type: str
    """

class OceanModifier(Modifier, bpy_struct):
    """Simulate an ocean surface"""

    bake_foam_fade: float
    """ How much foam accumulates over time (baked ocean only)

    :type: float
    """

    choppiness: float
    """ Choppiness of the wave's crest (adds some horizontal component to the displacement)

    :type: float
    """

    damping: float
    """ Damp reflected waves going in opposite direction to the wind

    :type: float
    """

    depth: float
    """ Depth of the solid ground below the water surface

    :type: float
    """

    filepath: str
    """ Path to a folder to store external baked images

    :type: str
    """

    foam_coverage: float
    """ Amount of generated foam

    :type: float
    """

    foam_layer_name: str
    """ Name of the vertex color layer used for foam

    :type: str
    """

    frame_end: int
    """ End frame of the ocean baking

    :type: int
    """

    frame_start: int
    """ Start frame of the ocean baking

    :type: int
    """

    geometry_mode: str
    """ Method of modifying geometry

    :type: str
    """

    is_cached: bool
    """ Whether the ocean is using cached data or simulating

    :type: bool
    """

    random_seed: int
    """ Seed of the random generator

    :type: int
    """

    repeat_x: int
    """ Repetitions of the generated surface in X

    :type: int
    """

    repeat_y: int
    """ Repetitions of the generated surface in Y

    :type: int
    """

    resolution: int
    """ Resolution of the generated surface

    :type: int
    """

    size: float
    """ Surface scale factor (does not affect the height of the waves)

    :type: float
    """

    spatial_size: int
    """ Size of the simulation domain (in meters), and of the generated geometry (in BU)

    :type: int
    """

    time: float
    """ Current time of the simulation

    :type: float
    """

    use_foam: bool
    """ Generate foam mask as a vertex color channel

    :type: bool
    """

    use_normals: bool
    """ Output normals for bump mapping - disabling can speed up performance if its not needed

    :type: bool
    """

    wave_alignment: float
    """ How much the waves are aligned to each other

    :type: float
    """

    wave_direction: float
    """ Main direction of the waves when they are (partially) aligned

    :type: float
    """

    wave_scale: float
    """ Scale of the displacement effect

    :type: float
    """

    wave_scale_min: float
    """ Shortest allowed wavelength

    :type: float
    """

    wind_velocity: float
    """ Wind speed

    :type: float
    """

class ParticleInstanceModifier(Modifier, bpy_struct):
    """Particle system instancing modifier"""

    axis: str
    """ Pole axis for rotation

    :type: str
    """

    object: Object
    """ Object that has the particle system

    :type: Object
    """

    particle_system_index: int
    """ 

    :type: int
    """

    position: float
    """ Position along path

    :type: float
    """

    random_position: float
    """ Randomize position along path

    :type: float
    """

    show_alive: bool
    """ Show instances when particles are alive

    :type: bool
    """

    show_dead: bool
    """ Show instances when particles are dead

    :type: bool
    """

    show_unborn: bool
    """ Show instances when particles are unborn

    :type: bool
    """

    use_children: bool
    """ Create instances from child particles

    :type: bool
    """

    use_normal: bool
    """ Create instances from normal particles

    :type: bool
    """

    use_path: bool
    """ Create instances along particle paths

    :type: bool
    """

    use_preserve_shape: bool
    """ Don't stretch the object

    :type: bool
    """

    use_size: bool
    """ Use particle size to scale the instances

    :type: bool
    """

class ParticleSystemModifier(Modifier, bpy_struct):
    """Particle system simulation modifier"""

    particle_system: ParticleSystem
    """ Particle System that this modifier controls

    :type: ParticleSystem
    """

class RemeshModifier(Modifier, bpy_struct):
    """Generate a new surface with regular topology that follows the shape of the input mesh"""

    mode: str
    """ 

    :type: str
    """

    octree_depth: int
    """ Resolution of the octree; higher values give finer details

    :type: int
    """

    scale: float
    """ The ratio of the largest dimension of the model over the size of the grid

    :type: float
    """

    sharpness: float
    """ Tolerance for outliers; lower values filter noise while higher values will reproduce edges closer to the input

    :type: float
    """

    threshold: float
    """ If removing disconnected pieces, minimum size of components to preserve as a ratio of the number of polygons in the largest component

    :type: float
    """

    use_remove_disconnected: bool
    """ 

    :type: bool
    """

    use_smooth_shade: bool
    """ Output faces with smooth shading rather than flat shaded

    :type: bool
    """

class ScrewModifier(Modifier, bpy_struct):
    """Revolve edges"""

    angle: float
    """ Angle of revolution

    :type: float
    """

    axis: str
    """ Screw axis

    :type: str
    """

    iterations: int
    """ Number of times to apply the screw operation

    :type: int
    """

    object: Object
    """ Object to define the screw axis

    :type: Object
    """

    render_steps: int
    """ Number of steps in the revolution

    :type: int
    """

    screw_offset: float
    """ Offset the revolution along its axis

    :type: float
    """

    steps: int
    """ Number of steps in the revolution

    :type: int
    """

    use_normal_calculate: bool
    """ Calculate the order of edges (needed for meshes, but not curves)

    :type: bool
    """

    use_normal_flip: bool
    """ Flip normals of lathed faces

    :type: bool
    """

    use_object_screw_offset: bool
    """ Use the distance between the objects to make a screw

    :type: bool
    """

    use_smooth_shade: bool
    """ Output faces with smooth shading rather than flat shaded

    :type: bool
    """

    use_stretch_u: bool
    """ Stretch the U coordinates between 0-1 when UV's are present

    :type: bool
    """

    use_stretch_v: bool
    """ Stretch the V coordinates between 0-1 when UV's are present

    :type: bool
    """

class ShrinkwrapModifier(Modifier, bpy_struct):
    """Shrink wrapping modifier to shrink wrap and object to a target"""

    auxiliary_target: Object
    """ Additional mesh target to shrink to

    :type: Object
    """

    cull_face: str
    """ Stop vertices from projecting to a face on the target when facing towards/away

    :type: str
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    offset: float
    """ Distance to keep from the target

    :type: float
    """

    project_limit: float
    """ Limit the distance used for projection (zero disables)

    :type: float
    """

    subsurf_levels: int
    """ Number of subdivisions that must be performed before extracting vertices' positions and normals

    :type: int
    """

    target: Object
    """ Mesh target to shrink to

    :type: Object
    """

    use_keep_above_surface: bool
    """ 

    :type: bool
    """

    use_negative_direction: bool
    """ Allow vertices to move in the negative direction of axis

    :type: bool
    """

    use_positive_direction: bool
    """ Allow vertices to move in the positive direction of axis

    :type: bool
    """

    use_project_x: bool
    """ 

    :type: bool
    """

    use_project_y: bool
    """ 

    :type: bool
    """

    use_project_z: bool
    """ 

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

    wrap_method: str
    """ 

    :type: str
    """

class SimpleDeformModifier(Modifier, bpy_struct):
    """Simple deformation modifier to apply effects such as twisting and bending"""

    angle: float
    """ Angle of deformation

    :type: float
    """

    deform_method: str
    """ 

    :type: str
    """

    factor: float
    """ Amount to deform object

    :type: float
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    limits: bpy_prop_array[float]
    """ Lower/Upper limits for deform

    :type: bpy_prop_array[float]
    """

    lock_x: bool
    """ Do not allow deformation along the X axis

    :type: bool
    """

    lock_y: bool
    """ Do not allow deformation along the Y axis

    :type: bool
    """

    origin: Object
    """ Offset the origin and orientation of the deformation

    :type: Object
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class SkinModifier(Modifier, bpy_struct):
    """Generate Skin"""

    branch_smoothing: float
    """ Smooth complex geometry around branches

    :type: float
    """

    use_smooth_shade: bool
    """ Output faces with smooth shading rather than flat shaded

    :type: bool
    """

    use_x_symmetry: bool
    """ Avoid making unsymmetrical quads across the X axis

    :type: bool
    """

    use_y_symmetry: bool
    """ Avoid making unsymmetrical quads across the Y axis

    :type: bool
    """

    use_z_symmetry: bool
    """ Avoid making unsymmetrical quads across the Z axis

    :type: bool
    """

class SmokeModifier(Modifier, bpy_struct):
    """Smoke simulation modifier"""

    coll_settings: SmokeCollSettings
    """ 

    :type: SmokeCollSettings
    """

    domain_settings: SmokeDomainSettings
    """ 

    :type: SmokeDomainSettings
    """

    flow_settings: SmokeFlowSettings
    """ 

    :type: SmokeFlowSettings
    """

    smoke_type: str
    """ 

    :type: str
    """

class SmoothModifier(Modifier, bpy_struct):
    """Smoothing effect modifier"""

    factor: float
    """ Strength of modifier effect

    :type: float
    """

    iterations: int
    """ 

    :type: int
    """

    use_x: bool
    """ Smooth object along X axis

    :type: bool
    """

    use_y: bool
    """ Smooth object along Y axis

    :type: bool
    """

    use_z: bool
    """ Smooth object along Z axis

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class SoftBodyModifier(Modifier, bpy_struct):
    """Soft body simulation modifier"""

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    settings: SoftBodySettings
    """ 

    :type: SoftBodySettings
    """

class SolidifyModifier(Modifier, bpy_struct):
    """Create a solid skin by extruding, compensating for sharp angles"""

    edge_crease_inner: float
    """ Assign a crease to inner edges

    :type: float
    """

    edge_crease_outer: float
    """ Assign a crease to outer edges

    :type: float
    """

    edge_crease_rim: float
    """ Assign a crease to the edges making up the rim

    :type: float
    """

    invert_vertex_group: bool
    """ Invert the vertex group influence

    :type: bool
    """

    material_offset: int
    """ Offset material index of generated faces

    :type: int
    """

    material_offset_rim: int
    """ Offset material index of generated rim faces

    :type: int
    """

    offset: float
    """ Offset the thickness from the center

    :type: float
    """

    thickness: float
    """ Thickness of the shell

    :type: float
    """

    thickness_clamp: float
    """ Offset clamp based on geometry scale

    :type: float
    """

    thickness_vertex_group: float
    """ Thickness factor to use for zero vertex group influence

    :type: float
    """

    use_even_offset: bool
    """ Maintain thickness by adjusting for sharp corners (slow, disable when not needed)

    :type: bool
    """

    use_flip_normals: bool
    """ Invert the face direction

    :type: bool
    """

    use_quality_normals: bool
    """ Calculate normals which result in more even thickness (slow, disable when not needed)

    :type: bool
    """

    use_rim: bool
    """ Create edge loops between the inner and outer surfaces on face edges (slow, disable when not needed)

    :type: bool
    """

    use_rim_only: bool
    """ Only add the rim to the original data

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class SubsurfModifier(Modifier, bpy_struct):
    """Subdivision surface modifier"""

    levels: int
    """ Number of subdivisions to perform

    :type: int
    """

    render_levels: int
    """ Number of subdivisions to perform when rendering

    :type: int
    """

    show_only_control_edges: bool
    """ Skip drawing/rendering of interior subdivided edges

    :type: bool
    """

    subdivision_type: str
    """ Select type of subdivision algorithm

    :type: str
    """

    use_opensubdiv: bool
    """ Use OpenSubdiv for the subdivisions (viewport only)

    :type: bool
    """

    use_subsurf_uv: bool
    """ Use subsurf to subdivide UVs

    :type: bool
    """

class SurfaceModifier(Modifier, bpy_struct):
    """Surface modifier defining modifier stack position used for surface fields"""

    ...

class TriangulateModifier(Modifier, bpy_struct):
    """Triangulate Mesh"""

    ngon_method: str
    """ Method for splitting the polygons into triangles

    :type: str
    """

    quad_method: str
    """ Method for splitting the quads into triangles

    :type: str
    """

class UVProjectModifier(Modifier, bpy_struct):
    """UV projection modifier to set UVs from a projector"""

    aspect_x: float
    """ 

    :type: float
    """

    aspect_y: float
    """ 

    :type: float
    """

    image: Image
    """ 

    :type: Image
    """

    projector_count: int
    """ Number of projectors to use

    :type: int
    """

    projectors: bpy_prop_collection[UVProjector]
    """ 

    :type: bpy_prop_collection[UVProjector]
    """

    scale_x: float
    """ 

    :type: float
    """

    scale_y: float
    """ 

    :type: float
    """

    use_image_override: bool
    """ Override faces' current images with the given image

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

class UVWarpModifier(Modifier, bpy_struct):
    """Add target position to uv coordinates"""

    axis_u: str
    """ Pole axis for rotation

    :type: str
    """

    axis_v: str
    """ Pole axis for rotation

    :type: str
    """

    bone_from: str
    """ Bone defining offset

    :type: str
    """

    bone_to: str
    """ Bone defining offset

    :type: str
    """

    center: bpy_prop_array[float]
    """ Center point for rotate/scale

    :type: bpy_prop_array[float]
    """

    object_from: Object
    """ Object defining offset

    :type: Object
    """

    object_to: Object
    """ Object defining offset

    :type: Object
    """

    uv_layer: str
    """ UV Layer name

    :type: str
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class VertexWeightEditModifier(Modifier, bpy_struct):
    """Edit the weights of vertices in a group"""

    add_threshold: float
    """ Lower bound for a vertex's weight to be added to the vgroup

    :type: float
    """

    default_weight: float
    """ Default weight a vertex will have if it is not in the vgroup

    :type: float
    """

    falloff_type: str
    """ How weights are mapped to their new values

    :type: str
    """

    map_curve: CurveMapping
    """ Custom mapping curve

    :type: CurveMapping
    """

    mask_constant: float
    """ Global influence of current modifications on vgroup

    :type: float
    """

    mask_tex_map_object: Object
    """ Which object to take texture coordinates from

    :type: Object
    """

    mask_tex_mapping: str
    """ Which texture coordinates to use for mapping

    :type: str
    """

    mask_tex_use_channel: str
    """ Which texture channel to use for masking

    :type: str
    """

    mask_tex_uv_layer: str
    """ UV map name

    :type: str
    """

    mask_texture: Texture
    """ Masking texture

    :type: Texture
    """

    mask_vertex_group: str
    """ Masking vertex group name

    :type: str
    """

    remove_threshold: float
    """ Upper bound for a vertex's weight to be removed from the vgroup

    :type: float
    """

    use_add: bool
    """ Add vertices with weight over threshold to vgroup

    :type: bool
    """

    use_remove: bool
    """ Remove vertices with weight below threshold from vgroup

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class VertexWeightMixModifier(Modifier, bpy_struct):
    """Mix the weights of two vertex groups"""

    default_weight_a: float
    """ Default weight a vertex will have if it is not in the first A vgroup

    :type: float
    """

    default_weight_b: float
    """ Default weight a vertex will have if it is not in the second B vgroup

    :type: float
    """

    mask_constant: float
    """ Global influence of current modifications on vgroup

    :type: float
    """

    mask_tex_map_object: Object
    """ Which object to take texture coordinates from

    :type: Object
    """

    mask_tex_mapping: str
    """ Which texture coordinates to use for mapping

    :type: str
    """

    mask_tex_use_channel: str
    """ Which texture channel to use for masking

    :type: str
    """

    mask_tex_uv_layer: str
    """ UV map name

    :type: str
    """

    mask_texture: Texture
    """ Masking texture

    :type: Texture
    """

    mask_vertex_group: str
    """ Masking vertex group name

    :type: str
    """

    mix_mode: str
    """ How weights from vgroup B affect weights of vgroup A

    :type: str
    """

    mix_set: str
    """ Which vertices should be affected

    :type: str
    """

    vertex_group_a: str
    """ First vertex group name

    :type: str
    """

    vertex_group_b: str
    """ Second vertex group name

    :type: str
    """

class VertexWeightProximityModifier(Modifier, bpy_struct):
    """Set the weights of vertices in a group from a target object's distance"""

    falloff_type: str
    """ How weights are mapped to their new values

    :type: str
    """

    mask_constant: float
    """ Global influence of current modifications on vgroup

    :type: float
    """

    mask_tex_map_object: Object
    """ Which object to take texture coordinates from

    :type: Object
    """

    mask_tex_mapping: str
    """ Which texture coordinates to use for mapping

    :type: str
    """

    mask_tex_use_channel: str
    """ Which texture channel to use for masking

    :type: str
    """

    mask_tex_uv_layer: str
    """ UV map name

    :type: str
    """

    mask_texture: Texture
    """ Masking texture

    :type: Texture
    """

    mask_vertex_group: str
    """ Masking vertex group name

    :type: str
    """

    max_dist: float
    """ Distance mapping to weight 1.0

    :type: float
    """

    min_dist: float
    """ Distance mapping to weight 0.0

    :type: float
    """

    proximity_geometry: set[str]
    """ Use the shortest computed distance to target object's geometry as weight

    :type: set[str]
    """

    proximity_mode: str
    """ Which distances to target object to use

    :type: str
    """

    target: Object
    """ Object to calculate vertices distances from

    :type: Object
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class WarpModifier(Modifier, bpy_struct):
    """Warp modifier"""

    falloff_curve: CurveMapping
    """ Custom Lamp Falloff Curve

    :type: CurveMapping
    """

    falloff_radius: float
    """ Radius to apply

    :type: float
    """

    falloff_type: str
    """ 

    :type: str
    """

    object_from: Object
    """ Object to transform from

    :type: Object
    """

    object_to: Object
    """ Object to transform to

    :type: Object
    """

    strength: float
    """ 

    :type: float
    """

    texture: Texture
    """ 

    :type: Texture
    """

    texture_coords: str
    """ 

    :type: str
    """

    texture_coords_object: Object
    """ Object to set the texture coordinates

    :type: Object
    """

    use_volume_preserve: bool
    """ Preserve volume when rotations are used

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    vertex_group: str
    """ Vertex group name for modulating the deform

    :type: str
    """

class WaveModifier(Modifier, bpy_struct):
    """Wave effect modifier"""

    damping_time: float
    """ Number of frames in which the wave damps out after it dies

    :type: float
    """

    falloff_radius: float
    """ Distance after which it fades out

    :type: float
    """

    height: float
    """ Height of the wave

    :type: float
    """

    lifetime: float
    """ Lifetime of the wave in frames, zero means infinite

    :type: float
    """

    narrowness: float
    """ Distance between the top and the base of a wave, the higher the value, the more narrow the wave

    :type: float
    """

    speed: float
    """ Speed of the wave, towards the starting point when negative

    :type: float
    """

    start_position_object: Object
    """ Object which defines the wave center

    :type: Object
    """

    start_position_x: float
    """ X coordinate of the start position

    :type: float
    """

    start_position_y: float
    """ Y coordinate of the start position

    :type: float
    """

    texture: Texture
    """ 

    :type: Texture
    """

    texture_coords: str
    """ 

    :type: str
    """

    texture_coords_object: Object
    """ Object to set the texture coordinates

    :type: Object
    """

    time_offset: float
    """ Either the starting frame (for positive speed) or ending frame (for negative speed.)

    :type: float
    """

    use_cyclic: bool
    """ Cyclic wave effect

    :type: bool
    """

    use_normal: bool
    """ Displace along normals

    :type: bool
    """

    use_normal_x: bool
    """ Enable displacement along the X normal

    :type: bool
    """

    use_normal_y: bool
    """ Enable displacement along the Y normal

    :type: bool
    """

    use_normal_z: bool
    """ Enable displacement along the Z normal

    :type: bool
    """

    use_x: bool
    """ X axis motion

    :type: bool
    """

    use_y: bool
    """ Y axis motion

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    vertex_group: str
    """ Vertex group name for modulating the wave

    :type: str
    """

    width: float
    """ Distance between the waves

    :type: float
    """

class WireframeModifier(Modifier, bpy_struct):
    """Wireframe effect modifier"""

    crease_weight: float
    """ Crease weight (if active)

    :type: float
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    material_offset: int
    """ Offset material index of generated faces

    :type: int
    """

    offset: float
    """ Offset the thickness from the center

    :type: float
    """

    thickness: float
    """ Thickness factor

    :type: float
    """

    thickness_vertex_group: float
    """ Thickness factor to use for zero vertex group influence

    :type: float
    """

    use_boundary: bool
    """ Support face boundaries

    :type: bool
    """

    use_crease: bool
    """ Crease hub edges for improved subsurf

    :type: bool
    """

    use_even_offset: bool
    """ Scale the offset to give more even thickness

    :type: bool
    """

    use_relative_offset: bool
    """ Scale the offset by surrounding geometry

    :type: bool
    """

    use_replace: bool
    """ Remove original geometry

    :type: bool
    """

    vertex_group: str
    """ Vertex group name for selecting the affected areas

    :type: str
    """

class NodeCustomGroup(Node, bpy_struct):
    """Base node type for custom registered node group types"""

    interface: PropertyGroup
    """ Interface socket data

    :type: PropertyGroup
    """

    node_tree: NodeTree
    """ 

    :type: NodeTree
    """

class NodeInternal(Node, bpy_struct):
    @classmethod
    def poll(cls, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node type can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """
        ...

    def poll_instance(self, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """
        ...

    def update(self):
        """Update on editor changes"""
        ...

    def draw_buttons(self, context: Context, layout: UILayout):
        """Draw node buttons

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_buttons_ext(self, context: Context, layout: UILayout):
        """Draw node buttons in the sidebar

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

class NodeSocketStandard(NodeSocket, bpy_struct):
    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

    def draw(
        self, context: Context, layout: UILayout, node: Node, text: str | typing.Any
    ):
        """Draw socket

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        :param node: Node, Node the socket belongs to
        :type node: Node
        :param text: Text, Text label to draw alongside properties
        :type text: str | typing.Any
        """
        ...

    def draw_color(self, context: Context, node: Node) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context
        :param node: Node, Node the socket belongs to
        :type node: Node
        :return: Color
        :rtype: bpy_prop_array[float]
        """
        ...

class NodeSocketInterfaceStandard(NodeSocketInterface, bpy_struct):
    type: str
    """ Data type

    :type: str
    """

    def draw(self, context: Context, layout: UILayout):
        """Draw template settings

        :param context:
        :type context: Context
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """
        ...

    def draw_color(self, context: Context) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context
        :return: Color
        :rtype: bpy_prop_array[float]
        """
        ...

class ImagePaint(Paint, bpy_struct):
    """Properties of image and texture painting mode"""

    canvas: Image
    """ Image used as canvas

    :type: Image
    """

    clone_image: Image
    """ Image used as clone source

    :type: Image
    """

    dither: float
    """ Amount of dithering when painting on byte images

    :type: float
    """

    invert_stencil: bool
    """ Invert the stencil layer

    :type: bool
    """

    missing_materials: bool
    """ The mesh is missing materials

    :type: bool
    """

    missing_stencil: bool
    """ Image Painting does not have a stencil

    :type: bool
    """

    missing_texture: bool
    """ Image Painting does not have a texture to paint on

    :type: bool
    """

    missing_uvs: bool
    """ A UV layer is missing on the mesh

    :type: bool
    """

    mode: str
    """ Mode of operation for projection painting

    :type: str
    """

    normal_angle: int
    """ Paint most on faces pointing towards the view according to this angle

    :type: int
    """

    screen_grab_size: bpy_prop_array[int]
    """ Size to capture the image for re-projecting

    :type: bpy_prop_array[int]
    """

    seam_bleed: int
    """ Extend paint beyond the faces UVs to reduce seams (in pixels, slower)

    :type: int
    """

    stencil_color: bpy_prop_array[float]
    """ Stencil color in the viewport

    :type: bpy_prop_array[float]
    """

    stencil_image: Image
    """ Image used as stencil

    :type: Image
    """

    use_backface_culling: bool
    """ Ignore faces pointing away from the view (faster)

    :type: bool
    """

    use_clone_layer: bool
    """ Use another UV map as clone source, otherwise use the 3D cursor as the source

    :type: bool
    """

    use_normal_falloff: bool
    """ Paint most on faces pointing towards the view

    :type: bool
    """

    use_occlude: bool
    """ Only paint onto the faces directly under the brush (slower)

    :type: bool
    """

    use_stencil_layer: bool
    """ Set the mask layer from the UV map buttons

    :type: bool
    """

    def detect_data(self) -> bool:
        """Check if required texpaint data exist

        :return:
        :rtype: bool
        """
        ...

class Sculpt(Paint, bpy_struct):
    constant_detail: float
    """ Maximum edge length for dynamic topology sculpting (as percentage of blender unit)

    :type: float
    """

    detail_percent: float
    """ Maximum edge length for dynamic topology sculpting (in brush percenage)

    :type: float
    """

    detail_refine_method: str
    """ In dynamic-topology mode, how to add or remove mesh detail

    :type: str
    """

    detail_size: float
    """ Maximum edge length for dynamic topology sculpting (in pixels)

    :type: float
    """

    detail_type_method: str
    """ In dynamic-topology mode, how mesh detail size is calculated

    :type: str
    """

    gravity: float
    """ Amount of gravity after each dab

    :type: float
    """

    gravity_object: Object
    """ Object whose Z axis defines orientation of gravity

    :type: Object
    """

    lock_x: bool
    """ Disallow changes to the X axis of vertices

    :type: bool
    """

    lock_y: bool
    """ Disallow changes to the Y axis of vertices

    :type: bool
    """

    lock_z: bool
    """ Disallow changes to the Z axis of vertices

    :type: bool
    """

    radial_symmetry: bpy_prop_array[int]
    """ Number of times to copy strokes across the surface

    :type: bpy_prop_array[int]
    """

    show_diffuse_color: bool
    """ Show diffuse color of object and overlay sculpt mask on top of it

    :type: bool
    """

    symmetrize_direction: str
    """ Source and destination for symmetrize operator

    :type: str
    """

    use_deform_only: bool
    """ Use only deformation modifiers (temporary disable all constructive modifiers except multi-resolution)

    :type: bool
    """

    use_smooth_shading: bool
    """ Show faces in dynamic-topology mode with smooth shading rather than flat shaded

    :type: bool
    """

    use_threaded: bool
    """ Take advantage of multiple CPU cores to improve sculpting performance

    :type: bool
    """

class UvSculpt(Paint, bpy_struct): ...

class VertexPaint(Paint, bpy_struct):
    """Properties of vertex and weight paint mode"""

    use_group_restrict: bool
    """ Restrict painting to vertices in the group

    :type: bool
    """

    use_normal: bool
    """ Apply the vertex normal before painting

    :type: bool
    """

    use_spray: bool
    """ Keep applying paint effect while holding mouse

    :type: bool
    """

class IMAGE_UV_sculpt(Panel, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """
        ...

    @staticmethod
    def paint_settings(context):
        """

        :param context:
        """
        ...

    @staticmethod
    def prop_unified_color(parent, context, brush, prop_name, text=""):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param text:
        """
        ...

    @staticmethod
    def prop_unified_color_picker(parent, context, brush, prop_name, value_slider=True):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param value_slider:
        """
        ...

    @staticmethod
    def prop_unified_size(
        parent, context, brush, prop_name, icon="NONE", text="", slider=False
    ):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param icon:
        :param text:
        :param slider:
        """
        ...

    @staticmethod
    def prop_unified_strength(
        parent, context, brush, prop_name, icon="NONE", text="", slider=False
    ):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param icon:
        :param text:
        :param slider:
        """
        ...

    @staticmethod
    def prop_unified_weight(
        parent, context, brush, prop_name, icon="NONE", text="", slider=False
    ):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param icon:
        :param text:
        :param slider:
        """
        ...

    @staticmethod
    def unified_paint_settings(parent, context):
        """

        :param parent:
        :param context:
        """
        ...

class IMAGE_UV_sculpt_curve(Panel, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """
        ...

class BoolProperty(Property, bpy_struct):
    """RNA boolean property definition"""

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: bool
    """ Default value for this number

    :type: bool
    """

    default_array: list[bool]
    """ Default value for this array

    :type: list[bool]
    """

    is_array: bool
    """ 

    :type: bool
    """

class CollectionProperty(Property, bpy_struct):
    """RNA collection property to define lists, arrays and mappings"""

    fixed_type: Struct
    """ Fixed pointer type, empty if variable type

    :type: Struct
    """

class EnumProperty(Property, bpy_struct):
    """RNA enumeration property definition, to choose from a number of predefined options"""

    default: str
    """ Default value for this enum

    :type: str
    """

    default_flag: set[str]
    """ Default value for this enum

    :type: set[str]
    """

    enum_items: bpy_prop_collection[EnumPropertyItem]
    """ Possible values for the property

    :type: bpy_prop_collection[EnumPropertyItem]
    """

    enum_items_static: bpy_prop_collection[EnumPropertyItem]
    """ Possible values for the property (never calls optional dynamic generation of those)

    :type: bpy_prop_collection[EnumPropertyItem]
    """

class FloatProperty(Property, bpy_struct):
    """RNA floating pointer number property definition"""

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: float
    """ Default value for this number

    :type: float
    """

    default_array: bpy_prop_array[float]
    """ Default value for this array

    :type: bpy_prop_array[float]
    """

    hard_max: float
    """ Maximum value used by buttons

    :type: float
    """

    hard_min: float
    """ Minimum value used by buttons

    :type: float
    """

    is_array: bool
    """ 

    :type: bool
    """

    precision: int
    """ Number of digits after the dot used by buttons

    :type: int
    """

    soft_max: float
    """ Maximum value used by buttons

    :type: float
    """

    soft_min: float
    """ Minimum value used by buttons

    :type: float
    """

    step: float
    """ Step size used by number buttons, for floats 1/100th of the step size

    :type: float
    """

class IntProperty(Property, bpy_struct):
    """RNA integer number property definition"""

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: int
    """ Default value for this number

    :type: int
    """

    default_array: bpy_prop_array[int]
    """ Default value for this array

    :type: bpy_prop_array[int]
    """

    hard_max: int
    """ Maximum value used by buttons

    :type: int
    """

    hard_min: int
    """ Minimum value used by buttons

    :type: int
    """

    is_array: bool
    """ 

    :type: bool
    """

    soft_max: int
    """ Maximum value used by buttons

    :type: int
    """

    soft_min: int
    """ Minimum value used by buttons

    :type: int
    """

    step: int
    """ Step size used by number buttons, for floats 1/100th of the step size

    :type: int
    """

class PointerProperty(Property, bpy_struct):
    """RNA pointer property to point to another RNA struct"""

    fixed_type: Struct
    """ Fixed pointer type, empty if variable type

    :type: Struct
    """

class StringProperty(Property, bpy_struct):
    """RNA text string property definition"""

    default: str
    """ string default value

    :type: str
    """

    length_max: int
    """ Maximum length of the string, 0 means unlimited

    :type: int
    """

class CyclesCameraSettings(PropertyGroup, bpy_struct):
    aperture_blades: int
    """ Number of blades in aperture for polygonal bokeh (at least 3)

    :type: int
    """

    aperture_fstop: float
    """ F-stop ratio (lower numbers give more defocus, higher numbers give a sharper image)

    :type: float
    """

    aperture_ratio: float
    """ Distortion to simulate anamorphic lens bokeh

    :type: float
    """

    aperture_rotation: float
    """ Rotation of blades in aperture

    :type: float
    """

    aperture_size: float
    """ Radius of the aperture for depth of field (higher values give more defocus)

    :type: float
    """

    aperture_type: str
    """ Use f-stop number or aperture radius

    :type: str
    """

    fisheye_fov: float
    """ Field of view for the fisheye lens

    :type: float
    """

    fisheye_lens: float
    """ Lens focal length (mm)

    :type: float
    """

    latitude_max: float
    """ Maximum latitude (vertical angle) for the equirectangular lens

    :type: float
    """

    latitude_min: float
    """ Minimum latitude (vertical angle) for the equirectangular lens

    :type: float
    """

    longitude_max: float
    """ Maximum longitude (horizontal angle) for the equirectangular lens

    :type: float
    """

    longitude_min: float
    """ Minimum longitude (horizontal angle) for the equirectangular lens

    :type: float
    """

    panorama_type: str
    """ Distortion to use for the calculation

    :type: str
    """

class CyclesCurveRenderSettings(PropertyGroup, bpy_struct):
    cull_backfacing: bool
    """ Do not test the back-face of each strand

    :type: bool
    """

    maximum_width: float
    """ Maximum extension that strand radius can be increased by

    :type: float
    """

    minimum_width: float
    """ Minimal pixel width for strands (0 - deactivated)

    :type: float
    """

    primitive: str
    """ Type of primitive used for hair rendering

    :type: str
    """

    resolution: int
    """ Resolution of generated mesh

    :type: int
    """

    shape: str
    """ Form of hair

    :type: str
    """

    subdivisions: int
    """ Number of subdivisions used in Cardinal curve intersection (power of 2)

    :type: int
    """

    use_curves: bool
    """ Activate Cycles hair rendering for particle system

    :type: bool
    """

class CyclesCurveSettings(PropertyGroup, bpy_struct):
    radius_scale: float
    """ Multiplier of width properties

    :type: float
    """

    root_width: float
    """ Strand's width at root

    :type: float
    """

    shape: float
    """ Strand shape parameter

    :type: float
    """

    tip_width: float
    """ Strand's width at tip

    :type: float
    """

    use_closetip: bool
    """ Set tip radius to zero

    :type: bool
    """

class CyclesDeviceSettings(PropertyGroup, bpy_struct):
    id: str
    """ 

    :type: str
    """

    name: str
    """ 

    :type: str
    """

    type: str
    """ 

    :type: str
    """

    use: bool
    """ 

    :type: bool
    """

class CyclesLampSettings(PropertyGroup, bpy_struct):
    cast_shadow: bool
    """ Lamp casts shadows

    :type: bool
    """

    is_portal: bool
    """ Use this area lamp to guide sampling of the background, note that this will make the lamp invisible

    :type: bool
    """

    max_bounces: int
    """ Maximum number of bounces the light will contribute to the render

    :type: int
    """

    samples: int
    """ Number of light samples to render for each AA sample

    :type: int
    """

    use_multiple_importance_sampling: bool
    """ Use multiple importance sampling for the lamp, reduces noise for area lamps and sharp glossy materials

    :type: bool
    """

class CyclesMaterialSettings(PropertyGroup, bpy_struct):
    displacement_method: str
    """ Method to use for the displacement

    :type: str
    """

    homogeneous_volume: bool
    """ When using volume rendering, assume volume has the same density everywhere (not using any textures), for faster rendering

    :type: bool
    """

    sample_as_light: bool
    """ Use multiple importance sampling for this material, disabling may reduce overall noise for large objects that emit little light compared to other light sources

    :type: bool
    """

    use_transparent_shadow: bool
    """ Use transparent shadows for this material if it contains a Transparent BSDF, disabling will render faster but not give accurate shadows

    :type: bool
    """

    volume_interpolation: str
    """ Interpolation method to use for smoke/fire volumes

    :type: str
    """

    volume_sampling: str
    """ Sampling method to use for volumes

    :type: str
    """

class CyclesMeshSettings(PropertyGroup, bpy_struct): ...

class CyclesObjectSettings(PropertyGroup, bpy_struct):
    dicing_rate: float
    """ Multiplier for scene dicing rate (located in the Geometry Panel)

    :type: float
    """

    motion_steps: int
    """ Control accuracy of deformation motion blur, more steps gives more memory usage (actual number of steps is 2^(steps - 1))

    :type: int
    """

    use_adaptive_subdivision: bool
    """ Use adaptive render time subdivision

    :type: bool
    """

    use_camera_cull: bool
    """ Allow this object and its duplicators to be culled by camera space culling

    :type: bool
    """

    use_deform_motion: bool
    """ Use deformation motion blur for this object

    :type: bool
    """

    use_distance_cull: bool
    """ Allow this object and its duplicators to be culled by distance from camera

    :type: bool
    """

    use_motion_blur: bool
    """ Use motion blur for this object

    :type: bool
    """

class CyclesRenderSettings(PropertyGroup, bpy_struct):
    aa_samples: int
    """ Number of antialiasing samples to render for each pixel

    :type: int
    """

    ao_samples: int
    """ Number of ambient occlusion samples to render for each AA sample

    :type: int
    """

    bake_type: str
    """ Type of pass to bake

    :type: str
    """

    blur_glossy: float
    """ Adaptively blur glossy shaders after blurry bounces, to reduce noise at the cost of accuracy

    :type: float
    """

    camera_cull_margin: float
    """ Margin for the camera space culling

    :type: float
    """

    caustics_reflective: bool
    """ Use reflective caustics, resulting in a brighter image (more noise but added realism)

    :type: bool
    """

    caustics_refractive: bool
    """ Use refractive caustics, resulting in a brighter image (more noise but added realism)

    :type: bool
    """

    debug_bvh_time_steps: int
    """ Split BVH primitives by this number of time steps to speed up render time in cost of memory

    :type: int
    """

    debug_bvh_type: str
    """ Choose between faster updates, or faster render

    :type: str
    """

    debug_cancel_timeout: float
    """ 

    :type: float
    """

    debug_opencl_device_type: str
    """ 

    :type: str
    """

    debug_opencl_kernel_type: str
    """ 

    :type: str
    """

    debug_reset_timeout: float
    """ 

    :type: float
    """

    debug_text_timeout: float
    """ 

    :type: float
    """

    debug_tile_size: int
    """ 

    :type: int
    """

    debug_use_cpu_avx: bool
    """ 

    :type: bool
    """

    debug_use_cpu_avx2: bool
    """ 

    :type: bool
    """

    debug_use_cpu_sse2: bool
    """ 

    :type: bool
    """

    debug_use_cpu_sse3: bool
    """ 

    :type: bool
    """

    debug_use_cpu_sse41: bool
    """ 

    :type: bool
    """

    debug_use_cuda_adaptive_compile: bool
    """ 

    :type: bool
    """

    debug_use_hair_bvh: bool
    """ Use special type BVH optimized for hair (uses more ram but renders faster)

    :type: bool
    """

    debug_use_opencl_debug: bool
    """ 

    :type: bool
    """

    debug_use_qbvh: bool
    """ 

    :type: bool
    """

    debug_use_spatial_splits: bool
    """ Use BVH spatial splits: longer builder time, faster render

    :type: bool
    """

    device: str
    """ Device to use for rendering

    :type: str
    """

    dicing_rate: float
    """ Size of a micropolygon in pixels

    :type: float
    """

    diffuse_bounces: int
    """ Maximum number of diffuse reflection bounces, bounded by total maximum

    :type: int
    """

    diffuse_samples: int
    """ Number of diffuse bounce samples to render for each AA sample

    :type: int
    """

    distance_cull_margin: float
    """ Cull objects which are further away from camera than this distance

    :type: float
    """

    feature_set: str
    """ Feature set to use for rendering

    :type: str
    """

    film_exposure: float
    """ Image brightness scale

    :type: float
    """

    film_transparent: bool
    """ World background is transparent with premultiplied alpha

    :type: bool
    """

    filter_type: str
    """ Pixel filter type

    :type: str
    """

    filter_width: float
    """ Pixel filter width

    :type: float
    """

    glossy_bounces: int
    """ Maximum number of glossy reflection bounces, bounded by total maximum

    :type: int
    """

    glossy_samples: int
    """ Number of glossy bounce samples to render for each AA sample

    :type: int
    """

    light_sampling_threshold: float
    """ Probabilistically terminate light samples when the light contribution is below this threshold (more noise but faster rendering). Zero disables the test and never ignores lights

    :type: float
    """

    max_bounces: int
    """ Total maximum number of bounces

    :type: int
    """

    max_subdivisions: int
    """ Stop subdividing when this level is reached even if the dice rate would produce finer tessellation

    :type: int
    """

    mesh_light_samples: int
    """ Number of mesh emission light samples to render for each AA sample

    :type: int
    """

    min_bounces: int
    """ Minimum number of bounces, setting this lower than the maximum enables probabilistic path termination (faster but noisier)

    :type: int
    """

    motion_blur_position: str
    """ Offset for the shutter's time interval, allows to change the motion blur trails

    :type: str
    """

    pixel_filter_type: str
    """ Pixel filter type

    :type: str
    """

    preview_aa_samples: int
    """ Number of antialiasing samples to render in the viewport, unlimited if 0

    :type: int
    """

    preview_active_layer: bool
    """ Preview active render layer in viewport

    :type: bool
    """

    preview_dicing_rate: float
    """ Size of a micropolygon in pixels during preview render

    :type: float
    """

    preview_pause: bool
    """ Pause all viewport preview renders

    :type: bool
    """

    preview_samples: int
    """ Number of samples to render in the viewport, unlimited if 0

    :type: int
    """

    preview_start_resolution: int
    """ Resolution to start rendering preview at, progressively increasing it to the full viewport size

    :type: int
    """

    progressive: str
    """ Method to sample lights and materials

    :type: str
    """

    rolling_shutter_duration: float
    """ Scanline "exposure" time for the rolling shutter effect

    :type: float
    """

    rolling_shutter_type: str
    """ Type of rolling shutter effect matching CMOS-based cameras

    :type: str
    """

    sample_all_lights_direct: bool
    """ Sample all lights (for direct samples), rather than randomly picking one

    :type: bool
    """

    sample_all_lights_indirect: bool
    """ Sample all lights (for indirect samples), rather than randomly picking one

    :type: bool
    """

    sample_clamp_direct: float
    """ If non-zero, the maximum value for a direct sample, higher values will be scaled down to avoid too much noise and slow convergence at the cost of accuracy

    :type: float
    """

    sample_clamp_indirect: float
    """ If non-zero, the maximum value for an indirect sample, higher values will be scaled down to avoid too much noise and slow convergence at the cost of accuracy

    :type: float
    """

    samples: int
    """ Number of samples to render for each pixel

    :type: int
    """

    sampling_pattern: str
    """ Random sampling pattern used by the integrator

    :type: str
    """

    seed: int
    """ Seed value for integrator to get different noise patterns

    :type: int
    """

    shading_system: bool
    """ Use Open Shading Language (CPU rendering only)

    :type: bool
    """

    subsurface_samples: int
    """ Number of subsurface scattering samples to render for each AA sample

    :type: int
    """

    texture_limit: str
    """ Limit texture size used by viewport rendering

    :type: str
    """

    texture_limit_render: str
    """ Limit texture size used by final rendering

    :type: str
    """

    tile_order: str
    """ Tile order for rendering

    :type: str
    """

    transmission_bounces: int
    """ Maximum number of transmission bounces, bounded by total maximum

    :type: int
    """

    transmission_samples: int
    """ Number of transmission bounce samples to render for each AA sample

    :type: int
    """

    transparent_max_bounces: int
    """ Maximum number of transparent bounces

    :type: int
    """

    transparent_min_bounces: int
    """ Minimum number of transparent bounces, setting this lower than the maximum enables probabilistic path termination (faster but noisier)

    :type: int
    """

    use_animated_seed: bool
    """ Use different seed values (and hence noise patterns) at different frames

    :type: bool
    """

    use_camera_cull: bool
    """ Allow objects to be culled based on the camera frustum

    :type: bool
    """

    use_distance_cull: bool
    """ Allow objects to be culled based on the distance from camera

    :type: bool
    """

    use_layer_samples: str
    """ How to use per render layer sample settings

    :type: str
    """

    use_progressive_refine: bool
    """ Instead of rendering each tile until it is finished, refine the whole image progressively (this renders somewhat slower, but time can be saved by manually stopping the render when the noise is low enough)

    :type: bool
    """

    use_square_samples: bool
    """ Square sampling values for easier artist control

    :type: bool
    """

    use_transparent_shadows: bool
    """ Use transparency of surfaces for rendering shadows

    :type: bool
    """

    volume_bounces: int
    """ Maximum number of volumetric scattering events

    :type: int
    """

    volume_max_steps: int
    """ Maximum number of steps through the volume before giving up, to avoid extremely long render times with big objects or small step sizes

    :type: int
    """

    volume_samples: int
    """ Number of volume scattering samples to render for each AA sample

    :type: int
    """

    volume_step_size: float
    """ Distance between volume shader samples when rendering the volume (lower values give more accurate and detailed results, but also increased render time)

    :type: float
    """

class CyclesVisibilitySettings(PropertyGroup, bpy_struct):
    camera: bool
    """ Object visibility for camera rays

    :type: bool
    """

    diffuse: bool
    """ Object visibility for diffuse reflection rays

    :type: bool
    """

    glossy: bool
    """ Object visibility for glossy reflection rays

    :type: bool
    """

    scatter: bool
    """ Object visibility for volume scatter rays

    :type: bool
    """

    shadow: bool
    """ Object visibility for shadow rays

    :type: bool
    """

    transmission: bool
    """ Object visibility for transmission rays

    :type: bool
    """

class CyclesWorldSettings(PropertyGroup, bpy_struct):
    homogeneous_volume: bool
    """ When using volume rendering, assume volume has the same density everywhere(not using any textures), for faster rendering

    :type: bool
    """

    max_bounces: int
    """ Maximum number of bounces the background light will contribute to the render

    :type: int
    """

    sample_as_light: bool
    """ Use multiple importance sampling for the environment, enabling for non-solid colors is recommended

    :type: bool
    """

    sample_map_resolution: int
    """ Importance map size is resolution x resolution; higher values potentially produce less noise, at the cost of memory and speed

    :type: int
    """

    samples: int
    """ Number of light samples to render for each AA sample

    :type: int
    """

    volume_interpolation: str
    """ Interpolation method to use for volumes

    :type: str
    """

    volume_sampling: str
    """ Sampling method to use for volumes

    :type: str
    """

class NodeSetting(PropertyGroup, bpy_struct):
    value: str
    """ Python expression to be evaluated as the initial node setting

    :type: str
    """

class OperatorFileListElement(PropertyGroup, bpy_struct):
    name: str
    """ Name of a file or directory within a file list

    :type: str
    """

class OperatorMousePath(PropertyGroup, bpy_struct):
    """Mouse path values for operators that record such paths"""

    loc: bpy_prop_array[float]
    """ Mouse location

    :type: bpy_prop_array[float]
    """

    time: float
    """ Time of mouse location

    :type: float
    """

class OperatorStrokeElement(PropertyGroup, bpy_struct):
    is_start: bool
    """ 

    :type: bool
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    mouse: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    pen_flip: bool
    """ 

    :type: bool
    """

    pressure: float
    """ Tablet pressure

    :type: float
    """

    size: float
    """ Brush Size in screen space

    :type: float
    """

    time: float
    """ 

    :type: float
    """

class SelectedUvElement(PropertyGroup, bpy_struct):
    element_index: int
    """ 

    :type: int
    """

    face_index: int
    """ 

    :type: int
    """

class CYCLES(RenderEngine, bpy_struct):
    def bake(
        self,
        scene,
        obj,
        pass_type,
        pass_filter,
        object_id,
        pixel_array,
        num_pixels,
        depth,
        result,
    ):
        """

        :param scene:
        :param obj:
        :param pass_type:
        :param pass_filter:
        :param object_id:
        :param pixel_array:
        :param num_pixels:
        :param depth:
        :param result:
        """
        ...

    def render(self, scene):
        """

        :param scene:
        """
        ...

    def update(self, data, scene):
        """

        :param data:
        :param scene:
        """
        ...

    def update_script_node(self, node):
        """

        :param node:
        """
        ...

    def view_draw(self, context):
        """

        :param context:
        """
        ...

    def view_update(self, context):
        """

        :param context:
        """
        ...

class ActuatorSensor(Sensor, bpy_struct):
    """Sensor to detect state modifications of actuators"""

    actuator: str
    """ Actuator name, actuator active state modifications will be detected

    :type: str
    """

class AlwaysSensor(Sensor, bpy_struct):
    """Sensor to generate continuous pulses"""

    ...

class ArmatureSensor(Sensor, bpy_struct):
    """Sensor to detect values and changes in values of IK solver"""

    bone: str
    """ Identify the bone to check value from

    :type: str
    """

    constraint: str
    """ Identify the bone constraint to check value from

    :type: str
    """

    test_type: str
    """ Type of value and test

    :type: str
    """

    value: float
    """ Value to be used in comparison

    :type: float
    """

class CollisionSensor(Sensor, bpy_struct):
    """Sensor to detect objects colliding with the current object, with more settings than the Touch sensor"""

    material: str
    """ Only look for objects with this material (blank = all objects)

    :type: str
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

    use_material: bool
    """ Toggle collision on material or property

    :type: bool
    """

    use_pulse: bool
    """ Change to the set of colliding objects generates pulse

    :type: bool
    """

class DelaySensor(Sensor, bpy_struct):
    """Sensor to send delayed events"""

    delay: int
    """ Delay in number of logic tics before the positive trigger (default 60 per second)

    :type: int
    """

    duration: int
    """ If >0, delay in number of logic tics before the negative trigger following the positive trigger

    :type: int
    """

    use_repeat: bool
    """ Toggle repeat option (if selected, the sensor restarts after Delay+Duration logic tics)

    :type: bool
    """

class JoystickSensor(Sensor, bpy_struct):
    """Sensor to detect joystick events"""

    axis_direction: str
    """ The direction of the axis

    :type: str
    """

    axis_number: int
    """ Which axis pair to use, 1 is usually the main direction input

    :type: int
    """

    axis_threshold: int
    """ Precision of the axis

    :type: int
    """

    button_number: int
    """ Which button to use

    :type: int
    """

    event_type: str
    """ The type of event this joystick sensor is triggered on

    :type: str
    """

    hat_direction: str
    """ Hat direction

    :type: str
    """

    hat_number: int
    """ Which hat to use

    :type: int
    """

    joystick_index: int
    """ Which joystick to use

    :type: int
    """

    single_axis_number: int
    """ Single axis (vertical/horizontal/other) to detect

    :type: int
    """

    use_all_events: bool
    """ Triggered by all events on this joystick's current type (axis/button/hat)

    :type: bool
    """

class KeyboardSensor(Sensor, bpy_struct):
    """Sensor to detect keyboard events"""

    key: str
    """ 

    :type: str
    """

    log: str
    """ Property that indicates whether to log keystrokes as a string

    :type: str
    """

    modifier_key_1: str
    """ Modifier key code

    :type: str
    """

    modifier_key_2: str
    """ Modifier key code

    :type: str
    """

    target: str
    """ Property that receives the keystrokes in case a string is logged

    :type: str
    """

    use_all_keys: bool
    """ Trigger this sensor on any keystroke

    :type: bool
    """

class MessageSensor(Sensor, bpy_struct):
    """Sensor to detect incoming messages"""

    subject: str
    """ Optional subject filter: only accept messages with this subject, or empty to accept all

    :type: str
    """

class MouseSensor(Sensor, bpy_struct):
    """Sensor to detect mouse events"""

    material: str
    """ Only look for objects with this material (blank = all objects)

    :type: str
    """

    mouse_event: str
    """ Type of event this mouse sensor should trigger on

    :type: str
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

    use_material: str
    """ Toggle collision on material or property

    :type: str
    """

    use_pulse: bool
    """ Moving the mouse over a different object generates a pulse

    :type: bool
    """

    use_x_ray: bool
    """ Toggle X-Ray option (see through objects that don't have the property)

    :type: bool
    """

class NearSensor(Sensor, bpy_struct):
    """Sensor to detect nearby objects"""

    distance: float
    """ Trigger distance

    :type: float
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

    reset_distance: float
    """ The distance where the sensor forgets the actor

    :type: float
    """

class PropertySensor(Sensor, bpy_struct):
    """Sensor to detect values and changes in values of properties"""

    evaluation_type: str
    """ Type of property evaluation

    :type: str
    """

    property: str
    """ 

    :type: str
    """

    value: str
    """ Check for this value in types in Equal, Not Equal, Less Than and Greater Than types

    :type: str
    """

    value_max: str
    """ Maximum value in Interval type

    :type: str
    """

    value_min: str
    """ Minimum value in Interval type

    :type: str
    """

class RadarSensor(Sensor, bpy_struct):
    """Sensor to detect objects in a cone shaped radar emanating from the current object"""

    angle: float
    """ Opening angle of the radar cone

    :type: float
    """

    axis: str
    """ Along which axis the radar cone is cast

    :type: str
    """

    distance: float
    """ Depth of the radar cone

    :type: float
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

class RandomSensor(Sensor, bpy_struct):
    """Sensor to send random events"""

    seed: int
    """ Initial seed of the generator (choose 0 for not random)

    :type: int
    """

class RaySensor(Sensor, bpy_struct):
    """Sensor to detect intersections with a ray emanating from the current object"""

    axis: str
    """ Along which axis the ray is cast

    :type: str
    """

    material: str
    """ Only look for objects with this material (blank = all objects)

    :type: str
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

    range: float
    """ Sense objects no farther than this distance

    :type: float
    """

    ray_type: str
    """ Toggle collision on material or property

    :type: str
    """

    use_x_ray: bool
    """ Toggle X-Ray option (see through objects that don't have the property)

    :type: bool
    """

class EffectSequence(Sequence, bpy_struct):
    """Sequence strip applying an effect on the images created by other strips"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class ImageSequence(Sequence, bpy_struct):
    """Sequence strip to load one or more images"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    directory: str
    """ 

    :type: str
    """

    elements: SequenceElements
    """ 

    :type: SequenceElements
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_multiview: bool
    """ Use Multiple Views (when available)

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

    views_format: str
    """ Mode to load image views

    :type: str
    """

class MaskSequence(Sequence, bpy_struct):
    """Sequence strip to load a video from a mask"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    mask: Mask
    """ Mask that this sequence uses

    :type: Mask
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class MetaSequence(Sequence, bpy_struct):
    """Sequence strip to group other strips as a single sequence strip"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    sequences: bpy_prop_collection[Sequence]
    """ 

    :type: bpy_prop_collection[Sequence]
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class MovieClipSequence(Sequence, bpy_struct):
    """Sequence strip to load a video from the clip editor"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    stabilize2d: bool
    """ Use the 2D stabilized version of the clip

    :type: bool
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    undistort: bool
    """ Use the undistorted version of the clip

    :type: bool
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class MovieSequence(Sequence, bpy_struct):
    """Sequence strip to load a video"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    elements: bpy_prop_collection[SequenceElement]
    """ 

    :type: bpy_prop_collection[SequenceElement]
    """

    filepath: str
    """ 

    :type: str
    """

    mpeg_preseek: int
    """ For MPEG movies, preseek this many frames

    :type: int
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    stream_index: int
    """ For files with several movie streams, use the stream with the given index

    :type: int
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_multiview: bool
    """ Use Multiple Views (when available)

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

    views_format: str
    """ Mode to load movie views

    :type: str
    """

class SceneSequence(Sequence, bpy_struct):
    """Sequence strip to used the rendered image of a scene"""

    alpha_mode: str
    """ Representation of alpha information in the RGBA pixels

    :type: str
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    scene: Scene
    """ Scene that this sequence uses

    :type: Scene
    """

    scene_camera: Object
    """ Override the scenes active camera

    :type: Object
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_grease_pencil: bool
    """ Show Grease Pencil strokes in OpenGL previews

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_sequence: bool
    """ Use scenes sequence strips directly, instead of rendering

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class SoundSequence(Sequence, bpy_struct):
    """Sequence strip defining a sound to be played over a period of time"""

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    pan: float
    """ Playback panning of the sound (only for Mono sources)

    :type: float
    """

    pitch: float
    """ Playback pitch of the sound

    :type: float
    """

    show_waveform: bool
    """ Whether to draw the sound's waveform

    :type: bool
    """

    sound: Sound
    """ Sound data-block used by this sequence

    :type: Sound
    """

    volume: float
    """ Playback volume of the sound

    :type: float
    """

class SequenceColorBalance(SequenceColorBalanceData, bpy_struct):
    """Color balance parameters for a sequence strip"""

    ...

class BrightContrastModifier(SequenceModifier, bpy_struct):
    """Bright/contrast modifier data for sequence strip"""

    bright: float
    """ 

    :type: float
    """

    contrast: float
    """ 

    :type: float
    """

class ColorBalanceModifier(SequenceModifier, bpy_struct):
    """Color balance modifier for sequence strip"""

    color_balance: SequenceColorBalanceData
    """ 

    :type: SequenceColorBalanceData
    """

    color_multiply: float
    """ 

    :type: float
    """

class CurvesModifier(SequenceModifier, bpy_struct):
    """RGB curves modifier for sequence strip"""

    curve_mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

class HueCorrectModifier(SequenceModifier, bpy_struct):
    """Hue correction modifier for sequence strip"""

    curve_mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

class SequencerTonemapModifierData(SequenceModifier, bpy_struct):
    """Tone mapping modifier"""

    adaptation: float
    """ If 0, global; if 1, based on pixel intensity

    :type: float
    """

    contrast: float
    """ Set to 0 to use estimate from input image

    :type: float
    """

    correction: float
    """ If 0, same for all channels; if 1, each independent

    :type: float
    """

    gamma: float
    """ If not used, set to 1

    :type: float
    """

    intensity: float
    """ If less than zero, darkens image; otherwise, makes it brighter

    :type: float
    """

    key: float
    """ The value the average luminance is mapped to

    :type: float
    """

    offset: float
    """ Normally always 1, but can be used as an extra control to alter the brightness curve

    :type: float
    """

    tonemap_type: str
    """ 

    :type: str
    """

class WhiteBalanceModifier(SequenceModifier, bpy_struct):
    """White balance modifier for sequence strip"""

    white_value: bpy_prop_array[float]
    """ This color defines white in the strip

    :type: bpy_prop_array[float]
    """

class SpaceClipEditor(Space, bpy_struct):
    """Clip editor space data"""

    clip: MovieClip
    """ Movie clip displayed and edited in this space

    :type: MovieClip
    """

    clip_user: MovieClipUser
    """ Parameters defining which frame of the movie clip is displayed

    :type: MovieClipUser
    """

    grease_pencil_source: str
    """ Where the grease pencil comes from

    :type: str
    """

    lock_selection: bool
    """ Lock viewport to selected markers during playback

    :type: bool
    """

    lock_time_cursor: bool
    """ Lock curves view to time cursor during playback and tracking

    :type: bool
    """

    mask: Mask
    """ Mask displayed and edited in this space

    :type: Mask
    """

    mask_draw_type: str
    """ Draw type for mask splines

    :type: str
    """

    mask_overlay_mode: str
    """ Overlay mode of rasterized mask

    :type: str
    """

    mode: str
    """ Editing context being displayed

    :type: str
    """

    path_length: int
    """ Length of displaying path, in frames

    :type: int
    """

    pivot_point: str
    """ Pivot center for rotation/scaling

    :type: str
    """

    scopes: MovieClipScopes
    """ Scopes to visualize movie clip statistics

    :type: MovieClipScopes
    """

    show_blue_channel: bool
    """ Show blue channel in the frame

    :type: bool
    """

    show_bundles: bool
    """ Show projection of 3D markers into footage

    :type: bool
    """

    show_disabled: bool
    """ Show disabled tracks from the footage

    :type: bool
    """

    show_filters: bool
    """ Show filters for graph editor

    :type: bool
    """

    show_graph_frames: bool
    """ Show curve for per-frame average error (camera motion should be solved first)

    :type: bool
    """

    show_graph_hidden: bool
    """ Include channels from objects/bone that aren't visible

    :type: bool
    """

    show_graph_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    show_graph_tracks_error: bool
    """ Display the reprojection error curve for selected tracks

    :type: bool
    """

    show_graph_tracks_motion: bool
    """ Display the speed curves (in "x" direction red, in "y" direction green) for the selected tracks

    :type: bool
    """

    show_grease_pencil: bool
    """ Show grease pencil for this view

    :type: bool
    """

    show_green_channel: bool
    """ Show green channel in the frame

    :type: bool
    """

    show_grid: bool
    """ Show grid showing lens distortion

    :type: bool
    """

    show_marker_pattern: bool
    """ Show pattern boundbox for markers

    :type: bool
    """

    show_marker_search: bool
    """ Show search boundbox for markers

    :type: bool
    """

    show_mask_overlay: bool
    """ 

    :type: bool
    """

    show_mask_smooth: bool
    """ 

    :type: bool
    """

    show_metadata: bool
    """ Show metadata of clip

    :type: bool
    """

    show_names: bool
    """ Show track names and status

    :type: bool
    """

    show_red_channel: bool
    """ Show red channel in the frame

    :type: bool
    """

    show_seconds: bool
    """ Show timing in seconds not frames

    :type: bool
    """

    show_stable: bool
    """ Show stable footage in editor (if stabilization is enabled)

    :type: bool
    """

    show_tiny_markers: bool
    """ Show markers in a more compact manner

    :type: bool
    """

    show_track_path: bool
    """ Show path of how track moves

    :type: bool
    """

    use_grayscale_preview: bool
    """ Display frame in grayscale mode

    :type: bool
    """

    use_manual_calibration: bool
    """ Use manual calibration helpers

    :type: bool
    """

    use_mute_footage: bool
    """ Mute footage and show black background instead

    :type: bool
    """

    view: str
    """ Type of the clip editor view

    :type: str
    """

    def draw_handler_add(self):
        """Undocumented"""
        ...

    def draw_handler_remove(self):
        """Undocumented"""
        ...

class SpaceConsole(Space, bpy_struct):
    """Interactive python console"""

    font_size: int
    """ Font size to use for displaying the text

    :type: int
    """

    history: bpy_prop_collection[ConsoleLine]
    """ Command history

    :type: bpy_prop_collection[ConsoleLine]
    """

    language: str
    """ Command line prompt language

    :type: str
    """

    prompt: str
    """ Command line prompt

    :type: str
    """

    scrollback: bpy_prop_collection[ConsoleLine]
    """ Command output

    :type: bpy_prop_collection[ConsoleLine]
    """

    select_end: int
    """ 

    :type: int
    """

    select_start: int
    """ 

    :type: int
    """

    def draw_handler_add(self):
        """Undocumented"""
        ...

    def draw_handler_remove(self):
        """Undocumented"""
        ...

class SpaceDopeSheetEditor(Space, bpy_struct):
    """Dope Sheet space data"""

    action: Action
    """ Action displayed and edited in this space

    :type: Action
    """

    auto_snap: str
    """ Automatic time snapping settings for transformations

    :type: str
    """

    dopesheet: DopeSheet
    """ Settings for filtering animation data

    :type: DopeSheet
    """

    mode: str
    """ Editing context being displayed

    :type: str
    """

    show_frame_indicator: bool
    """ Show frame number beside the current frame indicator line

    :type: bool
    """

    show_group_colors: bool
    """ Draw groups and channels with colors matching their corresponding groups (pose bones only currently)

    :type: bool
    """

    show_pose_markers: bool
    """ Show markers belonging to the active action instead of Scene markers (Action and Shape Key Editors only)

    :type: bool
    """

    show_seconds: bool
    """ Show timing in seconds not frames

    :type: bool
    """

    show_sliders: bool
    """ Show sliders beside F-Curve channels

    :type: bool
    """

    use_auto_merge_keyframes: bool
    """ Automatically merge nearby keyframes

    :type: bool
    """

    use_marker_sync: bool
    """ Sync Markers with keyframe edits

    :type: bool
    """

    use_realtime_update: bool
    """ When transforming keyframes, changes to the animation data are flushed to other views

    :type: bool
    """

    def draw_handler_add(self):
        """Undocumented"""
        ...

    def draw_handler_remove(self):
        """Undocumented"""
        ...

class SpaceFileBrowser(Space, bpy_struct):
    """File browser space data"""

    active_operator: Operator
    """ 

    :type: Operator
    """

    bookmarks: bpy_prop_collection[FileBrowserFSMenuEntry]
    """ User's bookmarks

    :type: bpy_prop_collection[FileBrowserFSMenuEntry]
    """

    bookmarks_active: int
    """ Index of active bookmark (-1 if none)

    :type: int
    """

    operator: Operator
    """ 

    :type: Operator
    """

    params: FileSelectParams
    """ Parameters and Settings for the Filebrowser

    :type: FileSelectParams
    """

    recent_folders: bpy_prop_collection[FileBrowserFSMenuEntry]
    """ 

    :type: bpy_prop_collection[FileBrowserFSMenuEntry]
    """

    recent_folders_active: int
    """ Index of active recent folder (-1 if none)

    :type: int
    """

    system_bookmarks: bpy_prop_collection[FileBrowserFSMenuEntry]
    """ System's bookmarks

    :type: bpy_prop_collection[FileBrowserFSMenuEntry]
    """

    system_bookmarks_active: int
    """ Index of active system bookmark (-1 if none)

    :type: int
    """

    system_folders: bpy_prop_collection[FileBrowserFSMenuEntry]
    """ System's folders (usually root, available hard drives, etc)

    :type: bpy_prop_collection[FileBrowserFSMenuEntry]
    """

    system_folders_active: int
    """ Index of active system folder (-1 if none)

    :type: int
    """

    def draw_handler_add(self):
        """Undocumented"""
        ...

    def draw_handler_remove(self):
        """Undocumented"""
        ...

class SpaceGraphEditor(Space, bpy_struct):
    """Graph Editor space data"""

    auto_snap: str
    """ Automatic time snapping settings for transformations

    :type: str
    """

    cursor_position_x: float
    """ Graph Editor 2D-Value cursor - X-Value component

    :type: float
    """

    cursor_position_y: float
    """ Graph Editor 2D-Value cursor - Y-Value component

    :type: float
    """

    dopesheet: DopeSheet
    """ Settings for filtering animation data

    :type: DopeSheet
    """

    has_ghost_curves: bool
    """ Graph Editor instance has some ghost curves stored

    :type: bool
    """

    mode: str
    """ Editing context being displayed

    :type: str
    """

    pivot_point: str
    """ Pivot center for rotation/scaling

    :type: str
    """

    show_cursor: bool
    """ Show 2D cursor

    :type: bool
    """

    show_frame_indicator: bool
    """ Show frame number beside the current frame indicator line

    :type: bool
    """

    show_group_colors: bool
    """ Draw groups and channels with colors matching their corresponding groups

    :type: bool
    """

    show_handles: bool
    """ Show handles of Bezier control points

    :type: bool
    """

    show_seconds: bool
    """ Show timing in seconds not frames

    :type: bool
    """

    show_sliders: bool
    """ Show sliders beside F-Curve channels

    :type: bool
    """

    use_auto_merge_keyframes: bool
    """ Automatically merge nearby keyframes

    :type: bool
    """

    use_auto_normalization: bool
    """ Automatically recalculate curve normalization on every curve edit

    :type: bool
    """

    use_beauty_drawing: bool
    """ Draw F-Curves using Anti-Aliasing and other fancy effects (disable for better performance)

    :type: bool
    """

    use_normalization: bool
    """ Display curves in normalized to -1..1 range, for easier editing of multiple curves with different ranges

    :type: bool
    """

    use_only_selected_curves_handles: bool
    """ Only keyframes of selected F-Curves are visible and editable

    :type: bool
    """

    use_only_selected_keyframe_handles: bool
    """ Only show and edit handles of selected keyframes

    :type: bool
    """

    use_realtime_update: bool
    """ When transforming keyframes, changes to the animation data are flushed to other views

    :type: bool
    """

    def draw_handler_add(self):
        """Undocumented"""
        ...

    def draw_handler_remove(self):
        """Undocumented"""
        ...

class SpaceImageEditor(Space, bpy_struct):
    """Image and UV editor space data"""

    cursor_location: bpy_prop_array[float]
    """ 2D cursor location for this view

    :type: bpy_prop_array[float]
    """

    draw_channels: str
    """ Channels of the image to draw

    :type: str
    """

    grease_pencil: GreasePencil
    """ Grease pencil data for this space

    :type: GreasePencil
    """

    image: Image
    """ Image displayed and edited in this space

    :type: Image
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    mask: Mask
    """ Mask displayed and edited in this space

    :type: Mask
    """

    mask_draw_type: str
    """ Draw type for mask splines

    :type: str
    """

    mask_overlay_mode: str
    """ Overlay mode of rasterized mask

    :type: str
    """

    mode: str
    """ Editing context being displayed

    :type: str
    """

    pivot_point: str
    """ Rotation/Scaling Pivot

    :type: str
    """

    sample_histogram: Histogram
    """ Sampled colors along line

    :type: Histogram
    """

    scopes: Scopes
    """ Scopes to visualize image statistics

    :type: Scopes
    """

    show_grease_pencil: bool
    """ Show grease pencil for this view

    :type: bool
    """

    show_mask_overlay: bool
    """ 

    :type: bool
    """

    show_mask_smooth: bool
    """ 

    :type: bool
    """

    show_maskedit: bool
    """ Show Mask editing related properties

    :type: bool
    """

    show_paint: bool
    """ Show paint related properties

    :type: bool
    """

    show_render: bool
    """ Show render related properties

    :type: bool
    """

    show_repeat: bool
    """ Draw the image repeated outside of the main view

    :type: bool
    """

    show_stereo_3d: bool
    """ Display the image in Stereo 3D

    :type: bool
    """

    show_uvedit: bool
    ""