"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
class LambdaToDynamoDB extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToDynamoDBProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckDynamoDBProps(props);
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        // Other permissions for constructs are accepted as arrays, turning tablePermissions into
        // an array to use the same validation function.
        if (props.tablePermissions) {
            defaults.CheckListValues(['All', 'Read', 'ReadWrite', 'Write'], [props.tablePermissions], 'table permission');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.DYNAMODB);
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc
        });
        // Since we are only invoking this function with an existing Table or tableProps,
        // (not a table interface), we know that the implementation will always return
        // a Table object and we can safely cast away the optional aspect of the type.
        const buildDynamoDBTableResponse = defaults.buildDynamoDBTable(this, {
            dynamoTableProps: props.dynamoTableProps,
            existingTableObj: props.existingTableObj
        });
        this.dynamoTable = buildDynamoDBTableResponse.tableObject;
        // Configure environment variables
        const tableEnvironmentVariableName = props.tableEnvironmentVariableName || 'DDB_TABLE_NAME';
        this.lambdaFunction.addEnvironment(tableEnvironmentVariableName, this.dynamoTable.tableName);
        // Add the requested or default table permissions
        if (props.tablePermissions) {
            const permissions = props.tablePermissions.toUpperCase();
            if (permissions === 'ALL') {
                this.dynamoTable.grantFullAccess(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'READ') {
                this.dynamoTable.grantReadData(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'READWRITE') {
                this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'WRITE') {
                this.dynamoTable.grantWriteData(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        }
        // Conditional metadata for cfn_nag
        if (props.dynamoTableProps?.billingMode === dynamodb.BillingMode.PROVISIONED) {
            defaults.addCfnSuppressRules(this.dynamoTable, [
                {
                    id: "W73",
                    reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`,
                },
            ]);
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
_a = JSII_RTTI_SYMBOL_1;
LambdaToDynamoDB[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-dynamodb.LambdaToDynamoDB", version: "2.90.3" };
//# sourceMappingURL=data:application/json;base64,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